from alyios_windows import (
    select_option,
    select_from_list,
    confirm,
    get_text_input,
    display_menu,
    get_click,
    get_key,
    get_mouse_position,
    select_file,
    select_folder,
    save_file_dialog
)
from enum import Enum


class DemoOptions(Enum):
    """Example enum for menu options"""
    CONSOLE_DEMO = "console"
    INPUT_DEMO = "input"
    DIALOG_DEMO = "dialog"
    EXIT = "exit"


def console_demo():
    """Demonstrate console interface features"""
    print("\n" + "=" * 60)
    print("CONSOLE INTERFACE DEMO")
    print("=" * 60)

    # Simple selection with list
    print("\n1. Simple Selection (List of Strings)")
    color = select_option(
        "Choose your favorite color:",
        ["Red", "Green", "Blue", "Yellow"]
    )
    print(f"You selected: {color}")

    # Selection with Enum
    print("\n2. Selection with Enum")
    class Fruits(Enum):
        APPLE = "apple"
        BANANA = "banana"
        ORANGE = "orange"

    fruit = select_option("Choose a fruit:", Fruits)
    print(f"You selected: {fruit}")

    # Multi-selection
    print("\n3. Multi-Selection")
    toppings = select_from_list(
        "Choose pizza toppings:",
        ["Cheese", "Pepperoni", "Mushrooms", "Olives", "Onions"],
        multi=True
    )
    print(f"You selected: {', '.join(toppings) if toppings else 'None'}")

    # Confirmation dialog
    print("\n4. Confirmation")
    if confirm("Do you want to continue?", default=True):
        print("User confirmed!")
    else:
        print("User declined!")

    # Text input
    print("\n5. Text Input")
    name = get_text_input("Enter your name:", default="User")
    print(f"Hello, {name}!")


def input_demo():
    """Demonstrate input capture features"""
    print("\n" + "=" * 60)
    print("INPUT CAPTURE DEMO")
    print("=" * 60)

    # Get current mouse position
    print("\n1. Current Mouse Position")
    x, y = get_mouse_position()
    print(f"Mouse is currently at: ({x}, {y})")

    # Wait for mouse click
    print("\n2. Mouse Click Capture")
    if confirm("Click somewhere on the screen?"):
        x, y, button = get_click()
        print(f"You clicked at ({x}, {y}) with {button} button")

    # Wait for key press
    print("\n3. Keyboard Capture")
    if confirm("Press any key?"):
        key = get_key()
        print(f"You pressed: {key}")


def dialog_demo():
    """Demonstrate file dialog features"""
    print("\n" + "=" * 60)
    print("FILE DIALOG DEMO")
    print("=" * 60)

    menu = {
        "Select a file": "file",
        "Select multiple files": "multi_file",
        "Select a folder": "folder",
        "Save file dialog": "save",
        "Back to main menu": "back"
    }

    while True:
        choice = display_menu("File Dialog Options", menu)

        if choice == "file":
            filepath = select_file(
                title="Select a File",
                filetypes=[("Text files", "*.txt"), ("Python files", "*.py"), ("All files", "*.*")]
            )
            if filepath:
                print(f"\nSelected file: {filepath}")
            else:
                print("\nNo file selected")

        elif choice == "multi_file":
            files = select_file(
                title="Select Multiple Files",
                filetypes=[("All files", "*.*")],
                multiple=True
            )
            if files:
                print(f"\nSelected {len(files)} file(s):")
                for f in files:
                    print(f"  - {f}")
            else:
                print("\nNo files selected")

        elif choice == "folder":
            folder = select_folder(title="Select a Folder")
            if folder:
                print(f"\nSelected folder: {folder}")
            else:
                print("\nNo folder selected")

        elif choice == "save":
            filepath = save_file_dialog(
                title="Save File As",
                initialfile="document.txt",
                defaultextension=".txt",
                filetypes=[("Text files", "*.txt"), ("All files", "*.*")]
            )
            if filepath:
                print(f"\nSave location: {filepath}")
            else:
                print("\nSave cancelled")

        elif choice == "back":
            break


def main():
    """Main demonstration menu"""
    print("\n" + "=" * 60)
    print("ALYIOS WINDOWS FUNCTIONS - DEMONSTRATION")
    print("=" * 60)
    print("\nThis package provides:")
    print("  - Interactive console menus with arrow key navigation")
    print("  - Mouse and keyboard input capture via Windows API")
    print("  - Native Windows file/folder dialogs")
    print("  - Pure Python - No external dependencies!")

    while True:
        print("\n")
        choice = select_option(
            "Select a demo to run:",
            DemoOptions
        )

        if choice == "console":
            console_demo()
        elif choice == "input":
            input_demo()
        elif choice == "dialog":
            dialog_demo()
        elif choice == "exit":
            print("\nThank you for trying AlyiosWindowsFunctions!")
            break

        input("\nPress Enter to continue...")


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\n\nExiting...")
    except Exception as e:
        print(f"\nError: {e}")
        import traceback
        traceback.print_exc()
