"""
Example: Detailed Input Capture

This demonstrates the enhanced get_input() function with:
1. Mouse position tracking
2. Button detection (left/right/middle)
3. Input type identification (mouse/keyboard)
4. Hold duration tracking
5. Modifier key states (Ctrl, Shift, Alt, Win)
6. Timestamps
"""

from alyios_windows import get_input
import time

print("=" * 70)
print("DETAILED INPUT CAPTURE EXAMPLES")
print("=" * 70)
print()

# Example 1: Simple mode (backwards compatible)
print("Example 1: Simple Mode (Original Behavior)")
print("-" * 70)
print("Click anywhere or press any key...")
result = get_input()

if result[0] == 'mouse':
    _, x, y, button = result
    print(f"✓ Mouse {button}-clicked at ({x}, {y})")
else:
    _, key = result
    print(f"✓ Key pressed: {key}")

print()
time.sleep(1)

# Example 2: Detailed mode with all information
print("Example 2: Detailed Mode (Maximum Information)")
print("-" * 70)
print("Click or press any key...")
result = get_input(detailed=True, verbose=False)

print(f"\n📊 Detailed Information:")
print(f"  Type: {result['type']}")
print(f"  Timestamp: {result['timestamp']:.3f}")
print(f"  Modifiers: {result['modifiers']}")

if result['type'] == 'mouse':
    print(f"  Position: ({result['x']}, {result['y']})")
    print(f"  Button: {result['button']}")
else:
    print(f"  Key: {result['key']}")
    print(f"  VK Code: {result['vk_code']}")

print()
time.sleep(1)

# Example 3: Track hold duration
print("Example 3: Hold Duration Tracking")
print("-" * 70)
print("Click and HOLD the mouse button (or hold a key)...")
print("Release when ready.")
result = get_input(track_duration=True, detailed=True, verbose=False)

print(f"\n⏱️  Hold Duration Information:")
print(f"  Type: {result['type']}")
print(f"  Duration: {result['duration']:.3f} seconds")

if result['type'] == 'mouse':
    print(f"  Button: {result['button']}")
    print(f"  Position: ({result['x']}, {result['y']})")
else:
    print(f"  Key: {result['key']}")

print()
time.sleep(1)

# Example 4: Detecting modifier keys
print("Example 4: Modifier Key Detection")
print("-" * 70)
print("Try clicking while holding Ctrl, Shift, or Alt...")
result = get_input(detailed=True, verbose=False)

print(f"\n🎹 Modifier Keys:")
for modifier, pressed in result['modifiers'].items():
    status = "✓ PRESSED" if pressed else "✗ not pressed"
    print(f"  {modifier.capitalize():6s}: {status}")

if result['type'] == 'mouse':
    print(f"\n  Mouse: {result['button']} button at ({result['x']}, {result['y']})")
else:
    print(f"\n  Key: {result['key']}")

print()
time.sleep(1)

# Example 5: Recording a sequence of inputs
print("Example 5: Recording Input Sequence")
print("-" * 70)
print("Perform 5 actions (any combination of clicks and keys)...")
print()

recorded_inputs = []

for i in range(5):
    print(f"Action {i+1}/5:", end=" ", flush=True)

    result = get_input(track_duration=True, detailed=True, verbose=False)
    recorded_inputs.append(result)

    if result['type'] == 'mouse':
        print(f"Mouse {result['button']} at ({result['x']}, {result['y']}) "
              f"[{result['duration']:.2f}s]")
    else:
        print(f"Key '{result['key']}' [{result['duration']:.2f}s]")

    time.sleep(0.3)

print()
print("📝 Recorded Sequence Summary:")
print("-" * 70)

for i, inp in enumerate(recorded_inputs, 1):
    mods = [k for k, v in inp['modifiers'].items() if v]
    mod_str = f"+{'+'.join(mods)}" if mods else ""

    if inp['type'] == 'mouse':
        print(f"  {i}. Mouse {inp['button']:6s} at ({inp['x']:4d}, {inp['y']:4d}) "
              f"held {inp['duration']:.2f}s {mod_str}")
    else:
        print(f"  {i}. Key '{inp['key']:8s}' held {inp['duration']:.2f}s {mod_str}")

print()
print("=" * 70)
print("All examples completed!")
print("=" * 70)
print()
print("💡 Use Cases:")
print("  • Macro recording & playback")
print("  • User input analytics")
print("  • Accessibility tools")
print("  • Automation testing")
print("  • Gaming input capture")
print("  • Gesture recognition")
