"""
Example script demonstrating the new features in AlyiosWindowsFunctions
Run this to see clipboard, screen, notifications, and audio in action!
"""

from alyios_windows import clipboard, screen, notifications, audio, select_option
import time


def demo_clipboard():
    """Demonstrate clipboard operations"""
    print("\n" + "="*60)
    print("CLIPBOARD DEMO")
    print("="*60)

    # Save original clipboard content
    original = clipboard.get_text()

    # Test text operations
    print("\n1. Testing text operations...")
    clipboard.set_text("Hello from AlyiosWindowsFunctions!")
    print(f"   Set clipboard to: 'Hello from AlyiosWindowsFunctions!'")

    text = clipboard.get_text()
    print(f"   Retrieved: '{text}'")

    # Test copy/paste aliases
    print("\n2. Testing copy/paste aliases...")
    clipboard.copy("Testing copy alias")
    pasted = clipboard.paste()
    print(f"   Copied and pasted: '{pasted}'")

    # Restore original
    if original:
        clipboard.set_text(original)
        print(f"\n   Restored original clipboard content")
    else:
        clipboard.clear()
        print(f"\n   Cleared clipboard")


def demo_screen():
    """Demonstrate screen capture and display info"""
    print("\n" + "="*60)
    print("SCREEN DEMO")
    print("="*60)

    # Get display info
    print("\n1. Display Information:")
    count = screen.get_screen_count()
    print(f"   Connected displays: {count}")

    width, height = screen.get_primary_resolution()
    print(f"   Primary resolution: {width}x{height}")

    # List all displays
    print("\n2. All Displays:")
    displays = screen.list_displays()
    if displays:
        for display in displays:
            print(f"   {display}")

    # Take a screenshot
    print("\n3. Taking screenshot...")
    if screen.screenshot("demo_screenshot.png"):
        print("   Screenshot saved as: demo_screenshot.png")

    # Get detailed info
    print("\n4. Detailed Display Info:")
    info = screen.get_display_info()
    if info:
        for i, scr in enumerate(info['screens']):
            print(f"   Screen {i}:")
            print(f"     Resolution: {scr['width']}x{scr['height']}")
            print(f"     Position: ({scr['x']}, {scr['y']})")
            print(f"     Primary: {scr['primary']}")
            print(f"     Bits per pixel: {scr['bits_per_pixel']}")


def demo_notifications():
    """Demonstrate notification system"""
    print("\n" + "="*60)
    print("NOTIFICATIONS DEMO")
    print("="*60)
    print("\nShowing various notifications...")
    print("(Watch your system tray!)")

    # Info notification
    print("\n1. Info notification...")
    notifications.show_info("Demo", "This is an info notification", timeout=3000)
    time.sleep(3.5)

    # Warning notification
    print("2. Warning notification...")
    notifications.show_warning("Demo", "This is a warning notification", timeout=3000)
    time.sleep(3.5)

    # Error notification
    print("3. Error notification...")
    notifications.show_error("Demo", "This is an error notification", timeout=3000)
    time.sleep(3.5)

    # Simple toast
    print("4. Simple toast...")
    notifications.toast("Demo complete!")
    time.sleep(3)


def demo_audio():
    """Demonstrate audio control"""
    print("\n" + "="*60)
    print("AUDIO DEMO")
    print("="*60)

    # Get current volume
    original_volume = audio.get_volume()
    print(f"\n1. Current volume: {original_volume}%")
    print(f"   Is muted: {audio.is_muted()}")

    # Test volume control
    print("\n2. Testing volume control...")
    print("   Setting volume to 30%...")
    audio.set_volume(30)
    time.sleep(1)
    print(f"   Current volume: {audio.get_volume()}%")

    print("   Increasing by 20%...")
    audio.increase_volume(20)
    time.sleep(1)
    print(f"   Current volume: {audio.get_volume()}%")

    print("   Decreasing by 10%...")
    audio.decrease_volume(10)
    time.sleep(1)
    print(f"   Current volume: {audio.get_volume()}%")

    # Test mute/unmute
    print("\n3. Testing mute/unmute...")
    print("   Muting...")
    audio.mute()
    time.sleep(1)
    print(f"   Is muted: {audio.is_muted()}")

    print("   Unmuting...")
    audio.unmute()
    time.sleep(1)
    print(f"   Is muted: {audio.is_muted()}")

    # Restore original volume
    print(f"\n4. Restoring original volume ({original_volume}%)...")
    audio.set_volume(original_volume)


def main():
    """Main demo menu"""
    print("""
    ╔════════════════════════════════════════════════════════════╗
    ║    Alyios Windows Functions - New Features Demo           ║
    ║                                                            ║
    ║    Demonstrating: Clipboard, Screen, Notifications, Audio ║
    ╚════════════════════════════════════════════════════════════╝
    """)

    demos = {
        "Clipboard Operations": demo_clipboard,
        "Screen Capture & Display": demo_screen,
        "Notifications": demo_notifications,
        "Audio Control": demo_audio,
        "Run All Demos": None,
        "Exit": None
    }

    while True:
        print("\n" + "="*60)
        choice = select_option(
            "Select a demo to run:",
            list(demos.keys())
        )

        if choice == "Exit":
            print("\nGoodbye!")
            break
        elif choice == "Run All Demos":
            print("\nRunning all demos...")
            demo_clipboard()
            demo_screen()
            demo_notifications()
            demo_audio()
            print("\n" + "="*60)
            print("All demos completed!")
            print("="*60)
        else:
            demos[choice]()

        print("\n" + "="*60)
        input("Press Enter to continue...")


if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\n\nDemo interrupted. Goodbye!")
    except Exception as e:
        print(f"\n\nError: {e}")
        import traceback
        traceback.print_exc()
