"""
Example: Using Pixel Color Detection for Automation

This demonstrates practical uses of pixel color detection:
- Detecting button state changes
- Waiting for UI elements to appear
- Finding specific colors on screen
- Monitoring visual changes
"""

from alyios_windows import screen, get_mouse_position
import time

print("=== Pixel Color Detection Examples ===\n")

# Example 1: Get color at specific position
print("Example 1: Get pixel color at coordinates")
print("-" * 40)
color = screen.get_pixel_color(100, 100)
if color:
    r, g, b = color
    print(f"Color at (100, 100): RGB({r}, {g}, {b}) - Hex: #{r:02x}{g:02x}{b:02x}")
print()

# Example 2: Get color at current mouse position
print("Example 2: Get color under mouse cursor")
print("-" * 40)
x, y = get_mouse_position()
color = screen.get_pixel_color(x, y)
if color:
    r, g, b = color
    print(f"Mouse at ({x}, {y})")
    print(f"Color: RGB({r}, {g}, {b}) - Hex: #{r:02x}{g:02x}{b:02x}")
print()

# Example 3: Wait for color change (useful for detecting button clicks, loading states)
print("Example 3: Wait for color change")
print("-" * 40)
print("This would wait for a pixel to change color:")
print("  screen.wait_for_color_change(500, 500, timeout=10)")
print("Use case: Detect when a loading spinner disappears")
print()

# Example 4: Wait for specific color (useful for waiting for UI elements)
print("Example 4: Wait for specific color")
print("-" * 40)
print("This would wait for a pixel to become a specific color:")
print("  target_color = (255, 0, 0)  # Red")
print("  screen.wait_for_color(500, 500, target_color, tolerance=10, timeout=30)")
print("Use case: Wait for a button to turn green (indicating ready state)")
print()

# Example 5: Find color on screen
print("Example 5: Find color on entire screen")
print("-" * 40)
print("This would search for a color on the screen:")
print("  target_color = (0, 120, 215)  # Windows blue")
print("  position = screen.find_color_on_screen(target_color, tolerance=10)")
print("  if position:")
print("      x, y = position")
print("      print(f'Found color at ({x}, {y})')")
print()

# Example 6: Find color in specific region (faster)
print("Example 6: Find color in specific region")
print("-" * 40)
print("This would search only in a specific area:")
print("  region = (0, 0, 800, 600)  # x, y, width, height")
print("  position = screen.find_color_on_screen(target_color, region=region)")
print("Use case: Search for a color only in the top-left corner")
print()

# Practical automation example
print("=" * 50)
print("PRACTICAL EXAMPLE: Button Click Detection")
print("=" * 50)
print("""
# Wait for a button to change from gray to blue (indicating it's clickable)
from alyios_windows import screen, click, get_mouse_position

# 1. Move mouse over the button and note its position
print("Move mouse over button, then press Enter")
input()
button_x, button_y = get_mouse_position()

# 2. Get the current color (gray, disabled state)
current_color = screen.get_pixel_color(button_x, button_y)
print(f"Button color (disabled): {current_color}")

# 3. Wait for button to turn blue (enabled state)
blue_color = (0, 120, 215)  # Windows blue
print("Waiting for button to become enabled...")
screen.wait_for_color(button_x, button_y, blue_color, tolerance=20, timeout=30)

# 4. Click the button
print("Button is ready! Clicking...")
click(button_x, button_y)
""")

print("\n" + "=" * 50)
print("All pixel detection features are ready to use!")
print("=" * 50)
