# Build Alyios Automation Studio - Professional Edition
# Compiles multi-file project into single executable

$scriptPath = Split-Path -Parent $MyInvocation.MyCommand.Path
$outputExe = Join-Path $scriptPath "AlyiosWindowsFunctions\AutomationStudio.exe"

Write-Host "================================================" -ForegroundColor Cyan
Write-Host "  Alyios Automation Studio - Professional Build" -ForegroundColor Cyan
Write-Host "================================================" -ForegroundColor Cyan
Write-Host ""

# Source files
$sourceFiles = @(
    "AlyiosWindowsFunctions\Automation_ActionModels.cs",
    "AlyiosWindowsFunctions\Automation_InputCore.cs",
    "AlyiosWindowsFunctions\Automation_ScriptEngine.cs"
)

# Check all files exist
$allFilesExist = $true
foreach ($file in $sourceFiles) {
    $fullPath = Join-Path $scriptPath $file
    if (-not (Test-Path $fullPath)) {
        Write-Host "ERROR: Missing file: $file" -ForegroundColor Red
        $allFilesExist = $false
    } else {
        Write-Host "[OK] Found: $file" -ForegroundColor Green
    }
}

if (-not $allFilesExist) {
    Write-Host ""
    Write-Host "Build cannot continue - missing source files!" -ForegroundColor Red
    exit 1
}

Write-Host ""
Write-Host "All source files found!" -ForegroundColor Green
Write-Host ""

# Find C# compiler
Write-Host "Locating C# compiler..." -ForegroundColor Yellow
$cscPath = $null
$frameworkPaths = @(
    "C:\Windows\Microsoft.NET\Framework64\v4.0.30319\csc.exe",
    "C:\Windows\Microsoft.NET\Framework\v4.0.30319\csc.exe"
)

foreach ($path in $frameworkPaths) {
    if (Test-Path $path) {
        $cscPath = $path
        break
    }
}

if (-not $cscPath) {
    Write-Host "ERROR: C# compiler (csc.exe) not found!" -ForegroundColor Red
    Write-Host "Please install .NET Framework 4.0 or later" -ForegroundColor Yellow
    exit 1
}

Write-Host "Using compiler: $cscPath" -ForegroundColor Green
Write-Host ""

# Build full source file list
$fullSourcePaths = @()
foreach ($file in $sourceFiles) {
    $fullSourcePaths += Join-Path $scriptPath $file
}

# Compile arguments
$compileArgs = @(
    "/target:library",
    "/out:$outputExe",
    "/reference:System.dll",
    "/reference:System.Core.dll",
    "/reference:System.Drawing.dll",
    "/reference:System.Windows.Forms.dll",
    "/reference:System.Web.Extensions.dll",
    "/optimize+",
    "/nologo",
    "/warn:0"
) + $fullSourcePaths

Write-Host "Compiling Automation Studio..." -ForegroundColor Yellow
Write-Host "Output: $outputExe" -ForegroundColor Gray
Write-Host ""

& $cscPath $compileArgs

if ($LASTEXITCODE -eq 0) {
    Write-Host ""
    Write-Host "================================================" -ForegroundColor Green
    Write-Host "  BUILD SUCCESSFUL!" -ForegroundColor Green
    Write-Host "================================================" -ForegroundColor Green
    Write-Host ""
    Write-Host "Created: $outputExe" -ForegroundColor White
    Write-Host "Size: $((Get-Item $outputExe).Length / 1KB) KB" -ForegroundColor Gray
    Write-Host ""
    Write-Host "Your professional automation library is ready!" -ForegroundColor Cyan
    Write-Host "Use it from Python with the automation module." -ForegroundColor Cyan
} else {
    Write-Host ""
    Write-Host "================================================" -ForegroundColor Red
    Write-Host "  BUILD FAILED!" -ForegroundColor Red
    Write-Host "================================================" -ForegroundColor Red
    exit 1
}
