"""
Windows notifications (toast notifications) using Windows API via C# helper
"""

import subprocess
import os
from pathlib import Path
from typing import Literal

# Path to the compiled helper
_HELPER_EXE = Path(__file__).parent / "helpers" / "WindowsHelper.exe"


def _ensure_helper_built():
    """Ensure the helper executable exists"""
    if not _HELPER_EXE.exists():
        raise FileNotFoundError(
            f"WindowsHelper.exe not found at {_HELPER_EXE}. "
            "Please run build_windows_helper.ps1 to compile it."
        )


def notify(
    title: str,
    message: str,
    icon: Literal["info", "warning", "error", "none"] = "info",
    timeout: int = 5000
) -> bool:
    """
    Show a Windows notification (balloon tip).

    Args:
        title: Notification title
        message: Notification message
        icon: Icon type - "info", "warning", "error", or "none"
        timeout: Duration to show notification in milliseconds (default: 5000)

    Returns:
        True if successful, False otherwise

    Examples:
        >>> notify("Hello", "This is a notification!")

        >>> notify("Warning", "Something needs attention", icon="warning")

        >>> notify("Error", "An error occurred", icon="error", timeout=10000)
    """
    _ensure_helper_built()

    try:
        cmd = [
            str(_HELPER_EXE),
            "notify",
            title,
            message,
            f"--icon={icon}",
            f"--timeout={timeout}"
        ]

        result = subprocess.run(
            cmd,
            capture_output=True,
            text=True,
            timeout=(timeout / 1000) + 5,  # Add 5 seconds buffer
            creationflags=subprocess.CREATE_NO_WINDOW if os.name == 'nt' else 0
        )

        return result.returncode == 0

    except Exception as e:
        print(f"Error showing notification: {e}")
        return False


def show_info(title: str, message: str, timeout: int = 5000) -> bool:
    """
    Show an info notification.

    Args:
        title: Notification title
        message: Notification message
        timeout: Duration in milliseconds

    Returns:
        True if successful, False otherwise

    Example:
        >>> show_info("Success", "Operation completed successfully")
    """
    return notify(title, message, icon="info", timeout=timeout)


def show_warning(title: str, message: str, timeout: int = 5000) -> bool:
    """
    Show a warning notification.

    Args:
        title: Notification title
        message: Notification message
        timeout: Duration in milliseconds

    Returns:
        True if successful, False otherwise

    Example:
        >>> show_warning("Warning", "Low disk space")
    """
    return notify(title, message, icon="warning", timeout=timeout)


def show_error(title: str, message: str, timeout: int = 5000) -> bool:
    """
    Show an error notification.

    Args:
        title: Notification title
        message: Notification message
        timeout: Duration in milliseconds

    Returns:
        True if successful, False otherwise

    Example:
        >>> show_error("Error", "Failed to connect to server")
    """
    return notify(title, message, icon="error", timeout=timeout)


def toast(message: str, title: str = "Notification") -> bool:
    """
    Show a simple toast notification with default settings.

    Args:
        message: Notification message
        title: Notification title (default: "Notification")

    Returns:
        True if successful, False otherwise

    Example:
        >>> toast("Task completed!")
    """
    return notify(title, message)
