#!/usr/bin/env python3
"""
Launcher for Alyios Automation Studio
"""
import os
import sys
import subprocess
from pathlib import Path


def get_executable_path():
    """Get the path to the automation-studio executable"""
    # Get the directory where this script is located
    package_dir = Path(__file__).parent
    bin_dir = package_dir / 'bin'

    # Determine executable name based on platform
    if sys.platform == 'win32':
        exe_name = 'automation-studio.exe'
    elif sys.platform == 'darwin':
        exe_name = 'automation-studio.app'
    else:
        exe_name = 'automation-studio'

    exe_path = bin_dir / exe_name

    if not exe_path.exists():
        raise FileNotFoundError(
            f"Automation Studio executable not found at {exe_path}\n"
            f"The executable may not be included in this installation."
        )

    return exe_path


def main():
    """Launch Automation Studio"""
    try:
        exe_path = get_executable_path()

        # Make executable on Unix systems
        if sys.platform != 'win32':
            os.chmod(exe_path, 0o755)

        # Launch the application
        print(f"Launching Automation Studio from: {exe_path}")

        if sys.platform == 'darwin' and str(exe_path).endswith('.app'):
            # macOS app bundle
            subprocess.Popen(['open', str(exe_path)])
        else:
            # Windows/Linux executable
            subprocess.Popen([str(exe_path)])

        print("Automation Studio launched successfully!")

    except FileNotFoundError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Failed to launch Automation Studio: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == '__main__':
    main()
