
from pydantic import BaseModel, ConfigDict
from typing import Any, Dict, Optional, Union, List


class Message(BaseModel):
    """Chat message class."""

    role: str
    """The role of the messages author (system, user, assistant or tool)."""

    content: Optional[Union[str, list[Union[str, dict]]]] = None
    """Content of the message."""

    name: Optional[str] = None
    """An optional name for the participant."""

    tool_call_id: Optional[str] = None
    """Tool call that this message is responding to."""
    
    tool_calls: Optional[List[Dict[str, Any]]] = None
    """The tool calls generated by the model, such as function calls."""

    thinking: Optional[str] = None
    """Thinking content from the model."""

    model_config = ConfigDict(extra="allow", populate_by_name=True, arbitrary_types_allowed=True)


    def __str__(self) -> str:
        return f"{self.role}: {self.content}"

    def to_dict(self, **kwargs: Any) -> Dict[str, Any]:
        message_dict = self.model_dump(**kwargs)
        message_dict = {
            k: v for k, v in message_dict.items() if v is not None and not (isinstance(v, (list, dict)) and len(v) == 0)
        }
        return message_dict
