# CleanFrames

## Overview

CleanFrames is an advanced tool designed to identify and remove duplicate or near-duplicate images from large datasets using multiple embedding models and sophisticated clustering techniques. It supports exact and perceptual duplicate detection, semantic similarity analysis via deep embeddings, and offers visualization and detailed tabulated reporting for thorough dataset cleaning.

## Features

- Multi-model embedding support: **Swin**, **CLIP**, **DINO**, **ResNet**.
- Exact duplicate detection using MD5 hashing.
- Semantic similarity detection with deep embeddings and clustering.
- Flexible cleaning modes: path-only, embedding-based, or custom embeddings.
- Clustering to group similar images and identify duplicates.
- Visualization tools for inspecting clusters and embeddings.
- Detailed tabulated reports with removed duplicates, retained images, and thresholds.
- Device support for CPU, CUDA GPU, and Apple MPS.
- Efficient caching system to store and reuse embeddings for faster processing.

## Installation

Install CleanFrames easily via pip:

```bash
pip install cleanframes
```

## Usage

### Basic Cleaning by Path

CleanFrames can process a folder of images, compute embeddings using the default Swin model, and remove duplicates.

```python
from cleanframes import CleanFrame

cleaner = CleanFrame(device='cuda')  # or 'cpu', 'mps'
input_folder = "path/to/images"

cleaner.cleanframe(input_folder)
```

This creates an output folder inside the specified directory (default: `frames_cleaned/`) containing the unique images after cleaning.

A detailed tabulated report is printed to the console summarizing the cleaning results.

### Generate Embeddings and Clean

Generate embeddings separately and then clean based on those embeddings:

```python
from cleanframes import CleanFrame

cleaner = CleanFrame(device='cuda')
input_folder = "path/to/images"

embeddings, paths = cleaner.SwinEmbedding(input_folder)

cleaner.cleanframe(paths, embeddings_list=[("swin", embeddings)], threshold=0.95)
```

### Clean Using Custom Embeddings

Supply your own embeddings (e.g., precomputed vectors) for cleaning:

```python
from cleanframes import CleanFrame
import numpy as np

cleaner = CleanFrame(device='cpu')
image_paths = [...]  # list of image file paths
custom_embeddings = np.load("custom_embeddings.npy")

cleaner.cleanframe(image_paths, embeddings_list=[("custom_model", custom_embeddings)], threshold=0.9)
```

## Clustering & Visualization

CleanFrames groups similar images using clustering algorithms on embeddings to identify duplicates and near-duplicates effectively.

You can also visualize clusters and embeddings to inspect dataset structure:

```python
cleaner.visualize_clusters(embeddings, image_paths)
```

This helps in understanding similarity groups and verifying cleaning results.

## Report

After cleaning, CleanFrames prints a comprehensive tabulated report including:

- Number of duplicates removed.
- Images retained.
- Threshold values used.
- Embedding models applied.
- Cluster information.

This report facilitates audit and reproducibility of dataset cleaning.

## Supported Models

- **Swin**: Hierarchical Vision Transformer for image representation.
- **CLIP**: Contrastive Language-Image Pretraining embeddings.
- **DINO**: Self-distillation with no labels for visual features.
- **ResNet**: Classic convolutional neural network embeddings.

Generate embeddings with corresponding methods like `cleaner.CLIPEmbedding()`, `cleaner.DINOEmbedding()`, etc.

## Device Support

CleanFrames supports multiple devices for accelerated embedding computation:

- **CPU**: Default fallback.
- **CUDA GPU**: For NVIDIA GPUs.
- **MPS**: Apple's Metal Performance Shaders for Macs with Apple Silicon.

Specify device during initialization:

```python
cleaner = CleanFrame(device='mps')  # or 'cuda', 'cpu'
```

## Caching System

CleanFrames includes a caching mechanism to save and load embeddings, clusters, and visualizations, reducing redundant computations on repeated runs:

- Automatically caches `.npz` files per folder and model inside the `.cleanframe_cache/` directory.
- Loads cached embeddings and clusters to speed up cleaning.
- Manages cache files for efficient storage and reuse.

Example:

```python
embeddings, paths = cleaner.SwinEmbedding(input_folder, use_cache=True)
```
