"""Formatting and whitespace normalization patterns."""

from .core import ValidatedPattern

PATTERNS: dict[str, ValidatedPattern] = {
    "fix_command_spacing": ValidatedPattern(
        name="fix_command_spacing",
        pattern=r"python\s*-\s*m\s+(\w+)",
        replacement=r"python -m \1",
        description="Fix spacing in 'python -m command' patterns",
        test_cases=[
            ("python - m crackerjack", "python -m crackerjack"),
            ("python -m crackerjack", "python -m crackerjack"),
            ("python - m pytest", "python -m pytest"),
            ("other python - m stuff", "other python -m stuff"),
        ],
    ),
    "fix_long_flag_spacing": ValidatedPattern(
        name="fix_long_flag_spacing",
        pattern=r"-\s*-\s*(\w+(?: -\w+)*)",
        replacement=r"--\1",
        description="Fix spacing in long flags like '--help'",
        test_cases=[
            ("- - help", "--help"),
            ("- - ai-fix", "--ai-fix"),
            ("--help", "--help"),
            ("- - start-websocket-server", "--start-websocket-server"),
        ],
    ),
    "fix_short_flag_spacing": ValidatedPattern(
        name="fix_short_flag_spacing",
        pattern=r"(?<!\w)-\s+(\w)(?!\w)",
        replacement=r"-\1",
        description="Fix spacing in short flags like '-t'",
        test_cases=[
            ("python -m crackerjack - t", "python -m crackerjack -t"),
            ("- q", "-q"),
            ("-t", "-t"),
            ("some - x flag", "some -x flag"),
        ],
    ),
    "fix_hyphenated_names": ValidatedPattern(
        name="fix_hyphenated_names",
        pattern=r"(\w+)\s*-\s*(\w+)",
        replacement=r"\1-\2",
        description="Fix spacing in hyphenated names and identifiers",
        test_cases=[
            ("python - pro", "python-pro"),
            (
                "pytest - hypothesis - specialist",
                "pytest-hypothesis - specialist",
            ),
            ("backend - architect", "backend-architect"),
            ("python-pro", "python-pro"),
            ("end - of - file-fixer", "end-of - file-fixer"),
        ],
    ),
    "fix_hyphenated_names_global": ValidatedPattern(
        name="fix_hyphenated_names_global",
        pattern=r"(\w+)\s+-\s+(\w+)",
        replacement=r"\1-\2",
        description="Globally fix spacing in hyphenated names (single pass only)",
        global_replace=True,
        test_cases=[
            ("python - pro", "python-pro"),
            ("end - of - file", "end-of - file"),
            ("already-hyphenated", "already-hyphenated"),
            ("start - middle - end", "start-middle - end"),
        ],
    ),
    "fix_spaced_hyphens": ValidatedPattern(
        name="fix_spaced_hyphens",
        pattern=r"(\w+)\s+-\s+(\w+)",
        replacement=r"\1-\2",
        description="Fix spaced hyphens with spaces around dashes (use apply_iteratively for multi-word)",
        global_replace=True,
        test_cases=[
            ("python - pro", "python-pro"),
            (
                "pytest - hypothesis - specialist",
                "pytest-hypothesis - specialist",
            ),
            (
                "end - of - file - fixer",
                "end-of - file-fixer",
            ),
            ("already-hyphenated", "already-hyphenated"),
            ("mixed-case with - spaces", "mixed-case with-spaces"),
        ],
    ),
    "fix_debug_log_pattern": ValidatedPattern(
        name="fix_debug_log_pattern",
        pattern=r"crackerjack\s*-\s*debug",
        replacement="crackerjack-debug",
        description="Fix spacing in debug log patterns",
        test_cases=[
            ("crackerjack - debug-12345.log", "crackerjack-debug-12345.log"),
            ("crackerjack-debug.log", "crackerjack-debug.log"),
            ("old crackerjack - debug files", "old crackerjack-debug files"),
        ],
    ),
    "fix_job_file_pattern": ValidatedPattern(
        name="fix_job_file_pattern",
        pattern=r"job\s*-\s*(\{[^}]+\}|\w+)",
        replacement=r"job-\1",
        description="Fix spacing in job file patterns",
        test_cases=[
            ("job - {self.web_job_id}.json", "job-{self.web_job_id}.json"),
            ("job - abc123.json", "job-abc123.json"),
            ("job-existing.json", "job-existing.json"),
        ],
    ),
    "fix_markdown_bold": ValidatedPattern(
        name="fix_markdown_bold",
        pattern=r"\*\s+\*(.+?)\s*\*\s+\*",
        replacement=r"**\1**",
        description="Fix spacing in markdown bold patterns",
        test_cases=[
            ("* *Bold Text * *", "**Bold Text**"),
            ("* *🧪 pytest-specialist * *", "**🧪 pytest-specialist**"),
            ("**Already Bold**", "**Already Bold**"),
        ],
    ),
    "remove_trailing_whitespace": ValidatedPattern(
        name="remove_trailing_whitespace",
        pattern=r"[ \t]+$",
        replacement="",
        description="Remove trailing whitespace from lines",
        global_replace=True,
        test_cases=[
            ("line with spaces ", "line with spaces"),
            ("line with tabs\t\t", "line with tabs"),
            ("normal line", "normal line"),
            ("mixed \t ", "mixed"),
            ("", ""),
        ],
    ),
    "normalize_multiple_newlines": ValidatedPattern(
        name="normalize_multiple_newlines",
        pattern=r"\n{3,}",
        replacement="\n\n",
        description="Normalize multiple consecutive newlines to maximum 2",
        global_replace=True,
        test_cases=[
            ("line1\n\n\nline2", "line1\n\nline2"),
            ("line1\n\n\n\n\nline2", "line1\n\nline2"),
            ("line1\n\nline2", "line1\n\nline2"),
            ("line1\nline2", "line1\nline2"),
        ],
    ),
    "normalize_assert_statements": ValidatedPattern(
        name="normalize_assert_statements",
        pattern=r"assert (.+?)\s*==\s*(.+)",
        replacement=r"assert \1 == \2",
        description="Normalize spacing around == in assert statements",
        global_replace=True,
        test_cases=[
            ("assert result==expected", "assert result == expected"),
            ("assert value == other", "assert value == other"),
            ("assert result== expected", "assert result == expected"),
            ("assert result ==expected", "assert result == expected"),
            (
                "assert result == expected",
                "assert result == expected",
            ),
        ],
    ),
    "spacing_after_comma": ValidatedPattern(
        name="spacing_after_comma",
        pattern=r",(?! |\n|$)",
        replacement=r", ",
        description="Add space after comma if missing (skip if already spaced, at end of line, or end of string)",
        global_replace=True,
        test_cases=[
            ("func(a,b,c)", "func(a, b, c)"),
            ("func(a, b, c)", "func(a, b, c)"),
            ("[1,2,3]", "[1, 2, 3]"),
            ("[1, 2, 3]", "[1, 2, 3]"),
            ("{'x':1,'y':2,'z':3}", "{'x':1, 'y':2, 'z':3}"),
            ("trailing,\nnewline", "trailing,\nnewline"),
            ("end_comma,", "end_comma,"),
        ],
    ),
    "spacing_after_colon": ValidatedPattern(
        name="spacing_after_colon",
        pattern=r"(?<!https)(?<!http)(?<!ftp)(?<!file)(?<!: )(\b[a-zA-Z_][a-zA-Z0-9_]*):([a-zA-Z0-9_][^ \n:]*)",
        replacement=r"\1: \2",
        global_replace=True,
        description="Add space after colon if missing (avoid double colons, URLs, and protocols)",
        test_cases=[
            ("def func(x: int, y: str): ", "def func(x: int, y: str): "),
            ("dict_item = {'key': 'value'}", "dict_item = {'key': 'value'}"),
            ("already: spaced: properly", "already: spaced: properly"),
            ("class::method", "class::method"),
            ("https://github.com", "https://github.com"),
            ("http://example.com", "http://example.com"),
            ("ftp://server.com", "ftp://server.com"),
            ("repo:local", "repo: local"),
        ],
    ),
    "multiple_spaces": ValidatedPattern(
        name="multiple_spaces",
        pattern=r" {2,}",
        replacement=r" ",
        description="Replace multiple spaces with single space",
        global_replace=True,
        test_cases=[
            ("def func( x, y ): ", "def func( x, y ): "),
            ("single space only", "single space only"),
            ("lots of spaces", "lots of spaces"),
            ("\tkeep\ttabs\tbut fix spaces", "\tkeep\ttabs\tbut fix spaces"),
        ],
    ),
    "normalize_whitespace": ValidatedPattern(
        name="normalize_whitespace",
        pattern=r"\s+",
        replacement=r" ",
        description="Normalize multiple whitespace to single space",
        global_replace=True,
        test_cases=[
            ("import module", "import module"),
            ("from pkg import item", "from pkg import item"),
            ("normal text", "normal text"),
        ],
    ),
}
