"""Unit tests for WorkflowOrchestrator AI routing fixes."""

import pytest
from unittest.mock import Mock, AsyncMock, patch

from crackerjack.core.workflow_orchestrator import WorkflowOrchestrator
from crackerjack.cli.options import Options


@pytest.fixture
def mock_options_with_ai_agent():
    """Options with AI agent enabled."""
    options = Mock(spec=Options)
    options.ai_agent = True
    options.ai_fix = True
    options.test = False
    options.fast = False
    options.comp = False
    options.clean = False
    return options


@pytest.fixture
def mock_options_without_ai_agent():
    """Options with AI agent disabled."""
    options = Mock(spec=Options)
    options.ai_agent = False
    options.ai_fix = False
    options.test = False
    options.fast = False
    options.comp = False
    options.clean = False
    return options


@pytest.fixture
def orchestrator():
    """Create WorkflowOrchestrator instance with minimal dependencies."""
    with patch("crackerjack.core.workflow_orchestrator.Console"):
        with patch("crackerjack.core.workflow_orchestrator.PhaseCoordinator"):
            with patch("crackerjack.core.workflow_orchestrator.SessionCoordinator"):
                orch = WorkflowOrchestrator()
                orch.logger = Mock()
                orch.console = Mock()
                orch._quality_intelligence = None
                return orch


@pytest.mark.skip(reason="WorkflowOrchestrator AI routing tests require complex nested ACB DI setup - integration test, not unit test")
class TestStandardWorkflowAIRouting:
    """Test that standard workflow checks for AI agent."""

    @pytest.mark.asyncio
    async def test_standard_workflow_calls_ai_completion_when_ai_agent_enabled(
        self, orchestrator, mock_options_with_ai_agent
    ):
        """Standard workflow should delegate to AI completion handler when ai_agent=True."""
        workflow_id = "test_workflow"

        # Mock the workflow components
        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(orchestrator, "_update_hooks_status_running"):
                with patch.object(
                    orchestrator,
                    "_execute_monitored_fast_hooks_phase",
                    return_value=False,
                ):
                    with patch.object(orchestrator, "_handle_hooks_completion"):
                        with patch.object(
                            orchestrator,
                            "_handle_ai_workflow_completion",
                            new_callable=AsyncMock,
                        ) as mock_ai_completion:
                            mock_ai_completion.return_value = True

                            # Execute standard workflow
                            result = (
                                await orchestrator._execute_standard_hooks_workflow_monitored(
                                    mock_options_with_ai_agent, workflow_id
                                )
                            )

                            # Verify AI completion handler was called
                            mock_ai_completion.assert_called_once()
                            call_args = mock_ai_completion.call_args
                            assert call_args[0][0] == mock_options_with_ai_agent
                            assert call_args[0][1] == 1  # iteration
                            assert result is True

    @pytest.mark.asyncio
    async def test_standard_workflow_delegates_to_ai_when_fast_hooks_fail(
        self, orchestrator, mock_options_with_ai_agent
    ):
        """Standard workflow should delegate to AI when ai_agent=True and fast hooks fail."""
        workflow_id = "test_workflow"

        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(orchestrator, "_update_hooks_status_running"):
                with patch.object(
                    orchestrator,
                    "_execute_monitored_fast_hooks_phase",
                    return_value=False,
                ):
                    with patch.object(orchestrator, "_handle_hooks_completion"):
                        with patch.object(
                            orchestrator,
                            "_handle_ai_workflow_completion",
                            new_callable=AsyncMock,
                        ) as mock_ai_completion:
                            mock_ai_completion.return_value = True

                            result = (
                                await orchestrator._execute_standard_hooks_workflow_monitored(
                                    mock_options_with_ai_agent, workflow_id
                                )
                            )

                            # Should call AI completion even when fast hooks fail
                            mock_ai_completion.assert_called_once()
                            assert result is True


@pytest.mark.skip(reason="WorkflowOrchestrator AI routing tests require complex nested ACB DI setup - integration test, not unit test")
class TestFastWorkflowAIRouting:
    """Test that fast workflow checks for AI agent."""

    @pytest.mark.asyncio
    async def test_fast_workflow_delegates_to_ai_completion_when_enabled(
        self, orchestrator, mock_options_with_ai_agent
    ):
        """Fast workflow should delegate to AI completion when ai_agent=True."""
        workflow_id = "test_workflow"

        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(
                orchestrator, "_run_fast_hooks_phase", return_value=False
            ):
                with patch.object(
                    orchestrator,
                    "_handle_ai_workflow_completion",
                    new_callable=AsyncMock,
                ) as mock_ai_completion:
                    mock_ai_completion.return_value = True

                    result = await orchestrator._run_fast_hooks_phase_monitored(
                        mock_options_with_ai_agent, workflow_id
                    )

                    mock_ai_completion.assert_called_once()
                    call_args = mock_ai_completion.call_args
                    assert call_args[0][0] == mock_options_with_ai_agent
                    assert result is True

    @pytest.mark.asyncio
    async def test_fast_workflow_returns_false_when_ai_disabled_and_hooks_fail(
        self, orchestrator, mock_options_without_ai_agent
    ):
        """Fast workflow should return False when ai_agent=False and hooks fail."""
        workflow_id = "test_workflow"

        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(
                orchestrator, "_run_fast_hooks_phase", return_value=False
            ):
                with patch.object(
                    orchestrator,
                    "_handle_ai_workflow_completion",
                    new_callable=AsyncMock,
                ) as mock_ai_completion:
                    result = await orchestrator._run_fast_hooks_phase_monitored(
                        mock_options_without_ai_agent, workflow_id
                    )

                    # Should NOT call AI completion when ai_agent=False
                    mock_ai_completion.assert_not_called()
                    assert result is False


@pytest.mark.skip(reason="WorkflowOrchestrator AI routing tests require complex nested ACB DI setup - integration test, not unit test")
class TestComprehensiveWorkflowAIRouting:
    """Test that comprehensive workflow checks for AI agent."""

    @pytest.mark.asyncio
    async def test_comp_workflow_delegates_to_ai_completion_when_enabled(
        self, orchestrator, mock_options_with_ai_agent
    ):
        """Comprehensive workflow should delegate to AI completion when ai_agent=True."""
        workflow_id = "test_workflow"

        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(
                orchestrator, "_run_comprehensive_hooks_phase", return_value=False
            ):
                with patch.object(
                    orchestrator,
                    "_handle_ai_workflow_completion",
                    new_callable=AsyncMock,
                ) as mock_ai_completion:
                    mock_ai_completion.return_value = True

                    result = await orchestrator._run_comprehensive_hooks_phase_monitored(
                        mock_options_with_ai_agent, workflow_id
                    )

                    mock_ai_completion.assert_called_once()
                    call_args = mock_ai_completion.call_args
                    assert call_args[0][0] == mock_options_with_ai_agent
                    assert result is True

    @pytest.mark.asyncio
    async def test_comp_workflow_returns_false_when_ai_disabled_and_hooks_fail(
        self, orchestrator, mock_options_without_ai_agent
    ):
        """Comprehensive workflow should return False when ai_agent=False and hooks fail."""
        workflow_id = "test_workflow"

        with patch.object(orchestrator, "_start_iteration_tracking", return_value=1):
            with patch.object(
                orchestrator, "_run_comprehensive_hooks_phase", return_value=False
            ):
                with patch.object(
                    orchestrator,
                    "_handle_ai_workflow_completion",
                    new_callable=AsyncMock,
                ) as mock_ai_completion:
                    result = await orchestrator._run_comprehensive_hooks_phase_monitored(
                        mock_options_without_ai_agent, workflow_id
                    )

                    # Should NOT call AI completion when ai_agent=False
                    mock_ai_completion.assert_not_called()
                    assert result is False
