import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { IFleet } from './fleet-base';
import { IGameSessionQueueDestination } from './game-session-queue';
/**
 * Represents a Gamelift Alias for a Gamelift fleet destination.
 */
export interface IAlias extends cdk.IResource, IGameSessionQueueDestination {
    /**
       * The Identifier of the alias.
       *
       * @attribute
       */
    readonly aliasId: string;
    /**
     * The ARN of the alias.
     *
     * @attribute
     */
    readonly aliasArn: string;
}
/**
 * Options for `gamelift.Alias`.
 */
export interface AliasOptions {
    /**
       * Description for the alias
       *
       * @default No description
       */
    readonly description?: string;
}
/**
 * A full specification of an alias that can be used to import it fluently into the CDK application.
 */
export interface AliasAttributes {
    /**
         * The ARN of the alias
         *
         * At least one of `aliasArn` and `aliasId` must be provided.
         *
         * @default derived from `aliasId`.
         */
    readonly aliasArn?: string;
    /**
       * The identifier of the alias
       *
       * At least one of `aliasId` and `aliasArn`  must be provided.
       *
       * @default derived from `aliasArn`.
       */
    readonly aliasId?: string;
}
/**
 * Properties for a new Fleet alias
 */
export interface AliasProps {
    /**
       * Name of this alias
       */
    readonly aliasName: string;
    /**
       * A human-readable description of the alias
       *
       * @default no description
       */
    readonly description?: string;
    /**
     * A fleet that the alias points to.
     * If specified, the alias resolves to one specific fleet.
     *
     * At least one of `fleet` and `terminalMessage` must be provided.
     *
     * @default no fleet that the alias points to.
     */
    readonly fleet?: IFleet;
    /**
     * The message text to be used with a terminal routing strategy.
     *
     * At least one of `fleet` and `terminalMessage` must be provided.
     *
     * @default no terminal message
     */
    readonly terminalMessage?: string;
}
/**
 * Base class for new and imported GameLift Alias.
 */
export declare abstract class AliasBase extends cdk.Resource implements IAlias {
    /**
   * The Identifier of the alias.
   */
    abstract readonly aliasId: string;
    /**
     * The ARN of the alias
     */
    abstract readonly aliasArn: string;
    /**
     * The ARN to put into the destination field of a game session queue
     */
    get resourceArnForDestination(): string;
}
/**
 * A Amazon GameLift alias is used to abstract a fleet designation.
 * Fleet designations tell GameLift where to search for available resources when creating new game sessions for players.
 * Use aliases instead of specific fleet IDs to seamlessly switch player traffic from one fleet to another by changing the alias's target location.
 *
 * Aliases are useful in games that don't use queues.
 * Switching fleets in a queue is a simple matter of creating a new fleet, adding it to the queue, and removing the old fleet, none of which is visible to players.
 * In contrast, game clients that don't use queues must specify which fleet to use when communicating with the GameLift service.
 * Without aliases, a fleet switch requires updates to your game code and possibly distribution of an updated game clients to players.
 *
 * When updating the fleet-id an alias points to, there is a transition period of up to 2 minutes where game sessions on the alias may end up on the old fleet.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/aliases-creating.html
 *
 * @resource AWS::GameLift::Alias
 */
export declare class Alias extends AliasBase {
    /**
     * Import an existing alias from its identifier.
     */
    static fromAliasId(scope: Construct, id: string, aliasId: string): IAlias;
    /**
     * Import an existing alias from its ARN.
     */
    static fromAliasArn(scope: Construct, id: string, aliasArn: string): IAlias;
    /**
     * Import an existing alias from its attributes.
     */
    static fromAliasAttributes(scope: Construct, id: string, attrs: AliasAttributes): IAlias;
    /**
     * The Identifier of the alias.
     */
    readonly aliasId: string;
    /**
      * The ARN of the alias.
      */
    readonly aliasArn: string;
    /**
     * A fleet that the alias points to.
     */
    readonly fleet?: IFleet;
    constructor(scope: Construct, id: string, props: AliasProps);
    private parseRoutingStrategy;
}
