"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = exports.AliasBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * Base class for new and imported GameLift Alias.
 */
class AliasBase extends cdk.Resource {
    /**
     * The ARN to put into the destination field of a game session queue
     */
    get resourceArnForDestination() {
        return this.aliasArn;
    }
}
exports.AliasBase = AliasBase;
_a = JSII_RTTI_SYMBOL_1;
AliasBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.AliasBase", version: "2.65.0-alpha.0" };
/**
 * A Amazon GameLift alias is used to abstract a fleet designation.
 * Fleet designations tell GameLift where to search for available resources when creating new game sessions for players.
 * Use aliases instead of specific fleet IDs to seamlessly switch player traffic from one fleet to another by changing the alias's target location.
 *
 * Aliases are useful in games that don't use queues.
 * Switching fleets in a queue is a simple matter of creating a new fleet, adding it to the queue, and removing the old fleet, none of which is visible to players.
 * In contrast, game clients that don't use queues must specify which fleet to use when communicating with the GameLift service.
 * Without aliases, a fleet switch requires updates to your game code and possibly distribution of an updated game clients to players.
 *
 * When updating the fleet-id an alias points to, there is a transition period of up to 2 minutes where game sessions on the alias may end up on the old fleet.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/aliases-creating.html
 *
 * @resource AWS::GameLift::Alias
 */
class Alias extends AliasBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_AliasProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Alias);
            }
            throw error;
        }
        if (!cdk.Token.isUnresolved(props.aliasName)) {
            if (props.aliasName.length > 1024) {
                throw new Error(`Alias name can not be longer than 1024 characters but has ${props.aliasName.length} characters.`);
            }
        }
        if (props.description && !cdk.Token.isUnresolved(props.description)) {
            if (props.description.length > 1024) {
                throw new Error(`Alias description can not be longer than 1024 characters but has ${props.description.length} characters.`);
            }
        }
        if (!props.terminalMessage && !props.fleet) {
            throw new Error('Either a terminal message or a fleet must be binded to this Alias.');
        }
        if (props.terminalMessage && props.fleet) {
            throw new Error('Either a terminal message or a fleet must be binded to this Alias, not both.');
        }
        const resource = new aws_gamelift_1.CfnAlias(this, 'Resource', {
            name: props.aliasName,
            description: props.description,
            routingStrategy: this.parseRoutingStrategy(props),
        });
        this.aliasId = this.getResourceNameAttribute(resource.ref);
        this.aliasArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'alias',
            resourceName: this.aliasId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
     * Import an existing alias from its identifier.
     */
    static fromAliasId(scope, id, aliasId) {
        return this.fromAliasAttributes(scope, id, { aliasId: aliasId });
    }
    /**
     * Import an existing alias from its ARN.
     */
    static fromAliasArn(scope, id, aliasArn) {
        return this.fromAliasAttributes(scope, id, { aliasArn: aliasArn });
    }
    /**
     * Import an existing alias from its attributes.
     */
    static fromAliasAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_AliasAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAliasAttributes);
            }
            throw error;
        }
        if (!attrs.aliasId && !attrs.aliasArn) {
            throw new Error('Either aliasId or aliasArn must be provided in AliasAttributes');
        }
        const aliasId = attrs.aliasId ??
            cdk.Stack.of(scope).splitArn(attrs.aliasArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!aliasId) {
            throw new Error(`No alias identifier found in ARN: '${attrs.aliasArn}'`);
        }
        const aliasArn = attrs.aliasArn ?? cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'alias',
            resourceName: attrs.aliasId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends AliasBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: aliasArn,
                });
                this.aliasId = aliasId;
                this.aliasArn = aliasArn;
            }
        }
        return new Import(scope, id);
    }
    parseRoutingStrategy(props) {
        if (props.fleet) {
            return {
                fleetId: props.fleet.fleetId,
                type: 'SIMPLE',
            };
        }
        return {
            message: props.terminalMessage,
            type: 'TERMINAL',
        };
    }
}
exports.Alias = Alias;
_b = JSII_RTTI_SYMBOL_1;
Alias[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.Alias", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxpYXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGlhcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxtQ0FBbUM7QUFJbkMsMkRBQW9EO0FBNkZwRDs7R0FFRztBQUNILE1BQXNCLFNBQVUsU0FBUSxHQUFHLENBQUMsUUFBUTtJQVNsRDs7T0FFRztJQUNILElBQVcseUJBQXlCO1FBQ2xDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQztLQUN0Qjs7QUFkSCw4QkFlQzs7O0FBRUQ7Ozs7Ozs7Ozs7Ozs7OztHQWVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsU0FBUztJQWdFbEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQjtRQUN6RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUztTQUM5QixDQUFDLENBQUM7Ozs7OzsrQ0FuRU0sS0FBSzs7OztRQXFFZCxJQUFJLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQzVDLElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEdBQUcsSUFBSSxFQUFFO2dCQUNqQyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sY0FBYyxDQUFDLENBQUM7YUFDcEg7U0FDRjtRQUVELElBQUksS0FBSyxDQUFDLFdBQVcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUNuRSxJQUFJLEtBQUssQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLElBQUksRUFBRTtnQkFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRUFBb0UsS0FBSyxDQUFDLFdBQVcsQ0FBQyxNQUFNLGNBQWMsQ0FBQyxDQUFDO2FBQzdIO1NBQ0Y7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUU7WUFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRUFBb0UsQ0FBQyxDQUFDO1NBQ3ZGO1FBRUQsSUFBSSxLQUFLLENBQUMsZUFBZSxJQUFJLEtBQUssQ0FBQyxLQUFLLEVBQUU7WUFDeEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw4RUFBOEUsQ0FBQyxDQUFDO1NBQ2pHO1FBRUQsTUFBTSxRQUFRLEdBQUcsSUFBSSx1QkFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxTQUFTO1lBQ3JCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixlQUFlLEVBQUUsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQztTQUNsRCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUM7UUFDM0QsSUFBSSxDQUFDLFFBQVEsR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDNUMsT0FBTyxFQUFFLFVBQVU7WUFDbkIsUUFBUSxFQUFFLE9BQU87WUFDakIsWUFBWSxFQUFFLElBQUksQ0FBQyxPQUFPO1lBQzFCLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7S0FDSjtJQXBHRDs7T0FFRztJQUNILE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsT0FBZTtRQUM5RCxPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7S0FDbEU7SUFFRDs7T0FFRztJQUNILE1BQU0sQ0FBQyxZQUFZLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBZ0I7UUFDaEUsT0FBTyxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO0tBQ3BFO0lBRUQ7O09BRUc7SUFDSCxNQUFNLENBQUMsbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7Ozs7Ozs7Ozs7UUFDN0UsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQztTQUNuRjtRQUNELE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPO1lBQzNCLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDLFFBQVEsQ0FBQyxLQUFLLENBQUMsUUFBUyxFQUFFLEdBQUcsQ0FBQyxTQUFTLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxZQUFZLENBQUM7UUFFaEcsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNaLE1BQU0sSUFBSSxLQUFLLENBQUMsc0NBQXNDLEtBQUssQ0FBQyxRQUFRLEdBQUcsQ0FBQyxDQUFDO1NBQzFFO1FBRUQsTUFBTSxRQUFRLEdBQUcsS0FBSyxDQUFDLFFBQVEsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLENBQUM7WUFDL0QsT0FBTyxFQUFFLFVBQVU7WUFDbkIsUUFBUSxFQUFFLE9BQU87WUFDakIsWUFBWSxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQzNCLFNBQVMsRUFBRSxHQUFHLENBQUMsU0FBUyxDQUFDLG1CQUFtQjtTQUM3QyxDQUFDLENBQUM7UUFDSCxNQUFNLE1BQU8sU0FBUSxTQUFTO1lBSTVCLFlBQVksQ0FBWSxFQUFFLENBQVM7Z0JBQ2pDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFO29CQUNWLGtCQUFrQixFQUFFLFFBQVE7aUJBQzdCLENBQUMsQ0FBQztnQkFOVyxZQUFPLEdBQUcsT0FBUSxDQUFDO2dCQUNuQixhQUFRLEdBQUcsUUFBUSxDQUFDO1lBTXBDLENBQUM7U0FDRjtRQUNELE9BQU8sSUFBSSxNQUFNLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0tBQzlCO0lBeURPLG9CQUFvQixDQUFDLEtBQWlCO1FBQzVDLElBQUksS0FBSyxDQUFDLEtBQUssRUFBRztZQUNoQixPQUFPO2dCQUNMLE9BQU8sRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLE9BQU87Z0JBQzVCLElBQUksRUFBRSxRQUFRO2FBQ2YsQ0FBQztTQUNIO1FBQ0QsT0FBTztZQUNMLE9BQU8sRUFBRSxLQUFLLENBQUMsZUFBZTtZQUM5QixJQUFJLEVBQUUsVUFBVTtTQUNqQixDQUFDO0tBQ0g7O0FBbkhILHNCQW9IQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IElGbGVldCB9IGZyb20gJy4vZmxlZXQtYmFzZSc7XG5pbXBvcnQgeyBJR2FtZVNlc3Npb25RdWV1ZURlc3RpbmF0aW9uIH0gZnJvbSAnLi9nYW1lLXNlc3Npb24tcXVldWUnO1xuaW1wb3J0IHsgQ2ZuQWxpYXMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtZ2FtZWxpZnQnO1xuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBHYW1lbGlmdCBBbGlhcyBmb3IgYSBHYW1lbGlmdCBmbGVldCBkZXN0aW5hdGlvbi5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQWxpYXMgZXh0ZW5kcyBjZGsuSVJlc291cmNlLCBJR2FtZVNlc3Npb25RdWV1ZURlc3RpbmF0aW9uIHtcblxuICAvKipcbiAgICAgKiBUaGUgSWRlbnRpZmllciBvZiB0aGUgYWxpYXMuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gIHJlYWRvbmx5IGFsaWFzSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgYWxpYXMuXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGFsaWFzQXJuOiBzdHJpbmc7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYGdhbWVsaWZ0LkFsaWFzYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBbGlhc09wdGlvbnMge1xuICAvKipcbiAgICAgKiBEZXNjcmlwdGlvbiBmb3IgdGhlIGFsaWFzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBkZXNjcmlwdGlvblxuICAgICAqL1xuICByZWFkb25seSBkZXNjcmlwdGlvbj86IHN0cmluZztcbn1cblxuLyoqXG4gKiBBIGZ1bGwgc3BlY2lmaWNhdGlvbiBvZiBhbiBhbGlhcyB0aGF0IGNhbiBiZSB1c2VkIHRvIGltcG9ydCBpdCBmbHVlbnRseSBpbnRvIHRoZSBDREsgYXBwbGljYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNBdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAgICAgKiBUaGUgQVJOIG9mIHRoZSBhbGlhc1xuICAgICAgICpcbiAgICAgICAqIEF0IGxlYXN0IG9uZSBvZiBgYWxpYXNBcm5gIGFuZCBgYWxpYXNJZGAgbXVzdCBiZSBwcm92aWRlZC5cbiAgICAgICAqXG4gICAgICAgKiBAZGVmYXVsdCBkZXJpdmVkIGZyb20gYGFsaWFzSWRgLlxuICAgICAgICovXG4gIHJlYWRvbmx5IGFsaWFzQXJuPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgICAqIFRoZSBpZGVudGlmaWVyIG9mIHRoZSBhbGlhc1xuICAgICAqXG4gICAgICogQXQgbGVhc3Qgb25lIG9mIGBhbGlhc0lkYCBhbmQgYGFsaWFzQXJuYCAgbXVzdCBiZSBwcm92aWRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGRlcml2ZWQgZnJvbSBgYWxpYXNBcm5gLlxuICAgICAqL1xuICByZWFkb25seSBhbGlhc0lkPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IEZsZWV0IGFsaWFzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNQcm9wcyB7XG4gIC8qKlxuICAgICAqIE5hbWUgb2YgdGhpcyBhbGlhc1xuICAgICAqL1xuICByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcblxuICAvKipcbiAgICAgKiBBIGh1bWFuLXJlYWRhYmxlIGRlc2NyaXB0aW9uIG9mIHRoZSBhbGlhc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgbm8gZGVzY3JpcHRpb25cbiAgICAgKi9cbiAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEEgZmxlZXQgdGhhdCB0aGUgYWxpYXMgcG9pbnRzIHRvLlxuICAgKiBJZiBzcGVjaWZpZWQsIHRoZSBhbGlhcyByZXNvbHZlcyB0byBvbmUgc3BlY2lmaWMgZmxlZXQuXG4gICAqXG4gICAqIEF0IGxlYXN0IG9uZSBvZiBgZmxlZXRgIGFuZCBgdGVybWluYWxNZXNzYWdlYCBtdXN0IGJlIHByb3ZpZGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBubyBmbGVldCB0aGF0IHRoZSBhbGlhcyBwb2ludHMgdG8uXG4gICAqL1xuICByZWFkb25seSBmbGVldD86IElGbGVldDtcblxuICAvKipcbiAgICogVGhlIG1lc3NhZ2UgdGV4dCB0byBiZSB1c2VkIHdpdGggYSB0ZXJtaW5hbCByb3V0aW5nIHN0cmF0ZWd5LlxuICAgKlxuICAgKiBBdCBsZWFzdCBvbmUgb2YgYGZsZWV0YCBhbmQgYHRlcm1pbmFsTWVzc2FnZWAgbXVzdCBiZSBwcm92aWRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgbm8gdGVybWluYWwgbWVzc2FnZVxuICAgKi9cbiAgcmVhZG9ubHkgdGVybWluYWxNZXNzYWdlPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhc2UgY2xhc3MgZm9yIG5ldyBhbmQgaW1wb3J0ZWQgR2FtZUxpZnQgQWxpYXMuXG4gKi9cbmV4cG9ydCBhYnN0cmFjdCBjbGFzcyBBbGlhc0Jhc2UgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJQWxpYXMge1xuICAvKipcbiAqIFRoZSBJZGVudGlmaWVyIG9mIHRoZSBhbGlhcy5cbiAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYWxpYXNJZDogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGUgYWxpYXNcbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhbGlhc0Fybjogc3RyaW5nO1xuICAvKipcbiAgICogVGhlIEFSTiB0byBwdXQgaW50byB0aGUgZGVzdGluYXRpb24gZmllbGQgb2YgYSBnYW1lIHNlc3Npb24gcXVldWVcbiAgICovXG4gIHB1YmxpYyBnZXQgcmVzb3VyY2VBcm5Gb3JEZXN0aW5hdGlvbigpIHtcbiAgICByZXR1cm4gdGhpcy5hbGlhc0FybjtcbiAgfVxufVxuXG4vKipcbiAqIEEgQW1hem9uIEdhbWVMaWZ0IGFsaWFzIGlzIHVzZWQgdG8gYWJzdHJhY3QgYSBmbGVldCBkZXNpZ25hdGlvbi5cbiAqIEZsZWV0IGRlc2lnbmF0aW9ucyB0ZWxsIEdhbWVMaWZ0IHdoZXJlIHRvIHNlYXJjaCBmb3IgYXZhaWxhYmxlIHJlc291cmNlcyB3aGVuIGNyZWF0aW5nIG5ldyBnYW1lIHNlc3Npb25zIGZvciBwbGF5ZXJzLlxuICogVXNlIGFsaWFzZXMgaW5zdGVhZCBvZiBzcGVjaWZpYyBmbGVldCBJRHMgdG8gc2VhbWxlc3NseSBzd2l0Y2ggcGxheWVyIHRyYWZmaWMgZnJvbSBvbmUgZmxlZXQgdG8gYW5vdGhlciBieSBjaGFuZ2luZyB0aGUgYWxpYXMncyB0YXJnZXQgbG9jYXRpb24uXG4gKlxuICogQWxpYXNlcyBhcmUgdXNlZnVsIGluIGdhbWVzIHRoYXQgZG9uJ3QgdXNlIHF1ZXVlcy5cbiAqIFN3aXRjaGluZyBmbGVldHMgaW4gYSBxdWV1ZSBpcyBhIHNpbXBsZSBtYXR0ZXIgb2YgY3JlYXRpbmcgYSBuZXcgZmxlZXQsIGFkZGluZyBpdCB0byB0aGUgcXVldWUsIGFuZCByZW1vdmluZyB0aGUgb2xkIGZsZWV0LCBub25lIG9mIHdoaWNoIGlzIHZpc2libGUgdG8gcGxheWVycy5cbiAqIEluIGNvbnRyYXN0LCBnYW1lIGNsaWVudHMgdGhhdCBkb24ndCB1c2UgcXVldWVzIG11c3Qgc3BlY2lmeSB3aGljaCBmbGVldCB0byB1c2Ugd2hlbiBjb21tdW5pY2F0aW5nIHdpdGggdGhlIEdhbWVMaWZ0IHNlcnZpY2UuXG4gKiBXaXRob3V0IGFsaWFzZXMsIGEgZmxlZXQgc3dpdGNoIHJlcXVpcmVzIHVwZGF0ZXMgdG8geW91ciBnYW1lIGNvZGUgYW5kIHBvc3NpYmx5IGRpc3RyaWJ1dGlvbiBvZiBhbiB1cGRhdGVkIGdhbWUgY2xpZW50cyB0byBwbGF5ZXJzLlxuICpcbiAqIFdoZW4gdXBkYXRpbmcgdGhlIGZsZWV0LWlkIGFuIGFsaWFzIHBvaW50cyB0bywgdGhlcmUgaXMgYSB0cmFuc2l0aW9uIHBlcmlvZCBvZiB1cCB0byAyIG1pbnV0ZXMgd2hlcmUgZ2FtZSBzZXNzaW9ucyBvbiB0aGUgYWxpYXMgbWF5IGVuZCB1cCBvbiB0aGUgb2xkIGZsZWV0LlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2dhbWVsaWZ0L2xhdGVzdC9kZXZlbG9wZXJndWlkZS9hbGlhc2VzLWNyZWF0aW5nLmh0bWxcbiAqXG4gKiBAcmVzb3VyY2UgQVdTOjpHYW1lTGlmdDo6QWxpYXNcbiAqL1xuZXhwb3J0IGNsYXNzIEFsaWFzIGV4dGVuZHMgQWxpYXNCYXNlIHtcblxuICAvKipcbiAgICogSW1wb3J0IGFuIGV4aXN0aW5nIGFsaWFzIGZyb20gaXRzIGlkZW50aWZpZXIuXG4gICAqL1xuICBzdGF0aWMgZnJvbUFsaWFzSWQoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYWxpYXNJZDogc3RyaW5nKTogSUFsaWFzIHtcbiAgICByZXR1cm4gdGhpcy5mcm9tQWxpYXNBdHRyaWJ1dGVzKHNjb3BlLCBpZCwgeyBhbGlhc0lkOiBhbGlhc0lkIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydCBhbiBleGlzdGluZyBhbGlhcyBmcm9tIGl0cyBBUk4uXG4gICAqL1xuICBzdGF0aWMgZnJvbUFsaWFzQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGFsaWFzQXJuOiBzdHJpbmcpOiBJQWxpYXMge1xuICAgIHJldHVybiB0aGlzLmZyb21BbGlhc0F0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IGFsaWFzQXJuOiBhbGlhc0FybiB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXhpc3RpbmcgYWxpYXMgZnJvbSBpdHMgYXR0cmlidXRlcy5cbiAgICovXG4gIHN0YXRpYyBmcm9tQWxpYXNBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBBbGlhc0F0dHJpYnV0ZXMpOiBJQWxpYXMge1xuICAgIGlmICghYXR0cnMuYWxpYXNJZCAmJiAhYXR0cnMuYWxpYXNBcm4pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRWl0aGVyIGFsaWFzSWQgb3IgYWxpYXNBcm4gbXVzdCBiZSBwcm92aWRlZCBpbiBBbGlhc0F0dHJpYnV0ZXMnKTtcbiAgICB9XG4gICAgY29uc3QgYWxpYXNJZCA9IGF0dHJzLmFsaWFzSWQgPz9cbiAgICAgIGNkay5TdGFjay5vZihzY29wZSkuc3BsaXRBcm4oYXR0cnMuYWxpYXNBcm4hLCBjZGsuQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZTtcblxuICAgIGlmICghYWxpYXNJZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBObyBhbGlhcyBpZGVudGlmaWVyIGZvdW5kIGluIEFSTjogJyR7YXR0cnMuYWxpYXNBcm59J2ApO1xuICAgIH1cblxuICAgIGNvbnN0IGFsaWFzQXJuID0gYXR0cnMuYWxpYXNBcm4gPz8gY2RrLlN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2dhbWVsaWZ0JyxcbiAgICAgIHJlc291cmNlOiAnYWxpYXMnLFxuICAgICAgcmVzb3VyY2VOYW1lOiBhdHRycy5hbGlhc0lkLFxuICAgICAgYXJuRm9ybWF0OiBjZGsuQXJuRm9ybWF0LlNMQVNIX1JFU09VUkNFX05BTUUsXG4gICAgfSk7XG4gICAgY2xhc3MgSW1wb3J0IGV4dGVuZHMgQWxpYXNCYXNlIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBhbGlhc0lkID0gYWxpYXNJZCE7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgYWxpYXNBcm4gPSBhbGlhc0FybjtcblxuICAgICAgY29uc3RydWN0b3IoczogQ29uc3RydWN0LCBpOiBzdHJpbmcpIHtcbiAgICAgICAgc3VwZXIocywgaSwge1xuICAgICAgICAgIGVudmlyb25tZW50RnJvbUFybjogYWxpYXNBcm4sXG4gICAgICAgIH0pO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBJZGVudGlmaWVyIG9mIHRoZSBhbGlhcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBhbGlhc0lkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAgKiBUaGUgQVJOIG9mIHRoZSBhbGlhcy5cbiAgICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgYWxpYXNBcm46IHN0cmluZztcblxuICAvKipcbiAgICogQSBmbGVldCB0aGF0IHRoZSBhbGlhcyBwb2ludHMgdG8uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZmxlZXQ/OiBJRmxlZXQ7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEFsaWFzUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuYWxpYXNOYW1lLFxuICAgIH0pO1xuXG4gICAgaWYgKCFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLmFsaWFzTmFtZSkpIHtcbiAgICAgIGlmIChwcm9wcy5hbGlhc05hbWUubGVuZ3RoID4gMTAyNCkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYEFsaWFzIG5hbWUgY2FuIG5vdCBiZSBsb25nZXIgdGhhbiAxMDI0IGNoYXJhY3RlcnMgYnV0IGhhcyAke3Byb3BzLmFsaWFzTmFtZS5sZW5ndGh9IGNoYXJhY3RlcnMuYCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmRlc2NyaXB0aW9uICYmICFjZGsuVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLmRlc2NyaXB0aW9uKSkge1xuICAgICAgaWYgKHByb3BzLmRlc2NyaXB0aW9uLmxlbmd0aCA+IDEwMjQpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBBbGlhcyBkZXNjcmlwdGlvbiBjYW4gbm90IGJlIGxvbmdlciB0aGFuIDEwMjQgY2hhcmFjdGVycyBidXQgaGFzICR7cHJvcHMuZGVzY3JpcHRpb24ubGVuZ3RofSBjaGFyYWN0ZXJzLmApO1xuICAgICAgfVxuICAgIH1cblxuICAgIGlmICghcHJvcHMudGVybWluYWxNZXNzYWdlICYmICFwcm9wcy5mbGVldCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdFaXRoZXIgYSB0ZXJtaW5hbCBtZXNzYWdlIG9yIGEgZmxlZXQgbXVzdCBiZSBiaW5kZWQgdG8gdGhpcyBBbGlhcy4nKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMudGVybWluYWxNZXNzYWdlICYmIHByb3BzLmZsZWV0KSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VpdGhlciBhIHRlcm1pbmFsIG1lc3NhZ2Ugb3IgYSBmbGVldCBtdXN0IGJlIGJpbmRlZCB0byB0aGlzIEFsaWFzLCBub3QgYm90aC4nKTtcbiAgICB9XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5BbGlhcyh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBuYW1lOiBwcm9wcy5hbGlhc05hbWUsXG4gICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICByb3V0aW5nU3RyYXRlZ3k6IHRoaXMucGFyc2VSb3V0aW5nU3RyYXRlZ3kocHJvcHMpLFxuICAgIH0pO1xuXG4gICAgdGhpcy5hbGlhc0lkID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocmVzb3VyY2UucmVmKTtcbiAgICB0aGlzLmFsaWFzQXJuID0gY2RrLlN0YWNrLm9mKHNjb3BlKS5mb3JtYXRBcm4oe1xuICAgICAgc2VydmljZTogJ2dhbWVsaWZ0JyxcbiAgICAgIHJlc291cmNlOiAnYWxpYXMnLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLmFsaWFzSWQsXG4gICAgICBhcm5Gb3JtYXQ6IGNkay5Bcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSxcbiAgICB9KTtcbiAgfVxuXG4gIHByaXZhdGUgcGFyc2VSb3V0aW5nU3RyYXRlZ3kocHJvcHM6IEFsaWFzUHJvcHMpOiBDZm5BbGlhcy5Sb3V0aW5nU3RyYXRlZ3lQcm9wZXJ0eSB7XG4gICAgaWYgKHByb3BzLmZsZWV0ICkge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgZmxlZXRJZDogcHJvcHMuZmxlZXQuZmxlZXRJZCxcbiAgICAgICAgdHlwZTogJ1NJTVBMRScsXG4gICAgICB9O1xuICAgIH1cbiAgICByZXR1cm4ge1xuICAgICAgbWVzc2FnZTogcHJvcHMudGVybWluYWxNZXNzYWdlLFxuICAgICAgdHlwZTogJ1RFUk1JTkFMJyxcbiAgICB9O1xuICB9XG59XG5cbiJdfQ==