import * as iam from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { IBuild } from './build';
import { FleetBase, FleetProps, IFleet } from './fleet-base';
import { Port, IPeer, IngressRule } from './ingress-rule';
/**
 * Represents a GameLift Fleet used to run a custom game build.
 */
export interface IBuildFleet extends IFleet {
}
/**
 * Properties for a new Gamelift build fleet
 */
export interface BuildFleetProps extends FleetProps {
    /**
       * A build to be deployed on the fleet.
       * The build must have been successfully uploaded to Amazon GameLift and be in a `READY` status.
       *
       * This fleet setting cannot be changed once the fleet is created.
       */
    readonly content: IBuild;
    /**
     * The allowed IP address ranges and port settings that allow inbound traffic to access game sessions on this fleet.
     *
     * This property must be set before players can connect to game sessions.
     *
     * @default no inbound traffic allowed
     */
    readonly ingressRules?: IngressRule[];
}
/**
 * A fleet contains Amazon Elastic Compute Cloud (Amazon EC2) instances that GameLift hosts.
 * A fleet uses the configuration and scaling logic that you define to run your game server build. You can use a fleet directly without a queue.
 * You can also associate multiple fleets with a GameLift queue.
 *
 * For example, you can use Spot Instance fleets configured with your preferred locations, along with a backup On-Demand Instance fleet with the same locations.
 * Using multiple Spot Instance fleets of different instance types reduces the chance of needing On-Demand Instance placement.
 *
 * @resource AWS::GameLift::Fleet
 */
export declare class BuildFleet extends FleetBase implements IBuildFleet {
    /**
     * Import an existing fleet from its identifier.
     */
    static fromBuildFleetId(scope: Construct, id: string, buildFleetId: string): IBuildFleet;
    /**
     * Import an existing fleet from its ARN.
     */
    static fromBuildFleetArn(scope: Construct, id: string, buildFleetArn: string): IBuildFleet;
    /**
     * The Identifier of the fleet.
     */
    readonly fleetId: string;
    /**
     * The ARN of the fleet.
     */
    readonly fleetArn: string;
    /**
     * The build content of the fleet
     */
    readonly content: IBuild;
    /**
     * The IAM role GameLift assumes by fleet instances to access AWS ressources.
     */
    readonly role: iam.IRole;
    /**
     * The principal this GameLift fleet is using.
     */
    readonly grantPrincipal: iam.IPrincipal;
    private readonly ingressRules;
    constructor(scope: Construct, id: string, props: BuildFleetProps);
    /**
     * Adds an ingress rule to allow inbound traffic to access game sessions on this fleet.
     *
     * @param source A range of allowed IP addresses
     * @param port The port range used for ingress traffic
     */
    addIngressRule(source: IPeer, port: Port): void;
    private addInternalIngressRule;
    private parseIngressRules;
}
