"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuildFleet = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const fleet_base_1 = require("./fleet-base");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * A fleet contains Amazon Elastic Compute Cloud (Amazon EC2) instances that GameLift hosts.
 * A fleet uses the configuration and scaling logic that you define to run your game server build. You can use a fleet directly without a queue.
 * You can also associate multiple fleets with a GameLift queue.
 *
 * For example, you can use Spot Instance fleets configured with your preferred locations, along with a backup On-Demand Instance fleet with the same locations.
 * Using multiple Spot Instance fleets of different instance types reduces the chance of needing On-Demand Instance placement.
 *
 * @resource AWS::GameLift::Fleet
 */
class BuildFleet extends fleet_base_1.FleetBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.fleetName,
        });
        this.ingressRules = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_BuildFleetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, BuildFleet);
            }
            throw error;
        }
        if (!cdk.Token.isUnresolved(props.fleetName)) {
            if (props.fleetName.length > 1024) {
                throw new Error(`Fleet name can not be longer than 1024 characters but has ${props.fleetName.length} characters.`);
            }
        }
        if (props.description && !cdk.Token.isUnresolved(props.description)) {
            if (props.description.length > 1024) {
                throw new Error(`Fleet description can not be longer than 1024 characters but has ${props.description.length} characters.`);
            }
        }
        if (props.minSize && props.minSize < 0) {
            throw new Error(`The minimum number of instances allowed in the Fleet cannot be lower than 0, given ${props.minSize}`);
        }
        if (props.maxSize && props.maxSize < 0) {
            throw new Error(`The maximum number of instances allowed in the Fleet cannot be lower than 0, given ${props.maxSize}`);
        }
        if (props.peerVpc) {
            this.warnVpcPeeringAuthorizations(this);
        }
        // Add all locations
        if (props.locations && props.locations?.length > 100) {
            throw new Error(`No more than 100 locations are allowed per fleet, given ${props.locations.length}`);
        }
        (props.locations || []).forEach(this.addInternalLocation.bind(this));
        // Add all Ingress rules
        if (props.ingressRules && props.ingressRules?.length > 50) {
            throw new Error(`No more than 50 ingress rules are allowed per fleet, given ${props.ingressRules.length}`);
        }
        (props.ingressRules || []).forEach(this.addInternalIngressRule.bind(this));
        this.content = props.content;
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('gamelift.amazonaws.com'), new iam.ServicePrincipal('ec2.amazonaws.com')),
        });
        this.grantPrincipal = this.role;
        const resource = new aws_gamelift_1.CfnFleet(this, 'Resource', {
            buildId: this.content.buildId,
            certificateConfiguration: {
                certificateType: props.useCertificate ? 'GENERATED' : 'DISABLED',
            },
            description: props.description,
            desiredEc2Instances: props.desiredCapacity,
            ec2InboundPermissions: cdk.Lazy.any({ produce: () => this.parseIngressRules() }),
            ec2InstanceType: props.instanceType.toString(),
            fleetType: props.useSpot ? 'SPOT' : 'ON_DEMAND',
            instanceRoleArn: this.role.roleArn,
            locations: cdk.Lazy.any({ produce: () => this.parseLocations() }),
            maxSize: props.maxSize ? props.maxSize : 1,
            minSize: props.minSize ? props.minSize : 0,
            name: this.physicalName,
            newGameSessionProtectionPolicy: props.protectNewGameSession ? 'FullProtection' : 'NoProtection',
            peerVpcAwsAccountId: props.peerVpc && props.peerVpc.env.account,
            peerVpcId: props.peerVpc && props.peerVpc.vpcId,
            resourceCreationLimitPolicy: this.parseResourceCreationLimitPolicy(props),
            runtimeConfiguration: this.parseRuntimeConfiguration(props),
        });
        this.fleetId = this.getResourceNameAttribute(resource.ref);
        this.fleetArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'fleet',
            resourceName: this.fleetId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
     * Import an existing fleet from its identifier.
     */
    static fromBuildFleetId(scope, id, buildFleetId) {
        return this.fromFleetAttributes(scope, id, { fleetId: buildFleetId });
    }
    /**
     * Import an existing fleet from its ARN.
     */
    static fromBuildFleetArn(scope, id, buildFleetArn) {
        return this.fromFleetAttributes(scope, id, { fleetArn: buildFleetArn });
    }
    /**
     * Adds an ingress rule to allow inbound traffic to access game sessions on this fleet.
     *
     * @param source A range of allowed IP addresses
     * @param port The port range used for ingress traffic
     */
    addIngressRule(source, port) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_IPeer(source);
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_Port(port);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addIngressRule);
            }
            throw error;
        }
        this.addInternalIngressRule({
            source: source,
            port: port,
        });
    }
    addInternalIngressRule(ingressRule) {
        if (this.ingressRules.length == 50) {
            throw new Error('No more than 50 ingress rules are allowed per fleet');
        }
        this.ingressRules.push(ingressRule);
    }
    parseIngressRules() {
        if (!this.ingressRules || this.ingressRules.length === 0) {
            return undefined;
        }
        return this.ingressRules.map(parseIngressRule);
        function parseIngressRule(ingressRule) {
            return {
                ...ingressRule.port.toJson(),
                ...ingressRule.source.toJson(),
            };
        }
    }
}
exports.BuildFleet = BuildFleet;
_a = JSII_RTTI_SYMBOL_1;
BuildFleet[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.BuildFleet", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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