import * as iam from 'aws-cdk-lib/aws-iam';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as s3_assets from 'aws-cdk-lib/aws-s3-assets';
import * as cdk from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { Content } from './content';
/**
 * Your custom-built game server software that runs on GameLift and hosts game sessions for your players.
 * A game build represents the set of files that run your game server on a particular operating system.
 * You can have many different builds, such as for different flavors of your game.
 * The game build must be integrated with the GameLift service.
 * You upload game build files to the GameLift service in the Regions where you plan to set up fleets.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-cli-uploading.html
 */
export interface IBuild extends cdk.IResource, iam.IGrantable {
    /**
     * The Identifier of the build.
     *
     * @attribute
     */
    readonly buildId: string;
    /**
     * The ARN of the build
     *
     * @attribute
     */
    readonly buildArn: string;
}
/**
 * Base class for new and imported GameLift server build.
 */
export declare abstract class BuildBase extends cdk.Resource implements IBuild {
    /**
       * The Identifier of the build.
       */
    abstract readonly buildId: string;
    /**
       * The ARN of the build.
       */
    abstract readonly buildArn: string;
    abstract readonly grantPrincipal: iam.IPrincipal;
}
/**
 * The operating system that the game server binaries are built to run on.
 */
export declare enum OperatingSystem {
    AMAZON_LINUX = "AMAZON_LINUX",
    AMAZON_LINUX_2 = "AMAZON_LINUX_2",
    WINDOWS_2012 = "WINDOWS_2012"
}
/**
 * Represents a Build content defined outside of this stack.
 */
export interface BuildAttributes {
    /**
       * The ARN of the build
       *
       * At least one of `buildArn` and `buildId` must be provided.
       *
       * @default derived from `buildId`.
       */
    readonly buildArn?: string;
    /**
      * The identifier of the build
      *
      * At least one of `buildId` and `buildArn`  must be provided.
      *
      * @default derived from `buildArn`.
      */
    readonly buildId?: string;
    /**
     * The IAM role assumed by GameLift to access server build in S3.
     * @default the imported fleet cannot be granted access to other resources as an `iam.IGrantable`.
     */
    readonly role?: iam.IRole;
}
/**
 * Properties for a new build
 */
export interface BuildProps {
    /**
       * Name of this build
       *
       * @default No name
       */
    readonly buildName?: string;
    /**
       * Version of this build
       *
       * @default No version
       */
    readonly buildVersion?: string;
    /**
       * The operating system that the game server binaries are built to run on.
       *
       * @default No version
       */
    readonly operatingSystem?: OperatingSystem;
    /**
       * The game build file storage
       */
    readonly content: Content;
    /**
      * The IAM role assumed by GameLift to access server build in S3.
      * If providing a custom role, it needs to trust the GameLift service principal (gamelift.amazonaws.com) and be granted sufficient permissions
      * to have Read access to a specific key content into a specific S3 bucket.
      * Below an example of required permission:
      * {
      *  "Version": "2012-10-17",
      *  "Statement": [{
      *        "Effect": "Allow",
      *        "Action": [
      *            "s3:GetObject",
      *            "s3:GetObjectVersion"
      *        ],
      *        "Resource": "arn:aws:s3:::bucket-name/object-name"
      *  }]
      *}
      *
      * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/security_iam_id-based-policy-examples.html#security_iam_id-based-policy-examples-access-storage-loc
      *
      * @default - a role will be created with default permissions.
      */
    readonly role?: iam.IRole;
}
/**
 * A GameLift build, that is installed and runs on instances in an Amazon GameLift fleet. It consists of
 * a zip file with all of the components of the game server build.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-cli-uploading.html
 *
 * @resource AWS::GameLift::Build
 */
export declare class Build extends BuildBase {
    /**
     * Create a new Build from s3 content
     */
    static fromBucket(scope: Construct, id: string, bucket: s3.IBucket, key: string, objectVersion?: string): Build;
    /**
     * Create a new Build from asset content
     */
    static fromAsset(scope: Construct, id: string, path: string, options?: s3_assets.AssetOptions): Build;
    /**
       * Import a build into CDK using its identifier
       */
    static fromBuildId(scope: Construct, id: string, buildId: string): IBuild;
    /**
       * Import a build into CDK using its ARN
       */
    static fromBuildArn(scope: Construct, id: string, buildArn: string): IBuild;
    /**
     * Import an existing build from its attributes.
     */
    static fromBuildAttributes(scope: Construct, id: string, attrs: BuildAttributes): IBuild;
    /**
     * The Identifier of the build.
     */
    readonly buildId: string;
    /**
     * The ARN of the build.
     */
    readonly buildArn: string;
    /**
     * The IAM role GameLift assumes to acccess server build content.
     */
    readonly role: iam.IRole;
    /**
     * The principal this GameLift Build is using.
     */
    readonly grantPrincipal: iam.IPrincipal;
    constructor(scope: Construct, id: string, props: BuildProps);
}
