"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Build = exports.OperatingSystem = exports.BuildBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const content_1 = require("./content");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * Base class for new and imported GameLift server build.
 */
class BuildBase extends cdk.Resource {
}
exports.BuildBase = BuildBase;
_a = JSII_RTTI_SYMBOL_1;
BuildBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.BuildBase", version: "2.65.0-alpha.0" };
/**
 * The operating system that the game server binaries are built to run on.
 */
var OperatingSystem;
(function (OperatingSystem) {
    OperatingSystem["AMAZON_LINUX"] = "AMAZON_LINUX";
    OperatingSystem["AMAZON_LINUX_2"] = "AMAZON_LINUX_2";
    OperatingSystem["WINDOWS_2012"] = "WINDOWS_2012";
})(OperatingSystem = exports.OperatingSystem || (exports.OperatingSystem = {}));
/**
 * A GameLift build, that is installed and runs on instances in an Amazon GameLift fleet. It consists of
 * a zip file with all of the components of the game server build.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-build-cli-uploading.html
 *
 * @resource AWS::GameLift::Build
 */
class Build extends BuildBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.buildName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_BuildProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Build);
            }
            throw error;
        }
        if (props.buildName && !cdk.Token.isUnresolved(props.buildName)) {
            if (props.buildName.length > 1024) {
                throw new Error(`Build name can not be longer than 1024 characters but has ${props.buildName.length} characters.`);
            }
        }
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('gamelift.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const content = props.content.bind(this, this.role);
        const resource = new aws_gamelift_1.CfnBuild(this, 'Resource', {
            name: props.buildName,
            version: props.buildVersion,
            operatingSystem: props.operatingSystem,
            storageLocation: {
                bucket: content.s3Location && content.s3Location.bucketName,
                key: content.s3Location && content.s3Location.objectKey,
                objectVersion: content.s3Location && content.s3Location.objectVersion,
                roleArn: this.role.roleArn,
            },
        });
        resource.node.addDependency(this.role);
        this.buildId = this.getResourceNameAttribute(resource.ref);
        this.buildArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'build',
            resourceName: this.buildId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
     * Create a new Build from s3 content
     */
    static fromBucket(scope, id, bucket, key, objectVersion) {
        return new Build(scope, id, {
            content: content_1.Content.fromBucket(bucket, key, objectVersion),
        });
    }
    /**
     * Create a new Build from asset content
     */
    static fromAsset(scope, id, path, options) {
        return new Build(scope, id, {
            content: content_1.Content.fromAsset(path, options),
        });
    }
    /**
       * Import a build into CDK using its identifier
       */
    static fromBuildId(scope, id, buildId) {
        return this.fromBuildAttributes(scope, id, { buildId });
    }
    /**
       * Import a build into CDK using its ARN
       */
    static fromBuildArn(scope, id, buildArn) {
        return this.fromBuildAttributes(scope, id, { buildArn });
    }
    /**
     * Import an existing build from its attributes.
     */
    static fromBuildAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_BuildAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromBuildAttributes);
            }
            throw error;
        }
        if (!attrs.buildId && !attrs.buildArn) {
            throw new Error('Either buildId or buildArn must be provided in BuildAttributes');
        }
        const buildId = attrs.buildId ??
            cdk.Stack.of(scope).splitArn(attrs.buildArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!buildId) {
            throw new Error(`No build identifier found in ARN: '${attrs.buildArn}'`);
        }
        const buildArn = attrs.buildArn ?? cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'build',
            resourceName: attrs.buildId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends BuildBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: buildArn,
                });
                this.buildId = buildId;
                this.buildArn = buildArn;
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
                this.role = attrs.role;
            }
        }
        return new Import(scope, id);
    }
}
exports.Build = Build;
_b = JSII_RTTI_SYMBOL_1;
Build[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.Build", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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