import * as iam from 'aws-cdk-lib/aws-iam';
import * as s3 from 'aws-cdk-lib/aws-s3';
import * as s3_assets from 'aws-cdk-lib/aws-s3-assets';
import { Construct } from 'constructs';
/**
 * Before deploying your GameLift-enabled multiplayer game servers for hosting with the GameLift service, you need to upload your game server files.
 * The class helps you on preparing and uploading custom game server build files or Realtime Servers server script files.
 */
export declare abstract class Content {
    /**
       * Game content as an S3 object.
       * @param bucket The S3 bucket
       * @param key The object key
       * @param objectVersion Optional S3 ob ject version
       */
    static fromBucket(bucket: s3.IBucket, key: string, objectVersion?: string): S3Content;
    /**
       * Loads the game content from a local disk path.
       *
       * @param path Either a directory with the game content bundle or a .zip file
       */
    static fromAsset(path: string, options?: s3_assets.AssetOptions): AssetContent;
    /**
       * Called when the Build is initialized to allow this object to bind
       */
    abstract bind(scope: Construct, role: iam.IRole): ContentConfig;
}
/**
 * Result of binding `Content` into a `Build`.
 */
export interface ContentConfig {
    /**
     * The location of the content in S3.
     */
    readonly s3Location: s3.Location;
}
/**
 * Game content from an S3 archive.
 */
export declare class S3Content extends Content {
    private readonly bucket;
    private key;
    private objectVersion?;
    constructor(bucket: s3.IBucket, key: string, objectVersion?: string | undefined);
    bind(_scope: Construct, role: iam.IRole): ContentConfig;
}
/**
 * Game content from a local directory.
 */
export declare class AssetContent extends Content {
    readonly path: string;
    private readonly options;
    private asset?;
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path: string, options?: s3_assets.AssetOptions);
    bind(scope: Construct, role: iam.IRole): ContentConfig;
}
