"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetContent = exports.S3Content = exports.Content = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const s3_assets = require("aws-cdk-lib/aws-s3-assets");
const cdk = require("aws-cdk-lib");
/**
 * Before deploying your GameLift-enabled multiplayer game servers for hosting with the GameLift service, you need to upload your game server files.
 * The class helps you on preparing and uploading custom game server build files or Realtime Servers server script files.
 */
class Content {
    /**
       * Game content as an S3 object.
       * @param bucket The S3 bucket
       * @param key The object key
       * @param objectVersion Optional S3 ob ject version
       */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Content(bucket, key, objectVersion);
    }
    /**
       * Loads the game content from a local disk path.
       *
       * @param path Either a directory with the game content bundle or a .zip file
       */
    static fromAsset(path, options) {
        return new AssetContent(path, options);
    }
}
exports.Content = Content;
_a = JSII_RTTI_SYMBOL_1;
Content[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.Content", version: "2.65.0-alpha.0" };
/**
 * Game content from an S3 archive.
 */
class S3Content extends Content {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
    }
    bind(_scope, role) {
        // Adding permission to access specific content
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [this.bucket.arnForObjects(this.key)],
            actions: ['s3:GetObject', 's3:GetObjectVersion'],
        }));
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Content = S3Content;
_b = JSII_RTTI_SYMBOL_1;
S3Content[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.S3Content", version: "2.65.0-alpha.0" };
/**
 * Game content from a local directory.
 */
class AssetContent extends Content {
    /**
     * @param path The path to the asset file or directory.
     */
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope, role) {
        // If the same AssetContent is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Content', {
                path: this.path,
                ...this.options,
            });
        }
        else if (cdk.Stack.of(this.asset) !== cdk.Stack.of(scope)) {
            throw new Error(`Asset is already associated with another stack '${cdk.Stack.of(this.asset).stackName}'. ` +
                'Create a new Content instance for every stack.');
        }
        const bucket = s3.Bucket.fromBucketName(scope, 'AssetBucket', this.asset.s3BucketName);
        // Adding permission to access specific content
        role.addToPrincipalPolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            resources: [bucket.arnForObjects(this.asset.s3ObjectKey)],
            actions: ['s3:GetObject', 's3:GetObjectVersion'],
        }));
        if (!this.asset.isZipArchive) {
            throw new Error(`Asset must be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.AssetContent = AssetContent;
_c = JSII_RTTI_SYMBOL_1;
AssetContent[_c] = { fqn: "@aws-cdk/aws-gamelift-alpha.AssetContent", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29udGVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNvbnRlbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSwyQ0FBMkM7QUFDM0MseUNBQXlDO0FBQ3pDLHVEQUF1RDtBQUN2RCxtQ0FBbUM7QUFHbkM7OztHQUdHO0FBQ0gsTUFBc0IsT0FBTztJQUMzQjs7Ozs7U0FLSztJQUNFLE1BQU0sQ0FBQyxVQUFVLENBQUMsTUFBa0IsRUFBRSxHQUFXLEVBQUUsYUFBc0I7UUFDOUUsT0FBTyxJQUFJLFNBQVMsQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLGFBQWEsQ0FBQyxDQUFDO0tBQ2xEO0lBR0Q7Ozs7U0FJSztJQUNFLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBWSxFQUFFLE9BQWdDO1FBQ3BFLE9BQU8sSUFBSSxZQUFZLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0tBQ3hDOztBQW5CSCwwQkEwQkM7OztBQVlEOztHQUVHO0FBQ0gsTUFBYSxTQUFVLFNBQVEsT0FBTztJQUVwQyxZQUE2QixNQUFrQixFQUFVLEdBQVcsRUFBVSxhQUFzQjtRQUNsRyxLQUFLLEVBQUUsQ0FBQztRQURtQixXQUFNLEdBQU4sTUFBTSxDQUFZO1FBQVUsUUFBRyxHQUFILEdBQUcsQ0FBUTtRQUFVLGtCQUFhLEdBQWIsYUFBYSxDQUFTO1FBRWxHLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztTQUNwRTtLQUNGO0lBRU0sSUFBSSxDQUFDLE1BQWlCLEVBQUUsSUFBZTtRQUM1QywrQ0FBK0M7UUFDL0MsSUFBSSxDQUFDLG9CQUFvQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUNoRCxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO1lBQ3hCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUNoRCxPQUFPLEVBQUUsQ0FBQyxjQUFjLEVBQUUscUJBQXFCLENBQUM7U0FDakQsQ0FBQyxDQUFDLENBQUM7UUFFSixPQUFPO1lBQ0wsVUFBVSxFQUFFO2dCQUNWLFVBQVUsRUFBRSxJQUFJLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQ2xDLFNBQVMsRUFBRSxJQUFJLENBQUMsR0FBRztnQkFDbkIsYUFBYSxFQUFFLElBQUksQ0FBQyxhQUFhO2FBQ2xDO1NBQ0YsQ0FBQztLQUNIOztBQXhCSCw4QkF5QkM7OztBQUVEOztHQUVHO0FBQ0gsTUFBYSxZQUFhLFNBQVEsT0FBTztJQUd2Qzs7T0FFRztJQUNILFlBQTRCLElBQVksRUFBbUIsVUFBa0MsRUFBRztRQUM5RixLQUFLLEVBQUUsQ0FBQztRQURrQixTQUFJLEdBQUosSUFBSSxDQUFRO1FBQW1CLFlBQU8sR0FBUCxPQUFPLENBQThCO0tBRS9GO0lBRU0sSUFBSSxDQUFDLEtBQWdCLEVBQUUsSUFBZTtRQUMzQyx3RkFBd0Y7UUFDeEYsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDZixJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksU0FBUyxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO2dCQUNqRCxJQUFJLEVBQUUsSUFBSSxDQUFDLElBQUk7Z0JBQ2YsR0FBRyxJQUFJLENBQUMsT0FBTzthQUNoQixDQUFDLENBQUM7U0FDSjthQUFNLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxLQUFLLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQzNELE1BQU0sSUFBSSxLQUFLLENBQUMsbURBQW1ELEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxTQUFTLEtBQUs7Z0JBQ3hHLGdEQUFnRCxDQUFDLENBQUM7U0FDckQ7UUFDRCxNQUFNLE1BQU0sR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFdkYsK0NBQStDO1FBQy9DLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDaEQsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztZQUN4QixTQUFTLEVBQUUsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUM7WUFDekQsT0FBTyxFQUFFLENBQUMsY0FBYyxFQUFFLHFCQUFxQixDQUFDO1NBQ2pELENBQUMsQ0FBQyxDQUFDO1FBRUosSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQzVCLE1BQU0sSUFBSSxLQUFLLENBQUMsNkNBQTZDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxDQUFDO1NBQzVFO1FBRUQsT0FBTztZQUNMLFVBQVUsRUFBRTtnQkFDVixVQUFVLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZO2dCQUNuQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXO2FBQ2xDO1NBQ0YsQ0FBQztLQUNIOztBQXhDSCxvQ0F5Q0MiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBzMyBmcm9tICdhd3MtY2RrLWxpYi9hd3MtczMnO1xuaW1wb3J0ICogYXMgczNfYXNzZXRzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMy1hc3NldHMnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG4vKipcbiAqIEJlZm9yZSBkZXBsb3lpbmcgeW91ciBHYW1lTGlmdC1lbmFibGVkIG11bHRpcGxheWVyIGdhbWUgc2VydmVycyBmb3IgaG9zdGluZyB3aXRoIHRoZSBHYW1lTGlmdCBzZXJ2aWNlLCB5b3UgbmVlZCB0byB1cGxvYWQgeW91ciBnYW1lIHNlcnZlciBmaWxlcy5cbiAqIFRoZSBjbGFzcyBoZWxwcyB5b3Ugb24gcHJlcGFyaW5nIGFuZCB1cGxvYWRpbmcgY3VzdG9tIGdhbWUgc2VydmVyIGJ1aWxkIGZpbGVzIG9yIFJlYWx0aW1lIFNlcnZlcnMgc2VydmVyIHNjcmlwdCBmaWxlcy5cbiAqL1xuZXhwb3J0IGFic3RyYWN0IGNsYXNzIENvbnRlbnQge1xuICAvKipcbiAgICAgKiBHYW1lIGNvbnRlbnQgYXMgYW4gUzMgb2JqZWN0LlxuICAgICAqIEBwYXJhbSBidWNrZXQgVGhlIFMzIGJ1Y2tldFxuICAgICAqIEBwYXJhbSBrZXkgVGhlIG9iamVjdCBrZXlcbiAgICAgKiBAcGFyYW0gb2JqZWN0VmVyc2lvbiBPcHRpb25hbCBTMyBvYiBqZWN0IHZlcnNpb25cbiAgICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0KGJ1Y2tldDogczMuSUJ1Y2tldCwga2V5OiBzdHJpbmcsIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpOiBTM0NvbnRlbnQge1xuICAgIHJldHVybiBuZXcgUzNDb250ZW50KGJ1Y2tldCwga2V5LCBvYmplY3RWZXJzaW9uKTtcbiAgfVxuXG5cbiAgLyoqXG4gICAgICogTG9hZHMgdGhlIGdhbWUgY29udGVudCBmcm9tIGEgbG9jYWwgZGlzayBwYXRoLlxuICAgICAqXG4gICAgICogQHBhcmFtIHBhdGggRWl0aGVyIGEgZGlyZWN0b3J5IHdpdGggdGhlIGdhbWUgY29udGVudCBidW5kbGUgb3IgYSAuemlwIGZpbGVcbiAgICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tQXNzZXQocGF0aDogc3RyaW5nLCBvcHRpb25zPzogczNfYXNzZXRzLkFzc2V0T3B0aW9ucyk6IEFzc2V0Q29udGVudCB7XG4gICAgcmV0dXJuIG5ldyBBc3NldENvbnRlbnQocGF0aCwgb3B0aW9ucyk7XG4gIH1cblxuICAvKipcbiAgICAgKiBDYWxsZWQgd2hlbiB0aGUgQnVpbGQgaXMgaW5pdGlhbGl6ZWQgdG8gYWxsb3cgdGhpcyBvYmplY3QgdG8gYmluZFxuICAgICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgYmluZChzY29wZTogQ29uc3RydWN0LCByb2xlOiBpYW0uSVJvbGUpOiBDb250ZW50Q29uZmlnO1xuXG59XG5cbi8qKlxuICogUmVzdWx0IG9mIGJpbmRpbmcgYENvbnRlbnRgIGludG8gYSBgQnVpbGRgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbnRlbnRDb25maWcge1xuICAvKipcbiAgICogVGhlIGxvY2F0aW9uIG9mIHRoZSBjb250ZW50IGluIFMzLlxuICAgKi9cbiAgcmVhZG9ubHkgczNMb2NhdGlvbjogczMuTG9jYXRpb247XG59XG5cbi8qKlxuICogR2FtZSBjb250ZW50IGZyb20gYW4gUzMgYXJjaGl2ZS5cbiAqL1xuZXhwb3J0IGNsYXNzIFMzQ29udGVudCBleHRlbmRzIENvbnRlbnQge1xuXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0LCBwcml2YXRlIGtleTogc3RyaW5nLCBwcml2YXRlIG9iamVjdFZlcnNpb24/OiBzdHJpbmcpIHtcbiAgICBzdXBlcigpO1xuICAgIGlmICghYnVja2V0LmJ1Y2tldE5hbWUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignYnVja2V0TmFtZSBpcyB1bmRlZmluZWQgZm9yIHRoZSBwcm92aWRlZCBidWNrZXQnKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYmluZChfc2NvcGU6IENvbnN0cnVjdCwgcm9sZTogaWFtLklSb2xlKTogQ29udGVudENvbmZpZyB7XG4gICAgLy8gQWRkaW5nIHBlcm1pc3Npb24gdG8gYWNjZXNzIHNwZWNpZmljIGNvbnRlbnRcbiAgICByb2xlLmFkZFRvUHJpbmNpcGFsUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGVmZmVjdDogaWFtLkVmZmVjdC5BTExPVyxcbiAgICAgIHJlc291cmNlczogW3RoaXMuYnVja2V0LmFybkZvck9iamVjdHModGhpcy5rZXkpXSxcbiAgICAgIGFjdGlvbnM6IFsnczM6R2V0T2JqZWN0JywgJ3MzOkdldE9iamVjdFZlcnNpb24nXSxcbiAgICB9KSk7XG5cbiAgICByZXR1cm4ge1xuICAgICAgczNMb2NhdGlvbjoge1xuICAgICAgICBidWNrZXROYW1lOiB0aGlzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICBvYmplY3RLZXk6IHRoaXMua2V5LFxuICAgICAgICBvYmplY3RWZXJzaW9uOiB0aGlzLm9iamVjdFZlcnNpb24sXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBHYW1lIGNvbnRlbnQgZnJvbSBhIGxvY2FsIGRpcmVjdG9yeS5cbiAqL1xuZXhwb3J0IGNsYXNzIEFzc2V0Q29udGVudCBleHRlbmRzIENvbnRlbnQge1xuICBwcml2YXRlIGFzc2V0PzogczNfYXNzZXRzLkFzc2V0O1xuXG4gIC8qKlxuICAgKiBAcGFyYW0gcGF0aCBUaGUgcGF0aCB0byB0aGUgYXNzZXQgZmlsZSBvciBkaXJlY3RvcnkuXG4gICAqL1xuICBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgcGF0aDogc3RyaW5nLCBwcml2YXRlIHJlYWRvbmx5IG9wdGlvbnM6IHMzX2Fzc2V0cy5Bc3NldE9wdGlvbnMgPSB7IH0pIHtcbiAgICBzdXBlcigpO1xuICB9XG5cbiAgcHVibGljIGJpbmQoc2NvcGU6IENvbnN0cnVjdCwgcm9sZTogaWFtLklSb2xlKTogQ29udGVudENvbmZpZyB7XG4gICAgLy8gSWYgdGhlIHNhbWUgQXNzZXRDb250ZW50IGlzIHVzZWQgbXVsdGlwbGUgdGltZXMsIHJldGFpbiBvbmx5IHRoZSBmaXJzdCBpbnN0YW50aWF0aW9uLlxuICAgIGlmICghdGhpcy5hc3NldCkge1xuICAgICAgdGhpcy5hc3NldCA9IG5ldyBzM19hc3NldHMuQXNzZXQoc2NvcGUsICdDb250ZW50Jywge1xuICAgICAgICBwYXRoOiB0aGlzLnBhdGgsXG4gICAgICAgIC4uLnRoaXMub3B0aW9ucyxcbiAgICAgIH0pO1xuICAgIH0gZWxzZSBpZiAoY2RrLlN0YWNrLm9mKHRoaXMuYXNzZXQpICE9PSBjZGsuU3RhY2sub2Yoc2NvcGUpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEFzc2V0IGlzIGFscmVhZHkgYXNzb2NpYXRlZCB3aXRoIGFub3RoZXIgc3RhY2sgJyR7Y2RrLlN0YWNrLm9mKHRoaXMuYXNzZXQpLnN0YWNrTmFtZX0nLiBgICtcbiAgICAgICAgJ0NyZWF0ZSBhIG5ldyBDb250ZW50IGluc3RhbmNlIGZvciBldmVyeSBzdGFjay4nKTtcbiAgICB9XG4gICAgY29uc3QgYnVja2V0ID0gczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHNjb3BlLCAnQXNzZXRCdWNrZXQnLCB0aGlzLmFzc2V0LnMzQnVja2V0TmFtZSk7XG5cbiAgICAvLyBBZGRpbmcgcGVybWlzc2lvbiB0byBhY2Nlc3Mgc3BlY2lmaWMgY29udGVudFxuICAgIHJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgcmVzb3VyY2VzOiBbYnVja2V0LmFybkZvck9iamVjdHModGhpcy5hc3NldC5zM09iamVjdEtleSldLFxuICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QnLCAnczM6R2V0T2JqZWN0VmVyc2lvbiddLFxuICAgIH0pKTtcblxuICAgIGlmICghdGhpcy5hc3NldC5pc1ppcEFyY2hpdmUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQXNzZXQgbXVzdCBiZSBhIC56aXAgZmlsZSBvciBhIGRpcmVjdG9yeSAoJHt0aGlzLnBhdGh9KWApO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBzM0xvY2F0aW9uOiB7XG4gICAgICAgIGJ1Y2tldE5hbWU6IHRoaXMuYXNzZXQuczNCdWNrZXROYW1lLFxuICAgICAgICBvYmplY3RLZXk6IHRoaXMuYXNzZXQuczNPYmplY3RLZXksXG4gICAgICB9LFxuICAgIH07XG4gIH1cbn1cbiJdfQ==