"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FleetBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const alias_1 = require("./alias");
const gamelift_canned_metrics_generated_1 = require("./gamelift-canned-metrics.generated");
/**
 * Base class for new and imported GameLift fleet.
 */
class FleetBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.locations = [];
    }
    /**
     * Import an existing fleet from its attributes.
     */
    static fromFleetAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_FleetAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromFleetAttributes);
            }
            throw error;
        }
        if (!attrs.fleetId && !attrs.fleetArn) {
            throw new Error('Either fleetId or fleetArn must be provided in FleetAttributes');
        }
        const fleetId = attrs.fleetId ??
            cdk.Stack.of(scope).splitArn(attrs.fleetArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!fleetId) {
            throw new Error(`No fleet identifier found in ARN: '${attrs.fleetArn}'`);
        }
        const fleetArn = attrs.fleetArn ?? cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'fleet',
            resourceName: attrs.fleetId,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends FleetBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: fleetArn,
                });
                this.fleetId = fleetId;
                this.fleetArn = fleetArn;
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
                this.role = attrs.role;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Defines an alias for this fleet.
     *
     * ```ts
     * declare const fleet: gamelift.FleetBase;
     *
     * fleet.addAlias('Live');
     *
     * // Is equivalent to
     *
     * new gamelift.Alias(this, 'AliasLive', {
     *   aliasName: 'Live',
     *   fleet: fleet,
     * });
     * ```
     *
     * @param aliasName The name of the alias
     * @param options Alias options
     */
    addAlias(aliasName, options = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_AliasOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addAlias);
            }
            throw error;
        }
        return new alias_1.Alias(this, `Alias${aliasName}`, {
            aliasName,
            fleet: this,
            ...options,
        });
    }
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            resourceArns: [this.fleetArn],
            grantee: grantee,
            actions: actions,
        });
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/GameLift',
            metricName: metricName,
            dimensionsMap: {
                FleetId: this.fleetId,
            },
            ...props,
        }).attachTo(this);
    }
    metricActiveInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.activeInstancesAverage, props);
    }
    metricPercentIdleInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.percentIdleInstancesAverage, props);
    }
    metricDesiredInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.desiredInstancesAverage, props);
    }
    metricIdleInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.idleInstancesAverage, props);
    }
    metricInstanceInterruptions(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.instanceInterruptionsSum, props);
    }
    metricMaxInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.maxInstancesAverage, props);
    }
    metricMinInstances(props) {
        return this.cannedMetric(gamelift_canned_metrics_generated_1.GameLiftMetrics.minInstancesAverage, props);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ FleetId: this.fleetId }),
            ...props,
        }).attachTo(this);
    }
    /**
     * The ARN to put into the destination field of a game session queue
     */
    get resourceArnForDestination() {
        return this.fleetArn;
    }
    /**
     * Adds a remote locations to deploy additional instances to and manage as part of the fleet.
     *
     * @param region The AWS region to add
     */
    addLocation(region, desiredCapacity, minSize, maxSize) {
        this.addInternalLocation({
            region: region,
            capacity: {
                desiredCapacity: desiredCapacity,
                minSize: minSize,
                maxSize: maxSize,
            },
        });
    }
    /**
     * Adds a remote locations to deploy additional instances to and manage as part of the fleet.
     *
     * @param location The location to add
     */
    addInternalLocation(location) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_Location(location);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addInternalLocation);
            }
            throw error;
        }
        if (this.locations.length == 100) {
            throw new Error('No more than 100 locations are allowed per fleet');
        }
        this.locations.push(location);
    }
    parseResourceCreationLimitPolicy(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_FleetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseResourceCreationLimitPolicy);
            }
            throw error;
        }
        if (!props.resourceCreationLimitPolicy ||
            (!props.resourceCreationLimitPolicy.newGameSessionsPerCreator
                && !props.resourceCreationLimitPolicy.policyPeriod)) {
            return undefined;
        }
        return {
            newGameSessionsPerCreator: props.resourceCreationLimitPolicy.newGameSessionsPerCreator,
            policyPeriodInMinutes: props.resourceCreationLimitPolicy.policyPeriod?.toMinutes(),
        };
    }
    parseLocations() {
        if (!this.locations || this.locations.length === 0) {
            return undefined;
        }
        const self = this;
        return this.locations.map(parseLocation);
        function parseLocation(location) {
            return {
                location: location.region,
                locationCapacity: self.parseLocationCapacity(location.capacity),
            };
        }
    }
    parseLocationCapacity(capacity) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_LocationCapacity(capacity);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseLocationCapacity);
            }
            throw error;
        }
        if (!capacity ||
            (!capacity.desiredCapacity
                && !capacity.minSize
                && !capacity.maxSize)) {
            return undefined;
        }
        return {
            desiredEc2Instances: capacity.desiredCapacity ?? 0,
            minSize: capacity.minSize ?? 0,
            maxSize: capacity.maxSize ?? 1,
        };
    }
    parseRuntimeConfiguration(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_FleetProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseRuntimeConfiguration);
            }
            throw error;
        }
        if (!props.runtimeConfiguration ||
            (!props.runtimeConfiguration.gameSessionActivationTimeout
                && !props.runtimeConfiguration.maxConcurrentGameSessionActivations
                && props.runtimeConfiguration.serverProcesses.length == 0)) {
            return undefined;
        }
        return {
            gameSessionActivationTimeoutSeconds: props.runtimeConfiguration.gameSessionActivationTimeout?.toSeconds(),
            maxConcurrentGameSessionActivations: props.runtimeConfiguration.maxConcurrentGameSessionActivations,
            serverProcesses: props.runtimeConfiguration.serverProcesses.map(parseServerProcess),
        };
        function parseServerProcess(serverProcess) {
            return {
                parameters: serverProcess.parameters,
                launchPath: serverProcess.launchPath,
                concurrentExecutions: serverProcess.concurrentExecutions ?? 1,
            };
        }
    }
    warnVpcPeeringAuthorizations(scope) {
        cdk.Annotations.of(scope).addWarning([
            'To authorize the VPC peering, call the GameLift service API CreateVpcPeeringAuthorization() or use the AWS CLI command create-vpc-peering-authorization.',
            'Make this call using the account that manages your non-GameLift resources.',
            'See: https://docs.aws.amazon.com/gamelift/latest/developerguide/vpc-peering.html',
        ].join('\n'));
    }
}
exports.FleetBase = FleetBase;
_a = JSII_RTTI_SYMBOL_1;
FleetBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.FleetBase", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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