"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GameServerGroup = exports.GameServerGroupBase = exports.BalancingStrategy = exports.DeleteOption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * The type of delete to perform.
 * To delete a game server group, specify the DeleteOption.
 */
var DeleteOption;
(function (DeleteOption) {
    /**
      * Terminates the game server group and Amazon EC2 Auto Scaling group only when it has no game servers that are in UTILIZED status.
      */
    DeleteOption["SAFE_DELETE"] = "SAFE_DELETE";
    /**
     * Terminates the game server group, including all active game servers regardless of their utilization status, and the Amazon EC2 Auto Scaling group.
     */
    DeleteOption["FORCE_DELETE"] = "FORCE_DELETE";
    /**
     * Does a safe delete of the game server group but retains the Amazon EC2 Auto Scaling group as is.
     */
    DeleteOption["RETAIN"] = "RETAIN";
})(DeleteOption = exports.DeleteOption || (exports.DeleteOption = {}));
/**
 * Indicates how GameLift FleetIQ balances the use of Spot Instances and On-Demand Instances in the game server group.
 */
var BalancingStrategy;
(function (BalancingStrategy) {
    /**
       * Only Spot Instances are used in the game server group.
       * If Spot Instances are unavailable or not viable for game hosting, the game server group provides no hosting capacity until Spot Instances can again be used.
       * Until then, no new instances are started, and the existing nonviable Spot Instances are terminated (after current gameplay ends) and are not replaced.
       */
    BalancingStrategy["SPOT_ONLY"] = "SPOT_ONLY";
    /**
     * Spot Instances are used whenever available in the game server group.
     * If Spot Instances are unavailable, the game server group continues to provide hosting capacity by falling back to On-Demand Instances.
     * Existing nonviable Spot Instances are terminated (after current gameplay ends) and are replaced with new On-Demand Instances.
     */
    BalancingStrategy["SPOT_PREFERRED"] = "SPOT_PREFERRED";
    /**
     * Only On-Demand Instances are used in the game server group.
     * No Spot Instances are used, even when available, while this balancing strategy is in force.
     */
    BalancingStrategy["ON_DEMAND_ONLY"] = "ON_DEMAND_ONLY";
})(BalancingStrategy = exports.BalancingStrategy || (exports.BalancingStrategy = {}));
/**
 * Base class for new and imported GameLift FleetIQ game server group.
 */
class GameServerGroupBase extends cdk.Resource {
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            resourceArns: [this.gameServerGroupArn],
            grantee: grantee,
            actions: actions,
        });
    }
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/GameLift',
            metricName: metricName,
            dimensionsMap: {
                GameServerGroupArn: this.gameServerGroupArn,
            },
            ...props,
        }).attachTo(this);
    }
}
exports.GameServerGroupBase = GameServerGroupBase;
_a = JSII_RTTI_SYMBOL_1;
GameServerGroupBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.GameServerGroupBase", version: "2.65.0-alpha.0" };
/**
 * Creates a GameLift FleetIQ game server group for managing game hosting on a collection of Amazon EC2 instances for game hosting.
 * This operation creates the game server group, creates an Auto Scaling group in your AWS account, and establishes a link between the two groups.
 * You can view the status of your game server groups in the GameLift console.
 * Game server group metrics and events are emitted to Amazon CloudWatch.
 * Before creating a new game server group, you must have the following:
 *  - An Amazon EC2 launch template that specifies how to launch Amazon EC2 instances with your game server build.
 *  - An IAM role that extends limited access to your AWS account to allow GameLift FleetIQ to create and interact with the Auto Scaling group.
 *
 * To create a new game server group, specify a unique group name, IAM role and Amazon EC2 launch template, and provide a list of instance types that can be used in the group.
 * You must also set initial maximum and minimum limits on the group's instance count.
 * You can optionally set an Auto Scaling policy with target tracking based on a GameLift FleetIQ metric.
 *
 * Once the game server group and corresponding Auto Scaling group are created, you have full access to change the Auto Scaling group's configuration as needed.
 * Several properties that are set when creating a game server group, including maximum/minimum size and auto-scaling policy settings, must be updated directly in the Auto Scaling group.
 * Keep in mind that some Auto Scaling group properties are periodically updated by GameLift FleetIQ as part of its balancing activities to optimize for availability and cost.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/fleetiqguide/gsg-intro.html
 *
 * @resource AWS::GameLift::GameServerGroup
 */
class GameServerGroup extends GameServerGroupBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.gameServerGroupName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameServerGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GameServerGroup);
            }
            throw error;
        }
        if (!cdk.Token.isUnresolved(props.gameServerGroupName)) {
            if (props.gameServerGroupName.length > 128) {
                throw new Error(`GameServerGroup name can not be longer than 128 characters but has ${props.gameServerGroupName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-\.]+$/.test(props.gameServerGroupName)) {
                throw new Error(`Game server group name ${props.gameServerGroupName} can contain only letters, numbers, hyphens, back slash or dot with no spaces.`);
            }
        }
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets ?? { subnetType: ec2.SubnetType.PUBLIC };
        const { subnetIds } = props.vpc.selectSubnets(this.vpcSubnets);
        if (props.minSize && props.minSize < 0) {
            throw new Error(`The minimum number of instances allowed in the Amazon EC2 Auto Scaling group cannot be lower than 0, given ${props.minSize}`);
        }
        if (props.maxSize && props.maxSize < 1) {
            throw new Error(`The maximum number of instances allowed in the Amazon EC2 Auto Scaling group cannot be lower than 1, given ${props.maxSize}`);
        }
        if (subnetIds.length > 20) {
            throw new Error(`No more than 20 subnets are allowed per game server group, given ${subnetIds.length}`);
        }
        // Add all instance definitions
        if (props.instanceDefinitions.length > 20) {
            throw new Error(`No more than 20 instance definitions are allowed per game server group, given ${props.instanceDefinitions.length}`);
        }
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.CompositePrincipal(new iam.ServicePrincipal('gamelift.amazonaws.com'), new iam.ServicePrincipal('autoscaling.amazonaws.com')),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('GameLiftGameServerGroupPolicy'),
            ],
        });
        this.grantPrincipal = this.role;
        const resource = new aws_gamelift_1.CfnGameServerGroup(this, 'Resource', {
            gameServerGroupName: this.physicalName,
            autoScalingPolicy: this.parseAutoScalingPolicy(props),
            deleteOption: props.deleteOption,
            balancingStrategy: props.balancingStrategy,
            gameServerProtectionPolicy: props.protectGameServer ? 'FULL_PROTECTION' : 'NO_PROTECTION',
            instanceDefinitions: this.parseInstanceDefinitions(props),
            launchTemplate: this.parseLaunchTemplate(props),
            minSize: props.minSize,
            maxSize: props.maxSize,
            roleArn: this.role.roleArn,
            vpcSubnets: subnetIds,
        });
        this.gameServerGroupName = this.getResourceNameAttribute(resource.ref);
        this.gameServerGroupArn = this.getResourceArnAttribute(resource.attrGameServerGroupArn, {
            service: 'gamelift',
            resource: 'gameservergroup',
            resourceName: this.physicalName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        this.autoScalingGroupArn = this.getResourceArnAttribute(resource.attrAutoScalingGroupArn, {
            service: 'autoscaling',
            resource: 'autoScalingGroup',
            resourceName: this.physicalName,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * Import an existing game server group from its attributes.
     */
    static fromGameServerGroupAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameServerGroupAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromGameServerGroupAttributes);
            }
            throw error;
        }
        if (!attrs.gameServerGroupArn && !attrs.gameServerGroupName) {
            throw new Error('Either gameServerGroupName or gameServerGroupArn must be provided in GameServerGroupAttributes');
        }
        const gameServerGroupName = attrs.gameServerGroupName ??
            cdk.Stack.of(scope).splitArn(attrs.gameServerGroupArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!gameServerGroupName) {
            throw new Error(`No game server group name found in ARN: '${attrs.gameServerGroupArn}'`);
        }
        const gameServerGroupArn = attrs.gameServerGroupArn ?? cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'gameservergroup',
            resourceName: attrs.gameServerGroupName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends GameServerGroupBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: gameServerGroupArn,
                });
                this.gameServerGroupName = gameServerGroupName;
                this.gameServerGroupArn = gameServerGroupArn;
                this.autoScalingGroupArn = attrs.autoScalingGroupArn;
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
                this.role = attrs.role;
            }
        }
        return new Import(scope, id);
    }
    parseLaunchTemplate(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameServerGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseLaunchTemplate);
            }
            throw error;
        }
        return {
            launchTemplateId: props.launchTemplate.launchTemplateId,
            launchTemplateName: props.launchTemplate.launchTemplateName,
            version: props.launchTemplate.versionNumber,
        };
    }
    parseAutoScalingPolicy(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameServerGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseAutoScalingPolicy);
            }
            throw error;
        }
        if (!props.autoScalingPolicy) {
            return undefined;
        }
        return {
            estimatedInstanceWarmup: props.autoScalingPolicy.estimatedInstanceWarmup?.toSeconds(),
            targetTrackingConfiguration: {
                targetValue: props.autoScalingPolicy.targetTrackingConfiguration,
            },
        };
    }
    parseInstanceDefinitions(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameServerGroupProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseInstanceDefinitions);
            }
            throw error;
        }
        return props.instanceDefinitions.map(parseInstanceDefinition);
        function parseInstanceDefinition(instanceDefinition) {
            return {
                instanceType: instanceDefinition.instanceType.toString(),
                weightedCapacity: instanceDefinition.weight?.toString(),
            };
        }
    }
}
exports.GameServerGroup = GameServerGroup;
_b = JSII_RTTI_SYMBOL_1;
GameServerGroup[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.GameServerGroup", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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