"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GameSessionQueue = exports.GameSessionQueueBase = exports.PriorityType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const cdk = require("aws-cdk-lib");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * Priority to condider when placing new game sessions
 */
var PriorityType;
(function (PriorityType) {
    /**
       * FleetIQ prioritizes locations where the average player latency (provided in each game session request) is lowest.
       */
    PriorityType["LATENCY"] = "LATENCY";
    /**
     * FleetIQ prioritizes destinations with the lowest current hosting costs. Cost is evaluated based on the location, instance type, and fleet type (Spot or On-Demand) for each destination in the queue.
     */
    PriorityType["COST"] = "COST";
    /**
     * FleetIQ prioritizes based on the order that destinations are listed in the queue configuration.
     */
    PriorityType["DESTINATION"] = "DESTINATION";
    /**
     * FleetIQ prioritizes based on the provided order of locations, as defined in `LocationOrder`
     */
    PriorityType["LOCATION"] = "LOCATION";
})(PriorityType = exports.PriorityType || (exports.PriorityType = {}));
/**
 * Base class for new and imported GameLift GameSessionQueue.
 */
class GameSessionQueueBase extends cdk.Resource {
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/GameLift',
            metricName: metricName,
            dimensionsMap: {
                GameSessionQueueName: this.gameSessionQueueName,
            },
            ...props,
        }).attachTo(this);
    }
    metricAverageWaitTime(props) {
        return this.metric('AverageWaitTime', props);
    }
    metricPlacementsCanceled(props) {
        return this.metric('PlacementsCanceled', props);
    }
    metricPlacementsFailed(props) {
        return this.metric('PlacementsFailed', props);
    }
    metricPlacementsStarted(props) {
        return this.metric('PlacementsStarted', props);
    }
    metricPlacementsSucceeded(props) {
        return this.metric('PlacementsSucceeded', props);
    }
    metricPlacementsTimedOut(props) {
        return this.metric('PlacementsTimedOut', props);
    }
}
exports.GameSessionQueueBase = GameSessionQueueBase;
_a = JSII_RTTI_SYMBOL_1;
GameSessionQueueBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.GameSessionQueueBase", version: "2.65.0-alpha.0" };
/**
 * The GameSessionQueue resource creates a placement queue that processes requests for new game sessions.
 * A queue uses FleetIQ algorithms to determine the best placement locations and find an available game server, then prompts the game server to start a new game session.
 * Queues can have destinations (GameLift fleets or gameSessionQueuees), which determine where the queue can place new game sessions.
 * A queue can have destinations with varied fleet type (Spot and On-Demand), instance type, and AWS Region.
 *
 * @resource AWS::GameLift::GameSessionQueue
 */
class GameSessionQueue extends GameSessionQueueBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.gameSessionQueueName,
        });
        this.destinations = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameSessionQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GameSessionQueue);
            }
            throw error;
        }
        if (!cdk.Token.isUnresolved(props.gameSessionQueueName)) {
            if (props.gameSessionQueueName.length > 128) {
                throw new Error(`GameSessionQueue name can not be longer than 128 characters but has ${props.gameSessionQueueName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-]+$/.test(props.gameSessionQueueName)) {
                throw new Error(`GameSessionQueue name ${props.gameSessionQueueName} can contain only letters, numbers, hyphens with no spaces.`);
            }
        }
        if (props.customEventData && props.customEventData.length > 256) {
            throw new Error(`GameSessionQueue custom event data can not be longer than 256 characters but has ${props.customEventData.length} characters.`);
        }
        if (props.allowedLocations && props.allowedLocations.length > 100) {
            throw new Error(`No more than 100 allowed locations are allowed per game session queue, given ${props.allowedLocations.length}`);
        }
        // Add all destinations
        (props.destinations || []).forEach(this.addDestination.bind(this));
        const resource = new aws_gamelift_1.CfnGameSessionQueue(this, 'Resource', {
            name: this.physicalName,
            customEventData: props.customEventData,
            destinations: cdk.Lazy.any({ produce: () => this.parseDestinations() }),
            filterConfiguration: this.parseFilterConfiguration(props),
            notificationTarget: props.notificationTarget && props.notificationTarget.topicArn,
            playerLatencyPolicies: this.parsePlayerLatencyPolicies(props),
            priorityConfiguration: this.parsePriorityConfiguration(props),
            timeoutInSeconds: props.timeout && props.timeout.toSeconds(),
        });
        this.gameSessionQueueName = this.getResourceNameAttribute(resource.ref);
        this.gameSessionQueueArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'gamesessionqueue',
            resourceName: this.gameSessionQueueName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
     * Import an existing gameSessionQueue from its name.
     */
    static fromGameSessionQueueName(scope, id, gameSessionQueueName) {
        return this.fromGameSessionQueueAttributes(scope, id, { gameSessionQueueName });
    }
    /**
     * Import an existing gameSessionQueue from its ARN.
     */
    static fromGameSessionQueueArn(scope, id, gameSessionQueueArn) {
        return this.fromGameSessionQueueAttributes(scope, id, { gameSessionQueueArn });
    }
    /**
     * Import an existing gameSessionQueue from its attributes.
     */
    static fromGameSessionQueueAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameSessionQueueAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromGameSessionQueueAttributes);
            }
            throw error;
        }
        if (!attrs.gameSessionQueueName && !attrs.gameSessionQueueArn) {
            throw new Error('Either gameSessionQueueName or gameSessionQueueArn must be provided in GameSessionQueueAttributes');
        }
        const gameSessionQueueName = attrs.gameSessionQueueName ??
            cdk.Stack.of(scope).splitArn(attrs.gameSessionQueueArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!gameSessionQueueName) {
            throw new Error(`No gameSessionQueue name found in ARN: '${attrs.gameSessionQueueArn}'`);
        }
        const gameSessionQueueArn = attrs.gameSessionQueueArn ?? cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'gamesessionqueue',
            resourceName: attrs.gameSessionQueueName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
        class Import extends GameSessionQueueBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: gameSessionQueueArn,
                });
                this.gameSessionQueueName = gameSessionQueueName;
                this.gameSessionQueueArn = gameSessionQueueArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a destination to fulfill requests for new game sessions
     *
     * @param destination A destination to add
     */
    addDestination(destination) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_IGameSessionQueueDestination(destination);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addDestination);
            }
            throw error;
        }
        this.destinations.push(destination);
    }
    parsePriorityConfiguration(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameSessionQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parsePriorityConfiguration);
            }
            throw error;
        }
        if (!props.priorityConfiguration) {
            return undefined;
        }
        if (props.priorityConfiguration.locationOrder.length > 100) {
            throw new Error(`No more than 100 locations are allowed per priority configuration, given ${props.priorityConfiguration.locationOrder.length}`);
        }
        if (props.priorityConfiguration.priorityOrder.length > 4) {
            throw new Error(`No more than 4 priorities are allowed per priority configuration, given ${props.priorityConfiguration.priorityOrder.length}`);
        }
        return {
            priorityOrder: props.priorityConfiguration.priorityOrder,
            locationOrder: props.priorityConfiguration.locationOrder,
        };
    }
    parsePlayerLatencyPolicies(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameSessionQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parsePlayerLatencyPolicies);
            }
            throw error;
        }
        if (!props.playerLatencyPolicies) {
            return undefined;
        }
        return props.playerLatencyPolicies.map(parsePlayerLatencyPolicy);
        function parsePlayerLatencyPolicy(playerLatencyPolicy) {
            return {
                maximumIndividualPlayerLatencyMilliseconds: playerLatencyPolicy.maximumIndividualPlayerLatency.toMilliseconds(),
                policyDurationSeconds: playerLatencyPolicy.policyDuration && playerLatencyPolicy.policyDuration.toSeconds(),
            };
        }
    }
    parseFilterConfiguration(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_GameSessionQueueProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.parseFilterConfiguration);
            }
            throw error;
        }
        if (!props.allowedLocations) {
            return undefined;
        }
        return {
            allowedLocations: props.allowedLocations,
        };
    }
    parseDestinations() {
        if (!this.destinations || this.destinations.length === 0) {
            return undefined;
        }
        return this.destinations.map(parseDestination);
        function parseDestination(destination) {
            return {
                destinationArn: destination.resourceArnForDestination,
            };
        }
    }
}
exports.GameSessionQueue = GameSessionQueue;
_b = JSII_RTTI_SYMBOL_1;
GameSessionQueue[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.GameSessionQueue", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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