/**
 * Protocol for use in Connection Rules
 *
 * https://www.iana.org/assignments/protocol-numbers/protocol-numbers.xhtml
 */
export declare enum Protocol {
    TCP = "TCP",
    UDP = "UDP"
}
/**
 * Properties to create a port range
 */
export interface PortProps {
    /**
       * The protocol for the range
       */
    readonly protocol: Protocol;
    /**
       * A starting value for a range of allowed port numbers.
       *
       * For fleets using Windows and Linux builds, only ports 1026-60000 are valid.
       */
    readonly fromPort: number;
    /**
       * An ending value for a range of allowed port numbers. Port numbers are end-inclusive.
       * This value must be higher than `fromPort`.
       *
       * For fleets using Windows and Linux builds, only ports 1026-60000 are valid.
       *
       * @default the `fromPort` value
       */
    readonly toPort?: number;
}
/**
 * Interface for classes that provide the connection-specification parts of a security group rule
 */
export declare class Port {
    private readonly props;
    /**
       * A single TCP port
       */
    static tcp(port: number): Port;
    /**
       * A TCP port range
       */
    static tcpRange(startPort: number, endPort: number): Port;
    /**
       * Any TCP traffic
       */
    static allTcp(): Port;
    /**
       * A single UDP port
       */
    static udp(port: number): Port;
    /**
       * A UDP port range
       */
    static udpRange(startPort: number, endPort: number): Port;
    /**
       * Any UDP traffic
       */
    static allUdp(): Port;
    constructor(props: PortProps);
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toJson(): any;
}
/**
 * Interface for classes that provide the peer-specification parts of an inbound permission
 */
export interface IPeer {
    /**
       * A unique identifier for this connection peer
       */
    readonly uniqueId: string;
    /**
       * Produce the ingress rule JSON for the given connection
       */
    toJson(): any;
}
/**
   * Peer object factories
   *
   * The static methods on this object can be used to create peer objects
   * which represent a connection partner in inbound permission rules.
   *
   * Use this object if you need to represent connection partners using plain IP addresses.
   */
export declare class Peer {
    /**
       * Create an IPv4 peer from a CIDR
       */
    static ipv4(cidrIp: string): IPeer;
    /**
       * Any IPv4 address
       */
    static anyIpv4(): IPeer;
    protected constructor();
}
/**
 * A range of IP addresses and port settings that allow inbound traffic to connect to server processes on an instance in a fleet.
 * New game sessions are assigned an IP address/port number combination, which must fall into the fleet's allowed ranges.
 *
 * Fleets with custom game builds must have permissions explicitly set.
 * For Realtime Servers fleets, GameLift automatically opens two port ranges, one for TCP messaging and one for UDP.
 */
export interface IngressRule {
    /**
       * The port range used for ingress traffic
       */
    readonly port: Port;
    /**
     * A range of allowed IP addresses .
     */
    readonly source: IPeer;
}
