import * as sns from 'aws-cdk-lib/aws-sns';
import { Construct } from 'constructs';
import { IGameSessionQueue } from './game-session-queue';
import { MatchmakingConfigurationProps, GameProperty, MatchmakingConfigurationBase, IMatchmakingConfiguration } from './matchmaking-configuration';
/**
 * Properties for a new queued matchmaking configuration
 */
export interface QueuedMatchmakingConfigurationProps extends MatchmakingConfigurationProps {
    /**
       * The number of player slots in a match to keep open for future players.
       * For example, if the configuration's rule set specifies a match for a single 12-person team, and the additional player count is set to 2, only 10 players are selected for the match.
       *
       * @default no additional player slots
       */
    readonly additionalPlayerCount?: number;
    /**
     * The method used to backfill game sessions that are created with this matchmaking configuration.
     * - Choose manual when your game manages backfill requests manually or does not use the match backfill feature.
     * - Otherwise backfill is settled to automatic to have GameLift create a `StartMatchBackfill` request whenever a game session has one or more open slots.
     *
     * @see https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-backfill.html
     *
     * @default automatic backfill mode
     */
    readonly manualBackfillMode?: boolean;
    /**
     * A set of custom properties for a game session, formatted as key-value pairs.
     * These properties are passed to a game server process with a request to start a new game session.
     *
     * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession
     *
     * @default no additional game properties
     */
    readonly gameProperties?: GameProperty[];
    /**
     * A set of custom game session properties, formatted as a single string value.
     * This data is passed to a game server process with a request to start a new game session.
     *
     * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-server-api.html#gamelift-sdk-server-startsession
     *
     * @default no additional game session data
     */
    readonly gameSessionData?: string;
    /**
     * Queues are used to start new GameLift-hosted game sessions for matches that are created with this matchmaking configuration.
     *
     * Queues can be located in any Region.
     */
    readonly gameSessionQueues: IGameSessionQueue[];
}
/**
 * A FlexMatch matchmaker process does the work of building a game match.
 * It manages the pool of matchmaking requests received, forms teams for a match, processes and selects players to find the best possible player groups, and initiates the process of placing and starting a game session for the match.
 * This topic describes the key aspects of a matchmaker and how to configure one customized for your game.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-configuration.html
 *
 * @resource AWS::GameLift::MatchmakingConfiguration
 */
export declare class QueuedMatchmakingConfiguration extends MatchmakingConfigurationBase {
    /**
     * Import an existing matchmaking configuration from its name.
     */
    static fromQueuedMatchmakingConfigurationName(scope: Construct, id: string, matchmakingConfigurationName: string): IMatchmakingConfiguration;
    /**
     * Import an existing matchmaking configuration from its ARN.
     */
    static fromQueuedMatchmakingConfigurationArn(scope: Construct, id: string, matchmakingConfigurationArn: string): IMatchmakingConfiguration;
    /**
       * The name of the matchmaking configuration.
       */
    readonly matchmakingConfigurationName: string;
    /**
       * The ARN of the matchmaking configuration.
       */
    readonly matchmakingConfigurationArn: string;
    /**
     * The notification target for matchmaking events
     */
    readonly notificationTarget?: sns.ITopic;
    /**
     * A list of game session queue destinations
     */
    private readonly gameSessionQueues;
    constructor(scope: Construct, id: string, props: QueuedMatchmakingConfigurationProps);
    /**
     * Adds a game session queue destination to the matchmaking configuration.
     *
     * @param gameSessionQueue A game session queue
     */
    addGameSessionQueue(gameSessionQueue: IGameSessionQueue): void;
    private parseGameSessionQueues;
    private parseGameProperties;
}
