"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.QueuedMatchmakingConfiguration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const cdk = require("aws-cdk-lib");
const gamelift = require("aws-cdk-lib/aws-gamelift");
const matchmaking_configuration_1 = require("./matchmaking-configuration");
/**
 * A FlexMatch matchmaker process does the work of building a game match.
 * It manages the pool of matchmaking requests received, forms teams for a match, processes and selects players to find the best possible player groups, and initiates the process of placing and starting a game session for the match.
 * This topic describes the key aspects of a matchmaker and how to configure one customized for your game.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-configuration.html
 *
 * @resource AWS::GameLift::MatchmakingConfiguration
 */
class QueuedMatchmakingConfiguration extends matchmaking_configuration_1.MatchmakingConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.matchmakingConfigurationName,
        });
        /**
         * A list of game session queue destinations
         */
        this.gameSessionQueues = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_QueuedMatchmakingConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, QueuedMatchmakingConfiguration);
            }
            throw error;
        }
        if (props.matchmakingConfigurationName && !cdk.Token.isUnresolved(props.matchmakingConfigurationName)) {
            if (props.matchmakingConfigurationName.length > 128) {
                throw new Error(`Matchmaking configuration name can not be longer than 128 characters but has ${props.matchmakingConfigurationName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-\.]+$/.test(props.matchmakingConfigurationName)) {
                throw new Error(`Matchmaking configuration name ${props.matchmakingConfigurationName} can contain only letters, numbers, hyphens, back slash or dot with no spaces.`);
            }
        }
        if (props.description && !cdk.Token.isUnresolved(props.description)) {
            if (props.description.length > 1024) {
                throw new Error(`Matchmaking configuration description can not be longer than 1024 characters but has ${props.description.length} characters.`);
            }
        }
        if (props.gameProperties && props.gameProperties.length > 16) {
            throw new Error(`The maximum number of game properties allowed in the matchmaking configuration cannot be higher than 16, given ${props.gameProperties.length}`);
        }
        if (props.gameSessionData && props.gameSessionData.length > 4096) {
            throw new Error(`Matchmaking configuration game session data can not be longer than 4096 characters but has ${props.gameSessionData.length} characters.`);
        }
        if (props.customEventData && props.customEventData.length > 256) {
            throw new Error(`Matchmaking configuration custom event data can not be longer than 256 characters but has ${props.customEventData.length} characters.`);
        }
        if (props.acceptanceTimeout && props.acceptanceTimeout.toSeconds() > 600) {
            throw new Error(`Matchmaking configuration acceptance timeout can not exceed 600 seconds, actual ${props.acceptanceTimeout.toSeconds()} seconds.`);
        }
        if (props.requestTimeout && props.requestTimeout.toSeconds() > 43200) {
            throw new Error(`Matchmaking configuration request timeout can not exceed 43200 seconds, actual ${props.requestTimeout.toSeconds()} seconds.`);
        }
        //Notification target
        this.notificationTarget = props.notificationTarget;
        if (!this.notificationTarget) {
            this.notificationTarget = new sns.Topic(this, 'Topic', {});
        }
        // Be sure to add the right TopicPolicy to enable gamelift publish action to given topic
        const topicPolicy = new sns.TopicPolicy(this, 'TopicPolicy', {
            topics: [this.notificationTarget],
        });
        topicPolicy.document.addStatements(new iam.PolicyStatement({
            actions: ['sns:Publish'],
            principals: [new iam.ServicePrincipal('gamelift.amazonaws.com')],
            resources: [this.notificationTarget.topicArn],
        }));
        // Add all queues
        (props.gameSessionQueues || []).forEach(this.addGameSessionQueue.bind(this));
        const resource = new gamelift.CfnMatchmakingConfiguration(this, 'Resource', {
            name: this.physicalName,
            acceptanceRequired: Boolean(props.requireAcceptance),
            acceptanceTimeoutSeconds: props.acceptanceTimeout && props.acceptanceTimeout.toSeconds(),
            additionalPlayerCount: props.additionalPlayerCount,
            backfillMode: props.manualBackfillMode ? 'MANUAL' : 'AUTOMATIC',
            customEventData: props.customEventData,
            description: props.description,
            flexMatchMode: 'WITH_QUEUE',
            gameProperties: this.parseGameProperties(props),
            gameSessionData: props.gameSessionData,
            gameSessionQueueArns: cdk.Lazy.list({ produce: () => this.parseGameSessionQueues() }),
            notificationTarget: this.notificationTarget.topicArn,
            requestTimeoutSeconds: props.requestTimeout && props.requestTimeout.toSeconds() || cdk.Duration.seconds(300).toSeconds(),
            ruleSetName: props.ruleSet.matchmakingRuleSetName,
        });
        this.matchmakingConfigurationName = this.getResourceNameAttribute(resource.ref);
        this.matchmakingConfigurationArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'matchmakingconfiguration',
            resourceName: this.matchmakingConfigurationName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
     * Import an existing matchmaking configuration from its name.
     */
    static fromQueuedMatchmakingConfigurationName(scope, id, matchmakingConfigurationName) {
        return this.fromMatchmakingConfigurationAttributes(scope, id, { matchmakingConfigurationName: matchmakingConfigurationName });
    }
    /**
     * Import an existing matchmaking configuration from its ARN.
     */
    static fromQueuedMatchmakingConfigurationArn(scope, id, matchmakingConfigurationArn) {
        return this.fromMatchmakingConfigurationAttributes(scope, id, { matchmakingConfigurationArn: matchmakingConfigurationArn });
    }
    /**
     * Adds a game session queue destination to the matchmaking configuration.
     *
     * @param gameSessionQueue A game session queue
     */
    addGameSessionQueue(gameSessionQueue) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_IGameSessionQueue(gameSessionQueue);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addGameSessionQueue);
            }
            throw error;
        }
        this.gameSessionQueues.push(gameSessionQueue);
    }
    parseGameSessionQueues() {
        if (!this.gameSessionQueues || this.gameSessionQueues.length === 0) {
            return undefined;
        }
        return this.gameSessionQueues.map((queue) => queue.gameSessionQueueArn);
    }
    parseGameProperties(props) {
        if (!props.gameProperties || props.gameProperties.length === 0) {
            return undefined;
        }
        return props.gameProperties.map(parseGameProperty);
        function parseGameProperty(gameProperty) {
            return {
                key: gameProperty.key,
                value: gameProperty.value,
            };
        }
    }
}
exports.QueuedMatchmakingConfiguration = QueuedMatchmakingConfiguration;
_a = JSII_RTTI_SYMBOL_1;
QueuedMatchmakingConfiguration[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.QueuedMatchmakingConfiguration", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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