"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Script = exports.ScriptBase = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const content_1 = require("./content");
const aws_gamelift_1 = require("aws-cdk-lib/aws-gamelift");
/**
 * Base class for new and imported GameLift realtime server script.
 */
class ScriptBase extends cdk.Resource {
}
exports.ScriptBase = ScriptBase;
_a = JSII_RTTI_SYMBOL_1;
ScriptBase[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.ScriptBase", version: "2.65.0-alpha.0" };
/**
 * A GameLift script, that is installed and runs on instances in an Amazon GameLift fleet. It consists of
 * a zip file with all of the components of the realtime game server script.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/developerguide/realtime-script-uploading.html
 *
 * @resource AWS::GameLift::Script
 */
class Script extends ScriptBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.scriptName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_ScriptProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Script);
            }
            throw error;
        }
        if (props.scriptName && !cdk.Token.isUnresolved(props.scriptName)) {
            if (props.scriptName.length > 1024) {
                throw new Error(`Script name can not be longer than 1024 characters but has ${props.scriptName.length} characters.`);
            }
        }
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('gamelift.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const content = props.content.bind(this, this.role);
        const resource = new aws_gamelift_1.CfnScript(this, 'Resource', {
            name: props.scriptName,
            version: props.scriptVersion,
            storageLocation: {
                bucket: content.s3Location && content.s3Location.bucketName,
                key: content.s3Location && content.s3Location.objectKey,
                objectVersion: content.s3Location && content.s3Location.objectVersion,
                roleArn: this.role.roleArn,
            },
        });
        this.scriptId = this.getResourceNameAttribute(resource.ref);
        this.scriptArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'gamelift',
            resource: `script/${this.physicalName}`,
            arnFormat: cdk.ArnFormat.COLON_RESOURCE_NAME,
        });
    }
    /**
     * Create a new realtime server script from s3 content
     */
    static fromBucket(scope, id, bucket, key, objectVersion) {
        return new Script(scope, id, {
            content: content_1.Content.fromBucket(bucket, key, objectVersion),
        });
    }
    /**
     * Create a new realtime server script from asset content
     */
    static fromAsset(scope, id, path, options) {
        return new Script(scope, id, {
            content: content_1.Content.fromAsset(path, options),
        });
    }
    /**
       * Import a script into CDK using its ARN
       */
    static fromScriptArn(scope, id, scriptArn) {
        return this.fromScriptAttributes(scope, id, { scriptArn });
    }
    /**
     * Import an existing realtime server script from its attributes.
     */
    static fromScriptAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_ScriptAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromScriptAttributes);
            }
            throw error;
        }
        const scriptArn = attrs.scriptArn;
        const scriptId = extractIdFromArn(attrs.scriptArn);
        const role = attrs.role;
        class Import extends ScriptBase {
            constructor(s, i) {
                super(s, i, {
                    environmentFromArn: scriptArn,
                });
                this.scriptArn = scriptArn;
                this.scriptId = scriptId;
                this.role = role;
                this.grantPrincipal = this.role || new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
}
exports.Script = Script;
_b = JSII_RTTI_SYMBOL_1;
Script[_b] = { fqn: "@aws-cdk/aws-gamelift-alpha.Script", version: "2.65.0-alpha.0" };
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the script
 * identifier from the ARN.
 *
 * Script ARNs look like this:
 *
 *   arn:aws:gamelift:region:account-id:script/script-identifier
 *
 * ..which means that in order to extract the `script-identifier` component from the ARN, we can
 * split the ARN using ":" and select the component in index 5 then split using "/" and select the component in index 1.
 *
 * @returns the script identifier from his ARN
 */
function extractIdFromArn(arn) {
    const splitValue = cdk.Fn.select(5, cdk.Fn.split(':', arn));
    return cdk.Fn.select(1, cdk.Fn.split('/', splitValue));
}
//# sourceMappingURL=data:application/json;base64,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