"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StandaloneMatchmakingConfiguration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const sns = require("aws-cdk-lib/aws-sns");
const cdk = require("aws-cdk-lib");
const gamelift = require("aws-cdk-lib/aws-gamelift");
const matchmaking_configuration_1 = require("./matchmaking-configuration");
/**
 * A FlexMatch matchmaker process does the work of building a game match.
 * It manages the pool of matchmaking requests received, forms teams for a match, processes and selects players to find the best possible player groups, and initiates the process of placing and starting a game session for the match.
 * This topic describes the key aspects of a matchmaker and how to configure one customized for your game.
 *
 * @see https://docs.aws.amazon.com/gamelift/latest/flexmatchguide/match-configuration.html
 *
 * @resource AWS::GameLift::MatchmakingConfiguration
 */
class StandaloneMatchmakingConfiguration extends matchmaking_configuration_1.MatchmakingConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.matchmakingConfigurationName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_gamelift_alpha_StandaloneMatchmakingConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, StandaloneMatchmakingConfiguration);
            }
            throw error;
        }
        if (props.matchmakingConfigurationName && !cdk.Token.isUnresolved(props.matchmakingConfigurationName)) {
            if (props.matchmakingConfigurationName.length > 128) {
                throw new Error(`Matchmaking configuration name can not be longer than 128 characters but has ${props.matchmakingConfigurationName.length} characters.`);
            }
            if (!/^[a-zA-Z0-9-\.]+$/.test(props.matchmakingConfigurationName)) {
                throw new Error(`Matchmaking configuration name ${props.matchmakingConfigurationName} can contain only letters, numbers, hyphens, back slash or dot with no spaces.`);
            }
        }
        if (props.description && !cdk.Token.isUnresolved(props.description)) {
            if (props.description.length > 1024) {
                throw new Error(`Matchmaking configuration description can not be longer than 1024 characters but has ${props.description.length} characters.`);
            }
        }
        if (props.customEventData && props.customEventData.length > 256) {
            throw new Error(`Matchmaking configuration custom event data can not be longer than 256 characters but has ${props.customEventData.length} characters.`);
        }
        if (props.acceptanceTimeout && props.acceptanceTimeout.toSeconds() > 600) {
            throw new Error(`Matchmaking configuration acceptance timeout can not exceed 600 seconds, actual ${props.acceptanceTimeout.toSeconds()} seconds.`);
        }
        if (props.requestTimeout && props.requestTimeout.toSeconds() > 43200) {
            throw new Error(`Matchmaking configuration request timeout can not exceed 43200 seconds, actual ${props.requestTimeout.toSeconds()} seconds.`);
        }
        //Notification target
        this.notificationTarget = props.notificationTarget;
        if (!this.notificationTarget) {
            this.notificationTarget = new sns.Topic(this, 'Topic', {});
        }
        // Be sure to add the right TopicPolicy to enable gamelift publish action to given topic
        const topicPolicy = new sns.TopicPolicy(this, 'TopicPolicy', {
            topics: [this.notificationTarget],
        });
        topicPolicy.document.addStatements(new iam.PolicyStatement({
            actions: ['sns:Publish'],
            principals: [new iam.ServicePrincipal('gamelift.amazonaws.com')],
            resources: [this.notificationTarget.topicArn],
        }));
        const resource = new gamelift.CfnMatchmakingConfiguration(this, 'Resource', {
            name: this.physicalName,
            acceptanceRequired: Boolean(props.requireAcceptance),
            acceptanceTimeoutSeconds: props.acceptanceTimeout && props.acceptanceTimeout.toSeconds(),
            customEventData: props.customEventData,
            description: props.description,
            flexMatchMode: 'STANDALONE',
            notificationTarget: this.notificationTarget.topicArn,
            requestTimeoutSeconds: props.requestTimeout && props.requestTimeout.toSeconds() || cdk.Duration.seconds(300).toSeconds(),
            ruleSetName: props.ruleSet.matchmakingRuleSetName,
        });
        this.matchmakingConfigurationName = this.getResourceNameAttribute(resource.ref);
        this.matchmakingConfigurationArn = cdk.Stack.of(scope).formatArn({
            service: 'gamelift',
            resource: 'matchmakingconfiguration',
            resourceName: this.matchmakingConfigurationName,
            arnFormat: cdk.ArnFormat.SLASH_RESOURCE_NAME,
        });
    }
    /**
       * Import an existing matchmaking configuration from its name.
       */
    static fromStandaloneMatchmakingConfigurationName(scope, id, matchmakingConfigurationName) {
        return this.fromMatchmakingConfigurationAttributes(scope, id, { matchmakingConfigurationName });
    }
    /**
       * Import an existing matchmaking configuration from its ARN.
       */
    static fromStandaloneMatchmakingConfigurationArn(scope, id, matchmakingConfigurationArn) {
        return this.fromMatchmakingConfigurationAttributes(scope, id, { matchmakingConfigurationArn });
    }
}
exports.StandaloneMatchmakingConfiguration = StandaloneMatchmakingConfiguration;
_a = JSII_RTTI_SYMBOL_1;
StandaloneMatchmakingConfiguration[_a] = { fqn: "@aws-cdk/aws-gamelift-alpha.StandaloneMatchmakingConfiguration", version: "2.65.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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