# -*- coding: utf-8 -*-
"""
Sources for Movie
"""
from __future__ import unicode_literals

import re

import simplejson as json
from bs4 import BeautifulSoup, Tag
from dateutil import parser
from lxml import html

from ..utils import KinopoiskPage, KinopoiskImagesPage


class MovieCareerLink(KinopoiskPage):
    """
    Parser movie info from person career links
    """
    xpath = {
        'id': './/@data-fid',
        'imdb_rating': './/@data-imdbr',
        'imdb_votes': './/@data-imdbv',
        'rating': './/div[@class="rating kp"]/a/text()',
        'votes': './/div[@class="rating kp"]/span/text()',
        'link': './/span[@class="name"]/a/text()',
        'role': './/span[@class="role"]/text()',
    }

    def parse(self):
        self.instance.id = self.extract('id')
        self.instance.imdb_rating = self.extract('imdb_rating', to_float=True)
        self.instance.imdb_votes = self.extract('imdb_votes', to_int=True)
        self.instance.rating = self.extract('rating', to_float=True)
        self.instance.votes = self.extract('votes', to_int=True)

        link = self.extract('link', to_str=True)
        role = self.extract('role').strip().split('...')
        title, year = re.findall(r'^(.+?)(?:\s+\(.*([0-9]{4})\))?$', link, re.M)[0]
        if role[0] == '':
            title = ''
            title_en = title
        else:
            title_en = role[0]

        self.instance.title = self.prepare_str(title)
        self.instance.title_en = self.prepare_str(title_en)
        if year:
            self.instance.year = self.prepare_int(year)

        self.instance.set_source('career_link')


class MoviePremierLink(KinopoiskPage):
    """
    Parser movie info from premiers links
    """

    def parse(self):

        if isinstance(self.content, Tag):
            premier_soup = self.content
        else:
            content_soup = BeautifulSoup(self.content, 'html.parser')
            premier_soup = content_soup.find('div', {'class': 'premier_item'})

        title_soup = premier_soup.find('span', {'class': 'name_big'}) or premier_soup.find('span', {'class': 'name'})

        self.instance.id = self.prepare_int(premier_soup['id'])
        self.instance.title = self.prepare_str(title_soup.find('a').contents[0])
        date = premier_soup.find('meta', {'itemprop': 'startDate'})['content']
        try:
            self.instance.release = parser.parse(date)
        except Exception:
            pass

        match = re.findall(r'^(.+) \((\d{4})\)$', title_soup.nextSibling.nextSibling.contents[0])
        if len(match):
            self.instance.title_en = self.prepare_str(match[0][0].strip())
            self.instance.year = self.prepare_int(match[0][1])

        try:
            self.instance.plot = self.prepare_str(premier_soup.find('span', {'class': 'sinopsys'}).contents[0])
        except Exception:
            pass

        self.instance.set_source('premier_link')


class MovieLink(KinopoiskPage):
    """
    Parser movie info from links
    """
    xpath = {
        'url': './/p[@class="name"]/a/@href',
        'title': './/p[@class="name"]/a/text()',
        'years': './/p[@class="name"]/span[@class="year"]/text()',
        'title_en': './/span[@class="gray"][1]/text()',
        'rating': './/div[starts-with(@class, "rating")]/@title',
    }

    def parse(self):
        self.content = html.fromstring(self.content)

        url = self.extract('url')
        title = self.extract('title', to_str=True)
        years = self.extract('years')
        title_en = self.extract('title_en', to_str=True)
        rating = self.extract('rating')

        self.instance.id = self.prepare_int(url.split('/')[2].split('-')[-1])
        self.instance.title = title.replace('(сериал)', '')
        self.instance.series = '(сериал)' in title

        if years:
            self.instance.year = self.prepare_int(years[:4])

        if 'мин' in title_en:
            values = title_en.split(', ')
            self.instance.runtime = self.prepare_int(values[-1].split(' ')[0])
            self.instance.title_en = ', '.join(values[:-1])
        else:
            self.instance.title_en = title_en

        if rating:
            rating = rating.split(' ')
            self.instance.rating = float(rating[0])
            self.instance.votes = self.prepare_int(rating[1][1:-1])

        self.instance.set_source('link')


class MovieSeries(KinopoiskPage):
    url = '/film/{id}/episodes/'

    def parse(self):
        soup = BeautifulSoup(self.content, 'html.parser')
        for season in soup.findAll('h1', attrs={'class': 'moviename-big'}):
            if '21px' not in season['style']:
                continue

            parts = season.nextSibling.split(',')
            if len(parts) == 2:
                year = self.prepare_int(parts[0])
            tbody = season.parent.parent.parent
            episodes = []
            for tr in tbody.findAll('tr')[1:]:
                if not tr.find('h1'):
                    continue

                raw_date = tr.find('td', attrs={'width': '20%'}).string
                if raw_date.strip().count('\xa0') == 2:
                    normalized_date = self.prepare_date(raw_date)
                else:
                    normalized_date = raw_date
                title = tr.find('h1').b.string
                if title.startswith('Эпизод #'):
                    title = None
                episodes.append((title, normalized_date))

            if episodes:
                self.instance.add_series_season(year, episodes)


class MovieMainPage(KinopoiskPage):
    """
    Parser of main movie page
    """
    url = '/film/{id}/'
    main_persons = {
        'режиссер': 'directors',
        'сценарий': 'screenwriters',
        'продюсер': 'producers',
        'оператор': 'operators',
        'композитор': 'composers',
        'художник': 'art_direction_by',
        'монтаж': 'editing_by',
    }
    main_profits = {
        'бюджет': 'budget',
        'маркетинг': 'marketing',
        'сборы в США': 'profit_usa',
        'сборы в России': 'profit_russia',
        'сборы в мире': 'profit_world',
    }

    xpath = {
        'url': './/meta[@property="og:url"]/@content',
        'title': './/h1/text()',
        'title_en': './/span[@itemprop="alternativeHeadline"]/text()',
        'plot': './/div[@itemprop="description"]/text()',
        'rating': './/span[@class="rating_ball"]/text()',
        'votes': './/div[@id="block_rating"]//div[@class="div1"]//span[@class="ratingCount"]/text()',
        'imdb': './/div[@id="block_rating"]//div[@class="block_2"]//div[last()]/text()',
    }

    def parse(self):

        trailers = re.findall(r'GetTrailerPreview\(([^\)]+)\)', self.content)
        if len(trailers):
            self.instance.add_trailer(json.loads(trailers[0].replace("'", '"')))

        content_info = BeautifulSoup(self.content, 'html.parser')

        table_info = content_info.find('table', {'class': re.compile(r'^info')})
        if table_info:
            for tr in table_info.findAll('tr'):
                tds = tr.findAll('td')
                name = tds[0].text
                value = tds[1].text
                if value == '-':
                    continue

                if name == 'слоган':
                    self.instance.tagline = self.prepare_str(value)
                elif name == 'время':
                    self.instance.runtime = self.prepare_int(value.split(' ')[0])
                elif name == 'год':
                    try:
                        self.instance.year = self.prepare_int(value.split('(')[0])
                    except ValueError:
                        pass
                    self.instance.series = 'сезон' in value
                elif name == 'страна':
                    for item in value.split(', '):
                        self.instance.countries.append(self.prepare_str(item))
                elif name == 'жанр':
                    genres = value.split(', ')
                    for genre in genres:
                        if genre != '...\nслова\n':
                            self.instance.genres.append(self.prepare_str(genre))
                elif name in self.main_profits:
                    self.parse_main_profit(self.main_profits[name], tds)
                elif name in self.main_persons:
                    self.parse_persons(self.main_persons[name], tds[1].contents)

        actors = content_info.find('div', {'id': 'actorList'})
        if actors and actors.ul:
            self.parse_persons('actors', [li.a for li in actors.ul.findAll('li')])

        self.content = html.fromstring(self.content)

        self.instance.id = self.prepare_int(self.extract('url').split('/')[-2].split('-')[-1])
        self.instance.title = self.extract('title', to_str=True)
        self.instance.title_en = self.extract('title_en', to_str=True)
        self.instance.plot = self.extract('plot', to_str=True)
        self.instance.rating = self.extract('rating', to_float=True)
        self.instance.votes = self.extract('votes', to_int=True)

        imdb = re.findall(r'^IMDb: ([0-9\.]+) \(([0-9 ]+)\)$', self.extract('imdb'))
        if imdb:
            self.instance.imdb_rating = float(imdb[0][0])
            self.instance.imdb_votes = self.prepare_int(imdb[0][1])

        self.instance.set_source('main_page')

    def parse_main_profit(self, field_name, value):
        setattr(self.instance, field_name, self.find_profit(value[1]))

    def parse_persons(self, field_name, links):
        from kinopoisk.person import Person
        for link in links:
            if isinstance(link, Tag) and link.text != "...":
                person = Person.get_parsed('short_link', link.decode())
                getattr(self.instance, field_name).append(person)


class MoviePostersPage(KinopoiskImagesPage):
    """
    Parser of movie posters page
    """
    url = '/film/{id}/posters/'
    field_name = 'posters'


class MovieTrailersPage(KinopoiskPage):
    """
    Parser of kinopoisk trailers page
    """
    url = '/film/{id}/video/'

    def parse(self):
        trailers = re.findall(r'GetTrailerPreview\(([^\)]+)\)', self.content)
        for trailer in trailers:
            self.instance.add_trailer(json.loads(trailer.replace("'", '"')))

        self.instance.youtube_ids = list(set(re.findall(r'//www.youtube.com/v/(.+)\?', self.content)))
        self.instance.set_source(self.content_name)
