"""
Maps all the MQTT topics to either attributes or metrics.
"""

from typing import List
from .constants import MetricKind, MetricNature, MetricType, ValueType, RangeType
from .data_classes import TopicDescriptor
from ._victron_enums import DESSReactiveStrategy, DESSStrategy, DeviceType, ESSMode, FluidType, InverterMode, GenericOnOff, EvChargerMode, InverterOverloadAlarmEnum, InverterState, MultiState, SolarChargerState, TemperatureStatus, TemperatureType, DESSErrorCode, DESSRestrictions

# Good sources for topics is:
# https://github.com/victronenergy/venus/wiki/dbus
# https://view.officeapps.live.com/op/view.aspx?src=https%3A%2F%2Fwww.victronenergy.com%2Fupload%2Fdocuments%2FCCGX-Modbus-TCP-register-list-3.60.xlsx
# mqtt-explorer

topics: List[TopicDescriptor] = [
    # generic device attributes
    TopicDescriptor(
        topic = "N/+/+/+/ProductName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="model",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/Serial",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        value_type=ValueType.STRING,
    ),
    # inverter hides its serial number away in a different topic
    TopicDescriptor(
        topic="N/+/vebus/+/Devices/0/SerialNumber",
        message_type=MetricKind.ATTRIBUTE,
        short_id="serial_number",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/Manufacturer",
        message_type=MetricKind.ATTRIBUTE,
        short_id="manufacturer",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/ProductId",
        message_type=MetricKind.ATTRIBUTE,
        short_id="victron_productid",
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/+/+/FirmwareVersion",
        message_type=MetricKind.ATTRIBUTE,
        short_id="firmware_version",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/+/+/CustomName",
        message_type=MetricKind.ATTRIBUTE,
        short_id="custom_name",
        value_type=ValueType.STRING,
    ),
    # grid
    TopicDescriptor(
        topic="N/+/system/+/Ac/Grid/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_phases",  # system attribute
        name="Grid phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT_DEFAULT_0,
    ),
    # individual grid phases
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}",
        name="Grid voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_{phase}",
        name="Grid current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power_{phase}",
        name="Grid power on {phase}",
        metric_type=MetricType.POWER,
    ),
    # VM-3P75CT
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward_{phase}",
        name="Grid consumption on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse_{phase}",
        name="Grid feed-in on {phase}",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/PENVoltage", 
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_pen",
        name="Grid voltage on PEN",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/N/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current_n",
        name="Grid current on N",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Frequency",
        message_type=MetricKind.SENSOR,
        short_id="grid_frequency",
        name="Grid frequency",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/{phase}/VoltageLineToLine",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage_{phase}_{next_phase}",
        name="Grid voltage {phase} to {next_phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    # total grid
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="grid_voltage",
        name="Grid voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Current",
        message_type=MetricKind.SENSOR,
        short_id="grid_current",
        name="Grid current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="grid_power",
        name="Grid power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_forward",
        name="Grid consumption",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/grid/+/Ac/Energy/Reverse",
        message_type=MetricKind.SENSOR,
        short_id="grid_energy_reverse",
        name="Grid feed-in",
        metric_type=MetricType.ENERGY,
    ),
    # solar / MPPT
    TopicDescriptor(
        topic="N/+/solarcharger/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_dc_voltage",
        name="DC (Batt) Bus voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_dc_current",
        name="DC (Batt) Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Load/I",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_load_current",
        name="Load Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Pv/V",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_voltage",
        name="PV Bus Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/State",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_state",
        name="Solar Charger State",
        value_type=ValueType.ENUM,
        enum=SolarChargerState,
    ),

    #System
    TopicDescriptor(
        topic="N/+/system/+/Dc/Pv/Power",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_power",
        name="PV Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/PV/Current",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_current",
        name="PV Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Yield/Power",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_power",
        name="PV Yield Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Yield/User",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_total",
        name="Total Yield",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/History/Daily/0/Yield",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_today",
        name="Yield Today",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/History/Daily/1/Yield",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_yield_yesterday",
        name="Yield Yesterday",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/History/Daily/0/MaxPower",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_max_power_today",
        name="Max Power Today",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/History/Daily/1/MaxPower",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_max_power_yesterday",
        name="Max Power Yesterday",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Load/State",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="solarcharger_load_state",
        name="Solar Charger Load State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Mode",
        message_type=MetricKind.SWITCH,
        short_id="solarcharger_mode",
        name="Solar Charger Mode",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/MppOperationMode",
        message_type=MetricKind.SENSOR,
        short_id="solarcharger_mppt_operation_mode",
        name="Solar Charger MPPT Operation Mode",
        value_type=ValueType.ENUM,
        enum=MultiState,
    ),
    TopicDescriptor(
        topic="N/+/solarcharger/+/Relay/0/State",
        message_type=MetricKind.SWITCH,
        short_id="solarcharger_relay_state",
        name="Solar Charger Relay State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    # batteries
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_voltage",
        name="DC Bus voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="battery_current",
        name="DC Bus current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="battery_power",
        name="Battery power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Dc/0/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="battery_temperature",
        name="Battery temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/History/DischargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_discharged_energy",
        name="Discharged energy",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/History/ChargedEnergy",
        message_type=MetricKind.SENSOR,
        short_id="battery_charged_energy",
        name="Charged energy",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/History/AutomaticSyncs",
        message_type=MetricKind.SENSOR,
        short_id="battery_automatic_syncs",
        name="Automatic syncs",
        unit_of_measurement="count",
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.CUMULATIVE,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Capacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_capacity",
        name="Battery capacity",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/InstalledCapacity",
        message_type=MetricKind.SENSOR,
        short_id="battery_installed_capacity",
        name="Battery installed capacity",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Soc",
        message_type=MetricKind.SENSOR,
        short_id="battery_soc",
        name="Battery charge",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/System/MinCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_min_cell_voltage",
        name="Battery minimum cell voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/System/MaxCellVoltage",
        message_type=MetricKind.SENSOR,
        short_id="battery_max_cell_voltage",
        name="Battery maximum cell voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/TimeToGo",
        message_type=MetricKind.SENSOR,
        short_id="battery_time_to_go",
        name="Battery time to go",
        unit_of_measurement="s",
        metric_type=MetricType.TIME,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Info/ChargeMode",
        message_type=MetricKind.SENSOR,
        short_id="charge_mode",
        name="Battery charge mode",
        value_type=ValueType.STRING,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/ConsumedAmphours",
        message_type=MetricKind.SENSOR,
        short_id="battery_consumed_amphours",
        name="Battery consumed amp-hours",
        metric_type=MetricType.ELECTRIC_STORAGE_CAPACITY,
    ),
    TopicDescriptor(
        topic="N/+/battery/+/Voltages/Cell{cell_id(1-4)}",
        message_type=MetricKind.SENSOR,
        short_id="battery_cell_{cell_id}_voltage",
        name="Battery cell {cell_id} voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    # inverter
    TopicDescriptor(
        topic="N/+/vebus/+/Mode",
        message_type=MetricKind.SELECT,
        short_id="inverter_mode",
        name="Inverter mode",
        value_type=ValueType.ENUM,
        enum=InverterMode,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/State",
        message_type=MetricKind.SENSOR,
        short_id="inverter_state",
        name="Inverter state",
        value_type=ValueType.ENUM,
        enum=InverterState,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_voltage_{phase}",
        name="Inverter input voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_power_{phase}",
        name="Inverter input power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_frequency_{phase}",
        name="Inverter input frequency {phase}",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="inverter_input_apparent_power_{phase}",
        name="Inverter input apparent power {phase}",
        metric_type=MetricType.APPARENT_POWER,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_power_{phase}",
        name="Inverter output power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/F",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_frequency_{phase}",
        name="Inverter output frequency {phase}",
        metric_type=MetricType.FREQUENCY,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/Out/{phase}/S",
        message_type=MetricKind.SENSOR,
        short_id="inverter_output_apparent_power_{phase}",
        name="Inverter output apparent power {phase}",
        metric_type=MetricType.APPARENT_POWER,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="inverter_connected",
        name="Inverter connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Alarms/Overload",
        message_type=MetricKind.SENSOR,
        short_id="inverter_alarm_overload",
        name="Inverter overload alarm",
        value_type=ValueType.ENUM,
        enum=InverterOverloadAlarmEnum,
    ),
    TopicDescriptor(
        topic="N/+/vebus/+/Ac/ActiveIn/CurrentLimit",
        message_type=MetricKind.NUMBER,
        short_id="inverter_current_limit",
        name="Inverter current limit",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        min=0,
        max=16,
        is_adjustable_suffix = "CurrentLimitIsAdjustable"
    ),
    # integrated system. Note that this might not be currently accurate for all systems
    #  as there are different physical configurations
    # and don't have access to any other for testing or feedback.
    TopicDescriptor(
        topic="N/+/system/+/Ac/ConsumptionOnOutput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_critical_loads_{phase}",
        name="Critical loads on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/ConsumptionOnInput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_ac_loads_{phase}",
        name="AC loads on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/System/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_consumption",
        name="DC Consumption",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/Battery/Soc",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_soc",
        name="DC Battery Charge",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/Battery/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_power",
        name="DC Battery Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/Battery/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_voltage",
        name="DC Battery Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Dc/Battery/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_dc_battery_current",
        name="DC Battery Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Relay/{relay}/State",
        message_type=MetricKind.SWITCH,
        short_id="system_relay_{relay}",
        name="Relay {relay} state",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/Available",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_available",
        name="Dynamic ESS Available",
        value_type=ValueType.ENUM,
        enum=GenericOnOff
    ),
    # there is at least one mode for DESS that is controlled by Node-Red (value=4)
    # it may show up here - as well as: GREEN Mode (1?)an Trade Mode (2?)
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/Active",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_active",
        name="Dynamic ESS Active",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/AllowGridFeedIn",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="system_dynamicess_allow_gridfeedin",
        name="Dynamic ESS Allow Grid Feed In",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/AvailableOverhead",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_available_overhead",
        name="Dynamic ESS Available Overhead",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/ErrorCode",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_error",
        name="Dynamic ESS Error",
        value_type=ValueType.ENUM,
        enum=DESSErrorCode,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/NumberOfSchedules",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_schedule_count",
        name="Dynamic ESS Number Of Schedules",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/TargetSoc",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_target_soc",
        name="Dynamic ESS Target SOC",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        precision=0,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/MinimumSoc",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_minimum_soc",
        name="Dynamic ESS Minimum SOC",
        unit_of_measurement="%",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        precision=0,
    ),
    # N/+/system/+/DynamicEss/Capabilities       - not documented - in my case always 31 (bitmask?)
    # N/+/system/+/DynamicEss/ChargeRate         - not documented - in my case alway 0
    # N/+/system/+/DynamicEss/Flags              - not documented - in my case always 0
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/LastScheduledStart",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_last_scheduled_start",
        name="Dynamic ESS Last Scheduled Start",
        value_type=ValueType.EPOCH,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/LastScheduledEnd",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_last_scheduled_end",
        name="Dynamic ESS Last Scheduled End",
        value_type=ValueType.EPOCH,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/Restrictions",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_restrictions",
        name="Dynamic ESS Restrictions",
        value_type=ValueType.ENUM,
        enum=DESSRestrictions,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/Strategy",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_strategy",
        name="Dynamic ESS Strategy",
        value_type=ValueType.ENUM,
        enum=DESSStrategy,
    ),
    TopicDescriptor(
        topic="N/+/system/+/DynamicEss/ReactiveStrategy",
        message_type=MetricKind.SENSOR,
        short_id="system_dynamicess_reactive_strategy",
        name="Dynamic ESS Reactive Strategy",
        value_type=ValueType.ENUM,
        enum=DESSReactiveStrategy,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/Genset/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_generator_load_{phase}",
        name="Genset Load {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/Grid/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_grid_power_{phase}",
        name="Grid Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/PvOnOutput/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_phases",
        name="PV on Output phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/PvOnOutput/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_power_{phase}",
        name="PV on Output Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/PvOnOutput/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="system_pv_on_output_current_{phase}",
        name="PV on Output Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/system/+/Ac/ConsumptionOnOutput/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="system_consumption_on_output_phases",
        name="Consumption On Output phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    # evcharger
    TopicDescriptor(
        topic="N/+/evcharger/+/Mode",
        message_type=MetricKind.SELECT,
        short_id="evcharger_mode",
        name="EV charger mode",
        value_type=ValueType.ENUM,
        enum=EvChargerMode,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/StartStop",
        message_type=MetricKind.SWITCH,
        short_id="evcharger_charge",
        name="EV charger charge",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="evcharger_connected",
        name="EV charger connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Current",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_current",
        name="EV charger current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="evcharger_power_{phase}",
        name="EV charger power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/evcharger/+/SetCurrent",
        message_type=MetricKind.NUMBER,
        short_id="evcharger_set_current",
        name="EV charger set current",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min=0,
        max=16,
    ),
    #pvinverter
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_voltage_{phase}",
        name="Voltage {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Current",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_current_{phase}",
        name="Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/Power",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_power_total",
        name="Power Total",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Power",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_power_{phase}",
        name="Power {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_yield_total",
        name="Total Yield",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/pvinverter/+/Ac/{phase}/Energy/Forward",
        message_type=MetricKind.SENSOR,
        short_id="pvinverter_yield_{phase}",
        name="Yield {phase}",
        metric_type=MetricType.ENERGY,
    ),
    # temperature devices
    TopicDescriptor(
        topic="N/+/temperature/+/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="temperature_temperature",
        name="Temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Status",
        message_type=MetricKind.SENSOR,
        short_id="temperature_status",
        name="Temperature sensor status",
        value_type=ValueType.ENUM,
        enum=TemperatureStatus,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/TemperatureType",
        message_type=MetricKind.SENSOR,
        short_id="temperature_type",
        name="Temperature sensor type",
        value_type=ValueType.ENUM,
        enum=TemperatureType,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Offset",
        message_type=MetricKind.NUMBER,
        short_id="temperature_offset",
        name="Temperature offset",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/+/temperature/+/Scale",
        message_type=MetricKind.NUMBER,
        short_id="temperature_scale",
        name="Temperature scale factor",
        unit_of_measurement=None,
        metric_type=MetricType.NONE,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    #Tanks
    TopicDescriptor(
        topic="N/+/tank/+/Level",
        message_type=MetricKind.SENSOR,
        short_id="tank_level",
        name="Level",
        unit_of_measurement="%",
        metric_type=MetricType.PERCENTAGE,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/tank/+/Remaining",
        message_type=MetricKind.SENSOR,
        short_id="tank_remaining",
        name="Remaining",
        unit_of_measurement="m³",
        metric_type=MetricType.LIQUID_VOLUME,
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=2,
    ),
    TopicDescriptor(
        topic="N/+/tank/+/Temperature",
        message_type=MetricKind.SENSOR,
        short_id="tank_temperature",
        name="Temperature",
        metric_type=MetricType.TEMPERATURE,
    ),
    TopicDescriptor(
        topic="N/+/tank/+/FluidType",
        message_type=MetricKind.SENSOR,
        short_id="tank_fluid_type",
        name="Fluid Type",
        value_type=ValueType.ENUM,
        enum=FluidType,
    ),
    #multirssolar
    #acin and acout
    TopicDescriptor(
        topic="N/+/multi/+/Ac/NumberOfPhases",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_phases",
        name="Phases",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Ac/In/1/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_voltage_{phase}",
        name="Voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Ac/In/1/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_current_{phase}",
        name="Current {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Ac/In/1/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acin_power_{phase}",
        name="Power on {phase}",
        metric_type=MetricType.POWER,
    ),
    #acout {output}
     TopicDescriptor(
        topic="N/+/multi/+/Ac/Out/{output}/{phase}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_voltage_{phase}",
        name="AC Out {output} Voltage on {phase}",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Ac/Out/{output}/{phase}/I",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_current_{phase}",
        name="AC Out {output} Current on {phase}",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Ac/Out/{output}/{phase}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_acout_{output}_power_{phase}",
        name="AC Out {output} Power on {phase}",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/State",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_state",
        name="State",
        value_type=ValueType.ENUM,
        enum=InverterState,
    ),
    #multirssolar mppt {mpptnumber} pv topics
        TopicDescriptor(
        topic="N/+/multi/+/Pv/{mpptnumber}/V",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_voltage",
        name="MPPT {mpptnumber} PV Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Pv/{mpptnumber}/P",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_power",
        name="MPPT {mpptnumber} Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Pv/{mpptnumber}/MppOperationMode",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_mppt_{mpptnumber}_state",
        name="MPPT {mpptnumber} state",
        value_type=ValueType.ENUM,
        enum=MultiState,
    ),     
    TopicDescriptor(
        topic="N/+/multi/+/Yield/Power",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_pv_power_total",
        name="PV Power Total",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Yield/User",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_total_pv_yield",
        name="Total PV Yield",
        metric_type=MetricType.ENERGY,
    ),
    TopicDescriptor(
        topic="N/+/multi/+/Settings/Ess/Mode",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_ess_mode",
        name="ESS Mode",
        value_type=ValueType.ENUM,
        enum=ESSMode,
    ),             
    #ESS Information
    TopicDescriptor(
        topic="N/+/multi/+/Ess/AcPowerSetpoint",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_ess_ac_power_set_point",
        name="ESS AC Power Setpoint",
        metric_type=MetricType.POWER,
    ),        
    TopicDescriptor(
        topic="N/+/multi/+/Ess/InverterPowerSetpoint",
        message_type=MetricKind.SENSOR,
        short_id="multirssolar_inverter_power_setpoint",
        name="Inverter Power Setpoint",
        metric_type=MetricType.POWER,
    ),
    # Orion Alternator (Charger) metrics
    TopicDescriptor(
        topic="N/+/alternator/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_voltage",
        name="DC Output Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/alternator/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_current",
        name="DC Output Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/alternator/+/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="alternator_dc_power",
        name="DC Output Power",
        metric_type=MetricType.POWER,
    ),
    TopicDescriptor(
        topic="N/+/alternator/+/In/V",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_voltage",
        name="Input Voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/alternator/+/In/I",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_current",
        name="Input Current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/alternator/+/In/P",
        message_type=MetricKind.SENSOR,
        short_id="alternator_input_power",
        name="Input Power",
        metric_type=MetricType.POWER,
    ),
    # Settings
    TopicDescriptor(
        topic="N/+/settings/+/Settings/CGwacs/AcPowerSetPoint",
        message_type=MetricKind.NUMBER,
        short_id="system_ac_power_set_point",
        name="AC Power Setpoint",
        metric_type=MetricType.POWER,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
        min=-10000,  # Dynamic range, depends on device model,
        max=10000,  # Dynamic range, depends on device model
    ),
    TopicDescriptor(
        topic="N/+/settings/+/Settings/CGwacs/OvervoltageFeedIn",
        message_type=MetricKind.SWITCH,
        short_id="system_settings_overvoltage_feedin",
        name="PV DC Overvoltage FeedIn",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/settings/+/Settings/CGwacs/BatteryLife/MinimumSocLimit",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_min_soc_limit",
        name="ESS min SOC limit",
        metric_type=MetricType.ELECTRIC_STORAGE_PERCENTAGE,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    TopicDescriptor(
        topic="N/+/settings/+/Settings/SystemSetup/MaxChargeCurrent",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_charge_current",
        name="ESS max charge current",
        metric_type=MetricType.CURRENT,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    TopicDescriptor(
        topic="N/+/settings/+/Settings/SystemSetup/MaxChargeVoltage",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_charge_voltage",
        name="ESS max charge voltage",
        metric_type=MetricType.VOLTAGE,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
    # DC Load
    TopicDescriptor(
        topic="N/+/dcload/+/Dc/0/Voltage",
        message_type=MetricKind.SENSOR,
        short_id="dcload_voltage",
        name="DC load voltage",
        metric_type=MetricType.VOLTAGE,
    ),
    TopicDescriptor(
        topic="N/+/dcload/+/Dc/0/Current",
        message_type=MetricKind.SENSOR,
        short_id="dcload_current",
        name="DC load current",
        metric_type=MetricType.CURRENT,
    ),
    TopicDescriptor(
        topic="N/+/dcload/+/Dc/0/Power",
        message_type=MetricKind.SENSOR,
        short_id="dcload_power",
        name="DC load power",
        metric_type=MetricType.POWER,
    ),
    # Switches
    TopicDescriptor(
        topic="N/+/switch/+/SwitchableOutput/output_{output(1-4)}/State",
        message_type=MetricKind.SWITCH,
        short_id="switch_{output}_state",
        name="Switch {output} State",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/switch/+/SwitchableOutput/output_{output(1-4)}/Dimming",
        message_type=MetricKind.NUMBER,
        short_id="switch_{output}_dimming",
        name="Switch {output} Dimming",
        value_type=ValueType.INT,
        min=0,
        max=100,
    ),
    # GPS
    TopicDescriptor(
        topic="N/+/gps/+/Altitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_altitude",
        name="GPS Altitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.FLOAT,
        precision=None # Use full precision of GPS device
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Connected",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="gps_connected",
        name="GPS Connected",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Course",
        message_type=MetricKind.SENSOR,
        short_id="gps_course",
        name="GPS Course",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.HEADING,
        value_type=ValueType.FLOAT,
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Fix",
        message_type=MetricKind.BINARY_SENSOR,
        short_id="gps_fix",
        name="GPS Fix",
        value_type=ValueType.ENUM,
        enum=GenericOnOff,
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Speed",
        message_type=MetricKind.SENSOR,
        short_id="gps_speed",
        name="GPS Speed",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.SPEED,
        value_type=ValueType.FLOAT,
    ),
    TopicDescriptor(
        topic="N/+/gps/+/NrOfSatellites",
        message_type=MetricKind.SENSOR,
        short_id="gps_nrofsatellites",
        name="GPS Nr Of Satellites",
        metric_nature=MetricNature.INSTANTANEOUS,
        value_type=ValueType.INT,
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Position/Latitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_latitude",
        name="GPS Latitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.LOCATION,
        value_type=ValueType.FLOAT,
        precision=None # Use full precision of GPS device
    ),
    TopicDescriptor(
        topic="N/+/gps/+/Position/Longitude",
        message_type=MetricKind.SENSOR,
        short_id="gps_longitude",
        name="GPS Longitude",
        metric_nature=MetricNature.INSTANTANEOUS,
        metric_type=MetricType.LOCATION,
        value_type=ValueType.FLOAT,
        precision=None # Use full precision of GPS device
    ),
    TopicDescriptor(
        topic="N/+/settings/+/Settings/CGwacs/MaxDischargePower",
        message_type=MetricKind.NUMBER,
        short_id="system_ess_max_inverter_power_limit",
        name="ESS max inverter power limit",
        metric_type=MetricType.POWER,
        value_type=ValueType.INT,
        min_max_range=RangeType.DYNAMIC,
    ),
]
