# adaone-utils

A Python utility package for working with AdaOne toolpaths. This package provides a high-level interface to read, modify, and write toolpaths generated by [AdaOne](https://adaone.adaxis.eu/), the software platform by [Adaxis](https://adaxis.eu/).

## Description

`adaone-utils` provides a Python interface using Polars DataFrames to read and manipulate *.ada3dp files. It leverages Rust for performance and uses the `pyo3` and `polars` libraries to provide a fast and memory-efficient implementation.

Key features:
- Read *.ada3dp files into Polars DataFrames
- Write modified toolpaths back to *.ada3dp format
- Full access to all toolpath data including:
  - Position, direction, and orientation
  - Layer information
  - Process parameters
  - External axes positions
  - Fan settings
  - User events

## Installation

```sh
pip install adaone-utils
```

## Usage

### Reading a *.ada3dp file

```python
from adaone_utils import Toolpath
import plotly.express as px

# Load the toolpath file
toolpath = Toolpath.from_file("path/to/your/file.ada3dp")

# The DataFrame contains columns for all toolpath properties
print(toolpath.data.columns)

# Example: Plot the toolpath
fig = px.line_3d(
    toolpath.data,
    x="position.x",
    y="position.y",
    z="position.z",
    color="segment_type",
    line_group="segmentID"
)
fig.show()
```

### Writing a *.ada3dp file

```python
from adaone_utils import Toolpath, Parameters, PathPlanningStrategy

# Create parameters
params = Parameters(
    layer_height=0.2,
    path_planning_strategy=PathPlanningStrategy.PLANAR_HORIZONTAL,
    posi_axis1_val=0.0,
    posi_axis2_val=0.0,
    posi_axis1_dynamic=False,
    posi_axis2_dynamic=False,
    deposition_width=0.4
)

# Create a new toolpath or modify an existing one
toolpath = Toolpath(data=modified_df, parameters=params)

# Write to file
toolpath.to_file("path/to/output.ada3dp")
```

## Data Format

This package reads and writes the ADA3DP format, which is based on Protocol Buffers. For details about the format, see the [AdaOne documentation](https://adaone.adaxis.eu/docs/functionality/experimental/read-edit-in-python#reading-a-tool-path-file).

The DataFrame structure provides easy access to all toolpath properties, which are organized in three scopes:

### Per Point Properties
Properties that can vary at each point along the toolpath:
- `position.{x,y,z}`: Tool position in mm
- `direction.{x,y,z}`: Tool direction vector (normalized)
- `orientation.{x,y,z,w}`: Tool orientation as quaternion
- `speed`: Feed rate in mm/s
- `deposition`: Material flow rate multiplier (0.0-1.0)
- `externalAxes`: List of external axis positions in degrees or mm
- `fans.{num,speed}`: Fan number and speed settings
- `userEvents.num`: User event IDs triggered at this point

### Per Segment Properties
Properties that apply to an entire segment (continuous toolpath):
- `segmentID`: Unique identifier for each segment (added by the package)
- `segment_type`: Type of path segment (wall, infill, etc.)
- `speedTCP`: Tool center point speed in mm/s
- `processOnDelay`: Delay before starting process in seconds
- `processOffDelay`: Delay after stopping process in seconds
- `startDelay`: Delay before starting movement in seconds
- `processHeadID`: ID of the equipment used
- `toolID`: ID of the tool used
- `materialID`: ID of the material used

### Per Layer Properties
Properties that apply to an entire layer:
- `layerIndex`: Layer number

When working with the DataFrame, all properties are unrolled into individual rows for easier manipulation. However, when writing back to a file, only the first row's value of segment and layer properties is used for each segment or layer.

## Development

For development, install the package in editable mode:

```sh
pip install -e ".[dev]"
```

To run tests:

```sh
pytest
```

## License

This project is licensed under the MIT License.