# Open Traffic Generator API 1.24.0
# License: MIT

import importlib
import logging
import json
import platform
import yaml
import requests
import urllib3
import io
import sys
import time
import grpc
import semantic_version
import types
import platform
from google.protobuf import json_format

try:
    from snappi import otg_pb2_grpc as pb2_grpc
except ImportError:
    import otg_pb2_grpc as pb2_grpc
try:
    from snappi import otg_pb2 as pb2
except ImportError:
    import otg_pb2 as pb2

try:
    from typing import Union, Dict, List, Any, Literal
except ImportError:
    from typing_extensions import Literal


if sys.version_info[0] == 3:
    unicode = str


openapi_warnings = []


class Transport:
    HTTP = "http"
    GRPC = "grpc"


def api(
    location=None,
    transport=None,
    verify=True,
    logger=None,
    loglevel=logging.INFO,
    ext=None,
    version_check=False,
):
    """Create an instance of an Api class

    generator.Generator outputs a base Api class with the following:
    - an abstract method for each OpenAPI path item object
    - a concrete properties for each unique OpenAPI path item parameter.

    generator.Generator also outputs an HttpApi class that inherits the base
    Api class, implements the abstract methods and uses the common HttpTransport
    class send_recv method to communicate with a REST based server.

    Args
    ----
    - location (str): The location of an Open Traffic Generator server.
    - transport (enum["http", "grpc"]): Transport Type
    - verify (bool): Verify the server's TLS certificate, or a string, in which
      case it must be a path to a CA bundle to use. Defaults to `True`.
      When set to `False`, requests will accept any TLS certificate presented by
      the server, and will ignore hostname mismatches and/or expired
      certificates, which will make your application vulnerable to
      man-in-the-middle (MitM) attacks. Setting verify to `False`
      may be useful during local development or testing.
    - logger (logging.Logger): A user defined logging.logger, if none is provided
      then a default logger with a stdout handler will be provided
    - loglevel (logging.loglevel): The logging package log level.
      The default loglevel is logging.INFO
    - ext (str): Name of an extension package
    """
    params = locals()
    transport_types = ["http", "grpc"]
    if ext is None:
        transport = "http" if transport is None else transport
        if transport not in transport_types:
            raise Exception(
                "{transport} is not within valid transport types {transport_types}".format(
                    transport=transport, transport_types=transport_types
                )
            )
        if transport == "http":
            return HttpApi(**params)
        else:
            return GrpcApi(**params)
    try:
        if transport is not None:
            raise Exception(
                "ext and transport are not mutually exclusive. Please configure one of them."
            )
        lib = importlib.import_module("snappi_{}.snappi_api".format(ext))
        return lib.Api(**params)
    except ImportError as err:
        msg = "Extension %s is not installed or invalid: %s"
        raise Exception(msg % (ext, err))


class HttpTransport(object):
    def __init__(self, **kwargs):
        """Use args from api() method to instantiate an HTTP transport"""
        self.location = (
            kwargs["location"]
            if "location" in kwargs and kwargs["location"] is not None
            else "https://localhost:443"
        )
        self.verify = kwargs["verify"] if "verify" in kwargs else False
        self.logger = kwargs["logger"] if "logger" in kwargs else None
        self.loglevel = kwargs["loglevel"] if "loglevel" in kwargs else logging.DEBUG
        if self.logger is None:
            stdout_handler = logging.StreamHandler(sys.stdout)
            formatter = logging.Formatter(
                fmt="%(asctime)s [%(name)s] [%(levelname)s] %(message)s",
                datefmt="%Y-%m-%d %H:%M:%S",
            )
            formatter.converter = time.gmtime
            stdout_handler.setFormatter(formatter)
            self.logger = logging.Logger(self.__module__, level=self.loglevel)
            self.logger.addHandler(stdout_handler)
        self.logger.debug(
            "HttpTransport args: {}".format(
                ", ".join(["{}={!r}".format(k, v) for k, v in kwargs.items()])
            )
        )
        self.set_verify(self.verify)
        self._session = requests.Session()

    def set_verify(self, verify):
        self.verify = verify
        if self.verify is False:
            urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
            self.logger.warning("Certificate verification is disabled")

    def _parse_response_error(self, response_code, response_text):
        error_response = ""
        try:
            error_response = yaml.safe_load(response_text)
        except Exception as _:
            error_response = response_text

        err_obj = Error()
        try:
            err_obj.deserialize(error_response)
        except Exception as _:
            err_obj.code = response_code
            err_obj.errors = [str(error_response)]

        raise Exception(err_obj)

    def send_recv(
        self,
        method,
        relative_url,
        payload=None,
        return_object=None,
        headers=None,
        request_class=None,
    ):
        url = "%s%s" % (self.location, relative_url)
        data = None
        headers = headers or {"Content-Type": "application/json"}
        if payload is not None:
            if isinstance(payload, bytes):
                data = payload
                headers["Content-Type"] = "application/octet-stream"
            elif isinstance(payload, (str, unicode)):
                if request_class is not None:
                    request_class().deserialize(payload)
                data = payload
            elif isinstance(payload, OpenApiBase):
                data = payload.serialize()
            else:
                raise Exception("Type of payload provided is unknown")
        response = self._session.request(
            method=method,
            url=url,
            data=data,
            verify=False,
            allow_redirects=True,
            # TODO: add a timeout here
            headers=headers,
        )
        if response.ok:
            if "application/json" in response.headers["content-type"]:
                # TODO: we might want to check for utf-8 charset and decode
                # accordingly, but current impl works for now
                response_dict = yaml.safe_load(response.text)
                if return_object is None:
                    # if response type is not provided, return dictionary
                    # instead of python object
                    return response_dict
                else:
                    return return_object.deserialize(response_dict)
            elif "application/octet-stream" in response.headers["content-type"]:
                return io.BytesIO(response.content)
            else:
                # TODO: for now, return bare response object for unknown
                # content types
                return response
        else:
            self._parse_response_error(response.status_code, response.text)


class OpenApiStatus:
    messages = {}
    # logger = logging.getLogger(__module__)

    @classmethod
    def warn(cls, key, object):
        if cls.messages.get(key) is not None:
            if cls.messages[key] in object.__warnings__:
                return
            # cls.logger.warning(cls.messages[key])
            logging.warning(cls.messages[key])
            object.__warnings__.append(cls.messages[key])
            # openapi_warnings.append(cls.messages[key])

    @staticmethod
    def deprecated(func_or_data):
        def inner(self, *args, **kwargs):
            OpenApiStatus.warn(
                "{}.{}".format(type(self).__name__, func_or_data.__name__),
                self,
            )
            return func_or_data(self, *args, **kwargs)

        if isinstance(func_or_data, types.FunctionType):
            return inner
        OpenApiStatus.warn(func_or_data)

    @staticmethod
    def under_review(func_or_data):
        def inner(self, *args, **kwargs):
            OpenApiStatus.warn(
                "{}.{}".format(type(self).__name__, func_or_data.__name__),
                self,
            )
            return func_or_data(self, *args, **kwargs)

        if isinstance(func_or_data, types.FunctionType):
            return inner
        OpenApiStatus.warn(func_or_data)


class OpenApiBase(object):
    """Base class for all generated classes"""

    JSON = "json"
    YAML = "yaml"
    DICT = "dict"

    __slots__ = ()

    __constraints__ = {"global": []}
    __validate_latter__ = {"unique": [], "constraint": []}

    def __init__(self):
        pass

    def serialize(self, encoding=JSON):
        """Serialize the current object according to a specified encoding.

        Args
        ----
        - encoding (str[json, yaml, dict]): The object will be recursively
            serialized according to the specified encoding.
            The supported encodings are json, yaml and python dict.

        Returns
        -------
        - obj(Union[str, dict]): A str or dict object depending on the specified
            encoding. The json and yaml encodings will return a str object and
            the dict encoding will return a python dict object.
        """
        # TODO: restore behavior
        # self._clear_globals()
        if encoding == OpenApiBase.JSON:
            data = json.dumps(self._encode(), indent=2, sort_keys=True)
        elif encoding == OpenApiBase.YAML:
            data = yaml.safe_dump(self._encode())
        elif encoding == OpenApiBase.DICT:
            data = self._encode()
        else:
            raise NotImplementedError("Encoding %s not supported" % encoding)
        # TODO: restore behavior
        # self._validate_coded()
        return data

    def _encode(self):
        raise NotImplementedError()

    def deserialize(self, serialized_object):
        """Deserialize a python object into the current object.

        If the input `serialized_object` does not match the current
        openapi object an exception will be raised.

        Args
        ----
        - serialized_object (Union[str, dict]): The object to deserialize.
            If the serialized_object is of type str then the internal encoding
            of the serialized_object must be json or yaml.

        Returns
        -------
        - obj(OpenApiObject): This object with all the
            serialized_object deserialized within.
        """
        # TODO: restore behavior
        # self._clear_globals()
        if isinstance(serialized_object, (str, unicode)):
            serialized_object = yaml.safe_load(serialized_object)
        self._decode(serialized_object)
        # TODO: restore behavior
        # self._validate_coded()
        return self

    def _decode(self, dict_object):
        raise NotImplementedError()

    def warnings(self):
        warns = list(self.__warnings__)
        if "2.7" in platform.python_version().rsplit(".", 1)[0]:
            del self.__warnings__[:]
        else:
            self.__warnings__.clear()
        return warns


class OpenApiValidator(object):

    __slots__ = ()

    _validation_errors = []

    def __init__(self):
        pass

    def _clear_errors(self):
        if "2.7" in platform.python_version().rsplit(".", 1)[0]:
            del self._validation_errors[:]
        else:
            self._validation_errors.clear()

    def validate_mac(self, mac):
        if mac is None or not isinstance(mac, (str, unicode)) or mac.count(" ") != 0:
            return False
        try:
            if len(mac) != 17:
                return False
            return all([0 <= int(oct, 16) <= 255 for oct in mac.split(":")])
        except Exception:
            return False

    def validate_ipv4(self, ip):
        if ip is None or not isinstance(ip, (str, unicode)) or ip.count(" ") != 0:
            return False
        if len(ip.split(".")) != 4:
            return False
        try:
            return all([0 <= int(oct) <= 255 for oct in ip.split(".", 3)])
        except Exception:
            return False

    def validate_ipv6(self, ip):
        if ip is None or not isinstance(ip, (str, unicode)):
            return False
        ip = ip.strip()
        if (
            ip.count(" ") > 0
            or ip.count(":") > 7
            or ip.count("::") > 1
            or ip.count(":::") > 0
        ):
            return False
        if (ip[0] == ":" and ip[:2] != "::") or (ip[-1] == ":" and ip[-2:] != "::"):
            return False
        if ip.count("::") == 0 and ip.count(":") != 7:
            return False
        if ip == "::":
            return True
        if ip[:2] == "::":
            ip = ip.replace("::", "0:")
        elif ip[-2:] == "::":
            ip = ip.replace("::", ":0")
        else:
            ip = ip.replace("::", ":0:")
        try:
            return all(
                [
                    True
                    if (0 <= int(oct, 16) <= 65535) and (1 <= len(oct) <= 4)
                    else False
                    for oct in ip.split(":")
                ]
            )
        except Exception:
            return False

    def validate_hex(self, hex):
        if hex is None or not isinstance(hex, (str, unicode)):
            return False
        try:
            int(hex, 16)
            return True
        except Exception:
            return False

    def validate_integer(self, value, min, max, type_format=None):
        if value is None or not isinstance(value, int):
            return False
        if min is not None and value < min:
            return False
        if max is not None and value > max:
            return False
        if type_format is not None:
            if type_format == "uint32" and (value < 0 or value > 4294967295):
                return False
            elif type_format == "uint64" and (
                value < 0 or value > 18446744073709551615
            ):
                return False
            elif type_format == "int32" and (value < -2147483648 or value > 2147483647):
                return False
            elif type_format == "int64" and (
                value < -9223372036854775808 or value > 9223372036854775807
            ):
                return False
        return True

    def validate_float(self, value):
        return isinstance(value, (int, float))

    def validate_string(self, value, min_length, max_length):
        if value is None or not isinstance(value, (str, unicode)):
            return False
        if min_length is not None and len(value) < min_length:
            return False
        if max_length is not None and len(value) > max_length:
            return False
        return True

    def validate_bool(self, value):
        return isinstance(value, bool)

    def validate_list(self, value, itemtype, min, max, min_length, max_length):
        if value is None or not isinstance(value, list):
            return False
        v_obj = getattr(self, "validate_{}".format(itemtype), None)
        if v_obj is None:
            raise AttributeError("{} is not a valid attribute".format(itemtype))
        v_obj_lst = []
        for item in value:
            if itemtype == "integer":
                v_obj_lst.append(v_obj(item, min, max))
            elif itemtype == "string":
                v_obj_lst.append(v_obj(item, min_length, max_length))
            else:
                v_obj_lst.append(v_obj(item))
        return v_obj_lst

    def validate_binary(self, value):
        if value is None or not isinstance(value, (str, unicode)):
            return False
        return all([True if int(bin) == 0 or int(bin) == 1 else False for bin in value])

    def validate_oid(self, value):
        segments = value.split(".")
        if len(segments) < 2:
            return False
        for segment in segments:
            if not segment.isnumeric():
                return False
            if not (0 <= int(segment) <= 4294967295):
                return False
        return True

    def types_validation(
        self,
        value,
        type_,
        err_msg,
        itemtype=None,
        min=None,
        max=None,
        min_length=None,
        max_length=None,
    ):
        type_map = {
            int: "integer",
            str: "string",
            float: "float",
            bool: "bool",
            list: "list",
            "int64": "integer",
            "int32": "integer",
            "uint64": "integer",
            "uint32": "integer",
            "double": "float",
        }
        type_format = type_
        if type_ in type_map:
            type_ = type_map[type_]
        if itemtype is not None and itemtype in type_map:
            itemtype = type_map[itemtype]
        v_obj = getattr(self, "validate_{}".format(type_), None)
        if v_obj is None:
            msg = "{} is not a valid or unsupported format".format(type_)
            raise TypeError(msg)
        if type_ == "list":
            verdict = v_obj(value, itemtype, min, max, min_length, max_length)
            if all(verdict) is True:
                return
            err_msg = "{} \n {} are not valid".format(
                err_msg,
                [value[index] for index, item in enumerate(verdict) if item is False],
            )
            verdict = False
        elif type_ == "integer":
            verdict = v_obj(value, min, max, type_format)
            if verdict is True:
                return
            min_max = ""
            if min is not None:
                min_max = ", expected min {}".format(min)
            if max is not None:
                min_max = min_max + ", expected max {}".format(max)
            err_msg = "{} \n got {} of type {} {}".format(
                err_msg, value, type(value), min_max
            )
        elif type_ == "string":
            verdict = v_obj(value, min_length, max_length)
            if verdict is True:
                return
            msg = ""
            if min_length is not None:
                msg = ", expected min {}".format(min_length)
            if max_length is not None:
                msg = msg + ", expected max {}".format(max_length)
            err_msg = "{} \n got {} of type {} {}".format(
                err_msg, value, type(value), msg
            )
        else:
            verdict = v_obj(value)
        if verdict is False:
            raise TypeError(err_msg)

    def _validate_unique_and_name(self, name, value, latter=False):
        if self._TYPES[name].get("unique") is None or value is None:
            return
        if latter is True:
            self.__validate_latter__["unique"].append(
                (self._validate_unique_and_name, name, value)
            )
            return
        class_name = type(self).__name__
        unique_type = self._TYPES[name]["unique"]
        if class_name not in self.__constraints__:
            self.__constraints__[class_name] = dict()
        if unique_type == "global":
            values = self.__constraints__["global"]
        else:
            values = self.__constraints__[class_name]
        if value in values:
            self._validation_errors.append(
                "{} with {} already exists".format(name, value)
            )
            return
        if isinstance(values, list):
            values.append(value)
        self.__constraints__[class_name].update({value: self})

    def _validate_constraint(self, name, value, latter=False):
        cons = self._TYPES[name].get("constraint")
        if cons is None or value is None:
            return
        if latter is True:
            self.__validate_latter__["constraint"].append(
                (self._validate_constraint, name, value)
            )
            return
        found = False
        for c in cons:
            klass, prop = c.split(".")
            names = self.__constraints__.get(klass, {})
            props = [obj._properties.get(prop) for obj in names.values()]
            if value in props:
                found = True
                break
        if found is not True:
            self._validation_errors.append(
                "{} is not a valid type of {}".format(value, "||".join(cons))
            )
            return

    def _validate_coded(self):
        for item in self.__validate_latter__["unique"]:
            item[0](item[1], item[2])
        for item in self.__validate_latter__["constraint"]:
            item[0](item[1], item[2])
        self._clear_vars()
        if len(self._validation_errors) > 0:
            errors = "\n".join(self._validation_errors)
            self._clear_errors()
            raise Exception(errors)

    def _clear_vars(self):
        if platform.python_version_tuple()[0] == "2":
            self.__validate_latter__["unique"] = []
            self.__validate_latter__["constraint"] = []
        else:
            self.__validate_latter__["unique"].clear()
            self.__validate_latter__["constraint"].clear()

    def _clear_globals(self):
        keys = list(self.__constraints__.keys())
        for k in keys:
            if k == "global":
                self.__constraints__["global"] = []
                continue
            del self.__constraints__[k]


class OpenApiObject(OpenApiBase, OpenApiValidator):
    """Base class for any /components/schemas object

    Every OpenApiObject is reuseable within the schema so it can
    exist in multiple locations within the hierarchy.
    That means it can exist in multiple locations as a
    leaf, parent/choice or parent.
    """

    __slots__ = ("__warnings__", "_properties", "_parent", "_choice")
    _DEFAULTS = {}
    _TYPES = {}
    _REQUIRED = []
    _STATUS = {}

    def __init__(self, parent=None, choice=None):
        super(OpenApiObject, self).__init__()
        self._parent = parent
        self._choice = choice
        self._properties = {}
        self.__warnings__ = []

    @property
    def parent(self):
        return self._parent

    def _set_choice(self, name):
        if self._has_choice(name):
            for enum in self._TYPES["choice"]["enum"]:
                if enum in self._properties and name != enum:
                    self._properties.pop(enum)
            self._properties["choice"] = name

    def _has_choice(self, name):
        if (
            "choice" in dir(self)
            and "_TYPES" in dir(self)
            and "choice" in self._TYPES
            and name in self._TYPES["choice"]["enum"]
        ):
            return True
        else:
            return False

    def _get_property(self, name, default_value=None, parent=None, choice=None):
        if name in self._properties and self._properties[name] is not None:
            return self._properties[name]
        if isinstance(default_value, type) is True:
            self._set_choice(name)
            if "_choice" in default_value.__slots__:
                self._properties[name] = default_value(parent=parent, choice=choice)
            else:
                self._properties[name] = default_value(parent=parent)
            if (
                "_DEFAULTS" in dir(self._properties[name])
                and "choice" in self._properties[name]._DEFAULTS
            ):
                choice_str = self._properties[name]._DEFAULTS["choice"]

                if choice_str in self._properties[name]._TYPES:
                    getattr(
                        self._properties[name],
                        self._properties[name]._DEFAULTS["choice"],
                    )
        else:
            if default_value is None and name in self._DEFAULTS:
                self._set_choice(name)
                self._properties[name] = self._DEFAULTS[name]
            else:
                self._properties[name] = default_value
        return self._properties[name]

    def _set_property(self, name, value, choice=None):
        if name == "choice":

            if (
                self.parent is None
                and value is not None
                and value not in self._TYPES["choice"]["enum"]
            ):
                raise Exception(
                    "%s is not a valid choice, valid choices are %s"
                    % (value, ", ".join(self._TYPES["choice"]["enum"]))
                )

            self._set_choice(value)
            if name in self._DEFAULTS and value is None:
                self._properties[name] = self._DEFAULTS[name]
        elif name in self._DEFAULTS and value is None:
            self._set_choice(name)
            self._properties[name] = self._DEFAULTS[name]
        else:
            self._set_choice(name)
            self._properties[name] = value
        # TODO: restore behavior
        # self._validate_unique_and_name(name, value)
        # self._validate_constraint(name, value)
        if self._parent is not None and self._choice is not None and value is not None:
            self._parent._set_property("choice", self._choice)

    def _encode(self):
        """Helper method for serialization"""
        output = {}
        self._raise_status_warnings(self, None)
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)
            # TODO: restore behavior
            # self._validate_unique_and_name(key, value, True)
            # self._validate_constraint(key, value, True)
            if isinstance(value, (OpenApiObject, OpenApiIter)):
                output[key] = value._encode()
                if isinstance(value, OpenApiObject):
                    self._raise_status_warnings(key, value)
            elif value is not None:
                if (
                    self._TYPES.get(key, {}).get("format", "") == "int64"
                    or self._TYPES.get(key, {}).get("format", "") == "uint64"
                ):
                    value = str(value)
                elif (
                    self._TYPES.get(key, {}).get("itemformat", "") == "int64"
                    or self._TYPES.get(key, {}).get("itemformat", "") == "uint64"
                ):
                    value = [str(v) for v in value]
                output[key] = value
                self._raise_status_warnings(key, value)
        return output

    def _decode(self, obj):
        dtypes = [list, str, int, float, bool]
        self._raise_status_warnings(self, None)
        for property_name, property_value in obj.items():
            if property_name in self._TYPES:
                ignore_warnings = False
                if isinstance(property_value, dict):
                    child = self._get_child_class(property_name)
                    if "choice" in child[1]._TYPES and "_parent" in child[1].__slots__:
                        property_value = child[1](self, property_name)._decode(
                            property_value
                        )
                    elif "_parent" in child[1].__slots__:
                        property_value = child[1](self)._decode(property_value)
                    else:
                        property_value = child[1]()._decode(property_value)
                elif (
                    isinstance(property_value, list)
                    and property_name in self._TYPES
                    and self._TYPES[property_name]["type"] not in dtypes
                ):
                    child = self._get_child_class(property_name, True)
                    openapi_list = child[0]()
                    for item in property_value:
                        item = child[1]()._decode(item)
                        openapi_list._items.append(item)
                    property_value = openapi_list
                    ignore_warnings = True
                elif property_name in self._DEFAULTS and property_value is None:
                    if isinstance(self._DEFAULTS[property_name], tuple(dtypes)):
                        property_value = self._DEFAULTS[property_name]
                self._set_choice(property_name)
                # convert int64(will be string on wire) to to int
                if (
                    self._TYPES[property_name].get("format", "") == "int64"
                    or self._TYPES[property_name].get("format", "") == "uint64"
                ):
                    property_value = int(property_value)
                elif (
                    self._TYPES[property_name].get("itemformat", "") == "int64"
                    or self._TYPES[property_name].get("itemformat", "") == "uint64"
                ):
                    property_value = [int(v) for v in property_value]
                self._properties[property_name] = property_value
                # TODO: restore behavior
                # OpenApiStatus.warn(
                #     "{}.{}".format(type(self).__name__, property_name), self
                # )
                if not ignore_warnings:
                    self._raise_status_warnings(property_name, property_value)
            self._validate_types(property_name, property_value)
            # TODO: restore behavior
            # self._validate_unique_and_name(property_name, property_value, True)
            # self._validate_constraint(property_name, property_value, True)
        self._validate_required()
        return self

    def _get_child_class(self, property_name, is_property_list=False):
        list_class = None
        class_name = self._TYPES[property_name]["type"]
        module = globals().get(self.__module__)
        if module is None:
            module = importlib.import_module(self.__module__)
            globals()[self.__module__] = module
        object_class = getattr(module, class_name)
        if is_property_list is True:
            list_class = object_class
            object_class = getattr(module, class_name[0:-4])
        return (list_class, object_class)

    def __str__(self):
        return self.serialize(encoding=self.YAML)

    def __deepcopy__(self, memo):
        """Creates a deep copy of the current object"""
        return self.__class__().deserialize(self.serialize())

    def __copy__(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)

    def __eq__(self, other):
        return self.__str__() == other.__str__()

    def clone(self):
        """Creates a deep copy of the current object"""
        return self.__deepcopy__(None)

    def _validate_required(self):
        """Validates the required properties are set
        Use getattr as it will set any defaults prior to validating
        """
        if getattr(self, "_REQUIRED", None) is None:
            return
        for name in self._REQUIRED:
            if self._properties.get(name) is None:
                msg = (
                    "{} is a mandatory property of {}"
                    " and should not be set to None".format(
                        name,
                        self.__class__,
                    )
                )
                raise ValueError(msg)

    def _validate_types(self, property_name, property_value):
        common_data_types = [list, str, int, float, bool]
        if property_name not in self._TYPES:
            # raise ValueError("Invalid Property {}".format(property_name))
            return
        details = self._TYPES[property_name]
        if (
            property_value is None
            and property_name not in self._DEFAULTS
            and property_name not in self._REQUIRED
        ):
            return
        if "enum" in details and property_value not in details["enum"]:
            raise_error = False
            if isinstance(property_value, list):
                for value in property_value:
                    if value not in details["enum"]:
                        raise_error = True
                        break
            elif property_value not in details["enum"]:
                raise_error = True

            if raise_error is True:
                msg = "property {} shall be one of these" " {} enum, but got {} at {}"
                raise TypeError(
                    msg.format(
                        property_name,
                        details["enum"],
                        property_value,
                        self.__class__,
                    )
                )
        if details["type"] in common_data_types and "format" not in details:
            msg = "property {} shall be of type {} at {}".format(
                property_name, details["type"], self.__class__
            )

            itemtype = (
                details.get("itemformat")
                if "itemformat" in details
                else details.get("itemtype")
            )
            self.types_validation(
                property_value,
                details["type"],
                msg,
                itemtype,
                details.get("minimum"),
                details.get("maximum"),
                details.get("minLength"),
                details.get("maxLength"),
            )

        if details["type"] not in common_data_types:
            class_name = details["type"]
            # TODO Need to revisit importlib
            module = importlib.import_module(self.__module__)
            object_class = getattr(module, class_name)
            if not isinstance(property_value, object_class):
                msg = "property {} shall be of type {}," " but got {} at {}"
                raise TypeError(
                    msg.format(
                        property_name,
                        class_name,
                        type(property_value),
                        self.__class__,
                    )
                )
        if "format" in details:
            msg = "Invalid {} format, expected {} at {}".format(
                property_value, details["format"], self.__class__
            )
            _type = details["type"] if details["type"] is list else details["format"]
            self.types_validation(
                property_value,
                _type,
                msg,
                details["format"],
                details.get("minimum"),
                details.get("maximum"),
                details.get("minLength"),
                details.get("maxLength"),
            )

    def validate(self):
        self._validate_required()
        for key, value in self._properties.items():
            self._validate_types(key, value)
        # TODO: restore behavior
        # self._validate_coded()

    def get(self, name, with_default=False):
        """
        getattr for openapi object
        """
        if self._properties.get(name) is not None:
            return self._properties[name]
        elif with_default:
            # TODO need to find a way to avoid getattr
            choice = self._properties.get("choice") if "choice" in dir(self) else None
            getattr(self, name)
            if "choice" in dir(self):
                if choice is None and "choice" in self._properties:
                    self._properties.pop("choice")
                else:
                    self._properties["choice"] = choice
            return self._properties.pop(name)
        return None

    def _raise_status_warnings(self, property_name, property_value):
        if len(self._STATUS) > 0:

            if isinstance(property_name, OpenApiObject):
                if "self" in self._STATUS and property_value is None:
                    print("[WARNING]: %s" % self._STATUS["self"], file=sys.stderr)

                return

            enum_key = "%s.%s" % (property_name, property_value)
            if property_name in self._STATUS:
                print(
                    "[WARNING]: %s" % self._STATUS[property_name],
                    file=sys.stderr,
                )
            elif enum_key in self._STATUS:
                print("[WARNING]: %s" % self._STATUS[enum_key], file=sys.stderr)


class OpenApiIter(OpenApiBase):
    """Container class for OpenApiObject

    Inheriting classes contain 0..n instances of an OpenAPI components/schemas
    object.
    - config.flows.flow(name="1").flow(name="2").flow(name="3")

    The __getitem__ method allows getting an instance using ordinal.
    - config.flows[0]
    - config.flows[1:]
    - config.flows[0:1]
    - f1, f2, f3 = config.flows

    The __iter__ method allows for iterating across the encapsulated contents
    - for flow in config.flows:
    """

    __slots__ = ("_index", "_items")
    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self):
        super(OpenApiIter, self).__init__()
        self._index = -1
        self._items = []

    def __len__(self):
        return len(self._items)

    def _getitem(self, key):
        found = None
        if isinstance(key, int):
            found = self._items[key]
        elif isinstance(key, slice) is True:
            start, stop, step = key.indices(len(self))
            sliced = self.__class__()
            for i in range(start, stop, step):
                sliced._items.append(self._items[i])
            return sliced
        elif isinstance(key, str):
            for item in self._items:
                if item.name == key:
                    found = item
        if found is None:
            raise IndexError()
        if (
            self._GETITEM_RETURNS_CHOICE_OBJECT is True
            and found._properties.get("choice") is not None
            and found._properties.get(found._properties["choice"]) is not None
        ):
            return found._properties[found._properties["choice"]]
        return found

    def _iter(self):
        self._index = -1
        return self

    def _next(self):
        if self._index + 1 >= len(self._items):
            raise StopIteration
        else:
            self._index += 1
        return self.__getitem__(self._index)

    def __getitem__(self, key):
        raise NotImplementedError("This should be overridden by the generator")

    def _add(self, item):
        self._items.append(item)
        self._index = len(self._items) - 1

    def remove(self, index):
        del self._items[index]
        self._index = len(self._items) - 1

    def append(self, item):
        """Append an item to the end of OpenApiIter
        TBD: type check, raise error on mismatch
        """
        self._instanceOf(item)
        self._add(item)
        return self

    def clear(self):
        del self._items[:]
        self._index = -1

    def set(self, index, item):
        self._instanceOf(item)
        self._items[index] = item
        return self

    def _encode(self):
        return [item._encode() for item in self._items]

    def _decode(self, encoded_list):
        item_class_name = self.__class__.__name__.replace("Iter", "")
        module = importlib.import_module(self.__module__)
        object_class = getattr(module, item_class_name)
        self.clear()
        for item in encoded_list:
            self._add(object_class()._decode(item))

    def __copy__(self):
        raise NotImplementedError(
            "Shallow copy of OpenApiIter objects is not supported"
        )

    def __deepcopy__(self, memo):
        raise NotImplementedError("Deep copy of OpenApiIter objects is not supported")

    def __str__(self):
        return yaml.safe_dump(self._encode())

    def __eq__(self, other):
        return self.__str__() == other.__str__()

    def _instanceOf(self, item):
        raise NotImplementedError("validating an OpenApiIter object is not supported")


class Config(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ports": {"type": "PortIter"},
        "lags": {"type": "LagIter"},
        "layer1": {"type": "Layer1Iter"},
        "captures": {"type": "CaptureIter"},
        "devices": {"type": "DeviceIter"},
        "flows": {"type": "FlowIter"},
        "events": {"type": "Event"},
        "options": {"type": "ConfigOptions"},
        "lldp": {"type": "LldpIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(Config, self).__init__()
        self._parent = parent

    @property
    def ports(self):
        # type: () -> PortIter
        """ports getter

        The ports that will be configured on the traffic generator.

        Returns: PortIter
        """
        return self._get_property("ports", PortIter, self._parent, self._choice)

    @property
    def lags(self):
        # type: () -> LagIter
        """lags getter

        The LAGs that will be configured on the traffic generator.

        Returns: LagIter
        """
        return self._get_property("lags", LagIter, self._parent, self._choice)

    @property
    def layer1(self):
        # type: () -> Layer1Iter
        """layer1 getter

        The layer1 settings that will be configured on the traffic generator.. Since layer1 settings usually vary across variety of test ports, these. most likely won't be portable.

        Returns: Layer1Iter
        """
        return self._get_property("layer1", Layer1Iter, self._parent, self._choice)

    @property
    def captures(self):
        # type: () -> CaptureIter
        """captures getter

        The capture settings that will be configured on the traffic generator.

        Returns: CaptureIter
        """
        return self._get_property("captures", CaptureIter, self._parent, self._choice)

    @property
    def devices(self):
        # type: () -> DeviceIter
        """devices getter

        The emulated devices that will be configured on the traffic generator.. Each device contains configurations for network interfaces and protocols running on top of those interfaces.

        Returns: DeviceIter
        """
        return self._get_property("devices", DeviceIter, self._parent, self._choice)

    @property
    def flows(self):
        # type: () -> FlowIter
        """flows getter

        The flows that will be configured on the traffic generator.

        Returns: FlowIter
        """
        return self._get_property("flows", FlowIter, self._parent, self._choice)

    @property
    def events(self):
        # type: () -> Event
        """events getter

        Under Review: Event configuration is currently under review for pending exploration on use cases.. Under Review: Event configuration is currently under review for pending exploration on use cases.. The optional container for event configuration.. Both cp_events.enable and dp_events.enable must be explicitly set to true to get control_plane_data_plane_convergence_us metric values for convergence metrics.Under Review: Event configuration is currently under review for pending exploration on use cases.. Under Review: Event configuration is currently under review for pending exploration on use cases.. The optional container for event configuration.. Both cp_events.enable and dp_events.enable must be explicitly set to true to get control_plane_data_plane_convergence_us metric values for convergence metrics.Under Review: Event configuration is currently under review for pending exploration on use cases.. Under Review: Event configuration is currently under review for pending exploration on use cases.. The optional container for event configuration.. Both cp_events.enable and dp_events.enable must be explicitly set to true to get control_plane_data_plane_convergence_us metric values for convergence metrics.

        Returns: Event
        """
        return self._get_property("events", Event)

    @property
    def options(self):
        # type: () -> ConfigOptions
        """options getter

        Global configuration options.Global configuration options.Global configuration options.

        Returns: ConfigOptions
        """
        return self._get_property("options", ConfigOptions)

    @property
    def lldp(self):
        # type: () -> LldpIter
        """lldp getter

        LLDP protocol that will be configured on traffic generator.

        Returns: LldpIter
        """
        return self._get_property("lldp", LldpIter, self._parent, self._choice)


class Port(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "location": {"type": str},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, location=None, name=None):
        super(Port, self).__init__()
        self._parent = parent
        self._set_property("location", location)
        self._set_property("name", name)

    def set(self, location=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def location(self):
        # type: () -> str
        """location getter

        The location of test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. physical appliance with multiple ports. physical chassis with multiple cards and ports. local interface. virtual machine, docker container, kubernetes cluster. The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        Returns: str
        """
        return self._get_property("location")

    @location.setter
    def location(self, value):
        """location setter

        The location of test port. It is the endpoint where packets will emit from.. Test port locations can be the following:. physical appliance with multiple ports. physical chassis with multiple cards and ports. local interface. virtual machine, docker container, kubernetes cluster. The test port location format is implementation specific. Use the /results/capabilities API to determine what formats an implementation supports for the location property.. Get the configured location state by using the /results/port API.

        value: str
        """
        self._set_property("location", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class PortIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(PortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Port]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> PortIter
        return self._iter()

    def __next__(self):
        # type: () -> Port
        return self._next()

    def next(self):
        # type: () -> Port
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Port):
            raise Exception("Item is not an instance of Port")

    def port(self, location=None, name=None):
        # type: (str,str) -> PortIter
        """Factory method that creates an instance of the Port class

        An abstract test port.

        Returns: PortIter
        """
        item = Port(parent=self._parent, location=location, name=name)
        self._add(item)
        return self

    def add(self, location=None, name=None):
        # type: (str,str) -> Port
        """Add method that creates and returns an instance of the Port class

        An abstract test port.

        Returns: Port
        """
        item = Port(parent=self._parent, location=location, name=name)
        self._add(item)
        return item


class Lag(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ports": {"type": "LagPortIter"},
        "protocol": {"type": "LagProtocol"},
        "min_links": {
            "type": int,
            "format": "uint32",
            "maximum": 32,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "min_links": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, min_links=1, name=None):
        super(Lag, self).__init__()
        self._parent = parent
        self._set_property("min_links", min_links)
        self._set_property("name", name)

    def set(self, min_links=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ports(self):
        # type: () -> LagPortIter
        """ports getter

        TBD

        Returns: LagPortIter
        """
        return self._get_property("ports", LagPortIter, self._parent, self._choice)

    @property
    def protocol(self):
        # type: () -> LagProtocol
        """protocol getter



        Returns: LagProtocol
        """
        return self._get_property("protocol", LagProtocol)

    @property
    def min_links(self):
        # type: () -> int
        """min_links getter

        Specifies the mininum number of member interfaces that must be active for the aggregate interface to be available.. If the aggregate interface is not available due to min-links criterion not being met, LACPDUs continue to be transmitted and received by the member interfaces if LACP is enabled, but other PDUs are not transmitted or received.

        Returns: int
        """
        return self._get_property("min_links")

    @min_links.setter
    def min_links(self, value):
        """min_links setter

        Specifies the mininum number of member interfaces that must be active for the aggregate interface to be available.. If the aggregate interface is not available due to min-links criterion not being met, LACPDUs continue to be transmitted and received by the member interfaces if LACP is enabled, but other PDUs are not transmitted or received.

        value: int
        """
        self._set_property("min_links", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class LagPort(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_name": {"type": str},
        "lacp": {"type": "LagPortLacp"},
        "ethernet": {"type": "DeviceEthernetBase"},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_name", "ethernet")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, port_name=None):
        super(LagPort, self).__init__()
        self._parent = parent
        self._set_property("port_name", port_name)

    def set(self, port_name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        The name of port object that will be part of the LAG. . x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        Returns: str
        """
        return self._get_property("port_name")

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        The name of port object that will be part of the LAG. . x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property port_name as None")
        self._set_property("port_name", value)

    @property
    def lacp(self):
        # type: () -> LagPortLacp
        """lacp getter

        The container for link aggregation control protocol settings of LAG member (port).The container for link aggregation control protocol settings of LAG member (port).The container for link aggregation control protocol settings of LAG member (port).

        Returns: LagPortLacp
        """
        return self._get_property("lacp", LagPortLacp)

    @property
    def ethernet(self):
        # type: () -> DeviceEthernetBase
        """ethernet getter

        Base Ethernet interface.Base Ethernet interface.Base Ethernet interface.

        Returns: DeviceEthernetBase
        """
        return self._get_property("ethernet", DeviceEthernetBase)


class LagPortLacp(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "actor_port_number": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "actor_port_priority": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "actor_activity": {
            "type": str,
            "enum": [
                "passive",
                "active",
            ],
        },
        "lacpdu_periodic_time_interval": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "lacpdu_timeout": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "actor_port_number": 0,
        "actor_port_priority": 1,
        "actor_activity": "active",
        "lacpdu_periodic_time_interval": 0,
        "lacpdu_timeout": 0,
    }  # type: Dict[str, Union(type)]

    PASSIVE = "passive"  # type: str
    ACTIVE = "active"  # type: str

    FAST = 1  #
    SLOW = 30  #
    AUTO = 0  #

    SHORT = 3  #
    LONG = 90  #
    AUTO = 0  #

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        actor_port_number=0,
        actor_port_priority=1,
        actor_activity="active",
        lacpdu_periodic_time_interval=0,
        lacpdu_timeout=0,
    ):
        super(LagPortLacp, self).__init__()
        self._parent = parent
        self._set_property("actor_port_number", actor_port_number)
        self._set_property("actor_port_priority", actor_port_priority)
        self._set_property("actor_activity", actor_activity)
        self._set_property(
            "lacpdu_periodic_time_interval", lacpdu_periodic_time_interval
        )
        self._set_property("lacpdu_timeout", lacpdu_timeout)

    def set(
        self,
        actor_port_number=None,
        actor_port_priority=None,
        actor_activity=None,
        lacpdu_periodic_time_interval=None,
        lacpdu_timeout=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def actor_port_number(self):
        # type: () -> int
        """actor_port_number getter

        The actor port number

        Returns: int
        """
        return self._get_property("actor_port_number")

    @actor_port_number.setter
    def actor_port_number(self, value):
        """actor_port_number setter

        The actor port number

        value: int
        """
        self._set_property("actor_port_number", value)

    @property
    def actor_port_priority(self):
        # type: () -> int
        """actor_port_priority getter

        The actor port priority

        Returns: int
        """
        return self._get_property("actor_port_priority")

    @actor_port_priority.setter
    def actor_port_priority(self, value):
        """actor_port_priority setter

        The actor port priority

        value: int
        """
        self._set_property("actor_port_priority", value)

    @property
    def actor_activity(self):
        # type: () -> Union[Literal["active"], Literal["passive"]]
        """actor_activity getter

        Sets the value of LACP actor activity as either passive or active.. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active.. Active indicates the port's preference to participate in the protocol regardless of the partner's control value.

        Returns: Union[Literal["active"], Literal["passive"]]
        """
        return self._get_property("actor_activity")

    @actor_activity.setter
    def actor_activity(self, value):
        """actor_activity setter

        Sets the value of LACP actor activity as either passive or active.. Passive indicates the port's preference for not transmitting LACPDUs unless its partner's control is Active.. Active indicates the port's preference to participate in the protocol regardless of the partner's control value.

        value: Union[Literal["active"], Literal["passive"]]
        """
        self._set_property("actor_activity", value)

    @property
    def lacpdu_periodic_time_interval(self):
        # type: () -> int
        """lacpdu_periodic_time_interval getter

        This field defines how frequently LACPDUs are sent to the link partner

        Returns: int
        """
        return self._get_property("lacpdu_periodic_time_interval")

    @lacpdu_periodic_time_interval.setter
    def lacpdu_periodic_time_interval(self, value):
        """lacpdu_periodic_time_interval setter

        This field defines how frequently LACPDUs are sent to the link partner

        value: int
        """
        self._set_property("lacpdu_periodic_time_interval", value)

    @property
    def lacpdu_timeout(self):
        # type: () -> int
        """lacpdu_timeout getter

        This timer is used to detect whether received protocol information has expired

        Returns: int
        """
        return self._get_property("lacpdu_timeout")

    @lacpdu_timeout.setter
    def lacpdu_timeout(self, value):
        """lacpdu_timeout setter

        This timer is used to detect whether received protocol information has expired

        value: int
        """
        self._set_property("lacpdu_timeout", value)


class DeviceEthernetBase(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mac": {
            "type": str,
            "format": "mac",
        },
        "mtu": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "vlans": {"type": "DeviceVlanIter"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, mac=None, mtu=1500, name=None):
        super(DeviceEthernetBase, self).__init__()
        self._parent = parent
        self._set_property("mac", mac)
        self._set_property("mtu", mtu)
        self._set_property("name", name)

    def set(self, mac=None, mtu=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media Access Control address.The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where 'mac' is not mandatory.

        Returns: str
        """
        return self._get_property("mac")

    @mac.setter
    def mac(self, value):
        """mac setter

        Media Access Control address.The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where 'mac' is not mandatory.

        value: str
        """
        self._set_property("mac", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum Transmission Unit.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum Transmission Unit.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property("vlans", DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceVlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "tpid": {
            "type": str,
            "enum": [
                "x8100",
                "x88A8",
                "x9100",
                "x9200",
                "x9300",
            ],
        },
        "priority": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "id": {
            "type": int,
            "format": "uint32",
            "maximum": 4095,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "tpid": "x8100",
        "priority": 0,
        "id": 1,
    }  # type: Dict[str, Union(type)]

    X8100 = "x8100"  # type: str
    X88A8 = "x88A8"  # type: str
    X9100 = "x9100"  # type: str
    X9200 = "x9200"  # type: str
    X9300 = "x9300"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, tpid="x8100", priority=0, id=1, name=None):
        super(DeviceVlan, self).__init__()
        self._parent = parent
        self._set_property("tpid", tpid)
        self._set_property("priority", priority)
        self._set_property("id", id)
        self._set_property("name", name)

    def set(self, tpid=None, priority=None, id=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def tpid(self):
        # type: () -> Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """tpid getter

        Tag protocol identifier

        Returns: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        return self._get_property("tpid")

    @tpid.setter
    def tpid(self, value):
        """tpid setter

        Tag protocol identifier

        value: Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]]
        """
        self._set_property("tpid", value)

    @property
    def priority(self):
        # type: () -> int
        """priority getter

        Priority code point

        Returns: int
        """
        return self._get_property("priority")

    @priority.setter
    def priority(self, value):
        """priority setter

        Priority code point

        value: int
        """
        self._set_property("priority", value)

    @property
    def id(self):
        # type: () -> int
        """id getter

        VLAN identifier

        Returns: int
        """
        return self._get_property("id")

    @id.setter
    def id(self, value):
        """id setter

        VLAN identifier

        value: int
        """
        self._set_property("id", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceVlanIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceVlanIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceVlanIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceVlan
        return self._next()

    def next(self):
        # type: () -> DeviceVlan
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceVlan):
            raise Exception("Item is not an instance of DeviceVlan")

    def vlan(self, tpid="x8100", priority=0, id=1, name=None):
        # type: (Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]],int,int,str) -> DeviceVlanIter
        """Factory method that creates an instance of the DeviceVlan class

        Emulated VLAN protocol.

        Returns: DeviceVlanIter
        """
        item = DeviceVlan(
            parent=self._parent, tpid=tpid, priority=priority, id=id, name=name
        )
        self._add(item)
        return self

    def add(self, tpid="x8100", priority=0, id=1, name=None):
        # type: (Union[Literal["x8100"], Literal["x88A8"], Literal["x9100"], Literal["x9200"], Literal["x9300"]],int,int,str) -> DeviceVlan
        """Add method that creates and returns an instance of the DeviceVlan class

        Emulated VLAN protocol.

        Returns: DeviceVlan
        """
        item = DeviceVlan(
            parent=self._parent, tpid=tpid, priority=priority, id=id, name=name
        )
        self._add(item)
        return item


class LagPortIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagPortIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[LagPort]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagPortIter
        return self._iter()

    def __next__(self):
        # type: () -> LagPort
        return self._next()

    def next(self):
        # type: () -> LagPort
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, LagPort):
            raise Exception("Item is not an instance of LagPort")

    def port(self, port_name=None):
        # type: (str) -> LagPortIter
        """Factory method that creates an instance of the LagPort class

        The container for port's ethernet interface and LAG protocol settings

        Returns: LagPortIter
        """
        item = LagPort(parent=self._parent, port_name=port_name)
        self._add(item)
        return self

    def add(self, port_name=None):
        # type: (str) -> LagPort
        """Add method that creates and returns an instance of the LagPort class

        The container for port's ethernet interface and LAG protocol settings

        Returns: LagPort
        """
        item = LagPort(parent=self._parent, port_name=port_name)
        self._add(item)
        return item


class LagProtocol(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "lacp",
                "static",
            ],
        },
        "lacp": {"type": "LagProtocolLacp"},
        "static": {"type": "LagProtocolStatic"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "lacp",
    }  # type: Dict[str, Union(type)]

    LACP = "lacp"  # type: str
    STATIC = "static"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(LagProtocol, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def lacp(self):
        # type: () -> LagProtocolLacp
        """Factory property that returns an instance of the LagProtocolLacp class

        The container for link aggregation control protocol settings of LAG (ports group).

        Returns: LagProtocolLacp
        """
        return self._get_property("lacp", LagProtocolLacp, self, "lacp")

    @property
    def static(self):
        # type: () -> LagProtocolStatic
        """Factory property that returns an instance of the LagProtocolStatic class

        The container for static link aggregation protocol settings.

        Returns: LagProtocolStatic
        """
        return self._get_property("static", LagProtocolStatic, self, "static")

    @property
    def choice(self):
        # type: () -> Union[Literal["lacp"], Literal["static"]]
        """choice getter

        The type of controlling protocol for the LAG (ports group).

        Returns: Union[Literal["lacp"], Literal["static"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of controlling protocol for the LAG (ports group).

        value: Union[Literal["lacp"], Literal["static"]]
        """
        self._set_property("choice", value)


class LagProtocolLacp(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "actor_system_id": {
            "type": str,
            "format": "mac",
        },
        "actor_system_priority": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "actor_key": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "actor_system_id": "00:00:00:00:00:00",
        "actor_system_priority": 0,
        "actor_key": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        actor_system_id="00:00:00:00:00:00",
        actor_system_priority=0,
        actor_key=0,
    ):
        super(LagProtocolLacp, self).__init__()
        self._parent = parent
        self._set_property("actor_system_id", actor_system_id)
        self._set_property("actor_system_priority", actor_system_priority)
        self._set_property("actor_key", actor_key)

    def set(self, actor_system_id=None, actor_system_priority=None, actor_key=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def actor_system_id(self):
        # type: () -> str
        """actor_system_id getter

        The actor system id

        Returns: str
        """
        return self._get_property("actor_system_id")

    @actor_system_id.setter
    def actor_system_id(self, value):
        """actor_system_id setter

        The actor system id

        value: str
        """
        self._set_property("actor_system_id", value)

    @property
    def actor_system_priority(self):
        # type: () -> int
        """actor_system_priority getter

        The actor system priority

        Returns: int
        """
        return self._get_property("actor_system_priority")

    @actor_system_priority.setter
    def actor_system_priority(self, value):
        """actor_system_priority setter

        The actor system priority

        value: int
        """
        self._set_property("actor_system_priority", value)

    @property
    def actor_key(self):
        # type: () -> int
        """actor_key getter

        The actor key

        Returns: int
        """
        return self._get_property("actor_key")

    @actor_key.setter
    def actor_key(self, value):
        """actor_key setter

        The actor key

        value: int
        """
        self._set_property("actor_key", value)


class LagProtocolStatic(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "lag_id": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "lag_id": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, lag_id=0):
        super(LagProtocolStatic, self).__init__()
        self._parent = parent
        self._set_property("lag_id", lag_id)

    def set(self, lag_id=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def lag_id(self):
        # type: () -> int
        """lag_id getter

        The static lag id

        Returns: int
        """
        return self._get_property("lag_id")

    @lag_id.setter
    def lag_id(self, value):
        """lag_id setter

        The static lag id

        value: int
        """
        self._set_property("lag_id", value)


class LagIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LagIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Lag]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LagIter
        return self._iter()

    def __next__(self):
        # type: () -> Lag
        return self._next()

    def next(self):
        # type: () -> Lag
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Lag):
            raise Exception("Item is not an instance of Lag")

    def lag(self, min_links=1, name=None):
        # type: (int,str) -> LagIter
        """Factory method that creates an instance of the Lag class

        The container for LAG (ports group) aggregation of multiple LAG members (ports)

        Returns: LagIter
        """
        item = Lag(parent=self._parent, min_links=min_links, name=name)
        self._add(item)
        return self

    def add(self, min_links=1, name=None):
        # type: (int,str) -> Lag
        """Add method that creates and returns an instance of the Lag class

        The container for LAG (ports group) aggregation of multiple LAG members (ports)

        Returns: Lag
        """
        item = Lag(parent=self._parent, min_links=min_links, name=name)
        self._add(item)
        return item


class Layer1(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_names": {
            "type": list,
            "itemtype": str,
        },
        "speed": {
            "type": str,
            "enum": [
                "speed_10_fd_mbps",
                "speed_10_hd_mbps",
                "speed_100_fd_mbps",
                "speed_100_hd_mbps",
                "speed_1_gbps",
                "speed_10_gbps",
                "speed_25_gbps",
                "speed_40_gbps",
                "speed_50_gbps",
                "speed_100_gbps",
                "speed_200_gbps",
                "speed_400_gbps",
                "speed_800_gbps",
            ],
        },
        "media": {
            "type": str,
            "enum": [
                "copper",
                "fiber",
                "sgmii",
            ],
        },
        "promiscuous": {"type": bool},
        "mtu": {
            "type": int,
            "format": "uint32",
            "minimum": 64,
            "maximum": 9000,
        },
        "ieee_media_defaults": {"type": bool},
        "auto_negotiate": {"type": bool},
        "auto_negotiation": {"type": "Layer1AutoNegotiation"},
        "flow_control": {"type": "Layer1FlowControl"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_names", "name")  # type: tuple(str)

    _DEFAULTS = {
        "promiscuous": True,
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    SPEED_10_FD_MBPS = "speed_10_fd_mbps"  # type: str
    SPEED_10_HD_MBPS = "speed_10_hd_mbps"  # type: str
    SPEED_100_FD_MBPS = "speed_100_fd_mbps"  # type: str
    SPEED_100_HD_MBPS = "speed_100_hd_mbps"  # type: str
    SPEED_1_GBPS = "speed_1_gbps"  # type: str
    SPEED_10_GBPS = "speed_10_gbps"  # type: str
    SPEED_25_GBPS = "speed_25_gbps"  # type: str
    SPEED_40_GBPS = "speed_40_gbps"  # type: str
    SPEED_50_GBPS = "speed_50_gbps"  # type: str
    SPEED_100_GBPS = "speed_100_gbps"  # type: str
    SPEED_200_GBPS = "speed_200_gbps"  # type: str
    SPEED_400_GBPS = "speed_400_gbps"  # type: str
    SPEED_800_GBPS = "speed_800_gbps"  # type: str

    COPPER = "copper"  # type: str
    FIBER = "fiber"  # type: str
    SGMII = "sgmii"  # type: str

    _STATUS = {
        "ieee_media_defaults": "ieee_media_defaults property in schema Layer1 is under-review, This field is currently under review for pending exploration on use cases",
        "auto_negotiate": "auto_negotiate property in schema Layer1 is under-review, This field is currently under review for pending exploration on use cases, given that a separate configuration called `AutoNegotiation` already exists.",
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        port_names=None,
        speed=None,
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        super(Layer1, self).__init__()
        self._parent = parent
        self._set_property("port_names", port_names)
        self._set_property("speed", speed)
        self._set_property("media", media)
        self._set_property("promiscuous", promiscuous)
        self._set_property("mtu", mtu)
        self._set_property("ieee_media_defaults", ieee_media_defaults)
        self._set_property("auto_negotiate", auto_negotiate)
        self._set_property("name", name)

    def set(
        self,
        port_names=None,
        speed=None,
        media=None,
        promiscuous=None,
        mtu=None,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        A list of unique names of port objects that will share the. choice settings. . x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        Returns: List[str]
        """
        return self._get_property("port_names")

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        A list of unique names of port objects that will share the. choice settings. . x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        value: List[str]
        """
        if value is None:
            raise TypeError("Cannot set required property port_names as None")
        self._set_property("port_names", value)

    @property
    def speed(self):
        # type: () -> Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"], Literal["speed_800_gbps"]]
        """speed getter

        Set the speed if supported. When no speed is explicitly set, the current. speed of underlying test interface shall be assumed.

        Returns: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"], Literal["speed_800_gbps"]]
        """
        return self._get_property("speed")

    @speed.setter
    def speed(self, value):
        """speed setter

        Set the speed if supported. When no speed is explicitly set, the current. speed of underlying test interface shall be assumed.

        value: Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"], Literal["speed_800_gbps"]]
        """
        self._set_property("speed", value)

    @property
    def media(self):
        # type: () -> Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """media getter

        Set the type of media for test interface if supported. When no media. type is explicitly set, the current media type of underlying test. interface shall be assumed.

        Returns: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        return self._get_property("media")

    @media.setter
    def media(self, value):
        """media setter

        Set the type of media for test interface if supported. When no media. type is explicitly set, the current media type of underlying test. interface shall be assumed.

        value: Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]]
        """
        self._set_property("media", value)

    @property
    def promiscuous(self):
        # type: () -> bool
        """promiscuous getter

        Enable promiscuous mode on test interface. warning shall be raised if. this field is set to `true`, even when it's not supported, ignoring. the setting altogether.

        Returns: bool
        """
        return self._get_property("promiscuous")

    @promiscuous.setter
    def promiscuous(self, value):
        """promiscuous setter

        Enable promiscuous mode on test interface. warning shall be raised if. this field is set to `true`, even when it's not supported, ignoring. the setting altogether.

        value: bool
        """
        self._set_property("promiscuous", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Set the maximum transmission unit size. warning shall be raised if. the specified value is valid but not supported, ignoring the setting altogether.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Set the maximum transmission unit size. warning shall be raised if. the specified value is valid but not supported, ignoring the setting altogether.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def ieee_media_defaults(self):
        # type: () -> bool
        """ieee_media_defaults getter

        Under Review: This field is currently under review for pending exploration on use cases. Under Review: This field is currently under review for pending exploration on use cases. Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        Returns: bool
        """
        return self._get_property("ieee_media_defaults")

    @ieee_media_defaults.setter
    def ieee_media_defaults(self, value):
        """ieee_media_defaults setter

        Under Review: This field is currently under review for pending exploration on use cases. Under Review: This field is currently under review for pending exploration on use cases. Set to true to override the auto_negotiate, link_training. and rs_fec settings for gigabit ethernet interfaces.

        value: bool
        """
        self._set_property("ieee_media_defaults", value)

    @property
    def auto_negotiate(self):
        # type: () -> bool
        """auto_negotiate getter

        Under Review: This field is currently under review for pending exploration on use cases, given that separate configuration called `AutoNegotiation` already exists.. Under Review: This field is currently under review for pending exploration on use cases, given that separate configuration called `AutoNegotiation` already exists.. Enable/disable auto negotiation.

        Returns: bool
        """
        return self._get_property("auto_negotiate")

    @auto_negotiate.setter
    def auto_negotiate(self, value):
        """auto_negotiate setter

        Under Review: This field is currently under review for pending exploration on use cases, given that separate configuration called `AutoNegotiation` already exists.. Under Review: This field is currently under review for pending exploration on use cases, given that separate configuration called `AutoNegotiation` already exists.. Enable/disable auto negotiation.

        value: bool
        """
        self._set_property("auto_negotiate", value)

    @property
    def auto_negotiation(self):
        # type: () -> Layer1AutoNegotiation
        """auto_negotiation getter

        Configuration for auto negotiation settingsConfiguration for auto negotiation settingsConfiguration for auto negotiation settings

        Returns: Layer1AutoNegotiation
        """
        return self._get_property("auto_negotiation", Layer1AutoNegotiation)

    @property
    def flow_control(self):
        # type: () -> Layer1FlowControl
        """flow_control getter

        A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be valid. object not null value.A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be valid. object not null value.A container for layer1 receive flow control settings.. To enable flow control settings on ports this object must be valid. object not null value.

        Returns: Layer1FlowControl
        """
        return self._get_property("flow_control", Layer1FlowControl)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class Layer1AutoNegotiation(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "advertise_1000_mbps": {"type": bool},
        "advertise_100_fd_mbps": {"type": bool},
        "advertise_100_hd_mbps": {"type": bool},
        "advertise_10_fd_mbps": {"type": bool},
        "advertise_10_hd_mbps": {"type": bool},
        "link_training": {"type": bool},
        "rs_fec": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "advertise_1000_mbps": True,
        "advertise_100_fd_mbps": True,
        "advertise_100_hd_mbps": True,
        "advertise_10_fd_mbps": True,
        "advertise_10_hd_mbps": True,
        "link_training": False,
        "rs_fec": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        advertise_1000_mbps=True,
        advertise_100_fd_mbps=True,
        advertise_100_hd_mbps=True,
        advertise_10_fd_mbps=True,
        advertise_10_hd_mbps=True,
        link_training=False,
        rs_fec=False,
    ):
        super(Layer1AutoNegotiation, self).__init__()
        self._parent = parent
        self._set_property("advertise_1000_mbps", advertise_1000_mbps)
        self._set_property("advertise_100_fd_mbps", advertise_100_fd_mbps)
        self._set_property("advertise_100_hd_mbps", advertise_100_hd_mbps)
        self._set_property("advertise_10_fd_mbps", advertise_10_fd_mbps)
        self._set_property("advertise_10_hd_mbps", advertise_10_hd_mbps)
        self._set_property("link_training", link_training)
        self._set_property("rs_fec", rs_fec)

    def set(
        self,
        advertise_1000_mbps=None,
        advertise_100_fd_mbps=None,
        advertise_100_hd_mbps=None,
        advertise_10_fd_mbps=None,
        advertise_10_hd_mbps=None,
        link_training=None,
        rs_fec=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def advertise_1000_mbps(self):
        # type: () -> bool
        """advertise_1000_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_1000_mbps")

    @advertise_1000_mbps.setter
    def advertise_1000_mbps(self, value):
        """advertise_1000_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_1000_mbps", value)

    @property
    def advertise_100_fd_mbps(self):
        # type: () -> bool
        """advertise_100_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_100_fd_mbps")

    @advertise_100_fd_mbps.setter
    def advertise_100_fd_mbps(self, value):
        """advertise_100_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_100_fd_mbps", value)

    @property
    def advertise_100_hd_mbps(self):
        # type: () -> bool
        """advertise_100_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_100_hd_mbps")

    @advertise_100_hd_mbps.setter
    def advertise_100_hd_mbps(self, value):
        """advertise_100_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_100_hd_mbps", value)

    @property
    def advertise_10_fd_mbps(self):
        # type: () -> bool
        """advertise_10_fd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_10_fd_mbps")

    @advertise_10_fd_mbps.setter
    def advertise_10_fd_mbps(self, value):
        """advertise_10_fd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_10_fd_mbps", value)

    @property
    def advertise_10_hd_mbps(self):
        # type: () -> bool
        """advertise_10_hd_mbps getter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        Returns: bool
        """
        return self._get_property("advertise_10_hd_mbps")

    @advertise_10_hd_mbps.setter
    def advertise_10_hd_mbps(self, value):
        """advertise_10_hd_mbps setter

        If auto_negotiate is true and the interface supports this option. then this speed will be advertised.

        value: bool
        """
        self._set_property("advertise_10_hd_mbps", value)

    @property
    def link_training(self):
        # type: () -> bool
        """link_training getter

        Enable/disable gigabit ethernet link training.

        Returns: bool
        """
        return self._get_property("link_training")

    @link_training.setter
    def link_training(self, value):
        """link_training setter

        Enable/disable gigabit ethernet link training.

        value: bool
        """
        self._set_property("link_training", value)

    @property
    def rs_fec(self):
        # type: () -> bool
        """rs_fec getter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        Returns: bool
        """
        return self._get_property("rs_fec")

    @rs_fec.setter
    def rs_fec(self, value):
        """rs_fec setter

        Enable/disable gigabit ethernet reed solomon forward error correction (RS FEC).

        value: bool
        """
        self._set_property("rs_fec", value)


class Layer1FlowControl(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "directed_address": {
            "type": str,
            "format": "mac",
        },
        "choice": {
            "type": str,
            "enum": [
                "ieee_802_1qbb",
                "ieee_802_3x",
            ],
        },
        "ieee_802_1qbb": {"type": "Layer1Ieee8021qbb"},
        "ieee_802_3x": {"type": "Layer1Ieee8023x"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "directed_address": "01:80:C2:00:00:01",
        "choice": "ieee_802_1qbb",
    }  # type: Dict[str, Union(type)]

    IEEE_802_1QBB = "ieee_802_1qbb"  # type: str
    IEEE_802_3X = "ieee_802_3x"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, directed_address="01:80:C2:00:00:01"):
        super(Layer1FlowControl, self).__init__()
        self._parent = parent
        self._set_property("directed_address", directed_address)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, directed_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ieee_802_1qbb(self):
        # type: () -> Layer1Ieee8021qbb
        """Factory property that returns an instance of the Layer1Ieee8021qbb class

        These settings enhance the existing 802.3x pause priority capabilities to enable flow control based on 802.1p priorities (classes of service).

        Returns: Layer1Ieee8021qbb
        """
        return self._get_property(
            "ieee_802_1qbb", Layer1Ieee8021qbb, self, "ieee_802_1qbb"
        )

    @property
    def ieee_802_3x(self):
        # type: () -> Layer1Ieee8023x
        """Factory property that returns an instance of the Layer1Ieee8023x class

        A container for ieee 802.3x rx pause settings

        Returns: Layer1Ieee8023x
        """
        return self._get_property("ieee_802_3x", Layer1Ieee8023x, self, "ieee_802_3x")

    @property
    def directed_address(self):
        # type: () -> str
        """directed_address getter

        The 48bit mac address that the layer1 port names will listen on. for directed pause.

        Returns: str
        """
        return self._get_property("directed_address")

    @directed_address.setter
    def directed_address(self, value):
        """directed_address setter

        The 48bit mac address that the layer1 port names will listen on. for directed pause.

        value: str
        """
        self._set_property("directed_address", value)

    @property
    def choice(self):
        # type: () -> Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """choice getter

        The type of priority flow control.

        Returns: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of priority flow control.

        value: Union[Literal["ieee_802_1qbb"], Literal["ieee_802_3x"]]
        """
        self._set_property("choice", value)


class Layer1Ieee8021qbb(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "pfc_delay": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "pfc_class_0": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_1": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_2": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_3": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_4": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_5": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_6": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "pfc_class_7": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "pfc_delay": 0,
        "pfc_class_0": 0,
        "pfc_class_1": 1,
        "pfc_class_2": 2,
        "pfc_class_3": 3,
        "pfc_class_4": 4,
        "pfc_class_5": 5,
        "pfc_class_6": 6,
        "pfc_class_7": 7,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        pfc_delay=0,
        pfc_class_0=0,
        pfc_class_1=1,
        pfc_class_2=2,
        pfc_class_3=3,
        pfc_class_4=4,
        pfc_class_5=5,
        pfc_class_6=6,
        pfc_class_7=7,
    ):
        super(Layer1Ieee8021qbb, self).__init__()
        self._parent = parent
        self._set_property("pfc_delay", pfc_delay)
        self._set_property("pfc_class_0", pfc_class_0)
        self._set_property("pfc_class_1", pfc_class_1)
        self._set_property("pfc_class_2", pfc_class_2)
        self._set_property("pfc_class_3", pfc_class_3)
        self._set_property("pfc_class_4", pfc_class_4)
        self._set_property("pfc_class_5", pfc_class_5)
        self._set_property("pfc_class_6", pfc_class_6)
        self._set_property("pfc_class_7", pfc_class_7)

    def set(
        self,
        pfc_delay=None,
        pfc_class_0=None,
        pfc_class_1=None,
        pfc_class_2=None,
        pfc_class_3=None,
        pfc_class_4=None,
        pfc_class_5=None,
        pfc_class_6=None,
        pfc_class_7=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def pfc_delay(self):
        # type: () -> int
        """pfc_delay getter

        The upper limit on the transmit time of queue after receiving . message to pause specified priority.. value of or null indicates that pfc delay will not be enabled.

        Returns: int
        """
        return self._get_property("pfc_delay")

    @pfc_delay.setter
    def pfc_delay(self, value):
        """pfc_delay setter

        The upper limit on the transmit time of queue after receiving . message to pause specified priority.. value of or null indicates that pfc delay will not be enabled.

        value: int
        """
        self._set_property("pfc_delay", value)

    @property
    def pfc_class_0(self):
        # type: () -> int
        """pfc_class_0 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_0")

    @pfc_class_0.setter
    def pfc_class_0(self, value):
        """pfc_class_0 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_0", value)

    @property
    def pfc_class_1(self):
        # type: () -> int
        """pfc_class_1 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_1")

    @pfc_class_1.setter
    def pfc_class_1(self, value):
        """pfc_class_1 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_1", value)

    @property
    def pfc_class_2(self):
        # type: () -> int
        """pfc_class_2 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_2")

    @pfc_class_2.setter
    def pfc_class_2(self, value):
        """pfc_class_2 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_2", value)

    @property
    def pfc_class_3(self):
        # type: () -> int
        """pfc_class_3 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_3")

    @pfc_class_3.setter
    def pfc_class_3(self, value):
        """pfc_class_3 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_3", value)

    @property
    def pfc_class_4(self):
        # type: () -> int
        """pfc_class_4 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_4")

    @pfc_class_4.setter
    def pfc_class_4(self, value):
        """pfc_class_4 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_4", value)

    @property
    def pfc_class_5(self):
        # type: () -> int
        """pfc_class_5 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_5")

    @pfc_class_5.setter
    def pfc_class_5(self, value):
        """pfc_class_5 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_5", value)

    @property
    def pfc_class_6(self):
        # type: () -> int
        """pfc_class_6 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_6")

    @pfc_class_6.setter
    def pfc_class_6(self, value):
        """pfc_class_6 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_6", value)

    @property
    def pfc_class_7(self):
        # type: () -> int
        """pfc_class_7 getter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        Returns: int
        """
        return self._get_property("pfc_class_7")

    @pfc_class_7.setter
    def pfc_class_7(self, value):
        """pfc_class_7 setter

        The valid values are null, - 7.. null value indicates there is no setting for this pfc class.

        value: int
        """
        self._set_property("pfc_class_7", value)


class Layer1Ieee8023x(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {}  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(Layer1Ieee8023x, self).__init__()
        self._parent = parent


class Layer1Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(Layer1Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Layer1]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Layer1Iter
        return self._iter()

    def __next__(self):
        # type: () -> Layer1
        return self._next()

    def next(self):
        # type: () -> Layer1
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Layer1):
            raise Exception("Item is not an instance of Layer1")

    def layer1(
        self,
        port_names=None,
        speed=None,
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        # type: (List[str],Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"], Literal["speed_800_gbps"]],Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]],bool,int,bool,bool,str) -> Layer1Iter
        """Factory method that creates an instance of the Layer1 class

        A container for layer1 settings.

        Returns: Layer1Iter
        """
        item = Layer1(
            parent=self._parent,
            port_names=port_names,
            speed=speed,
            media=media,
            promiscuous=promiscuous,
            mtu=mtu,
            ieee_media_defaults=ieee_media_defaults,
            auto_negotiate=auto_negotiate,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        port_names=None,
        speed=None,
        media=None,
        promiscuous=True,
        mtu=1500,
        ieee_media_defaults=None,
        auto_negotiate=None,
        name=None,
    ):
        # type: (List[str],Union[Literal["speed_100_fd_mbps"], Literal["speed_100_gbps"], Literal["speed_100_hd_mbps"], Literal["speed_10_fd_mbps"], Literal["speed_10_gbps"], Literal["speed_10_hd_mbps"], Literal["speed_1_gbps"], Literal["speed_200_gbps"], Literal["speed_25_gbps"], Literal["speed_400_gbps"], Literal["speed_40_gbps"], Literal["speed_50_gbps"], Literal["speed_800_gbps"]],Union[Literal["copper"], Literal["fiber"], Literal["sgmii"]],bool,int,bool,bool,str) -> Layer1
        """Add method that creates and returns an instance of the Layer1 class

        A container for layer1 settings.

        Returns: Layer1
        """
        item = Layer1(
            parent=self._parent,
            port_names=port_names,
            speed=speed,
            media=media,
            promiscuous=promiscuous,
            mtu=mtu,
            ieee_media_defaults=ieee_media_defaults,
            auto_negotiate=auto_negotiate,
            name=name,
        )
        self._add(item)
        return item


class Capture(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "port_names": {
            "type": list,
            "itemtype": str,
        },
        "filters": {"type": "CaptureFilterIter"},
        "overwrite": {"type": bool},
        "packet_size": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "format": {
            "type": str,
            "enum": [
                "pcap",
                "pcapng",
            ],
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("port_names", "name")  # type: tuple(str)

    _DEFAULTS = {
        "overwrite": True,
        "format": "pcap",
    }  # type: Dict[str, Union(type)]

    PCAP = "pcap"  # type: str
    PCAPNG = "pcapng"  # type: str

    _STATUS = {
        "self": "Capture is under_review, There may be changes in filter configuration",
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        super(Capture, self).__init__()
        self._parent = parent
        self._set_property("port_names", port_names)
        self._set_property("overwrite", overwrite)
        self._set_property("packet_size", packet_size)
        self._set_property("format", format)
        self._set_property("name", name)

    def set(
        self, port_names=None, overwrite=None, packet_size=None, format=None, name=None
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def port_names(self):
        # type: () -> List[str]
        """port_names getter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        Returns: List[str]
        """
        return self._get_property("port_names")

    @port_names.setter
    def port_names(self, value):
        """port_names setter

        The unique names of ports that the capture settings will apply to. Port_names cannot be duplicated between capture objects.. x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        value: List[str]
        """
        if value is None:
            raise TypeError("Cannot set required property port_names as None")
        self._set_property("port_names", value)

    @property
    def filters(self):
        # type: () -> CaptureFilterIter
        """filters getter

        A list of filters to apply to the capturing ports. If no filters are specified then all packets will be captured. capture can have multiple filters. The number of filters supported is determined by the implementation which can be retrieved using the capabilities API.. When multiple filters are specified the capture implementation must && (and) all the filters.

        Returns: CaptureFilterIter
        """
        return self._get_property(
            "filters", CaptureFilterIter, self._parent, self._choice
        )

    @property
    def overwrite(self):
        # type: () -> bool
        """overwrite getter

        Overwrite the capture buffer.

        Returns: bool
        """
        return self._get_property("overwrite")

    @overwrite.setter
    def overwrite(self, value):
        """overwrite setter

        Overwrite the capture buffer.

        value: bool
        """
        self._set_property("overwrite", value)

    @property
    def packet_size(self):
        # type: () -> int
        """packet_size getter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        Returns: int
        """
        return self._get_property("packet_size")

    @packet_size.setter
    def packet_size(self, value):
        """packet_size setter

        The maximum size of each captured packet. If no value is specified or it is null then the entire packet will be captured.

        value: int
        """
        self._set_property("packet_size", value)

    @property
    def format(self):
        # type: () -> Union[Literal["pcap"], Literal["pcapng"]]
        """format getter

        The format of the capture file.

        Returns: Union[Literal["pcap"], Literal["pcapng"]]
        """
        return self._get_property("format")

    @format.setter
    def format(self, value):
        """format setter

        The format of the capture file.

        value: Union[Literal["pcap"], Literal["pcapng"]]
        """
        self._set_property("format", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class CaptureFilter(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "custom",
                "ethernet",
                "vlan",
                "ipv4",
                "ipv6",
            ],
        },
        "custom": {"type": "CaptureCustom"},
        "ethernet": {"type": "CaptureEthernet"},
        "vlan": {"type": "CaptureVlan"},
        "ipv4": {"type": "CaptureIpv4"},
        "ipv6": {"type": "CaptureIpv6"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "custom",
    }  # type: Dict[str, Union(type)]

    CUSTOM = "custom"  # type: str
    ETHERNET = "ethernet"  # type: str
    VLAN = "vlan"  # type: str
    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(CaptureFilter, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def custom(self):
        # type: () -> CaptureCustom
        """Factory property that returns an instance of the CaptureCustom class

        TBD

        Returns: CaptureCustom
        """
        return self._get_property("custom", CaptureCustom, self, "custom")

    @property
    def ethernet(self):
        # type: () -> CaptureEthernet
        """Factory property that returns an instance of the CaptureEthernet class

        TBD

        Returns: CaptureEthernet
        """
        return self._get_property("ethernet", CaptureEthernet, self, "ethernet")

    @property
    def vlan(self):
        # type: () -> CaptureVlan
        """Factory property that returns an instance of the CaptureVlan class

        TBD

        Returns: CaptureVlan
        """
        return self._get_property("vlan", CaptureVlan, self, "vlan")

    @property
    def ipv4(self):
        # type: () -> CaptureIpv4
        """Factory property that returns an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureIpv4
        """
        return self._get_property("ipv4", CaptureIpv4, self, "ipv4")

    @property
    def ipv6(self):
        # type: () -> CaptureIpv6
        """Factory property that returns an instance of the CaptureIpv6 class

        TBD

        Returns: CaptureIpv6
        """
        return self._get_property("ipv6", CaptureIpv6, self, "ipv6")

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """choice getter

        The type of capture filter.

        Returns: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of capture filter.

        value: Union[Literal["custom"], Literal["ethernet"], Literal["ipv4"], Literal["ipv6"], Literal["vlan"]]
        """
        self._set_property("choice", value)


class CaptureCustom(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "offset": {
            "type": int,
            "format": "uint32",
        },
        "bit_length": {
            "type": int,
            "format": "uint32",
        },
        "value": {
            "type": str,
            "format": "hex",
        },
        "mask": {
            "type": str,
            "format": "hex",
        },
        "negate": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "bit_length": 8,
        "value": "00",
        "mask": "00",
        "negate": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        offset=None,
        bit_length=8,
        value="00",
        mask="00",
        negate=False,
    ):
        super(CaptureCustom, self).__init__()
        self._parent = parent
        self._set_property("offset", offset)
        self._set_property("bit_length", bit_length)
        self._set_property("value", value)
        self._set_property("mask", mask)
        self._set_property("negate", negate)

    def set(self, offset=None, bit_length=None, value=None, mask=None, negate=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def offset(self):
        # type: () -> int
        """offset getter

        The bit offset of field to filter on

        Returns: int
        """
        return self._get_property("offset")

    @offset.setter
    def offset(self, value):
        """offset setter

        The bit offset of field to filter on

        value: int
        """
        self._set_property("offset", value)

    @property
    def bit_length(self):
        # type: () -> int
        """bit_length getter

        The bit length of field to filter on

        Returns: int
        """
        return self._get_property("bit_length")

    @bit_length.setter
    def bit_length(self, value):
        """bit_length setter

        The bit length of field to filter on

        value: int
        """
        self._set_property("bit_length", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property("mask")

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property("mask", value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property("negate")

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property("negate", value)


class CaptureEthernet(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
        "ether_type": {"type": "CaptureField"},
        "pfc_queue": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureEthernet, self).__init__()
        self._parent = parent

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)

    @property
    def ether_type(self):
        # type: () -> CaptureField
        """ether_type getter



        Returns: CaptureField
        """
        return self._get_property("ether_type", CaptureField)

    @property
    def pfc_queue(self):
        # type: () -> CaptureField
        """pfc_queue getter



        Returns: CaptureField
        """
        return self._get_property("pfc_queue", CaptureField)


class CaptureField(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "format": "hex",
        },
        "mask": {
            "type": str,
            "format": "hex",
        },
        "negate": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": "00",
        "mask": "00",
        "negate": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value="00", mask="00", negate=False):
        super(CaptureField, self).__init__()
        self._parent = parent
        self._set_property("value", value)
        self._set_property("mask", mask)
        self._set_property("negate", negate)

    def set(self, value=None, mask=None, negate=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value)

    @property
    def mask(self):
        # type: () -> str
        """mask getter

        TBD

        Returns: str
        """
        return self._get_property("mask")

    @mask.setter
    def mask(self, value):
        """mask setter

        TBD

        value: str
        """
        self._set_property("mask", value)

    @property
    def negate(self):
        # type: () -> bool
        """negate getter

        TBD

        Returns: bool
        """
        return self._get_property("negate")

    @negate.setter
    def negate(self, value):
        """negate setter

        TBD

        value: bool
        """
        self._set_property("negate", value)


class CaptureVlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "priority": {"type": "CaptureField"},
        "cfi": {"type": "CaptureField"},
        "id": {"type": "CaptureField"},
        "protocol": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureVlan, self).__init__()
        self._parent = parent

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter



        Returns: CaptureField
        """
        return self._get_property("priority", CaptureField)

    @property
    def cfi(self):
        # type: () -> CaptureField
        """cfi getter



        Returns: CaptureField
        """
        return self._get_property("cfi", CaptureField)

    @property
    def id(self):
        # type: () -> CaptureField
        """id getter



        Returns: CaptureField
        """
        return self._get_property("id", CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter



        Returns: CaptureField
        """
        return self._get_property("protocol", CaptureField)


class CaptureIpv4(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "CaptureField"},
        "header_length": {"type": "CaptureField"},
        "priority": {"type": "CaptureField"},
        "total_length": {"type": "CaptureField"},
        "identification": {"type": "CaptureField"},
        "reserved": {"type": "CaptureField"},
        "dont_fragment": {"type": "CaptureField"},
        "more_fragments": {"type": "CaptureField"},
        "fragment_offset": {"type": "CaptureField"},
        "time_to_live": {"type": "CaptureField"},
        "protocol": {"type": "CaptureField"},
        "header_checksum": {"type": "CaptureField"},
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureIpv4, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> CaptureField
        """version getter



        Returns: CaptureField
        """
        return self._get_property("version", CaptureField)

    @property
    def header_length(self):
        # type: () -> CaptureField
        """header_length getter



        Returns: CaptureField
        """
        return self._get_property("header_length", CaptureField)

    @property
    def priority(self):
        # type: () -> CaptureField
        """priority getter



        Returns: CaptureField
        """
        return self._get_property("priority", CaptureField)

    @property
    def total_length(self):
        # type: () -> CaptureField
        """total_length getter



        Returns: CaptureField
        """
        return self._get_property("total_length", CaptureField)

    @property
    def identification(self):
        # type: () -> CaptureField
        """identification getter



        Returns: CaptureField
        """
        return self._get_property("identification", CaptureField)

    @property
    def reserved(self):
        # type: () -> CaptureField
        """reserved getter



        Returns: CaptureField
        """
        return self._get_property("reserved", CaptureField)

    @property
    def dont_fragment(self):
        # type: () -> CaptureField
        """dont_fragment getter



        Returns: CaptureField
        """
        return self._get_property("dont_fragment", CaptureField)

    @property
    def more_fragments(self):
        # type: () -> CaptureField
        """more_fragments getter



        Returns: CaptureField
        """
        return self._get_property("more_fragments", CaptureField)

    @property
    def fragment_offset(self):
        # type: () -> CaptureField
        """fragment_offset getter



        Returns: CaptureField
        """
        return self._get_property("fragment_offset", CaptureField)

    @property
    def time_to_live(self):
        # type: () -> CaptureField
        """time_to_live getter



        Returns: CaptureField
        """
        return self._get_property("time_to_live", CaptureField)

    @property
    def protocol(self):
        # type: () -> CaptureField
        """protocol getter



        Returns: CaptureField
        """
        return self._get_property("protocol", CaptureField)

    @property
    def header_checksum(self):
        # type: () -> CaptureField
        """header_checksum getter



        Returns: CaptureField
        """
        return self._get_property("header_checksum", CaptureField)

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)


class CaptureIpv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "CaptureField"},
        "traffic_class": {"type": "CaptureField"},
        "flow_label": {"type": "CaptureField"},
        "payload_length": {"type": "CaptureField"},
        "next_header": {"type": "CaptureField"},
        "hop_limit": {"type": "CaptureField"},
        "src": {"type": "CaptureField"},
        "dst": {"type": "CaptureField"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(CaptureIpv6, self).__init__()
        self._parent = parent

    @property
    def version(self):
        # type: () -> CaptureField
        """version getter



        Returns: CaptureField
        """
        return self._get_property("version", CaptureField)

    @property
    def traffic_class(self):
        # type: () -> CaptureField
        """traffic_class getter



        Returns: CaptureField
        """
        return self._get_property("traffic_class", CaptureField)

    @property
    def flow_label(self):
        # type: () -> CaptureField
        """flow_label getter



        Returns: CaptureField
        """
        return self._get_property("flow_label", CaptureField)

    @property
    def payload_length(self):
        # type: () -> CaptureField
        """payload_length getter



        Returns: CaptureField
        """
        return self._get_property("payload_length", CaptureField)

    @property
    def next_header(self):
        # type: () -> CaptureField
        """next_header getter



        Returns: CaptureField
        """
        return self._get_property("next_header", CaptureField)

    @property
    def hop_limit(self):
        # type: () -> CaptureField
        """hop_limit getter



        Returns: CaptureField
        """
        return self._get_property("hop_limit", CaptureField)

    @property
    def src(self):
        # type: () -> CaptureField
        """src getter



        Returns: CaptureField
        """
        return self._get_property("src", CaptureField)

    @property
    def dst(self):
        # type: () -> CaptureField
        """dst getter



        Returns: CaptureField
        """
        return self._get_property("dst", CaptureField)


class CaptureFilterIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(CaptureFilterIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[CaptureCustom, CaptureEthernet, CaptureFilter, CaptureIpv4, CaptureIpv6, CaptureVlan]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureFilterIter
        return self._iter()

    def __next__(self):
        # type: () -> CaptureFilter
        return self._next()

    def next(self):
        # type: () -> CaptureFilter
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, CaptureFilter):
            raise Exception("Item is not an instance of CaptureFilter")

    def filter(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureFilter class

        Configuration for capture filters

        Returns: CaptureFilterIter
        """
        item = CaptureFilter(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> CaptureFilter
        """Add method that creates and returns an instance of the CaptureFilter class

        Configuration for capture filters

        Returns: CaptureFilter
        """
        item = CaptureFilter(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def custom(self, offset=None, bit_length=8, value="00", mask="00", negate=False):
        # type: (int,int,str,str,bool) -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureCustom class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.custom
        item.choice = "custom"
        self._add(item)
        return self

    def ethernet(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureEthernet class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ethernet
        item.choice = "ethernet"
        self._add(item)
        return self

    def vlan(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureVlan class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.vlan
        item.choice = "vlan"
        self._add(item)
        return self

    def ipv4(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureIpv4 class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ipv4
        item.choice = "ipv4"
        self._add(item)
        return self

    def ipv6(self):
        # type: () -> CaptureFilterIter
        """Factory method that creates an instance of the CaptureIpv6 class

        TBD

        Returns: CaptureFilterIter
        """
        item = CaptureFilter()
        item.ipv6
        item.choice = "ipv6"
        self._add(item)
        return self


class CaptureIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(CaptureIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Capture]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> CaptureIter
        return self._iter()

    def __next__(self):
        # type: () -> Capture
        return self._next()

    def next(self):
        # type: () -> Capture
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Capture):
            raise Exception("Item is not an instance of Capture")

    def capture(
        self,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        # type: (List[str],bool,int,Union[Literal["pcap"], Literal["pcapng"]],str) -> CaptureIter
        """Factory method that creates an instance of the Capture class

        Under Review: There may be changes in filter configuration. Under Review: There may be changes in filter configuration. Configuration for capture settings.

        Returns: CaptureIter
        """
        item = Capture(
            parent=self._parent,
            port_names=port_names,
            overwrite=overwrite,
            packet_size=packet_size,
            format=format,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        port_names=None,
        overwrite=True,
        packet_size=None,
        format="pcap",
        name=None,
    ):
        # type: (List[str],bool,int,Union[Literal["pcap"], Literal["pcapng"]],str) -> Capture
        """Add method that creates and returns an instance of the Capture class

        Under Review: There may be changes in filter configuration. Under Review: There may be changes in filter configuration. Configuration for capture settings.

        Returns: Capture
        """
        item = Capture(
            parent=self._parent,
            port_names=port_names,
            overwrite=overwrite,
            packet_size=packet_size,
            format=format,
            name=name,
        )
        self._add(item)
        return item


class Device(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ethernets": {"type": "DeviceEthernetIter"},
        "ipv4_loopbacks": {"type": "DeviceIpv4LoopbackIter"},
        "ipv6_loopbacks": {"type": "DeviceIpv6LoopbackIter"},
        "isis": {"type": "DeviceIsisRouter"},
        "bgp": {"type": "DeviceBgpRouter"},
        "vxlan": {"type": "DeviceVxlan"},
        "name": {"type": str},
        "rsvp": {"type": "DeviceRsvp"},
        "dhcp_server": {"type": "DeviceDhcpServer"},
        "ospfv2": {"type": "DeviceOspfv2Router"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None):
        super(Device, self).__init__()
        self._parent = parent
        self._set_property("name", name)

    def set(self, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ethernets(self):
        # type: () -> DeviceEthernetIter
        """ethernets getter

        Ethernet configuration for one or more emulated or simulated network interfaces.

        Returns: DeviceEthernetIter
        """
        return self._get_property(
            "ethernets", DeviceEthernetIter, self._parent, self._choice
        )

    @property
    def ipv4_loopbacks(self):
        # type: () -> DeviceIpv4LoopbackIter
        """ipv4_loopbacks getter

        IPv4 Loopback interface that can be attached to an Ethernet in the same device or to an Ethernet in another device.

        Returns: DeviceIpv4LoopbackIter
        """
        return self._get_property(
            "ipv4_loopbacks", DeviceIpv4LoopbackIter, self._parent, self._choice
        )

    @property
    def ipv6_loopbacks(self):
        # type: () -> DeviceIpv6LoopbackIter
        """ipv6_loopbacks getter

        IPv6 Loopback interface that can be attached to an Ethernet in the same device or to an Ethernet in another device.

        Returns: DeviceIpv6LoopbackIter
        """
        return self._get_property(
            "ipv6_loopbacks", DeviceIpv6LoopbackIter, self._parent, self._choice
        )

    @property
    def isis(self):
        # type: () -> DeviceIsisRouter
        """isis getter

        A container of properties for an ISIS router and its interfaces.A container of properties for an ISIS router and its interfaces.A container of properties for an ISIS router and its interfaces.The properties of an IS-IS router and its children, such as IS-IS interfaces and route ranges.

        Returns: DeviceIsisRouter
        """
        return self._get_property("isis", DeviceIsisRouter)

    @property
    def bgp(self):
        # type: () -> DeviceBgpRouter
        """bgp getter

        Configuration for one or more IPv4 or IPv6 BGP peers.Configuration for one or more IPv4 or IPv6 BGP peers.Configuration for one or more IPv4 or IPv6 BGP peers.The properties of BGP router and its children, such as BGPv4, BGPv6 peers and their route ranges.

        Returns: DeviceBgpRouter
        """
        return self._get_property("bgp", DeviceBgpRouter)

    @property
    def vxlan(self):
        # type: () -> DeviceVxlan
        """vxlan getter

        Configuration of VXLAN tunnel interfaces RFC Ref: https://datatracker.ietf.org/doc/html/rfc7348

        Returns: DeviceVxlan
        """
        return self._get_property("vxlan", DeviceVxlan)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def rsvp(self):
        # type: () -> DeviceRsvp
        """rsvp getter

        Configuration for one or more RSVP interfaces, ingress and egress LSPs. In this model, currently IPv4 RSVP and point-to-point LSPs are supported as per RFC3209 and related specifications.Configuration for one or more RSVP interfaces, ingress and egress LSPs. In this model, currently IPv4 RSVP and point-to-point LSPs are supported as per RFC3209 and related specifications.Configuration for one or more RSVP interfaces, ingress and egress LSPs. In this model, currently IPv4 RSVP and point-to-point LSPs are supported as per RFC3209 and related specifications.The properties of an RSVP router and its children.

        Returns: DeviceRsvp
        """
        return self._get_property("rsvp", DeviceRsvp)

    @property
    def dhcp_server(self):
        # type: () -> DeviceDhcpServer
        """dhcp_server getter

        Configuration for one or more IPv4 or IPv6 DHCP servers.Configuration for one or more IPv4 or IPv6 DHCP servers.Configuration for one or more IPv4 or IPv6 DHCP servers.The properties of DHCP Server and its children, such as DHCPv4, DHCPv6 servers.

        Returns: DeviceDhcpServer
        """
        return self._get_property("dhcp_server", DeviceDhcpServer)

    @property
    def ospfv2(self):
        # type: () -> DeviceOspfv2Router
        """ospfv2 getter

        Under Review: OSPFv2 is currently under review for pending exploration on use cases.. Under Review: OSPFv2 is currently under review for pending exploration on use cases.. A container of properties for an OSPFv2 router and its interfaces Route Ranges.Under Review: OSPFv2 is currently under review for pending exploration on use cases.. Under Review: OSPFv2 is currently under review for pending exploration on use cases.. A container of properties for an OSPFv2 router and its interfaces Route Ranges.Under Review: OSPFv2 is currently under review for pending exploration on use cases.. Under Review: OSPFv2 is currently under review for pending exploration on use cases.. A container of properties for an OSPFv2 router and its interfaces Route Ranges.Configuration for OSPFv2 router.

        Returns: DeviceOspfv2Router
        """
        return self._get_property("ospfv2", DeviceOspfv2Router)


class DeviceEthernet(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "connection": {"type": "EthernetConnection"},
        "ipv4_addresses": {"type": "DeviceIpv4Iter"},
        "ipv6_addresses": {"type": "DeviceIpv6Iter"},
        "mac": {
            "type": str,
            "format": "mac",
        },
        "mtu": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "vlans": {"type": "DeviceVlanIter"},
        "name": {"type": str},
        "dhcpv4_interfaces": {"type": "DeviceDhcpv4clientIter"},
        "dhcpv6_interfaces": {"type": "DeviceDhcpv6clientIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "mtu": 1500,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, mac=None, mtu=1500, name=None):
        super(DeviceEthernet, self).__init__()
        self._parent = parent
        self._set_property("mac", mac)
        self._set_property("mtu", mtu)
        self._set_property("name", name)

    def set(self, mac=None, mtu=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def connection(self):
        # type: () -> EthernetConnection
        """connection getter

        Ethernet interface connection to port, LAG, VXLAN tunnel or Simulated Internal Link used to create simulated topologies behind an emulated router.Ethernet interface connection to port, LAG, VXLAN tunnel or Simulated Internal Link used to create simulated topologies behind an emulated router.Ethernet interface connection to port, LAG, VXLAN tunnel or Simulated Internal Link used to create simulated topologies behind an emulated router.Device connection to physical, LAG or another device.

        Returns: EthernetConnection
        """
        return self._get_property("connection", EthernetConnection)

    @property
    def ipv4_addresses(self):
        # type: () -> DeviceIpv4Iter
        """ipv4_addresses getter

        List of IPv4 addresses and their gateways.

        Returns: DeviceIpv4Iter
        """
        return self._get_property(
            "ipv4_addresses", DeviceIpv4Iter, self._parent, self._choice
        )

    @property
    def ipv6_addresses(self):
        # type: () -> DeviceIpv6Iter
        """ipv6_addresses getter

        List of global IPv6 addresses and their gateways.. The Link Local IPv6 address will be automatically generated.

        Returns: DeviceIpv6Iter
        """
        return self._get_property(
            "ipv6_addresses", DeviceIpv6Iter, self._parent, self._choice
        )

    @property
    def mac(self):
        # type: () -> str
        """mac getter

        Media Access Control address.The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where 'mac' is not mandatory.

        Returns: str
        """
        return self._get_property("mac")

    @mac.setter
    def mac(self, value):
        """mac setter

        Media Access Control address.The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where 'mac' is not mandatory.

        value: str
        """
        self._set_property("mac", value)

    @property
    def mtu(self):
        # type: () -> int
        """mtu getter

        Maximum Transmission Unit.

        Returns: int
        """
        return self._get_property("mtu")

    @mtu.setter
    def mtu(self, value):
        """mtu setter

        Maximum Transmission Unit.

        value: int
        """
        self._set_property("mtu", value)

    @property
    def vlans(self):
        # type: () -> DeviceVlanIter
        """vlans getter

        List of VLANs

        Returns: DeviceVlanIter
        """
        return self._get_property("vlans", DeviceVlanIter, self._parent, self._choice)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def dhcpv4_interfaces(self):
        # type: () -> DeviceDhcpv4clientIter
        """dhcpv4_interfaces getter

        List of DHCPv4 Clients Configuration.

        Returns: DeviceDhcpv4clientIter
        """
        return self._get_property(
            "dhcpv4_interfaces", DeviceDhcpv4clientIter, self._parent, self._choice
        )

    @property
    def dhcpv6_interfaces(self):
        # type: () -> DeviceDhcpv6clientIter
        """dhcpv6_interfaces getter

        List of DHCPv6 Clients Configuration.

        Returns: DeviceDhcpv6clientIter
        """
        return self._get_property(
            "dhcpv6_interfaces", DeviceDhcpv6clientIter, self._parent, self._choice
        )


class EthernetConnection(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "port_name",
                "lag_name",
                "vxlan_name",
                "simulated_link",
            ],
        },
        "port_name": {"type": str},
        "lag_name": {"type": str},
        "vxlan_name": {"type": str},
        "simulated_link": {"type": "EthernetSimulatedLink"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    PORT_NAME = "port_name"  # type: str
    LAG_NAME = "lag_name"  # type: str
    VXLAN_NAME = "vxlan_name"  # type: str
    SIMULATED_LINK = "simulated_link"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, choice=None, port_name=None, lag_name=None, vxlan_name=None
    ):
        super(EthernetConnection, self).__init__()
        self._parent = parent
        self._set_property("port_name", port_name)
        self._set_property("lag_name", lag_name)
        self._set_property("vxlan_name", vxlan_name)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, port_name=None, lag_name=None, vxlan_name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def simulated_link(self):
        # type: () -> EthernetSimulatedLink
        """Factory property that returns an instance of the EthernetSimulatedLink class

        Details of the internal link which can be used to create simulated device topologies behind an emulated router. MAC, VLAN and MTU information for the internal links are not used for purposes of emulating Simulated Topologies e.g. by ISIS Emulated Router behind which this is configured )

        Returns: EthernetSimulatedLink
        """
        return self._get_property(
            "simulated_link", EthernetSimulatedLink, self, "simulated_link"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["lag_name"], Literal["port_name"], Literal["simulated_link"], Literal["vxlan_name"]]
        """choice getter

        port_name, lag_name, vxlan_name or simulated_link

        Returns: Union[Literal["lag_name"], Literal["port_name"], Literal["simulated_link"], Literal["vxlan_name"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        port_name, lag_name, vxlan_name or simulated_link

        value: Union[Literal["lag_name"], Literal["port_name"], Literal["simulated_link"], Literal["vxlan_name"]]
        """
        self._set_property("choice", value)

    @property
    def port_name(self):
        # type: () -> str
        """port_name getter

        Name of the port that the Ethernet interface is configured on.. x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        Returns: str
        """
        return self._get_property("port_name")

    @port_name.setter
    def port_name(self, value):
        """port_name setter

        Name of the port that the Ethernet interface is configured on.. x-constraint:. /components/schemas/Port/properties/name. . x-constraint:. /components/schemas/Port/properties/name.

        value: str
        """
        self._set_property("port_name", value, "port_name")

    @property
    def lag_name(self):
        # type: () -> str
        """lag_name getter

        Name of the LAG that the Ethernet interface is configured on.. x-constraint:. /components/schemas/Lag/properties/name. . x-constraint:. /components/schemas/Lag/properties/name.

        Returns: str
        """
        return self._get_property("lag_name")

    @lag_name.setter
    def lag_name(self, value):
        """lag_name setter

        Name of the LAG that the Ethernet interface is configured on.. x-constraint:. /components/schemas/Lag/properties/name. . x-constraint:. /components/schemas/Lag/properties/name.

        value: str
        """
        self._set_property("lag_name", value, "lag_name")

    @property
    def vxlan_name(self):
        # type: () -> str
        """vxlan_name getter

        Name of the VXLAN instance (or VXLAN tunnel) that this Ethernet interface is connected to.. x-constraint:. #/components/schemas/Vxlan.V4Tunnel/properties/name. #/components/schemas/Vxlan.V6Tunnel/properties/name. . x-constraint:. #/components/schemas/Vxlan.V4Tunnel/properties/name. #/components/schemas/Vxlan.V6Tunnel/properties/name.

        Returns: str
        """
        return self._get_property("vxlan_name")

    @vxlan_name.setter
    def vxlan_name(self, value):
        """vxlan_name setter

        Name of the VXLAN instance (or VXLAN tunnel) that this Ethernet interface is connected to.. x-constraint:. #/components/schemas/Vxlan.V4Tunnel/properties/name. #/components/schemas/Vxlan.V6Tunnel/properties/name. . x-constraint:. #/components/schemas/Vxlan.V4Tunnel/properties/name. #/components/schemas/Vxlan.V6Tunnel/properties/name.

        value: str
        """
        self._set_property("vxlan_name", value, "vxlan_name")


class EthernetSimulatedLink(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_simulated_link": {"type": str},
        "link_type": {
            "type": str,
            "enum": [
                "primary",
                "secondary",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "link_type": "primary",
    }  # type: Dict[str, Union(type)]

    PRIMARY = "primary"  # type: str
    SECONDARY = "secondary"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, remote_simulated_link=None, link_type="primary"):
        super(EthernetSimulatedLink, self).__init__()
        self._parent = parent
        self._set_property("remote_simulated_link", remote_simulated_link)
        self._set_property("link_type", link_type)

    def set(self, remote_simulated_link=None, link_type=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def remote_simulated_link(self):
        # type: () -> str
        """remote_simulated_link getter

        Name of the remote end of the simulated interface which also must be simulated_link on device which might be acting either as an unconnected device in simulated topology ( all ethernet links of type simulated_link or an emulated device connected to the Device Under Test (has at atleast one ethernet interface with connection to the port or. lag connected to the DUT). x-constraint:. #/components/schemas/Device.Ethernet/properties/name. . x-constraint:. #/components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("remote_simulated_link")

    @remote_simulated_link.setter
    def remote_simulated_link(self, value):
        """remote_simulated_link setter

        Name of the remote end of the simulated interface which also must be simulated_link on device which might be acting either as an unconnected device in simulated topology ( all ethernet links of type simulated_link or an emulated device connected to the Device Under Test (has at atleast one ethernet interface with connection to the port or. lag connected to the DUT). x-constraint:. #/components/schemas/Device.Ethernet/properties/name. . x-constraint:. #/components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        self._set_property("remote_simulated_link", value)

    @property
    def link_type(self):
        # type: () -> Union[Literal["primary"], Literal["secondary"]]
        """link_type getter

        By default, simulated links are treated as Primary links which means that the intention is for connected device to advertise this and full topology of devices connected to it.. e.g. when advertised as ISIS Simulated Topology.. All simulated links inside one topology subset would normally can point to only other unconnected devices in the same topology or to the 'root' emulated device.. If link is designated as secondary only that link information will be advertised by the IGP e.g. ISIS and not the entire topology behind it.. The optional secondary option allows emulation of external link scenarios where simulated device (e.g. part of ISIS simulated topology is advertised as reachable part of the topology by the emulated router behind which this is configured as well as the other end of the secondary link which could be - 1) either simulated device behind different emulated router.. 2) or an emulated router on same or different port.. This allows emulation of scenarios where one device/router is emulated to be reachable from different Emulated Routers connected to the Device Under Test. (e.g. for FRR scenarios). If an implementation does not support multiple primary links from same simulated topology i.e. full topology advertised via multiple emulated routers, it should return an error during set_config operation with such topology.

        Returns: Union[Literal["primary"], Literal["secondary"]]
        """
        return self._get_property("link_type")

    @link_type.setter
    def link_type(self, value):
        """link_type setter

        By default, simulated links are treated as Primary links which means that the intention is for connected device to advertise this and full topology of devices connected to it.. e.g. when advertised as ISIS Simulated Topology.. All simulated links inside one topology subset would normally can point to only other unconnected devices in the same topology or to the 'root' emulated device.. If link is designated as secondary only that link information will be advertised by the IGP e.g. ISIS and not the entire topology behind it.. The optional secondary option allows emulation of external link scenarios where simulated device (e.g. part of ISIS simulated topology is advertised as reachable part of the topology by the emulated router behind which this is configured as well as the other end of the secondary link which could be - 1) either simulated device behind different emulated router.. 2) or an emulated router on same or different port.. This allows emulation of scenarios where one device/router is emulated to be reachable from different Emulated Routers connected to the Device Under Test. (e.g. for FRR scenarios). If an implementation does not support multiple primary links from same simulated topology i.e. full topology advertised via multiple emulated routers, it should return an error during set_config operation with such topology.

        value: Union[Literal["primary"], Literal["secondary"]]
        """
        self._set_property("link_type", value)


class DeviceIpv4(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "gateway": {
            "type": str,
            "format": "ipv4",
        },
        "gateway_mac": {"type": "DeviceIpv4GatewayMAC"},
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "prefix": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 32,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("gateway", "address", "name")  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 24,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, gateway=None, address=None, prefix=24, name=None):
        super(DeviceIpv4, self).__init__()
        self._parent = parent
        self._set_property("gateway", gateway)
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("name", name)

    def set(self, gateway=None, address=None, prefix=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv4 address of the gateway

        Returns: str
        """
        return self._get_property("gateway")

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv4 address of the gateway

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property gateway as None")
        self._set_property("gateway", value)

    @property
    def gateway_mac(self):
        # type: () -> DeviceIpv4GatewayMAC
        """gateway_mac getter

        By default auto(resolved gateway mac) is set. Setting value would mean that ARP will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv4 endpoint. whenever applicable.By default auto(resolved gateway mac) is set. Setting value would mean that ARP will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv4 endpoint. whenever applicable.By default auto(resolved gateway mac) is set. Setting value would mean that ARP will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv4 endpoint. whenever applicable.

        Returns: DeviceIpv4GatewayMAC
        """
        return self._get_property("gateway_mac", DeviceIpv4GatewayMAC)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv4 address

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv4 address

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property address as None")
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The prefix of the IPv4 address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The prefix of the IPv4 address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceIpv4GatewayMAC(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "auto",
                "value",
            ],
        },
        "auto": {
            "type": str,
            "format": "mac",
        },
        "value": {
            "type": str,
            "format": "mac",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "auto": "00:00:00:00:00:00",
        "value": "00:00:00:00:00:00",
    }  # type: Dict[str, Union(type)]

    AUTO = "auto"  # type: str
    VALUE = "value"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        auto="00:00:00:00:00:00",
        value="00:00:00:00:00:00",
    ):
        super(DeviceIpv4GatewayMAC, self).__init__()
        self._parent = parent
        self._set_property("auto", auto)
        self._set_property("value", value)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, auto=None, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["value"]]
        """choice getter

        auto or configured value.

        Returns: Union[Literal["auto"], Literal["value"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        auto or configured value.

        value: Union[Literal["auto"], Literal["value"]]
        """
        self._set_property("choice", value)

    @property
    def auto(self):
        # type: () -> str
        """auto getter

        The OTG implementation can provide system generated value for this property. If the OTG is unable to generate value the default value must be used.

        Returns: str
        """
        return self._get_property("auto")

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")


class DeviceIpv4Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv4Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv4]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv4Iter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv4
        return self._next()

    def next(self):
        # type: () -> DeviceIpv4
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceIpv4):
            raise Exception("Item is not an instance of DeviceIpv4")

    def ipv4(self, gateway=None, address=None, prefix=24, name=None):
        # type: (str,str,int,str) -> DeviceIpv4Iter
        """Factory method that creates an instance of the DeviceIpv4 class

        An IPv4 interface with gateway

        Returns: DeviceIpv4Iter
        """
        item = DeviceIpv4(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return self

    def add(self, gateway=None, address=None, prefix=24, name=None):
        # type: (str,str,int,str) -> DeviceIpv4
        """Add method that creates and returns an instance of the DeviceIpv4 class

        An IPv4 interface with gateway

        Returns: DeviceIpv4
        """
        item = DeviceIpv4(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return item


class DeviceIpv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "gateway": {
            "type": str,
            "format": "ipv6",
        },
        "gateway_mac": {"type": "DeviceIpv6GatewayMAC"},
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "prefix": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 128,
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("gateway", "address", "name")  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 64,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, gateway=None, address=None, prefix=64, name=None):
        super(DeviceIpv6, self).__init__()
        self._parent = parent
        self._set_property("gateway", gateway)
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("name", name)

    def set(self, gateway=None, address=None, prefix=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def gateway(self):
        # type: () -> str
        """gateway getter

        The IPv6 gateway address.

        Returns: str
        """
        return self._get_property("gateway")

    @gateway.setter
    def gateway(self, value):
        """gateway setter

        The IPv6 gateway address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property gateway as None")
        self._set_property("gateway", value)

    @property
    def gateway_mac(self):
        # type: () -> DeviceIpv6GatewayMAC
        """gateway_mac getter

        By default auto(resolved gateway mac) is set. Setting value would mean that ND will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv6 endpoint. whenever applicable.By default auto(resolved gateway mac) is set. Setting value would mean that ND will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv6 endpoint. whenever applicable.By default auto(resolved gateway mac) is set. Setting value would mean that ND will not be used for learning MAC of connected device. The user-configured MAC address will be used for auto-filling the destination. MAC address in the control and data packets sent from this IPv6 endpoint. whenever applicable.

        Returns: DeviceIpv6GatewayMAC
        """
        return self._get_property("gateway_mac", DeviceIpv6GatewayMAC)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv6 address.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv6 address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property address as None")
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The network prefix.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The network prefix.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceIpv6GatewayMAC(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "auto",
                "value",
            ],
        },
        "auto": {
            "type": str,
            "format": "mac",
        },
        "value": {
            "type": str,
            "format": "mac",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "auto",
        "auto": "00:00:00:00:00:00",
        "value": "00:00:00:00:00:00",
    }  # type: Dict[str, Union(type)]

    AUTO = "auto"  # type: str
    VALUE = "value"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        auto="00:00:00:00:00:00",
        value="00:00:00:00:00:00",
    ):
        super(DeviceIpv6GatewayMAC, self).__init__()
        self._parent = parent
        self._set_property("auto", auto)
        self._set_property("value", value)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, auto=None, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["auto"], Literal["value"]]
        """choice getter

        auto or configured value.

        Returns: Union[Literal["auto"], Literal["value"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        auto or configured value.

        value: Union[Literal["auto"], Literal["value"]]
        """
        self._set_property("choice", value)

    @property
    def auto(self):
        # type: () -> str
        """auto getter

        The OTG implementation can provide system generated value for this property. If the OTG is unable to generate value the default value must be used.

        Returns: str
        """
        return self._get_property("auto")

    @property
    def value(self):
        # type: () -> str
        """value getter

        TBD

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        TBD

        value: str
        """
        self._set_property("value", value, "value")


class DeviceIpv6Iter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv6Iter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv6]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv6Iter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv6
        return self._next()

    def next(self):
        # type: () -> DeviceIpv6
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceIpv6):
            raise Exception("Item is not an instance of DeviceIpv6")

    def ipv6(self, gateway=None, address=None, prefix=64, name=None):
        # type: (str,str,int,str) -> DeviceIpv6Iter
        """Factory method that creates an instance of the DeviceIpv6 class

        An IPv6 interface with gateway.

        Returns: DeviceIpv6Iter
        """
        item = DeviceIpv6(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return self

    def add(self, gateway=None, address=None, prefix=64, name=None):
        # type: (str,str,int,str) -> DeviceIpv6
        """Add method that creates and returns an instance of the DeviceIpv6 class

        An IPv6 interface with gateway.

        Returns: DeviceIpv6
        """
        item = DeviceIpv6(
            parent=self._parent,
            gateway=gateway,
            address=address,
            prefix=prefix,
            name=name,
        )
        self._add(item)
        return item


class DeviceDhcpv4client(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "name": {"type": str},
        "choice": {
            "type": str,
            "enum": [
                "first_server",
                "server_address",
            ],
        },
        "server_address": {
            "type": str,
            "format": "ipv4",
        },
        "broadcast": {"type": bool},
        "parameters_request_list": {"type": "Dhcpv4ClientParams"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "choice": "first_server",
        "broadcast": False,
    }  # type: Dict[str, Union(type)]

    FIRST_SERVER = "first_server"  # type: str
    SERVER_ADDRESS = "server_address"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, choice=None, name=None, server_address=None, broadcast=False
    ):
        super(DeviceDhcpv4client, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("server_address", server_address)
        self._set_property("broadcast", broadcast)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, name=None, server_address=None, broadcast=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def choice(self):
        # type: () -> Union[Literal["first_server"], Literal["server_address"]]
        """choice getter

        The client receives one or more DHCPOFFER messages from one or more servers and client may choose to wait for multiple responses.. The client chooses one server from which to request configuration. parameters, based on the configuration parameters offered in the DHCPOFFER messages.. first_server: if selected, the subnet accepts the IP addresses offered by the first server to respond with an offer of IP addresses.. server_address: The address of the DHCP server from which the subnet will accept IP addresses.

        Returns: Union[Literal["first_server"], Literal["server_address"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The client receives one or more DHCPOFFER messages from one or more servers and client may choose to wait for multiple responses.. The client chooses one server from which to request configuration. parameters, based on the configuration parameters offered in the DHCPOFFER messages.. first_server: if selected, the subnet accepts the IP addresses offered by the first server to respond with an offer of IP addresses.. server_address: The address of the DHCP server from which the subnet will accept IP addresses.

        value: Union[Literal["first_server"], Literal["server_address"]]
        """
        self._set_property("choice", value)

    @property
    def server_address(self):
        # type: () -> str
        """server_address getter

        The address of the DHCP server.

        Returns: str
        """
        return self._get_property("server_address")

    @server_address.setter
    def server_address(self, value):
        """server_address setter

        The address of the DHCP server.

        value: str
        """
        self._set_property("server_address", value, "server_address")

    @property
    def broadcast(self):
        # type: () -> bool
        """broadcast getter

        If the broadcast bit is set, then the server and relay agent broadcast DHCPOFFER and DHCPACK messages.

        Returns: bool
        """
        return self._get_property("broadcast")

    @broadcast.setter
    def broadcast(self, value):
        """broadcast setter

        If the broadcast bit is set, then the server and relay agent broadcast DHCPOFFER and DHCPACK messages.

        value: bool
        """
        self._set_property("broadcast", value)

    @property
    def parameters_request_list(self):
        # type: () -> Dhcpv4ClientParams
        """parameters_request_list getter

        Configuration Parameter request list by emulated DHCPv4 Client.Configuration Parameter request list by emulated DHCPv4 Client.Configuration Parameter request list by emulated DHCPv4 Client.Optional parameters field request list of DHCPv4 Client.

        Returns: Dhcpv4ClientParams
        """
        return self._get_property("parameters_request_list", Dhcpv4ClientParams)


class Dhcpv4ClientParams(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "subnet_mask": {"type": bool},
        "router": {"type": bool},
        "renewal_timer": {"type": bool},
        "rebinding_timer": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "subnet_mask": True,
        "router": True,
        "renewal_timer": False,
        "rebinding_timer": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        subnet_mask=True,
        router=True,
        renewal_timer=False,
        rebinding_timer=False,
    ):
        super(Dhcpv4ClientParams, self).__init__()
        self._parent = parent
        self._set_property("subnet_mask", subnet_mask)
        self._set_property("router", router)
        self._set_property("renewal_timer", renewal_timer)
        self._set_property("rebinding_timer", rebinding_timer)

    def set(
        self, subnet_mask=None, router=None, renewal_timer=None, rebinding_timer=None
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def subnet_mask(self):
        # type: () -> bool
        """subnet_mask getter

        Request for the subnet mask option specifies the client's subnet mask as per RFC950.

        Returns: bool
        """
        return self._get_property("subnet_mask")

    @subnet_mask.setter
    def subnet_mask(self, value):
        """subnet_mask setter

        Request for the subnet mask option specifies the client's subnet mask as per RFC950.

        value: bool
        """
        self._set_property("subnet_mask", value)

    @property
    def router(self):
        # type: () -> bool
        """router getter

        Request for the router option that specifies list of IP addresses for routers on the client's subnet.

        Returns: bool
        """
        return self._get_property("router")

    @router.setter
    def router(self, value):
        """router setter

        Request for the router option that specifies list of IP addresses for routers on the client's subnet.

        value: bool
        """
        self._set_property("router", value)

    @property
    def renewal_timer(self):
        # type: () -> bool
        """renewal_timer getter

        Request for the renewal timer, T1. When the timer expires, the client transitions from the BOUND state to the RENEWING state.

        Returns: bool
        """
        return self._get_property("renewal_timer")

    @renewal_timer.setter
    def renewal_timer(self, value):
        """renewal_timer setter

        Request for the renewal timer, T1. When the timer expires, the client transitions from the BOUND state to the RENEWING state.

        value: bool
        """
        self._set_property("renewal_timer", value)

    @property
    def rebinding_timer(self):
        # type: () -> bool
        """rebinding_timer getter

        Request for the rebinding timer (T2). When expires, the client transitions to the REBINDING state.

        Returns: bool
        """
        return self._get_property("rebinding_timer")

    @rebinding_timer.setter
    def rebinding_timer(self, value):
        """rebinding_timer setter

        Request for the rebinding timer (T2). When expires, the client transitions to the REBINDING state.

        value: bool
        """
        self._set_property("rebinding_timer", value)


class DeviceDhcpv4clientIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceDhcpv4clientIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceDhcpv4client]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceDhcpv4clientIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceDhcpv4client
        return self._next()

    def next(self):
        # type: () -> DeviceDhcpv4client
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceDhcpv4client):
            raise Exception("Item is not an instance of DeviceDhcpv4client")

    def dhcpv4client(self, name=None, server_address=None, broadcast=False):
        # type: (str,str,bool) -> DeviceDhcpv4clientIter
        """Factory method that creates an instance of the DeviceDhcpv4client class

        Configuration for emulated DHCPv4 Client on single Interface. https://www.rfc-editor.org/rfc/rfc2131.html

        Returns: DeviceDhcpv4clientIter
        """
        item = DeviceDhcpv4client(
            parent=self._parent,
            choice=self._choice,
            name=name,
            server_address=server_address,
            broadcast=broadcast,
        )
        self._add(item)
        return self

    def add(self, name=None, server_address=None, broadcast=False):
        # type: (str,str,bool) -> DeviceDhcpv4client
        """Add method that creates and returns an instance of the DeviceDhcpv4client class

        Configuration for emulated DHCPv4 Client on single Interface. https://www.rfc-editor.org/rfc/rfc2131.html

        Returns: DeviceDhcpv4client
        """
        item = DeviceDhcpv4client(
            parent=self._parent,
            choice=self._choice,
            name=name,
            server_address=server_address,
            broadcast=broadcast,
        )
        self._add(item)
        return item


class DeviceDhcpv6client(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "name": {"type": str},
        "rapid_commit": {"type": bool},
        "ia_type": {"type": "DeviceDhcpv6clientIaType"},
        "duid_type": {"type": "DeviceDhcpv6clientDuidType"},
        "options_request": {"type": "DeviceDhcpv6ClientOptionsRequest"},
        "options": {"type": "DeviceDhcpv6ClientOptions"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "rapid_commit": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None, rapid_commit=False):
        super(DeviceDhcpv6client, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("rapid_commit", rapid_commit)

    def set(self, name=None, rapid_commit=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def rapid_commit(self):
        # type: () -> bool
        """rapid_commit getter

        If Rapid Commit is set, client initiates Rapid Commit two-message exchange by including Rapid Commit option in Solicit message.

        Returns: bool
        """
        return self._get_property("rapid_commit")

    @rapid_commit.setter
    def rapid_commit(self, value):
        """rapid_commit setter

        If Rapid Commit is set, client initiates Rapid Commit two-message exchange by including Rapid Commit option in Solicit message.

        value: bool
        """
        self._set_property("rapid_commit", value)

    @property
    def ia_type(self):
        # type: () -> DeviceDhcpv6clientIaType
        """ia_type getter

        Each IA has an associated IAID. Differnet IA options represent different types of IPv6 addresses and parameters accepted by DHCPv6 clients each used in different context by an IPv6 node.

        Returns: DeviceDhcpv6clientIaType
        """
        return self._get_property("ia_type", DeviceDhcpv6clientIaType)

    @property
    def duid_type(self):
        # type: () -> DeviceDhcpv6clientDuidType
        """duid_type getter

        Each DHCP client and server has DUID. DHCP clients and servers use DUIDs to identify each other.

        Returns: DeviceDhcpv6clientDuidType
        """
        return self._get_property("duid_type", DeviceDhcpv6clientDuidType)

    @property
    def options_request(self):
        # type: () -> DeviceDhcpv6ClientOptionsRequest
        """options_request getter

        DHCP client options, these configured options are sent in Dhcp client messages.DHCP client options, these configured options are sent in Dhcp client messages.DHCP client options, these configured options are sent in Dhcp client messages.The options requested by client from server in the options request option.

        Returns: DeviceDhcpv6ClientOptionsRequest
        """
        return self._get_property("options_request", DeviceDhcpv6ClientOptionsRequest)

    @property
    def options(self):
        # type: () -> DeviceDhcpv6ClientOptions
        """options getter

        DHCP client options, these configured options are sent in Dhcp client messages.DHCP client options, these configured options are sent in Dhcp client messages.DHCP client options, these configured options are sent in Dhcp client messages.Optional DHCPv4 Client options that are sent in Dhcp client messages.

        Returns: DeviceDhcpv6ClientOptions
        """
        return self._get_property("options", DeviceDhcpv6ClientOptions)


class DeviceDhcpv6clientIaType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "iana",
                "iata",
                "iapd",
                "ianapd",
            ],
        },
        "iana": {"type": "DeviceDhcpv6clientIaTimeValue"},
        "iapd": {"type": "DeviceDhcpv6clientIaTimeValue"},
        "ianapd": {"type": "DeviceDhcpv6clientIaTimeValue"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "iana",
    }  # type: Dict[str, Union(type)]

    IANA = "iana"  # type: str
    IATA = "iata"  # type: str
    IAPD = "iapd"  # type: str
    IANAPD = "ianapd"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(DeviceDhcpv6clientIaType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def iana(self):
        # type: () -> DeviceDhcpv6clientIaTimeValue
        """Factory property that returns an instance of the DeviceDhcpv6clientIaTimeValue class

        The container for the suggested times at which the client contacts the server or any available server.

        Returns: DeviceDhcpv6clientIaTimeValue
        """
        return self._get_property("iana", DeviceDhcpv6clientIaTimeValue, self, "iana")

    @property
    def iapd(self):
        # type: () -> DeviceDhcpv6clientIaTimeValue
        """Factory property that returns an instance of the DeviceDhcpv6clientIaTimeValue class

        The container for the suggested times at which the client contacts the server or any available server.

        Returns: DeviceDhcpv6clientIaTimeValue
        """
        return self._get_property("iapd", DeviceDhcpv6clientIaTimeValue, self, "iapd")

    @property
    def ianapd(self):
        # type: () -> DeviceDhcpv6clientIaTimeValue
        """Factory property that returns an instance of the DeviceDhcpv6clientIaTimeValue class

        The container for the suggested times at which the client contacts the server or any available server.

        Returns: DeviceDhcpv6clientIaTimeValue
        """
        return self._get_property(
            "ianapd", DeviceDhcpv6clientIaTimeValue, self, "ianapd"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["iana"], Literal["ianapd"], Literal["iapd"], Literal["iata"]]
        """choice getter

        Identity Association: Each IA has an associated IAID. IA_NA and IA_TA options represent different types of IPv6 addresses and parameters accepted by DHCPv6 clients each used in different context by an IPv6 node. IA_NA is the Identity Association for Non-temporary Addresses option. IA_TA is the Identity Association for Temporary Addresses option. IA_PD and IA_NAPD options represent one or more IPv6 prefix and parameters. IA_PD is the Identity Association for Prefix Delegation and IA_NAPD the Identity Association for Temporary Prefix Delegation.

        Returns: Union[Literal["iana"], Literal["ianapd"], Literal["iapd"], Literal["iata"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Identity Association: Each IA has an associated IAID. IA_NA and IA_TA options represent different types of IPv6 addresses and parameters accepted by DHCPv6 clients each used in different context by an IPv6 node. IA_NA is the Identity Association for Non-temporary Addresses option. IA_TA is the Identity Association for Temporary Addresses option. IA_PD and IA_NAPD options represent one or more IPv6 prefix and parameters. IA_PD is the Identity Association for Prefix Delegation and IA_NAPD the Identity Association for Temporary Prefix Delegation.

        value: Union[Literal["iana"], Literal["ianapd"], Literal["iapd"], Literal["iata"]]
        """
        self._set_property("choice", value)


class DeviceDhcpv6clientIaTimeValue(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "t1": {
            "type": int,
            "format": "uint32",
            "maximum": 4294967295,
        },
        "t2": {
            "type": int,
            "format": "uint32",
            "maximum": 4294967295,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "t1": 302400,
        "t2": 483840,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, t1=302400, t2=483840):
        super(DeviceDhcpv6clientIaTimeValue, self).__init__()
        self._parent = parent
        self._set_property("t1", t1)
        self._set_property("t2", t2)

    def set(self, t1=None, t2=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def t1(self):
        # type: () -> int
        """t1 getter

        The suggested time at which the client contacts the server from which the addresses were obtained to extend the lifetimes of the addresses assigned. T1 is time duration relative to the current time expressed in units of seconds. If set to server will ignore it. If the maximum value is specified it means infinite time.

        Returns: int
        """
        return self._get_property("t1")

    @t1.setter
    def t1(self, value):
        """t1 setter

        The suggested time at which the client contacts the server from which the addresses were obtained to extend the lifetimes of the addresses assigned. T1 is time duration relative to the current time expressed in units of seconds. If set to server will ignore it. If the maximum value is specified it means infinite time.

        value: int
        """
        self._set_property("t1", value)

    @property
    def t2(self):
        # type: () -> int
        """t2 getter

        The suggested time at which the client contacts any available server to extend the lifetimes of the addresses assigned. T2 is time duration relative to the current time expressed in units of seconds. If set to server will ignore it. If the maximum value is specified it means infinite time

        Returns: int
        """
        return self._get_property("t2")

    @t2.setter
    def t2(self, value):
        """t2 setter

        The suggested time at which the client contacts any available server to extend the lifetimes of the addresses assigned. T2 is time duration relative to the current time expressed in units of seconds. If set to server will ignore it. If the maximum value is specified it means infinite time

        value: int
        """
        self._set_property("t2", value)


class DeviceDhcpv6clientDuidType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "llt",
                "en",
                "ll",
            ],
        },
        "llt": {"type": "DeviceDhcpv6clientNoDuid"},
        "en": {"type": "DeviceDhcpv6clientDuidValue"},
        "ll": {"type": "DeviceDhcpv6clientNoDuid"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "llt",
    }  # type: Dict[str, Union(type)]

    LLT = "llt"  # type: str
    EN = "en"  # type: str
    LL = "ll"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(DeviceDhcpv6clientDuidType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def llt(self):
        # type: () -> DeviceDhcpv6clientNoDuid
        """Factory property that returns an instance of the DeviceDhcpv6clientNoDuid class

        The container for DUID-LL and DUID-LLT.

        Returns: DeviceDhcpv6clientNoDuid
        """
        return self._get_property("llt", DeviceDhcpv6clientNoDuid, self, "llt")

    @property
    def en(self):
        # type: () -> DeviceDhcpv6clientDuidValue
        """Factory property that returns an instance of the DeviceDhcpv6clientDuidValue class

        The container for the DUID-EN. This consists of the 4-octet vendor's registered Private Enterprise Number as maintained by IANA [IANA-PEN] followed by unique identifier assigned by the vendor.

        Returns: DeviceDhcpv6clientDuidValue
        """
        return self._get_property("en", DeviceDhcpv6clientDuidValue, self, "en")

    @property
    def ll(self):
        # type: () -> DeviceDhcpv6clientNoDuid
        """Factory property that returns an instance of the DeviceDhcpv6clientNoDuid class

        The container for DUID-LL and DUID-LLT.

        Returns: DeviceDhcpv6clientNoDuid
        """
        return self._get_property("ll", DeviceDhcpv6clientNoDuid, self, "ll")

    @property
    def choice(self):
        # type: () -> Union[Literal["en"], Literal["ll"], Literal["llt"]]
        """choice getter

        Each DHCP client and server has DUID. DHCP clients use DUIDs to identify server in messages where server needs to be identified.

        Returns: Union[Literal["en"], Literal["ll"], Literal["llt"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Each DHCP client and server has DUID. DHCP clients use DUIDs to identify server in messages where server needs to be identified.

        value: Union[Literal["en"], Literal["ll"], Literal["llt"]]
        """
        self._set_property("choice", value)


class DeviceDhcpv6clientNoDuid(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {}  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(DeviceDhcpv6clientNoDuid, self).__init__()
        self._parent = parent


class DeviceDhcpv6clientDuidValue(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enterprise_id": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 2147483647,
        },
        "vendor_id": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 2147483647,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enterprise_id": 10,
        "vendor_id": 10,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, enterprise_id=10, vendor_id=10):
        super(DeviceDhcpv6clientDuidValue, self).__init__()
        self._parent = parent
        self._set_property("enterprise_id", enterprise_id)
        self._set_property("vendor_id", vendor_id)

    def set(self, enterprise_id=None, vendor_id=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enterprise_id(self):
        # type: () -> int
        """enterprise_id getter

        4-octet vendor's registered Private Enterprise Number as maintained by IANA [IANA-PEN].

        Returns: int
        """
        return self._get_property("enterprise_id")

    @enterprise_id.setter
    def enterprise_id(self, value):
        """enterprise_id setter

        4-octet vendor's registered Private Enterprise Number as maintained by IANA [IANA-PEN].

        value: int
        """
        self._set_property("enterprise_id", value)

    @property
    def vendor_id(self):
        # type: () -> int
        """vendor_id getter

        Unique identifier assigned by the vendor.

        Returns: int
        """
        return self._get_property("vendor_id")

    @vendor_id.setter
    def vendor_id(self, value):
        """vendor_id setter

        Unique identifier assigned by the vendor.

        value: int
        """
        self._set_property("vendor_id", value)


class DeviceDhcpv6ClientOptionsRequest(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "request": {"type": "Dhcpv6ClientOptionsOptionsRequestIter"},
        "associated_dhcp_messages": {"type": "Dhcpv6ClientOptionsIncludedMessages"},
    }  # type: Dict[str, str]

    _REQUIRED = ("request", "associated_dhcp_messages")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(DeviceDhcpv6ClientOptionsRequest, self).__init__()
        self._parent = parent

    @property
    def request(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequestIter
        """request getter

        List of options requested by client from server.

        Returns: Dhcpv6ClientOptionsOptionsRequestIter
        """
        return self._get_property(
            "request", Dhcpv6ClientOptionsOptionsRequestIter, self._parent, self._choice
        )

    @property
    def associated_dhcp_messages(self):
        # type: () -> Dhcpv6ClientOptionsIncludedMessages
        """associated_dhcp_messages getter

        The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The list of dhcpv6 client messages where this option is included.

        Returns: Dhcpv6ClientOptionsIncludedMessages
        """
        return self._get_property(
            "associated_dhcp_messages", Dhcpv6ClientOptionsIncludedMessages
        )


class Dhcpv6ClientOptionsOptionsRequest(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "vendor_information",
                "name_servers",
                "fqdn",
                "bootfile_url",
                "sztp",
                "custom",
            ],
        },
        "custom": {"type": "Dhcpv6ClientOptionsCustom"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "vendor_information",
    }  # type: Dict[str, Union(type)]

    VENDOR_INFORMATION = "vendor_information"  # type: str
    NAME_SERVERS = "name_servers"  # type: str
    FQDN = "fqdn"  # type: str
    BOOTFILE_URL = "bootfile_url"  # type: str
    SZTP = "sztp"  # type: str
    CUSTOM = "custom"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsOptionsRequest, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def custom(self):
        # type: () -> Dhcpv6ClientOptionsCustom
        """Factory property that returns an instance of the Dhcpv6ClientOptionsCustom class

        The Custom option is used to provide not so well known option in the message between client and server.

        Returns: Dhcpv6ClientOptionsCustom
        """
        return self._get_property("custom", Dhcpv6ClientOptionsCustom, self, "custom")

    @property
    def choice(self):
        # type: () -> Union[Literal["bootfile_url"], Literal["custom"], Literal["fqdn"], Literal["name_servers"], Literal["sztp"], Literal["vendor_information"]]
        """choice getter

        The Option Request option is used to identify list of options in message between client and server. The option code is 6. Vendor_specific information option, requested by clients for vendor-specific informations from servers. DNS Recursive Name Server Option, requested by clients to get the list ofIPv6 addresses of DNS recursive name servers to which DNS queries may be sent by the client resolver in order of preference.. Client FQDN option indicates whether the client or the DHCP server should update DNS with the AAAA record corresponding to the assigned IPv6 address and the FQDN provided in this option. The DHCP server always updates the PTR record.. bootfile_url, if client is configured for network booting then the client must use this option to obtain the boot file url from the server.. sztp. Securely provision networking device when it is booting in factory-default state.

        Returns: Union[Literal["bootfile_url"], Literal["custom"], Literal["fqdn"], Literal["name_servers"], Literal["sztp"], Literal["vendor_information"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The Option Request option is used to identify list of options in message between client and server. The option code is 6. Vendor_specific information option, requested by clients for vendor-specific informations from servers. DNS Recursive Name Server Option, requested by clients to get the list ofIPv6 addresses of DNS recursive name servers to which DNS queries may be sent by the client resolver in order of preference.. Client FQDN option indicates whether the client or the DHCP server should update DNS with the AAAA record corresponding to the assigned IPv6 address and the FQDN provided in this option. The DHCP server always updates the PTR record.. bootfile_url, if client is configured for network booting then the client must use this option to obtain the boot file url from the server.. sztp. Securely provision networking device when it is booting in factory-default state.

        value: Union[Literal["bootfile_url"], Literal["custom"], Literal["fqdn"], Literal["name_servers"], Literal["sztp"], Literal["vendor_information"]]
        """
        self._set_property("choice", value)


class Dhcpv6ClientOptionsCustom(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("type",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type=None):
        super(Dhcpv6ClientOptionsCustom, self).__init__()
        self._parent = parent
        self._set_property("type", type)

    def set(self, type=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> int
        """type getter

        The type of the Custom option TLV.

        Returns: int
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        The type of the Custom option TLV.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property type as None")
        self._set_property("type", value)


class Dhcpv6ClientOptionsOptionsRequestIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsOptionsRequestIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Dhcpv6ClientOptionsCustom, Dhcpv6ClientOptionsOptionsRequest]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequestIter
        return self._iter()

    def __next__(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequest
        return self._next()

    def next(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequest
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Dhcpv6ClientOptionsOptionsRequest):
            raise Exception(
                "Item is not an instance of Dhcpv6ClientOptionsOptionsRequest"
            )

    def optionsrequest(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequestIter
        """Factory method that creates an instance of the Dhcpv6ClientOptionsOptionsRequest class

        TBD

        Returns: Dhcpv6ClientOptionsOptionsRequestIter
        """
        item = Dhcpv6ClientOptionsOptionsRequest(
            parent=self._parent, choice=self._choice
        )
        self._add(item)
        return self

    def add(self):
        # type: () -> Dhcpv6ClientOptionsOptionsRequest
        """Add method that creates and returns an instance of the Dhcpv6ClientOptionsOptionsRequest class

        TBD

        Returns: Dhcpv6ClientOptionsOptionsRequest
        """
        item = Dhcpv6ClientOptionsOptionsRequest(
            parent=self._parent, choice=self._choice
        )
        self._add(item)
        return item

    def custom(self, type=None):
        # type: (int) -> Dhcpv6ClientOptionsOptionsRequestIter
        """Factory method that creates an instance of the Dhcpv6ClientOptionsCustom class

        The Custom option is used to provide not so well known option in the message between client and server.

        Returns: Dhcpv6ClientOptionsOptionsRequestIter
        """
        item = Dhcpv6ClientOptionsOptionsRequest()
        item.custom
        item.choice = "custom"
        self._add(item)
        return self


class Dhcpv6ClientOptionsIncludedMessages(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "all",
                "msg_types",
            ],
        },
        "msg_types": {"type": "Dhcpv6ClientOptionsMessageTypeIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "all",
    }  # type: Dict[str, Union(type)]

    ALL = "all"  # type: str
    MSG_TYPES = "msg_types"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsIncludedMessages, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def choice(self):
        # type: () -> Union[Literal["all"], Literal["msg_types"]]
        """choice getter

        The client message name where the option is included, by default it is all.

        Returns: Union[Literal["all"], Literal["msg_types"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The client message name where the option is included, by default it is all.

        value: Union[Literal["all"], Literal["msg_types"]]
        """
        self._set_property("choice", value)

    @property
    def msg_types(self):
        # type: () -> Dhcpv6ClientOptionsMessageTypeIter
        """msg_types getter

        User must specify the Dhcpv6 message type.

        Returns: Dhcpv6ClientOptionsMessageTypeIter
        """
        return self._get_property(
            "msg_types", Dhcpv6ClientOptionsMessageTypeIter, self._parent, self._choice
        )


class Dhcpv6ClientOptionsMessageType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "solicit",
                "request",
                "inform_request",
                "release",
                "renew",
                "rebind",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "solicit",
    }  # type: Dict[str, Union(type)]

    SOLICIT = "solicit"  # type: str
    REQUEST = "request"  # type: str
    INFORM_REQUEST = "inform_request"  # type: str
    RELEASE = "release"  # type: str
    RENEW = "renew"  # type: str
    REBIND = "rebind"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsMessageType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def choice(self):
        # type: () -> Union[Literal["inform_request"], Literal["rebind"], Literal["release"], Literal["renew"], Literal["request"], Literal["solicit"]]
        """choice getter

        The client message name where the option is included, by default it is all.

        Returns: Union[Literal["inform_request"], Literal["rebind"], Literal["release"], Literal["renew"], Literal["request"], Literal["solicit"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The client message name where the option is included, by default it is all.

        value: Union[Literal["inform_request"], Literal["rebind"], Literal["release"], Literal["renew"], Literal["request"], Literal["solicit"]]
        """
        self._set_property("choice", value)


class Dhcpv6ClientOptionsMessageTypeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsMessageTypeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Dhcpv6ClientOptionsMessageType]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Dhcpv6ClientOptionsMessageTypeIter
        return self._iter()

    def __next__(self):
        # type: () -> Dhcpv6ClientOptionsMessageType
        return self._next()

    def next(self):
        # type: () -> Dhcpv6ClientOptionsMessageType
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Dhcpv6ClientOptionsMessageType):
            raise Exception("Item is not an instance of Dhcpv6ClientOptionsMessageType")

    def messagetype(self):
        # type: () -> Dhcpv6ClientOptionsMessageTypeIter
        """Factory method that creates an instance of the Dhcpv6ClientOptionsMessageType class

        The dhcpv6 client messages where the option will be included.

        Returns: Dhcpv6ClientOptionsMessageTypeIter
        """
        item = Dhcpv6ClientOptionsMessageType(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> Dhcpv6ClientOptionsMessageType
        """Add method that creates and returns an instance of the Dhcpv6ClientOptionsMessageType class

        The dhcpv6 client messages where the option will be included.

        Returns: Dhcpv6ClientOptionsMessageType
        """
        item = Dhcpv6ClientOptionsMessageType(parent=self._parent, choice=self._choice)
        self._add(item)
        return item


class DeviceDhcpv6ClientOptions(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "server_identifier": {"type": "Dhcpv6ClientOptionsServerIdentifier"},
        "vendor_class": {"type": "Dhcpv6ClientOptionsVendorClass"},
        "vendor_info": {"type": "Dhcpv6ClientOptionsVendorInfo"},
        "fqdn": {"type": "Dhcpv6ClientOptionsFqdn"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(DeviceDhcpv6ClientOptions, self).__init__()
        self._parent = parent

    @property
    def server_identifier(self):
        # type: () -> Dhcpv6ClientOptionsServerIdentifier
        """server_identifier getter

        A client uses multicast to reach all servers or an individual server. An individual server is indicated by specifying that server's DUID in Server Identifier option in the client's message (all servers will receive this message but only the indicated server will respond). All servers are indicated by not supplying this option.

        Returns: Dhcpv6ClientOptionsServerIdentifier
        """
        return self._get_property(
            "server_identifier", Dhcpv6ClientOptionsServerIdentifier
        )

    @property
    def vendor_class(self):
        # type: () -> Dhcpv6ClientOptionsVendorClass
        """vendor_class getter

        This option is used by client to identify the vendor that manufactured the hardware on which the client is running. The option code is 16.This option is used by client to identify the vendor that manufactured the hardware on which the client is running. The option code is 16.This option is used by client to identify the vendor that manufactured the hardware on which the client is running. The option code is 16.The vendor class option is used by client to identify the vendor that manufactured the hardware on which the client is running. The information contained in the data area of this option is contained in one or more opaque fields that identify details of the hardware configuration.

        Returns: Dhcpv6ClientOptionsVendorClass
        """
        return self._get_property("vendor_class", Dhcpv6ClientOptionsVendorClass)

    @property
    def vendor_info(self):
        # type: () -> Dhcpv6ClientOptionsVendorInfo
        """vendor_info getter

        This option is used by clients to exchange vendor-specific information. The option code is 17.This option is used by clients to exchange vendor-specific information. The option code is 17.This option is used by clients to exchange vendor-specific information. The option code is 17.This option is used by clients to exchange vendor-specific information with servers.

        Returns: Dhcpv6ClientOptionsVendorInfo
        """
        return self._get_property("vendor_info", Dhcpv6ClientOptionsVendorInfo)

    @property
    def fqdn(self):
        # type: () -> Dhcpv6ClientOptionsFqdn
        """fqdn getter

        DHCPv6 server needs to know the FQDN of the client for the addresses for the client's IA_NA bindings in order to update the IPv6-address-to-FQDN mapping. This option allows the client to convey its FQDN to the server. The Client FQDN option also contains Flags that DHCPv6 clients and servers use to negotiate who does which updates. The option code is 39.DHCPv6 server needs to know the FQDN of the client for the addresses for the client's IA_NA bindings in order to update the IPv6-address-to-FQDN mapping. This option allows the client to convey its FQDN to the server. The Client FQDN option also contains Flags that DHCPv6 clients and servers use to negotiate who does which updates. The option code is 39.DHCPv6 server needs to know the FQDN of the client for the addresses for the client's IA_NA bindings in order to update the IPv6-address-to-FQDN mapping. This option allows the client to convey its FQDN to the server. The Client FQDN option also contains Flags that DHCPv6 clients and servers use to negotiate who does which updates. The option code is 39.DHCPv6 server needs to know the FQDN of the client for the addresses for the client's IA_NA bindings in order to update the IPv6-address-to-FQDN mapping. This option allows the client to convey its FQDN to the server. The Client FQDN option also contains Flags that DHCPv6 clients and servers use to negotiate who does which update.

        Returns: Dhcpv6ClientOptionsFqdn
        """
        return self._get_property("fqdn", Dhcpv6ClientOptionsFqdn)


class Dhcpv6ClientOptionsServerIdentifier(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "duid_llt",
                "duid_en",
                "duid_ll",
                "duid_uuid",
            ],
        },
        "duid_llt": {"type": "Dhcpv6ClientOptionsDuidLlt"},
        "duid_en": {"type": "Dhcpv6ClientOptionsDuidEn"},
        "duid_ll": {"type": "Dhcpv6ClientOptionsDuidLl"},
        "duid_uuid": {"type": "Dhcpv6ClientOptionsDuidUuid"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "duid_ll",
    }  # type: Dict[str, Union(type)]

    DUID_LLT = "duid_llt"  # type: str
    DUID_EN = "duid_en"  # type: str
    DUID_LL = "duid_ll"  # type: str
    DUID_UUID = "duid_uuid"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsServerIdentifier, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def duid_llt(self):
        # type: () -> Dhcpv6ClientOptionsDuidLlt
        """Factory property that returns an instance of the Dhcpv6ClientOptionsDuidLlt class

        DUID based on Link Layer address plus time. Hardware Type will be auto assigned to ethernet type.

        Returns: Dhcpv6ClientOptionsDuidLlt
        """
        return self._get_property(
            "duid_llt", Dhcpv6ClientOptionsDuidLlt, self, "duid_llt"
        )

    @property
    def duid_en(self):
        # type: () -> Dhcpv6ClientOptionsDuidEn
        """Factory property that returns an instance of the Dhcpv6ClientOptionsDuidEn class

        DUID assigned by vendor based on enterprise number.

        Returns: Dhcpv6ClientOptionsDuidEn
        """
        return self._get_property("duid_en", Dhcpv6ClientOptionsDuidEn, self, "duid_en")

    @property
    def duid_ll(self):
        # type: () -> Dhcpv6ClientOptionsDuidLl
        """Factory property that returns an instance of the Dhcpv6ClientOptionsDuidLl class

        DUID based on Link Layer address. Hardware Type will be auto assigned to ethernet type.

        Returns: Dhcpv6ClientOptionsDuidLl
        """
        return self._get_property("duid_ll", Dhcpv6ClientOptionsDuidLl, self, "duid_ll")

    @property
    def duid_uuid(self):
        # type: () -> Dhcpv6ClientOptionsDuidUuid
        """Factory property that returns an instance of the Dhcpv6ClientOptionsDuidUuid class

        DUID embedded Universally Unique IDentifier (UUID). UUID is an identifier that is unique across both space and time, with respect to the space of all UUIDs.

        Returns: Dhcpv6ClientOptionsDuidUuid
        """
        return self._get_property(
            "duid_uuid", Dhcpv6ClientOptionsDuidUuid, self, "duid_uuid"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["duid_en"], Literal["duid_ll"], Literal["duid_llt"], Literal["duid_uuid"]]
        """choice getter

        The Identifier option is used to carry DUID. The option code is 2. The server identifier identifies server. This option is used when client wants to contact particular server.

        Returns: Union[Literal["duid_en"], Literal["duid_ll"], Literal["duid_llt"], Literal["duid_uuid"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The Identifier option is used to carry DUID. The option code is 2. The server identifier identifies server. This option is used when client wants to contact particular server.

        value: Union[Literal["duid_en"], Literal["duid_ll"], Literal["duid_llt"], Literal["duid_uuid"]]
        """
        self._set_property("choice", value)


class Dhcpv6ClientOptionsDuidLlt(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "time": {
            "type": int,
            "format": "uint32",
        },
        "link_layer_address": {"type": "Dhcpv6ClientOptionsLinkLayerAddress"},
    }  # type: Dict[str, str]

    _REQUIRED = ("time", "link_layer_address")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, time=None):
        super(Dhcpv6ClientOptionsDuidLlt, self).__init__()
        self._parent = parent
        self._set_property("time", time)

    def set(self, time=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def time(self):
        # type: () -> int
        """time getter

        The time value is the time that the DUID is generated represented in seconds since midnight (UTC), January 1, 2000, modulo 2^32. The DUID generatation time will the current time when dhcpv6 client contacts the server.

        Returns: int
        """
        return self._get_property("time")

    @time.setter
    def time(self, value):
        """time setter

        The time value is the time that the DUID is generated represented in seconds since midnight (UTC), January 1, 2000, modulo 2^32. The DUID generatation time will the current time when dhcpv6 client contacts the server.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property time as None")
        self._set_property("time", value)

    @property
    def link_layer_address(self):
        # type: () -> Dhcpv6ClientOptionsLinkLayerAddress
        """link_layer_address getter

        The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address is stored in canonical form, as described in RFC 2464.

        Returns: Dhcpv6ClientOptionsLinkLayerAddress
        """
        return self._get_property(
            "link_layer_address", Dhcpv6ClientOptionsLinkLayerAddress
        )


class Dhcpv6ClientOptionsLinkLayerAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "format": "mac",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("value",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=None):
        super(Dhcpv6ClientOptionsLinkLayerAddress, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        The MAC address that becomes part of DUID llt or DUID ll.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        The MAC address that becomes part of DUID llt or DUID ll.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property value as None")
        self._set_property("value", value)


class Dhcpv6ClientOptionsDuidEn(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enterprise_number": {
            "type": int,
            "format": "uint32",
            "maximum": 4294967295,
        },
        "identifier": {
            "type": str,
            "format": "hex",
            "minLength": 1,
            "maxLength": 8,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("enterprise_number", "identifier")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, enterprise_number=None, identifier=None):
        super(Dhcpv6ClientOptionsDuidEn, self).__init__()
        self._parent = parent
        self._set_property("enterprise_number", enterprise_number)
        self._set_property("identifier", identifier)

    def set(self, enterprise_number=None, identifier=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enterprise_number(self):
        # type: () -> int
        """enterprise_number getter

        Vendor's registered private enterprise number as maintained by IANA.

        Returns: int
        """
        return self._get_property("enterprise_number")

    @enterprise_number.setter
    def enterprise_number(self, value):
        """enterprise_number setter

        Vendor's registered private enterprise number as maintained by IANA.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property enterprise_number as None")
        self._set_property("enterprise_number", value)

    @property
    def identifier(self):
        # type: () -> str
        """identifier getter

        The unique identifier assigned by the vendor.

        Returns: str
        """
        return self._get_property("identifier")

    @identifier.setter
    def identifier(self, value):
        """identifier setter

        The unique identifier assigned by the vendor.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property identifier as None")
        self._set_property("identifier", value)


class Dhcpv6ClientOptionsDuidLl(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "link_layer_address": {"type": "Dhcpv6ClientOptionsLinkLayerAddress"},
    }  # type: Dict[str, str]

    _REQUIRED = ("link_layer_address",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(Dhcpv6ClientOptionsDuidLl, self).__init__()
        self._parent = parent

    @property
    def link_layer_address(self):
        # type: () -> Dhcpv6ClientOptionsLinkLayerAddress
        """link_layer_address getter

        The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address configured in DUID llt or DUID ll.The link-layer address is stored in canonical form, as described in RFC 2464.

        Returns: Dhcpv6ClientOptionsLinkLayerAddress
        """
        return self._get_property(
            "link_layer_address", Dhcpv6ClientOptionsLinkLayerAddress
        )


class Dhcpv6ClientOptionsDuidUuid(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "version": {"type": "Dhcpv6ClientOptionsDuidUuidVersion"},
        "variant": {"type": "Dhcpv6ClientOptionsDuidUuidVariant"},
        "time_low": {
            "type": int,
            "format": "uint32",
        },
        "time_mid": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "time_hi_and_version": {
            "type": int,
            "format": "uint32",
            "maximum": 4095,
        },
        "clock_seq_hi_and_reserved": {
            "type": int,
            "format": "uint32",
            "maximum": 31,
        },
        "clock_seq_low": {
            "type": int,
            "format": "uint32",
            "maximum": 127,
        },
        "node": {
            "type": str,
            "format": "mac",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "time_low": 0,
        "time_mid": 0,
        "time_hi_and_version": 0,
        "clock_seq_hi_and_reserved": 0,
        "clock_seq_low": 0,
        "node": "00:00:00:00:00:00",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        time_low=0,
        time_mid=0,
        time_hi_and_version=0,
        clock_seq_hi_and_reserved=0,
        clock_seq_low=0,
        node="00:00:00:00:00:00",
    ):
        super(Dhcpv6ClientOptionsDuidUuid, self).__init__()
        self._parent = parent
        self._set_property("time_low", time_low)
        self._set_property("time_mid", time_mid)
        self._set_property("time_hi_and_version", time_hi_and_version)
        self._set_property("clock_seq_hi_and_reserved", clock_seq_hi_and_reserved)
        self._set_property("clock_seq_low", clock_seq_low)
        self._set_property("node", node)

    def set(
        self,
        time_low=None,
        time_mid=None,
        time_hi_and_version=None,
        clock_seq_hi_and_reserved=None,
        clock_seq_low=None,
        node=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def version(self):
        # type: () -> Dhcpv6ClientOptionsDuidUuidVersion
        """version getter

        The version number is in the most significant bits of the timestamp (bits through of the time_hi_and_version field).The version number is in the most significant bits of the timestamp (bits through of the time_hi_and_version field).The version number is in the most significant bits of the timestamp (bits through of the time_hi_and_version field).The version number is in the most significant bits of the timestamp (bits through of the time_hi_and_version field).The version number is in the most significant bits of the timestamp (bits through of the time_hi_and_version field).

        Returns: Dhcpv6ClientOptionsDuidUuidVersion
        """
        return self._get_property("version", Dhcpv6ClientOptionsDuidUuidVersion)

    @property
    def variant(self):
        # type: () -> Dhcpv6ClientOptionsDuidUuidVariant
        """variant getter

        The variant field determines the layout of the UUID. That is, the interpretation of all other bits in the UUID depends on the setting of the bits in the variant field).The variant field determines the layout of the UUID. That is, the interpretation of all other bits in the UUID depends on the setting of the bits in the variant field).The variant field determines the layout of the UUID. That is, the interpretation of all other bits in the UUID depends on the setting of the bits in the variant field).The variant field determines the layout of the UUID. That is, the interpretation of all other bits in the UUID depends on the setting of the bits in the variant field).The variant field determines the layout of the UUID. It is multiplexed with clock_seq_hi_and_reserved.

        Returns: Dhcpv6ClientOptionsDuidUuidVariant
        """
        return self._get_property("variant", Dhcpv6ClientOptionsDuidUuidVariant)

    @property
    def time_low(self):
        # type: () -> int
        """time_low getter

        The low field of the timestamp.

        Returns: int
        """
        return self._get_property("time_low")

    @time_low.setter
    def time_low(self, value):
        """time_low setter

        The low field of the timestamp.

        value: int
        """
        self._set_property("time_low", value)

    @property
    def time_mid(self):
        # type: () -> int
        """time_mid getter

        The middle field of the timestamp.

        Returns: int
        """
        return self._get_property("time_mid")

    @time_mid.setter
    def time_mid(self, value):
        """time_mid setter

        The middle field of the timestamp.

        value: int
        """
        self._set_property("time_mid", value)

    @property
    def time_hi_and_version(self):
        # type: () -> int
        """time_hi_and_version getter

        The high field of the timestamp multiplexed with the version number.

        Returns: int
        """
        return self._get_property("time_hi_and_version")

    @time_hi_and_version.setter
    def time_hi_and_version(self, value):
        """time_hi_and_version setter

        The high field of the timestamp multiplexed with the version number.

        value: int
        """
        self._set_property("time_hi_and_version", value)

    @property
    def clock_seq_hi_and_reserved(self):
        # type: () -> int
        """clock_seq_hi_and_reserved getter

        The high field of the clock sequence multiplexed with the variant.

        Returns: int
        """
        return self._get_property("clock_seq_hi_and_reserved")

    @clock_seq_hi_and_reserved.setter
    def clock_seq_hi_and_reserved(self, value):
        """clock_seq_hi_and_reserved setter

        The high field of the clock sequence multiplexed with the variant.

        value: int
        """
        self._set_property("clock_seq_hi_and_reserved", value)

    @property
    def clock_seq_low(self):
        # type: () -> int
        """clock_seq_low getter

        The low field of the clock sequence.

        Returns: int
        """
        return self._get_property("clock_seq_low")

    @clock_seq_low.setter
    def clock_seq_low(self, value):
        """clock_seq_low setter

        The low field of the clock sequence.

        value: int
        """
        self._set_property("clock_seq_low", value)

    @property
    def node(self):
        # type: () -> str
        """node getter

        The spatially unique node identifier.

        Returns: str
        """
        return self._get_property("node")

    @node.setter
    def node(self, value):
        """node setter

        The spatially unique node identifier.

        value: str
        """
        self._set_property("node", value)


class Dhcpv6ClientOptionsDuidUuidVersion(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "v_1",
                "v_2",
                "v_3",
                "v_4",
                "v_5",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "v_1",
    }  # type: Dict[str, Union(type)]

    V_1 = "v_1"  # type: str
    V_2 = "v_2"  # type: str
    V_3 = "v_3"  # type: str
    V_4 = "v_4"  # type: str
    V_5 = "v_5"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsDuidUuidVersion, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def choice(self):
        # type: () -> Union[Literal["v_1"], Literal["v_2"], Literal["v_3"], Literal["v_4"], Literal["v_5"]]
        """choice getter

        The version values are from to in the most significant bits of the timestamp (bits through of the time_hi_and_version field).

        Returns: Union[Literal["v_1"], Literal["v_2"], Literal["v_3"], Literal["v_4"], Literal["v_5"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The version values are from to in the most significant bits of the timestamp (bits through of the time_hi_and_version field).

        value: Union[Literal["v_1"], Literal["v_2"], Literal["v_3"], Literal["v_4"], Literal["v_5"]]
        """
        self._set_property("choice", value)


class Dhcpv6ClientOptionsDuidUuidVariant(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "ncs",
                "dce",
                "guid",
                "var_reserved",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "ncs",
    }  # type: Dict[str, Union(type)]

    NCS = "ncs"  # type: str
    DCE = "dce"  # type: str
    GUID = "guid"  # type: str
    VAR_RESERVED = "var_reserved"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6ClientOptionsDuidUuidVariant, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def choice(self):
        # type: () -> Union[Literal["dce"], Literal["guid"], Literal["ncs"], Literal["var_reserved"]]
        """choice getter

        The current variants are ncs, dce,microsoft guid and reserved.

        Returns: Union[Literal["dce"], Literal["guid"], Literal["ncs"], Literal["var_reserved"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The current variants are ncs, dce,microsoft guid and reserved.

        value: Union[Literal["dce"], Literal["guid"], Literal["ncs"], Literal["var_reserved"]]
        """
        self._set_property("choice", value)


class Dhcpv6ClientOptionsVendorClass(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enterprise_number": {
            "type": int,
            "format": "uint32",
            "maximum": 4294967295,
        },
        "class_data": {
            "type": list,
            "itemtype": str,
        },
        "associated_dhcp_messages": {"type": "Dhcpv6ClientOptionsIncludedMessages"},
    }  # type: Dict[str, str]

    _REQUIRED = (
        "enterprise_number",
        "class_data",
        "associated_dhcp_messages",
    )  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, enterprise_number=None, class_data=None):
        super(Dhcpv6ClientOptionsVendorClass, self).__init__()
        self._parent = parent
        self._set_property("enterprise_number", enterprise_number)
        self._set_property("class_data", class_data)

    def set(self, enterprise_number=None, class_data=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enterprise_number(self):
        # type: () -> int
        """enterprise_number getter

        The vendor's registered Enterprise Number as registered with IANA.

        Returns: int
        """
        return self._get_property("enterprise_number")

    @enterprise_number.setter
    def enterprise_number(self, value):
        """enterprise_number setter

        The vendor's registered Enterprise Number as registered with IANA.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property enterprise_number as None")
        self._set_property("enterprise_number", value)

    @property
    def class_data(self):
        # type: () -> List[str]
        """class_data getter

        The opaque data representing the hardware configuration of the host on which the client is running. Examples of class data instances might include the version of the operating system the client is running or the amount of memory installed on the client.

        Returns: List[str]
        """
        return self._get_property("class_data")

    @class_data.setter
    def class_data(self, value):
        """class_data setter

        The opaque data representing the hardware configuration of the host on which the client is running. Examples of class data instances might include the version of the operating system the client is running or the amount of memory installed on the client.

        value: List[str]
        """
        if value is None:
            raise TypeError("Cannot set required property class_data as None")
        self._set_property("class_data", value)

    @property
    def associated_dhcp_messages(self):
        # type: () -> Dhcpv6ClientOptionsIncludedMessages
        """associated_dhcp_messages getter

        The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where this option is included.

        Returns: Dhcpv6ClientOptionsIncludedMessages
        """
        return self._get_property(
            "associated_dhcp_messages", Dhcpv6ClientOptionsIncludedMessages
        )


class Dhcpv6ClientOptionsVendorInfo(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enterprise_number": {
            "type": int,
            "format": "uint32",
            "maximum": 4294967295,
        },
        "option_data": {"type": "Dhcpv6OptionsVendorSpecificOptionsIter"},
        "associated_dhcp_messages": {"type": "Dhcpv6ClientOptionsIncludedMessages"},
    }  # type: Dict[str, str]

    _REQUIRED = (
        "enterprise_number",
        "option_data",
        "associated_dhcp_messages",
    )  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, enterprise_number=None):
        super(Dhcpv6ClientOptionsVendorInfo, self).__init__()
        self._parent = parent
        self._set_property("enterprise_number", enterprise_number)

    def set(self, enterprise_number=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enterprise_number(self):
        # type: () -> int
        """enterprise_number getter

        The vendor's registered Enterprise Number as registered with IANA.

        Returns: int
        """
        return self._get_property("enterprise_number")

    @enterprise_number.setter
    def enterprise_number(self, value):
        """enterprise_number setter

        The vendor's registered Enterprise Number as registered with IANA.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property enterprise_number as None")
        self._set_property("enterprise_number", value)

    @property
    def option_data(self):
        # type: () -> Dhcpv6OptionsVendorSpecificOptionsIter
        """option_data getter

        An opaque object of octets,interpreted by vendor-specific code on the clients and servers.

        Returns: Dhcpv6OptionsVendorSpecificOptionsIter
        """
        return self._get_property(
            "option_data",
            Dhcpv6OptionsVendorSpecificOptionsIter,
            self._parent,
            self._choice,
        )

    @property
    def associated_dhcp_messages(self):
        # type: () -> Dhcpv6ClientOptionsIncludedMessages
        """associated_dhcp_messages getter

        The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The list of dhcpv6 client messages where this option is included.

        Returns: Dhcpv6ClientOptionsIncludedMessages
        """
        return self._get_property(
            "associated_dhcp_messages", Dhcpv6ClientOptionsIncludedMessages
        )


class Dhcpv6OptionsVendorSpecificOptions(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "code": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "data": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("code", "data")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, code=None, data=None):
        super(Dhcpv6OptionsVendorSpecificOptions, self).__init__()
        self._parent = parent
        self._set_property("code", code)
        self._set_property("data", data)

    def set(self, code=None, data=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def code(self):
        # type: () -> int
        """code getter

        The code for the encapsulated option.

        Returns: int
        """
        return self._get_property("code")

    @code.setter
    def code(self, value):
        """code setter

        The code for the encapsulated option.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property code as None")
        self._set_property("code", value)

    @property
    def data(self):
        # type: () -> str
        """data getter

        The data for the encapsulated option.

        Returns: str
        """
        return self._get_property("data")

    @data.setter
    def data(self, value):
        """data setter

        The data for the encapsulated option.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property data as None")
        self._set_property("data", value)


class Dhcpv6OptionsVendorSpecificOptionsIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(Dhcpv6OptionsVendorSpecificOptionsIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[Dhcpv6OptionsVendorSpecificOptions]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> Dhcpv6OptionsVendorSpecificOptionsIter
        return self._iter()

    def __next__(self):
        # type: () -> Dhcpv6OptionsVendorSpecificOptions
        return self._next()

    def next(self):
        # type: () -> Dhcpv6OptionsVendorSpecificOptions
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, Dhcpv6OptionsVendorSpecificOptions):
            raise Exception(
                "Item is not an instance of Dhcpv6OptionsVendorSpecificOptions"
            )

    def vendorspecificoptions(self, code=None, data=None):
        # type: (int,str) -> Dhcpv6OptionsVendorSpecificOptionsIter
        """Factory method that creates an instance of the Dhcpv6OptionsVendorSpecificOptions class

        The encapsulated vendor-specific options field is encoded as sequence of code/length/value fields of identical format to the DHCP options field. The option codes are defined by the vendor identified in the enterprise-number field and are not managed by IANA.

        Returns: Dhcpv6OptionsVendorSpecificOptionsIter
        """
        item = Dhcpv6OptionsVendorSpecificOptions(
            parent=self._parent, code=code, data=data
        )
        self._add(item)
        return self

    def add(self, code=None, data=None):
        # type: (int,str) -> Dhcpv6OptionsVendorSpecificOptions
        """Add method that creates and returns an instance of the Dhcpv6OptionsVendorSpecificOptions class

        The encapsulated vendor-specific options field is encoded as sequence of code/length/value fields of identical format to the DHCP options field. The option codes are defined by the vendor identified in the enterprise-number field and are not managed by IANA.

        Returns: Dhcpv6OptionsVendorSpecificOptions
        """
        item = Dhcpv6OptionsVendorSpecificOptions(
            parent=self._parent, code=code, data=data
        )
        self._add(item)
        return item


class Dhcpv6ClientOptionsFqdn(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flag_s": {"type": bool},
        "flag_o": {"type": bool},
        "flag_n": {"type": bool},
        "domain_name": {"type": str},
        "associated_dhcp_messages": {"type": "Dhcpv6ClientOptionsIncludedMessages"},
    }  # type: Dict[str, str]

    _REQUIRED = ("domain_name",)  # type: tuple(str)

    _DEFAULTS = {
        "flag_s": True,
        "flag_o": False,
        "flag_n": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flag_s=True, flag_o=False, flag_n=False, domain_name=None
    ):
        super(Dhcpv6ClientOptionsFqdn, self).__init__()
        self._parent = parent
        self._set_property("flag_s", flag_s)
        self._set_property("flag_o", flag_o)
        self._set_property("flag_n", flag_n)
        self._set_property("domain_name", domain_name)

    def set(self, flag_s=None, flag_o=None, flag_n=None, domain_name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flag_s(self):
        # type: () -> bool
        """flag_s getter

        The "S" bit indicates whether the server should or should not perform the AAAA RR (FQDN-to-address) DNS updates. client sets the bit to to indicate that the server should not perform the updates and to indicate that the server should perform the updates. The state of the bit in the reply from the server indicates the action to be taken by the server. If it is 1, the server has taken responsibility for AAAA RR updates for the FQDN.

        Returns: bool
        """
        return self._get_property("flag_s")

    @flag_s.setter
    def flag_s(self, value):
        """flag_s setter

        The "S" bit indicates whether the server should or should not perform the AAAA RR (FQDN-to-address) DNS updates. client sets the bit to to indicate that the server should not perform the updates and to indicate that the server should perform the updates. The state of the bit in the reply from the server indicates the action to be taken by the server. If it is 1, the server has taken responsibility for AAAA RR updates for the FQDN.

        value: bool
        """
        self._set_property("flag_s", value)

    @property
    def flag_o(self):
        # type: () -> bool
        """flag_o getter

        The "O" bit indicates whether the server has overridden the client's preference for the "S" bit. client must set this bit to 0. server must set this bit to if the "S" bit in its reply to the client does not match the "S" bit received from the client.

        Returns: bool
        """
        return self._get_property("flag_o")

    @flag_o.setter
    def flag_o(self, value):
        """flag_o setter

        The "O" bit indicates whether the server has overridden the client's preference for the "S" bit. client must set this bit to 0. server must set this bit to if the "S" bit in its reply to the client does not match the "S" bit received from the client.

        value: bool
        """
        self._set_property("flag_o", value)

    @property
    def flag_n(self):
        # type: () -> bool
        """flag_n getter

        The "N" bit indicates whether the server should not perform any DNS updates. client sets this bit to to request that the server should perform updates (the PTR RR and possibly the AAAA RR based on the "S" bit) or to to request that the server should not perform any DNS updates. server sets the "N" bit to indicate whether the server shall (0) or shall not (1) perform DNS updates. If the "N" bit is 1, the "S" bit MUST be 0.

        Returns: bool
        """
        return self._get_property("flag_n")

    @flag_n.setter
    def flag_n(self, value):
        """flag_n setter

        The "N" bit indicates whether the server should not perform any DNS updates. client sets this bit to to request that the server should perform updates (the PTR RR and possibly the AAAA RR based on the "S" bit) or to to request that the server should not perform any DNS updates. server sets the "N" bit to indicate whether the server shall (0) or shall not (1) perform DNS updates. If the "N" bit is 1, the "S" bit MUST be 0.

        value: bool
        """
        self._set_property("flag_n", value)

    @property
    def domain_name(self):
        # type: () -> str
        """domain_name getter

        The Domain Name part of the option carries all or part of the FQDN of DHCPv6 client. client MAY also leave the Domain Name field empty if it desires the server to provide name. fully qualified domain name (FQDN) is the complete address of an internet host or computer. It provides its exact location within the domain name system (DNS) by specifying the hostname, domain name and top-level domain (TLD). An FQDN isn't the same as URL but rather is part of it that fully identifies the server to which the request is addressed. An FQDN doesn't carry the TCP/IP protocol information, such as Hypertext Transfer Protocol (HTTP) or Hypertext Transfer Protocol Secure (HTTPS), which is always used at the beginning of URL. Therefore, adding the prefix http:// or https:// to the FQDN turns it into full URL. One example can be microsoft.com.

        Returns: str
        """
        return self._get_property("domain_name")

    @domain_name.setter
    def domain_name(self, value):
        """domain_name setter

        The Domain Name part of the option carries all or part of the FQDN of DHCPv6 client. client MAY also leave the Domain Name field empty if it desires the server to provide name. fully qualified domain name (FQDN) is the complete address of an internet host or computer. It provides its exact location within the domain name system (DNS) by specifying the hostname, domain name and top-level domain (TLD). An FQDN isn't the same as URL but rather is part of it that fully identifies the server to which the request is addressed. An FQDN doesn't carry the TCP/IP protocol information, such as Hypertext Transfer Protocol (HTTP) or Hypertext Transfer Protocol Secure (HTTPS), which is always used at the beginning of URL. Therefore, adding the prefix http:// or https:// to the FQDN turns it into full URL. One example can be microsoft.com.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property domain_name as None")
        self._set_property("domain_name", value)

    @property
    def associated_dhcp_messages(self):
        # type: () -> Dhcpv6ClientOptionsIncludedMessages
        """associated_dhcp_messages getter

        The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The dhcpv6 client messages where the option will be included. If all is selected the selected option will be added in the all the Dhcpv6 client messages, else based on the selection in particular Dhcpv6 client messages the option will be included.The list of dhcpv6 client messages where this option is included.

        Returns: Dhcpv6ClientOptionsIncludedMessages
        """
        return self._get_property(
            "associated_dhcp_messages", Dhcpv6ClientOptionsIncludedMessages
        )


class DeviceDhcpv6clientIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceDhcpv6clientIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceDhcpv6client]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceDhcpv6clientIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceDhcpv6client
        return self._next()

    def next(self):
        # type: () -> DeviceDhcpv6client
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceDhcpv6client):
            raise Exception("Item is not an instance of DeviceDhcpv6client")

    def dhcpv6client(self, name=None, rapid_commit=False):
        # type: (str,bool) -> DeviceDhcpv6clientIter
        """Factory method that creates an instance of the DeviceDhcpv6client class

        Configuration for emulated DHCPv6 Client on single Interface. If the DHCPv6 Client receives one or more DHCPv6 ADVERTISE messages from one or more servers then the client chooses one server from which to request configuration parameters, based on the configuration parameters offered by the server in the DHCPv6 ADVERTISE messages. If all configuration parameters match then the first server will be chosen. https://www.rfc-editor.org/rfc/rfc8415.html

        Returns: DeviceDhcpv6clientIter
        """
        item = DeviceDhcpv6client(
            parent=self._parent, name=name, rapid_commit=rapid_commit
        )
        self._add(item)
        return self

    def add(self, name=None, rapid_commit=False):
        # type: (str,bool) -> DeviceDhcpv6client
        """Add method that creates and returns an instance of the DeviceDhcpv6client class

        Configuration for emulated DHCPv6 Client on single Interface. If the DHCPv6 Client receives one or more DHCPv6 ADVERTISE messages from one or more servers then the client chooses one server from which to request configuration parameters, based on the configuration parameters offered by the server in the DHCPv6 ADVERTISE messages. If all configuration parameters match then the first server will be chosen. https://www.rfc-editor.org/rfc/rfc8415.html

        Returns: DeviceDhcpv6client
        """
        item = DeviceDhcpv6client(
            parent=self._parent, name=name, rapid_commit=rapid_commit
        )
        self._add(item)
        return item


class DeviceEthernetIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceEthernetIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceEthernet]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceEthernetIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceEthernet
        return self._next()

    def next(self):
        # type: () -> DeviceEthernet
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceEthernet):
            raise Exception("Item is not an instance of DeviceEthernet")

    def ethernet(self, mac=None, mtu=1500, name=None):
        # type: (str,int,str) -> DeviceEthernetIter
        """Factory method that creates an instance of the DeviceEthernet class

        An Ethernet interface with IPv4 and IPv6 addresses. The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where MAC is not mandatory.

        Returns: DeviceEthernetIter
        """
        item = DeviceEthernet(parent=self._parent, mac=mac, mtu=mtu, name=name)
        self._add(item)
        return self

    def add(self, mac=None, mtu=1500, name=None):
        # type: (str,int,str) -> DeviceEthernet
        """Add method that creates and returns an instance of the DeviceEthernet class

        An Ethernet interface with IPv4 and IPv6 addresses. The implementation should ensure that the 'mac' field is explicitly configured by the user for all types of interfaces as denoted by 'connection' attribute except 'simulated_link' where MAC is not mandatory.

        Returns: DeviceEthernet
        """
        item = DeviceEthernet(parent=self._parent, mac=mac, mtu=mtu, name=name)
        self._add(item)
        return item


class DeviceIpv4Loopback(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, eth_name=None, address="0.0.0.0", name=None):
        super(DeviceIpv4Loopback, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("address", address)
        self._set_property("name", name)

    def set(self, eth_name=None, address=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property eth_name as None")
        self._set_property("eth_name", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv4 Loopback address with prefix length of 32.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv4 Loopback address with prefix length of 32.

        value: str
        """
        self._set_property("address", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceIpv4LoopbackIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv4LoopbackIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv4Loopback]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv4LoopbackIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv4Loopback
        return self._next()

    def next(self):
        # type: () -> DeviceIpv4Loopback
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceIpv4Loopback):
            raise Exception("Item is not an instance of DeviceIpv4Loopback")

    def ipv4loopback(self, eth_name=None, address="0.0.0.0", name=None):
        # type: (str,str,str) -> DeviceIpv4LoopbackIter
        """Factory method that creates an instance of the DeviceIpv4Loopback class

        An IPv4 Loopback interface.

        Returns: DeviceIpv4LoopbackIter
        """
        item = DeviceIpv4Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return self

    def add(self, eth_name=None, address="0.0.0.0", name=None):
        # type: (str,str,str) -> DeviceIpv4Loopback
        """Add method that creates and returns an instance of the DeviceIpv4Loopback class

        An IPv4 Loopback interface.

        Returns: DeviceIpv4Loopback
        """
        item = DeviceIpv4Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return item


class DeviceIpv6Loopback(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "address": "::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, eth_name=None, address="::0", name=None):
        super(DeviceIpv6Loopback, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("address", address)
        self._set_property("name", name)

    def set(self, eth_name=None, address=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface behind which this Loopback interface will be created.. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property eth_name as None")
        self._set_property("eth_name", value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The IPv6 Loopback address with prefix length of 128.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The IPv6 Loopback address with prefix length of 128.

        value: str
        """
        self._set_property("address", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class DeviceIpv6LoopbackIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(DeviceIpv6LoopbackIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[DeviceIpv6Loopback]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> DeviceIpv6LoopbackIter
        return self._iter()

    def __next__(self):
        # type: () -> DeviceIpv6Loopback
        return self._next()

    def next(self):
        # type: () -> DeviceIpv6Loopback
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, DeviceIpv6Loopback):
            raise Exception("Item is not an instance of DeviceIpv6Loopback")

    def ipv6loopback(self, eth_name=None, address="::0", name=None):
        # type: (str,str,str) -> DeviceIpv6LoopbackIter
        """Factory method that creates an instance of the DeviceIpv6Loopback class

        An IPv6 Loopback interface

        Returns: DeviceIpv6LoopbackIter
        """
        item = DeviceIpv6Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return self

    def add(self, eth_name=None, address="::0", name=None):
        # type: (str,str,str) -> DeviceIpv6Loopback
        """Add method that creates and returns an instance of the DeviceIpv6Loopback class

        An IPv6 Loopback interface

        Returns: DeviceIpv6Loopback
        """
        item = DeviceIpv6Loopback(
            parent=self._parent, eth_name=eth_name, address=address, name=name
        )
        self._add(item)
        return item


class DeviceIsisRouter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "instance": {"type": "DeviceIsisMultiInstance"},
        "system_id": {
            "type": str,
            "format": "hex",
        },
        "interfaces": {"type": "IsisInterfaceIter"},
        "basic": {"type": "IsisBasic"},
        "advanced": {"type": "IsisAdvanced"},
        "router_auth": {"type": "IsisAuthentication"},
        "v4_routes": {"type": "IsisV4RouteRangeIter"},
        "v6_routes": {"type": "IsisV6RouteRangeIter"},
        "name": {"type": str},
        "segment_routing": {"type": "IsisSegmentRouting"},
    }  # type: Dict[str, str]

    _REQUIRED = ("system_id", "interfaces", "name")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, system_id=None, name=None):
        super(DeviceIsisRouter, self).__init__()
        self._parent = parent
        self._set_property("system_id", system_id)
        self._set_property("name", name)

    def set(self, system_id=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def instance(self):
        # type: () -> DeviceIsisMultiInstance
        """instance getter

        This container properties of an Multi-Instance-capable router (MI-RTR). This container properties of an Multi-Instance-capable router (MI-RTR). This container properties of an Multi-Instance-capable router (MI-RTR). This contains the properties of Multi-Instance-capable routers or MI-RTR. Each router can emulate one ISIS instance at time.

        Returns: DeviceIsisMultiInstance
        """
        return self._get_property("instance", DeviceIsisMultiInstance)

    @property
    def system_id(self):
        # type: () -> str
        """system_id getter

        The System ID for this emulated ISIS router, e.g. "640100010000".

        Returns: str
        """
        return self._get_property("system_id")

    @system_id.setter
    def system_id(self, value):
        """system_id setter

        The System ID for this emulated ISIS router, e.g. "640100010000".

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property system_id as None")
        self._set_property("system_id", value)

    @property
    def interfaces(self):
        # type: () -> IsisInterfaceIter
        """interfaces getter

        List of ISIS interfaces for this router.

        Returns: IsisInterfaceIter
        """
        return self._get_property(
            "interfaces", IsisInterfaceIter, self._parent, self._choice
        )

    @property
    def basic(self):
        # type: () -> IsisBasic
        """basic getter

        This contains ISIS router basic properties.This contains ISIS router basic properties.This contains ISIS router basic properties.Contains basic properties of an ISIS Router.

        Returns: IsisBasic
        """
        return self._get_property("basic", IsisBasic)

    @property
    def advanced(self):
        # type: () -> IsisAdvanced
        """advanced getter

        Contains ISIS router advanced properties.Contains ISIS router advanced properties.Contains ISIS router advanced properties.Contains advance properties of an ISIS Router..

        Returns: IsisAdvanced
        """
        return self._get_property("advanced", IsisAdvanced)

    @property
    def router_auth(self):
        # type: () -> IsisAuthentication
        """router_auth getter

        This contains ISIS Area/Domain authentication properties.This contains ISIS Area/Domain authentication properties.This contains ISIS Area/Domain authentication properties.ISIS Router authentication properties.

        Returns: IsisAuthentication
        """
        return self._get_property("router_auth", IsisAuthentication)

    @property
    def v4_routes(self):
        # type: () -> IsisV4RouteRangeIter
        """v4_routes getter

        Emulated ISIS IPv4 routes.

        Returns: IsisV4RouteRangeIter
        """
        return self._get_property(
            "v4_routes", IsisV4RouteRangeIter, self._parent, self._choice
        )

    @property
    def v6_routes(self):
        # type: () -> IsisV6RouteRangeIter
        """v6_routes getter

        Emulated ISIS IPv6 routes.

        Returns: IsisV6RouteRangeIter
        """
        return self._get_property(
            "v6_routes", IsisV6RouteRangeIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def segment_routing(self):
        # type: () -> IsisSegmentRouting
        """segment_routing getter

        Segment Routing (SR) allows for flexible definition of end-to-end paths within IGP topologies by encoding paths as sequences of topological sub-paths, called "segments". These segments are advertised by the link-state routing protocols (IS-IS and OSPF). Prefix segments represent an ECMP-aware shortest path to prefix (or node), as per the state of the IGP topology. Adjacency segments represent hop over specific adjacency between two nodes in the IGP. A prefix segment is typically multi-hop path while an adjacency segment, in most of the cases, is one-hop path. These segments act as topological sub-paths that can be combined together to form the required path.. Reference: https://datatracker.ietf.org/doc/html/rfc8667.:w. An implementation may advertise Router Capability with default values if user does not even set the properties of Router Capability and Segment Routing Capability. Segment Routing (SR) allows for flexible definition of end-to-end paths within IGP topologies by encoding paths as sequences of topological sub-paths, called "segments". These segments are advertised by the link-state routing protocols (IS-IS and OSPF). Prefix segments represent an ECMP-aware shortest path to prefix (or node), as per the state of the IGP topology. Adjacency segments represent hop over specific adjacency between two nodes in the IGP. A prefix segment is typically multi-hop path while an adjacency segment, in most of the cases, is one-hop path. These segments act as topological sub-paths that can be combined together to form the required path.. Reference: https://datatracker.ietf.org/doc/html/rfc8667.:w. An implementation may advertise Router Capability with default values if user does not even set the properties of Router Capability and Segment Routing Capability. Segment Routing (SR) allows for flexible definition of end-to-end paths within IGP topologies by encoding paths as sequences of topological sub-paths, called "segments". These segments are advertised by the link-state routing protocols (IS-IS and OSPF). Prefix segments represent an ECMP-aware shortest path to prefix (or node), as per the state of the IGP topology. Adjacency segments represent hop over specific adjacency between two nodes in the IGP. A prefix segment is typically multi-hop path while an adjacency segment, in most of the cases, is one-hop path. These segments act as topological sub-paths that can be combined together to form the required path.. Reference: https://datatracker.ietf.org/doc/html/rfc8667.:w. An implementation may advertise Router Capability with default values if user does not even set the properties of Router Capability and Segment Routing Capability. Optional Segment Routing (SR).

        Returns: IsisSegmentRouting
        """
        return self._get_property("segment_routing", IsisSegmentRouting)


class DeviceIsisMultiInstance(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "iid": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "itids": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "iid": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, iid=1, itids=None):
        super(DeviceIsisMultiInstance, self).__init__()
        self._parent = parent
        self._set_property("iid", iid)
        self._set_property("itids", itids)

    def set(self, iid=None, itids=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def iid(self):
        # type: () -> int
        """iid getter

        Instance Identifier (IID) TLV will associate PDU with an ISIS instance by using unique 16-bit number and including one or more Instance-Specific Topology Identifiers (ITIDs).

        Returns: int
        """
        return self._get_property("iid")

    @iid.setter
    def iid(self, value):
        """iid setter

        Instance Identifier (IID) TLV will associate PDU with an ISIS instance by using unique 16-bit number and including one or more Instance-Specific Topology Identifiers (ITIDs).

        value: int
        """
        self._set_property("iid", value)

    @property
    def itids(self):
        # type: () -> List[int]
        """itids getter

        This contains one or more ITIDs that will be advertised in IID TLV.

        Returns: List[int]
        """
        return self._get_property("itids")

    @itids.setter
    def itids(self, value):
        """itids setter

        This contains one or more ITIDs that will be advertised in IID TLV.

        value: List[int]
        """
        self._set_property("itids", value)


class IsisInterface(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "eth_name": {"type": str},
        "metric": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "network_type": {
            "type": str,
            "enum": [
                "broadcast",
                "point_to_point",
            ],
        },
        "level_type": {
            "type": str,
            "enum": [
                "level_1",
                "level_2",
                "level_1_2",
            ],
        },
        "l1_settings": {"type": "IsisInterfaceLevel"},
        "l2_settings": {"type": "IsisInterfaceLevel"},
        "multi_topology_ids": {"type": "IsisMTIter"},
        "traffic_engineering": {"type": "LinkStateTEIter"},
        "authentication": {"type": "IsisInterfaceAuthentication"},
        "advanced": {"type": "IsisInterfaceAdvanced"},
        "link_protection": {"type": "IsisInterfaceLinkProtection"},
        "srlg_values": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "maximum": 16777215,
        },
        "name": {"type": str},
        "adjacency_sids": {"type": "IsisInterfaceAdjacencySidIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("eth_name", "name")  # type: tuple(str)

    _DEFAULTS = {
        "metric": 10,
        "network_type": "broadcast",
        "level_type": "level_2",
    }  # type: Dict[str, Union(type)]

    BROADCAST = "broadcast"  # type: str
    POINT_TO_POINT = "point_to_point"  # type: str

    LEVEL_1 = "level_1"  # type: str
    LEVEL_2 = "level_2"  # type: str
    LEVEL_1_2 = "level_1_2"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        super(IsisInterface, self).__init__()
        self._parent = parent
        self._set_property("eth_name", eth_name)
        self._set_property("metric", metric)
        self._set_property("network_type", network_type)
        self._set_property("level_type", level_type)
        self._set_property("srlg_values", srlg_values)
        self._set_property("name", name)

    def set(
        self,
        eth_name=None,
        metric=None,
        network_type=None,
        level_type=None,
        srlg_values=None,
        name=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def eth_name(self):
        # type: () -> str
        """eth_name getter

        The unique name of the Ethernet interface on which ISIS is running. Two ISIS interfaces cannot share the same Ethernet. The underlying Ethernet Interface can an emulated or simulated interface. simulated ethernet interface can be assumed to be connected by primary (internal to simulated topology) or secondary link (connected to device behind different simulated topology).. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        Returns: str
        """
        return self._get_property("eth_name")

    @eth_name.setter
    def eth_name(self, value):
        """eth_name setter

        The unique name of the Ethernet interface on which ISIS is running. Two ISIS interfaces cannot share the same Ethernet. The underlying Ethernet Interface can an emulated or simulated interface. simulated ethernet interface can be assumed to be connected by primary (internal to simulated topology) or secondary link (connected to device behind different simulated topology).. x-constraint:. /components/schemas/Device.Ethernet/properties/name. . x-constraint:. /components/schemas/Device.Ethernet/properties/name.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property eth_name as None")
        self._set_property("eth_name", value)

    @property
    def metric(self):
        # type: () -> int
        """metric getter

        The default metric cost for the interface.

        Returns: int
        """
        return self._get_property("metric")

    @metric.setter
    def metric(self, value):
        """metric setter

        The default metric cost for the interface.

        value: int
        """
        self._set_property("metric", value)

    @property
    def network_type(self):
        # type: () -> Union[Literal["broadcast"], Literal["point_to_point"]]
        """network_type getter

        The type of network link.

        Returns: Union[Literal["broadcast"], Literal["point_to_point"]]
        """
        return self._get_property("network_type")

    @network_type.setter
    def network_type(self, value):
        """network_type setter

        The type of network link.

        value: Union[Literal["broadcast"], Literal["point_to_point"]]
        """
        self._set_property("network_type", value)

    @property
    def level_type(self):
        # type: () -> Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """level_type getter

        This indicates whether this router is participating in Level-1 (L1), Level-2 (L2) or both L1 and L2 domains on this interface.

        Returns: Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """
        return self._get_property("level_type")

    @level_type.setter
    def level_type(self, value):
        """level_type setter

        This indicates whether this router is participating in Level-1 (L1), Level-2 (L2) or both L1 and L2 domains on this interface.

        value: Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]]
        """
        self._set_property("level_type", value)

    @property
    def l1_settings(self):
        # type: () -> IsisInterfaceLevel
        """l1_settings getter

        Configuration for the properties of Level Hello.Configuration for the properties of Level Hello.Configuration for the properties of Level Hello.Settings of Level Hello.

        Returns: IsisInterfaceLevel
        """
        return self._get_property("l1_settings", IsisInterfaceLevel)

    @property
    def l2_settings(self):
        # type: () -> IsisInterfaceLevel
        """l2_settings getter

        Configuration for the properties of Level Hello.Configuration for the properties of Level Hello.Configuration for the properties of Level Hello.Settings of Level Hello.

        Returns: IsisInterfaceLevel
        """
        return self._get_property("l2_settings", IsisInterfaceLevel)

    @property
    def multi_topology_ids(self):
        # type: () -> IsisMTIter
        """multi_topology_ids getter

        Contains the properties of multiple topologies.

        Returns: IsisMTIter
        """
        return self._get_property(
            "multi_topology_ids", IsisMTIter, self._parent, self._choice
        )

    @property
    def traffic_engineering(self):
        # type: () -> LinkStateTEIter
        """traffic_engineering getter

        Contains list of Traffic Engineering attributes.

        Returns: LinkStateTEIter
        """
        return self._get_property(
            "traffic_engineering", LinkStateTEIter, self._parent, self._choice
        )

    @property
    def authentication(self):
        # type: () -> IsisInterfaceAuthentication
        """authentication getter

        Optional container for circuit authentication properties.Optional container for circuit authentication properties.Optional container for circuit authentication properties.The Circuit authentication method used for the interfaces on this emulated ISIS v4/v6 router.

        Returns: IsisInterfaceAuthentication
        """
        return self._get_property("authentication", IsisInterfaceAuthentication)

    @property
    def advanced(self):
        # type: () -> IsisInterfaceAdvanced
        """advanced getter

        Optional container for advanced interface properties.Optional container for advanced interface properties.Optional container for advanced interface properties.Optional container for advanced interface properties.

        Returns: IsisInterfaceAdvanced
        """
        return self._get_property("advanced", IsisInterfaceAdvanced)

    @property
    def link_protection(self):
        # type: () -> IsisInterfaceLinkProtection
        """link_protection getter

        Optional container for the link protection sub TLV (type 20).Optional container for the link protection sub TLV (type 20).Optional container for the link protection sub TLV (type 20).Link protection on the ISIS link between two interfaces.

        Returns: IsisInterfaceLinkProtection
        """
        return self._get_property("link_protection", IsisInterfaceLinkProtection)

    @property
    def srlg_values(self):
        # type: () -> List[int]
        """srlg_values getter

        This contains list of SRLG values for the link between two interfaces.

        Returns: List[int]
        """
        return self._get_property("srlg_values")

    @srlg_values.setter
    def srlg_values(self, value):
        """srlg_values setter

        This contains list of SRLG values for the link between two interfaces.

        value: List[int]
        """
        self._set_property("srlg_values", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def adjacency_sids(self):
        # type: () -> IsisInterfaceAdjacencySidIter
        """adjacency_sids getter

        List of Adjacency Segment Identifier (Adj-SID) sub-TLVs.

        Returns: IsisInterfaceAdjacencySidIter
        """
        return self._get_property(
            "adjacency_sids", IsisInterfaceAdjacencySidIter, self._parent, self._choice
        )


class IsisInterfaceLevel(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "priority": {
            "type": int,
            "format": "uint32",
        },
        "hello_interval": {
            "type": int,
            "format": "uint32",
        },
        "dead_interval": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "priority": 0,
        "hello_interval": 10,
        "dead_interval": 30,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, priority=0, hello_interval=10, dead_interval=30):
        super(IsisInterfaceLevel, self).__init__()
        self._parent = parent
        self._set_property("priority", priority)
        self._set_property("hello_interval", hello_interval)
        self._set_property("dead_interval", dead_interval)

    def set(self, priority=None, hello_interval=None, dead_interval=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def priority(self):
        # type: () -> int
        """priority getter

        The Priority setting in Level LAN Hellos for Designated Router election.

        Returns: int
        """
        return self._get_property("priority")

    @priority.setter
    def priority(self, value):
        """priority setter

        The Priority setting in Level LAN Hellos for Designated Router election.

        value: int
        """
        self._set_property("priority", value)

    @property
    def hello_interval(self):
        # type: () -> int
        """hello_interval getter

        The Hello interval for Level Hello messages, in seconds.

        Returns: int
        """
        return self._get_property("hello_interval")

    @hello_interval.setter
    def hello_interval(self, value):
        """hello_interval setter

        The Hello interval for Level Hello messages, in seconds.

        value: int
        """
        self._set_property("hello_interval", value)

    @property
    def dead_interval(self):
        # type: () -> int
        """dead_interval getter

        The Dead (Holding Time) interval for Level Hello messages, in seconds.

        Returns: int
        """
        return self._get_property("dead_interval")

    @dead_interval.setter
    def dead_interval(self, value):
        """dead_interval setter

        The Dead (Holding Time) interval for Level Hello messages, in seconds.

        value: int
        """
        self._set_property("dead_interval", value)


class IsisMT(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mt_id": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "link_metric": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "mt_id": 0,
        "link_metric": 10,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, mt_id=0, link_metric=10):
        super(IsisMT, self).__init__()
        self._parent = parent
        self._set_property("mt_id", mt_id)
        self._set_property("link_metric", link_metric)

    def set(self, mt_id=None, link_metric=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def mt_id(self):
        # type: () -> int
        """mt_id getter

        The Multi Topology ID for one of the topologies supported on the ISIS interface.

        Returns: int
        """
        return self._get_property("mt_id")

    @mt_id.setter
    def mt_id(self, value):
        """mt_id setter

        The Multi Topology ID for one of the topologies supported on the ISIS interface.

        value: int
        """
        self._set_property("mt_id", value)

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        Specifies the link metric for this topology on the ISIS interface.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        Specifies the link metric for this topology on the ISIS interface.

        value: int
        """
        self._set_property("link_metric", value)


class IsisMTIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisMTIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisMT]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisMTIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisMT
        return self._next()

    def next(self):
        # type: () -> IsisMT
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisMT):
            raise Exception("Item is not an instance of IsisMT")

    def mt(self, mt_id=0, link_metric=10):
        # type: (int,int) -> IsisMTIter
        """Factory method that creates an instance of the IsisMT class

        Configuration of properties per interface per topology when multiple topologies are configured in an ISIS router.. in ISIS router.

        Returns: IsisMTIter
        """
        item = IsisMT(parent=self._parent, mt_id=mt_id, link_metric=link_metric)
        self._add(item)
        return self

    def add(self, mt_id=0, link_metric=10):
        # type: (int,int) -> IsisMT
        """Add method that creates and returns an instance of the IsisMT class

        Configuration of properties per interface per topology when multiple topologies are configured in an ISIS router.. in ISIS router.

        Returns: IsisMT
        """
        item = IsisMT(parent=self._parent, mt_id=mt_id, link_metric=link_metric)
        self._add(item)
        return item


class LinkStateTE(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "administrative_group": {
            "type": str,
            "format": "hex",
        },
        "metric_level": {
            "type": int,
            "format": "uint32",
        },
        "max_bandwith": {
            "type": int,
            "format": "uint32",
        },
        "max_reservable_bandwidth": {
            "type": int,
            "format": "uint32",
        },
        "priority_bandwidths": {"type": "LinkStatepriorityBandwidths"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "administrative_group": "00000000",
        "metric_level": 0,
        "max_bandwith": 125000000,
        "max_reservable_bandwidth": 125000000,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        super(LinkStateTE, self).__init__()
        self._parent = parent
        self._set_property("administrative_group", administrative_group)
        self._set_property("metric_level", metric_level)
        self._set_property("max_bandwith", max_bandwith)
        self._set_property("max_reservable_bandwidth", max_reservable_bandwidth)

    def set(
        self,
        administrative_group=None,
        metric_level=None,
        max_bandwith=None,
        max_reservable_bandwidth=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def administrative_group(self):
        # type: () -> str
        """administrative_group getter

        The Administrative group sub-TLV (sub-TLV 3). It is 4-octet user-defined bit mask used to assign administrative group numbers to the interface, for use in assigning colors and resource classes. Each set bit corresponds to single administrative group for this interface. The settings translate into Group numbers, which range from to 31 (integers).

        Returns: str
        """
        return self._get_property("administrative_group")

    @administrative_group.setter
    def administrative_group(self, value):
        """administrative_group setter

        The Administrative group sub-TLV (sub-TLV 3). It is 4-octet user-defined bit mask used to assign administrative group numbers to the interface, for use in assigning colors and resource classes. Each set bit corresponds to single administrative group for this interface. The settings translate into Group numbers, which range from to 31 (integers).

        value: str
        """
        self._set_property("administrative_group", value)

    @property
    def metric_level(self):
        # type: () -> int
        """metric_level getter

        The user-assigned link metric for Traffic Engineering.

        Returns: int
        """
        return self._get_property("metric_level")

    @metric_level.setter
    def metric_level(self, value):
        """metric_level setter

        The user-assigned link metric for Traffic Engineering.

        value: int
        """
        self._set_property("metric_level", value)

    @property
    def max_bandwith(self):
        # type: () -> int
        """max_bandwith getter

        The maximum link bandwidth (sub-TLV 9) in bytes/sec allowed for this link for direction.

        Returns: int
        """
        return self._get_property("max_bandwith")

    @max_bandwith.setter
    def max_bandwith(self, value):
        """max_bandwith setter

        The maximum link bandwidth (sub-TLV 9) in bytes/sec allowed for this link for direction.

        value: int
        """
        self._set_property("max_bandwith", value)

    @property
    def max_reservable_bandwidth(self):
        # type: () -> int
        """max_reservable_bandwidth getter

        The maximum link bandwidth (sub-TLV 10) in bytes/sec allowed for this link in direction.

        Returns: int
        """
        return self._get_property("max_reservable_bandwidth")

    @max_reservable_bandwidth.setter
    def max_reservable_bandwidth(self, value):
        """max_reservable_bandwidth setter

        The maximum link bandwidth (sub-TLV 10) in bytes/sec allowed for this link in direction.

        value: int
        """
        self._set_property("max_reservable_bandwidth", value)

    @property
    def priority_bandwidths(self):
        # type: () -> LinkStatepriorityBandwidths
        """priority_bandwidths getter

        Specifies the amount of bandwidth that can be reserved with setup priority of . through 7, arranged in increasing order with priority having highest priority. In ISIS, this is sent in sub-TLV (11) of Extended IS Reachability TLV. Specifies the amount of bandwidth that can be reserved with setup priority of . through 7, arranged in increasing order with priority having highest priority. In ISIS, this is sent in sub-TLV (11) of Extended IS Reachability TLV. Specifies the amount of bandwidth that can be reserved with setup priority of . through 7, arranged in increasing order with priority having highest priority. In ISIS, this is sent in sub-TLV (11) of Extended IS Reachability TLV. Configuration of bandwidths of priority through priority 7.

        Returns: LinkStatepriorityBandwidths
        """
        return self._get_property("priority_bandwidths", LinkStatepriorityBandwidths)


class LinkStatepriorityBandwidths(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "pb0": {
            "type": int,
            "format": "uint32",
        },
        "pb1": {
            "type": int,
            "format": "uint32",
        },
        "pb2": {
            "type": int,
            "format": "uint32",
        },
        "pb3": {
            "type": int,
            "format": "uint32",
        },
        "pb4": {
            "type": int,
            "format": "uint32",
        },
        "pb5": {
            "type": int,
            "format": "uint32",
        },
        "pb6": {
            "type": int,
            "format": "uint32",
        },
        "pb7": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "pb0": 125000000,
        "pb1": 125000000,
        "pb2": 125000000,
        "pb3": 125000000,
        "pb4": 125000000,
        "pb5": 125000000,
        "pb6": 125000000,
        "pb7": 125000000,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        pb0=125000000,
        pb1=125000000,
        pb2=125000000,
        pb3=125000000,
        pb4=125000000,
        pb5=125000000,
        pb6=125000000,
        pb7=125000000,
    ):
        super(LinkStatepriorityBandwidths, self).__init__()
        self._parent = parent
        self._set_property("pb0", pb0)
        self._set_property("pb1", pb1)
        self._set_property("pb2", pb2)
        self._set_property("pb3", pb3)
        self._set_property("pb4", pb4)
        self._set_property("pb5", pb5)
        self._set_property("pb6", pb6)
        self._set_property("pb7", pb7)

    def set(
        self,
        pb0=None,
        pb1=None,
        pb2=None,
        pb3=None,
        pb4=None,
        pb5=None,
        pb6=None,
        pb7=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def pb0(self):
        # type: () -> int
        """pb0 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 0.

        Returns: int
        """
        return self._get_property("pb0")

    @pb0.setter
    def pb0(self, value):
        """pb0 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 0.

        value: int
        """
        self._set_property("pb0", value)

    @property
    def pb1(self):
        # type: () -> int
        """pb1 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 1.

        Returns: int
        """
        return self._get_property("pb1")

    @pb1.setter
    def pb1(self, value):
        """pb1 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 1.

        value: int
        """
        self._set_property("pb1", value)

    @property
    def pb2(self):
        # type: () -> int
        """pb2 getter

        Specify the amount of bandwidth that can be reserved for the Priority 2.

        Returns: int
        """
        return self._get_property("pb2")

    @pb2.setter
    def pb2(self, value):
        """pb2 setter

        Specify the amount of bandwidth that can be reserved for the Priority 2.

        value: int
        """
        self._set_property("pb2", value)

    @property
    def pb3(self):
        # type: () -> int
        """pb3 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 3.

        Returns: int
        """
        return self._get_property("pb3")

    @pb3.setter
    def pb3(self, value):
        """pb3 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 3.

        value: int
        """
        self._set_property("pb3", value)

    @property
    def pb4(self):
        # type: () -> int
        """pb4 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 4.

        Returns: int
        """
        return self._get_property("pb4")

    @pb4.setter
    def pb4(self, value):
        """pb4 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 4.

        value: int
        """
        self._set_property("pb4", value)

    @property
    def pb5(self):
        # type: () -> int
        """pb5 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 5.

        Returns: int
        """
        return self._get_property("pb5")

    @pb5.setter
    def pb5(self, value):
        """pb5 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 5.

        value: int
        """
        self._set_property("pb5", value)

    @property
    def pb6(self):
        # type: () -> int
        """pb6 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 6.

        Returns: int
        """
        return self._get_property("pb6")

    @pb6.setter
    def pb6(self, value):
        """pb6 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 6.

        value: int
        """
        self._set_property("pb6", value)

    @property
    def pb7(self):
        # type: () -> int
        """pb7 getter

        Specifies the amount of bandwidth that can be reserved for the Priority 7.

        Returns: int
        """
        return self._get_property("pb7")

    @pb7.setter
    def pb7(self, value):
        """pb7 setter

        Specifies the amount of bandwidth that can be reserved for the Priority 7.

        value: int
        """
        self._set_property("pb7", value)


class LinkStateTEIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(LinkStateTEIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[LinkStateTE]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> LinkStateTEIter
        return self._iter()

    def __next__(self):
        # type: () -> LinkStateTE
        return self._next()

    def next(self):
        # type: () -> LinkStateTE
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, LinkStateTE):
            raise Exception("Item is not an instance of LinkStateTE")

    def te(
        self,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        # type: (str,int,int,int) -> LinkStateTEIter
        """Factory method that creates an instance of the LinkStateTE class

        A container for Traffic Engineering properties on interface.

        Returns: LinkStateTEIter
        """
        item = LinkStateTE(
            parent=self._parent,
            administrative_group=administrative_group,
            metric_level=metric_level,
            max_bandwith=max_bandwith,
            max_reservable_bandwidth=max_reservable_bandwidth,
        )
        self._add(item)
        return self

    def add(
        self,
        administrative_group="00000000",
        metric_level=0,
        max_bandwith=125000000,
        max_reservable_bandwidth=125000000,
    ):
        # type: (str,int,int,int) -> LinkStateTE
        """Add method that creates and returns an instance of the LinkStateTE class

        A container for Traffic Engineering properties on interface.

        Returns: LinkStateTE
        """
        item = LinkStateTE(
            parent=self._parent,
            administrative_group=administrative_group,
            metric_level=metric_level,
            max_bandwith=max_bandwith,
            max_reservable_bandwidth=max_reservable_bandwidth,
        )
        self._add(item)
        return item


class IsisInterfaceAuthentication(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auth_type": {
            "type": str,
            "enum": [
                "md5",
                "password",
            ],
        },
        "md5": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
        "password": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("auth_type",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    MD5 = "md5"  # type: str
    PASSWORD = "password"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, auth_type=None, md5=None, password=None):
        super(IsisInterfaceAuthentication, self).__init__()
        self._parent = parent
        self._set_property("auth_type", auth_type)
        self._set_property("md5", md5)
        self._set_property("password", password)

    def set(self, auth_type=None, md5=None, password=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def auth_type(self):
        # type: () -> Union[Literal["md5"], Literal["password"]]
        """auth_type getter

        The circuit authentication method.

        Returns: Union[Literal["md5"], Literal["password"]]
        """
        return self._get_property("auth_type")

    @auth_type.setter
    def auth_type(self, value):
        """auth_type setter

        The circuit authentication method.

        value: Union[Literal["md5"], Literal["password"]]
        """
        if value is None:
            raise TypeError("Cannot set required property auth_type as None")
        self._set_property("auth_type", value)

    @property
    def md5(self):
        # type: () -> str
        """md5 getter

        MD5 key to be used for authentication.

        Returns: str
        """
        return self._get_property("md5")

    @md5.setter
    def md5(self, value):
        """md5 setter

        MD5 key to be used for authentication.

        value: str
        """
        self._set_property("md5", value)

    @property
    def password(self):
        # type: () -> str
        """password getter

        The password, in clear text, to be used for Authentication.

        Returns: str
        """
        return self._get_property("password")

    @password.setter
    def password(self, value):
        """password setter

        The password, in clear text, to be used for Authentication.

        value: str
        """
        self._set_property("password", value)


class IsisInterfaceAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auto_adjust_mtu": {"type": bool},
        "auto_adjust_area": {"type": bool},
        "auto_adjust_supported_protocols": {"type": bool},
        "enable_3way_handshake": {"type": bool},
        "p2p_hellos_to_unicast_mac": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "auto_adjust_mtu": True,
        "auto_adjust_area": True,
        "auto_adjust_supported_protocols": False,
        "enable_3way_handshake": True,
        "p2p_hellos_to_unicast_mac": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        auto_adjust_mtu=True,
        auto_adjust_area=True,
        auto_adjust_supported_protocols=False,
        enable_3way_handshake=True,
        p2p_hellos_to_unicast_mac=False,
    ):
        super(IsisInterfaceAdvanced, self).__init__()
        self._parent = parent
        self._set_property("auto_adjust_mtu", auto_adjust_mtu)
        self._set_property("auto_adjust_area", auto_adjust_area)
        self._set_property(
            "auto_adjust_supported_protocols", auto_adjust_supported_protocols
        )
        self._set_property("enable_3way_handshake", enable_3way_handshake)
        self._set_property("p2p_hellos_to_unicast_mac", p2p_hellos_to_unicast_mac)

    def set(
        self,
        auto_adjust_mtu=None,
        auto_adjust_area=None,
        auto_adjust_supported_protocols=None,
        enable_3way_handshake=None,
        p2p_hellos_to_unicast_mac=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def auto_adjust_mtu(self):
        # type: () -> bool
        """auto_adjust_mtu getter

        If padded Hello message is received on the interface, the length of the Hello packets sent out on that interface is adjusted to match.

        Returns: bool
        """
        return self._get_property("auto_adjust_mtu")

    @auto_adjust_mtu.setter
    def auto_adjust_mtu(self, value):
        """auto_adjust_mtu setter

        If padded Hello message is received on the interface, the length of the Hello packets sent out on that interface is adjusted to match.

        value: bool
        """
        self._set_property("auto_adjust_mtu", value)

    @property
    def auto_adjust_area(self):
        # type: () -> bool
        """auto_adjust_area getter

        If Level Hello is received on this emulated router for an area not currently in its area list, an area from the received Hello is added to that list. This ensures an area match for all future Level Hellos from the source L1 router.

        Returns: bool
        """
        return self._get_property("auto_adjust_area")

    @auto_adjust_area.setter
    def auto_adjust_area(self, value):
        """auto_adjust_area setter

        If Level Hello is received on this emulated router for an area not currently in its area list, an area from the received Hello is added to that list. This ensures an area match for all future Level Hellos from the source L1 router.

        value: bool
        """
        self._set_property("auto_adjust_area", value)

    @property
    def auto_adjust_supported_protocols(self):
        # type: () -> bool
        """auto_adjust_supported_protocols getter

        If Hello message listing supported protocols is received on this emulated router, the supported protocols advertised by this router are changed to match exactly.

        Returns: bool
        """
        return self._get_property("auto_adjust_supported_protocols")

    @auto_adjust_supported_protocols.setter
    def auto_adjust_supported_protocols(self, value):
        """auto_adjust_supported_protocols setter

        If Hello message listing supported protocols is received on this emulated router, the supported protocols advertised by this router are changed to match exactly.

        value: bool
        """
        self._set_property("auto_adjust_supported_protocols", value)

    @property
    def enable_3way_handshake(self):
        # type: () -> bool
        """enable_3way_handshake getter

        If it is true, the Point-to-Point circuit will include 3-way TLV in its Point-to-Point IIH and attempt to establish the adjacency as specified in RFC 5303. This field is not applicable if network_type is set to 'broadcast' type in ISIS interface.

        Returns: bool
        """
        return self._get_property("enable_3way_handshake")

    @enable_3way_handshake.setter
    def enable_3way_handshake(self, value):
        """enable_3way_handshake setter

        If it is true, the Point-to-Point circuit will include 3-way TLV in its Point-to-Point IIH and attempt to establish the adjacency as specified in RFC 5303. This field is not applicable if network_type is set to 'broadcast' type in ISIS interface.

        value: bool
        """
        self._set_property("enable_3way_handshake", value)

    @property
    def p2p_hellos_to_unicast_mac(self):
        # type: () -> bool
        """p2p_hellos_to_unicast_mac getter

        If it is true, the Point-to-Point Hello messages will be sent to the unicast MAC address.

        Returns: bool
        """
        return self._get_property("p2p_hellos_to_unicast_mac")

    @p2p_hellos_to_unicast_mac.setter
    def p2p_hellos_to_unicast_mac(self, value):
        """p2p_hellos_to_unicast_mac setter

        If it is true, the Point-to-Point Hello messages will be sent to the unicast MAC address.

        value: bool
        """
        self._set_property("p2p_hellos_to_unicast_mac", value)


class IsisInterfaceLinkProtection(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "extra_traffic": {"type": bool},
        "unprotected": {"type": bool},
        "shared": {"type": bool},
        "dedicated_1_to_1": {"type": bool},
        "dedicated_1_plus_1": {"type": bool},
        "enhanced": {"type": bool},
        "reserved_40": {"type": bool},
        "reserved_80": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "extra_traffic": False,
        "unprotected": False,
        "shared": False,
        "dedicated_1_to_1": False,
        "dedicated_1_plus_1": False,
        "enhanced": False,
        "reserved_40": False,
        "reserved_80": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        extra_traffic=False,
        unprotected=False,
        shared=False,
        dedicated_1_to_1=False,
        dedicated_1_plus_1=False,
        enhanced=False,
        reserved_40=False,
        reserved_80=False,
    ):
        super(IsisInterfaceLinkProtection, self).__init__()
        self._parent = parent
        self._set_property("extra_traffic", extra_traffic)
        self._set_property("unprotected", unprotected)
        self._set_property("shared", shared)
        self._set_property("dedicated_1_to_1", dedicated_1_to_1)
        self._set_property("dedicated_1_plus_1", dedicated_1_plus_1)
        self._set_property("enhanced", enhanced)
        self._set_property("reserved_40", reserved_40)
        self._set_property("reserved_80", reserved_80)

    def set(
        self,
        extra_traffic=None,
        unprotected=None,
        shared=None,
        dedicated_1_to_1=None,
        dedicated_1_plus_1=None,
        enhanced=None,
        reserved_40=None,
        reserved_80=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def extra_traffic(self):
        # type: () -> bool
        """extra_traffic getter

        Enable this to protect other link or links. LSPs on link of this type are lost. if any of the links fail.

        Returns: bool
        """
        return self._get_property("extra_traffic")

    @extra_traffic.setter
    def extra_traffic(self, value):
        """extra_traffic setter

        Enable this to protect other link or links. LSPs on link of this type are lost. if any of the links fail.

        value: bool
        """
        self._set_property("extra_traffic", value)

    @property
    def unprotected(self):
        # type: () -> bool
        """unprotected getter

        Enabling this signifies that there is no other link protecting this link. LSPs on link of this type are lost if the link fails.

        Returns: bool
        """
        return self._get_property("unprotected")

    @unprotected.setter
    def unprotected(self, value):
        """unprotected setter

        Enabling this signifies that there is no other link protecting this link. LSPs on link of this type are lost if the link fails.

        value: bool
        """
        self._set_property("unprotected", value)

    @property
    def shared(self):
        # type: () -> bool
        """shared getter

        Enable this to share the Extra Traffic links between one or more links of type Shared.There are one or more disjoint links of type Extra Traffic that are protecting this link.

        Returns: bool
        """
        return self._get_property("shared")

    @shared.setter
    def shared(self, value):
        """shared setter

        Enable this to share the Extra Traffic links between one or more links of type Shared.There are one or more disjoint links of type Extra Traffic that are protecting this link.

        value: bool
        """
        self._set_property("shared", value)

    @property
    def dedicated_1_to_1(self):
        # type: () -> bool
        """dedicated_1_to_1 getter

        Enabling this signifies that there is one dedicated disjoint link of type Extra Traffic that is protecting this link.

        Returns: bool
        """
        return self._get_property("dedicated_1_to_1")

    @dedicated_1_to_1.setter
    def dedicated_1_to_1(self, value):
        """dedicated_1_to_1 setter

        Enabling this signifies that there is one dedicated disjoint link of type Extra Traffic that is protecting this link.

        value: bool
        """
        self._set_property("dedicated_1_to_1", value)

    @property
    def dedicated_1_plus_1(self):
        # type: () -> bool
        """dedicated_1_plus_1 getter

        Enabling this signifies that dedicated disjoint link is protecting this link. However, the protecting link is not advertised in the link state database and is therefore not available for the routing of LSPs.

        Returns: bool
        """
        return self._get_property("dedicated_1_plus_1")

    @dedicated_1_plus_1.setter
    def dedicated_1_plus_1(self, value):
        """dedicated_1_plus_1 setter

        Enabling this signifies that dedicated disjoint link is protecting this link. However, the protecting link is not advertised in the link state database and is therefore not available for the routing of LSPs.

        value: bool
        """
        self._set_property("dedicated_1_plus_1", value)

    @property
    def enhanced(self):
        # type: () -> bool
        """enhanced getter

        Enabling this signifies that protection scheme that is more reliable than Dedicated 1+1.

        Returns: bool
        """
        return self._get_property("enhanced")

    @enhanced.setter
    def enhanced(self, value):
        """enhanced setter

        Enabling this signifies that protection scheme that is more reliable than Dedicated 1+1.

        value: bool
        """
        self._set_property("enhanced", value)

    @property
    def reserved_40(self):
        # type: () -> bool
        """reserved_40 getter

        This is Protection Scheme with value 0x40.

        Returns: bool
        """
        return self._get_property("reserved_40")

    @reserved_40.setter
    def reserved_40(self, value):
        """reserved_40 setter

        This is Protection Scheme with value 0x40.

        value: bool
        """
        self._set_property("reserved_40", value)

    @property
    def reserved_80(self):
        # type: () -> bool
        """reserved_80 getter

        This is Protection Scheme with value 0x80.

        Returns: bool
        """
        return self._get_property("reserved_80")

    @reserved_80.setter
    def reserved_80(self, value):
        """reserved_80 setter

        This is Protection Scheme with value 0x80.

        value: bool
        """
        self._set_property("reserved_80", value)


class IsisInterfaceAdjacencySid(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "sid_values",
                "sid_indices",
            ],
        },
        "sid_values": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "minimum": 16,
            "maximum": 1048575,
        },
        "sid_indices": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "f_flag": {"type": bool},
        "b_flag": {"type": bool},
        "l_flag": {"type": bool},
        "s_flag": {"type": bool},
        "p_flag": {"type": bool},
        "weight": {
            "type": int,
            "format": "uint32",
            "minimum": 0,
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "sid_values",
        "f_flag": True,
        "b_flag": False,
        "l_flag": True,
        "s_flag": False,
        "p_flag": False,
        "weight": 0,
    }  # type: Dict[str, Union(type)]

    SID_VALUES = "sid_values"  # type: str
    SID_INDICES = "sid_indices"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        sid_values=None,
        sid_indices=None,
        f_flag=True,
        b_flag=False,
        l_flag=True,
        s_flag=False,
        p_flag=False,
        weight=0,
    ):
        super(IsisInterfaceAdjacencySid, self).__init__()
        self._parent = parent
        self._set_property("sid_values", sid_values)
        self._set_property("sid_indices", sid_indices)
        self._set_property("f_flag", f_flag)
        self._set_property("b_flag", b_flag)
        self._set_property("l_flag", l_flag)
        self._set_property("s_flag", s_flag)
        self._set_property("p_flag", p_flag)
        self._set_property("weight", weight)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(
        self,
        sid_values=None,
        sid_indices=None,
        f_flag=None,
        b_flag=None,
        l_flag=None,
        s_flag=None,
        p_flag=None,
        weight=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["sid_indices"], Literal["sid_values"]]
        """choice getter

        Choice of whether Adjacency-SID carries and absolute value or relative index to the SRGB/SRLB Ranges.. Please refer to device.isis.segment_routing.router_capability.sr_capability.srgb_ranges for the Segment Routing Global Block (SRGB) Descriptor and. device.isis.segment_routing.router_capability.srlb_ranges for the SR Local Block (SRLB).. user needs to configure at least one entry of SID value or SID index.. If no entry is configured, then an implementation may advertise appropriate default SID Value/Index based on the choice. e.g. the first value. from the SRGB or SRLB range.. sid_values: Adjacency-SID carries one or more values and then v_flag is set by the implementation.. sid_indices: Adjacency-SID carries one or more indices and then v_flag is unset by the implementation.

        Returns: Union[Literal["sid_indices"], Literal["sid_values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Choice of whether Adjacency-SID carries and absolute value or relative index to the SRGB/SRLB Ranges.. Please refer to device.isis.segment_routing.router_capability.sr_capability.srgb_ranges for the Segment Routing Global Block (SRGB) Descriptor and. device.isis.segment_routing.router_capability.srlb_ranges for the SR Local Block (SRLB).. user needs to configure at least one entry of SID value or SID index.. If no entry is configured, then an implementation may advertise appropriate default SID Value/Index based on the choice. e.g. the first value. from the SRGB or SRLB range.. sid_values: Adjacency-SID carries one or more values and then v_flag is set by the implementation.. sid_indices: Adjacency-SID carries one or more indices and then v_flag is unset by the implementation.

        value: Union[Literal["sid_indices"], Literal["sid_values"]]
        """
        self._set_property("choice", value)

    @property
    def sid_values(self):
        # type: () -> List[int]
        """sid_values getter

        The corresponding Adjacency SID as one or more absolute Values for the link.

        Returns: List[int]
        """
        return self._get_property("sid_values")

    @sid_values.setter
    def sid_values(self, value):
        """sid_values setter

        The corresponding Adjacency SID as one or more absolute Values for the link.

        value: List[int]
        """
        self._set_property("sid_values", value, "sid_values")

    @property
    def sid_indices(self):
        # type: () -> List[int]
        """sid_indices getter

        One or more adjacency Indices are relative to ranges defined for SRGB or SRLB.

        Returns: List[int]
        """
        return self._get_property("sid_indices")

    @sid_indices.setter
    def sid_indices(self, value):
        """sid_indices setter

        One or more adjacency Indices are relative to ranges defined for SRGB or SRLB.

        value: List[int]
        """
        self._set_property("sid_indices", value, "sid_indices")

    @property
    def f_flag(self):
        # type: () -> bool
        """f_flag getter

        The address family flag. If unset, then the Adj-SID refers. to an adjacency with outgoing IPv4 encapsulation. If set then. the Adj-SID refers to an adjacency with outgoing IPv6. encapsulation.

        Returns: bool
        """
        return self._get_property("f_flag")

    @f_flag.setter
    def f_flag(self, value):
        """f_flag setter

        The address family flag. If unset, then the Adj-SID refers. to an adjacency with outgoing IPv4 encapsulation. If set then. the Adj-SID refers to an adjacency with outgoing IPv6. encapsulation.

        value: bool
        """
        self._set_property("f_flag", value)

    @property
    def b_flag(self):
        # type: () -> bool
        """b_flag getter

        The backup flag. If set, the Adj-SID is eligible for protection.

        Returns: bool
        """
        return self._get_property("b_flag")

    @b_flag.setter
    def b_flag(self, value):
        """b_flag setter

        The backup flag. If set, the Adj-SID is eligible for protection.

        value: bool
        """
        self._set_property("b_flag", value)

    @property
    def l_flag(self):
        # type: () -> bool
        """l_flag getter

        The local flag. If set, then the value/index carried by. the Adj-SID has local significance. In this case, Adjacency_sid is from device.isis.segment_routing.router_capability.srlb_ranges.

        Returns: bool
        """
        return self._get_property("l_flag")

    @l_flag.setter
    def l_flag(self, value):
        """l_flag setter

        The local flag. If set, then the value/index carried by. the Adj-SID has local significance. In this case, Adjacency_sid is from device.isis.segment_routing.router_capability.srlb_ranges.

        value: bool
        """
        self._set_property("l_flag", value)

    @property
    def s_flag(self):
        # type: () -> bool
        """s_flag getter

        The set flag. When set, the S-Flag indicates that the. Adj-SID refers to set of adjacencies (and therefore MAY be. assigned to other adjacencies as well).

        Returns: bool
        """
        return self._get_property("s_flag")

    @s_flag.setter
    def s_flag(self, value):
        """s_flag setter

        The set flag. When set, the S-Flag indicates that the. Adj-SID refers to set of adjacencies (and therefore MAY be. assigned to other adjacencies as well).

        value: bool
        """
        self._set_property("s_flag", value)

    @property
    def p_flag(self):
        # type: () -> bool
        """p_flag getter

        The persistent flag. When set, the P-Flag indicates that. the Adj-SID is persistently allocated, i.e., the Adj-SID value. remains consistent across router restart and/or interface flap.

        Returns: bool
        """
        return self._get_property("p_flag")

    @p_flag.setter
    def p_flag(self, value):
        """p_flag setter

        The persistent flag. When set, the P-Flag indicates that. the Adj-SID is persistently allocated, i.e., the Adj-SID value. remains consistent across router restart and/or interface flap.

        value: bool
        """
        self._set_property("p_flag", value)

    @property
    def weight(self):
        # type: () -> int
        """weight getter

        The value represents the weight of the Adj-SID for the purpose of load balancing.

        Returns: int
        """
        return self._get_property("weight")

    @weight.setter
    def weight(self, value):
        """weight setter

        The value represents the weight of the Adj-SID for the purpose of load balancing.

        value: int
        """
        self._set_property("weight", value)


class IsisInterfaceAdjacencySidIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisInterfaceAdjacencySidIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisInterfaceAdjacencySid]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisInterfaceAdjacencySidIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisInterfaceAdjacencySid
        return self._next()

    def next(self):
        # type: () -> IsisInterfaceAdjacencySid
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisInterfaceAdjacencySid):
            raise Exception("Item is not an instance of IsisInterfaceAdjacencySid")

    def adjacencysid(
        self,
        sid_values=None,
        sid_indices=None,
        f_flag=True,
        b_flag=False,
        l_flag=True,
        s_flag=False,
        p_flag=False,
        weight=0,
    ):
        # type: (List[int],List[int],bool,bool,bool,bool,bool,int) -> IsisInterfaceAdjacencySidIter
        """Factory method that creates an instance of the IsisInterfaceAdjacencySid class

        Optional container for segment routing MPLS settings.. If the container exists then the adjacency SID (segment identifier). sub TLV will be part of the packet.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-adjacency-segment-identifie.

        Returns: IsisInterfaceAdjacencySidIter
        """
        item = IsisInterfaceAdjacencySid(
            parent=self._parent,
            choice=self._choice,
            sid_values=sid_values,
            sid_indices=sid_indices,
            f_flag=f_flag,
            b_flag=b_flag,
            l_flag=l_flag,
            s_flag=s_flag,
            p_flag=p_flag,
            weight=weight,
        )
        self._add(item)
        return self

    def add(
        self,
        sid_values=None,
        sid_indices=None,
        f_flag=True,
        b_flag=False,
        l_flag=True,
        s_flag=False,
        p_flag=False,
        weight=0,
    ):
        # type: (List[int],List[int],bool,bool,bool,bool,bool,int) -> IsisInterfaceAdjacencySid
        """Add method that creates and returns an instance of the IsisInterfaceAdjacencySid class

        Optional container for segment routing MPLS settings.. If the container exists then the adjacency SID (segment identifier). sub TLV will be part of the packet.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-adjacency-segment-identifie.

        Returns: IsisInterfaceAdjacencySid
        """
        item = IsisInterfaceAdjacencySid(
            parent=self._parent,
            choice=self._choice,
            sid_values=sid_values,
            sid_indices=sid_indices,
            f_flag=f_flag,
            b_flag=b_flag,
            l_flag=l_flag,
            s_flag=s_flag,
            p_flag=p_flag,
            weight=weight,
        )
        self._add(item)
        return item


class IsisInterfaceIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisInterfaceIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisInterface]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisInterfaceIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisInterface
        return self._next()

    def next(self):
        # type: () -> IsisInterface
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisInterface):
            raise Exception("Item is not an instance of IsisInterface")

    def interface(
        self,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        # type: (str,int,Union[Literal["broadcast"], Literal["point_to_point"]],Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]],List[int],str) -> IsisInterfaceIter
        """Factory method that creates an instance of the IsisInterface class

        Configuration for single ISIS interface.

        Returns: IsisInterfaceIter
        """
        item = IsisInterface(
            parent=self._parent,
            eth_name=eth_name,
            metric=metric,
            network_type=network_type,
            level_type=level_type,
            srlg_values=srlg_values,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        eth_name=None,
        metric=10,
        network_type="broadcast",
        level_type="level_2",
        srlg_values=None,
        name=None,
    ):
        # type: (str,int,Union[Literal["broadcast"], Literal["point_to_point"]],Union[Literal["level_1"], Literal["level_1_2"], Literal["level_2"]],List[int],str) -> IsisInterface
        """Add method that creates and returns an instance of the IsisInterface class

        Configuration for single ISIS interface.

        Returns: IsisInterface
        """
        item = IsisInterface(
            parent=self._parent,
            eth_name=eth_name,
            metric=metric,
            network_type=network_type,
            level_type=level_type,
            srlg_values=srlg_values,
            name=name,
        )
        self._add(item)
        return item


class IsisBasic(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_te_router_id": {
            "type": str,
            "format": "ipv4",
        },
        "hostname": {"type": str},
        "enable_wide_metric": {"type": bool},
        "learned_lsp_filter": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enable_wide_metric": True,
        "learned_lsp_filter": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        ipv4_te_router_id=None,
        hostname=None,
        enable_wide_metric=True,
        learned_lsp_filter=False,
    ):
        super(IsisBasic, self).__init__()
        self._parent = parent
        self._set_property("ipv4_te_router_id", ipv4_te_router_id)
        self._set_property("hostname", hostname)
        self._set_property("enable_wide_metric", enable_wide_metric)
        self._set_property("learned_lsp_filter", learned_lsp_filter)

    def set(
        self,
        ipv4_te_router_id=None,
        hostname=None,
        enable_wide_metric=None,
        learned_lsp_filter=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ipv4_te_router_id(self):
        # type: () -> str
        """ipv4_te_router_id getter

        IPv4 Traffic Engineering(TE) router id. This address should be configured as an IPv4 Loopback address in 'ipv4_loopbacks' in the Device.

        Returns: str
        """
        return self._get_property("ipv4_te_router_id")

    @ipv4_te_router_id.setter
    def ipv4_te_router_id(self, value):
        """ipv4_te_router_id setter

        IPv4 Traffic Engineering(TE) router id. This address should be configured as an IPv4 Loopback address in 'ipv4_loopbacks' in the Device.

        value: str
        """
        self._set_property("ipv4_te_router_id", value)

    @property
    def hostname(self):
        # type: () -> str
        """hostname getter

        Host name for the router. The host name is transmitted in all the packets sent from the router.

        Returns: str
        """
        return self._get_property("hostname")

    @hostname.setter
    def hostname(self, value):
        """hostname setter

        Host name for the router. The host name is transmitted in all the packets sent from the router.

        value: str
        """
        self._set_property("hostname", value)

    @property
    def enable_wide_metric(self):
        # type: () -> bool
        """enable_wide_metric getter

        When set to true, it allows sending of more detailed metric information for the routes using 32-bit wide values using TLV 135 IP reachability and more detailed reachability information for IS reachability by using TLV 22. The detailed usage is described in RFC3784.

        Returns: bool
        """
        return self._get_property("enable_wide_metric")

    @enable_wide_metric.setter
    def enable_wide_metric(self, value):
        """enable_wide_metric setter

        When set to true, it allows sending of more detailed metric information for the routes using 32-bit wide values using TLV 135 IP reachability and more detailed reachability information for IS reachability by using TLV 22. The detailed usage is described in RFC3784.

        value: bool
        """
        self._set_property("enable_wide_metric", value)

    @property
    def learned_lsp_filter(self):
        # type: () -> bool
        """learned_lsp_filter getter

        Configuration for controlling storage of ISIS learned LSPs are received from the neighbors.

        Returns: bool
        """
        return self._get_property("learned_lsp_filter")

    @learned_lsp_filter.setter
    def learned_lsp_filter(self, value):
        """learned_lsp_filter setter

        Configuration for controlling storage of ISIS learned LSPs are received from the neighbors.

        value: bool
        """
        self._set_property("learned_lsp_filter", value)


class IsisAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enable_hello_padding": {"type": bool},
        "max_area_addresses": {
            "type": int,
            "format": "uint32",
            "maximum": 254,
        },
        "area_addresses": {
            "type": list,
            "itemtype": str,
            "itemformat": "hex",
        },
        "lsp_refresh_rate": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "lsp_lifetime": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "psnp_interval": {
            "type": int,
            "format": "uint32",
            "maximum": 60000,
        },
        "csnp_interval": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 65535000,
        },
        "max_lsp_size": {
            "type": int,
            "format": "uint32",
            "minimum": 64,
            "maximum": 9216,
        },
        "lsp_mgroup_min_trans_interval": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 60000,
        },
        "enable_attached_bit": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enable_hello_padding": True,
        "max_area_addresses": 3,
        "lsp_refresh_rate": 600,
        "lsp_lifetime": 1200,
        "psnp_interval": 2000,
        "csnp_interval": 10000,
        "max_lsp_size": 1492,
        "lsp_mgroup_min_trans_interval": 5000,
        "enable_attached_bit": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        enable_hello_padding=True,
        max_area_addresses=3,
        area_addresses=None,
        lsp_refresh_rate=600,
        lsp_lifetime=1200,
        psnp_interval=2000,
        csnp_interval=10000,
        max_lsp_size=1492,
        lsp_mgroup_min_trans_interval=5000,
        enable_attached_bit=True,
    ):
        super(IsisAdvanced, self).__init__()
        self._parent = parent
        self._set_property("enable_hello_padding", enable_hello_padding)
        self._set_property("max_area_addresses", max_area_addresses)
        self._set_property("area_addresses", area_addresses)
        self._set_property("lsp_refresh_rate", lsp_refresh_rate)
        self._set_property("lsp_lifetime", lsp_lifetime)
        self._set_property("psnp_interval", psnp_interval)
        self._set_property("csnp_interval", csnp_interval)
        self._set_property("max_lsp_size", max_lsp_size)
        self._set_property(
            "lsp_mgroup_min_trans_interval", lsp_mgroup_min_trans_interval
        )
        self._set_property("enable_attached_bit", enable_attached_bit)

    def set(
        self,
        enable_hello_padding=None,
        max_area_addresses=None,
        area_addresses=None,
        lsp_refresh_rate=None,
        lsp_lifetime=None,
        psnp_interval=None,
        csnp_interval=None,
        max_lsp_size=None,
        lsp_mgroup_min_trans_interval=None,
        enable_attached_bit=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enable_hello_padding(self):
        # type: () -> bool
        """enable_hello_padding getter

        It enables padding of Hello message to MTU size.

        Returns: bool
        """
        return self._get_property("enable_hello_padding")

    @enable_hello_padding.setter
    def enable_hello_padding(self, value):
        """enable_hello_padding setter

        It enables padding of Hello message to MTU size.

        value: bool
        """
        self._set_property("enable_hello_padding", value)

    @property
    def max_area_addresses(self):
        # type: () -> int
        """max_area_addresses getter

        The Number of Area Addresses permitted, with valid range from to 254. zero indicates maximum of addresses.

        Returns: int
        """
        return self._get_property("max_area_addresses")

    @max_area_addresses.setter
    def max_area_addresses(self, value):
        """max_area_addresses setter

        The Number of Area Addresses permitted, with valid range from to 254. zero indicates maximum of addresses.

        value: int
        """
        self._set_property("max_area_addresses", value)

    @property
    def area_addresses(self):
        # type: () -> List[str]
        """area_addresses getter

        Its combination of the ISP and HO-DSP.Usually all nodes within an area have the same area address. If no area addresses are configured, default area of "490001" will be advertised.

        Returns: List[str]
        """
        return self._get_property("area_addresses")

    @area_addresses.setter
    def area_addresses(self, value):
        """area_addresses setter

        Its combination of the ISP and HO-DSP.Usually all nodes within an area have the same area address. If no area addresses are configured, default area of "490001" will be advertised.

        value: List[str]
        """
        self._set_property("area_addresses", value)

    @property
    def lsp_refresh_rate(self):
        # type: () -> int
        """lsp_refresh_rate getter

        The rate at which LSPs are re-sent in seconds.

        Returns: int
        """
        return self._get_property("lsp_refresh_rate")

    @lsp_refresh_rate.setter
    def lsp_refresh_rate(self, value):
        """lsp_refresh_rate setter

        The rate at which LSPs are re-sent in seconds.

        value: int
        """
        self._set_property("lsp_refresh_rate", value)

    @property
    def lsp_lifetime(self):
        # type: () -> int
        """lsp_lifetime getter

        The MaxAge for retaining learned LSP on this router in seconds.

        Returns: int
        """
        return self._get_property("lsp_lifetime")

    @lsp_lifetime.setter
    def lsp_lifetime(self, value):
        """lsp_lifetime setter

        The MaxAge for retaining learned LSP on this router in seconds.

        value: int
        """
        self._set_property("lsp_lifetime", value)

    @property
    def psnp_interval(self):
        # type: () -> int
        """psnp_interval getter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        Returns: int
        """
        return self._get_property("psnp_interval")

    @psnp_interval.setter
    def psnp_interval(self, value):
        """psnp_interval setter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        value: int
        """
        self._set_property("psnp_interval", value)

    @property
    def csnp_interval(self):
        # type: () -> int
        """csnp_interval getter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        Returns: int
        """
        return self._get_property("csnp_interval")

    @csnp_interval.setter
    def csnp_interval(self, value):
        """csnp_interval setter

        The number of milliseconds between transmissions of Partial Sequence Number PDU.

        value: int
        """
        self._set_property("csnp_interval", value)

    @property
    def max_lsp_size(self):
        # type: () -> int
        """max_lsp_size getter

        The maximum size in bytes of any LSP that can be transmitted over link of equal or less than maximum MTU size.

        Returns: int
        """
        return self._get_property("max_lsp_size")

    @max_lsp_size.setter
    def max_lsp_size(self, value):
        """max_lsp_size setter

        The maximum size in bytes of any LSP that can be transmitted over link of equal or less than maximum MTU size.

        value: int
        """
        self._set_property("max_lsp_size", value)

    @property
    def lsp_mgroup_min_trans_interval(self):
        # type: () -> int
        """lsp_mgroup_min_trans_interval getter

        The number of seconds between transmissions of LSPs/MGROUP-PDUs.

        Returns: int
        """
        return self._get_property("lsp_mgroup_min_trans_interval")

    @lsp_mgroup_min_trans_interval.setter
    def lsp_mgroup_min_trans_interval(self, value):
        """lsp_mgroup_min_trans_interval setter

        The number of seconds between transmissions of LSPs/MGROUP-PDUs.

        value: int
        """
        self._set_property("lsp_mgroup_min_trans_interval", value)

    @property
    def enable_attached_bit(self):
        # type: () -> bool
        """enable_attached_bit getter

        If the Attached bit is enabled, it indicates that the ISIS router is attached to another area or the Level backbone. The purpose of an Attached-Bit is to accomplish Inter-Area Routing. When an L1/L2 router is connected to more than one area, it sets the Attached-bit on its L1 LSP. This can cause default route 0.0.0.0/0 to be installed by the receiving router.

        Returns: bool
        """
        return self._get_property("enable_attached_bit")

    @enable_attached_bit.setter
    def enable_attached_bit(self, value):
        """enable_attached_bit setter

        If the Attached bit is enabled, it indicates that the ISIS router is attached to another area or the Level backbone. The purpose of an Attached-Bit is to accomplish Inter-Area Routing. When an L1/L2 router is connected to more than one area, it sets the Attached-bit on its L1 LSP. This can cause default route 0.0.0.0/0 to be installed by the receiving router.

        value: bool
        """
        self._set_property("enable_attached_bit", value)


class IsisAuthentication(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ignore_receive_md5": {"type": bool},
        "area_auth": {"type": "IsisAuthenticationBase"},
        "domain_auth": {"type": "IsisAuthenticationBase"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ignore_receive_md5": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ignore_receive_md5=True):
        super(IsisAuthentication, self).__init__()
        self._parent = parent
        self._set_property("ignore_receive_md5", ignore_receive_md5)

    def set(self, ignore_receive_md5=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ignore_receive_md5(self):
        # type: () -> bool
        """ignore_receive_md5 getter

        Do not verify MD5 checksum in received LSPs.

        Returns: bool
        """
        return self._get_property("ignore_receive_md5")

    @ignore_receive_md5.setter
    def ignore_receive_md5(self, value):
        """ignore_receive_md5 setter

        Do not verify MD5 checksum in received LSPs.

        value: bool
        """
        self._set_property("ignore_receive_md5", value)

    @property
    def area_auth(self):
        # type: () -> IsisAuthenticationBase
        """area_auth getter

        Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.The Area authentication method used for the emulated ISIS router.. This is used for L1 LSPs.

        Returns: IsisAuthenticationBase
        """
        return self._get_property("area_auth", IsisAuthenticationBase)

    @property
    def domain_auth(self):
        # type: () -> IsisAuthenticationBase
        """domain_auth getter

        Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.Optional container for ISIS authentication properties.The Domain authentication method used for the emulated ISIS router.. This is used for L2 LSPs.

        Returns: IsisAuthenticationBase
        """
        return self._get_property("domain_auth", IsisAuthenticationBase)


class IsisAuthenticationBase(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "auth_type": {
            "type": str,
            "enum": [
                "md5",
                "password",
            ],
        },
        "md5": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
        "password": {
            "type": str,
            "minLength": 0,
            "maxLength": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("auth_type",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    MD5 = "md5"  # type: str
    PASSWORD = "password"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, auth_type=None, md5=None, password=None):
        super(IsisAuthenticationBase, self).__init__()
        self._parent = parent
        self._set_property("auth_type", auth_type)
        self._set_property("md5", md5)
        self._set_property("password", password)

    def set(self, auth_type=None, md5=None, password=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def auth_type(self):
        # type: () -> Union[Literal["md5"], Literal["password"]]
        """auth_type getter

        The authentication method.

        Returns: Union[Literal["md5"], Literal["password"]]
        """
        return self._get_property("auth_type")

    @auth_type.setter
    def auth_type(self, value):
        """auth_type setter

        The authentication method.

        value: Union[Literal["md5"], Literal["password"]]
        """
        if value is None:
            raise TypeError("Cannot set required property auth_type as None")
        self._set_property("auth_type", value)

    @property
    def md5(self):
        # type: () -> str
        """md5 getter

        Authentication as an MD5 key.

        Returns: str
        """
        return self._get_property("md5")

    @md5.setter
    def md5(self, value):
        """md5 setter

        Authentication as an MD5 key.

        value: str
        """
        self._set_property("md5", value)

    @property
    def password(self):
        # type: () -> str
        """password getter

        Authentication as clear text password.

        Returns: str
        """
        return self._get_property("password")

    @password.setter
    def password(self, value):
        """password setter

        Authentication as clear text password.

        value: str
        """
        self._set_property("password", value)


class IsisV4RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V4RouteAddressIter"},
        "link_metric": {
            "type": int,
            "format": "uint32",
            "minimum": 0,
            "maximum": 16777215,
        },
        "origin_type": {
            "type": str,
            "enum": [
                "internal",
                "external",
            ],
        },
        "redistribution_type": {
            "type": str,
            "enum": [
                "up",
                "down",
            ],
        },
        "name": {"type": str},
        "prefix_attr_enabled": {"type": bool},
        "x_flag": {"type": bool},
        "r_flag": {"type": bool},
        "n_flag": {"type": bool},
        "prefix_sids": {"type": "IsisSRPrefixSidIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "link_metric": 0,
        "origin_type": "internal",
        "redistribution_type": "up",
        "prefix_attr_enabled": False,
        "x_flag": False,
        "r_flag": False,
        "n_flag": False,
    }  # type: Dict[str, Union(type)]

    INTERNAL = "internal"  # type: str
    EXTERNAL = "external"  # type: str

    UP = "up"  # type: str
    DOWN = "down"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        super(IsisV4RouteRange, self).__init__()
        self._parent = parent
        self._set_property("link_metric", link_metric)
        self._set_property("origin_type", origin_type)
        self._set_property("redistribution_type", redistribution_type)
        self._set_property("name", name)
        self._set_property("prefix_attr_enabled", prefix_attr_enabled)
        self._set_property("x_flag", x_flag)
        self._set_property("r_flag", r_flag)
        self._set_property("n_flag", n_flag)

    def set(
        self,
        link_metric=None,
        origin_type=None,
        redistribution_type=None,
        name=None,
        prefix_attr_enabled=None,
        x_flag=None,
        r_flag=None,
        n_flag=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def addresses(self):
        # type: () -> V4RouteAddressIter
        """addresses getter

        A list of group of IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        return self._get_property(
            "addresses", V4RouteAddressIter, self._parent, self._choice
        )

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        The user-defined metric associated with this route range.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        The user-defined metric associated with this route range.

        value: int
        """
        self._set_property("link_metric", value)

    @property
    def origin_type(self):
        # type: () -> Union[Literal["external"], Literal["internal"]]
        """origin_type getter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: Internal-for intra-area routes, through Level LSPs. External-for inter-area routes redistributed within L1, through Level. LSPs.

        Returns: Union[Literal["external"], Literal["internal"]]
        """
        return self._get_property("origin_type")

    @origin_type.setter
    def origin_type(self, value):
        """origin_type setter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: Internal-for intra-area routes, through Level LSPs. External-for inter-area routes redistributed within L1, through Level. LSPs.

        value: Union[Literal["external"], Literal["internal"]]
        """
        self._set_property("origin_type", value)

    @property
    def redistribution_type(self):
        # type: () -> Union[Literal["down"], Literal["up"]]
        """redistribution_type getter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information.. Up (0)-used when prefix is initially advertised within the ISIS L3. hierarchy, and for all other prefixes in L1 and L2 LSPs. (default) Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs.. The prefixes are being advertised from higher level (L2) down to lower level (L1).

        Returns: Union[Literal["down"], Literal["up"]]
        """
        return self._get_property("redistribution_type")

    @redistribution_type.setter
    def redistribution_type(self, value):
        """redistribution_type setter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information.. Up (0)-used when prefix is initially advertised within the ISIS L3. hierarchy, and for all other prefixes in L1 and L2 LSPs. (default) Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs.. The prefixes are being advertised from higher level (L2) down to lower level (L1).

        value: Union[Literal["down"], Literal["up"]]
        """
        self._set_property("redistribution_type", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def prefix_attr_enabled(self):
        # type: () -> bool
        """prefix_attr_enabled getter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        Returns: bool
        """
        return self._get_property("prefix_attr_enabled")

    @prefix_attr_enabled.setter
    def prefix_attr_enabled(self, value):
        """prefix_attr_enabled setter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        value: bool
        """
        self._set_property("prefix_attr_enabled", value)

    @property
    def x_flag(self):
        # type: () -> bool
        """x_flag getter

        External Prefix Flag (Bit 0)

        Returns: bool
        """
        return self._get_property("x_flag")

    @x_flag.setter
    def x_flag(self, value):
        """x_flag setter

        External Prefix Flag (Bit 0)

        value: bool
        """
        self._set_property("x_flag", value)

    @property
    def r_flag(self):
        # type: () -> bool
        """r_flag getter

        Re-advertisement Flag (Bit 1)

        Returns: bool
        """
        return self._get_property("r_flag")

    @r_flag.setter
    def r_flag(self, value):
        """r_flag setter

        Re-advertisement Flag (Bit 1)

        value: bool
        """
        self._set_property("r_flag", value)

    @property
    def n_flag(self):
        # type: () -> bool
        """n_flag getter

        Node Flag (Bit 2)

        Returns: bool
        """
        return self._get_property("n_flag")

    @n_flag.setter
    def n_flag(self, value):
        """n_flag setter

        Node Flag (Bit 2)

        value: bool
        """
        self._set_property("n_flag", value)

    @property
    def prefix_sids(self):
        # type: () -> IsisSRPrefixSidIter
        """prefix_sids getter

        A list of SID paramters for group of IPv4 route addresses.

        Returns: IsisSRPrefixSidIter
        """
        return self._get_property(
            "prefix_sids", IsisSRPrefixSidIter, self._parent, self._choice
        )


class V4RouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv4",
        },
        "prefix": {
            "type": int,
            "format": "uint32",
            "maximum": 32,
        },
        "count": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
        "step": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 24,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=24, count=1, step=1):
        super(V4RouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    def set(self, address=None, prefix=None, count=None, step=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property address as None")
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv4 network prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv4 network prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the network address prefixes within route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the network address prefixes within route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class V4RouteAddressIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(V4RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[V4RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> V4RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> V4RouteAddress
        return self._next()

    def next(self):
        # type: () -> V4RouteAddress
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, V4RouteAddress):
            raise Exception("Item is not an instance of V4RouteAddress")

    def v4routeaddress(self, address=None, prefix=24, count=1, step=1):
        # type: (str,int,int,int) -> V4RouteAddressIter
        """Factory method that creates an instance of the V4RouteAddress class

        A container for IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        item = V4RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return self

    def add(self, address=None, prefix=24, count=1, step=1):
        # type: (str,int,int,int) -> V4RouteAddress
        """Add method that creates and returns an instance of the V4RouteAddress class

        A container for IPv4 route addresses.

        Returns: V4RouteAddress
        """
        item = V4RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return item


class IsisSRPrefixSid(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "sid_values",
                "sid_indices",
            ],
        },
        "sid_values": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "minimum": 16,
            "maximum": 1048575,
        },
        "sid_indices": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "minimum": 0,
            "maximum": 4294967295,
        },
        "r_flag": {"type": bool},
        "n_flag": {"type": bool},
        "p_flag": {"type": bool},
        "e_flag": {"type": bool},
        "l_flag": {"type": bool},
        "algorithm": {
            "type": int,
            "format": "uint32",
            "minimum": 0,
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "sid_indices",
        "r_flag": False,
        "n_flag": False,
        "p_flag": False,
        "e_flag": False,
        "l_flag": False,
        "algorithm": 0,
    }  # type: Dict[str, Union(type)]

    SID_VALUES = "sid_values"  # type: str
    SID_INDICES = "sid_indices"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        sid_values=None,
        sid_indices=None,
        r_flag=False,
        n_flag=False,
        p_flag=False,
        e_flag=False,
        l_flag=False,
        algorithm=0,
    ):
        super(IsisSRPrefixSid, self).__init__()
        self._parent = parent
        self._set_property("sid_values", sid_values)
        self._set_property("sid_indices", sid_indices)
        self._set_property("r_flag", r_flag)
        self._set_property("n_flag", n_flag)
        self._set_property("p_flag", p_flag)
        self._set_property("e_flag", e_flag)
        self._set_property("l_flag", l_flag)
        self._set_property("algorithm", algorithm)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(
        self,
        sid_values=None,
        sid_indices=None,
        r_flag=None,
        n_flag=None,
        p_flag=None,
        e_flag=None,
        l_flag=None,
        algorithm=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["sid_indices"], Literal["sid_values"]]
        """choice getter

        Choice of whether Prefix-SID carries and absolute value or relative index to the SRGB/SRLB Ranges.. Please refer to device.isis.segment_routing.router_capability.sr_capability.srgb_ranges for the Segment Routing Global Block (SRGB) Descriptor and. device.isis.segment_routing.router_capability.srlb_ranges for the SR Local Block (SRLB).. user needs to configure at least one entry of SID value or SID index.. If no entry is configured, then an implementation may advertise appropriate default SID Value/Index based on the choice. e.g. the first value. from the SRGB or SRLB range.. sid_values: Prefix-SID carries one or more values and then v_flag is set by the implementation.. sid_indices: Prefix-SID carries one or more indices and then v_flag is unset by the implementation.

        Returns: Union[Literal["sid_indices"], Literal["sid_values"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Choice of whether Prefix-SID carries and absolute value or relative index to the SRGB/SRLB Ranges.. Please refer to device.isis.segment_routing.router_capability.sr_capability.srgb_ranges for the Segment Routing Global Block (SRGB) Descriptor and. device.isis.segment_routing.router_capability.srlb_ranges for the SR Local Block (SRLB).. user needs to configure at least one entry of SID value or SID index.. If no entry is configured, then an implementation may advertise appropriate default SID Value/Index based on the choice. e.g. the first value. from the SRGB or SRLB range.. sid_values: Prefix-SID carries one or more values and then v_flag is set by the implementation.. sid_indices: Prefix-SID carries one or more indices and then v_flag is unset by the implementation.

        value: Union[Literal["sid_indices"], Literal["sid_values"]]
        """
        self._set_property("choice", value)

    @property
    def sid_values(self):
        # type: () -> List[int]
        """sid_values getter

        SID/Label as one or more absolute values that are associated with the IGP Prefix segment attached to the specific IPv4 or IPv6 prefix.

        Returns: List[int]
        """
        return self._get_property("sid_values")

    @sid_values.setter
    def sid_values(self, value):
        """sid_values setter

        SID/Label as one or more absolute values that are associated with the IGP Prefix segment attached to the specific IPv4 or IPv6 prefix.

        value: List[int]
        """
        self._set_property("sid_values", value, "sid_values")

    @property
    def sid_indices(self):
        # type: () -> List[int]
        """sid_indices getter

        One or more SID/Label Indices that are associated with the IGP Prefix segment attached to the specific IPv4 or IPv6 prefix.

        Returns: List[int]
        """
        return self._get_property("sid_indices")

    @sid_indices.setter
    def sid_indices(self, value):
        """sid_indices setter

        One or more SID/Label Indices that are associated with the IGP Prefix segment attached to the specific IPv4 or IPv6 prefix.

        value: List[int]
        """
        self._set_property("sid_indices", value, "sid_indices")

    @property
    def r_flag(self):
        # type: () -> bool
        """r_flag getter

        R-Flag: Re-advertisement Flag. If set, then the prefix to which this Prefix-SID is attached has been propagated by the router from either another level (i.e., from Level-1 to Level-2 or the opposite) or redistribution (e.g., from another protocol).

        Returns: bool
        """
        return self._get_property("r_flag")

    @r_flag.setter
    def r_flag(self, value):
        """r_flag setter

        R-Flag: Re-advertisement Flag. If set, then the prefix to which this Prefix-SID is attached has been propagated by the router from either another level (i.e., from Level-1 to Level-2 or the opposite) or redistribution (e.g., from another protocol).

        value: bool
        """
        self._set_property("r_flag", value)

    @property
    def n_flag(self):
        # type: () -> bool
        """n_flag getter

        N-Flag: Node-SID flag. If set, then the Prefix-SID refers to the router identified by the prefix. Typically, the N-Flag is set on Prefix-SIDs that are attached to router loopback address. The N-Flag is set when the Prefix-SID is Node-SID as described in [RFC8402].

        Returns: bool
        """
        return self._get_property("n_flag")

    @n_flag.setter
    def n_flag(self, value):
        """n_flag setter

        N-Flag: Node-SID flag. If set, then the Prefix-SID refers to the router identified by the prefix. Typically, the N-Flag is set on Prefix-SIDs that are attached to router loopback address. The N-Flag is set when the Prefix-SID is Node-SID as described in [RFC8402].

        value: bool
        """
        self._set_property("n_flag", value)

    @property
    def p_flag(self):
        # type: () -> bool
        """p_flag getter

        P-Flag: No-PHP (No Penultimate Hop-Popping) Flag.. If set, then the penultimate hop MUST NOT pop the Prefix-SID before delivering the packet to the node that advertised the Prefix-SID.

        Returns: bool
        """
        return self._get_property("p_flag")

    @p_flag.setter
    def p_flag(self, value):
        """p_flag setter

        P-Flag: No-PHP (No Penultimate Hop-Popping) Flag.. If set, then the penultimate hop MUST NOT pop the Prefix-SID before delivering the packet to the node that advertised the Prefix-SID.

        value: bool
        """
        self._set_property("p_flag", value)

    @property
    def e_flag(self):
        # type: () -> bool
        """e_flag getter

        E-Flag: Explicit-Null Flag.. If set, any upstream neighbor of the Prefix-SID originator MUST replace the Prefix-SID with Prefix-SID that has an Explicit NULL value (0 for IPv4 and for IPv6) before forwarding the packet.

        Returns: bool
        """
        return self._get_property("e_flag")

    @e_flag.setter
    def e_flag(self, value):
        """e_flag setter

        E-Flag: Explicit-Null Flag.. If set, any upstream neighbor of the Prefix-SID originator MUST replace the Prefix-SID with Prefix-SID that has an Explicit NULL value (0 for IPv4 and for IPv6) before forwarding the packet.

        value: bool
        """
        self._set_property("e_flag", value)

    @property
    def l_flag(self):
        # type: () -> bool
        """l_flag getter

        The local flag. If set, then the value/index carried by. the Prefix-SID has local significance and the Prefix SID is from Please refer to device.isis.segment_routing.router_capability.srlb_ranges.

        Returns: bool
        """
        return self._get_property("l_flag")

    @l_flag.setter
    def l_flag(self, value):
        """l_flag setter

        The local flag. If set, then the value/index carried by. the Prefix-SID has local significance and the Prefix SID is from Please refer to device.isis.segment_routing.router_capability.srlb_ranges.

        value: bool
        """
        self._set_property("l_flag", value)

    @property
    def algorithm(self):
        # type: () -> int
        """algorithm getter

        The Isis may use various algorithms when calculating reachability to other nodes or to prefixes attached to these nodes.. Refernce: https://datatracker.ietf.org/doc/html/rfc8667#SRALGOSUBTLV

        Returns: int
        """
        return self._get_property("algorithm")

    @algorithm.setter
    def algorithm(self, value):
        """algorithm setter

        The Isis may use various algorithms when calculating reachability to other nodes or to prefixes attached to these nodes.. Refernce: https://datatracker.ietf.org/doc/html/rfc8667#SRALGOSUBTLV

        value: int
        """
        self._set_property("algorithm", value)


class IsisSRPrefixSidIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisSRPrefixSidIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisSRPrefixSid]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisSRPrefixSidIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisSRPrefixSid
        return self._next()

    def next(self):
        # type: () -> IsisSRPrefixSid
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisSRPrefixSid):
            raise Exception("Item is not an instance of IsisSRPrefixSid")

    def prefixsid(
        self,
        sid_values=None,
        sid_indices=None,
        r_flag=False,
        n_flag=False,
        p_flag=False,
        e_flag=False,
        l_flag=False,
        algorithm=0,
    ):
        # type: (List[int],List[int],bool,bool,bool,bool,bool,int) -> IsisSRPrefixSidIter
        """Factory method that creates an instance of the IsisSRPrefixSid class

        This contains the properties of IS-IS Prefix-SID and its attributes for the extended Ipv4 and Ipv6 reachability. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-prefix-segment-identifier-p.

        Returns: IsisSRPrefixSidIter
        """
        item = IsisSRPrefixSid(
            parent=self._parent,
            choice=self._choice,
            sid_values=sid_values,
            sid_indices=sid_indices,
            r_flag=r_flag,
            n_flag=n_flag,
            p_flag=p_flag,
            e_flag=e_flag,
            l_flag=l_flag,
            algorithm=algorithm,
        )
        self._add(item)
        return self

    def add(
        self,
        sid_values=None,
        sid_indices=None,
        r_flag=False,
        n_flag=False,
        p_flag=False,
        e_flag=False,
        l_flag=False,
        algorithm=0,
    ):
        # type: (List[int],List[int],bool,bool,bool,bool,bool,int) -> IsisSRPrefixSid
        """Add method that creates and returns an instance of the IsisSRPrefixSid class

        This contains the properties of IS-IS Prefix-SID and its attributes for the extended Ipv4 and Ipv6 reachability. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-prefix-segment-identifier-p.

        Returns: IsisSRPrefixSid
        """
        item = IsisSRPrefixSid(
            parent=self._parent,
            choice=self._choice,
            sid_values=sid_values,
            sid_indices=sid_indices,
            r_flag=r_flag,
            n_flag=n_flag,
            p_flag=p_flag,
            e_flag=e_flag,
            l_flag=l_flag,
            algorithm=algorithm,
        )
        self._add(item)
        return item


class IsisV4RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisV4RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisV4RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisV4RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisV4RouteRange
        return self._next()

    def next(self):
        # type: () -> IsisV4RouteRange
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisV4RouteRange):
            raise Exception("Item is not an instance of IsisV4RouteRange")

    def v4routerange(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV4RouteRangeIter
        """Factory method that creates an instance of the IsisV4RouteRange class

        Emulated ISIS IPv4 routes.

        Returns: IsisV4RouteRangeIter
        """
        item = IsisV4RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return self

    def add(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV4RouteRange
        """Add method that creates and returns an instance of the IsisV4RouteRange class

        Emulated ISIS IPv4 routes.

        Returns: IsisV4RouteRange
        """
        item = IsisV4RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return item


class IsisV6RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V6RouteAddressIter"},
        "link_metric": {
            "type": int,
            "format": "uint32",
            "minimum": 0,
            "maximum": 16777215,
        },
        "origin_type": {
            "type": str,
            "enum": [
                "internal",
                "external",
            ],
        },
        "redistribution_type": {
            "type": str,
            "enum": [
                "up",
                "down",
            ],
        },
        "name": {"type": str},
        "prefix_attr_enabled": {"type": bool},
        "x_flag": {"type": bool},
        "r_flag": {"type": bool},
        "n_flag": {"type": bool},
        "prefix_sids": {"type": "IsisSRPrefixSidIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "link_metric": 0,
        "origin_type": "internal",
        "redistribution_type": "up",
        "prefix_attr_enabled": False,
        "x_flag": False,
        "r_flag": False,
        "n_flag": False,
    }  # type: Dict[str, Union(type)]

    INTERNAL = "internal"  # type: str
    EXTERNAL = "external"  # type: str

    UP = "up"  # type: str
    DOWN = "down"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        super(IsisV6RouteRange, self).__init__()
        self._parent = parent
        self._set_property("link_metric", link_metric)
        self._set_property("origin_type", origin_type)
        self._set_property("redistribution_type", redistribution_type)
        self._set_property("name", name)
        self._set_property("prefix_attr_enabled", prefix_attr_enabled)
        self._set_property("x_flag", x_flag)
        self._set_property("r_flag", r_flag)
        self._set_property("n_flag", n_flag)

    def set(
        self,
        link_metric=None,
        origin_type=None,
        redistribution_type=None,
        name=None,
        prefix_attr_enabled=None,
        x_flag=None,
        r_flag=None,
        n_flag=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def addresses(self):
        # type: () -> V6RouteAddressIter
        """addresses getter

        A list of group of IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        return self._get_property(
            "addresses", V6RouteAddressIter, self._parent, self._choice
        )

    @property
    def link_metric(self):
        # type: () -> int
        """link_metric getter

        The user-defined metric associated with this route range.

        Returns: int
        """
        return self._get_property("link_metric")

    @link_metric.setter
    def link_metric(self, value):
        """link_metric setter

        The user-defined metric associated with this route range.

        value: int
        """
        self._set_property("link_metric", value)

    @property
    def origin_type(self):
        # type: () -> Union[Literal["external"], Literal["internal"]]
        """origin_type getter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: Internal-for intra-area routes, through Level LSPs. External-for inter-area routes redistributed within L1, through Level. LSPs.

        Returns: Union[Literal["external"], Literal["internal"]]
        """
        return self._get_property("origin_type")

    @origin_type.setter
    def origin_type(self, value):
        """origin_type setter

        The origin of the advertised route-internal or external to the ISIS area. Options include the following: Internal-for intra-area routes, through Level LSPs. External-for inter-area routes redistributed within L1, through Level. LSPs.

        value: Union[Literal["external"], Literal["internal"]]
        """
        self._set_property("origin_type", value)

    @property
    def redistribution_type(self):
        # type: () -> Union[Literal["down"], Literal["up"]]
        """redistribution_type getter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information.. Up (0)-used when prefix is initially advertised within the ISIS L3. hierarchy, and for all other prefixes in L1 and L2 LSPs. (default) Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs.. The prefixes are being advertised from higher level (L2) down to lower level (L1).

        Returns: Union[Literal["down"], Literal["up"]]
        """
        return self._get_property("redistribution_type")

    @redistribution_type.setter
    def redistribution_type(self, value):
        """redistribution_type setter

        Defines the Up/Down (Redistribution) bit defined for TLVs 128 and 130 by RFC 2966. It is used for domain-wide advertisement of prefix information.. Up (0)-used when prefix is initially advertised within the ISIS L3. hierarchy, and for all other prefixes in L1 and L2 LSPs. (default) Down (1)-used when an L1/L2 router advertises L2 prefixes in L1 LSPs.. The prefixes are being advertised from higher level (L2) down to lower level (L1).

        value: Union[Literal["down"], Literal["up"]]
        """
        self._set_property("redistribution_type", value)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def prefix_attr_enabled(self):
        # type: () -> bool
        """prefix_attr_enabled getter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        Returns: bool
        """
        return self._get_property("prefix_attr_enabled")

    @prefix_attr_enabled.setter
    def prefix_attr_enabled(self, value):
        """prefix_attr_enabled setter

        Specifies whether the sub-TLV for IPv4/IPv6 Extended Reachability Attribute Flags. will be advertised or not.

        value: bool
        """
        self._set_property("prefix_attr_enabled", value)

    @property
    def x_flag(self):
        # type: () -> bool
        """x_flag getter

        External Prefix Flag (Bit 0)

        Returns: bool
        """
        return self._get_property("x_flag")

    @x_flag.setter
    def x_flag(self, value):
        """x_flag setter

        External Prefix Flag (Bit 0)

        value: bool
        """
        self._set_property("x_flag", value)

    @property
    def r_flag(self):
        # type: () -> bool
        """r_flag getter

        Re-advertisement Flag (Bit 1)

        Returns: bool
        """
        return self._get_property("r_flag")

    @r_flag.setter
    def r_flag(self, value):
        """r_flag setter

        Re-advertisement Flag (Bit 1)

        value: bool
        """
        self._set_property("r_flag", value)

    @property
    def n_flag(self):
        # type: () -> bool
        """n_flag getter

        Node Flag (Bit 2)

        Returns: bool
        """
        return self._get_property("n_flag")

    @n_flag.setter
    def n_flag(self, value):
        """n_flag setter

        Node Flag (Bit 2)

        value: bool
        """
        self._set_property("n_flag", value)

    @property
    def prefix_sids(self):
        # type: () -> IsisSRPrefixSidIter
        """prefix_sids getter

        A list of SID parameters for group of IPv6 route addresses.

        Returns: IsisSRPrefixSidIter
        """
        return self._get_property(
            "prefix_sids", IsisSRPrefixSidIter, self._parent, self._choice
        )


class V6RouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "ipv6",
        },
        "prefix": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "count": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
        "step": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 64,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=64, count=1, step=1):
        super(V6RouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    def set(self, address=None, prefix=None, count=None, step=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the network.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the network.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property address as None")
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The IPv6 network prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The IPv6 network prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the network address prefixes within route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the network address prefixes within route range where multiple routes are present. The value is incremented according to the Prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class V6RouteAddressIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(V6RouteAddressIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[V6RouteAddress]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> V6RouteAddressIter
        return self._iter()

    def __next__(self):
        # type: () -> V6RouteAddress
        return self._next()

    def next(self):
        # type: () -> V6RouteAddress
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, V6RouteAddress):
            raise Exception("Item is not an instance of V6RouteAddress")

    def v6routeaddress(self, address=None, prefix=64, count=1, step=1):
        # type: (str,int,int,int) -> V6RouteAddressIter
        """Factory method that creates an instance of the V6RouteAddress class

        A container for IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        item = V6RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return self

    def add(self, address=None, prefix=64, count=1, step=1):
        # type: (str,int,int,int) -> V6RouteAddress
        """Add method that creates and returns an instance of the V6RouteAddress class

        A container for IPv6 route addresses.

        Returns: V6RouteAddress
        """
        item = V6RouteAddress(
            parent=self._parent, address=address, prefix=prefix, count=count, step=step
        )
        self._add(item)
        return item


class IsisV6RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisV6RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisV6RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisV6RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisV6RouteRange
        return self._next()

    def next(self):
        # type: () -> IsisV6RouteRange
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisV6RouteRange):
            raise Exception("Item is not an instance of IsisV6RouteRange")

    def v6routerange(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV6RouteRangeIter
        """Factory method that creates an instance of the IsisV6RouteRange class

        Emulated ISIS IPv6 routes.

        Returns: IsisV6RouteRangeIter
        """
        item = IsisV6RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return self

    def add(
        self,
        link_metric=0,
        origin_type="internal",
        redistribution_type="up",
        name=None,
        prefix_attr_enabled=False,
        x_flag=False,
        r_flag=False,
        n_flag=False,
    ):
        # type: (int,Union[Literal["external"], Literal["internal"]],Union[Literal["down"], Literal["up"]],str,bool,bool,bool,bool) -> IsisV6RouteRange
        """Add method that creates and returns an instance of the IsisV6RouteRange class

        Emulated ISIS IPv6 routes.

        Returns: IsisV6RouteRange
        """
        item = IsisV6RouteRange(
            parent=self._parent,
            link_metric=link_metric,
            origin_type=origin_type,
            redistribution_type=redistribution_type,
            name=name,
            prefix_attr_enabled=prefix_attr_enabled,
            x_flag=x_flag,
            r_flag=r_flag,
            n_flag=n_flag,
        )
        self._add(item)
        return item


class IsisSegmentRouting(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "router_capability": {"type": "IsisRouterCapability"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(IsisSegmentRouting, self).__init__()
        self._parent = parent

    @property
    def router_capability(self):
        # type: () -> IsisRouterCapability
        """router_capability getter

        Container for the configuration of IS-IS Router CAPABILITY TLV.. https://datatracker.ietf.org/doc/html/rfc7981#section-2.. An implementation should set default values appropriately if any mandatory item is not configured by user.Container for the configuration of IS-IS Router CAPABILITY TLV.. https://datatracker.ietf.org/doc/html/rfc7981#section-2.. An implementation should set default values appropriately if any mandatory item is not configured by user.Container for the configuration of IS-IS Router CAPABILITY TLV.. https://datatracker.ietf.org/doc/html/rfc7981#section-2.. An implementation should set default values appropriately if any mandatory item is not configured by user.Optional IS-IS TLV named CAPABILITY, formed of multiple sub-TLVs, which allows router to announce its. capabilities within an IS-IS level or the entire routing domain.

        Returns: IsisRouterCapability
        """
        return self._get_property("router_capability", IsisRouterCapability)


class IsisRouterCapability(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "ipv4_te_router_id",
                "interface_ip",
                "custom_router_cap_id",
            ],
        },
        "custom_router_cap_id": {
            "type": str,
            "format": "ipv4",
        },
        "s_bit": {
            "type": str,
            "enum": [
                "flood",
                "not_flood",
            ],
        },
        "d_bit": {
            "type": str,
            "enum": [
                "down",
                "not_down",
            ],
        },
        "sr_capability": {"type": "IsisSRCapability"},
        "algorithms": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "minimum": 0,
            "maximum": 255,
        },
        "srlb_ranges": {"type": "IsisSRSrlbIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "ipv4_te_router_id",
        "custom_router_cap_id": "0.0.0.0",
        "s_bit": "not_flood",
        "d_bit": "not_down",
    }  # type: Dict[str, Union(type)]

    IPV4_TE_ROUTER_ID = "ipv4_te_router_id"  # type: str
    INTERFACE_IP = "interface_ip"  # type: str
    CUSTOM_ROUTER_CAP_ID = "custom_router_cap_id"  # type: str

    FLOOD = "flood"  # type: str
    NOT_FLOOD = "not_flood"  # type: str

    DOWN = "down"  # type: str
    NOT_DOWN = "not_down"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        custom_router_cap_id="0.0.0.0",
        s_bit="not_flood",
        d_bit="not_down",
        algorithms=None,
    ):
        super(IsisRouterCapability, self).__init__()
        self._parent = parent
        self._set_property("custom_router_cap_id", custom_router_cap_id)
        self._set_property("s_bit", s_bit)
        self._set_property("d_bit", d_bit)
        self._set_property("algorithms", algorithms)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, custom_router_cap_id=None, s_bit=None, d_bit=None, algorithms=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["custom_router_cap_id"], Literal["interface_ip"], Literal["ipv4_te_router_id"]]
        """choice getter

        The Router Capability ID SHOULD be identical to the value advertised in the Traffic Engineering Router ID TLV [RFC5305].. If no Traffic Engineering Router ID is assigned, the Router ID SHOULD be identical to an IP Interface Address [RFC1195] advertised by the originating IS.. If the originating node does not support IPv4, then the reserved value 0.0.0.0 MUST be used in the Router ID field, and the IPv6 TE Router ID sub-TLV [RFC5316] MUST be present in the TLV.. ipv4_te_router_id: IPv4 Traffic Engineering(TE) router id (defined in isis.basic.ipv4_te_router_id) to be used as Router Capability ID.. interface_ip: When this is chosen the first IPv4 address of the first eth interface associated with this isis router instance should be assigned as the Router Capability ID.. custom_router_cap_id: This option should be chosen when Router Capability ID needs to be configured different from above two options.

        Returns: Union[Literal["custom_router_cap_id"], Literal["interface_ip"], Literal["ipv4_te_router_id"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The Router Capability ID SHOULD be identical to the value advertised in the Traffic Engineering Router ID TLV [RFC5305].. If no Traffic Engineering Router ID is assigned, the Router ID SHOULD be identical to an IP Interface Address [RFC1195] advertised by the originating IS.. If the originating node does not support IPv4, then the reserved value 0.0.0.0 MUST be used in the Router ID field, and the IPv6 TE Router ID sub-TLV [RFC5316] MUST be present in the TLV.. ipv4_te_router_id: IPv4 Traffic Engineering(TE) router id (defined in isis.basic.ipv4_te_router_id) to be used as Router Capability ID.. interface_ip: When this is chosen the first IPv4 address of the first eth interface associated with this isis router instance should be assigned as the Router Capability ID.. custom_router_cap_id: This option should be chosen when Router Capability ID needs to be configured different from above two options.

        value: Union[Literal["custom_router_cap_id"], Literal["interface_ip"], Literal["ipv4_te_router_id"]]
        """
        self._set_property("choice", value)

    @property
    def custom_router_cap_id(self):
        # type: () -> str
        """custom_router_cap_id getter

        Router Capability ID in IPv4 address format.

        Returns: str
        """
        return self._get_property("custom_router_cap_id")

    @custom_router_cap_id.setter
    def custom_router_cap_id(self, value):
        """custom_router_cap_id setter

        Router Capability ID in IPv4 address format.

        value: str
        """
        self._set_property("custom_router_cap_id", value, "custom_router_cap_id")

    @property
    def s_bit(self):
        # type: () -> Union[Literal["flood"], Literal["not_flood"]]
        """s_bit getter

        S bit (0x01): If the bit is set(1), the IS-IS Router CAPABILITY TLV. MUST be flooded across the entire routing domain. If the bit is. not set(0), the TLV MUST NOT be leaked between levels. This bit MUST. NOT be altered during the TLV leaking.

        Returns: Union[Literal["flood"], Literal["not_flood"]]
        """
        return self._get_property("s_bit")

    @s_bit.setter
    def s_bit(self, value):
        """s_bit setter

        S bit (0x01): If the bit is set(1), the IS-IS Router CAPABILITY TLV. MUST be flooded across the entire routing domain. If the bit is. not set(0), the TLV MUST NOT be leaked between levels. This bit MUST. NOT be altered during the TLV leaking.

        value: Union[Literal["flood"], Literal["not_flood"]]
        """
        self._set_property("s_bit", value)

    @property
    def d_bit(self):
        # type: () -> Union[Literal["down"], Literal["not_down"]]
        """d_bit getter

        D bit (0x02): When the IS-IS Router CAPABILITY TLV is leaked from. Level (L2) to Level (L1), the bit MUST be set. Otherwise, this. bit MUST be clear. IS-IS Router CAPABILITY TLVs with the bit set. MUST NOT be leaked from Level to Level 2. This is to prevent TLV looping.

        Returns: Union[Literal["down"], Literal["not_down"]]
        """
        return self._get_property("d_bit")

    @d_bit.setter
    def d_bit(self, value):
        """d_bit setter

        D bit (0x02): When the IS-IS Router CAPABILITY TLV is leaked from. Level (L2) to Level (L1), the bit MUST be set. Otherwise, this. bit MUST be clear. IS-IS Router CAPABILITY TLVs with the bit set. MUST NOT be leaked from Level to Level 2. This is to prevent TLV looping.

        value: Union[Literal["down"], Literal["not_down"]]
        """
        self._set_property("d_bit", value)

    @property
    def sr_capability(self):
        # type: () -> IsisSRCapability
        """sr_capability getter

        Container for the configuration of IS-IS SR-CAPABILITY TLV that Segment Routing requires each router to advertise its SR data plane capability and the range of MPLS label values it uses for Segment Routing in the case where global SIDs are allocated (i.e., global indexes).. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-sr-capabilities-sub-tlv.. An implementation should set default values appropriately if any mandatory item is not configured by user.Container for the configuration of IS-IS SR-CAPABILITY TLV that Segment Routing requires each router to advertise its SR data plane capability and the range of MPLS label values it uses for Segment Routing in the case where global SIDs are allocated (i.e., global indexes).. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-sr-capabilities-sub-tlv.. An implementation should set default values appropriately if any mandatory item is not configured by user.Container for the configuration of IS-IS SR-CAPABILITY TLV that Segment Routing requires each router to advertise its SR data plane capability and the range of MPLS label values it uses for Segment Routing in the case where global SIDs are allocated (i.e., global indexes).. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-sr-capabilities-sub-tlv.. An implementation should set default values appropriately if any mandatory item is not configured by user.SR-Capabilities.

        Returns: IsisSRCapability
        """
        return self._get_property("sr_capability", IsisSRCapability)

    @property
    def algorithms(self):
        # type: () -> List[int]
        """algorithms getter

        This contains one or more Segment Routing Algorithm that router may use various algorithms when calculating reachability to other nodes or to prefixes attached to these nodes.. The Isis may use various algorithms when calculating reachability to other nodes or to prefixes attached to these. nodes. Examples of these algorithms are metric-based SPF, various sorts of Constrained SPF, etc.. 0: SPF algorithm based on link metric.. 1: Strict SPF algorithm based on link metric.. Reference: https://datatracker.ietf.org/doc/html/rfc8665#name-igp-algorithm-types-registr.. When the originating router does not advertise the SR-Algorithm sub-TLV, it implies that algorithm is the only algorithm supported by the routers.. When the originating router does advertise the SR-Algorithm sub-TLV, then algorithm MUST be present while non-zero algorithms MAY be present.

        Returns: List[int]
        """
        return self._get_property("algorithms")

    @algorithms.setter
    def algorithms(self, value):
        """algorithms setter

        This contains one or more Segment Routing Algorithm that router may use various algorithms when calculating reachability to other nodes or to prefixes attached to these nodes.. The Isis may use various algorithms when calculating reachability to other nodes or to prefixes attached to these. nodes. Examples of these algorithms are metric-based SPF, various sorts of Constrained SPF, etc.. 0: SPF algorithm based on link metric.. 1: Strict SPF algorithm based on link metric.. Reference: https://datatracker.ietf.org/doc/html/rfc8665#name-igp-algorithm-types-registr.. When the originating router does not advertise the SR-Algorithm sub-TLV, it implies that algorithm is the only algorithm supported by the routers.. When the originating router does advertise the SR-Algorithm sub-TLV, then algorithm MUST be present while non-zero algorithms MAY be present.

        value: List[int]
        """
        self._set_property("algorithms", value)

    @property
    def srlb_ranges(self):
        # type: () -> IsisSRSrlbIter
        """srlb_ranges getter

        This contains the list of SR Local Block (SRLB)

        Returns: IsisSRSrlbIter
        """
        return self._get_property(
            "srlb_ranges", IsisSRSrlbIter, self._parent, self._choice
        )


class IsisSRCapability(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "IsisSRCapabilityFlags"},
        "srgb_ranges": {"type": "IsisSRSrgbIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(IsisSRCapability, self).__init__()
        self._parent = parent

    @property
    def flags(self):
        # type: () -> IsisSRCapabilityFlags
        """flags getter

        Container for the configuration of IS-IS SR-CAPABILITY flags.. The Router Capability TLV specifies flags that control its advertisement. The SR-Capabilities sub-TLV MUST be propagated throughout the level and MUST NOT be advertised across level boundaries.. Therefore, Router Capability TLV distribution flags are set accordingly, i.e., the S-Flag in the Router Capability TLV [RFC7981] MUST be unset.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#section-3.1-5.1.1.6.1.Container for the configuration of IS-IS SR-CAPABILITY flags.. The Router Capability TLV specifies flags that control its advertisement. The SR-Capabilities sub-TLV MUST be propagated throughout the level and MUST NOT be advertised across level boundaries.. Therefore, Router Capability TLV distribution flags are set accordingly, i.e., the S-Flag in the Router Capability TLV [RFC7981] MUST be unset.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#section-3.1-5.1.1.6.1.Container for the configuration of IS-IS SR-CAPABILITY flags.. The Router Capability TLV specifies flags that control its advertisement. The SR-Capabilities sub-TLV MUST be propagated throughout the level and MUST NOT be advertised across level boundaries.. Therefore, Router Capability TLV distribution flags are set accordingly, i.e., the S-Flag in the Router Capability TLV [RFC7981] MUST be unset.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#section-3.1-5.1.1.6.1.1 octet of flags.

        Returns: IsisSRCapabilityFlags
        """
        return self._get_property("flags", IsisSRCapabilityFlags)

    @property
    def srgb_ranges(self):
        # type: () -> IsisSRSrgbIter
        """srgb_ranges getter

        This contains the list of SRGB.. If no SRGB range is configured, implementation should send one SRGB range with default values.

        Returns: IsisSRSrgbIter
        """
        return self._get_property(
            "srgb_ranges", IsisSRSrgbIter, self._parent, self._choice
        )


class IsisSRCapabilityFlags(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_mpls": {"type": bool},
        "ipv6_mpls": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ipv4_mpls": True,
        "ipv6_mpls": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ipv4_mpls=True, ipv6_mpls=True):
        super(IsisSRCapabilityFlags, self).__init__()
        self._parent = parent
        self._set_property("ipv4_mpls", ipv4_mpls)
        self._set_property("ipv6_mpls", ipv6_mpls)

    def set(self, ipv4_mpls=None, ipv6_mpls=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ipv4_mpls(self):
        # type: () -> bool
        """ipv4_mpls getter

        I-Flag for the MPLS IPv4 Flag. If set, then the router is capable of processing SR-MPLS-encapsulated IPv4 packets on all interfaces.

        Returns: bool
        """
        return self._get_property("ipv4_mpls")

    @ipv4_mpls.setter
    def ipv4_mpls(self, value):
        """ipv4_mpls setter

        I-Flag for the MPLS IPv4 Flag. If set, then the router is capable of processing SR-MPLS-encapsulated IPv4 packets on all interfaces.

        value: bool
        """
        self._set_property("ipv4_mpls", value)

    @property
    def ipv6_mpls(self):
        # type: () -> bool
        """ipv6_mpls getter

        V-Flag for the MPLS IPv6 Flag. If set, then the router is capable of processing SR-MPLS-encapsulated IPv6 packets on all interfaces.

        Returns: bool
        """
        return self._get_property("ipv6_mpls")

    @ipv6_mpls.setter
    def ipv6_mpls(self, value):
        """ipv6_mpls setter

        V-Flag for the MPLS IPv6 Flag. If set, then the router is capable of processing SR-MPLS-encapsulated IPv6 packets on all interfaces.

        value: bool
        """
        self._set_property("ipv6_mpls", value)


class IsisSRSrgb(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "starting_sid": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 4294967295,
        },
        "range": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 4294967295,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "starting_sid": 16000,
        "range": 8000,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, starting_sid=16000, range=8000):
        super(IsisSRSrgb, self).__init__()
        self._parent = parent
        self._set_property("starting_sid", starting_sid)
        self._set_property("range", range)

    def set(self, starting_sid=None, range=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def starting_sid(self):
        # type: () -> int
        """starting_sid getter

        The SID/Label sub-TLV contains the first value of the SRGB while the range contains the number of SRGB elements.

        Returns: int
        """
        return self._get_property("starting_sid")

    @starting_sid.setter
    def starting_sid(self, value):
        """starting_sid setter

        The SID/Label sub-TLV contains the first value of the SRGB while the range contains the number of SRGB elements.

        value: int
        """
        self._set_property("starting_sid", value)

    @property
    def range(self):
        # type: () -> int
        """range getter

        This represents the number of SID in SRGB range.

        Returns: int
        """
        return self._get_property("range")

    @range.setter
    def range(self, value):
        """range setter

        This represents the number of SID in SRGB range.

        value: int
        """
        self._set_property("range", value)


class IsisSRSrgbIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisSRSrgbIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisSRSrgb]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisSRSrgbIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisSRSrgb
        return self._next()

    def next(self):
        # type: () -> IsisSRSrgb
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisSRSrgb):
            raise Exception("Item is not an instance of IsisSRSrgb")

    def srgb(self, starting_sid=16000, range=8000):
        # type: (int,int) -> IsisSRSrgbIter
        """Factory method that creates an instance of the IsisSRSrgb class

        This contains the propeties of Segment Routing Global Block (SRGB) Descriptor. Reference: https://datatracker.ietf.org/doc/html/rfc8667#section-3.1-7.1.1

        Returns: IsisSRSrgbIter
        """
        item = IsisSRSrgb(parent=self._parent, starting_sid=starting_sid, range=range)
        self._add(item)
        return self

    def add(self, starting_sid=16000, range=8000):
        # type: (int,int) -> IsisSRSrgb
        """Add method that creates and returns an instance of the IsisSRSrgb class

        This contains the propeties of Segment Routing Global Block (SRGB) Descriptor. Reference: https://datatracker.ietf.org/doc/html/rfc8667#section-3.1-7.1.1

        Returns: IsisSRSrgb
        """
        item = IsisSRSrgb(parent=self._parent, starting_sid=starting_sid, range=range)
        self._add(item)
        return item


class IsisSRSrlb(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "starting_sid": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 4294967295,
        },
        "range": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
            "maximum": 4294967295,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "starting_sid": 16000,
        "range": 8000,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, starting_sid=16000, range=8000):
        super(IsisSRSrlb, self).__init__()
        self._parent = parent
        self._set_property("starting_sid", starting_sid)
        self._set_property("range", range)

    def set(self, starting_sid=None, range=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def starting_sid(self):
        # type: () -> int
        """starting_sid getter

        The SID/Label sub-TLV contains the first value of the SRLB while the range contains the number of SRLB elements.

        Returns: int
        """
        return self._get_property("starting_sid")

    @starting_sid.setter
    def starting_sid(self, value):
        """starting_sid setter

        The SID/Label sub-TLV contains the first value of the SRLB while the range contains the number of SRLB elements.

        value: int
        """
        self._set_property("starting_sid", value)

    @property
    def range(self):
        # type: () -> int
        """range getter

        This represents the number of SIDs in SRLB range.

        Returns: int
        """
        return self._get_property("range")

    @range.setter
    def range(self, value):
        """range setter

        This represents the number of SIDs in SRLB range.

        value: int
        """
        self._set_property("range", value)


class IsisSRSrlbIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(IsisSRSrlbIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[IsisSRSrlb]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> IsisSRSrlbIter
        return self._iter()

    def __next__(self):
        # type: () -> IsisSRSrlb
        return self._next()

    def next(self):
        # type: () -> IsisSRSrlb
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, IsisSRSrlb):
            raise Exception("Item is not an instance of IsisSRSrlb")

    def srlb(self, starting_sid=16000, range=8000):
        # type: (int,int) -> IsisSRSrlbIter
        """Factory method that creates an instance of the IsisSRSrlb class

        The SR Local Block (SRLB) sub-TLV contains the range of labels the node has reserved for Local SIDs. Local SIDs are used for Adjacency SIDs or locally significant Prefix SIDs and may also be allocated by components other than the IS-IS protocol. As an example, an application or controller may instruct the router to allocate specific Local SID. Therefore, in order for such applications or controllers to know what Local SIDs are available in the router, it is required that the router advertises its SRLB.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-sr-local-block-sub-tlv.

        Returns: IsisSRSrlbIter
        """
        item = IsisSRSrlb(parent=self._parent, starting_sid=starting_sid, range=range)
        self._add(item)
        return self

    def add(self, starting_sid=16000, range=8000):
        # type: (int,int) -> IsisSRSrlb
        """Add method that creates and returns an instance of the IsisSRSrlb class

        The SR Local Block (SRLB) sub-TLV contains the range of labels the node has reserved for Local SIDs. Local SIDs are used for Adjacency SIDs or locally significant Prefix SIDs and may also be allocated by components other than the IS-IS protocol. As an example, an application or controller may instruct the router to allocate specific Local SID. Therefore, in order for such applications or controllers to know what Local SIDs are available in the router, it is required that the router advertises its SRLB.. Reference: https://datatracker.ietf.org/doc/html/rfc8667#name-sr-local-block-sub-tlv.

        Returns: IsisSRSrlb
        """
        item = IsisSRSrlb(parent=self._parent, starting_sid=starting_sid, range=range)
        self._add(item)
        return item


class DeviceBgpRouter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "router_id": {
            "type": str,
            "format": "ipv4",
        },
        "ipv4_interfaces": {"type": "BgpV4InterfaceIter"},
        "ipv6_interfaces": {"type": "BgpV6InterfaceIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("router_id",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, router_id=None):
        super(DeviceBgpRouter, self).__init__()
        self._parent = parent
        self._set_property("router_id", router_id)

    def set(self, router_id=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def router_id(self):
        # type: () -> str
        """router_id getter

        The BGP router ID is unique identifier used by BGP. It is 32-bit value that is often represented by an IPv4 address.

        Returns: str
        """
        return self._get_property("router_id")

    @router_id.setter
    def router_id(self, value):
        """router_id setter

        The BGP router ID is unique identifier used by BGP. It is 32-bit value that is often represented by an IPv4 address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property router_id as None")
        self._set_property("router_id", value)

    @property
    def ipv4_interfaces(self):
        # type: () -> BgpV4InterfaceIter
        """ipv4_interfaces getter

        This contains an array of references to IPv4 interfaces, each of which will have list of peers to different destinations.

        Returns: BgpV4InterfaceIter
        """
        return self._get_property(
            "ipv4_interfaces", BgpV4InterfaceIter, self._parent, self._choice
        )

    @property
    def ipv6_interfaces(self):
        # type: () -> BgpV6InterfaceIter
        """ipv6_interfaces getter

        This contains an array of references to IPv6 interfaces, each of which will have list of peers to different destinations.

        Returns: BgpV6InterfaceIter
        """
        return self._get_property(
            "ipv6_interfaces", BgpV6InterfaceIter, self._parent, self._choice
        )


class BgpV4Interface(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_name": {"type": str},
        "peers": {"type": "BgpV4PeerIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_name",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ipv4_name=None):
        super(BgpV4Interface, self).__init__()
        self._parent = parent
        self._set_property("ipv4_name", ipv4_name)

    def set(self, ipv4_name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ipv4_name(self):
        # type: () -> str
        """ipv4_name getter

        The unique name of the IPv4, Loopback IPv4 interface or DHCPv4 client used as the source IP for this list of BGP peers.. x-constraint:. /components/schemas/Device.Ipv4/properties/name. /components/schemas/Device.Ipv4Loopback/properties/name. /components/schemas/Device.Dhcpv4client/properties/name. . x-constraint:. /components/schemas/Device.Ipv4/properties/name. /components/schemas/Device.Ipv4Loopback/properties/name. /components/schemas/Device.Dhcpv4client/properties/name.

        Returns: str
        """
        return self._get_property("ipv4_name")

    @ipv4_name.setter
    def ipv4_name(self, value):
        """ipv4_name setter

        The unique name of the IPv4, Loopback IPv4 interface or DHCPv4 client used as the source IP for this list of BGP peers.. x-constraint:. /components/schemas/Device.Ipv4/properties/name. /components/schemas/Device.Ipv4Loopback/properties/name. /components/schemas/Device.Dhcpv4client/properties/name. . x-constraint:. /components/schemas/Device.Ipv4/properties/name. /components/schemas/Device.Ipv4Loopback/properties/name. /components/schemas/Device.Dhcpv4client/properties/name.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv4_name as None")
        self._set_property("ipv4_name", value)

    @property
    def peers(self):
        # type: () -> BgpV4PeerIter
        """peers getter

        This contains the list of BGPv4 peers configured on this interface.

        Returns: BgpV4PeerIter
        """
        return self._get_property("peers", BgpV4PeerIter, self._parent, self._choice)


class BgpV4Peer(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "peer_address": {
            "type": str,
            "format": "ipv4",
        },
        "evpn_ethernet_segments": {"type": "BgpV4EthernetSegmentIter"},
        "as_type": {
            "type": str,
            "enum": [
                "ibgp",
                "ebgp",
            ],
        },
        "as_number": {
            "type": int,
            "format": "uint32",
        },
        "as_number_width": {
            "type": str,
            "enum": [
                "two",
                "four",
            ],
        },
        "advanced": {"type": "BgpAdvanced"},
        "capability": {"type": "BgpCapability"},
        "learned_information_filter": {"type": "BgpLearnedInformationFilter"},
        "v4_routes": {"type": "BgpV4RouteRangeIter"},
        "v6_routes": {"type": "BgpV6RouteRangeIter"},
        "v4_srte_policies": {"type": "BgpSrteV4PolicyIter"},
        "v6_srte_policies": {"type": "BgpSrteV6PolicyIter"},
        "name": {"type": str},
        "graceful_restart": {"type": "BgpGracefulRestart"},
        "replay_updates": {"type": "BgpUpdateReplay"},
    }  # type: Dict[str, str]

    _REQUIRED = ("peer_address", "as_type", "as_number", "name")  # type: tuple(str)

    _DEFAULTS = {
        "as_number_width": "four",
    }  # type: Dict[str, Union(type)]

    IBGP = "ibgp"  # type: str
    EBGP = "ebgp"  # type: str

    TWO = "two"  # type: str
    FOUR = "four"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width="four",
        name=None,
    ):
        super(BgpV4Peer, self).__init__()
        self._parent = parent
        self._set_property("peer_address", peer_address)
        self._set_property("as_type", as_type)
        self._set_property("as_number", as_number)
        self._set_property("as_number_width", as_number_width)
        self._set_property("name", name)

    def set(
        self,
        peer_address=None,
        as_type=None,
        as_number=None,
        as_number_width=None,
        name=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def peer_address(self):
        # type: () -> str
        """peer_address getter

        IPv4 address of the BGP peer for the session.

        Returns: str
        """
        return self._get_property("peer_address")

    @peer_address.setter
    def peer_address(self, value):
        """peer_address setter

        IPv4 address of the BGP peer for the session.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property peer_address as None")
        self._set_property("peer_address", value)

    @property
    def evpn_ethernet_segments(self):
        # type: () -> BgpV4EthernetSegmentIter
        """evpn_ethernet_segments getter

        This contains the list of Ethernet Virtual Private Network (EVPN) Ethernet Segments (ES) Per BGP Peer for IPv4 Address Family Identifier (AFI).. Each Ethernet Segment contains list of EVPN Instances (EVIs) . Each EVI contains list of Broadcast Domains. Each Broadcast Domain contains list of MAC/IP Ranges. . <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Ethernet Auto-discovery Route Per EVI (Type 1).. <Ethernet Segment, EVI> is responsible for advertising Ethernet Auto-discovery Route Per Ethernet Segment (Type 1).. <Ethernet Segment, EVI, Broadcast Domain, MAC/IP> is responsible for advertising MAC/IP Advertisement Route (Type 2).. <Ethernet Segment, EVI, Broadcast Domain> is responsible for advertising Inclusive Multicast Ethernet Tag Route (Type 3).. Ethernet Segment is responsible for advertising Ethernet Segment Route (Type 4).

        Returns: BgpV4EthernetSegmentIter
        """
        return self._get_property(
            "evpn_ethernet_segments",
            BgpV4EthernetSegmentIter,
            self._parent,
            self._choice,
        )

    @property
    def as_type(self):
        # type: () -> Union[Literal["ebgp"], Literal["ibgp"]]
        """as_type getter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        Returns: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        return self._get_property("as_type")

    @as_type.setter
    def as_type(self, value):
        """as_type setter

        The type of BGP autonomous system. External BGP is used for BGP links between two or more autonomous systems (ebgp). Internal BGP is used within single autonomous system (ibgp). BGP property defaults are aligned with this object defined as an internal BGP peer. If the as_type is specified as 'ebgp' then other properties will need to be specified as per an external BGP peer. Specifically, for 'ebgp', 'as_set_mode' attribute in 'as_path' field in any Route Range should be changed from default value 'do_not_include_local_as' to any other value.

        value: Union[Literal["ebgp"], Literal["ibgp"]]
        """
        if value is None:
            raise TypeError("Cannot set required property as_type as None")
        self._set_property("as_type", value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous System Number (AS number or ASN)

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous System Number (AS number or ASN)

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property as_number as None")
        self._set_property("as_number", value)

    @property
    def as_number_width(self):
        # type: () -> Union[Literal["four"], Literal["two"]]
        """as_number_width getter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        Returns: Union[Literal["four"], Literal["two"]]
        """
        return self._get_property("as_number_width")

    @as_number_width.setter
    def as_number_width(self, value):
        """as_number_width setter

        The width in bytes of the as_number values. Any as_number values that exceeds the width MUST result in an error.

        value: Union[Literal["four"], Literal["two"]]
        """
        self._set_property("as_number_width", value)

    @property
    def advanced(self):
        # type: () -> BgpAdvanced
        """advanced getter

        Configuration for BGP advanced settings.Configuration for BGP advanced settings.Configuration for BGP advanced settings.

        Returns: BgpAdvanced
        """
        return self._get_property("advanced", BgpAdvanced)

    @property
    def capability(self):
        # type: () -> BgpCapability
        """capability getter

        Configuration for BGP capability settings.Configuration for BGP capability settings.Configuration for BGP capability settings.

        Returns: BgpCapability
        """
        return self._get_property("capability", BgpCapability)

    @property
    def learned_information_filter(self):
        # type: () -> BgpLearnedInformationFilter
        """learned_information_filter getter

        Configuration for controlling storage of BGP learned information recieved from the peer.Configuration for controlling storage of BGP learned information recieved from the peer.Configuration for controlling storage of BGP learned information recieved from the peer.

        Returns: BgpLearnedInformationFilter
        """
        return self._get_property(
            "learned_information_filter", BgpLearnedInformationFilter
        )

    @property
    def v4_routes(self):
        # type: () -> BgpV4RouteRangeIter
        """v4_routes getter

        Emulated BGPv4 route ranges.

        Returns: BgpV4RouteRangeIter
        """
        return self._get_property(
            "v4_routes", BgpV4RouteRangeIter, self._parent, self._choice
        )

    @property
    def v6_routes(self):
        # type: () -> BgpV6RouteRangeIter
        """v6_routes getter

        Emulated BGPv6 route ranges.

        Returns: BgpV6RouteRangeIter
        """
        return self._get_property(
            "v6_routes", BgpV6RouteRangeIter, self._parent, self._choice
        )

    @property
    def v4_srte_policies(self):
        # type: () -> BgpSrteV4PolicyIter
        """v4_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv4 Address Family Identifier (AFI).

        Returns: BgpSrteV4PolicyIter
        """
        return self._get_property(
            "v4_srte_policies", BgpSrteV4PolicyIter, self._parent, self._choice
        )

    @property
    def v6_srte_policies(self):
        # type: () -> BgpSrteV6PolicyIter
        """v6_srte_policies getter

        Segment Routing Traffic Engineering (SR TE) Policies for IPv6 Address Family Identifier (AFI).

        Returns: BgpSrteV6PolicyIter
        """
        return self._get_property(
            "v6_srte_policies", BgpSrteV6PolicyIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.. Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.. Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def graceful_restart(self):
        # type: () -> BgpGracefulRestart
        """graceful_restart getter

        The Graceful Restart Capability (RFC 4724) is BGP capability that can be used by BGP speaker to indicate its ability to preserve its forwarding state during BGP restart. The Graceful Restart (GR) capability is advertised in OPEN messages sent between BGP peers. After BGP session has been established, and the initial routing update has been completed, an End-of-RIB (Routing Information Base) marker is sent in an UPDATE message to convey information about routing convergence.The Graceful Restart Capability (RFC 4724) is BGP capability that can be used by BGP speaker to indicate its ability to preserve its forwarding state during BGP restart. The Graceful Restart (GR) capability is advertised in OPEN messages sent between BGP peers. After BGP session has been established, and the initial routing update has been completed, an End-of-RIB (Routing Information Base) marker is sent in an UPDATE message to convey information about routing convergence.The Graceful Restart Capability (RFC 4724) is BGP capability that can be used by BGP speaker to indicate its ability to preserve its forwarding state during BGP restart. The Graceful Restart (GR) capability is advertised in OPEN messages sent between BGP peers. After BGP session has been established, and the initial routing update has been completed, an End-of-RIB (Routing Information Base) marker is sent in an UPDATE message to convey information about routing convergence.

        Returns: BgpGracefulRestart
        """
        return self._get_property("graceful_restart", BgpGracefulRestart)

    @property
    def replay_updates(self):
        # type: () -> BgpUpdateReplay
        """replay_updates getter

        Ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.Ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.Ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.BGP Updates to be sent to the peer as specified after the session is established.

        Returns: BgpUpdateReplay
        """
        return self._get_property("replay_updates", BgpUpdateReplay)


class BgpV4EthernetSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "df_election": {"type": "BgpEthernetSegmentDfElection"},
        "evis": {"type": "BgpV4EvpnEvisIter"},
        "esi": {
            "type": str,
            "format": "hex",
        },
        "active_mode": {
            "type": str,
            "enum": [
                "single_active",
                "all_active",
            ],
        },
        "esi_label": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "esi": "00000000000000000000",
        "active_mode": "all_active",
        "esi_label": 0,
    }  # type: Dict[str, Union(type)]

    SINGLE_ACTIVE = "single_active"  # type: str
    ALL_ACTIVE = "all_active"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        esi="00000000000000000000",
        active_mode="all_active",
        esi_label=0,
    ):
        super(BgpV4EthernetSegment, self).__init__()
        self._parent = parent
        self._set_property("esi", esi)
        self._set_property("active_mode", active_mode)
        self._set_property("esi_label", esi_label)

    def set(self, esi=None, active_mode=None, esi_label=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def df_election(self):
        # type: () -> BgpEthernetSegmentDfElection
        """df_election getter

        Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Configuration for Designated Forwarder (DF) election among the Provider Edge (PE) routers on the same Ethernet Segment.Designated Forwarder (DF) election configuration.

        Returns: BgpEthernetSegmentDfElection
        """
        return self._get_property("df_election", BgpEthernetSegmentDfElection)

    @property
    def evis(self):
        # type: () -> BgpV4EvpnEvisIter
        """evis getter

        This contains the list of EVIs.

        Returns: BgpV4EvpnEvisIter
        """
        return self._get_property("evis", BgpV4EvpnEvisIter, self._parent, self._choice)

    @property
    def esi(self):
        # type: () -> str
        """esi getter

        10-octet Ethernet Segment Identifier (ESI) Example For multi-home scenario nonZero ESI is '10000000000000000000' .

        Returns: str
        """
        return self._get_property("esi")

    @esi.setter
    def esi(self, value):
        """esi setter

        10-octet Ethernet Segment Identifier (ESI) Example For multi-home scenario nonZero ESI is '10000000000000000000' .

        value: str
        """
        self._set_property("esi", value)

    @property
    def active_mode(self):
        # type: () -> Union[Literal["all_active"], Literal["single_active"]]
        """active_mode getter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        Returns: Union[Literal["all_active"], Literal["single_active"]]
        """
        return self._get_property("active_mode")

    @active_mode.setter
    def active_mode(self, value):
        """active_mode setter

        Single Active or All Active mode Redundancy mode selection for Multi-home.

        value: Union[Literal["all_active"], Literal["single_active"]]
        """
        self._set_property("active_mode", value)

    @property
    def esi_label(self):
        # type: () -> int
        """esi_label getter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by router.

        Returns: int
        """
        return self._get_property("esi_label")

    @esi_label.setter
    def esi_label(self, value):
        """esi_label setter

        The label value to be advertised as ESI Label in ESI Label Extended Community. This is included in Ethernet Auto-discovery per ES Routes advertised by router.

        value: int
        """
        self._set_property("esi_label", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpEthernetSegmentDfElection(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "election_timer": {
            "type": int,
            "format": "uint32",
            "maximum": 300,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "election_timer": 3,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, election_timer=3):
        super(BgpEthernetSegmentDfElection, self).__init__()
        self._parent = parent
        self._set_property("election_timer", election_timer)

    def set(self, election_timer=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def election_timer(self):
        # type: () -> int
        """election_timer getter

        The DF election timer in seconds.

        Returns: int
        """
        return self._get_property("election_timer")

    @election_timer.setter
    def election_timer(self, value):
        """election_timer setter

        The DF election timer in seconds.

        value: int
        """
        self._set_property("election_timer", value)


class BgpV4EvpnEvis(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "evi_vxlan",
            ],
        },
        "evi_vxlan": {"type": "BgpV4EviVxlan"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "evi_vxlan",
    }  # type: Dict[str, Union(type)]

    EVI_VXLAN = "evi_vxlan"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpV4EvpnEvis, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def evi_vxlan(self):
        # type: () -> BgpV4EviVxlan
        """Factory property that returns an instance of the BgpV4EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes . Type - Inclusive Multicast Ethernet Tag Route. Type - Ethernet Auto-discovery Route (Per EVI). Type - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV4EviVxlan
        """
        return self._get_property("evi_vxlan", BgpV4EviVxlan, self, "evi_vxlan")

    @property
    def choice(self):
        # type: () -> Union[Literal["evi_vxlan"]]
        """choice getter

        TBD

        Returns: Union[Literal["evi_vxlan"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["evi_vxlan"]]
        """
        self._set_property("choice", value)


class BgpV4EviVxlan(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "broadcast_domains": {"type": "BgpV4EviVxlanBroadcastDomainIter"},
        "replication_type": {
            "type": str,
            "enum": [
                "ingress_replication",
            ],
        },
        "pmsi_label": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "ad_label": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "route_distinguisher": {"type": "BgpRouteDistinguisher"},
        "route_target_export": {"type": "BgpRouteTargetIter"},
        "route_target_import": {"type": "BgpRouteTargetIter"},
        "l3_route_target_export": {"type": "BgpRouteTargetIter"},
        "l3_route_target_import": {"type": "BgpRouteTargetIter"},
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "replication_type": "ingress_replication",
        "pmsi_label": 16,
        "ad_label": 0,
    }  # type: Dict[str, Union(type)]

    INGRESS_REPLICATION = "ingress_replication"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        replication_type="ingress_replication",
        pmsi_label=16,
        ad_label=0,
    ):
        super(BgpV4EviVxlan, self).__init__()
        self._parent = parent
        self._set_property("replication_type", replication_type)
        self._set_property("pmsi_label", pmsi_label)
        self._set_property("ad_label", ad_label)

    def set(self, replication_type=None, pmsi_label=None, ad_label=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def broadcast_domains(self):
        # type: () -> BgpV4EviVxlanBroadcastDomainIter
        """broadcast_domains getter

        This contains the list of Broadcast Domains to be configured per EVI.

        Returns: BgpV4EviVxlanBroadcastDomainIter
        """
        return self._get_property(
            "broadcast_domains",
            BgpV4EviVxlanBroadcastDomainIter,
            self._parent,
            self._choice,
        )

    @property
    def replication_type(self):
        # type: () -> Union[Literal["ingress_replication"]]
        """replication_type getter

        This model only supports Ingress Replication

        Returns: Union[Literal["ingress_replication"]]
        """
        return self._get_property("replication_type")

    @replication_type.setter
    def replication_type(self, value):
        """replication_type setter

        This model only supports Ingress Replication

        value: Union[Literal["ingress_replication"]]
        """
        self._set_property("replication_type", value)

    @property
    def pmsi_label(self):
        # type: () -> int
        """pmsi_label getter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type Inclusive Multicast Ethernet Tag Route.

        Returns: int
        """
        return self._get_property("pmsi_label")

    @pmsi_label.setter
    def pmsi_label(self, value):
        """pmsi_label setter

        Downstream assigned VNI to be carried as Part of P-Multicast Service Interface Tunnel attribute (PMSI Tunnel Attribute) in Type Inclusive Multicast Ethernet Tag Route.

        value: int
        """
        self._set_property("pmsi_label", value)

    @property
    def ad_label(self):
        # type: () -> int
        """ad_label getter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        Returns: int
        """
        return self._get_property("ad_label")

    @ad_label.setter
    def ad_label(self, value):
        """ad_label setter

        The Auto-discovery Route label (AD label) value, which gets advertised in the Ethernet Auto-discovery Route per <EVI, ESI>

        value: int
        """
        self._set_property("ad_label", value)

    @property
    def route_distinguisher(self):
        # type: () -> BgpRouteDistinguisher
        """route_distinguisher getter

        BGP Route Distinguisher.BGP Route Distinguisher.BGP Route Distinguisher.BGP Route Distinguisher.Colon separated Extended Community value of Bytes "AS number: Value" identifying an EVI. Example for the as_2octet "60005:100".

        Returns: BgpRouteDistinguisher
        """
        return self._get_property("route_distinguisher", BgpRouteDistinguisher)

    @property
    def route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """route_target_export getter

        List of Layer Virtual Network Identifier (L2VNI) export targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """route_target_import getter

        List of L2VNI import targets associated with this EVI.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_export(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_export getter

        List of Layer Virtual Network Identifier (L3VNI) Export Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_export", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def l3_route_target_import(self):
        # type: () -> BgpRouteTargetIter
        """l3_route_target_import getter

        List of L3VNI Import Route Targets.

        Returns: BgpRouteTargetIter
        """
        return self._get_property(
            "l3_route_target_import", BgpRouteTargetIter, self._parent, self._choice
        )

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)


class BgpV4EviVxlanBroadcastDomain(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "cmac_ip_range": {"type": "BgpCMacIpRangeIter"},
        "ethernet_tag_id": {
            "type": int,
            "format": "uint32",
        },
        "vlan_aware_service": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ethernet_tag_id": 0,
        "vlan_aware_service": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, ethernet_tag_id=0, vlan_aware_service=False):
        super(BgpV4EviVxlanBroadcastDomain, self).__init__()
        self._parent = parent
        self._set_property("ethernet_tag_id", ethernet_tag_id)
        self._set_property("vlan_aware_service", vlan_aware_service)

    def set(self, ethernet_tag_id=None, vlan_aware_service=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def cmac_ip_range(self):
        # type: () -> BgpCMacIpRangeIter
        """cmac_ip_range getter

        This contains the list of Customer MAC/IP Ranges to be configured per Broadcast Domain. . Advertises following route . Type - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRangeIter
        """
        return self._get_property(
            "cmac_ip_range", BgpCMacIpRangeIter, self._parent, self._choice
        )

    @property
    def ethernet_tag_id(self):
        # type: () -> int
        """ethernet_tag_id getter

        The Ethernet Tag ID of the Broadcast Domain.

        Returns: int
        """
        return self._get_property("ethernet_tag_id")

    @ethernet_tag_id.setter
    def ethernet_tag_id(self, value):
        """ethernet_tag_id setter

        The Ethernet Tag ID of the Broadcast Domain.

        value: int
        """
        self._set_property("ethernet_tag_id", value)

    @property
    def vlan_aware_service(self):
        # type: () -> bool
        """vlan_aware_service getter

        VLAN-Aware service to be enabled or disabled.

        Returns: bool
        """
        return self._get_property("vlan_aware_service")

    @vlan_aware_service.setter
    def vlan_aware_service(self, value):
        """vlan_aware_service setter

        VLAN-Aware service to be enabled or disabled.

        value: bool
        """
        self._set_property("vlan_aware_service", value)


class BgpCMacIpRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "mac_addresses": {"type": "MACRouteAddress"},
        "l2vni": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "ipv4_addresses": {"type": "V4RouteAddress"},
        "ipv6_addresses": {"type": "V6RouteAddress"},
        "l3vni": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "include_default_gateway": {"type": bool},
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "name": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "l2vni": 0,
        "l3vni": 0,
        "include_default_gateway": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, l2vni=0, l3vni=0, include_default_gateway=False, name=None
    ):
        super(BgpCMacIpRange, self).__init__()
        self._parent = parent
        self._set_property("l2vni", l2vni)
        self._set_property("l3vni", l3vni)
        self._set_property("include_default_gateway", include_default_gateway)
        self._set_property("name", name)

    def set(self, l2vni=None, l3vni=None, include_default_gateway=None, name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def mac_addresses(self):
        # type: () -> MACRouteAddress
        """mac_addresses getter

        A container for MAC route addresses.A container for MAC route addresses.A container for MAC route addresses.Host MAC address range per Broadcast Domain.

        Returns: MACRouteAddress
        """
        return self._get_property("mac_addresses", MACRouteAddress)

    @property
    def l2vni(self):
        # type: () -> int
        """l2vni getter

        Layer Virtual Network Identifier (L2VNI) to be advertised with MAC/IP Advertisement Route (Type 2)

        Returns: int
        """
        return self._get_property("l2vni")

    @l2vni.setter
    def l2vni(self, value):
        """l2vni setter

        Layer Virtual Network Identifier (L2VNI) to be advertised with MAC/IP Advertisement Route (Type 2)

        value: int
        """
        self._set_property("l2vni", value)

    @property
    def ipv4_addresses(self):
        # type: () -> V4RouteAddress
        """ipv4_addresses getter

        A container for IPv4 route addresses.A container for IPv4 route addresses.A container for IPv4 route addresses.Host IPv4 address range per Broadcast Domain.

        Returns: V4RouteAddress
        """
        return self._get_property("ipv4_addresses", V4RouteAddress)

    @property
    def ipv6_addresses(self):
        # type: () -> V6RouteAddress
        """ipv6_addresses getter

        A container for IPv6 route addresses.A container for IPv6 route addresses.A container for IPv6 route addresses.Host IPv6 address range per Broadcast Domain.

        Returns: V6RouteAddress
        """
        return self._get_property("ipv6_addresses", V6RouteAddress)

    @property
    def l3vni(self):
        # type: () -> int
        """l3vni getter

        Layer Virtual Network Identifier (L3VNI) to be advertised with MAC/IP Advertisement Route (Type 2).

        Returns: int
        """
        return self._get_property("l3vni")

    @l3vni.setter
    def l3vni(self, value):
        """l3vni setter

        Layer Virtual Network Identifier (L3VNI) to be advertised with MAC/IP Advertisement Route (Type 2).

        value: int
        """
        self._set_property("l3vni", value)

    @property
    def include_default_gateway(self):
        # type: () -> bool
        """include_default_gateway getter

        Include default Gateway Extended Community in MAC/IP Advertisement Route (Type 2).

        Returns: bool
        """
        return self._get_property("include_default_gateway")

    @include_default_gateway.setter
    def include_default_gateway(self, value):
        """include_default_gateway setter

        Include default Gateway Extended Community in MAC/IP Advertisement Route (Type 2).

        value: bool
        """
        self._set_property("include_default_gateway", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.Optional AS PATH settings.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)


class MACRouteAddress(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "address": {
            "type": str,
            "format": "mac",
        },
        "prefix": {
            "type": int,
            "format": "uint32",
            "minimum": 0,
            "maximum": 48,
        },
        "count": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
        "step": {
            "type": int,
            "format": "uint32",
            "minimum": 1,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("address",)  # type: tuple(str)

    _DEFAULTS = {
        "prefix": 48,
        "count": 1,
        "step": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, address=None, prefix=48, count=1, step=1):
        super(MACRouteAddress, self).__init__()
        self._parent = parent
        self._set_property("address", address)
        self._set_property("prefix", prefix)
        self._set_property("count", count)
        self._set_property("step", step)

    def set(self, address=None, prefix=None, count=None, step=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def address(self):
        # type: () -> str
        """address getter

        The starting address of the MAC Range.

        Returns: str
        """
        return self._get_property("address")

    @address.setter
    def address(self, value):
        """address setter

        The starting address of the MAC Range.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property address as None")
        self._set_property("address", value)

    @property
    def prefix(self):
        # type: () -> int
        """prefix getter

        The MAC prefix length to be applied to the address.

        Returns: int
        """
        return self._get_property("prefix")

    @prefix.setter
    def prefix(self, value):
        """prefix setter

        The MAC prefix length to be applied to the address.

        value: int
        """
        self._set_property("prefix", value)

    @property
    def count(self):
        # type: () -> int
        """count getter

        The total number of mac addresses in the range.

        Returns: int
        """
        return self._get_property("count")

    @count.setter
    def count(self, value):
        """count setter

        The total number of mac addresses in the range.

        value: int
        """
        self._set_property("count", value)

    @property
    def step(self):
        # type: () -> int
        """step getter

        Increments the mac address prefixes within mac range where multiple routes are present. The value is incremented according to the mac prefix Length and Step.

        Returns: int
        """
        return self._get_property("step")

    @step.setter
    def step(self, value):
        """step setter

        Increments the mac address prefixes within mac range where multiple routes are present. The value is incremented according to the mac prefix Length and Step.

        value: int
        """
        self._set_property("step", value)


class BgpRouteAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "include_multi_exit_discriminator": {"type": bool},
        "multi_exit_discriminator": {
            "type": int,
            "format": "uint32",
        },
        "include_origin": {"type": bool},
        "origin": {
            "type": str,
            "enum": [
                "igp",
                "egp",
                "incomplete",
            ],
        },
        "include_local_preference": {"type": bool},
        "local_preference": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "include_multi_exit_discriminator": True,
        "include_origin": True,
        "origin": "igp",
        "include_local_preference": True,
        "local_preference": 100,
    }  # type: Dict[str, Union(type)]

    IGP = "igp"  # type: str
    EGP = "egp"  # type: str
    INCOMPLETE = "incomplete"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        include_multi_exit_discriminator=True,
        multi_exit_discriminator=None,
        include_origin=True,
        origin="igp",
        include_local_preference=True,
        local_preference=100,
    ):
        super(BgpRouteAdvanced, self).__init__()
        self._parent = parent
        self._set_property(
            "include_multi_exit_discriminator", include_multi_exit_discriminator
        )
        self._set_property("multi_exit_discriminator", multi_exit_discriminator)
        self._set_property("include_origin", include_origin)
        self._set_property("origin", origin)
        self._set_property("include_local_preference", include_local_preference)
        self._set_property("local_preference", local_preference)

    def set(
        self,
        include_multi_exit_discriminator=None,
        multi_exit_discriminator=None,
        include_origin=None,
        origin=None,
        include_local_preference=None,
        local_preference=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def include_multi_exit_discriminator(self):
        # type: () -> bool
        """include_multi_exit_discriminator getter

        BGP Multi Exit Discriminator attribute sent to the peer to help in the route selection process. If set to true, the Multi Exit Discriminator attribute will be included in the route advertisement.

        Returns: bool
        """
        return self._get_property("include_multi_exit_discriminator")

    @include_multi_exit_discriminator.setter
    def include_multi_exit_discriminator(self, value):
        """include_multi_exit_discriminator setter

        BGP Multi Exit Discriminator attribute sent to the peer to help in the route selection process. If set to true, the Multi Exit Discriminator attribute will be included in the route advertisement.

        value: bool
        """
        self._set_property("include_multi_exit_discriminator", value)

    @property
    def multi_exit_discriminator(self):
        # type: () -> int
        """multi_exit_discriminator getter

        The multi exit discriminator (MED) value used for route selection sent to the peer.

        Returns: int
        """
        return self._get_property("multi_exit_discriminator")

    @multi_exit_discriminator.setter
    def multi_exit_discriminator(self, value):
        """multi_exit_discriminator setter

        The multi exit discriminator (MED) value used for route selection sent to the peer.

        value: int
        """
        self._set_property("multi_exit_discriminator", value)

    @property
    def include_origin(self):
        # type: () -> bool
        """include_origin getter

        If set to true, the Origin attribute will be included in the route advertisement.

        Returns: bool
        """
        return self._get_property("include_origin")

    @include_origin.setter
    def include_origin(self, value):
        """include_origin setter

        If set to true, the Origin attribute will be included in the route advertisement.

        value: bool
        """
        self._set_property("include_origin", value)

    @property
    def origin(self):
        # type: () -> Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """origin getter

        The origin attribute of prefix can take three values: the prefix originates from an interior routing protocol 'igp', it originates from 'egp' or the origin is 'incomplete', if the prefix is learned through other means.

        Returns: Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """
        return self._get_property("origin")

    @origin.setter
    def origin(self, value):
        """origin setter

        The origin attribute of prefix can take three values: the prefix originates from an interior routing protocol 'igp', it originates from 'egp' or the origin is 'incomplete', if the prefix is learned through other means.

        value: Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """
        self._set_property("origin", value)

    @property
    def include_local_preference(self):
        # type: () -> bool
        """include_local_preference getter

        BGP Local Preference attribute sent to the peer to indicate the degree of preference for externally learned routes. If set to true, the Local Preference attribute will be included in the route advertisement. This should be included only for internal peers.

        Returns: bool
        """
        return self._get_property("include_local_preference")

    @include_local_preference.setter
    def include_local_preference(self, value):
        """include_local_preference setter

        BGP Local Preference attribute sent to the peer to indicate the degree of preference for externally learned routes. If set to true, the Local Preference attribute will be included in the route advertisement. This should be included only for internal peers.

        value: bool
        """
        self._set_property("include_local_preference", value)

    @property
    def local_preference(self):
        # type: () -> int
        """local_preference getter

        Value to be set in Local Preference attribute if include_local_preference is set to true. It is used for the selection of the path for the traffic leaving the AS. The route with the highest local preference value is preferred.

        Returns: int
        """
        return self._get_property("local_preference")

    @local_preference.setter
    def local_preference(self, value):
        """local_preference setter

        Value to be set in Local Preference attribute if include_local_preference is set to true. It is used for the selection of the path for the traffic leaving the AS. The route with the highest local preference value is preferred.

        value: int
        """
        self._set_property("local_preference", value)


class BgpCommunity(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "manual_as_number",
                "no_export",
                "no_advertised",
                "no_export_subconfed",
                "llgr_stale",
                "no_llgr",
            ],
        },
        "as_number": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "as_custom": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_number": 0,
        "as_custom": 0,
    }  # type: Dict[str, Union(type)]

    MANUAL_AS_NUMBER = "manual_as_number"  # type: str
    NO_EXPORT = "no_export"  # type: str
    NO_ADVERTISED = "no_advertised"  # type: str
    NO_EXPORT_SUBCONFED = "no_export_subconfed"  # type: str
    LLGR_STALE = "llgr_stale"  # type: str
    NO_LLGR = "no_llgr"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type=None, as_number=0, as_custom=0):
        super(BgpCommunity, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_number", as_number)
        self._set_property("as_custom", as_custom)

    def set(self, type=None, as_number=None, as_custom=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """type getter

        The type of community AS number.

        Returns: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        The type of community AS number.

        value: Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        self._set_property("type", value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        First two octets of 32 bit community AS number.

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        First two octets of 32 bit community AS number.

        value: int
        """
        self._set_property("as_number", value)

    @property
    def as_custom(self):
        # type: () -> int
        """as_custom getter

        Last two octets of the community value.

        Returns: int
        """
        return self._get_property("as_custom")

    @as_custom.setter
    def as_custom(self, value):
        """as_custom setter

        Last two octets of the community value.

        value: int
        """
        self._set_property("as_custom", value)


class BgpCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpCommunity
        return self._next()

    def next(self):
        # type: () -> BgpCommunity
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpCommunity):
            raise Exception("Item is not an instance of BgpCommunity")

    def community(self, type=None, as_number=0, as_custom=0):
        # type: (Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]],int,int) -> BgpCommunityIter
        """Factory method that creates an instance of the BgpCommunity class

        BGP communities provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is 32-bit number which is broken into 16-bit AS number and 16-bit custom value.

        Returns: BgpCommunityIter
        """
        item = BgpCommunity(
            parent=self._parent, type=type, as_number=as_number, as_custom=as_custom
        )
        self._add(item)
        return self

    def add(self, type=None, as_number=0, as_custom=0):
        # type: (Union[Literal["llgr_stale"], Literal["manual_as_number"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]],int,int) -> BgpCommunity
        """Add method that creates and returns an instance of the BgpCommunity class

        BGP communities provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is 32-bit number which is broken into 16-bit AS number and 16-bit custom value.

        Returns: BgpCommunity
        """
        item = BgpCommunity(
            parent=self._parent, type=type, as_number=as_number, as_custom=as_custom
        )
        self._add(item)
        return item


class BgpExtCommunity(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "administrator_as_2octet",
                "administrator_ipv4_address",
                "administrator_as_4octet",
                "opaque",
                "evpn",
                "administrator_as_2octet_link_bandwidth",
            ],
        },
        "subtype": {
            "type": str,
            "enum": [
                "route_target",
                "origin",
                "extended_bandwidth",
                "color",
                "encapsulation",
                "mac_address",
            ],
        },
        "value": {
            "type": str,
            "format": "hex",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    ADMINISTRATOR_AS_2OCTET = "administrator_as_2octet"  # type: str
    ADMINISTRATOR_IPV4_ADDRESS = "administrator_ipv4_address"  # type: str
    ADMINISTRATOR_AS_4OCTET = "administrator_as_4octet"  # type: str
    OPAQUE = "opaque"  # type: str
    EVPN = "evpn"  # type: str
    ADMINISTRATOR_AS_2OCTET_LINK_BANDWIDTH = (
        "administrator_as_2octet_link_bandwidth"
    )  # type: str

    ROUTE_TARGET = "route_target"  # type: str
    ORIGIN = "origin"  # type: str
    EXTENDED_BANDWIDTH = "extended_bandwidth"  # type: str
    COLOR = "color"  # type: str
    ENCAPSULATION = "encapsulation"  # type: str
    MAC_ADDRESS = "mac_address"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type=None, subtype=None, value=None):
        super(BgpExtCommunity, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("subtype", subtype)
        self._set_property("value", value)

    def set(self, type=None, subtype=None, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """type getter

        Extended Community Type field of Byte.. administrator_as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. administrator_ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. administrator_as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).. opaque: Opaque Extended Community (RFC 7432).. evpn: EVPN Extended Community (RFC 7153). - administrator_as_2octet_link_bandwidth Link Bandwidth Extended Community (RFC 7153).

        Returns: Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        Extended Community Type field of Byte.. administrator_as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. administrator_ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. administrator_as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).. opaque: Opaque Extended Community (RFC 7432).. evpn: EVPN Extended Community (RFC 7153). - administrator_as_2octet_link_bandwidth Link Bandwidth Extended Community (RFC 7153).

        value: Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]]
        """
        self._set_property("type", value)

    @property
    def subtype(self):
        # type: () -> Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """subtype getter

        Extended Community Sub Type field of Byte.. route_target: Route Target.. origin: Origin.. extended_bandwidth: Specifies the link bandwidth.. color: Specifies the color value.. encapsulation: Specifies the Encapsulation Extended Community.. mac_address: Specifies the Extended community MAC address.

        Returns: Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """
        return self._get_property("subtype")

    @subtype.setter
    def subtype(self, value):
        """subtype setter

        Extended Community Sub Type field of Byte.. route_target: Route Target.. origin: Origin.. extended_bandwidth: Specifies the link bandwidth.. color: Specifies the color value.. encapsulation: Specifies the Encapsulation Extended Community.. mac_address: Specifies the Extended community MAC address.

        value: Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]]
        """
        self._set_property("subtype", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        Extended Community value of Bytes. Example for the Opaque type and Color subtype value can be '0000000000c8' for the color value 200.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Extended Community value of Bytes. Example for the Opaque type and Color subtype value can be '0000000000c8' for the color value 200.

        value: str
        """
        self._set_property("value", value)


class BgpExtCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpExtCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpExtCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpExtCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpExtCommunity
        return self._next()

    def next(self):
        # type: () -> BgpExtCommunity
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpExtCommunity):
            raise Exception("Item is not an instance of BgpExtCommunity")

    def extcommunity(self, type=None, subtype=None, value=None):
        # type: (Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]],Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]],str) -> BgpExtCommunityIter
        """Factory method that creates an instance of the BgpExtCommunity class

        The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtCommunityIter
        """
        item = BgpExtCommunity(
            parent=self._parent, type=type, subtype=subtype, value=value
        )
        self._add(item)
        return self

    def add(self, type=None, subtype=None, value=None):
        # type: (Union[Literal["administrator_as_2octet"], Literal["administrator_as_2octet_link_bandwidth"], Literal["administrator_as_4octet"], Literal["administrator_ipv4_address"], Literal["evpn"], Literal["opaque"]],Union[Literal["color"], Literal["encapsulation"], Literal["extended_bandwidth"], Literal["mac_address"], Literal["origin"], Literal["route_target"]],str) -> BgpExtCommunity
        """Add method that creates and returns an instance of the BgpExtCommunity class

        The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtCommunity
        """
        item = BgpExtCommunity(
            parent=self._parent, type=type, subtype=subtype, value=value
        )
        self._add(item)
        return item


class BgpAsPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_set_mode": {
            "type": str,
            "enum": [
                "do_not_include_local_as",
                "include_as_seq",
                "include_as_set",
                "include_as_confed_seq",
                "include_as_confed_set",
                "prepend_to_first_segment",
            ],
        },
        "segments": {"type": "BgpAsPathSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_set_mode": "do_not_include_local_as",
    }  # type: Dict[str, Union(type)]

    DO_NOT_INCLUDE_LOCAL_AS = "do_not_include_local_as"  # type: str
    INCLUDE_AS_SEQ = "include_as_seq"  # type: str
    INCLUDE_AS_SET = "include_as_set"  # type: str
    INCLUDE_AS_CONFED_SEQ = "include_as_confed_seq"  # type: str
    INCLUDE_AS_CONFED_SET = "include_as_confed_set"  # type: str
    PREPEND_TO_FIRST_SEGMENT = "prepend_to_first_segment"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, as_set_mode="do_not_include_local_as"):
        super(BgpAsPath, self).__init__()
        self._parent = parent
        self._set_property("as_set_mode", as_set_mode)

    def set(self, as_set_mode=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def as_set_mode(self):
        # type: () -> Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """as_set_mode getter

        Defines how the Local AS should be included in the MP REACH NLRI. For iBGP sessions, "Do Not Include Local AS" must be chosen. For eBGP sessions, any choice other than "Do Not Include Local AS" can be chosen.

        Returns: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        return self._get_property("as_set_mode")

    @as_set_mode.setter
    def as_set_mode(self, value):
        """as_set_mode setter

        Defines how the Local AS should be included in the MP REACH NLRI. For iBGP sessions, "Do Not Include Local AS" must be chosen. For eBGP sessions, any choice other than "Do Not Include Local AS" can be chosen.

        value: Union[Literal["do_not_include_local_as"], Literal["include_as_confed_seq"], Literal["include_as_confed_set"], Literal["include_as_seq"], Literal["include_as_set"], Literal["prepend_to_first_segment"]]
        """
        self._set_property("as_set_mode", value)

    @property
    def segments(self):
        # type: () -> BgpAsPathSegmentIter
        """segments getter

        The additional AS path segments to be added in the NLRI. By default, an empty AS path is always included and the local AS is added to it as per the value of 'as_set_mode' attribute.

        Returns: BgpAsPathSegmentIter
        """
        return self._get_property(
            "segments", BgpAsPathSegmentIter, self._parent, self._choice
        )


class BgpAsPathSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "as_seq",
                "as_set",
                "as_confed_seq",
                "as_confed_set",
            ],
        },
        "as_numbers": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "type": "as_seq",
    }  # type: Dict[str, Union(type)]

    AS_SEQ = "as_seq"  # type: str
    AS_SET = "as_set"  # type: str
    AS_CONFED_SEQ = "as_confed_seq"  # type: str
    AS_CONFED_SET = "as_confed_set"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type="as_seq", as_numbers=None):
        super(BgpAsPathSegment, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_numbers", as_numbers)

    def set(self, type=None, as_numbers=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """type getter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        Returns: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        AS sequence is the most common type of AS_PATH, it contains the list of ASNs starting with the most recent ASN being added read from left to right.. The other three AS_PATH types are used for Confederations AS_SET is the type of AS_PATH attribute that summarizes routes using using the aggregate-address command, allowing AS_PATHs to be summarized in the update as well. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most recent ASN to be added reading left to right AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent in BGP Updates.

        value: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        self._set_property("type", value)

    @property
    def as_numbers(self):
        # type: () -> List[int]
        """as_numbers getter

        The AS numbers in this AS path segment.

        Returns: List[int]
        """
        return self._get_property("as_numbers")

    @as_numbers.setter
    def as_numbers(self, value):
        """as_numbers setter

        The AS numbers in this AS path segment.

        value: List[int]
        """
        self._set_property("as_numbers", value)


class BgpAsPathSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAsPathSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAsPathSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAsPathSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAsPathSegment
        return self._next()

    def next(self):
        # type: () -> BgpAsPathSegment
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAsPathSegment):
            raise Exception("Item is not an instance of BgpAsPathSegment")

    def aspathsegment(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAsPathSegmentIter
        """Factory method that creates an instance of the BgpAsPathSegment class

        Configuration for single BGP AS path segment

        Returns: BgpAsPathSegmentIter
        """
        item = BgpAsPathSegment(parent=self._parent, type=type, as_numbers=as_numbers)
        self._add(item)
        return self

    def add(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAsPathSegment
        """Add method that creates and returns an instance of the BgpAsPathSegment class

        Configuration for single BGP AS path segment

        Returns: BgpAsPathSegment
        """
        item = BgpAsPathSegment(parent=self._parent, type=type, as_numbers=as_numbers)
        self._add(item)
        return item


class BgpCMacIpRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpCMacIpRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpCMacIpRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpCMacIpRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpCMacIpRange
        return self._next()

    def next(self):
        # type: () -> BgpCMacIpRange
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpCMacIpRange):
            raise Exception("Item is not an instance of BgpCMacIpRange")

    def cmaciprange(self, l2vni=0, l3vni=0, include_default_gateway=False, name=None):
        # type: (int,int,bool,str) -> BgpCMacIpRangeIter
        """Factory method that creates an instance of the BgpCMacIpRange class

        Configuration for MAC/IP Ranges per Broadcast Domain. . Advertises following route -. Type - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRangeIter
        """
        item = BgpCMacIpRange(
            parent=self._parent,
            l2vni=l2vni,
            l3vni=l3vni,
            include_default_gateway=include_default_gateway,
            name=name,
        )
        self._add(item)
        return self

    def add(self, l2vni=0, l3vni=0, include_default_gateway=False, name=None):
        # type: (int,int,bool,str) -> BgpCMacIpRange
        """Add method that creates and returns an instance of the BgpCMacIpRange class

        Configuration for MAC/IP Ranges per Broadcast Domain. . Advertises following route -. Type - MAC/IP Advertisement Route.

        Returns: BgpCMacIpRange
        """
        item = BgpCMacIpRange(
            parent=self._parent,
            l2vni=l2vni,
            l3vni=l3vni,
            include_default_gateway=include_default_gateway,
            name=name,
        )
        self._add(item)
        return item


class BgpV4EviVxlanBroadcastDomainIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4EviVxlanBroadcastDomainIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EviVxlanBroadcastDomain]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EviVxlanBroadcastDomainIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EviVxlanBroadcastDomain
        return self._next()

    def next(self):
        # type: () -> BgpV4EviVxlanBroadcastDomain
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpV4EviVxlanBroadcastDomain):
            raise Exception("Item is not an instance of BgpV4EviVxlanBroadcastDomain")

    def broadcastdomain(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV4EviVxlanBroadcastDomainIter
        """Factory method that creates an instance of the BgpV4EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV4EviVxlanBroadcastDomainIter
        """
        item = BgpV4EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return self

    def add(self, ethernet_tag_id=0, vlan_aware_service=False):
        # type: (int,bool) -> BgpV4EviVxlanBroadcastDomain
        """Add method that creates and returns an instance of the BgpV4EviVxlanBroadcastDomain class

        Configuration for Broadcast Domains per EVI.

        Returns: BgpV4EviVxlanBroadcastDomain
        """
        item = BgpV4EviVxlanBroadcastDomain(
            parent=self._parent,
            ethernet_tag_id=ethernet_tag_id,
            vlan_aware_service=vlan_aware_service,
        )
        self._add(item)
        return item


class BgpRouteDistinguisher(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "rd_type": {
            "type": str,
            "enum": [
                "as_2octet",
                "ipv4_address",
                "as_4octet",
            ],
        },
        "auto_config_rd_ip_addr": {"type": bool},
        "rd_value": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "rd_type": "as_2octet",
        "auto_config_rd_ip_addr": False,
    }  # type: Dict[str, Union(type)]

    AS_2OCTET = "as_2octet"  # type: str
    IPV4_ADDRESS = "ipv4_address"  # type: str
    AS_4OCTET = "as_4octet"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        rd_type="as_2octet",
        auto_config_rd_ip_addr=False,
        rd_value=None,
    ):
        super(BgpRouteDistinguisher, self).__init__()
        self._parent = parent
        self._set_property("rd_type", rd_type)
        self._set_property("auto_config_rd_ip_addr", auto_config_rd_ip_addr)
        self._set_property("rd_value", rd_value)

    def set(self, rd_type=None, auto_config_rd_ip_addr=None, rd_value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def rd_type(self):
        # type: () -> Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """rd_type getter

        Route Distinguisher Type field of Byte.. as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        Returns: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        return self._get_property("rd_type")

    @rd_type.setter
    def rd_type(self, value):
        """rd_type setter

        Route Distinguisher Type field of Byte.. as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        value: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        self._set_property("rd_type", value)

    @property
    def auto_config_rd_ip_addr(self):
        # type: () -> bool
        """auto_config_rd_ip_addr getter

        Allow to automatically configure RD IP address from local ip.

        Returns: bool
        """
        return self._get_property("auto_config_rd_ip_addr")

    @auto_config_rd_ip_addr.setter
    def auto_config_rd_ip_addr(self, value):
        """auto_config_rd_ip_addr setter

        Allow to automatically configure RD IP address from local ip.

        value: bool
        """
        self._set_property("auto_config_rd_ip_addr", value)

    @property
    def rd_value(self):
        # type: () -> str
        """rd_value getter

        Colon separated Extended Community value of Bytes "AS number: Value". Example for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        Returns: str
        """
        return self._get_property("rd_value")

    @rd_value.setter
    def rd_value(self, value):
        """rd_value setter

        Colon separated Extended Community value of Bytes "AS number: Value". Example for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        value: str
        """
        self._set_property("rd_value", value)


class BgpRouteTarget(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "rt_type": {
            "type": str,
            "enum": [
                "as_2octet",
                "ipv4_address",
                "as_4octet",
            ],
        },
        "rt_value": {"type": str},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    AS_2OCTET = "as_2octet"  # type: str
    IPV4_ADDRESS = "ipv4_address"  # type: str
    AS_4OCTET = "as_4octet"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, rt_type=None, rt_value=None):
        super(BgpRouteTarget, self).__init__()
        self._parent = parent
        self._set_property("rt_type", rt_type)
        self._set_property("rt_value", rt_value)

    def set(self, rt_type=None, rt_value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def rt_type(self):
        # type: () -> Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """rt_type getter

        Extended Community Type field of Byte.. as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        Returns: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        return self._get_property("rt_type")

    @rt_type.setter
    def rt_type(self, value):
        """rt_type setter

        Extended Community Type field of Byte.. as_2octet: Two-Octet AS Specific Extended Community (RFC 4360).. ipv4_address: IPv4 Address Specific Extended Community (RFC 4360).. as_4octet: 4-Octet AS Specific Extended Community (RFC 5668).

        value: Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]]
        """
        self._set_property("rt_type", value)

    @property
    def rt_value(self):
        # type: () -> str
        """rt_value getter

        Colon separated Extended Community value of Bytes AS number: Assigned Number. Example for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        Returns: str
        """
        return self._get_property("rt_value")

    @rt_value.setter
    def rt_value(self, value):
        """rt_value setter

        Colon separated Extended Community value of Bytes AS number: Assigned Number. Example for the as_2octet or as_4octet "60005:100", for ipv4_address "1.1.1.1:100"

        value: str
        """
        self._set_property("rt_value", value)


class BgpRouteTargetIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpRouteTargetIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpRouteTarget]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpRouteTargetIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpRouteTarget
        return self._next()

    def next(self):
        # type: () -> BgpRouteTarget
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpRouteTarget):
            raise Exception("Item is not an instance of BgpRouteTarget")

    def routetarget(self, rt_type=None, rt_value=None):
        # type: (Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]],str) -> BgpRouteTargetIter
        """Factory method that creates an instance of the BgpRouteTarget class

        BGP Route Target.

        Returns: BgpRouteTargetIter
        """
        item = BgpRouteTarget(parent=self._parent, rt_type=rt_type, rt_value=rt_value)
        self._add(item)
        return self

    def add(self, rt_type=None, rt_value=None):
        # type: (Union[Literal["as_2octet"], Literal["as_4octet"], Literal["ipv4_address"]],str) -> BgpRouteTarget
        """Add method that creates and returns an instance of the BgpRouteTarget class

        BGP Route Target.

        Returns: BgpRouteTarget
        """
        item = BgpRouteTarget(parent=self._parent, rt_type=rt_type, rt_value=rt_value)
        self._add(item)
        return item


class BgpV4EvpnEvisIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(BgpV4EvpnEvisIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EviVxlan, BgpV4EvpnEvis]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EvpnEvisIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EvpnEvis
        return self._next()

    def next(self):
        # type: () -> BgpV4EvpnEvis
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpV4EvpnEvis):
            raise Exception("Item is not an instance of BgpV4EvpnEvis")

    def evpnevis(self):
        # type: () -> BgpV4EvpnEvisIter
        """Factory method that creates an instance of the BgpV4EvpnEvis class

        This contains list of different flavors of EVPN. For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV4EvpnEvisIter
        """
        item = BgpV4EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> BgpV4EvpnEvis
        """Add method that creates and returns an instance of the BgpV4EvpnEvis class

        This contains list of different flavors of EVPN. For example EVPN over VXLAN or EVPN over MPLS etc to be configured per Ethernet segment. Need to instantiate correct type of EVPN instance as per requirement.

        Returns: BgpV4EvpnEvis
        """
        item = BgpV4EvpnEvis(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def evi_vxlan(
        self, replication_type="ingress_replication", pmsi_label=16, ad_label=0
    ):
        # type: (Union[Literal["ingress_replication"]],int,int) -> BgpV4EvpnEvisIter
        """Factory method that creates an instance of the BgpV4EviVxlan class

        Configuration for BGP EVPN EVI. Advertises following routes . Type - Inclusive Multicast Ethernet Tag Route. Type - Ethernet Auto-discovery Route (Per EVI). Type - Ethernet Auto-discovery Route (Per ES)

        Returns: BgpV4EvpnEvisIter
        """
        item = BgpV4EvpnEvis()
        item.evi_vxlan
        item.choice = "evi_vxlan"
        self._add(item)
        return self


class BgpV4EthernetSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4EthernetSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4EthernetSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4EthernetSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4EthernetSegment
        return self._next()

    def next(self):
        # type: () -> BgpV4EthernetSegment
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpV4EthernetSegment):
            raise Exception("Item is not an instance of BgpV4EthernetSegment")

    def ethernetsegment(
        self, esi="00000000000000000000", active_mode="all_active", esi_label=0
    ):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV4EthernetSegmentIter
        """Factory method that creates an instance of the BgpV4EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes . Type - Ethernet Segment Route

        Returns: BgpV4EthernetSegmentIter
        """
        item = BgpV4EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return self

    def add(self, esi="00000000000000000000", active_mode="all_active", esi_label=0):
        # type: (str,Union[Literal["all_active"], Literal["single_active"]],int) -> BgpV4EthernetSegment
        """Add method that creates and returns an instance of the BgpV4EthernetSegment class

        Configuration for BGP Ethernet Segment ranges. Advertises following routes . Type - Ethernet Segment Route

        Returns: BgpV4EthernetSegment
        """
        item = BgpV4EthernetSegment(
            parent=self._parent, esi=esi, active_mode=active_mode, esi_label=esi_label
        )
        self._add(item)
        return item


class BgpAdvanced(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "hold_time_interval": {
            "type": int,
            "format": "uint32",
        },
        "keep_alive_interval": {
            "type": int,
            "format": "uint32",
        },
        "update_interval": {
            "type": int,
            "format": "uint32",
        },
        "time_to_live": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
        "md5_key": {"type": str},
        "passive_mode": {"type": bool},
        "listen_port": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "neighbor_port": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "hold_time_interval": 90,
        "keep_alive_interval": 30,
        "update_interval": 0,
        "time_to_live": 64,
        "passive_mode": False,
        "listen_port": 179,
        "neighbor_port": 179,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        hold_time_interval=90,
        keep_alive_interval=30,
        update_interval=0,
        time_to_live=64,
        md5_key=None,
        passive_mode=False,
        listen_port=179,
        neighbor_port=179,
    ):
        super(BgpAdvanced, self).__init__()
        self._parent = parent
        self._set_property("hold_time_interval", hold_time_interval)
        self._set_property("keep_alive_interval", keep_alive_interval)
        self._set_property("update_interval", update_interval)
        self._set_property("time_to_live", time_to_live)
        self._set_property("md5_key", md5_key)
        self._set_property("passive_mode", passive_mode)
        self._set_property("listen_port", listen_port)
        self._set_property("neighbor_port", neighbor_port)

    def set(
        self,
        hold_time_interval=None,
        keep_alive_interval=None,
        update_interval=None,
        time_to_live=None,
        md5_key=None,
        passive_mode=None,
        listen_port=None,
        neighbor_port=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def hold_time_interval(self):
        # type: () -> int
        """hold_time_interval getter

        Number of seconds the sender proposes for the value of the Hold Timer.

        Returns: int
        """
        return self._get_property("hold_time_interval")

    @hold_time_interval.setter
    def hold_time_interval(self, value):
        """hold_time_interval setter

        Number of seconds the sender proposes for the value of the Hold Timer.

        value: int
        """
        self._set_property("hold_time_interval", value)

    @property
    def keep_alive_interval(self):
        # type: () -> int
        """keep_alive_interval getter

        Number of seconds between transmissions of Keepalive messages by this peer.

        Returns: int
        """
        return self._get_property("keep_alive_interval")

    @keep_alive_interval.setter
    def keep_alive_interval(self, value):
        """keep_alive_interval setter

        Number of seconds between transmissions of Keepalive messages by this peer.

        value: int
        """
        self._set_property("keep_alive_interval", value)

    @property
    def update_interval(self):
        # type: () -> int
        """update_interval getter

        The time interval at which Update messages are sent to the DUT, expressed as the number of milliseconds between Update messages. The update interval implies to send all the updates as fast as possible.

        Returns: int
        """
        return self._get_property("update_interval")

    @update_interval.setter
    def update_interval(self, value):
        """update_interval setter

        The time interval at which Update messages are sent to the DUT, expressed as the number of milliseconds between Update messages. The update interval implies to send all the updates as fast as possible.

        value: int
        """
        self._set_property("update_interval", value)

    @property
    def time_to_live(self):
        # type: () -> int
        """time_to_live getter

        The limited number of iterations that unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        Returns: int
        """
        return self._get_property("time_to_live")

    @time_to_live.setter
    def time_to_live(self, value):
        """time_to_live setter

        The limited number of iterations that unit of data can experience before the data is discarded. This is placed in the TTL field in the IP header of the transmitted packets.

        value: int
        """
        self._set_property("time_to_live", value)

    @property
    def md5_key(self):
        # type: () -> str
        """md5_key getter

        The value to be used as secret MD5 key for authentication. If not configured, MD5 authentication will not be enabled.

        Returns: str
        """
        return self._get_property("md5_key")

    @md5_key.setter
    def md5_key(self, value):
        """md5_key setter

        The value to be used as secret MD5 key for authentication. If not configured, MD5 authentication will not be enabled.

        value: str
        """
        self._set_property("md5_key", value)

    @property
    def passive_mode(self):
        # type: () -> bool
        """passive_mode getter

        If set to true, the local BGP peer will wait for the remote peer to initiate the BGP session. by establishing the TCP connection, rather than initiating sessions from the local peer.

        Returns: bool
        """
        return self._get_property("passive_mode")

    @passive_mode.setter
    def passive_mode(self, value):
        """passive_mode setter

        If set to true, the local BGP peer will wait for the remote peer to initiate the BGP session. by establishing the TCP connection, rather than initiating sessions from the local peer.

        value: bool
        """
        self._set_property("passive_mode", value)

    @property
    def listen_port(self):
        # type: () -> int
        """listen_port getter

        The TCP port number on which to accept BGP connections from the remote peer.

        Returns: int
        """
        return self._get_property("listen_port")

    @listen_port.setter
    def listen_port(self, value):
        """listen_port setter

        The TCP port number on which to accept BGP connections from the remote peer.

        value: int
        """
        self._set_property("listen_port", value)

    @property
    def neighbor_port(self):
        # type: () -> int
        """neighbor_port getter

        Destination TCP port number of the BGP peer when initiating a. session from the local BGP peer.

        Returns: int
        """
        return self._get_property("neighbor_port")

    @neighbor_port.setter
    def neighbor_port(self, value):
        """neighbor_port setter

        Destination TCP port number of the BGP peer when initiating a. session from the local BGP peer.

        value: int
        """
        self._set_property("neighbor_port", value)


class BgpCapability(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "ipv4_unicast": {"type": bool},
        "ipv4_multicast": {"type": bool},
        "ipv6_unicast": {"type": bool},
        "ipv6_multicast": {"type": bool},
        "vpls": {"type": bool},
        "route_refresh": {"type": bool},
        "route_constraint": {"type": bool},
        "link_state_non_vpn": {"type": bool},
        "link_state_vpn": {"type": bool},
        "evpn": {"type": bool},
        "extended_next_hop_encoding": {"type": bool},
        "ipv4_multicast_vpn": {"type": bool},
        "ipv4_mpls_vpn": {"type": bool},
        "ipv4_mdt": {"type": bool},
        "ipv4_multicast_mpls_vpn": {"type": bool},
        "ipv4_unicast_flow_spec": {"type": bool},
        "ipv4_sr_te_policy": {"type": bool},
        "ipv4_unicast_add_path": {"type": bool},
        "ipv6_multicast_vpn": {"type": bool},
        "ipv6_mpls_vpn": {"type": bool},
        "ipv6_mdt": {"type": bool},
        "ipv6_multicast_mpls_vpn": {"type": bool},
        "ipv6_unicast_flow_spec": {"type": bool},
        "ipv6_sr_te_policy": {"type": bool},
        "ipv6_unicast_add_path": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ipv4_unicast": True,
        "ipv4_multicast": False,
        "ipv6_unicast": True,
        "ipv6_multicast": False,
        "vpls": False,
        "route_refresh": True,
        "route_constraint": False,
        "link_state_non_vpn": False,
        "link_state_vpn": False,
        "evpn": False,
        "extended_next_hop_encoding": False,
        "ipv4_multicast_vpn": False,
        "ipv4_mpls_vpn": False,
        "ipv4_mdt": False,
        "ipv4_multicast_mpls_vpn": False,
        "ipv4_unicast_flow_spec": False,
        "ipv4_sr_te_policy": False,
        "ipv4_unicast_add_path": False,
        "ipv6_multicast_vpn": False,
        "ipv6_mpls_vpn": False,
        "ipv6_mdt": False,
        "ipv6_multicast_mpls_vpn": False,
        "ipv6_unicast_flow_spec": False,
        "ipv6_sr_te_policy": False,
        "ipv6_unicast_add_path": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        ipv4_unicast=True,
        ipv4_multicast=False,
        ipv6_unicast=True,
        ipv6_multicast=False,
        vpls=False,
        route_refresh=True,
        route_constraint=False,
        link_state_non_vpn=False,
        link_state_vpn=False,
        evpn=False,
        extended_next_hop_encoding=False,
        ipv4_multicast_vpn=False,
        ipv4_mpls_vpn=False,
        ipv4_mdt=False,
        ipv4_multicast_mpls_vpn=False,
        ipv4_unicast_flow_spec=False,
        ipv4_sr_te_policy=False,
        ipv4_unicast_add_path=False,
        ipv6_multicast_vpn=False,
        ipv6_mpls_vpn=False,
        ipv6_mdt=False,
        ipv6_multicast_mpls_vpn=False,
        ipv6_unicast_flow_spec=False,
        ipv6_sr_te_policy=False,
        ipv6_unicast_add_path=False,
    ):
        super(BgpCapability, self).__init__()
        self._parent = parent
        self._set_property("ipv4_unicast", ipv4_unicast)
        self._set_property("ipv4_multicast", ipv4_multicast)
        self._set_property("ipv6_unicast", ipv6_unicast)
        self._set_property("ipv6_multicast", ipv6_multicast)
        self._set_property("vpls", vpls)
        self._set_property("route_refresh", route_refresh)
        self._set_property("route_constraint", route_constraint)
        self._set_property("link_state_non_vpn", link_state_non_vpn)
        self._set_property("link_state_vpn", link_state_vpn)
        self._set_property("evpn", evpn)
        self._set_property("extended_next_hop_encoding", extended_next_hop_encoding)
        self._set_property("ipv4_multicast_vpn", ipv4_multicast_vpn)
        self._set_property("ipv4_mpls_vpn", ipv4_mpls_vpn)
        self._set_property("ipv4_mdt", ipv4_mdt)
        self._set_property("ipv4_multicast_mpls_vpn", ipv4_multicast_mpls_vpn)
        self._set_property("ipv4_unicast_flow_spec", ipv4_unicast_flow_spec)
        self._set_property("ipv4_sr_te_policy", ipv4_sr_te_policy)
        self._set_property("ipv4_unicast_add_path", ipv4_unicast_add_path)
        self._set_property("ipv6_multicast_vpn", ipv6_multicast_vpn)
        self._set_property("ipv6_mpls_vpn", ipv6_mpls_vpn)
        self._set_property("ipv6_mdt", ipv6_mdt)
        self._set_property("ipv6_multicast_mpls_vpn", ipv6_multicast_mpls_vpn)
        self._set_property("ipv6_unicast_flow_spec", ipv6_unicast_flow_spec)
        self._set_property("ipv6_sr_te_policy", ipv6_sr_te_policy)
        self._set_property("ipv6_unicast_add_path", ipv6_unicast_add_path)

    def set(
        self,
        ipv4_unicast=None,
        ipv4_multicast=None,
        ipv6_unicast=None,
        ipv6_multicast=None,
        vpls=None,
        route_refresh=None,
        route_constraint=None,
        link_state_non_vpn=None,
        link_state_vpn=None,
        evpn=None,
        extended_next_hop_encoding=None,
        ipv4_multicast_vpn=None,
        ipv4_mpls_vpn=None,
        ipv4_mdt=None,
        ipv4_multicast_mpls_vpn=None,
        ipv4_unicast_flow_spec=None,
        ipv4_sr_te_policy=None,
        ipv4_unicast_add_path=None,
        ipv6_multicast_vpn=None,
        ipv6_mpls_vpn=None,
        ipv6_mdt=None,
        ipv6_multicast_mpls_vpn=None,
        ipv6_unicast_flow_spec=None,
        ipv6_sr_te_policy=None,
        ipv6_unicast_add_path=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ipv4_unicast(self):
        # type: () -> bool
        """ipv4_unicast getter

        Support for the IPv4 Unicast address family.

        Returns: bool
        """
        return self._get_property("ipv4_unicast")

    @ipv4_unicast.setter
    def ipv4_unicast(self, value):
        """ipv4_unicast setter

        Support for the IPv4 Unicast address family.

        value: bool
        """
        self._set_property("ipv4_unicast", value)

    @property
    def ipv4_multicast(self):
        # type: () -> bool
        """ipv4_multicast getter

        Support for the IPv4 Multicast address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast")

    @ipv4_multicast.setter
    def ipv4_multicast(self, value):
        """ipv4_multicast setter

        Support for the IPv4 Multicast address family.

        value: bool
        """
        self._set_property("ipv4_multicast", value)

    @property
    def ipv6_unicast(self):
        # type: () -> bool
        """ipv6_unicast getter

        Support for the IPv4 Unicast address family.

        Returns: bool
        """
        return self._get_property("ipv6_unicast")

    @ipv6_unicast.setter
    def ipv6_unicast(self, value):
        """ipv6_unicast setter

        Support for the IPv4 Unicast address family.

        value: bool
        """
        self._set_property("ipv6_unicast", value)

    @property
    def ipv6_multicast(self):
        # type: () -> bool
        """ipv6_multicast getter

        Support for the IPv6 Multicast address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast")

    @ipv6_multicast.setter
    def ipv6_multicast(self, value):
        """ipv6_multicast setter

        Support for the IPv6 Multicast address family.

        value: bool
        """
        self._set_property("ipv6_multicast", value)

    @property
    def vpls(self):
        # type: () -> bool
        """vpls getter

        Support for VPLS as below. RFC4761 Virtual Private LAN Service (VPLS) using BGP for Auto-Discovery. and Signaling. RFC6624 Layer Virtual Private Networks using BGP for Auto-Discovery and Signaling.

        Returns: bool
        """
        return self._get_property("vpls")

    @vpls.setter
    def vpls(self, value):
        """vpls setter

        Support for VPLS as below. RFC4761 Virtual Private LAN Service (VPLS) using BGP for Auto-Discovery. and Signaling. RFC6624 Layer Virtual Private Networks using BGP for Auto-Discovery and Signaling.

        value: bool
        """
        self._set_property("vpls", value)

    @property
    def route_refresh(self):
        # type: () -> bool
        """route_refresh getter

        Support for the route refresh capabilities. Route Refresh allows the dynamic exchange of route refresh requests and routing information between BGP peers and the subsequent re-advertisement of the outbound or inbound routing table.

        Returns: bool
        """
        return self._get_property("route_refresh")

    @route_refresh.setter
    def route_refresh(self, value):
        """route_refresh setter

        Support for the route refresh capabilities. Route Refresh allows the dynamic exchange of route refresh requests and routing information between BGP peers and the subsequent re-advertisement of the outbound or inbound routing table.

        value: bool
        """
        self._set_property("route_refresh", value)

    @property
    def route_constraint(self):
        # type: () -> bool
        """route_constraint getter

        Supports for the route constraint capabilities. Route Constraint allows the advertisement of Route Target Membership information. The BGP peers exchange Route Target Reachability Information, which is used to build route distribution graph. This limits the propagation of VPN Network Layer Reachability Information (NLRI) between different autonomous systems or distinct clusters of the same autonomous system. This is supported for Layer Virtual Private Network scenario.

        Returns: bool
        """
        return self._get_property("route_constraint")

    @route_constraint.setter
    def route_constraint(self, value):
        """route_constraint setter

        Supports for the route constraint capabilities. Route Constraint allows the advertisement of Route Target Membership information. The BGP peers exchange Route Target Reachability Information, which is used to build route distribution graph. This limits the propagation of VPN Network Layer Reachability Information (NLRI) between different autonomous systems or distinct clusters of the same autonomous system. This is supported for Layer Virtual Private Network scenario.

        value: bool
        """
        self._set_property("route_constraint", value)

    @property
    def link_state_non_vpn(self):
        # type: () -> bool
        """link_state_non_vpn getter

        Support for BGP Link State for ISIS and OSPF.

        Returns: bool
        """
        return self._get_property("link_state_non_vpn")

    @link_state_non_vpn.setter
    def link_state_non_vpn(self, value):
        """link_state_non_vpn setter

        Support for BGP Link State for ISIS and OSPF.

        value: bool
        """
        self._set_property("link_state_non_vpn", value)

    @property
    def link_state_vpn(self):
        # type: () -> bool
        """link_state_vpn getter

        Capability advertisement of BGP Link State for VPNs.

        Returns: bool
        """
        return self._get_property("link_state_vpn")

    @link_state_vpn.setter
    def link_state_vpn(self, value):
        """link_state_vpn setter

        Capability advertisement of BGP Link State for VPNs.

        value: bool
        """
        self._set_property("link_state_vpn", value)

    @property
    def evpn(self):
        # type: () -> bool
        """evpn getter

        Support for the EVPN address family.

        Returns: bool
        """
        return self._get_property("evpn")

    @evpn.setter
    def evpn(self, value):
        """evpn setter

        Support for the EVPN address family.

        value: bool
        """
        self._set_property("evpn", value)

    @property
    def extended_next_hop_encoding(self):
        # type: () -> bool
        """extended_next_hop_encoding getter

        Support for extended Next Hop Encoding for Nexthop field in IPv4 routes advertisement. This allows IPv4 routes being advertised by IPv6 peers to include an IPv6 Nexthop.

        Returns: bool
        """
        return self._get_property("extended_next_hop_encoding")

    @extended_next_hop_encoding.setter
    def extended_next_hop_encoding(self, value):
        """extended_next_hop_encoding setter

        Support for extended Next Hop Encoding for Nexthop field in IPv4 routes advertisement. This allows IPv4 routes being advertised by IPv6 peers to include an IPv6 Nexthop.

        value: bool
        """
        self._set_property("extended_next_hop_encoding", value)

    @property
    def ipv4_multicast_vpn(self):
        # type: () -> bool
        """ipv4_multicast_vpn getter

        Support for the IPv4 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast_vpn")

    @ipv4_multicast_vpn.setter
    def ipv4_multicast_vpn(self, value):
        """ipv4_multicast_vpn setter

        Support for the IPv4 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv4_multicast_vpn", value)

    @property
    def ipv4_mpls_vpn(self):
        # type: () -> bool
        """ipv4_mpls_vpn getter

        Support for the IPv4 MPLS L3VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_mpls_vpn")

    @ipv4_mpls_vpn.setter
    def ipv4_mpls_vpn(self, value):
        """ipv4_mpls_vpn setter

        Support for the IPv4 MPLS L3VPN address family.

        value: bool
        """
        self._set_property("ipv4_mpls_vpn", value)

    @property
    def ipv4_mdt(self):
        # type: () -> bool
        """ipv4_mdt getter

        Supports for IPv4 MDT address family messages.

        Returns: bool
        """
        return self._get_property("ipv4_mdt")

    @ipv4_mdt.setter
    def ipv4_mdt(self, value):
        """ipv4_mdt setter

        Supports for IPv4 MDT address family messages.

        value: bool
        """
        self._set_property("ipv4_mdt", value)

    @property
    def ipv4_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv4_multicast_mpls_vpn getter

        Support for the IPv4 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv4_multicast_mpls_vpn")

    @ipv4_multicast_mpls_vpn.setter
    def ipv4_multicast_mpls_vpn(self, value):
        """ipv4_multicast_mpls_vpn setter

        Support for the IPv4 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv4_multicast_mpls_vpn", value)

    @property
    def ipv4_unicast_flow_spec(self):
        # type: () -> bool
        """ipv4_unicast_flow_spec getter

        Support for propagation of IPv4 unicast flow specification rules.

        Returns: bool
        """
        return self._get_property("ipv4_unicast_flow_spec")

    @ipv4_unicast_flow_spec.setter
    def ipv4_unicast_flow_spec(self, value):
        """ipv4_unicast_flow_spec setter

        Support for propagation of IPv4 unicast flow specification rules.

        value: bool
        """
        self._set_property("ipv4_unicast_flow_spec", value)

    @property
    def ipv4_sr_te_policy(self):
        # type: () -> bool
        """ipv4_sr_te_policy getter

        Support for IPv4 SRTE policy.

        Returns: bool
        """
        return self._get_property("ipv4_sr_te_policy")

    @ipv4_sr_te_policy.setter
    def ipv4_sr_te_policy(self, value):
        """ipv4_sr_te_policy setter

        Support for IPv4 SRTE policy.

        value: bool
        """
        self._set_property("ipv4_sr_te_policy", value)

    @property
    def ipv4_unicast_add_path(self):
        # type: () -> bool
        """ipv4_unicast_add_path getter

        Support for IPv4 Unicast Add Path Capability.

        Returns: bool
        """
        return self._get_property("ipv4_unicast_add_path")

    @ipv4_unicast_add_path.setter
    def ipv4_unicast_add_path(self, value):
        """ipv4_unicast_add_path setter

        Support for IPv4 Unicast Add Path Capability.

        value: bool
        """
        self._set_property("ipv4_unicast_add_path", value)

    @property
    def ipv6_multicast_vpn(self):
        # type: () -> bool
        """ipv6_multicast_vpn getter

        Support for the IPv6 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast_vpn")

    @ipv6_multicast_vpn.setter
    def ipv6_multicast_vpn(self, value):
        """ipv6_multicast_vpn setter

        Support for the IPv6 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv6_multicast_vpn", value)

    @property
    def ipv6_mpls_vpn(self):
        # type: () -> bool
        """ipv6_mpls_vpn getter

        Support for the IPv6 MPLS L3VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_mpls_vpn")

    @ipv6_mpls_vpn.setter
    def ipv6_mpls_vpn(self, value):
        """ipv6_mpls_vpn setter

        Support for the IPv6 MPLS L3VPN address family.

        value: bool
        """
        self._set_property("ipv6_mpls_vpn", value)

    @property
    def ipv6_mdt(self):
        # type: () -> bool
        """ipv6_mdt getter

        Support for IPv6 MDT address family messages.

        Returns: bool
        """
        return self._get_property("ipv6_mdt")

    @ipv6_mdt.setter
    def ipv6_mdt(self, value):
        """ipv6_mdt setter

        Support for IPv6 MDT address family messages.

        value: bool
        """
        self._set_property("ipv6_mdt", value)

    @property
    def ipv6_multicast_mpls_vpn(self):
        # type: () -> bool
        """ipv6_multicast_mpls_vpn getter

        Support for the IPv6 Multicast VPN address family.

        Returns: bool
        """
        return self._get_property("ipv6_multicast_mpls_vpn")

    @ipv6_multicast_mpls_vpn.setter
    def ipv6_multicast_mpls_vpn(self, value):
        """ipv6_multicast_mpls_vpn setter

        Support for the IPv6 Multicast VPN address family.

        value: bool
        """
        self._set_property("ipv6_multicast_mpls_vpn", value)

    @property
    def ipv6_unicast_flow_spec(self):
        # type: () -> bool
        """ipv6_unicast_flow_spec getter

        Support for propagation of IPv6 unicast flow specification rules.

        Returns: bool
        """
        return self._get_property("ipv6_unicast_flow_spec")

    @ipv6_unicast_flow_spec.setter
    def ipv6_unicast_flow_spec(self, value):
        """ipv6_unicast_flow_spec setter

        Support for propagation of IPv6 unicast flow specification rules.

        value: bool
        """
        self._set_property("ipv6_unicast_flow_spec", value)

    @property
    def ipv6_sr_te_policy(self):
        # type: () -> bool
        """ipv6_sr_te_policy getter

        Support for IPv6 SRTE policy.

        Returns: bool
        """
        return self._get_property("ipv6_sr_te_policy")

    @ipv6_sr_te_policy.setter
    def ipv6_sr_te_policy(self, value):
        """ipv6_sr_te_policy setter

        Support for IPv6 SRTE policy.

        value: bool
        """
        self._set_property("ipv6_sr_te_policy", value)

    @property
    def ipv6_unicast_add_path(self):
        # type: () -> bool
        """ipv6_unicast_add_path getter

        Support for IPv6 Unicast Add Path Capability.

        Returns: bool
        """
        return self._get_property("ipv6_unicast_add_path")

    @ipv6_unicast_add_path.setter
    def ipv6_unicast_add_path(self, value):
        """ipv6_unicast_add_path setter

        Support for IPv6 Unicast Add Path Capability.

        value: bool
        """
        self._set_property("ipv6_unicast_add_path", value)


class BgpLearnedInformationFilter(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "unicast_ipv4_prefix": {"type": bool},
        "unicast_ipv6_prefix": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "unicast_ipv4_prefix": False,
        "unicast_ipv6_prefix": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, unicast_ipv4_prefix=False, unicast_ipv6_prefix=False
    ):
        super(BgpLearnedInformationFilter, self).__init__()
        self._parent = parent
        self._set_property("unicast_ipv4_prefix", unicast_ipv4_prefix)
        self._set_property("unicast_ipv6_prefix", unicast_ipv6_prefix)

    def set(self, unicast_ipv4_prefix=None, unicast_ipv6_prefix=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def unicast_ipv4_prefix(self):
        # type: () -> bool
        """unicast_ipv4_prefix getter

        If enabled, will store the information related to Unicast IPv4 Prefixes recieved from the peer.

        Returns: bool
        """
        return self._get_property("unicast_ipv4_prefix")

    @unicast_ipv4_prefix.setter
    def unicast_ipv4_prefix(self, value):
        """unicast_ipv4_prefix setter

        If enabled, will store the information related to Unicast IPv4 Prefixes recieved from the peer.

        value: bool
        """
        self._set_property("unicast_ipv4_prefix", value)

    @property
    def unicast_ipv6_prefix(self):
        # type: () -> bool
        """unicast_ipv6_prefix getter

        If enabled, will store the information related to Unicast IPv6 Prefixes recieved from the peer.

        Returns: bool
        """
        return self._get_property("unicast_ipv6_prefix")

    @unicast_ipv6_prefix.setter
    def unicast_ipv6_prefix(self, value):
        """unicast_ipv6_prefix setter

        If enabled, will store the information related to Unicast IPv6 Prefixes recieved from the peer.

        value: bool
        """
        self._set_property("unicast_ipv6_prefix", value)


class BgpV4RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V4RouteAddressIter"},
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "add_path": {"type": "BgpAddPath"},
        "name": {"type": str},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "extended_communities": {"type": "BgpExtendedCommunityIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv4",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {
        "ext_communities": "ext_communities property in schema BgpV4RouteRange is deprecated, This property is deprecated in favor of property extended_communities",
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        super(BgpV4RouteRange, self).__init__()
        self._parent = parent
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)

    def set(
        self,
        next_hop_mode=None,
        next_hop_address_type=None,
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def addresses(self):
        # type: () -> V4RouteAddressIter
        """addresses getter

        A list of group of IPv4 route addresses.

        Returns: V4RouteAddressIter
        """
        return self._get_property(
            "addresses", V4RouteAddressIter, self._parent, self._choice
        )

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP. peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP. peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Deprecated: This property is deprecated in favor of property extended_communities. Deprecated: This property is deprecated in favor of property extended_communities. Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address. Note evpn type is defined mainly for use with evpn route updates and not for IPv4 and IPv6 route updates.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def extended_communities(self):
        # type: () -> BgpExtendedCommunityIter
        """extended_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an eight byte value. It is divided into two main parts. The first two bytes of the community encode type and sub-type fields and the last six bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtendedCommunityIter
        """
        return self._get_property(
            "extended_communities", BgpExtendedCommunityIter, self._parent, self._choice
        )


class BgpAddPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "path_id": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "path_id": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, path_id=1):
        super(BgpAddPath, self).__init__()
        self._parent = parent
        self._set_property("path_id", path_id)

    def set(self, path_id=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def path_id(self):
        # type: () -> int
        """path_id getter

        The id of the additional path.

        Returns: int
        """
        return self._get_property("path_id")

    @path_id.setter
    def path_id(self, value):
        """path_id setter

        The id of the additional path.

        value: int
        """
        self._set_property("path_id", value)


class BgpExtendedCommunity(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "transitive_2octet_as_type",
                "transitive_ipv4_address_type",
                "transitive_4octet_as_type",
                "transitive_opaque_type",
                "transitive_evpn_type",
                "non_transitive_2octet_as_type",
                "custom",
            ],
        },
        "transitive_2octet_as_type": {
            "type": "BgpExtendedCommunityTransitive2OctetAsType"
        },
        "transitive_ipv4_address_type": {
            "type": "BgpExtendedCommunityTransitiveIpv4AddressType"
        },
        "transitive_4octet_as_type": {
            "type": "BgpExtendedCommunityTransitive4OctetAsType"
        },
        "transitive_opaque_type": {"type": "BgpExtendedCommunityTransitiveOpaqueType"},
        "transitive_evpn_type": {"type": "BgpExtendedCommunityTransitiveEvpnType"},
        "non_transitive_2octet_as_type": {
            "type": "BgpExtendedCommunityNonTransitive2OctetAsType"
        },
        "custom": {"type": "BgpExtendedCommunityCustomType"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "transitive_2octet_as_type",
    }  # type: Dict[str, Union(type)]

    TRANSITIVE_2OCTET_AS_TYPE = "transitive_2octet_as_type"  # type: str
    TRANSITIVE_IPV4_ADDRESS_TYPE = "transitive_ipv4_address_type"  # type: str
    TRANSITIVE_4OCTET_AS_TYPE = "transitive_4octet_as_type"  # type: str
    TRANSITIVE_OPAQUE_TYPE = "transitive_opaque_type"  # type: str
    TRANSITIVE_EVPN_TYPE = "transitive_evpn_type"  # type: str
    NON_TRANSITIVE_2OCTET_AS_TYPE = "non_transitive_2octet_as_type"  # type: str
    CUSTOM = "custom"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunity, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def transitive_2octet_as_type(self):
        # type: () -> BgpExtendedCommunityTransitive2OctetAsType
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive2OctetAsType class

        The Transitive Two-Octet AS-Specific Extended Community is sent as type 0x00

        Returns: BgpExtendedCommunityTransitive2OctetAsType
        """
        return self._get_property(
            "transitive_2octet_as_type",
            BgpExtendedCommunityTransitive2OctetAsType,
            self,
            "transitive_2octet_as_type",
        )

    @property
    def transitive_ipv4_address_type(self):
        # type: () -> BgpExtendedCommunityTransitiveIpv4AddressType
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveIpv4AddressType class

        The Transitive IPv4 Address Specific Extended Community is sent as type 0x01.

        Returns: BgpExtendedCommunityTransitiveIpv4AddressType
        """
        return self._get_property(
            "transitive_ipv4_address_type",
            BgpExtendedCommunityTransitiveIpv4AddressType,
            self,
            "transitive_ipv4_address_type",
        )

    @property
    def transitive_4octet_as_type(self):
        # type: () -> BgpExtendedCommunityTransitive4OctetAsType
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive4OctetAsType class

        The Transitive Four-Octet AS-Specific Extended Community is sent as type 0x02. It is defined in RFC 5668.

        Returns: BgpExtendedCommunityTransitive4OctetAsType
        """
        return self._get_property(
            "transitive_4octet_as_type",
            BgpExtendedCommunityTransitive4OctetAsType,
            self,
            "transitive_4octet_as_type",
        )

    @property
    def transitive_opaque_type(self):
        # type: () -> BgpExtendedCommunityTransitiveOpaqueType
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveOpaqueType class

        The Transitive Opaque Extended Community is sent as type 0x03.

        Returns: BgpExtendedCommunityTransitiveOpaqueType
        """
        return self._get_property(
            "transitive_opaque_type",
            BgpExtendedCommunityTransitiveOpaqueType,
            self,
            "transitive_opaque_type",
        )

    @property
    def transitive_evpn_type(self):
        # type: () -> BgpExtendedCommunityTransitiveEvpnType
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveEvpnType class

        The Transitive EVPN Extended Community is sent as type 0x06

        Returns: BgpExtendedCommunityTransitiveEvpnType
        """
        return self._get_property(
            "transitive_evpn_type",
            BgpExtendedCommunityTransitiveEvpnType,
            self,
            "transitive_evpn_type",
        )

    @property
    def non_transitive_2octet_as_type(self):
        # type: () -> BgpExtendedCommunityNonTransitive2OctetAsType
        """Factory property that returns an instance of the BgpExtendedCommunityNonTransitive2OctetAsType class

        The Non-Transitive Two-Octet AS-Specific Extended Community is sent as type 0x40.

        Returns: BgpExtendedCommunityNonTransitive2OctetAsType
        """
        return self._get_property(
            "non_transitive_2octet_as_type",
            BgpExtendedCommunityNonTransitive2OctetAsType,
            self,
            "non_transitive_2octet_as_type",
        )

    @property
    def custom(self):
        # type: () -> BgpExtendedCommunityCustomType
        """Factory property that returns an instance of the BgpExtendedCommunityCustomType class

        Add custom Extended Community with combination of types sub-types and values not explicitly specified above or not defined yet.

        Returns: BgpExtendedCommunityCustomType
        """
        return self._get_property(
            "custom", BgpExtendedCommunityCustomType, self, "custom"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["custom"], Literal["non_transitive_2octet_as_type"], Literal["transitive_2octet_as_type"], Literal["transitive_4octet_as_type"], Literal["transitive_evpn_type"], Literal["transitive_ipv4_address_type"], Literal["transitive_opaque_type"]]
        """choice getter

        TBD

        Returns: Union[Literal["custom"], Literal["non_transitive_2octet_as_type"], Literal["transitive_2octet_as_type"], Literal["transitive_4octet_as_type"], Literal["transitive_evpn_type"], Literal["transitive_ipv4_address_type"], Literal["transitive_opaque_type"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["custom"], Literal["non_transitive_2octet_as_type"], Literal["transitive_2octet_as_type"], Literal["transitive_4octet_as_type"], Literal["transitive_evpn_type"], Literal["transitive_ipv4_address_type"], Literal["transitive_opaque_type"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitive2OctetAsType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "route_target_subtype",
                "route_origin_subtype",
            ],
        },
        "route_target_subtype": {
            "type": "BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget"
        },
        "route_origin_subtype": {
            "type": "BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "route_target_subtype",
    }  # type: Dict[str, Union(type)]

    ROUTE_TARGET_SUBTYPE = "route_target_subtype"  # type: str
    ROUTE_ORIGIN_SUBTYPE = "route_origin_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityTransitive2OctetAsType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def route_target_subtype(self):
        # type: () -> BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget class

        The Route Target Community identifies one or more routers that may receive set of routes (that carry this Community) carried by BGP. It is sent with sub-type as 0x02.

        Returns: BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget
        """
        return self._get_property(
            "route_target_subtype",
            BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget,
            self,
            "route_target_subtype",
        )

    @property
    def route_origin_subtype(self):
        # type: () -> BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin class

        The Route Origin Community identifies one or more routers that inject set of routes (that carry this Community) into BGP. It is sent with sub-type as 0x03 .

        Returns: BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin
        """
        return self._get_property(
            "route_origin_subtype",
            BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin,
            self,
            "route_origin_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_2byte_as": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "local_4byte_admin": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_2byte_as": 100,
        "local_4byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_2byte_as=100, local_4byte_admin=1):
        super(BgpExtendedCommunityTransitive2OctetAsTypeRouteTarget, self).__init__()
        self._parent = parent
        self._set_property("global_2byte_as", global_2byte_as)
        self._set_property("local_4byte_admin", local_4byte_admin)

    def set(self, global_2byte_as=None, local_4byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_2byte_as(self):
        # type: () -> int
        """global_2byte_as getter

        The two octet IANA assigned AS value assigned to the Autonomous System.

        Returns: int
        """
        return self._get_property("global_2byte_as")

    @global_2byte_as.setter
    def global_2byte_as(self, value):
        """global_2byte_as setter

        The two octet IANA assigned AS value assigned to the Autonomous System.

        value: int
        """
        self._set_property("global_2byte_as", value)

    @property
    def local_4byte_admin(self):
        # type: () -> int
        """local_4byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_4byte_admin")

    @local_4byte_admin.setter
    def local_4byte_admin(self, value):
        """local_4byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_4byte_admin", value)


class BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_2byte_as": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "local_4byte_admin": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_2byte_as": 100,
        "local_4byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_2byte_as=100, local_4byte_admin=1):
        super(BgpExtendedCommunityTransitive2OctetAsTypeRouteOrigin, self).__init__()
        self._parent = parent
        self._set_property("global_2byte_as", global_2byte_as)
        self._set_property("local_4byte_admin", local_4byte_admin)

    def set(self, global_2byte_as=None, local_4byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_2byte_as(self):
        # type: () -> int
        """global_2byte_as getter

        The two octet IANA assigned AS value assigned to the Autonomous System.

        Returns: int
        """
        return self._get_property("global_2byte_as")

    @global_2byte_as.setter
    def global_2byte_as(self, value):
        """global_2byte_as setter

        The two octet IANA assigned AS value assigned to the Autonomous System.

        value: int
        """
        self._set_property("global_2byte_as", value)

    @property
    def local_4byte_admin(self):
        # type: () -> int
        """local_4byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_4byte_admin")

    @local_4byte_admin.setter
    def local_4byte_admin(self, value):
        """local_4byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_4byte_admin", value)


class BgpExtendedCommunityTransitiveIpv4AddressType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "route_target_subtype",
                "route_origin_subtype",
            ],
        },
        "route_target_subtype": {
            "type": "BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget"
        },
        "route_origin_subtype": {
            "type": "BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "route_target_subtype",
    }  # type: Dict[str, Union(type)]

    ROUTE_TARGET_SUBTYPE = "route_target_subtype"  # type: str
    ROUTE_ORIGIN_SUBTYPE = "route_origin_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityTransitiveIpv4AddressType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def route_target_subtype(self):
        # type: () -> BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget class

        The Route Target Community identifies one or more routers that may receive set of routes (that carry this Community) carried by BGP. It is sent with sub-type as 0x02.

        Returns: BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget
        """
        return self._get_property(
            "route_target_subtype",
            BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget,
            self,
            "route_target_subtype",
        )

    @property
    def route_origin_subtype(self):
        # type: () -> BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin class

        The Route Origin Community identifies one or more routers that inject set of routes (that carry this Community) into BGP It is sent with sub-type as 0x03.

        Returns: BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin
        """
        return self._get_property(
            "route_origin_subtype",
            BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin,
            self,
            "route_origin_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_ipv4_admin": {
            "type": str,
            "format": "ipv4",
        },
        "local_2byte_admin": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_ipv4_admin": "0.0.0.0",
        "local_2byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_ipv4_admin="0.0.0.0", local_2byte_admin=1):
        super(BgpExtendedCommunityTransitiveIpv4AddressTypeRouteTarget, self).__init__()
        self._parent = parent
        self._set_property("global_ipv4_admin", global_ipv4_admin)
        self._set_property("local_2byte_admin", local_2byte_admin)

    def set(self, global_ipv4_admin=None, local_2byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_ipv4_admin(self):
        # type: () -> str
        """global_ipv4_admin getter

        An IPv4 unicast address assigned by one of the Internet registries.

        Returns: str
        """
        return self._get_property("global_ipv4_admin")

    @global_ipv4_admin.setter
    def global_ipv4_admin(self, value):
        """global_ipv4_admin setter

        An IPv4 unicast address assigned by one of the Internet registries.

        value: str
        """
        self._set_property("global_ipv4_admin", value)

    @property
    def local_2byte_admin(self):
        # type: () -> int
        """local_2byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the IP address carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_2byte_admin")

    @local_2byte_admin.setter
    def local_2byte_admin(self, value):
        """local_2byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the IP address carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_2byte_admin", value)


class BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_ipv4_admin": {
            "type": str,
            "format": "ipv4",
        },
        "local_2byte_admin": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_ipv4_admin": "0.0.0.0",
        "local_2byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_ipv4_admin="0.0.0.0", local_2byte_admin=1):
        super(BgpExtendedCommunityTransitiveIpv4AddressTypeRouteOrigin, self).__init__()
        self._parent = parent
        self._set_property("global_ipv4_admin", global_ipv4_admin)
        self._set_property("local_2byte_admin", local_2byte_admin)

    def set(self, global_ipv4_admin=None, local_2byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_ipv4_admin(self):
        # type: () -> str
        """global_ipv4_admin getter

        An IPv4 unicast address assigned by one of the Internet registries.

        Returns: str
        """
        return self._get_property("global_ipv4_admin")

    @global_ipv4_admin.setter
    def global_ipv4_admin(self, value):
        """global_ipv4_admin setter

        An IPv4 unicast address assigned by one of the Internet registries.

        value: str
        """
        self._set_property("global_ipv4_admin", value)

    @property
    def local_2byte_admin(self):
        # type: () -> int
        """local_2byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the IP address carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_2byte_admin")

    @local_2byte_admin.setter
    def local_2byte_admin(self, value):
        """local_2byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the IP address carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_2byte_admin", value)


class BgpExtendedCommunityTransitive4OctetAsType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "route_target_subtype",
                "route_origin_subtype",
            ],
        },
        "route_target_subtype": {
            "type": "BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget"
        },
        "route_origin_subtype": {
            "type": "BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "route_target_subtype",
    }  # type: Dict[str, Union(type)]

    ROUTE_TARGET_SUBTYPE = "route_target_subtype"  # type: str
    ROUTE_ORIGIN_SUBTYPE = "route_origin_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityTransitive4OctetAsType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def route_target_subtype(self):
        # type: () -> BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget class

        The Route Target Community identifies one or more routers that may receive set of routes (that carry this Community) carried by BGP. It is sent with sub-type as 0x02

        Returns: BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget
        """
        return self._get_property(
            "route_target_subtype",
            BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget,
            self,
            "route_target_subtype",
        )

    @property
    def route_origin_subtype(self):
        # type: () -> BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin
        """Factory property that returns an instance of the BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin class

        The Route Origin Community identifies one or more routers that inject set of routes (that carry this Community) into BGP. It is sent with sub-type as 0x03.

        Returns: BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin
        """
        return self._get_property(
            "route_origin_subtype",
            BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin,
            self,
            "route_origin_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["route_origin_subtype"], Literal["route_target_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_4byte_as": {
            "type": int,
            "format": "uint32",
        },
        "local_2byte_admin": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_4byte_as": 100,
        "local_2byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_4byte_as=100, local_2byte_admin=1):
        super(BgpExtendedCommunityTransitive4OctetAsTypeRouteTarget, self).__init__()
        self._parent = parent
        self._set_property("global_4byte_as", global_4byte_as)
        self._set_property("local_2byte_admin", local_2byte_admin)

    def set(self, global_4byte_as=None, local_2byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_4byte_as(self):
        # type: () -> int
        """global_4byte_as getter

        The four octet IANA assigned AS value assigned to the Autonomous System.

        Returns: int
        """
        return self._get_property("global_4byte_as")

    @global_4byte_as.setter
    def global_4byte_as(self, value):
        """global_4byte_as setter

        The four octet IANA assigned AS value assigned to the Autonomous System.

        value: int
        """
        self._set_property("global_4byte_as", value)

    @property
    def local_2byte_admin(self):
        # type: () -> int
        """local_2byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_2byte_admin")

    @local_2byte_admin.setter
    def local_2byte_admin(self, value):
        """local_2byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_2byte_admin", value)


class BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_4byte_as": {
            "type": int,
            "format": "uint32",
        },
        "local_2byte_admin": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_4byte_as": 100,
        "local_2byte_admin": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_4byte_as=100, local_2byte_admin=1):
        super(BgpExtendedCommunityTransitive4OctetAsTypeRouteOrigin, self).__init__()
        self._parent = parent
        self._set_property("global_4byte_as", global_4byte_as)
        self._set_property("local_2byte_admin", local_2byte_admin)

    def set(self, global_4byte_as=None, local_2byte_admin=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_4byte_as(self):
        # type: () -> int
        """global_4byte_as getter

        The four octet IANA assigned AS value assigned to the Autonomous System.

        Returns: int
        """
        return self._get_property("global_4byte_as")

    @global_4byte_as.setter
    def global_4byte_as(self, value):
        """global_4byte_as setter

        The four octet IANA assigned AS value assigned to the Autonomous System.

        value: int
        """
        self._set_property("global_4byte_as", value)

    @property
    def local_2byte_admin(self):
        # type: () -> int
        """local_2byte_admin getter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        Returns: int
        """
        return self._get_property("local_2byte_admin")

    @local_2byte_admin.setter
    def local_2byte_admin(self, value):
        """local_2byte_admin setter

        The Local Administrator sub-field contains number from numbering space that is administered by the organization to which the Autonomous System number carried in the Global Administrator sub-field has been assigned by an appropriate authority.

        value: int
        """
        self._set_property("local_2byte_admin", value)


class BgpExtendedCommunityTransitiveOpaqueType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "color_subtype",
                "encapsulation_subtype",
            ],
        },
        "color_subtype": {"type": "BgpExtendedCommunityTransitiveOpaqueTypeColor"},
        "encapsulation_subtype": {
            "type": "BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "color_subtype",
    }  # type: Dict[str, Union(type)]

    COLOR_SUBTYPE = "color_subtype"  # type: str
    ENCAPSULATION_SUBTYPE = "encapsulation_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityTransitiveOpaqueType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def color_subtype(self):
        # type: () -> BgpExtendedCommunityTransitiveOpaqueTypeColor
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveOpaqueTypeColor class

        The Color Community contains locally administrator defined 'color' value which is used in conjunction with Encapsulation attribute to decide whether data packet can be transmitted on certain tunnel or not. It is defined in RFC9012 and sent with sub-type as 0x0b.

        Returns: BgpExtendedCommunityTransitiveOpaqueTypeColor
        """
        return self._get_property(
            "color_subtype",
            BgpExtendedCommunityTransitiveOpaqueTypeColor,
            self,
            "color_subtype",
        )

    @property
    def encapsulation_subtype(self):
        # type: () -> BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation class

        This identifies the type of tunneling technology being signalled. It is defined in RFC9012 and sent with sub-type as 0x0c.

        Returns: BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation
        """
        return self._get_property(
            "encapsulation_subtype",
            BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation,
            self,
            "encapsulation_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["color_subtype"], Literal["encapsulation_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["color_subtype"], Literal["encapsulation_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["color_subtype"], Literal["encapsulation_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitiveOpaqueTypeColor(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "color": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "flags": 0,
        "color": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=0, color=0):
        super(BgpExtendedCommunityTransitiveOpaqueTypeColor, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("color", color)

    def set(self, flags=None, color=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> int
        """flags getter

        Two octet flag values.

        Returns: int
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        Two octet flag values.

        value: int
        """
        self._set_property("flags", value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        The color value is user defined and configured locally and used to determine whether data packet can be transmitted on certain tunnel or not in conjunction with the Encapsulation attribute. It is defined in RFC9012.

        Returns: int
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        The color value is user defined and configured locally and used to determine whether data packet can be transmitted on certain tunnel or not in conjunction with the Encapsulation attribute. It is defined in RFC9012.

        value: int
        """
        self._set_property("color", value)


class BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "reserved": {
            "type": int,
            "format": "uint32",
        },
        "tunnel_type": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "reserved": 0,
        "tunnel_type": 1,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, reserved=0, tunnel_type=1):
        super(BgpExtendedCommunityTransitiveOpaqueTypeEncapsulation, self).__init__()
        self._parent = parent
        self._set_property("reserved", reserved)
        self._set_property("tunnel_type", tunnel_type)

    def set(self, reserved=None, tunnel_type=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def reserved(self):
        # type: () -> int
        """reserved getter

        Four bytes of reserved values. Normally set to on transmit and ignored on receive.

        Returns: int
        """
        return self._get_property("reserved")

    @reserved.setter
    def reserved(self, value):
        """reserved setter

        Four bytes of reserved values. Normally set to on transmit and ignored on receive.

        value: int
        """
        self._set_property("reserved", value)

    @property
    def tunnel_type(self):
        # type: () -> int
        """tunnel_type getter

        Identifies the type of tunneling technology being signalled. Initially defined in RFC5512 and extended in RFC9012. Some of the important tunnel types include L2TPv3 over IP [RFC9012], 2 GRE [RFC9012]. IP in IP [RFC9012]. VXLAN Encapsulation [RFC8365]. NVGRE Encapsulation [RFC8365]. 10 MPLS Encapsulation [RFC8365]. 15 SR TE Policy Type [draft-ietf-idr-segment-routing-te-policy]. 19 Geneve Encapsulation [RFC8926]

        Returns: int
        """
        return self._get_property("tunnel_type")

    @tunnel_type.setter
    def tunnel_type(self, value):
        """tunnel_type setter

        Identifies the type of tunneling technology being signalled. Initially defined in RFC5512 and extended in RFC9012. Some of the important tunnel types include L2TPv3 over IP [RFC9012], 2 GRE [RFC9012]. IP in IP [RFC9012]. VXLAN Encapsulation [RFC8365]. NVGRE Encapsulation [RFC8365]. 10 MPLS Encapsulation [RFC8365]. 15 SR TE Policy Type [draft-ietf-idr-segment-routing-te-policy]. 19 Geneve Encapsulation [RFC8926]

        value: int
        """
        self._set_property("tunnel_type", value)


class BgpExtendedCommunityTransitiveEvpnType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "router_mac_subtype",
            ],
        },
        "router_mac_subtype": {
            "type": "BgpExtendedCommunityTransitiveEvpnTypeRouterMac"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "router_mac_subtype",
    }  # type: Dict[str, Union(type)]

    ROUTER_MAC_SUBTYPE = "router_mac_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityTransitiveEvpnType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def router_mac_subtype(self):
        # type: () -> BgpExtendedCommunityTransitiveEvpnTypeRouterMac
        """Factory property that returns an instance of the BgpExtendedCommunityTransitiveEvpnTypeRouterMac class

        The Router MAC EVPN Community is defined in RFC9135 and normally sent only for EVPN Type-2 Routes It is sent with sub-type 0x03.

        Returns: BgpExtendedCommunityTransitiveEvpnTypeRouterMac
        """
        return self._get_property(
            "router_mac_subtype",
            BgpExtendedCommunityTransitiveEvpnTypeRouterMac,
            self,
            "router_mac_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["router_mac_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["router_mac_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["router_mac_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityTransitiveEvpnTypeRouterMac(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "router_mac": {
            "type": str,
            "format": "mac",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "router_mac": "0:0:0:0:0:0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, router_mac="0:0:0:0:0:0"):
        super(BgpExtendedCommunityTransitiveEvpnTypeRouterMac, self).__init__()
        self._parent = parent
        self._set_property("router_mac", router_mac)

    def set(self, router_mac=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def router_mac(self):
        # type: () -> str
        """router_mac getter

        MAC Address of the PE Router.

        Returns: str
        """
        return self._get_property("router_mac")

    @router_mac.setter
    def router_mac(self, value):
        """router_mac setter

        MAC Address of the PE Router.

        value: str
        """
        self._set_property("router_mac", value)


class BgpExtendedCommunityNonTransitive2OctetAsType(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "link_bandwidth_subtype",
            ],
        },
        "link_bandwidth_subtype": {
            "type": "BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "link_bandwidth_subtype",
    }  # type: Dict[str, Union(type)]

    LINK_BANDWIDTH_SUBTYPE = "link_bandwidth_subtype"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityNonTransitive2OctetAsType, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def link_bandwidth_subtype(self):
        # type: () -> BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth
        """Factory property that returns an instance of the BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth class

        The Link Bandwidth Extended Community attribute is defined in draft-ietf-idr-link-bandwidth. It is sent with sub-type as 0x04.

        Returns: BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth
        """
        return self._get_property(
            "link_bandwidth_subtype",
            BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth,
            self,
            "link_bandwidth_subtype",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["link_bandwidth_subtype"]]
        """choice getter

        TBD

        Returns: Union[Literal["link_bandwidth_subtype"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["link_bandwidth_subtype"]]
        """
        self._set_property("choice", value)


class BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "global_2byte_as": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "bandwidth": {
            "type": float,
            "format": "float",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "global_2byte_as": 100,
        "bandwidth": 0.0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, global_2byte_as=100, bandwidth=0):
        super(
            BgpExtendedCommunityNonTransitive2OctetAsTypeLinkBandwidth, self
        ).__init__()
        self._parent = parent
        self._set_property("global_2byte_as", global_2byte_as)
        self._set_property("bandwidth", bandwidth)

    def set(self, global_2byte_as=None, bandwidth=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def global_2byte_as(self):
        # type: () -> int
        """global_2byte_as getter

        The value of the Global Administrator subfield should represent the Autonomous System of the router that attaches the Link Bandwidth Community. If four octet AS numbering scheme is used, AS_TRANS (23456) should be used.

        Returns: int
        """
        return self._get_property("global_2byte_as")

    @global_2byte_as.setter
    def global_2byte_as(self, value):
        """global_2byte_as setter

        The value of the Global Administrator subfield should represent the Autonomous System of the router that attaches the Link Bandwidth Community. If four octet AS numbering scheme is used, AS_TRANS (23456) should be used.

        value: int
        """
        self._set_property("global_2byte_as", value)

    @property
    def bandwidth(self):
        # type: () -> float
        """bandwidth getter

        Bandwidth of the link in bytes per second. 1 Kbps is 1000 bytes per second and Mbps is 1000 Kbps per second )

        Returns: float
        """
        return self._get_property("bandwidth")

    @bandwidth.setter
    def bandwidth(self, value):
        """bandwidth setter

        Bandwidth of the link in bytes per second. 1 Kbps is 1000 bytes per second and Mbps is 1000 Kbps per second )

        value: float
        """
        self._set_property("bandwidth", value)


class BgpExtendedCommunityCustomType(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "community_type": {
            "type": str,
            "format": "hex",
            "maxLength": 2,
        },
        "community_subtype": {
            "type": str,
            "format": "hex",
            "maxLength": 2,
        },
        "value": {
            "type": str,
            "format": "hex",
            "maxLength": 12,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "community_type": "00",
        "community_subtype": "00",
        "value": "000000000000",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        community_type="00",
        community_subtype="00",
        value="000000000000",
    ):
        super(BgpExtendedCommunityCustomType, self).__init__()
        self._parent = parent
        self._set_property("community_type", community_type)
        self._set_property("community_subtype", community_subtype)
        self._set_property("value", value)

    def set(self, community_type=None, community_subtype=None, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def community_type(self):
        # type: () -> str
        """community_type getter

        The type to be set in the Extended Community attribute. Accepts hexadecimal input upto ff

        Returns: str
        """
        return self._get_property("community_type")

    @community_type.setter
    def community_type(self, value):
        """community_type setter

        The type to be set in the Extended Community attribute. Accepts hexadecimal input upto ff

        value: str
        """
        self._set_property("community_type", value)

    @property
    def community_subtype(self):
        # type: () -> str
        """community_subtype getter

        The sub-type to be set in the Extended Community attribute. For certain types with no sub-type this byte can also be used as part of an extended value field. Accepts hexadecimal input upto ff.

        Returns: str
        """
        return self._get_property("community_subtype")

    @community_subtype.setter
    def community_subtype(self, value):
        """community_subtype setter

        The sub-type to be set in the Extended Community attribute. For certain types with no sub-type this byte can also be used as part of an extended value field. Accepts hexadecimal input upto ff.

        value: str
        """
        self._set_property("community_subtype", value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        6 byte hex value to be carried in the last bytes of the Extended Community. Accepts hexadecimal input upto ffffffffffff.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        6 byte hex value to be carried in the last bytes of the Extended Community. Accepts hexadecimal input upto ffffffffffff.

        value: str
        """
        self._set_property("value", value)


class BgpExtendedCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(BgpExtendedCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpExtendedCommunity, BgpExtendedCommunityCustomType, BgpExtendedCommunityNonTransitive2OctetAsType, BgpExtendedCommunityTransitive2OctetAsType, BgpExtendedCommunityTransitive4OctetAsType, BgpExtendedCommunityTransitiveEvpnType, BgpExtendedCommunityTransitiveIpv4AddressType, BgpExtendedCommunityTransitiveOpaqueType]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpExtendedCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpExtendedCommunity
        return self._next()

    def next(self):
        # type: () -> BgpExtendedCommunity
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpExtendedCommunity):
            raise Exception("Item is not an instance of BgpExtendedCommunity")

    def extendedcommunity(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunity class

        The Extended Communities Attribute is optional BGP attribute,defined in RFC4360 with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value.It is divided into two main parts. The first Bytes of the community encode type and optonal sub-type field. The last bytes (or bytes for types without sub-type) carry unique set of data in format defined by the type and optional sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> BgpExtendedCommunity
        """Add method that creates and returns an instance of the BgpExtendedCommunity class

        The Extended Communities Attribute is optional BGP attribute,defined in RFC4360 with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value.It is divided into two main parts. The first Bytes of the community encode type and optonal sub-type field. The last bytes (or bytes for types without sub-type) carry unique set of data in format defined by the type and optional sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtendedCommunity
        """
        item = BgpExtendedCommunity(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def transitive_2octet_as_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityTransitive2OctetAsType class

        The Transitive Two-Octet AS-Specific Extended Community is sent as type 0x00

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.transitive_2octet_as_type
        item.choice = "transitive_2octet_as_type"
        self._add(item)
        return self

    def transitive_ipv4_address_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityTransitiveIpv4AddressType class

        The Transitive IPv4 Address Specific Extended Community is sent as type 0x01.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.transitive_ipv4_address_type
        item.choice = "transitive_ipv4_address_type"
        self._add(item)
        return self

    def transitive_4octet_as_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityTransitive4OctetAsType class

        The Transitive Four-Octet AS-Specific Extended Community is sent as type 0x02. It is defined in RFC 5668.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.transitive_4octet_as_type
        item.choice = "transitive_4octet_as_type"
        self._add(item)
        return self

    def transitive_opaque_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityTransitiveOpaqueType class

        The Transitive Opaque Extended Community is sent as type 0x03.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.transitive_opaque_type
        item.choice = "transitive_opaque_type"
        self._add(item)
        return self

    def transitive_evpn_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityTransitiveEvpnType class

        The Transitive EVPN Extended Community is sent as type 0x06

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.transitive_evpn_type
        item.choice = "transitive_evpn_type"
        self._add(item)
        return self

    def non_transitive_2octet_as_type(self):
        # type: () -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityNonTransitive2OctetAsType class

        The Non-Transitive Two-Octet AS-Specific Extended Community is sent as type 0x40.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.non_transitive_2octet_as_type
        item.choice = "non_transitive_2octet_as_type"
        self._add(item)
        return self

    def custom(self, community_type="00", community_subtype="00", value="000000000000"):
        # type: (str,str,str) -> BgpExtendedCommunityIter
        """Factory method that creates an instance of the BgpExtendedCommunityCustomType class

        Add custom Extended Community with combination of types sub-types and values not explicitly specified above or not defined yet.

        Returns: BgpExtendedCommunityIter
        """
        item = BgpExtendedCommunity()
        item.custom
        item.choice = "custom"
        self._add(item)
        return self


class BgpV4RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV4RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV4RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV4RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV4RouteRange
        return self._next()

    def next(self):
        # type: () -> BgpV4RouteRange
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpV4RouteRange):
            raise Exception("Item is not an instance of BgpV4RouteRange")

    def v4routerange(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV4RouteRangeIter
        """Factory method that creates an instance of the BgpV4RouteRange class

        Emulated BGPv4 route range.

        Returns: BgpV4RouteRangeIter
        """
        item = BgpV4RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV4RouteRange
        """Add method that creates and returns an instance of the BgpV4RouteRange class

        Emulated BGPv4 route range.

        Returns: BgpV4RouteRange
        """
        item = BgpV4RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return item


class BgpV6RouteRange(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "addresses": {"type": "V6RouteAddressIter"},
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "communities": {"type": "BgpCommunityIter"},
        "as_path": {"type": "BgpAsPath"},
        "add_path": {"type": "BgpAddPath"},
        "name": {"type": str},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "extended_communities": {"type": "BgpExtendedCommunityIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv6",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {
        "ext_communities": "ext_communities property in schema BgpV6RouteRange is deprecated, This property is deprecated in favor of property extended_communities",
    }  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        super(BgpV6RouteRange, self).__init__()
        self._parent = parent
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)

    def set(
        self,
        next_hop_mode=None,
        next_hop_address_type=None,
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def addresses(self):
        # type: () -> V6RouteAddressIter
        """addresses getter

        A list of group of IPv6 route addresses.

        Returns: V6RouteAddressIter
        """
        return self._get_property(
            "addresses", V6RouteAddressIter, self._parent, self._choice
        )

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP. peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Specify the NextHop in MP REACH NLRI. The mode for setting the IP address of the NextHop in the MP REACH NLRI can be one of the following:. Local IP: Automatically fills the Nexthop with the Local IP of the BGP. peer.. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        If the Nexthop Mode is Manual, it sets the type of the NextHop IP address.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode is manual and the Nexthop type is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Deprecated: This property is deprecated in favor of property extended_communities. Deprecated: This property is deprecated in favor of property extended_communities. Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address. Note evpn type is defined mainly for use with evpn route updates and not for IPv4 and IPv6 route updates.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def extended_communities(self):
        # type: () -> BgpExtendedCommunityIter
        """extended_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an eight byte value. It is divided into two main parts. The first two bytes of the community encode type and sub-type fields and the last six bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtendedCommunityIter
        """
        return self._get_property(
            "extended_communities", BgpExtendedCommunityIter, self._parent, self._choice
        )


class BgpV6RouteRangeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpV6RouteRangeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpV6RouteRange]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpV6RouteRangeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpV6RouteRange
        return self._next()

    def next(self):
        # type: () -> BgpV6RouteRange
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpV6RouteRange):
            raise Exception("Item is not an instance of BgpV6RouteRange")

    def v6routerange(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV6RouteRangeIter
        """Factory method that creates an instance of the BgpV6RouteRange class

        Emulated BGPv6 route range.

        Returns: BgpV6RouteRangeIter
        """
        item = BgpV6RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return self

    def add(
        self,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
    ):
        # type: (Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str) -> BgpV6RouteRange
        """Add method that creates and returns an instance of the BgpV6RouteRange class

        Emulated BGPv6 route range.

        Returns: BgpV6RouteRange
        """
        item = BgpV6RouteRange(
            parent=self._parent,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
        )
        self._add(item)
        return item


class BgpSrteV4Policy(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "distinguisher": {
            "type": int,
            "format": "uint32",
        },
        "color": {
            "type": int,
            "format": "uint32",
        },
        "ipv4_endpoint": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "add_path": {"type": "BgpAddPath"},
        "as_path": {"type": "BgpAsPath"},
        "communities": {"type": "BgpCommunityIter"},
        "ext_communities": {"type": "BgpExtCommunityIter"},
        "tunnel_tlvs": {"type": "BgpSrteV4TunnelTlvIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_endpoint", "name")  # type: tuple(str)

    _DEFAULTS = {
        "distinguisher": 1,
        "color": 100,
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv4",
        "active": True,
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        super(BgpSrteV4Policy, self).__init__()
        self._parent = parent
        self._set_property("distinguisher", distinguisher)
        self._set_property("color", color)
        self._set_property("ipv4_endpoint", ipv4_endpoint)
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)
        self._set_property("active", active)

    def set(
        self,
        distinguisher=None,
        color=None,
        ipv4_endpoint=None,
        next_hop_mode=None,
        next_hop_address_type=None,
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def distinguisher(self):
        # type: () -> int
        """distinguisher getter

        4-octet value uniquely identifying the policy in the context of (color, endpoint) tuple. It is used by the SR Policy originator to make unique (from an NLRI perspective) both for multiple candidate paths of the same SR Policy as well as candidate paths of different SR Policies (i.e. with different segment list) with the same Color and Endpoint but meant for different head-ends.

        Returns: int
        """
        return self._get_property("distinguisher")

    @distinguisher.setter
    def distinguisher(self, value):
        """distinguisher setter

        4-octet value uniquely identifying the policy in the context of (color, endpoint) tuple. It is used by the SR Policy originator to make unique (from an NLRI perspective) both for multiple candidate paths of the same SR Policy as well as candidate paths of different SR Policies (i.e. with different segment list) with the same Color and Endpoint but meant for different head-ends.

        value: int
        """
        self._set_property("distinguisher", value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        Policy color is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        Returns: int
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Policy color is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        value: int
        """
        self._set_property("color", value)

    @property
    def ipv4_endpoint(self):
        # type: () -> str
        """ipv4_endpoint getter

        Specifies single node or set of nodes (e.g. an anycast address). It is selected on the basis of the SR Policy type (AFI).

        Returns: str
        """
        return self._get_property("ipv4_endpoint")

    @ipv4_endpoint.setter
    def ipv4_endpoint(self, value):
        """ipv4_endpoint setter

        Specifies single node or set of nodes (e.g. an anycast address). It is selected on the basis of the SR Policy type (AFI).

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv4_endpoint as None")
        self._set_property("ipv4_endpoint", value)

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the next hop if the Nexthop type 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the next hop if the Nexthop type 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional Community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def ext_communities(self):
        # type: () -> BgpExtCommunityIter
        """ext_communities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "ext_communities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def tunnel_tlvs(self):
        # type: () -> BgpSrteV4TunnelTlvIter
        """tunnel_tlvs getter

        List Tunnel Encapsulation Attributes.

        Returns: BgpSrteV4TunnelTlvIter
        """
        return self._get_property(
            "tunnel_tlvs", BgpSrteV4TunnelTlvIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV4TunnelTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_endpoint_sub_tlv": {"type": "BgpSrteRemoteEndpointSubTlv"},
        "color_sub_tlv": {"type": "BgpSrteColorSubTlv"},
        "binding_sub_tlv": {"type": "BgpSrteBindingSubTlv"},
        "preference_sub_tlv": {"type": "BgpSrtePreferenceSubTlv"},
        "policy_priority_sub_tlv": {"type": "BgpSrtePolicyPrioritySubTlv"},
        "policy_name_sub_tlv": {"type": "BgpSrtePolicyNameSubTlv"},
        "explicit_null_label_policy_sub_tlv": {
            "type": "BgpSrteExplicitNullLabelPolicySubTlv"
        },
        "segment_lists": {"type": "BgpSrteSegmentListIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None, active=True):
        super(BgpSrteV4TunnelTlv, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("active", active)

    def set(self, name=None, active=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def remote_endpoint_sub_tlv(self):
        # type: () -> BgpSrteRemoteEndpointSubTlv
        """remote_endpoint_sub_tlv getter

        Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.

        Returns: BgpSrteRemoteEndpointSubTlv
        """
        return self._get_property(
            "remote_endpoint_sub_tlv", BgpSrteRemoteEndpointSubTlv
        )

    @property
    def color_sub_tlv(self):
        # type: () -> BgpSrteColorSubTlv
        """color_sub_tlv getter

        Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure.

        Returns: BgpSrteColorSubTlv
        """
        return self._get_property("color_sub_tlv", BgpSrteColorSubTlv)

    @property
    def binding_sub_tlv(self):
        # type: () -> BgpSrteBindingSubTlv
        """binding_sub_tlv getter

        Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.

        Returns: BgpSrteBindingSubTlv
        """
        return self._get_property("binding_sub_tlv", BgpSrteBindingSubTlv)

    @property
    def preference_sub_tlv(self):
        # type: () -> BgpSrtePreferenceSubTlv
        """preference_sub_tlv getter

        Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.

        Returns: BgpSrtePreferenceSubTlv
        """
        return self._get_property("preference_sub_tlv", BgpSrtePreferenceSubTlv)

    @property
    def policy_priority_sub_tlv(self):
        # type: () -> BgpSrtePolicyPrioritySubTlv
        """policy_priority_sub_tlv getter

        Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.

        Returns: BgpSrtePolicyPrioritySubTlv
        """
        return self._get_property(
            "policy_priority_sub_tlv", BgpSrtePolicyPrioritySubTlv
        )

    @property
    def policy_name_sub_tlv(self):
        # type: () -> BgpSrtePolicyNameSubTlv
        """policy_name_sub_tlv getter

        Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.

        Returns: BgpSrtePolicyNameSubTlv
        """
        return self._get_property("policy_name_sub_tlv", BgpSrtePolicyNameSubTlv)

    @property
    def explicit_null_label_policy_sub_tlv(self):
        # type: () -> BgpSrteExplicitNullLabelPolicySubTlv
        """explicit_null_label_policy_sub_tlv getter

        Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.

        Returns: BgpSrteExplicitNullLabelPolicySubTlv
        """
        return self._get_property(
            "explicit_null_label_policy_sub_tlv", BgpSrteExplicitNullLabelPolicySubTlv
        )

    @property
    def segment_lists(self):
        # type: () -> BgpSrteSegmentListIter
        """segment_lists getter

        TBD

        Returns: BgpSrteSegmentListIter
        """
        return self._get_property(
            "segment_lists", BgpSrteSegmentListIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteRemoteEndpointSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_number": {
            "type": int,
            "format": "uint32",
        },
        "address_family": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_number": 0,
        "address_family": "ipv4",
        "ipv4_address": "0.0.0.0",
        "ipv6_address": "::0",
    }  # type: Dict[str, Union(type)]

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        as_number=0,
        address_family="ipv4",
        ipv4_address="0.0.0.0",
        ipv6_address="::0",
    ):
        super(BgpSrteRemoteEndpointSubTlv, self).__init__()
        self._parent = parent
        self._set_property("as_number", as_number)
        self._set_property("address_family", address_family)
        self._set_property("ipv4_address", ipv4_address)
        self._set_property("ipv6_address", ipv6_address)

    def set(
        self, as_number=None, address_family=None, ipv4_address=None, ipv6_address=None
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        Autonomous system (AS) number

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        Autonomous system (AS) number

        value: int
        """
        self._set_property("as_number", value)

    @property
    def address_family(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """address_family getter

        Determines the address type

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("address_family")

    @address_family.setter
    def address_family(self, value):
        """address_family setter

        Determines the address type

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("address_family", value)

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address of the Remote Endpoint.

        Returns: str
        """
        return self._get_property("ipv4_address")

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address of the Remote Endpoint.

        value: str
        """
        self._set_property("ipv4_address", value)

    @property
    def ipv6_address(self):
        # type: () -> str
        """ipv6_address getter

        The IPv6 address of the Remote Endpoint.

        Returns: str
        """
        return self._get_property("ipv6_address")

    @ipv6_address.setter
    def ipv6_address(self, value):
        """ipv6_address setter

        The IPv6 address of the Remote Endpoint.

        value: str
        """
        self._set_property("ipv6_address", value)


class BgpSrteColorSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "color": {
            "type": str,
            "format": "hex",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, color=None):
        super(BgpSrteColorSubTlv, self).__init__()
        self._parent = parent
        self._set_property("color", color)

    def set(self, color=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def color(self):
        # type: () -> str
        """color getter

        Six octet values. Example: 000000000064 for color value 100.

        Returns: str
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Six octet values. Example: 000000000064 for color value 100.

        value: str
        """
        self._set_property("color", value)


class BgpSrteBindingSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "binding_sid_type": {
            "type": str,
            "enum": [
                "no_binding",
                "four_octet_sid",
                "ipv6_sid",
            ],
        },
        "four_octet_sid": {
            "type": int,
            "format": "uint32",
        },
        "ipv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "s_flag": {"type": bool},
        "i_flag": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "binding_sid_type": "no_binding",
        "s_flag": False,
        "i_flag": False,
    }  # type: Dict[str, Union(type)]

    NO_BINDING = "no_binding"  # type: str
    FOUR_OCTET_SID = "four_octet_sid"  # type: str
    IPV6_SID = "ipv6_sid"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        binding_sid_type="no_binding",
        four_octet_sid=None,
        ipv6_sid=None,
        s_flag=False,
        i_flag=False,
    ):
        super(BgpSrteBindingSubTlv, self).__init__()
        self._parent = parent
        self._set_property("binding_sid_type", binding_sid_type)
        self._set_property("four_octet_sid", four_octet_sid)
        self._set_property("ipv6_sid", ipv6_sid)
        self._set_property("s_flag", s_flag)
        self._set_property("i_flag", i_flag)

    def set(
        self,
        binding_sid_type=None,
        four_octet_sid=None,
        ipv6_sid=None,
        s_flag=None,
        i_flag=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def binding_sid_type(self):
        # type: () -> Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """binding_sid_type getter

        Type of the binding SID. Supported types are "No Binding SID" or "Four Octets Sid" or "IPv6 SID".

        Returns: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        return self._get_property("binding_sid_type")

    @binding_sid_type.setter
    def binding_sid_type(self, value):
        """binding_sid_type setter

        Type of the binding SID. Supported types are "No Binding SID" or "Four Octets Sid" or "IPv6 SID".

        value: Union[Literal["four_octet_sid"], Literal["ipv6_sid"], Literal["no_binding"]]
        """
        self._set_property("binding_sid_type", value)

    @property
    def four_octet_sid(self):
        # type: () -> int
        """four_octet_sid getter

        Binding SID is encoded in octets.

        Returns: int
        """
        return self._get_property("four_octet_sid")

    @four_octet_sid.setter
    def four_octet_sid(self, value):
        """four_octet_sid setter

        Binding SID is encoded in octets.

        value: int
        """
        self._set_property("four_octet_sid", value)

    @property
    def ipv6_sid(self):
        # type: () -> str
        """ipv6_sid getter

        IPv6 SID value.

        Returns: str
        """
        return self._get_property("ipv6_sid")

    @ipv6_sid.setter
    def ipv6_sid(self, value):
        """ipv6_sid setter

        IPv6 SID value.

        value: str
        """
        self._set_property("ipv6_sid", value)

    @property
    def s_flag(self):
        # type: () -> bool
        """s_flag getter

        S-Flag encodes the "Specified-BSID-only" behavior.

        Returns: bool
        """
        return self._get_property("s_flag")

    @s_flag.setter
    def s_flag(self, value):
        """s_flag setter

        S-Flag encodes the "Specified-BSID-only" behavior.

        value: bool
        """
        self._set_property("s_flag", value)

    @property
    def i_flag(self):
        # type: () -> bool
        """i_flag getter

        I-Flag encodes the "Drop Upon Invalid" behavior.

        Returns: bool
        """
        return self._get_property("i_flag")

    @i_flag.setter
    def i_flag(self, value):
        """i_flag setter

        I-Flag encodes the "Drop Upon Invalid" behavior.

        value: bool
        """
        self._set_property("i_flag", value)


class BgpSrtePreferenceSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "preference": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "preference": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, preference=0):
        super(BgpSrtePreferenceSubTlv, self).__init__()
        self._parent = parent
        self._set_property("preference", preference)

    def set(self, preference=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def preference(self):
        # type: () -> int
        """preference getter

        The preference value of the SR Policy candidate path.

        Returns: int
        """
        return self._get_property("preference")

    @preference.setter
    def preference(self, value):
        """preference setter

        The preference value of the SR Policy candidate path.

        value: int
        """
        self._set_property("preference", value)


class BgpSrtePolicyPrioritySubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "policy_priority": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, policy_priority=None):
        super(BgpSrtePolicyPrioritySubTlv, self).__init__()
        self._parent = parent
        self._set_property("policy_priority", policy_priority)

    def set(self, policy_priority=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def policy_priority(self):
        # type: () -> int
        """policy_priority getter

        One-octet Priority value.

        Returns: int
        """
        return self._get_property("policy_priority")

    @policy_priority.setter
    def policy_priority(self, value):
        """policy_priority setter

        One-octet Priority value.

        value: int
        """
        self._set_property("policy_priority", value)


class BgpSrtePolicyNameSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "policy_name": {
            "type": str,
            "minLength": 1,
            "maxLength": 32,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, policy_name=None):
        super(BgpSrtePolicyNameSubTlv, self).__init__()
        self._parent = parent
        self._set_property("policy_name", policy_name)

    def set(self, policy_name=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def policy_name(self):
        # type: () -> str
        """policy_name getter

        Symbolic name for the policy that should be string of printable ASCII characters, without NULL terminator.

        Returns: str
        """
        return self._get_property("policy_name")

    @policy_name.setter
    def policy_name(self, value):
        """policy_name setter

        Symbolic name for the policy that should be string of printable ASCII characters, without NULL terminator.

        value: str
        """
        self._set_property("policy_name", value)


class BgpSrteExplicitNullLabelPolicySubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "explicit_null_label_policy": {
            "type": str,
            "enum": [
                "reserved_enlp",
                "push_ipv4_enlp",
                "push_ipv6_enlp",
                "push_ipv4_ipv6_enlp",
                "do_not_push_enlp",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "explicit_null_label_policy": "do_not_push_enlp",
    }  # type: Dict[str, Union(type)]

    RESERVED_ENLP = "reserved_enlp"  # type: str
    PUSH_IPV4_ENLP = "push_ipv4_enlp"  # type: str
    PUSH_IPV6_ENLP = "push_ipv6_enlp"  # type: str
    PUSH_IPV4_IPV6_ENLP = "push_ipv4_ipv6_enlp"  # type: str
    DO_NOT_PUSH_ENLP = "do_not_push_enlp"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, explicit_null_label_policy="do_not_push_enlp"):
        super(BgpSrteExplicitNullLabelPolicySubTlv, self).__init__()
        self._parent = parent
        self._set_property("explicit_null_label_policy", explicit_null_label_policy)

    def set(self, explicit_null_label_policy=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def explicit_null_label_policy(self):
        # type: () -> Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """explicit_null_label_policy getter

        The value of the explicit null label policy

        Returns: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        return self._get_property("explicit_null_label_policy")

    @explicit_null_label_policy.setter
    def explicit_null_label_policy(self, value):
        """explicit_null_label_policy setter

        The value of the explicit null label policy

        value: Union[Literal["do_not_push_enlp"], Literal["push_ipv4_enlp"], Literal["push_ipv4_ipv6_enlp"], Literal["push_ipv6_enlp"], Literal["reserved_enlp"]]
        """
        self._set_property("explicit_null_label_policy", value)


class BgpSrteSegmentList(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "weight": {
            "type": int,
            "format": "uint32",
        },
        "segments": {"type": "BgpSrteSegmentIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "weight": 0,
        "active": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, weight=0, name=None, active=True):
        super(BgpSrteSegmentList, self).__init__()
        self._parent = parent
        self._set_property("weight", weight)
        self._set_property("name", name)
        self._set_property("active", active)

    def set(self, weight=None, name=None, active=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def weight(self):
        # type: () -> int
        """weight getter

        The Weight associated with given path and the sub-TLV is optional.

        Returns: int
        """
        return self._get_property("weight")

    @weight.setter
    def weight(self, value):
        """weight setter

        The Weight associated with given path and the sub-TLV is optional.

        value: int
        """
        self._set_property("weight", value)

    @property
    def segments(self):
        # type: () -> BgpSrteSegmentIter
        """segments getter

        TBD

        Returns: BgpSrteSegmentIter
        """
        return self._get_property(
            "segments", BgpSrteSegmentIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "segment_type": {
            "type": str,
            "enum": [
                "type_a",
                "type_b",
                "type_c",
                "type_d",
                "type_e",
                "type_f",
                "type_g",
                "type_h",
                "type_i",
                "type_j",
                "type_k",
            ],
        },
        "type_a": {"type": "BgpSrteSegmentATypeSubTlv"},
        "type_b": {"type": "BgpSrteSegmentBTypeSubTlv"},
        "type_c": {"type": "BgpSrteSegmentCTypeSubTlv"},
        "type_d": {"type": "BgpSrteSegmentDTypeSubTlv"},
        "type_e": {"type": "BgpSrteSegmentETypeSubTlv"},
        "type_f": {"type": "BgpSrteSegmentFTypeSubTlv"},
        "type_g": {"type": "BgpSrteSegmentGTypeSubTlv"},
        "type_h": {"type": "BgpSrteSegmentHTypeSubTlv"},
        "type_i": {"type": "BgpSrteSegmentITypeSubTlv"},
        "type_j": {"type": "BgpSrteSegmentJTypeSubTlv"},
        "type_k": {"type": "BgpSrteSegmentKTypeSubTlv"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("segment_type", "name")  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    TYPE_A = "type_a"  # type: str
    TYPE_B = "type_b"  # type: str
    TYPE_C = "type_c"  # type: str
    TYPE_D = "type_d"  # type: str
    TYPE_E = "type_e"  # type: str
    TYPE_F = "type_f"  # type: str
    TYPE_G = "type_g"  # type: str
    TYPE_H = "type_h"  # type: str
    TYPE_I = "type_i"  # type: str
    TYPE_J = "type_j"  # type: str
    TYPE_K = "type_k"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, segment_type=None, name=None, active=True):
        super(BgpSrteSegment, self).__init__()
        self._parent = parent
        self._set_property("segment_type", segment_type)
        self._set_property("name", name)
        self._set_property("active", active)

    def set(self, segment_type=None, name=None, active=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def segment_type(self):
        # type: () -> Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """segment_type getter

        Specify one of the segment type.. https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Node Address with optional SID.. Type D: IPv6 Node Address with optional SID for SR MPLS.. Type E: IPv4 Address and index with optional SID.. Type F: IPv4 Local and Remote addresses with optional SID.. Type G: IPv6 Address and index for local and remote pair with optional. SID for SR MPLS.. Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.. Type I: IPv6 Node Address with optional SID for SRv6.. Type J: IPv6 Address and index for local and remote pair with optional. SID for SRv6.. Type K: IPv6 Local and Remote addresses for SRv6.

        Returns: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        return self._get_property("segment_type")

    @segment_type.setter
    def segment_type(self, value):
        """segment_type setter

        Specify one of the segment type.. https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Node Address with optional SID.. Type D: IPv6 Node Address with optional SID for SR MPLS.. Type E: IPv4 Address and index with optional SID.. Type F: IPv4 Local and Remote addresses with optional SID.. Type G: IPv6 Address and index for local and remote pair with optional. SID for SR MPLS.. Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.. Type I: IPv6 Node Address with optional SID for SRv6.. Type J: IPv6 Address and index for local and remote pair with optional. SID for SRv6.. Type K: IPv6 Local and Remote addresses for SRv6.

        value: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        if value is None:
            raise TypeError("Cannot set required property segment_type as None")
        self._set_property("segment_type", value)

    @property
    def type_a(self):
        # type: () -> BgpSrteSegmentATypeSubTlv
        """type_a getter

        Type A: SID only, in the form of MPLS Label.Type A: SID only, in the form of MPLS Label.Type A: SID only, in the form of MPLS Label.

        Returns: BgpSrteSegmentATypeSubTlv
        """
        return self._get_property("type_a", BgpSrteSegmentATypeSubTlv)

    @property
    def type_b(self):
        # type: () -> BgpSrteSegmentBTypeSubTlv
        """type_b getter

        Type B: SID only, in the form of IPv6 address.Type B: SID only, in the form of IPv6 address.Type B: SID only, in the form of IPv6 address.

        Returns: BgpSrteSegmentBTypeSubTlv
        """
        return self._get_property("type_b", BgpSrteSegmentBTypeSubTlv)

    @property
    def type_c(self):
        # type: () -> BgpSrteSegmentCTypeSubTlv
        """type_c getter

        Type C: IPv4 Node Address with optional SID.Type C: IPv4 Node Address with optional SID.Type C: IPv4 Node Address with optional SID.

        Returns: BgpSrteSegmentCTypeSubTlv
        """
        return self._get_property("type_c", BgpSrteSegmentCTypeSubTlv)

    @property
    def type_d(self):
        # type: () -> BgpSrteSegmentDTypeSubTlv
        """type_d getter

        Type D: IPv6 Node Address with optional SID for SR MPLS.Type D: IPv6 Node Address with optional SID for SR MPLS.Type D: IPv6 Node Address with optional SID for SR MPLS.

        Returns: BgpSrteSegmentDTypeSubTlv
        """
        return self._get_property("type_d", BgpSrteSegmentDTypeSubTlv)

    @property
    def type_e(self):
        # type: () -> BgpSrteSegmentETypeSubTlv
        """type_e getter

        Type E: IPv4 Address and Local Interface ID with optional SIDType E: IPv4 Address and Local Interface ID with optional SIDType E: IPv4 Address and Local Interface ID with optional SID

        Returns: BgpSrteSegmentETypeSubTlv
        """
        return self._get_property("type_e", BgpSrteSegmentETypeSubTlv)

    @property
    def type_f(self):
        # type: () -> BgpSrteSegmentFTypeSubTlv
        """type_f getter

        Type F: IPv4 Local and Remote addresses with optional SID.Type F: IPv4 Local and Remote addresses with optional SID.Type F: IPv4 Local and Remote addresses with optional SID.

        Returns: BgpSrteSegmentFTypeSubTlv
        """
        return self._get_property("type_f", BgpSrteSegmentFTypeSubTlv)

    @property
    def type_g(self):
        # type: () -> BgpSrteSegmentGTypeSubTlv
        """type_g getter

        Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.

        Returns: BgpSrteSegmentGTypeSubTlv
        """
        return self._get_property("type_g", BgpSrteSegmentGTypeSubTlv)

    @property
    def type_h(self):
        # type: () -> BgpSrteSegmentHTypeSubTlv
        """type_h getter

        Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.

        Returns: BgpSrteSegmentHTypeSubTlv
        """
        return self._get_property("type_h", BgpSrteSegmentHTypeSubTlv)

    @property
    def type_i(self):
        # type: () -> BgpSrteSegmentITypeSubTlv
        """type_i getter

        Type I: IPv6 Node Address with optional SRv6 SID.Type I: IPv6 Node Address with optional SRv6 SID.Type I: IPv6 Node Address with optional SRv6 SID.

        Returns: BgpSrteSegmentITypeSubTlv
        """
        return self._get_property("type_i", BgpSrteSegmentITypeSubTlv)

    @property
    def type_j(self):
        # type: () -> BgpSrteSegmentJTypeSubTlv
        """type_j getter

        Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.

        Returns: BgpSrteSegmentJTypeSubTlv
        """
        return self._get_property("type_j", BgpSrteSegmentJTypeSubTlv)

    @property
    def type_k(self):
        # type: () -> BgpSrteSegmentKTypeSubTlv
        """type_k getter

        Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.

        Returns: BgpSrteSegmentKTypeSubTlv
        """
        return self._get_property("type_k", BgpSrteSegmentKTypeSubTlv)

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteSegmentATypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "label": {
            "type": int,
            "format": "uint32",
            "maximum": 1048575,
        },
        "tc": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "s_bit": {"type": bool},
        "ttl": {
            "type": int,
            "format": "uint32",
            "maximum": 225,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, label=None, tc=None, s_bit=None, ttl=None
    ):
        super(BgpSrteSegmentATypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("label", label)
        self._set_property("tc", tc)
        self._set_property("s_bit", s_bit)
        self._set_property("ttl", ttl)

    def set(self, flags=None, label=None, tc=None, s_bit=None, ttl=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def label(self):
        # type: () -> int
        """label getter

        Label value in [0, 2^20 -1].

        Returns: int
        """
        return self._get_property("label")

    @label.setter
    def label(self, value):
        """label setter

        Label value in [0, 2^20 -1].

        value: int
        """
        self._set_property("label", value)

    @property
    def tc(self):
        # type: () -> int
        """tc getter

        Traffic class in bits.

        Returns: int
        """
        return self._get_property("tc")

    @tc.setter
    def tc(self, value):
        """tc setter

        Traffic class in bits.

        value: int
        """
        self._set_property("tc", value)

    @property
    def s_bit(self):
        # type: () -> bool
        """s_bit getter

        Bottom-of-Stack bit.

        Returns: bool
        """
        return self._get_property("s_bit")

    @s_bit.setter
    def s_bit(self, value):
        """s_bit setter

        Bottom-of-Stack bit.

        value: bool
        """
        self._set_property("s_bit", value)

    @property
    def ttl(self):
        # type: () -> int
        """ttl getter

        Time To Live.

        Returns: int
        """
        return self._get_property("ttl")

    @ttl.setter
    def ttl(self, value):
        """ttl setter

        Time To Live.

        value: int
        """
        self._set_property("ttl", value)


class BgpSrteSegmentBTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("srv6_sid",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, srv6_sid=None):
        super(BgpSrteSegmentBTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("srv6_sid", srv6_sid)

    def set(self, flags=None, srv6_sid=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        SRv6 SID.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property srv6_sid as None")
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSRv6SIDEndpointBehaviorAndStructure(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "lb_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "ln_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "func_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "arg_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "lb_length": 0,
        "ln_length": 0,
        "func_length": 0,
        "arg_length": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, lb_length=0, ln_length=0, func_length=0, arg_length=0
    ):
        super(BgpSrteSRv6SIDEndpointBehaviorAndStructure, self).__init__()
        self._parent = parent
        self._set_property("lb_length", lb_length)
        self._set_property("ln_length", ln_length)
        self._set_property("func_length", func_length)
        self._set_property("arg_length", arg_length)

    def set(self, lb_length=None, ln_length=None, func_length=None, arg_length=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def lb_length(self):
        # type: () -> int
        """lb_length getter

        SRv6 SID Locator Block length in bits.

        Returns: int
        """
        return self._get_property("lb_length")

    @lb_length.setter
    def lb_length(self, value):
        """lb_length setter

        SRv6 SID Locator Block length in bits.

        value: int
        """
        self._set_property("lb_length", value)

    @property
    def ln_length(self):
        # type: () -> int
        """ln_length getter

        SRv6 SID Locator Node length in bits.

        Returns: int
        """
        return self._get_property("ln_length")

    @ln_length.setter
    def ln_length(self, value):
        """ln_length setter

        SRv6 SID Locator Node length in bits.

        value: int
        """
        self._set_property("ln_length", value)

    @property
    def func_length(self):
        # type: () -> int
        """func_length getter

        SRv6 SID Function length in bits.

        Returns: int
        """
        return self._get_property("func_length")

    @func_length.setter
    def func_length(self, value):
        """func_length setter

        SRv6 SID Function length in bits.

        value: int
        """
        self._set_property("func_length", value)

    @property
    def arg_length(self):
        # type: () -> int
        """arg_length getter

        SRv6 SID Arguments length in bits.

        Returns: int
        """
        return self._get_property("arg_length")

    @arg_length.setter
    def arg_length(self, value):
        """arg_length setter

        SRv6 SID Arguments length in bits.

        value: int
        """
        self._set_property("arg_length", value)


class BgpSrteSegmentCTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, sr_algorithm=0, ipv4_node_address=None):
        super(BgpSrteSegmentCTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv4_node_address", ipv4_node_address)

    def set(self, flags=None, sr_algorithm=None, ipv4_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv4_node_address(self):
        # type: () -> str
        """ipv4_node_address getter

        IPv4 address representing node.

        Returns: str
        """
        return self._get_property("ipv4_node_address")

    @ipv4_node_address.setter
    def ipv4_node_address(self, value):
        """ipv4_node_address setter

        IPv4 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv4_node_address as None")
        self._set_property("ipv4_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSrMplsSid(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "label": {
            "type": int,
            "format": "uint32",
            "maximum": 1048575,
        },
        "tc": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "s_bit": {"type": bool},
        "ttl": {
            "type": int,
            "format": "uint32",
            "maximum": 225,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, label=None, tc=None, s_bit=None, ttl=None):
        super(BgpSrteSrMplsSid, self).__init__()
        self._parent = parent
        self._set_property("label", label)
        self._set_property("tc", tc)
        self._set_property("s_bit", s_bit)
        self._set_property("ttl", ttl)

    def set(self, label=None, tc=None, s_bit=None, ttl=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def label(self):
        # type: () -> int
        """label getter

        Label value in [0, 2^20 -1].

        Returns: int
        """
        return self._get_property("label")

    @label.setter
    def label(self, value):
        """label setter

        Label value in [0, 2^20 -1].

        value: int
        """
        self._set_property("label", value)

    @property
    def tc(self):
        # type: () -> int
        """tc getter

        Traffic class in bits.

        Returns: int
        """
        return self._get_property("tc")

    @tc.setter
    def tc(self, value):
        """tc setter

        Traffic class in bits.

        value: int
        """
        self._set_property("tc", value)

    @property
    def s_bit(self):
        # type: () -> bool
        """s_bit getter

        Bottom-of-Stack bit.

        Returns: bool
        """
        return self._get_property("s_bit")

    @s_bit.setter
    def s_bit(self, value):
        """s_bit setter

        Bottom-of-Stack bit.

        value: bool
        """
        self._set_property("s_bit", value)

    @property
    def ttl(self):
        # type: () -> int
        """ttl getter

        Time To Live.

        Returns: int
        """
        return self._get_property("ttl")

    @ttl.setter
    def ttl(self, value):
        """ttl setter

        Time To Live.

        value: int
        """
        self._set_property("ttl", value)


class BgpSrteSegmentDTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
        "ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, sr_algorithm=0, ipv6_node_address=None):
        super(BgpSrteSegmentDTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv6_node_address", ipv6_node_address)

    def set(self, flags=None, sr_algorithm=None, ipv6_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        specifying SR Algorithm when when A-Flag as defined in above flags.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        specifying SR Algorithm when when A-Flag as defined in above flags.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv6_node_address(self):
        # type: () -> str
        """ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("ipv6_node_address")

    @ipv6_node_address.setter
    def ipv6_node_address(self, value):
        """ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv6_node_address as None")
        self._set_property("ipv6_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentETypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv4_node_address",)  # type: tuple(str)

    _DEFAULTS = {
        "local_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_interface_id=0, ipv4_node_address=None
    ):
        super(BgpSrteSegmentETypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("ipv4_node_address", ipv4_node_address)

    def set(self, flags=None, local_interface_id=None, ipv4_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def ipv4_node_address(self):
        # type: () -> str
        """ipv4_node_address getter

        IPv4 address representing node.

        Returns: str
        """
        return self._get_property("ipv4_node_address")

    @ipv4_node_address.setter
    def ipv4_node_address(self, value):
        """ipv4_node_address setter

        IPv4 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv4_node_address as None")
        self._set_property("ipv4_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentFTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "remote_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv4_address", "remote_ipv4_address")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_ipv4_address=None, remote_ipv4_address=None
    ):
        super(BgpSrteSegmentFTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_ipv4_address", local_ipv4_address)
        self._set_property("remote_ipv4_address", remote_ipv4_address)

    def set(self, flags=None, local_ipv4_address=None, remote_ipv4_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_ipv4_address(self):
        # type: () -> str
        """local_ipv4_address getter

        Local IPv4 Address.

        Returns: str
        """
        return self._get_property("local_ipv4_address")

    @local_ipv4_address.setter
    def local_ipv4_address(self, value):
        """local_ipv4_address setter

        Local IPv4 Address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property local_ipv4_address as None")
        self._set_property("local_ipv4_address", value)

    @property
    def remote_ipv4_address(self):
        # type: () -> str
        """remote_ipv4_address getter

        Remote IPv4 Address.

        Returns: str
        """
        return self._get_property("remote_ipv4_address")

    @remote_ipv4_address.setter
    def remote_ipv4_address(self, value):
        """remote_ipv4_address setter

        Remote IPv4 Address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property remote_ipv4_address as None")
        self._set_property("remote_ipv4_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentGTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "local_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "remote_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = (
        "local_ipv6_node_address",
        "remote_ipv6_node_address",
    )  # type: tuple(str)

    _DEFAULTS = {
        "local_interface_id": 0,
        "remote_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        local_interface_id=0,
        local_ipv6_node_address=None,
        remote_interface_id=0,
        remote_ipv6_node_address=None,
    ):
        super(BgpSrteSegmentGTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("local_ipv6_node_address", local_ipv6_node_address)
        self._set_property("remote_interface_id", remote_interface_id)
        self._set_property("remote_ipv6_node_address", remote_ipv6_node_address)

    def set(
        self,
        flags=None,
        local_interface_id=None,
        local_ipv6_node_address=None,
        remote_interface_id=None,
        remote_ipv6_node_address=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def local_ipv6_node_address(self):
        # type: () -> str
        """local_ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("local_ipv6_node_address")

    @local_ipv6_node_address.setter
    def local_ipv6_node_address(self, value):
        """local_ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError(
                "Cannot set required property local_ipv6_node_address as None"
            )
        self._set_property("local_ipv6_node_address", value)

    @property
    def remote_interface_id(self):
        # type: () -> int
        """remote_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("remote_interface_id")

    @remote_interface_id.setter
    def remote_interface_id(self, value):
        """remote_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("remote_interface_id", value)

    @property
    def remote_ipv6_node_address(self):
        # type: () -> str
        """remote_ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("remote_ipv6_node_address")

    @remote_ipv6_node_address.setter
    def remote_ipv6_node_address(self, value):
        """remote_ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError(
                "Cannot set required property remote_ipv6_node_address as None"
            )
        self._set_property("remote_ipv6_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentHTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "local_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpSrteSrMplsSid"},
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv6_address", "remote_ipv6_address")  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flags=None, local_ipv6_address=None, remote_ipv6_address=None
    ):
        super(BgpSrteSegmentHTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("local_ipv6_address", local_ipv6_address)
        self._set_property("remote_ipv6_address", remote_ipv6_address)

    def set(self, flags=None, local_ipv6_address=None, remote_ipv6_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def local_ipv6_address(self):
        # type: () -> str
        """local_ipv6_address getter

        Local IPv6 Address.

        Returns: str
        """
        return self._get_property("local_ipv6_address")

    @local_ipv6_address.setter
    def local_ipv6_address(self, value):
        """local_ipv6_address setter

        Local IPv6 Address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property local_ipv6_address as None")
        self._set_property("local_ipv6_address", value)

    @property
    def remote_ipv6_address(self):
        # type: () -> str
        """remote_ipv6_address getter

        Remote IPv6 Address.

        Returns: str
        """
        return self._get_property("remote_ipv6_address")

    @remote_ipv6_address.setter
    def remote_ipv6_address(self, value):
        """remote_ipv6_address setter

        Remote IPv6 Address.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property remote_ipv6_address as None")
        self._set_property("remote_ipv6_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpSrteSrMplsSid
        """sr_mpls_sid getter

        Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Configuration for SR-MPLS with Label, TC, Bottom-of-Stack and TTL.Optional SR-MPLS SID.

        Returns: BgpSrteSrMplsSid
        """
        return self._get_property("sr_mpls_sid", BgpSrteSrMplsSid)


class BgpSrteSegmentITypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_node_address",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, flags=None, ipv6_node_address=None, srv6_sid=None):
        super(BgpSrteSegmentITypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("ipv6_node_address", ipv6_node_address)
        self._set_property("srv6_sid", srv6_sid)

    def set(self, flags=None, ipv6_node_address=None, srv6_sid=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def ipv6_node_address(self):
        # type: () -> str
        """ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("ipv6_node_address")

    @ipv6_node_address.setter
    def ipv6_node_address(self, value):
        """ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv6_node_address as None")
        self._set_property("ipv6_node_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentJTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
        },
        "local_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "local_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "remote_ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = (
        "local_ipv6_node_address",
        "remote_ipv6_node_address",
    )  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
        "local_interface_id": 0,
        "remote_interface_id": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        sr_algorithm=0,
        local_interface_id=0,
        local_ipv6_node_address=None,
        remote_interface_id=0,
        remote_ipv6_node_address=None,
        srv6_sid=None,
    ):
        super(BgpSrteSegmentJTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("local_ipv6_node_address", local_ipv6_node_address)
        self._set_property("remote_interface_id", remote_interface_id)
        self._set_property("remote_ipv6_node_address", remote_ipv6_node_address)
        self._set_property("srv6_sid", srv6_sid)

    def set(
        self,
        flags=None,
        sr_algorithm=None,
        local_interface_id=None,
        local_ipv6_node_address=None,
        remote_interface_id=None,
        remote_ipv6_node_address=None,
        srv6_sid=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def local_interface_id(self):
        # type: () -> int
        """local_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("local_interface_id")

    @local_interface_id.setter
    def local_interface_id(self, value):
        """local_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("local_interface_id", value)

    @property
    def local_ipv6_node_address(self):
        # type: () -> str
        """local_ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("local_ipv6_node_address")

    @local_ipv6_node_address.setter
    def local_ipv6_node_address(self, value):
        """local_ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError(
                "Cannot set required property local_ipv6_node_address as None"
            )
        self._set_property("local_ipv6_node_address", value)

    @property
    def remote_interface_id(self):
        # type: () -> int
        """remote_interface_id getter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        Returns: int
        """
        return self._get_property("remote_interface_id")

    @remote_interface_id.setter
    def remote_interface_id(self, value):
        """remote_interface_id setter

        Local Interface ID: The Interface Index as defined in [RFC8664].

        value: int
        """
        self._set_property("remote_interface_id", value)

    @property
    def remote_ipv6_node_address(self):
        # type: () -> str
        """remote_ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("remote_ipv6_node_address")

    @remote_ipv6_node_address.setter
    def remote_ipv6_node_address(self, value):
        """remote_ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError(
                "Cannot set required property remote_ipv6_node_address as None"
            )
        self._set_property("remote_ipv6_node_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentKTypeSubTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {
            "type": str,
            "format": "hex",
        },
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
        },
        "local_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "remote_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_sid_endpoint_behavior": {
            "type": "BgpSrteSRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("local_ipv6_address", "remote_ipv6_address")  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flags=None,
        sr_algorithm=0,
        local_ipv6_address=None,
        remote_ipv6_address=None,
        srv6_sid=None,
    ):
        super(BgpSrteSegmentKTypeSubTlv, self).__init__()
        self._parent = parent
        self._set_property("flags", flags)
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("local_ipv6_address", local_ipv6_address)
        self._set_property("remote_ipv6_address", remote_ipv6_address)
        self._set_property("srv6_sid", srv6_sid)

    def set(
        self,
        flags=None,
        sr_algorithm=None,
        local_ipv6_address=None,
        remote_ipv6_address=None,
        srv6_sid=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> str
        """flags getter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        Returns: str
        """
        return self._get_property("flags")

    @flags.setter
    def flags(self, value):
        """flags setter

        One octet bitmap for flags including V-Flag, A-Flag, S-Flag, B-Flag etc. as defined in https://datatracker.ietf.org/doc/html/draft-ietf-idr-segment-routing-te-policy-13#section-2.4.4.2.12

        value: str
        """
        self._set_property("flags", value)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def local_ipv6_address(self):
        # type: () -> str
        """local_ipv6_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("local_ipv6_address")

    @local_ipv6_address.setter
    def local_ipv6_address(self, value):
        """local_ipv6_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property local_ipv6_address as None")
        self._set_property("local_ipv6_address", value)

    @property
    def remote_ipv6_address(self):
        # type: () -> str
        """remote_ipv6_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("remote_ipv6_address")

    @remote_ipv6_address.setter
    def remote_ipv6_address(self, value):
        """remote_ipv6_address setter

        IPv6 address representing node.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property remote_ipv6_address as None")
        self._set_property("remote_ipv6_address", value)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        Optional SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        Optional SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_sid_endpoint_behavior(self):
        # type: () -> BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """srv6_sid_endpoint_behavior getter

        Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Configuration for SRv6 Endpoint Behavior and SID Structure. Its optional. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128.Optional SRv6 Endpoint Behavior and SID Structure.

        Returns: BgpSrteSRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_sid_endpoint_behavior", BgpSrteSRv6SIDEndpointBehaviorAndStructure
        )


class BgpSrteSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteSegment
        return self._next()

    def next(self):
        # type: () -> BgpSrteSegment
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteSegment):
            raise Exception("Item is not an instance of BgpSrteSegment")

    def segment(self, segment_type=None, name=None, active=True):
        # type: (Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]],str,bool) -> BgpSrteSegmentIter
        """Factory method that creates an instance of the BgpSrteSegment class

        A Segment sub-TLV describes single segment in segment list i.e., single element of the explicit path. The Segment sub-TLVs are optional.

        Returns: BgpSrteSegmentIter
        """
        item = BgpSrteSegment(
            parent=self._parent, segment_type=segment_type, name=name, active=active
        )
        self._add(item)
        return self

    def add(self, segment_type=None, name=None, active=True):
        # type: (Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]],str,bool) -> BgpSrteSegment
        """Add method that creates and returns an instance of the BgpSrteSegment class

        A Segment sub-TLV describes single segment in segment list i.e., single element of the explicit path. The Segment sub-TLVs are optional.

        Returns: BgpSrteSegment
        """
        item = BgpSrteSegment(
            parent=self._parent, segment_type=segment_type, name=name, active=active
        )
        self._add(item)
        return item


class BgpSrteSegmentListIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteSegmentListIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteSegmentList]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteSegmentListIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteSegmentList
        return self._next()

    def next(self):
        # type: () -> BgpSrteSegmentList
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteSegmentList):
            raise Exception("Item is not an instance of BgpSrteSegmentList")

    def segmentlist(self, weight=0, name=None, active=True):
        # type: (int,str,bool) -> BgpSrteSegmentListIter
        """Factory method that creates an instance of the BgpSrteSegmentList class

        Optional configuration for BGP SR TE Policy segment list. The Segment List sub-TLV encodes single explicit path towards the Endpoint.

        Returns: BgpSrteSegmentListIter
        """
        item = BgpSrteSegmentList(
            parent=self._parent, weight=weight, name=name, active=active
        )
        self._add(item)
        return self

    def add(self, weight=0, name=None, active=True):
        # type: (int,str,bool) -> BgpSrteSegmentList
        """Add method that creates and returns an instance of the BgpSrteSegmentList class

        Optional configuration for BGP SR TE Policy segment list. The Segment List sub-TLV encodes single explicit path towards the Endpoint.

        Returns: BgpSrteSegmentList
        """
        item = BgpSrteSegmentList(
            parent=self._parent, weight=weight, name=name, active=active
        )
        self._add(item)
        return item


class BgpSrteV4TunnelTlvIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV4TunnelTlvIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV4TunnelTlv]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV4TunnelTlvIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV4TunnelTlv
        return self._next()

    def next(self):
        # type: () -> BgpSrteV4TunnelTlv
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteV4TunnelTlv):
            raise Exception("Item is not an instance of BgpSrteV4TunnelTlv")

    def v4tunneltlv(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV4TunnelTlvIter
        """Factory method that creates an instance of the BgpSrteV4TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV4TunnelTlvIter
        """
        item = BgpSrteV4TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return self

    def add(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV4TunnelTlv
        """Add method that creates and returns an instance of the BgpSrteV4TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV4TunnelTlv
        """
        item = BgpSrteV4TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return item


class BgpSrteV4PolicyIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV4PolicyIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV4Policy]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV4PolicyIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV4Policy
        return self._next()

    def next(self):
        # type: () -> BgpSrteV4Policy
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteV4Policy):
            raise Exception("Item is not an instance of BgpSrteV4Policy")

    def v4policy(
        self,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV4PolicyIter
        """Factory method that creates an instance of the BgpSrteV4Policy class

        Configuration for BGP Segment Routing Traffic Engineering(SRTE) policy..

        Returns: BgpSrteV4PolicyIter
        """
        item = BgpSrteV4Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv4_endpoint=ipv4_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return self

    def add(
        self,
        distinguisher=1,
        color=100,
        ipv4_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv4",
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV4Policy
        """Add method that creates and returns an instance of the BgpSrteV4Policy class

        Configuration for BGP Segment Routing Traffic Engineering(SRTE) policy..

        Returns: BgpSrteV4Policy
        """
        item = BgpSrteV4Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv4_endpoint=ipv4_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return item


class BgpSrteV6Policy(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "distinguisher": {
            "type": int,
            "format": "uint32",
        },
        "color": {
            "type": int,
            "format": "uint32",
        },
        "ipv6_endpoint": {
            "type": str,
            "format": "ipv6",
        },
        "next_hop_mode": {
            "type": str,
            "enum": [
                "local_ip",
                "manual",
            ],
        },
        "next_hop_address_type": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
            ],
        },
        "next_hop_ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
        "next_hop_ipv6_address": {
            "type": str,
            "format": "ipv6",
        },
        "advanced": {"type": "BgpRouteAdvanced"},
        "add_path": {"type": "BgpAddPath"},
        "as_path": {"type": "BgpAsPath"},
        "communities": {"type": "BgpCommunityIter"},
        "extcommunities": {"type": "BgpExtCommunityIter"},
        "tunnel_tlvs": {"type": "BgpSrteV6TunnelTlvIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("ipv6_endpoint", "name")  # type: tuple(str)

    _DEFAULTS = {
        "distinguisher": 1,
        "color": 100,
        "next_hop_mode": "local_ip",
        "next_hop_address_type": "ipv6",
        "next_hop_ipv4_address": "0.0.0.0",
        "next_hop_ipv6_address": "::0",
        "active": True,
    }  # type: Dict[str, Union(type)]

    LOCAL_IP = "local_ip"  # type: str
    MANUAL = "manual"  # type: str

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        super(BgpSrteV6Policy, self).__init__()
        self._parent = parent
        self._set_property("distinguisher", distinguisher)
        self._set_property("color", color)
        self._set_property("ipv6_endpoint", ipv6_endpoint)
        self._set_property("next_hop_mode", next_hop_mode)
        self._set_property("next_hop_address_type", next_hop_address_type)
        self._set_property("next_hop_ipv4_address", next_hop_ipv4_address)
        self._set_property("next_hop_ipv6_address", next_hop_ipv6_address)
        self._set_property("name", name)
        self._set_property("active", active)

    def set(
        self,
        distinguisher=None,
        color=None,
        ipv6_endpoint=None,
        next_hop_mode=None,
        next_hop_address_type=None,
        next_hop_ipv4_address=None,
        next_hop_ipv6_address=None,
        name=None,
        active=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def distinguisher(self):
        # type: () -> int
        """distinguisher getter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        Returns: int
        """
        return self._get_property("distinguisher")

    @distinguisher.setter
    def distinguisher(self, value):
        """distinguisher setter

        Identifies the policy in the context of (color and endpoint) tuple. It is used by the SR Policy originator to make unique multiple occurrences of the same SR Policy.

        value: int
        """
        self._set_property("distinguisher", value)

    @property
    def color(self):
        # type: () -> int
        """color getter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        Returns: int
        """
        return self._get_property("color")

    @color.setter
    def color(self, value):
        """color setter

        Identifies the policy. It is used to match the color of the destination prefixes to steer traffic into the SR Policy.

        value: int
        """
        self._set_property("color", value)

    @property
    def ipv6_endpoint(self):
        # type: () -> str
        """ipv6_endpoint getter

        Specifies single node or set of nodes (e.g., an anycast address). It is selected on the basis of the SR Policy type (AFI).

        Returns: str
        """
        return self._get_property("ipv6_endpoint")

    @ipv6_endpoint.setter
    def ipv6_endpoint(self, value):
        """ipv6_endpoint setter

        Specifies single node or set of nodes (e.g., an anycast address). It is selected on the basis of the SR Policy type (AFI).

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property ipv6_endpoint as None")
        self._set_property("ipv6_endpoint", value)

    @property
    def next_hop_mode(self):
        # type: () -> Union[Literal["local_ip"], Literal["manual"]]
        """next_hop_mode getter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        Returns: Union[Literal["local_ip"], Literal["manual"]]
        """
        return self._get_property("next_hop_mode")

    @next_hop_mode.setter
    def next_hop_mode(self, value):
        """next_hop_mode setter

        Mode for choosing the NextHop in MP REACH NLRI. Available modes are Local IP: Automatically fills the Nexthop with the Local IP of the BGP peer. For IPv6 BGP peer the Nexthop Encoding capability should be enabled. Manual: Override the Nexthop with any arbitrary IPv4/IPv6 address.

        value: Union[Literal["local_ip"], Literal["manual"]]
        """
        self._set_property("next_hop_mode", value)

    @property
    def next_hop_address_type(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"]]
        """next_hop_address_type getter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        Returns: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        return self._get_property("next_hop_address_type")

    @next_hop_address_type.setter
    def next_hop_address_type(self, value):
        """next_hop_address_type setter

        Type of next hop IP address to be used when 'next_hop_mode' is set to 'manual'.

        value: Union[Literal["ipv4"], Literal["ipv6"]]
        """
        self._set_property("next_hop_address_type", value)

    @property
    def next_hop_ipv4_address(self):
        # type: () -> str
        """next_hop_ipv4_address getter

        The IPv4 address of the Nexthop if the 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        Returns: str
        """
        return self._get_property("next_hop_ipv4_address")

    @next_hop_ipv4_address.setter
    def next_hop_ipv4_address(self, value):
        """next_hop_ipv4_address setter

        The IPv4 address of the Nexthop if the 'next_hop_mode' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv4. If BGP peer is of type IPv6, Nexthop Encoding capability extended_next_hop_encoding should be enabled.

        value: str
        """
        self._set_property("next_hop_ipv4_address", value)

    @property
    def next_hop_ipv6_address(self):
        # type: () -> str
        """next_hop_ipv6_address getter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        Returns: str
        """
        return self._get_property("next_hop_ipv6_address")

    @next_hop_ipv6_address.setter
    def next_hop_ipv6_address(self, value):
        """next_hop_ipv6_address setter

        The IPv6 address of the next hop if the Nexthop Mode 'next_hop_address_type' is 'manual' and the Nexthop type 'next_hop_address_type' is IPv6.

        value: str
        """
        self._set_property("next_hop_ipv6_address", value)

    @property
    def advanced(self):
        # type: () -> BgpRouteAdvanced
        """advanced getter

        Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.Configuration for advanced BGP route range settings.

        Returns: BgpRouteAdvanced
        """
        return self._get_property("advanced", BgpRouteAdvanced)

    @property
    def add_path(self):
        # type: () -> BgpAddPath
        """add_path getter

        The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.The BGP Additional Paths feature is BGP extension that allows the advertisement of multiple paths for the same prefix without the new paths implicitly replacing any previous paths.

        Returns: BgpAddPath
        """
        return self._get_property("add_path", BgpAddPath)

    @property
    def as_path(self):
        # type: () -> BgpAsPath
        """as_path getter

        This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.This attribute identifies the autonomous systems through which routing information carried in this UPDATE message has passed. This contains the configuration of how to include the Local AS in the AS path attribute of the MP REACH NLRI. It also contains optional configuration of additional AS Path Segments that can be included in the AS Path attribute. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that routing information passes through to reach the destination.

        Returns: BgpAsPath
        """
        return self._get_property("as_path", BgpAsPath)

    @property
    def communities(self):
        # type: () -> BgpCommunityIter
        """communities getter

        Optional community settings.

        Returns: BgpCommunityIter
        """
        return self._get_property(
            "communities", BgpCommunityIter, self._parent, self._choice
        )

    @property
    def extcommunities(self):
        # type: () -> BgpExtCommunityIter
        """extcommunities getter

        Optional Extended Community settings. The Extended Communities Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an 8-Bytes value. It is divided into two main parts. The first Bytes of the community encode type and sub-type fields and the last Bytes carry unique set of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities. When type is administrator_as_2octet or administrator_as_4octet, the valid sub types are route target and origin. The valid value for administrator_as_2octet and administrator_as_4octet type is either two byte AS followed by four byte local administrator id or four byte AS followed by two byte local administrator id. When type is administrator_ipv4_address the valid sub types are route target and origin. The valid value for administrator_ipv4_address is four byte IPv4 address followed by two byte local administrator id. When type is opaque, valid sub types are color and encapsulation. When sub type is color, first two bytes of the value field contain flags and last four bytes contains the value of the color. When sub type is encapsulation the first four bytes of value field are reserved and last two bytes carries the tunnel type from IANA's "ETHER TYPES" registry e.g IPv4 (protocol type 0x0800), IPv6 (protocol type 0x86dd), and MPLS (protocol type 0x8847). When type is administrator_as_2octet_link_bandwidth the valid sub type is extended_bandwidth. The first two bytes of the value field contains the AS number and the last four bytes contains the bandwidth in IEEE floating point format. When type is evpn the valid subtype is mac_address. In the value field the low-order bit of the first byte(Flags) is defined as the "Sticky/static" flag and may be set to 1, indicating the MAC address is static and cannot move. The second byte is reserved and the last four bytes contain the sequence number which is used to ensure that PEs retain the correct MAC/IP Advertisement route when multiple updates occur for the same MAC address.

        Returns: BgpExtCommunityIter
        """
        return self._get_property(
            "extcommunities", BgpExtCommunityIter, self._parent, self._choice
        )

    @property
    def tunnel_tlvs(self):
        # type: () -> BgpSrteV6TunnelTlvIter
        """tunnel_tlvs getter

        List of optional tunnel TLV settings.

        Returns: BgpSrteV6TunnelTlvIter
        """
        return self._get_property(
            "tunnel_tlvs", BgpSrteV6TunnelTlvIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV6TunnelTlv(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "remote_endpoint_sub_tlv": {"type": "BgpSrteRemoteEndpointSubTlv"},
        "color_sub_tlv": {"type": "BgpSrteColorSubTlv"},
        "binding_sub_tlv": {"type": "BgpSrteBindingSubTlv"},
        "preference_sub_tlv": {"type": "BgpSrtePreferenceSubTlv"},
        "policy_priority_sub_tlv": {"type": "BgpSrtePolicyPrioritySubTlv"},
        "policy_name_sub_tlv": {"type": "BgpSrtePolicyNameSubTlv"},
        "explicit_null_label_policy_sub_tlv": {
            "type": "BgpSrteExplicitNullLabelPolicySubTlv"
        },
        "segment_lists": {"type": "BgpSrteSegmentListIter"},
        "name": {"type": str},
        "active": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ("name",)  # type: tuple(str)

    _DEFAULTS = {
        "active": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, name=None, active=True):
        super(BgpSrteV6TunnelTlv, self).__init__()
        self._parent = parent
        self._set_property("name", name)
        self._set_property("active", active)

    def set(self, name=None, active=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def remote_endpoint_sub_tlv(self):
        # type: () -> BgpSrteRemoteEndpointSubTlv
        """remote_endpoint_sub_tlv getter

        Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.Configuration for the BGP remote endpoint sub TLV.

        Returns: BgpSrteRemoteEndpointSubTlv
        """
        return self._get_property(
            "remote_endpoint_sub_tlv", BgpSrteRemoteEndpointSubTlv
        )

    @property
    def color_sub_tlv(self):
        # type: () -> BgpSrteColorSubTlv
        """color_sub_tlv getter

        Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure. Configuration for the Policy Color attribute sub-TLV. The Color sub-TLV MAY be used as way to "color" the corresponding Tunnel TLV. The Value field of the sub-TLV is eight octets long and consists of Color Extended Community. First two octets of its Value field are 0x030b as type and subtype of extended community. Remaining six octets are are exposed to configure.

        Returns: BgpSrteColorSubTlv
        """
        return self._get_property("color_sub_tlv", BgpSrteColorSubTlv)

    @property
    def binding_sub_tlv(self):
        # type: () -> BgpSrteBindingSubTlv
        """binding_sub_tlv getter

        Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.Configuration for the binding SID sub-TLV. This is used to signal the binding SID related information of the SR Policy candidate path.

        Returns: BgpSrteBindingSubTlv
        """
        return self._get_property("binding_sub_tlv", BgpSrteBindingSubTlv)

    @property
    def preference_sub_tlv(self):
        # type: () -> BgpSrtePreferenceSubTlv
        """preference_sub_tlv getter

        Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.Configuration for BGP preference sub TLV of the SR Policy candidate path.

        Returns: BgpSrtePreferenceSubTlv
        """
        return self._get_property("preference_sub_tlv", BgpSrtePreferenceSubTlv)

    @property
    def policy_priority_sub_tlv(self):
        # type: () -> BgpSrtePolicyPrioritySubTlv
        """policy_priority_sub_tlv getter

        Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.Configuration for the Policy Priority sub-TLV. The Policy Priority to indicate the order in which the SR policies are re-computed upon topological change.

        Returns: BgpSrtePolicyPrioritySubTlv
        """
        return self._get_property(
            "policy_priority_sub_tlv", BgpSrtePolicyPrioritySubTlv
        )

    @property
    def policy_name_sub_tlv(self):
        # type: () -> BgpSrtePolicyNameSubTlv
        """policy_name_sub_tlv getter

        Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.Configuration for the Policy Name sub-TLV. The Policy Name sub-TLV is used to attach symbolic name to the SR Policy candidate path.

        Returns: BgpSrtePolicyNameSubTlv
        """
        return self._get_property("policy_name_sub_tlv", BgpSrtePolicyNameSubTlv)

    @property
    def explicit_null_label_policy_sub_tlv(self):
        # type: () -> BgpSrteExplicitNullLabelPolicySubTlv
        """explicit_null_label_policy_sub_tlv getter

        Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.Configuration for BGP explicit null label policy sub TLV settings.

        Returns: BgpSrteExplicitNullLabelPolicySubTlv
        """
        return self._get_property(
            "explicit_null_label_policy_sub_tlv", BgpSrteExplicitNullLabelPolicySubTlv
        )

    @property
    def segment_lists(self):
        # type: () -> BgpSrteSegmentListIter
        """segment_lists getter

        TBD

        Returns: BgpSrteSegmentListIter
        """
        return self._get_property(
            "segment_lists", BgpSrteSegmentListIter, self._parent, self._choice
        )

    @property
    def name(self):
        # type: () -> str
        """name getter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        Returns: str
        """
        return self._get_property("name")

    @name.setter
    def name(self, value):
        """name setter

        Globally unique name of an object. It also serves as the primary key for arrays of objects.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property name as None")
        self._set_property("name", value)

    @property
    def active(self):
        # type: () -> bool
        """active getter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        Returns: bool
        """
        return self._get_property("active")

    @active.setter
    def active(self, value):
        """active setter

        If enabled means that this part of the configuration including any active 'children' nodes will be advertised to peer. If disabled, this means that though config is present, it is not taking any part of the test but can be activated at run-time to advertise just this part of the configuration to the peer.

        value: bool
        """
        self._set_property("active", value)


class BgpSrteV6TunnelTlvIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV6TunnelTlvIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV6TunnelTlv]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV6TunnelTlvIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV6TunnelTlv
        return self._next()

    def next(self):
        # type: () -> BgpSrteV6TunnelTlv
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteV6TunnelTlv):
            raise Exception("Item is not an instance of BgpSrteV6TunnelTlv")

    def v6tunneltlv(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV6TunnelTlvIter
        """Factory method that creates an instance of the BgpSrteV6TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV6TunnelTlvIter
        """
        item = BgpSrteV6TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return self

    def add(self, name=None, active=True):
        # type: (str,bool) -> BgpSrteV6TunnelTlv
        """Add method that creates and returns an instance of the BgpSrteV6TunnelTlv class

        Configuration for BGP SRTE Tunnel TLV.

        Returns: BgpSrteV6TunnelTlv
        """
        item = BgpSrteV6TunnelTlv(parent=self._parent, name=name, active=active)
        self._add(item)
        return item


class BgpSrteV6PolicyIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpSrteV6PolicyIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpSrteV6Policy]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpSrteV6PolicyIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpSrteV6Policy
        return self._next()

    def next(self):
        # type: () -> BgpSrteV6Policy
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpSrteV6Policy):
            raise Exception("Item is not an instance of BgpSrteV6Policy")

    def v6policy(
        self,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV6PolicyIter
        """Factory method that creates an instance of the BgpSrteV6Policy class

        Configuration for BGP Segment Routing Traffic Engineering policy..

        Returns: BgpSrteV6PolicyIter
        """
        item = BgpSrteV6Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv6_endpoint=ipv6_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return self

    def add(
        self,
        distinguisher=1,
        color=100,
        ipv6_endpoint=None,
        next_hop_mode="local_ip",
        next_hop_address_type="ipv6",
        next_hop_ipv4_address="0.0.0.0",
        next_hop_ipv6_address="::0",
        name=None,
        active=True,
    ):
        # type: (int,int,str,Union[Literal["local_ip"], Literal["manual"]],Union[Literal["ipv4"], Literal["ipv6"]],str,str,str,bool) -> BgpSrteV6Policy
        """Add method that creates and returns an instance of the BgpSrteV6Policy class

        Configuration for BGP Segment Routing Traffic Engineering policy..

        Returns: BgpSrteV6Policy
        """
        item = BgpSrteV6Policy(
            parent=self._parent,
            distinguisher=distinguisher,
            color=color,
            ipv6_endpoint=ipv6_endpoint,
            next_hop_mode=next_hop_mode,
            next_hop_address_type=next_hop_address_type,
            next_hop_ipv4_address=next_hop_ipv4_address,
            next_hop_ipv6_address=next_hop_ipv6_address,
            name=name,
            active=active,
        )
        self._add(item)
        return item


class BgpGracefulRestart(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "enable_gr": {"type": bool},
        "restart_time": {
            "type": int,
            "format": "uint32",
            "maximum": 4096,
        },
        "enable_llgr": {"type": bool},
        "stale_time": {
            "type": int,
            "format": "uint32",
            "maximum": 16777215,
        },
        "enable_notification": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "enable_gr": False,
        "restart_time": 45,
        "enable_llgr": False,
        "stale_time": 10,
        "enable_notification": True,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        enable_gr=False,
        restart_time=45,
        enable_llgr=False,
        stale_time=10,
        enable_notification=True,
    ):
        super(BgpGracefulRestart, self).__init__()
        self._parent = parent
        self._set_property("enable_gr", enable_gr)
        self._set_property("restart_time", restart_time)
        self._set_property("enable_llgr", enable_llgr)
        self._set_property("stale_time", stale_time)
        self._set_property("enable_notification", enable_notification)

    def set(
        self,
        enable_gr=None,
        restart_time=None,
        enable_llgr=None,
        stale_time=None,
        enable_notification=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def enable_gr(self):
        # type: () -> bool
        """enable_gr getter

        If enabled, Graceful Restart capability is advertised in BGP OPEN messages.

        Returns: bool
        """
        return self._get_property("enable_gr")

    @enable_gr.setter
    def enable_gr(self, value):
        """enable_gr setter

        If enabled, Graceful Restart capability is advertised in BGP OPEN messages.

        value: bool
        """
        self._set_property("enable_gr", value)

    @property
    def restart_time(self):
        # type: () -> int
        """restart_time getter

        This is the estimated duration (in seconds) it will take for the BGP session to be re-established after restart. This can be used to speed up routing convergence by its peer in case the BGP speaker does not come back after restart.

        Returns: int
        """
        return self._get_property("restart_time")

    @restart_time.setter
    def restart_time(self, value):
        """restart_time setter

        This is the estimated duration (in seconds) it will take for the BGP session to be re-established after restart. This can be used to speed up routing convergence by its peer in case the BGP speaker does not come back after restart.

        value: int
        """
        self._set_property("restart_time", value)

    @property
    def enable_llgr(self):
        # type: () -> bool
        """enable_llgr getter

        If enabled, the "Long-lived Graceful Restart Capability", or "LLGR Capability". will be advertised.. This capability MUST be advertised in conjunction with the Graceful Restart capability.

        Returns: bool
        """
        return self._get_property("enable_llgr")

    @enable_llgr.setter
    def enable_llgr(self, value):
        """enable_llgr setter

        If enabled, the "Long-lived Graceful Restart Capability", or "LLGR Capability". will be advertised.. This capability MUST be advertised in conjunction with the Graceful Restart capability.

        value: bool
        """
        self._set_property("enable_llgr", value)

    @property
    def stale_time(self):
        # type: () -> int
        """stale_time getter

        Duration (in seconds) specifying how long stale information (for the AFI/SAFI) may be retained. This is three byte field and is applicable only if 'enable_llgr' is set to 'true'.

        Returns: int
        """
        return self._get_property("stale_time")

    @stale_time.setter
    def stale_time(self, value):
        """stale_time setter

        Duration (in seconds) specifying how long stale information (for the AFI/SAFI) may be retained. This is three byte field and is applicable only if 'enable_llgr' is set to 'true'.

        value: int
        """
        self._set_property("stale_time", value)

    @property
    def enable_notification(self):
        # type: () -> bool
        """enable_notification getter

        If enabled, the flag will be set in the Graceful Restart capability in the Open message. If both peers in BGP connection has this enabled, Graceful Restart procedures are performed. even if the peer goes down due to sending of Notification Message as per RFC8538.

        Returns: bool
        """
        return self._get_property("enable_notification")

    @enable_notification.setter
    def enable_notification(self, value):
        """enable_notification setter

        If enabled, the flag will be set in the Graceful Restart capability in the Open message. If both peers in BGP connection has this enabled, Graceful Restart procedures are performed. even if the peer goes down due to sending of Notification Message as per RFC8538.

        value: bool
        """
        self._set_property("enable_notification", value)


class BgpUpdateReplay(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "structured_pdus",
                "raw_bytes",
            ],
        },
        "structured_pdus": {"type": "BgpStructuredPdus"},
        "raw_bytes": {"type": "BgpRawBytes"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "structured_pdus",
    }  # type: Dict[str, Union(type)]

    STRUCTURED_PDUS = "structured_pdus"  # type: str
    RAW_BYTES = "raw_bytes"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpUpdateReplay, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def structured_pdus(self):
        # type: () -> BgpStructuredPdus
        """Factory property that returns an instance of the BgpStructuredPdus class

        Ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.

        Returns: BgpStructuredPdus
        """
        return self._get_property(
            "structured_pdus", BgpStructuredPdus, self, "structured_pdus"
        )

    @property
    def raw_bytes(self):
        # type: () -> BgpRawBytes
        """Factory property that returns an instance of the BgpRawBytes class

        Ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.

        Returns: BgpRawBytes
        """
        return self._get_property("raw_bytes", BgpRawBytes, self, "raw_bytes")

    @property
    def choice(self):
        # type: () -> Union[Literal["raw_bytes"], Literal["structured_pdus"]]
        """choice getter

        TBD

        Returns: Union[Literal["raw_bytes"], Literal["structured_pdus"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["raw_bytes"], Literal["structured_pdus"]]
        """
        self._set_property("choice", value)


class BgpStructuredPdus(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "updates": {"type": "BgpOneStructuredUpdateReplayIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpStructuredPdus, self).__init__()
        self._parent = parent

    @property
    def updates(self):
        # type: () -> BgpOneStructuredUpdateReplayIter
        """updates getter

        Array of ordered BGP Updates including both Advertise and Withdraws to be sent in the order given in the input to the peer after the BGP session is established.

        Returns: BgpOneStructuredUpdateReplayIter
        """
        return self._get_property(
            "updates", BgpOneStructuredUpdateReplayIter, self._parent, self._choice
        )


class BgpOneStructuredUpdateReplay(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "time_gap": {
            "type": int,
            "format": "uint32",
        },
        "path_attributes": {"type": "BgpAttributes"},
        "traditional_unreach_nlris": {"type": "BgpOneTraditionalNlriPrefixIter"},
        "traditional_reach_nlris": {"type": "BgpOneTraditionalNlriPrefixIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "time_gap": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, time_gap=0):
        super(BgpOneStructuredUpdateReplay, self).__init__()
        self._parent = parent
        self._set_property("time_gap", time_gap)

    def set(self, time_gap=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def time_gap(self):
        # type: () -> int
        """time_gap getter

        Minimum time interval in milliseconds from previous Update from the sequence of BGP Updates to be replayed.

        Returns: int
        """
        return self._get_property("time_gap")

    @time_gap.setter
    def time_gap(self, value):
        """time_gap setter

        Minimum time interval in milliseconds from previous Update from the sequence of BGP Updates to be replayed.

        value: int
        """
        self._set_property("time_gap", value)

    @property
    def path_attributes(self):
        # type: () -> BgpAttributes
        """path_attributes getter

        Attributes carried in the Update packet alongwith the reach/unreach prefixes.Attributes carried in the Update packet alongwith the reach/unreach prefixes.Attributes carried in the Update packet alongwith the reach/unreach prefixes.Attributes carried in the Update packet alongwith the reach/unreach prefixes.

        Returns: BgpAttributes
        """
        return self._get_property("path_attributes", BgpAttributes)

    @property
    def traditional_unreach_nlris(self):
        # type: () -> BgpOneTraditionalNlriPrefixIter
        """traditional_unreach_nlris getter

        The IPv4 prefixes to be included in the traditional UNREACH_NLRI.

        Returns: BgpOneTraditionalNlriPrefixIter
        """
        return self._get_property(
            "traditional_unreach_nlris",
            BgpOneTraditionalNlriPrefixIter,
            self._parent,
            self._choice,
        )

    @property
    def traditional_reach_nlris(self):
        # type: () -> BgpOneTraditionalNlriPrefixIter
        """traditional_reach_nlris getter

        The IPv4 prefixes to be included in the traditional REACH_NLRI.

        Returns: BgpOneTraditionalNlriPrefixIter
        """
        return self._get_property(
            "traditional_reach_nlris",
            BgpOneTraditionalNlriPrefixIter,
            self._parent,
            self._choice,
        )


class BgpAttributes(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "other_attributes": {"type": "BgpAttributesOtherAttributeIter"},
        "origin": {
            "type": str,
            "enum": [
                "igp",
                "egp",
                "incomplete",
            ],
        },
        "as_path": {"type": "BgpAttributesAsPath"},
        "as4_path": {"type": "BgpAttributesAs4Path"},
        "next_hop": {"type": "BgpAttributesNextHop"},
        "multi_exit_discriminator": {"type": "BgpAttributesMultiExitDiscriminator"},
        "local_preference": {"type": "BgpAttributesLocalPreference"},
        "include_atomic_aggregator": {"type": bool},
        "aggregator": {"type": "BgpAttributesAggregator"},
        "as4_aggregator": {"type": "BgpAttributesAs4Aggregator"},
        "community": {"type": "BgpAttributesCommunityIter"},
        "originator_id": {"type": "BgpAttributesOriginatorId"},
        "cluster_ids": {
            "type": list,
            "itemtype": str,
            "itemformat": "ipv4",
        },
        "extended_communities": {"type": "BgpExtendedCommunityIter"},
        "tunnel_encapsulation": {"type": "BgpAttributesTunnelEncapsulation"},
        "mp_reach": {"type": "BgpAttributesMpReachNlri"},
        "mp_unreach": {"type": "BgpAttributesMpUnreachNlri"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "origin": "incomplete",
        "include_atomic_aggregator": False,
    }  # type: Dict[str, Union(type)]

    IGP = "igp"  # type: str
    EGP = "egp"  # type: str
    INCOMPLETE = "incomplete"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        origin="incomplete",
        include_atomic_aggregator=False,
        cluster_ids=None,
    ):
        super(BgpAttributes, self).__init__()
        self._parent = parent
        self._set_property("origin", origin)
        self._set_property("include_atomic_aggregator", include_atomic_aggregator)
        self._set_property("cluster_ids", cluster_ids)

    def set(self, origin=None, include_atomic_aggregator=None, cluster_ids=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def other_attributes(self):
        # type: () -> BgpAttributesOtherAttributeIter
        """other_attributes getter

        Any attributes not present in the list of configurable attributes should be added to the list of unknown attributes.

        Returns: BgpAttributesOtherAttributeIter
        """
        return self._get_property(
            "other_attributes",
            BgpAttributesOtherAttributeIter,
            self._parent,
            self._choice,
        )

    @property
    def origin(self):
        # type: () -> Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """origin getter

        The ORIGIN attribute is mandatory attribute which can take three values: the prefix originates from an interior routing protocol 'igp', it originates from 'egp' or the origin is 'incomplete',if the prefix is learned through other means.

        Returns: Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """
        return self._get_property("origin")

    @origin.setter
    def origin(self, value):
        """origin setter

        The ORIGIN attribute is mandatory attribute which can take three values: the prefix originates from an interior routing protocol 'igp', it originates from 'egp' or the origin is 'incomplete',if the prefix is learned through other means.

        value: Union[Literal["egp"], Literal["igp"], Literal["incomplete"]]
        """
        self._set_property("origin", value)

    @property
    def as_path(self):
        # type: () -> BgpAttributesAsPath
        """as_path getter

        The AS_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. There are two modes in which AS numbers can be encoded in the AS Path Segments. When the AS Path is being exchanged between old and new BGP speakers or between two old BGP speakers the AS numbers are encoded as byte values.. When the AS Path is being exchanged between two new BGP speakers supporting byte AS the AS numbers are encoded as byte values.The AS_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. There are two modes in which AS numbers can be encoded in the AS Path Segments. When the AS Path is being exchanged between old and new BGP speakers or between two old BGP speakers the AS numbers are encoded as byte values.. When the AS Path is being exchanged between two new BGP speakers supporting byte AS the AS numbers are encoded as byte values.The AS_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. There are two modes in which AS numbers can be encoded in the AS Path Segments. When the AS Path is being exchanged between old and new BGP speakers or between two old BGP speakers the AS numbers are encoded as byte values.. When the AS Path is being exchanged between two new BGP speakers supporting byte AS the AS numbers are encoded as byte values.AS_PATH attribute to be included in the Update.

        Returns: BgpAttributesAsPath
        """
        return self._get_property("as_path", BgpAttributesAsPath)

    @property
    def as4_path(self):
        # type: () -> BgpAttributesAs4Path
        """as4_path getter

        The AS4_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. AS4_PATH is only exchanged in two scenarios:. When an old BGP speaker has to forward received AS4_PATH containing byte AS numbers to new BGP speaker.. When new BGP speaker is connected to an old BGP speaker and has to propagate byte AS numbers via the old BGP speaker.. Its usage is described in RFC4893.The AS4_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. AS4_PATH is only exchanged in two scenarios:. When an old BGP speaker has to forward received AS4_PATH containing byte AS numbers to new BGP speaker.. When new BGP speaker is connected to an old BGP speaker and has to propagate byte AS numbers via the old BGP speaker.. Its usage is described in RFC4893.The AS4_PATH attribute identifies the autonomous systems through which routing information. carried in this UPDATE message has passed.. This contains the configuration of how to include the Local AS in the AS path. attribute of the MP REACH NLRI. It also contains optional configuration of. additional AS Path Segments that can be included in the AS Path attribute.. The AS Path consists of Set or Sequence of Autonomous Systems (AS) numbers that. routing information passes through to reach the destination.. AS4_PATH is only exchanged in two scenarios:. When an old BGP speaker has to forward received AS4_PATH containing byte AS numbers to new BGP speaker.. When new BGP speaker is connected to an old BGP speaker and has to propagate byte AS numbers via the old BGP speaker.. Its usage is described in RFC4893.AS4_PATH attribute to be included in the Update.

        Returns: BgpAttributesAs4Path
        """
        return self._get_property("as4_path", BgpAttributesAs4Path)

    @property
    def next_hop(self):
        # type: () -> BgpAttributesNextHop
        """next_hop getter

        Next hop to be sent inside MP_REACH NLRI or as the NEXT_HOP attribute if advertised as traditional NLRI. Next hop to be sent inside MP_REACH NLRI or as the NEXT_HOP attribute if advertised as traditional NLRI. Next hop to be sent inside MP_REACH NLRI or as the NEXT_HOP attribute if advertised as traditional NLRI.

        Returns: BgpAttributesNextHop
        """
        return self._get_property("next_hop", BgpAttributesNextHop)

    @property
    def multi_exit_discriminator(self):
        # type: () -> BgpAttributesMultiExitDiscriminator
        """multi_exit_discriminator getter

        Optional MULTI_EXIT_DISCRIMINATOR attribute sent to the peer to help in the route selection process. Optional MULTI_EXIT_DISCRIMINATOR attribute sent to the peer to help in the route selection process. Optional MULTI_EXIT_DISCRIMINATOR attribute sent to the peer to help in the route selection process.

        Returns: BgpAttributesMultiExitDiscriminator
        """
        return self._get_property(
            "multi_exit_discriminator", BgpAttributesMultiExitDiscriminator
        )

    @property
    def local_preference(self):
        # type: () -> BgpAttributesLocalPreference
        """local_preference getter

        Optional LOCAL_PREFERENCE attribute sent to the peer to indicate the degree of preference for externally learned routes.This should be included only for internal peers.It is used for the selection of the path for the traffic leaving the AS.The route with the highest local preference value is preferred.Optional LOCAL_PREFERENCE attribute sent to the peer to indicate the degree of preference for externally learned routes.This should be included only for internal peers.It is used for the selection of the path for the traffic leaving the AS.The route with the highest local preference value is preferred.Optional LOCAL_PREFERENCE attribute sent to the peer to indicate the degree of preference for externally learned routes.This should be included only for internal peers.It is used for the selection of the path for the traffic leaving the AS.The route with the highest local preference value is preferred.

        Returns: BgpAttributesLocalPreference
        """
        return self._get_property("local_preference", BgpAttributesLocalPreference)

    @property
    def include_atomic_aggregator(self):
        # type: () -> bool
        """include_atomic_aggregator getter

        If enabled, it indicates that the ATOMIC_AGGREGATOR attribute should be included in the Update.. Presence of this attribute Indicates that this route might not be getting sent on fully optimized path since some intermediate BGP speaker has aggregated the route.

        Returns: bool
        """
        return self._get_property("include_atomic_aggregator")

    @include_atomic_aggregator.setter
    def include_atomic_aggregator(self, value):
        """include_atomic_aggregator setter

        If enabled, it indicates that the ATOMIC_AGGREGATOR attribute should be included in the Update.. Presence of this attribute Indicates that this route might not be getting sent on fully optimized path since some intermediate BGP speaker has aggregated the route.

        value: bool
        """
        self._set_property("include_atomic_aggregator", value)

    @property
    def aggregator(self):
        # type: () -> BgpAttributesAggregator
        """aggregator getter

        Optional AGGREGATOR attribute which maybe be added by BGP speaker which performs route aggregation.. When AGGREGATOR attribute is being sent to new BGP speaker the AS number is encoded as 4 byte value.. When AGGREGATOR attribute is being exchanged between new and an old BGP speaker or between two old BGP speakers, the AS number is encoded as 2 byte value.. It contain the AS number and IP address of the speaker performing the aggregation. Optional AGGREGATOR attribute which maybe be added by BGP speaker which performs route aggregation.. When AGGREGATOR attribute is being sent to new BGP speaker the AS number is encoded as 4 byte value.. When AGGREGATOR attribute is being exchanged between new and an old BGP speaker or between two old BGP speakers, the AS number is encoded as 2 byte value.. It contain the AS number and IP address of the speaker performing the aggregation. Optional AGGREGATOR attribute which maybe be added by BGP speaker which performs route aggregation.. When AGGREGATOR attribute is being sent to new BGP speaker the AS number is encoded as 4 byte value.. When AGGREGATOR attribute is being exchanged between new and an old BGP speaker or between two old BGP speakers, the AS number is encoded as 2 byte value.. It contain the AS number and IP address of the speaker performing the aggregation.

        Returns: BgpAttributesAggregator
        """
        return self._get_property("aggregator", BgpAttributesAggregator)

    @property
    def as4_aggregator(self):
        # type: () -> BgpAttributesAs4Aggregator
        """as4_aggregator getter

        Optional AS4_AGGREGATOR attribute which maybe be added by BGP speaker in one of two cases:. If it is new BGP speaker speaking to an old BGP speaker and needs to send 4 byte value for the AS number of the BGP route aggregator.. If it is old BGP speaker speaking to new BGP speaker and has to transparently forward received AS4_AGGREGATOR from some other peer.. Its usage is described in RFC4893. Optional AS4_AGGREGATOR attribute which maybe be added by BGP speaker in one of two cases:. If it is new BGP speaker speaking to an old BGP speaker and needs to send 4 byte value for the AS number of the BGP route aggregator.. If it is old BGP speaker speaking to new BGP speaker and has to transparently forward received AS4_AGGREGATOR from some other peer.. Its usage is described in RFC4893. Optional AS4_AGGREGATOR attribute which maybe be added by BGP speaker in one of two cases:. If it is new BGP speaker speaking to an old BGP speaker and needs to send 4 byte value for the AS number of the BGP route aggregator.. If it is old BGP speaker speaking to new BGP speaker and has to transparently forward received AS4_AGGREGATOR from some other peer.. Its usage is described in RFC4893.

        Returns: BgpAttributesAs4Aggregator
        """
        return self._get_property("as4_aggregator", BgpAttributesAs4Aggregator)

    @property
    def community(self):
        # type: () -> BgpAttributesCommunityIter
        """community getter

        TBD

        Returns: BgpAttributesCommunityIter
        """
        return self._get_property(
            "community", BgpAttributesCommunityIter, self._parent, self._choice
        )

    @property
    def originator_id(self):
        # type: () -> BgpAttributesOriginatorId
        """originator_id getter

        Optional ORIGINATOR_ID attribute (type code 9) carries the Router Id of the route's originator in the local AS.Optional ORIGINATOR_ID attribute (type code 9) carries the Router Id of the route's originator in the local AS.Optional ORIGINATOR_ID attribute (type code 9) carries the Router Id of the route's originator in the local AS.

        Returns: BgpAttributesOriginatorId
        """
        return self._get_property("originator_id", BgpAttributesOriginatorId)

    @property
    def cluster_ids(self):
        # type: () -> List[str]
        """cluster_ids getter

        When Route Reflector reflects route, it prepends the local CLUSTER_ID to the CLUSTER_LIST as defined in RFC4456.

        Returns: List[str]
        """
        return self._get_property("cluster_ids")

    @cluster_ids.setter
    def cluster_ids(self, value):
        """cluster_ids setter

        When Route Reflector reflects route, it prepends the local CLUSTER_ID to the CLUSTER_LIST as defined in RFC4456.

        value: List[str]
        """
        self._set_property("cluster_ids", value)

    @property
    def extended_communities(self):
        # type: () -> BgpExtendedCommunityIter
        """extended_communities getter

        Optional EXTENDED_COMMUNITY attribute settings.. The EXTENDED_COMMUNITY Attribute is transitive optional BGP attribute, with the Type Code 16. Community and Extended Communities attributes. are utilized to trigger routing decisions, such as acceptance, rejection, preference, or redistribution. An extended community is an eight byte value.. It is divided into two main parts. The first two bytes of the community encode type and sub-type fields and the last six bytes carry unique set. of data in format defined by the type and sub-type field. Extended communities provide larger range for grouping or categorizing communities.

        Returns: BgpExtendedCommunityIter
        """
        return self._get_property(
            "extended_communities", BgpExtendedCommunityIter, self._parent, self._choice
        )

    @property
    def tunnel_encapsulation(self):
        # type: () -> BgpAttributesTunnelEncapsulation
        """tunnel_encapsulation getter

        The TUNNEL_ENCAPSULATION attribute is used by BGP speaker to inform other BGP speakers how to encapsulate packets that need to be sent to it.. It is defined in RFC9012 and is assigned Type code of 23.The TUNNEL_ENCAPSULATION attribute is used by BGP speaker to inform other BGP speakers how to encapsulate packets that need to be sent to it.. It is defined in RFC9012 and is assigned Type code of 23.The TUNNEL_ENCAPSULATION attribute is used by BGP speaker to inform other BGP speakers how to encapsulate packets that need to be sent to it.. It is defined in RFC9012 and is assigned Type code of 23.

        Returns: BgpAttributesTunnelEncapsulation
        """
        return self._get_property(
            "tunnel_encapsulation", BgpAttributesTunnelEncapsulation
        )

    @property
    def mp_reach(self):
        # type: () -> BgpAttributesMpReachNlri
        """mp_reach getter

        The MP_REACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 draft-ietf-idr-sr-policy-safi-02 Section 2.1 ). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73The MP_REACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 draft-ietf-idr-sr-policy-safi-02 Section 2.1 ). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73The MP_REACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 draft-ietf-idr-sr-policy-safi-02 Section 2.1 ). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73

        Returns: BgpAttributesMpReachNlri
        """
        return self._get_property("mp_reach", BgpAttributesMpReachNlri)

    @property
    def mp_unreach(self):
        # type: () -> BgpAttributesMpUnreachNlri
        """mp_unreach getter

        The MP_UNREACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 (draft-ietf-idr-sr-policy-safi-02 Section 2.1). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73The MP_UNREACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 (draft-ietf-idr-sr-policy-safi-02 Section 2.1). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73The MP_UNREACH attribute is an optional attribute which can be included in the attributes of BGP Update message as defined in https://datatracker.ietf.org/doc/html/rfc4760#section-3.. The following AFI SAFI combinations are supported:. IPv4 Unicast with AFI as and SAFI as . IPv6 Unicast with AFI as and SAFI as . Segment Routing Policy for IPv4 Unicast with AFI as and SAFI as 73 (draft-ietf-idr-sr-policy-safi-02 Section 2.1). Segment Routing Policy for IPv6 Unicast with AFI as and SAFI as 73

        Returns: BgpAttributesMpUnreachNlri
        """
        return self._get_property("mp_unreach", BgpAttributesMpUnreachNlri)


class BgpAttributesOtherAttribute(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flag_optional": {"type": bool},
        "flag_transitive": {"type": bool},
        "flag_partial": {"type": bool},
        "flag_extended_length": {"type": bool},
        "type": {
            "type": int,
            "format": "uint32",
        },
        "raw_value": {
            "type": str,
            "format": "hex",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("type", "raw_value")  # type: tuple(str)

    _DEFAULTS = {
        "flag_optional": False,
        "flag_transitive": False,
        "flag_partial": False,
        "flag_extended_length": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flag_optional=False,
        flag_transitive=False,
        flag_partial=False,
        flag_extended_length=False,
        type=None,
        raw_value=None,
    ):
        super(BgpAttributesOtherAttribute, self).__init__()
        self._parent = parent
        self._set_property("flag_optional", flag_optional)
        self._set_property("flag_transitive", flag_transitive)
        self._set_property("flag_partial", flag_partial)
        self._set_property("flag_extended_length", flag_extended_length)
        self._set_property("type", type)
        self._set_property("raw_value", raw_value)

    def set(
        self,
        flag_optional=None,
        flag_transitive=None,
        flag_partial=None,
        flag_extended_length=None,
        type=None,
        raw_value=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flag_optional(self):
        # type: () -> bool
        """flag_optional getter

        Optional flag in the BGP attribute.

        Returns: bool
        """
        return self._get_property("flag_optional")

    @flag_optional.setter
    def flag_optional(self, value):
        """flag_optional setter

        Optional flag in the BGP attribute.

        value: bool
        """
        self._set_property("flag_optional", value)

    @property
    def flag_transitive(self):
        # type: () -> bool
        """flag_transitive getter

        Transitive flag in the BGP attribute.

        Returns: bool
        """
        return self._get_property("flag_transitive")

    @flag_transitive.setter
    def flag_transitive(self, value):
        """flag_transitive setter

        Transitive flag in the BGP attribute.

        value: bool
        """
        self._set_property("flag_transitive", value)

    @property
    def flag_partial(self):
        # type: () -> bool
        """flag_partial getter

        Partial flag in the BGP attribute.

        Returns: bool
        """
        return self._get_property("flag_partial")

    @flag_partial.setter
    def flag_partial(self, value):
        """flag_partial setter

        Partial flag in the BGP attribute.

        value: bool
        """
        self._set_property("flag_partial", value)

    @property
    def flag_extended_length(self):
        # type: () -> bool
        """flag_extended_length getter

        Extended length flag in the BGP attribute.

        Returns: bool
        """
        return self._get_property("flag_extended_length")

    @flag_extended_length.setter
    def flag_extended_length(self, value):
        """flag_extended_length setter

        Extended length flag in the BGP attribute.

        value: bool
        """
        self._set_property("flag_extended_length", value)

    @property
    def type(self):
        # type: () -> int
        """type getter

        The value of the Type field in the attribute.

        Returns: int
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        The value of the Type field in the attribute.

        value: int
        """
        if value is None:
            raise TypeError("Cannot set required property type as None")
        self._set_property("type", value)

    @property
    def raw_value(self):
        # type: () -> str
        """raw_value getter

        Contents of the value field the contents after the initial two bytes containing the Flags and Type field of the attribute in hex bytes. It includes the contents of length of the extended length field if included.

        Returns: str
        """
        return self._get_property("raw_value")

    @raw_value.setter
    def raw_value(self, value):
        """raw_value setter

        Contents of the value field the contents after the initial two bytes containing the Flags and Type field of the attribute in hex bytes. It includes the contents of length of the extended length field if included.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property raw_value as None")
        self._set_property("raw_value", value)


class BgpAttributesOtherAttributeIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesOtherAttributeIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAttributesOtherAttribute]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAttributesOtherAttributeIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAttributesOtherAttribute
        return self._next()

    def next(self):
        # type: () -> BgpAttributesOtherAttribute
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAttributesOtherAttribute):
            raise Exception("Item is not an instance of BgpAttributesOtherAttribute")

    def otherattribute(
        self,
        flag_optional=False,
        flag_transitive=False,
        flag_partial=False,
        flag_extended_length=False,
        type=None,
        raw_value=None,
    ):
        # type: (bool,bool,bool,bool,int,str) -> BgpAttributesOtherAttributeIter
        """Factory method that creates an instance of the BgpAttributesOtherAttribute class

        One unknown attribute stored as hex bytes.

        Returns: BgpAttributesOtherAttributeIter
        """
        item = BgpAttributesOtherAttribute(
            parent=self._parent,
            flag_optional=flag_optional,
            flag_transitive=flag_transitive,
            flag_partial=flag_partial,
            flag_extended_length=flag_extended_length,
            type=type,
            raw_value=raw_value,
        )
        self._add(item)
        return self

    def add(
        self,
        flag_optional=False,
        flag_transitive=False,
        flag_partial=False,
        flag_extended_length=False,
        type=None,
        raw_value=None,
    ):
        # type: (bool,bool,bool,bool,int,str) -> BgpAttributesOtherAttribute
        """Add method that creates and returns an instance of the BgpAttributesOtherAttribute class

        One unknown attribute stored as hex bytes.

        Returns: BgpAttributesOtherAttribute
        """
        item = BgpAttributesOtherAttribute(
            parent=self._parent,
            flag_optional=flag_optional,
            flag_transitive=flag_transitive,
            flag_partial=flag_partial,
            flag_extended_length=flag_extended_length,
            type=type,
            raw_value=raw_value,
        )
        self._add(item)
        return item


class BgpAttributesAsPath(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "four_byte_as_path",
                "two_byte_as_path",
            ],
        },
        "four_byte_as_path": {"type": "BgpAttributesAsPathFourByteAsPath"},
        "two_byte_as_path": {"type": "BgpAttributesAsPathTwoByteAsPath"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "four_byte_as_path",
    }  # type: Dict[str, Union(type)]

    FOUR_BYTE_AS_PATH = "four_byte_as_path"  # type: str
    TWO_BYTE_AS_PATH = "two_byte_as_path"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesAsPath, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def four_byte_as_path(self):
        # type: () -> BgpAttributesAsPathFourByteAsPath
        """Factory property that returns an instance of the BgpAttributesAsPathFourByteAsPath class

        AS Paths with byte AS numbers can be exchanged only if both BGP speakers support byte AS number extensions.

        Returns: BgpAttributesAsPathFourByteAsPath
        """
        return self._get_property(
            "four_byte_as_path",
            BgpAttributesAsPathFourByteAsPath,
            self,
            "four_byte_as_path",
        )

    @property
    def two_byte_as_path(self):
        # type: () -> BgpAttributesAsPathTwoByteAsPath
        """Factory property that returns an instance of the BgpAttributesAsPathTwoByteAsPath class

        AS Paths with byte AS numbers is used when any of the two scenarios occur :. An old BGP speaker and new BGP speaker are sending BGP Updates to one another.. Two old BGP speakers are sending BGP Updates to one another.

        Returns: BgpAttributesAsPathTwoByteAsPath
        """
        return self._get_property(
            "two_byte_as_path",
            BgpAttributesAsPathTwoByteAsPath,
            self,
            "two_byte_as_path",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["four_byte_as_path"], Literal["two_byte_as_path"]]
        """choice getter

        TBD

        Returns: Union[Literal["four_byte_as_path"], Literal["two_byte_as_path"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["four_byte_as_path"], Literal["two_byte_as_path"]]
        """
        self._set_property("choice", value)


class BgpAttributesAsPathFourByteAsPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "segments": {"type": "BgpAttributesFourByteAsPathSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesAsPathFourByteAsPath, self).__init__()
        self._parent = parent

    @property
    def segments(self):
        # type: () -> BgpAttributesFourByteAsPathSegmentIter
        """segments getter

        The AS path segments containing byte AS numbers to be added in the AS Path attribute. By default, an empty AS path should always be included and for EBGP at minimum the local AS number should be present in the AS Path.

        Returns: BgpAttributesFourByteAsPathSegmentIter
        """
        return self._get_property(
            "segments",
            BgpAttributesFourByteAsPathSegmentIter,
            self._parent,
            self._choice,
        )


class BgpAttributesFourByteAsPathSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "as_seq",
                "as_set",
                "as_confed_seq",
                "as_confed_set",
            ],
        },
        "as_numbers": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "type": "as_seq",
    }  # type: Dict[str, Union(type)]

    AS_SEQ = "as_seq"  # type: str
    AS_SET = "as_set"  # type: str
    AS_CONFED_SEQ = "as_confed_seq"  # type: str
    AS_CONFED_SET = "as_confed_set"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type="as_seq", as_numbers=None):
        super(BgpAttributesFourByteAsPathSegment, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_numbers", as_numbers)

    def set(self, type=None, as_numbers=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """type getter

        AS sequence is the most common type of AS_PATH, it contains the list. of ASNs starting with the most recent ASN being added read from left. to right.. The other three AS_PATH types are used for Confederations. AS_SET is the type of AS_PATH attribute that summarizes routes using. using the aggregate-address command, allowing AS_PATHs to be summarized. in the update as well.. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most. recent ASN to be added reading left to right. AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent. in BGP Updates.

        Returns: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        AS sequence is the most common type of AS_PATH, it contains the list. of ASNs starting with the most recent ASN being added read from left. to right.. The other three AS_PATH types are used for Confederations. AS_SET is the type of AS_PATH attribute that summarizes routes using. using the aggregate-address command, allowing AS_PATHs to be summarized. in the update as well.. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most. recent ASN to be added reading left to right. AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent. in BGP Updates.

        value: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        self._set_property("type", value)

    @property
    def as_numbers(self):
        # type: () -> List[int]
        """as_numbers getter

        The byte AS numbers in this AS path segment.

        Returns: List[int]
        """
        return self._get_property("as_numbers")

    @as_numbers.setter
    def as_numbers(self, value):
        """as_numbers setter

        The byte AS numbers in this AS path segment.

        value: List[int]
        """
        self._set_property("as_numbers", value)


class BgpAttributesFourByteAsPathSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesFourByteAsPathSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAttributesFourByteAsPathSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAttributesFourByteAsPathSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAttributesFourByteAsPathSegment
        return self._next()

    def next(self):
        # type: () -> BgpAttributesFourByteAsPathSegment
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAttributesFourByteAsPathSegment):
            raise Exception(
                "Item is not an instance of BgpAttributesFourByteAsPathSegment"
            )

    def fourbyteaspathsegment(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAttributesFourByteAsPathSegmentIter
        """Factory method that creates an instance of the BgpAttributesFourByteAsPathSegment class

        Configuration for single BGP AS path segment containing byte AS numbers.

        Returns: BgpAttributesFourByteAsPathSegmentIter
        """
        item = BgpAttributesFourByteAsPathSegment(
            parent=self._parent, type=type, as_numbers=as_numbers
        )
        self._add(item)
        return self

    def add(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAttributesFourByteAsPathSegment
        """Add method that creates and returns an instance of the BgpAttributesFourByteAsPathSegment class

        Configuration for single BGP AS path segment containing byte AS numbers.

        Returns: BgpAttributesFourByteAsPathSegment
        """
        item = BgpAttributesFourByteAsPathSegment(
            parent=self._parent, type=type, as_numbers=as_numbers
        )
        self._add(item)
        return item


class BgpAttributesAsPathTwoByteAsPath(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "segments": {"type": "BgpAttributesTwoByteAsPathSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesAsPathTwoByteAsPath, self).__init__()
        self._parent = parent

    @property
    def segments(self):
        # type: () -> BgpAttributesTwoByteAsPathSegmentIter
        """segments getter

        The AS path segments containing byte AS numbers to be added in the AS Path attribute. By default, an empty AS path should always be included and for EBGP the sender's AS number should be prepended to the AS Path.

        Returns: BgpAttributesTwoByteAsPathSegmentIter
        """
        return self._get_property(
            "segments",
            BgpAttributesTwoByteAsPathSegmentIter,
            self._parent,
            self._choice,
        )


class BgpAttributesTwoByteAsPathSegment(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "type": {
            "type": str,
            "enum": [
                "as_seq",
                "as_set",
                "as_confed_seq",
                "as_confed_set",
            ],
        },
        "as_numbers": {
            "type": list,
            "itemtype": int,
            "itemformat": "uint32",
            "maximum": 65535,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "type": "as_seq",
    }  # type: Dict[str, Union(type)]

    AS_SEQ = "as_seq"  # type: str
    AS_SET = "as_set"  # type: str
    AS_CONFED_SEQ = "as_confed_seq"  # type: str
    AS_CONFED_SET = "as_confed_set"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, type="as_seq", as_numbers=None):
        super(BgpAttributesTwoByteAsPathSegment, self).__init__()
        self._parent = parent
        self._set_property("type", type)
        self._set_property("as_numbers", as_numbers)

    def set(self, type=None, as_numbers=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def type(self):
        # type: () -> Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """type getter

        AS sequence is the most common type of AS_PATH, it contains the list. of ASNs starting with the most recent ASN being added read from left. to right.. The other three AS_PATH types are used for Confederations. AS_SET is the type of AS_PATH attribute that summarizes routes using. using the aggregate-address command, allowing AS_PATHs to be summarized. in the update as well.. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most. recent ASN to be added reading left to right. AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent. in BGP Updates.

        Returns: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        return self._get_property("type")

    @type.setter
    def type(self, value):
        """type setter

        AS sequence is the most common type of AS_PATH, it contains the list. of ASNs starting with the most recent ASN being added read from left. to right.. The other three AS_PATH types are used for Confederations. AS_SET is the type of AS_PATH attribute that summarizes routes using. using the aggregate-address command, allowing AS_PATHs to be summarized. in the update as well.. AS_CONFED_SEQ gives the list of ASNs in the path starting with the most. recent ASN to be added reading left to right. AS_CONFED_SET will allow summarization of multiple AS PATHs to be sent. in BGP Updates.

        value: Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]]
        """
        self._set_property("type", value)

    @property
    def as_numbers(self):
        # type: () -> List[int]
        """as_numbers getter

        The byte AS numbers in this AS path segment.

        Returns: List[int]
        """
        return self._get_property("as_numbers")

    @as_numbers.setter
    def as_numbers(self, value):
        """as_numbers setter

        The byte AS numbers in this AS path segment.

        value: List[int]
        """
        self._set_property("as_numbers", value)


class BgpAttributesTwoByteAsPathSegmentIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesTwoByteAsPathSegmentIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAttributesTwoByteAsPathSegment]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAttributesTwoByteAsPathSegmentIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAttributesTwoByteAsPathSegment
        return self._next()

    def next(self):
        # type: () -> BgpAttributesTwoByteAsPathSegment
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAttributesTwoByteAsPathSegment):
            raise Exception(
                "Item is not an instance of BgpAttributesTwoByteAsPathSegment"
            )

    def twobyteaspathsegment(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAttributesTwoByteAsPathSegmentIter
        """Factory method that creates an instance of the BgpAttributesTwoByteAsPathSegment class

        Configuration for single BGP AS path segment containing byte AS numbers.

        Returns: BgpAttributesTwoByteAsPathSegmentIter
        """
        item = BgpAttributesTwoByteAsPathSegment(
            parent=self._parent, type=type, as_numbers=as_numbers
        )
        self._add(item)
        return self

    def add(self, type="as_seq", as_numbers=None):
        # type: (Union[Literal["as_confed_seq"], Literal["as_confed_set"], Literal["as_seq"], Literal["as_set"]],List[int]) -> BgpAttributesTwoByteAsPathSegment
        """Add method that creates and returns an instance of the BgpAttributesTwoByteAsPathSegment class

        Configuration for single BGP AS path segment containing byte AS numbers.

        Returns: BgpAttributesTwoByteAsPathSegment
        """
        item = BgpAttributesTwoByteAsPathSegment(
            parent=self._parent, type=type, as_numbers=as_numbers
        )
        self._add(item)
        return item


class BgpAttributesAs4Path(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "segments": {"type": "BgpAttributesFourByteAsPathSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesAs4Path, self).__init__()
        self._parent = parent

    @property
    def segments(self):
        # type: () -> BgpAttributesFourByteAsPathSegmentIter
        """segments getter

        The AS path segments containing byte AS numbers to be added in the AS4_PATH attribute.

        Returns: BgpAttributesFourByteAsPathSegmentIter
        """
        return self._get_property(
            "segments",
            BgpAttributesFourByteAsPathSegmentIter,
            self._parent,
            self._choice,
        )


class BgpAttributesNextHop(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "ipv4",
                "ipv6",
                "ipv6_two_addresses",
            ],
        },
        "ipv4": {
            "type": str,
            "format": "ipv4",
        },
        "ipv6": {
            "type": str,
            "format": "ipv6",
        },
        "ipv6_two_addresses": {"type": "BgpAttributesNextHopIpv6TwoAddresses"},
    }  # type: Dict[str, str]

    _REQUIRED = ("choice",)  # type: tuple(str)

    _DEFAULTS = {
        "ipv4": "0.0.0.0",
        "ipv6": "0::0",
    }  # type: Dict[str, Union(type)]

    IPV4 = "ipv4"  # type: str
    IPV6 = "ipv6"  # type: str
    IPV6_TWO_ADDRESSES = "ipv6_two_addresses"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None, ipv4="0.0.0.0", ipv6="0::0"):
        super(BgpAttributesNextHop, self).__init__()
        self._parent = parent
        self._set_property("ipv4", ipv4)
        self._set_property("ipv6", ipv6)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, ipv4=None, ipv6=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def ipv6_two_addresses(self):
        # type: () -> BgpAttributesNextHopIpv6TwoAddresses
        """Factory property that returns an instance of the BgpAttributesNextHopIpv6TwoAddresses class

        There is specific scenario in which it is possible to receive Global and Link Local address in the Next Hop field in MP_REACH attribute or in the NEXT_HOP attribute(RFC2545: Section 3).

        Returns: BgpAttributesNextHopIpv6TwoAddresses
        """
        return self._get_property(
            "ipv6_two_addresses",
            BgpAttributesNextHopIpv6TwoAddresses,
            self,
            "ipv6_two_addresses",
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["ipv4"], Literal["ipv6"], Literal["ipv6_two_addresses"]]
        """choice getter

        The type of the next HOP.

        Returns: Union[Literal["ipv4"], Literal["ipv6"], Literal["ipv6_two_addresses"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of the next HOP.

        value: Union[Literal["ipv4"], Literal["ipv6"], Literal["ipv6_two_addresses"]]
        """
        if value is None:
            raise TypeError("Cannot set required property choice as None")
        self._set_property("choice", value)

    @property
    def ipv4(self):
        # type: () -> str
        """ipv4 getter

        The byte IPv4 address of the next-hop from which the route was received.

        Returns: str
        """
        return self._get_property("ipv4")

    @ipv4.setter
    def ipv4(self, value):
        """ipv4 setter

        The byte IPv4 address of the next-hop from which the route was received.

        value: str
        """
        self._set_property("ipv4", value, "ipv4")

    @property
    def ipv6(self):
        # type: () -> str
        """ipv6 getter

        The 16 byte IPv6 address of the next-hop from which the route was received.

        Returns: str
        """
        return self._get_property("ipv6")

    @ipv6.setter
    def ipv6(self, value):
        """ipv6 setter

        The 16 byte IPv6 address of the next-hop from which the route was received.

        value: str
        """
        self._set_property("ipv6", value, "ipv6")


class BgpAttributesNextHopIpv6TwoAddresses(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "first": {
            "type": str,
            "format": "ipv6",
        },
        "second": {
            "type": str,
            "format": "ipv6",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "first": "0::0",
        "second": "0::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, first="0::0", second="0::0"):
        super(BgpAttributesNextHopIpv6TwoAddresses, self).__init__()
        self._parent = parent
        self._set_property("first", first)
        self._set_property("second", second)

    def set(self, first=None, second=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def first(self):
        # type: () -> str
        """first getter

        The first IPv6 next hop in the 32 byte IPv6 Next Hop.

        Returns: str
        """
        return self._get_property("first")

    @first.setter
    def first(self, value):
        """first setter

        The first IPv6 next hop in the 32 byte IPv6 Next Hop.

        value: str
        """
        self._set_property("first", value)

    @property
    def second(self):
        # type: () -> str
        """second getter

        The second IPv6 next hop in the 32 byte IPv6 Next Hop.

        Returns: str
        """
        return self._get_property("second")

    @second.setter
    def second(self, value):
        """second setter

        The second IPv6 next hop in the 32 byte IPv6 Next Hop.

        value: str
        """
        self._set_property("second", value)


class BgpAttributesMultiExitDiscriminator(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=0):
        super(BgpAttributesMultiExitDiscriminator, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        The multi exit discriminator (MED) value used for route selection sent to the peer.

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        The multi exit discriminator (MED) value used for route selection sent to the peer.

        value: int
        """
        self._set_property("value", value)


class BgpAttributesLocalPreference(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": 100,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=100):
        super(BgpAttributesLocalPreference, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        Value to be set in the LOCAL_PREFERENCE attribute The multi exit discriminator (MED) value used for route selection sent to the peer.

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value to be set in the LOCAL_PREFERENCE attribute The multi exit discriminator (MED) value used for route selection sent to the peer.

        value: int
        """
        self._set_property("value", value)


class BgpAttributesAggregator(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "four_byte_as",
                "two_byte_as",
            ],
        },
        "four_byte_as": {
            "type": int,
            "format": "uint32",
        },
        "two_byte_as": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "four_byte_as",
        "four_byte_as": 65536,
        "two_byte_as": 1,
        "ipv4_address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    FOUR_BYTE_AS = "four_byte_as"  # type: str
    TWO_BYTE_AS = "two_byte_as"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        choice=None,
        four_byte_as=65536,
        two_byte_as=1,
        ipv4_address="0.0.0.0",
    ):
        super(BgpAttributesAggregator, self).__init__()
        self._parent = parent
        self._set_property("four_byte_as", four_byte_as)
        self._set_property("two_byte_as", two_byte_as)
        self._set_property("ipv4_address", ipv4_address)
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    def set(self, four_byte_as=None, two_byte_as=None, ipv4_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def choice(self):
        # type: () -> Union[Literal["four_byte_as"], Literal["two_byte_as"]]
        """choice getter

        TBD

        Returns: Union[Literal["four_byte_as"], Literal["two_byte_as"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        TBD

        value: Union[Literal["four_byte_as"], Literal["two_byte_as"]]
        """
        self._set_property("choice", value)

    @property
    def four_byte_as(self):
        # type: () -> int
        """four_byte_as getter

        The value of the byte AS number of the BGP speaker which aggregated the route. If the value of the AS number is less than octets 65535 or less), the AS4_AGGREGATOR object should not be sent.

        Returns: int
        """
        return self._get_property("four_byte_as")

    @four_byte_as.setter
    def four_byte_as(self, value):
        """four_byte_as setter

        The value of the byte AS number of the BGP speaker which aggregated the route. If the value of the AS number is less than octets 65535 or less), the AS4_AGGREGATOR object should not be sent.

        value: int
        """
        self._set_property("four_byte_as", value, "four_byte_as")

    @property
    def two_byte_as(self):
        # type: () -> int
        """two_byte_as getter

        The value of the byte AS number of the BGP speaker which aggregated the route.

        Returns: int
        """
        return self._get_property("two_byte_as")

    @two_byte_as.setter
    def two_byte_as(self, value):
        """two_byte_as setter

        The value of the byte AS number of the BGP speaker which aggregated the route.

        value: int
        """
        self._set_property("two_byte_as", value, "two_byte_as")

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address of the BGP speaker which aggregated the route.

        Returns: str
        """
        return self._get_property("ipv4_address")

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address of the BGP speaker which aggregated the route.

        value: str
        """
        self._set_property("ipv4_address", value)


class BgpAttributesAs4Aggregator(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_num": {
            "type": int,
            "format": "uint32",
        },
        "ipv4_address": {
            "type": str,
            "format": "ipv4",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "ipv4_address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, as_num=None, ipv4_address="0.0.0.0"):
        super(BgpAttributesAs4Aggregator, self).__init__()
        self._parent = parent
        self._set_property("as_num", as_num)
        self._set_property("ipv4_address", ipv4_address)

    def set(self, as_num=None, ipv4_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def as_num(self):
        # type: () -> int
        """as_num getter

        The value of the byte AS number of the BGP speaker which aggregated the route.

        Returns: int
        """
        return self._get_property("as_num")

    @as_num.setter
    def as_num(self, value):
        """as_num setter

        The value of the byte AS number of the BGP speaker which aggregated the route.

        value: int
        """
        self._set_property("as_num", value)

    @property
    def ipv4_address(self):
        # type: () -> str
        """ipv4_address getter

        The IPv4 address of the BGP speaker which aggregated the route.

        Returns: str
        """
        return self._get_property("ipv4_address")

    @ipv4_address.setter
    def ipv4_address(self, value):
        """ipv4_address setter

        The IPv4 address of the BGP speaker which aggregated the route.

        value: str
        """
        self._set_property("ipv4_address", value)


class BgpAttributesCommunity(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "custom_community",
                "no_export",
                "no_advertised",
                "no_export_subconfed",
                "llgr_stale",
                "no_llgr",
            ],
        },
        "custom_community": {"type": "BgpAttributesCustomCommunity"},
    }  # type: Dict[str, str]

    _REQUIRED = ("choice",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    CUSTOM_COMMUNITY = "custom_community"  # type: str
    NO_EXPORT = "no_export"  # type: str
    NO_ADVERTISED = "no_advertised"  # type: str
    NO_EXPORT_SUBCONFED = "no_export_subconfed"  # type: str
    LLGR_STALE = "llgr_stale"  # type: str
    NO_LLGR = "no_llgr"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesCommunity, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def custom_community(self):
        # type: () -> BgpAttributesCustomCommunity
        """Factory property that returns an instance of the BgpAttributesCustomCommunity class

        User defined COMMUNITY attribute containing byte AS and custom byte value defined by the administrator of the domain.

        Returns: BgpAttributesCustomCommunity
        """
        return self._get_property(
            "custom_community", BgpAttributesCustomCommunity, self, "custom_community"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["custom_community"], Literal["llgr_stale"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """choice getter

        The type of community AS number.

        Returns: Union[Literal["custom_community"], Literal["llgr_stale"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of community AS number.

        value: Union[Literal["custom_community"], Literal["llgr_stale"], Literal["no_advertised"], Literal["no_export"], Literal["no_export_subconfed"], Literal["no_llgr"]]
        """
        if value is None:
            raise TypeError("Cannot set required property choice as None")
        self._set_property("choice", value)


class BgpAttributesCustomCommunity(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "as_number": {
            "type": int,
            "format": "uint32",
            "maximum": 65535,
        },
        "custom": {
            "type": str,
            "format": "hex",
            "maxLength": 4,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "as_number": 0,
        "custom": "0000",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, as_number=0, custom="0000"):
        super(BgpAttributesCustomCommunity, self).__init__()
        self._parent = parent
        self._set_property("as_number", as_number)
        self._set_property("custom", custom)

    def set(self, as_number=None, custom=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def as_number(self):
        # type: () -> int
        """as_number getter

        First two octets of the community value containing 2 byte AS number.

        Returns: int
        """
        return self._get_property("as_number")

    @as_number.setter
    def as_number(self, value):
        """as_number setter

        First two octets of the community value containing 2 byte AS number.

        value: int
        """
        self._set_property("as_number", value)

    @property
    def custom(self):
        # type: () -> str
        """custom getter

        Last two octets of the community value in hex. If user provides less than hex bytes, it should be left-padded with 0s.

        Returns: str
        """
        return self._get_property("custom")

    @custom.setter
    def custom(self, value):
        """custom setter

        Last two octets of the community value in hex. If user provides less than hex bytes, it should be left-padded with 0s.

        value: str
        """
        self._set_property("custom", value)


class BgpAttributesCommunityIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = True

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesCommunityIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAttributesCommunity, BgpAttributesCustomCommunity]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAttributesCommunityIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAttributesCommunity
        return self._next()

    def next(self):
        # type: () -> BgpAttributesCommunity
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAttributesCommunity):
            raise Exception("Item is not an instance of BgpAttributesCommunity")

    def community(self):
        # type: () -> BgpAttributesCommunityIter
        """Factory method that creates an instance of the BgpAttributesCommunity class

        The COMMUNITY attribute provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is 32-bit number which is broken into 16-bit AS number and . 16-bit custom value or it contains some pre-defined well known values.

        Returns: BgpAttributesCommunityIter
        """
        item = BgpAttributesCommunity(parent=self._parent, choice=self._choice)
        self._add(item)
        return self

    def add(self):
        # type: () -> BgpAttributesCommunity
        """Add method that creates and returns an instance of the BgpAttributesCommunity class

        The COMMUNITY attribute provide additional capability for tagging routes and for modifying BGP routing policy on upstream and downstream routers. BGP community is 32-bit number which is broken into 16-bit AS number and . 16-bit custom value or it contains some pre-defined well known values.

        Returns: BgpAttributesCommunity
        """
        item = BgpAttributesCommunity(parent=self._parent, choice=self._choice)
        self._add(item)
        return item

    def custom_community(self, as_number=0, custom="0000"):
        # type: (int,str) -> BgpAttributesCommunityIter
        """Factory method that creates an instance of the BgpAttributesCustomCommunity class

        User defined COMMUNITY attribute containing byte AS and custom byte value defined by the administrator of the domain.

        Returns: BgpAttributesCommunityIter
        """
        item = BgpAttributesCommunity()
        item.custom_community
        item.choice = "custom_community"
        self._add(item)
        return self


class BgpAttributesOriginatorId(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "format": "ipv4",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value="0.0.0.0"):
        super(BgpAttributesOriginatorId, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        The value of the originator's Router Id.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        The value of the originator's Router Id.

        value: str
        """
        self._set_property("value", value)


class BgpAttributesTunnelEncapsulation(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "sr_policy",
            ],
        },
        "sr_policy": {"type": "BgpAttributesSegmentRoutingPolicy"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "sr_policy",
    }  # type: Dict[str, Union(type)]

    SR_POLICY = "sr_policy"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesTunnelEncapsulation, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def sr_policy(self):
        # type: () -> BgpAttributesSegmentRoutingPolicy
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicy class

        Optional Segment Routing Policy information as defined in draft-ietf-idr-sr-policy-safi-02.. This information is carried in TUNNEL_ENCAPSULATION attribute with type set to SR Policy (15).

        Returns: BgpAttributesSegmentRoutingPolicy
        """
        return self._get_property(
            "sr_policy", BgpAttributesSegmentRoutingPolicy, self, "sr_policy"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["sr_policy"]]
        """choice getter

        Identifies type of tunnel. The field contains values from the IANA registry "BGP Tunnel Encapsulation Attribute Tunnel Types".

        Returns: Union[Literal["sr_policy"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Identifies type of tunnel. The field contains values from the IANA registry "BGP Tunnel Encapsulation Attribute Tunnel Types".

        value: Union[Literal["sr_policy"]]
        """
        self._set_property("choice", value)


class BgpAttributesSegmentRoutingPolicy(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "binding_segment_identifier": {"type": "BgpAttributesBsid"},
        "srv6_binding_segment_identifier": {"type": "BgpAttributesSrv6BsidIter"},
        "preference": {"type": "BgpAttributesSrPolicyPreference"},
        "priority": {"type": "BgpAttributesSrPolicyPriority"},
        "policy_name": {"type": "BgpAttributesSrPolicyPolicyName"},
        "policy_candidate_name": {"type": "BgpAttributesSrPolicyPolicyCandidateName"},
        "explicit_null_label_policy": {
            "type": "BgpAttributesSrPolicyExplicitNullPolicy"
        },
        "segment_list": {"type": "BgpAttributesSrPolicySegmentListIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesSegmentRoutingPolicy, self).__init__()
        self._parent = parent

    @property
    def binding_segment_identifier(self):
        # type: () -> BgpAttributesBsid
        """binding_segment_identifier getter

        The Binding Segment Identifier is an optional sub-tlv of type 13 that can be sent with SR Policy Tunnel Encapsulation attribute.. When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID if this value (label in MPLS, IPv6 address in SRv6) is available.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 . It is recommended that if SRv6 Binding SID is desired to be signalled, the SRv6 Binding SID sub-TLV that enables the specification of the SRv6 Endpoint Behavior should be used.The Binding Segment Identifier is an optional sub-tlv of type 13 that can be sent with SR Policy Tunnel Encapsulation attribute.. When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID if this value (label in MPLS, IPv6 address in SRv6) is available.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 . It is recommended that if SRv6 Binding SID is desired to be signalled, the SRv6 Binding SID sub-TLV that enables the specification of the SRv6 Endpoint Behavior should be used.The Binding Segment Identifier is an optional sub-tlv of type 13 that can be sent with SR Policy Tunnel Encapsulation attribute.. When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID if this value (label in MPLS, IPv6 address in SRv6) is available.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 . It is recommended that if SRv6 Binding SID is desired to be signalled, the SRv6 Binding SID sub-TLV that enables the specification of the SRv6 Endpoint Behavior should be used.The Binding Segment Identifier is an optional sub-tlv of type 13 that can be sent with SR Policy Tunnel Encapsulation attribute.. When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID if this value (label in MPLS, IPv6 address in SRv6) is available.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 . It is recommended that if SRv6 Binding SID is desired to be signalled, the SRv6 Binding SID sub-TLV that enables the specification of the SRv6 Endpoint Behavior should be used.

        Returns: BgpAttributesBsid
        """
        return self._get_property("binding_segment_identifier", BgpAttributesBsid)

    @property
    def srv6_binding_segment_identifier(self):
        # type: () -> BgpAttributesSrv6BsidIter
        """srv6_binding_segment_identifier getter

        The SRv6 Binding SID sub-TLV is an optional sub-TLV of type 20 that is used to signal the SRv6 Binding SID. related information of an SR Policy candidate path. - More than one SRv6 Binding SID sub-TLVs MAY be signaled in the same SR Policy encoding to indicate one. or more SRv6 SIDs, each with potentially different SRv6 Endpoint Behaviors to be instantiated.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.3

        Returns: BgpAttributesSrv6BsidIter
        """
        return self._get_property(
            "srv6_binding_segment_identifier",
            BgpAttributesSrv6BsidIter,
            self._parent,
            self._choice,
        )

    @property
    def preference(self):
        # type: () -> BgpAttributesSrPolicyPreference
        """preference getter

        Optional Preference sub-tlv (Type 12) is used to select the best candidate path for an SR Policy.. It is defined in Section 2.4.1 of draft-ietf-idr-sr-policy-safi-02 .. Optional Preference sub-tlv (Type 12) is used to select the best candidate path for an SR Policy.. It is defined in Section 2.4.1 of draft-ietf-idr-sr-policy-safi-02 .. Optional Preference sub-tlv (Type 12) is used to select the best candidate path for an SR Policy.. It is defined in Section 2.4.1 of draft-ietf-idr-sr-policy-safi-02 .. Optional Preference sub-tlv (Type 12) is used to select the best candidate path for an SR Policy.. It is defined in Section 2.4.1 of draft-ietf-idr-sr-policy-safi-02 ..

        Returns: BgpAttributesSrPolicyPreference
        """
        return self._get_property("preference", BgpAttributesSrPolicyPreference)

    @property
    def priority(self):
        # type: () -> BgpAttributesSrPolicyPriority
        """priority getter

        Optional Priority sub-tlv (Type 15) used to select the order in which policies should be re-computed.. It is defined in Section 2.4.6 of draft-ietf-idr-sr-policy-safi-02 .Optional Priority sub-tlv (Type 15) used to select the order in which policies should be re-computed.. It is defined in Section 2.4.6 of draft-ietf-idr-sr-policy-safi-02 .Optional Priority sub-tlv (Type 15) used to select the order in which policies should be re-computed.. It is defined in Section 2.4.6 of draft-ietf-idr-sr-policy-safi-02 .Optional Priority sub-tlv (Type 15) used to select the order in which policies should be re-computed.. It is defined in Section 2.4.6 of draft-ietf-idr-sr-policy-safi-02 .

        Returns: BgpAttributesSrPolicyPriority
        """
        return self._get_property("priority", BgpAttributesSrPolicyPriority)

    @property
    def policy_name(self):
        # type: () -> BgpAttributesSrPolicyPolicyName
        """policy_name getter

        Optional Policy Name sub-tlv (Type 130) which carries the symbolic name for the SR Policy for which the candidate path is being advertised for debugging. - It is defined in Section 2.4.8 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Name sub-tlv (Type 130) which carries the symbolic name for the SR Policy for which the candidate path is being advertised for debugging. - It is defined in Section 2.4.8 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Name sub-tlv (Type 130) which carries the symbolic name for the SR Policy for which the candidate path is being advertised for debugging. - It is defined in Section 2.4.8 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Name sub-tlv (Type 130) which carries the symbolic name for the SR Policy for which the candidate path is being advertised for debugging. - It is defined in Section 2.4.8 of draft-ietf-idr-sr-policy-safi-02 .

        Returns: BgpAttributesSrPolicyPolicyName
        """
        return self._get_property("policy_name", BgpAttributesSrPolicyPolicyName)

    @property
    def policy_candidate_name(self):
        # type: () -> BgpAttributesSrPolicyPolicyCandidateName
        """policy_candidate_name getter

        Optional Policy Candidate Path Name sub-tlv (Type 129) which carries the symbolic name for the SR Policy candidate path for debugging. - It is defined in Section 2.4.7 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Candidate Path Name sub-tlv (Type 129) which carries the symbolic name for the SR Policy candidate path for debugging. - It is defined in Section 2.4.7 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Candidate Path Name sub-tlv (Type 129) which carries the symbolic name for the SR Policy candidate path for debugging. - It is defined in Section 2.4.7 of draft-ietf-idr-sr-policy-safi-02 .Optional Policy Candidate Path Name sub-tlv (Type 129) which carries the symbolic name for the SR Policy candidate path for debugging. - It is defined in Section 2.4.7 of draft-ietf-idr-sr-policy-safi-02 .

        Returns: BgpAttributesSrPolicyPolicyCandidateName
        """
        return self._get_property(
            "policy_candidate_name", BgpAttributesSrPolicyPolicyCandidateName
        )

    @property
    def explicit_null_label_policy(self):
        # type: () -> BgpAttributesSrPolicyExplicitNullPolicy
        """explicit_null_label_policy getter

        This is an optional sub-tlv (Type 14) which indicates whether an Explicit NULL Label must be pushed on an unlabeled IP. packet before other labels for IPv4 or IPv6 flows.. It is defined in Section 2.4.5 of draft-ietf-idr-sr-policy-safi-02.This is an optional sub-tlv (Type 14) which indicates whether an Explicit NULL Label must be pushed on an unlabeled IP. packet before other labels for IPv4 or IPv6 flows.. It is defined in Section 2.4.5 of draft-ietf-idr-sr-policy-safi-02.This is an optional sub-tlv (Type 14) which indicates whether an Explicit NULL Label must be pushed on an unlabeled IP. packet before other labels for IPv4 or IPv6 flows.. It is defined in Section 2.4.5 of draft-ietf-idr-sr-policy-safi-02.This is an optional sub-tlv (Type 14) which indicates whether an Explicit NULL Label must be pushed on an unlabeled IP. packet before other labels for IPv4 or IPv6 flows.. It is defined in Section 2.4.5 of draft-ietf-idr-sr-policy-safi-02.

        Returns: BgpAttributesSrPolicyExplicitNullPolicy
        """
        return self._get_property(
            "explicit_null_label_policy", BgpAttributesSrPolicyExplicitNullPolicy
        )

    @property
    def segment_list(self):
        # type: () -> BgpAttributesSrPolicySegmentListIter
        """segment_list getter

        TBD

        Returns: BgpAttributesSrPolicySegmentListIter
        """
        return self._get_property(
            "segment_list",
            BgpAttributesSrPolicySegmentListIter,
            self._parent,
            self._choice,
        )


class BgpAttributesBsid(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "mpls",
                "srv6",
            ],
        },
        "mpls": {"type": "BgpAttributesBsidMpls"},
        "srv6": {"type": "BgpAttributesBsidSrv6"},
    }  # type: Dict[str, str]

    _REQUIRED = ("choice",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    MPLS = "mpls"  # type: str
    SRV6 = "srv6"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesBsid, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def mpls(self):
        # type: () -> BgpAttributesBsidMpls
        """Factory property that returns an instance of the BgpAttributesBsidMpls class

        When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID defined as MPLS label.The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 .

        Returns: BgpAttributesBsidMpls
        """
        return self._get_property("mpls", BgpAttributesBsidMpls, self, "mpls")

    @property
    def srv6(self):
        # type: () -> BgpAttributesBsidSrv6
        """Factory property that returns an instance of the BgpAttributesBsidSrv6 class

        When the active candidate path has specified Binding Segment Identifier, the SR Policy uses that BSID defined as an IPv6 Address.The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.2 .

        Returns: BgpAttributesBsidSrv6
        """
        return self._get_property("srv6", BgpAttributesBsidSrv6, self, "srv6")

    @property
    def choice(self):
        # type: () -> Union[Literal["mpls"], Literal["srv6"]]
        """choice getter

        The type of Segment Identifier.

        Returns: Union[Literal["mpls"], Literal["srv6"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The type of Segment Identifier.

        value: Union[Literal["mpls"], Literal["srv6"]]
        """
        if value is None:
            raise TypeError("Cannot set required property choice as None")
        self._set_property("choice", value)


class BgpAttributesBsidMpls(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flag_specified_bsid_only": {"type": bool},
        "flag_drop_upon_invalid": {"type": bool},
        "mpls_sid": {"type": "BgpAttributesSidMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "flag_specified_bsid_only": False,
        "flag_drop_upon_invalid": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, flag_specified_bsid_only=False, flag_drop_upon_invalid=False
    ):
        super(BgpAttributesBsidMpls, self).__init__()
        self._parent = parent
        self._set_property("flag_specified_bsid_only", flag_specified_bsid_only)
        self._set_property("flag_drop_upon_invalid", flag_drop_upon_invalid)

    def set(self, flag_specified_bsid_only=None, flag_drop_upon_invalid=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flag_specified_bsid_only(self):
        # type: () -> bool
        """flag_specified_bsid_only getter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_specified_bsid_only")

    @flag_specified_bsid_only.setter
    def flag_specified_bsid_only(self, value):
        """flag_specified_bsid_only setter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        value: bool
        """
        self._set_property("flag_specified_bsid_only", value)

    @property
    def flag_drop_upon_invalid(self):
        # type: () -> bool
        """flag_drop_upon_invalid getter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_drop_upon_invalid")

    @flag_drop_upon_invalid.setter
    def flag_drop_upon_invalid(self, value):
        """flag_drop_upon_invalid setter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        value: bool
        """
        self._set_property("flag_drop_upon_invalid", value)

    @property
    def mpls_sid(self):
        # type: () -> BgpAttributesSidMpls
        """mpls_sid getter

        This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value.

        Returns: BgpAttributesSidMpls
        """
        return self._get_property("mpls_sid", BgpAttributesSidMpls)


class BgpAttributesSidMpls(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "label": {
            "type": int,
            "format": "uint32",
            "maximum": 1048576,
        },
        "traffic_class": {
            "type": int,
            "format": "uint32",
            "maximum": 7,
        },
        "flag_bos": {"type": bool},
        "ttl": {
            "type": int,
            "format": "uint32",
            "maximum": 63,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "label": 16,
        "traffic_class": 0,
        "flag_bos": True,
        "ttl": 63,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, label=16, traffic_class=0, flag_bos=True, ttl=63):
        super(BgpAttributesSidMpls, self).__init__()
        self._parent = parent
        self._set_property("label", label)
        self._set_property("traffic_class", traffic_class)
        self._set_property("flag_bos", flag_bos)
        self._set_property("ttl", ttl)

    def set(self, label=None, traffic_class=None, flag_bos=None, ttl=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def label(self):
        # type: () -> int
        """label getter

        20 bit MPLS Label value.

        Returns: int
        """
        return self._get_property("label")

    @label.setter
    def label(self, value):
        """label setter

        20 bit MPLS Label value.

        value: int
        """
        self._set_property("label", value)

    @property
    def traffic_class(self):
        # type: () -> int
        """traffic_class getter

        3 bits of Traffic Class.

        Returns: int
        """
        return self._get_property("traffic_class")

    @traffic_class.setter
    def traffic_class(self, value):
        """traffic_class setter

        3 bits of Traffic Class.

        value: int
        """
        self._set_property("traffic_class", value)

    @property
    def flag_bos(self):
        # type: () -> bool
        """flag_bos getter

        Bottom of Stack

        Returns: bool
        """
        return self._get_property("flag_bos")

    @flag_bos.setter
    def flag_bos(self, value):
        """flag_bos setter

        Bottom of Stack

        value: bool
        """
        self._set_property("flag_bos", value)

    @property
    def ttl(self):
        # type: () -> int
        """ttl getter

        8 bits Time to Live

        Returns: int
        """
        return self._get_property("ttl")

    @ttl.setter
    def ttl(self, value):
        """ttl setter

        8 bits Time to Live

        value: int
        """
        self._set_property("ttl", value)


class BgpAttributesBsidSrv6(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flag_specified_bsid_only": {"type": bool},
        "flag_drop_upon_invalid": {"type": bool},
        "ipv6_addr": {
            "type": str,
            "format": "ipv6",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "flag_specified_bsid_only": False,
        "flag_drop_upon_invalid": False,
        "ipv6_addr": "0::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flag_specified_bsid_only=False,
        flag_drop_upon_invalid=False,
        ipv6_addr="0::0",
    ):
        super(BgpAttributesBsidSrv6, self).__init__()
        self._parent = parent
        self._set_property("flag_specified_bsid_only", flag_specified_bsid_only)
        self._set_property("flag_drop_upon_invalid", flag_drop_upon_invalid)
        self._set_property("ipv6_addr", ipv6_addr)

    def set(
        self, flag_specified_bsid_only=None, flag_drop_upon_invalid=None, ipv6_addr=None
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flag_specified_bsid_only(self):
        # type: () -> bool
        """flag_specified_bsid_only getter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_specified_bsid_only")

    @flag_specified_bsid_only.setter
    def flag_specified_bsid_only(self, value):
        """flag_specified_bsid_only setter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        value: bool
        """
        self._set_property("flag_specified_bsid_only", value)

    @property
    def flag_drop_upon_invalid(self):
        # type: () -> bool
        """flag_drop_upon_invalid getter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_drop_upon_invalid")

    @flag_drop_upon_invalid.setter
    def flag_drop_upon_invalid(self, value):
        """flag_drop_upon_invalid setter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        value: bool
        """
        self._set_property("flag_drop_upon_invalid", value)

    @property
    def ipv6_addr(self):
        # type: () -> str
        """ipv6_addr getter

        IPv6 address denoting the SRv6 SID.

        Returns: str
        """
        return self._get_property("ipv6_addr")

    @ipv6_addr.setter
    def ipv6_addr(self, value):
        """ipv6_addr setter

        IPv6 address denoting the SRv6 SID.

        value: str
        """
        self._set_property("ipv6_addr", value)


class BgpAttributesSrv6Bsid(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flag_specified_bsid_only": {"type": bool},
        "flag_drop_upon_invalid": {"type": bool},
        "flag_srv6_endpoint_behavior": {"type": bool},
        "ipv6_addr": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_endpoint_behavior": {
            "type": "BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "flag_specified_bsid_only": False,
        "flag_drop_upon_invalid": False,
        "flag_srv6_endpoint_behavior": False,
        "ipv6_addr": "0::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        flag_specified_bsid_only=False,
        flag_drop_upon_invalid=False,
        flag_srv6_endpoint_behavior=False,
        ipv6_addr="0::0",
    ):
        super(BgpAttributesSrv6Bsid, self).__init__()
        self._parent = parent
        self._set_property("flag_specified_bsid_only", flag_specified_bsid_only)
        self._set_property("flag_drop_upon_invalid", flag_drop_upon_invalid)
        self._set_property("flag_srv6_endpoint_behavior", flag_srv6_endpoint_behavior)
        self._set_property("ipv6_addr", ipv6_addr)

    def set(
        self,
        flag_specified_bsid_only=None,
        flag_drop_upon_invalid=None,
        flag_srv6_endpoint_behavior=None,
        ipv6_addr=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flag_specified_bsid_only(self):
        # type: () -> bool
        """flag_specified_bsid_only getter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_specified_bsid_only")

    @flag_specified_bsid_only.setter
    def flag_specified_bsid_only(self, value):
        """flag_specified_bsid_only setter

        S-Flag: This flag encodes the "Specified-BSID-only" behavior. It's usage is described in section 6.2.3 in [RFC9256].

        value: bool
        """
        self._set_property("flag_specified_bsid_only", value)

    @property
    def flag_drop_upon_invalid(self):
        # type: () -> bool
        """flag_drop_upon_invalid getter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        Returns: bool
        """
        return self._get_property("flag_drop_upon_invalid")

    @flag_drop_upon_invalid.setter
    def flag_drop_upon_invalid(self, value):
        """flag_drop_upon_invalid setter

        I-Flag: This flag encodes the "Drop Upon Invalid" behavior. It's usage is described in section 8.2 in [RFC9256].

        value: bool
        """
        self._set_property("flag_drop_upon_invalid", value)

    @property
    def flag_srv6_endpoint_behavior(self):
        # type: () -> bool
        """flag_srv6_endpoint_behavior getter

        B-Flag: This flag, when set, indicates the presence of the SRv6 Endpoint Behavior and SID Structure encoding specified in Section 2.4.4.2.4 of draft-ietf-idr-sr-policy-safi-02.

        Returns: bool
        """
        return self._get_property("flag_srv6_endpoint_behavior")

    @flag_srv6_endpoint_behavior.setter
    def flag_srv6_endpoint_behavior(self, value):
        """flag_srv6_endpoint_behavior setter

        B-Flag: This flag, when set, indicates the presence of the SRv6 Endpoint Behavior and SID Structure encoding specified in Section 2.4.4.2.4 of draft-ietf-idr-sr-policy-safi-02.

        value: bool
        """
        self._set_property("flag_srv6_endpoint_behavior", value)

    @property
    def ipv6_addr(self):
        # type: () -> str
        """ipv6_addr getter

        IPv6 address denoting the SRv6 SID.

        Returns: str
        """
        return self._get_property("ipv6_addr")

    @ipv6_addr.setter
    def ipv6_addr(self, value):
        """ipv6_addr setter

        IPv6 address denoting the SRv6 SID.

        value: str
        """
        self._set_property("ipv6_addr", value)

    @property
    def srv6_endpoint_behavior(self):
        # type: () -> BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure
        """srv6_endpoint_behavior getter

        Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4

        Returns: BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_endpoint_behavior",
            BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure,
        )


class BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure(
    OpenApiObject
):
    __slots__ = "_parent"

    _TYPES = {
        "endpoint_behaviour": {
            "type": str,
            "format": "hex",
            "maxLength": 4,
        },
        "lb_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "ln_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "func_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
        "arg_length": {
            "type": int,
            "format": "uint32",
            "maximum": 128,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "endpoint_behaviour": "ffff",
        "lb_length": 0,
        "ln_length": 0,
        "func_length": 0,
        "arg_length": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self,
        parent=None,
        endpoint_behaviour="ffff",
        lb_length=0,
        ln_length=0,
        func_length=0,
        arg_length=0,
    ):
        super(
            BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure, self
        ).__init__()
        self._parent = parent
        self._set_property("endpoint_behaviour", endpoint_behaviour)
        self._set_property("lb_length", lb_length)
        self._set_property("ln_length", ln_length)
        self._set_property("func_length", func_length)
        self._set_property("arg_length", arg_length)

    def set(
        self,
        endpoint_behaviour=None,
        lb_length=None,
        ln_length=None,
        func_length=None,
        arg_length=None,
    ):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def endpoint_behaviour(self):
        # type: () -> str
        """endpoint_behaviour getter

        This is 2-octet field that is used to specify the SRv6 Endpoint Behavior code point for the SRv6 SID as defined in section 9.2 of [RFC8986]. When set with the value 0xFFFF (i.e., Opaque), the choice of SRv6 Endpoint Behavior is left to the headend. Well known 16-bit values for this field are available at https://www.iana.org/assignments/segment-routing/segment-routing.xhtml .

        Returns: str
        """
        return self._get_property("endpoint_behaviour")

    @endpoint_behaviour.setter
    def endpoint_behaviour(self, value):
        """endpoint_behaviour setter

        This is 2-octet field that is used to specify the SRv6 Endpoint Behavior code point for the SRv6 SID as defined in section 9.2 of [RFC8986]. When set with the value 0xFFFF (i.e., Opaque), the choice of SRv6 Endpoint Behavior is left to the headend. Well known 16-bit values for this field are available at https://www.iana.org/assignments/segment-routing/segment-routing.xhtml .

        value: str
        """
        self._set_property("endpoint_behaviour", value)

    @property
    def lb_length(self):
        # type: () -> int
        """lb_length getter

        SRv6 SID Locator Block length in bits.

        Returns: int
        """
        return self._get_property("lb_length")

    @lb_length.setter
    def lb_length(self, value):
        """lb_length setter

        SRv6 SID Locator Block length in bits.

        value: int
        """
        self._set_property("lb_length", value)

    @property
    def ln_length(self):
        # type: () -> int
        """ln_length getter

        SRv6 SID Locator Node length in bits.

        Returns: int
        """
        return self._get_property("ln_length")

    @ln_length.setter
    def ln_length(self, value):
        """ln_length setter

        SRv6 SID Locator Node length in bits.

        value: int
        """
        self._set_property("ln_length", value)

    @property
    def func_length(self):
        # type: () -> int
        """func_length getter

        SRv6 SID Function length in bits.

        Returns: int
        """
        return self._get_property("func_length")

    @func_length.setter
    def func_length(self, value):
        """func_length setter

        SRv6 SID Function length in bits.

        value: int
        """
        self._set_property("func_length", value)

    @property
    def arg_length(self):
        # type: () -> int
        """arg_length getter

        SRv6 SID Arguments length in bits.

        Returns: int
        """
        return self._get_property("arg_length")

    @arg_length.setter
    def arg_length(self, value):
        """arg_length setter

        SRv6 SID Arguments length in bits.

        value: int
        """
        self._set_property("arg_length", value)


class BgpAttributesSrv6BsidIter(OpenApiIter):
    __slots__ = ("_parent", "_choice")

    _GETITEM_RETURNS_CHOICE_OBJECT = False

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesSrv6BsidIter, self).__init__()
        self._parent = parent
        self._choice = choice

    def __getitem__(self, key):
        # type: (str) -> Union[BgpAttributesSrv6Bsid]
        return self._getitem(key)

    def __iter__(self):
        # type: () -> BgpAttributesSrv6BsidIter
        return self._iter()

    def __next__(self):
        # type: () -> BgpAttributesSrv6Bsid
        return self._next()

    def next(self):
        # type: () -> BgpAttributesSrv6Bsid
        return self._next()

    def _instanceOf(self, item):
        if not isinstance(item, BgpAttributesSrv6Bsid):
            raise Exception("Item is not an instance of BgpAttributesSrv6Bsid")

    def srv6bsid(
        self,
        flag_specified_bsid_only=False,
        flag_drop_upon_invalid=False,
        flag_srv6_endpoint_behavior=False,
        ipv6_addr="0::0",
    ):
        # type: (bool,bool,bool,str) -> BgpAttributesSrv6BsidIter
        """Factory method that creates an instance of the BgpAttributesSrv6Bsid class

        The SRv6 Binding SID sub-TLV is an optional sub-TLV of type 20 that is used to signal the SRv6 Binding SID. related information of an SR Policy candidate path. - More than one SRv6 Binding SID sub-TLVs MAY be signaled in the same SR Policy encoding to indicate one or. more SRv6 SIDs, each with potentially different SRv6 Endpoint Behaviors to be instantiated.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.3 .

        Returns: BgpAttributesSrv6BsidIter
        """
        item = BgpAttributesSrv6Bsid(
            parent=self._parent,
            flag_specified_bsid_only=flag_specified_bsid_only,
            flag_drop_upon_invalid=flag_drop_upon_invalid,
            flag_srv6_endpoint_behavior=flag_srv6_endpoint_behavior,
            ipv6_addr=ipv6_addr,
        )
        self._add(item)
        return self

    def add(
        self,
        flag_specified_bsid_only=False,
        flag_drop_upon_invalid=False,
        flag_srv6_endpoint_behavior=False,
        ipv6_addr="0::0",
    ):
        # type: (bool,bool,bool,str) -> BgpAttributesSrv6Bsid
        """Add method that creates and returns an instance of the BgpAttributesSrv6Bsid class

        The SRv6 Binding SID sub-TLV is an optional sub-TLV of type 20 that is used to signal the SRv6 Binding SID. related information of an SR Policy candidate path. - More than one SRv6 Binding SID sub-TLVs MAY be signaled in the same SR Policy encoding to indicate one or. more SRv6 SIDs, each with potentially different SRv6 Endpoint Behaviors to be instantiated.. The format of the sub-TLV is defined in draft-ietf-idr-sr-policy-safi-02 Section 2.4.3 .

        Returns: BgpAttributesSrv6Bsid
        """
        item = BgpAttributesSrv6Bsid(
            parent=self._parent,
            flag_specified_bsid_only=flag_specified_bsid_only,
            flag_drop_upon_invalid=flag_drop_upon_invalid,
            flag_srv6_endpoint_behavior=flag_srv6_endpoint_behavior,
            ipv6_addr=ipv6_addr,
        )
        self._add(item)
        return item


class BgpAttributesSrPolicyPreference(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=0):
        super(BgpAttributesSrPolicyPreference, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        Value to be carried in the Preference sub-tlv.

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value to be carried in the Preference sub-tlv.

        value: int
        """
        self._set_property("value", value)


class BgpAttributesSrPolicyPriority(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=0):
        super(BgpAttributesSrPolicyPriority, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        Value to be carried in the Priority sub-tlv.

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value to be carried in the Priority sub-tlv.

        value: int
        """
        self._set_property("value", value)


class BgpAttributesSrPolicyPolicyName(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "maxLength": 500,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("value",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=None):
        super(BgpAttributesSrPolicyPolicyName, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        Value of the symbolic policy name carried in the Policy Name sub-tlv.. It is recommended that the size of the name is limited to 255 bytes.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value of the symbolic policy name carried in the Policy Name sub-tlv.. It is recommended that the size of the name is limited to 255 bytes.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property value as None")
        self._set_property("value", value)


class BgpAttributesSrPolicyPolicyCandidateName(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": str,
            "maxLength": 500,
        },
    }  # type: Dict[str, str]

    _REQUIRED = ("value",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=None):
        super(BgpAttributesSrPolicyPolicyCandidateName, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> str
        """value getter

        Value of the symbolic Policy Candidate Path Name carried in the Policy Candidate Path Name sub-tlv.. It is recommended that the size of the name is limited to 255 bytes.

        Returns: str
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value of the symbolic Policy Candidate Path Name carried in the Policy Candidate Path Name sub-tlv.. It is recommended that the size of the name is limited to 255 bytes.

        value: str
        """
        if value is None:
            raise TypeError("Cannot set required property value as None")
        self._set_property("value", value)


class BgpAttributesSrPolicyExplicitNullPolicy(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "unknown",
                "push_ipv4",
                "push_ipv6",
                "push_ipv4_and_ipv6",
                "donot_push",
            ],
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "choice": "push_ipv4_and_ipv6",
    }  # type: Dict[str, Union(type)]

    UNKNOWN = "unknown"  # type: str
    PUSH_IPV4 = "push_ipv4"  # type: str
    PUSH_IPV6 = "push_ipv6"  # type: str
    PUSH_IPV4_AND_IPV6 = "push_ipv4_and_ipv6"  # type: str
    DONOT_PUSH = "donot_push"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesSrPolicyExplicitNullPolicy, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def choice(self):
        # type: () -> Union[Literal["donot_push"], Literal["push_ipv4"], Literal["push_ipv4_and_ipv6"], Literal["push_ipv6"], Literal["unknown"]]
        """choice getter

        The Explicit NULL Label policy.

        Returns: Union[Literal["donot_push"], Literal["push_ipv4"], Literal["push_ipv4_and_ipv6"], Literal["push_ipv6"], Literal["unknown"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        The Explicit NULL Label policy.

        value: Union[Literal["donot_push"], Literal["push_ipv4"], Literal["push_ipv4_and_ipv6"], Literal["push_ipv6"], Literal["unknown"]]
        """
        self._set_property("choice", value)


class BgpAttributesSrPolicySegmentList(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "weight": {"type": "BgpAttributesSegmentRoutingPolicySegmentListWeight"},
        "segments": {"type": "BgpAttributesSegmentRoutingPolicySegmentListSegmentIter"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesSrPolicySegmentList, self).__init__()
        self._parent = parent

    @property
    def weight(self):
        # type: () -> BgpAttributesSegmentRoutingPolicySegmentListWeight
        """weight getter

        The optional Weight sub-TLV (Type 9) specifies the weight associated with given segment list. The weight is used for weighted multipath.The optional Weight sub-TLV (Type 9) specifies the weight associated with given segment list. The weight is used for weighted multipath.The optional Weight sub-TLV (Type 9) specifies the weight associated with given segment list. The weight is used for weighted multipath.

        Returns: BgpAttributesSegmentRoutingPolicySegmentListWeight
        """
        return self._get_property(
            "weight", BgpAttributesSegmentRoutingPolicySegmentListWeight
        )

    @property
    def segments(self):
        # type: () -> BgpAttributesSegmentRoutingPolicySegmentListSegmentIter
        """segments getter

        TBD

        Returns: BgpAttributesSegmentRoutingPolicySegmentListSegmentIter
        """
        return self._get_property(
            "segments",
            BgpAttributesSegmentRoutingPolicySegmentListSegmentIter,
            self._parent,
            self._choice,
        )


class BgpAttributesSegmentRoutingPolicySegmentListWeight(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "value": {
            "type": int,
            "format": "uint32",
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "value": 0,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, value=0):
        super(BgpAttributesSegmentRoutingPolicySegmentListWeight, self).__init__()
        self._parent = parent
        self._set_property("value", value)

    def set(self, value=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def value(self):
        # type: () -> int
        """value getter

        Value of the weight.

        Returns: int
        """
        return self._get_property("value")

    @value.setter
    def value(self, value):
        """value setter

        Value of the weight.

        value: int
        """
        self._set_property("value", value)


class BgpAttributesSegmentRoutingPolicySegmentListSegment(OpenApiObject):
    __slots__ = ("_parent", "_choice")

    _TYPES = {
        "choice": {
            "type": str,
            "enum": [
                "type_a",
                "type_b",
                "type_c",
                "type_d",
                "type_e",
                "type_f",
                "type_g",
                "type_h",
                "type_i",
                "type_j",
                "type_k",
            ],
        },
        "type_a": {"type": "BgpAttributesSegmentRoutingPolicyTypeA"},
        "type_b": {"type": "BgpAttributesSegmentRoutingPolicyTypeB"},
        "type_c": {"type": "BgpAttributesSegmentRoutingPolicyTypeC"},
        "type_d": {"type": "BgpAttributesSegmentRoutingPolicyTypeD"},
        "type_e": {"type": "BgpAttributesSegmentRoutingPolicyTypeE"},
        "type_f": {"type": "BgpAttributesSegmentRoutingPolicyTypeF"},
        "type_g": {"type": "BgpAttributesSegmentRoutingPolicyTypeG"},
        "type_h": {"type": "BgpAttributesSegmentRoutingPolicyTypeH"},
        "type_i": {"type": "BgpAttributesSegmentRoutingPolicyTypeI"},
        "type_j": {"type": "BgpAttributesSegmentRoutingPolicyTypeJ"},
        "type_k": {"type": "BgpAttributesSegmentRoutingPolicyTypeK"},
    }  # type: Dict[str, str]

    _REQUIRED = ("choice",)  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    TYPE_A = "type_a"  # type: str
    TYPE_B = "type_b"  # type: str
    TYPE_C = "type_c"  # type: str
    TYPE_D = "type_d"  # type: str
    TYPE_E = "type_e"  # type: str
    TYPE_F = "type_f"  # type: str
    TYPE_G = "type_g"  # type: str
    TYPE_H = "type_h"  # type: str
    TYPE_I = "type_i"  # type: str
    TYPE_J = "type_j"  # type: str
    TYPE_K = "type_k"  # type: str

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, choice=None):
        super(BgpAttributesSegmentRoutingPolicySegmentListSegment, self).__init__()
        self._parent = parent
        if (
            "choice" in self._DEFAULTS
            and choice is None
            and self._DEFAULTS["choice"] in self._TYPES
        ):
            getattr(self, self._DEFAULTS["choice"])
        else:
            self._set_property("choice", choice)

    @property
    def type_a(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeA
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeA class

        Type A: SID only, in the form of MPLS Label.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeA
        """
        return self._get_property(
            "type_a", BgpAttributesSegmentRoutingPolicyTypeA, self, "type_a"
        )

    @property
    def type_b(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeB
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeB class

        Type B: SID only, in the form of IPv6 address.. It is encoded as Segment of Type 13 in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeB
        """
        return self._get_property(
            "type_b", BgpAttributesSegmentRoutingPolicyTypeB, self, "type_b"
        )

    @property
    def type_c(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeC
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeC class

        Type C: IPv4 Node Address with optional SID.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeC
        """
        return self._get_property(
            "type_c", BgpAttributesSegmentRoutingPolicyTypeC, self, "type_c"
        )

    @property
    def type_d(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeD
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeD class

        Type D: IPv6 Node Address with optional SID for SR MPLS.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeD
        """
        return self._get_property(
            "type_d", BgpAttributesSegmentRoutingPolicyTypeD, self, "type_d"
        )

    @property
    def type_e(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeE
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeE class

        Type E: IPv4 Address and Local Interface ID with optional SID. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeE
        """
        return self._get_property(
            "type_e", BgpAttributesSegmentRoutingPolicyTypeE, self, "type_e"
        )

    @property
    def type_f(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeF
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeF class

        Type F: IPv4 Local and Remote addresses with optional SR-MPLS SID.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeF
        """
        return self._get_property(
            "type_f", BgpAttributesSegmentRoutingPolicyTypeF, self, "type_f"
        )

    @property
    def type_g(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeG
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeG class

        Type G: IPv6 Address, Interface ID for local and remote pair with optional SID for SR MPLS.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeG
        """
        return self._get_property(
            "type_g", BgpAttributesSegmentRoutingPolicyTypeG, self, "type_g"
        )

    @property
    def type_h(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeH
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeH class

        Type H: IPv6 Local and Remote addresses with optional SID for SR MPLS.. It is encoded as Segment of Type in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeH
        """
        return self._get_property(
            "type_h", BgpAttributesSegmentRoutingPolicyTypeH, self, "type_h"
        )

    @property
    def type_i(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeI
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeI class

        Type I: IPv6 Node Address with optional SR Algorithm and optional SRv6 SID.. It is encoded as Segment of Type 14 in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeI
        """
        return self._get_property(
            "type_i", BgpAttributesSegmentRoutingPolicyTypeI, self, "type_i"
        )

    @property
    def type_j(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeJ
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeJ class

        Type J: IPv6 Address, Interface ID for local and remote pair for SRv6 with optional SID.. It is encoded as Segment of Type 15 in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeJ
        """
        return self._get_property(
            "type_j", BgpAttributesSegmentRoutingPolicyTypeJ, self, "type_j"
        )

    @property
    def type_k(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeK
        """Factory property that returns an instance of the BgpAttributesSegmentRoutingPolicyTypeK class

        Type K: IPv6 Local and Remote addresses for SRv6 with optional SID.. It is encoded as Segment of Type 16 in the SEGMENT_LIST sub-tlv.

        Returns: BgpAttributesSegmentRoutingPolicyTypeK
        """
        return self._get_property(
            "type_k", BgpAttributesSegmentRoutingPolicyTypeK, self, "type_k"
        )

    @property
    def choice(self):
        # type: () -> Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """choice getter

        Specify one of the segment types as defined in ietf-idr-segment-routing-te-policy. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Prefix with optional SR Algorithm.. Type D: IPv6 Global Prefix with optional SR Algorithm for SR-MPLS.. Type E: IPv4 Prefix with Local Interface ID.. Type F: IPv4 Addresses for link endpoints as Local, Remote pair.. Type G: IPv6 Prefix and Interface ID for link endpoints as Local, Remote pair for SR-MPLS.. Type H: IPv6 Addresses for link endpoints as Local, Remote pair for SR-MPLS.. Type I: IPv6 Global Prefix with optional SR Algorithm for SRv6.. Type J: IPv6 Prefix and Interface ID for link endpoints as Local, Remote pair for SRv6.. Type K: IPv6 Addresses for link endpoints as Local, Remote pair for SRv6.

        Returns: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        return self._get_property("choice")

    @choice.setter
    def choice(self, value):
        """choice setter

        Specify one of the segment types as defined in ietf-idr-segment-routing-te-policy. Type A: SID only, in the form of MPLS Label.. Type B: SID only, in the form of IPv6 Address.. Type C: IPv4 Prefix with optional SR Algorithm.. Type D: IPv6 Global Prefix with optional SR Algorithm for SR-MPLS.. Type E: IPv4 Prefix with Local Interface ID.. Type F: IPv4 Addresses for link endpoints as Local, Remote pair.. Type G: IPv6 Prefix and Interface ID for link endpoints as Local, Remote pair for SR-MPLS.. Type H: IPv6 Addresses for link endpoints as Local, Remote pair for SR-MPLS.. Type I: IPv6 Global Prefix with optional SR Algorithm for SRv6.. Type J: IPv6 Prefix and Interface ID for link endpoints as Local, Remote pair for SRv6.. Type K: IPv6 Addresses for link endpoints as Local, Remote pair for SRv6.

        value: Union[Literal["type_a"], Literal["type_b"], Literal["type_c"], Literal["type_d"], Literal["type_e"], Literal["type_f"], Literal["type_g"], Literal["type_h"], Literal["type_i"], Literal["type_j"], Literal["type_k"]]
        """
        if value is None:
            raise TypeError("Cannot set required property choice as None")
        self._set_property("choice", value)


class BgpAttributesSegmentRoutingPolicyTypeA(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "BgpAttributesSegmentRoutingPolicyTypeFlags"},
        "mpls_sid": {"type": "BgpAttributesSidMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {}  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None):
        super(BgpAttributesSegmentRoutingPolicyTypeA, self).__init__()
        self._parent = parent

    @property
    def flags(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeFlags
        """flags getter

        Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.

        Returns: BgpAttributesSegmentRoutingPolicyTypeFlags
        """
        return self._get_property("flags", BgpAttributesSegmentRoutingPolicyTypeFlags)

    @property
    def mpls_sid(self):
        # type: () -> BgpAttributesSidMpls
        """mpls_sid getter

        This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value.

        Returns: BgpAttributesSidMpls
        """
        return self._get_property("mpls_sid", BgpAttributesSidMpls)


class BgpAttributesSegmentRoutingPolicyTypeFlags(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "v_flag": {"type": bool},
        "a_flag": {"type": bool},
        "s_flag": {"type": bool},
        "b_flag": {"type": bool},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "v_flag": False,
        "a_flag": False,
        "s_flag": False,
        "b_flag": False,
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(
        self, parent=None, v_flag=False, a_flag=False, s_flag=False, b_flag=False
    ):
        super(BgpAttributesSegmentRoutingPolicyTypeFlags, self).__init__()
        self._parent = parent
        self._set_property("v_flag", v_flag)
        self._set_property("a_flag", a_flag)
        self._set_property("s_flag", s_flag)
        self._set_property("b_flag", b_flag)

    def set(self, v_flag=None, a_flag=None, s_flag=None, b_flag=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def v_flag(self):
        # type: () -> bool
        """v_flag getter

        Indicates verification of segment data in is enabled.

        Returns: bool
        """
        return self._get_property("v_flag")

    @v_flag.setter
    def v_flag(self, value):
        """v_flag setter

        Indicates verification of segment data in is enabled.

        value: bool
        """
        self._set_property("v_flag", value)

    @property
    def a_flag(self):
        # type: () -> bool
        """a_flag getter

        Indicates presence of SR Algorithm field applicable to Segment Types 3, 4, and 9.

        Returns: bool
        """
        return self._get_property("a_flag")

    @a_flag.setter
    def a_flag(self, value):
        """a_flag setter

        Indicates presence of SR Algorithm field applicable to Segment Types 3, 4, and 9.

        value: bool
        """
        self._set_property("a_flag", value)

    @property
    def s_flag(self):
        # type: () -> bool
        """s_flag getter

        This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type.

        Returns: bool
        """
        return self._get_property("s_flag")

    @s_flag.setter
    def s_flag(self, value):
        """s_flag setter

        This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type.

        value: bool
        """
        self._set_property("s_flag", value)

    @property
    def b_flag(self):
        # type: () -> bool
        """b_flag getter

        Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding specified in Section 2.4.4.2.4. of draft-ietf-idr-sr-policy-safi-02.

        Returns: bool
        """
        return self._get_property("b_flag")

    @b_flag.setter
    def b_flag(self, value):
        """b_flag setter

        Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding specified in Section 2.4.4.2.4. of draft-ietf-idr-sr-policy-safi-02.

        value: bool
        """
        self._set_property("b_flag", value)


class BgpAttributesSegmentRoutingPolicyTypeB(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "BgpAttributesSegmentRoutingPolicyTypeFlags"},
        "srv6_sid": {
            "type": str,
            "format": "ipv6",
        },
        "srv6_endpoint_behavior": {
            "type": "BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure"
        },
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "srv6_sid": "0::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, srv6_sid="0::0"):
        super(BgpAttributesSegmentRoutingPolicyTypeB, self).__init__()
        self._parent = parent
        self._set_property("srv6_sid", srv6_sid)

    def set(self, srv6_sid=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeFlags
        """flags getter

        Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.

        Returns: BgpAttributesSegmentRoutingPolicyTypeFlags
        """
        return self._get_property("flags", BgpAttributesSegmentRoutingPolicyTypeFlags)

    @property
    def srv6_sid(self):
        # type: () -> str
        """srv6_sid getter

        SRv6 SID.

        Returns: str
        """
        return self._get_property("srv6_sid")

    @srv6_sid.setter
    def srv6_sid(self, value):
        """srv6_sid setter

        SRv6 SID.

        value: str
        """
        self._set_property("srv6_sid", value)

    @property
    def srv6_endpoint_behavior(self):
        # type: () -> BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure
        """srv6_endpoint_behavior getter

        Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4Configuration for optional SRv6 Endpoint Behavior and SID Structure. Summation of lengths for Locator Block, Locator Node, Function, and Argument MUST be less than or equal to 128. This is specified in draft-ietf-idr-sr-policy-safi-02 Section 2.4.4.2.4

        Returns: BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure
        """
        return self._get_property(
            "srv6_endpoint_behavior",
            BgpAttributesSegmentRoutingPolicySRv6SIDEndpointBehaviorAndStructure,
        )


class BgpAttributesSegmentRoutingPolicyTypeC(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "BgpAttributesSegmentRoutingPolicyTypeFlags"},
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpAttributesSidMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
        "ipv4_node_address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, sr_algorithm=0, ipv4_node_address="0.0.0.0"):
        super(BgpAttributesSegmentRoutingPolicyTypeC, self).__init__()
        self._parent = parent
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv4_node_address", ipv4_node_address)

    def set(self, sr_algorithm=None, ipv4_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeFlags
        """flags getter

        Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.

        Returns: BgpAttributesSegmentRoutingPolicyTypeFlags
        """
        return self._get_property("flags", BgpAttributesSegmentRoutingPolicyTypeFlags)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on. If A-flag is not enabled, it should be set to on transmission and ignored on receipt.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on. If A-flag is not enabled, it should be set to on transmission and ignored on receipt.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv4_node_address(self):
        # type: () -> str
        """ipv4_node_address getter

        IPv4 address representing node.

        Returns: str
        """
        return self._get_property("ipv4_node_address")

    @ipv4_node_address.setter
    def ipv4_node_address(self, value):
        """ipv4_node_address setter

        IPv4 address representing node.

        value: str
        """
        self._set_property("ipv4_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpAttributesSidMpls
        """sr_mpls_sid getter

        This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. Optional SR-MPLS SID.

        Returns: BgpAttributesSidMpls
        """
        return self._get_property("sr_mpls_sid", BgpAttributesSidMpls)


class BgpAttributesSegmentRoutingPolicyTypeD(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "BgpAttributesSegmentRoutingPolicyTypeFlags"},
        "sr_algorithm": {
            "type": int,
            "format": "uint32",
            "maximum": 255,
        },
        "ipv6_node_address": {
            "type": str,
            "format": "ipv6",
        },
        "sr_mpls_sid": {"type": "BgpAttributesSidMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "sr_algorithm": 0,
        "ipv6_node_address": "0::0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, sr_algorithm=0, ipv6_node_address="0::0"):
        super(BgpAttributesSegmentRoutingPolicyTypeD, self).__init__()
        self._parent = parent
        self._set_property("sr_algorithm", sr_algorithm)
        self._set_property("ipv6_node_address", ipv6_node_address)

    def set(self, sr_algorithm=None, ipv6_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeFlags
        """flags getter

        Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.

        Returns: BgpAttributesSegmentRoutingPolicyTypeFlags
        """
        return self._get_property("flags", BgpAttributesSegmentRoutingPolicyTypeFlags)

    @property
    def sr_algorithm(self):
        # type: () -> int
        """sr_algorithm getter

        SR Algorithm identifier when A-Flag in on. If A-flag is not enabled, it should be set to on transmission and ignored on receipt.

        Returns: int
        """
        return self._get_property("sr_algorithm")

    @sr_algorithm.setter
    def sr_algorithm(self, value):
        """sr_algorithm setter

        SR Algorithm identifier when A-Flag in on. If A-flag is not enabled, it should be set to on transmission and ignored on receipt.

        value: int
        """
        self._set_property("sr_algorithm", value)

    @property
    def ipv6_node_address(self):
        # type: () -> str
        """ipv6_node_address getter

        IPv6 address representing node.

        Returns: str
        """
        return self._get_property("ipv6_node_address")

    @ipv6_node_address.setter
    def ipv6_node_address(self, value):
        """ipv6_node_address setter

        IPv6 address representing node.

        value: str
        """
        self._set_property("ipv6_node_address", value)

    @property
    def sr_mpls_sid(self):
        # type: () -> BgpAttributesSidMpls
        """sr_mpls_sid getter

        This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. This carries 20 bit Multi Protocol Label Switching alongwith bits traffic class, bit indicating presence. or absence of Bottom-Of-Stack and bits carrying the Time to Live value. Optional SR-MPLS SID.

        Returns: BgpAttributesSidMpls
        """
        return self._get_property("sr_mpls_sid", BgpAttributesSidMpls)


class BgpAttributesSegmentRoutingPolicyTypeE(OpenApiObject):
    __slots__ = "_parent"

    _TYPES = {
        "flags": {"type": "BgpAttributesSegmentRoutingPolicyTypeFlags"},
        "local_interface_id": {
            "type": int,
            "format": "uint32",
        },
        "ipv4_node_address": {
            "type": str,
            "format": "ipv4",
        },
        "sr_mpls_sid": {"type": "BgpAttributesSidMpls"},
    }  # type: Dict[str, str]

    _REQUIRED = ()  # type: tuple(str)

    _DEFAULTS = {
        "local_interface_id": 0,
        "ipv4_node_address": "0.0.0.0",
    }  # type: Dict[str, Union(type)]

    _STATUS = {}  # type: Dict[str, Union(type)]

    def __init__(self, parent=None, local_interface_id=0, ipv4_node_address="0.0.0.0"):
        super(BgpAttributesSegmentRoutingPolicyTypeE, self).__init__()
        self._parent = parent
        self._set_property("local_interface_id", local_interface_id)
        self._set_property("ipv4_node_address", ipv4_node_address)

    def set(self, local_interface_id=None, ipv4_node_address=None):
        for property_name, property_value in locals().items():
            if property_name != "self" and property_value is not None:
                self._set_property(property_name, property_value)

    @property
    def flags(self):
        # type: () -> BgpAttributesSegmentRoutingPolicyTypeFlags
        """flags getter

        Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of the SR-MPLS or SRv6 SID depending on the segment type. (draft-ietf-idr-bgp-sr-segtypes-ext-03 Section 2.10).. This flag is applicable for Segment Types C, D, E, F, G, H, I, J, and K.Flags for each Segment in SEGMENT_LIST sub-tlv.. V-flag. Indicates verification is enabled. See section 5, of https://datatracker.ietf.org/doc/html/rfc9256. A-flag. Indicates presence of SR Algorithm field applicable to Segment Types C, , , and .. B-Flag. Indicates presence of SRv6 Endpoint Behavior and SID Structure encoding applicable to Segment Types , , and .. S-Flag: This flag, when set, indicates the presence of th