"""This module contains the high-level API for the multi-day suite2p processing pipeline."""

import os
from typing import Any
from pathlib import Path
from functools import partial
from concurrent.futures import ThreadPoolExecutor

from tqdm import tqdm
import numpy as np
from natsort import natsorted
from ataraxis_time import PrecisionTimer
from ataraxis_base_utilities import LogLevel, console, ensure_directory_exists

from .version import version, sl_version, python_version
from .multiday import (
    import_sessions,
    register_sessions,
    extract_session_traces,
    export_masks_and_images,
    generate_template_masks,
    backward_transform_masks,
    extract_unique_components,
)
from .configuration import MultiDayS2PConfiguration, SingleDayS2PConfiguration, generate_default_multiday_ops

# THe minimum number of sessions required for multi-day processing
_MINIMUM_SESSION_COUNT = 2


def _resolve_session_data(input_directory: Path, output_directory: Path) -> Path:
    """Resolves the multi-day configuration data and output data hierarchy for the input session.

    This worker function is used as part of the main 'resolve_multiday_ops' function to efficiently process multiple
    sessions in parallel.

    Args:
        input_directory: The path to the directory that stores the single-day suite2p data for the processed session.
        output_directory: The path to the directory where to save the data generated during the multi-day processing.

    Returns:
        The path to the 'combined' single-day pipeline folder of the processed session.
    """
    # There should be exactly one 'combined' folder for each session folder. If this is false, then the folder path
    # is not valid
    combined_path_list = list(input_directory.rglob("combined"))
    if len(combined_path_list) != 1:
        message = (
            f"Unable to run the multi-day processing pipeline. Expected each path provided under the "
            f"'session_directories' multi-day configuration file field to point to a directory tree with exactly one "
            f"'combined' folder. Instead, encountered at least one path pointing to a directory tree with "
            f"{len(combined_path_list)} 'combined' folders."
        )
        console.error(message=message, error=RuntimeError)
        raise RuntimeError(message)  # Fallback to appease mypy, should not be reachable

    # Extracts the path to the 'combined' folder parent: the root suite2p output folder.
    combined_path = combined_path_list.pop()
    suite2p_path = combined_path.parent
    # Uses the path to load the single-day configuration file for the processed session.
    original_ops: dict[str, Any] = SingleDayS2PConfiguration.from_yaml(
        combined_path.parent.joinpath("single_day_ss2p_configuration.yaml")
    ).to_ops()

    # If necessary, overrides the paths to the single-day pipeline's output folder(s) to match the actual location of
    # the single-day output data.
    original_ops["save_folder"] = str(suite2p_path.stem)
    original_ops["save_path0"] = str(suite2p_path.parent)

    # Saves the updated single-day config to the session's multi-day output folder, as it is reused during
    # the multi-day pipeline
    single_day_config = SingleDayS2PConfiguration.from_ops(ops_dict=original_ops)
    single_day_config.to_config(file_path=output_directory.joinpath("single_day_ss2p_configuration.yaml"))

    return combined_path


def resolve_multiday_ops(ops: dict[str, Any], db: dict[str, Any]) -> Path:
    """Generates the output directory hierarchy and the multiday 'ops.npy' file for the multi-day suite2p pipeline,
    using the configuration parameters from ops and db.

    This function should be used before the first step (multi-day registration) of the multi-day suite2p pipeline to
    generate the necessary directories and files used by the further pipeline steps. During step-wise pipeline
    execution, this function can also be used between steps to update the 'ops.npy' file with new runtime parameters.

    Notes:
        All multi-day pipeline functions require the path generated by this function as the 'ops_path' argument.

        Since the multi-day pipeline uses the outputs of the single-day pipeline, this function merges the multi-day
        configuration parameters into the single-day configuration file of each processed session. Both 'ops' and 'db'
        can contain single-day and multi-day parameters to override them as necessary to support runtime.

        If both 'ops' and 'db' do not contain some of the expected multiday parameters, they are set using
        the 'default' dictionary generated using the MultiDayS2PConfiguration.

    Args:
        ops: A dictionary that contains the single-day configuration parameters.
        db: An optional dictionary that contains the same keys as 'ops'. Values from this dictionary are used to
            override the matching keys in the 'ops' dictionary.

    Returns:
        The path to the generated 'ops.npy' file.
    """
    # Since this step takes a noticeable amount of time, notifies the user about the progress of this step.
    console.echo(f"Resolving the multi-day 'ops' dictionary for {len(ops['session_directories'])} sessions...")

    # Builds up the multiday 'ops' dictionary. First fills the dictionary with the 'default' keys. Then
    # overwrites all default keys with keys from the input 'ops' dictionary. Finally, overwrites any keys from the
    # input 'db' dictionary with values from 'db'. This way, there is the following order of precedence:
    # db > ops > default.
    ops = {
        **generate_default_multiday_ops(as_dict=True),
        **ops,
        **db,
        # Actualizes version information
        "base_suite2p_version": version,
        "sl_suite2p_version": sl_version,
        "python_version": python_version,
    }

    # If the user does not specify the maximum parallel worker limit, sets it based on the number of available
    # CPU cores
    if ops["parallel_workers"] < 1:
        # noinspection PyUnresolvedReferences
        ops["parallel_workers"] = os.process_cpu_count()

    # Ensures that the resolved 'ops' file contains at least two sessions to process.
    if len(ops["session_directories"]) < _MINIMUM_SESSION_COUNT:
        message = (
            f"Unable to run multi-day processing using the input parameters. Specifically, expected at least two "
            f"items provided under the 'session_directories' configuration file field, but instead encountered "
            f"{len(ops['session_directories'])}."
        )
        console.error(message=message, error=RuntimeError)

    # Builds a list of unique path components for each session. Assuming all sessions use unique names, this generates
    # a list of unique session IDs from their paths.
    sessions = natsorted([Path(session) for session in ops["session_directories"]])
    ops["session_ids"] = extract_unique_components(paths=sessions)

    # Resolves and generates the output directory for the multi-day runtime.
    output_folder = Path(ops["multiday_save_path"]).joinpath(ops["multiday_save_folder"])
    ensure_directory_exists(output_folder)

    # Extracts the paths to the processed session directories.
    session_directories = ops["session_directories"]

    # Processes all sessions in parallel.
    last_combined_path = Path()
    with ThreadPoolExecutor(max_workers=ops["parallel_workers"]) as executor:
        # Submits processing tasks using partial to create properly parameterized function calls.
        futures = []
        for i, directory in enumerate(session_directories):
            # Generates the session-specific output directory under the main multi-day output folder, using the
            # session ID.
            multi_day_path = output_folder.joinpath(ops["session_ids"][i])
            ensure_directory_exists(multi_day_path)

            # Uses partial to specialize the processing function for runtime.
            session_worker = partial(
                _resolve_session_data, input_directory=Path(directory), output_directory=multi_day_path
            )

            # Submit the specialized function for processing.
            futures.append(executor.submit(session_worker))

        # Processes completed tasks with a progress bar if progress bars are enabled.
        with tqdm(
            total=len(futures), desc="Processing sessions", disable=not ops["progress_bars"], unit="session"
        ) as pbar:
            for future in futures:
                last_combined_path = future.result()  # Keeps track of the last processed path
                pbar.update(1)

    # Resolves the path to the 'combined' folder ops.npy file for the last session processed by the loop above. This
    # file contains all single-day processing parameters, some of which are reused at the end of the multi-day pipeline
    # (during fluorescence extraction). Then, loads the combined ops into memory as a dictionary. Note, this approach
    # EXPECTS all sessions to share the same single-day parameters and many of the descriptive characteristics (such as
    # imaging fields) to work as expected.
    ops_combined: dict[str, Any] = np.load(last_combined_path.joinpath("ops.npy"), allow_pickle=True).item()

    # Merges the multi-day runtime parameters into the single-day ops dictionary. This generates a
    # combined config that holds both the single-day and the multi-day processing parameters. Also, since 'ops'
    # overrides single-day 'ops_combined', users can overwrite the parameters inside the single-day 'ops' dictionary
    # as necessary to support their runtime.
    multiday_ops = {
        **ops_combined,
        **ops,
        "allow_overlap": True,  # Required for multi-day signal extraction to work as expected
    }

    # Saves the combined single-day and multi-day ops.npy file to the root output folder. During the rest of the
    # runtime, this file is loaded back into memory for various processing steps.
    ops_path = output_folder.joinpath("ops.npy")
    np.save(ops_path, multiday_ops, allow_pickle=True)

    # Also generates a 'yaml' snapshot of the multi-day configuration parameters. This does not use the 'combined' ops
    # as it stores many fields that are not directly translatable into a YAML format.
    multi_day_config = MultiDayS2PConfiguration.from_ops(ops_dict=ops)
    multi_day_config.to_config(file_path=output_folder.joinpath("multi_day_ss2p_configuration.yaml"))

    # Returns the path to the final multi-day config.
    return ops_path


def run_s2p_multiday(ops_path: Path) -> None:
    """Executes the multi-day suite2p processing pipeline using the parameters stored in the target ops.npy file.

    This function sequentially calls all steps of the suite2p multi-day processing pipeline, extracting cell
    fluorescence data from cells tracked across multiple sessions (days) pre-processed with the single-day suite2p
    pipeline.

    Notes:
        This function works as a high-level API wrapper over the two pipeline step functions: 'discover_multiday_cells'
        and 'extract_multiday_fluorescence'.

    Args:
         ops_path: The path to the ops.npy file that stores the multi-day suite2p processing parameters. Compatible
            ops.npy files are generated by the resolve_multiday_ops() function.
    """
    # Guards against invalid inputs.
    if not ops_path.exists() or not ops_path.is_file() or ops_path.suffix != ".npy":
        message = (
            f"Unable to run the multi-day suite2p pipeline, as the 'ops.npy' file does not exist at the specified "
            f"path {ops_path}."
        )
        console.error(message=message, error=FileNotFoundError)

    # Instantiates and resets the execution timer
    timer = PrecisionTimer("s")
    timer.reset()

    console.echo(message="Initializing multi-day suite2p runtime...", level=LogLevel.INFO)

    # Loads the configuration data into memory
    ops = np.load(ops_path, allow_pickle=True).item()

    # Step 1: Tracks cells active across processed sessions and generates the ROI masks for the tracked cells in the
    # visual space of each processed session.
    discover_multiday_cells(ops_path)

    # Step 2: Extracts the across-day-tracked cell fluorescence for each processed session.
    for session in ops["session_ids"]:
        extract_multiday_fluorescence(ops_path, session)

    message = f"Multi-day suite2p runtime: Complete. Total time: {timer.elapsed} seconds."
    console.echo(message=message, level=LogLevel.SUCCESS)


def discover_multiday_cells(ops_path: Path) -> None:
    """Discovers reliably identifiable cells and tracks them across the processed set of sessions.

    Notes:
        As part of its runtime, this function fills the output folder for each session with the results of
        across-day cell tracking, including the masks used by the second pipeline step.

    Args:
        ops_path: The path to the ops.npy file that stores multi-day suite2p processing parameters. Compatible ops.npy
            files are generated by the resolve_multiday_ops() function.
    """
    # Verifies the input ops.npy file.
    if not ops_path.exists() or not ops_path.is_file() or ops_path.suffix != ".npy":
        message = (
            f"Unable to run the multi-day suite2p pipeline, as the 'ops.npy' file does not exist at the specified "
            f"path {ops_path}."
        )
        console.error(message=message, error=FileNotFoundError)
    ops = np.load(ops_path, allow_pickle=True).item()

    # Initializes the runtime timer
    timer = PrecisionTimer("s")
    step_timer = PrecisionTimer("s")  # Times the whole registration processing step.
    step_timer.reset()

    # Imports the data required to carry out registration for each session.
    console.echo(message="Importing requested session data for multi-day registration...")
    timer.reset()
    sessions_data = import_sessions(ops=ops)
    console.echo(message=f"Session data: imported. Time taken: {timer.elapsed} seconds.", level=LogLevel.SUCCESS)

    # Registers all sessions across days (to the same deformed visual field) by generating and applying the deformation
    # offsets to all reference images and cell masks of every session.
    console.echo(message="Registering sessions across days...")
    timer.reset()
    sessions_data = register_sessions(ops=ops, data=sessions_data)
    console.echo(message=f"Registration: complete. Time taken: {timer.elapsed} seconds.", level=LogLevel.SUCCESS)

    # Tracks cells across sessions in the deformed visual space and computes the template masks for cells that can be
    # identified across sessions to apply to the original visual space of each session
    console.echo(message="Computing across-session cell masks...")
    timer.reset()
    sessions_data = generate_template_masks(ops=ops, data=sessions_data)
    console.echo(
        message=f"Across-session cell masks: computed. Time taken: {timer.elapsed} seconds.", level=LogLevel.SUCCESS
    )

    # Transforms the template cell masks from the shared multi-session (deformed) visual space to the original
    # (unregistered) visual space of each session. This is necessary to re-extract the fluorescence of multi-day-tracked
    # cells from each session during the second step of the multi-day pipeline.
    console.echo(message="Transforming template cell masks from multi-day visual space to single-day visual space...")
    timer.reset()
    sessions_data = backward_transform_masks(ops=ops, data=sessions_data)
    console.echo(message=f"Cell masks: transformed. Time taken: {timer.elapsed} seconds.", level=LogLevel.SUCCESS)

    # Exports all data generate during the first (registration) step to disk. The data is then reloaded as part of the
    # second (extraction) step.
    console.echo(message="Appending multi-day registration data to each session's suite2p (output) folder...")
    export_masks_and_images(ops=ops, data=sessions_data)
    console.echo(
        message=f"Multi-day registration: complete. Time taken: {step_timer.elapsed} seconds.", level=LogLevel.SUCCESS
    )


def extract_multiday_fluorescence(ops_path: Path, session_id: str) -> None:
    """Extracts the fluorescence data from cells tracked across the processed set of sessions.

    Args:
        ops_path: The path to the ops.npy file that stores multi-day suite2p processing parameters. Compatible ops.npy
            files are generated by the resolve_multiday_ops() function.
        session_id: The unique ID of the session for which to extract the across-day-tracked cell fluorescence. The ID
            has to match one of the unique session IDs stored under the 'session_ids' field of the dictionary loaded
            from the ops.npy file.
    """
    # Verifies the input ops.npy file.
    if not ops_path.exists() or not ops_path.is_file() or ops_path.suffix != ".npy":
        message = (
            f"Unable to run the multi-day suite2p pipeline, as the 'ops.npy' file does not exist at the specified "
            f"path {ops_path}."
        )
        console.error(message=message, error=FileNotFoundError)

    ops = np.load(ops_path, allow_pickle=True).item()

    # Reconstructs the path to the output folder from 'ops' parameters
    output_folder = Path(ops["multiday_save_path"]).joinpath(ops["multiday_save_folder"])

    # The output folder contains .npy and .yaml files and directories named after each processed session ID.
    # This re-generates the list of session IDs from the directories stored in the output folder.
    session_ids = [folder.stem for folder in output_folder.glob("*") if folder.is_dir()]

    # Sorts session IDs for consistency
    session_ids = natsorted(session_ids)

    if session_id not in session_ids:
        message = (
            f"Unable to extract across-day-tracked cell activity data for session '{session_id}' as this ID is not "
            f"inside the session_ids list loaded from the ops.npy file specified by the {ops_path} path."
        )
        console.error(message=message, error=ValueError)

    # Loops over the output session directories and reconstructs the path to the single-day output folder of each
    # session using the data inside the single-day s2p config from the multi-day folder.
    sessions = []
    for session in session_ids:
        configuration: SingleDayS2PConfiguration = SingleDayS2PConfiguration.from_yaml(
            output_folder.joinpath(session, "single_day_ss2p_configuration.yaml")
        )
        session_folder = Path(configuration.file_io.save_path0).joinpath(configuration.file_io.save_folder)
        sessions.append(session_folder)

    # Uses the input ID to resolve the target session folder
    session_index = session_ids.index(session_id)
    target_session_folder = sessions[session_index]

    # Calls the trace extraction function
    extract_session_traces(ops=ops, session_folder=target_session_folder, session_id=session_id)
