from datetime import datetime
from pathlib import Path
from typing import Any, Dict, List, Optional

from fastapi import Depends, FastAPI, HTTPException, status
from fastapi.responses import HTMLResponse
from pydantic import BaseModel, Field
from sqlalchemy import JSON, Column, DateTime, Integer, String, Text, UniqueConstraint, create_engine
from sqlalchemy.exc import IntegrityError
from sqlalchemy.orm import Session, declarative_base, sessionmaker


# --- 1. Define the Data Models -------------------------------------------------


class ConfigModel(BaseModel):
    """
    Defines the technical parameters for the LLM provider.
    This structure matches the config.json referenced by agents.
    """

    model_provider: str = Field(..., description="e.g., 'openai', 'anthropic', 'google'")
    model_name: str = Field(..., description="e.g., 'gpt-4o', 'claude-3-opus'")
    temperature: float = Field(0.7, ge=0.0, le=2.0, description="Controls randomness; higher = more creative")
    max_tokens: int = Field(2048, gt=0, description="Maximum length of the model response")
    top_p: float = Field(1.0, ge=0.0, le=1.0, description="Nucleus sampling parameter")
    stop_sequences: Optional[List[str]] = Field(default_factory=list, description="Strings that halt generation")


class RagManifestModel(BaseModel):
    """
    Defines the "source of truth" for the RAG system.
    Points to the data backing each agent/context pair.
    """

    version: str
    sources: List[Dict[str, Any]]


class ContextPackage(BaseModel):
    """
    Bundle of persona/system prompt, technical config, and RAG manifest.
    Returned to agents at runtime and written by developers/operators.
    """

    system_prompt: str
    config: ConfigModel
    rag_manifest: RagManifestModel


class ContextEnvelope(ContextPackage):
    """Context plus metadata stored in the persistence layer."""

    agent_name: str
    label: str
    created_at: datetime
    updated_at: datetime


class PromptCreate(BaseModel):
    """Schema for recording prompts/responses generated by the agents."""

    agent_name: str
    label: Optional[str] = Field(None, description="Context label used when generating the prompt")
    prompt_text: str = Field(..., description="Prompt sent to the model")
    response_text: Optional[str] = Field(None, description="Optional model response to persist")
    metadata: Optional[Dict[str, Any]] = Field(default_factory=dict)


class PromptRecordModel(PromptCreate):
    """API representation of a persisted prompt."""

    id: int
    created_at: datetime


class MemoryCreate(BaseModel):
    """Schema for storing general or user-specific agent memories."""

    agent_name: str = Field(..., description="Agent the memory belongs to")
    user_id: Optional[str] = Field(
        None,
        description="Identifier for the user this memory is scoped to; leave empty for general agent memories",
    )
    memory_text: str = Field(..., description="Memory content")
    metadata: Optional[Dict[str, Any]] = Field(default_factory=dict)


class MemoryRecordModel(MemoryCreate):
    """API representation of a persisted memory."""

    id: int
    created_at: datetime
    updated_at: datetime


# --- 2. Database Layer ---------------------------------------------------------


BASE_DIR = Path(__file__).resolve().parent
DATABASE_PATH = BASE_DIR / "context_control.db"
FRONTEND_DIR = BASE_DIR / "frontend"
DATABASE_URL = f"sqlite:///{DATABASE_PATH}"


engine = create_engine(
    DATABASE_URL,
    connect_args={"check_same_thread": False},
    future=True,
)
SessionLocal = sessionmaker(bind=engine, autoflush=False, autocommit=False, future=True)
Base = declarative_base()


class ContextRecord(Base):
    __tablename__ = "contexts"
    __table_args__ = (
        UniqueConstraint("agent_name", "label", name="uq_agent_label"),
    )

    id = Column(Integer, primary_key=True, index=True)
    agent_name = Column(String(100), nullable=False)
    label = Column(String(100), nullable=False)
    system_prompt = Column(Text, nullable=False)
    config = Column(JSON, nullable=False)
    rag_manifest = Column(JSON, nullable=False)
    created_at = Column(DateTime, default=datetime.utcnow, nullable=False)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow, nullable=False)


class PromptRecord(Base):
    __tablename__ = "prompts"

    id = Column(Integer, primary_key=True, index=True)
    agent_name = Column(String(100), nullable=False)
    label = Column(String(100), nullable=True)
    prompt_text = Column(Text, nullable=False)
    response_text = Column(Text, nullable=True)
    payload = Column(JSON, nullable=False, default=dict)
    created_at = Column(DateTime, default=datetime.utcnow, nullable=False)


class MemoryRecord(Base):
    __tablename__ = "memories"
    __table_args__ = (
        UniqueConstraint("agent_name", "user_id", "memory_text", name="uq_agent_user_text"),
    )

    id = Column(Integer, primary_key=True, index=True)
    agent_name = Column(String(100), nullable=False)
    user_id = Column(String(255), nullable=True)
    memory_text = Column(Text, nullable=False)
    metadata_json = Column("metadata", JSON, nullable=False, default=dict)
    created_at = Column(DateTime, default=datetime.utcnow, nullable=False)
    updated_at = Column(DateTime, default=datetime.utcnow, onupdate=datetime.utcnow, nullable=False)


def init_db() -> None:
    """Create database tables if they do not already exist."""

    Base.metadata.create_all(bind=engine)


def get_session() -> Session:
    """FastAPI dependency that yields a SQLAlchemy session."""

    session = SessionLocal()
    try:
        yield session
    finally:
        session.close()


def record_to_envelope(record: ContextRecord) -> ContextEnvelope:
    """Convert a persistence record into a response payload."""

    return ContextEnvelope(
        agent_name=record.agent_name,
        label=record.label,
        system_prompt=record.system_prompt,
        config=ConfigModel(**record.config),
        rag_manifest=RagManifestModel(**record.rag_manifest),
        created_at=record.created_at,
        updated_at=record.updated_at,
    )


def prompt_record_to_model(record: PromptRecord) -> PromptRecordModel:
    return PromptRecordModel(
        id=record.id,
        agent_name=record.agent_name,
        label=record.label,
        prompt_text=record.prompt_text,
        response_text=record.response_text,
        metadata=record.payload or {},
        created_at=record.created_at,
    )


def memory_record_to_model(record: MemoryRecord) -> MemoryRecordModel:
    return MemoryRecordModel(
        id=record.id,
        agent_name=record.agent_name,
        user_id=record.user_id,
        memory_text=record.memory_text,
        metadata=record.metadata_json or {},
        created_at=record.created_at,
        updated_at=record.updated_at,
    )


# --- 3. FastAPI Application ----------------------------------------------------


app = FastAPI(
    title="Context Control API",
    description="Store, version, and retrieve AI context packages and prompt telemetry.",
)


@app.on_event("startup")
def on_startup() -> None:
    """Ensure tables exist when the API starts."""

    init_db()


@app.get("/", summary="Server Health Check")
def get_root() -> Dict[str, str]:
    return {"message": "Context Control API is running."}


@app.get("/dashboard", response_class=HTMLResponse, summary="Embedded dashboard UI")
def get_dashboard() -> HTMLResponse:
    html_path = FRONTEND_DIR / "index.html"
    if not html_path.exists():
        raise HTTPException(status_code=status.HTTP_503_SERVICE_UNAVAILABLE, detail="Dashboard not available")
    return HTMLResponse(content=html_path.read_text(encoding="utf-8"))


@app.get(
    "/contexts",
    response_model=List[ContextEnvelope],
    summary="List all contexts",
)
def list_contexts(session: Session = Depends(get_session)) -> List[ContextEnvelope]:
    records = session.query(ContextRecord).order_by(ContextRecord.agent_name, ContextRecord.label).all()
    return [record_to_envelope(record) for record in records]


@app.get(
    "/contexts/{agent_name}",
    response_model=List[ContextEnvelope],
    summary="List contexts for an agent",
)
def list_agent_contexts(agent_name: str, session: Session = Depends(get_session)) -> List[ContextEnvelope]:
    records = session.query(ContextRecord).filter(ContextRecord.agent_name == agent_name).order_by(ContextRecord.label).all()
    if not records:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Agent not found")
    return [record_to_envelope(record) for record in records]


@app.get(
    "/contexts/{agent_name}/{label}",
    response_model=ContextEnvelope,
    summary="Read a specific context",
)
def read_context(agent_name: str, label: str, session: Session = Depends(get_session)) -> ContextEnvelope:
    record = (
        session.query(ContextRecord)
        .filter(ContextRecord.agent_name == agent_name, ContextRecord.label == label)
        .one_or_none()
    )
    if not record:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Context not found")
    return record_to_envelope(record)


@app.post(
    "/contexts/{agent_name}/{label}",
    response_model=ContextEnvelope,
    summary="Create or update a context",
)
def upsert_context(
    agent_name: str,
    label: str,
    package: ContextPackage,
    session: Session = Depends(get_session),
) -> ContextEnvelope:
    record = (
        session.query(ContextRecord)
        .filter(ContextRecord.agent_name == agent_name, ContextRecord.label == label)
        .one_or_none()
    )

    data = {
        "system_prompt": package.system_prompt,
        "config": package.config.model_dump(),
        "rag_manifest": package.rag_manifest.model_dump(),
    }

    if record:
        for field, value in data.items():
            setattr(record, field, value)
        record.updated_at = datetime.utcnow()
    else:
        record = ContextRecord(
            agent_name=agent_name,
            label=label,
            **data,
        )
        session.add(record)

    session.commit()
    session.refresh(record)
    return record_to_envelope(record)


@app.delete(
    "/contexts/{agent_name}/{label}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="Delete a context",
)
def delete_context(agent_name: str, label: str, session: Session = Depends(get_session)) -> None:
    record = (
        session.query(ContextRecord)
        .filter(ContextRecord.agent_name == agent_name, ContextRecord.label == label)
        .one_or_none()
    )
    if not record:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Context not found")

    session.delete(record)
    session.commit()


@app.post(
    "/prompts",
    response_model=PromptRecordModel,
    status_code=status.HTTP_201_CREATED,
    summary="Log a prompt/response pair",
)
def create_prompt(entry: PromptCreate, session: Session = Depends(get_session)) -> PromptRecordModel:
    record = PromptRecord(
        agent_name=entry.agent_name,
        label=entry.label,
        prompt_text=entry.prompt_text,
        response_text=entry.response_text,
        payload=entry.metadata or {},
    )
    session.add(record)
    session.commit()
    session.refresh(record)
    return prompt_record_to_model(record)


@app.get(
    "/prompts",
    response_model=List[PromptRecordModel],
    summary="List recorded prompts",
)
def list_prompts(
    agent_name: Optional[str] = None,
    label: Optional[str] = None,
    session: Session = Depends(get_session),
) -> List[PromptRecordModel]:
    query = session.query(PromptRecord).order_by(PromptRecord.created_at.desc())
    if agent_name:
        query = query.filter(PromptRecord.agent_name == agent_name)
    if label:
        query = query.filter(PromptRecord.label == label)

    records = query.limit(200).all()
    return [prompt_record_to_model(record) for record in records]


@app.post(
    "/memories",
    response_model=MemoryRecordModel,
    status_code=status.HTTP_201_CREATED,
    summary="Create a general or user-specific memory",
)
def create_memory(entry: MemoryCreate, session: Session = Depends(get_session)) -> MemoryRecordModel:
    record = MemoryRecord(
        agent_name=entry.agent_name,
        user_id=entry.user_id,
        memory_text=entry.memory_text,
        metadata_json=entry.metadata or {},
    )
    session.add(record)
    try:
        session.commit()
    except IntegrityError as exc:
        session.rollback()
        raise HTTPException(
            status_code=status.HTTP_409_CONFLICT,
            detail="Memory already exists for this agent/user combination",
        ) from exc
    session.refresh(record)
    return memory_record_to_model(record)


@app.get(
    "/memories",
    response_model=List[MemoryRecordModel],
    summary="List memories with optional filtering",
)
def list_memories(
    agent_name: Optional[str] = None,
    user_id: Optional[str] = None,
    include_general: bool = True,
    limit: int = 200,
    session: Session = Depends(get_session),
) -> List[MemoryRecordModel]:
    query = session.query(MemoryRecord).order_by(MemoryRecord.created_at.desc())

    if agent_name:
        query = query.filter(MemoryRecord.agent_name == agent_name)
    if user_id:
        if include_general:
            query = query.filter(
                (MemoryRecord.user_id == user_id) | (MemoryRecord.user_id.is_(None))
            )
        else:
            query = query.filter(MemoryRecord.user_id == user_id)
    elif not include_general:
        query = query.filter(MemoryRecord.user_id.isnot(None))

    records = query.limit(max(1, min(limit, 500))).all()
    return [memory_record_to_model(record) for record in records]


@app.delete(
    "/memories/{memory_id}",
    status_code=status.HTTP_204_NO_CONTENT,
    summary="Delete a stored memory",
)
def delete_memory(memory_id: int, session: Session = Depends(get_session)) -> None:
    record = session.get(MemoryRecord, memory_id)
    if not record:
        raise HTTPException(status_code=status.HTTP_404_NOT_FOUND, detail="Memory not found")

    session.delete(record)
    session.commit()


"""Compatibility module that exposes the FastAPI app from the new package."""

from context_control.api import create_app

app = create_app()
