import lambda = require('@aws-cdk/aws-lambda');
import s3 = require('@aws-cdk/aws-s3');
import cdk = require('@aws-cdk/cdk');
import { CfnDistribution } from './cloudfront.generated';
import { IDistribution } from './distribution';
export declare enum HttpVersion {
    HTTP1_1 = "http1.1",
    HTTP2 = "http2"
}
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
export declare enum PriceClass {
    PriceClass100 = "PriceClass_100",
    PriceClass200 = "PriceClass_200",
    PriceClassAll = "PriceClass_All"
}
/**
 * How HTTPs should be handled with your distribution.
 */
export declare enum ViewerProtocolPolicy {
    HTTPSOnly = "https-only",
    RedirectToHTTPS = "redirect-to-https",
    AllowAll = "allow-all"
}
/**
 * Configuration for custom domain names
 *
 * CloudFront can use a custom domain that you provide instead of a
 * "cloudfront.net" domain. To use this feature you must provide the list of
 * additional domains, and the ACM Certificate that CloudFront should use for
 * these additional domains.
 */
export interface AliasConfiguration {
    /**
     * ARN of an AWS Certificate Manager (ACM) certificate.
     */
    readonly acmCertRef: string;
    /**
     * Domain names on the certificate
     *
     * Both main domain name and Subject Alternative Names.
     */
    readonly names: string[];
    /**
     * How CloudFront should serve HTTPS requests.
     *
     * See the notes on SSLMethod if you wish to use other SSL termination types.
     *
     * @default SSLMethod.SNI
     * @see https://docs.aws.amazon.com/cloudfront/latest/APIReference/API_ViewerCertificate.html
     */
    readonly sslMethod?: SSLMethod;
    /**
     * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
     *
     * CloudFront serves your objects only to browsers or devices that support at
     * least the SSL version that you specify.
     *
     * @default - SSLv3 if sslMethod VIP, TLSv1 if sslMethod SNI
     */
    readonly securityPolicy?: SecurityPolicyProtocol;
}
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
export declare enum SSLMethod {
    SNI = "sni-only",
    VIP = "vip"
}
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
export declare enum SecurityPolicyProtocol {
    SSLv3 = "SSLv3",
    TLSv1 = "TLSv1",
    TLSv1_2016 = "TLSv1_2016",
    TLSv1_1_2016 = "TLSv1.1_2016",
    TLSv1_2_2018 = "TLSv1.2_2018"
}
/**
 * Logging configuration for incoming requests
 */
export interface LoggingConfiguration {
    /**
     * Bucket to log requests to
     *
     * @default - A logging bucket is automatically created.
     */
    readonly bucket?: s3.IBucket;
    /**
     * Whether to include the cookies in the logs
     *
     * @default false
     */
    readonly includeCookies?: boolean;
    /**
     * Where in the bucket to store logs
     *
     * @default - No prefix.
     */
    readonly prefix?: string;
}
/**
 * A source configuration is a wrapper for CloudFront origins and behaviors.
 * An origin is what CloudFront will "be in front of" - that is, CloudFront will pull it's assets from an origin.
 *
 * If you're using s3 as a source - pass the `s3Origin` property, otherwise, pass the `customOriginSource` property.
 *
 * One or the other must be passed, and it is invalid to pass both in the same SourceConfiguration.
 */
export interface SourceConfiguration {
    /**
     * An s3 origin source - if you're using s3 for your assets
     */
    readonly s3OriginSource?: S3OriginConfig;
    /**
     * A custom origin source - for all non-s3 sources.
     */
    readonly customOriginSource?: CustomOriginConfig;
    /**
     * The behaviors associated with this source.
     * At least one (default) behavior must be included.
     */
    readonly behaviors: Behavior[];
    /**
     * The relative path to the origin root to use for sources.
     *
     * @default /
     */
    readonly originPath?: string;
    /**
     * Any additional headers to pass to the origin
     *
     * @default - No additional headers are passed.
     */
    readonly originHeaders?: {
        [key: string]: string;
    };
}
/**
 * A custom origin configuration
 */
export interface CustomOriginConfig {
    /**
     * The domain name of the custom origin. Should not include the path - that should be in the parent SourceConfiguration
     */
    readonly domainName: string;
    /**
     * The origin HTTP port
     *
     * @default 80
     */
    readonly httpPort?: number;
    /**
     * The origin HTTPS port
     *
     * @default 443
     */
    readonly httpsPort?: number;
    /**
     * The keep alive timeout when making calls in seconds.
     *
     * @default 5
     */
    readonly originKeepaliveTimeoutSeconds?: number;
    /**
     * The protocol (http or https) policy to use when interacting with the origin.
     *
     * @default OriginProtocolPolicy.HttpsOnly
     */
    readonly originProtocolPolicy?: OriginProtocolPolicy;
    /**
     * The read timeout when calling the origin in seconds
     *
     * @default 30
     */
    readonly originReadTimeoutSeconds?: number;
    /**
     * The SSL versions to use when interacting with the origin.
     *
     * @default OriginSslPolicy.TLSv1_2
     */
    readonly allowedOriginSSLVersions?: OriginSslPolicy[];
}
export declare enum OriginSslPolicy {
    SSLv3 = "SSLv3",
    TLSv1 = "TLSv1",
    TLSv1_1 = "TLSv1.1",
    TLSv1_2 = "TLSv1.2"
}
export declare enum OriginProtocolPolicy {
    HttpOnly = "http-only",
    MatchViewer = "match-viewer",
    HttpsOnly = "https-only"
}
export interface S3OriginConfig {
    /**
     * The source bucket to serve content from
     */
    readonly s3BucketSource: s3.IBucket;
    /**
     * The optional ID of the origin identity cloudfront will use when calling your s3 bucket.
     */
    readonly originAccessIdentityId?: string;
}
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
export declare enum CloudFrontAllowedMethods {
    GET_HEAD = "GH",
    GET_HEAD_OPTIONS = "GHO",
    ALL = "ALL"
}
/**
 * Enums for the methods CloudFront can cache.
 */
export declare enum CloudFrontAllowedCachedMethods {
    GET_HEAD = "GH",
    GET_HEAD_OPTIONS = "GHO"
}
/**
 * A CloudFront behavior wrapper.
 */
export interface Behavior {
    /**
     * If CloudFront should automatically compress some content types.
     *
     * @default true
     */
    readonly compress?: boolean;
    /**
     * If this behavior is the default behavior for the distribution.
     *
     * You must specify exactly one default distribution per CloudFront distribution.
     * The default behavior is allowed to omit the "path" property.
     */
    readonly isDefaultBehavior?: boolean;
    /**
     * Trusted signers is how CloudFront allows you to serve private content.
     * The signers are the account IDs that are allowed to sign cookies/presigned URLs for this distribution.
     *
     * If you pass a non empty value, all requests for this behavior must be signed (no public access will be allowed)
     */
    readonly trustedSigners?: string[];
    /**
     *
     * The default amount of time CloudFront will cache an object.
     *
     * This value applies only when your custom origin does not add HTTP headers,
     * such as Cache-Control max-age, Cache-Control s-maxage, and Expires to objects.
     * @default 86400 (1 day)
     *
     */
    readonly defaultTtlSeconds?: number;
    /**
     * The method this CloudFront distribution responds do.
     *
     * @default GET_HEAD
     */
    readonly allowedMethods?: CloudFrontAllowedMethods;
    /**
     * The path this behavior responds to.
     * Required for all non-default behaviors. (The default behavior implicitly has "*" as the path pattern. )
     *
     */
    readonly pathPattern?: string;
    /**
     * Which methods are cached by CloudFront by default.
     *
     * @default GET_HEAD
     */
    readonly cachedMethods?: CloudFrontAllowedCachedMethods;
    /**
     * The values CloudFront will forward to the origin when making a request.
     *
     * @default none (no cookies - no headers)
     *
     */
    readonly forwardedValues?: CfnDistribution.ForwardedValuesProperty;
    /**
     * The minimum amount of time that you want objects to stay in the cache
     * before CloudFront queries your origin.
     */
    readonly minTtlSeconds?: number;
    /**
     * The max amount of time you want objects to stay in the cache
     * before CloudFront queries your origin.
     *
     * @default 31536000 (one year)
     */
    readonly maxTtlSeconds?: number;
    /**
     * Declares associated lambda@edge functions for this distribution behaviour.
     *
     * @default No lambda function associated
     */
    readonly lambdaFunctionAssociations?: LambdaFunctionAssociation[];
}
export interface LambdaFunctionAssociation {
    /**
     * The lambda event type defines at which event the lambda
     * is called during the request lifecycle
     */
    readonly eventType: LambdaEdgeEventType;
    /**
     * A version of the lambda to associate
     */
    readonly lambdaFunction: lambda.IVersion;
}
export declare enum LambdaEdgeEventType {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    OriginRequest = "origin-request",
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    OriginResponse = "origin-response",
    /**
     * The viewer-request specifies the incoming request
     */
    ViewerRequest = "viewer-request",
    /**
     * The viewer-response specifies the outgoing reponse
     */
    ViewerResponse = "viewer-response"
}
export interface ErrorConfiguration {
    /**
     * The error code matched from the origin
     */
    readonly originErrorCode: number;
    /**
     * The error code that is sent to the caller.
     */
    readonly respondWithErrorCode: number;
    /**
     * The path to service instead
     */
    readonly respondWithPage: string;
    /**
     * How long before this error is retried.
     */
    readonly cacheTtl?: number;
}
export interface CloudFrontWebDistributionProps {
    /**
     * AliasConfiguration is used to configured CloudFront to respond to requests on custom domain names.
     *
     * @default - None.
     */
    readonly aliasConfiguration?: AliasConfiguration;
    /**
     * A comment for this distribution in the CloudFront console.
     *
     * @default - No comment is added to distribution.
     */
    readonly comment?: string;
    /**
     * The default object to serve.
     *
     * @default - "index.html" is served.
     */
    readonly defaultRootObject?: string;
    /**
     * If your distribution should have IPv6 enabled.
     *
     * @default true
     */
    readonly enableIpV6?: boolean;
    /**
     * The max supported HTTP Versions.
     *
     * @default HttpVersion.HTTP2
     */
    readonly httpVersion?: HttpVersion;
    /**
     * The price class for the distribution (this impacts how many locations CloudFront uses for your distribution, and billing)
     *
     * @default PriceClass.PriceClass100 the cheapest option for CloudFront is picked by default.
     */
    readonly priceClass?: PriceClass;
    /**
     * The default viewer policy for incoming clients.
     *
     * @default RedirectToHTTPs
     */
    readonly viewerProtocolPolicy?: ViewerProtocolPolicy;
    /**
     * The origin configurations for this distribution. Behaviors are a part of the origin.
     */
    readonly originConfigs: SourceConfiguration[];
    /**
     * Optional - if we should enable logging.
     * You can pass an empty object ({}) to have us auto create a bucket for logging.
     * Omission of this property indicates no logging is to be enabled.
     *
     * @default - no logging is enabled by default.
     */
    readonly loggingConfig?: LoggingConfiguration;
    /**
     * How CloudFront should handle requests that are not successful (eg PageNotFound)
     *
     * By default, CloudFront does not replace HTTP status codes in the 4xx and 5xx range
     * with custom error messages. CloudFront does not cache HTTP status codes.
     *
     * @default - No custom error configuration.
     */
    readonly errorConfigurations?: CfnDistribution.CustomErrorResponseProperty[];
    /**
     * Unique identifier that specifies the AWS WAF web ACL to associate with this CloudFront distribution.
     * @see https://docs.aws.amazon.com/waf/latest/developerguide/what-is-aws-waf.html
     *
     * @default - No AWS Web Application Firewall web access control list (web ACL).
     */
    readonly webACLId?: string;
}
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFront } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
export declare class CloudFrontWebDistribution extends cdk.Construct implements IDistribution {
    /**
     * The logging bucket for this CloudFront distribution.
     * If logging is not enabled for this distribution - this property will be undefined.
     */
    readonly loggingBucket?: s3.IBucket;
    /**
     * The domain name created by CloudFront for this distribution.
     * If you are using aliases for your distribution, this is the domainName your DNS records should point to.
     * (In Route53, you could create an ALIAS record to this value, for example. )
     */
    readonly domainName: string;
    /**
     * The distribution ID for this distribution.
     */
    readonly distributionId: string;
    /**
     * Maps our methods to the string arrays they are
     */
    private readonly METHOD_LOOKUP_MAP;
    /**
     * Maps for which SecurityPolicyProtocol are available to which SSLMethods
     */
    private readonly VALID_SSL_PROTOCOLS;
    constructor(scope: cdk.Construct, id: string, props: CloudFrontWebDistributionProps);
    private toBehavior;
}
