"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PriceClass100"] = "PriceClass_100";
    PriceClass["PriceClass200"] = "PriceClass_200";
    PriceClass["PriceClassAll"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPSOnly"] = "https-only";
    ViewerProtocolPolicy["RedirectToHTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["AllowAll"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSLv3"] = "SSLv3";
    SecurityPolicyProtocol["TLSv1"] = "TLSv1";
    SecurityPolicyProtocol["TLSv1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLSv1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLSv1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSLv3"] = "SSLv3";
    OriginSslPolicy["TLSv1"] = "TLSv1";
    OriginSslPolicy["TLSv1_1"] = "TLSv1.1";
    OriginSslPolicy["TLSv1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HttpOnly"] = "http-only";
    OriginProtocolPolicy["MatchViewer"] = "match-viewer";
    OriginProtocolPolicy["HttpsOnly"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["OriginRequest"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["OriginResponse"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["ViewerRequest"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["ViewerResponse"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFront } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ["GET", "HEAD"],
            GHO: ["GET", "HEAD", "OPTIONS"],
            ALL: ["DELETE", "GET", "HEAD", "OPTIONS", "PATCH", "POST", "PUT"],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            "sni-only": [
                SecurityPolicyProtocol.TLSv1, SecurityPolicyProtocol.TLSv1_1_2016,
                SecurityPolicyProtocol.TLSv1_2016, SecurityPolicyProtocol.TLSv1_2_2018
            ],
            "vip": [SecurityPolicyProtocol.SSLv3, SecurityPolicyProtocol.TLSv1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : "index.html",
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PriceClass100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error("There must be at least one origin source - either an s3OriginSource or a customOriginSource");
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error("There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.");
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key]
                    };
                    originHeaders.push(oHeader);
                });
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig: originConfig.s3OriginSource && originConfig.s3OriginSource.originAccessIdentityId
                    ? { originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentityId}` }
                    : originConfig.s3OriginSource
                        ? {}
                        : undefined,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeoutSeconds || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeoutSeconds || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HttpsOnly,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLSv1_2]
                    }
                    : undefined
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error("There can only be one default behavior across all sources. [ One default behavior per distribution ].");
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error("pathPattern is required for all non-default behaviors");
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors };
        if (props.aliasConfiguration) {
            const minimumProtocolVersion = props.aliasConfiguration.securityPolicy;
            const sslSupportMethod = props.aliasConfiguration.sslMethod || SSLMethod.SNI;
            const acmCertificateArn = props.aliasConfiguration.acmCertRef;
            distributionConfig = {
                ...distributionConfig,
                aliases: props.aliasConfiguration.names,
                viewerCertificate: {
                    acmCertificateArn,
                    sslSupportMethod,
                    minimumProtocolVersion
                }
            };
            if (minimumProtocolVersion !== undefined) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod.toString()];
                if (validProtocols === undefined) {
                    throw new Error(`Invalid sslMethod. ${sslSupportMethod.toString()} is not fully implemented yet.`);
                }
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(", ")}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true }
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, `LoggingBucket`);
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix
                }
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.refAsString;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress,
            defaultTtl: input.defaultTtlSeconds,
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: "none" } },
            maxTtl: input.maxTtlSeconds,
            minTtl: input.minTtlSeconds,
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.RedirectToHTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                }))
            });
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoid2ViX2Rpc3RyaWJ1dGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIndlYl9kaXN0cmlidXRpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFDQSxzQ0FBdUM7QUFDdkMsb0NBQXFDO0FBQ3JDLGlFQUF5RDtBQUd6RCxJQUFZLFdBR1g7QUFIRCxXQUFZLFdBQVc7SUFDckIsa0NBQW1CLENBQUE7SUFDbkIsOEJBQWUsQ0FBQTtBQUNqQixDQUFDLEVBSFcsV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUFHdEI7QUFFRDs7R0FFRztBQUNILElBQVksVUFJWDtBQUpELFdBQVksVUFBVTtJQUNwQiw4Q0FBZ0MsQ0FBQTtJQUNoQyw4Q0FBZ0MsQ0FBQTtJQUNoQyw4Q0FBZ0MsQ0FBQTtBQUNsQyxDQUFDLEVBSlcsVUFBVSxHQUFWLGtCQUFVLEtBQVYsa0JBQVUsUUFJckI7QUFFRDs7R0FFRztBQUNILElBQVksb0JBSVg7QUFKRCxXQUFZLG9CQUFvQjtJQUM5QixnREFBd0IsQ0FBQTtJQUN4Qiw2REFBcUMsQ0FBQTtJQUNyQyw4Q0FBc0IsQ0FBQTtBQUN4QixDQUFDLEVBSlcsb0JBQW9CLEdBQXBCLDRCQUFvQixLQUFwQiw0QkFBb0IsUUFJL0I7QUE0Q0Q7Ozs7Ozs7Ozs7Ozs7OztHQWVHO0FBQ0gsSUFBWSxTQUdYO0FBSEQsV0FBWSxTQUFTO0lBQ25CLDZCQUFnQixDQUFBO0lBQ2hCLHdCQUFXLENBQUE7QUFDYixDQUFDLEVBSFcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUFHcEI7QUFFRDs7O0dBR0c7QUFDSCxJQUFZLHNCQU1YO0FBTkQsV0FBWSxzQkFBc0I7SUFDaEMseUNBQWUsQ0FBQTtJQUNmLHlDQUFlLENBQUE7SUFDZixtREFBeUIsQ0FBQTtJQUN6Qix1REFBNkIsQ0FBQTtJQUM3Qix1REFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBTlcsc0JBQXNCLEdBQXRCLDhCQUFzQixLQUF0Qiw4QkFBc0IsUUFNakM7QUF5SEQsSUFBWSxlQUtYO0FBTEQsV0FBWSxlQUFlO0lBQ3pCLGtDQUFlLENBQUE7SUFDZixrQ0FBZSxDQUFBO0lBQ2Ysc0NBQW1CLENBQUE7SUFDbkIsc0NBQW1CLENBQUE7QUFDckIsQ0FBQyxFQUxXLGVBQWUsR0FBZix1QkFBZSxLQUFmLHVCQUFlLFFBSzFCO0FBRUQsSUFBWSxvQkFJWDtBQUpELFdBQVksb0JBQW9CO0lBQzlCLDhDQUFzQixDQUFBO0lBQ3RCLG9EQUE0QixDQUFBO0lBQzVCLGdEQUF3QixDQUFBO0FBQzFCLENBQUMsRUFKVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQUkvQjtBQWNEOztHQUVHO0FBQ0gsSUFBWSx3QkFJWDtBQUpELFdBQVksd0JBQXdCO0lBQ2xDLDJDQUFlLENBQUE7SUFDZixvREFBd0IsQ0FBQTtJQUN4Qix1Q0FBVyxDQUFBO0FBQ2IsQ0FBQyxFQUpXLHdCQUF3QixHQUF4QixnQ0FBd0IsS0FBeEIsZ0NBQXdCLFFBSW5DO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLDhCQUdYO0FBSEQsV0FBWSw4QkFBOEI7SUFDeEMsaURBQWUsQ0FBQTtJQUNmLDBEQUF3QixDQUFBO0FBQzFCLENBQUMsRUFIVyw4QkFBOEIsR0FBOUIsc0NBQThCLEtBQTlCLHNDQUE4QixRQUd6QztBQTJHRCxJQUFZLG1CQW1CWDtBQW5CRCxXQUFZLG1CQUFtQjtJQUM3Qjs7O09BR0c7SUFDSCx1REFBZ0MsQ0FBQTtJQUNoQzs7O09BR0c7SUFDSCx5REFBa0MsQ0FBQTtJQUNsQzs7T0FFRztJQUNILHVEQUFnQyxDQUFBO0lBQ2hDOztPQUVHO0lBQ0gseURBQWtDLENBQUE7QUFDcEMsQ0FBQyxFQW5CVyxtQkFBbUIsR0FBbkIsMkJBQW1CLEtBQW5CLDJCQUFtQixRQW1COUI7QUFpSEQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBNkJHO0FBQ0gsTUFBYSx5QkFBMEIsU0FBUSxHQUFHLENBQUMsU0FBUztJQXVDMUQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFxQztRQUNqRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBckJuQjs7V0FFRztRQUNjLHNCQUFpQixHQUFHO1lBQ25DLEVBQUUsRUFBRSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUM7WUFDbkIsR0FBRyxFQUFFLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRSxTQUFTLENBQUM7WUFDL0IsR0FBRyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDO1NBQ2xFLENBQUM7UUFFRjs7V0FFRztRQUNjLHdCQUFtQixHQUFnQztZQUNsRSxVQUFVLEVBQUU7Z0JBQ1Ysc0JBQXNCLENBQUMsS0FBSyxFQUFFLHNCQUFzQixDQUFDLFlBQVk7Z0JBQ2pFLHNCQUFzQixDQUFDLFVBQVUsRUFBRSxzQkFBc0IsQ0FBQyxZQUFZO2FBQ3ZFO1lBQ0QsS0FBSyxFQUFFLENBQUMsc0JBQXNCLENBQUMsS0FBSyxFQUFFLHNCQUFzQixDQUFDLEtBQUssQ0FBQztTQUNwRSxDQUFDO1FBS0EsSUFBSSxrQkFBa0IsR0FBK0M7WUFDbkUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLE9BQU8sRUFBRSxJQUFJO1lBQ2IsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQixLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxZQUFZO1lBQ2pHLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxJQUFJLFdBQVcsQ0FBQyxLQUFLO1lBQ25ELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxJQUFJLFVBQVUsQ0FBQyxhQUFhO1lBQ3hELFdBQVcsRUFBRSxDQUFDLEtBQUssQ0FBQyxVQUFVLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLElBQUk7WUFDdkUsMkNBQTJDO1lBQzNDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxtQkFBbUI7WUFDL0MsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1NBQ3pCLENBQUM7UUFFRixNQUFNLFNBQVMsR0FBeUIsRUFBRSxDQUFDO1FBRTNDLE1BQU0sT0FBTyxHQUFxQyxFQUFFLENBQUM7UUFFckQsSUFBSSxXQUFXLEdBQUcsQ0FBQyxDQUFDO1FBQ3BCLEtBQUssTUFBTSxZQUFZLElBQUksS0FBSyxDQUFDLGFBQWEsRUFBRTtZQUM5QyxNQUFNLFFBQVEsR0FBRyxTQUFTLFdBQVcsRUFBRSxDQUFDO1lBQ3hDLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxJQUFJLENBQUMsWUFBWSxDQUFDLGtCQUFrQixFQUFFO2dCQUNwRSxNQUFNLElBQUksS0FBSyxDQUFDLDZGQUE2RixDQUFDLENBQUM7YUFDaEg7WUFDRCxJQUFJLFlBQVksQ0FBQyxrQkFBa0IsSUFBSSxZQUFZLENBQUMsY0FBYyxFQUFFO2dCQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLGtHQUFrRyxDQUFDLENBQUM7YUFDckg7WUFFRCxNQUFNLGFBQWEsR0FBaUQsRUFBRSxDQUFDO1lBQ3ZFLElBQUksWUFBWSxDQUFDLGFBQWEsRUFBRTtnQkFDOUIsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxFQUFFO29CQUNwRCxNQUFNLE9BQU8sR0FBK0M7d0JBQzFELFVBQVUsRUFBRSxHQUFHO3dCQUNmLFdBQVcsRUFBRSxZQUFZLENBQUMsYUFBYyxDQUFDLEdBQUcsQ0FBQztxQkFDOUMsQ0FBQztvQkFDRixhQUFhLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUM5QixDQUFDLENBQUMsQ0FBQzthQUNKO1lBRUQsTUFBTSxjQUFjLEdBQW1DO2dCQUNyRCxFQUFFLEVBQUUsUUFBUTtnQkFDWixVQUFVLEVBQUUsWUFBWSxDQUFDLGNBQWM7b0JBQ3JDLENBQUMsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLGNBQWMsQ0FBQyx3QkFBd0I7b0JBQ3JFLENBQUMsQ0FBQyxZQUFZLENBQUMsa0JBQW1CLENBQUMsVUFBVTtnQkFDL0MsVUFBVSxFQUFFLFlBQVksQ0FBQyxVQUFVO2dCQUNuQyxtQkFBbUIsRUFBRSxhQUFhLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUN6RSxjQUFjLEVBQUUsWUFBWSxDQUFDLGNBQWMsSUFBSSxZQUFZLENBQUMsY0FBYyxDQUFDLHNCQUFzQjtvQkFDL0YsQ0FBQyxDQUFDLEVBQUUsb0JBQW9CLEVBQUUscUNBQXFDLFlBQVksQ0FBQyxjQUFjLENBQUMsc0JBQXNCLEVBQUUsRUFBRTtvQkFDckgsQ0FBQyxDQUFDLFlBQVksQ0FBQyxjQUFjO3dCQUM3QixDQUFDLENBQUMsRUFBRzt3QkFDTCxDQUFDLENBQUMsU0FBUztnQkFDYixrQkFBa0IsRUFBRSxZQUFZLENBQUMsa0JBQWtCO29CQUNqRCxDQUFDLENBQUM7d0JBQ0EsUUFBUSxFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLElBQUksRUFBRTt3QkFDeEQsU0FBUyxFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLElBQUksR0FBRzt3QkFDM0Qsc0JBQXNCLEVBQUUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLDZCQUE2QixJQUFJLENBQUM7d0JBQzFGLGlCQUFpQixFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyx3QkFBd0IsSUFBSSxFQUFFO3dCQUNqRixvQkFBb0IsRUFBRSxZQUFZLENBQUMsa0JBQWtCLENBQUMsb0JBQW9CLElBQUksb0JBQW9CLENBQUMsU0FBUzt3QkFDNUcsa0JBQWtCLEVBQUUsWUFBWSxDQUFDLGtCQUFrQixDQUFDLHdCQUF3QixJQUFJLENBQUMsZUFBZSxDQUFDLE9BQU8sQ0FBQztxQkFDMUc7b0JBQ0QsQ0FBQyxDQUFDLFNBQVM7YUFDZCxDQUFDO1lBRUYsS0FBSyxNQUFNLFFBQVEsSUFBSSxZQUFZLENBQUMsU0FBUyxFQUFFO2dCQUM3QyxTQUFTLENBQUMsSUFBSSxDQUFDLEVBQUUsR0FBRyxRQUFRLEVBQUUsY0FBYyxFQUFFLFFBQVEsRUFBRSxDQUFDLENBQUM7YUFDM0Q7WUFFRCxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxDQUFDO1lBQzdCLFdBQVcsRUFBRSxDQUFDO1NBQ2Y7UUFFRCxPQUFPLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxJQUFJLENBQUMsTUFBTSxDQUFDLGtCQUFrQixFQUFFO2dCQUN4RCxNQUFNLElBQUksS0FBSyxDQUFDLFVBQVUsTUFBTSxDQUFDLFVBQVUsd0ZBQXdGLENBQUMsQ0FBQzthQUN0STtRQUNILENBQUMsQ0FBQyxDQUFDO1FBQ0gsa0JBQWtCLEdBQUc7WUFDbkIsR0FBRyxrQkFBa0I7WUFDckIsT0FBTztTQUNSLENBQUM7UUFFRixNQUFNLGdCQUFnQixHQUFHLFNBQVMsQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUMsQ0FBQztRQUNsRixJQUFJLGdCQUFnQixDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDakMsTUFBTSxJQUFJLEtBQUssQ0FBQyx1R0FBdUcsQ0FBQyxDQUFDO1NBQzFIO1FBRUQsa0JBQWtCLEdBQUcsRUFBRSxHQUFHLGtCQUFrQixFQUFFLG9CQUFvQixFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUFDLEVBQUUsQ0FBQztRQUV2SSxNQUFNLGNBQWMsR0FBNEMsRUFBRSxDQUFDO1FBQ25FLEtBQUssTUFBTSxRQUFRLElBQUksU0FBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLGlCQUFpQixDQUFDLEVBQUU7WUFDbEUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUU7Z0JBQ3pCLE1BQU0sSUFBSSxLQUFLLENBQUMsdURBQXVELENBQUMsQ0FBQzthQUMxRTtZQUNELGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixDQUEwQyxDQUFDLENBQUM7U0FDckg7UUFFRCxrQkFBa0IsR0FBRyxFQUFFLEdBQUcsa0JBQWtCLEVBQUUsY0FBYyxFQUFFLGNBQWMsRUFBRSxDQUFDO1FBRS9FLElBQUksS0FBSyxDQUFDLGtCQUFrQixFQUFFO1lBQzVCLE1BQU0sc0JBQXNCLEdBQUcsS0FBSyxDQUFDLGtCQUFrQixDQUFDLGNBQWMsQ0FBQztZQUN2RSxNQUFNLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLElBQUksU0FBUyxDQUFDLEdBQUcsQ0FBQztZQUM3RSxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxVQUFVLENBQUM7WUFFOUQsa0JBQWtCLEdBQUc7Z0JBQ25CLEdBQUcsa0JBQWtCO2dCQUNyQixPQUFPLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEtBQUs7Z0JBQ3ZDLGlCQUFpQixFQUFFO29CQUNqQixpQkFBaUI7b0JBQ2pCLGdCQUFnQjtvQkFDaEIsc0JBQXNCO2lCQUN2QjthQUNGLENBQUM7WUFFRixJQUFJLHNCQUFzQixLQUFLLFNBQVMsRUFBRTtnQkFDeEMsTUFBTSxjQUFjLEdBQUcsSUFBSSxDQUFDLG1CQUFtQixDQUFDLGdCQUFnQixDQUFDLFFBQVEsRUFBRSxDQUFDLENBQUM7Z0JBRTdFLElBQUksY0FBYyxLQUFLLFNBQVMsRUFBRTtvQkFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzQkFBc0IsZ0JBQWdCLENBQUMsUUFBUSxFQUFFLGdDQUFnQyxDQUFDLENBQUM7aUJBQ3BHO2dCQUVELElBQUksY0FBYyxDQUFDLE9BQU8sQ0FBQyxzQkFBc0IsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO29CQUNwRSwyQ0FBMkM7b0JBQzNDLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxzQkFBc0Isc0NBQXNDLGdCQUFnQiw2QkFBNkIsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7aUJBQzFKO2FBQ0Y7U0FDRjthQUFNO1lBQ0wsa0JBQWtCLEdBQUcsRUFBRSxHQUFHLGtCQUFrQjtnQkFDMUMsaUJBQWlCLEVBQUUsRUFBRSw0QkFBNEIsRUFBRSxJQUFJLEVBQUU7YUFDMUQsQ0FBQztTQUNIO1FBRUQsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNLElBQUksSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxlQUFlLENBQUMsQ0FBQztZQUN4RixrQkFBa0IsR0FBRztnQkFDbkIsR0FBRyxrQkFBa0I7Z0JBQ3JCLE9BQU8sRUFBRTtvQkFDUCxNQUFNLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyx3QkFBd0I7b0JBQ25ELGNBQWMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLGNBQWMsSUFBSSxLQUFLO29CQUMzRCxNQUFNLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNO2lCQUNuQzthQUNGLENBQUM7U0FDSDtRQUVELE1BQU0sWUFBWSxHQUFHLElBQUksc0NBQWUsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsRUFBRSxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDekYsSUFBSSxDQUFDLFVBQVUsR0FBRyxZQUFZLENBQUMsY0FBYyxDQUFDO1FBQzlDLElBQUksQ0FBQyxjQUFjLEdBQUcsWUFBWSxDQUFDLFdBQVcsQ0FBQztJQUNqRCxDQUFDO0lBRU8sVUFBVSxDQUFDLEtBQXlCLEVBQUUsV0FBa0M7UUFDOUUsSUFBSSxRQUFRLEdBQUc7WUFDYixjQUFjLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxjQUFjLElBQUksd0JBQXdCLENBQUMsUUFBUSxDQUFDO1lBQ2pHLGFBQWEsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSw4QkFBOEIsQ0FBQyxRQUFRLENBQUM7WUFDckcsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLFVBQVUsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQ25DLGVBQWUsRUFBRSxLQUFLLENBQUMsZUFBZSxJQUFJLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxPQUFPLEVBQUUsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLEVBQUU7WUFDOUYsTUFBTSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQzNCLE1BQU0sRUFBRSxLQUFLLENBQUMsYUFBYTtZQUMzQixjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLG9CQUFvQixFQUFFLFdBQVcsSUFBSSxvQkFBb0IsQ0FBQyxlQUFlO1NBQzFFLENBQUM7UUFDRixJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQixFQUFFO1lBQzVCLFFBQVEsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQztTQUN4RTtRQUNELElBQUksS0FBSyxDQUFDLDBCQUEwQixFQUFFO1lBQ3BDLFFBQVEsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsRUFBRTtnQkFDakMsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLDBCQUEwQjtxQkFDekQsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDWCxTQUFTLEVBQUUsR0FBRyxDQUFDLFNBQVM7b0JBQ3hCLGlCQUFpQixFQUFFLEdBQUcsQ0FBQyxjQUFjLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxXQUFXO2lCQUN4RSxDQUFDLENBQUM7YUFDTixDQUFDLENBQUM7U0FDSjtRQUNELE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7Q0FDRjtBQXZORCw4REF1TkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgbGFtYmRhID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLWxhbWJkYScpO1xuaW1wb3J0IHMzID0gcmVxdWlyZSgnQGF3cy1jZGsvYXdzLXMzJyk7XG5pbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBDZm5EaXN0cmlidXRpb24gfSBmcm9tICcuL2Nsb3VkZnJvbnQuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElEaXN0cmlidXRpb24gfSBmcm9tICcuL2Rpc3RyaWJ1dGlvbic7XG5cbmV4cG9ydCBlbnVtIEh0dHBWZXJzaW9uIHtcbiAgSFRUUDFfMSA9IFwiaHR0cDEuMVwiLFxuICBIVFRQMiA9IFwiaHR0cDJcIlxufVxuXG4vKipcbiAqIFRoZSBwcmljZSBjbGFzcyBkZXRlcm1pbmVzIGhvdyBtYW55IGVkZ2UgbG9jYXRpb25zIENsb3VkRnJvbnQgd2lsbCB1c2UgZm9yIHlvdXIgZGlzdHJpYnV0aW9uLlxuICovXG5leHBvcnQgZW51bSBQcmljZUNsYXNzIHtcbiAgUHJpY2VDbGFzczEwMCA9IFwiUHJpY2VDbGFzc18xMDBcIixcbiAgUHJpY2VDbGFzczIwMCA9IFwiUHJpY2VDbGFzc18yMDBcIixcbiAgUHJpY2VDbGFzc0FsbCA9IFwiUHJpY2VDbGFzc19BbGxcIlxufVxuXG4vKipcbiAqIEhvdyBIVFRQcyBzaG91bGQgYmUgaGFuZGxlZCB3aXRoIHlvdXIgZGlzdHJpYnV0aW9uLlxuICovXG5leHBvcnQgZW51bSBWaWV3ZXJQcm90b2NvbFBvbGljeSB7XG4gIEhUVFBTT25seSA9IFwiaHR0cHMtb25seVwiLFxuICBSZWRpcmVjdFRvSFRUUFMgPSBcInJlZGlyZWN0LXRvLWh0dHBzXCIsXG4gIEFsbG93QWxsID0gXCJhbGxvdy1hbGxcIlxufVxuXG4vKipcbiAqIENvbmZpZ3VyYXRpb24gZm9yIGN1c3RvbSBkb21haW4gbmFtZXNcbiAqXG4gKiBDbG91ZEZyb250IGNhbiB1c2UgYSBjdXN0b20gZG9tYWluIHRoYXQgeW91IHByb3ZpZGUgaW5zdGVhZCBvZiBhXG4gKiBcImNsb3VkZnJvbnQubmV0XCIgZG9tYWluLiBUbyB1c2UgdGhpcyBmZWF0dXJlIHlvdSBtdXN0IHByb3ZpZGUgdGhlIGxpc3Qgb2ZcbiAqIGFkZGl0aW9uYWwgZG9tYWlucywgYW5kIHRoZSBBQ00gQ2VydGlmaWNhdGUgdGhhdCBDbG91ZEZyb250IHNob3VsZCB1c2UgZm9yXG4gKiB0aGVzZSBhZGRpdGlvbmFsIGRvbWFpbnMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNDb25maWd1cmF0aW9uIHtcbiAgLyoqXG4gICAqIEFSTiBvZiBhbiBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlciAoQUNNKSBjZXJ0aWZpY2F0ZS5cbiAgICovXG4gIHJlYWRvbmx5IGFjbUNlcnRSZWY6IHN0cmluZztcblxuICAvKipcbiAgICogRG9tYWluIG5hbWVzIG9uIHRoZSBjZXJ0aWZpY2F0ZVxuICAgKlxuICAgKiBCb3RoIG1haW4gZG9tYWluIG5hbWUgYW5kIFN1YmplY3QgQWx0ZXJuYXRpdmUgTmFtZXMuXG4gICAqL1xuICByZWFkb25seSBuYW1lczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIEhvdyBDbG91ZEZyb250IHNob3VsZCBzZXJ2ZSBIVFRQUyByZXF1ZXN0cy5cbiAgICpcbiAgICogU2VlIHRoZSBub3RlcyBvbiBTU0xNZXRob2QgaWYgeW91IHdpc2ggdG8gdXNlIG90aGVyIFNTTCB0ZXJtaW5hdGlvbiB0eXBlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgU1NMTWV0aG9kLlNOSVxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jbG91ZGZyb250L2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1ZpZXdlckNlcnRpZmljYXRlLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHNzbE1ldGhvZD86IFNTTE1ldGhvZDtcblxuICAvKipcbiAgICogVGhlIG1pbmltdW0gdmVyc2lvbiBvZiB0aGUgU1NMIHByb3RvY29sIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byB1c2UgZm9yIEhUVFBTIGNvbm5lY3Rpb25zLlxuICAgKlxuICAgKiBDbG91ZEZyb250IHNlcnZlcyB5b3VyIG9iamVjdHMgb25seSB0byBicm93c2VycyBvciBkZXZpY2VzIHRoYXQgc3VwcG9ydCBhdFxuICAgKiBsZWFzdCB0aGUgU1NMIHZlcnNpb24gdGhhdCB5b3Ugc3BlY2lmeS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBTU0x2MyBpZiBzc2xNZXRob2QgVklQLCBUTFN2MSBpZiBzc2xNZXRob2QgU05JXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eVBvbGljeT86IFNlY3VyaXR5UG9saWN5UHJvdG9jb2w7XG59XG5cbi8qKlxuICogVGhlIFNTTCBtZXRob2QgQ2xvdWRGcm9udCB3aWxsIHVzZSBmb3IgeW91ciBkaXN0cmlidXRpb24uXG4gKlxuICogU2VydmVyIE5hbWUgSW5kaWNhdGlvbiAoU05JKSAtIGlzIGFuIGV4dGVuc2lvbiB0byB0aGUgVExTIGNvbXB1dGVyIG5ldHdvcmtpbmcgcHJvdG9jb2wgYnkgd2hpY2ggYSBjbGllbnQgaW5kaWNhdGVzXG4gKiAgd2hpY2ggaG9zdG5hbWUgaXQgaXMgYXR0ZW1wdGluZyB0byBjb25uZWN0IHRvIGF0IHRoZSBzdGFydCBvZiB0aGUgaGFuZHNoYWtpbmcgcHJvY2Vzcy4gVGhpcyBhbGxvd3MgYSBzZXJ2ZXIgdG8gcHJlc2VudFxuICogIG11bHRpcGxlIGNlcnRpZmljYXRlcyBvbiB0aGUgc2FtZSBJUCBhZGRyZXNzIGFuZCBUQ1AgcG9ydCBudW1iZXIgYW5kIGhlbmNlIGFsbG93cyBtdWx0aXBsZSBzZWN1cmUgKEhUVFBTKSB3ZWJzaXRlc1xuICogKG9yIGFueSBvdGhlciBzZXJ2aWNlIG92ZXIgVExTKSB0byBiZSBzZXJ2ZWQgYnkgdGhlIHNhbWUgSVAgYWRkcmVzcyB3aXRob3V0IHJlcXVpcmluZyBhbGwgdGhvc2Ugc2l0ZXMgdG8gdXNlIHRoZSBzYW1lIGNlcnRpZmljYXRlLlxuICpcbiAqIENsb3VkRnJvbnQgY2FuIHVzZSBTTkkgdG8gaG9zdCBtdWx0aXBsZSBkaXN0cmlidXRpb25zIG9uIHRoZSBzYW1lIElQIC0gd2hpY2ggYSBsYXJnZSBtYWpvcml0eSBvZiBjbGllbnRzIHdpbGwgc3VwcG9ydC5cbiAqXG4gKiBJZiB5b3VyIGNsaWVudHMgY2Fubm90IHN1cHBvcnQgU05JIGhvd2V2ZXIgLSBDbG91ZEZyb250IGNhbiB1c2UgZGVkaWNhdGVkIElQcyBmb3IgeW91ciBkaXN0cmlidXRpb24gLSBidXQgdGhlcmUgaXMgYSBwcm9yYXRlZCBtb250aGx5IGNoYXJnZSBmb3JcbiAqIHVzaW5nIHRoaXMgZmVhdHVyZS4gQnkgZGVmYXVsdCwgd2UgdXNlIFNOSSAtIGJ1dCB5b3UgY2FuIG9wdGlvbmFsbHkgZW5hYmxlIGRlZGljYXRlZCBJUHMgKFZJUCkuXG4gKlxuICogU2VlIHRoZSBDbG91ZEZyb250IFNTTCBmb3IgbW9yZSBkZXRhaWxzIGFib3V0IHByaWNpbmcgOiBodHRwczovL2F3cy5hbWF6b24uY29tL2Nsb3VkZnJvbnQvY3VzdG9tLXNzbC1kb21haW5zL1xuICpcbiAqL1xuZXhwb3J0IGVudW0gU1NMTWV0aG9kIHtcbiAgU05JID0gXCJzbmktb25seVwiLFxuICBWSVAgPSBcInZpcFwiXG59XG5cbi8qKlxuICogVGhlIG1pbmltdW0gdmVyc2lvbiBvZiB0aGUgU1NMIHByb3RvY29sIHRoYXQgeW91IHdhbnQgQ2xvdWRGcm9udCB0byB1c2UgZm9yIEhUVFBTIGNvbm5lY3Rpb25zLlxuICogQ2xvdWRGcm9udCBzZXJ2ZXMgeW91ciBvYmplY3RzIG9ubHkgdG8gYnJvd3NlcnMgb3IgZGV2aWNlcyB0aGF0IHN1cHBvcnQgYXQgbGVhc3QgdGhlIFNTTCB2ZXJzaW9uIHRoYXQgeW91IHNwZWNpZnkuXG4gKi9cbmV4cG9ydCBlbnVtIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wge1xuICBTU0x2MyA9IFwiU1NMdjNcIixcbiAgVExTdjEgPSBcIlRMU3YxXCIsXG4gIFRMU3YxXzIwMTYgPSBcIlRMU3YxXzIwMTZcIixcbiAgVExTdjFfMV8yMDE2ID0gXCJUTFN2MS4xXzIwMTZcIixcbiAgVExTdjFfMl8yMDE4ID0gXCJUTFN2MS4yXzIwMThcIlxufVxuXG4vKipcbiAqIExvZ2dpbmcgY29uZmlndXJhdGlvbiBmb3IgaW5jb21pbmcgcmVxdWVzdHNcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2dnaW5nQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBCdWNrZXQgdG8gbG9nIHJlcXVlc3RzIHRvXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gQSBsb2dnaW5nIGJ1Y2tldCBpcyBhdXRvbWF0aWNhbGx5IGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ/OiBzMy5JQnVja2V0LFxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGluY2x1ZGUgdGhlIGNvb2tpZXMgaW4gdGhlIGxvZ3NcbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGluY2x1ZGVDb29raWVzPzogYm9vbGVhbixcblxuICAvKipcbiAgICogV2hlcmUgaW4gdGhlIGJ1Y2tldCB0byBzdG9yZSBsb2dzXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gcHJlZml4LlxuICAgKi9cbiAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nXG59XG5cbi8qKlxuICogQSBzb3VyY2UgY29uZmlndXJhdGlvbiBpcyBhIHdyYXBwZXIgZm9yIENsb3VkRnJvbnQgb3JpZ2lucyBhbmQgYmVoYXZpb3JzLlxuICogQW4gb3JpZ2luIGlzIHdoYXQgQ2xvdWRGcm9udCB3aWxsIFwiYmUgaW4gZnJvbnQgb2ZcIiAtIHRoYXQgaXMsIENsb3VkRnJvbnQgd2lsbCBwdWxsIGl0J3MgYXNzZXRzIGZyb20gYW4gb3JpZ2luLlxuICpcbiAqIElmIHlvdSdyZSB1c2luZyBzMyBhcyBhIHNvdXJjZSAtIHBhc3MgdGhlIGBzM09yaWdpbmAgcHJvcGVydHksIG90aGVyd2lzZSwgcGFzcyB0aGUgYGN1c3RvbU9yaWdpblNvdXJjZWAgcHJvcGVydHkuXG4gKlxuICogT25lIG9yIHRoZSBvdGhlciBtdXN0IGJlIHBhc3NlZCwgYW5kIGl0IGlzIGludmFsaWQgdG8gcGFzcyBib3RoIGluIHRoZSBzYW1lIFNvdXJjZUNvbmZpZ3VyYXRpb24uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU291cmNlQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBBbiBzMyBvcmlnaW4gc291cmNlIC0gaWYgeW91J3JlIHVzaW5nIHMzIGZvciB5b3VyIGFzc2V0c1xuICAgKi9cbiAgcmVhZG9ubHkgczNPcmlnaW5Tb3VyY2U/OiBTM09yaWdpbkNvbmZpZ1xuXG4gIC8qKlxuICAgKiBBIGN1c3RvbSBvcmlnaW4gc291cmNlIC0gZm9yIGFsbCBub24tczMgc291cmNlcy5cbiAgICovXG4gIHJlYWRvbmx5IGN1c3RvbU9yaWdpblNvdXJjZT86IEN1c3RvbU9yaWdpbkNvbmZpZyxcblxuICAvKipcbiAgICogVGhlIGJlaGF2aW9ycyBhc3NvY2lhdGVkIHdpdGggdGhpcyBzb3VyY2UuXG4gICAqIEF0IGxlYXN0IG9uZSAoZGVmYXVsdCkgYmVoYXZpb3IgbXVzdCBiZSBpbmNsdWRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGJlaGF2aW9yczogQmVoYXZpb3JbXTtcblxuICAvKipcbiAgICogVGhlIHJlbGF0aXZlIHBhdGggdG8gdGhlIG9yaWdpbiByb290IHRvIHVzZSBmb3Igc291cmNlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgL1xuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luUGF0aD86IHN0cmluZyxcblxuICAvKipcbiAgICogQW55IGFkZGl0aW9uYWwgaGVhZGVycyB0byBwYXNzIHRvIHRoZSBvcmlnaW5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIGhlYWRlcnMgYXJlIHBhc3NlZC5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbkhlYWRlcnM/OiB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG4vKipcbiAqIEEgY3VzdG9tIG9yaWdpbiBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tT3JpZ2luQ29uZmlnIHtcbiAgLyoqXG4gICAqIFRoZSBkb21haW4gbmFtZSBvZiB0aGUgY3VzdG9tIG9yaWdpbi4gU2hvdWxkIG5vdCBpbmNsdWRlIHRoZSBwYXRoIC0gdGhhdCBzaG91bGQgYmUgaW4gdGhlIHBhcmVudCBTb3VyY2VDb25maWd1cmF0aW9uXG4gICAqL1xuICByZWFkb25seSBkb21haW5OYW1lOiBzdHJpbmcsXG5cbiAgLyoqXG4gICAqIFRoZSBvcmlnaW4gSFRUUCBwb3J0XG4gICAqXG4gICAqIEBkZWZhdWx0IDgwXG4gICAqL1xuICByZWFkb25seSBodHRwUG9ydD86IG51bWJlcixcblxuICAvKipcbiAgICogVGhlIG9yaWdpbiBIVFRQUyBwb3J0XG4gICAqXG4gICAqIEBkZWZhdWx0IDQ0M1xuICAgKi9cbiAgcmVhZG9ubHkgaHR0cHNQb3J0PzogbnVtYmVyLFxuXG4gIC8qKlxuICAgKiBUaGUga2VlcCBhbGl2ZSB0aW1lb3V0IHdoZW4gbWFraW5nIGNhbGxzIGluIHNlY29uZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IDVcbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbktlZXBhbGl2ZVRpbWVvdXRTZWNvbmRzPzogbnVtYmVyLFxuXG4gIC8qKlxuICAgKiBUaGUgcHJvdG9jb2wgKGh0dHAgb3IgaHR0cHMpIHBvbGljeSB0byB1c2Ugd2hlbiBpbnRlcmFjdGluZyB3aXRoIHRoZSBvcmlnaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IE9yaWdpblByb3RvY29sUG9saWN5Lkh0dHBzT25seVxuICAgKi9cbiAgcmVhZG9ubHkgb3JpZ2luUHJvdG9jb2xQb2xpY3k/OiBPcmlnaW5Qcm90b2NvbFBvbGljeSxcblxuICAvKipcbiAgICogVGhlIHJlYWQgdGltZW91dCB3aGVuIGNhbGxpbmcgdGhlIG9yaWdpbiBpbiBzZWNvbmRzXG4gICAqXG4gICAqIEBkZWZhdWx0IDMwXG4gICAqL1xuICByZWFkb25seSBvcmlnaW5SZWFkVGltZW91dFNlY29uZHM/OiBudW1iZXJcblxuICAvKipcbiAgICogVGhlIFNTTCB2ZXJzaW9ucyB0byB1c2Ugd2hlbiBpbnRlcmFjdGluZyB3aXRoIHRoZSBvcmlnaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IE9yaWdpblNzbFBvbGljeS5UTFN2MV8yXG4gICAqL1xuICByZWFkb25seSBhbGxvd2VkT3JpZ2luU1NMVmVyc2lvbnM/OiBPcmlnaW5Tc2xQb2xpY3lbXTtcblxufVxuXG5leHBvcnQgZW51bSBPcmlnaW5Tc2xQb2xpY3kge1xuICBTU0x2MyA9IFwiU1NMdjNcIixcbiAgVExTdjEgPSBcIlRMU3YxXCIsXG4gIFRMU3YxXzEgPSBcIlRMU3YxLjFcIixcbiAgVExTdjFfMiA9IFwiVExTdjEuMlwiLFxufVxuXG5leHBvcnQgZW51bSBPcmlnaW5Qcm90b2NvbFBvbGljeSB7XG4gIEh0dHBPbmx5ID0gXCJodHRwLW9ubHlcIixcbiAgTWF0Y2hWaWV3ZXIgPSBcIm1hdGNoLXZpZXdlclwiLFxuICBIdHRwc09ubHkgPSBcImh0dHBzLW9ubHlcIixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBTM09yaWdpbkNvbmZpZyB7XG4gIC8qKlxuICAgKiBUaGUgc291cmNlIGJ1Y2tldCB0byBzZXJ2ZSBjb250ZW50IGZyb21cbiAgICovXG4gIHJlYWRvbmx5IHMzQnVja2V0U291cmNlOiBzMy5JQnVja2V0O1xuXG4gIC8qKlxuICAgKiBUaGUgb3B0aW9uYWwgSUQgb2YgdGhlIG9yaWdpbiBpZGVudGl0eSBjbG91ZGZyb250IHdpbGwgdXNlIHdoZW4gY2FsbGluZyB5b3VyIHMzIGJ1Y2tldC5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbkFjY2Vzc0lkZW50aXR5SWQ/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQW4gZW51bSBmb3IgdGhlIHN1cHBvcnRlZCBtZXRob2RzIHRvIGEgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gKi9cbmV4cG9ydCBlbnVtIENsb3VkRnJvbnRBbGxvd2VkTWV0aG9kcyB7XG4gIEdFVF9IRUFEID0gXCJHSFwiLFxuICBHRVRfSEVBRF9PUFRJT05TID0gXCJHSE9cIixcbiAgQUxMID0gXCJBTExcIlxufVxuXG4vKipcbiAqIEVudW1zIGZvciB0aGUgbWV0aG9kcyBDbG91ZEZyb250IGNhbiBjYWNoZS5cbiAqL1xuZXhwb3J0IGVudW0gQ2xvdWRGcm9udEFsbG93ZWRDYWNoZWRNZXRob2RzIHtcbiAgR0VUX0hFQUQgPSBcIkdIXCIsXG4gIEdFVF9IRUFEX09QVElPTlMgPSBcIkdIT1wiLFxufVxuXG4vKipcbiAqIEEgQ2xvdWRGcm9udCBiZWhhdmlvciB3cmFwcGVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJlaGF2aW9yIHtcblxuICAvKipcbiAgICogSWYgQ2xvdWRGcm9udCBzaG91bGQgYXV0b21hdGljYWxseSBjb21wcmVzcyBzb21lIGNvbnRlbnQgdHlwZXMuXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNvbXByZXNzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogSWYgdGhpcyBiZWhhdmlvciBpcyB0aGUgZGVmYXVsdCBiZWhhdmlvciBmb3IgdGhlIGRpc3RyaWJ1dGlvbi5cbiAgICpcbiAgICogWW91IG11c3Qgc3BlY2lmeSBleGFjdGx5IG9uZSBkZWZhdWx0IGRpc3RyaWJ1dGlvbiBwZXIgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAqIFRoZSBkZWZhdWx0IGJlaGF2aW9yIGlzIGFsbG93ZWQgdG8gb21pdCB0aGUgXCJwYXRoXCIgcHJvcGVydHkuXG4gICAqL1xuICByZWFkb25seSBpc0RlZmF1bHRCZWhhdmlvcj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRydXN0ZWQgc2lnbmVycyBpcyBob3cgQ2xvdWRGcm9udCBhbGxvd3MgeW91IHRvIHNlcnZlIHByaXZhdGUgY29udGVudC5cbiAgICogVGhlIHNpZ25lcnMgYXJlIHRoZSBhY2NvdW50IElEcyB0aGF0IGFyZSBhbGxvd2VkIHRvIHNpZ24gY29va2llcy9wcmVzaWduZWQgVVJMcyBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqXG4gICAqIElmIHlvdSBwYXNzIGEgbm9uIGVtcHR5IHZhbHVlLCBhbGwgcmVxdWVzdHMgZm9yIHRoaXMgYmVoYXZpb3IgbXVzdCBiZSBzaWduZWQgKG5vIHB1YmxpYyBhY2Nlc3Mgd2lsbCBiZSBhbGxvd2VkKVxuICAgKi9cbiAgcmVhZG9ubHkgdHJ1c3RlZFNpZ25lcnM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICpcbiAgICogVGhlIGRlZmF1bHQgYW1vdW50IG9mIHRpbWUgQ2xvdWRGcm9udCB3aWxsIGNhY2hlIGFuIG9iamVjdC5cbiAgICpcbiAgICogVGhpcyB2YWx1ZSBhcHBsaWVzIG9ubHkgd2hlbiB5b3VyIGN1c3RvbSBvcmlnaW4gZG9lcyBub3QgYWRkIEhUVFAgaGVhZGVycyxcbiAgICogc3VjaCBhcyBDYWNoZS1Db250cm9sIG1heC1hZ2UsIENhY2hlLUNvbnRyb2wgcy1tYXhhZ2UsIGFuZCBFeHBpcmVzIHRvIG9iamVjdHMuXG4gICAqIEBkZWZhdWx0IDg2NDAwICgxIGRheSlcbiAgICpcbiAgICovXG4gIHJlYWRvbmx5IGRlZmF1bHRUdGxTZWNvbmRzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWV0aG9kIHRoaXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24gcmVzcG9uZHMgZG8uXG4gICAqXG4gICAqIEBkZWZhdWx0IEdFVF9IRUFEXG4gICAqL1xuICByZWFkb25seSBhbGxvd2VkTWV0aG9kcz86IENsb3VkRnJvbnRBbGxvd2VkTWV0aG9kcztcblxuICAvKipcbiAgICogVGhlIHBhdGggdGhpcyBiZWhhdmlvciByZXNwb25kcyB0by5cbiAgICogUmVxdWlyZWQgZm9yIGFsbCBub24tZGVmYXVsdCBiZWhhdmlvcnMuIChUaGUgZGVmYXVsdCBiZWhhdmlvciBpbXBsaWNpdGx5IGhhcyBcIipcIiBhcyB0aGUgcGF0aCBwYXR0ZXJuLiApXG4gICAqXG4gICAqL1xuICByZWFkb25seSBwYXRoUGF0dGVybj86IHN0cmluZztcblxuICAvKipcbiAgICogV2hpY2ggbWV0aG9kcyBhcmUgY2FjaGVkIGJ5IENsb3VkRnJvbnQgYnkgZGVmYXVsdC5cbiAgICpcbiAgICogQGRlZmF1bHQgR0VUX0hFQURcbiAgICovXG4gIHJlYWRvbmx5IGNhY2hlZE1ldGhvZHM/OiBDbG91ZEZyb250QWxsb3dlZENhY2hlZE1ldGhvZHM7XG5cbiAgLyoqXG4gICAqIFRoZSB2YWx1ZXMgQ2xvdWRGcm9udCB3aWxsIGZvcndhcmQgdG8gdGhlIG9yaWdpbiB3aGVuIG1ha2luZyBhIHJlcXVlc3QuXG4gICAqXG4gICAqIEBkZWZhdWx0IG5vbmUgKG5vIGNvb2tpZXMgLSBubyBoZWFkZXJzKVxuICAgKlxuICAgKi9cbiAgcmVhZG9ubHkgZm9yd2FyZGVkVmFsdWVzPzogQ2ZuRGlzdHJpYnV0aW9uLkZvcndhcmRlZFZhbHVlc1Byb3BlcnR5O1xuXG4gIC8qKlxuICAgKiBUaGUgbWluaW11bSBhbW91bnQgb2YgdGltZSB0aGF0IHlvdSB3YW50IG9iamVjdHMgdG8gc3RheSBpbiB0aGUgY2FjaGVcbiAgICogYmVmb3JlIENsb3VkRnJvbnQgcXVlcmllcyB5b3VyIG9yaWdpbi5cbiAgICovXG4gIHJlYWRvbmx5IG1pblR0bFNlY29uZHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtYXggYW1vdW50IG9mIHRpbWUgeW91IHdhbnQgb2JqZWN0cyB0byBzdGF5IGluIHRoZSBjYWNoZVxuICAgKiBiZWZvcmUgQ2xvdWRGcm9udCBxdWVyaWVzIHlvdXIgb3JpZ2luLlxuICAgKlxuICAgKiBAZGVmYXVsdCAzMTUzNjAwMCAob25lIHllYXIpXG4gICAqL1xuICByZWFkb25seSBtYXhUdGxTZWNvbmRzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBEZWNsYXJlcyBhc3NvY2lhdGVkIGxhbWJkYUBlZGdlIGZ1bmN0aW9ucyBmb3IgdGhpcyBkaXN0cmlidXRpb24gYmVoYXZpb3VyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBsYW1iZGEgZnVuY3Rpb24gYXNzb2NpYXRlZFxuICAgKi9cbiAgcmVhZG9ubHkgbGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnM/OiBMYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uW107XG5cbn1cblxuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uIHtcblxuICAvKipcbiAgICogVGhlIGxhbWJkYSBldmVudCB0eXBlIGRlZmluZXMgYXQgd2hpY2ggZXZlbnQgdGhlIGxhbWJkYVxuICAgKiBpcyBjYWxsZWQgZHVyaW5nIHRoZSByZXF1ZXN0IGxpZmVjeWNsZVxuICAgKi9cbiAgcmVhZG9ubHkgZXZlbnRUeXBlOiBMYW1iZGFFZGdlRXZlbnRUeXBlO1xuXG4gIC8qKlxuICAgKiBBIHZlcnNpb24gb2YgdGhlIGxhbWJkYSB0byBhc3NvY2lhdGVcbiAgICovXG4gIHJlYWRvbmx5IGxhbWJkYUZ1bmN0aW9uOiBsYW1iZGEuSVZlcnNpb247XG59XG5cbmV4cG9ydCBlbnVtIExhbWJkYUVkZ2VFdmVudFR5cGUge1xuICAvKipcbiAgICogVGhlIG9yaWdpbi1yZXF1ZXN0IHNwZWNpZmllcyB0aGUgcmVxdWVzdCB0byB0aGVcbiAgICogb3JpZ2luIGxvY2F0aW9uIChlLmcuIFMzKVxuICAgKi9cbiAgT3JpZ2luUmVxdWVzdCA9IFwib3JpZ2luLXJlcXVlc3RcIixcbiAgLyoqXG4gICAqIFRoZSBvcmlnaW4tcmVzcG9uc2Ugc3BlY2lmaWVzIHRoZSByZXNwb25zZSBmcm9tIHRoZVxuICAgKiBvcmlnaW4gbG9jYXRpb24gKGUuZy4gUzMpXG4gICAqL1xuICBPcmlnaW5SZXNwb25zZSA9IFwib3JpZ2luLXJlc3BvbnNlXCIsXG4gIC8qKlxuICAgKiBUaGUgdmlld2VyLXJlcXVlc3Qgc3BlY2lmaWVzIHRoZSBpbmNvbWluZyByZXF1ZXN0XG4gICAqL1xuICBWaWV3ZXJSZXF1ZXN0ID0gXCJ2aWV3ZXItcmVxdWVzdFwiLFxuICAvKipcbiAgICogVGhlIHZpZXdlci1yZXNwb25zZSBzcGVjaWZpZXMgdGhlIG91dGdvaW5nIHJlcG9uc2VcbiAgICovXG4gIFZpZXdlclJlc3BvbnNlID0gXCJ2aWV3ZXItcmVzcG9uc2VcIixcbn1cblxuZXhwb3J0IGludGVyZmFjZSBFcnJvckNvbmZpZ3VyYXRpb24ge1xuICAvKipcbiAgICogVGhlIGVycm9yIGNvZGUgbWF0Y2hlZCBmcm9tIHRoZSBvcmlnaW5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbkVycm9yQ29kZTogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIGVycm9yIGNvZGUgdGhhdCBpcyBzZW50IHRvIHRoZSBjYWxsZXIuXG4gICAqL1xuICByZWFkb25seSByZXNwb25kV2l0aEVycm9yQ29kZTogbnVtYmVyO1xuICAvKipcbiAgICogVGhlIHBhdGggdG8gc2VydmljZSBpbnN0ZWFkXG4gICAqL1xuICByZWFkb25seSByZXNwb25kV2l0aFBhZ2U6IHN0cmluZztcbiAgLyoqXG4gICAqIEhvdyBsb25nIGJlZm9yZSB0aGlzIGVycm9yIGlzIHJldHJpZWQuXG4gICAqL1xuICByZWFkb25seSBjYWNoZVR0bD86IG51bWJlcjtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBDbG91ZEZyb250V2ViRGlzdHJpYnV0aW9uUHJvcHMge1xuXG4gIC8qKlxuICAgKiBBbGlhc0NvbmZpZ3VyYXRpb24gaXMgdXNlZCB0byBjb25maWd1cmVkIENsb3VkRnJvbnQgdG8gcmVzcG9uZCB0byByZXF1ZXN0cyBvbiBjdXN0b20gZG9tYWluIG5hbWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vbmUuXG4gICAqL1xuICByZWFkb25seSBhbGlhc0NvbmZpZ3VyYXRpb24/OiBBbGlhc0NvbmZpZ3VyYXRpb247XG5cbiAgLyoqXG4gICAqIEEgY29tbWVudCBmb3IgdGhpcyBkaXN0cmlidXRpb24gaW4gdGhlIENsb3VkRnJvbnQgY29uc29sZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjb21tZW50IGlzIGFkZGVkIHRvIGRpc3RyaWJ1dGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBkZWZhdWx0IG9iamVjdCB0byBzZXJ2ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBcImluZGV4Lmh0bWxcIiBpcyBzZXJ2ZWQuXG4gICAqL1xuICByZWFkb25seSBkZWZhdWx0Um9vdE9iamVjdD86IHN0cmluZztcblxuICAvKipcbiAgICogSWYgeW91ciBkaXN0cmlidXRpb24gc2hvdWxkIGhhdmUgSVB2NiBlbmFibGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbmFibGVJcFY2PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG1heCBzdXBwb3J0ZWQgSFRUUCBWZXJzaW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgSHR0cFZlcnNpb24uSFRUUDJcbiAgICovXG4gIHJlYWRvbmx5IGh0dHBWZXJzaW9uPzogSHR0cFZlcnNpb247XG5cbiAgLyoqXG4gICAqIFRoZSBwcmljZSBjbGFzcyBmb3IgdGhlIGRpc3RyaWJ1dGlvbiAodGhpcyBpbXBhY3RzIGhvdyBtYW55IGxvY2F0aW9ucyBDbG91ZEZyb250IHVzZXMgZm9yIHlvdXIgZGlzdHJpYnV0aW9uLCBhbmQgYmlsbGluZylcbiAgICpcbiAgICogQGRlZmF1bHQgUHJpY2VDbGFzcy5QcmljZUNsYXNzMTAwIHRoZSBjaGVhcGVzdCBvcHRpb24gZm9yIENsb3VkRnJvbnQgaXMgcGlja2VkIGJ5IGRlZmF1bHQuXG4gICAqL1xuICByZWFkb25seSBwcmljZUNsYXNzPzogUHJpY2VDbGFzcztcblxuICAvKipcbiAgICogVGhlIGRlZmF1bHQgdmlld2VyIHBvbGljeSBmb3IgaW5jb21pbmcgY2xpZW50cy5cbiAgICpcbiAgICogQGRlZmF1bHQgUmVkaXJlY3RUb0hUVFBzXG4gICAqL1xuICByZWFkb25seSB2aWV3ZXJQcm90b2NvbFBvbGljeT86IFZpZXdlclByb3RvY29sUG9saWN5O1xuXG4gIC8qKlxuICAgKiBUaGUgb3JpZ2luIGNvbmZpZ3VyYXRpb25zIGZvciB0aGlzIGRpc3RyaWJ1dGlvbi4gQmVoYXZpb3JzIGFyZSBhIHBhcnQgb2YgdGhlIG9yaWdpbi5cbiAgICovXG4gIHJlYWRvbmx5IG9yaWdpbkNvbmZpZ3M6IFNvdXJjZUNvbmZpZ3VyYXRpb25bXTtcblxuICAvKipcbiAgICogT3B0aW9uYWwgLSBpZiB3ZSBzaG91bGQgZW5hYmxlIGxvZ2dpbmcuXG4gICAqIFlvdSBjYW4gcGFzcyBhbiBlbXB0eSBvYmplY3QgKHt9KSB0byBoYXZlIHVzIGF1dG8gY3JlYXRlIGEgYnVja2V0IGZvciBsb2dnaW5nLlxuICAgKiBPbWlzc2lvbiBvZiB0aGlzIHByb3BlcnR5IGluZGljYXRlcyBubyBsb2dnaW5nIGlzIHRvIGJlIGVuYWJsZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gbm8gbG9nZ2luZyBpcyBlbmFibGVkIGJ5IGRlZmF1bHQuXG4gICAqL1xuICByZWFkb25seSBsb2dnaW5nQ29uZmlnPzogTG9nZ2luZ0NvbmZpZ3VyYXRpb247XG5cbiAgLyoqXG4gICAqIEhvdyBDbG91ZEZyb250IHNob3VsZCBoYW5kbGUgcmVxdWVzdHMgdGhhdCBhcmUgbm90IHN1Y2Nlc3NmdWwgKGVnIFBhZ2VOb3RGb3VuZClcbiAgICpcbiAgICogQnkgZGVmYXVsdCwgQ2xvdWRGcm9udCBkb2VzIG5vdCByZXBsYWNlIEhUVFAgc3RhdHVzIGNvZGVzIGluIHRoZSA0eHggYW5kIDV4eCByYW5nZVxuICAgKiB3aXRoIGN1c3RvbSBlcnJvciBtZXNzYWdlcy4gQ2xvdWRGcm9udCBkb2VzIG5vdCBjYWNoZSBIVFRQIHN0YXR1cyBjb2Rlcy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjdXN0b20gZXJyb3IgY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGVycm9yQ29uZmlndXJhdGlvbnM/OiBDZm5EaXN0cmlidXRpb24uQ3VzdG9tRXJyb3JSZXNwb25zZVByb3BlcnR5W107XG5cbiAgLyoqXG4gICAqIFVuaXF1ZSBpZGVudGlmaWVyIHRoYXQgc3BlY2lmaWVzIHRoZSBBV1MgV0FGIHdlYiBBQ0wgdG8gYXNzb2NpYXRlIHdpdGggdGhpcyBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbi5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vd2FmL2xhdGVzdC9kZXZlbG9wZXJndWlkZS93aGF0LWlzLWF3cy13YWYuaHRtbFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIEFXUyBXZWIgQXBwbGljYXRpb24gRmlyZXdhbGwgd2ViIGFjY2VzcyBjb250cm9sIGxpc3QgKHdlYiBBQ0wpLlxuICAgKi9cbiAgcmVhZG9ubHkgd2ViQUNMSWQ/OiBzdHJpbmc7XG5cbn1cblxuLyoqXG4gKiBJbnRlcm5hbCBvbmx5IC0ganVzdCBhZGRzIHRoZSBvcmlnaW5JZCBzdHJpbmcgdG8gdGhlIEJlaGF2aW9yXG4gKi9cbmludGVyZmFjZSBCZWhhdmlvcldpdGhPcmlnaW4gZXh0ZW5kcyBCZWhhdmlvciB7XG4gIHJlYWRvbmx5IHRhcmdldE9yaWdpbklkOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQW1hem9uIENsb3VkRnJvbnQgaXMgYSBnbG9iYWwgY29udGVudCBkZWxpdmVyeSBuZXR3b3JrIChDRE4pIHNlcnZpY2UgdGhhdCBzZWN1cmVseSBkZWxpdmVycyBkYXRhLCB2aWRlb3MsXG4gKiBhcHBsaWNhdGlvbnMsIGFuZCBBUElzIHRvIHlvdXIgdmlld2VycyB3aXRoIGxvdyBsYXRlbmN5IGFuZCBoaWdoIHRyYW5zZmVyIHNwZWVkcy5cbiAqIENsb3VkRnJvbnQgZnJvbnRzIHVzZXIgcHJvdmlkZWQgY29udGVudCBhbmQgY2FjaGVzIGl0IGF0IGVkZ2UgbG9jYXRpb25zIGFjcm9zcyB0aGUgd29ybGQuXG4gKlxuICogSGVyZSdzIGhvdyB5b3UgY2FuIHVzZSB0aGlzIGNvbnN0cnVjdDpcbiAqXG4gKiBgYGB0c1xuICogaW1wb3J0IHsgQ2xvdWRGcm9udCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250J1xuICpcbiAqIGNvbnN0IHNvdXJjZUJ1Y2tldCA9IG5ldyBCdWNrZXQodGhpcywgJ0J1Y2tldCcpO1xuICpcbiAqIGNvbnN0IGRpc3RyaWJ1dGlvbiA9IG5ldyBDbG91ZEZyb250RGlzdHJpYnV0aW9uKHRoaXMsICdNeURpc3RyaWJ1dGlvbicsIHtcbiAqICBvcmlnaW5Db25maWdzOiBbXG4gKiAgICB7XG4gKiAgICAgIHMzT3JpZ2luU291cmNlOiB7XG4gKiAgICAgIHMzQnVja2V0U291cmNlOiBzb3VyY2VCdWNrZXRcbiAqICAgICAgfSxcbiAqICAgICAgYmVoYXZpb3JzIDogWyB7aXNEZWZhdWx0QmVoYXZpb3J9XVxuICogICAgfVxuICogIF1cbiAqIH0pO1xuICogYGBgXG4gKlxuICogVGhpcyB3aWxsIGNyZWF0ZSBhIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uIHRoYXQgdXNlcyB5b3VyIFMzQnVja2V0IGFzIGl0J3Mgb3JpZ2luLlxuICpcbiAqIFlvdSBjYW4gY3VzdG9taXplIHRoZSBkaXN0cmlidXRpb24gdXNpbmcgYWRkaXRpb25hbCBwcm9wZXJ0aWVzIGZyb20gdGhlIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcyBpbnRlcmZhY2UuXG4gKlxuICpcbiAqL1xuZXhwb3J0IGNsYXNzIENsb3VkRnJvbnRXZWJEaXN0cmlidXRpb24gZXh0ZW5kcyBjZGsuQ29uc3RydWN0IGltcGxlbWVudHMgSURpc3RyaWJ1dGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgbG9nZ2luZyBidWNrZXQgZm9yIHRoaXMgQ2xvdWRGcm9udCBkaXN0cmlidXRpb24uXG4gICAqIElmIGxvZ2dpbmcgaXMgbm90IGVuYWJsZWQgZm9yIHRoaXMgZGlzdHJpYnV0aW9uIC0gdGhpcyBwcm9wZXJ0eSB3aWxsIGJlIHVuZGVmaW5lZC5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBsb2dnaW5nQnVja2V0PzogczMuSUJ1Y2tldDtcblxuICAvKipcbiAgICogVGhlIGRvbWFpbiBuYW1lIGNyZWF0ZWQgYnkgQ2xvdWRGcm9udCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqIElmIHlvdSBhcmUgdXNpbmcgYWxpYXNlcyBmb3IgeW91ciBkaXN0cmlidXRpb24sIHRoaXMgaXMgdGhlIGRvbWFpbk5hbWUgeW91ciBETlMgcmVjb3JkcyBzaG91bGQgcG9pbnQgdG8uXG4gICAqIChJbiBSb3V0ZTUzLCB5b3UgY291bGQgY3JlYXRlIGFuIEFMSUFTIHJlY29yZCB0byB0aGlzIHZhbHVlLCBmb3IgZXhhbXBsZS4gKVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRvbWFpbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGRpc3RyaWJ1dGlvbiBJRCBmb3IgdGhpcyBkaXN0cmlidXRpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgZGlzdHJpYnV0aW9uSWQ6IHN0cmluZztcblxuICAvKipcbiAgICogTWFwcyBvdXIgbWV0aG9kcyB0byB0aGUgc3RyaW5nIGFycmF5cyB0aGV5IGFyZVxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBNRVRIT0RfTE9PS1VQX01BUCA9IHtcbiAgICBHSDogW1wiR0VUXCIsIFwiSEVBRFwiXSxcbiAgICBHSE86IFtcIkdFVFwiLCBcIkhFQURcIiwgXCJPUFRJT05TXCJdLFxuICAgIEFMTDogW1wiREVMRVRFXCIsIFwiR0VUXCIsIFwiSEVBRFwiLCBcIk9QVElPTlNcIiwgXCJQQVRDSFwiLCBcIlBPU1RcIiwgXCJQVVRcIl0sXG4gIH07XG5cbiAgLyoqXG4gICAqIE1hcHMgZm9yIHdoaWNoIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wgYXJlIGF2YWlsYWJsZSB0byB3aGljaCBTU0xNZXRob2RzXG4gICAqL1xuICBwcml2YXRlIHJlYWRvbmx5IFZBTElEX1NTTF9QUk9UT0NPTFM6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nW10gfSA9IHtcbiAgICBcInNuaS1vbmx5XCI6IFtcbiAgICAgIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTdjEsIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTdjFfMV8yMDE2LFxuICAgICAgU2VjdXJpdHlQb2xpY3lQcm90b2NvbC5UTFN2MV8yMDE2LCBTZWN1cml0eVBvbGljeVByb3RvY29sLlRMU3YxXzJfMjAxOFxuICAgIF0sXG4gICAgXCJ2aXBcIjogW1NlY3VyaXR5UG9saWN5UHJvdG9jb2wuU1NMdjMsIFNlY3VyaXR5UG9saWN5UHJvdG9jb2wuVExTdjFdLFxuICB9O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGxldCBkaXN0cmlidXRpb25Db25maWc6IENmbkRpc3RyaWJ1dGlvbi5EaXN0cmlidXRpb25Db25maWdQcm9wZXJ0eSA9IHtcbiAgICAgIGNvbW1lbnQ6IHByb3BzLmNvbW1lbnQsXG4gICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgZGVmYXVsdFJvb3RPYmplY3Q6IHByb3BzLmRlZmF1bHRSb290T2JqZWN0ICE9PSB1bmRlZmluZWQgPyBwcm9wcy5kZWZhdWx0Um9vdE9iamVjdCA6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgaHR0cFZlcnNpb246IHByb3BzLmh0dHBWZXJzaW9uIHx8IEh0dHBWZXJzaW9uLkhUVFAyLFxuICAgICAgcHJpY2VDbGFzczogcHJvcHMucHJpY2VDbGFzcyB8fCBQcmljZUNsYXNzLlByaWNlQ2xhc3MxMDAsXG4gICAgICBpcHY2RW5hYmxlZDogKHByb3BzLmVuYWJsZUlwVjYgIT09IHVuZGVmaW5lZCkgPyBwcm9wcy5lbmFibGVJcFY2IDogdHJ1ZSxcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTptYXgtbGluZS1sZW5ndGhcbiAgICAgIGN1c3RvbUVycm9yUmVzcG9uc2VzOiBwcm9wcy5lcnJvckNvbmZpZ3VyYXRpb25zLCAvLyBUT0RPOiB2YWxpZGF0aW9uIDogaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtY2xvdWRmcm9udC1kaXN0cmlidXRpb24tY3VzdG9tZXJyb3JyZXNwb25zZS5odG1sI2Nmbi1jbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1jdXN0b21lcnJvcnJlc3BvbnNlLWVycm9yY2FjaGluZ21pbnR0bFxuICAgICAgd2ViQWNsSWQ6IHByb3BzLndlYkFDTElkLFxuICAgIH07XG5cbiAgICBjb25zdCBiZWhhdmlvcnM6IEJlaGF2aW9yV2l0aE9yaWdpbltdID0gW107XG5cbiAgICBjb25zdCBvcmlnaW5zOiBDZm5EaXN0cmlidXRpb24uT3JpZ2luUHJvcGVydHlbXSA9IFtdO1xuXG4gICAgbGV0IG9yaWdpbkluZGV4ID0gMTtcbiAgICBmb3IgKGNvbnN0IG9yaWdpbkNvbmZpZyBvZiBwcm9wcy5vcmlnaW5Db25maWdzKSB7XG4gICAgICBjb25zdCBvcmlnaW5JZCA9IGBvcmlnaW4ke29yaWdpbkluZGV4fWA7XG4gICAgICBpZiAoIW9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZSAmJiAhb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXCJUaGVyZSBtdXN0IGJlIGF0IGxlYXN0IG9uZSBvcmlnaW4gc291cmNlIC0gZWl0aGVyIGFuIHMzT3JpZ2luU291cmNlIG9yIGEgY3VzdG9tT3JpZ2luU291cmNlXCIpO1xuICAgICAgfVxuICAgICAgaWYgKG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UgJiYgb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcIlRoZXJlIGNhbm5vdCBiZSBib3RoIGFuIHMzT3JpZ2luU291cmNlIGFuZCBhIGN1c3RvbU9yaWdpblNvdXJjZSBpbiB0aGUgc2FtZSBTb3VyY2VDb25maWd1cmF0aW9uLlwiKTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3JpZ2luSGVhZGVyczogQ2ZuRGlzdHJpYnV0aW9uLk9yaWdpbkN1c3RvbUhlYWRlclByb3BlcnR5W10gPSBbXTtcbiAgICAgIGlmIChvcmlnaW5Db25maWcub3JpZ2luSGVhZGVycykge1xuICAgICAgICBPYmplY3Qua2V5cyhvcmlnaW5Db25maWcub3JpZ2luSGVhZGVycykuZm9yRWFjaChrZXkgPT4ge1xuICAgICAgICAgIGNvbnN0IG9IZWFkZXI6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5DdXN0b21IZWFkZXJQcm9wZXJ0eSA9IHtcbiAgICAgICAgICAgIGhlYWRlck5hbWU6IGtleSxcbiAgICAgICAgICAgIGhlYWRlclZhbHVlOiBvcmlnaW5Db25maWcub3JpZ2luSGVhZGVycyFba2V5XVxuICAgICAgICAgIH07XG4gICAgICAgICAgb3JpZ2luSGVhZGVycy5wdXNoKG9IZWFkZXIpO1xuICAgICAgICB9KTtcbiAgICAgIH1cblxuICAgICAgY29uc3Qgb3JpZ2luUHJvcGVydHk6IENmbkRpc3RyaWJ1dGlvbi5PcmlnaW5Qcm9wZXJ0eSA9IHtcbiAgICAgICAgaWQ6IG9yaWdpbklkLFxuICAgICAgICBkb21haW5OYW1lOiBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2VcbiAgICAgICAgICA/IG9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZS5zM0J1Y2tldFNvdXJjZS5idWNrZXRSZWdpb25hbERvbWFpbk5hbWVcbiAgICAgICAgICA6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UhLmRvbWFpbk5hbWUsXG4gICAgICAgIG9yaWdpblBhdGg6IG9yaWdpbkNvbmZpZy5vcmlnaW5QYXRoLFxuICAgICAgICBvcmlnaW5DdXN0b21IZWFkZXJzOiBvcmlnaW5IZWFkZXJzLmxlbmd0aCA+IDAgPyBvcmlnaW5IZWFkZXJzIDogdW5kZWZpbmVkLFxuICAgICAgICBzM09yaWdpbkNvbmZpZzogb3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlICYmIG9yaWdpbkNvbmZpZy5zM09yaWdpblNvdXJjZS5vcmlnaW5BY2Nlc3NJZGVudGl0eUlkXG4gICAgICAgICAgPyB7IG9yaWdpbkFjY2Vzc0lkZW50aXR5OiBgb3JpZ2luLWFjY2Vzcy1pZGVudGl0eS9jbG91ZGZyb250LyR7b3JpZ2luQ29uZmlnLnMzT3JpZ2luU291cmNlLm9yaWdpbkFjY2Vzc0lkZW50aXR5SWR9YCB9XG4gICAgICAgICAgOiBvcmlnaW5Db25maWcuczNPcmlnaW5Tb3VyY2VcbiAgICAgICAgICA/IHsgfVxuICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICBjdXN0b21PcmlnaW5Db25maWc6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2VcbiAgICAgICAgICA/IHtcbiAgICAgICAgICAgIGh0dHBQb3J0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLmh0dHBQb3J0IHx8IDgwLFxuICAgICAgICAgICAgaHR0cHNQb3J0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLmh0dHBzUG9ydCB8fCA0NDMsXG4gICAgICAgICAgICBvcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiBvcmlnaW5Db25maWcuY3VzdG9tT3JpZ2luU291cmNlLm9yaWdpbktlZXBhbGl2ZVRpbWVvdXRTZWNvbmRzIHx8IDUsXG4gICAgICAgICAgICBvcmlnaW5SZWFkVGltZW91dDogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5vcmlnaW5SZWFkVGltZW91dFNlY29uZHMgfHwgMzAsXG4gICAgICAgICAgICBvcmlnaW5Qcm90b2NvbFBvbGljeTogb3JpZ2luQ29uZmlnLmN1c3RvbU9yaWdpblNvdXJjZS5vcmlnaW5Qcm90b2NvbFBvbGljeSB8fCBPcmlnaW5Qcm90b2NvbFBvbGljeS5IdHRwc09ubHksXG4gICAgICAgICAgICBvcmlnaW5Tc2xQcm90b2NvbHM6IG9yaWdpbkNvbmZpZy5jdXN0b21PcmlnaW5Tb3VyY2UuYWxsb3dlZE9yaWdpblNTTFZlcnNpb25zIHx8IFtPcmlnaW5Tc2xQb2xpY3kuVExTdjFfMl1cbiAgICAgICAgICB9XG4gICAgICAgICAgOiB1bmRlZmluZWRcbiAgICAgIH07XG5cbiAgICAgIGZvciAoY29uc3QgYmVoYXZpb3Igb2Ygb3JpZ2luQ29uZmlnLmJlaGF2aW9ycykge1xuICAgICAgICBiZWhhdmlvcnMucHVzaCh7IC4uLmJlaGF2aW9yLCB0YXJnZXRPcmlnaW5JZDogb3JpZ2luSWQgfSk7XG4gICAgICB9XG5cbiAgICAgIG9yaWdpbnMucHVzaChvcmlnaW5Qcm9wZXJ0eSk7XG4gICAgICBvcmlnaW5JbmRleCsrO1xuICAgIH1cblxuICAgIG9yaWdpbnMuZm9yRWFjaChvcmlnaW4gPT4ge1xuICAgICAgaWYgKCFvcmlnaW4uczNPcmlnaW5Db25maWcgJiYgIW9yaWdpbi5jdXN0b21PcmlnaW5Db25maWcpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPcmlnaW4gJHtvcmlnaW4uZG9tYWluTmFtZX0gaXMgbWlzc2luZyBlaXRoZXIgUzNPcmlnaW5Db25maWcgb3IgQ3VzdG9tT3JpZ2luQ29uZmlnLiBBdCBsZWFzdCAxIG11c3QgYmUgc3BlY2lmaWVkLmApO1xuICAgICAgfVxuICAgIH0pO1xuICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHtcbiAgICAgIC4uLmRpc3RyaWJ1dGlvbkNvbmZpZyxcbiAgICAgIG9yaWdpbnNcbiAgICB9O1xuXG4gICAgY29uc3QgZGVmYXVsdEJlaGF2aW9ycyA9IGJlaGF2aW9ycy5maWx0ZXIoYmVoYXZpb3IgPT4gYmVoYXZpb3IuaXNEZWZhdWx0QmVoYXZpb3IpO1xuICAgIGlmIChkZWZhdWx0QmVoYXZpb3JzLmxlbmd0aCAhPT0gMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFwiVGhlcmUgY2FuIG9ubHkgYmUgb25lIGRlZmF1bHQgYmVoYXZpb3IgYWNyb3NzIGFsbCBzb3VyY2VzLiBbIE9uZSBkZWZhdWx0IGJlaGF2aW9yIHBlciBkaXN0cmlidXRpb24gXS5cIik7XG4gICAgfVxuXG4gICAgZGlzdHJpYnV0aW9uQ29uZmlnID0geyAuLi5kaXN0cmlidXRpb25Db25maWcsIGRlZmF1bHRDYWNoZUJlaGF2aW9yOiB0aGlzLnRvQmVoYXZpb3IoZGVmYXVsdEJlaGF2aW9yc1swXSwgcHJvcHMudmlld2VyUHJvdG9jb2xQb2xpY3kpIH07XG5cbiAgICBjb25zdCBvdGhlckJlaGF2aW9yczogQ2ZuRGlzdHJpYnV0aW9uLkNhY2hlQmVoYXZpb3JQcm9wZXJ0eVtdID0gW107XG4gICAgZm9yIChjb25zdCBiZWhhdmlvciBvZiBiZWhhdmlvcnMuZmlsdGVyKGIgPT4gIWIuaXNEZWZhdWx0QmVoYXZpb3IpKSB7XG4gICAgICBpZiAoIWJlaGF2aW9yLnBhdGhQYXR0ZXJuKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcihcInBhdGhQYXR0ZXJuIGlzIHJlcXVpcmVkIGZvciBhbGwgbm9uLWRlZmF1bHQgYmVoYXZpb3JzXCIpO1xuICAgICAgfVxuICAgICAgb3RoZXJCZWhhdmlvcnMucHVzaCh0aGlzLnRvQmVoYXZpb3IoYmVoYXZpb3IsIHByb3BzLnZpZXdlclByb3RvY29sUG9saWN5KSBhcyBDZm5EaXN0cmlidXRpb24uQ2FjaGVCZWhhdmlvclByb3BlcnR5KTtcbiAgICB9XG5cbiAgICBkaXN0cmlidXRpb25Db25maWcgPSB7IC4uLmRpc3RyaWJ1dGlvbkNvbmZpZywgY2FjaGVCZWhhdmlvcnM6IG90aGVyQmVoYXZpb3JzIH07XG5cbiAgICBpZiAocHJvcHMuYWxpYXNDb25maWd1cmF0aW9uKSB7XG4gICAgICBjb25zdCBtaW5pbXVtUHJvdG9jb2xWZXJzaW9uID0gcHJvcHMuYWxpYXNDb25maWd1cmF0aW9uLnNlY3VyaXR5UG9saWN5O1xuICAgICAgY29uc3Qgc3NsU3VwcG9ydE1ldGhvZCA9IHByb3BzLmFsaWFzQ29uZmlndXJhdGlvbi5zc2xNZXRob2QgfHwgU1NMTWV0aG9kLlNOSTtcbiAgICAgIGNvbnN0IGFjbUNlcnRpZmljYXRlQXJuID0gcHJvcHMuYWxpYXNDb25maWd1cmF0aW9uLmFjbUNlcnRSZWY7XG5cbiAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHtcbiAgICAgICAgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLFxuICAgICAgICBhbGlhc2VzOiBwcm9wcy5hbGlhc0NvbmZpZ3VyYXRpb24ubmFtZXMsXG4gICAgICAgIHZpZXdlckNlcnRpZmljYXRlOiB7XG4gICAgICAgICAgYWNtQ2VydGlmaWNhdGVBcm4sXG4gICAgICAgICAgc3NsU3VwcG9ydE1ldGhvZCxcbiAgICAgICAgICBtaW5pbXVtUHJvdG9jb2xWZXJzaW9uXG4gICAgICAgIH1cbiAgICAgIH07XG5cbiAgICAgIGlmIChtaW5pbXVtUHJvdG9jb2xWZXJzaW9uICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgY29uc3QgdmFsaWRQcm90b2NvbHMgPSB0aGlzLlZBTElEX1NTTF9QUk9UT0NPTFNbc3NsU3VwcG9ydE1ldGhvZC50b1N0cmluZygpXTtcblxuICAgICAgICBpZiAodmFsaWRQcm90b2NvbHMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBzc2xNZXRob2QuICR7c3NsU3VwcG9ydE1ldGhvZC50b1N0cmluZygpfSBpcyBub3QgZnVsbHkgaW1wbGVtZW50ZWQgeWV0LmApO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHZhbGlkUHJvdG9jb2xzLmluZGV4T2YobWluaW11bVByb3RvY29sVmVyc2lvbi50b1N0cmluZygpKSA9PT0gLTEpIHtcbiAgICAgICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6bWF4LWxpbmUtbGVuZ3RoXG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAke21pbmltdW1Qcm90b2NvbFZlcnNpb259IGlzIG5vdCBjb21wYWJ0aWJsZSB3aXRoIHNzbE1ldGhvZCAke3NzbFN1cHBvcnRNZXRob2R9LlxcblxcdFZhbGlkIFByb3RvY29scyBhcmU6ICR7dmFsaWRQcm90b2NvbHMuam9pbihcIiwgXCIpfWApO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSBlbHNlIHtcbiAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHsgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLFxuICAgICAgICB2aWV3ZXJDZXJ0aWZpY2F0ZTogeyBjbG91ZEZyb250RGVmYXVsdENlcnRpZmljYXRlOiB0cnVlIH1cbiAgICAgIH07XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLmxvZ2dpbmdDb25maWcpIHtcbiAgICAgIHRoaXMubG9nZ2luZ0J1Y2tldCA9IHByb3BzLmxvZ2dpbmdDb25maWcuYnVja2V0IHx8IG5ldyBzMy5CdWNrZXQodGhpcywgYExvZ2dpbmdCdWNrZXRgKTtcbiAgICAgIGRpc3RyaWJ1dGlvbkNvbmZpZyA9IHtcbiAgICAgICAgLi4uZGlzdHJpYnV0aW9uQ29uZmlnLFxuICAgICAgICBsb2dnaW5nOiB7XG4gICAgICAgICAgYnVja2V0OiB0aGlzLmxvZ2dpbmdCdWNrZXQuYnVja2V0UmVnaW9uYWxEb21haW5OYW1lLFxuICAgICAgICAgIGluY2x1ZGVDb29raWVzOiBwcm9wcy5sb2dnaW5nQ29uZmlnLmluY2x1ZGVDb29raWVzIHx8IGZhbHNlLFxuICAgICAgICAgIHByZWZpeDogcHJvcHMubG9nZ2luZ0NvbmZpZy5wcmVmaXhcbiAgICAgICAgfVxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBkaXN0cmlidXRpb24gPSBuZXcgQ2ZuRGlzdHJpYnV0aW9uKHRoaXMsICdDRkRpc3RyaWJ1dGlvbicsIHsgZGlzdHJpYnV0aW9uQ29uZmlnIH0pO1xuICAgIHRoaXMuZG9tYWluTmFtZSA9IGRpc3RyaWJ1dGlvbi5hdHRyRG9tYWluTmFtZTtcbiAgICB0aGlzLmRpc3RyaWJ1dGlvbklkID0gZGlzdHJpYnV0aW9uLnJlZkFzU3RyaW5nO1xuICB9XG5cbiAgcHJpdmF0ZSB0b0JlaGF2aW9yKGlucHV0OiBCZWhhdmlvcldpdGhPcmlnaW4sIHByb3RvUG9saWN5PzogVmlld2VyUHJvdG9jb2xQb2xpY3kpIHtcbiAgICBsZXQgdG9SZXR1cm4gPSB7XG4gICAgICBhbGxvd2VkTWV0aG9kczogdGhpcy5NRVRIT0RfTE9PS1VQX01BUFtpbnB1dC5hbGxvd2VkTWV0aG9kcyB8fCBDbG91ZEZyb250QWxsb3dlZE1ldGhvZHMuR0VUX0hFQURdLFxuICAgICAgY2FjaGVkTWV0aG9kczogdGhpcy5NRVRIT0RfTE9PS1VQX01BUFtpbnB1dC5jYWNoZWRNZXRob2RzIHx8IENsb3VkRnJvbnRBbGxvd2VkQ2FjaGVkTWV0aG9kcy5HRVRfSEVBRF0sXG4gICAgICBjb21wcmVzczogaW5wdXQuY29tcHJlc3MsXG4gICAgICBkZWZhdWx0VHRsOiBpbnB1dC5kZWZhdWx0VHRsU2Vjb25kcyxcbiAgICAgIGZvcndhcmRlZFZhbHVlczogaW5wdXQuZm9yd2FyZGVkVmFsdWVzIHx8IHsgcXVlcnlTdHJpbmc6IGZhbHNlLCBjb29raWVzOiB7IGZvcndhcmQ6IFwibm9uZVwiIH0gfSxcbiAgICAgIG1heFR0bDogaW5wdXQubWF4VHRsU2Vjb25kcyxcbiAgICAgIG1pblR0bDogaW5wdXQubWluVHRsU2Vjb25kcyxcbiAgICAgIHRydXN0ZWRTaWduZXJzOiBpbnB1dC50cnVzdGVkU2lnbmVycyxcbiAgICAgIHRhcmdldE9yaWdpbklkOiBpbnB1dC50YXJnZXRPcmlnaW5JZCxcbiAgICAgIHZpZXdlclByb3RvY29sUG9saWN5OiBwcm90b1BvbGljeSB8fCBWaWV3ZXJQcm90b2NvbFBvbGljeS5SZWRpcmVjdFRvSFRUUFMsXG4gICAgfTtcbiAgICBpZiAoIWlucHV0LmlzRGVmYXVsdEJlaGF2aW9yKSB7XG4gICAgICB0b1JldHVybiA9IE9iamVjdC5hc3NpZ24odG9SZXR1cm4sIHsgcGF0aFBhdHRlcm46IGlucHV0LnBhdGhQYXR0ZXJuIH0pO1xuICAgIH1cbiAgICBpZiAoaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnMpIHtcbiAgICAgIHRvUmV0dXJuID0gT2JqZWN0LmFzc2lnbih0b1JldHVybiwge1xuICAgICAgICBsYW1iZGFGdW5jdGlvbkFzc29jaWF0aW9uczogaW5wdXQubGFtYmRhRnVuY3Rpb25Bc3NvY2lhdGlvbnNcbiAgICAgICAgICAubWFwKGZuYSA9PiAoe1xuICAgICAgICAgICAgZXZlbnRUeXBlOiBmbmEuZXZlbnRUeXBlLFxuICAgICAgICAgICAgbGFtYmRhRnVuY3Rpb25Bcm46IGZuYS5sYW1iZGFGdW5jdGlvbiAmJiBmbmEubGFtYmRhRnVuY3Rpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgICAgfSkpXG4gICAgICB9KTtcbiAgICB9XG4gICAgcmV0dXJuIHRvUmV0dXJuO1xuICB9XG59XG4iXX0=