"use strict";
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'distribution with custom origin adds custom origin'(test) {
        const stack = new cdk.Stack();
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    originHeaders: {
                        "X-Custom-Header": "somevalue",
                    },
                    customOriginSource: {
                        domainName: "myorigin.com",
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ],
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "CacheBehaviors": [],
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "ForwardedValues": {
                                    "Cookies": {
                                        "Forward": "none"
                                    },
                                    "QueryString": false
                                },
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https"
                            },
                            "DefaultRootObject": "index.html",
                            "Enabled": true,
                            "HttpVersion": "http2",
                            "IPV6Enabled": true,
                            "Origins": [
                                {
                                    "CustomOriginConfig": {
                                        "HTTPPort": 80,
                                        "HTTPSPort": 443,
                                        "OriginKeepaliveTimeout": 5,
                                        "OriginProtocolPolicy": "https-only",
                                        "OriginReadTimeout": 30,
                                        "OriginSSLProtocols": [
                                            "TLSv1.2"
                                        ]
                                    },
                                    "DomainName": "myorigin.com",
                                    "Id": "origin1",
                                    "OriginCustomHeaders": [
                                        {
                                            "HeaderName": "X-Custom-Header",
                                            "HeaderValue": "somevalue"
                                        }
                                    ]
                                }
                            ],
                            "PriceClass": "PriceClass_100",
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'most basic distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                }
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2",
                            "CacheBehaviors": []
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with trusted signers on default distribution'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                            trustedSigners: ["1234"],
                        },
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "redirect-to-https",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                },
                                "TrustedSigners": [
                                    "1234"
                                ]
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2",
                            "CacheBehaviors": []
                        }
                    }
                }
            }
        });
        test.done();
    },
    'distribution with ViewerProtocolPolicy set to a non-default value'(test) {
        const stack = new cdk.Stack();
        const sourceBucket = new s3.Bucket(stack, 'Bucket');
        new lib_1.CloudFrontWebDistribution(stack, 'AnAmazingWebsiteProbably', {
            viewerProtocolPolicy: lib_1.ViewerProtocolPolicy.AllowAll,
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: sourceBucket
                    },
                    behaviors: [
                        {
                            isDefaultBehavior: true,
                        }
                    ]
                }
            ]
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "Bucket83908E77": {
                    "Type": "AWS::S3::Bucket",
                    "DeletionPolicy": "Retain",
                },
                "AnAmazingWebsiteProbablyCFDistribution47E3983B": {
                    "Type": "AWS::CloudFront::Distribution",
                    "Properties": {
                        "DistributionConfig": {
                            "DefaultRootObject": "index.html",
                            "Origins": [
                                {
                                    "DomainName": {
                                        "Fn::GetAtt": [
                                            "Bucket83908E77",
                                            "RegionalDomainName"
                                        ]
                                    },
                                    "Id": "origin1",
                                    "S3OriginConfig": {}
                                }
                            ],
                            "ViewerCertificate": {
                                "CloudFrontDefaultCertificate": true
                            },
                            "PriceClass": "PriceClass_100",
                            "DefaultCacheBehavior": {
                                "AllowedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "CachedMethods": [
                                    "GET",
                                    "HEAD"
                                ],
                                "TargetOriginId": "origin1",
                                "ViewerProtocolPolicy": "allow-all",
                                "ForwardedValues": {
                                    "QueryString": false,
                                    "Cookies": { "Forward": "none" }
                                }
                            },
                            "Enabled": true,
                            "IPV6Enabled": true,
                            "HttpVersion": "http2",
                            "CacheBehaviors": []
                        }
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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