import functools
from importlib import resources
from importlib.abc import Traversable
import json
from pathlib import Path
from typing import Any

from mach_client import ChainId

# Only used for swaps between similar tokens (ie. WETH -> ETH, USDC -> USDT, etc.)
max_slippage = 0.01

# Relative to the root of the repository
abi_path = resources.files("abi")

# Time to wait between polls of the destination balance to check if the desired amount has been received
poll_timeout = 10

# Maximum number of polls
max_polls = 30

solidity_uint_max = 2**256 - 1

log_dir = Path("logs")

log_files = {
    # All logs
    "app": log_dir / "app.log",
    # Source funds not withdrawn
    "delayed_transactions": log_dir / "delayed_transactions.log",
    # Source funds withdrawn, destination funds not received
    "stuck_transactions": log_dir / "stuck_transactions.log",
    # Filled for less than order was placed for or received less than order was quoted for
    "improper_fills": log_dir / "improper_fills.log",
    # Slippage for similar tokens (ie. both USD stablecoins, both ETH variants, etc.)
    "slippage": log_dir / "slippage.log",
}


def load_abi(path: Traversable) -> Any:
    with path.open("r") as abi:
        return json.load(abi)


order_book_abi = load_abi(abi_path / "mach" / "order_book.json")

erc20_abi = load_abi(abi_path / "ethereum" / "erc20.json")

aave_pool_addresses = {
    ChainId.ETHEREUM: "0x87870Bca3F3fD6335C3F4ce8392D69350B4fA4E2",
    ChainId.OP: "0x794a61358D6845594F94dc1DB02A252b5b4814aD",
    ChainId.POLYGON: "0x794a61358D6845594F94dc1DB02A252b5b4814aD",
    ChainId.OPBNB: "0x6807dc923806fE8Fd134338EABCA509979a7e0cB",
    # ChainId.MANTLE
    ChainId.BASE: "0xA238Dd80C259a72e81d7e4664a9801593F98d1c5",
    # ChainId.MODE
    ChainId.ARBITRUM: "0x794a61358D6845594F94dc1DB02A252b5b4814aD",
    # ChainId.CELO
    ChainId.AVALANCHE_C_CHAIN: "0x794a61358D6845594F94dc1DB02A252b5b4814aD",
    # ChainId.BLAST
    ChainId.SCROLL: "0x11fCfe756c05AD438e312a7fd934381537D3cFfe",
}

aave_pool_data_provider_addresses = {
    ChainId.ETHEREUM: "0x7B4EB56E7CD4b454BA8ff71E4518426369a138a3",
    ChainId.OP: "0x69FA688f1Dc47d4B5d8029D5a35FB7a548310654",
    ChainId.POLYGON: "0x69FA688f1Dc47d4B5d8029D5a35FB7a548310654",
    ChainId.OPBNB: "0xff75B6da14FfbbfD355Daf7a2731456b3562Ba6D",
    # ChainId.MANTLE
    ChainId.BASE: "0x2d8A3C5677189723C4cB8873CfC9C8976FDF38Ac",
    # ChainId.MODE
    ChainId.ARBITRUM: "0x69FA688f1Dc47d4B5d8029D5a35FB7a548310654",
    # ChainId.CELO
    ChainId.AVALANCHE_C_CHAIN: "0x69FA688f1Dc47d4B5d8029D5a35FB7a548310654",
    # ChainId.BLAST
    ChainId.SCROLL: "0xa99F4E69acF23C6838DE90dD1B5c02EA928A53ee",
}

aave_pool_data_provider_abi = load_abi(abi_path / "aave" / "pool_data_provider.json")

@functools.cache
def aave_pool_abi(chain: ChainId) -> Any:
    """
    Load a JSON abi from a file.

    :param envvar: Environment variable pointing to the path.
    :param default_path: The path to check if environment var isn't set.
    """

    abi_path_full = abi_path / "aave" / "pool" / f"{chain}.json"

    return load_abi(abi_path_full)
