#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Пример использования pyQuantumAI с моделями NativeMind

Использует модели из https://huggingface.co/nativemind/models
© 2025 NativeMind
"""

import sys
from pathlib import Path

# Добавляем путь к модулям
sys.path.insert(0, str(Path(__file__).parent.parent / "src"))

from quantum_sync.equalizer import QuantumBatteryEqualizer, ModelChannel
from quantum_sync.pyramid import QuantumPyramid
from quantum_sync.school import QuantumSchool


def example_quantum_equalizer():
    """Пример использования квантового эквалайзера батарей"""
    print("🔋 Пример: Квантовый Эквалайзер Батарей")
    print("=" * 80)
    
    # Модели из NativeMind на Hugging Face
    channels = [
        ModelChannel(
            name="Mozgach108-Battery",
            model_path="nativemind/mozgach108",
            capacitance=1.0,  # Фарады
            voltage=440.0,    # Вольты
            charge=0.0,
            entanglement_level=0.0,
            face=0
        ),
        ModelChannel(
            name="Braindler-Battery",
            model_path="nativemind/braindler_final_model", 
            capacitance=1.2,
            voltage=440.0,
            charge=0.0,
            entanglement_level=0.0,
            face=1
        ),
        ModelChannel(
            name="Shridhar-Battery",
            model_path="nativemind/shridhar_8k",
            capacitance=0.8,
            voltage=440.0,
            charge=0.0,
            entanglement_level=0.0,
            face=2
        ),
        ModelChannel(
            name="Sphere028-Battery",
            model_path="nativemind/sphere-028-support",
            capacitance=1.5,
            voltage=440.0,
            charge=0.0,
            entanglement_level=0.0,
            face=3
        )
    ]
    
    # Создаем эквалайзер батарей
    equalizer = QuantumBatteryEqualizer(
        channels=channels,
        resonance_voltage=440.0
    )
    
    # Анализируем квантовые состояния батарей
    print("\n📊 Анализ квантовых состояний...")
    signatures = equalizer.extract_all_signatures()
    
    # Рассчитываем интерференцию
    print("\n🌊 Расчет интерференции...")
    interference = equalizer.calculate_interference_pattern("Sphere028-Battery")
    
    # Заряжаем квантовые батареи
    print("\n⚡ Зарядка квантовых батарей...")
    result = equalizer.balance(
        target_model="Sphere028-Battery",
        learning_rate=0.05,
        cycles=10
    )
    
    print(f"\n✅ Результат зарядки: {result['final_sync']:.1%}")
    return result


def example_quantum_pyramid():
    """Пример использования квантовой пирамиды с частотным анализом"""
    print("\n🔺 Пример: Квантовая Пирамида с Частотным Анализом")
    print("=" * 80)
    
    # Создаем пирамиду
    pyramid = QuantumPyramid(
        base_side=50.8,
        height=48.05,
        resonance_freq=440.0
    )
    
    # Размещаем модели на гранях
    models = [
        ("Mozgach108", "nativemind/mozgach108", 0, "teacher"),
        ("Braindler", "nativemind/braindler_final_model", 1, "teacher"),
        ("Shridhar", "nativemind/shridhar_8k", 2, "teacher"),
        ("Sphere028", "nativemind/sphere-028-support", 3, "student")
    ]
    
    for name, path, face, role in models:
        pyramid.place_model(name, path, face, role)
    
    # Частотный анализ
    print("\n📊 Частотный анализ моделей...")
    for name, _, _, _ in models:
        spectrum = pyramid.analyze_frequency_spectrum(name)
        print(f"  {name}: {spectrum['dominant_frequency']:.2f} Hz")
    
    # Расчет квантовой запутанности
    print("\n🔗 Расчет квантовой запутанности...")
    entanglement = pyramid.calculate_quantum_entanglement("Mozgach108", "Braindler")
    print(f"  Mozgach108 ↔ Braindler: {entanglement:.1%}")
    
    # Оптимизация резонансной частоты
    print("\n🎵 Оптимизация резонансной частоты...")
    optimal_freq = pyramid.optimize_resonance_frequency()
    print(f"  Оптимальная частота: {optimal_freq:.2f} Hz")
    
    # Синхронизация
    print("\n⚡ Синхронизация через пирамиду...")
    result = pyramid.synchronize(
        target="Sphere028",
        cycles=10,
        learning_rate=0.05
    )
    
    print(f"\n✅ Результат синхронизации: {result['final_sync']:.1%}")
    return result


def example_quantum_school():
    """Пример использования квантовой школы"""
    print("\n🏫 Пример: Квантовая Школа")
    print("=" * 80)
    
    # Учителя из NativeMind
    teachers = [
        "nativemind/mozgach108",
        "nativemind/braindler_final_model",
        "nativemind/shridhar_8k"
    ]
    
    # Ученики из NativeMind
    students = [
        {
            "name": "Sphere027",
            "path": "nativemind/sphere-027-sales-engineer",
            "output": "./quantum_school/sphere_027"
        },
        {
            "name": "Sphere028",
            "path": "nativemind/sphere-028-support", 
            "output": "./quantum_school/sphere_028"
        },
        {
            "name": "Sphere049",
            "path": "nativemind/sphere-049-judge",
            "output": "./quantum_school/sphere_049"
        }
    ]
    
    # Создаем школу
    school = QuantumSchool(
        base_frequency=440.0,
        method="equalizer",
        max_workers=2,
        debug=True
    )
    
    # Обучаем класс
    print("\n📚 Обучение класса...")
    result = school.train_class(
        teachers=teachers,
        students=students,
        cycles=5,  # Короткий пример
        learning_rate=0.05,
        parallel=True
    )
    
    print(f"\n✅ Результаты обучения:")
    print(f"  Успешность: {result['success_rate']:.1%}")
    print(f"  Средняя синхронизация: {result['synchronization']['average']:.1%}")
    
    # Статистика школы
    stats = school.get_statistics()
    print(f"\n📊 Статистика школы:")
    print(f"  Обучено учеников: {stats['students_trained']}")
    print(f"  Средняя синхронизация: {stats['average_sync']:.1%}")
    
    return result


def main():
    """Главная функция с примерами"""
    print("🚀 pyQuantumAI - Примеры использования с моделями NativeMind")
    print("=" * 80)
    print("Модели из: https://huggingface.co/nativemind/models")
    print()
    
    try:
        # Пример 1: Квантовый эквалайзер батарей
        equalizer_result = example_quantum_equalizer()
        
        # Пример 2: Квантовая пирамида с частотным анализом
        pyramid_result = example_quantum_pyramid()
        
        # Пример 3: Квантовая школа
        school_result = example_quantum_school()
        
        print("\n🎉 Все примеры выполнены успешно!")
        print("=" * 80)
        
    except Exception as e:
        print(f"\n⚠️ Ошибка в примерах: {e}")
        print("Это нормально для демонстрации - модели могут быть недоступны")


if __name__ == "__main__":
    main()
