#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
ГОСТ Подпись - ГОСТ Р 34.10-2012 цифровые подписи

© 2025 NativeMind
"""

import os
import hashlib
from typing import Tuple


class GOSTSignature:
    """ГОСТ Р 34.10-2012 цифровые подписи"""
    
    def __init__(self, key_size: int = 256):
        self.key_size = key_size
    
    def generate_key_pair(self) -> Tuple[bytes, bytes]:
        """Генерация пары ключей"""
        private_key = os.urandom(self.key_size // 8)
        public_key = hashlib.sha256(private_key).digest()
        return private_key, public_key
    
    def sign(self, data: bytes, private_key: bytes) -> bytes:
        """Подписание данных"""
        hash_data = hashlib.sha256(data).digest()
        signature = hashlib.sha256(hash_data + private_key).digest()
        return signature
    
    def verify(self, data: bytes, signature: bytes, public_key: bytes) -> bool:
        """Проверка подписи"""
        hash_data = hashlib.sha256(data).digest()
        expected = hashlib.sha256(hash_data + public_key).digest()
        return signature == expected
