function my = GetMy(alpha,beta,fi,drn,del,dnos,Wx,Wy,V,l)
% --------------------------------------------------------------
%       Вычисление коэффициента момента рыскания my.
% --------------------------------------------------------------
% my = GetMy(alpha,beta,fi,drn,del,Wx,Wy,V)  вычисляет коэффициент
%    момента рыскания в зависимости от величин:
% alpha - угол атаки, рад.
% beta - угол скольжения, рад.
% fi - отклонение стабилизатора, рад.
% drn - отклонение руля направления, рад.
% del - отклонение элеронов (дифференциальное), рад.
% dnos - отклонение носков крыла, рад.
% V - воздушная скорость, м/с
% Wx - угловая скорость крена, рад/с
% Wy - угловая скорость рыскания, рад/с
%
% Зависимости заданы в виде таблиц.
% Таблицы интерполяции хранятся в оперативной памяти для ускорения
% численного расчёта.
% --------------------------------------------------------------
persistent init_mdl
persistent alpha1 alpha2 beta1 fi2 my1 my_nos1 mydel20 mydel20_nos
persistent mydrn30 dmybt1 mywy1 dmywy_nos1 mywx1 dmywx_nos1
persistent interpmy interpmy_nos interpdmybt interpmywx1 interpmywx2 interpmywy1 interpmywy2 interpmydel interpmydel_nos interpmydrn

% Инициализация
if isempty(init_mdl)

init_mdl = 1;

alpha1 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45 50 55 60 70 80 90]);
alpha2 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45]);
beta1 = deg2rad([-30 -25 -20 -15 -10 -8 -6 -4 -2 0 2 4 6 8 10 15 20 25 30]);
fi2 = deg2rad([-25 0 25]);

% Коэффициент момента рыскания от углов атаки и скольжения.
% my = f(alpha, beta, fi)

% fi = -25 grad
my1(:,:,1) = [-0.06330	-0.06670	-0.05650	-0.04180	-0.01750	-0.00930	-0.00600	0.00470	0.00340	0.00000	-0.00480	-0.01060	-0.00740	-0.00150	0.00520	0.02970	0.04430	0.05450	0.05100;
    -0.06210	-0.05790	-0.04540	-0.02850	-0.01810	-0.01330	-0.00670	-0.00100	0.00100	0.00000	0.00040	0.00280	0.00710	0.01510	0.02000	0.03030	0.04730	0.06020	0.06410;
    -0.06728	-0.05880	-0.04930	-0.03930	-0.02420	-0.01670	-0.00980	-0.00220	0.00220	0.00000	0.00470	0.00960	0.01630	0.02450	0.03200	0.04730	0.05720	0.06660	0.07540;
    -0.08500	-0.07610	-0.06390	-0.04780	-0.03540	-0.02630	-0.01840	-0.01140	-0.00550	0.00000	0.00540	0.01120	0.01890	0.02900	0.03920	0.05160	0.06800	0.08000	0.08860;
    -0.09950	-0.08690	-0.07950	-0.05280	-0.03750	-0.02800	-0.01930	-0.01180	-0.00530	0.00000	0.00550	0.01220	0.02080	0.03020	0.03930	0.05470	0.07060	0.08910	0.10340;
    -0.10440	-0.08240	-0.06910	-0.05210	-0.03520	-0.02800	-0.01930	-0.01210	-0.00500	0.00000	0.00560	0.01320	0.02100	0.03010	0.03830	0.05530	0.07210	0.08580	0.10750;
    -0.09810	-0.07590	-0.06310	-0.04780	-0.03580	-0.02830	-0.02010	-0.01250	-0.00540	0.00000	0.00540	0.01310	0.02250	0.03090	0.03910	0.05120	0.06680	0.07980	0.10180;
    -0.09760	-0.06180	-0.04750	-0.04470	-0.03390	-0.02670	-0.01800	-0.01140	-0.00450	0.00000	0.00550	0.01290	0.02230	0.03040	0.03720	0.04800	0.05090	0.06470	0.09090;
    -0.06770	-0.05060	-0.02900	-0.02760	-0.02590	-0.02160	-0.01510	-0.00880	-0.00400	0.00000	-0.00220	0.00210	0.00990	0.01610	0.02100	0.02260	0.02410	0.04600	0.06270;
    -0.04880	-0.03510	-0.01630	-0.01280	-0.01550	-0.01150	-0.00720	-0.00370	-0.00160	0.00000	0.00130	0.00470	0.00850	0.01320	0.01570	0.01320	0.01620	0.03470	0.04870;
    -0.01020	0.01550	0.02870	0.02560	0.02940	0.00670	0.00400	0.00460	0.00380	0.00000	-0.00420	-0.00500	-0.00690	-0.00900	-0.01150	-0.02140	-0.02180	0.00550	0.04170;
    -0.00280	0.03140	0.05720	0.07120	0.05450	0.05370	0.04130	0.02540	0.01450	0.00000	-0.01040	-0.01620	-0.02230	-0.03120	-0.05060	-0.06700	-0.05360	-0.02760	0.00690;
    -0.00370	0.01670	0.07700	0.08030	0.05730	0.04330	0.02920	0.01840	0.00680	0.00000	-0.00480	-0.01150	-0.02330	-0.03320	-0.04920	-0.07620	-0.07270	-0.01250	0.00790;
    -0.01200	0.00270	0.03970	0.05770	0.03990	0.03040	0.02000	0.01470	0.00620	0.00000	-0.01450	-0.03560	-0.04420	-0.05800	-0.06980	-0.09000	-0.07040	-0.03360	-0.01910;
    -0.03730	-0.02740	-0.00960	0.02160	0.03190	0.02960	0.02980	0.01570	0.01040	0.00000	-0.00820	-0.02550	-0.04410	-0.06190	-0.07880	-0.06930	-0.03840	-0.02170	-0.01200;
    -0.00449	-0.03240	0.01020	-0.00770	-0.01610	-0.00900	-0.00570	-0.00650	0.00400	0.00000	-0.00190	-0.01520	-0.02750	-0.03150	-0.03050	-0.03860	-0.05640	-0.01370	-0.00170;
    -0.00550	0.00680	0.03740	0.01190	0.02340	0.01270	-0.00160	-0.01200	-0.00290	0.00000	0.00520	0.00570	-0.01010	-0.02150	-0.02210	-0.02630	-0.03580	-0.00550	0.00660;
    0.02320	0.02800	0.02030	0.01270	0.00070	-0.00310	-0.00700	-0.01370	-0.01680	0.00000	0.00280	0.01330	0.01380	0.00830	0.00180	-0.01000	-0.01730	-0.02510	-0.02070;
    0.02360	0.02370	0.01610	0.01160	0.00990	0.01100	0.01080	0.00870	0.00590	0.00000	-0.00130	0.00350	-0.00540	-0.00690	-0.00540	-0.00750	-0.01170	-0.01920	-0.01910;
    0.03190	0.01990	0.01080	0.00180	0.00790	0.00620	0.00390	0.00290	0.00180	0.00000	-0.00640	-0.00510	-0.00980	-0.00970	-0.01010	-0.00380	-0.00720	-0.01590	-0.02770];

% fi = 0 grad
my1(:,:,2) = [-0.05510	-0.05880	-0.04960	-0.04060	-0.02190	-0.01450	-0.00750	-0.00120	0.00020	0.00000	-0.00900	0.00120	0.00590	0.00990	0.01410	0.03330	0.04250	0.05160	0.04770;
    -0.05610	-0.05270	-0.04560	-0.03330	-0.02480	-0.01790	-0.01270	-0.00570	-0.00180	0.00000	0.00250	0.00580	0.01110	0.01800	0.02380	0.03300	0.04500	0.05210	0.05530;
    -0.06660	-0.06370	-0.05450	-0.04680	-0.02970	-0.02330	-0.01450	-0.00790	-0.00310	0.00000	0.00280	0.00750	0.01500	0.02210	0.03020	0.04740	0.05520	0.06460	0.06740;
    -0.09020	-0.08120	-0.06640	-0.05230	-0.03660	-0.02770	-0.01940	-0.01170	-0.00550	0.00000	0.00630	0.01270	0.02140	0.02970	0.03980	0.05530	0.06930	0.08430	0.09330;
    -0.10580	-0.09160	-0.07490	-0.05780	-0.04130	-0.03170	-0.02260	-0.01380	-0.00660	0.00000	0.00610	0.01350	0.02250	0.03240	0.04140	0.05790	0.07460	0.09140	0.10550;
    -0.10740	-0.09160	-0.07540	-0.05870	-0.04150	-0.03290	-0.02270	-0.01450	-0.00640	0.00000	0.00610	0.01480	0.02310	0.03350	0.04210	0.05940	0.07620	0.09140	0.10790;
    -0.09810	-0.07980	-0.07180	-0.05680	-0.04160	-0.03260	-0.02320	-0.01460	-0.00620	0.00000	0.00630	0.01470	0.02400	0.03320	0.04270	0.05790	0.07270	0.08090	0.09950;
    -0.08120	-0.05920	-0.05370	-0.05130	-0.03750	-0.03010	-0.02120	-0.01210	-0.00520	0.00000	0.00630	0.01410	0.02430	0.03340	0.04040	0.05410	0.05660	0.06220	0.08400;
    -0.06840	-0.04910	-0.02900	-0.03210	-0.03080	-0.02620	-0.01790	-0.01020	-0.00420	0.00000	0.00180	0.00680	0.01520	0.02330	0.02960	0.03090	0.02790	0.04790	0.06740;
    -0.05280	-0.04110	-0.02230	-0.02290	-0.02400	-0.01880	-0.01290	-0.00720	-0.00290	0.00000	0.00330	0.00880	0.01470	0.02160	0.02580	0.02510	0.02420	0.04290	0.05470;
    -0.03000	0.00020	0.01150	0.01640	0.00910	-0.00370	-0.00240	0.00900	0.00250	0.00000	-0.00290	-0.00230	-0.00130	-0.00300	-0.00190	-0.00970	-0.00420	0.00690	0.03700;
    -0.00980	0.01680	0.03920	0.05140	0.03960	0.03400	0.01630	0.01030	0.00690	0.00000	-0.00970	-0.01470	-0.01570	-0.01890	-0.02950	-0.04150	-0.02910	-0.00680	0.01940;
    -0.00250	0.00540	0.06830	0.07440	0.05060	0.03510	0.02070	0.01310	0.00520	0.00000	-0.00710	-0.01360	-0.02160	-0.03290	-0.04400	-0.06770	-0.06220	0.00120	0.02020;
    -0.01110	0.00100	0.02940	0.06120	0.04510	0.03690	0.02930	0.02010	0.01160	0.00000	-0.02370	-0.03750	-0.04600	-0.05650	-0.06940	-0.08470	-0.05300	-0.02460	-0.01260;
    -0.02560	-0.01360	0.00580	0.02870	0.02540	0.02310	0.02330	0.01050	0.00780	0.00000	-0.00630	-0.02170	-0.03550	-0.04560	-0.05480	-0.05740	-0.03460	-0.01610	-0.00460;
    -0.03020	-0.02280	0.01300	0.01400	0.00400	0.00270	-0.00230	-0.00700	0.00430	0.00000	0.00280	-0.00580	-0.01720	-0.02390	-0.02580	-0.03610	-0.03550	0.00140	0.00730;
    -0.01880	-0.00750	0.02110	0.00800	-0.00610	-0.01000	-0.01740	-0.02190	-0.00790	0.00000	0.00750	0.01030	0.00430	-0.00130	-0.00190	-0.01560	-0.02840	0.00040	0.01100;
    0.02960	0.03160	0.02100	0.00920	0.00030	-0.00620	-0.01280	-0.01930	-0.01870	0.00000	0.00390	0.01510	0.01630	0.01160	0.00590	-0.00230	-0.01410	-0.02460	-0.02280;
    0.02640	0.03510	0.02540	0.01800	0.01330	0.01260	0.01070	0.00790	0.00550	0.00000	-0.00010	0.00600	-0.00330	-0.00690	-0.00750	-0.01300	-0.01980	-0.02420	-0.02090;
    0.02740	0.01280	0.01180	0.00590	0.00510	0.00440	0.00310	0.00270	0.00170	0.00000	-0.00180	-0.00230	-0.00310	-0.00480	-0.00480	-0.00540	-0.01110	-0.01210	-0.01630];

% fi = 25 grad
my1(:,:,3) = [-0.04880	-0.05150	-0.04420	-0.04280	-0.02150	-0.01360	-0.00460	-0.00180	0.00010	0.00000	-0.00050	-0.00030	0.00480	0.00840	0.01430	0.03560	0.03690	0.04410	0.04250;
    -0.04990	-0.04630	-0.04020	-0.03240	-0.02010	-0.01540	-0.00950	-0.00290	-0.00130	0.00000	0.00050	0.00310	0.00930	0.01450	0.01950	0.03180	0.03980	0.04590	0.04920;
    -0.05740	-0.05340	-0.04770	-0.04240	-0.02770	-0.02080	-0.01340	-0.00730	-0.00250	0.00000	0.00180	0.00750	0.01400	0.02220	0.02770	0.04210	0.04760	0.05340	0.05720;
    -0.07580	-0.07140	-0.06170	-0.05070	-0.03680	-0.02900	-0.02080	-0.01280	-0.00610	0.00000	0.00640	0.01390	0.02220	0.03040	0.03990	0.05360	0.06450	0.07420	0.07870;
    -0.09190	-0.08180	-0.06940	-0.05600	-0.04020	-0.03110	-0.02230	-0.01410	-0.00650	0.00000	0.00690	0.01470	0.02300	0.03190	0.04080	0.05650	0.06960	0.08240	0.09240;
    -0.08600	-0.07490	-0.06590	-0.05310	-0.04060	-0.03220	-0.02230	-0.01270	-0.00470	0.00000	0.00420	0.01240	0.02210	0.03230	0.04150	0.05670	0.06940	0.07860	0.08920;
    -0.08210	-0.07230	-0.06530	-0.05340	-0.04030	-0.03280	-0.02330	-0.01350	-0.00610	0.00000	0.00490	0.01260	0.02180	0.03100	0.04000	0.05310	0.06490	0.07180	0.08140;
    -0.06710	-0.05160	-0.04860	-0.04960	-0.03570	-0.02890	-0.01950	-0.01070	-0.00480	0.00000	0.00380	0.01080	0.02080	0.03060	0.03810	0.05100	0.04990	0.05320	0.06840;
    -0.03980	-0.03550	-0.02370	-0.02840	-0.03110	-0.02700	-0.01830	-0.00910	-0.00350	0.00000	0.00280	0.00520	0.01780	0.02680	0.03320	0.03050	0.02590	0.03740	0.04170;
    -0.02730	-0.02100	-0.01320	-0.01480	-0.02190	-0.01960	-0.01590	-0.00890	-0.00330	0.00000	0.00430	0.01030	0.01790	0.02640	0.03110	0.02390	0.02240	0.03020	0.03620;
    -0.01160	0.01420	0.02730	0.02420	0.01110	-0.00660	-0.00630	-0.00200	0.00090	0.00000	-0.00100	-0.00060	0.00180	0.00390	0.00180	-0.01110	-0.01460	-0.00120	0.02440;
    0.00180	0.02820	0.04990	0.05500	0.04300	0.03820	0.01930	0.00990	0.00690	0.00000	-0.00860	-0.01260	-0.01540	-0.01810	-0.02880	-0.04560	-0.04020	-0.01850	0.00710;
    0.00030	-0.01930	0.06980	0.07880	0.05340	0.03720	0.02520	0.01690	0.00730	0.00000	-0.00840	-0.01470	-0.02480	-0.03620	-0.04800	-0.07330	-0.06410	-0.00520	0.00510;
    -0.01490	-0.00070	0.02260	0.05690	0.04550	0.03630	0.02880	0.01880	0.00890	0.00000	-0.02520	-0.04030	-0.05110	-0.06210	-0.07120	-0.08040	-0.04800	-0.02470	-0.02070;
    -0.02190	-0.01740	-0.00770	0.01710	0.03100	0.03070	0.03280	0.01890	0.01200	0.00000	-0.00580	-0.02510	-0.04080	-0.05430	-0.06640	-0.05300	-0.02920	-0.01870	-0.01510;
    -0.05180	-0.04350	-0.00530	-0.03070	-0.02310	-0.01080	-0.00220	-0.00160	0.00650	0.00000	-0.00260	-0.00850	-0.02230	-0.02570	-0.02440	-0.01710	-0.04240	-0.00420	0.00310;
    -0.02700	-0.02070	0.00420	-0.01370	-0.01370	-0.01380	-0.01730	-0.02030	-0.00710	0.00000	0.00930	0.01380	0.00670	-0.00280	-0.00730	-0.00720	-0.02540	-0.00080	0.00560;
    0.01580	0.02700	0.02520	0.01170	-0.00100	-0.00390	-0.00680	-0.01320	-0.01590	0.00000	-0.00390	0.01100	0.00880	0.00840	0.00690	-0.00590	-0.01910	-0.02090	-0.01040;
    0.01060	0.01820	0.01820	0.01170	0.00810	0.00960	0.00990	0.00810	0.00560	0.00000	-0.00100	0.00420	-0.00430	0.00160	-0.00350	-0.01120	-0.01350	-0.01350	-0.00580;
    0.01180	0.01010	0.01170	0.00360	0.00600	0.00530	0.00410	0.00350	0.00210	0.00000	-0.00020	0.00080	0.00080	-0.00080	-0.00080	0.00140	-0.00680	-0.00540	-0.00720];

my1 = -my1;
    
% Коэффициент момента рыскания от углов атаки и скольжения, носки выпущены
% my_nos = f(alpha2, beta)
my_nos1 = [-0.05410	-0.05630	-0.04610	-0.04950	-0.02960	-0.02080	-0.01730	-0.01000	-0.00430	0.00000	0.00370	0.00760	0.01210	0.01860	0.02310	0.04280	0.03930	0.04940	0.04680;
    -0.06780	-0.07280	-0.06580	-0.05390	-0.03580	-0.02820	-0.02040	-0.01260	-0.00580	0.00000	0.00570	0.01250	0.02060	0.02880	0.03640	0.05390	0.06580	0.07300	0.06830;
    -0.07800	-0.07730	-0.06290	-0.05550	-0.03700	-0.02890	-0.02180	-0.01420	-0.00680	0.00000	0.00690	0.01410	0.02240	0.03010	0.03870	0.05720	0.06450	0.07150	0.07940;
    -0.08810	-0.08510	-0.07530	-0.05560	-0.04020	-0.03080	-0.02540	-0.01410	-0.00670	0.00000	0.00670	0.01440	0.02340	0.03280	0.04300	0.05860	0.07850	0.08720	0.09130;
    -0.10600	-0.09290	-0.07540	-0.05930	-0.04200	-0.03190	-0.02220	-0.01350	-0.00620	0.00000	0.00660	0.01430	0.02340	0.03400	0.04290	0.05950	0.07590	0.09380	0.10660;
    -0.10510	-0.08770	-0.07280	-0.05730	-0.04100	-0.03240	-0.02250	-0.01400	-0.00610	0.00000	0.00620	0.01490	0.02290	0.03360	0.04230	0.05890	0.07470	0.08770	0.10690;
    -0.09260	-0.07970	-0.07310	-0.05800	-0.04240	-0.03270	-0.02350	-0.01540	-0.00640	0.00000	0.00640	0.01150	0.02430	0.03300	0.04250	0.05790	0.07300	0.07960	0.09250;
    -0.06320	-0.06700	-0.06530	-0.05490	-0.04140	-0.03160	-0.02230	-0.01350	-0.00590	0.00000	0.00550	0.01430	0.02320	0.03250	0.04170	0.05330	0.06560	0.06730	0.06350;
    -0.03590	-0.01910	-0.01730	-0.02300	-0.02160	-0.01740	-0.00760	-0.00580	-0.00150	0.00000	0.00300	0.00870	0.01590	0.02270	0.02790	0.02920	0.02370	0.02530	0.04210;
    -0.03420	-0.02080	-0.00170	0.00630	-0.00590	-0.00940	-0.00610	-0.00290	-0.00120	0.00000	0.00080	0.00380	0.00690	0.00780	0.00340	-0.00880	-0.00080	0.01830	0.03170;
    -0.02650	-0.00470	0.01280	0.02490	0.01980	0.01140	0.00550	0.00570	0.00300	0.00000	-0.00320	-0.00770	-0.01170	-0.02010	-0.02760	-0.03320	-0.02090	-0.00340	0.01880;
    0.01380	0.03910	0.05330	0.05530	0.04340	0.03970	0.02630	0.02060	0.01190	0.00000	-0.00900	-0.01340	-0.01900	-0.02630	-0.03280	-0.04440	-0.04270	-0.02790	-0.00210;
    0.03020	0.03570	0.06750	0.06450	0.04450	0.03300	0.02140	0.01560	0.00650	0.00000	-0.00600	-0.01360	-0.01550	-0.02660	-0.03300	-0.05320	-0.05610	-0.02400	0.00140;
    0.00030	-0.00380	0.02140	0.04000	0.03260	0.02610	0.01990	0.01300	0.00470	0.00000	-0.01700	-0.03690	-0.04640	-0.05330	-0.06110	-0.06810	-0.04950	-0.02450	-0.02800];

my_nos1 = -my_nos1;

% Коэффициент момента рыскания при максимальном отклонении элеронов (20 град)
% mydel20 = f(alpha1,beta)
mydel20 = [-0.06390	-0.06280	-0.06160	-0.05500	-0.03590	-0.02670	-0.01880	-0.01190	-0.00930	-0.00890	-0.00810	-0.00710	-0.00430	-0.00400	0.00200	0.02110	0.02770	0.02890	0.03000;
    -0.06190	-0.05540	-0.04900	-0.03840	-0.03360	-0.02790	-0.02320	-0.01740	-0.01370	-0.00980	-0.00660	-0.00420	0.00020	0.00880	0.01340	0.01820	0.02880	0.03520	0.04170;
    -0.06790	-0.05990	-0.05440	-0.04650	-0.03960	-0.03220	-0.02540	-0.01930	-0.01390	-0.00910	-0.00550	-0.00070	0.00470	0.01200	0.01860	0.02550	0.03340	0.03890	0.04690;
    -0.10800	-0.09940	-0.08380	-0.06770	-0.04600	-0.03980	-0.03210	-0.02480	-0.01760	-0.01110	-0.00540	0.00080	0.00740	0.01590	0.02460	0.04630	0.06240	0.07800	0.08660;
    -0.12340	-0.10940	-0.09150	-0.07210	-0.04980	-0.04480	-0.03770	-0.02770	-0.01930	-0.01200	-0.00560	0.00150	0.00920	0.01750	0.02680	0.04910	0.06850	0.08640	0.10040;
    -0.12450	-0.11000	-0.09390	-0.07300	-0.04960	-0.04400	-0.03600	-0.02650	-0.01760	-0.01050	-0.00370	0.00240	0.01090	0.01940	0.02850	0.05190	0.07280	0.08890	0.10340;
    -0.11180	-0.10200	-0.08940	-0.06900	-0.04860	-0.04400	-0.03490	-0.02670	-0.01710	-0.00900	-0.00200	0.00470	0.01320	0.02200	0.03180	0.05220	0.07260	0.08520	0.09500;
    -0.09670	-0.08070	-0.07370	-0.06280	-0.04720	-0.04160	-0.03790	-0.02340	-0.01360	-0.00660	-0.00030	0.00690	0.01580	0.02310	0.03280	0.04840	0.05930	0.06630	0.08230;
    -0.06700	-0.05610	-0.05050	-0.04720	-0.03580	-0.02690	-0.01980	-0.01110	-0.00290	-0.00010	0.00150	0.00520	0.01210	0.02040	0.02730	0.03870	0.04200	0.04760	0.05850;
    -0.03530	-0.03160	-0.02010	-0.02430	-0.01750	-0.01300	-0.00790	-0.00370	0.00120	0.00450	0.00720	0.01060	0.01590	0.02170	0.02770	0.03450	0.03030	0.04180	0.04550;
    -0.01870	0.00910	0.02300	0.01960	0.01320	0.00260	0.00210	0.00560	0.00820	0.00650	0.00390	0.00220	0.00300	0.00550	0.00450	-0.00191	-0.00530	0.00860	0.03640;
    0.00700	0.03570	0.05480	0.06580	0.04680	0.03830	0.02190	0.01780	0.01380	0.00990	0.00110	-0.00520	-0.00820	-0.00980	-0.01740	-0.03640	-0.02540	-0.00630	0.02240;
    0.00560	0.03220	0.08310	0.08810	0.05630	0.03950	0.02710	0.01870	0.01270	0.00440	-0.00090	-0.00600	-0.01310	-0.02240	-0.03280	-0.06460	-0.05780	-0.00870	0.01790;
    0.00460	0.01410	0.04040	0.06420	0.05130	0.04160	0.03190	0.02520	0.01640	0.00970	-0.00620	-0.02830	-0.03860	-0.04750	-0.05830	-0.07120	-0.04740	-0.02110	-0.01160;
    -0.10900	-0.00430	0.01570	0.03850	0.03860	0.03570	0.02820	0.02290	0.01960	0.01300	0.00710	-0.01400	-0.02110	-0.01980	-0.04520	-0.04510	-0.02230	-0.00230	0.00430;
    -0.01000	-0.01240	0.02560	0.03030	0.02370	0.02330	0.01660	0.01320	0.01930	0.01670	0.01750	-0.00250	-0.00420	-0.00600	-0.01830	-0.02490	-0.02020	0.01780	0.01540;
    0.00470	-0.00080	0.02810	0.02570	0.01650	0.01690	0.01150	0.00920	0.02070	0.01820	0.02360	0.01950	0.01580	0.01120	0.00390	-0.00530	-0.00770	0.02120	0.01570;
    0.04700	0.04260	0.03080	0.03010	0.02530	0.01860	0.01600	0.02060	0.01900	0.01540	0.02450	0.02160	0.02830	0.02180	0.01560	0.01080	0.01010	-0.00170	-0.00610;
    0.04100	0.04140	0.03680	0.03140	0.02510	0.02480	0.02330	0.01840	0.01560	0.01380	0.01540	0.01330	0.01010	0.00750	0.00810	0.00180	-0.00360	-0.00820	-0.00780;
    0.03200	0.02870	0.02370	0.01650	0.01650	0.01530	0.01510	0.01550	0.01380	0.01250	0.01130	0.01100	0.01010	0.01010	0.01170	0.01170	0.00450	-0.00500	-0.00380];

mydel20 = -mydel20;

% --//--//--//--//--//--//--//--//--, носки выпущены.
% mydel20_nos = f(alpha2,beta)
mydel20_nos = [-0.06830	-0.06150	-0.05560	-0.05190	-0.03930	-0.03140	-0.02640	-0.01990	-0.01400	-0.00950	-0.00540	-0.00290	0.00190	0.00740	0.01240	0.02510	0.02930	0.03540	0.04210;
    -0.07330	-0.07020	-0.06630	-0.05510	-0.04370	-0.03720	-0.03010	-0.02330	-0.01700	-0.01080	-0.00460	0.00170	0.00820	0.01590	0.02300	0.03430	0.04550	0.04970	0.05280;
    -0.07750	-0.06830	-0.06100	-0.05270	-0.04340	-0.03850	-0.03010	-0.02400	-0.01750	-0.01080	-0.00400	0.00270	0.00890	0.01610	0.02360	0.03270	0.04070	0.04790	0.05750;
    -0.11490	-0.10670	-0.08980	-0.07160	-0.04820	-0.04290	-0.03590	-0.02670	-0.01880	-0.01130	-0.00500	0.00240	0.00930	0.01860	0.02780	0.05110	0.06890	0.08560	0.09320;
    -0.12250	-0.11060	-0.09090	-0.07220	-0.04820	-0.04280	-0.03590	-0.02560	-0.01700	-0.00990	-0.00270	0.00420	0.01210	0.01970	0.02920	0.05330	0.07200	0.09150	0.10340;
    -0.11620	-0.10300	-0.08730	-0.06770	-0.04650	-0.04060	-0.03280	-0.02400	-0.01450	-0.00770	-0.00080	0.00550	0.01340	0.02220	0.03160	0.05390	0.07290	0.08880	0.10240;
    -0.10240	-0.09440	-0.08270	-0.06580	-0.04500	-0.04010	-0.03070	-0.02240	-0.01370	-0.00560	0.00150	0.00790	0.01640	0.02510	0.03450	0.05500	0.07190	0.08380	0.09170;
    -0.07990	-0.08160	-0.07890	-0.06080	-0.04330	-0.03780	-0.02860	-0.02010	-0.01040	-0.00370	0.00240	0.00800	0.01590	0.02490	0.03410	0.05130	0.06970	0.07210	0.06830;
    -0.03640	-0.02850	-0.03040	-0.03550	-0.02730	-0.02330	-0.01670	-0.01060	-0.00560	-0.00260	0.00040	0.00450	0.00950	0.01640	0.02290	0.03120	0.02600	0.02420	0.03230;
    -0.03700	-0.01630	-0.00250	0.00280	-0.00870	-0.01050	-0.00710	-0.00490	-0.00190	-0.00060	0.00040	0.00240	0.00410	0.00550	0.00150	-0.00980	-0.00500	0.00890	0.03070;
    -0.01690	0.00370	0.02100	0.03030	0.02110	0.01330	0.00960	0.01000	0.00810	0.00430	-0.00050	-0.00440	-0.00780	-0.01550	-0.02400	-0.03180	-0.02420	-0.00660	0.01390;
    0.02130	0.05430	0.06020	0.06590	0.05150	0.04390	0.03110	0.02360	0.01780	0.00680	0.00020	-0.00470	-0.00960	-0.01950	-0.02750	-0.04190	-0.03610	-0.02980	0.00290;
    0.01890	0.04630	0.08030	0.07860	0.05190	0.03920	0.02870	0.02090	0.01270	0.00620	-0.00170	-0.00790	-0.01050	-0.01610	-0.02210	-0.04860	-0.04970	-0.01550	0.01160;
    0.00550	0.00450	0.02240	0.04320	0.04190	0.00355	0.02740	0.02020	0.01410	0.00690	-0.01050	-0.03210	-0.03750	-0.04680	-0.05360	-0.05490	-0.03440	-0.01640	-0.01740];

mydel20_nos = -mydel20_nos;

% Коэффициент момента рыскания при максимальном отклонении РН (-30 градусов)
% mydrn30 = f(alpha1,beta)
mydrn30 = [-0.07870	-0.08150	-0.07410	-0.06560	-0.06200	-0.06270	-0.06160	-0.05510	-0.05200	-0.04810	-0.04940	-0.04860	-0.04650	-0.03960	-0.03100	-0.02700	-0.01910	-0.01130	-0.01430;
    -0.07580	-0.07450	-0.07080	-0.06100	-0.06230	-0.06580	-0.06490	-0.05800	-0.05220	-0.04840	-0.04650	-0.04370	-0.03950	-0.02980	-0.02020	-0.02160	-0.01170	-0.00820	-0.00690;
    -0.08500	-0.08330	-0.08280	-0.07490	-0.06700	-0.06850	-0.06570	-0.05900	-0.05200	-0.04760	-0.04470	-0.04070	-0.03380	-0.02380	-0.01250	-0.00460	0.00330	0.00430	0.00560;
    -0.14220	-0.12700	-0.11700	-0.09320	-0.07740	-0.07450	-0.06710	-0.05990	-0.05220	-0.04490	-0.04010	-0.03370	-0.02580	-0.01470	-0.00450	0.01520	0.03520	0.04980	0.05980;
    -0.15760	-0.13810	-0.11810	-0.09810	-0.07910	-0.07830	-0.06930	-0.06100	-0.05270	-0.04510	-0.03890	-0.03230	-0.02300	-0.01160	0.00000	0.01900	0.03890	0.05840	0.07840;
    -0.15910	-0.14060	-0.12160	-0.10260	-0.08190	-0.07930	-0.06960	-0.06100	-0.05200	-0.04500	-0.03880	-0.03110	-0.02200	-0.01000	0.00090	0.02150	0.04050	0.05940	0.07940;
    -0.15200	-0.13500	-0.11700	-0.09900	-0.08160	-0.07790	-0.06900	-0.06000	-0.05130	-0.04410	-0.03820	-0.03090	-0.02000	-0.00830	0.00031	0.02070	0.03810	0.05500	0.07250;
    -0.13060	-0.10910	-0.10260	-0.09060	-0.07520	-0.07590	-0.06940	-0.06050	-0.05170	-0.04460	-0.03860	-0.03200	-0.02010	-0.00880	0.00190	0.01730	0.02890	0.03490	0.05690;
    -0.12710	-0.10710	-0.08660	-0.08360	-0.06770	-0.06850	-0.06760	-0.06280	-0.05430	-0.04750	-0.04310	-0.04040	-0.03210	-0.02300	-0.00470	0.00120	0.00430	0.02490	0.04470;
    -0.10410	-0.09250	-0.07380	-0.06830	-0.05420	-0.06000	-0.06200	-0.05890	-0.05270	-0.04830	-0.04510	-0.04110	-0.03330	-0.02540	-0.01790	-0.00420	0.00130	0.01970	0.03170;
    -0.05980	-0.02950	-0.01830	-0.00980	-0.00490	-0.02810	-0.04220	-0.04750	-0.04740	-0.04940	-0.05100	-0.05140	-0.05040	-0.04370	-0.03670	-0.03250	-0.02420	-0.01300	0.01800;
    -0.04670	-0.02010	0.00610	0.01860	0.01590	0.01230	-0.00850	-0.02430	-0.03630	-0.04490	-0.05270	-0.05710	-0.06070	-0.05830	-0.05850	-0.06110	-0.04510	-0.02280	0.01040;
    -0.02890	-0.01110	0.03860	0.04840	0.03210	0.01450	0.00130	-0.01030	-0.02430	-0.00328	-0.04050	-0.04490	-0.04960	-0.05700	-0.06590	-0.08240	-0.07260	-0.02270	-0.00530;
    -0.02430	-0.01290	0.02130	0.04470	0.03250	0.02480	0.01400	0.00470	-0.00530	-0.01620	-0.04100	-0.05450	-0.06170	-0.06970	-0.07840	-0.09120	-0.06720	-0.03329	-0.02220;
    -0.03950	-0.02470	-0.00630	0.01770	0.01960	0.01490	0.00820	0.00220	0.00030	-0.00810	-0.01660	-0.03000	-0.04380	-0.05580	-0.06550	-0.06320	-0.03940	-0.02020	-0.00580;
    -0.03640	-0.03050	0.00880	0.00670	0.00060	-0.00180	-0.00750	-0.00750	0.00040	-0.00400	-0.00120	-0.00890	-0.02030	-0.02890	-0.03180	-0.03830	-0.04010	-0.00110	0.00510;
    -0.01620	-0.01270	0.01810	0.00260	-0.00840	-0.01210	-0.01950	-0.01930	-0.00820	-0.00120	0.00660	0.00960	0.00460	-0.00330	-0.00480	-0.01630	-0.03170	-0.00120	0.00260;
    0.02670	0.02970	0.01770	0.00690	-0.00160	-0.00810	-0.01560	-0.02030	-0.01520	-0.00150	0.00150	0.01430	0.01570	0.00940	0.00490	-0.00350	-0.01480	-0.02630	-0.02360;
    0.02230	0.02610	0.02150	0.01670	0.01090	0.00840	0.00500	0.00160	-0.00020	-0.00610	-0.00550	-0.00890	-0.00960	-0.01050	-0.01140	-0.01720	-0.02250	-0.02680	-0.02330;
    0.00890	0.00770	0.00680	0.00140	-0.00360	-0.00440	-0.00570	-0.00100	-0.00090	-0.00240	-0.00420	-0.00470	-0.00540	-0.00580	-0.00570	-0.01090	-0.01570	-0.01700	-0.01810];

mydrn30 = -mydrn30;
% При этом отклонение руля -30 град.

% Какая-то поправка на производную my по углу скольжения
% dmybt = f(alpha1)
dmybt1 = [0 0 0 0 0 0 0 0 0 -0.0008 0.0010 0 0 0 0 0 0 0 0 0]';
dmybt1 = -dmybt1;

% Производная my по угловой скорости рыскания
% mywy = f(alpha1)
mywy1 = [-0.51700 -0.51700 -0.51700 -0.46100 -0.41400 -0.39700 -0.37300 -0.45500 -0.55000 -0.58200 -0.59500 -0.63700 -1.02000 -0.84000 -0.54100 -0.35000 -0.35000 -0.07000 -0.15000 -0.15000]'; 

% Приращение mywy при выпуске носков
% dmywy_nos = f(alpha2)
dmywy_nos1 = [0.1370    0.1370    0.1370    0.0980    0.0370    0.0160    0.0070    0.0140   -0.1030   -0.0980   -0.3100   -0.4370    0.1670    0.0840]';

% Производная my по угловой скорости крена
% mywx = f(alpha1)
mywx1 = [-0.00060 -0.00060 -0.00060 0.04240 -0.00750 -0.02140 -0.03200 -0.03200 0.05000 0.15000 0.13000 0.15800 0.24000 0.15000 0.00000 -0.20000 -0.30000 0.15000 0.00000 0.00000]';
mywx1 = -mywx1;

% Приращение mywx при выпуске носков
% dmywx_nos = f(alpha2)
dmywx_nos1 = [0.0615    0.0615    0.0615    0.0091    0.0610    0.0129    0.0439    0.0512   -0.0294    0.0017    0.0584    0.2110    0.3920    0.1960]';
dmywx_nos1 = -dmywx_nos1;

interpmy = csapi({alpha1, beta1, fi2}, my1);
interpmy_nos = csaps({alpha2, beta1}, my_nos1, 1.0 - 1.0e-5);
interpdmybt = pchip(alpha1, dmybt1);
interpmywx1 = pchip(alpha1, mywx1);
interpmywx2 = pchip(alpha2, dmywx_nos1);
interpmywy1 = pchip(alpha1, mywy1);
interpmywy2 = pchip(alpha2, dmywy_nos1);
interpmydel = csaps({alpha1, beta1}, mydel20, 1.0 - 1.0e-5);
interpmydel_nos = csaps({alpha2, beta1}, mydel20_nos, 1.0 - 1.0e-5);
interpmydrn = csaps({alpha1, beta1}, mydrn30, 1.0 - 1.0e-5);
end

ntraj = size(alpha, 2);
my = zeros(1, ntraj);
my0 = zeros(1, ntraj);
my_nos = zeros(1, ntraj);
dmybt = zeros(1, ntraj);
mywx = zeros(1, ntraj);
mywy = zeros(1, ntraj);
mydel = zeros(1, ntraj);
mydel_nos = zeros(1, ntraj);
mydrn = zeros(1, ntraj);
for i = 1 : ntraj
    my(i) = fnval(interpmy, {(alpha(i)), (beta(i)), (fi(i))});
    my0(i) = fnval(interpmy, {(alpha(i)), (beta(i)), (0)});
    my_nos(i) = fnval(interpmy_nos, {(alpha(i)), (beta(i))});
    dmybt(i) = fnval(interpdmybt, alpha(i));
    mywx(i) = fnval(interpmywx1, alpha(i)) + fnval(interpmywx2, alpha(i)).*(dnos / deg2rad(25));
    mywy(i) = fnval(interpmywy1, alpha(i)) + fnval(interpmywy2, alpha(i)).*(dnos / deg2rad(25));
    mydel(i) = fnval(interpmydel, {(alpha(i)), (beta(i))});
    mydel_nos(i) = fnval(interpmydel_nos, {(alpha(i)), (beta(i))});
    mydrn(i) = fnval(interpmydrn, {(alpha(i)), (beta(i))});
end

% Cборка
dmy_nos = my_nos - my0;
dmydel = mydel - my0;
dmydel_nos = mydel_nos - my_nos - dmydel;
dmydrn = mydrn - my0;

my = my + dmy_nos.*(dnos / deg2rad(25)) + ...
          (dmydel + dmydel_nos.*(dnos / deg2rad(25))).*(del / deg2rad(20)) + ...
          dmydrn.*(drn / deg2rad(-30)) + ...
          mywx.*((Wx.*l)./(2 * V)) + mywy.*((Wy.*l)./(2 * V)) + dmybt.*beta;
                              
