function mz = GetMz(alpha,beta,fi,dnos,Wz,V,ba,sb)
% --------------------------------------------------------------
%        Вычисление коэффициента момента тангажа mz.
% --------------------------------------------------------------
% mz = GetMz(alpha,beta,fi,dnos,Wz,V)  вычисляет коэффициент момента
%    тангажа в зависимости от величин:
% alpha - угол атаки, рад.
% beta - угол скольжения, рад.
% fi - отклонение стабилизатора, рад.
% dnos - отклонение носков крыла, рад.
% V - воздушная скорость, м/с
% Wz - угловая скорость тангажа, рад/с
%
% Зависимости заданы в виде таблиц.
% Таблицы интерполяции хранятся в оперативной памяти для ускорения
% численного расчёта.
% init - признак инициализации.
%        0 - задавать таблицы, а затем интерполировать
%        1 - сразу интерполировать (таблицы должны быть заданы)
% --------------------------------------------------------------
persistent init_mdl
persistent alpha1 alpha2 fi1 fi2 beta1 mz1 mz_nos1 mzwz1 dmzwz_nos1 dmz1 dmz_sb1 eta_fi1 dmz_ds1
persistent interpmz interpmz_nos interpdmz interpmzwz1 interpmzwz2 interpdmz_sb interpeta_fi interpdmz_ds

% Инициализация
if isempty(init_mdl)

init_mdl = 1;

alpha1 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45 50 55 60 70 80 90]);
alpha2 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45]);
beta1 = deg2rad([-30 -25 -20 -15 -10 -8 -6 -4 -2 0 2 4 6 8 10 15 20 25 30]);
fi1 = deg2rad([-25 -10 0 10 25]);
fi2 = deg2rad([-25 -10 0 10 15 20 25]);

% Коэффициент момента тангажа в зависимости от угла атаки,
% угла скольжения и отклонения стабилизатора.
% mz = f(alpha1, beta1, fi1)

% fi = -25 grad
mz1(:,:,1) = [0.2059    0.1937    0.1918    0.1850    0.1692    0.1693    0.1770    0.1746    0.1742    0.1750    0.1721    0.1758    0.1801    0.1826    0.1912   0.2070    0.2140    0.2160    0.2270;
    0.1698    0.1650    0.1733    0.1723    0.1533    0.1618    0.1639    0.1607    0.1597    0.1584    0.1589    0.1615    0.1573    0.1534    0.1464   0.1650    0.1670    0.1580    0.1630;
    0.1426    0.1579    0.1807    0.1641    0.1533    0.1586    0.1595    0.1629    0.1615    0.1590    0.1566    0.1534    0.1523    0.1489    0.1493   0.1580    0.1750    0.1530    0.1380;
    0.1620    0.1770    0.1530    0.1450    0.1380    0.1365    0.1329    0.1269    0.1242    0.1216    0.1183    0.1212    0.1236    0.1267    0.1303   0.1373    0.1444    0.1679    0.1538;
    0.1530    0.1540    0.1480    0.1450    0.1445    0.1438    0.1430    0.1411    0.1412    0.1409    0.1410    0.1409    0.1403    0.1409    0.1414   0.1448    0.1453    0.1523    0.1513;
    0.1470    0.1530    0.1560    0.1570    0.1586    0.1595    0.1585    0.1577    0.1580    0.1580    0.1591    0.1584    0.1576    0.1572    0.1573   0.1568    0.1557    0.1540    0.1472;
    0.1500    0.1620    0.1650    0.1700    0.1746    0.1758    0.1768    0.1778    0.1833    0.1845    0.1840    0.1824    0.1811    0.1797    0.1784   0.1750    0.1701    0.1663    0.1547;
    0.1670    0.1760    0.1910    0.1960    0.2000    0.2012    0.2041    0.2062    0.2069    0.2087    0.2070    0.2066    0.2055    0.2022    0.2000   0.1958    0.1901    0.1755    0.1668;
    0.1510    0.1700    0.1900    0.2020    0.2073    0.2098    0.2122    0.2129    0.2137    0.2152    0.2133    0.2118    0.2109    0.2082    0.2061   0.1991    0.1878    0.1618    0.1479;
    0.1200    0.1470    0.1750    0.1940    0.2043    0.2028    0.2028    0.1991    0.1981    0.1978    0.1969    0.1957    0.1958    0.1948    0.1953   0.1877    0.1687    0.1406    0.1130;
    0.1080    0.0670    0.0980    0.1500    0.1704    0.1930    0.1985    0.2009    0.2022    0.2022    0.2021    0.2007    0.1972    0.1947    0.1901   0.1696    0.1186    0.0889    0.1270;
    0.0820    0.0470    0.0680    0.0810    0.1174    0.1233    0.1522    0.1713    0.1789    0.1814    0.1815    0.1799    0.1790    0.1703    0.1459    0.1124    0.0962    0.0757    0.1132;
    0.1130    0.0500    0.0600    0.0870    0.1131    0.1279    0.1341    0.1433    0.1483    0.1478    0.1291    0.1312    0.1245    0.1025    0.1028    0.0745    0.0495    0.0406    0.1024;
    0.0930    0.0660    0.0650    0.0530    0.0734    0.0914    0.0968    0.0848    0.0935    0.0922    0.0940    0.0838    0.0610    0.0491    0.0420   0.0208    0.0343    0.0338    0.0600;
   -0.0150   -0.0110   -0.0250    0.0150    0.0663    0.0644    0.0498    0.0407    0.0521    0.0745    0.0670    0.0453    0.0373    0.0320    0.0397  -0.0114   -0.0514   -0.0371   -0.0408;
    0.0190    0.0170   -0.0860   -0.0040    0.0794    0.0494    0.0174    0.0530    0.0292    0.0713    0.0404    0.0007   -0.0024    0.0165    0.0281  -0.0562   -0.1373   -0.0343   -0.0335;
   -0.0360   -0.0230   -0.0750   -0.0600   -0.0627   -0.0705   -0.0556   -0.0534   -0.0549   -0.0540   -0.0618   -0.0674   -0.0828   -0.0849   -0.1004  -0.0976   -0.1117   -0.0599   -0.0714;
   -0.3070   -0.3080   -0.2850   -0.3050   -0.2769   -0.2648   -0.1828   -0.2115   -0.2032   -0.2244   -0.2264   -0.2195   -0.2054   -0.2203   -0.2191  -0.2479   -0.2276   -0.2518   -0.2503;
   -0.3650   -0.3980   -0.4030   -0.3870   -0.3411   -0.3344   -0.3425   -0.3455   -0.3254   -0.3389   -0.3522   -0.3187   -0.3262   -0.3283   -0.3285  -0.3763   -0.3923   -0.3857   -0.3532;
   -0.5260   -0.5270   -0.5150   -0.5040   -0.4900   -0.5157   -0.4801   -0.4970   -0.4831   -0.4723   -0.4830   -0.4818   -0.4911   -0.5074   -0.4863  -0.5001   -0.5124   -0.5227   -0.5219];

% fi = -10 grad
mz1(:,:,2) = [0.1469    0.1272    0.1210    0.1075    0.0798    0.0756    0.0800    0.0827    0.0853    0.0864    0.0782    0.0811    0.0821    0.0847    0.0965   0.1240    0.1376    0.1439    0.1631;
    0.1087    0.0956    0.0947    0.0885    0.0581    0.0549    0.0505    0.0427    0.0378    0.0328    0.0353    0.0426    0.0481    0.0499    0.0524   0.0823    0.0891    0.0898    0.1002;
    0.0784    0.0743    0.0852    0.0619    0.0390    0.0344    0.0290    0.0249    0.0177    0.0041    0.0169    0.0227    0.0280    0.0311    0.0357   0.0585    0.0820    0.0707    0.0752;
    0.0570    0.0620    0.0440    0.0320    0.0170    0.0160    0.0120    0.0080    0.0100    0.0076    0.0070    0.0080    0.0100    0.0110    0.0120   0.0270    0.0390    0.0580    0.0520;
    0.0520    0.0540    0.0430    0.0390    0.0420    0.0410    0.0420    0.0430    0.0430    0.0430    0.0420    0.0430    0.0370    0.0380    0.0378   0.0430    0.0450    0.0570    0.0500;
    0.0520    0.0420    0.0500    0.0530    0.0540    0.0530    0.0540    0.0530    0.0520    0.0501    0.0520    0.0510    0.0510    0.0510    0.0510   0.0510    0.0490    0.0430    0.0520;
    0.0280    0.0350    0.0400    0.0400    0.0470    0.0480    0.0500    0.0500    0.0510    0.0553    0.0520    0.0530    0.0520    0.0520    0.0510   0.0430    0.0420    0.0380    0.0300;
    0.0430    0.0400    0.0530    0.0600    0.0630    0.0630    0.0670    0.0690    0.0720    0.0706    0.0710    0.0700    0.0700    0.0680    0.0630   0.0590    0.0530    0.0400    0.0420;
    0.0270    0.0250    0.0400    0.0500    0.0570    0.0560    0.0580    0.0600    0.0650    0.0674    0.0690    0.0660    0.0620    0.0550    0.0520   0.0460    0.0360    0.0200    0.0220;
    0.0100    0.0080    0.0230    0.0380    0.0470    0.0480    0.0480    0.0460    0.0480    0.0492    0.0460    0.0470    0.0440    0.0430    0.0430   0.0340    0.0190    0.0020    0.0050;
    0.0150   -0.0350   -0.0170    0.0030    0.0200    0.0400    0.0470    0.0490    0.0510    0.0528    0.0480    0.0480    0.0450    0.0400    0.0330   0.0160   -0.0050   -0.0240    0.0280;
    0.0160   -0.0270   -0.0340   -0.0240   -0.0060    0.0040    0.0160    0.0240    0.0310    0.0278    0.0280    0.0250    0.0120    0.0130    0.0030  -0.0210   -0.0260   -0.0200    0.0230;
    0.0680    0.0190   -0.0160   -0.0130   -0.0080   -0.0070   -0.0060   -0.0050   -0.0060   -0.0094   -0.0220   -0.0220   -0.0440   -0.0380   -0.0410  -0.0470   -0.0500   -0.0130    0.0330;
    0.0250   -0.0210   -0.0270   -0.0540   -0.0500   -0.0390   -0.0530   -0.0540   -0.0390   -0.0411   -0.0470   -0.0580   -0.0720   -0.0750   -0.0810  -0.0850   -0.0560   -0.0510   -0.0060;
   -0.0111         0   -0.0070   -0.0105    0.0073   -0.0085   -0.0371   -0.0519   -0.0379   -0.0129   -0.0221   -0.0455   -0.0542   -0.0594   -0.0515  -0.0693   -0.0658   -0.0588   -0.0699;
    0.0002    0.0043   -0.0936   -0.0425    0.0359    0.0134   -0.0110   -0.0169   -0.0113    0.0202   -0.0131   -0.0553   -0.0602   -0.0424   -0.0319  -0.1104   -0.1614   -0.0635   -0.0676;
   -0.0879   -0.0315   -0.0384   -0.1757   -0.0962   -0.1050   -0.0912   -0.0857   -0.0794   -0.0708   -0.0887   -0.1045   -0.1247   -0.1264   -0.1414  -0.2209   -0.0836   -0.0767   -0.1331;
   -0.3429   -0.3579   -0.3430   -0.3564   -0.3520   -0.3363   -0.2691   -0.3005   -0.2924   -0.3137   -0.3113   -0.3001   -0.2868   -0.3076   -0.3124  -0.3168   -0.3034   -0.3182   -0.3033;
   -0.4294   -0.4715   -0.4877   -0.4833   -0.4315   -0.4235   -0.4238   -0.4321   -0.4110   -0.4236   -0.4445   -0.4185   -0.4268   -0.4231   -0.4175  -0.4693   -0.4737   -0.4575   -0.4154;
   -0.6208   -0.6173   -0.6028   -0.5959   -0.5532   -0.5881   -0.5617   -0.5859   -0.5773   -0.5718   -0.5728   -0.5618   -0.5680   -0.5878   -0.5702  -0.5789   -0.5858   -0.6003   -0.6038];

% fi = 0 grad
mz1(:,:,3) = [0.0978    0.0719    0.0621    0.0430    0.0054   -0.0023   -0.0006    0.0062    0.0114    0.0127    0.0001    0.0023    0.0006    0.0033    0.0177   0.0550    0.0740    0.0840    0.1100;
    0.0560    0.0357    0.0264    0.0163   -0.0240   -0.0372   -0.0472   -0.0590   -0.0674   -0.0755   -0.0712   -0.0600   -0.0460   -0.0393   -0.0287   0.0110    0.0220    0.0310    0.0460;
    0.0342    0.0167    0.0194   -0.0089   -0.0410   -0.0510   -0.0608   -0.0700   -0.0813   -0.1025   -0.0793   -0.0673   -0.0576   -0.0500   -0.0424  -0.0100    0.0180    0.0140    0.0320;
   -0.0240   -0.0240   -0.0390   -0.0550   -0.0758   -0.0773   -0.0802   -0.0802   -0.0774   -0.0744   -0.0774   -0.0782   -0.0784   -0.0782   -0.0770  -0.0572   -0.0400   -0.0251   -0.0260;
   -0.0550   -0.0460   -0.0590   -0.0640   -0.0660   -0.0660   -0.0639   -0.0615   -0.0605   -0.0598   -0.0600   -0.0606   -0.0608   -0.0617   -0.0621  -0.0606   -0.0587   -0.0484   -0.0517;
   -0.0460   -0.0640   -0.0550   -0.0520   -0.0514   -0.0507   -0.0509   -0.0501   -0.0499   -0.0498   -0.0500   -0.0518   -0.0526   -0.0532   -0.0537  -0.0545   -0.0564   -0.0619   -0.0651;
   -0.0670   -0.0620   -0.0560   -0.0530   -0.0495   -0.0484   -0.0467   -0.0457   -0.0444   -0.0437   -0.0448   -0.0458   -0.0480   -0.0490   -0.0498  -0.0534   -0.0555   -0.0619   -0.0658;
   -0.0670   -0.0770   -0.0680   -0.0590   -0.0536   -0.0514   -0.0489   -0.0456   -0.0419   -0.0407   -0.0410   -0.0422   -0.0432   -0.0447   -0.0484  -0.0536   -0.0609   -0.0715   -0.0613;
   -0.0570   -0.0710   -0.0620   -0.0520   -0.0478   -0.0518   -0.0498   -0.0463   -0.0384   -0.0342   -0.0329   -0.0366   -0.0426   -0.0532   -0.0555  -0.0620   -0.0705   -0.0800   -0.0660;
   -0.0640   -0.0880   -0.0770   -0.0670   -0.0548   -0.0539   -0.0530   -0.0520   -0.0499   -0.0507   -0.0501   -0.0506   -0.0526   -0.0539   -0.0560  -0.0649   -0.0761   -0.0888   -0.0633;
   -0.0450   -0.1050   -0.0920   -0.0920   -0.0782   -0.0608   -0.0529   -0.0500   -0.0471   -0.0459   -0.0510   -0.0520   -0.0542   -0.0612   -0.0680  -0.0847   -0.0849   -0.0971   -0.0364;
   -0.0220   -0.0720   -0.0920   -0.0880   -0.0738   -0.0639   -0.0594   -0.0572   -0.0567   -0.0605   -0.0605   -0.0625   -0.0729   -0.0747   -0.0804  -0.0930   -0.0974   -0.0775   -0.0279;
    0.0450    0.0050   -0.0520   -0.0610   -0.0662   -0.0729   -0.0739   -0.0789   -0.0820   -0.0835   -0.0917   -0.0971   -0.1252   -0.1071   -0.1116  -0.1057   -0.0979   -0.0402    0.0022;
   -0.0010   -0.0520   -0.0600   -0.0920   -0.0927   -0.0861   -0.1056   -0.0966   -0.0862   -0.0923   -0.0975   -0.1080   -0.1168   -0.1209   -0.1243  -0.1234   -0.0897   -0.0820   -0.0294;
   -0.0090   -0.0130   -0.0170   -0.0350   -0.0780   -0.0713   -0.0774   -0.0890   -0.0913   -0.0826   -0.0898   -0.1112   -0.1201   -0.1277   -0.1222  -0.1220   -0.0852   -0.0648   -0.0624;
   -0.0510   -0.0180   -0.0650   -0.0530   -0.0477   -0.0520   -0.0583   -0.0663   -0.0830   -0.0738   -0.0851   -0.1053   -0.1050   -0.0988   -0.1000  -0.1076   -0.1152   -0.0589   -0.1047;
   -0.1830   -0.1480   -0.1730   -0.1720   -0.1512   -0.1428   -0.1118   -0.1094   -0.1266   -0.1414   -0.1436   -0.1437   -0.1521   -0.1459   -0.1530  -0.1709   -0.1741   -0.1475   -0.1841;
   -0.3830   -0.3980   -0.3820   -0.3870   -0.3869   -0.3637   -0.2706   -0.2967   -0.2944   -0.3216   -0.3252   -0.3199   -0.3123   -0.3385   -0.3487  -0.3486   -0.3445   -0.3593   -0.3444;
   -0.4830   -0.5180   -0.5280   -0.5060   -0.4850   -0.4785   -0.4804   -0.4869   -0.4605   -0.4678   -0.4883   -0.4620   -0.4744   -0.4792   -0.4821  -0.5022   -0.5242   -0.5145   -0.4788;
   -0.6330   -0.6300   -0.6160   -0.6160   -0.6067   -0.6366   -0.6053   -0.6281   -0.6217   -0.6184   -0.6163   -0.6022   -0.6073   -0.6281   -0.6115  -0.6209   -0.6210   -0.6351   -0.6381];

% fi = 10 grad
mz1(:,:,4) = [0.0200   -0.0036   -0.0107   -0.0334   -0.0778   -0.0944   -0.0926   -0.0855   -0.0815   -0.0835   -0.0955   -0.0930   -0.0943   -0.0881   -0.0743  -0.0265   -0.0040    0.0034    0.0268;
   -0.0153   -0.0385   -0.0525   -0.0743   -0.1233   -0.1376   -0.1466   -0.1551   -0.1663   -0.1719   -0.1683   -0.1568   -0.1437   -0.1371   -0.1295  -0.0798   -0.0589   -0.0445   -0.0241;
   -0.0549   -0.0792   -0.0932   -0.1226   -0.1521   -0.1609   -0.1688   -0.1774   -0.1880   -0.2153   -0.1839   -0.1738   -0.1648   -0.1594   -0.1525  -0.1225   -0.0931   -0.1276   -0.0548;
   -0.1120   -0.1240   -0.1520   -0.1680   -0.1830   -0.1880   -0.1910   -0.1920   -0.1890   -0.1888   -0.1900   -0.1930   -0.1930   -0.1910   -0.1830  -0.1680   -0.1520   -0.1250   -0.1120;
   -0.1170   -0.1270   -0.1520   -0.1590   -0.1600   -0.1600   -0.1590   -0.1570   -0.1620   -0.1610   -0.1620   -0.1630   -0.1670   -0.1620   -0.1620  -0.1620   -0.1530   -0.1260   -0.1170;
   -0.1050   -0.1330   -0.1440   -0.1550   -0.1550   -0.1550   -0.1550   -0.1550   -0.1580   -0.1606   -0.1610   -0.1620   -0.1570   -0.1570   -0.1560  -0.1560   -0.1450   -0.1330   -0.1180;
   -0.0970   -0.1120   -0.1220   -0.1350   -0.1420   -0.1420   -0.1510   -0.1530   -0.1570   -0.1548   -0.1550   -0.1520   -0.1530   -0.1450   -0.1460  -0.1400   -0.1260   -0.1170   -0.1040;
   -0.0970   -0.1180   -0.1330   -0.1510   -0.1520   -0.1500   -0.1550   -0.1550   -0.1520   -0.1452   -0.1480   -0.1550   -0.1550   -0.1570   -0.1520  -0.1510   -0.1340   -0.1180   -0.0960;
   -0.0620   -0.0830   -0.0970   -0.1060   -0.1340   -0.1420   -0.1380   -0.1340   -0.1300   -0.1264   -0.1260   -0.1260   -0.1510   -0.1610   -0.1660  -0.1370   -0.1260   -0.1150   -0.0940;
   -0.0750   -0.1030   -0.1130   -0.1080   -0.1370   -0.1440   -0.1530   -0.1540   -0.1540   -0.1530   -0.1550   -0.1500   -0.1500   -0.1600   -0.1640  -0.1330   -0.1380   -0.1290   -0.0990;
   -0.0880   -0.1680   -0.1650   -0.1720   -0.1710   -0.1550   -0.1500   -0.1470   -0.1440   -0.1440   -0.1450   -0.1460   -0.1530   -0.1570   -0.1560  -0.1590   -0.1520   -0.1550   -0.0750;
   -0.1050   -0.1611   -0.1862   -0.2095   -0.1951   -0.1760   -0.1514   -0.1444   -0.1427   -0.1411   -0.1450   -0.1513   -0.1565   -0.1627   -0.1705  -0.1836   -0.1611   -0.1363   -0.0815;
   -0.0438   -0.1079   -0.1281   -0.1485   -0.1405   -0.1272   -0.1301   -0.1367   -0.1555   -0.1450   -0.1543   -0.1595   -0.1527   -0.1644   -0.1682  -0.1791   -0.1553   -0.1362   -0.0744;
   -0.1448   -0.0931   -0.1319   -0.1793   -0.1518   -0.1264   -0.1053   -0.1575   -0.1807   -0.1411   -0.1635   -0.1655   -0.1635   -0.1815   -0.1872  -0.2206   -0.1644   -0.1320   -0.1935;
   -0.1530   -0.1330   -0.1280   -0.1470   -0.1077   -0.1030   -0.1111   -0.1154   -0.1161   -0.1008   -0.1060   -0.1254   -0.1273   -0.1228   -0.1052  -0.1442   -0.1253   -0.1286   -0.1498;
   -0.0760   -0.0630   -0.1520   -0.0570   -0.0075   -0.0460   -0.0865   -0.0614   -0.0976   -0.0679   -0.0922   -0.1253   -0.1221   -0.0972   -0.0797  -0.1301   -0.2255   -0.1358   -0.1481;
   -0.1710   -0.1200   -0.1350   -0.1400   -0.1588   -0.1634   -0.1455   -0.1444   -0.1512   -0.1556   -0.1653   -0.1719   -0.1866   -0.1859   -0.1985  -0.1808   -0.1758   -0.1621   -0.2116;
   -0.4001   -0.4044   -0.3789   -0.4050   -0.3419   -0.3364   -0.2610   -0.2714   -0.2201   -0.1983   -0.2363   -0.2655   -0.2695   -0.2844   -0.2833  -0.3734   -0.3473   -0.3728   -0.3685;
   -0.5082   -0.5338   -0.5333   -0.5253   -0.4877   -0.4848   -0.4902   -0.4970   -0.4677   -0.4721   -0.4929   -0.4669   -0.4776   -0.4787   -0.4779  -0.5155   -0.5235   -0.5240   -0.4984;
   -0.6368   -0.6326   -0.6174   -0.6217   -0.5909   -0.6214   -0.5906   -0.6146   -0.6099   -0.6083   -0.6080   -0.5958   -0.5979   -0.6109   -0.5865  -0.6173   -0.6130   -0.6282   -0.6324];

% fi = 25 grad
mz1(:,:,5) = [-0.0818   -0.1023   -0.1060   -0.1334   -0.1866   -0.2149   -0.2128   -0.2055   -0.2030   -0.2093   -0.2204   -0.2176   -0.2185   -0.2077   -0.1946  -0.1330   -0.1060   -0.1020   -0.0820;
   -0.1160   -0.1432   -0.1646   -0.2020   -0.2635   -0.2792   -0.2868   -0.2906   -0.3059   -0.3079   -0.3052   -0.2933   -0.2816   -0.2750   -0.2717  -0.2080   -0.1730   -0.1510   -0.1230;
   -0.1527   -0.1845   -0.2168   -0.2480   -0.2740   -0.2816   -0.2874   -0.2952   -0.3025   -0.3391   -0.2988   -0.2907   -0.2825   -0.2794   -0.2734  -0.2460   -0.2150   -0.1830   -0.1500;
   -0.1770   -0.2000   -0.2370   -0.2520   -0.2630   -0.2698   -0.2734   -0.2737   -0.2738   -0.2741   -0.2761   -0.2782   -0.2785   -0.2756   -0.2632  -0.2527   -0.2370   -0.2002   -0.1772;
   -0.1740   -0.1970   -0.2340   -0.2470   -0.2487   -0.2486   -0.2493   -0.2489   -0.2539   -0.2527   -0.2524   -0.2524   -0.2532   -0.2517   -0.2491  -0.2491   -0.2359   -0.1983   -0.1748;
   -0.1640   -0.1920   -0.2190   -0.2430   -0.2429   -0.2425   -0.2441   -0.2476   -0.2540   -0.2562   -0.2589   -0.2581   -0.2482   -0.2428   -0.2427  -0.2434   -0.2199   -0.1939   -0.1612;
   -0.1280   -0.1620   -0.1880   -0.2160   -0.2297   -0.2289   -0.2391   -0.2519   -0.2626   -0.2554   -0.2599   -0.2530   -0.2501   -0.2367   -0.2330  -0.2220   -0.1938   -0.1664   -0.1338;
   -0.1160   -0.1480   -0.1810   -0.2150   -0.2186   -0.2174   -0.2272   -0.2283   -0.2258   -0.2157   -0.2184   -0.2297   -0.2305   -0.2310   -0.2190  -0.2175   -0.1838   -0.1502   -0.1195;
   -0.0680   -0.0930   -0.1240   -0.1540   -0.2203   -0.2311   -0.2272   -0.2205   -0.2205   -0.2165   -0.2182   -0.2138   -0.2589   -0.2705   -0.2751  -0.2111   -0.1811   -0.1482   -0.1227;
   -0.0750   -0.1090   -0.1320   -0.1330   -0.1882   -0.2123   -0.2264   -0.2304   -0.2337   -0.2325   -0.2322   -0.2269   -0.2243   -0.2382   -0.2465  -0.1828   -0.1848   -0.1595   -0.1250;
   -0.0970   -0.1860   -0.1860   -0.1980   -0.1989   -0.1828   -0.1798   -0.1762   -0.1751   -0.1740   -0.1732   -0.1782   -0.1855   -0.1875   -0.1852  -0.1824   -0.1732   -0.1478   -0.0814;
   -0.1040   -0.1600   -0.1850   -0.2080   -0.1936   -0.1746   -0.1503   -0.1433   -0.1416   -0.1401   -0.1440   -0.1502   -0.1555   -0.1616   -0.1694  -0.1825   -0.1603   -0.1356   -0.0808;
   -0.0250   -0.0840   -0.1120   -0.1300   -0.1248   -0.1157   -0.1182   -0.1245   -0.1400   -0.1320   -0.1411   -0.1463   -0.1532   -0.1523   -0.1562  -0.1636   -0.1432   -0.1159   -0.0582;
   -0.0570   -0.0680   -0.0880   -0.1260   -0.1157   -0.1018   -0.1055   -0.1203   -0.1230   -0.1113   -0.1232   -0.1304   -0.1350   -0.1445   -0.1488  -0.1618   -0.1188   -0.1003   -0.0933;
   -0.1080   -0.0930   -0.0930   -0.0870   -0.0745   -0.0894   -0.1198   -0.1388   -0.1366   -0.1234   -0.1254   -0.1416   -0.1463   -0.1508   -0.1421 -0.1550   -0.1585   -0.1588   -0.1771;
   -0.1250   -0.1150   -0.2070   -0.1030   -0.0588   -0.0831   -0.1095   -0.0791   -0.1189   -0.0929   -0.1186   -0.1533   -0.1523   -0.1304   -0.1158  -0.1580   -0.2612   -0.1702   -0.1812;
   -0.1430   -0.0820   -0.0850   -0.0910   -0.1251   -0.1492   -0.1507   -0.1570   -0.1589   -0.1584   -0.1689   -0.1773   -0.1947   -0.1982   -0.2150  -0.1808   -0.1737   -0.1719   -0.2333;
   -0.4220   -0.4380   -0.4250   -0.4330   -0.3390   -0.3231   -0.2373   -0.2547   -0.2277   -0.2303   -0.3505   -0.1931   -0.1880   -0.2371   -0.2701  -0.3635   -0.3563   -0.3697   -0.3534;
   -0.4500   -0.5000   -0.5240   -0.5140   -0.4633   -0.4648   -0.4746   -0.4862   -0.4621   -0.4716   -0.4474   -0.3916   -0.4082   -0.4299   -0.4593  -0.5113   -0.5202   -0.4961   -0.4460;
   -0.5600   -0.5920   -0.5130   -0.5930   -0.5674   -0.6030   -0.5774   -0.6021   -0.5938   -0.5886   -0.5839   -0.5673   -0.5700   -0.5885   -0.5696  -0.5961   -0.6158   -0.5951   -0.5634];


% Коэффициент момента тангажа, носки выпущены.
% mz_nos = f(alpha2,beta)
mz_nos1 = [0.0922    0.0559    0.0525   -0.0338   -0.0518   -0.0605   -0.0574   -0.0554   -0.0550   -0.0503   -0.0521   -0.0483   -0.0459   -0.0404   -0.0373  -0.0193    0.0670    0.0704    0.1067;
    0.0372    0.0062   -0.0067   -0.0217   -0.0702   -0.0860   -0.1001   -0.1000   -0.1002   -0.1012   -0.0974   -0.0939   -0.0839   -0.0837   -0.0759  -0.0274   -0.0124    0.0005    0.0315;
    0.0251    0.0006    0.0014   -0.0229   -0.0536   -0.0634   -0.0654   -0.0656   -0.0652   -0.0647   -0.0653   -0.0659   -0.0654   -0.0631   -0.0570  -0.0263   -0.0020   -0.0028    0.0217;
   -0.0006   -0.0193   -0.0234   -0.0321   -0.0386   -0.0389   -0.0385   -0.0386   -0.0388   -0.0387   -0.0389   -0.0387   -0.0388   -0.0392   -0.0386  -0.0321   -0.0234   -0.0193   -0.0006;
   -0.0273   -0.0246   -0.0230   -0.0231   -0.0259   -0.0255   -0.0286   -0.0271   -0.0271   -0.0267   -0.0266   -0.0272   -0.0280   -0.0267   -0.0270  -0.0242   -0.0241   -0.0257   -0.0284;
   -0.0319   -0.0272   -0.0204   -0.0170   -0.0152   -0.0148   -0.0145   -0.0138   -0.0127   -0.0128   -0.0133   -0.0141   -0.0149   -0.0157   -0.0164  -0.0182   -0.0216   -0.0287   -0.0331;
   -0.0446   -0.0368   -0.0266   -0.0166   -0.0127   -0.0113   -0.0092   -0.0057   -0.0033   -0.0016   -0.0017   -0.0025   -0.0038   -0.0049   -0.0085  -0.0124   -0.0224   -0.0326   -0.0404;
   -0.0682   -0.0587   -0.0425   -0.0197         0    0.0026    0.0078    0.0158    0.0243    0.0323    0.0328    0.0290    0.0189    0.0120    0.0061  -0.0136   -0.0364   -0.0526   -0.0621;
   -0.0947   -0.0851   -0.0642   -0.0536   -0.0308   -0.0293   -0.0275   -0.0234   -0.0188   -0.0161   -0.0141   -0.0136   -0.0154   -0.0180   -0.0273  -0.0501   -0.0607   -0.0816   -0.0912;
   -0.1090   -0.1235   -0.0938   -0.0777   -0.0674   -0.0648   -0.1607   -0.0558   -0.0526   -0.0455   -0.0471   -0.0479   -0.0530   -0.0563   -0.0610  -0.0713   -0.0874   -0.1171   -0.1026;
   -0.0135   -0.0857   -0.0907   -0.1013   -0.0875   -0.0983   -0.0951   -0.0913   -0.0902   -0.0871   -0.0865   -0.0896   -0.0962   -0.0997   -0.1060  -0.1198   -0.1092   -0.1042   -0.0320;
    0.0202   -0.0510   -0.0891   -0.1086   -0.1018   -0.1014   -0.1105   -0.1117   -0.1127   -0.1151   -0.1167   -0.1230   -0.1301   -0.1387   -0.1402  -0.1470   -0.1275   -0.0894   -0.0142;
   -0.0116   -0.0639   -0.0971   -0.1156   -0.1170   -0.1142   -0.1182   -0.1160   -0.1178   -0.1206   -0.1280   -0.1347   -0.1436   -0.1512   -0.1516  -0.1502   -0.1317   -0.0985   -0.0462;
   -0.0023   -0.0164   -0.0417   -0.0987   -0.0985   -0.0975   -0.1278   -0.1042   -0.1156   -0.0979   -0.1122   -0.1225   -0.1444   -0.1340   -0.1461  -0.1463   -0.0893   -0.0640   -0.0499];

% Какая-то поправка
% dmz = f(alpha1)
dmz1 = [0.0190    0.0190    0.0190    0.0190    0.0190    0.0190    0.0200    0.0400    0.0400    0.0500    0.0600    0.0600    0.0600    0.0600    0.0600   0.0600    0.0600    0.0600    0.0600    0.0600]';


% Производная коэффициента момента тангажа по угловой скорости тангажа.
% mzwz = f(alpha1)
mzwz1 = [-6.8400   -6.8400   -6.8400   -3.4200   -5.4800   -5.4500   -6.0200   -6.7000   -5.6900   -6.0000   -6.2000   -6.4000  -6.6000  -6.0000  -5.5000   -5.0000   -4.5000    -3.5000   -5.6000   -4.0400]';
% Приращение mzwz при выпуске носков.
% dmzwz_nos = f(alpha2)
dmzwz_nos1 = [-0.3670   -0.3670   -0.3670    2.8800    0.2500    0.2700   -0.2100    0.3600   -1.2600   -2.5100   -1.6600   -1.7200   -1.2000   -0.6000]';

dmz_sb1 = [-0.00340 -0.00340 -0.00340 -0.00340 -0.00340 0.02890 0.02150 0.01220 0.02410 0.02630 -0.01630 -0.04280 -0.07040 -0.08440 -0.07890 -0.06030 -0.04500 -0.05780 -0.01070 -0.01070];

eta_fi1 = [1.00 1.00 1.00 1.00 0.95];

dmz_ds1 = [0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   0.0000   0.0000   0.0000;
           0.0000   0.0000   0.0000   0.0000   -0.0113   -0.0132   0.0000;
           0.0000   0.0100   0.0100   0.0200   0.0041   0.0297   0.0254;
           0.0480   0.0570   0.0640   0.0700   0.0706   0.0506   0.0407;
           0.0390   0.0540   0.1050   0.0750   0.0792   0.0747   0.0822;
           0.0000   0.0250   0.0750   0.0400   0.0416   0.0500   0.0583;
           0.0840   0.0800   0.1060   0.1000   0.0886   0.0776   0.0785;
           0.0990   0.0750   0.1320   0.0200   0.0288   0.0361   0.0468;
           0.0200   0.0100   0.0500   0.0000   0.0218   -0.0251   -0.0210;
           0.0000   0.0100   0.0400   0.0000   -0.0345   -0.0378   -0.0378];

interpmz = csaps({alpha1, beta1, fi1}, mz1, 1.0 - 1.0e-5);
interpmz_nos = csaps({alpha2, beta1}, mz_nos1, 1.0 - 1.0e-5);
interpdmz = csaps(alpha1, dmz1, 1.0 - 1.0e-5);
interpmzwz1 = pchip(alpha1, mzwz1);
interpmzwz2 = pchip(alpha2, dmzwz_nos1);
interpdmz_sb = pchip(alpha1, dmz_sb1);
interpeta_fi = pchip(fi1, eta_fi1);
interpdmz_ds = csaps({alpha1, fi2}, dmz_ds1, 1.0 - 1.0e-5);
end

ntraj = size(alpha, 2);
mz = zeros(1, ntraj);
mz0 = zeros(1, ntraj);
mz_nos = zeros(1, ntraj);
dmz = zeros(1, ntraj);
mzwz = zeros(1, ntraj);
dmz_sb = zeros(1, ntraj);
eta_fi = zeros(1, ntraj);
dmz_ds = zeros(1, ntraj);
for i = 1 : ntraj
    mz(i) = fnval(interpmz, {(alpha(i)), (beta(i)), (fi(i))});
    mz0(i) = fnval(interpmz, {(alpha(i)), (beta(i)), (0)});
    mz_nos(i) = fnval(interpmz_nos, {(alpha(i)), (beta(i))});
    dmz(i) = fnval(interpdmz, alpha(i));
    mzwz(i) = fnval(interpmzwz1, alpha(i)) + fnval(interpmzwz2, alpha(i)).*(dnos / deg2rad(25));
    dmz_sb(i) = fnval(interpdmz_sb, alpha(i));
    eta_fi(i) = fnval(interpeta_fi, fi(i));
    dmz_ds(i) = fnval(interpdmz_ds, {(alpha(i)), (fi(i))});
end

% Сборка
dmz_nos = mz_nos - mz0;

mz = mz.*eta_fi + dmz_nos.*(dnos / deg2rad(25)) + dmz + mzwz.*((Wz.*ba)./(2 * V)) + dmz_sb.*(sb / deg2rad(60)) + dmz_ds;
