function Cx=GetCx(alpha,beta,fi,dnos,Wz,V,ba,sb)
% --------------------------------------------------------------
%       Вычисление коэффициента продольной силы Cx.
% --------------------------------------------------------------
% Cx = GetCx(alpha,beta,fi,dnos,Wz,V) вычисляет коэффициент продольной силы в 
%    зависимости от углов атаки, скольжения, отклонения стабилизатора, а 
%    также от угловой скорости тангажа.
% alpha - угол атаки, рад.
% beta - угол скольжения, рад.
% fi - отклонение стабилизатора, рад.
% dnos - отклонение носков крыла, рад.
% V - воздушная скорость, м/с
% Wz - угловая скорость тангажа, рад/с
%
% Зависимости заданы в виде таблиц.
% Таблицы интерполяции хранятся в оперативной памяти для ускорения
% численного расчёта.
% --------------------------------------------------------------
persistent init_mdl
persistent alpha1 alpha2 beta1 fi1 Cx1 Cx_nos1 Cxwz1 dCxwz_nos1 dCx_sb1
persistent interpCx interpCx_nos interpCxwz1 interpCxwz2 interpdCx_sb

% Инициализация
if isempty(init_mdl)

init_mdl = 1;   
    
alpha1 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45 50 55 60 70 80 90]);
alpha2 = deg2rad([-20 -15 -10 -5 0 5 10 15 20 25 30 35 40 45]);
beta1 = deg2rad([-30 -25 -20 -15 -10 -8 -6 -4 -2 0 2 4 6 8 10 15 20 25 30]);
fi1 = deg2rad([-25 -10 0 10 25]);

% Зависимость коэффициента продольной силы от угла атаки, угла скольжения
% и отклонения стабилизатора.
% Cx = f(alpha, beta, fi)

% fi = -25 grad
Cx1(:,:,1) = [-0.1837   -0.1853   -0.1904   -0.1899   -0.1949   -0.1914   -0.1872   -0.1860   -0.1860   -0.1868   -0.1899   -0.1902   -0.1900   -0.1896   -0.1883   -0.1833   -0.1838   -0.1787   -0.1771;
   -0.1714   -0.1765   -0.1792   -0.1827   -0.1816   -0.1834   -0.1852   -0.1853   -0.1877   -0.1875   -0.1898   -0.1876   -0.1868   -0.1848   -0.1841  -0.1852   -0.1817   -0.1790   -0.1739;
   -0.1531   -0.1627   -0.1692   -0.1718   -0.1695   -0.1693   -0.1707   -0.1735   -0.1772   -0.1787   -0.1769   -0.1729   -0.1711   -0.1706   -0.1698  -0.1721   -0.1695   -0.1630   -0.1534;
   -0.1151   -0.1232   -0.1276   -0.1317   -0.1390   -0.1415   -0.1420   -0.1425   -0.1437   -0.1432   -0.1425   -0.1422   -0.1410   -0.1397   -0.1372  -0.1299   -0.1258   -0.1214   -0.1133;
   -0.0907   -0.0985   -0.1043   -0.1093   -0.1120   -0.1115   -0.1122   -0.1124   -0.1130   -0.1132   -0.1129   -0.1119   -0.1110   -0.1102   -0.1092  -0.1065   -0.1015   -0.0957   -0.0879;
   -0.0514   -0.0567   -0.0603   -0.0640   -0.0653   -0.0661   -0.0668   -0.0675   -0.0690   -0.0693   -0.0686   -0.0680   -0.0664   -0.0650   -0.0649  -0.0631   -0.0594   -0.0558   -0.0505;
   -0.0079   -0.0108   -0.0099   -0.0101   -0.0074   -0.0070   -0.0078   -0.0090   -0.0116   -0.0120   -0.0123   -0.0106   -0.0088   -0.0083   -0.0080  -0.0107   -0.0105   -0.0114   -0.0085;
    0.0354    0.0358    0.0388    0.0402    0.0477    0.0503    0.0535    0.0553    0.0538    0.0537    0.0533    0.0536    0.0527    0.0509    0.0485  0.0410    0.0396    0.0366    0.0362;
    0.0740    0.0756    0.0746    0.0745    0.0867    0.0888    0.0924    0.0941    0.0948    0.0951    0.0975    0.0939    0.0913    0.0867    0.0824  0.0702    0.0703    0.0713    0.0697;
    0.1092    0.1124    0.1102    0.1067    0.1101    0.1121    0.1126    0.1129    0.1123    0.1111    0.1122    0.1125    0.1136    0.1115    0.1075  0.1041    0.1076    0.1098    0.1066;
    0.0915    0.1010    0.0975    0.1079    0.1188    0.1333    0.1399    0.1422    0.1443    0.1435    0.1431    0.1407    0.1379    0.1359    0.1323  0.1214    0.1110    0.1145    0.1050;
    0.1079    0.1137    0.1198    0.1278    0.1402    0.1425    0.1478    0.1570    0.1623    0.1663    0.1667    0.1664    0.1637    0.1560    0.1460  0.1336    0.1256    0.1195    0.1137;
    0.1306    0.1437    0.1350    0.1441    0.1574    0.1585    0.1601    0.1682    0.1726    0.1739    0.1711    0.1699    0.1655    0.1611    0.1567  0.1434    0.1343    0.1430    0.1299;
    0.1535    0.1603    0.1605    0.1604    0.1637    0.1671    0.1664    0.1639    0.1674    0.1659    0.1649    0.1650    0.1625    0.1597    0.1573  0.1540    0.1541    0.1539    0.1471;
    0.1471    0.1584    0.1646    0.1671    0.1712    0.1712    0.1676    0.1644    0.1656    0.1693    0.1714    0.1728    0.1749    0.1725    0.1730  0.1537    0.1457    0.1435    0.1362;
    0.1554    0.1615    0.1568    0.1661    0.1778    0.1769    0.1765    0.1749    0.1762    0.1804    0.1743    0.1666    0.1677    0.1724    0.1761  0.1722    0.1347    0.1448    0.1442;
    0.1501    0.1599    0.1647    0.1525    0.1664    0.1662    0.1704    0.1710    0.1719    0.1718    0.1728    0.1730    0.1734    0.1721    0.1688  0.1471    0.1462    0.1486    0.1460;
    0.1501    0.1536    0.1569    0.1420    0.1573    0.1595    0.1788    0.1715    0.1738    0.1695    0.1710    0.1712    0.1730    0.1720    0.1686  0.1474    0.1567    0.1557    0.1545;
    0.1685    0.1615    0.1559    0.1520    0.1521    0.1521    0.1535    0.1585    0.1566    0.1598    0.1573    0.1563    0.1586    0.1558    0.1572  0.1410    0.1410    0.1467    0.1538;
    0.1712    0.1651    0.1608    0.1648    0.1676    0.1660    0.1686    0.1667    0.1669    0.1660    0.1672    0.1662    0.1664    0.1711    0.1677  0.1531    0.1493    0.1549    0.1624];

% fi = -10 grad
Cx1(:,:,2) = [-0.1362   -0.1351   -0.1419   -0.1386   -0.1374   -0.1330   -0.1268   -0.1249   -0.1222   -0.1223   -0.1246   -0.1247   -0.1252   -0.1257   -0.1282  -0.1294   -0.1327   -0.1259   -0.1270;
   -0.1216   -0.1245   -0.1235   -0.1208   -0.1176   -0.1176   -0.1170   -0.1177   -0.1184   -0.1188   -0.1185   -0.1187   -0.1182   -0.1178   -0.1184  -0.1216   -0.1243   -0.1253   -0.1224;
   -0.1018   -0.1066   -0.1068   -0.1071   -0.1061   -0.1068   -0.1072   -0.1083   -0.1094   -0.1147   -0.1095   -0.1084   -0.1077   -0.1063   -0.1069  -0.1079   -0.1076   -0.1074   -0.1026;
   -0.0655   -0.0706   -0.0746   -0.0771   -0.0836   -0.0864   -0.0876   -0.0887   -0.0889   -0.0893   -0.0885   -0.0875   -0.0859   -0.0842   -0.0812  -0.0747   -0.0722   -0.0682   -0.0631;
   -0.0483   -0.0509   -0.0532   -0.0544   -0.0578   -0.0589   -0.0597   -0.0606   -0.0613   -0.0617   -0.0611   -0.0603   -0.0595   -0.0577   -0.0561  -0.0527   -0.0515   -0.0492   -0.0466;
   -0.0118   -0.0106   -0.0096   -0.0102   -0.0142   -0.0148   -0.0155   -0.0161   -0.0177   -0.0172   -0.0178   -0.0167   -0.0156   -0.0141   -0.0133  -0.0093   -0.0087   -0.0106   -0.0127;
    0.0268    0.0328    0.0367    0.0399    0.0412    0.0417    0.0408    0.0413    0.0406    0.0399    0.0399    0.0409    0.0415    0.0414    0.0412  0.0399    0.0367    0.0328    0.0268;
    0.0735    0.0800    0.0887    0.0934    0.0983    0.1006    0.1024    0.1034    0.1033    0.1027    0.1031    0.1027    0.1018    0.1008    0.0983  0.0934    0.0887    0.0800    0.0735;
    0.1222    0.1275    0.1258    0.1249    0.1326    0.1347    0.1350    0.1349    0.1325    0.1322    0.1332    0.1338    0.1343    0.1310    0.1298  0.1221    0.1230    0.1247    0.1194;
    0.1374    0.1474    0.1466    0.1454    0.1465    0.1485    0.1485    0.1453    0.1429    0.1407    0.1418    0.1443    0.1457    0.1442    0.1439  0.1428    0.1440    0.1448    0.1348;
    0.1056    0.1261    0.1297    0.1437    0.1500    0.1619    0.1655    0.1660    0.1663    0.1651    0.1640    0.1643    0.1624    0.1615    0.1593  0.1530    0.1390    0.1354    0.1149;
    0.1075    0.1154    0.1299    0.1377    0.1523    0.1581    0.1722    0.1789    0.1801    0.1795    0.1793    0.1804    0.1782    0.1749    0.1675  0.1529    0.1451    0.1306    0.1227;
    0.1335    0.1412    0.1365    0.1456    0.1597    0.1622    0.1725    0.1762    0.1798    0.1798    0.1810    0.1771    0.1710    0.1702    0.1659  0.1518    0.1427    0.1474    0.1397;
    0.1521    0.1486    0.1517    0.1520    0.1608    0.1613    0.1597    0.1671    0.1667    0.1671    0.1664    0.1653    0.1629    0.1597    0.1569  0.1481    0.1478    0.1447    0.1482;
    0.1346    0.1410    0.1422    0.1486    0.1561    0.1570    0.1538    0.1511    0.1515    0.1544    0.1549    0.1547    0.1560    0.1538    0.1544  0.1469    0.1405    0.1393    0.1329;
    0.1375    0.1367    0.1251    0.1336    0.1467    0.1472    0.1475    0.1465    0.1462    0.1488    0.1433    0.1361    0.1370    0.1405    0.1431  0.1300    0.1215    0.1331    0.1339;
    0.1316    0.1360    0.1355    0.1154    0.1285    0.1289    0.1336    0.1351    0.1372    0.1383    0.1356    0.1320    0.1387    0.1323    0.1310  0.1179    0.1380    0.1385    0.1341;
    0.1171    0.1174    0.1185    0.1108    0.1161    0.1187    0.1376    0.1312    0.1353    0.1328    0.1301    0.1263    0.1270    0.1281    0.1268  0.1215    0.1292    0.1281    0.1278;
    0.1201    0.1161    0.1136    0.1124    0.1158    0.1148    0.1149    0.1194    0.1177    0.1211    0.1195    0.1195    0.1225    0.1204    0.1177  0.1143    0.1155    0.1180    0.1220;
    0.1287    0.1241    0.1214    0.1221    0.1265    0.1256    0.1257    0.1236    0.1248    0.1247    0.1262    0.1256    0.1256    0.1297    0.1257  0.1213    0.1206    0.1233    0.1279];

% fi = 0 grad
Cx1(:,:,3) = [-0.1072   -0.1061   -0.1129   -0.1096   -0.1084   -0.1040   -0.0978   -0.0959   -0.0932   -0.0933   -0.0956   -0.0957   -0.0962   -0.0967   -0.0992  -0.1004   -0.1037   -0.0969   -0.0980;
   -0.1006   -0.1035   -0.1025   -0.0998   -0.0966   -0.0966   -0.0960   -0.0967   -0.0974   -0.0978   -0.0975   -0.0977   -0.0972   -0.0968   -0.0974  -0.1006   -0.1033   -0.1043   -0.1014;
   -0.0853   -0.0901   -0.0903   -0.0906   -0.0896   -0.0903   -0.0907   -0.0918   -0.0929   -0.0982   -0.0930   -0.0919   -0.0912   -0.0898   -0.0904  -0.0914   -0.0911   -0.0909   -0.0861;
   -0.0546   -0.0597   -0.0637   -0.0662   -0.0727   -0.0755   -0.0767   -0.0778   -0.0780   -0.0784   -0.0776   -0.0766   -0.0750   -0.0733   -0.0703  -0.0638   -0.0613   -0.0573   -0.0522;
   -0.0355   -0.0381   -0.0404   -0.0416   -0.0450   -0.0461   -0.0469   -0.0478   -0.0485   -0.0489   -0.0483   -0.0475   -0.0467   -0.0449   -0.0433  -0.0399   -0.0387   -0.0364   -0.0338;
   -0.0012         0    0.0010    0.0004   -0.0036   -0.0042   -0.0049   -0.0055   -0.0071   -0.0066   -0.0072   -0.0061   -0.0050   -0.0035   -0.0027  0.0013    0.0019         0   -0.0021;
    0.0359    0.0491    0.0458    0.0490    0.0503    0.0508    0.0499    0.0509    0.0497    0.0490    0.0490    0.0500    0.0506    0.0505    0.0503  0.0490    0.0458    0.0419    0.0359;
    0.0780    0.0845    0.0932    0.0979    0.1028    0.1051    0.1069    0.1079    0.1078    0.1072    0.1076    0.1072    0.1063    0.1053    0.1028  0.0979    0.0932    0.0845    0.0780;
    0.1183    0.1236    0.1219    0.1210    0.1287    0.1308    0.1311    0.1310    0.1286    0.1283    0.1293    0.1299    0.1304    0.1271    0.1259  0.1182    0.1191    0.1208    0.1155;
    0.1267    0.1367    0.1359    0.1347    0.1358    0.1378    0.1378    0.1346    0.1322    0.1300    0.1311    0.1336    0.1350    0.1335    0.1332  0.1321    0.1333    0.1341    0.1241;
    0.0941    0.1146    0.1182    0.1322    0.1385    0.1504    0.1540    0.1545    0.1548    0.1536    0.1525    0.1528    0.1509    0.1500    0.1478  0.1415    0.1275    0.1239    0.1034;
    0.0885    0.0964    0.1109    0.1187    0.1333    0.1391    0.1532    0.1599    0.1611    0.1605    0.1603    0.1614    0.1592    0.1559    0.1485  0.1339    0.1261    0.1116    0.1037;
    0.1089    0.1166    0.1119    0.1210    0.1351    0.1376    0.1479    0.1516    0.1552    0.1552    0.1564    0.1525    0.1464    0.1456    0.1413  0.1272    0.1181    0.1228    0.1151;
    0.1232    0.1197    0.1228    0.1231    0.1319    0.1324    0.1308    0.1332    0.1378    0.1382    0.1375    0.1364    0.1340    0.1308    0.1280  0.1192    0.1189    0.1158    0.1193;
    0.1135    0.1185    0.1184    0.1171    0.1243    0.1279    0.1279    0.1258    0.1257    0.1281    0.1258    0.1228    0.1221    0.1186    0.1180  0.1108    0.1121    0.1122    0.1072;
    0.1137    0.1195    0.1146    0.1161    0.1209    0.1211    0.1211    0.1195    0.1183    0.1200    0.1185    0.1153    0.1160    0.1152    0.1135  0.1087    0.1072    0.1121    0.1063;
    0.1037    0.1090    0.1094    0.1049    0.1109    0.1123    0.1181    0.1184    0.1170    0.1147    0.1141    0.1126    0.1129    0.1129    0.1109  0.1049    0.1094    0.1090    0.1037;
    0.0857    0.0858    0.0857    0.0796    0.0851    0.0919    0.1150    0.1087    0.1089    0.1025    0.1022    0.1007    0.1012    0.0994    0.0952  0.0897    0.0958    0.0959    0.0958;
    0.0842    0.0807    0.0787    0.0778    0.0791    0.0793    0.0805    0.0846    0.0808    0.0821    0.0802    0.0799    0.0826    0.0800    0.0709  0.0756    0.0765    0.0785    0.0820;
    0.0847    0.0813    0.0798    0.0824    0.0843    0.0843    0.0853    0.0841    0.0858    0.0864    0.0857    0.0828    0.0817    0.0857    0.0816  0.0797    0.0771    0.0786    0.0820];

% fi = 10 grad
Cx1(:,:,4) = [-0.1023   -0.1012   -0.1080   -0.1047   -0.1035   -0.0991   -0.0929   -0.0910   -0.0884   -0.0884   -0.0907   -0.0908   -0.0913   -0.0918   -0.0943  -0.0955   -0.0988   -0.0920   -0.0931;
   -0.1038   -0.1067   -0.1057   -0.1030   -0.0998   -0.0998   -0.0992   -0.0999   -0.1006   -0.1010   -0.1007   -0.1009   -0.1004   -0.1000   -0.1006  -0.1038   -0.1065   -0.1075   -0.1046;
   -0.0963   -0.1011   -0.1013   -0.1016   -0.1006   -0.1013   -0.1017   -0.1028   -0.1039   -0.1092   -0.1040   -0.1029   -0.1022   -0.1008   -0.1014  -0.1024   -0.1021   -0.1019   -0.0971;
   -0.0664   -0.0715   -0.0755   -0.0780   -0.0845   -0.0873   -0.0885   -0.0896   -0.0898   -0.0902   -0.0894   -0.0884   -0.0868   -0.0851   -0.0821  -0.0756   -0.0731   -0.0691   -0.0640;
   -0.0472   -0.0498   -0.0521   -0.0533   -0.0567   -0.0578   -0.0586   -0.0595   -0.0602   -0.0606   -0.0600   -0.0592   -0.0584   -0.0566   -0.0550  -0.0516   -0.0504   -0.0481   -0.0455;
   -0.0146   -0.0134   -0.0124   -0.0130   -0.0170   -0.0176   -0.0183   -0.0189   -0.0205   -0.0200   -0.0206   -0.0195   -0.0184   -0.0169   -0.0161  -0.0121   -0.0115   -0.0134   -0.0155;
    0.0182    0.0242    0.0281    0.0313    0.0326    0.0331    0.0322    0.0327    0.0320    0.0313    0.0313    0.0323    0.0329    0.0328    0.0326  0.0313    0.0281    0.0242    0.0182;
    0.0537    0.0602    0.0689    0.0736    0.0785    0.0808    0.0826    0.0836    0.0835    0.0829    0.0833    0.0829    0.0820    0.0810    0.0785  0.0736    0.0689    0.0602    0.0537;
    0.0871    0.0924    0.0907    0.0898    0.0975    0.0996    0.0999    0.0998    0.0974    0.0971    0.0981    0.0987    0.0992    0.0959    0.0947  0.0870    0.0879    0.0896    0.0843;
    0.0916    0.1016    0.1008    0.0996    0.1007    0.1027    0.1027    0.0995    0.0971    0.0949    0.0960    0.0985    0.0999    0.0984    0.0981  0.0970    0.0982    0.0990    0.0890;
    0.0509    0.0714    0.0750    0.0890    0.0953    0.1072    0.1108    0.1113    0.1116    0.1104    0.1093    0.1096    0.1077    0.1068    0.1046  0.0983    0.0843    0.0807    0.0602;
    0.0481    0.0560    0.0705    0.0783    0.0929    0.0987    0.1128    0.1195    0.1207    0.1201    0.1199    0.1210    0.1188    0.1155    0.1081  0.0935    0.0857    0.0712    0.0633;
    0.0664    0.0741    0.0694    0.0785    0.0926    0.0951    0.1054    0.1091    0.1127    0.1127    0.1139    0.1100    0.1039    0.1031    0.0988  0.0847    0.0756    0.0803    0.0726;
    0.0846    0.0811    0.0842    0.0845    0.0933    0.0938    0.0922    0.0946    0.0992    0.0996    0.0989    0.0978    0.0954    0.0922    0.0894  0.0806    0.0803    0.0772    0.0807;
    0.0908    0.0985    0.1011    0.0999    0.1063    0.1061    0.1018    0.0996    0.1021    0.1071    0.1071    0.1064    0.1070    0.1036    0.1032  0.0968    0.0980    0.0954    0.0877;
    0.0842    0.0869    0.0790    0.0882    0.1025    0.1010    0.0993    0.0980    0.0991    0.1030    0.0972    0.0897    0.0914    0.0969    0.1015  0.0872    0.0780    0.0859    0.0832;
    0.0749    0.0823    0.0849    0.0794    0.0831    0.0841    0.0896    0.0908    0.0915    0.0914    0.0908    0.0893    0.0895    0.0889    0.0868  0.0831    0.0886    0.0860    0.0786;
    0.0504    0.0500    0.0504    0.0467    0.0813    0.0811    0.0972    0.0950    0.1075    0.1190    0.1101    0.1001    0.0967    0.0958    0.0931  0.0585    0.0622    0.0618    0.0622;
    0.0421    0.0380    0.0355    0.0397    0.0420    0.0417    0.0424    0.0478    0.0473    0.0519    0.0484    0.0465    0.0489    0.0472    0.0450  0.0427    0.0385    0.0410    0.0451;
    0.0433    0.0404    0.0395    0.0467    0.0495    0.0492    0.0499    0.0484    0.0500    0.0504    0.0495    0.0463    0.0457    0.0510    0.0482  0.0454    0.0382    0.0391    0.0420];

% fi = 25 grad
Cx1(:,:,5) = [-0.1068   -0.1102   -0.1160   -0.1176   -0.1291   -0.1289   -0.1244   -0.1158   -0.1137   -0.1141   -0.1164   -0.1192   -0.1200   -0.1240   -0.1243  -0.1128   -0.1112   -0.1054   -0.1020;
   -0.1122   -0.1180   -0.1227   -0.1292   -0.1365   -0.1397   -0.1406   -0.1416   -0.1442   -0.1450   -0.1448   -0.1428   -0.1408   -0.1440   -0.1397  -0.1324   -0.1259   -0.1212   -0.1154;
   -0.1102   -0.1212   -0.1319   -0.1359   -0.1403   -0.1427   -0.1454   -0.1480   -0.1520   -0.1633   -0.1518   -0.1482   -0.1457   -0.1438   -0.1419  -0.1375   -0.1335   -0.1228   -0.1118;
   -0.0911   -0.1027   -0.1093   -0.1144   -0.1244   -0.1304   -0.1316   -0.1320   -0.1333   -0.1337   -0.1340   -0.1322   -0.1309   -0.1280   -0.1243  -0.1143   -0.1092   -0.1026   -0.0910;
   -0.0811   -0.0889   -0.0955   -0.0996   -0.1015   -0.1037   -0.1056   -0.1065   -0.1077   -0.1075   -0.1072   -0.1061   -0.1045   -0.1024   -0.1003  -0.0984   -0.0943   -0.0877   -0.0799;
   -0.0575   -0.0588   -0.0631   -0.0676   -0.0671   -0.0694   -0.0715   -0.0739   -0.0775   -0.0785   -0.0787   -0.0744   -0.0704   -0.0688   -0.0669  -0.0674   -0.0624   -0.0586   -0.0573;
   -0.0183   -0.0188   -0.0211   -0.0241   -0.0226   -0.0254   -0.0291   -0.0333   -0.0370   -0.0336   -0.0345   -0.0326   -0.0283   -0.0247   -0.0236  -0.0251   -0.0221   -0.0198   -0.0193;
    0.0195    0.0186    0.0204    0.0186    0.0194    0.0181    0.0154    0.0162    0.0198    0.0212    0.0157    0.0131    0.0136    0.0158    0.0179  0.0171    0.0189    0.0171    0.0180;
    0.0494    0.0626    0.0562    0.0477    0.0323    0.0279    0.0289    0.0263    0.0204    0.0187    0.0173    0.0255    0.0183    0.0165    0.0119  0.0273    0.0358    0.0422    0.0290;
    0.0699    0.0695    0.0627    0.0557    0.0366    0.0316    0.0263    0.0207    0.0160    0.0198    0.0165    0.0218    0.0244    0.0228    0.0214  0.0405    0.0475    0.0543    0.0547;
    0.0207    0.0324    0.0323    0.0293    0.0304    0.0404    0.0419    0.0404    0.0385    0.0381    0.0374    0.0379    0.0389    0.0417    0.0446  0.0435    0.0465    0.0466    0.0349;
    0.0211    0.0282    0.0309    0.0263    0.0307    0.0334    0.0437    0.0466    0.0458    0.0479    0.0495    0.0495    0.0487    0.0467    0.0434  0.0390    0.0436    0.0409    0.0338;
    0.0386    0.0462    0.0331    0.0339    0.0365    0.0407    0.0394    0.0411    0.0407    0.0418    0.0431    0.0426    0.0392    0.0405    0.0381  0.0355    0.0347    0.0478    0.0402;
    0.0460    0.0438    0.0341    0.0311    0.0348    0.0373    0.0362    0.0335    0.0338    0.0363    0.0325    0.0340    0.0342    0.0356    0.0338  0.0301    0.0331    0.0428    0.0450;
    0.0394    0.0479    0.0513    0.0447    0.0538    0.0528    0.0483    0.0441    0.0444    0.0472    0.0488    0.0497    0.0507    0.0487    0.0495  0.0478    0.0525    0.0476    0.0376;
    0.0336    0.0411    0.0380    0.0471    0.0543    0.0508    0.0471    0.0445    0.0450    0.0484    0.0442    0.0383    0.0410    0.0471    0.0522  0.0432    0.0272    0.0347    0.0315;
    0.0158    0.0284    0.0361    0.0335    0.0487    0.0443    0.0442    0.0432    0.0451    0.0460    0.0451    0.0433    0.0435    0.0438    0.0416  0.0363    0.0397    0.0340    0.0246;
   -0.0186   -0.0121   -0.0057   -0.0070    0.0410    0.0451    0.0655    0.0604    0.0655    0.0641    0.0677    0.0701    0.0702    0.0636    0.0546  0.0033    0.0020   -0.0005    0.0058;
   -0.0242   -0.0267   -0.0277   -0.0200   -0.0215   -0.0224   -0.0223   -0.0180   -0.0202   -0.0173   -0.0046    0.0281    0.0311    0.0053   -0.0210  -0.0288   -0.0312   -0.0240   -0.0152;
   -0.0208   -0.0271   -0.0315   -0.0229   -0.0156   -0.0165   -0.0141   -0.0184   -0.0173   -0.0173   -0.0168   -0.0185   -0.0183   -0.0130   -0.0157 -0.0237   -0.0323   -0.0246   -0.0150];

% Сила сопротивления положительна, направлена назад
Cx1 = -Cx1;

% Коэффициент продольной силы, носки выпущены.  носки УБРАНЫ!!!
% Cx_nos = f(alpha2,beta)
Cx_nos1 = [-0.0277   -0.0285   -0.0318   -0.0256   -0.0184   -0.0156   -0.0159   -0.0162   -0.0174   -0.0181   -0.0179   -0.0167   -0.0168   -0.0156   -0.0153  -0.0225   -0.0287   -0.0254   -0.0246;
   -0.0314   -0.0310   -0.0259   -0.0191   -0.0161   -0.0157   -0.0162   -0.0173   -0.0189   -0.0193   -0.0186   -0.0186   -0.0170   -0.0155   -0.0154  -0.0184   -0.0252   -0.0303   -0.0307;
   -0.0295   -0.0298   -0.0260   -0.0233   -0.0209   -0.0215   -0.0214   -0.0224   -0.0230   -0.0224   -0.0220   -0.0217   -0.0213   -0.0205   -0.0199  -0.0223   -0.0250   -0.0288   -0.0285;
   -0.0148   -0.0153   -0.0163   -0.0150   -0.0167   -0.0173   -0.0185   -0.0189   -0.0193   -0.0196   -0.0192   -0.0185   -0.0179   -0.0178   -0.0162  -0.0155   -0.0168   -0.0158   -0.0153;
   -0.0136   -0.0149   -0.0143   -0.0136   -0.0168   -0.0178   -0.0182   -0.0188   -0.0197   -0.0202   -0.0196   -0.0188   -0.0180   -0.0172   -0.0160  -0.0144   -0.0151   -0.0149   -0.0144;
   -0.0029   -0.0010   -0.0003   -0.0005   -0.0004   -0.0006   -0.0017   -0.0027   -0.0033   -0.0033   -0.0033   -0.0024   -0.0014   -0.0004    0.0004  0.0013    0.0011   -0.0002   -0.0021;
    0.0085    0.0104    0.0116    0.0121    0.0131    0.0125    0.0122    0.0119    0.0104    0.0099    0.0096    0.0106    0.0117    0.0126    0.0127  0.0117    0.0112    0.0100    0.0081;
    0.0145    0.0168    0.0196    0.0218    0.0225    0.0231    0.0238    0.0238    0.0231    0.0224    0.0224    0.0226    0.0227    0.0223    0.0222  0.0215    0.0193    0.0165    0.0142;
    0.0165    0.0170    0.0205    0.0226    0.0252    0.0245    0.0236    0.0232    0.0233    0.0221    0.0232    0.0241    0.0250    0.0267    0.0276  0.0250    0.0229    0.0212    0.0189;
    0.0138    0.0172    0.0157    0.0178    0.0226    0.0251    0.0264    0.0274    0.0271    0.0278    0.0275    0.0271    0.0267    0.0249    0.0252  0.0203    0.0183    0.0198    0.0164;
    0.0092    0.0122    0.0129    0.0165    0.0202    0.0253    0.0279    0.0295    0.0296    0.0301    0.0309    0.0306    0.0278    0.0261    0.0247  0.0200    0.0174    0.0167    0.0137;
    0.0099    0.0134    0.0162    0.0149    0.0208    0.0229    0.0273    0.0286    0.0303    0.0305    0.0286    0.0307    0.0292    0.0259    0.0253  0.0194    0.0207    0.0179    0.0144;
    0.0206    0.0202    0.0236    0.0246    0.0289    0.0293    0.0290    0.0320    0.0317    0.0328    0.0314    0.0305    0.0289    0.0281    0.0262  0.0219    0.0209    0.0175    0.0179;
    0.0257    0.0274    0.0266    0.0236    0.0266    0.0283    0.0236    0.0298    0.0268    0.0309    0.0307    0.0280    0.0238    0.0284    0.0254  0.0224    0.0254    0.0262    0.0245];

Cx_nos1 = -Cx_nos1;

% Производная продольной силы по угловой скорости тангажа
% Cxwz = f(alpha1)
Cxwz1 = [0.9530    0.9530    0.9530    1.5500    1.9000    2.4600    2.9200    3.3000    2.7600    2.0500    1.5000    1.4900    1.8300    1.2100    1.3300   1.6100    0.9100    3.4300    0.6170    0.2730]';
Cxwz1 = -Cxwz1;

% Приращение Cxwz при выпуске носков
% dCxwz_nos = f(alpha2)
dCxwz_nos1 = [-1.2200   -1.2200   -1.2200   -1.6600   -1.6200   -1.5800   -1.9600   -2.5100   -2.0400   -1.6400   -0.8240   -0.8170   -1.1000   -0.5500]';
dCxwz_nos1 = -dCxwz_nos1;

dCx_sb1 = [-0.01010 -0.01010 -0.01010 -0.01010 -0.01010 -0.03580 -0.07900 -0.12270 -0.18270 -0.18920 -0.19880 -0.20000 -0.18740 -0.16730 -0.14760 -0.13100 -0.12790 -0.13250 -0.12500 -0.12500];
dCx_sb1 = -dCx_sb1;

interpCx = csaps({alpha1, beta1, fi1}, Cx1, 1.0 - 1.0e-5);
interpCx_nos = csaps({alpha2, beta1}, Cx_nos1, 1.0 - 1.0e-5);
interpCxwz1 = pchip(alpha1, Cxwz1);
interpCxwz2 = pchip(alpha2, dCxwz_nos1);
interpdCx_sb = pchip(alpha1, dCx_sb1);
end

ntraj = size(alpha, 2);
Cx = zeros(1, ntraj);
Cx0 = zeros(1, ntraj);
Cx_nos = zeros(1, ntraj);
Cxwz = zeros(1, ntraj);
dCx_sb = zeros(1, ntraj);
for i = 1 : ntraj
    Cx(i) = fnval(interpCx, {(alpha(i)), (beta(i)), (fi(i))});
    Cx0(i) = fnval(interpCx, {(alpha(i)), (beta(i)), (0)});
    Cx_nos(i) = fnval(interpCx_nos, {(alpha(i)), (beta(i))});
    Cxwz(i) = fnval(interpCxwz1, alpha(i)) + fnval(interpCxwz2, alpha(i)).*(dnos / deg2rad(25));
    dCx_sb(i) = fnval(interpdCx_sb, alpha(i));
end

% Сборка
dCx_nos = Cx_nos - Cx0;

Cx = Cx + dCx_nos.*(dnos / deg2rad(25)) + Cxwz.*((Wz.*ba)./(2 * V)) + dCx_sb.*(sb / deg2rad(60));
