# Control System Benchmark - Расширенные метрики качества

Улучшенная система бенчмаркинга для анализа качества систем управления с расширенным набором метрик и современной визуализацией.

## 🚀 Новые возможности

### 📊 Расширенные метрики качества

#### Временные характеристики
- **Время нарастания (Rise Time)** - время перехода от 10% до 90% установившегося значения
- **Время пика (Peak Time)** - время достижения первого максимума
- **Время установления (Settling Time)** - время достижения зоны ±2% от установившегося значения

#### Характеристики переходного процесса
- **Перерегулирование (Overshoot)** - максимальное превышение установившегося значения в %
- **Максимальное отклонение (Maximum Deviation)** - наибольшее отклонение от установившегося значения
- **Количество колебаний (Oscillation Count)** - число пересечений установившегося значения
- **Степень затухания (Damping Degree)** - характеристика затухания колебаний
- **Установившееся значение (Steady State Value)** - финальное значение сигнала

#### Интегральные критерии качества
- **IAE (Integral Absolute Error)** - интегральная абсолютная ошибка
- **ISE (Integral Squared Error)** - интегральная квадратичная ошибка  
- **ITAE (Integral Time Absolute Error)** - интегральная абсолютная ошибка, взвешенная по времени
- **Комплексный индекс качества** - обобщенная оценка качества системы

### 🎨 Улучшенная визуализация

- **Современный дизайн** с использованием Seaborn стилей
- **Зона установления** с визуальным выделением
- **Аннотации ключевых точек** (время установления, максимум)
- **Таблица метрик** прямо на графике
- **График ошибки** для детального анализа
- **Сравнительные таблицы** для множественного анализа

## 📋 Использование

### Базовый анализ одной системы

```python
from bench import ControlBenchmark
import numpy as np

# Создание экземпляра
benchmark = ControlBenchmark()

# Подготовка данных
time_points = np.linspace(0, 10, 1000)
dt = time_points[1] - time_points[0]
control_signal = np.ones_like(time_points)
system_signal = your_system_response  # Ваш отклик системы

# Построение графика с метриками
benchmark.plot(
    control_signal, 
    system_signal, 
    signal_val=1.0, 
    dt=dt,
    title="Анализ системы управления"
)

# Генерация детального отчета
report = benchmark.generate_report(
    control_signal, 
    system_signal, 
    signal_val=1.0, 
    dt=dt,
    system_name="Моя система"
)
print(report)
```

### Сравнение нескольких систем

```python
# Подготовка данных для сравнения
systems_data = {
    "Система A": (control_signal_1, system_signal_1),
    "Система B": (control_signal_2, system_signal_2),
    "Система C": (control_signal_3, system_signal_3)
}

# Сравнительный анализ
benchmark.compare_systems(
    systems_data, 
    signal_val=1.0, 
    dt=dt,
    title="Сравнение систем управления"
)
```

### Получение метрик программно

```python
# Вычисление всех метрик
metrics = benchmark.becnchmarking_one_step(
    control_signal, 
    system_signal, 
    signal_val=1.0, 
    dt=dt
)

# Доступные метрики:
print(f"Время нарастания: {metrics['rise_time']:.3f} с")
print(f"Время пика: {metrics['peak_time']:.3f} с")
print(f"Перерегулирование: {metrics['overshoot']:.2f}%")
print(f"IAE: {metrics['iae']:.2f}")
print(f"Индекс качества: {metrics['performance_index']:.3f}")
# ... и многие другие
```

## 📁 Структура файлов

- `bench.py` - Основной класс ControlBenchmark с расширенными возможностями
- `function.py` - Функции для вычисления метрик качества
- `example_benchmark_usage.py` - Демонстрация всех возможностей
- `README.md` - Данная документация

## 🔧 Зависимости

```python
numpy
matplotlib
seaborn
scipy
typing
```

## 📊 Интерпретация метрик

### Временные характеристики
- **Время нарастания** < 1с - быстрая система
- **Время установления** < 3с - хорошая динамика
- **Время пика** ≈ 0.5 × время установления - оптимально

### Качество переходного процесса
- **Перерегулирование** < 10% - отличное качество
- **Перерегулирование** 10-20% - хорошее качество
- **Перерегулирование** > 30% - требует настройки

### Интегральные критерии
- **IAE** - чем меньше, тем лучше точность
- **ISE** - чем меньше, тем лучше энергетическая эффективность
- **ITAE** - чем меньше, тем быстрее система
- **Индекс качества** < 1.0 - отличная система

## 🎯 Примеры применения

1. **Настройка ПИД-регуляторов** - оптимизация параметров
2. **Сравнение алгоритмов управления** - выбор лучшего подхода
3. **Анализ робастности** - оценка устойчивости к возмущениям
4. **Документирование результатов** - генерация отчетов
5. **Обучение и исследования** - визуализация теории управления

## 🚀 Запуск примера

```bash
python example_benchmark_usage.py
```

Эта команда продемонстрирует все новые возможности системы бенчмаркинга.