proc __copyAndLoad__ {offset size package} {
    set mchan [open [info script] rb]
    set lchan [file tempfile tempname]
    chan configure $lchan -translation binary
    try {
        chan seek $mchan $offset start
        chan copy $mchan $lchan -size $size
    } finally {
        chan close $mchan
        chan close $lchan
    }

    try {
        load $tempname $package
    } finally {
        catch {file delete $tempname}
    }
}
if {$::tcl_platform(os) eq "Linux"} {
    __copyAndLoad__ 77782 238944 ral
}
if {$::tcl_platform(os) eq "Windows NT"} {
    __copyAndLoad__ 316726 497381 ral
}
if {$::tcl_platform(os) eq "Darwin"} {
    __copyAndLoad__ 814107 324292 ral
}
rename __copyAndLoad__ {}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       
# This software is copyrighted 2004 - 2014 by G. Andrew Mangogna.
# The following terms apply to all files associated with the software unless
# explicitly disclaimed in individual files.
# 
# The authors hereby grant permission to use, copy, modify, distribute,
# and license this software and its documentation for any purpose, provided
# that existing copyright notices are retained in all copies and that this
# notice is included verbatim in any distributions. No written agreement,
# license, or royalty fee is required for any of the authorized uses.
# Modifications to this software may be copyrighted by their authors and
# need not follow the licensing terms described here, provided that the
# new terms are clearly indicated on the first page of each file where
# they apply.
# 
# IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING
# OUT OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES
# THEREOF, EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
# 
# THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE
# IS PROVIDED ON AN "AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE
# NO OBLIGATION TO PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS,
# OR MODIFICATIONS.
# 
# GOVERNMENT USE: If you are acquiring this software on behalf of the
# U.S. government, the Government shall have only "Restricted Rights"
# in the software and related documentation as defined in the Federal
# Acquisition Regulations (FARs) in Clause 52.227.19 (c) (2).  If you
# are acquiring the software on behalf of the Department of Defense,
# the software shall be classified as "Commercial Computer Software"
# and the Government shall have only "Restricted Rights" as defined in
# Clause 252.227-7013 (c) (1) of DFARs.  Notwithstanding the foregoing,
# the authors grant the U.S. Government and others acting in its behalf
# permission to use and distribute the software in accordance with the
# terms specified in this license.
# 
#  *++
# MODULE:
# ral.tcl -- Tcl scripts portion of TclRAL
# 
# ABSTRACT:
# This file contains the Tcl script portions of the TclRAL package.
#  *--

namespace eval ::ral {
    namespace export tuple2matrix
    namespace export relation2matrix
    namespace export relformat
    namespace export tupleformat
    namespace export serialize
    namespace export serializeToFile
    namespace export deserialize
    namespace export deserializeFromFile
    namespace export deserialize-0.8.X
    namespace export deserializeFromFile-0.8.X
    namespace export merge
    namespace export mergeFromFile
    namespace export storeToMk
    namespace export loadFromMk
    namespace export mergeFromMk
    namespace export storeToSQLite
    namespace export loadFromSQLite
    namespace export mergeFromSQLite
    namespace export dump
    namespace export dumpToFile
    namespace export csv
    namespace export csvToFile
    if {[package vsatisfies [package require Tcl] 8.5]} {
        # Configure the exported commands into the namespace ensemble
        # for the other package commands.
        set map [namespace ensemble configure ::ral -map]
        set map [dict merge $map {
            tuple2matrix ::ral::tuple2matrix
            relation2matrix ::ral::relation2matrix
            relformat ::ral::relformat
            tupleformat ::ral::tupleformat
            serialize ::ral::serialize
            serializeToFile ::ral::serializeToFile
            deserialize ::ral::deserialize
            deserializeFromFile ::ral::deserializeFromFile
            deserialize-0.8.X ::ral::deserialize-0.8.X
            deserializeFromFile-0.8.X ::ral::deserializeFromFile-0.8.X
            merge ::ral::merge
            mergeFromFile ::ral::mergeFromFile
            storeToMk ::ral::storeToMk
            loadFromMk ::ral::loadFromMk
            mergeFromMk ::ral::mergeFromMk
            storeToSQLite ::ral::storeToSQLite
            loadFromSQLite ::ral::loadFromSQLite
            mergeFromSQLite ::ral::mergeFromSQLite
            dump ::ral::dump
            dumpToFile ::ral::dumpToFile
            csv ::ral::csv
            csvToFile ::ral::csvToFile
        }]
        namespace ensemble configure ::ral -map $map
    } else {
        namespace export rcount
        namespace export rcountd
        namespace export rsum
        namespace export rsumd
        namespace export ravg
        namespace export ravgd
        namespace export rmin
        namespace export rmax
    }

    if {![catch {package require report}]} {
        # Default report style for Tuple types
        ::report::defstyle ::ral::tupleAsTable {{capRows 2}} {
            data set [split [string repeat "| " [columns]]|]
            set sepTemplate [split [string repeat "+ - " [columns]]+]
            top set $sepTemplate
            top enable
            bottom set [top get]
            bottom enable
            topdata set [data get]
            topcapsep set [top get]
            topcapsep enable
            tcaption $capRows
        }
        namespace export tupleAsTable

        # Default report style for Relation types
        ::report::defstyle ::ral::relationAsTable {{capRows 2}} {
            ::ral::tupleAsTable $capRows
            top set [top get]
            bottom set [top get]
            topcapsep set [top get]
        }
        namespace export relationAsTable
    }

    variable reportCounter 0 ; # used to make unique names
    # Maximum width of a column that holds a scalar value.
    # Relation and Tuple values are always shown in their "normal" width.
    variable maxColLen 30
    # The internal representation of NULL for the benefit of SQLite.
    variable nullValue {}

    # Define a proc to determine if the version of a serialized file
    # is compatible with the library. We use "pkgconfig" if it is
    # available, "package require" if not. If building for some older
    # versions of Tcl, "pkgconfig" may not be available.
    if {[llength [info commands ::ral::pkgconfig]]} {
        proc getVersion {} {
            return [::ral::pkgconfig get version]
        }
    } else {
        proc getVersion {} {
            return [package require ral]
        }
    }
    proc getCompatVersion {} {
        lassign [split [getVersion] .] maj min rev
        incr maj
        return 0.9-$maj.$min
    }

    variable sqlTypeMap ; array set sqlTypeMap {
        int         integer
        double      real
        bytearray   blob
    }

    # This is the SQL schema required to store the ral meta-data into SQL so
    # that we will have all the required information to reconstruct the relvars
    # when we load it back in.
    variable ralSQLSchema {
-- SQL Tables to hold RAL specific meta-data
create table __ral_version (
    Vnum text not null,
    Date timestamp not null,
    constraint __ral_version_id unique (Vnum)) ;
create table __ral_relvar (
    Vname text not null,
    constraint __ral_relvar_id unique (Vname)) ;
create table __ral_attribute (
    Vname text not null,
    Aname text not null,
    Type text not null,
    constraint __ral_attribute_id unique (Vname, Aname),
    constraint __ral_attribute_ref foreign key (Vname) references
        __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_attribute_ref_index on __ral_attribute (Vname) ;
create table __ral_identifier (
    IdNum integer not null,
    Vname text not null,
    Attr text not null,
    constraint __ral_identifier_id unique (IdNum, Vname, Attr),
    constraint __ral_identifier_ref foreign key (Vname) references
        __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred,
    constraint __ral_identifier_ref2 foreign key (Vname, Attr) references
        __ral_attribute(Vname, Aname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_identifier_ref_index on __ral_identifier (Vname) ;
create index __ral_identifier_ref2_index on __ral_identifier (Vname, Attr) ;
create table __ral_association (
    Cname text not null,
    Referring text not null,
    ReferringAttrs text not null,
    RefToSpec text not null,
    RefTo text not null,
    RefToAttrs text not null,
    ReferringSpec text not null,
    constraint __ral_association_id unique (Cname),
    constraint __ral_association_one foreign key (Referring)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred,
    constraint __ral_association_other foreign key (RefTo)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_association_oneindex on __ral_association (Referring) ;
create index __ral_association_otherindex on __ral_association (RefTo) ;
create table __ral_partition (
    Cname text not null,
    SuperType text not null,
    SuperAttrs text not null,
    constraint __ral_partition_id unique (Cname),
    constraint __ral_partition_ref foreign key (SuperType)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_partition_ref on __ral_partition (SuperType) ;
create table __ral_subtype (
    Cname text not null,
    SubType text not null,
    SubAttrs text not null,
    constraint __ral_subtype_id unique (Cname, SubType),
    constraint __ral_subtype_ref foreign key (SubType)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_subtype_refindex on __ral_subtype (SubType) ;
create table __ral_correlation (
    Cname text not null,
    IsComplete integer not null,
    AssocRelvar text not null,
    OneRefAttrs text not null,
    OneRefSpec text not null,
    OneRelvar text not null,
    OneAttrs text not null,
    OtherRefAttrs text not null,
    OtherRefSpec text not null,
    OtherRelvar text not null,
    OtherAttrs text not null,
    constraint __ral_correlation_id unique (Cname),
    constraint __ral_correlation_assoc foreign key (AssocRelvar)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred,
    constraint __ral_correlation_oneref foreign key (OneRelvar)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred,
    constraint __ral_correlation_otherref foreign key (OtherRelvar)
        references __ral_relvar (Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
create index __ral_correlation_assocref on __ral_correlation (AssocRelvar) ;
create index __ral_correlation_oneref on __ral_correlation (OneRelvar) ;
create index __ral_correlation_otherref on __ral_correlation (OtherRelvar) ;
create table __ral_procedural (
    Cname text not null,
    Script text not null,
    constraint __ral_procedural_id unique(Cname)) ;
create table __ral_proc_participant (
    Cname text not null,
    ParticipantRelvar text not null,
    constraint __ral_proc_participant_id unique (Cname, ParticipantRelvar),
    constraint __ral_proc_participant_proc foreign key (Cname)
        references __ral_procedural (Cname) on delete cascade on update cascade
        deferrable initially deferred,
    constraint __ral_proc_participant_relvar foreign key (ParticipantRelvar)
        references __ral_relvar(Vname) on delete cascade on update cascade
        deferrable initially deferred) ;
    }
}

# Convert a tuple into a matrix.
proc ::ral::tuple2matrix {tupleValue {noheading 0}} {
    package require struct::matrix 2

    set m [::struct::matrix]
    $m add columns [tuple degree $tupleValue]
    set heading [tuple heading $tupleValue]
    if {!$noheading} {
        addHeading $m $heading
    }
    addTuple $m $tupleValue [getFormatMap $heading]

    return $m
}

# Convert a relation into a matrix.
proc ::ral::relation2matrix {relValue {sortAttr {}} {noheading 0}} {
    package require struct::matrix 2

    set m [::struct::matrix]
    $m add columns [relation degree $relValue]
    set heading [relation heading $relValue]
    set attrReportMap [getFormatMap $heading]
    if {!$noheading} {
        addHeading $m $heading
    }
    relation foreach r $relValue -ascending $sortAttr {
        addTuple $m [relation tuple $r] $attrReportMap
    }

    return $m
}

# This procedure is intended to be to relations what "parray" is to arrays.
# It provides a very simple text formatting of relation values into a
# tabular layout. Rather than writing the output to a channel "relformat"
# returns a string containing the formatted relation.
# There is much more that could be done here, e.g. adding a "-format" option
# where you could specify the details of the formatting.
proc ::ral::relformat {relValue {title {}} {sortAttrs {}} {noheading 0}} {
    package require report

    variable reportCounter
    set reportName rep[incr reportCounter]
    ::report::report $reportName [relation degree $relValue]\
        style ::ral::relationAsTable [expr {$noheading ? 0 : 2}]
    set m [relation2matrix $relValue $sortAttrs $noheading]
    set result [string trimright [$reportName printmatrix $m]]
    if {$title ne ""} {
        append result "\n" $title "\n" [string repeat - [string length $title]]
    }

    $reportName destroy
    $m destroy

    return $result
}

# Return a string that pretty prints a tuple.
proc ::ral::tupleformat {tupleValue {title {}} {noheading 0}} {
    package require report

    variable reportCounter
    set reportName rep[incr reportCounter]
    ::report::report $reportName [tuple degree $tupleValue]\
        style ::ral::tupleAsTable [expr {$noheading ? 0 : 2}]
    set m [tuple2matrix $tupleValue $noheading]
    set result [string trimright [$reportName printmatrix $m]]
    if {$title ne ""} {
        append result "\n" $title "\n" [string repeat - [string length $title]]
    }

    $reportName destroy
    $m destroy

    return $result
}

# Serialization format:
# Dictionary with keys:
#   Version <library version>
#   Relvars {<list of relvar defs>}
#   Constraints {<list of constraints>}
#   Values {<list of relvar names/relation values >}
#
#   <list of relvar defs> :
#       {<relvar name> <relation heading> <list of relvar identfiers}
#
#   <list of constraints> :
#       {association | partition | correlation | procedural <constraint detail>}
#   <association constaint detail> :
#       <association name> <relvar name> {<attr list>} <mult/cond>\
#           <relvar name> {<attr list>} <mult/cond>
#   <partition constraint detail> :
#       <partition name> <supertype> {<attr list>} <subtype1> {<attr list>} ...
#   <correlation constaint detail> :
#       <?-complete?> <correlation name> <correl relvar>
#           {<attr list>} <mult/cond> <relvarA> {<attr list>}
#           {<attr list>} <mult/cond> <relvarB> {<attr list>}
#   <procedural constraint detail> :
#       <procedural name> {<relvar name list>} {<script>}
#
#   <list of relvar names/relation values> :
#       {<relvar name> <relation value>}

# Generate a string that encodes all the relvars.

proc ::ral::serialize {{pattern *}} {
    set pattern [SerializePattern $pattern]
    set rnames [BaseNames [relvar names $pattern]]
    set cnames [BaseNames [relvar constraint names $pattern]]

    set result [list]

    lappend result Version [getVersion]

    # Gather the headings and identifiers
    set relNameList [list]
    relation foreach rname $rnames {
        relation assign $rname
        lappend relNameList $Base\
            [relation heading [relvar set $Qualified]]\
            [relvar identifiers $Qualified]
    }
    lappend result Relvars $relNameList

    # Gather the contraints
    set constraints [list]
    relation foreach cname $cnames {
        relation assign $cname
        lappend constraints [getRelativeConstraintInfo $Qualified]
    }
    lappend result Constraints $constraints

    # Gather the bodies
    set bodies [list]
    relation foreach rname $rnames {
        relation assign $rname
        lappend bodies [list $Base [relvar set $Qualified]]
    }

    lappend result Values $bodies

    return $result
}

proc ::ral::serializeToFile {fileName {pattern *}} {
    set pattern [SerializePattern $pattern]
    set chan [::open $fileName w]
    catch {puts $chan [serialize $pattern]} result opts
    ::close $chan
    return -options $opts $result
}

# Restore the relvar values from a string.
proc ::ral::deserialize {value {ns ::}} {
    if {[llength $value] == 4} {
        # Assume it is 0.8.X style serialization.
        deserialize-0.8.X $value [expr {$ns eq {} ? "::" : $ns}]
        return
    }
    set ns [DeserialNS $ns]

    if {[dict size $value] != 4} {
        error "bad value format, expected dictionary of 4 items,\
                got [dict size $value] items"
    }

    if {![package vsatisfies [dict get $value Version] [getCompatVersion]]} {
        error "incompatible version number, \"$versionNumber\",\
            current library version is, \"[getVersion]\""
    }

    foreach {rvName rvHead rvIds} [dict get $value Relvars] {
        ::ral::relvar create ${ns}$rvName $rvHead {*}$rvIds
    }

    foreach constraint [dict get $value Constraints] {
        ::ral::relvar {*}[setRelativeConstraintInfo $ns $constraint]
    }

    relvar eval {
        foreach body [dict get $value Values] {
            foreach {relvarName relvarBody} $body {
                ::ral::relvar set ${ns}$relvarName $relvarBody
            }
        }
    }

    return
}

proc ::ral::deserializeFromFile {fileName {ns ::}} {
    set ns [DeserialNS $ns]
    set chan [::open $fileName r]
    catch {deserialize [read $chan] $ns} result opts
    ::close $chan
    return -options $opts $result
}

proc ::ral::deserialize-0.8.X {value {ns ::}} {
    set ns [DeserialNS $ns]
    if {[llength $value] != 4} {
        error "bad value format, expected list of 4 items,\
                got [llength $value] items"
    }
    lassign $value version relvars constraints bodies

    lassign $version versionKeyWord verNum
    if {$versionKeyWord ne "Version"} {
	error "expected keyword \"Version\", got \"$versionKeyWord\""
    }
    if {![package vsatisfies $verNum 0.8-]} {
	error "incompatible version number, \"$verNum\",\
            while attempting to deserialize version 0.8 data"
    }

    lassign $relvars relvarKeyWord revarDefs
    if {$relvarKeyWord ne "Relvars"} {
	error "expected keyword \"Relvars\", got \"$revarKeyWord\""
    }
    # In version 0.8.X, relvar headings consisted of a list
    # of 3 items: a) the "Relation" keyword, b) the relation heading
    # and c) a list of identifiers. We must adapt this to 0.9.X
    # syntax
    foreach {rvName rvHead} $revarDefs {
        lassign $rvHead keyword heading ids
        # In 0.8.X, relation valued attributes have an embedded relation
        # heading that includes a list of identifiers. So we have to
        # examine the heading for any relation valued attribute and
        # patch things up accordingly. Fortunately, the tuple valued
        # attrbutes don't have any syntax change.
        set newHeading [list]
        foreach {attrName attrType} $heading {
            lappend newHeading $attrName\
                [expr {[lindex $attrType 0] eq "Relation" ?\
                [lrange $attrType 0 1] : $attrType}]
        }
	namespace eval $ns [list ::ral::relvar create $rvName $newHeading] $ids
    }

    # Constraint syntax is unmodified between 0.8.X and 0.9.X
    lassign $constraints cnstrKeyWord cnstrDef
    if {$cnstrKeyWord ne "Constraints"} {
	error "expected keyword \"Constraints\", got \"$cnstrKeyWord\""
    }
    foreach constraint $cnstrDef {
	namespace eval $ns ::ral::relvar $constraint
    }

    lassign $bodies bodyKeyWord bodyDefs
    if {$bodyKeyWord ne "Bodies"} {
	error "expected keyword \"Bodies\", got \"$bodyKeyWord\""
    }

    # The 0.8.X serialization format uses a list of tuple values
    # to represent the body. This works fine with "relvar insert".
    relvar eval {
	foreach body $bodyDefs {
	    foreach {relvarName relvarBody} $body {
		namespace eval $ns ::ral::relvar insert [list $relvarName]\
		    $relvarBody
	    }
	}
    }

    return
}

proc ::ral::deserializeFromFile-0.8.X {fileName {ns ::}} {
    set chan [::open $fileName r]
    catch {deserialize-0.8.X [read $chan] $ns} result opts
    ::close $chan
    return -options $opts $result
}

proc ::ral::merge {value {ns ::}} {
    set ns [DeserialNS $ns]
    if {[dict size $value] != 4} {
        error "bad value format, expected dictionary of 4 items,\
                got [dict size $value] items"
    }
    if {![package vsatisfies [dict get $value Version] [getCompatVersion]]} {
        error "incompatible version number, \"$versionNumber\",\
            current library version is, \"[getVersion]\""
    }

    set newRelvars [list]
    set matchingRelvars [list]
    set mismatched [list]
    foreach {rvName rvHead rvIds} [dict get $value Relvars] {
        set fullName ${ns}$rvName
        if {![relvar exists $fullName]} {
            # New relvars are just created.
            ::ral::relvar create $fullName $rvHead {*}$rvIds
            lappend newRelvars $rvName
        } else {
            # For existing relvars we test if the type is the same.
            set mergeRelation [relation create $rvHead]
            if {[relation issametype $mergeRelation [relvar set $fullName]]} {
                lappend matchingRelvars $rvName
            } else {
                lappend mismatched $rvName
            }
        }
    }

    foreach constraint [dict get $value Constraints] {
        set cname [lindex $constraint 1]
        if {$cname eq "-complete"} {
            set cname [lindex $constraint 2]
        }
        set cname ${ns}$cname
        if {![relvar constraint exists $cname]} {
            eval ::ral::relvar [setRelativeConstraintInfo $ns $constraint]
        }
    }

    relvar eval {
        foreach body [dict get $value Values] {
            foreach {relvarName relvarBody} $body {
                if {$relvarName in $matchingRelvars} {
                    relvar uinsert ${ns}$relvarName\
                            {*}[relation body $relvarBody]
                } elseif {$relvarName in $newRelvars} {
                    relvar insert ${ns}$relvarName\
                            {*}[relation body $relvarBody]
                }
            }
        }
    }

    return $mismatched
}

proc ::ral::mergeFromFile {fileName {ns ::}} {
    set ns [DeserialNS $ns]
    set chan [::open $fileName r]
    catch {merge [read $chan] $ns} result opts
    ::close $chan
    return -options $opts $result
}

proc ::ral::storeToMk {fileName {pattern *}} {
    package require Mk4tcl
    set pattern [SerializePattern $pattern]
    set rnames [BaseNames [relvar names $pattern]]
    set cnames [BaseNames [relvar constraint names $pattern]]

    # Back up an existing file
    catch {file rename -force -- $fileName $fileName~}

    ::mk::file open db $fileName
    catch {
        # Add some versioning information into a view. Just a sanity check
        # when the data is loaded later.
        ::mk::view layout db.__ral_version {Version_ral Date_ral Comment_ral}
        ::mk::row append db.__ral_version\
            Version_ral [getVersion]\
            Date_ral [clock format [clock seconds]]\
            Comment_ral "Created by: \"[info level 0]\""
        # Create a set of views that are used as catalogs to hold
        # the relvar info that will be needed to reconstruct the values.
        ::mk::view layout db.__ral_relvar\
                {Name_ral Heading_ral Ids_ral View_ral}
        ::mk::view layout db.__ral_constraint Constraint_ral

        # Get the names of the relvars and insert them into the catalog.
        # Convert the names to be relative before inserting.  Also create the
        # views that will hold the values.  Metakit doesn't like "::" and some
        # other characters in view names.
        relation foreach rname $rnames {
            relation assign $rname
            set heading [relation heading [relvar set $Qualified]]
            set ids [relvar identifiers $Qualified]
            ::mk::row append db.__ral_relvar Name_ral $Base\
                Heading_ral $heading Ids_ral $ids View_ral $Base
            # Determine the structure of the view that will hold the relvar
            # value.  Special attention is required for Tuple and Relation
            # valued attributes.
            set relvarLayout [list]
            foreach {attr type} $heading {
                lappend relvarLayout [mkHeading $attr $type]
            }
            ::mk::view layout db.$Base $relvarLayout
        }
        # Get the constraints and put them into the catalog.
        relation foreach cname $cnames {
            relation assign $cname
            # Constrain names must be made relative.
            ::mk::row append db.__ral_constraint Constraint_ral\
                [getRelativeConstraintInfo $Qualified]
        }
        # Populate the views for each relavar.
        relation foreach rname $rnames {
            relation assign $rname
            ::mk::cursor create cursor db.$Base 0
            relation foreach r [relvar set $Qualified] {
                ::mk::row insert $cursor
                mkStoreTuple $cursor [relation tuple $r]
                ::mk::cursor incr cursor
            }
        }
    } result opts
    ::mk::file close db
    return -options $opts $result
}

proc ::ral::loadFromMk {fileName {ns ::}} {
    package require Mk4tcl

    set ns [DeserialNS $ns]
    ::mk::file open db $fileName -readonly
    catch {
        # Check that a "version" view exists and that the information
        # is consistent before we proceed.
        set version [mkCheckVersion db]
        # determine the relvar names and types by reading the catalog
        ::mk::loop rvCursor db.__ral_relvar {
            array set relvarInfo [::mk::get $rvCursor]
            # Check for old style serialization that included namespace
            # information.
            if {[string first : $relvarInfo(Name_ral)] != -1} {
                set relvarName ${ns}[namespace tail $relvarInfo(Name_ral)]
            } else {
                set relvarName ${ns}$relvarInfo(Name_ral)
            }
            ::ral::relvar create $relvarName\
                    $relvarInfo(Heading_ral) {*}$relvarInfo(Ids_ral)
        }
        # create the constraints
        ::mk::loop cnstrCursor db.__ral_constraint {
            ::ral::relvar {*}[setRelativeConstraintInfo $ns\
                    [lindex [::mk::get $cnstrCursor] 1]]
        }
        # fetch the relation values from the views
        relvar eval {
            ::mk::loop rvCursor db.__ral_relvar {
                array set relvarInfo [::mk::get $rvCursor]
                #parray relvarInfo
                ::mk::loop vCursor db.$relvarInfo(View_ral) {
                    ::ral::relvar insert ${ns}$relvarInfo(Name_ral)\
                        [mkLoadTuple $vCursor $relvarInfo(Heading_ral)]
                }
            }
        }
        set version
    } result opts
    ::mk::file close db
    return -options $opts $result
}

# Merge data from a metakit store of relvars.
# All relvars that are in the file but not currently defined are created.
# All relvars whose names and headings match currently defined relvars
# will have their relation values unioned with those in the file.
proc ::ral::mergeFromMk {fileName {ns ::}} {
    package require Mk4tcl

    set ns [DeserialNS $ns]
    ::mk::file open db $fileName -readonly
    catch {
        mkCheckVersion db
        # determine the relvar names and types by reading the catalog
        set newRelvars [list]
        set sameTypeRelvars [list]
        set mismatched [list]
        ::mk::loop rvCursor db.__ral_relvar {
            array set relvarInfo [::mk::get $rvCursor]
            set relvarName ${ns}$relvarInfo(Name_ral)
            # Check if the relvar already exists
            if {![relvar exists $relvarName]} {
                # New relvar
                ::ral::relvar create $relvarName $relvarInfo(Heading_ral)]\
                        {*}$relvarInfo(Ids_ral)
            } else {
                set typeRelation [::ral::relation create\
                        $relvarInfo(Heading_ral)]
                if {[relation issametype $typeRelation\
                        [relvar set $relvarName]]} {
                    lappend sameTypeRelvars $relvarInfo(Name_ral)
                } else {
                    lappend mismatched $relvarInfo(Name_ral)
                }
            }
        }
        # create the constraints
        ::mk::loop cnstrCursor db.__ral_constraint {
            set cinfo [lindex [::mk::get $cnstrCursor] 1]
            set cName ${ns}[lindex $cinfo 1]
            if {![relvar constraint exists $cName]} {
                ::ral::relvar {*}[setRelativeConstraintInfo $ns $cinfo]
            }
        }
        # fetch the relation values from the views
        set failedMerge [list]
        relvar eval {
            ::mk::loop rvCursor db.__ral_relvar {
                array set relvarInfo [::mk::get $rvCursor]
                if {$relvarInfo(Name_ral) in $newRelvars} {
                    set op insert
                } elseif {$relvarInfo(Name_ral) in $sameTypeRelvars} {
                    set op uinsert
                } else {
                    continue
                }
                ::mk::loop vCursor db.$relvarInfo(View_ral) {
                    ::ral::relvar $op ${ns}$relvarInfo(Name_ral)\
                        [mkLoadTuple $vCursor $relvarInfo(Heading_ral)]
                }
            }
        }
        set mismatched
    } result opts

    ::mk::file close db

    return -options $opts $result
}

# Save all the relvars that match "pattern" in to a SQLite database named
# "filename".
proc ::ral::storeToSQLite {filename {pattern *}} {
    package require sqlite3

    set pattern [SerializePattern $pattern]
    set rnames [BaseNames [relvar names $pattern]]
    set cnames [BaseNames [relvar constraint names $pattern]]

    catch {file rename -force -- $filename $filename~}
    sqlite3 [namespace current]::sqlitedb $filename
    sqlitedb eval "pragma foreign_keys=ON;"

    catch {
        # First we insert the meta-data schema and populate it.
        variable ralSQLSchema
        sqlitedb eval $ralSQLSchema
        sqlitedb transaction {
            set version [getVersion]
            sqlitedb eval {insert into __ral_version (Vnum, Date)\
                values ($version, CURRENT_TIMESTAMP) ;}
            relation foreach rname $rnames {
                relation assign $rname
                # First the relvars and their attributes and identifiers
                sqlitedb eval {insert into __ral_relvar (Vname)\
                        values ($Base) ;}
                foreach {attrName type} [relation heading\
                        [relvar set $Qualified]] {
                    sqlitedb eval {insert into __ral_attribute (Vname, Aname,\
                            Type) values ($Base, $attrName, $type) ;}
                }
                set idCounter 0
                foreach identifier [relvar identifiers $Qualified] {
                    foreach idattr $identifier {
                        sqlitedb eval {insert into\
                            __ral_identifier (IdNum, Vname, Attr)\
                            values ($idCounter, $Base, $idattr) ;}
                    }
                    incr idCounter
                }
                # Next the constraints
                foreach constraint [relvar constraint member $Qualified] {
                    set cinfo [lassign [relvar constraint info $constraint]\
                        ctype]
                    switch -exact -- $ctype {
                    association {
                        lassign $cinfo cname rfering a1 c1 refto a2 c2
                        set cname [namespace tail $cname]
                        set rfering [namespace tail $rfering]
                        set refto [namespace tail $refto]
                        sqlitedb eval {insert or ignore into __ral_association\
                            (Cname, Referring, ReferringAttrs, RefToSpec,\
                            RefTo, RefToAttrs, ReferringSpec) values\
                            ($cname, $rfering, $a1, $c1, $refto, $a2, $c2) ;}
                    }
                    partition {
                        set subs [lassign $cinfo cname super sattrs]
                        set cname [namespace tail $cname]
                        set super [namespace tail $super]
                        sqlitedb eval {insert or ignore into __ral_partition\
                            (Cname, SuperType, SuperAttrs) values\
                            ($cname, $super, $sattrs) ;}
                        foreach {subtype subattrs} $subs {
                            set subtype [namespace tail $subtype]
                            sqlitedb eval {insert or ignore into __ral_subtype\
                                (Cname, SubType, SubAttrs) values\
                                ($cname, $subtype, $subattrs) ;}
                        }
                    }
                    correlation {
                        if {[lindex $cinfo 0] eq "-complete"} {
                            set comp 1
                            set cinfo [lrange $cinfo 1 end]
                        } else {
                            set comp 0
                        }
                        lassign $cinfo cname correl ref1Attr c1 rel1 rel1Attr\
                            ref2Attr c2 rel2 rel2Attr
                        set cname [namespace tail $cname]
                        set correl [namespace tail $correl]
                        set rel1 [namespace tail $rel1]
                        set rel2 [namespace tail $rel2]
                        sqlitedb eval {insert or ignore into __ral_correlation\
                            (Cname, IsComplete, AssocRelvar, OneRefAttrs,\
                            OneRefSpec, OneRelvar, OneAttrs, OtherRefAttrs,\
                            OtherRefSpec, OtherRelvar, OtherAttrs) values\
                            ($cname, $comp, $correl, $ref1Attr, $c1, $rel1,\
                            $rel1Attr, $ref2Attr, $c2, $rel2, $rel2Attr) ;}
                    }
                    procedural {
                        set cname [namespace tail [lindex $cinfo 0]]
                        set script [lindex $cinfo end]
                        sqlitedb eval {insert or ignore into __ral_procedural\
                            (Cname, Script) values ($cname, $script) ;}
                        foreach participant [lrange $cinfo 1 end-1] {
                            set basename [namespace tail $participant]
                            sqlitedb eval {insert or ignore into\
                                __ral_proc_participant\
                                (Cname, ParticipantRelvar) values\
                                ($cname, $basename) ;}
                        }
                    }
                    default {
                        error "unknown constraint type, \"$ctype\""
                    }
                    }
                }
            }
        }
        # Next we insert the schema that corresponds to the "pattern" as an SQL
        # schema. This gives us the ability to manipulate the relvar data as
        # SQL tables.
        sqlitedb transaction exclusive {
            sqlitedb eval [sqlSchema $pattern]
        }
        # Finally we insert the values of the relvars.  There is a tricky part
        # here. Since ral attributes can be any Tcl string and since SQL is
        # particular about names, we have to account for that.  Also, we have
        # to deal with the "run time" nature of what we are doing.  We
        # construct the "insert" statement in order to fill in the attributes
        # in the required order. We also want to use the ability of the SQLite
        # Tcl bindings to fetch the values of Tcl variables. Unfortunately, (as
        # determined by experiment) SQLite cannot deal with Tcl variable syntax
        # of the form "${name}". Consequently, any character in an attribute
        # name that would terminate scanning a variable reference of the form
        # "$name" creates a problem. The solution is to use the ability of
        # "relvar assign" to assign attributes to explicitly named variables.
        # We choose as the alternate variable names the names given to the
        # SQLite columns.
        sqlitedb transaction exclusive {
            relation foreach rname $rnames {
                relation assign $rname
                set relValue [relvar set $Qualified]
                set sqlTableName [mapNamesToSQL $Base]
                set attrNames [relation attributes $relValue]
                # Map the attribute names to a set of SQLite column names.
                set sqlCols [mapNamesToSQL {*}$attrNames]
                set statement "insert into $sqlTableName ([join $sqlCols {, }])\
                    values ("
                # Create two lists here, one is used to get "relvar assign" to
                # assign attributes to variable names that SQLite can later
                # resolve. The other is just a list of those resolvable names so
                # that we can finish out the "insert" statement.
                set assignVars [list]
                set sqlVars [list]
                foreach attr $attrNames sqlCol $sqlCols {
                    lappend assignVars [list $attr $sqlCol]
                    lappend sqlVars :$sqlCol
                }
                # Finally after all of that, the actual populating of the
                # SQLite tables is trivial.
                set condAttrs [FindNullableAttrs $Qualified]
                variable nullValue
                relation foreach row $relValue {
                    # "relation assign" for a tuple or relation valued
                    # attribute gives the heading and the body. Later when we
                    # are loading the results, the "relvar insert" command will
                    # want that stripped off.
                    relation assign $row {*}$assignVars
                    # Here we have to consider the case where an attribute is
                    # referential and the relationship is condition.  In SQL we
                    # will represent that by a NULL value. Since there are not
                    # NULL values in Tcl, we have to specify the value for
                    # these types of attribute. That is done in the "nullValue"
                    # variable. So we will pass a NULL value to SQLite only if
                    # the attribute is referential to a conditional
                    # relationship and its value matches the contents of the
                    # "nullValue" variable.
                    if {[llength $condAttrs] != 0} {
                        set valueClause {}
                        foreach attr $attrNames sqlVar $sqlVars {
                            if {$attr in $condAttrs &&
                                    [set $attr] eq $nullValue} {
                                append valueClause NULL ,
                            } else {
                                append valueClause $sqlVar ,
                            }
                        }
                        set valueClause [string trimright $valueClause ,]
                    } else {
                        # This completes the composition of the "insert"
                        # statement.  We just managed to add a list of variable
                        # references (i.e.  variable names each of which is
                        # preceded by a ":").
                        set valueClause [join $sqlVars ,]
                    }
                    append valueClause ") ;"
                    sqlitedb eval $statement$valueClause
                }
            }
        }
    } result opts

    sqlitedb close
    return -options $opts $result
}

proc ::ral::loadFromSQLite {filename {ns ::}} {
    if {![file exists $filename]} {
        error "no such file, \"$filename\""
    }
    set ns [DeserialNS $ns]

    package require sqlite3
    sqlite3 [namespace current]::sqlitedb $filename
    catch {
        # First we query the meta data and rebuild the relvars and constraints.
        set version [sqlitedb onecolumn {select Vnum from __ral_version ;}]
        if {![package vsatisfies $version [getCompatVersion]]} {
            error "incompatible version number, \"$version\",\
                current library version is, \"[getVersion]\""
        }
        set relvarNames [sqlitedb eval {select Vname from __ral_relvar}]
        sqlitedb transaction {
            # The relvars
            foreach relvarName $relvarNames {
                createRelvarFromSQLite sqlitedb $ns $relvarName
            }
            # The association constraints
            foreach {cname referring referringAttrs refToSpec refTo refToAttrs\
                    referringSpec} [sqlitedb eval {select Cname, Referring,\
                    ReferringAttrs, RefToSpec, RefTo, RefToAttrs, ReferringSpec\
                    from __ral_association}] {
                ::ral::relvar association $ns$cname $ns$referring\
                        $referringAttrs $refToSpec $ns$refTo $refToAttrs\
                        $referringSpec
            }
            # The partition constraints
            foreach {cname superType superAttrs} [sqlitedb eval {select Cname,\
                    SuperType, SuperAttrs from __ral_partition}] {
                set cmd [list ::ral::relvar partition $ns$cname $ns$superType\
                        $superAttrs]
                foreach {subtype subattr} [sqlitedb eval {select Subtype,\
                        SubAttrs from __ral_subtype where Cname = $cname}] {
                    lappend cmd $ns$subtype $subattr
                }
                eval $cmd
            }
            # The correlation contraints
            foreach {cname isComplete assocRelvar oneRefAttrs oneRefSpec\
                    oneRelvar oneAttrs otherRefAttrs otherRefSpec otherRelvar\
                    otherAttrs}\
                    [sqlitedb eval {select Cname, IsComplete, AssocRelvar,\
                    OneRefAttrs, OneRefSpec, OneRelvar, OneAttrs,\
                    OtherRefAttrs, OtherRefSpec, OtherRelvar, OtherAttrs\
                    from __ral_correlation}] {
                set corrCmd [list\
                    ::ral::relvar correlation $ns$cname\
                        $ns$assocRelvar $oneRefAttrs $oneRefSpec $ns$oneRelvar\
                        $oneAttrs $otherRefAttrs $otherRefSpec $ns$otherRelvar\
                        $otherAttrs]
                if {$isComplete} {
                    set corrCmd [linsert $corrCmd 3 -complete]
                }
                eval $corrCmd
            }
            # The procedural contraints
            foreach {cname script}\
                [sqlitedb eval {select Cname, Script from __ral_procedural}] {
                # Create a list of the participating relvars
                set partRels [list]
                foreach participantRelvar\
                    [sqlitedb eval {select ParticipantRelvar from\
                        __ral_proc_participant where (Cname = $cname)}] {
                    lappend partRels $ns$participantRelvar
                }
                ::ral::relvar procedural $ns$cname {*}$partRels $script
            }
        }
        # Now insert the data from the tables.
        # We are careful here not to depend upon SQL column ordering.
        relvar eval {
            foreach vname $relvarNames {
                loadRelvarFromSQLite sqlitedb $ns $vname insert
            }
        }
    } result opts

    sqlitedb close
    return -options $opts $result
}
# Merge data from a SQLite store of relvars.
# All relvars that are in the database but not currently defined are created.
# All relvars whose names and headings match currently defined relvars
# will have their relation values unioned with those in the file.
proc ::ral::mergeFromSQLite {filename {ns ::}} {
    package require sqlite3

    if {![file exists $filename]} {
        error "no such file, \"$filename\""
    }
    set ns [DeserialNS $ns]
    sqlite3 [namespace current]::sqlitedb $filename
    catch {
        # First we query database to find out its version and the
        # relvars it contains.
        set version [sqlitedb onecolumn {select Vnum from __ral_version ;}]
        if {![package vsatisfies $version [getCompatVersion]]} {
            error "incompatible version number, \"$version\",\
                current library version is, \"[getVersion]\""
        }
        # Compare the contents of the database to that already present.  We are
        # looking for things in the database that are not already present.
        # First we look for new relvars.
        set dbrelvars [sqlitedb eval {select Vname from __ral_relvar}]
        set newRelvars [list]
        set sameTypeRelvars [list]
        set mismatched [list]
        foreach dbrelvar $dbrelvars {
            set qualRelvar ${ns}$dbrelvar
            if {![::ral::relvar exists $qualRelvar]} {
                # New relvar, create it.
                createRelvarFromSQLite sqlitedb $ns $dbrelvar
                lappend newRelvars $dbrelvar
            } else {
                # Determine if the heading of the relvar in the database
                # matches that of the existing relvar. We will do this
                # by creating an empty relation with each heading and
                # invoke "relation issametype" to do the comparison.
                # If that returns true, the we will keep a list of those
                # that have the same type.
                set heading [sqlitedb eval {select Aname, Type from\
                        __ral_attribute where Vname = $dbrelvar}]
                set dbrel [::ral::relation create $heading]
                if {[::ral::relation issametype $dbrel\
                        [relvar set $qualRelvar]]} {
                    lappend sameTypeRelvars $dbrelvar
                }
            }
        }
        # Next do the same for constraints.
        # We consider each different type of association.
        # First associations
        set dbassocs [sqlitedb eval {select Cname from __ral_association}]
        foreach dbassoc $dbassocs {
            if {![::ral::relvar constraint exists ${ns}$dbassoc]} {
                createAssocFromSQLite sqlitedb $ns $dbassoc
            }
        }
        # Next partitions
        set dbparts [sqlitedb eval {select Cname from __ral_partition}]
        foreach dbpart $dbparts {
            if {![::ral::relvar constraint exists ${ns}$dbpart]} {
                createPartitionFromSQLite dsqlitedb $ns $dbpart
            }
        }
        # Next correlations
        set dbcorrels [sqlitedb eval {select Cname from __ral_correlation}]
        foreach dbcorrel $dbcorrels {
            if {![::ral::relvar constraint exists ${ns}$dbcorrel]} {
                createCorrelFromSQLite dbsqlitedb $ns $dbcorrel
            }
        }
        # Finally procedural constraints
        set dbprocs [sqlitedb eval {select Cname from __ral_procedural}]
        foreach dbproc $dbprocs {
            if {![::ral::relvar constraint exists ${ns}$dbproc]} {
                createProcFromSQLite dbsqlitedb $ns $dbproc
            }
        }
        # Now we populate the merge from the SQLite data base.  Newly created
        # relvars, as contained in the "newRelvars" variable can be loaded
        # straight in. For other relvars, we must verify that he heading is the
        # same as that in the database. For those that match, we union in the
        # data base contents.  N.B. that we use "uinsert" to merge in data base
        # contents.  Using "relvar union" has some corner cases that are not
        # what you might expect (i.e. the union might fail identity contraints
        # for those tuples which have the same identifier values but different
        # values for the other attributes.
        relvar eval {
            foreach dbrelvar $dbrelvars {
                if {$dbrelvar in $newRelvars} {
                    loadRelvarFromSQLite sqlitedb $ns $dbrelvar insert
                } elseif {$dbrelvar in $sameTypeRelvars} {
                    loadRelvarFromSQLite sqlitedb $ns $dbrelvar uinsert
                }
                # Else it must be that the type of the relvar in the database
                # did not match that of an existing relvar. So it is skipped.
            }
        }
        set mismatched
    } result opts

    sqlitedb close
    return -options $opts $result
}

proc ::ral::createRelvarFromSQLite {db ns relvarName} {
    set heading [list]
    foreach {aname type} [$db eval {select Aname, Type from\
            __ral_attribute where Vname = $relvarName}] {
        lappend heading $aname $type
    }
    set idents [dict create]
    foreach {idnum attrName} [$db eval {select IdNum, Attr\
            from __ral_identifier where Vname = $relvarName}] {
        dict lappend idents $idnum $attrName
    }
    ::ral::relvar create $ns$relvarName $heading {*}[dict values $idents]
}

proc ::ral::loadRelvarFromSQLite {db ns relvarName operation} {
    set sqlTableName [mapNamesToSQL $relvarName]
    set qualName $ns$relvarName
    set heading [relation heading [relvar set $qualName]]
    set attrNames [relation attributes [relvar set $qualName]]
    set sqlColNames [mapNamesToSQL {*}$attrNames]
    $db eval "select [join $sqlColNames {, }]\
            from $sqlTableName" valArray {
        # Build up the insert tuple
        set insert [list]
        # Note we have to adjust the values for Tuple and
        # Relation valued attributes. The "relvar insert"
        # command just wants the bodies of the values.
        foreach {attr type} $heading sqlCol $sqlColNames {
            lappend insert $attr
            switch -glob -- $type {
                Relation* {
                    lappend insert\
                            [relation body $valArray($sqlCol)]
                }
                Tuple* {
                    lappend insert [tuple get $valArray($sqlCol)]
                }
                default {
                    lappend insert $valArray($sqlCol)
                }
            }
        }
        ::ral::relvar $operation $qualName $insert
    }
}

proc ::ral::createAssocFromSQLite {db ns assocName} {
    # Fetch association from the data base.
    set assocElems [$db eval {select Cname, Referring,\
            ReferringAttrs, RefToSpec, RefTo, RefToAttrs, ReferringSpec\
            from __ral_association where Cname = $assocName}]
    # Put the keyword at the beginning.
    set assocElems [linsert $assocElems 0 association]
    # Create the association making the names fully qualified.
    eval ::ral::relvar [setRelativeConstraintInfo $ns $assocElems]
}

proc ::ral::createPartitionFromSQLite {db ns partitionName} {
    set partitionElems [$db eval {select Cname, SuperType, SuperAttrs\
            from __ral_partition where Cname = $partitionName}]
    foreach {subtype subattr} [$db eval {select Subtype,\
            SubAttrs from __ral_subtype where Cname = $cname}] {
        lappend partitionElems $subtype $subattr
    }
    set paritionElems [linsert $partitionElems 0 partition]
    eval ::ral::relvar [setRelativeConstraintInfo $ns $partitionElems]
}

proc ::ral::createCorrelFromSQLite {db ns correlName} {
    set correlElems [$db eval {select Cname, IsComplete, AssocRelvar,\
            OneRefAttrs, OneRefSpec, OneRelvar, OneAttrs,\
            OtherRefAttrs, OtherRefSpec, OtherRelvar, OtherAttrs\
            from __ral_correlation where Cname = $correlName}]
    if {[lindex $correlElems 2] == "true"} {
        lset correlElems 2 -complete
    }
    set correlElems [linsert $correlElems 0 correlation]
    eval ::ral::relvar [setRelativeConstraintInfo $ns $correlElems]
}

proc ::ral::createProcFromSQLite {db ns procName} {
    set script [$db eval {select Script from __ral__procedural\
            where Cname = $procName}]
    set participants [$db eval {select ParticipantRelvar from\
            __ral__proc_participant where Cname = $procName}]
    set procElems [list procedural $procName {*}$participants $scrip]
    eval ::ral::relvar [setRelativeConstraintInfo $ns $procElems]
}

proc ::ral::dump {{pattern *}} {
    set pattern [SerializePattern $pattern]
    set result {}
    set names [lsort [relvar names $pattern]]

    append result "# Generated via ::ral::dump\n"
    append result "package require ral [getVersion]\n"

    # Convert the names to be relative
    # Strip any leading "::" from names so that if the script is
    # sourced in, names are created relative to the namespace of the
    # source.
    foreach name $names {
        set rName [namespace tail $name]
        append result "::ral::relvar create $rName\
            [list [relation heading [relvar set $name]]]\
            [relvar identifiers $name]" \n
    }

    foreach cname [lsort [relvar constraint names $pattern]] {
        append result "::ral::relvar [getRelativeConstraintInfo $cname]\n"
    }

    # perform the inserts inside of a transaction.
    append result "::ral::relvar eval \{\n"
    foreach name $names {
        append result\
            "    ::ral::relvar set [namespace tail  $name]\
                [list [relvar set $name]]" \n
    }
    append result "\}"

    return $result
}

proc ::ral::dumpToFile {fileName {pattern *}} {
    set chan [::open $fileName w]
    catch {puts $chan [dump $pattern]} result opts
    ::close $chan
    return -options $opts $result
}

proc ::ral::csv {relValue {sortAttr {}} {noheading 0}} {
    package require csv

    set m [relation2matrix $relValue $sortAttr $noheading]
    catch {::csv::report printmatrix $m} result opts
    $m destroy
    return -options $opts $result
}

proc ::ral::csvToFile {relValue fileName {sortAttr {}} {noheading 0}} {
    package require csv

    set m [relation2matrix $relValue $sortAttr $noheading]
    set chan [::open $fileName w]
    catch {::csv::report printmatrix2channel $m $chan} result opts
    $m destroy
    ::close $chan
    return -options $opts $result
}

proc ::ral::FindNullableAttrs {name} {
    # Find the attributes of the relvar that participate in
    # a conditional association. This is the one and only case
    # where we will allow the attribute to be NULL.
    set condAttrs [list]
    foreach constraint [relvar constraint member $name] {
        set info [lassign [relvar constraint info $constraint] ctype]
        if {$ctype eq "association"} {
            lassign $info cname rfering a1 c1 refto a2 c2
            if {$name eq $rfering && $c2 eq "?"} {
                lappend condAttrs {*}$a1
            }
        }
    }
    return $condAttrs
}

# Returns a string that can be fed to SQLite that will create the necessary
# tables and contraints for the relvars that match "pattern".
proc ::ral::sqlSchema {{pattern *}} {
    set names [lsort [relvar names $pattern]]

    append result "-- Generated via ::ral::sqlSchema\n"
    append result "pragma foreign_keys=ON;\n"

    # Convert the names to be relative.  Strip any leading namespace qualifiers
    # from names so that if the script is sourced in, names are created
    # relative to the namespace of the source.
    foreach name $names {
        set indices {}
        set value [relvar set $name]
        set baseName [namespace tail $name]
        set sqlBaseName [mapNamesToSQL $baseName]

        append result "create table $sqlBaseName (\n"
        set condAttrs [FindNullableAttrs $name]
        # Define table attributes.
        foreach {attr type} [relation heading $value] {
            # N.B. need to map type from Tcl type to SQL type
            # Need to figure out what to do with Tuple and Relation type
            # attributes.
            append result "    [mapNamesToSQL $attr] [mapTypeToSQL $type]"
            if {$attr ni $condAttrs} {
                append result " not null"
            }
            append result ",\n"
        }
        # Define identification constraints.
        set idNum 0
        foreach id [relvar identifiers $name] {
            set id [join [mapNamesToSQL {*}$id] {, }]
            append result "    constraint ${sqlBaseName}_ID[incr idNum]\
                    unique ($id),\n"
        }
        # Define the referential constraints.
        foreach constraint [relvar constraint member $name] {
            set info [relvar constraint info $constraint]
            set info [lassign $info ctype]
            switch -exact -- $ctype {
                association {
                    lassign $info cname rfering a1 c1 refto a2 c2
                    set rfering [namespace tail $rfering]
                    if {$rfering eq $baseName} {
                        set a1 [join [mapNamesToSQL {*}$a1] {, }]
                        set cname [mapNamesToSQL [namespace tail $cname]]
                        set refto [namespace tail $refto]
                        append result "    constraint ${sqlBaseName}_$cname\
                            foreign key ($a1)\
                            references $refto\
                                ([join [mapNamesToSQL {*}$a2] {, }])\n"\
                            "        on delete [expr {$c2 eq "?" ?\
                                "set null" : "cascade"}]\
                            on update cascade\
                            deferrable initially deferred,\n"
                        append indices "create index\
                            ${sqlBaseName}_${cname}INDEX\
                            on $sqlBaseName ($a1) ;\n"
                    }
                }
                partition {
                    # The best we can do for a partition is to make it one to
                    # one conditional subtypes referring to the supertype. This
                    # will not enforce disjointedness, but it's SQL after all.
                    set subs [lassign $info cname super sattrs]
                    foreach {subrel subattrs} $subs {
                        set rfering [namespace tail $subrel]
                        if {$baseName eq $rfering} {
                            set cname [mapNamesToSQL [namespace tail $cname]]
                            set refto [namespace tail $super]
                            set subattrs [join [mapNamesToSQL {*}$subattrs] {, }]
                            append result "    constraint ${sqlBaseName}_$cname\
                                foreign key ($subattrs)\
                                references $refto\
                                    ([join [mapNamesToSQL {*}$sattrs] {, }])\n"\
                                "        on delete cascade on update cascade\
                                deferrable initially deferred,\n"
                            append indices "create index\
                                ${sqlBaseName}_${cname}INDEX on\
                                $sqlBaseName ($subattrs) ;\n"
                            break
                        }
                    }
                }
                correlation {
                    # The correlation constraint shows up as two foreign
                    # key references on the correlation table.
                    lassign $info cname correl ref1Attr c1 rel1 rel1Attr\
                        ref2Attr c2 rel2 rel2Attr
                    set correl [mapNamesToSQL [namespace tail $correl]]
                    if {$baseName eq $correl} {
                        set cname [mapNamesToSQL [namespace tail $cname]]
                        set rel1 [namespace tail $rel1]
                        set ref1Attr [join [mapNamesToSQL {*}$ref1Attr] {, }]
                        append result "    constraint\
                                ${sqlBaseName}_${cname}_${rel1}\
                            foreign key ($ref1Attr)\
                            references $rel1\
                                ([join [mapNamesToSQL {*}$rel1Attr] {, }])\n"\
                            "        on delete cascade on update cascade\
                            deferrable initially deferred,\n"
                        append indices "create index\
                            ${sqlBaseName}_${cname}INDEX0 on $sqlBaseName\
                            ($ref1Attr) ;\n"

                        set rel2 [namespace tail $rel2]
                        set ref2Attr [join [mapNamesToSQL {*}$ref2Attr] {, }]
                        append result "    constraint\
                                ${sqlBaseName}_${cname}_${rel2}\
                            foreign key ($ref2Attr)\
                            references $rel2\
                                ([join [mapNamesToSQL {*}$rel2Attr] {, }])\n"\
                            "        on delete cascade on update cascade\
                            deferrable initially deferred,\n"
                        append indices "create index\
                            ${sqlBaseName}_${cname}INDEX1 on $sqlBaseName\
                            ($ref2Attr) ;\n"
                    }
                }
                procedural {
                    # Procedural constraints do not add anything to the SQL
                    # schema since they are not "foreign key" oriented.
                }
                default {
                    error "unknown constraint type, \"$ctype\""
                }
            }
        }
        set result [string trimright $result ,\n]
        append result ") ;\n"
        append result $indices
    }

    return $result
}

# With Tcl 8.5, then we can supply some "aggregate scalar functions" that
# are useful and have "expr" syntax.
# Count the number of tuples in a relation
proc ::tcl::mathfunc::rcount {relation} {
    return [::ral::relation cardinality $relation]
}
# Count the number of distinct values of an attribute in a relation
proc ::tcl::mathfunc::rcountd {relation attr} {
    return [::ral::relation cardinality\
        [::ral::relation project $relation $attr]]
}
# Compute the sum over an attribute
proc ::tcl::mathfunc::rsum {relation attr} {
    set result 0
    foreach v [::ral::relation list $relation $attr] {
        incr result $v
    }
    return $result
}
# Compute the sum over the distinct values of an attribute.
proc ::tcl::mathfunc::rsumd {relation attr} {
    set result 0
    ::ral::relation foreach v [::ral::relation list\
        [::ral::relation project $relation $attr]] {
        incr result $v
    }
    return $result
}
# Compute the average of the values of an attribute
proc ::tcl::mathfunc::ravg {relation attr} {
    return [expr {rsum($relation, $attr) / rcount($relation)}]
}
# Compute the average of the distinct values of an attribute
proc ::tcl::mathfunc::ravgd {relation attr} {
    return [expr {rsumd($relation, $attr) / rcount($relation)}]
}
# Compute the minimum. N.B. this does not handle "empty" relations properly.
proc ::tcl::mathfunc::rmin {relation attr} {
    set values [::ral::relation list $relation $attr]
    set min [lindex $values 0]
    foreach v [lrange $values 1 end] {
        if {$v < $min} {
            set min $v
        }
    }
    return $min
}
# Again, empty relations are not handled properly.
proc ::tcl::mathfunc::rmax {relation attr} {
    set values [::ral::relation list $relation $attr]
    set max [lindex $values 0]
    foreach v [lrange $values 1 end] {
        if {$v > $max} {
            set max $v
        }
    }
    return $max
}

# PRIVATE PROCS

# Clean up a namespace name for deserialization and append trailing colons.
# Eval a null script in the namespace to make sure to create it
# paying careful attention to non-qualified names.
proc ::ral::DeserialNS {ns} {
    if {[string range $ns 0 1] ne "::"} {
        set callerns [string trimright [uplevel 2 namespace current] :]
        set ns ${callerns}::$ns
    }
    set ns [string trimright $ns :]::
    namespace eval $ns {}
    return $ns
}

proc ::ral::SerializePattern {pattern} {
    if {[string range $pattern 0 1] ne "::"} {
        set callerns [string trimright [uplevel 2 namespace current] :]
        set pattern ${callerns}::$pattern
    }
    return $pattern
}

# Returns a relation value with the heading {Qualified string Base string}
# that represents the fully qualified relvar names and their corresponding
# leaf base names. Checks that all the relvars that "names" have unique base
# names.
proc ::ral::BaseNames {names} {
    set rnames [relation extend [relation fromlist $names Qualified string]\
        rn Base string {
            [namespace tail [tuple extract $rn Qualified]]
        }]

    set bnames [relation list [relation project $rnames Base]]

    if {[relation cardinality $rnames] != [llength $bnames]} {
        error "set of relvar names results in duplicated base names, \"$names\""
    }

    return $rnames
}

# Add heading rows to the matrix.
proc ::ral::addHeading {matrix heading} {
    set attrNames [list]
    set attrTypes [list]
    foreach {name type} $heading {
        lappend attrNames $name
        # For Relation and Tuple types, just use the keyword.
        # The components of the types will be apparent from the headings
        # of the relation or tuple valued attributes. This saves quite
        # a bit of column space for these types of nested attributes.
        lappend attrTypes [lindex $type 0]
    }
    $matrix add row $attrNames
    $matrix add row $attrTypes

    return
}

# Returns a relation mapping attribute names to a formatting function.
# Ordinary scalar values attributes are not contained in the mapping. Relation
# and tuple valued attributes will be in the relation keyed by the attribute
# name with values corresponding to the "relformat" or "tupleformat" command.
proc ::ral::getFormatMap {heading} {
    set attrReportMap {{AttrName string AttrFunc string} {}}
    foreach {name type} $heading {
        set typeKey [lindex $type 0]
        if {$typeKey eq "Tuple"} {
            set attrReportMap [relation insert $attrReportMap\
                    [list AttrName $name AttrFunc ::ral::tupleformat]]
        } elseif {$typeKey eq "Relation"} {
            set attrReportMap [relation insert $attrReportMap\
                    [list AttrName $name AttrFunc ::ral::relformat]]
        }
    }
    return $attrReportMap
}

# Add tuple values to a matrix
proc ::ral::addTuple {matrix tupleValue attrMap} {
    set values [list]
    foreach {attr value} [tuple get $tupleValue] {
        set mapping [relation restrictwith $attrMap {$AttrName eq $attr}]
        if {[relation isnotempty $mapping]} {
            set attrfunc [relation extract $mapping AttrFunc]
            set value [$attrfunc $value]
        } else {
            # Limit the width of scalar values. We use the "textutil"
            # package to wrap the text to "maxColLen" characters.
            variable maxColLen
            if {[string length $value] > $maxColLen} {
                package require textutil
                set value [::textutil::adjust $value -justify left\
                    -length $maxColLen -strictlength true]
            }
        }
        lappend values $value
    }
    $matrix add row $values

    return
}

proc ::ral::tupleValue {tuple} {
    set result [list]
    foreach {attr type} [tuple heading $tuple]\
        {attr value} [tuple get $tuple] {
        switch [lindex $type 0] {
            Tuple {
                set value [tupleValue $value]
            }
            Relation {
                set value [relationValue $value]
            }
        }
        lappend result $attr $value
    }

    return $result
}

proc ::ral::relationValue {relation} {
    set result [list]
    relation foreach r $relation {
        lappend result [tupleValue [relation tuple $r]]
    }

    return $result
}

proc ::ral::mkCheckVersion {dbName} {
    # Check that a "version" view exists and that the information
    # is consistent before we proceed.
    set views [::mk::file views $dbName]
    if {[lsearch $views __ral_version] < 0} {
        error "Cannot find TclRAL catalogs:\
            file may not contain relvar information"
    }
    set result [tuple create\
        {Version_ral string Date_ral string Comment_ral string}\
        [::mk::get $dbName.__ral_version!0]]
    set verNum [tuple extract $result Version_ral]
    if {![package vsatisfies $verNum [getCompatVersion]]} {
        error "incompatible version number, \"$verNum\",\
            current library version is, \"[getVersion]\""
    }
    return $result
}

proc ::ral::mkHeading {attr type} {
    switch [lindex $type 0] {
        Tuple -
        Relation {
            set subHead [list]
            foreach {subattr subtype} [lindex $type 1] {
                lappend subHead [mkHeading $subattr $subtype]
            }
            set result [list $attr $subHead]
        }
        default {
            set result $attr
        }
    }
    return $result
}

proc ::ral::mkStoreTuple {cursor tuple} {
    foreach {attr type} [tuple heading $tuple] {attr value} [tuple get $tuple] {
        switch -exact [lindex $type 0] {
            Tuple {
                ::mk::cursor create tupCursor $cursor.$attr
                ::mk::row insert $tupCursor
                mkStoreTuple $tupCursor [tuple extract $tuple $attr]
            }
            Relation {
                mkStoreRelation $cursor.$attr [tuple extract $tuple $attr]
            }
            default {
                ::mk::set $cursor $attr $value
            }
        }
    }
    return
}

proc ::ral::mkStoreRelation {cursor relation} {
    ::mk::cursor create relCursor $cursor 0
    ::mk::row insert $relCursor [relation cardinality $relation]
    relation foreach r $relation {
        mkStoreTuple $relCursor [relation tuple $r]
        ::mk::cursor incr relCursor
    }
    return
}

proc ::ral::mkLoadTuple {cursor heading} {
    set value [list]
    foreach {attr type} $heading {
        switch -exact [lindex $type 0] {
            Tuple {
                lappend value $attr [mkLoadTuple $cursor.$attr!0\
                        [lindex $type 1]]
            }
            Relation {
                lappend value $attr [mkLoadRelation $cursor.$attr\
                        [lindex $type 1]]
            }
            default {
                lappend value $attr [::mk::get $cursor $attr]
            }
        }
    }
    return $value
}

proc ::ral::mkLoadRelation {cursor heading} {
    set value [list]
    ::mk::loop rCursor $cursor {
        lappend value [mkLoadTuple $rCursor $heading]
    }
    return $value
}

namespace eval ::ral {
    # The set of indices into the constraint information where fully qualified
    # path names exist.
    variable assocIndices {1 2 5}
    variable correlIndices {1 2 5 9}
    variable compCorrelIndices {2 3 6 10}
}

proc ::ral::getRelativeConstraintInfo {cname} {
    set cinfo [relvar constraint info $cname]
    switch -exact [lindex $cinfo 0] {
        association {
            variable assocIndices
            foreach index $assocIndices {
                lset cinfo $index [namespace tail [lindex $cinfo $index]]
            }
        }
        partition {
            lset cinfo 1 [namespace tail [lindex $cinfo 1]]
            lset cinfo 2 [namespace tail [lindex $cinfo 2]]
            for {set index 4} {$index < [llength $cinfo]} {incr index 2} {
                lset cinfo $index [namespace tail [lindex $cinfo $index]]
            }
        }
        correlation {
            variable correlIndices
            variable compCorrelIndices
            set cIndices [expr {[lindex $cinfo 1] eq "-complete" ?\
                $compCorrelIndices : $correlIndices}]
            foreach index $cIndices {
                lset cinfo $index [namespace tail [lindex $cinfo $index]]
            }
        }
        procedural {
            set endIndex [expr {[llength $cinfo] - 1}]
            for {set index 1} {$index < $endIndex} {incr index} {
                lset cinfo $index [namespace tail [lindex $cinfo $index]]
            }
        }
        default {
            error "unknown constraint type, \"[lindex $cinfo 0]\""
        }
    }
    return $cinfo
}

proc ::ral::setRelativeConstraintInfo {ns cinfo} {
    switch -exact -- [lindex $cinfo 0] {
        association {
            variable assocIndices
            foreach index $assocIndices {
                lset cinfo $index ${ns}[lindex $cinfo $index]
            }
        }
        partition {
            lset cinfo 1 ${ns}[lindex $cinfo 1]
            lset cinfo 2 ${ns}[lindex $cinfo 2]
            for {set index 4} {$index < [llength $cinfo]} {incr index 2} {
                lset cinfo $index ${ns}[lindex $cinfo $index]
            }
        }
        correlation {
            variable correlIndices
            variable compCorrelIndices
            set cIndices [expr {[lindex $cinfo 1] eq "-complete" ?\
                $compCorrelIndices : $correlIndices}]
            foreach index $cIndices {
                lset cinfo $index ${ns}[lindex $cinfo $index]
            }
        }
        procedural {
            set endIndex [expr {[llength $cinfo] - 1}]
            for {set index 1} {$index < $endIndex} {incr index} {
                lset cinfo $index\
                        ${ns}[lindex $cinfo $index]
            }
        }
        default {
            error "unknown constraint type, \"[lindex $cinfo 0]\""
        }
    }

    return $cinfo
}

# Handling SQL names is a pain. We must be prepared to deal with all the
# keywords and other characters. You might be tempted to use the SQL quoting
# mechanisms but this interfers with Tcl quoting in strange ways especially
# with embedded white space. We are luck in the sense that we only have to map
# the names to something SQL likes and not the inverse. It will make the names
# in a SQLite database appear different than those in relvars, but they are not
# so different as to be unrecognizable.
#
# So the convention is that SQL keywords have an "_K" append to them and all
# other names have anything that is _not_ an alphanumber or the underscore
# changed to an underscore.

namespace eval ::ral {
    # This list is sorted alphabetically and the "lsearch" invocation below
    # depends upon that sorting.
    variable sqlKeywords {
        ABORT ACTION ADD AFTER ALL ALTER ANALYZE AND AS ASC ATTACH AUTOINCREMENT
        BEFORE BEGIN BETWEEN BY
        CASCADE CASE CAST CHECK COLLATE COLUMN COMMIT CONFLICT CONSTRAINT
            CREATE CROSS CURRENT_DATE CURRENT_TIME CURRENT_TIMESTAMP
        DATABASE DEFAULT DEFERRABLE DEFERRED DELETE DESC DETACH DISTINCT DROP
        EACH ELSE END ESCAPE EXCEPT EXCLUSIVE EXISTS EXPLAIN
        FAIL FOR FOREIGN FROM FULL
        GLOB GROUP
        HAVING
        IF IGNORE IMMEDIATE IN INDEX INDEXED INITIALLY INNER INSERT INSTEAD
            INTERSECT INTO IS ISNULL
        JOIN
        KEY
        LEFT LIKE LIMIT
        MATCH
        NATURAL NO NOT NOTNULL NULL
        OF OFFSET ON OR ORDER OUTER
        PLAN PRAGMA PRIMARY
        QUERY
        RAISE RECURSIVE REFERENCES REGEXP REINDEX RELEASE RENAME REPLACE
            RESTRICT RIGHT ROLLBACK ROW
        SAVEPOINT SELECT SET
        TABLE TEMP TEMPORARY THEN TO TRANSACTION TRIGGER
        UNION UNIQUE UPDATE USING
        VACUUM VALUES VIEW VIRTUAL
        WHEN WHERE WITH WITHOUT
    }
}

proc ::ral::mapNamesToSQL {args} {
    variable sqlKeywords
    set newNames [list]
    foreach name $args {
        if {[lsearch -exact -sorted -increasing -ascii -nocase $sqlKeywords\
                $name] != -1} {
            set name ${name}_K
        } else {
            set name [regsub -all -- {[^[:alnum:]_]} $name _]
        }
        lappend newNames $name
    }
    return $newNames
}

proc ::ral::mapTypeToSQL {type} {
    variable sqlTypeMap
    return [expr {[info exists sqlTypeMap($type)] ?\
            $sqlTypeMap($type) : "text"}]
}

package provide ral 0.12.2
ELF          >    К      @                @ 8  @                                                              #     #            \                    -     -#     -#                                                $       $              Ptd   Н     Н     Н     l      l             Qtd                                                  Rtd        #     #     x      x                      GNU *f%꘲B&BH                 F@O$G
  b  b 	@  @Fډf-   %(  	  IP @ $
@ B
ti9e  !fD DQ    A@" "A  "D$ @ F@D ( Q9ơ #0@2 ,	y X `&@G.2  >'F!B! @ @@lP! DD#!r^AQ0$	 J`
%i*                                                                        #               %           '   )   *   ,   -   .   /   0   2               3   7       8   ;       <   ?       B   C       D       E   F   G   I                       J       L   M   O       P       Q   R   S       T   U       V   W   X       Y   [   ^       _                   `   a   b   d   e   g   h   k   m       n   q       s           u           v   w   y   {   |       ~                                                                                                                                                                                                                                                                                                                                                                                                                             	  
                                 !  $  &      )  *      h*
MiSXndtt
<[N(-
%_ i#;Q+Cp6~7)XGzTN
	̟sgZ['ٽx![)ljёȪ/?MGn-K4C|ͧ%7S6t4
KV{Yk"TM]Mpx!|.w"1XPSw˭YwO:.s!2s~$s@"}z
S{H%e|<\+7+ljT\:->YGi݁%!otBCEoeYH#DY/\\8|-)`S>>_E0Q"MT#C솟Ce[do9hVAy̪V+igf8V+ұŏTqP8iv
[mPt	_O
fIж(Uvn>I*8U;1e'c]OlCDM۝9vPAOy*l%-1KoUJ<b 4o;׍]E@d(ύ:J}Ye_xJ mpUҎAG**D	:X;1(@qX/lrl<Ǭo^xb=!{=U>V;dx"?lۡj'm]%
[~Mlz*o\[Jw1(eӻeS:9y+	'm0S8J!k`|˥OZDVFAu?]xqJ6
(v?"Yx7@GJ2Hw+_i;ƤsG JAʾ*jK	penנ}DUWz
v^pg+ŐƯ<	2_tegl󡕦EjnQRQn>
mIp#߷o&zboJV6$Q}DAJD7:$!ZHzxhPOmB.h5y@vcLl]K'T*C-mo&@"&.4/m|e                                                                                             7                     
                                           k                                                                                                                                                       a                       8                       R   "                   3    @I            h                     f     V       0    \     3       
    t                p                             
                     \     F           0_            ?    t     )       b           g       ~          J       
    `     5           u     !       Y     c            Q    p      D           w     i                 O           ;     -           P      n       s    @c     }           x     i       
    @      s       4                                 U    b     P                :       ~    _     @            `      d       V          C           a     E           0     }           @                       l       \          h           {#                 u     ,                                  A           p     e           Y     O           pb     t           Z     :       k    ^            +     k     A           j     "       G    0      S            \     B                b                t       3    0g            W    ]           
    Pg     C                                                      d    H8#     	       
    C     Z      S	    p      4           b     n       v         y       #          B           0             y          S           
            K                X    g     2       "     f     j       v                 Z                         k      
    K                 d            
    ph     H       /    w     O           :#                  W     J           `                 0     P                      t    	               p                @=                     v      
                                l    @x     Q       t    0h     4           J     @      L    p      X           k     ~       ^         F       z	          )          `s     -           s     $                p          Ќ                y                О                 P[     Y           j     5       &    0      C       e
    PJ     7       ;          C       u     Л                     ;       
    `                	     C                 k       ,                   `                `N                                @      ^       E    [                 @`            h    0     {       W    P	     J       
     M     X          0S                 f     :       `          C                R       m           C           P@            o
    `     \           0{     A            m            o          X       k    i               0      &                           е                 T                    1       	                s                     0                 Y                      W                           
 p                 ;     J      h     >     d           а      S                 R           ?     :           G     I                                  d       	    @F                	                   A            x                	          R           `     %          pk     P       
    @     %       T    P      y           V            P
    pB     a      '
    X     p       R    `     K               \       P    ``           r    P      y                           @l            >    `b            :    :#     (            j            E    g     B       d                     :#                             ,    0                h     D       =          T           [     B           `      m       
    e     `                0           H            $          (       L    w     i           v     E       |
    L     t        	    _     4       P         4       	          p          W               `     T           0      T           j     2          0                 Y            s    u                     C       
                    P                ;            	    8#     (           p\     1                 N       R
    _                @     v                            p>     '          Pv     i                 n       '	     @     &                L       $
           -      4                           @      ,          4          H            B    @]     D       M           C      C    `     "       g    PW     B       5          V            s                pt                            X    t     #       &     9#     (       	    P      ^           0[                 0^            u
                                       в      b           `a     C       :         T           a     @                {       ?
    0                     ;      H                {                	          8      *    PQ           f         :       	                 #          )       <	    P;     P                :           0     Q       B    p     b           G            >     A            0    p            6                                  ;
     Y     p            v     O       5    @8#            A    @     N                                      

    c                     R           0P                                 c     ]           Pr           ~      :            u               L          g           P      9           `      d           p                       |       c	    ?     <       c    `                {           2     Z     j                              e                :     h           Z     V                        __gmon_start__ _init _fini _ITM_deregisterTMCloneTable _ITM_registerTMCloneTable __cxa_finalize _Jv_RegisterClasses Ral_Init tupleCmd relationCmd Ral_RelvarNewInfo relvarCmd Ral_SafeInit Ral_Unload Ral_SafeUnload __stack_chk_fail strcmp memcmp Ral_AttributeNewTclType strlen strcpy Ral_AttributeNewTupleType ral_tupleTypeName Ral_AttributeNewRelationType ral_relationTypeName Ral_AttributeDelete Ral_TupleHeadingUnreference Ral_AttributeDup Ral_AttributeRename Ral_AttributeTypeEqual Ral_TupleHeadingEqual Ral_AttributeEqual Ral_AttributeValueEqual Ral_RelationObjType Ral_TupleObjType Ral_TupleEqualValues Ral_RelationEqual Ral_AttributeValueCompare Ral_AttributeValueObj Ral_RelationValueStringOf Ral_TupleValueStringOf Ral_AttributeNewFromObjs Ral_TupleHeadingNewFromObj Ral_ErrorInfoSetError Ral_InterpSetError Ral_AttributeConvertValueToType Ral_RelationObjConvert Ral_ErrorInfoSetErrorObj Ral_TupleObjConvert Ral_AttributeHashValue Ral_AttributeScanName Ral_AttributeConvertName Ral_AttributeScanType Ral_TupleHeadingScan Ral_AttributeConvertType Ral_TupleHeadingConvert Ral_AttributeScanValue Ral_RelationScanValue Ral_TupleScanValue Ral_AttributeConvertValue Ral_RelationConvertValue Ral_TupleConvertValue Ral_AttributeTypeScanFlagsFree Ral_AttributeValueScanFlagsFree Ral_AttributeToString Ral_JoinMapNew Ral_JoinMapDelete Ral_JoinMapAttrReserve Ral_JoinMapTupleReserve Ral_JoinMapAddAttrMapping Ral_JoinMapAddTupleMapping Ral_JoinMapGetAttr Ral_IntVectorNewEmpty Ral_IntVectorPushBack Ral_JoinMapAttrMap Ral_IntVectorNew Ral_IntVectorStore Ral_JoinMapFindAttr Ral_JoinMapSortAttr qsort Ral_JoinMapTupleMap Ral_JoinMapTupleCounts Ral_JoinMapMatchingTupleSet Ral_IntVectorSetAdd Ral_TupleUnreference Ral_TupleAttrEqual Ral_TupleHashAttr Ral_TupleHash Ral_IntVectorDelete Ral_IntVectorFetch Ral_TupleHeadingFetch Ral_IntVectorExchange Ral_TupleHeadingStringOf Ral_RelationNew Ral_RelationDelete Ral_RelationReserve Ral_RelationPushBack Ral_TupleDupOrdered Ral_RelationDup Ral_TupleHeadingDup Ral_TupleNew Ral_TupleCopyValues Ral_RelationShallowCopy Ral_RelationUpdate Ral_TupleDelete Ral_RelationTimes Ral_TupleHeadingUnion Ral_RelationProject Ral_TupleHeadingSubset Ral_TupleSubset Ral_RelationGroup Ral_IntVectorSetComplement Ral_TupleHeadingNew Ral_TupleHeadingAppend Ral_TupleHeadingPushBack Ral_RelationObjNew Ral_RelationUngroup Ral_TupleHeadingFind Ral_TupleHeadingDelete Ral_RelationTag Ral_TupleHeadingExtend Ral_RelationTagWithin tupleAttrHashType Ral_RelationUnwrap Ral_RelationDisjointCopy Ral_RelationUnionCopy Ral_RelationUnion Ral_TupleHeadingNewOrderMap Ral_TupleStringOf Ral_RelationFindJoinTuples Ral_RelationJoin Ral_TupleHeadingJoin Ral_TupleObjNew Ral_RelationCompose Ral_TupleHeadingCompose Ral_RelationSemiJoin Ral_RelationSemiMinus Ral_RelationTclose Ral_PtrVectorNew Ral_PtrVectorPushBack memset Ral_PtrVectorFetch Ral_PtrVectorDelete Ral_RelationSort Ral_IntVectorFillConsecutive Ral_RelationFind Ral_RelationIntersect Ral_RelationMinus Ral_RelationDivide Ral_TupleHeadingCommonAttributes Ral_RelationExtract Ral_RelationErase memmove Ral_RelationCompare Ral_RelationNotEqual Ral_RelationSubsetOf Ral_RelationProperSubsetOf Ral_RelationSupersetOf Ral_RelationProperSupersetOf Ral_RelationRenameAttribute Ral_TupleHeadingStore Ral_RelationScan Ral_RelationConvert Ral_RelationStringOf Ral_ErrorInfoSetCmd Ral_RelationObjParseJoinArgs Ral_InterpErrorInfo Ral_InterpErrorInfoObj Ral_RelationInsertTupleValue Ral_TupleDup Ral_TupleHeadingAttrsFromObj Ral_TupleSetFromValueList Ral_TupleHeadingAttrsFromVect Ral_TupleHeadingIndexOf Ral_IntVectorFind Ral_TupleUpdateAttrValue Ral_TupleAssignToVars Ral_TupleSetFromObj Ral_RelationInsertTupleObj Ral_RelationUpdateTupleObj Ral_PtrVectorBack Ral_PtrVectorPopBack Ral_RelvarDeleteInfo Ral_IntVectorOffsetOf Ral_RelvarNew Ral_RelvarDelete Ral_RelvarFind Ral_RelvarIsTransOnGoing Ral_RelvarTransModifiedRelvar Ral_PtrVectorSetAdd Ral_RelvarNewTransaction Ral_RelvarStartTransaction Ral_RelvarStartCommand Ral_RelvarDeleteTransaction Ral_ConstraintFindByName Ral_ConstraintNewAssociation Ral_ConstraintAssocCreate Ral_ConstraintNewPartition Ral_ConstraintPartitionCreate Ral_ConstraintNewCorrelation Ral_ConstraintCorrelationCreate Ral_ConstraintNewProcedural Ral_ConstraintProceduralCreate Ral_ConstraintDelete Ral_PtrVectorFind Ral_PtrVectorErase Ral_ConstraintDeleteByName Ral_RelvarDeclConstraintEval Ral_RelvarFindIdentifier Ral_IntVectorDup Ral_IntVectorSort Ral_IntVectorEqual Ral_RelvarFindById Ral_RelvarIdIndexTuple Ral_RelvarSetRelation Ral_RelvarInsertTuple Ral_RelvarIdUnindexTuple Ral_RelvarDeleteTuple Ral_RelvarRestorePrev Ral_RelvarDiscardPrev Ral_RelvarTraceAdd Ral_RelvarTraceRemove Ral_RelvarObjFindRelvar Ral_RelvarObjCopyOnShared Ral_RelvarObjInsertTuple Ral_RelvarObjEndCmd Ral_TupleDupShallow Ral_RelvarObjTraceUpdate Ral_RelvarObjKeyTuple Ral_RelvarObjUpdateTuple Ral_RelvarObjDelete Ral_RelvarObjExecSetTraces Ral_RelvarObjExecEvalTraces Ral_RelvarObjEndTrans Ral_RelvarObjTraceEvalAdd Ral_RelvarObjTraceEvalInfo Ral_RelvarObjTraceEvalRemove Ral_RelvarObjTraceVarInfo Ral_RelvarObjTraceVarRemove Ral_RelvarObjTraceVarAdd Ral_RelvarObjTraceVarSuspend Ral_RelvarObjCreateProcedural Ral_RelvarObjCreatePartition Ral_RelvarObjExecDeleteTraces Ral_RelvarObjNew Ral_RelvarObjCreateCorrelation Ral_RelvarObjConstraintNames Ral_RelvarObjConstraintDelete Ral_RelvarObjConstraintPath Ral_RelvarObjFindConstraint Ral_RelvarObjConstraintInfo Ral_RelvarObjConstraintMember Ral_RelvarObjCreateAssoc strrchr Ral_IntVectorSubsetOf Ral_PtrVectorReserve Ral_RelvarObjExecInsertTraces Ral_TuplePartialSetFromObj Ral_RelvarObjExecUpdateTraces Ral_ErrorInfoGetOption Ral_ErrorInfoGetCommand Ral_RelvarObjExecUnsetTraces Ral_TupleEqual Ral_TupleGetAttrValue Ral_TupleExtend Ral_TupleCopy Ral_TupleScan Ral_TupleConvert Ral_TupleHeadingIntersect Ral_TupleHeadingMapIndices Ral_TupleUpdateFromObj memcpy Ral_IntVectorReserve Ral_IntVectorFill Ral_IntVectorFront Ral_IntVectorBack Ral_IntVectorPopBack Ral_IntVectorInsert Ral_IntVectorErase Ral_IntVectorBooleanMap Ral_IntVectorIndexOf Ral_IntVectorIntersect Ral_IntVectorMinus Ral_IntVectorContainsAny Ral_IntVectorCopy Ral_IntVectorPrint __snprintf_chk Ral_PtrVectorDup Ral_PtrVectorFill Ral_PtrVectorStore Ral_PtrVectorFront Ral_PtrVectorInsert Ral_PtrVectorSort Ral_PtrVectorEqual Ral_PtrVectorSubsetOf Ral_PtrVectorCopy Ral_PtrVectorPrint libc.so.6 _edata __bss_start _end GLIBC_2.14 GLIBC_2.4 GLIBC_2.2.5 GLIBC_2.3.4                                                                                                                                                                                                                                                                                                                                      ii
        ui	        ti	   "      #                  #            `      #            u     #            `      #            Щ      #                  #            P      #            v     #            p      #                  #            p      #                  #            
v     #                   #                  #            `      #                  #            ~      #            0      (#            P      0#                  8#                   @#            v     H#                  P#            `      X#                  `#                  h#                 p#                   x#            p      #                  #            p      #                 #                  #            P      #            @      #                  #            v     #                  #            `      #                  #            `      #            #v     #                  #                   #                    #                   #            *v      #            @       #                    #                  ( #                  0 #            0      8 #            @      @ #            <~     P #            G~     ` #            S~     p #            [~      #            v      #            P      #            b~      #            \      #            i~      #                  #            t~      #            \      #            y~      #                   #            ~      #                   !#            ~     !#            0,     !#            ~     !#                  !#            ~     (!#            `N     0!#            ~     8!#            Z     @!#            ~     H!#                 P!#            ~     X!#            F     `!#            ~     h!#            p
     p!#            ~     x!#            ;     !#            ~     !#            0L     !#            ~     !#            #     !#            ~     !#            pZ     !#            ~     !#            pV     !#                 !#            @T     !#            ~     !#            pS     !#                 !#                 !#            ~     !#            Q      "#            ~     "#                 "#            ~     "#                  "#                 ("#            P     0"#                 8"#            0     @"#            E     H"#            0      P"#                 X"#            0I     `"#            N     h"#             H     p"#                 x"#            F     "#                 "#            pB     "#            $     "#            @     "#            +     "#                 "#            4     "#                 "#            A     "#                  "#            J     "#                  "#            T     "#            5     "#            ^     "#            P/      ##            j     ##            -     ##            p     ##             (      ##            t     (##            "     0##            {     8##             !     @##            w     H##                  P##                 X##            0     `##                 h##                  p##            ~     x##            0     ##                 ##                  ##                 ##            
     ##                 ##                  ##            <~     ##            G~      $#            q     $#                 $#            }     $#                  $#                 ($#                 0$#                 8$#                 @$#                 H$#                 P$#            ~     X$#                 `$#            M     h$#                 p$#                 x$#            p     $#            ~     $#                 $#            ʃ     $#            @     $#            T     $#            0     $#            σ     $#            Ж     $#                 $#            0     $#            ~     $#            p     $#            N     $#                  $#            `     $#                  %#            ۃ     %#                 %#            m     %#                  %#                 (%#            0     0%#                 8%#            P     @%#                 H%#                  P%#                 X%#                 `%#                 h%#                 p%#                 x%#                 %#            ~     %#                 %#                 %#            0     %#                 %#                 %#                 %#                 %#                 %#                 %#            q     %#                 %#            }     %#                  &#            M     &#                  &#                 0&#                 @&#                 `&#            b~     h&#             9     p&#            i~     x&#            @"     &#            ~     &#             8     &#            ~     &#            #     &#            ~     &#            @6     &#                 &#            `!     &#            ~     &#            4     &#            ~     &#            '     &#            ~     &#            @     &#                 &#            P$      '#            ~     '#            @3     '#                 '#            1      '#            Hv     ('#            0     0'#            $     8'#            .     @'#                 H'#            %     P'#                 X'#            ,     `'#                 h'#            )     '#                 '#                 '#                 '#            #     '#            /     '#            :     '#            D     '#            M     '#            Y     '#            i     '#            x     '#                 '#                 '#                 '#                 '#            Ϗ      (#            ޏ     (#                 (#                  (#                  (#                 ((#            &     0(#            6     8(#            H     @(#            W     H(#            c     P(#            t     X(#                 `(#                 h(#                 p(#                 x(#                 (#            ː     (#            ݐ     (#                 (#                 (#                 (#                 (#            !     (#            4     (#            H     (#            Q     (#            ^     (#            i     (#            y     (#                 (#                  )#                 )#                 )#            ɑ     )#                  )#                 ()#                 0)#                 8)#            1     @)#            x     H)#            B     P)#            T     X)#                 `)#                 h)#                 p)#            H     x)#            x     )#            m     )#            }     )#                 )#            ؔ     )#                  )#                 )#            (     )#            X     )#                 )#                 )#                 )#                  )#                 )#            h     )#                 )#                  *#                 *#            8     *#                 *#            ؒ      *#            p     (*#                 0*#                 8*#                 @*#                 H*#                  P*#            0     X*#            p     `*#                 h*#                 p*#                 *#            5     *#            v     *#            b~     *#                 *#            t~     *#            y~     *#            ~     *#                 *#                 *#            ~     *#            ~     *#            M     *#                 *#            :     *#            ~     *#            ~      +#            ~     +#            ~     +#            ~     +#                  +#            ʃ     (+#            ~     0+#            ~     8+#            ~     @+#            ~     H+#            ~     P+#                 X+#                 `+#            ~     h+#            σ     p+#                 x+#            ~     +#            ~     +#            ~     +#                 +#                 +#            E     +#                 +#            N     +#            `     +#            ۃ     +#                 +#                 +#                 +#            Hv     +#            $     +#            +     +#                  ,#            4     ,#            A     ,#            J     ,#                  ,#            T     (,#            ^     0,#            j     8,#            p     @,#            t     H,#            {     P,#                 X,#            w     `,#                 h,#                 p,#            ~     x,#                 ,#                 ,#                 ,#                 ,#                 ,#                 ,#            B     ,#            w     ,#            Hv     ,#            R     ,#            Y     ,#            d      -#                 -#            ~      -#            ~     0-#            ~     @-#            ~     P-#            !~     `-#            #~     p-#            4~     -#            ~     -#            6~     -#            )~     -#            9~     7#            7#      8#            q     8#            *r     8#            q     8#            #r      8#            $q     (8#            q     h8#            @      p8#                   x8#                  8#            p      8#            @      8#                   8#                  8#                  8#            `      8#            P      8#            P       9#                  (9#            `_     09#            0_     89#            _     @9#            pa     `9#            M     h9#            T     p9#            [     x9#            `     9#            f     9#            m     9#            r     9#                 9#            {     9#                 9#            [     9#                  :#                 :#                 :#                  :#            q     0:#                 @:#            }     P:#                 :#            0J     :#             J     :#            I     :#            S     -#                   -#                   (-#                   8-#                   H-#        y           X-#        y           h-#                   x-#                   -#        w           -#        w           -#        /           -#        /           /#                  :#                  /#                   /#                  /#                   /#        J            9#        J           /#                   /#                   /#                   /#                   /#        (          /#                   /#                   /#                   /#                   0#                    0#        	          (0#                   00#        3           80#                   @0#        G           H0#                   P0#                   X0#                   `0#                   h0#                  p0#                   x0#        F           0#                   0#                   0#        !          0#        *          0#        [           0#                    0#        
          0#                   0#        "          0#                   0#        ~           0#        k           0#                   0#                   0#        $          0#                    1#        W           1#        {           1#                   1#                    1#        q           (1#                  01#        *           81#                  @1#                   H1#        0           P1#        <           X1#                   `1#                   h1#        #          p1#                   x1#        g           1#        n           1#                   1#        \           1#        Y           1#        o           1#                   1#        8           1#                   1#        r           1#        h           1#                  1#                   1#        b           1#                   1#                   1#                    2#        !           2#        X           2#                  2#                    2#                   (2#                   02#                   82#                  @2#                   H2#                   P2#                  X2#                   `2#                   h2#        I           p2#                   x2#        %           2#        ^           2#                   2#                   2#                   2#                   2#                   2#                   2#                   2#        S           2#                   2#                   2#        )           2#        O           2#        p           2#                  2#        &           3#                   3#        ,           3#                   3#        s            3#                   (3#        '           03#        @           83#                  @3#                   H3#        L           P3#                  X3#                  `3#                   h3#        	           p3#                   x3#                   3#                   3#        #           3#        |           3#                  3#        _           3#        
           3#        (           3#        $           3#        d           3#        A           3#        c           3#        &           3#                   3#        a           3#                   3#        N            4#        E           4#                   4#                   4#                    4#        B           (4#        .           04#        +           84#                   @4#                  H4#        4           P4#        m           X4#                   `4#                   h4#        e           p4#                   x4#                   4#        v           4#        2           4#                   4#        Q           4#        >           4#                   4#                   4#                  4#                   4#        H           4#                   4#                   4#        i           4#        9           4#                   4#        P            5#        
          5#        R           5#        '          5#        %           5#                  (5#                   05#                   85#                   @5#                   H5#                   P5#        C           X5#                   `5#                   h5#                  p5#                   x5#        z           5#                   5#        K           5#                   5#                   5#                   5#                   5#                   5#                   5#                   5#                   5#                   5#        
           5#                  5#                  5#                   5#                   6#        f           6#                   6#                   6#                    6#        6           (6#                   06#                   86#                   @6#        t           H6#                   P6#                   X6#                  `6#        :           h6#                  p6#                   x6#        V           6#                   6#                   6#                   6#                   6#        +          6#                   6#                   6#                   6#                   6#        `           6#                   6#                   6#                   6#                   6#                   6#        }            7#        u           7#                   7#        "           7#                    7#        ,          (7#                   07#                   87#                  @7#                   H7#                   P7#        j           X7#                   `7#                   h7#                  p7#                   x7#                  7#                   7#                   7#        1           7#                   7#                   7#                  7#        x           7#                   7#                   7#                   7#                   HH" HtH         5Ҥ" %Ԥ" @ %Ҥ" h    %ʤ" h   %¤" h   %" h   %" h   %" h   %" h   %" h   p%" h   `%" h	   P%" h
   @%z" h   0%r" h    %j" h
   %b" h    %Z" h   %R" h   %J" h   %B" h   %:" h   %2" h   %*" h   %"" h   %" h   p%" h   `%
" h   P%" h   @%" h   0%" h    %" h   %" h    %ڣ" h   %ң" h    %ʣ" h!   %£" h"   %" h#   %" h$   %" h%   %" h&   %" h'   p%" h(   `%" h)   P%" h*   @%z" h+   0%r" h,    %j" h-   %b" h.    %Z" h/   %R" h0   %J" h1   %B" h2   %:" h3   %2" h4   %*" h5   %"" h6   %" h7   p%" h8   `%
" h9   P%" h:   @%" h;   0%" h<    %" h=   %" h>    %ڢ" h?   %Ң" h@   %ʢ" hA   %¢" hB   %" hC   %" hD   %" hE   %" hF   %" hG   p%" hH   `%" hI   P%" hJ   @%z" hK   0%r" hL    %j" hM   %b" hN    %Z" hO   %R" hP   %J" hQ   %B" hR   %:" hS   %2" hT   %*" hU   %"" hV   %" hW   p%" hX   `%
" hY   P%" hZ   @%" h[   0%" h\    %" h]   %" h^    %ڡ" h_   %ҡ" h`   %ʡ" ha   %¡" hb   %" hc   %" hd   %" he   %" hf   %" hg   p%" hh   `%" hi   P%" hj   @%z" hk   0%r" hl    %j" hm   %b" hn    %Z" ho   %R" hp   %J" hq   %B" hr   %:" hs   %2" ht   %*" hu   %"" hv   %" hw   p%" hx   `%
" hy   P%" hz   @%" h{   0%" h|    %" h}   %" h~    %ڠ" h   %Ҡ" h   %ʠ" h   % " h   %" h   %" h   %" h   %" h   %" h   p%" h   `%" h   P%" h   @%z" h   0%r" h    %j" h   %b" h    %Z" h   %R" h   %J" h   %B" h   %:" h   %2" h   %*" h   %"" h   %" h   p%" h   `%
" h   P%" h   @%" h   0%" h    %" h   %" h    %ڟ" h   %ҟ" h   %ʟ" h   %" h   %" h   %" h   %" h   %" h   %" h   p%" h   `%" h   P%" h   @%z" h   0%r" h    %j" h   %b" h    %Z" h   %R" h   %J" h   %B" h   %:" h   %2" h   %*" h   %"" h   %" h   p%" h   `%
" h   P%" h   @%" h   0%" h    %" h   %" h    %ڞ" h   %Ҟ" h   %ʞ" h   %" h   %" h   %" h   %" h   %" h   %" h   p%" h   `%" h   P%" h   @%z" h   0%r" h    %j" h   %b" h    %Z" h   %R" h   %J" h   %B" h   %:" h   %2" h   %*" h   %"" h   %" h   p%" h   `%
" h   P%" h   @%" h   0%" h    %" h   %" h    %ڝ" h   %ҝ" h   %ʝ" h   %" h   %" h   %" h   %" h   %" h   %" h   p%" h   `%" h   P%" h   @%z" h   0%r" h    %j" h   %b" h    %Z" h   %R" h   %J" h   %B" h   %:" h   %2" h   %*" h   %"" h   %" h   p%*" f        H=џ" Hџ" UH)HHvH" Ht	]fD  ]@ f.     H=" H5" UH)HHHH?HHtH" Ht]f     ]@ f.     =Y"  u'H="  UHtH=b" =H]0" @ f.     H=" H? u^fD  H" HtUH]@AWAVH5% AUAT1USHH H   H5 11H H   H-" 11H5o HHE   IHE   IHE E11HM" H5. H  HE 1H LH  t0A$PA$  A   HD[]A\A]A^A_@ HE H= L    IHE H=   LHLHAօuHE E11Hߒ" H5` H  HE 1H2 LH  FHE H=& L    IHE H=   LHLHAօH= H:LM E1HH1" H5 HA  HE 1H{ LH  HE H=_ L    IHE H=7   LHLHAօZHE 1LH5 H  HM LHH  A#HE HH
 H" H5   HE 1H H5 HPfHE L   @ f.     ;f.     1 f.     f.     USHH1HH" dH%(   HD$1HH  t1HL$dH3%(   u/H[]fH1HT$H  u΋D$9$D  USHH1HHs" dH%(   HD$1HH  u;H1HT$H  u$$+D$HL$dH3%(   uH[]fD  d@ USHH1HH-" dH%(   HD$1D$    HT$HE   u8D$T$D$D$HL$dH3%(   u/H[]    HE HHh 
  H= H1fHH]" H1H   1HHdH%(   HD$1H-" Ht$H   HD$tH1Ht
fHJH9uH|$dH3<%(   uH7    HdH%(   HD$1H" HT$H @  HL$dH3%(   uH f.     USHH1HHs" dH%(   HD$1HH@  t1HL$dH3%(   u/H[]fH1HT$H@  u΋D$9$eD  USHH1HH" dH%(   HD$1HH@  u;H1HT$H@  u$$+D$HL$dH3%(   uH[]fD  @ USHH1HH-s" dH%(   HD$1D$    HT$HE @  u8D$T$D$D$HL$dH3%(   u/H[]    HE HHh 
  H= H1BfHdH%(   HD$1H" HH H  HL$dH3%(   uHD  f.     USHH1H(H" dH%(   HD$1HT$HH  t1HL$dH3%(   u0H([]H1HT$HH  uHD$H9D$rfUSHH1H(H" dH%(   HD$1HT$HH  u9H1HT$HH  u"HD$+D$HL$dH3%(   uH([]f@ ATUHSH1HL%" dH%(   HD$1H$    HHI$H  u=Hs1H@ HJH9uH|$dH3<%(   u3H[]A\     I$HL` 
  H=` H1AIf     HdH%(   HD$1H" HH (  HL$dH3%(   uHD  f.     USHH1H(H" dH%(   HD$1HT$H(  t1HL$dH3%(   u4H([]H1HT$H(  uD$1f.D$Dn@ f.     USHH1H(H" dH%(   HD$1HT$H(  uYH1HT$H(  uBD$   L$f.w1f.HL$dH3%(   uH([]f     @ ATUHSH1HL%Q" dH%(   HD$1H$    HHI$(  u=Hs1H@ HJH9uH|$dH3<%(   u3H[]A\     I$HL` 
  H=P H1Af     HdH%(   HD$1H" HH H  HL$dH3%(   uHD  f.     USHH1H(HS" dH%(   HD$1HT$HH  t1HL$dH3%(   u0H([]H1HT$HH  uHD$H9D$BfUSHH1H(H" dH%(   HD$1HT$HH  uYH1HT$HH  uBHD$H9D$   HL$H9L$OHL$dH3%(   uH([]    @ ATUHSH1HL%1" dH%(   HD$1H$    HHI$H  u=Hs1H@ HJH9uH|$dH3<%(   u3H[]A\     I$HL` 
  H=X H1Af     SH0dH%(   HD$(1H|" H\$HH   uHh" D$HHR@D$HL$(dH3%(   uH0[ff.     AVAUIATUHS11H@L%" dH%(   HD$81HI$  t"HL$8dH3%(   uhH@[]A\A]A^ I$Lt$ 1LHL  L%" tI$HP@D  I$HL1PPI$LP@ AVAUIATUHS1AH@H-G" dH%(   HD$81HHE   u/HE Ld$ 1LHL  H-" t1HE HP@HL$8dH3%(   Du0H@[]A\A]A^    HE HLPPAHE LP@    USH11H(dH%(   HD$1H" HH   u;$HHT$ۍCtHL1 HBH9uHJ" HH P@H|$dH3<%(   uH([]s H(dH%(   HD$1H" HT$HL$H x  HT$dH3%(   uH((     HdH%(   HD$1H" HT$H   HL$dH3%(   uH f.     1 f.     HdH%(   HD$1HM" Ht$H X  HD$tH1Ht
fHJH9uH|$dH3<%(   uHW    @ f.     {f.     AWAVIAUATL-nq" USA
   1HI,LHHM| I7x!t/HkL9rH1[]A\A]A^A_D  H9IwfD  HL[]A\A]A^A_@ f.     ATUISH:" HH
  HHL
  [HH]A\x     @ f.     ATUISH" HH
  HHL
  HH,[]A\D  f.     @ f.     ATUISHH" dH%(   HD$1HH  HHHt$L  $9T$HNT$HHc$HL$dH3%(   u	H[]A\vfD  ATUISHH" dH%(   HD$1HH  HHHt$L  Hc$1;T$t"HL$dH3%(   u!H[]A\fD  HHf     AVAUIATUHSHL%o" dH%(   HD$1Lt$I$L@  uq|$v?H   t5I$1LL   X  H= H1AI$HHh  HL$dH3%(   u.H[]A\A]A^f.     I$LLH  @ kf.     f.     ATUISHHtLLHx!Ho" H P(Hx HLHHE C    HC    HCH[]A\1ff.     ATUISH!x!H" H P(Hx HLHH~" C   HkHCE H[]A\ f.     ATUISHx!H" H P(Hx HL"HHH~" C   HkHCE H[]A\ f.     USHHwt)vH-L" H= 1HU R fHH-(" HE HH@0H[] f.     wtPr>uHwH?eD  HH" H= H1R 1H    HwH?@ HwH?ff.     HwtMr;uHwHfHHm" H= H1R 1H    HwH@ HwHff.     HGt=v H" H=h 1HR 1HÐHvHH?    HvH;H@ f.     H9tSUSHHHH6H?uU9StH[]ÐH1[]    HHH[]       fAUATIUSHHwdH%(   HD$1   rruML-" H|" 1HIE H     1HL$dH3%(   9  H[]A\A]@ H" H=b H1R      L-" HHHIE X  IE Ht$LX  $   D$xH{HLH   PZD  L-I" H{" 1HIE H   +IE 1HL   It$ H}   T$1IE 1HL   It$ H}  SHHwtAv$H" 1H=T HR H1[fD  HHH[     HHT$rHHT$tH@HH[H    ATUSvHӃtA      H-" Hy" HHE    }   H{ ,@ H-" Hy" HHE    uOH{ IHE L  HHE LP0H[]A\fD  Hy" H=z H1R 1H[]A\     AVAUIATUISHHH0H9" dH%(   HD$(1H
  IHHL$ HT$LHx     D$   urHT$ HH:
  H=jx" HH/   H=px" H   HD$ LHHpLHt>HL3f.     HL
     HLtLH1HL$(dH3%(      H0[]A\A]A^fD  HT$ HH:
  LHHqHuHھ   LHD$LH,HD$D  HD$ LHHp|HjHL\ Hھ   7@ f.     AWAVIAUATIUSIHHvdH%(   HD$1B     t?H@" H=y H1R HHL$dH3%(   H?  H[]A\A]A^A_Hw" H9B  :F  L=" HI   I|$MHHLH   IH`  H   HE I|$H0^Hھ   L1LL<fHa" Ht$HH X  D$I|$bH HLPHھ   fD  Hu" H9B  :   L=" HI   I|$MHHLHr   E PU    1{I|$ILHuH" HHH `  fD  I|$ILHuHR" HHH `   HE I|$Hpf     IH1   D  IH`  Hu1H H3+f.     ATUHSHwdH%(   HD$1tNw)1HL$dH3%(      H[]A\f.     Hq" H= H1R 1fD  L%Q" HHHI$X  $1tH{THtHP     I$Ht$HX  HD$HUHt
1D  HJH9u5'    SGHH?HvFH" H   C[f.     H" RH?H H      HGtbr@t#H\" H= 1HR 1H@ H/H
    H!" HHH H  HHH
D  f.     ATUHSIHw      u[H5tr" H}E {H]HT$.E	 E
{I|$HT$HHH }@}HH)[]A\f     Ha" H= H1R H1[]A\    H9" RHHH H  H[]A\ H5q" H}E {HT$H]E E{PfD  USIHGt}HrHuH~ HH[]LfD  H" H= 1HR H1[]    H" HH H  
  HHsH[H]@ H~ HH[]Lf.     USHHHHOtyr?uHx HL[]H" H=  H1R H1[]    Hٻ" AhHHt$H
H  
  Ht$HHH[]Hx HL[]>@ f.     7tZw]ATUISGH~)hHHHD  H_ H9HuI|$H;" H P0ID$    []A\fD  H" H=r H HP 1D  f.     7tZw]ATUISGH~)HD@H,    H_H9HuI|$H" H P0ID$    []A\fD  H" H= H HP 1D  f.     ATUHSH0HdH%(   HD$(1=HAHA|H+" H P(HHHIlHcHHL HH<HL$(dH3%(   uH0L[]A\+f.     +Ðf.     G+Ff     AVAULcL-" AT0   USLcIE P(HIE B<    P(HCHJ<    HCIE P(HC HCJHC(H[]A\A]A^fUSHHH?H-%" HtHE P0H{HtHE P0HE HH@0H[]@ f.     ATULcSHHGH)HL9}7HoHø" HHH)H HHcP8HHJHSHC[]A\f.     ATULcSHWHG(H)HL9}8Ho Hb" HHH)H HHcP8HHCJHS HC([]A\     USHHHOHGH9shHE H9t!;t;;PuD@ 9t,9Pt7HH9uQHHMH1[]     H   []@ H   []@ H+T$HtHMT$nfD  HG HO(ATAUSHH9rH+OHt
HC D hHHC []A\fD  AUATUSHHcHHH+;HLkHII9t4LHI9uHL[]A\A]@ f.     AUATUSHcH   HLkHII9tD  41LH/I9uHL[]A\A]ff.     LGHI9t"Hc;uD  9tHI9u ƃ)HcƋfHI[" HcHHHw   HH)H    AUATUSHcH   HLk H[II9t@ 41LHOI9uHL[]A\A]ff.     LG HH
I9t1HHcf.     HcHI9uHI)IA@1ff.     AWAVAAUATUSHcH   HIǍELc H[D,AA)I9t0Mc    HI9tD94uB4LHI9uHL[]A\A]A^A_f.     H" SH(   H P(HX H@    [f     Hq" S8   HH P(HP(H@    HP HHP(HSHP0[fD  SHG HH8H!" H[H H@0ff.     SHH H" H[H H@0D  f.     HF HwH?HHH9f     HHvH81Hv 7    H(     SHG HH80H{'Hh" H[H H@0 USH8   HHC" H P(HH@(   HC  HCHE HC( HEHC0HH[]D  f.     AWAVIAUATUSHHHzHHXHL$IHL,rHL$IGHhL0L9u[f.     L9t0IcH|$IIT$IuHHH4tAOڅEH[]A\A]A^A_1f.     AVAUATUl?S9}`AAI2D  LDy=I}lDAA9~#]A9~ˉLN     []A\A]A^    AVAUIATUISHusL|LIqHL   IHL%k" H5 HI$  I$LH  I$LP0I$LP0[]A\A]A^fD  AWAVIAUATIUSHHHcn" HHdH%(   HD$81H۰" H H
  M,$L;m HD$ HD$t@Lt$IE LHt$HL|$(HD$ SHLI+4$HxIHL;m uHD$8dH3%(   uHH[]A\A]A^A_fATL%O" USHx   I$P(HxHH@p    H1Hn" HH)xHH+H{ I$E H
  H[]A\f     USHHH_H;_tD  H;H|H9]uH" H} Hx  H}HtHP0H} HHH@0H[]     ATULcSHWHGH)HL9}8HoHR" HHH)H HHcP8HHCJHSHC[]A\     AVAUATUISHH dH%(   HD$1HGH9G   A$LHA$tH3LHŋ LkHT$E H{ HShDt$EtL+kILhHCHPHSH(E H3L+HL$dH3%(   Du9H []A\A]A^ H+GHT$Ht	HT$?zf.     AWAVAUATUSHHH?HIHuH+uHIHH]H;]tD L;LH9IIIGIVHHpH+pZ1LLH9]uHL[]A\A]A^A_D  ATUHSH?QHuH+uHIH*H]H;]tH31LH?H9]uL[]A\ÐAWAVIAUATIUSHHH(dH%(   HD$1H9w   L#Lm LLU`HtHج" HHp  MLtHu LLHHD$H3LHD$HUhD|$EtJHH+ULA   HHPH 'HL$dH3%(   DugH([]A\A]A^A_D  H;I9tHT$L#LLUhT$tH+]HHX@ H	" H= H1R H     AWAVAUATIUSHH(H?Ht$H6HHD$   HIt$HI+t$HD$HCH+CHH\I\$I;\$   @ HD$LxL;xtf     H+H|$IMw軼H}IHEIUHHpH+pI4$I~IFHVH+VHHpIUH+pH|$1LHD$L9xuHI9\$cHD$H([]A\A]A^A_HD$    ff.     AVAUIATUHSH?ZHIoHuH+uHIHHH]H;]t+fD  H;LLH1HLIH9]u[L]A\A]A^    AWAVIAUATUSHH   L'H|$HdH%(   H$   1It$I+t$HLLIHD$(H$IFI+IT$I+T$HHH|I/I;oHD$0t0Ll$(L|$0HcU ID$LLHH4HVI9muH4$HҿH\$0HHH*HD$8HD$HL$(HPHpHD$pHHHD$HH" Ht$PHHt$@H   HHZ  $   HU uHU H|$0L"AHD$ HPHD$(L(L;hd  I IcM ID$HZIH<HwBM9oHuH<$`HIHD$ H LHEHpHH+0HD$HL#L;ctufHD$Ic$H<$H@H,薹M>M;~IHPt+fD  IcHEHZIH<HwM9~HuH|$1LIغHD$L9`uHt$ H|$81輺H" H|$@H   HHH|$(AH" H|$HH x  H$   dH3%(   HD$8uH   []A\A]A^A_H蕻D  AWAVAUATUSHHHXHHT$HIHHD$8$I;FHD$0T  HD$0H x  HXHE HPH+PHCH+CHH|
Lt$8Ld$0HIHD$ IvLLIVIt$LLHSHsLH  H|$ MHMH$HEHL$(H9HD$@  fD  HD$(H\$01H-[" H HhH@IL`L+`HD$8H+XH" Ld$H HL+L      M} H<$IwI+wHIGMwHKHL$L9HD$tw@ H|$ M>ILhHHILBHHt$H|$Ml L)MWIHIT HIrI+r
H<$1LLL9t$uHD$(HD$(H9D$@H$HX[]A\A]A^A_H|$HL   jH<$衽H$    H|$HHھ   H$    H|$HH    չH|$ H$    H|$HHھ   诹H$    cfAWAVIAUATMUSHH5& H߉H(HT$üI>Iž   CLIHHD$лI;D$   H|$HD$HD$L(L;h   @ IcU IFIH|$L<fHXIIIGD}HHHpH+p耾HDHH" H   H 1H|$L蠶HD$L9huHD$H([]A\A]A^A_Hھ   L聸H|$跽HD$    ff.     AWAVIAUATMUSHH5 HDH   HT$HL$(dH%(   H$   1gI?Iľ   LIHHD$tI;F  H|$HD$HD$pHgW" HHD$ H" H H
  HD$HL$\HL$0HL$`H(H;hHL$8HcHL$H   L|$@lD  I^IIGLHHHpH+pH߃Ml H*" HcH   IE  1H|$I^LHD$H9htqHD$@HcU H|$H@L<VLhIHD$(L|$`HT$0Ht$8H|$ HD$h$   IƋD$\CHD$HHIF6f.     H" H|$ H x  H$   dH3%(   HD$u7H   []A\A]A^A_H   LQH|$臻HD$    w    AWAVAUATIUSH8H/H|$HT$(H
H;E#  HH x  LpHD$H HPH+PIFI+FHH|HuHHHIHD$	HUHsLHIVIvLLs  H|$ALt$HHD$IvI+vHH+]InHI;n   HcH    HD$         H|$Mw HnLhI|$HIL藺HH{Ml ID$LHpH+pIt$rMNI~HIT HIqI+qSH|$1L蔲HD$H9htZLe H\$ 1Ho" H8T" I\$H L;L   DH|$(L   袽H|$طHD$    HD$H8[]A\A]A^A_ H|$(L   HD$     H|$(H=    H|$(HD$    H|$(L   ˳HD$    AUATIUSHH)HHHHI[H9u-@ HH9tH3LLiuHH[]A\A]HfAUATIUSHH)HHHHIH9tfD  H3LLHH9uH[]A\A]@ f.     AVAUAATIUSHH6H?HI1   H;蜼HsH{1HH7H;I4$[EHIt$I|$HHt[I;D$tTH8߷   HLIlHM" LH P0H1H[H]A\A]A^fD  賯H˯[H]A\A]A^D  f.     AWAVAUATIUSHH   IH   dH%(   H$   1覷HSHsH|$0HHD$H|$1L}I\$I;\$IHD$ HD$t`@ HHt$L|$(H|$HD$ T$pHt1HhLu L;ut#fHI+t$ALIH襵L9uuHI9\$uH" H|$H x  L跮H|$譮H$   dH3%(   uHĘ   []A\A]A^A_# AWAVIAUATIUSHXHH.dH%(   HD$H1HGHT$0HL$8HD$HFHD$ HL$@MLHH?HHD$(q  H9Iw I+wHHD$HI_I;_    fHcHL$H|$(L4HcCHL$ L<HD$@L(uI~L`HD$IFHLHpH+p葵IWHIwM$HBH9BLvtDD  II~LA}tLLSIwHIWM$HHBIH+BH9uHt$H|$1ptAHD$0HH9X )H|$@HL$HdH3%(   HD$uUHX[]A\A]A^A_H|$H|$@ǬH|$}H|$8H¾.   [HD$    HD$    0AWAVIAUATIUSHhHH.dH%(   HD$X1HGHT$8HD$ HFHD$(HL$HLD$PMLHH蟴HHD$0  HIIw I+wHHD$H IGI;G HD$'  D  HL$H\$ H|$0HcL4HcAHL$(H耪IVHD$LxHD$HIvL HBH9BtMLnf     II}LA|$tLLrIvHIVM<HHBIH+BH9uHD$PHSHsL0HBH9BLntAfII}LA~tLLHsHHSM<HHBIH+BH9uHt$H|$10HL$8HD$HD$H9A H|$H輪H|$P貪HL$XdH3%(   HD$uHh[]A\A]A^A_HD$    @ f.     AUATUSHHHLnH;Hu H+uHIHXH]H;] t#fD  HcC1LHIt aH9] uHL[]A\A] f.     AVAUATUISH荳I<$IT$I+T$H߾   IH覬I\$I;\$IH(u0fHI9\$t#HEtH31LHɩI9\$uLj[L]A\A]A^@ f.     AWAVAUATUSH   HGLgdH%(   H$   1HHl$@H|$ IH$HQ" H\$HHH P
  H{H+{HdM9HD$  HD$<LE1Ld$(I'Mt$LHL$   T$<uMHH9$thHLHL`M4$L$   L$<tA}LhL|$H|$Lcl$A}LhLHAH|$McئH9$uHD$Ld$(LpL+0Hb" IH EEDP(Ic1HI親Dt$H$MIHIL`I4$$   HXIt$H$   AXL94$HcAuDt$H" HH x  E  A]IcE1E1HE1ELt9@ HH<HL) HH)BB"	HH9uEHE9|IIE9H|$MH\$ HHD$HsH+sHLt$Hl$ L$    L|$(E1
IE9~TB|%  tH|$腥4$HXLIH DLIHC 1H|$LئE9$Hl$ $D9uL|$(H" LH P0H|$#H$   dH3%(   HD$u{HĨ   []A\A]A^A_HD$LpL+0H^" IH EEDP(Ic1HI袨IH|$H\$ HHD$HsH+sHS[f.     AWAVIAUATIUSAH8LgL+gHnH+.1dH%(   HD$(1IDHȰ1HHD$	H|$   H|$Dl$DL<$Lt$HoH+/HHAx!I@ LDuH|$A~*ID  11LH|$uHL$(dH3%(   HuH8[]A\A]A^A_?D  f.     HATUSHHtIH3ɨIH{ HS`LHqHtHcUHC[]A\H HC[]A\    HC HS`Hf     AWAVE1AUATUSHHHLgLvH?H6t[H3H} H;IH[II9t0    H3LHI9t
H31LHI9uL藣HL[]A\A]A^A_D  AWAVE1AUATUSHHHLgLvH?H6tNH3H} VH;IkH[II9t#    H3LHzI9t%HI9uLHL[]A\A]A^A_fH31L3̐AWAVIAUATIUSHXH.HHL$0H1H|$HELmLaL+aHHL$HD$HFIHD$@HFHHD$HHCHD$ HCHD$8芫HL$  1HHHL$kA9HL$}  L+l$1HHHL$FIA9Y  HD$ H+D$8EHA9?  HL$H"HL$HHD$8H=IH蒠HLHԬHL$HD$HAHHD$HD$@H+D$HHHD$ HH;X   fD  HD$H]H HHxHD$0H@HHpH+p\HMnL$1M;ntOf.     IE LHxH@HHpH+p!HT$HLqI9GIM9nu9\$ tmHL$HD$HD$H9ASHH|$ɠ&    H|$0Hl    
HD$8    HD$8HX[]A\A]A^A_D  Ht$0H|$81ǠH|$0H!    迢HD$8    @ AUATIUSHHH?hHuH+u HIHAH] H;]t&    HcIELHH41GH9]uHL[]A\A] AWAVAUATIUSHIHHGH9  H;w
  L9   L;m   M9   t;L5Ռ" L} LfD  H3LHU`IHp  H{^I9uHULLLL)TLHEL)Lm I9HEt,D  H3LU`HtHH+UHHPHH9]uHL[]A\A]A^A_    L5)" 1H=  IR M93L5
" H=3 1IR !fD  L5" 1H= IR HEf     AWAVAUATIUSHHLoL~H?H6tJH3I<$1RH[II9t(fD  H3LL肦I9HI9uLH[]A\A]A^A_D  USHHH菦x'HUHKH+UH+KHH99!H[] f.     HIH    USHHHx'HUHKH+UH+KHH99!H[] f.     USHHHϥx'HUHKH+UH+KHH99!H[] f.     USHHHx'HUHKH+UH+KHH99!H[] f.     USHHH/x'HUHKH+UH+KHH99!H[] f.     AVAUIATUISHIH脦H;Ct6H8HL萚HHHH;C   t/[]A\A]A^@ LL   81[]A\A]A^ L   L1@ AWAVIAUATUSHH(HGH+GL'H|$Ht$HFB@H" H P(1HHEHcI~INL}H9H|$          H|$I\$I;\$HH|$LrHWtULf     rIH;IHj HDl0Iv謡I9\$ADHLuI;\$tH|$HOHD$H|$H9pH|$HWH)HQHH([]A\A]A^A_Hʸ    f.     AUATIUSHIHH?腦LXLHtH[]A\A]    AWAVHAUATUSHHLyHT$0HH|$({HHt$8HT$HHH;xH|$        HD$ HZH HhHD$0HHHD${LpL;p   M&HHHLmHL$ILHL$HMHLHX  IGHq HHD$Ht$L蚣H|$HHL$HL|$  HXL9wuH|$(HD$ HP }@ Ht$ H9w4HP}HBH+D$8HH[]A\A]A^A_HAVIAUATUISH^{H?HIHHcLHHLH^}F H舤HHHL)L []A\A]A^D  AUATUSHHXdH%(   HD$H1H" Ld$ HHD$     HD$0    LHD$8    H$    H HD$(    D$    HD$    HD$    $   HX((xHLHHHϖHL$HdH3%(   H uHXH[]A\A]FfD  AUATUSHHXH?Ld$ HD$     H$    dH%(   HD$H1LHD$(    HD$0    HD$8    D$    HD$    HD$    $   fHg" HLHH HX(JxHLHHHHL$HdH3%(   H uHXH[]A\A]h     AWAVAUATUSHH(  dH%(   H$  1Ƀt$,HT$   HD$ L5Ã" H8" HhIH   t8A   H$  dH3%(   D  H(  []A\A]A^A_fD  HD$ Hm HH]8" L`IL   u11M|$ CIHD$02      HHD$IQHD$,HT$ l$,HD$  HHHT$HD$HAUMMH訙A^_  LHL=LH蒘D$,PT$,.  Dl$(L|$v 11Ml$ 蒗HHIAWHT$MHt$ HI.ZY   LLHƠHILD$,PT$,   LHD$ HHPL IHT$ H7" L   ^H薚AH" H
B~    A   H P  f     LA   肗D  MDl$(ILHh  HHIǕ    AWAVAUATUSHH(  dH%(   H$  1Ƀt$,HT$   HD$ L5#" H6" HhIH   t8A   H$  dH3%(   D  H(  []A\A]A^A_fD  HD$ Hm HH5" L`IL   u11M|$ 裕IHD$01      HHD$I豒HD$,HT$ l$,HD$  HHHT$HD$HAUMMHA^_  LHL蝗LHD$,PT$,.  Dl$(L|$v 11Ml$ HHIAWHT$MHt$ HI莖ZY   LLH&HIKLsD$,PT$,   LHD$ HHPL IHT$ Hg4" L   ^HAHA" H
{    A   H P  f     LA   D  MDl$(ILHh  ~HHI'    AWAVAUATUSHH(  dH%(   H$  1Ƀt$,HT$   HD$ L5~" H|3" HhIH   t8A   H$  dH3%(   D-  H(  []A\A]A^A_fD  HD$ Hm HH3" L`IL   u11M|$ IHD$0$      l$,HD$  HHD$IHD$,HT$ HHMMHT$HD$HAUHhA^_;  HL$HLLxLHMH,  Dl$(L|$   D  HD$ HHPL IHT$ H:2" L     11Ml$ #HHIAWHT$MHt$ HI迓ZY   LLLHԖHIyL衒M   LD$,PT$,RIHDl$(Lh  +HHA#D  Hy|" H
y    A   H P  LA   "D  Ht$HA   uHȔn@ f.     AWAVAUATUSHH(  dH%(   H$  1Ƀt$,HT$   HD$ L5{" H0" HhIH   t8A   H$  dH3%(   D-  H(  []A\A]A^A_fD  HD$ Hm HH]0" L`IL   u11M|$ CIHD$0      HHD$IQHD$,HT$ l$,HD$  HHHT$HD$HAUMMH訑A^_;  HL$HLLؙLH荐H,  Dl$(L|$   D  HD$ HHPL IHT$ Hz/" L     11Ml$ cHHIAWHT$MHt$ HIZY   LLLH4HI蹒LM   LD$,PT$,RIHDl$(Lh  kHHA#D  Hy" H
v    A   H P  LA   bD  Ht$HA   赕H讍@ f.     AWAVAUATUSHHt:H+y" A   H
!    H P  HD[]A\A]A^A_fD  Hx" LbH-" HL   AtA    ID$ 11L H  M|$IID$L9HD$urfIL9|$teIHH:  IHLLHp  tA$PA$~<APAkHL   ZD  HLHh  HL   @ f.     AUATUSHHt.Hw"    H
~    H P  H[]A\A]L%w" LjH," I$L   tH   []A\A]f     IU I$HzH+zLh  H  HHAH[]A\A]f     AUATUSHHt.Hw"    H
~    H P  H[]A\A]L-v" LbH+" IE L   tH   []A\A]f     IT$ IE HLh  HzH+zH  HHAH[]A\A]D  AUATUSHHt.H_v"    H
V}    H P  H[]A\A]L%1v" LjH&+" I$L   tH   []A\A]f     IM I$1HAH+ALh  HH@  HHAH[]A\A]D  f.     AUATUSHHt.Hu"    H
|    H P  H[]A\A]L%au" LjHV*" I$L   tH   []A\A]f     IM I$1HAH+ALh  HH@  HHAH[]A\A]D  f.     AWAVAUATIUSHH  dH%(   H$8  1H|$ c  Ht" HZAH)" H|$ H H   D$t=D$   H$8  dH3%(   D$  HH  []A\A]A^A_     HC HD$8HIEHD$AFAǉǉD$(YDHD$ܐEI   AFIm Mt(=HL$HPHHL  LH+sLHH݆L9   Hs" H} H 
  HHI袐H;CIuH|$ύL'M   H|$ @      [D$   fD  H9s" H
o    H P  D$       HCH+CH+D$(x@HsH+sLHD$@H藒H(H;hILt$@t+fD  HcU HCLHHH4HV!I;muHr" H|$H 
  Ht$HٍLt$@HHLI;Fk  H|$@BH\$8HHD$HHsH+sHHCH9CHD$   HD$H|$@H(莃M} M;}HD$(HXt* IcHEIH<HHw蝌M;}HHuH|$GM<$M;|$HD$0Lpt+ IcHEIH<LHwUM;|$HM4uH|$0H 1Ht$(H|$HwHL$8HD$HD$H;A-LLH>q" H|$HH Hh  H|$ H H|$fL较M   HT$Hl$P   HJH|$ HHH|$@H|$LiLaD$   /D  f.     AUATUSHHt.Hop"    H
fw    H P  H[]A\A]L-Ap" LbH6%" IE L   tH   []A\A]f     IE Lh  ID$ H8؎H耍HHAH[]A\A]f.     AWAVAUATUSHHHt7Ho"    H
Aq    H P  H[]A\A]A^A_D  L%qo" LrL-f$" I$LL   t
    L{I$LMv HL   uI$I>L  Lh  IG H0^1@AHHA_D  AVAUATUHSHHdH%(   HD$1tIHn"    H
gp    H P  HL$dH3%(     H[]A\A]A^D  L%qn" LjHf#" I$L   t
   fD  HI$Mm HT$HuE1ɹ   Lo HRH! " 	  ZYuLu I$HH"" L   uHcD$H" Iv LHTx'I$@@Lh    HHAM          H$2ځf.     AWAVAUATIUSHHXdH%(   HD$H1tOHHm" H
1j    H P  D$   HL$HdH3%(   D$  HX[]A\A]A^A_@ L-l" HjH!" IE H   D$tD$       HKHE HL$ HK H8HD$HL$(HK(HÃHL$8xHkH;kHD$]  HD$DHD$0\f     Lu HD$IvH8H|$HILHL~P  HD$HH;h   H} 脋 HIE 1A   HHt$ L0  HP  IE HT$0LHt$(P  1  T$DMIE 1Ht$8L8	      IE L@  IIE HB " LL   u:Mw HD$IvH8襊M   A      LÃ D$   IE H|$ H   
  11HLHL$8P   HL$ P   HL$(P   D$tOH|$U}IE H   nfD  PXIE H   FfD  IE H|$Hh  LHfD  IE H   ofD  IE H   BfD  IE H   tf   IE H=Pg Hh    LH IE LH   H     H=f 1LHkIE H=f Hh    LHIU H==k 1Hh     LHx} AWAVAUATIUSHHXdH%(   HD$H1tOHi" H
j    H P  D$   HL$HdH3%(   D$T  HX[]A\A]A^A_@ L%h" HjH" I$H   D$tD$       L} I?HGHoHD$ IEIHD$KIOHD$(IGAHL$H9HD$0  HD$DHl$HD$8HD$Hl$L0M~IFMLhL+hM9u   fD  IM9   HHuI$1A   IHH6	  HuHl$H;l$ Ll$ t#fHu I$H11HH6   I9uHL$P   H|$( D$    I$HT$8HHt$P  ~D$D   HD$HD$H9D$0Hl$H;l$ Ll$ t'fD  Hu I$H11HH6   I9uH|$P~NI$H|$(Hh  aHH@ H|$(1LygI$H|$   I$H|$   zfAWAVAUATUSHHH8dH%(   HD$(1tOHHf" A   H
Ah    H P  HL$(dH3%(   D  H8[]A\A]A^A_    L%e" LjH" I$L   AtA   f.     Mm HC L{I} HH$蔄HD$I]IEAH9HD$   HD$$Dt$I HH9\$   L+LքM$1A   HLHA0  Ht5I$LH4$HP  uD$$tH|$1Lx@ I$LH   
  11HHH$P   APA   H|$A   Z}D  Dt$I$LH   
  11HHH$P~?APA~"I$H|$Hh  HHI$L   I$H   I$L   QI$H<$   ,Sx AWAVAUATIUSHH   dH%(   H$   1tQHc"    H
e    H P  H$   dH3%(      H   []A\A]A^A_D  L-qc" LrHf" IE L   t   @ IE Mv H}
  @   IǾ   HYuHLL{wHt&IU HHh  蓀LHFfD  HL    +&wfD  AWAVIAUATUSHH  dH%(   H$   1Ƀt$Q  L-b" HZH" IE H   t6A   H$   dH3%(   Da  H  []A\A]A^A_@ MgIE HH[ H"" L   AuHD$Md$ >      HH$*tH$1LHixHI   D$I_ ID HD$RfD  HL{IE H" HL      Iw H$1LxLIzMtWMH;\$uIE LHh  ~HHD  H9a" H
^    A   H P  H4$HT}    LA   y@uAWAVIAUATUSHH  dH%(   H$   1Ƀt$Y  L-`" HZH" IE H   t6A   H$   dH3%(   Dq  H  []A\A]A^A_@ MgIE HH[ HB" L   AuHD$Md$       HH$JrH$L   HvHI   D$I_ ID HD$R HL{IE H" HL      Iw H$   L$vLI9xMtdMH;\$uIE LHh  }HHf.     HQ_" H
\    A   H P  f     H4$Hd{    LA   wuPsAWAVAUATIUSHH   dH%(   H$   1tQH^"    H
`    H P  H$   dH3%(      H   []A\A]A^A_D  L-q^" LrHf" IE L   t   @ IE Mv H}
  =   IǾ   HYpHLLKzHt&IU HHh  {LHFfD  HL    z+&rfD  AWAVIAUATUSHH  dH%(   H$   1  H]" Hjt$H" H H   AċL$t8A   H$   dH3%(   D   H  []A\A]A^A_fD  AH} L$M~AAtDDl$Ll$HHw<      L'oD$L$t&AMt  IIWLHHxM9uH\" HH Lh  <zHHA4fD  H\" H
BZ    A   H P  pff.     AWAVIAUATUSHH  dH%(   H$   1   H\" HjAH" H H   At5A   H$   dH3%(   D   H  []A\A]A^A_ H} AMorDHHvH|$      OtH|$mIIUHL$HHwudM9uH`[" HH Lh  xHHAI     H1[" H
X    A   H P  f     HA   sHo     AUATUSHHt.HZ"    H
a    H P  H[]A\A]L%Z" LjH" I$L   usIE HPH@H)HHw2H:I$Lh  mHyHHAH[]A\A]ÐM   ;      H   r_@ H   []A\A]fD  AWAVAUATIUSHH4$  L5Y" HZH" IH   tA   HD[]A\A]A^A_fD  Lc IHI]H" H   AuHs LsHII   $I] ID HD$Tf.     HLkIH&" HL      Iu LsLH$qH$HtYIH;\$uILHh  mvHHHX" H
RV    A   H P  f     M   9      HqLq AVAUATUHSt0HQX"    H
S    H P  []A\A]A^@ L%!X" LjH
" I$L      Mu M   IHPH@H)HHwKHrH:glt3I$LLh  nHHuHHAkf.     M   H   8      p[]A\A]A^fD     []A\A]A^AWAVAUATIUSHHHdH%(   HD$81F   HBL-W" LrH" HHD$ IE    uRHD$ Lx     HIE Hs HT$<E1ɹ   LY LRH|! 	  ZYtX   HL$8dH3%(     HH[]A\A]A^A_fD  HqV" H
RT       H P  D  HS(I?LqHHtHcD$4H! HLH|kHHD$hHC0HD$HD$ AL HD$ IE   IGHHD$HH;X  Lt$IŃ  HD$HH;Xz  I?zt   HHZpHc1HI4yhHs HIE 1A   HHt$L0  H   IE 1Ht$L8	  U JM _IU D$,H   D$,ED  HS I?LpHH51HLiHHD${gHC(HD$D  E Lt$PU =  IG   HIGH)HHwtH|$*gAPA   HL$ P   HL$PIE H   @ IGLt$Hf.     IE LH   
  11HLg II+1Hq1HHD$&mHC HD$     Lt$t4tbIOIE H   1IE L   
IG1HIE H      IGH1IE LH   H     H=EQ 1LHiUgD  AWAVAUATIUSHHX  dH%(   H$H  1FD$X      L%R" LrH" I$L   D$t9D$   H$H  dH3%(   D$  HX  []A\A]A^A_@ IF 7      M} HD$HD$`HHD$@gdI$I}
  HD$HH"     H"       EHD$0    HD$     Lt$\H-$! HHLL-" HD$HD$XHD$8V    W  6  AD    I$HIwHT$8@  W      IL;|$  I$HE1AV   I7LS HH	  Y^HcD$\AT e  HHtteW"    AD    HHD$IWH8lHH   1H|$HHL$(eHL$(HD$ Hfc9v"    AD    HHD$IWH8)lHHtY   f.     I$H=%O 1R  AD    HD$HIWH8kHHD$0D  D$   HD$ HtHbHD$0HoHbb HO" H
:N    H P  D$   6    Lt$@I
   LnLHks H|$     H|$0 tHLL$@DD$XHL$0HT$ Ht$HH|$qbHt?I$HHh  lHHLD$@L$XHT$ Ht$HH|$eHt$@HPkHD$0    HD$1HxH+xHl1HHD$ ;hJ!cAWAVIAUATUSHH  dH%(   H$   1   Ld$	   A   L`IwHL$mHǸ   HtbBmIAFI_LƉD$iD$   AFMt fHL9toHLHL6jtLf   H$   dH3%(   u_H  []A\A]A^A_HM" H
*P    H P     D  HM" LH Hh  /kHH1a@ f.     AWAVIAUATUSHH  dH%(   H$   1Ƀ4$
  Ld$6      Lc_IwLHkIŸ   M   LI_kIƋ$LƉD$gD$   $ID H$& 1LL_   HH;$   L^HLHHI!ktLEaLe        H$   dH3%(      H  []A\A]A^A_    H1L" H
K    H P     D  H	L" LH Hh  iHH1 H   L@jLHhLmd   S`@ f.     AWAVIAUATUSHH  dH%(   H$  1  H-nK" LbAH` " HE L   t7D$   H$  dH3%(   D$  HĘ  []A\A]A^A_fID$ H$   H$   HsLHD$L HE x  uH$   5      HHD$ \AEVUUUD$x)R)L$V  $   HC E1HD$8[HD$(HD$$   HpH+pHl_$   HD$H$   B  Lt$0H\$@HLt$(M@ HPLHLe~  LI+T$H|$DHAH}eD;$      HE H;
  LHfI;D$IuLt$0H   H\$ HhHLeH|$(cH|$d_D$   8    HqI" H
H    H P  D$       H\$ HH    HG^HLleD$       Lt$0H\$@D$xLk(AtvH\$0L|$(H\$ (fD  HLm`I;G  AIE~8IUIu HL`HuHt$ @ Lt$0H   H\$0Ll$(L\gH|$   HD$XHcH|$HIeLIHD$`ZHD$8HT$LHt$ dIIOH$   Ht$0HBH+B1IEI;OHL$HHD$h  H\$pHD$HH|$1L BD$|dH|$HIXaHPeH|$(I#YLHD$PHXRZHE 1A   LHt$8L0  H  ID$I|$HHHpH+pbHL<HD$pL`8D$x  AH\$h#f.     AIIEIG   HE HT$0LI4$x  upHHsHL$ H$   LcHt0H$    JHU HD$@   HD$@{H$   P
HE    H|$PC[L|$8HE LH   
  11HLAPA   H|$`^H|$X^IU Ht$PH|$X1YHL$`HD$HHD$HT$|H;AL|$8HE LH   
  11HLAPA~uH|$`^^HE H|$XHh  9cLHMHE H|$8   <YH|$P9ZAE PAU HE L   HE H|$8   wD  f.     AWAVIAUATUSHHx  dH%(   H$h  1  HD" LjL%! H LL   t6A   H$h  dH3%(   DQ  Hx  []A\A]A^A_@ IE MnLHHD$L8LHxD" H    uIE 4      L HD$H$   HH$yVHL$I|$I+|$HqH+qHH%YLHLHD$_IT$I+T$HHH99  EVUUUD$$A)RA)   IF LMf(HD$(UIŋD$$  Lt$0ML$$)D  HL[I;E  I   IVI6LH[HuH4$HA   _L]H|$	Yd@ H!C" H
2B    A   H P  9f     L<$H]B    LWLHA   _H|$X     L<$L   Laf.     Lt$0LsaHD$HHD$(Hl$HT$Ht$ H^HU HMHt$xHBH+B1IEH9MHL$8HD$P*  D|$lLl$XILt$`HD$8H|$1L(BD$h_H|$HI[H_H|$XI^SLHD$@HhTHA" 1A   LHt$(HH 0  H  IEI}HHHpH+pF\HLt HD$`L`8D$$0  AHl$PAIIEIF
  HHA" LI4$HH x     HHuH$HT$xH]HtSH|$x JHD$0H@" H   HD$0t@ L<$I   L,_LkH|$xPH@" H     H|$@VUH@" L|$(H LH   
  11HHAPA   H|$HXtHt$@H|$H19SHL$HD$8HD$8T$hH;AD|$lH@" Lt$(H LH   
  11HHAPA~}H?" H|$HH Hh  R]HHH?" H|$(H    BH|$@QTA$PA$Hn?" LH    SHQ?" H|$(H    ifD  AWAVAUATUSHHh  dH%(   H$X  1H$\  H$L=>" AH! HhIH   D$    t3H$X  dH3%(   D$ B  Hh  []A\A]A^A_fD  H$Lm       H@Im HD$AD$AHD$pHHD$PDVUUUDDDt$$)R)L$    HD$H $PHH$Lp D$$A   Ll$Ll$,fD  HHUH;E  AIE   IVI6LHUHuHXD$    fD  H=" H
BB    H P  D$        L|$H<    LwRLHYD$    t    Ll$H\IU HD$@IEHHD$XHBH+BHEHD$PIEHHD$0HD$hHD$HH9  Hl$HHD$0L LN\H|$HIANHD$8Hh1IA   LHt$H0  H  ID$I|$HHHpH+p6WHLl H$L`0D$$+  AHl$PAIIEIE  IHT$HI4$x     HHuHL$HT$hHXHtSH|$h JIHD$(   HD$(fL|$I   L+ZLH X H|$hPI   fD  H|$8NPLt$ILH   
  11HHAPA   H|$@SD$    fHt$8H|$@1/NHD$0HD$0H9D$X+Lt$ILH   
  11HHAPA~YIH|$@Hh  cXHHIH|$   [AE PAU IL    IH|$   Nf     AWAVIAUATIUSH  dH%(   H$   1%  H.:" LbAH ! HL   t6   H$   dH3%(   Y  H  []A\A]A^A_fD  ID$ H|$Eg-      H<$HD$KA   H|$WEHD$   AMfAIO|>(fIM9   HI|$
  IHI<$
  H$H|$LHJuH4$L   lUH|$QD  H	9" H
8       H P  fH$H8 	      HMHLU H|$HHh  <VLHLf.     AWAVIAUATUSHXdH%(   HD$H1FH|$(vRHS8" H
d8    H P  D$$   HL$HdH3%(   D$$/  HX[]A\A]A^A_    H8" HZAH! H|$(H H   D$$tD$$        Hk H7" McIK|7Le H 
  LHHTI;D$)  HI+D$K|7HL$HHD$Hi7" H 
  H5@ HO   IL9ILHIHD$0NI;F  H|$0E1UAHD$8M  L}Hc\$HEL|$I9L$       D  HL$I9H	HQHL$N,"   L1MID  IL;mt7IE H|$LH@J4 NAu1҅IL;muɃH|$0GLxIHD$HxH@LHHpH+pPHM<H*6" H   I 1H|$8LIHD$HEHL$H9t	L}H5" H|$8H Hh  lSH|$(H   QH5" HIwHT$LE1ɹ   L7 H RHU! H|$8	  ZYuvHcD$DH8! HDtZH|$(Iع   +      NND$$   H|$(Iع   +      %NH|$0kOD$$   H|$8MD$$   BIfAWAVIAUATIUSH   L-4" LzH! IE L   t   H[]A\A]A^A_fD  M UIL$LI?GHHtHLQHIGIE LHh  QLHfD  H)4" H
6       H P  kfAWAVIAUATIUSH   L-3" LzH! IE L   t   H[]A\A]A^A_fD  M UIL$LI?FHHtIHHpH+pH*SHIFLLPLI	FIE LHh  PLHrf     H!3" H
5       H P  Cf.     AWAVAUATUSHHH   dH%(   H$   1tQH2"    H
[4    H P  H$   dH3%(      H   []A\A]A^A_D  L%q2" LrL-f! I$LL   t
    L{I$LMv HL   uM &      HIDHLLKFHt!I$HLh  OHHA5HHN{ FAWAVIAUATUSHH(dH%(   HD$1Fh  H-1" LrAH! HE L   D$  AMn T  HE I
  I} HI'CD$4  AC  A  IW E1I} HLHI   DLHELICHE 11  Lcd$M>H$IM;~u  fD  IM;~   IcIEHH4$HH@J HE p  tH$P]  LCD$   HL$dH3%(   D$n  H([]A\A]A^A_    H)0" H
z0    H P  D$   fIU D$    HBH+BHH   I}I+}1HM1HI.If     HE HH4$h  LWB<fHHE Iw HT$E1ɹ   L1 HRH5! 	  ZYHcD$H! E1H|IW(A M   %      HGD$   HE H   M   %      HGD$   z+Cf.     AWAVAUATIUSHH(c  H." Lrt$H! HL   ADL$  IV HJHBH)HHt  HAL9HD$'  HDL$11Lu  DL$HD$AAHDHD$K    HII~
  H|$HH?xCIWHHt$LHHp  u>L;t$uHHt$Lh  Pf.     I         LFHL$PHH   f     A   H(D[]A\A]A^A_     HA-" H
-    A   H P  @ HH}
  H|$HH>x<IWHHHD$& M         LA   `EaI         LA   E<fD  AWAVAUATIUSHHt7Hx,"    H
/    H P  H[]A\A]A^A_D  L%A," LrH6! I$L   t   @ IF I}L0HD$I$
  II$I} 
  LLH$=LcME   H4$L=LcE   I$II  H$HD$LxL;xu   D  HD$IL;xtqIH4$HH@J0J(I$  tH$PI$H߽      fL$         HDI$H4$Hh  f.     AWAVIAUATUSHH(t7H*"    H
+    H P  H([]A\A]A^A_D  L%*" LjH! I$L   t   @ M} IvI~ IHt$HD$I$
  II$I~(
  H|$LI<M   H|$LD$;T$xqMwM9wtUHLcHIHD$@ IHL$A   Ht$HH@HJ(I$0  H8IM;wuI$H  M         H}B     AWAVIAUATUSHH  dH%(   H$   1Ƀt$Q  L-*)" HZH! IE H   t6A   H$   dH3%(   Da  H  []A\A]A^A_@ MgIE HH[ H! L   AuHD$Md$       HH$:H$LHGHI   D$I_ ID HD$R     HL{IE HE! HL      Iw H$LIGLI@MtYMH;\$uIE LHh  EHH    H'" H
j%    A   H P  H4$HC    LA   :@;AVAUATUHSt0Hq'"    H
h.    H P  []A\A]A^@ L%A'" LjH6! I$L   t   []A\A]A^ IE H8:II$L  II$LP0I$LHh  []A\A]A^D  f.     AWAVAUATUSHH8H|$  H-&" LbAH! H|$HE L   AtA   H8D[]A\A]A^A_ ID$ AOLs ωL$HD$(H HD$HCHD$ BL$Ią   AGL|(LAM9   HE II~
  H|$HH7yIع   H|$      A   >L]8;     H%" H
z&    A   H P  f     HL$IT$I+$HAH+AHHH9}}HE H|$ 
  H|$HH7xL`DI;D$Iع   >H|$(LH5Ht9HU HHh  uBH|$HifD  L%    H|$L\'          A   ='@ AWAVAUATIUSHH  dH%(   H$   1tQHR$"    H
'    H P  H$   dH3%(   \  H  []A\A]A^A_D  H$" HL$HT$HsH x  t   D  Ll$      L5HS HsLL;HI      4LHH";HjBHËD$)uD  1LHk6D$HD$D$~NH;4IHD$I6LLHE?uLL   a?L7H;@ H"" HH Lh  @LHALL   ?7fD  AWAVIAUAT   USH   Hh  Hl$pH|$ dH%(   H$X  1H4HS HsHL:H  HS0Hs(HH|$ If:HI     p3LIH9LL9L@IHD$<L=!" HsH|$ IHD$HD$HIHD$HD$@HHD$HD$PHH$I  D$,tl    I<$3HT$@Iu HHH=t|HT$HI6HH=te1HLh4IHL$HT$Ht$H<$  D$<tIH<$  ILHh  >H|$ H;f     H|$ HC=H5IH<$  L9D$,   H$X  dH3%(   D$,u.Hh  []A\A]A^A_D  H|$ H<D$,   4@ t+HHx " H
!    H P     HÐH     AWAVAUATUSHHH  dH%(   H$   1tQH " A   H
"    H P  H$   dH3%(   D   H  []A\A]A^A_ L%" LjL=! I$LL   t
A   fLuI$LMm HL   uHm I$LMv HH   AuHE Hl$      HHD$o1HT$HLL1Ht I$HHh  <HHHH<;MB3fAVAUATUHSt0H"    H
%    H P  []A\A]A^@ L%" LjH! I$L   t   []A\A]A^ I} 5HId2pI$I}  II$LP0I$LHh  []A\A]A^Ðf.     AWAVAUATIUSHH  dH%(   H$   1   H" LrH! A   H L   t3H$   dH3%(   D   H  []A\A]A^A_    IF L|$      LHD$/HD$HxH@H)HHwVH?HMSML-Avf     H!" H
    A   H P  Ff     LL   P;LL%9+1f.     ATUISHHdH%(   HD$1   H" HHT$HqE1ɹ   L$ HH RH! 	  Z   Yt%H|$dH3<%(   u\H[]A\f.     HcD$H
! LHHTH" LH
    HH P     B0fSHH 5HCH/C[fSH H9HtHC H! HC[ f.     SHH 34HC     HC    [ÐHq" SHH   H]! HX H@    @    HP[ff.     AWAVIAUATIUSIIHH?,LLLHHJ2uN1HL-tH[]A\A]A^A_fL   L09LL    7fD  H   k/f     AVAUIATUISMHH dH%(   HD$1S9HHa" LHL$HT$HA   H x  t&H\$dH3%(   D   H []A\A]A^ft$HA4*    HD$LHHHPHT$H	6u=D$PT$IFHtH@HtLH! In IFe HA   :2O-ATUHSHH   dH%(   H$   1H]" HL$HT$H x  t.   H$   dH3%(      H   []A\fD  Ld$1Ҿ   L<+|$t%LH   U7LH*5     HD$LHH0}7HvHT$MHHHHR-\,AWAVIAUATIUSHIHLHHq" H:!    H    tH[]A\A]A^A_    HM I} HqHL$5HL$HLIH+tL*@ H   LX6LL   (4fD  AWAVMAUATIUSHIHHHH" Hz! HA   H    tHD[]A\A]A^A_f.     HS I<$AHrHT$'5HT$HLLH$ ,tH<$)륐Hھ   L5LHA   _3 f.     AWAVIAUATUSHHdH%(   HD$81L$   uAI>I0LE12H\$8dH3%(   D-  HH[]A\A]A^A_f.     H-" LILL$LD$IHHE I:L$
  H=P Hƹ   L$LD$LL$lIBL$HL$0HT$,LHIHt$HD$I HD$HE x  A   D$,L$   LL$/D$,   HD$0L$W       LωDL$/1L$      D$,PHD$0T$,HHD$0   HU LL$H8
  H|$H9'$HD$0HxHE 
  H|$H'D$LL$E\HD$0   LHo3D  HT$	   LV3LL+1A   A/HHD$0   LHP3HD$0   LH3HD$0   LHP2(    USHHHH-d" P~HE HH@0H[]D  HE    ff.     AWAVAUATIUSHӺHH-" LnL>Lv(HHHE   HE H5 H  HE Iu HL-ż!   HE H5 H  IcT$IcD$HHPIt HE   HE H5f H  HE I7H  AD$HtHE H52 H  HE HߺH5   IcT$0IcD$4HHPIt HE   HE HߺH5   HE I6Hߺ  HE HߺH5 H  H[]A\A]A^A_f.     USHH&xHp" H P(HxHHHx#HH[]D  AUATUSHHHH   L%%" P!   fD  +H*H}%H}HGH+HHwh%H}_%H} HtfD  HHHHuE0HU8t;H]@H@HLlhH{ tI$Hx  H{3$H`I9uI$H} P0I$HH@0H[]A\A]     L%A" BI$   3D  ATUS7H      tC   LgI<$}$I|$HtEH-" P9HE    -@ LgI|$ Ht&I|$@Ht~&H-" HE LP0HE H[]A\H@0@ H-y" H=: 1HU R     LgI|$(Hu     LgI|$#AUATIUSH(H?dH%(   HD$1Ht#H-" Ml$HHLHE   HtfHxHE H  HuHE LMl$`x  HE HL  Ht!fD  HxoHE H  HuHE Lx  I$   Hu9HE HLP0Mt&H{L+PHE        HE LP0HD$dH3%(   uH([]A\A]^#@ f.     AWAVAUATUSHLHI9   LD$IAAH   ;f.     u$EuHHL!H|$*1fD  HI9t.t~EuHHH!LX*1I9uH[]A\A]A^A_ø   f.     AWHcAVAUATL$RUSIIH8Ho Nl&@Ht$H1dH%(   HD$(1(H]H;]HD$tNLD$Lt$MHLLHD$A$   HtHH+uHPLH'HH;]uHD$HG HD$(dH3%(   uH8[]A\A]A^A_! AWAVIAUATIUSHIH5Y HH;
" L$LL$H  HLH  HH5/ H  Mu L;4$   AIcIIH<
(IHDH5 H  HDLH  HLP0HDH5h/ H  HDHt$H  HDH5l. H  L;4$oH[]A\A]A^A_AWAVAUATIUSIHH(I H-"" LL$LL$`Ht$H53 H@ LL$H$HE   HE M<$HMt$Ht$  HE HߺH5   HE I7HߺL=!   HE HߺH5   IcT$IcD$HHPI4HE   HE HߺH5   IcT$IcD$HHPI4HE   HE HߺH5   HE I6Hߺ  HE HߺH5G   HL$H$HIu LL$LAH(HP[]A\A]A^A_G    AWAVAUATIUSMHH(L=
" HD$`HL$Ht$H5 HD$I H@ H$I  Mt$II,$Ht$HMf  IH5 H  IHu H߽  IH5A H  M6M9t>f     IHIH0IH6  IH5 H  M9uˋCHߍpI  IHߺH5   HD$H$LMELL$HH0H(H[]A\A]A^A_D  f.     AWAVIAUATIUSHMHͺHI H5t Ll$PH@ HD$H" H   HLLHD$MD$MHu HLH[]A\A]A^A_HPJf.     AWAVHAUATHUSIIHc1HdH%(   HL$1HT$PPT$   L=]" Iƍ|mIǘ   P(LHDxIP(LHHL'H%HC    <1HC1HTm H{8HCHC     C,    1C(    k0H0I^HL$dH3%(   HuH[]A\A]A^A_D  HSHHH6PHH" HHp  H[# HHHPHHt
H@H@ 1ff.     AUATIUSH   HL-" IE P(HxHHH     Hǀ       1HH)   H   HIE H{1  IE H{`1  IE HLHHc!   HH[]A\A]    HHBH+HÐf.     SH?H4 HHx[    HA" S   H P(   H`HCH[    USHHH;(H[]HM f.     ATUHSHH?HGH+HHw2H߾   % H;HHA   uD[]A\k E1tHEHx H}HD[]A\D  f.     SHHHD" H[H H@0D  f.     HHHH~`H   HtH@H 1@ SjH     H" 0   H P(HxHCH     H@(    HH)H01HH[ÐUSHHH~`HHdH%(   HL$1HT$   T$t*HHO HEHL$dH3%(   uH[]@ 1    USHH    H"    H P(   HHC9HEHH[]Ðf.     USHHH~`HHdH%(   HL$1HT$   T$t*HHoHEHL$dH3%(   uH[]@ 1    SH    Hb" P   H P(HxHCH     H@H    HH)HP1HH[ÐUSHHH~`HHdH%(   HL$1HT$   T$t*HHHEHL$dH3%(   uH[]@ 1    USH%H    H"    H P(   HHCHE HCH@    HH[]D  f.     USHHH~`HHdH%(   HL$1HT$   T$t*HHHEHL$dH3%(   uH[]@ 1G    AVAUATUS7HJ  4  to   LoIE H(H;ht;D  Le HI|$I|$H;GtHPHH IE HH9huH[]A\A]A^:f.     HGHL(L`Hh(I}UI}H;GtHPHI|$H2I|$H;GtHPHH}HH}H;GwHPHH[]A\A]A^D  H " H= H1R H[]A\A]A^zf.     HGL HhaLwHI.H}H}H;GtHPH'IFLhH(I9 HE HL I|$LI|$H;GtHPHHI9uIFL`H(I9     HE HxHtHI9uH[]A\A]A^D  HHSH~`H   H1Ht!H{H! HH p     [D  f.     AWAVAUATUSHHH4$7    t0HI! H= H1R E1HHD[]A\A]A^A_D  HoHGL}HD$ HE HM ULHxHD$IGH_ H@ IHD$0I~I+~1HH} H¾   I11IUuLILILD$<L$u6H<$ Mgt+IEI+E HH?	  IFI+HH  LLH{H+{1H;H} 1HI*11I11ILLILAǉD$(Eu;H<$ HD$L`t+IEI+E HH  IFI+HHY  LMLEHE L}(HM@HPLHP IGU8H@ IHD$8HD$HxI~I+~1HZH}@H¾   IF11IU0u4LILILD$L$u6H<$ Mgt+IEI+E HH  IFI+HH  LlLdH{H+{1HH}@1HI1Y1IO11ILLIjLAEu;H<$ HD$L`t+IEI+E HH  IFI+HH3  LLHE@uHHPHP   L$T$(D$EA   ~LoHGM]HD$Mu IM(AU ICLL\$I~HX L FH{H+{1HI}(H¾   Ho1(1HAUAuIHHI5H߉D$
D\$Eu=H<$ L\$I[t-HEH+E HH  ID$I+$HH'  H
L
II+1HI}(1HI1w1HmAUAuIHLHLA)
Eu[H<$ MftPHEH+E HH  HCH+HHv)H  HLIMLPHt$ H|$XZHHIE(HPHP D$E1EAA!fHGHO1A   HHD$HL$(HH@H@HL$IHD$ HAH@ HxH+xHHD$HCH(I9  D  HE Ht$HXL(HD$I}HًPLw HT$   HN
IVI+VHHH9      IVI+V1HIHL0L;pIu6fD  IM;wt&AtLLIpLM;wuLhH<$ tID$I+$HH   LE1?HH9l$ HCHC  1A1H7HHD$L0L;ptVI
E1H^IM;ut4A>tLL
HE14IM;uuf.     H<$ HD$Lpt+HCH+HH+  HEH+E HH   Hm
He
H|$[
gfD  H%  HMEMLPHT$ Ht$8H|$AZA[HL$0HSH+SH\$8HAH+AHKH+KHHHHH9:H#! L<$H5  HL  H|$ LHHH5  L  H{  HIMPHL$(HT$ Ht$8H|$^_H!  HIMPHL$(HT$ Ht$8H|$AXAYH  HMMLHPHt$0H|$AZA[H  HMMHPHL$Ht$0H|$^_wHm  HMMLHPHt$0H|$A[A\(Hq  HMMHPHL$Ht$0H|$_AXH  HIMLLPHt$ H|$D_AXH  HLIMLPHt$ H|$Y^H  HILILPHt$ H|$L\$(AYAZL\$+HM  HHMMLPHt$0H|$XZH`  HMMHPHL$Ht$0H|$AXAY
H  HHMMLPHt$0H|$]XZH   HMMHPHL$Ht$0H|$-AYAZ0@ AUATUSHHH{HHpC0tILk8Ld@LIM
H`L9t'H3H|tL)Hi۫f     HH[]A\A]fD  SHcH dH%(   HD$1HGH$HX HvHHHH|@HJ8HL$   Ht$HcPHCHH\$dH3%(   uH [fHCD  AWAVAUATUSHHdH%(   HD$81G0HL$Ht$    IHw8H@L@Ld$Hl$ HcHLlh I`HXM9tcIGLHLHD$(AWHDD$EuL|$Mt?LDD$
   HLHH! HH P0DD$fA   HL$8dH3%(   DuHH[]A\A]A^A_ AWAVIAUATIUSIIHG0tXHW8H-! H@H_@HHDhHD$ HE Hx  HE HH! H`H
  H;\$uII;] tNHf     HI;] t6IHH3LH)LHuH[]A\A]A^A_     H   []A\A]A^A_ff.     HFATUSHIHHHu'HCH1Hp HVH[]A\@ H{   HGPH{H! HGH! Lg H `  []A\@ f.     AVAUIATUISHGH^ Lh H3I} jHtEI} HyH{HsHHIH+[LLL]A\A]A^D  HxI AVAUIATUISHGIHh H/t[HEHULH)HHcH4Lt   []A\A]A^ÐHUHHrp[]A\A]A^D  L6
LH¾   HH! HH P0[]A\A]A^     AUATUSH(dH%(   HD$1G0H4$tSHW8H-U! H@H_@IHLlhf     HCHLHD$S@HHE H`p  L9uHD$dH3%(   uH([]A\A]H     AUATIUSHHH6ID$HSHHh HPH;EIt4H@ H3LUHH+UH31LHH	H9]uHL[]A\A]D  USHHH.H{HBHHH[]1D  f.     SHHHH{[H! ATAUSHHӿ   H P(HU D`HXHHE []A\fD  AWAVIAUATAUSE1HH-! H_ HHE 
  HH$t`E1 IHHtMD9kuHE H{
  H4$HuMHt9IHHD$HAH\$HuHD[]A\A]A^A_fD  HD$IF D  AWAVAUATUSH8dH%(   HD$(1FH<$   H.  L=! 1I11Ld$  HD$IH}L  HthMDLp QuLp HtIHL  t+ILLp    Ht$HH<$AՅurIL  HuIHt$H<$h  1H\$(dH3%(      H8[]A\A]A^A_fH! H
     H P     D  H\$B   I$H   $fD  L=! HzI
  H    AWAVAUATIUSHH8  dH%(   H$(  1HD$8       L=5! IHzAI
  LHHHHtHpIHH!    taA   H$(  dH3%(   D   H8  []A\A]A^A_    H! H
     A   H P  @ HLHuHD$@<      HHD$HL_  E(D$$  ID$E(   HD$HEH@ H8	HD$(AEMdHD$8HD$L9d$H|$8tiHD$LD$HHD$HL$HHPHItHT$8Hp H|$(AE PAU    D$$L9d$H|$8uE1E(    LHA   D$$t
IH}`  IH|$(Hh  HHMIT$-   L|$LLH^ f     IL   T    IT$,   @ H|$(~$@ AWAVAUATIUSHH8  dH%(   H$(  1HL$HD$8       L=! HzAI
  Ht$HHHHtHpIHHG!    t]A   H$(  dH3%(   D  H8  []A\A]A^A_ H! H
"     A   H P  @ Ht$HHuHD$@      HHD$H|$H  De(E   HEE(   IH@ H8[AHD$  AFDd$,E1D$(HD$8HD$ AD;t$(A  H|$8IIULD$HHL$ HHI~   Hp HT$8H|$   A$PA$IL   D  IU-   L|$LLHef.     IU,   D  Dd$,H|$81Ht$E(       H   E   IH|$Hh  HHfD  Ll$L   HDd$,LsLHHx H|$8Dd$,Ht$1E(    HKu?IH}`  nH|$8kHt$1E(    HFD  H|$D$*D$A?f     ATUSHt,HU! H
     H P     []A\fD  L%)! HHzI$
  HHHH¸   HtI$H:Hh    HH[1]A\ f.     ATUSHt,H! H
6     H P  [   ]A\fD  L%! HHzI$
  HHH"I$HHh  t"     HH[1]A\     1  H AWAVAUATUSHH8t:H! H
     H P  D$   D$H8[]A\A]A^A_ H-! IHzHE 
  LHHjHIt HpHE HH!    D$tD$        ID$11Lx HE   HD$ ID$8HD$AD$0   HD@HHD$HD$(HD$11L(HE   Me M;eHD$u   D  IM;etvA4$I?rHU H8Lp    Ht$HHAօtHL$P~vHL$ PHE H       HE HT$HHt$ p  uHD$`HD$H;D$( HE Ht$ Hh  JHE H   x@ f.     AWAVAUATUSH  dH%(   H$x  1H|$HT$(HL$tPH!    H
     H P  H$x  dH3%(     HĈ  []A\A]A^A_@ HD$(HxH! H 
  Ht$H|$HHHD$0t)HD$0HO! H|$HpH?! H    t
   iHT$0Ht$H|$,uHD$0H! H|$H@H@ HD$ HD$(HXH! H H   uL{ Ld$ IHXH;Xu$^fH0HU  IHH;Xt<H+I<$Hu I$H;BuLE    H|$C      i*H\$0{0{0HD$8HD$PC0HK8D$@    HL$HIH@H   IE1HxH+8HHD$IHH;XuCx    I?fLH{tEH|$AHIH;X  HD$ 3H8'I?H0IyuH|$8E  HD$0I`D$@@0H@HHD$HI98HL$8HAH+HH  IHxH+xHHHD$8Lt$8L L;`t2@ M,$I] I;]tf3HHI;]uIM;fuIHHpH+pH{HHD$XnH$   C      HHD$@Ht$0H|$R  HD$0@(  HD$ H8KHD$xIGI;GHD$(   H$   Hl$0L|$HHD$@ HD$(HL$PL(HD$8L1L8HpI9  LE1IHLID  L$   HE Ht$H$   IcH@HI|@A   HtH@t9  AHcHD$8HIH;huH0IMcLLH)HL9   HL$HHD$(HD$(H;A&H|$XHD$0Ht$1H|$@(      H|$x   H|$PHD$8Ld$8H(H;htH} HI;l$uH|$8bD  QHt$H|$8t$@H|$P/HL$8LMcHcLyH1	LE    	HD$ H@L<I;~HHHD$XL L;@   Hl$hL|$pLILt$XLf.     HH I;^   HMHcHUIHIL,HH0	IU H|$IHL$@H0IWL+bIMcILIWH:Htq7HD$`H! H   LHD$`IW\LHl$hL|$pHLL$@LD$xHH|$LHH|$XHD$0Ht$1H|$@(    H! H|$xH Hh  H|$HHD$(H|$   C         L@QH|$8   -H|$PSHD$(H|$   C      L@E1HH|$XHD$(-   HPH\$@HH|$H{H|$XHD$(,   HPg    AWAVIAUATUSHH8  dH%(   H$(  1tQH! A   H
     H P  H$(  dH3%(   D5  H8  []A\A]A^A_ L%! HHzI$
  HHHHItHpI$HHH!    tA   | LHH:uIGLl$@B      LH@ HD$2I$HL$8HT$0Iv Hx  uHL$8T$0LD$4MLHHxt$4HLHD$
HL$HD$HXLH  AG(D$T  HD$AG(   H8@HD$ HD$HT$H;PF  H:IFHD$I I$LLT$(1A   Ht$H0  HLT$(B  HIN(MAULL$0LHT$(HLT$8AAXZLT$(   I$LT$I`  LT$I$LT$H|$L   
  11HHALT$APA   T$AG(    HHunAthAI$H|$ Hh  HHbMF-   B      H1AG(    HH\AtH|$ {MF,   D$   AD$   A   PAI$LLT$   LT$I$L   f.     AWAVIAUATUSHHX  dH%(   H$H  1tQH"!    H
     H P  H$H  dH3%(     HX  []A\A]A^A_D  L%! IHzI$
  LHHjHItHpI$HH!    t   |@ LLHuIFA      H@ HD$HD$`HHD$ }LL  An(>  IGAF(   HD$ IG  HD$IG(L|$ I?HD$0IOI9OHT$\HD$(e  l$LLl$@HLt$8L|$I"fAPA  HI;o  H}  II$1A   LHt$H0  H  I$LHt$HP    t$\tD$HHt$(LL$8MHL$@Ht$HH=ZYD\$NMLl$@Lt$8AJA  1tE1ۃA@ AF(    DLH2HL$(   EHAH+AHH-  I$H|$L   
  11HHAHL$0P  HL$P&  HL$P      H|$(I$Lh  HHA    I$L   HfD  IW-   L|$ L:LH4f.     MLl$@Lt$8APA   A      IW,   Lt$8l$LLl$@E1I$I~`  H|$(>I$H   I$H   I$H   I$L   eQI$D\$ LD$   D\$ D$ f.     AUATIUSH~?FIHZHlD  HHSLL5uH9u1H[]A\A]Hp! H
z     H P     ff.     AWAVAUATIUSHH(  dH%(   H$  1H$~}H-! HzAHE 
  H4$HHHItHpHE HHԅ!    tZ   H$  dH3%(   _  H(  []A\A]A^A_@ H! H
2     H P     D  H4$LHyuID$H|$0      H|$H@ HD$nH<$L  AD$(D$$9  AGAD$(   MuLd$(IDLl$Ic@ IM~HE HՄ! HL     Iw HL$1L6L9l$tLHD$BHD$H  IM9uLLd$( HL$HLHIHtHT$HLuHt$HD$$   AE PAU    T$$H4$HAD$(    {6HU $HIt$h  $fD  IU-   Lt$L*LHf.     Ld$(D$$   zfD  IU,   D  Ht$HLd$(D$$   EHE L   3fAWAVAUATIUSHH(  dH%(   H$  1H$~}H-! HzAHE 
  H4$HHHItHpHE HHԂ!    tZ   H$  dH3%(   _  H(  []A\A]A^A_@ H! H
2     H P     D  H4$LHyuID$H|$0>      H|$H@ HD$nH<$L  AD$(D$$9  AGAD$(   MuLd$(IDLl$If@ IM~HE HՁ! HL     Iw HL$   L3L9l$tLHD$?HD$H	  IM9uLLd$( HL$HLHIHtHT$HL~uHt$H}D$$   AE PAU    T$$H4$HAD$(    x3HU $HIt$h  $ IU-   Lt$L*LHf.     Ld$(D$$   }fD  IU,   D  Ht$HLd$(D$$   HHE L   6fAUATUSHHdH%(   HD$1tFH!    H
:     H P  H|$dH3<%(   (  H[]A\A]@ L-! HHrHT$E1IL     IE RH! 	  ÅXZuht$t'   tbIU H=G  1R pD  I$1LHHHH+H,1LH/;        )fD  LI$   LHHHH+H@ I$1LHHHH+H   LHf.     AVAUIATIUS   HHω1H   HLHHH+HLtIuL-d! }IE    HIM 1L8	     H1HLHHH+H1҃HL   E[]A\A]A^    IuL-!  H!    H
     H P  []A\A]A^@ IE LL   L     H=v  1ALHA?AWAVAUATIUSHHdH%(   HD$1  L5D! HAHrHT$E1IL     IRH߆! 	  ^_t3   HL$dH3%(     H[]A\A]A^A_f     HIIt$HT$E1ɹ   L'  HRHU! 	  ZYu\$      4$     j  M  IH=  Lh    HHA8fD  H1! H
r  H L   H   P  @ IH=  1R A  IH
  L   HP  II|$ 
  LHHdH$   ~]      A  IH
  CfuIA   IH
  L   HP  4fuWA  IH
8  IH=  1R A   IH
  L   HP  IH=d  1   R A   IH
  L   HP  AtOIH
  \IT$ LH+fLHYTIT$ LH=HH+IL$0IT$(HHIL$0IT$(HHhIT$(HH5D  AWAVAUATIUSHH  dH%(   H$  1FvNH! A   H
     H P  H$  dH3%(   D_  H  []A\A]A^A_HQ! HHzAH 
  HHHHIt"HpH! HHy! H    tA   w LHHAuIEH|$ 3      H|$H@ HD$AtH! IuHH h   H! M|$HHx! H L   hIW HD$H2H8HT$HT$   LH  AE(D$   AE(   Ld$LALHL+H   LHLHD$HL$   P   APA   T$AE(    HHA
H:A   kL|$H¾   HLLHHY! HH P0IT$-   L|$LLHqIT$,   D$   A^Ht$HAD$   HL$
H! LH    H! HH    D  f.     AWAVIAUATUSHH  dH%(   H$  1   L-n! HHzAIE 
  HHHHItHpIE HH2v!    t^   H$  dH3%(   0  H  []A\A]A^A_fD  H! H
        H P  D  IFLL$ /      LL$H@ HD$AT$L$   IOLD$LHbHIHHD$H8It$LLILH$HL$H$H;At
H01L/IE LLh  HHAfD  H  LϾ	   L$HL%+f.     ~HcIȃHLH@ HH! H
     H P     HfD  SHH ~@u-H^! HH
     H P  H    [ H1! HL$HT$t$H   HT$t$HHL$H [HAWAVIAUATUSHH  dH%(   H$  1tQH¾! H
     H P     H$  dH3%(     H  []A\A]A^A_    L%q! HHzI$
  HHH
HItHpI$HH8s!    t   |@ LHH*uIGMnHHr! H@ LHD$I$   uIE Ll$ &      LHD$LH   AG(D$   Ht$H|$LAG(   H   H LHLHIHtLHLuLHD$   APA~~T$AG(    HH^I$D$HIwh  D$@IV-   L?LHIV,   LHD$   I$L   p     AWAVAUATIUSHH(  dH%(   H$  1H$~}H-N! HzAHE 
  H4$HHHItHpHE HHq!    tZ   H$  dH3%(   _  H(  []A\A]A^A_@ Hѻ! H
r     H P     D  H4$LHuID$H|$0      H|$H@ HD$H<$L  AD$(D$$9  AGAD$(   MuLd$(IDLl$Ia@ IM~HE Hp! HL     Iw HT$LL9l$tLHD$HD$H  IM9uLLd$(Q HL$HLHIHtHT$HLuHt$HD$$   AE PAU    T$$H4$HAD$(    8HU $HIt$h  $     IU-   Lt$LjLH?f.     Ld$(D$$   xfD  IU,   D  Ht$HLd$(D$$   CHE L   1fAWAVAUATUSHHH  dH%(   H$  1}   L5N! IHzAI
  LHHHItHpIHHn!    tZ   H$  dH3%(     H  []A\A]A^A_fHѸ! H
r        H P  D  LLHuIGLL$       LL$H@ HD$LLL$  AG($   AT$HMLD$AG(   LHHIĺ   tjt$HL{LHHD$1H;htCH} j HLHI&   A$HA$   AG(    LHI<$Lh    HHAufHU-   LL$L$HLE    HU,   LHLD$II`  $   T$:IT$L   T$5H     AWAVIAUATUSHHHdH%(   HD$81tOH! A   H
;     H P  HL$8dH3%(   D  HH[]A\A]A^A_    H-i! IHzHE 
  LHHHItHpHE HH0k!    tA   fD  LLH"uIELLH@ IX  AE(D$$   IFAE(   H$ IF Mw M;wHD$2  HL$4Ll$Ld$(HL$'D  IA$PA$   M9w  I> IHE 1A   LH4$H0  HtoHE HT$HHt$P  AuQT$4tHt$LHQu5H|$LD$$I_ HE L   ^fD  MLl$Ld$(APA&  A   L<$HE DD$AE(    L   L
  11HHAADD$PA   HL$P   LDHAD$$u_HE |$$Hh    HHmMF,         HELd$(Ll$-MF-   HE I}`  HE D$H   D$VHE H<$   DD$)HE L   E1@ AWAVAUATUSHHH   dH%(   H$   1   H.! IHzAIH 
  	      HI;HsHHH   HtHLK ED$AULLHZYH$   dH3%(   u8H   []A\A]A^A_ H! H
z     H P     v0HHe! H
n     H P     HfD  HrHff.     AWAVAUATIUSHH(dH%(   HD$1   L5! t$HHrHT$E1IL     IRH.p!    	  ZYue\$  H
پ  DT$HcHf     A  AH59  tII}
  HLH@ HL$dH3%(     H([]A\A]A^A_ H!! H
        H P  D  AN  IH
t  L   H   P  @ A  IH
  fA   IH
  L   HP  >fD  ABMeMl)III|$
  LHH   M9ufD  IuLHfIH=ܼ  11R      II}
  LHHEI1H@Hh    HH1uD  IuLHZfIuLHaBf.     IH
l  !@ 	t+HHX! H
9     H P     HÐHrH@ AUATIUSHHdH%(   HD$1   H! HHT$HqE1ɹ   IL!  H HRHW! 	  Z   Yt HL$dH3%(   u_H[]A\A] HcD$LW! LLHHAT @ Hi! LH
U     HH P     f.     AUATIUSIHHHH! H  HLA     Hx  tHpHE1H  HHH5    HLH  HD[]A\A]D  f.     AWAVAUATUSHH8H-! dH%(   HD$(1HT$HL$ HT$HE x     |$    A   HD$ E1Ll$L%z  7D  HcD$HX! HD|D$PHD$ T$HHD$ ~^L] HE1AU   HW! MH0HA	  ZYtA   H|$(dH3<%(   Du>H8[]A\A]A^A_E1@ HD$D8fD  HE 1H5  HP  f.     AWAVAAUATIUS11HEW! HL%2! HT$I$  HŸ   HCt]AtI$H;Lp    HHLAՅtǋU BE    &I$D$H   D$D  HD$H(1H[]A\A]A^A_D  f.     AWAVIAUATMUSHHXHj! I>dH4%(   Ht$H1MHT$HL$HLDL$   Av,HT$8HIHD$t<APA  E1HL$HdH3%(   L  HX[]A\A]A^A_ÐH|$8M~ M  HD$@HD$ HD$4HD$(
M?M  AGAF,tH11  IHIWLHh    HHT$8LHp  {  HHT$LHp  _  H|$ tHHT$LHp  ;  H<$ tHH$LHp    HHL$ LHT$(Hx     H   HT$@t$4H0	     H|$ t^MtAE PAU -  HH@   IHH  HLHHT$   uH<$    L,$A$PA$HL   M?Mx@ H|$8P   HL$PHH   D  A$PA$~yH|$8P~yHL$PHH   H|$ tXLl$6HH|$E1   a     HL   HL   vH   yHD$H$H   _D  f.     ATUISHHHHv;:t MtHHLHE []A\@ {:uL%! HI$  I$HH  HE []A\     AVAUATUSDH   dH%(   H$   1A    IIHLHH:LHIHt_L5! HHAH  1LLI	  IH  H$   dH3%(   HuCH   []A\A]A^    1L5! D  H! H=д  1H1R     AWAVIAUATIUSH  dH%(   H$   1>}   HN! Hl$ HH  HL:At;HH  H$   dH3%(   D  H  []A\A]A^A_ HHL      H^Ln:   D$    H蓺H
  LxL9E  H! H=  H  It$HL HHH)   H11H5ɲ  H   HIuH 	  H11H5  H   H   HL8	  U JM @  t!p     Dl$D  HLH  @  HT$HLՅuHL  Dl$EHIl$LH   @  E1HH
  HH5  1Dl$FD  IuHU!    HL8	  I}J%HD$   D$    HL  Il$HLH   @  E1HH
o  HH5  1@ HD$H   D$    IuH! 1҃HF     HL  HIl$LH   @  E1HH
  HH5  1Dl$ 蒷fAWAVIAUATHUSIHH-! L`I11HE   HIE M?H@ M9H HD$uVf     IM9tDA7H|$JHU H8Lp    HHLAՅtP~1HH[]A\A]A^A_HE H1   AWAVIAUATHUSHIDHH  H$  Ht$LL$HD$dH%(   H$8  1  LD$`HLALLD$(H|$LHHٷHHD$ LD$(  L%ǡ! LI$  HD$ H8HHD$80  CDt$LIHD@HHHD$@HD$XHD$0HD$THD$(H\$I$LHL$0HT$(H3x    |$T}  HJ@ I$H
  HD$XHHH8HL$XLHH  H"  D$TPT$THHD$8L9Hu*  f     H;HuuH`L9   H3HYtHD$H|$
   H޾H&Ht$ H|$跳Ht$L蚼A   ,fH\$H4  )   A   HAHLfH$8  dH3%(   D  HH  []A\A]A^A_f.     H|$Hھ   [f     HD$H|$   H6@ I$I.I~HFT! I`H
  HD$L;t$@Dt$LHt$ I$1A   LHNH6	  H   H\$ I$LLL$L"   1H3  I$HsLh  H\$H"   LD$A   HHL&Hמ! LD$H L  HD$H\$   HHHLHt$ H|$6Ht$ H|$A   ˱PAWAVE1AUATUSHHZ;HD[]A\A]A^A_@ L=1! HIIHI   HE P~qIM1Hu LL{"   E1  IHM1Hu LA   	  HIMAL="   1Hu L  N@ IH   fAVAUIATUHSIHH   dH%(   H$   1]Hv	;:~   Hl$HLHHt$LAHu	E   L5! IHD$H  HD$HtEH$   dH3%(      H   []A\A]A^@ {:xHL蛲HuIع#   1Ҿ   LHD$蚵HD$ L5! HI  IH  IHH5    IHH  Ht$L~@ f.     AWAVMAUATUSHH8H|$ Ht$LD$(  HD$׉H@Lh 觸HD$I} 9I   EL=! HHHHD$-H|$贮HSLHL蒷t:HH;\$tcIH;
  I} HH8yH   LTH|$ LwH|$ݭLE1H8L[]A\A]A^A_Ht$H|$x:HL$(HtH|$藭HLϾ	   E12H|$ LH     L迯ff.     AWAVAUATIUSHHHH8  dH%(   H$(  1+uL%X! I}I$
  HHHHItHpI$HH"O!    t`A   H$(  dH3%(   D  H8  []A\A]A^A_fD  L~  )         HA   ĲfIFHL$@HT$0IuHH@ H$I$x  qHI$IuHT$<E1HL     RHX! 	  ^_6I$I} 
  HHHHHpHD$HI$HN!    L\$ICHL$HHT$8Iu(HH@ HD$I$x  HI$Iu0HT$DE1ɹ   L  HRHW! 	  ZY|HcD$<H
W! L\$HD|E@  D$89D$0t ME($         H/11L\$(AHD$H$L\$(H HD$HD$H HD$ D$0PT$0   HD$@J@ (  H|$DL$eT$0HD$@HD$HL$rHHD$@t$0   I$L\$H8
  H|$H{HT$H$I$H:
  H|$ HYD$L\$EgHD$@L    Hߺ      دH|$ME0(   vH|$   L$H|$   L$HHD$LL\$˫HT$$H苩<$ L\$i  H|$   L\$ 踵I$I} 
  HL$PHHHHL$HHHD$2HIL\$    HcT$4H
U! H@I~L\$ L0HHʋrRLXPHcT$<pHHʋ$rRH HL$pPLHH(_L\$ I>I3L$[L$uBLHXtI$H|$  HD$HdH|$HA   I{LLD$%         HA   聮H|$GI$H|$  IME(   D  AWAVIAUATIUSHIH  dH%(   H$x  14$ب   H! I|$H
  LHLHD$蘨HHtHpHLHI!    teA   H$x  dH3%(   D  HĈ  []A\A]A^A_ Ld  )   (      LA   lf.     HEH$   H$   It$LH@ HD$Hx  Ab$   HL$D|$ ILd$(HD$Hl$H	IR    HH
  H$   HHH8H$   LHH蟥  L蝯-  $   P$   Hl$Ht$D|$ Ld$(HTa  HL$D$I<$HAH+HHD$0H
  H$   HLHHL$`+LHHD$hKHHD$@L$/  Ht$@H}HFH(HHD$H<   HD$ ID$HD$$  D|$|Ll$(L4$HID(HD$pH$   HD$PH$   HD$XHL$HH9
  H|$HH˩   H|$ H覫  Lt$(H4$HLHI  HpHLHG!    m  IGHL$PLHT$XHh HD$HpHx  >  D$0;$   ^  H   P(1I1L8赦ID$IHD$HLHxHD$Lu H(PfD  HH
  H$   HHH8H$   LHI7   HUL辬$   P$      LIHt$@^HD$HD$H;D$plD|$|Ll$(L4$Ht$@LM  H|${H|$ AHH|$`  fD  I   L(      莩H|$4    I   fD  Ll$(M   (      L4$LEH|$A   H|$ 諣H|$hLNHH|$`  LMD$   (      LA   yH|$菢M$%   (      LOH|$eHH|$`  L4$TLl$(I"   (      L4$Lv,HD$Ll$($   (      L4$L@LاLl$(LD$*   (      L4$LdH|$hLA   1'ff.     AWAVAUATIUSHHHH  dH%(   H$x  1{   L%! AH;I$
  H=  Hƹ
   u
HA   I$H{
  LHHHItHpI$HHAC!    t_   H$x  dH3%(   $  HĈ  []A\A]A^A_    L  )         H    IGH$   HT$hHsHH@ HD$I$x  nI$H$   HT$lHs0Hx  HHI$HsHT$|E1ɹ   L  HRH|L! 	  ZY
E  HcD$tH
WL! HHDXEt	DPEuLC+         H蓥HI$Hs8H$   E1HL     RHK! 	  ^_HcD$|H
K! HHDHEt	D@EuLC8+         HI$H{ 
  LHHHHD$HpI$HHAA!    HD$H$   HT$pHs(HH@H@ HD$0I$x  I$H{@
  LHH藟HHD$ HpI$HH@!    zHD$ H$   HT$xHsHHH@H@ HD$HI$x  @D$h9D$p&  D$l9D$xtoLCH$         Hܣ    HI$Hs8H$   E1ɹ   LM  HRH2J! 	  ZY}HI$L$H;
  H$   HHHHL$@LHHD$8|HHD$(L$  HD$(HT$1L$\H@HPHL8HcD$tHI! HL$HT$PHHЋp@qA1\HL$HT$P11HA HD$ HA(HcD$|HHBA0BA4$HL$HA@HD$DqHLq L$\H L$PHD$HD$0H HD$0zH$   I$H:
  H|$H譛H$   D$\I$H:
  H|$0H臛L$\  Q  LH$   H$   D$pPT$ps   LU   L(H|$HHD$0HL$HT$0AHћHL$y L$P  L   L$PHD$L$PLp@HD$HL$HH HD$0zH$   I$H:
  H|$H蛚H$   D$PI$H:
  H|$0HuL$P    LH$   H$   D$xPT$xsL$H   LL$0;   LH|$ HHD$HL$HT$A8H跚HL$y8 L$0N     LL$H\$Ht$(H{ HD$ H;H0 L$tHD$ Ht$(HxL$H\$(IL$HܘHHL$uH|$8L蜜   I$L$H|$@  L$LC(H$   L          H谟H|$8LCI$H|$@     H$   LD$8H%         ԟI$H|$@  H$   yH$   lLCH         HfLC(AWAVIAUATMUSHLHH  dH%(   H$  1t$D$(X  L%)! I>I$
  HT$0HHHT$LHHD$ _HHD$q  HD$LxD$I_I?ƉD$,I^蘣D$,  D$IDHD$If     HpI$HH:!    uLI?LɖI~Ht$軖H;\$   I$HH{
  LHHHIuH|$ L[I$H|$  D$(   H$  dH3%(   D$(   H  []A\A]A^A_    L  )   )      HD$(        Ht$HcI$H|$  s    M%   )      HI$H|$  D$(   6蕘D  AWAVAUATIUSH1H8H! dH%(   HD$(1H|$1Ld$H  H$HH}`L  HtjD     Lp QuLp HLL  t+HLLp    H4$HH|$AׅuLHL  HuHH4$H|$h  1HL$(dH3%(   uBH8[]A\A]A^A_     H$B   HD$H   D$`AWAVAUATIUSHHHH-! HE 
  HHL芖HtsHHE 11  IHCLpHI9ug    HI9tWHHE HzLp    LHLAׅtAE PAU ~;H   []A\A]A^A_fD  HE LLh  H1[]A\A]A^A_HE L        AVAUIATUHSIHH   dH%(   H$   1ݕHv	;:~   Hl$HLH{H|$LAHu	E   L5! IHD$H  HD$HtEH$   dH3%(      H   []A\A]A^@ {:xLH苝HuIع#   1Ҿ   LHD$HD$ L5! HI  IH  IHH5'    IHH  H|$L@ f.     ATUISHHHHؘHtHxH觕[1]A\MH&         V[   ]A\ff.     AWAVAUATIUSIHHHH! H
  LHLXH  IH11  A4$HŃO       usHH=  Mt$Lp    HHLAׅv  fD  U BE    EH$H   $H[]A\A]A^A_@ HH=  1R HHLh  1H[]A\A]A^A_f     HH=Ň  Mt$Lp    HHLAׅ[AFHt0HH=^  Lp    HHLAׅ#HI|$Lp    HHLAׅIHH:Lp    HHLAԅIIV 1LHpHHHHLp  IcNIcVL%(! HHJLp  I<  HHLAׅNIVHH:Lp    HHLAׅ IFIV    LHpWHHHHLp  IIV@1LHpHHHHLp  IcF0IcN4HHLp  HI<  HHLAׅsIV(HH:Lp    HHLAԅEIv(IV@   LH|H#HHHLp  	K@ Mt$H=Z  IFH@H$HLp    HHLAׅHI|$Lp    HHLAׅIFIL H:I$HD$HLp    HHLAׅRHD$I6   LHPHH,HHHLp  L9$$uRN    I6IV1LH;HHHHLp  IL9$$M4$HILp  H:  HHLAׅtfD  HH=  Mt$Lp    HHLAׅKHI|$Lp    HHLAׅIHH:Lp    HHLAԅIIV(1LHp)HHHHLp  IcFIcNL%%! HHLp  HI<  HHLAׅvIVHH:Lp    HHLAׅHIFIV(   LHpH&HHHLp  IcFIcNHHLp  HI<  HHLAׅfD  IL&         ΑH   []A\A]A^A_f.     HI|$Lp    HHLAׅbIL L;`tCfD  I$HH:Lp    HHLAׅ IIL9`uHIVHLp  : AUATIUSHHHH-^x! HE 
  LHHI蛐Ht6HU HxHh    HHH1[]A\A]    MH߹&         H   []A\A]fD  AWAVAUATIUSHH>HGH+HGHw! H=D  H Hh    LHӻ   H[]A\A]A^A_fD  I>H$t?H$H@HhHH9tfH;H脉H9u   H<$I>1PHH$H@HHL L9HL$t;@ I$H@HXL8L9t@ I7HI)L9uIL9d$uLeH] I9u/f.     HI9tH3LuH%HH$H@HhHH9t    H;HH9u1ATIUSH>H u1[]A\f.     HL[]A\D  f.     USHHHH(HT$dH%(   HD$17t#   HL$dH3%(   u'H([]    t$HH߉D$轋D$fATUHSHI̻   HHT$dH%(   HD$1ut$LHHL$dH3%(   u	H[]A\{f.     AWAVAUATUSHH11HHL-t! dH%(   HD$81IE   H[ IH   HD$ Lt$HD$LfD  HD$Ht$   HD$ HCHD$(IE Lp    LHHAׅuHHtJsLHtA$PA$~@   HL$8dH3%(   u:HH[]A\A]A^A_@ IE LHh  1@ IE L   T@ AVAUL-s! ATIUSHE,@   H1IE 8	  tE,    []A\A]A^@ IE LL   L     H=V  1ALHA@ USHHӿ   HH`s! H P(H   @    HXHH   H1[]f.     AWAVHAUATIUSHL%
s! H   I$
  HtnI1     HHHtT{ uI$H{
  LH&uHt=HHE L;H{P~/I$HLP0HuH1[]A\A]A^A_L;M   I$        AUATIUSH11HH-=r! HE   H   IHuZf.     HHtHHE HSLLp  tA$BA$   -HU D$L   D$@ HE LLh  1H[]A\A] 1H~  t?N,tfD  SHE11F,   HHC,    [fD  fD  H~  HtV,SHt[D  H&! F,   E1HC,    [ f.     AWAVAUATIUSHMHHFHHL$LLp I>莎Hti HHLHŎIǋP   Mt=I_ I>HsuAL   LLH辌APA~HE1HL[]A\A]A^A_ HsI>HL$HHLHLӁtD  Hp! LH     Hp! HH    F     H~  HtN,SHt[D  HH$! F,   I'C,    [ f.     AWAVIAUATIUSHHH(HFL-so! LD$LL$HH/$! H@ H$IE    t   H([]A\A]A^A_ I>蝎 HHLHHD$dLD$IA PA  P   MtIE H#! LH   uIG HpH$H8uNH\$L   HHHΊAPA+IE L   f.     I6LLD$H<$LHLtXAPAIE L   @ IE H   fD  IE L   fD  HL$H|$LH{~H$LI61LH+PH-mH\$L   HHH   Cf.     AWAVMAUATIUSIH1HHH-Bm! Ld$PHE 8	  At[ttHD[]A\A]A^A_D  HE H@  HMMLLH[]A\A]A^A_H^fD  D$HE HH   L     LALI#}DHLH=|  1AHHDD$Uff.     AWAVAUATUSHHH~  t
F,ItHH[]A\A]A^A_f     IHH!! E1F,   HHHAE,       L5k! Lx HIuH ! I   ugIEI?H@ H0qI?   HLI豀HLևILP0P~i1/f.     IH@  LH¾   贉LH艇    H4  L   <LHa@ IH1   D  H~  tSF,   HE111RC,    [     AVAUATUISHHHw~   L5j! LI
  HHH>~HI   HpIHHi!    A   IT$HBH+HHwjLH_IILI4$H߹"   1  I11I4$H   LHO}IH  [D]A\A]A^    M$'         HłA   [D]A\A]A^ Lr  HA   )   ?      茂[D]A\A]A^AWAVAUATUSHHdH%(   HD$81H      HEi! AIHAξH=q  H  EIžH  H=q    H$HD  H$AE LHD$ H@  L    IHD$0HD$HD$,MHD$u   M?M   H11  HHIWHLh  uHLHLp  uHH$HLp  uHHT$HLp  uHHL$HHT$Lx    E PU UHH   D@ AE PAU    H$P   HL$P~wHLLh  APAHL   fD  HD$8dH3%(      HH[]A\A]A^A_D  H=o    H$Cf.     HH   x    HH   O    HL   #H   HT$0t$,L0	  +{f.     Hf! ATIUSH_H+_H P(HcH1H{HEE     LeA$HEH[]A\D  f.     USHHHHwH{HHpH9t'LE     Hc
HHIHNH9uH[]@ f.     H   AUATIUSHH_HL%e! HHoH+oH9u=f     HH9t'H;HtPI$H   H9uI}zI$LH@0H[]A\A]@ f.     Ht~fD  z AWAVAUATUSHHGLvHoLnL`HXI9uK HI9t?L;LHI7vHHuLIT BuH[]A\A]A^A_    H   []A\A]A^A_@ H9tCUSHHHHvHuH[]    HHH[]z       fAUATE1USHLoHoI]I;]tHHHuH{;~AI9]uHD[]A\A]@ AUATUSHHH;^tDIHE1@ HcHUHH4HUHRH<}AI9]uHD[]A\A]E1@ AVAUIATUISHIH;^H)uIHHI9]t:HcM IVHcHIL$H4IL$HIH<踁u[]A\A]A^ [   ]A\A]A^f.     AWAVIAUATIUSHIH?~Hc! H=cs  H1R LeLLI;D$HtbH01LLH   I+\$HHcHHHUH:Htq7~FH    H[]A\A]A^A_@ LL   wH1[]A\A]A^A_    HYb! HD$H   HHD$HUD  L   L萀1ff.     USHHH_HH;CtH+CHUHH[]f1ff.     AUATUSHH9tTHIHE1 HHSHAHUI9t*H} HtۋBHa! H    E1HD[]A\A]ff.     USHHHrH}HHEHSHHpH+p{HH[]fD  AWAVIAUATUSHHHQHGHIH_HT$HLLhLaLqP|t&HT$IM)M)M)J4;J<Lk{   H[]A\A]A^A_    ATUISH_H{H+{HqHqHSHsHLHfxtH[]A\f.     H1u@ USHHHqH{HHCHUHHpH+pzHH[]D  USHHHH\qH{HH9t+LEHp     Hc
HHIHNH9uH[]@ f.     AWAVAUATIUSHHFLoLw@BHl_! H P(Hc1HEHtI]I;]IT$HMtL   rIH;HiLb HD|0IvxI9]ADLHu˃H[]A\A]A^A_ø   fAUATIUSHIHH}LXLHyH[]A\A]fD  AWAVAUATUSH^H(HGLwHjLa{Ht$LxL;xHD$    M/IIFHHILHD$uHcHMHLH H^ Ht$IH#{HL$HcH HBL9yHu}H+D$H([]A\A]A^A_HD$HH AVIAUATUISH^{HHIHnHcLHHLH^}F H7nHHHL)xLq[]A\A]A^@ AUATUSHHXLd$ HHD$     dH%(   HD$H1LH$    HD$(    HD$0    HD$8    D$    HD$    HD$    $   FmxH\! H P(HLHHHBmHL$HdH3%(   H uHXH[]A\A]pf     AUATUSHHXHLd$ HD$     H$    dH%(   HD$H1LHD$(    HD$0    HD$8    D$    HD$    HD$    $   {H\! HLHH HX(IvxHLHHHlHL$HdH3%(   H uHXH[]A\A]p    AWAVAUATIUSnH  dH%(   H$  1HT$VUUU)R)L$   HD$H|$ Ӿ      H|$Lp|mlIW  +fD  HLrI;D$
  I   IVHL$LI6sHIuL7uD$   H$  dH3%(   D$  H  []A\A]A^A_fHZ! H
$k  HT$   H P  D$   D  LkM|$Ld$IHXI  IHIH~wHL$LI$I7wHuLnD$   :f     H\$I   HoLH0vL(sHKkI     HY! LH Hh  'yLHmD  AWAVAUATUSHHt:H{Y!    H
j     H P  H[]A\A]A^A_     L%AY! LjH! LrI$LH   t       I$HLH   uIv I$I} L  Lh  o1@AHHA^    AWAVAUATUSHHt:HX! A   H
=i     H P  HD[]A\A]A^A_fD  HaX! LbH&
! HL   AtA    ID$ 11LxH  MwM;wHD$uyf     IM;wtfIHH:  IHHt$LHp  tA$PA$~=HL$PgHH   VHHt$Hh  HL   ff.     AUATUSHHt.HOW!    H
g     H P  H[]A\A]L-!W! LbH! IE L   tH   []A\A]f     IT$ IE HRLh  HzH+zH  HHAH[]A\A]@ AWAVAUATUSHHt:HV! H
3g     H P  D$   D$H[]A\A]A^A_ L%QV! HjH! I$H   D$tD$       HE 11HHHhI$IHL$  M~M9~Iu4   @ I$HHULHp  u;HD$IL;xtUII$H:Lp    LHHAօtAE PAU LI$L   : I$LHh  f.     AWAVAUATUSHHHt7H(U!    H
e     H P  H[]A\A]A^A_D  L%T! LjH	! I$L   t
   fD  Mu I$H}Mn
  HHLqI;EII   tH xt"I   @      HxmfD  LI+EIVHHHHI$H	! HH$   H$ULA IFIHHPH+PLD$HAH+AHH|@eHheIuHLLH$kLD$tI@H$LHPHpku1LO     @      H߽   lH<$QhvIUH$IwLxktH<$I$Lh  rHHA<fAWAVAUATIUSHH(3  H$S! Ljt$H! HL   D$DD$   Mu AIFHD$  HDD$11L}  DD$IA@HDHD$E HII
  H|$HHwdx;IVHLLHHp  u8L;|$uHLLh  FfD  I         L>kAE PAU HL   fD$   D$H([]A\A]A^A_D  HQ! H
b     H P  D$   f.     HH}
  H|$HHcxIVHL,1I         LjD$   af     AWAVAUATIUSH8  dH%(   H$(  1H|$;  L-,Q! HZAH! H|$IE H   D$t<D$   H$(  dH3%(   D$  H8  []A\A]A^A_    HC H$HhID$HD$(AGǉÉD$ aHHD$bI   AGI\$ M|()H<$HPLHmh  HL9   IE H;
  HHIMmH;EuLdM   H|$@      iD$   fD  HO! H
a     H P  D$       L8oHD$0H$HPHBH+BH+D$ x`HHD$8aHD$ HEH;ELxu
YIL9etNI_H|$MwMH3>ayHL$ H<$LLFguIU H3H=t`  R     H\$@D      HAaIE H|$(
  Ht$HHBjLt$8HLrfI;F   HT$0H|$ HHjuTH|$ ucHL$0P~rH|$HjD$   f.     L8cM   DIE H|$ Hh  mH|$HCH   HGcsIE H   |kbf.     AWAVAUATUSH  dH%(   H$   1H<$  HM! LbAHT$H<$H! HL      E~LL$A   I|$ Ld$LL$_A      LI_ELL$   AMyAIOt1( IM9   HI?
  IWLHL-iuH<$L   HiLafD     H$   dH3%(      H  []A\A]A^A_     HL! H
^        H P  D  H<$L^  	   A         te|    HLHh  kH<$HU` AWAVIAUATUSH8H|$ Q  H-L! HZAH ! H|$ HE H   D$   AVA  H{ T$^L`T$HD$(ID$HD$E  AI_AIKD7(HD$9f.     H8L\LHL`H9D$$  H9\$   HE H;H
  IHE H{
  LLIAhH9D$IuH|$ M   -      dH|$(_D$       D$   D$H8[]A\A]A^A_D  HJ! H
z\     H P  D$   f.     H|$ Lt\  	   -      cD$   fD  HE H|$(Hh  iH|$ Hd@ H|$ M   -      ,c    AVAUATUHSt0HJ!    H
Z     H P  []A\A]A^@ L%I! LjH  I$L   t   []A\A]A^ Mu I~{h   IH[dL1L~\I$LLh  fHHAԉ[]A\A]A^    AWAVIAUATUSH(H|$   H-"I! LbH  H|$HE L   AtA   H(D[]A\A]A^A_@ ID$ KMwωL$HD$H@H$eL$Ią   CM|  LdM9   HE II~
  H<$HHTZyH|$Iع   *      A   NaLZCH1H! H
Y     A   H P  f     H<$LbH|$LH\LIZHE LHh  FgH|$H    AVAUATUHSt0HG!    H
DX     H P  []A\A]A^@ L%qG! LjH6  I$L   t   []A\A]A^ IE Hx[II$L  II$LP0I$LHh  []A\A]A^@ f.     AWAVIAUATIUSH  dH%(   H$  1~VUUU)R9tXHF! H
X  L   LH P  D$   H$  dH3%(   D$  H  []A\A]A^A_ÐHAF! MwH  H L   D$   =  Mn II})WLHIQeH|$ Hź      HD$H|$ XHEHHhH+hHiD  IWHL$LI7]HI   HL]I;FtgIWHL$LLSbHtnHHEI HIE! H|$H Hh  dLHD$    H\$I   HkcHL@aH|$YD$   kf     HD! LLH h  GYAWAVIAUATUSH(H|$   H-D! LbHU  H|$HE L   D$   ߃D  ID$ M~HD$L`aIōCMt L&`M9   HE II
  LHHUyH|$Iع         \LVD$    D$   D$H([]A\A]A^A_D  HC! H
T     H P  D$   fIt$I+t$LH*cLHVHL$^H|$HHWHIUHE LHh  bH|$HW    HE LH|$h  9f     AVAUATUISHH   dH%(   H$   1tSHB!    H
S     H P  H$   dH3%(      H   []A\A]A^f     	      HTIt$HH   	aHtHSIT$HHHIvYu0H!B! LH Lh  aHHAWf     L   V<9Vf     AVAUATUISHH   dH%(   H$   1   HA! LjHc  H L   ¸   t$H$   dH3%(   udH   []A\A]A^ÐMm H      zSHMSILLuQ H!A! H
JS     H P     XU     ATUISHHdH%(   HD$1   H@! HHT$HqE1ɹ   LH  HH RH  	  Z   Yt%H|$dH3<%(   u\H[]A\f.     HcD$H
  LHHTHA@! LH
-C     HH P     rTfAUATUSHcD$x   HL-?! DIE P(IcH1HJUHCxH{ 1HCHCHHCIE   HH[]A\A]     USHHH_H;_tD  H;HXH9]uHw?! H} Hx  HHH@0H[]Ht~fD  Y SHGHcHH;_rH?! H=xQ  H1R H[ AWAVAUATIUSHHH(dH%(   HD$1H9w   L%>! L} HT$Iu LUhIƋD$   HLHD$H0U`HI   HH+EIwHH9tM$H=Q  1AP H|$HcWI$Lp  L+HInHL$dH3%(   uqH([]A\A]A^A_D  I$Lp  HED  L%>! H+w1H=P  I$HR @ HD$I$H=P  H01R 2RfD  ATUISHHHoHOdH%(   HD$1HGHH)H)HHHH9}UHT$H{ I4$ShT$tHSHJHKL"HH+SHHPHt$dH34%(   Hu$H[]A\ H=! H=zP  H1R [Qf.     AUATIUSHHH~H+>H NLcHHI9t'3LHBTHYHH?TI9uHH[]A\A]ff.     AUATIUSIHHHHGH9   L9   I)HEIIcHH9ErlEtOAD$Ll L9t;HH{fYHHISH;EuLJUH1[]A\A]D  H   []A\A]     H;! H=O  H1R yD  H;! LI+uH=3O  H1HR 6f.     H;! H+wH=O  H1HR IEf     USHHHH+{HLHSHsHHHVHH[]fD  USHHHH+{HDLHSHsHHH^VHH[]@ SHH S`HtHcPHC[HfHC[f.     AUATUSHHOH_HFH+FHH)HHH9tH1[]A\A]fD  H9IIw; H0HVtHI9]vH+LHu HWI;D$u    H   []A\A]     HHH P`Ht	H@HHf.     ATUISHHH+{HH~H+~HHJHSHsHHHUt6I\$I;\$u2@ HI9\$t#H;VHHPH;EuH1SH[]A\fAWAVIAUATIUSHHI+~HH~H+~HH9MWJI^I;^Iu^     HI9^tLL#LI4$UI;EHtH8LsUtH} VLHKPI;GuLE1RHL[]A\A]A^A_     AWAVIAUATIUSHHH8HBHH+H+{LnLD$(HL$ HHA)LH+FHzIHSHsHHHD$SIT$I+T$H   H'II\$HD$IH(I9tjHH     IEHHM9tAEL{tHL$HLLH\$SLD$t-DuHAHM9uHD$H8[]A\A]A^A_@ HCH|$(   HjLH|$QH|$IHD$ HD$    H     D  f.     AWAVIAUATIUSIHHHH~H+~HBH+HT$HL$8LD$LL$(HI|$I+|$H)HHIT$I+T$1HH$HGM\$I\$HD$0IH(I9  E1Ld$Lt$ HHME IEHHM9t?ULktH$H|$HLIQ   DuHAHM9uELt$ IVI+V   H|$HGMVI^HL$HD$L I9HtiLt$IHMIAD$HIM9t>AD$HktH$H|$HHIPtoEl$HAIM9uH$HH[]A\A]A^A_fD  IE H|$(   HJJH<$OH|$0GHD$8H     H$    먐HCH|$(   H
JH<$AOH|$GHD$H     H$    eE1fD  AWAVAUATIUSHHI+}Ht$HRI]IIEH9tl1E1    HH|$EgH0DF1D9DLHՉEKIEH9uIUHcH)HH9tHL[]A\A]A^A_H1LE1F@ AWAVAUATUSHH_H;_   HT$IIE1fD  L;LI7PI;D$Ht<H0LMPt-HD$HtHHI+T$I+uHHH OAHI9]uHD[]A\A]A^A_E1f.     AUATUSHHnHLgH9t$I HcLHIH0DCH9uH[]A\A]    AVH2! IAUATE1USHH_H+_H H^P(HcHE1HHI^I;^Hms8    L#HHLKHLFl(H DI9^Elw[D]A\A]A^Ðf.     AWAVIAUATUSHHLL;Ht$Lbsgf     M/LHILkIHLLHI Hk H}PHHE  M9~H]wHCH+D$H[]A\A]A^A_À~ HuH\$fATUISH0dH%(   HD$(1H1! HH$    HD$    HD$    HD$    $   H HX(UPxHHLH?BHH LHL$(dH3%(   uH0H[]A\pESHH KHCHEC[fSH HCHC  H  HC[D  f.     SHH sJHC     HC    [ÐH0! SHH   H]  HX HPH@    @    [f     AUATIUSIHH(dH%(   HD$1H>0! HL$HT$H x     D$   QAHŋD$~`HD$5    HH}GH;EtgD$PHD$T$HHD$~$HPH0LLGHHuH1IHL$dH3%(   Hu^H([]A\A]     HL   DLLKHH 1Hھ	   LMLL1wKCAWAVIAUATIUS׉HHt$KH~jCL-.! MtfD  HJM9tDIE II
  LHH@yH|$Iع   11GH1MAHH[]A\A]A^A_Ðf.     USHHHHH(dH%(   HD$1HO.! HT$HL$H x  1uHL$T$HHhAH|$dH3<%(   uH([]\Bff.     AWAVIAUATIUSHHLHH8dH%(   HD$(1H-! HL$ HT$H x     AIEHcL$LpH@L)HHH9   HM}u2   f     IU IIGHBH+BIEI9}   HD$ IIvHHHPHT$ HJHtNI?HtJHD$H,! H   HD$    HL@KHHIA   H\$(dH3%(   DuH8[]A\A]A^A_@     AWAVAUATUSHHHH8L5z,! dH%(   HD$(1H4$HL$HT$IHL$ Hx  C  D$;  q=HI=T$IHD$ p   HPLHHnG   ID$HL$LH<$HpHXHHt{I+\$IWH D$PHD$ T$HHD$ ~~IH8
  HHD$ HHLhHH;EiHھ   H\$Ht@H<$HGL@1HL$(dH3%(   u_H8[]A\A]A^A_    LJfD  Hھ   fD  1HH\$	   HOIH<$H#G1*?f.     AWAVIAUATIUSIHLLH(H*! A   dH%(   HD$1HL$HT$Hx  uqT$   AHD$ VD$PHD$T$HHD$~8HH8
  HT$HHLHR0FuHLA   KFH\$dH3%(   Du;H([]A\A]A^A_f     L	   H0HHLA   E>     AWAVIAUATHUSIL   H8L%)! dH%(   HD$(1HT$H$HT$I$HL$ x  uJD$uqIEHcHPH+PHH9t`HT$   L<$LzGLLOEHL$(dH3%(     H8[]A\A]A^A_f.     HT$	   @ 1FT$HHD$ H   @ HV>      DH.@D$PHD$ T$HHD$ ~YI$H8
  I}HH5:AyH<$H   M=H߻   :H4$LdD    H:H$HT$LL|:D  H<$H   <(<     AVAUMATUISIHH#9LLLHH>uCHUHtHRHt
D$HҋD$H=  H] HUH[]A\A]A^D  H<H   []A\A]A^@ f.     ATUHSHH   dH%(   H$   1H&! HL$HT$H x  t.   H$   dH3%(      H   []A\fD  Ld$1Ҿ   L8|$t%LH   DLHB     HD$LHH0EHvHT$MHHHHRE\:AWAVIAUATUSHHHdH%(   HD$81 LD$     HD$0IE1L=%! HD$HD$,H$}@ D$,@  HT$0IH:
  I~HHD$7HcL$,HT$0Ht7  IVHA   HHI10  Ht4AID9twIHL$HH$I4$x  ffD     H\$8dH3%(      HH[]A\A]A^A_     HGLwL=$! LhL`M9u0@ ILh    HHA1떐IM9tI4$II1INA   HH6	  HuW    H\$I$	   HBHH@*f.     H\$HT$   HY9HH~@1L=($! ;v8fD  Ht7WG     f.     HtCATUIH-#! SH_wHHE   HE LH[]A\H  D  fD  Ht;USHHHH#! H 
  HH[]H8f.     @ f.     AWAVAUATUSHHHtHu5H   L%-#! I$H{H  H[]A\A]A^A_fD  L%#! LnHI$  KH  I$HL   L<@  E1HLQ0  LHE0  L1A֋SH-  sM$HMj HHЋHM  HH  L1H5u6  A0  XZ2    H[]A\A]A^A_ÐAUATMUSAHH   HdH%(   H$   1O4LDH!7HHF>H$   dH3%(   uH   []A\A]06AUATMUSAHH   HdH%(   H$   13LDH@HH=H$   dH3%(   uH   []A\A]5H'  HfWH  HÐHH+f     +Ðf.     AUATLcUS   HL%!! I$P(HI$B<    P(HHCJHCHH[]A\A]ÐAUATIUSHL- ! H_H+   IE P(HIE P(HHE I4$HHHcHHUHUHc8HH[]A\A]fD  Ht3USHHH-S ! H?HE P0HE HH@0H[]fD  fD  ATULcSHHGH)HL9}7HoH ! HHH)H HHcP8HHJHSHC[]A\f.     HHWH9s@ 0HH9r AVAULcATL%! AUS   I$P(HI$B<    P(HJDHHCHC3>H[]A\A]A^    HHWH9s@ 0HH9rUSHcHHHHH;Gr H! HH=6=  H1R HHE  H[]@ f.     ATUASHcHHHH;Gr H! HH==  H1R HHE D []A\@ f.     HH;Gt @ Hq! SHH==  H1R H[ D  f.     HGH9t@ H1! SHH=<  H1R HC[@ f.     USHHHWHGH9rH+Ht/;HSHBHC*H[]f.     SHGHH9tHHC[f     H! H=b<  H1R HCHHC[ff.     AWAVAAUATIUSHHHWH9r~IE HHH)H)HHIIMDH)H9   McIMcIH9rjLEIUtAVHTD  HkH9uHL[]A\A]A^A_fH! H+7H=;  H1HR IU\f.     J<;H)HQ8IUIE z@ LMc9IE IIUJ JATUISHHWHH9rJH9v%H6! HH+u H=p;  H1HR HUH)HL7L)H)]L[]A\ H! H+7H=/;  H1HR HU f.     ATUISHcHHHHHH9vNHcHH(H9r H! H=9  H1Q I$H(H؋10H[]A\f.     HQ! T$H=~9  H1Q I$I|$T$    ATUSH1Lc,9HHI9t3   HH2I9uH[]A\HHwH
   H)HH,@ f.     AUATIUSHHGH+H)8LIH7I4$I|$1H9t6f.     9}f.     HS9HQH9u9~HS9uHL[]A\A]@ HHGH9s';2ufD  92tHH9rf     HÐf.     USHH91H;CtH[]    H,H   []HHWH9s!;1t+H
fD  90tHH9r@ H)H1 f.     USHHHHH9wH;wrH'! H=8  H1R HE H)HHH[]Ðf.     AWAVAUATIUSHHLg   Lv5HII9t+    +L7I9t
L+HI9uHL[]A\A]A^A_fAWAVAUATIUSHHLg   Lv:5HII9t!    +L7I9tHI9uHL[]A\A]A^A_@ L>+ff.     LHWHNH>L)HH)HcHH9t1fD  HHHL.HAWAVAUATUSHLoHL~I9tXII1f.     3L6I9HI9uI^IHcH)1HH9H[]A\A]A^A_H1 AUATUSHHLfHLoI9u1HI9t'3HM6I9tH   []A\A]     H1[]A\A] AVAUIATUISHGHHMH9   H9wgM9l$s!H! LI+4$H=6  H1HR H)LLH1ɉHc.H    HH.[]A\A]A^f.     HA! HI+6H=5  H1HR t H! H+7H=5  H1HR IFFfAWAVIAUATUSHL+IH;w   L5! H  L-7  II AHLH)޺   HMHAiP1IQ&HAHM9|$XZuHY  H9tH HHB  []A\A]A^A_H0  @ f.     AUATUSHc߿   HL-! IE P(HIE <    HP(HHEHE 1HV*H] HH]H[]A\A]AUATIUSHL-! H_H+   IE P(HIE P(HHE I4$HHHcHHUHUHc,HH[]A\A]fD  Ht3USHHH-S! H?HE P0HE HH@0H[]fD  fD  ATULcSHHGH)HL9}7HoH! HHH)H HHcP8HHJHSHC[]A\f.     HH;Gs    H0HH9GwÐUSHcHHHHH;Gr H! HH=3  H1R HHE H H[] f.     ATUISHcHHHH;Gr H7! HH=3  H1R HHE L []A\@ f.     HH;GtH  H! SHH=3  H1R H[H @ f.     HGH9tH@fH! SHH=F3  H1R HC[H@ff.     USHHHHWHGH9rH+Ht.0HSHBHCH*H[]     SHGHH9tHHC[f     H! H=2  H1R HCHHC[ff.     AWAVAAUATIUSHHHHWH9   IE IMHIH)H)I)HHIDH9   McIMcIH9reLEIUtAFHTHHkH9uLIE H[]A\A]A^A_D  H9! H+7H=/2  H1HR IUYfJ<;H)H,IU McLI.LIUI] Xf     ATUISHHWHH9rJH9v%H! HH+u H=1  H1HR HUH)HLR,L)H)]L[]A\ Hq! H+7H=1  H1HR HU f.     HHwH
   H)HH."@ f.     HHGH9s'H;2uD  H92tHH9r     HÐf.     USHHH(1H;CtH[]fD  HH H   []D  f.     LHWHNH>L)HH)HcHH9t1fD  HHHL-&HAWAVAUATUSHLoHL~I9tYII1f.     H3L(I9HI9uI^IHcH)1HH9H[]A\A]A^A_H1fAVAUIATUISHGHHMH9   H9wgM9l$s!H! LI+4$H=+0  H1HR H)LLH1ɉHcf$H    HH&[]A\A]A^f.     H!! HI+6H=/  H1HR t H
! H+7H=g/  H1HR IFFfAWAVIAUATUSHL+IH;w   L5  H  L-/  II IHLH)޺   HMHAiP1I1HAHM9|$XZuH9  H9tH HH"  []A\A]A^A_H  f.     @ AUATUSHHH_dH%(   HD$1H$    H  ;Ϻ  1IAIHH5/  SH"  Et|A<$@   @L1D  0H	u   @88HLt   fD  @:2   H1H@u0	   H$H-! HRH   H
H
! HJHRH
! H! H|$dH3<%(      H[]A\A]f     H-  HE    HE 1f     E1   LH5-  HSH_1 H}!     Hj!     HW!     jfE1   LH5-  HS1If.     fAWAVAAUAT1USHIAHHL%! H5-  dH%(   HD$1H$    II$PHHt0H$H   PD9uDH$PD9   H$H
! HT$dH3%(   H   H[]A\A]A^A_ L5,  I$H  H?-  M$j AVL
,  MPSH
 -  H,  H5-  H11A@  H |D  L5,      L5U,  b  HH       8.6 ::ral iso8859-1 pkgname version copyright                   relvar  ::ral::relvar   relation                ::ral::relation tuple   ::ral::tuple ::ral      This software is copyrighted 2004 - 2014 by G. Andrew Mangogna. Terms and conditions for use are distributed with the source code. 0.12.2 ral   booleanHash: cannot convert, "%s"       longHash: cannot convert, "%s"  doubleHash: cannot convert, "%s"        wideIntHash: cannot convert, "%s"       Ral_AttributeDelete: unknown attribute type: %d Ral_AttributeDup: unknown attribute type: %d    Ral_AttributeRename: unknown attribute type: %d Ral_AttributeTypeEqual: unknown attribute type: %d      Ral_AttributeValueEqual: unknown attribute type: %d     Ral_AttributeValueCompare: unknown attribute type: %d   Ral_AttributeValueObj: unknown attribute type: %d       Ral_AttributeConvertValueToType: unknown attribute type: %d     Ral_AttributeHashValue: unknown attribute type: %d      Ral_AttributeScanType: unknown attribute type: %d       Ral_AttributeScanValue: unknown attribute type: %d      Ral_AttributeConvertValue: unknown attribute type: %d   Ral_AttributeTypeScanFlagsFree: unknown flags type: %d  Ral_AttributeValueScanFlagsFree: unknown flags type: %d intHash: cannot convert, "%s" bad boolean value, "%s" bignum boolean bytearray double long string wideInt  !=  while ungrouping relation while unwrapping tuple while computing quotient        Ral_RelationUpdate: attempt to update non-existant tuple        Ral_RelationErase: first iterator out of bounds Ral_RelationErase: last iterator out of bounds  Ral_RelationErase: first iterator greater than last     relation1 relation2 ?-using joinAttrs?  relation1 relation2 ?-using joinAttrs relation3 ... ?   relationValue newAttr ?attr attr2 ...?  relationValue tupleVarName expr script  
    ("in ::ral::relation update" body line %d) invoked "break" outside of a loop       invoked "continue" outside of a loop    relation1 relation2 ?relation3 ...?     relationValue ?name-value-list ...?     tupleVarName relationValue ?-ascending | -descending? ?attr-list?script 
    ("::ral::relation foreach" body line %d)   relation attrName ?-ascending | -descending sort-attr-list? ?-within attr-list -start tag-value?        Ral_TagCmd: unknown option, "%d"        heading ?value-list1 value-list2 ...?   relationValue perRelation relationVarName ?attr1 type1 expr1 ... attrN typeN exprN?     attribute / type / expression arguments must be given in triples        relationValue ?oldname newname ... ?    oldname / newname arguments must be given in pairs      relationValue ?-ascending | -descending? rankAttr newAttr       relationValue ?attrName ? ?-ascending | -descending? sortAttrList? ?    relationValue attrName ?attrName2 ...?  relation arrayName keyAttr valueAttr    relation newattribute ?attr1 attr2 ...? attempt to group all attributes in the relation dictValue keyattr keytype valueattr valuetype   relationValue ?attrName | attr-var-list ...? relation1 relation2 relation1 compareop relation2 compareop unknown return code %d relValue expr relValue tupleVarName expr relationValue attribute relation attribute ordering tag option heading ?tuple1 tuple2 ...? relationValue ?attr ... ? relation keyAttr valueAttr during group operation list attribute type dividend divisor mediator subcommand ?arg? ... == notequal != propersubsetof < propersupersetof > <= >= -ascending -descending -within -start assign attributes body cardinality compose create degree divide dunion eliminate emptyof extend extract foreach fromdict fromlist heading intersect is isempty isnotempty issametype project rank rename restrict restrictwith semijoin semiminus summarize summarizeby table tag tclose times uinsert ungroup unwrap update  relationValue tupleVarName ?attr1 type1 expr1 ... attrN typeN exprN? -using (  <== [ ]   (Complete) ] ==>  , in relvar  tuple  for association  ] ==> [ for partition   is partitioned [ ])  |  for correlation  references multiple tuples references no tuple unknown constraint type, %d + * relvarConstraintCleanup: unknown constraint type, %d    Ral_ConstraintDelete: unknown constraint type, %d       is referenced by multiple tuples        is not referenced by any tuple  is referred to by multiple tuples       is not referred to by any tuple  does not form a complete correlation ?pattern? relvarName relationValue ?relvar1 relvar2 ...? transaction option arg ?arg ...? option type ?arg arg ...? trace option trace type info variable relvarName Unknown trace option, %d add transaction cmdPrefix remove transaction cmdPrefix info transaction Unknown trace type, %d relvar ?relationValue? relvarName tupleVarName expr constraint subcommand delete exists info names member path variable add remove suspend begin rollback association constraint correlation deleteone eval identifiers partition procedural restrictone trace unset updateone updateper     relvarName ?name-value-list ...?        relvarName tupleVarName idValueList script      relvarName tupleVarName expr script     relvarName ?relationValue ...?  Unknown transaction option, %d  
    ("in ::ral::relvar eval" body line %d)     add variable relvarName ops cmdPrefix   remove variable relvarName ops cmdPrefix        suspend variable relvarName script      suspending eval traces not implemented  relvarValue attr value ?attr2 value 2 ...?      attribute / value arguments must be given in pairs      name relvarName1 ?relvarName2 ...? script       name super super-attrs sub1 sub1-attrs ?sub2 sub2-attrs sub3 sub3-attrs ...?    relvarName relvarName ?attrName1 value1 attrName2 value2 ...?   relvarName heading id1 ?id2 id3 ...?    ?-complete? name corrRelvar corr-attr-list1 relvar1 attr-list1 spec1 corr-attr-list2 relvar2 attr-list2 spec2   delete | info | names ?args? | member <relvar> | path <name>    Unknown association command type, %d    name refrngRelvar refrngAttrList refToSpec refToRelvar refToAttrList refrngSpec @BBA@AAB:: traceOp namespace eval   { } ", failed procedural contraint, " -complete     bad operation list: must be one or more of delete, insert, update, set or unset relvar may only be modified using "::ral::relvar" command       relvarTraceProc: trace on non-write, flags = %#x
       returned "continue" from procedural contraint script for constraint, "  returned "break" from procedural contraint script for constraint, "     relvar creation not allowed during a transaction        during identifier construction operation        Ral_RelvarObjConstraintInfo: unknown constraint type, %d        end transaction with no beginning       
    ("in ::ral::relvar trace suspend variable" body line %d)   
    ("in ::ral::%s %s" body line %d)           multiplicity specification      Ral_TupleUpdate: attempt to update a shared tuple attr1 type1 value1 ... tuple1 tuple2 tupleValue tupleValue tupleAttribute tupleValue attr ?...? tupleValue ?attr? ... heading name-value-list get     tupleValue newAttr ?attr attr2 ...?     failed to copy attribute, "%s"  tupleValue ?attr1 value1 attr2 value2?  for attribute name / attribute value arguments  tupleValue ?oldname newname ... ?       for oldname / newname arguments tupleValue ?name type value ... ?       tupleValue ?attrName | attr-var-pair ... ?      Ral_TupleHeadingFetch: out of bounds access at offset "%d"      Ral_TupleHeadingStore: out of bounds access at offset "%d"      Ral_TupleHeadingStore: cannot find attribute name, "%s", in hash table  Ral_TupleHeadingStore: inconsistent index, expected %u, got %u  Ral_TupleHeadingPushBack: overflow      Ral_TupleHeadingAppend: out of bounds access at source offset "%d"      Ral_TupleHeadingAppend: overflow of destination RAL OK UNKNOWN_ATTR DUPLICATE_ATTR HEADING_ERR FORMAT_ERR BAD_VALUE BAD_TYPE BAD_KEYWORD WRONG_NUM_ATTRS BAD_PAIRS_LIST DUPLICATE_OPTION NO_IDENTIFIER IDENTIFIER_FORMAT IDENTIFIER_SUBSET IDENTITY_CONSTRAINT DUP_ATTR_IN_ID DUPLICATE_TUPLE HEADING_NOT_EQUAL DEGREE_ONE DEGREE_TWO CARDINALITY_ONE BAD_TRIPLE_LIST NOT_AN_IDENTIFIER NOT_A_RELATION NOT_A_TUPLE NOT_A_PROJECTION NOT_DISJOINT NOT_UNION TOO_MANY_ATTRS TYPE_MISMATCH SINGLE_IDENTIFIER SINGLE_ATTRIBUTE WITHIN_NOT_SUBSET BAD_RANK_TYPE DUP_NAME UNKNOWN_NAME REFATTR_MISMATCH DUP_CONSTRAINT UNKNOWN_CONSTRAINT CONSTRAINTS_PRESENT BAD_MULT BAD_TRANS_OP SUPER_NAME INCOMPLETE_SPEC ONGOING_CMD ONGOING_MODIFICATION INTERNAL_ERROR no error unknown attribute name duplicate attribute name bad heading format bad value format bad value type for value unknown data type bad type keyword bad list of pairs duplicate command option duplicate tuple headings not equal bad list of triples too many attributes specified duplicate relvar name unknown relvar name duplicate constraint name unknown constraint name serious internal error NONE destroy unknown command relvar setfromany updatefromobj       wrong number of attributes specified    relations of non-zero degree must have at least one identifier  identifiers must have at least one attribute    identifiers must not be subsets of other identifiers    tuple has duplicate values for an identifier    duplicate attribute name in identifier attribute set    relation must have degree of one        relation must have degree of two        relation must have cardinality of one   attributes do not constitute an identifier      attribute must be of a Relation type    attribute must be of a Tuple type       relation is not a projection of the summarized relation divisor heading must be disjoint from the dividend heading      mediator heading must be a union of the dividend and divisor headings   attributes must have the same type      only a single identifier may be specified       identifier must have only a single attribute    "-within" option attributes are not the subset of any identifier        attribute is not a valid type for rank operation        mismatch between referential attributes relvar has constraints in place referred to identifiers can not have non-singular multiplicities        operation is not allowed during "eval" command  a super set relvar may not be named as one of its own sub sets  correlation spec is not available for a "-complete" correlation recursively invoking a relvar command outside of a transaction  recursive attempt to modify a relvar already being changed      Ral_IntVectorFetch: out of bounds access at offset "%d" Ral_IntVectorStore: out of bounds access at offset "%d" Ral_IntVectorFront: access to empty vector      Ral_IntVectorPopBack: access to empty vector    Ral_IntVectorInsert: out of bounds access at offset "%d"        Ral_IntVectorErase: out of bounds access at offset "%d" Ral_IntVectorOffsetOf: out of bounds access     Ral_IntVectorCopy: out of bounds access at source offset "%d"   Ral_IntVectorCopy: out of bounds access at dest offset "%d"     Ral_PtrVectorFetch: out of bounds access at offset "%d" Ral_PtrVectorStore: out of bounds access at offset "%d" Ral_PtrVectorFront: access to empty vector      Ral_PtrVectorPopBack: access to empty vector    Ral_PtrVectorInsert: out of bounds access at offset "%d"        Ral_PtrVectorErase: out of bounds access at offset "%d" Ral_PtrVectorCopy: out of bounds access at source offset "%d"   Ral_PtrVectorCopy: out of bounds access at dest offset "%d" %d: %d
 %d: %X
     interpreter uses an incompatible stubs mechanism        Tcl missing stub table pointer epoch number mismatch requires a later revision tcl::tommath , actual version   (requested version  Error loading  ):    ;l    `           0  H   `           P(  H   x    @    (  X  @         p	H  	h  @
  
      (  p  
    `  (  @  0`  @x  P        08  h        p8  P  h     `        8   P  px        8  p    `  `@  `  x  @  @     P  !  0"  "   `#8  p#P  #h  #  @$  $    %0  %x   &  `&  &  '0  @'H  '  '  (  (   (    )@  P)`  p)x  )  )  )  )  0*  *X  p+  ,  ,(   @-X   -   .   /   /@!   0p!  @1!  2"   3P"  5"  8"  9@#  <#  >#  > $  @?X$  0@$  `A$  PCH%  pE%  E%  F&  0J`&  PK&  K&  `L0'   M'  O'   P(  `QX(  Q(  @R(  `R(  R)   S0)  PSX)  S)  0T)  U *  UX*   W*  pW*  @X(+   Yh+  [+  `^H,   a,  c(-  0ex-  e-  f8.  g.  Ph.  lH/  `m/  @n/  oH0  s0  v0  0y81  `z1  @|1  0~(2  `x2  Ѐ2  @3  0x3  3  Ѕ(4  04  `4  85  5   5  (6  x6  6  @(7   x7  07  `8  `x8  8  @9  h9  9  p:  h:  p:  ;   ;  @p;  ;  ` <  @`<  `<  <  <  <  0=  x=  =  P >   P>  >   >   ?  H?   ?  ?  0?  H@  @  @  0A  A  A  B  8B  XB  B  B  B  B  @C  HC   hC  0C  C  C  @ D  0D   PD  pD  D  @D  `PE  pE  F  `F  G  `G  G  `G  8H  H  @H  I   8I   XI  `I   I  (J  `xJ  J  ` K   @K  K   K  @L  `L  L  0M  M  pM  N  PN  N  ` O  8O  0`O   O  # P  @&PP  )P  * Q   +Q  -pQ   .Q  .Q  /R  0`R  1R  05R  50S  6xS  9S  :T  >hT  ?T  A U  BPU  GU  @NV  VV  XV   Z0W   [W  \W  \X  dX  PeX  f8Y  0gpY  gY  hY  0i(Z  ihZ   jZ  jZ  k[  k8[   l`[  @m[  m[  o(\  p\  0r\  `r ]  s`]  `v]  v]   w^  wH^  w`^  px^  x^  y0_  pyp_  y_  {(`  P{X`  {`  |`  | a  |0a  0}Xa  }a  @~a  ~b  PXb  b  b  pc  hc  pc  d  hd  d  e  Xe   e  e  Hf  f  pf  @Xg  pg  0g  P@h  0h  h   i  Н(i  @i   `i  pi  0i   j  Сpj  j  Pj  j   0k  PPk  k  k  l  phl  0l  m  0@m  Ъm  pm  n  0(n  `Hn  hn  n   n  o  0@o  o  `o  0p  `p   p  q  `q  0xq  q  йq  Hr  Pr  r  лr  r  s   (s  P`s  s   s  `s  t  Pt   ht  Pt  t  t    u  p(u  Hu  u  `u    v  @0v  pHv  v  Pv  v  v  0w  Xw   w  pw  x  P`x  0x  x  P0y  hy   y  `y  y  z   8z  `Xz  xz  z  @z  P{  @{  X{  Pp{  {  {  p|  PP|  |  |             zR x  $         FJw ?;*3$"       D   8           L   \   0   BBI B(C0A8G@
8D0A(B BBBE                                                        ,      {    AAL0}
AAC      ,   $  |    AAL0Y
AAG     ,   T  X    AAL0`
AAH               D[      i    D _
A           C    D y
A      ,     P{    AAL0}
AAC      ,     |    AAL0Y
AAG     ,   <      AAL0`
AAH        l  `A    D w
A      ,     ~    AAL@
AAA      ,     |    AAL@]
AAC     4     0    BAD I0c
 AABI        $  A    D w
A      ,   D      AAL@
AAA      ,   t  8    AAL@v
AAJ     4         BAD I0c
 AABI           A    D w
A      ,     P~    AAL@
AAA      ,   ,      AAL@x
AAH     4   \      BAD I0c
 AABI     $     d    AD@X
AA     D         BBE A(D0Hp~
0A(A BBBD      D     H    BBE A(D0Lpn
0A(A BBBH     ,   L      AAK@x
AAA        |   H    D0~
A           PC    D y
A                           xi    D _
A                                     \   $      BBE B(H0A8L@n
8C0A(B BBBFT8D0A(B BBB ,     8    BAD cGB                       ,     A    BAD sDB                       4     z    BAD D0e
 AABA     4   L  P    BAD D0Z
 AABG     D         BBE A(D0D@
0A(A BBBK          0                (           ,      d    BAD U
ABA   ,   ,  `S    BAD HAB      ,   \  S    BAD HAB      $     S    AAG FAA     d    d\      Pd    d\ $     b    Dj
BL
LV     D   	  ^    FAJ [
AABD
CAHDGA   <   T	     BBD A(G@e
(A ABBE     4   	  xy    AG f
CGD
GM`D       4   	      BAA 
ABG\AB  D   
  X    BBE A(D0J`	
0A(A BBBG    L   L
     BBE B(D0A8JP[
8A0A(B BBBA     4   
  P    BAD D0v
 AABK         
  &    Ad          
  (           ,     0    Do
EM
K]
CM       T   <      BAD G0^
 GABJZ
 CABH_
 AABE       L     8    AAG \
DAN\
CAH^
HDFHDA     L         AAM0W
DAE\
CAHl
IABHDA     ,   4      MAD KAB   ,   d  X    MAD KAB   4         BAD DP}
 DABA                                      <     n    BBL F(A0O(A BBB      $   <
  @B    AAG uAA ,   d
  hV    BAD KAB      ,   
  X    BAD MAB      D   
      AAI0H
CAID
FAED
FAE     ,     0	J    JDC uAB       4   <  P	R    BBA A(J0x(D ABB 4   t  x	T    BBA A(Q0s(D ABB      	N                	)           4     	T    BBA A(Q0s(D ABB      
D           D   ,  P
    BBE B(A0A8O@\8D0A(B BBB   t  
'    H^            
:    Hq            
$    AZ            
!    AW            
                
              $  
	              <  
              T  
-    Ac       $   t  
Q    AAL }DA L          BBE B(A0A8HP
8A0A(B BBBA     <     y    BBB A(E0e(A BBB      <   ,      BBE A(D0(A BBB      L   l  0    BBE B(D0A8V
8A0A(B BBBA    ,     g    BHA XAB      $     h    AAG [AA,     (
X    BAD MAB      D   D  X
    BBB A(D0GP
0A(A BBBD     D         BBB B(A0A8G@|8D0A(B BBB,     hO    BAD DAB      L     8   BBE B(D0A8J`
8A0A(B BBBF     L   T  x4   BBB B(D0A8G`

8A0A(B BBBA    <     hy    BBE A(D0`(D BBB      L        BBE B(A0A8J
8A0A(B BBBA   L   4  H   BBB B(A0A8J
8A0A(B BBBA   L     4   BBE B(D0A8S`
8A0A(B BBBA     L     )   BBE B(D0A8W
8A0A(B BBBA   L   $  xp   BBB B(D0A8Dp
8A0A(B BBBD    <   t  ^    BBD A(P0v
(D ABBA      4     R    BBD A(P0r(A ABB L         BBE D(A0
(D BBBGN(D BBB    L   <  -   BBB B(D0A8U
8A0A(B BBBA    L     `   BBE B(D0A8D
8A0A(B BBBA   L         BBE B(D0A8D
8A0A(B BBBA   4   ,  s    BBA A(J0Y(D ABB<   d       BBB A(D0|(D BBB      L     x    BBB B(A0A8G
8A0A(B BBBA   L     #   BBE B(D0A8Gp
8A0A(B BBBA     4   D  $g    EAA v
ABHE
ABHD   |  $    BBE B(A0A8J@v8D0A(B BBBL     (%    BBE B(A0A8J@i
8D0A(B BBBC     L     x%   BBE B(D0A8D4
8A0A(B BBBF   4   d  'm    BBD A(G0S(D ABBL     'W   BBB B(D0A8J@
8D0A(B BBBH     D      )    BBB B(D0A8L@e8C0A(B BBB$   4  H)C    AAJ tAA    \  p)    DT $   t  x)C    AAJ tAA $     )C    AAJ tAA $     )C    AAJ tAA $     )C    AAJ tAA L     *    BBE A(D0|
(C BBBES
(C BBBD  L   d  X*C   BBE B(A0A8G`
8A0D(B BBBA    4     X+9    BBD A(J0](C ABB L     `+@   BBE B(A0A8D
8A0A(B BBBA   <   <  P,k    BEB A(D0U(A BBB      <   |  ,    BBA A(G
(D ABBA    <     -    BBA A(G
(D ABBA    l     -   BBB B(A0A8Js
8A0A(B BBBGpRTASH[A  l   l  /   BBB B(A0A8Js
8A0A(B BBBGpRTASH[A  l     2   BBB B(A0A8Js
8A0A(B BBBGpXNAH[A  l   L  `4   BBB B(A0A8Js
8A0A(B BBBGpRTAH[A  L     6B   BBB B(A0A8GPk
8D0A(B BBBG      \     7    BBA A(G0j
(C ABBAi
(H ABBJn(C ABB     \   l  8    BBA A(G0j
(C ABBAi
(H ABBJr(C ABB     \     p8    BBA A(G0j
(C ABBAi
(H ABBJx(C ABB     \   ,   8    BBA A(G0j
(C ABBAi
(H ABBJx(C ABB     L      P9A   BBB B(D0A8Gy
8A0A(B BBBI    \      P=    BBA A(G0j
(C ABBAi
(H ABBJm(C ABB     L   <!  =    BBB B(A0A8J@j
8C0A(B BBBF      L   !  @>f   BBB A(D0G@P
0A(A BBBFtHdPPHA@ L   !  `?=   BBB B(D0A8GU
8A0A(B BBBE    L   ,"  PC   BBB B(D0A8GU
8A0A(B BBBE    L   |"  E}   BBB B(A0A8JpR
8A0A(B BBBH     L   "  G*   BBB B(D0A8JY
8A0A(B BBBF    L   #  H   BBE B(A0A8Jj
8A0A(B BBBE    L   l#  `J   BBE B(A0A8Jj
8A0A(B BBBE    L   #   L*   BBB B(D0A8JY
8A0A(B BBBF    L   $  Ld   BBE B(A0A8Jp
8A0A(B BBBG    L   \$   Nh   BBE B(A0A8Jk
8A0A(B BBBD    \   $   O    BBA A(G0j
(C ABBAf
(C ABBBl(H ABB    L   %  O   BBB B(D0A8GPx
8D0A(B BBBG      \   \%  P   EBB A(D0f
(A BBBE
(A BBBGH(A BBB  \   %  Q[   BBB B(D0A8Gi`PAa
8A0A(B BBBG    \   &  U/   BBB B(D0A8Jz
8A0A(B BBBEE^A   L   |&  pYB   BBE B(A0A8J
8A0A(B BBBA    L   &  pZ   BBE B(A0A8J
8A0A(B BBBH   L   '  [q   BBE B(A0A8Jl
8A0A(B BBBC    L   l'   b   BBE B(A0A8Jj
8A0A(B BBBE    L   '  g   BBB B(A0A8Js
8A0A(B BBBG    L   (  @l   BBE B(D0A8Gh
8A0A(B BBBG    \   \(  m   BBE B(A0A8D]
8A0A(B BBBHX\UA  L   (  q    BBE B(D0A8D@y
8C0A(B BBBG      L   )  q   BBE B(D0A8D@y
8C0A(B BBBG      L   \)  `r0   BBB B(A0A8MY
8A0A(B BBBF    \   )  @s   BBE B(A0A8G`r
8A0A(B BBBHh`pPhA`       L   *  u*   BBB B(D0A8G`V
8D0A(B BBBI    L   \*  w   BBB B(D0A8GPj
8C0A(B BBBF      L   *   yx   BBE B(A0A8G`j
8C0A(B BBBF      L   *  Pz   BBE B(A0A8Jj
8A0A(B BBBE    \   L+  {    EBB A(D0f
(A BBBEm
(A BBBDA(A BBB  L   +  P|,   BBB B(A0A8GpF
8D0A(B BBBD     L   +  0~   BBB B(D0A8JY
8A0A(B BBBF    L   L,  ,   BBE B(F0A8O
8A0A(B BBBF      ,  8    Ie L   ,  ^   BBB B(A0A8M[
8A0A(B BBBD    \   -  Ȃ    EBB A(D0f
(A BBBEm
(A BBBDE(A BBB  L   d-  8E   BBB B(D0A8Jg
8A0A(B BBBH    <   -  8    BAD G0d8a@P8F0X
 AABK     -  ؄    A\          .  ؄#    Aa          4.      A]          T.  4    Hk       L   t.      BBE B(D0A8J@w
8C0A(B BBBC      D   .  X   BBE A(D0JP]
0A(A BBBC     4   /       BAD JT
 AABG    L   D/  ؆    BBE B(D0A8PPd
8C0A(B BBBH      L   /  H    BBE B(D0A8PPh
8D0A(B BBBK      L   /  ȇ   BBE B(A0A8DJ
8A0A(B BBBK    ,   40  D    AAG f
AAG      L   d0  (   BBB B(D0A8L@p8A0A(B BBB       $   0  x;    AAG lDA <   0     BBA A(G0
(A ABBJ     ,   1  p    BAA 
ABJ   <   L1  02   BBD A(DP
(A ABBA    L   1  0    BBB B(A0A8DP
8A0A(B BBBA     L   1      BEB B(E0A8Kp
8A0A(B BBBA     D   ,2       BBE B(D0A8VP8A0A(B BBBL   t2     BBB B(D0A8O`j8E0A(B BBB       L   2  Ha   BBB B(D0A8O`28E0A(B BBB       D   3  h    BBE B(D0A8RPK8G0A(B BBBL   \3     BBE B(E0A8OP
8A0A(B BBBA        3  p-    Dd          3  $    GT
E      4   3      BBD A(L0(D ABB   $4                <4      AS          \4  )    H`       $   |4   #    AAI NAA 4   4  (    BAD E
ABAmAB     4  !    AW          4  ,    JZ
D         5  O    AM      ,   <5  Дi    AAN0I
AAE     $   l5  E    AAD yDA ,   5  8i    AAN0I
AAE        5  xO    AM      ,   5  i    AAN0I
AAE     $   6  Q    AAD EDA,   <6   i    AAN0I
AAE     t   l6  `   BBB A(A0r
(A BBBO
(A BBBJZ
(A BBBO(A BBB         6  A    Gy       <  7  8   BBB B(A0A8Dw
8D0A(B BBBFMPAKVBGZAPG[BPMQBPJUAPMQBPJUBPMPBPMPAPMVBUMPAPJVBPMPAPJVB4   D8      BBA A(J0q(C ABB$   |8  p{    AG0d
AC     L   8  Ȣ    BBB B(A0A8D
8A0A(B BBBA    \   8  x    BBE B(D0A8JP
8A0A(B BBBID8F0A(B BBB4   T9      FAA x
ABEvAB   <   9  `}    BBE A(D0H
(J BBBJ   \   9      BBE A(D0I
(A BBBBS
(A BBBFp(A BBB  <   ,:       BBA A(DP
(A ABBA     4   l:  `{    BBD A(G0a(D ABB$   :  1    AAG YGA    :      AY       ,   :  :    IDA hAB       L   ;  Х    BBE B(D0A8GP
8D0A(B BBBG     L   l;  @   BBB B(A0A8Dp
8A0A(B BBBC     L   ;     BBB B(D0A8J
8A0A(B BBBH    L   <  W   BBB B(D0A8J
8A0A(B BBBD    4   \<      EAA g
ABGUCB  <   <  X    EAA b
FBGJ
CBI       L   <     BBB B(A0A8Gpq
8A0A(B BBBD      L   $=  x	   BBB B(A0A8Gh
8A0A(B BBBE    \   t=  8   BBE B(A0A8J[
8A0A(B BBBDIbA  \   =     BBE B(A0A8JY
8A0A(B BBBFG[C  <   4>  t    BBD A(D0}
(A ABBA      L   t>     BBB B(D0A8J
8A0A(B BBBE    L   >     BBB B(D0A8J
8A0A(B BBBE    D   ?  X   BBA A(G@P
(A ABBEKH`PRHA@ L   \?  ?   EBE D(A0
(C BBBHr
(C BBBE d   ?     BBB B(D0A8GPdXb`PXAPa
8A0A(B BBBJDX``PXAP      L   @  !   BBB B(D0A8J^
8A0A(B BBBA    L   d@     BBE B(A0A8J
8A0A(B BBBG       @  8J    de $   @  p    AG0n
FDsAL   @  h   BBE B(A0A8JW
8A0A(B BBBH    L   DA     BBB B(D0A8J
8A0A(B BBBE    L   A     BBB B(A0A8M
8A0A(B BBBC    L   A  |   BBE B(A0A8GR
8A0A(B BBBH    \   4B      BBB B(A0A8MnKOAZ
8A0A(B BBBD       B  D    Le T   B     BBB B(D0A8G`hh_pUhA`
8A0A(B BBBD    C  H<    Ie D   C  p    BBD A(G@dHdPPHF@X
(A ABBD  4   dC      BBD A(M0q(D ABBT   C  5   BBB B(A0A8GpxE_xAp_
8A0A(B BBBAD   C  h    BBE B(D0A8OP8A0A(B BBBL   <D     BBE B(D0A8G
8A0A(B BBBB    4   D  0x    BAD q
ABEtAB   D   D  x    BBB A(A0J
0A(A BBBH    L   E   N   BBE B(D0A8Gs
8A0A(B BBBD    L   \E       BBE B(D0A8IP
8D0A(B BBBA     L   E     BBE B(D0A8PK
8A0A(B BBBK   L   E  P    BBE B(A0A8D@M
8D0A(B BBBE      D   LF  b   BBE A(D0M
0A(A BBBE    L   F  v   BBE B(A0A8Gp
8D0A(B BBBA     l   F  8K   BBB B(D0A8P
8A0A(B BBBG[`PA`PA  L   TG  T   BBE B(D0A8M
8A0A(B BBBD    |   G  (p   BBB B(D0A8P
8A0A(B BBBH`PAScPAcPAL   $H   ;   BBE B(D0A8Pn
8A0A(B BBBH   L   tH  0   BBB B(D0A8Ip
8A0A(B BBBI     d   H      BBB B(D0A8J@
8F0A(B BBBGT
8C0A(B BBBA     D   ,I  b   BBE A(D0M
0A(A BBBE    4   tI  T    BAD c
CBA[FB   |   I     BBB B(D0A8MP
8A0A(B BBBEd
8A0A(B BBBJ
8F0A(B BBBK   L   ,J      BBD A(J0M
(C ABBH^(F ABB      L   |J  `   BBB B(D0A8DPD
8C0A(B BBBG     4   J  
A    BDA X
DBKIAB   ,   K  
n    AAM@w
AAH      4   4K  
e    BAD O0E
 AABA     L   lK  0   BBB B(A0A8N
8A0A(B BBBE    <   K   |    BBI D(A0j
(A BBBE    $   K  @F    AAO pCA L   $L  h    BBE B(D0A8D@
8C0A(B BBBA     4   tL      BBD A(K@(A ABB   L  `J    Yh      $   L  C    QHGa     L   L     BBB B(D0A8JP
8D0A(B BBBD     $   DM  C    QHGa     L   lM  %   BBE B(D0A8J`~
8C0A(B BBBD      d   M      BBE B(D0A8OPn
8D0A(B BBBFQ
8P0A(B BBBN      L   $N  (k   BBB B(A0A8G@X
8D0A(B BBBJ         tN  H(    H^      \   N  XP   BBB A(D0
(D BBBHb
(D BBBDe(D BBB  L   N  H   BBB B(A0A8D
8A0A(B BBBF   ,   DO  Q    IDA AB       $   tO  R    AAM @AA<   O      KBD A(D0r(A ABB       O  p           \   O  x    BBB B(A0A8D@X
8A0A(B BBBHD8F0A(B BBB4   TP  N    FAJ U
AAHDGA 4   P  L    BBD A(D0u(D ABB <   P  \    BBA A(D0C
(D ABBA     L   Q  v    BBE A(D0O
(A BBBDA(F BBB    d   TQ  (   BBE B(D0A8JP
8A0A(B BBBET
8C0A(B BBBH     ,   Q  4    AAG b
AAC      4   Q  t    BBA A(D0`(D ABB$   $R  8:    AAJ hDA D   LR  Py    BBE B(A0A8GPZ8A0A(B BBB,   R  \    BAD {
ABK    $   R  ;    AAG lDA $   R  R    AAM @AAL   S      BBB B(D0A8G@
8A0A(B BBBA     4   dS  h:    BBD A(J0^(C ABB L   S  p    BBB B(A0A8H`
8A0A(B BBBA     <   S  l    BEB A(D0V(A BBB      <   ,T       BBA A(G
(D ABBA    <   lT      BBA A(G
(D ABBA    L   T  P   BBB B(D0A8J
8A0A(B BBBC    L   T        BBB B(A0A8G@j
8C0A(B BBBI      L   LU   D   BBB B(A0A8GPk
8D0A(B BBBG      \   U  !    BBA A(G0j
(C ABBAi
(H ABBJs(C ABB     L   U  "U   BBB B(A0A8GPq
8A0A(B BBBD      L   LV   #   BBB B(A0A8JPj
8C0A(B BBBF      L   V  $   BBB B(D0A8G`,
8A0A(B BBBF    L   V  `&U   BBB B(D0A8Gz
8A0A(B BBBH    L   <W  p)   BBB B(A0A8G,
8A0A(B BBBI   L   W  *   BBE B(A0A8DpO
8A0A(B BBBF    \   W  ,    EBB A(D0f
(A BBBEm
(A BBBDI(A BBB  L   <X  0-   BBE B(A0A8D`E
8D0A(B BBBE     \   X  p.    EBB A(D0f
(A BBBEm
(A BBBDB(A BBB  L   X  .0   BBE B(D0A8G}
8A0A(B BBBB    L   <Y  0   BBE B(A0A8D`
8A0A(B BBBF     D   Y  02   BBB A(D0JY
0A(A BBBJ    D   Y  3    BBB A(D0Ja
0A(A BBBB    <   Z  3    BAD G0d8a@P8F0X
 AABK  4   \Z  @4h    BBA A(O0I(D ABB$   Z  x4P    AAG CAA   Z  4              Z  4-    Ak       L   Z  4J   BBB B(D0A8J`
8A0A(B BBBF     4   D[  5    BAD G0
 AABD     4   |[  @6d    BBD A(G0J(D ABBL   [  x6'   BBD A(M0t
(C ABBFD
(F ABBI   $   \  X7:    AAG kDA $   ,\  p7<    AAG mDA    T\  7&    AX
GE L   t\  7    BBA A(D0g
(C ABBGL(F ABB         \  7%    GT
A      ,   \  7    BAD AB      D   ]  X8    BBE B(D0A8D@8D0A(B BBBL   \]  8a   BBE B(D0A8Jp
8A0A(B BBBE     L   ]  9Z   BBE B(D0A8J
8A0A(B BBBG   L   ]   <    BBB B(D0A8DP
8D0A(B BBBA     L   L^  p<    BBB B(A0A8DP
8D0A(B BBBA     4   ^  <I    BBA A(D0x(A ABB <   ^  <    BLB D(A0u(D BBB      L   _  H=    BBE B(A0A8GPo
8A0A(B BBBA     4   d_  =    BAD DP
 DABA        _   >    A\          _   >!    A_          _  >    A]          _  >7    Hn       <   `  0>@   BBD A(JP
(A ABBI     D   \`  0?    BBE B(D0A8HP8D0A(B BBB,   `  ?t    AAP@Z
AAA     L   `  ?X   BBE B(D0A8Pp(
8A0A(B BBBA    L   $a  @   BBB B(A0A8MpO
8A0A(B BBBH    L   ta  xB   BBE B(D0A8P`
8A0A(B BBBJ     L   a  HC   BBE B(D0A8Op
8A0A(B BBBK     T   b  D    BBE A(D0J@R
0A(A BBBFL0F(A BBB   4   lb   E    BAD JT
 AABG    L   b  E   BBE B(A0A8I
8A0A(B BBBI      b  G           ,   c  GJ    GAK _AB    ,   <c  GB    FAL TFA       l   lc  G   BBB B(A0A8G@m
8A0A(B BBBHnHEPpHA@P8A0A(B BBB    <   c  Hp    BBD A(MN
(A ABBA    <   d  Hp    BBD A(MN
(A ABBA       \d  H              td  H              d  H              d  H           4   d  HO    BBD A(I0s(D ABB 4   d  Hj    BBD A(D0S(D ABB,   ,e   I:    FAG `AA       ,   \e  0IV    BAD KAB         e  `I           <   e  hIY    BBE K(A0|(A BBB          e  I            $   e  IB    AAK rAA ,   $f  IB    BAD wAB          Tf  I1    XV         tf  I3    XW      $   f  JF    AAI xAA    f  @JD    AU
Jc L   f  pJ   BBE B(D0A8I@w
8D0A(B BBBC     ,   ,g  0K    BAD R
ABD   4   \g  K    BAD K0O
 AABK     ,   g  K@    BAA xAB          g  L"           4   g   L    BBD A(F0(D ABB   h  L5           4   ,h  L@    AAI Q
CAHNHA        dh  LC           $   |h  LE    AAJ oHA D   h  Mn    BBB B(D0A8G@L8D0A(B BBBL   h  @Mt    BBB B(D0A8G@B
8D0A(B BBBE        <i  pMP    tX L   Ti  M}    BBB B(A0A8D@]
8A0A(B BBBA     L   i  M]    BBA A(G0o
(F ABBID(C ABB       <   i  M    BBE A(D0p
(A BBBK   T   4j  N    BBE B(A0A8D@vH]PYHA@Y
8H0A(B BBBA  4   j  N`    BBA A(L0G(A ABB4   j  Oj    BBD A(D0S(D ABB,   j  PO:    FAG `AA       ,   ,k  `OV    BAD KAB         \k  O           $   tk  OC    AAK sAA ,   k  OB    BAD wAB          k  O2    XV         k   P4    XW      $   l   PH    AAJ yAA    4l  HPD    AU
Jc L   Tl  xP   BBE B(D0A8J@~
8A0A(B BBBF     ,   l  8Q    BAD R
ABD      l  Q"              l  Q5           4   m  QA    AAJ Q
CAGOHA        <m  QP    tX L   Tm  (R~    BBB B(A0A8D@^
8A0A(B BBBA     <   m  XR    BBE A(D0p
(A BBBK   T   m  R    BBE B(A0A8D@vH]PYHA@Y
8H0A(B BBBA  <   <n  `S   BBA A(G@#
(A ABBJ    \   |n   U.   BBE B(C0A8PP
8A0A(B BBBDaXB`KhApgP                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            `              u     `      Щ            P      v     p            p            
v                 `            ~     0      P                   v           `                              p            p                 P      @            v           `            `      #v                              *v     @                        0      @      <~             G~            S~            [~                                            v     P     b~     \     i~          t~     \     y~           ~           ~     0,     ~          ~     `N     ~     Z     ~          ~     F     ~     p
     ~     ;     ~     0L     ~     #     ~     pZ     ~     pV          @T     ~     pS               ~     Q     ~          ~               P          0     E     0           0I     N      H          F          pB     $     @     +          4          A           J           T     5     ^     P/     j     -     p      (     t     "     {      !     w                0                ~     0                     
                                <~             G~                                            q          }                                        ~          M               p     ~          ʃ     @     T     0     σ     Ж          0     ~     p     N           `          ۃ          m               0          P                                              ~               0                                                   q          }          M                                                                            b~      9     i~     @"     ~      8     ~     #     ~     @6          `!     ~     4     ~     '     ~     @          P$     ~     @3          1     Hv     0     $     .          %          ,          )                                    #     /     :     D     M     Y     i     x                         Ϗ     ޏ                          &     6     H     W     c     t                              ː     ݐ                         !     4     H     Q     ^     i     y                                 ɑ                         1     x     B     T                    H     x     m     }          ؔ                (     X                               h                    8          ؒ     p                               0     p                            5     v     b~          t~     y~     ~               ~     ~     M          :     ~     ~     ~     ~     ~          ʃ     ~     ~     ~     ~     ~               ~     σ          ~     ~     ~               E          N     `     ۃ                    Hv     $     +          4     A     J          T     ^     j     p     t     {          w               ~                                                           B     w     Hv     R     Y     d                                  ~             ~             ~             ~             !~             #~             4~             ~             6~             )~             9~                                                       
       p            #                          #                   o                 '                   
       .                            0#            @                           s             C             00      	              o    PC      o           o    @      o                                                                                                                                                                                                          -#                     F      V      f      v                              Ƌ      ֋                              &      6      F      V      f      v                              ƌ      ֌                              &      6      F      V      f      v                              ƍ      ֍                              &      6      F      V      f      v                              Ǝ      ֎                              &      6      F      V      f      v                              Ə      ֏                              &      6      F      V      f      v                              Ɛ      ֐                              &      6      F      V      f      v                              Ƒ      ֑                              &      6      F      V      f      v                              ƒ      ֒                              &      6      F      V      f      v                              Ɠ      ֓                              &      6      F      V      f      v                              Ɣ      ֔                              &      6      F      V      f      v                              ƕ      ֕                              &      6      F      V      f      v                              Ɩ      ֖                              &      6      F      V      f      v                              Ɨ      ֗                              &      6      F      V      f      v                              Ƙ      ֘                              &      6      F      V      f      v                              ƙ      ֙                              &      6      F      V      f      v                                      7#                             q     *r     q     #r     $q     q                     Tuple   Relation                      @                   p                                    @                                                       `      P      P                                            `_     0_     _     pa                             M     T     [     `     f     m                     r                          {          [                                                                 q                         }                                                               0J      J     I     S     GCC: (Ubuntu 6.3.0-12ubuntu2) 6.3.0 20170406                                                                                              '                    @                    PC                    C                    s                   	                    
 0                                        К                   
 p                    q                   Н                   @                   #                   #                   #                   #                   -#                   /#                    0#                   7#                   :#                                                            #                  К                                 .     `              D     :#            S     #             z                        #                                      q                 q     
            q                 pq                 Xq     	           *r                Hq            "    @q            1     8#     @       <    #r            H    q            V                   f          {       s    p      |                                                    i                  C           p      {                 |           p                       A           `      ~                 |           `                       A           `             
                                  #    @      A       .          ~       ;                 J                 V    `      d       `    Щ             l                 z    P                       H           0      C                                  i                                                                #                     8                                   A           P                 `      z                        "    p             -    @             9    P             C                   Q    0             \    @             g    0 #            u                       P      '                 :                 $                 !                             @                 P      	           `             ,    p      -       H          Q       e                  v          y           @                                  `8#     (           8#     (                                                               0            $                7         B      M                 d                v                    P                      A          p
                0                     f           -#                
     =                              }      2          *      D    0           U               g          *      z    0     d               h                            !               "               #     [          ##     0            (     /          :#                @ #     P           0,     B      .    -           ?    P/     q      V    5           k    ;           }         E           @               pB               F                F                H     0          0I               0L     *          `N               P     x      '    Q           <    pS            O    @T     ,      `    pV           t    @X     ,          pZ     8           Z     ^          \                \     E           #                                   _            	    0_     #       &	    `_            >	    pa            Q	                   ^	    f     D       q	    f           	     $#             	    ph     ;       	    h           	    i            	     l            	    l            
    m            4
    n           `
    Pp     a      
    q            
                
                
                   
                                  0     W      )                7    0            G    Ж           \         	      o                                  0     t                                                         :#                 @     ?                     
    9#     (           9#            ,          !      9    P           N    0     J       b                u          h          p               p                    |                               D                          `9#     8       
         <       
     $#           *
                   :
                Q
    `     5      m
     &#     `       v
                
    p           
          x       
                
    p     N      
                    0                 :#     P           %#             0                   <                   K    @           \    `!            j    @"     D      }    #                P$     U          %               '               )     U          ,               .               0                1               @3                4     0      "    @6           4     8           C     9            R    `&#            `                   s                       I                 J     !           0J                S                                    )#     x          *#     (          ,#     0           '#     x                             Y            )    Y            9    Z#             B    :#             K                   X                                       l                 z    #                                      z#                 {#                z#                7#                 m               z#                -#                 z#                 Н                 :#                  0#             0    o     .      J    T           `         :       n    u                      l                 N           Y                                P                j     "           g     2           `b            *                     F    t     )       _                u    H                Pg     C                 C           0      T           `      d           @8#                                      :     h           ph     H       1     c            L    0     Q       Y                     m    p                                       k           p      X           s                @     %           ?     <                                             .                H         ;       \    V            p         1                p                                           pb     t           0P                     R           a     E       $    Z     :       8    `     \       E    0            [          |       x          C           P      9                                  )          \     F           w     O           :#             	    K            '           C       D    `s     -       U    0     {       h         A       |    f     :           Ќ                @x     Q           p     b          pt                	               @I            %                 2    h     D       G   
 p             M                     a     b     n       x    p      D                k                            0                                            p          t            )                @    0g            R    в      b       i    g     B       |         R           0      C                 8                                                                                P      n            W     J       (    ``           ?     :            H    0{     A       c    ^            v    P	     J           ;                                     ;     J               \                                 T           [     B       (                F          B       X    s     $       g    c     ]           Pr               {               i                    :           e     `           y                    C           x     i       5    0      S       R                 d          W      v    p      4          @     N           c               u     !                                                @l                8#     (       !    \     3       3    H8#     	       H          R       \                 v    pk     P                 O           `                PQ               `                                      @      s       
    ;     -            @=            9     \     B       L    p            e    @c     }       {                   k     ~           е                 j     2          @]     D                       	           C               y       -                      <    ]           P                i    P      y                                     b          `      d           b     P                {            Y     p       
          h                F       7    w     i       W    pB     a      l         4                v          :#     (           0      &           X     p           j     5           @                       V                  S       %      m            8      k     A       L     [             i           J            p\     1                              `      m                                             v     E       !    0h     4       !    p     e       :!    0            K!    0             b!     A            x!     f     j       !    
            !     Z     j       !    {#             !    `            !    P;     P       !    t     #       !                "         0      ,"          4      >"         T      ["          t       o"     >     d       "          X       "    :#             "          R       "    0_            "    Л            "           g       "    C     Z      #    `            #    p             7#    а      S       Q#    0             e#    p>     '      |#                 #         L       #    L     t       #    Pv     i       #                     #     G            $    _     @       )$     v     O       F$     a     @       Z$               v$               $    _            $                $    W           $    u     ,       $           -      %    @     v       %    `     %      3%    О             >%    @      ^       Q%                      e%    P      y       y%               %    Y     O       %    p            %                 %                 %                 &     @     &        &    `     T       >&    `     "       P&          d       a&    Z     V       v&          n       &                 &    Y            &     e            &    `     K      &    _     4       &    0     P      '    `N           *'                4'    @F            P'    f     V       e'    ?     :       y'                 '    	     C       '    P[     Y       '          (       '          C       '    w     i       
(                      $(         ;      B(                Z(    PW     B       s(     d            (    A            (                 (                     0^            (    G     I       (     M     X      )          @      )    J     @      7)     j            J)    0[            \)          g       l)     H            )          )       )  "                   )    `a     C       )   	               )                 )     9#     (       )                 
*    0     }        *    `     5       2*    P@            H*         :       X*    PJ     7       h*    P      ^       *                *    0S            *    @`             crtstuff.c __JCR_LIST__ deregister_tm_clones __do_global_dtors_aux completed.7561 __do_global_dtors_aux_fini_array_entry frame_dummy __frame_dummy_init_array_entry ral.c pkgNamespace.9478 tupleCmdName.9472 tupleStr.9473 relationCmdName.9474 relationStr.9475 ral_pkgname relvarCmdName.9476 relvarStr.9477 ral_config ral_version ral_copyright ral_attribute.c booleanEqual booleanCompare booleanHash isAByteArray byteArrayHash isAnInt intEqual intCompare intHash isALong longEqual longCompare longHash isADouble doubleEqual doubleCompare doubleHash isAWideInt wideIntEqual wideIntCompare wideIntHash isABignum bignumEqual bignumCompare bignumHash isAList isADict isAString stringHash dictHash listHash findRalType Ral_Types stringCompare dictCompare stringEqual dictEqual byteArrayCompare byteArrayEqual isABoolean listCompare listEqual ral_joinmap.c attr_0_cmp attr_1_cmp cmpFuncs.8347 ral_relation.c tupleHashEntryAlloc tupleAttrHashEntryAlloc tupleAttrHashEntryFree tupleHashEntryFree tupleAttrHashCompareKeys tupleAttrHashGenKey tupleHashCompareKeys tupleHashGenKey tupleAttrHashMultiEntryFree tupleAttrHashMultiEntryAlloc Ral_TupleCompare Ral_DownHeap Ral_HeadingMatch Ral_RelationIndexByAttrs.isra.1 tupleAttrHashMultiType tupleHashType ral_relationcmd.c RelationSemiminusCmd RelationSemijoinCmd RelationJoinCmd RelationComposeCmd RelationAttributesCmd RelationCardinalityCmd RelationDegreeCmd RelationIsemptyCmd RelationIsnotemptyCmd RelationWrapCmd RelationEmptyofCmd RelationIssametypeCmd RelationIsCmd cmdTable.9199 RelationUpdateCmd RelationRestrictWithCmd RelationRestrictCmd RelationUnwrapCmd RelationUnionCmd RelationDunionCmd RelationUngroupCmd RelationUinsertCmd RelationInsertCmd RelationTupleCmd RelationTimesCmd RelationTcloseCmd RelationForeachCmd orderOptions RelationTagCmd gotOpt.9552 optTable.9561 RelationCreateCmd RelationTableCmd RelationSummarizebyCmd RelationSummarizeCmd RelationExtendCmd usage.9017 RelationRenameCmd RelationRankCmd RelationProjectCmd RelationEliminateCmd RelationMinusCmd RelationListCmd RelationExtractCmd RelationDictCmd RelationArrayCmd RelationIntersectCmd RelationHeadingCmd RelationGroupCmd RelationFromlistCmd RelationFromdictCmd.part.40 RelationFromdictCmd RelationDivideCmd RelationBodyCmd RelationAssignCmd cmdTable.8854 ral_relationobj.c UpdateStringOfRelation DupRelationInternalRep FreeRelationInternalRep SetRelationFromAny ral_relvar.c relvarTraceCleanup relvarCorrelationConstraintToString condMultStrings Ral_ConstraintNew relvarCleanup relvarConstraintCleanup relvarEvalAssocTupleCounts relvarFindJoinTuples.isra.1 relvarConstraintErrorMsg.isra.2 relvarAssocConstraintErrorMsg.isra.3.part.4 relvarPartitionConstraintErrorMsg.isra.5.part.6 relvarCorrelationConstraintErrorMsg.isra.7.part.8 relvarIndexIds.isra.10 relvarSetIntRep ral_relvarcmd.c RelvarNamesCmd RelvarUinsertCmd RelvarInsertCmd RelvarPathCmd RelvarExistsCmd RelvarIdentifiersCmd RelvarUpdatePerCmd RelvarUpdateOneCmd RelvarUpdateCmd RelvarUnsetCmd RelvarUnionCmd RelvarDunionCmd RelvarTransactionCmd transactionOptions.9448 RelvarEvalCmd RelvarTraceCmd traceOptions.9417 traceTypes.9421 RelvarSetCmd RelvarRestrictOneCmd RelvarProceduralCmd RelvarPartitionCmd RelvarMinusCmd RelvarIntersectCmd RelvarDeleteOneCmd RelvarDeleteCmd RelvarCreateCmd RelvarCorrelationCmd RelvarConstraintCmd constraintCmds.9155 RelvarAssociationCmd cmdTable.9134 ral_relvarobj.c relvarGetNamespaceName Ral_RelvarObjDecodeTraceOps opsTable Ral_RelvarObjEncodeTraceFlag Ral_RelvarObjExecTraces relvarResolveName relvarTraceProc relvarObjConstraintEval relvarConstraintAttrNames.isra.3 specErrMsg specTable condMultStrings.9599 ral_tuple.c ral_tuplecmd.c TupleFromListCmd TupleEqualCmd TupleAttributesCmd TupleDegreeCmd TupleGetCmd TupleUnwrapCmd TupleExtractCmd TupleWrapCmd TupleUpdateCmd TupleRenameCmd TupleRelationCmd TupleProjectCmd TupleHeadingCmd TupleExtendCmd TupleEliminateCmd TupleCreateCmd TupleAssignCmd cmdTable.8433 ral_tupleheading.c ral_tupleobj.c UpdateStringOfTuple DupTupleInternalRep FreeTupleInternalRep SetTupleFromAny ral_utils.c resultStrings optStrings cmdStrings errorStrings ral_vector.c ptr_ind_compare int_ind_compare buf.8698 buf.8821 tclStubLib.c tclTomMathStubLib.c __FRAME_END__ __JCR_END__ tclStubsPtr tclTomMathStubsPtr tclPlatStubsPtr __dso_handle Tcl_InitStubs tclIntStubsPtr _DYNAMIC tclIntPlatStubsPtr __GNU_EH_FRAME_HDR __TMC_END__ _GLOBAL_OFFSET_TABLE_ TclTomMathInitializeStubs Ral_TupleAssignToVars Ral_TupleScan Ral_RelvarStartCommand Ral_TupleConvert Ral_JoinMapFindAttr Ral_ErrorInfoGetCommand Ral_RelationGroup Ral_TupleConvertValue Ral_PtrVectorSort Ral_PtrVectorFront Ral_RelationInsertTupleObj __snprintf_chk@@GLIBC_2.3.4 Ral_RelvarNewTransaction Ral_RelvarObjEndTrans Ral_TupleHeadingConvert Ral_PtrVectorFetch Ral_RelationEqual Ral_JoinMapAttrMap Ral_AttributeRename ral_tupleTypeName _ITM_deregisterTMCloneTable Ral_TupleHeadingNew Ral_PtrVectorPushBack Ral_RelationUpdateTupleObj Ral_TupleNew strcpy@@GLIBC_2.2.5 Ral_RelationRenameAttribute Ral_RelvarInsertTuple Ral_RelationConvert Ral_JoinMapTupleReserve Ral_RelvarNewInfo Ral_TupleHeadingIndexOf Ral_TupleHeadingExtend Ral_SafeUnload qsort@@GLIBC_2.2.5 Ral_RelvarObjCopyOnShared Ral_TupleDupShallow Ral_ErrorInfoSetCmd Ral_RelvarRestorePrev Ral_RelvarObjCreateCorrelation Ral_RelationDivide Ral_AttributeScanType Ral_IntVectorMinus Ral_TupleUpdateFromObj Ral_RelationUnionCopy Ral_IntVectorOffsetOf Ral_IntVectorDelete Ral_TupleDup Ral_RelvarTraceRemove Ral_RelvarObjTraceVarSuspend Ral_RelationProperSubsetOf Ral_RelationScan Ral_RelationPushBack Ral_RelationTagWithin Ral_IntVectorPushBack Ral_ConstraintNewCorrelation _edata Ral_TupleHeadingAttrsFromVect Ral_RelationProperSupersetOf Ral_RelvarDelete Ral_RelvarFindById Ral_RelvarObjEndCmd Ral_PtrVectorDelete Ral_RelvarDiscardPrev Ral_ConstraintNewProcedural Ral_RelvarObjFindRelvar Ral_RelvarIsTransOnGoing Ral_RelvarObjInsertTuple Ral_TupleHeadingStringOf Ral_SafeInit Ral_PtrVectorPopBack _fini strlen@@GLIBC_2.2.5 Ral_IntVectorIntersect Ral_JoinMapTupleCounts Ral_RelvarObjExecSetTraces Ral_AttributeValueScanFlagsFree Ral_RelationMinus __stack_chk_fail@@GLIBC_2.4 Ral_RelationUnwrap Ral_RelvarTransModifiedRelvar Ral_RelvarIdIndexTuple Ral_PtrVectorFill Ral_AttributeTypeEqual Ral_PtrVectorStore Ral_TupleSubset Ral_RelationSubsetOf Ral_RelationUpdate Ral_TupleDelete strrchr@@GLIBC_2.2.5 Ral_RelvarFindIdentifier Ral_RelationSort Ral_JoinMapNew Ral_ErrorInfoSetError Ral_RelationObjConvert tupleCmd Ral_ConstraintDeleteByName Ral_IntVectorErase Ral_RelvarObjExecDeleteTraces Ral_TupleHeadingUnreference memset@@GLIBC_2.2.5 Ral_TupleHeadingStore Ral_TupleHashAttr Ral_TupleEqualValues Ral_JoinMapTupleMap Ral_IntVectorFetch Ral_RelvarObjConstraintMember Ral_JoinMapDelete Ral_RelvarFind Ral_IntVectorContainsAny Ral_RelvarNew Ral_RelvarDeclConstraintEval Ral_PtrVectorInsert Ral_RelvarTraceAdd Ral_PtrVectorNew Ral_ConstraintDelete Ral_RelvarObjExecUpdateTraces Ral_ConstraintProceduralCreate Ral_AttributeNewRelationType Ral_RelationUnion Ral_RelationErase Ral_RelationTag Ral_TupleEqual Ral_RelationObjParseJoinArgs Ral_RelvarDeleteTransaction memcmp@@GLIBC_2.2.5 Ral_RelvarObjTraceEvalInfo Ral_PtrVectorCopy tupleAttrHashType Ral_IntVectorBack ral_relationTypeName Ral_TupleDupOrdered Ral_RelationValueStringOf Ral_PtrVectorEqual Ral_RelationShallowCopy Ral_RelationTclose Ral_TupleSetFromObj Ral_AttributeConvertName strcmp@@GLIBC_2.2.5 Ral_RelationSemiJoin Ral_TupleHeadingFetch Ral_TupleHeadingPushBack Ral_IntVectorStore Ral_RelvarIdUnindexTuple Ral_IntVectorSubsetOf Ral_RelvarObjExecEvalTraces Ral_PtrVectorSubsetOf Ral_AttributeValueObj Ral_RelvarDeleteInfo Ral_IntVectorPopBack Ral_RelvarObjConstraintPath Ral_RelationScanValue Ral_TupleCopy __gmon_start__ Ral_IntVectorInsert Ral_AttributeNewFromObjs Ral_AttributeValueCompare memcpy@@GLIBC_2.14 Ral_RelvarObjFindConstraint Ral_AttributeNewTclType Ral_IntVectorEqual Ral_RelvarDeleteTuple Ral_InterpErrorInfoObj Ral_RelationDelete Ral_RelvarObjTraceEvalAdd Ral_ConstraintCorrelationCreate Ral_TupleHeadingJoin Ral_TupleGetAttrValue Ral_RelvarObjKeyTuple Ral_TupleObjType Ral_AttributeScanName Ral_InterpErrorInfo Ral_PtrVectorFind Ral_RelationStringOf Ral_JoinMapAttrReserve Ral_AttributeDelete Ral_PtrVectorPrint Ral_PtrVectorSetAdd Ral_IntVectorFillConsecutive Ral_JoinMapAddTupleMapping Ral_IntVectorFront Ral_AttributeScanValue Ral_RelationExtract Ral_RelvarObjTraceVarInfo Ral_TupleUnreference Ral_ConstraintNewPartition Ral_PtrVectorBack Ral_RelvarObjTraceVarRemove Ral_RelationJoin Ral_AttributeHashValue Ral_TupleHeadingUnion Ral_PtrVectorDup Ral_RelvarObjUpdateTuple Ral_IntVectorDup _end Ral_TupleScanValue Ral_TupleHeadingDelete Ral_RelvarStartTransaction Ral_TupleStringOf Ral_RelvarObjConstraintNames Ral_RelationTimes Ral_RelvarObjCreatePartition Ral_TupleCopyValues Ral_TupleHeadingSubset Ral_RelationReserve __bss_start Ral_JoinMapGetAttr Ral_IntVectorExchange Ral_Init Ral_RelationFind Ral_TupleHeadingCompose Ral_TupleValueStringOf Ral_AttributeTypeScanFlagsFree Ral_AttributeNewTupleType Ral_RelationCompare Ral_TupleHeadingAppend Ral_JoinMapAddAttrMapping Ral_TupleHash Ral_TupleHeadingAttrsFromObj Ral_ConstraintAssocCreate memmove@@GLIBC_2.2.5 Ral_TupleHeadingCommonAttributes Ral_IntVectorBooleanMap Ral_ConstraintNewAssociation Ral_IntVectorSetAdd Ral_RelvarObjConstraintInfo Ral_TupleUpdateAttrValue Ral_RelationInsertTupleValue Ral_RelationUngroup Ral_InterpSetError Ral_ConstraintFindByName Ral_RelationFindJoinTuples Ral_TupleAttrEqual Ral_RelvarObjTraceUpdate Ral_Unload Ral_AttributeEqual _Jv_RegisterClasses Ral_RelationProject Ral_RelvarObjNew Ral_IntVectorNewEmpty Ral_AttributeConvertValueToType Ral_JoinMapMatchingTupleSet Ral_RelationNotEqual Ral_AttributeConvertValue Ral_TupleHeadingFind Ral_RelvarObjConstraintDelete Ral_IntVectorSort Ral_AttributeDup Ral_IntVectorReserve Ral_RelvarObjTraceVarAdd Ral_AttributeToString Ral_ErrorInfoGetOption Ral_IntVectorPrint Ral_RelvarObjCreateAssoc Ral_RelationObjNew Ral_RelvarObjDelete Ral_TuplePartialSetFromObj relvarCmd Ral_TupleHeadingNewOrderMap Ral_PtrVectorReserve Ral_TupleHeadingDup Ral_RelationDup Ral_RelvarObjExecInsertTraces Ral_IntVectorNew Ral_RelvarObjExecUnsetTraces Ral_RelationSupersetOf Ral_ConstraintPartitionCreate _ITM_registerTMCloneTable Ral_RelvarObjCreateProcedural Ral_AttributeValueEqual Ral_ErrorInfoSetErrorObj Ral_IntVectorCopy Ral_TupleHeadingIntersect Ral_RelationSemiMinus Ral_AttributeConvertType Ral_TupleHeadingMapIndices Ral_TupleSetFromValueList Ral_RelationConvertValue Ral_TupleHeadingNewFromObj Ral_PtrVectorErase Ral_IntVectorFill Ral_RelationNew Ral_TupleHeadingScan Ral_JoinMapSortAttr __cxa_finalize@@GLIBC_2.2.5 Ral_IntVectorIndexOf _init Ral_RelationCompose Ral_RelationObjType Ral_RelationIntersect Ral_RelvarSetRelation Ral_IntVectorFind Ral_TupleHeadingEqual Ral_TupleExtend Ral_TupleObjNew Ral_RelationDisjointCopy Ral_RelvarObjTraceEvalRemove Ral_TupleObjConvert Ral_IntVectorSetComplement  .symtab .strtab .shstrtab .note.gnu.build-id .gnu.hash .dynsym .dynstr .gnu.version .gnu.version_r .rela.dyn .rela.plt .init .plt.got .text .fini .rodata .eh_frame_hdr .eh_frame .init_array .fini_array .jcr .data.rel.ro .dynamic .got.plt .data .bss .comment                                                                                          $                              .   o                   	                            8                         8                          @             '      '      .                             H   o       @      @      Z                           U   o       PC      PC      P                            d             C      C      00                           n      B       s      s      @                          x                                                       s             0      0                                  ~                                                                    К      К                                               p     p     	                                            q      q     ,                                           Н     Н     l                                          @     @     n                                          #                                                    #                                                    #                                                     #          0                                           -#     -                                             /#     /     p                                           0#      0                                              7#     7                                                :#     :     H@                                    0               :     -                                                   :     87         !                	                      r     *                                                   ݜ                                  MZ                @                                       	!L!This program cannot be run in DOS mode.

$       PE  d kπY     &    n           m                        p    <  `                                      u     l           P             x                            (                   ܱ                           .text   H                     ` P`.data                        @ `.rdata  @I      J                @ `@.pdata     P     *             @ 0@.xdata     p     B             @ 0@.bss    P
                         `.edata  u         `             @ 0@.idata  l        b             @ 0.CRT    X         j             @ @.tls    h         l             @ `.reloc  x        n             @ 0B/4               t             @ B/19     @       x             @ B/31     ~         B             @ B/45              b             @ B/57     	     
                @ @B/70     	                      @ B/81     8.   0  0                @ B/92        `                  @ B                                                                                                                                                                                                                                                                                                                                                                                                SH     HH H3 HHH3 Ht1H    H [ø   H [ AUATUWVSH(IMuz   H3 1z    H-ڡ   HH3HuH=2       t    H([^_]A\A]f.        ueH%0   H2 Hp1H-e     H9    HH3Hu1H=j2           H1 H Ht
M   LЃ~    H([^_]A\A]1/fD  L%	2 I$` HHtGL-2 IM H HHH9wHHtHH9vH IE     I$    1    H   H([^_]A\A]@    fD  1H$fD  H1 H
1         Hl1 H
U1     D  ATUWVSH H50 HωLŉuaK} t?B I1H Aątu$IHl IHA\uE1DH [^_]A\Ð CIHw&tIH AuuI1Hf.     uL I   H AvI1H I1H I1HI   H A    HHH50      t
HHLD$8T$4HL$(}  LD$8T$4HL$(HHqUH]f.     UHH H=  t0H
  Ht/H H Ht	H
 H
   H ]
 H@ UH]ÐAUATUWVSH8H E1H2 H   Hr E1E1HԷ H   H5T/ E1E1H HH  HH  HHE1Lo Hp HHD$       HE1LI HH  t)P
     H8[^_]A\A]D  HH
	 L    IHH
   MIHHAԅuHE1L  H HHD$       HE1Lv HH  IHH
c L    IHH
4   MIHHAԅH
 H  LIL H HHD$     A  HE1L HH  HH
 L    IHH
v   MIHHAԅ\HE1IH9 H  LIHHA  &HHL
 L H   HE1L H HPHH   @ f.     f.     1 f.     1ÐVSH8Hs, HLD$(H1H  t	1H8[^H1LD$,H  uD$,9D$(H8[^@ VSH8H, HLD$(H1H  u)H1LD$,H  uD$(+D$,H8[^ H8[^@ VSH8H5+ HHD$,    1LD$,H  u!D$-T$,D$.D$/H8[^HHHp 
  H
 HfH(HE+ H1H   1H(H8H%+ HT$,H   HD$,HtLD
1fHJL9uH8ff.     H8H* LD$,H @  H8fVSH8H* HLD$(H1H@  t	1H8[^H1LD$,H@  uD$,9D$(H8[^@ VSH8HS* HLD$(H1H@  u)H1LD$,H@  uD$(+D$,H8[^ H8[^@ VSH8H5) HHD$,    1LD$,H@  u!D$-T$,D$.D$/H8[^HHHp 
  H
 HfH8H) LD$,H H  H8fVSH8Hc) HLD$(H1HH  t	1H8[^H1LD$,HH  uD$,9D$(H8[^@ VSH8H) HLD$(H1HH  u)H1LD$,HH  uD$(+D$,H8[^ H8[^@ VSH8H5( HHD$,    1LD$,HH  u!D$-T$,D$.D$/H8[^HHHp 
  H
 HfH8H5( LD$(H (  H8fVSH8H( HLD$ H1H(  t	1H8[^H1LD$(H(  uD$ 1f.D$(DH8[^@ f.     VSH8H' HLD$ H1H(  uIH1LD$(H(  u2D$    L$(f.w1f.H8[^f     H8[^@ WVSH0H="' H\$(HHHD$(    1HI(  u,1I    IAPHt$0I9uH0[^_ HHHx 
  H
Y HfH8H& LD$(H H  H8fVSH8H& HLD$ H1HH  t	1H8[^H1LD$(HH  uHD$(H9D$ H8[^fVSH8H#& HLD$ H1HH  uIH1LD$(HH  u2HD$(H9D$    Ht$(H9t$ OH8[^    H8[^@ WVSH0H=% H\$(HHHD$(    1HIH  u,1I    IAPHt$0I9uH0[^_ HHHx 
  H
 HfSHPH$% H\$0H I  uH% D$,HHR@D$,HP[ÐATUWVSH`H=$ Ht$ HH1HI  t1ۉH`[^_]A\ÐHLd$@1HM  H=$ tHHP@H`[^_]A\f     HHL1PPHLP@fATUWVSH`H5?$ H\$ IH1HI  u*HH|$@1LI  H5$ tHHP@H`[^_]A\fD  HHHPPHHP@    VSHHH# 1Ht$ H1H I  u6\$ HT$0ۍCtHL1HBH9uHz# HH P@HH[^     H8HE# LL$(LD$$H x  H8 f.     H8H# LD$,H   H8f1 f.     H8H" HT$,H X  HD$,HtLD
1fHJL9uH8ff.     @ f.     @ f.     HH	 D  WVSH Hb" HHH
  HHH
  HHH [^_ f@ f.     WVSH H" HHH
  HHH
  HHQ H [^_f     @ f.     WVSH0H! HHT$(H  HHHT$,H  DD$(D9D$,HDND$,HMc H0[^_    WVSH0HR! HHT$(H  HHHHT$,  LcD$(1D;D$,tH0[^_@ HH蝿 H0[^_D  ATUWVSH0H=  Ld$,HHHM@  uR|$,v<H   t2H1HL   X  H
 HALHHAh  H0[^_]A\fHMHH  ÉH0[^_]A\D  f.     f.     f.     WVSH`HA(   A
   HT$0HHW HL$0HD$ 訾 HHtPH0 H!H H P(HH HH HHC    HC    HCHH`[^_D  1ff.     WVSH HHƽ H!Hd H P(HH HH诽 HHE C   HsHCHH [^_     WVSH HHf H!H H P(HH HHO HH C   HsHCHH [^_     VSH(QH˅t)vH5 H
 HP D  HIgt H5 HHH@0H([^H f.     H(QtDr2uHQH	H(H6 H
g H P 1H(fHQH	H(HQH	H(H(HЋQtQr?uHQHH(f.     H H
* H P 1H(D  HQHH(HQHH(H(At6v Hy H
 LAP 1H(fHRHIH(Os HRHI誻 H(D  VSH(H9HHt@HH	| uV9StH([^Ð1H([^    HHH([^O   H([^ UWVSHxH֋QHLǃ   rQu/H 1LL HH   (  1Hx[^_]Hv H
? H P     H-Y HT$8HHE X  HE HHT$<X  L$8   D$<tHCHL$@Hq A(   A
   HD$@HyHD$ 诺 HHH   PR     H 1L HH   $H1L` H   HW HN RM T$<1Hx[^_]f     H1L H   HW HN f8  fD  VSHhHӋQLƅt=wLHHh[^fD  H H
 H P 1Hh[^ HAH% HL$0A(   A
   HD$0H(HD$ ^ HHHtPHh[^f     WVSH RLÃt0ruu{H5` L	 HH   uqHK =  'H57 L HH   uHHK GT HHH  HHHP0HH [^_H H
 H P 1HH [^_Ðf.     ATUWVSH@H LHLHH
  IHLL$8LD$4HHx     D$4   utHT$8HH

  H
L HH詷    H
: H蒷    HD$8IHHP} Ht@HLH@[^_]A\ HH
     IH HH詉 1H@[^_]A\HT$8HH

  LHH}HuIغ   HHD$(蓈 HHX HD$(몐HD$8IHHPL} HtHL    Iغ   W AUATUWVSHxH׋RHLM̃\     t1H H
k H P HHHx[^_]A\A]f.     Hy I9@T  A8x  L- LIE    HOLd$ IIHH藝     IE H`  H  HF HOHYm gIغ   L1诇 LH EH( HT$<LH X  D$<HGHL$@H[ A(   A
   HD$@HcHD$ 虵 HHHPIغ   ifHi I9@/  A8   L- LIE    HOLd$ IIHH    P   1HHx[^_]A\A]HOLL$ HM9  uH HHH `  f.     HOLL$ HM茂 uH HHH `  fHF HOHPk fHHx[^_]A\A]    IE H1   ?IE H`  Hu1& LL WVSHpH֋QH˅t-w
1Hp[^_H6 H
 H P 1Hp[^_H= HT$<HHX  T$<1tHCH\$@HM A(   A
   HHD$@HRHD$ 舳 HtHP Hp[^_D  HHHX  HD$@HJLD
1     HJL9u'ff.     SH AH	HHRBHD H   CH [fH) E@H	H H  HD  H(AtVr4tH H
 LAP 1H(HIDv 
H(H HIHH H  H(HHIv 
H(Ðf.     WVSH0H֋QHσ   ruuPHNHm {H^LD$(贱 F	 F
{HOLD$(HKv HH }@}H)H0[^_ÐH H
 H P 1H0[^_fH HIHE@H H  H0[^_Hf.     HNHž {LD$(H^ F F{[    SH0AILAtzr8uIJ MH0[1  fHi H
 LAP 1H0[ÐHI LL$(LH H  
  LL$(HHIQH0[Hf.     IJ MH0[OI D  f.     WVSH ALMLL$`tvr9uHJ HH [^_2  H H
 LAP 1H [^_H AyHH H  
  AHHHH [^_H    HJ HH [^_I  f.     WVSH HυtJwMAHI~%pHHHfHY H9HuHOH H P0HG    H [^_H H
h H H@ H [^_HD  f.     WVSH HυtJwMAHI~%HD@H4@ HYH9HuHOHd H P0HG    H [^_HG H
 H H@ H [^_HD  f.     UWVSHHH- AHt$ HHT$$H	D$ HE   HHىD$(L8HE P(HHE DD$$HH  HcIHH HH\HHH[^_]Ë+Ðf.     A+Bf     ATUWVSH H-_ Lc0   HcHE P(HHE B    P(HCHJ    HCHE P(HC HCHHC(HH [^_]A\@ f.     VSH(H5 HH	HtHP0HKHtHP0HHH@0H([^HWVSH HH	HcHCH)HH9}1HsH H)H HHcP8HHHHSHCH [^_    WVSH HHIHcHC(H)HH9}2Hs H$ H)H HHcP8HHCHHS HC(H [^_D  VSH8HAHQHH9shHH9t#;t<D;@uD@ 9t,D9@t6HH9uDB1HHVH8[^fD     H8[^@    H8[^@ H+DD$,HTHVDD$,l@ VSH8HA HQ(HH9r(H+QDD$,HTHC DD$,0D@HHC H8[^Ðf.     UWVSH(HHIHcH+HB HkHHH9t HH H9uHH([^_]UWVSH(HHcD    HkHHH9tfD  E1HH讁 H9uHH([^_] f.     LIH	I9t#HcD;u$@ D9tHI9uf.     )Hcf.     LA   HcHQH
 LLL)H fD  UWVSH(HHcD   ~ Hk H[HH9tD  E1HH辀 H9uHH([^_] f.     LI LQM M9t2LHcf.     HcHAI9uIM)IAA1Ðf.     AUATUWVSH(HHc   EQ~ IōFH{ H[,)H9Hcu#HH9tD9$uLH莃 H9uLH([^_]A\A]ÐSH Hd
 (   HH P(HX H@    H [ÐSH H4
 8   HH P(HP(H@    HP HHSHH(HP0H [@ f.     SH HA HH; H	 HH H@0H [H SH HHI _; H	 HH H@0H [HfD  HB HQH	LHL < f     HHRHa< HR ';     H;      SH HA HH: HKc} H	 HH H@0H [Hf.     VSH(H 8   HH P(HH@(   HC P| HCHHC( HFHC0HH([^AVAUATUWVSH I ։IHMHXL(} INHL$} HH,IFHxH0H9uOH9t.HcLH^ HMIT$LHHtANڅEH [^_]A\A]A^1ff.     ATUWVSH t	AL9|1X@ I"yBHMtA؉* A9~)^A9~͉IINމxH [^_]A\ÐATUWVSH HHLh] uqHj HIj HM   HHHw H=1 AH4 HH  HAHH  HLP0HHP0H [^_]A\@ AVAUATUWVSH@H HLHMLز H Ll$,Ld$0H
  H] H;t=    HMLHLt$8HD$0VHHH+U HHHH| H;uH@[^_]A\A]A^Ðf.     UWVSH(H-1 Hιx   HE P(HxHH@p    H1L HH)xHH3HK HE H
  HH([^_]fVSH(HYH;YHtHHa7 H9^uH HN Hx  HNHtHP0H`[ HHH@0H([^H f.     WVSH HHIHcHCH)HH9}2HsH4 H)H HHcP8HHCHHSHCH [^_D  ATUWVSH@HAH9AHH   HMtHH; HƋ HkHK LD$<HShDd$<EtH+kHHhHCHPHSH0H+6 H#6 DH@[^_]A\D  H+ALD$(HTLD$(U    AUATUWVSH(HH	I^ HIHWH+WHHHH_H;_tBL+LH>4 HIMIELFHHPH+P8 E1HHH9_uHH([^_]A\A]Ðf.     WVSH HH	>HVH+VHHHH^H;^tHE1HHKH9^uHH [^_f.     AVAUATUWVSH0H9QHHMM   H;Hn HHV`HtH HHp  MLtHML9 HLt$,HHMVhDd$,Et5HH+VHA   HHPH =4 DH0[^_]A\A]A^HI9t3 HH;MHVhT$,tH+^HHXf     H9 H
J H P     AWAVAUATUWVSH8IHH	H] HI   HIT$HI+T$HD$(HEH+EHHaI|$I;|$   H]H;]tufD  L/LL;1 HHIMIELFHHPH+P6 I$IOIGLBL+BHHPLFH+Pw6 HL$(E1H7H9]uHI9|$rHD$(H8[^_]A\A]A^A_HD$(    ߐATUWVSH HH	IhY HHHVH+VHHHfH^H;^t&HMHHj1 E1HHH9^uHH [^_]A\     AWAVAUATUWVSH   H)HLILHUH+UHy HHHHD$8X HD$ HFH+HUH+UHHHLBU HD$@HHH;Xt/Ld$8Ll$@HcHEMHHHLBX I9\$uHT$ HH|$@HHW HmLL$8HD$HH$   IVMFHHHD$XHD$`HHHD$PHl H   HHY  $   HW uHW HL$@H*/ HD$0L@HD$8L L;`R  ID  Ic$HEIXIHHQ#4 M9eIuHL$ HIHD$(  HLoL IUI+U HRIm I;mtw     HcU IFHL$ H<. L>L;~IL@t/f.     IcHGIXIHHQ3 L9~IuHL$(E1LH7I9muHT$0HL$HE1H HL$PH   HHHL$84r H HL$XH x  HD$HH   [^_]A\A]A^A_L@ f.     AWAVAUATUWVSHhHHHL$   HHHD$H/X H;GHD$@Y  HD$@H x  HXHHPH+PHCH+CHHLfR H|$HL|$@IIHD$0HWMH2V LGIWMHV HSLCMHV   HL$0H~HD$ HFH|$8H9HD$X  Hc HD$PHD$8Ht$@1LD$PH HxH@HHhH+hHD$HH+pHw H HHH      H[ HL$ HSH+SHLcH[HFHD$(I9toHL$0L3H|, LxHHIM71 HHL$(HM<M!1 MFINHHIPI+PM1 HL$ E1LI9uHD$8HD$8H9D$XHD$ Hh[^_]A\A]A^A_H$   Iغ   k HL$ HD$     H$   Iغ   ck HD$     H$   L    ?k HL$0P HD$     oH$   Iغ   k HD$     L@ AWAVAUATUWVSH8HHA IHLDI$Hź   $U HIHHD$ R I;G   HL$ MHH;_HD$(}   L-t @ HcID$HHL$ H,* LpIHMHEMHHPH+Pd/ HI,IE DvD  HE  E1HL$(LH9_uHD$(H8[^_]A\A]A^A_H$   Iغ   i HL$ YO HD$(    @ f.     AWAVAUATUWVSH   HH IHL$0  L$8  XIHƺ   S HHHHD$ EQ H;Gs  HL$ L-: HD$(HD$pL HHD$0IE H
  H$0  H0HD$\HD$@HD$`HD$8Hc$@  HD$HH$0  H;puo    I\$IOIGIHHHPH+P- HكHl IE Hc  HE  E1HL$(I\$HwH$0  H9ptiHcIFHL$ L<( HhHH$8  L|$`LD$@HT$8HL$0HD$h$   IċD$\FHD$HHID$9@ IE HL$0x  HD$(H   [^_]A\A]A^A_H$H  Iغ   g HL$ IM HD$(    @ f.     AWAVAUATUWVSHHHIHL$   H'R H;CH(  H x  HpI$HPH+PHFH+FHHLdL HSIIHHHD$(:P LCHUIH'P HVLFIHP |  HL$(IT$I+T$HHD$ HgH+kI|$HI;|$   HcL= H    HD$0HQ HD$8       HL$(Lm H& LpHNHHM+ HHKM4HFMHPH+PHVf+ MEIMHHIPI+PMH+ HL$ E1HI9|$tRH7H\$01ILD$8H^H+H   WH$   I   2f HL$ HD$     HD$ HH[^_]A\A]A^A_f     H$   I   e HD$     H$   Lи    ge HL$(J HD$     H$   I   =e HD$     pD  f.     UWVSH(HH)HHLLLH9u-fD  HH9tHIHuHH([^_]H@ UWVSH(HH)HHLLL-H9t     HIHHnH9uH([^_]ATUWVSH 1HHH	HEML   HHSHKE1IHdHHyT EHHWHOIItVH;GtPH`-    IHHc H HH P0H1Hg HH [^_]A\Hg HH [^_]A\fAWAVAUATUWVSH   HHϺ   LLd$@LrHSLCILHD$(Lt$0U1HKH_H;_ItXHLLl$8LHD$0$   Ht/HpL>L;~t"fHH+WEHIHL9~uHH9_uH LH x  Lf HL$(f HĨ   [^_]A\A]A^A_ff.     AWAVAUATUWVSHhHAHML$   H2HD$8HBHD$@H$   LL$XMHHHD$ N HHD$Hh  HIT$ I+T$HHD$0HI|$I;|$        HcHt$8HL$HL,HcGHt$@L<HD$XH0" IMHXHIEHIHPH+P& IWHMGL,HBH9BIXtF    HHKIރ~tMH& MGHIWMl LHBHH+BI9uHL$0E1H!t/HI9|$ .HL$XDe HD$0Hh[^_]A\A]A^A_HL$0HL$Xe HU H$   I.   -a HD$0    HD$0        AWAVAUATUWVSHhHAHH2LL$   HD$8HBHD$@HD$XL$   LL$PH|$(HHHD$ !N HHD$H  HH$   HHD$0HW H+WHH$   HhH;h   @ HcE H|$8HL$HL,HcEH|$@H<,  IUIHpHD$PMEL0HBH9BtJIX    IHKIA~tIH$ MEHIUH4LHBHH+BI9uHD$XHWLGL(HBH9BIXtB IHKIA}tIHS$ LGHHWH4LHBHH+BI9uHL$0E1LHH$   H9h HL$Pc HL$Xc HD$0Hh[^_]A\A]A^A_HD$0    fD  UWVSH(HjHL	HAHV H+VHHH
H^H;^ tHcCE1HHHT HH9^ uHH([^_]fD  ATUWVSH HLHLGL+GHٺ   HIWH_H;_IH0u2D  HH9_t#HFtHE1HHH9_uLa HH [^_]A\ f.     AWAVAUATUWVSH   HHyHt$`LqH$  HD$(HY HH P
  H$  HHH+HH4l L9HD$0w  H1Ll$\H|$8%LgMHL$   T$\uEHI9t]HHMHxL'L$   L$\tHL$0D}HhLIcfn HL$0HhD}LHIcHn I9uHD$0H|$8L`L+ H{ IH DEDP(Mc1HI׊ HHHLxI$   LpIWH$   DDpH9IcAD uH HH x    }HcE1E1H1ELt6fILHM) HL)BB"HI9uHD9|IID9HL$(AE1PH$  HHD$@HWH+WHHH	Lt$0H\$8Ll$H1fD  HA9~RA|5  tHL$( HxDLHk H LHk HG E1HHA9ALl$8E9uLl$HH LH P0HL$0j HD$@H   [^_]A\A]A^A_HD$0L`L+ H IH DEDP(Mc1HI \HL$()H$  HHD$@HSH+SHa AUATUWVSHHHyH+yHrH+2IH1EHH] 1HHD$0^ HL$0   HL$0Hl$(Ll$ Dd$8HqH+1HHƉx$Ld$ fD  MuHL$0~(Ld$ 1Aa 1MtHL$0uHHH[^_]A\A]f.     WVSH MHHtNH&  HHK HS`HH> HtHcVHCHH [^_D  HCH [^_@ HC HS`HÐAUATUWVSH(HyLbE1HHH	H8tVHHI HH^H[IH9t,fHIH2I9tHE1LoHH9uH\ LH([^_]A\A]@ f.     AUATUWVSH(HyLbE1HHH	HtHHHyH HHH[IH9tfHIHI9t!HH9uH\ LH([^_]A\A]HE1Lff.     AWAVAUATUWVSHXHBM L9L*HH$   LE1HD$@HBI_I+_LLL$   ImHD$HID$MuHHD$0ID$HD$ yH   E1LLcH 9k  L)E1LHLFH 9N  HD$0H+D$ H96  LLLHD$8A HH HLHG IH$   H$   H@HHD$0HD$@H+D$HHH;YHD$    f     HD$0H]L8IIGIOHHPH+Pi HLwL$1L;wtI    IMHHH@HHPH+P2 MHHH9FIL9wu9\$ txH$   HD$0HD$0H9C[H LY +f     H$   L    U HD$8    HD$8HX[^_]A\A]A^A_     HL$8E1LPsH$   L¨    BU HD$8        UWVSH(HH	HHVH+HHHHH;^t HcHEE1HHHH9^uHH([^_]D  f.     AUATUWVSH(HAHHLH9  H;Q  H9   H;n   H9L%g    t7Ln H     HLHV`I$Hp  HK H9uLFHHHI)蛂 HHFH)Hn H9HFt'HHV`HtHH+VHHPHH9^uHH([^_]A\A]D  L% H
 I$P H9L% :I$H
 P )     L%y H
 I$P HFfAUATUWVSH(HiLjHHH	H69 tLHH1C H[IH9t+f     HMHI9HH9uL&W H([^_]A\A]    VSH(HH_x'HVHKH+VH+KHH99!H([^ f.     H(IH(    VSH(HHx'HVHKH+VH+KHH99!H([^ f.     VSH(HHx'HVHKH+VH+KHH99!H([^ f.     VSH(HHOx'HVHKH+VH+KHH99!H([^ f.     VSH(HHx'HVHKH+VH+KHH99!H([^ f.     ATUWVSH HHLMH; H;CHt9HHHIH7 H;C   t5H [^_]A\     I   LP 1҉H [^_]A\Ð   ILP 1ff.     AWAVAUATUWVSH(HAH+AH)LIHT$xHBA@@HD H P(1HFLcH~ MeIUL~I9      I$H]H;]HqHL$xLAtFMAPHHIIx HDtHV(H9]ADIMuH;]tIUIL9uIMH)HHH([^_]A\A]A^A_HѸ   @ f.     WVSH0HH	HLD$(A LD$(XHHH0[^_f     AWAVAUATUWVSHXHMyH$   H$   L$   HD$@HHR {HAH;AHD$H   HD$HHZH LpH$   HxHD$@{HpH;p   H.IHIMnHLg HLHIHHLLHX  IGL|$ IHD$8^HL$@HL|$8H  HXH9quH$   HD$HHP }@ Ht$HH9q;HPHBH+$   }HX[^_]A\A]A^A_H믐ATUWVSH IHZ{H	HLHL@ HcIIHLHZ}B HvHHHH)FHH [^_]A\ÐUWVSHhHa H|$@Hl$ HHD$@    HD$P    H IHHD$X    HD$     HD$H    D$@   HX(HD$(    HD$0    D$    HIIHHHH HHh[^_]     UWVSHhHH|$@H	HD$@    HD$     Hl$ HHD$H    HD$P    HD$X    D$@   HD$(    HD$0    D$    > HJ IHHH HX(HIIHHHH HHh[^_]ÐAWAVAUATUWVSH  BHΉMǄ$         H- IXLl HE H   D$Lt'D$L   D$LHĈ  [^_]A\A]A^A_    H$   A7      L{ I\$ HHD$pGK HE IL$
  HD$xH'     H'       GHD$`    HD$P    L$   L- H= HHLL%& HD$@H$   HD$hPG  "  A   HE HHSLD$h@  =       HH;\$@  HE A   Lt$0D$(    Ll$ IHH	  Hc$   AJ  HHTt]TLCIH%    A   @ HI   E1LLLL$XLL$XHD$PLN =LCIH%    A   l@ HItIA   HE H
A P f     LCIHA   )@ HHD$`D$L   HD$PHtHM HD$`HHkM fD  H L
R    H P  D$L   X    H|$pL
   HKI HHI pH|$P    H|$` HD$ptLHD$($   LLL$`LD$PHT$xD$ HtAHU HHh  ]  HHD$   LD$PLHT$xHD$ WHT$pHI HD$`    IOI+O1H'K 1HHD$PM D AWAVAUATUWVSHH  Hˉ$  L$    H$  L% Ll HpI$H   t   HH  [^_]A\A]A^A_H$  Hv HL& HxI$H   uLt$P11Lo L$  A2      LH!G H$  Lt$0H|$(Ll$ IMHHك$  H$   HD$@3`  9  LHIHH$  P$    H` HD$H}   fD  Lo 11#HT$@Lt$0IHD$(Ll$ MHH_     LIH(HI}HU$  P$     LH$  LD$HHHPH8I$H$  H   UHA     H L
       H P  f.     H   II$LHh  Z  HHf.     AWAVAUATUWVSHH  Hˉ$  L$    H$  L%3 L HpI$H   t   HH  [^_]A\A]A^A_H$  Hv HL HxI$H   uLt$P11Lo L$  GA1      LHD H$  Lt$0H|$(Ll$ IMHHك$  H$   HD$@]  9  LHIHHP$  P$    H HD$H}   fD  Lo 11HT$@Lt$0IHD$(Ll$ MHH]     LIH(HIH$  P$     LH$  LD$HHHPH8I$H$  H   UHA     HI L
       H P  f.     H   #II$LHh  X  HHf.     AWAVAUATUWVSHH  Hˉ$  L$    H$  L% LL HpI$H   t   HH  [^_]A\A]A^A_H$  Hv HL HxI$H   uLt$P11Lo L$  A$      LHǃ$  H$   A H$  Lt$0H|$(IMHHLl$ HD$H[  9  HMIL8HHH   !  H$  L@ HHPH8I$H$  H   	  Lo 11HT$HLt$0IHD$(Ll$ MHHwZ     LMIHHICHM   L$  P$  GI$HHh  U  HHDfD  H L
       H P  f.     H   LHٽ   NA f     H AWAVAUATUWVSHH  Hˉ$  L$    H$  L% L HpI$H   t   HH  [^_]A\A]A^A_H$  Hv HLv HxI$H   uLt$P11Lo L$  'A      LHq? H$  Lt$0H|$(Ll$ IMHHك$  H$   HD$HX  9  HMILHH-H   !  H$  L HHPH8I$H$  H   	  Lo 11\HT$HLt$0IHD$(Ll$ MHHW     LMIHHIH苿M   L$  P$  GI$HHh  cS  HHDfD  H) L
       H P  f.     H   LHٽ   > f     H AWAVAUATUWVSH(Ht8H A   L
ʒ    H P  DH([^_]A\A]A^A_fH5i IXL HH   AtA    HC 11AHH  LsH[II9ut@ HI9tgHHDH	  HHLIHp  t̋E PU ~BAE PAU oHL   ^f     HLHh  HH   @ f.     UWVSH(Ht0HY L
       H P  H([^_]@ H=) IhL HH   t   H([^_]HU HHJH+JHh  H  HH׉H([^_]f     UWVSH(Ht0H L
ؐ       H P  H([^_]@ H-y IxL HE H   t   H([^_]HW HE HHh  HJH+JH  HHՉH([^_]@ UWVSH(Ht0H L
(       H P  H([^_]@ H= IhLn HH   t   H([^_]HM HHAH+A1Hh  HH  HH׉H([^_]UWVSH(Ht0HI L
x       H P  H([^_]@ H= IhL HH   t   H([^_]HM HHAH+A1Hh  HH  HH׉H([^_]AWAVAUATUWVSHh  H$  ALI  L% IXH$  L I$H   D$<t'D$<   D$<Hh  [^_]A\A]A^A_    HC EuDHD$XHHEHD$@? DHD$P< EH   AEHu Dt$HL|$PLl(:L@MHH!   LH+SHHH> L9   I$H
  HHH8# H;CIuHL$P H< Hl$ A   H$  A@      :: D$<   H6 L
w    H P  D$<   Dt$HHCH+CHD)H( HSH+SHHD$`HA H0H;pHLl$`t$HcHCMHHHLB  H;uuI$HL$@
  HT$PHHt$`HHHk H;Fc  HL$`'Ht$XHHD$hHVH+VHLnL9n   HL$`Iu   L} L;}HD$@HXt1f.     IcHFIIHHQ%  L;}HHuHL$P?  L7L;wHD$HLxt'IcHFMIHHQ  L;wHM<uHL$HIH+ H E1HT$@HL$hHD$XL;h6H: H: I$HL$hHh  lK  H$  Hf.     HL$P H^: Hl$ A   LD$@Ht$p   Hg6 H$  H6 HHL$`e HL$P[ H: H9 D$<   _@ f.     UWVSH(Ht0H L
       H P  H([^_]@ H-Y IxL HE H   t   H([^_]HE Hh  HG HԾH,J  HHՉH([^_]f     AUATUWVSH(HLt9H    L
5    H P  H([^_]A\A]f     H= IhL> HH   t   H([^_]A\A]f.     Le HHHkL H   uHI$L  Hh  HE H 1ɅAHHL@ ATUWVSHPHLt.H    L
[    H P  HP[^_]A\H= IhL[ HH   t   HP[^_]A\f     HL$LHHm HVA   D$(    HL$0H
 L HL$ H	  uLf HHL L   uHcD$LLю IT$ HHAT x#HHh    HHLd$ A   A       H5 fD  AWAVAUATUWVSHxHLt9H L
7    H P  D$<   D$<Hx[^_]A\A]A^A_H-U IpL HE H   D$<t
D$<   HCLv HD$@ HC I HD$HHC( HD$X軻IvI;vIG  HD$lHD$PGL.IUI$>" LIIL轼L5 Pv  HI;v   H?&  HHE E1D$    IHT$@H0  HC  HE LD$PHHT$HP  $  T$lWHE E1HT$XH8	    p  HE H@  IHE L LH   u6Mo IIU L|$ A   AA      H3 D$<   HE HL$@H   
  E1E1HHHt$XP   Ht$@P   Ht$HP~iD$<tALdHE H   xPbHE H   PHE LHh  E  HHjHE H   f     HE H   YHE H   2t]uHE H
 Hh    HHHE HH   H     H
# HHHE H
2 Hh    HHHE H
X Hh     HH|f     AWAVAUATUWVSHxHLt6H A   L
    H P  DHx[^_]A\A]A^A_H= IpL- HH   At
A   fL~ HEIIHD$0HqLi IOHD$8IGAH9HD$@  HD$lLl$PIDd$\HD$HM7IMfIFLHhH+hI9u       IL9   IIUHE1D$    HM$H	  HuLl$PL9t!HHHE1E1HH   I9uHt$0P   HL$8A   舷 HLD$HHHT$0P  uD$luqIL9|$@Ll$PDd$\L9t!HHHE1E1HH   I9uHt$0P~IHHL$8Hh  A  HHHL$8E1L護zHHL$0   -HHL$0   D  AWAVAUATUWVSHXHLt6H^ A   L
˂    H P  DHX[^_]A\A]A^A_H=( IhLͺ HH   At
A   fHm Ls LkHM 諵L}H]AHD$8AE I9   HD$LHD$0
HI9   H+HX  LE1ID$    LHA0  Ht5HLD$0LHP  uD$LtHL$8E1HU HLH   
  E1E1HHAPA   AE PAU    HL$8A   )@ HLH   
  E1E1HHAPA~?AE PAU ~!HHL$8Hh  ?  HHMHL   HL   HL   ZHL   5fAUATUWVSH  HLt6H/    L
    H P  H  [^_]A\A] H- M`L HE L   t   @ HE HNHt$ Md$ 
  A@   Iź   H) ILLHt HU HHh  >  HHZHH   ?* Ef.     AWAVAUATUWVSH  HAM  H-$ IXLɷ HE H   tA   DH  [^_]A\A]A^A_H{ HE HI^L H   AuH[ Ll$ A>      L( E1HMH8HH   AGI^ Mt LHL{HE L HL      IW E1MHHIiMtTLL9uHE HHh  9=  HH
H L
~    A   H P  D  LH( HA    f.     AWAVAUATUWVSH  HAM/  H- IXL) HE H   tA   DH  [^_]A\A]A^A_H{ HE HI^L H   AuH[ Ll$ A      L& HMA   HHH   AGI^ Mt Yf.     HL{HE Lm HL      IW MA   H4HI蹰MtTLL9uHE HHh  ;  HHHU L
|    A   H P  D  LH' HA   B f.     AUATUWVSH  HLt6Hߵ    L
|    H P  H  [^_]A\A] H- M`LN HE L   t   @ HE HNHt$ Md$ 
  A=   Iź   HB% ILL蔷Ht HU HHh  L:  HHZHH   % Ef.     AWAVAUATUWVSH(  HAM   L5Դ IxLy IH   t   H(  [^_]A\A]A^A_HO AEI_ADDd$,Ld$0HHA<      LM$ D$,t$AEMl HLCMHH9  L9uIHHh  ?9  HH`fH	 L
z       H P  8f.     AWAVAUATUWVSH  HΉM   L- IhLZ IE H   At&A   DH  [^_]A\A]A^A_f     HM Lt$ I_˯HHA      L+# I|@ HLCMHHz8  uVH9uIE HHh  8  HHaH L
y    A   H P  9f     HA    f.     UWVSH8Ht0H    L
w    H P  H8[^_]@ H-Y IxL HE H   u{HG HPH@H)HHw2HE H
Hh  _  H HH׉H8[^_]@ H|$ A   A;      H   -$ Y        H8[^_]AWAVAUATUWVSH8HAL   L% IXL, I$H   tA   DH8[^_]A\A]A^A_ÐH{ I$HH]L H   AuHS HHH   AFH] Lt E HHkI$L HH   u}HU H豮HIMtqLL9uI$HHh  5  HH4f     H L
2w    A   H P  f     H航 Hl$ A   A9      H" H\$ fD  ATUWVSH0Ht.H    L
w    H P  H0[^_]A\H-ٯ IxL~ HE H      Lg I$HHH@H)HHw:HQH	tlHE LHh  H4  HHufH|$ A   A8      H   ! H0[^_]A\û   H0[^_]A\H|$ A   뼐AWAVAUATUWVSHxBHωLÃ   I@H- M`L HHD$PHE    u[HD$PLh     HL$lHE HS A   D$(    LC HL$0H
u HL$ H	  tE   Hx[^_]A\A]A^A_HF L
Wu       H P  f.     LC(IM H HHtHcD$lH E1LH|HAHI" HC0HD$HHD$PA$H HD$H HE   II;^M}u'X  f     ƃ  HI;^l  IM .   HHHcE1HILHd2   HHE E1D$    ILH0  H   HE E1HT$HH8	  JbHU D$\H   D$\H     LC IM H HHOE1HLfHI  HC(HD$HP!  M}   IEL)HHwcL  A$PA$   H|$PP   H|$HPHE H   M}D  HE LH   
  E1E1HHvf     IMI+M1H 1HI  HC HD$Hft4t\M}HE H   GHE L    M}1HE H      M}1HE HH   H     H
r HH    AVAUATUWVSH  HAM   H|$ A	      H IUHI H   HtXEt$I][DHH-EtlAD$Md 	HL9tWLIHH/  tH膤   H  [^_]A\A]A^H8 L
q    H P     H HH Hh  #/  HH1    AWAVAUATUWVSH  HAL   Hl$ A6      H HVIH Iĸ   M   E~LH^CDHIE   AFLt  E1HLR   HL9ttL  LIHHH5 tH9  LA   H  [^_]A\A]A^A_     H L
p    H P     Hƨ LH Hh  -  HH1     L   H  HHe L譢   g AWAVAUATUWVSH  HAL  H=D IhL HH   tA   DHĨ  [^_]A\A]A^A_HE HVL$   L$   HHD$0H(Hx  uH$      A5   HHD$8 AD$VUUUAD$TDE)RA)P  $   HF E1HD$X  HD$@HD$0$   HPH+PHHD$H$   L$   7  D|$`Lt$@L|$HH$   GL@MHH.  j  H+uDLAIIID;$      HIM 
  HHr H;EHuME    H|$8H HHA   r HL$@  HL$H莗Vf     Hi L
Jn    A   H P  &f     H|$8Ltn    A   H1 HH D|$`H$   D$THn(AtfLl$@Lt$8*     HL  I;E  AHE~0LEHU MHHuHT$8ME    Ll$@LAHL$H   HD$p荘Ld$0HHLZHIHD$x HD$XLD$HLL 貱IINE1HBH+BH$   IEI;NHL$`H$     D$   L|$8IH$   HD$`HL$H1H(A@$   ֙HL$0HI薿H)  HL$@I  LHD$hHp HE1D$    MHT$XH0  H  HEHMIHHPH+P}  HL,H$   Hh8D$T  AH$   fAIHEIE   HMHU Hx  ukHHVL$   MH襋Ht.H$    DBDHHD$8   HD$8H$   P	H   HL$h  Lt$XHLH   
  E1E1HHAPA   HL$xiHL$p_LE HL$pHT$hE1HL$xHD$`HD$`D$   H;AD$   Lt$XHLH   
  E1E1HHAPA~mHL$xלHHL$pHh  '  HHcHHL$X   0HL$h  A$PA$HL   HHL$X   @ AWAVAUATUWVSH  HˉL  L= IhL IH   t   HĘ  [^_]A\A]A^A_Lu IHHoLj M&H   uHE A4      L(HD$@H$   HHD$0n IVIMI+VI+MHHLILHD$8 IUI+UHHH97  FVUUUD$L)ƍv)   HG LHo(HD$Pj  IŋD$LA
  Lt$XLt$0&HL  I;E  AHE   LEHU MHHuHT$0Hپ   l L  HL$8芑D  Hi L
Jh       H P  ef.     H|$0Lth    H7 HHپ    HL$8#     H|$0I   H[ f     Lt$XL蓙HD$hHD$PLHl$@LD$8 H1HU HME1HBH+BH$   IEH9MHL$XHD$p8  Lt$xLt$0L$   $   IH$  HD$XHL$81H(A@$   KHL$xHIHC$  H$   Hs  LHD$`HpB IE1D$    IHT$PH0  H  HEHMIHHPH+P  HL$H$  Hx8D$L   Ht$p      IHID$   IMHHx     HHVL$   MHHtLH$    DBDIHD$0   HD$0H|$0LE    Hj HYH$   PI   D  HL$`  H|$PIHH   
  E1E1HH֋P   HL$h趗HT$`HL$hE1oH|$@HD$XHD$XD$   H;G$   Lt$PILH   
  E1E1HHAPA~_IHL$hHh  "  HHIHL$P   NHL$`  PIH   IHL$P    AWAVAUATUWVSH  HˉL$  (  H$  L=h A   L HpIH   tDHĈ  [^_]A\A]A^A_H$  Ln H$      A   HH@Iu HD$8EA DVUUUDt$xD)RA)   HD$8H 3  HH$  Hh D$xAu/   @ HH  H;F  AHE   LEHU IH见HuHA     HH L
af    A   H P       LYc    HA    HH fD  H訔IU HD$XIEHHD$pHBH+BHFHD$hIEHHD$HH$   IHH9  Dt$|Ht$`MHD$HH(H2  HL$`I  HpHD$PE1IMD$    HT$8H0  H  HEHMIHHPH+PY  HL$H$  Hh0D$x0  AHt$hAIHEID$  IMHU Hx     HHVL$   IH老Ht\H$    DBDIHD$@   HD$@zf.     LE H   	 HH4
 H$   PI   fHL$PF  H|$8IHH   
  E1E1HH֋P   HL$XA   HT$PHL$XE1ɓHD$HHD$HH9D$p!Dt$|H|$8IHH   
  E1E1HH֋P~YIHL$XHh    HHIHL$8   XA$PA$IL   IHL$8    AWAVAUATUWVSH  I͉M   H5 IXL HH   t   H  [^_]A\A]A^A_Ld$ Ls _A-      L~    LMI   I_HI|?(
HH9   HHK
  IHH
  MMHLuLL    L8@ H L
z_       H P  f.     Ly_ 	   L    LL     HLHh    LHAWAVAUATUWVSHxBH$   AMƃv:HS L
4_    H P  D$L   D$LHx[^_]A\A]A^A_ÐH IXH$   L H H   D$LtD$L        Hs HՕ McIKL.H>H 
  HHH_  H;G  L H+GKL.HHH H 
  H^ HSuHIź     LHHHD$P@  H;G  HL$PE1AHD$X;  LvHFHcH<    I9Lt$@M   fD  HL$@I9L1IVH,:   1 IL;~t4IILH@H8yAu1҅IL;~ũHL$P  INLxHIFHMHPH+P  HM4Hb H   I E1HL$XHCHD$@HFH\$@H9t	L~&H HL$XH Hh  +  H$   H   RH HL$lH=<[ IVA   Le H HL$0D$(    H|$ H$   	     HcD$lHTe HDl\H$   Hl$ A   A+      S D$L    H$   A   A+      Hl$ # HL$P  D$L   HL$XBD$L   D  AUATUWVSH(HΉL   L-ܒ M`L IE L   t   H([^_]A\A]f.     Md$ DGLMHI$  HHtHLtHH IE HHh  v  HH׉H([^_]A\A]H8 L
W[       H P  H([^_]A\A]D  f.     AUATUWVSH(HΉL   L-ܑ M`L IE L   t   H([^_]A\A]f.     Md$ DGLMHI$  HHtI$HHPH+PH
 HH HLQHI IE LHh  S  HH׉H([^_]A\A]H L
4Z       H P  H([^_]A\A]ff.     AUATUWVSH  HLt6H    L
"V    H P  H  [^_]A\A] H= IhL. HH   t       Le HHHkL H   uLm Hl$ A&      H  ILLҧHtHHHh    HHIHH  p@ f.     AWAVAUATUWVSHXBHΉMŃ<  H= MpLJ HL   A   Mf *  HIM
  I$HIJ  Hc    ]  ME E1I$H  HH   LEHФHIE H11H  I] I;]Iu   fHI;]   HcID$HHHPHL*Lp  tAPAR  L A   DHX[^_]A\A]A^A_    Hi L
W    A   H P  I$1HBH+BHH   IL$I+L$1H+ 1HI f     HHLh  L) Y@ HL$LHIU A   D$(    L_ HL$0H
U HL$ H	  HcD$LHQ_ E1ME(H|AB     Lt$ A   A%      HA     HL   Lt$ A   A%      HA      f.     AWAVAUATUWVSHHHAL>  H׌ M`L| HL      IT$ HJHBH)HHK  HAL1HD$8  H11Ln  IAGL|G@ HIIM
  HL$8HH/  x;IVLHHLLAp  u;M9uHLHh  L    Ht$ A   A      H  A$PA$HL   fD     HH[^_]A\A]A^A_     H L
U       H P  HHN
  HL$8HHP  x7IVHL$5Ld$ A   A      H     jHt$ A   A      H     B f.     AWAVAUATUWVSH8HLt5HΊ    L
qT    H P  H8[^_]A\A]A^A_ÐH- M`L> HE L   t   @ Mt$ HE HKM.
  IHE HK 
  LLH+  LcE   LH  LcE   HE II  I^I;^Iug    HI;^tVHLU LHH@N(N A  tAPA5HE L      f     HE LHh  L|$ A   A      HM       H\$ f     AWAVAUATUWVSHHHLt5H    L
R    H P  HH[^_]A\A]A^A_ÐH- M`L HE L   t   @ Mt$ HE HK L{M.
  IHE HK(
  LLHw     HLD$<`  DD$<   I^I9^tSHMcL,    I     HLHH@N(N HE D$    0  HEHI;^uHE H  @ Ld$ A   A      H  H\$ fD  AWAVAUATUWVSH  HAM  H- IXL9 HE H   tA   DH  [^_]A\A]A^A_H{ HE HI^L H   AuH[ Ll$ A      L	  HMH+HH   AGI^ Mt IHL{HE L HL      IW MHڝHI߀MtZLL9uHE HHh    HHfHy L
"M    A   H P  f     LH%  HA   b f.     ATUWVSH Ht.H L
6K       H P  H [^_]A\H=م IhL~ HH   t   H [^_]A\f.     HE H4  HHH  IHHP0LLHAh  H [^_]A\    AWAVAUATUWVSHXIAL   H5' IxL̃ HH   t   HX[^_]A\A]A^A_fHG AUL{ щT$<HD$HL HCHD$@H  T$<Hǅ   AELl(H  M9tHIIO
  LHHp  yH\$ A   A      LN  H     GHE L
6N       H P  !fD  HWID$H+I+D$HHH9}dHHL$@
  LHH  xH  H;GtzHL$HIHBHt6HHHh    LHHM A   HD$ HM A   A      L   HD$ M  SH\$ A        AVAUATUWVSH   HLt5H    L
xM    H P  H   [^_]A\A]A^L5 IPLL$(LD$$Ix  t   @ Ld$0A      L  HSLC MHhHI        LHHa  H)|HƋD$$%nE1HHN}D$$HD$(D$$~KH葲  HHD$(IU MHL   uLH     H  H#|IHHh    HHLH     f.     AWAVAUATUWVSH  A   H$   HH$     Hg  HSLC IH$  gHH  HS(LC0IH$  {gHI     Lt$pL|$hK  HHH  LH  HzL- HHD$`HSH\$\MIHD$@H$  IE H\$8H\$(L|$   D$Ltq   f     H  LD$`HU IHHa  t}LD$hI$IHI  teE1HHW{IE LL$8MHT$@L  D$\tIE L  IE HHh  5  H$  H;fD  H$  H  H  IE L  HzD$L   D$LHĈ  [^_]A\A]A^A_ H$  H  D$L   fD  H(t'H L
I    H P     H(ÐLH(@ AVAUATUWVSH  HLt5H-    L
I    H P  H  [^_]A\A]A^H=~ IhL} HH   t
   fD  Le HHHnLk} H   uLm HHHn LI} H   uLu Hl$ A      H^  IMLL轖HtHHHh  f  HH&HH  Lf     ATUWVSH Ht.H~ L
6C       H P  H [^_]A\H=} IhL~| HH   t   H [^_]A\f.     HM נHH PHHM  IHHP0LLHAh  H [^_]A\ AVAUATUWVSH   HˉL   H'} M`   L{ H L   tH   [^_]A\A]A^ Mt$ Ll$0A      L  INIFH)HHwNLl$ H	LMDGH  fH| L
ZG       H P  sf.     LM     LH5  IWVSHPAHDL   H/| HL$LIQLJ A   H HL$0H
G D$(    HL$ H	     tHP[^_ÐHcD$LL
tJ IHHATHP[^_    H{ L
F I   HH P     HP[^_ÐSH HHI HCH CH [f.     SH HI HvHtHC H' HCH [D  SH HHI /uHC     HC    H [f     SH Hz HH   H' HX H@    @    HPH [@ AUATUWVSH(HH	IML  IMLHH|  uVE1HHXutH([^_]A\A]D  MH        HL   H([^_]A\A]ÐH   #  H([^_]A\A]@ ATUWVSH0H$   LHMsHHy HLL$(LD$$HH x  t   H0[^_]A\D  T$$H$t%fHD$(IHHPL HT$(HuMD$$PT$$ID$HtH@HtLH& It$ ID$H0[^_]A\     H   3sH0[^_]A\fD  WVSH0  Hx HHLL$8LD$4H x  t   H0  [^_fH|$@E1   H  |$4t$HI     HHY      HD$8IHHM  HtHT$8IHLBH|$ HLH0  [^_ÐAVAUATUWVSH Hx HLHMLv HH H   t   H [^_]A\A]A^Lv HM IV<  LIHIrt
L  I   L
  LH   m  Ԑf.     AVAUATUWVSH Hkw LHILLv LH H   t   H [^_]A\A]A^Lv HM IV  MILHIstL<  f.     H$   I      F  H$   H  @ AWAVAUATUWVSHHHՋIMąu1H$   L$   1I	H  HH[^_]A\A]A^A_ÐL-iv IH=_H LL$(IE H
     HLL$(uH$   H{LD$4I1LLL$8L8IE Hx    D$4   H$   4gD$4   HD$8K   H$   g      D$4PHD$8T$4HHD$8   IU H
  HH^  HD$8HHIE 
  LH>  AsHD$8H$      L       H$   I	   {  H$   L     mm I$_HD$8H$      L@8  HD$8H$      L   HD$8H$      L@  낐VSH(H5st HHIP~HHH@0H([^HH   fAUATUWVSH(H5-t HjLHL*Lb(AHHH  HAHF H  HHU AHH-H   HAHE H  HcWHcGAHHPHT H  HAHE H  HAIU H  GHtHAH{E H  HHAHkE   HcW0HcG4HAHPHT H  HHAH2E   HI$HA  HHE AHH  H([^_]A\A]HfVSH(H HHr H P(HHHHHH HH([^D  UWVSH(H=ar HHIHt.H=Lr P     H`lHN'  HNHAH+HHw  HN  HN HtHsHHuF0HV8t9H^@H@HHlhH{ tHHx  HK  H`H9uHHP0HHH@0H([^_]HH   Tf.     WVSH H˃      t@uHyH  HOHtFH5-q P:H   /     HyHO HtbHO@Ht bH5p HHP0HHH@0H [^_HH5p H
C HP fD  HyHO(Hu뱐HyHOc  롐UWVSHHHH	HtH  H5qp HoH\$ HHH  Ht@ HHHH  HuHHHo`x  HHH  HtfHHHH  HuHHx  H   Hu/HHHP0HtHKH+PH   HHP0HH[^_]f.     AVAUATUWVSH HqH   HωEML$   H9u+VuEuHHx  L>  1HH9t-t~΅uHHHG  L
  1H9uH [^_]A\A]A^    AVAUATUWVSH0Hq McILK<@MHHl:@1aH^H;^HD$(tNLd$ L    HLHHD$    HtHH+VD@LH0aHH;^uHD$(H  H0[^_]A\A]A^f     AWAVAUATUWVSH(H)n L5@ L-@ LHHMH@ AH  HAHH  HAH^@ H  L'L;$      D  Ic$IIH̧  HHALH  HAHH  HHP0HALH  HAH$   H  HAH? H  L;$   oH([^_]A\A]A^A_@ f.     AWAVAUATUWVSH8H$   H5l LHMH$   HT$ AH L$   HH? Lp H  LOHHL'HT$ ALL$(  HHAH>   HI$HAL%@   HHAHr>   HcWHcGHAHPIH  HHAH>   HcWHcGHAHPIH  HHAH=   LL$(HHAI  HHAH=   L$   IU MFHEIHH$   H8[^_]A\A]A^A_@ AWAVAUATUWVSH8H$   L-1k LHLL$(HAH= H L$   L$   H@ HD$ IE   HoIE HL7AHپH}  IE AH< H  IE IAH  IE AHF= H  Lu H-G= L9t5IAHIHIE H  IE AHH  L9uˋCHٍPIE   IE HAH<   L$   ID$MHH$   HD$ L@HD$(HH8[^_]A\A]A^A_ f.     AVAUATUWVSH H$   H$   L$   HIMLAH HT< Hh Hsi H   ILLL$   HLEHFIHH$   H [^_]A\A]A^     AUATUWVSH81HHLIcHILD$,PPIċD$,   L-h LvIE    P(HH HIE P(HHH
 HebHHC      1HC  LvHK8HCHC     C,    1C(    s0I I\$HH8[^_]A\A]ff.     SH HHHIHPHHh HHp  HH [f.     H(HHIPHHt
H@H(@ 1ff.     ATUWVSH L%g HH͹   I$P(HxHHH     Hǀ       1HH)   H   b  HI$HK1  I$HK`1  I$LIHH  HH [^_]A\    HHBH+HÐf.     SH H	H@  HHHH [  D  f.     SH Hf    H P(   H  HCHH [D  f.     VSH(HˉHH0H([^
   f.     WVSH HH	HHAH+HHw?jH    H  Hq  HHٿ   uH [^_ K   1tHFHH CbHNHw  H [^_ f.     SH HHIo  He HH H@0H [HfD  H(IHHJ`L   Ht	H@H(1    WSH(H     H5e 0   H P(HxHCH     H@(    HH)H01HHH([_    VSH8HHHJ`LD$,H   T$,H1tHmHFH8[^fVSH(H    Hd    H P(   HHCy  HFHH([^Ðf.     VSH8HHHJ`LD$,H   T$,H1tH}HFH8[^fWSH(eH    Hd P   H P(HxHCH     H@H    HH)HP1HHH([_    VSH8HHHJ`LD$,H   T$,H1tHmHFH8[^fVSH(H    Hec    H P(   HHCI  HHCH@    HH([^VSH8HHHJ`LD$,H   T$,H1tH}HFH8[^fATUWVSH H˃    tb   HiHE H0H;pt6 H>HHO  HOH;AtL@H  HE HH9puHH [^_]A\HAHH(HxHp(HM  HMH;AtL@H  HOH~  HOH;AtL@Hh  HNH\  HNH;AtL@HF  tHa H
4 H P Z    HAH8Hp LaHI4$HN  HNH;AtL@H  ID$HhH0H9HHH8HO  HOH;AtL@H  HH9uID$HxH0H9f.     HHHHt/RHH9uSH IHHJ`L   H1Ht!HKH` HH p     H [    AWAVAUATUWVSHxH$   q    s  HYHALsHD$PH;LK DCIFLHOH@ Hq IHD$hbIOI+O1HO  HK I   HT1  1I  DCSMHHD$ IpHAǉD$X1  Eu<H$    Int-ID$I+$HHP	  IEI+E HH  L  L  HNH+N1H  HK 1IH<T1  1I  E11MHHD$ IHAƉD$d  Eu<H$    Hot-ID$I+$HH  IEI+E HH[  L;  L3  HC Ls(LK@DC8HOHPHP IFLLx 1IOI+OH  HK@I   H`S1  1I  DC0S4MHHD$ IHD$H  DL$HEu<H$    Int-ID$I+$HH)  IEI+E HH  L[  LS  HNH+N1H  HK@1IHR1h  1I^  E11MHHD$ I6HA  Eu;H$    Hot,ID$I+$HH  IFI+HH9  L  L  HC@DCHHPEHP z  L$XtT$dtD$Ht
E   u1Hx[^_]A\A]A^A_H] H
#1 H P D  HYHALsHD$HL+LK(DC IFLIMHp Ly HNH+N1H  HK(I   HeQ1  1H  DCSIHHD$ HHA  Eu:H$    Ivt+HGH+HH  HEH+E HH!  Hh  H`  IOI+O1H  HK(1IHP1u  1Hk  DCSIHHD$ HAHA  EucH$    ImtTHGH+HH  HFH+HHv.H/ HT$HH$   Ht$(Hl$ MHD$0I^H  H  HC(EHPHP 1E@!LiHA1HD$XIE HHD$HIEH@HHD$PHFH@ HHH+HH  HIEH0H   H9   HEEHT$HL HXIL$ILq    IHYOIVI+VHHH9        MFM+F1HHINL8L;xIu
2IM;~t'EEtLLI_  He  M;~uLW  H$    tHGH+HH   H1-  HH9t$PHCHC 	1O  1HE  Le L;eHu#KfD    1H  IL;et*A<$tLH  H1  IL;euH$    HD$HL`t*HCH+HH2  HFH+HH   He  H]  HU  H, HT$XH$   H|$(MMHD$0ID$HD$ HVH+VHt$hIOI+OHFH+FHHHHH9PH5X H$   AHf, H  L$   HL$PH(HAHH, H$     H+ LL$HHT$XH$   Ht$(MHD$0Ld$ H+ LL$HHT$XH$   H\$(MHD$0Ld$ H+ HT$PH$   Ll$(Hl$ MHD$0IH#+ HT$PH$   Ll$(Hl$ IHD$0IrH* HT$PH$   Ll$(Hl$ MHD$0IH* HT$PH$   Lt$(Hl$ IHD$0ITHP* HT$HH$   Hl$(Ht$ MHD$0IH<* HT$HH$   H|$(Hl$ MHD$0IH) HT$HH$   H|$(Ht$ MHD$0I{0H) HT$PH$   Ld$(Hl$ MHD$0I}H) HT$PH$   Ld$(Hl$ IHD$0IUH)) HT$PH$   Ld$(Hl$ MHD$0I"H=) HT$PH$   Ld$(Hl$ IHD$0If.     UWVSH(HH]  HHHk8  C0tGHHt@HH H`H9t'HH  tH)Hi۫f     Hc  H([^_]     SH0HAHcLD$ HX HRHLHL@IP8HT$(HT$ A   HtHcPHCHH0[HCH0[@ AWAVAUATUWVSHHA0IHT$0MHQ8   HY@Ld$,H@Hl$0IcHLlh H`HxL9tSHCMHHHD$8SHt$,uMt6L詍  LI   H&  HT HH P0	@    HH[^_]A\A]A^A_     AWAVAUATUWVSH8A0IHLMHQ8tVH5S HY@H@AHL|hHS HD$(HHx  HHH`LD$(DH
  L9uI$H;] tIH@ HH;] t6I$IHMI)HIzuH8[^_]A\A]A^A_D     f     WVSH HMHRLGHu-HCE1HH@ HPL@H [^_     HK   HAPHKHEQ HAHR Hy H `  H [^_ ATUWVSH HAHZ HMHh HHM 踲  HHt@HM KHKHSIIHq]HY  MHHH [^_]A\HhNH ATUWVSH HAHHMHp HLtgHFLFMHI)IAIcHt   H [^_]A\f.     LFHIPlH [^_]A\ H؊     ILHU  H>Q HH P0H [^_]A\     UWVSH8A0HT$ HQ8tOH5P HY@H@H|$ HHlh     HCHHHD$(S@HHH`p  H9uH8[^_]UWVSH(HHHzHGLCHHp HkH;FHHt5    HHEIL+FHE1HHIH9^uHH([^_]fD  VSH(HHI~  HKH  E1HHH([^@ f.     SH HHI?  HJHKH [  f     WVSH HO LHι   H P(xHXHV HHF H [^_fAWAVAUATUWVSH(H-iO HY E1IΉLHE 
  HItW1@ HHHtD9{uHE HK
  LH  uHt1HHL;HLAvHuDH([^_]A\A]A^A_L;M~ ϐAWAVAUATUWVSHXBILσ   1HN   H11Hl$0  IHHOH  HtffODLx QuLx HtHHL  t(HLLp    LILAօu`HH  HuHLLh  1HX[^_]A\A]A^A_f.     HM L
2"    H P     AU BAE    HD$,L   D$,fHIH
  Hf.     AWAVAUATUWVSHX  HˉLMHD$X    ~dL-=M IHIE 
  LIHQH  HHtHPIE HLK    t@   HX  [^_]A\A]A^A_fHL L
B!       H P  ILHF  uL|$`A<      L胼  HL84  F(D$D   HFF(   LwH@ HFHD$HEH|HD$XHD$8I9HL$Xt^Il  L|$ LL$8HMFH#J  HHtHP LD$XHL$HFE PU    D$DI9HL$Xu  E1F(    LHr  uwD$DtIE HN`  IE HL$HHh  HHLG-   L޻  LHC  aIE H   fLG,   ːHL$HfE9AWAVAUATUWVSHX  HALMHD$X    ~cL5J IHI
  LIHF  HHtHPIHLxI    tA   HX  [^_]A\A]A^A_ HJ L
       H P  ILHD  uHD$`A      HHD$0>  HLk  o(   HGG(   HH@ HCAHD$8x  AD$E1D$LHD$XHD$@AD;d$L  HL$XH  HD$0LL$@HHHD$ LCG  HI   HP LD$XHL$8DtPAPAIL   @ LC-   H|$0Hʹ  HH/  f.     H\$0   MH蛹  HH   HL$XV  G(    A   LH>p     EuTIHL$8Hh  HHLC,   [fHL$X  E1G(    LHo  u=IHO`  HL$Xļ  E1G(    LHo  v    HL$8D$0BD$0    VSH8Ht2H+H L
    H P     H8[^f.     H5G IHLL$(H
  LL$(HILC  H   HtHH	Hh    HH1H8[^@ f.     VSH8Ht2H{G L
    H P     H8[^f.     H5IG IHLL$(H
  LL$(IHLTB  HHHh  t%     HH1H8[^f     1  H AWAVAUATUWVSHHHt8HF A   L
E    H P  DHH[^_]A\A]A^A_fH5F IHLL$(H
  LL$(IHLA  HHtHPHHLD    AtA   fD  HG11Lh H  HD$0HG8HD$(G0   HD@HHD$(HD$8HD$(11H(H  L} L;}Huy     IL;}tgAIM J  LHMp  A  HIHAօtP~jH|$0P"HH   HIHT$0Hp  uHD$(`HD$(H;D$85HHT$0Hh  eHH    f.     AWAVAUATUWVSH  H$  ML$  t7HD    L
N    H P  HĨ  [^_]A\A]A^A_HD IHH 
  H$  H$  I?  HIt'HPHOD LB H$  H    t   H$  H$  MD>  uIGI]LB H$  H@ HHD$`HC H    uHk H|$`HE HXH;Xu#cHH&
  HE HH;XtBH3HHA  HH;BuHA   HD$ H$  AC      R  *AO0T  AO0HD$Xƶ  HD$PIG8D$8    IHD$HAG0H@H   L|$@L|$`L$   f.     IE1HHH+Hj  IIHH;XuDsf     HM e  HH%t?LA  IHH;X  I-  HM HHn  uL诶  E  HD$@I`D$8@0H@HHD$HI9?L|$@L$   H|$XHGH+HH  HE HHH+HH胵  HHD$XLt$XH8H;xt-L'I$I;\$tHHѺ  I;\$uHI;~uHE HHPH+PH  HHD$H۵  H$   AC      HHD$hk  H$  Lv  AG(5  HD$`H
;H$   HEH;EHD$8   Hl$@Hl$XL$   HD$8Lm HUH8HD$PI9L   1H H$   IE LH$   Ic$H@HIL@A   HtH@t9   HcIIL;muHU HcMI)II9   H|$@HD$8HD$8H;GEHL$H蘴  H$  H$  E1AG(    xg  P  H$      L:HL$PR  HD$XH|$XH0H;ptHH3  H;wuHL$X賿  :pHL$XLy  T$8HL$P  PHcLmHU HcHA   HD$ HD$`H@HL(HD$xA} ~Lv  IHD$HH0H;p   H$   L$   ML$   ILd$HHl$hH$  D  HH I;t$   HcIUINL<IUHRHH蘙  MHHIH&IVH+ZHHcHHIVH
HtDEHAD	HD$pH> H   I^HD$pH\ML$   H$   L$   L	  IHD$hLD$xH$  LHD$(H$   HD$ :?  HL$HV  H$  H$  E1AG(    A/e  H= H$   H Hh  H$  HL$   H$  A   AC         IEHD$ گ  YHL$X   6  HL$P蜱  IEH$  A   AC      HD$ 蕯  1HHL$HX  ME-   H|$hHr  H$  Hҭ  HL$H#  ME,        AWAVAUATUWVSHh  HMLt?H<    L
T    H P  Hh  [^_]A\A]A^A_     H=i< IHH
  HIH~7  HItHPHHL:    t   @ MHHB6  uIGHl$pAB      HL` ޫ  HIV LL$hLD$`Hx  uHD$dLL$hDD$`Hl$(LHHD$ <;  HIuT$dILLHD$Hl  LH)  Eo(E&  I$AG(   5HD$PHD$HI;D$'  INHL$XHݟ   IHE1D$    MHT$XH0  H/  HD$PMN(LLD$HHl$0HLd$ HD$(7>     HIO`  HHL$XL   
  E1E1HHAA$PA$   AG(    EHHa  urtmHHL$PHh  kHHIFA-   HD$ AB      H|  E1AG(    HHAa  tHL$P!4IFA,   HD$ A   A$A      PA$HL   HL   D  f.     AWAVAUATUWVSH  HMLt?HX9    L
4    H P  HĈ  [^_]A\A]A^A_     H=9 IHH
  HIH.4  HItHPHHL7    t   @ MHH2  uID$AA      Lx H$   HHD$H腨  LH:  At$(F  IEAD$(   I HD$hIE  HD$pIE( HD$X 2MoM9oHD$PH$   s  t$|H$  ILd$`Ht$hHl$p'D  A$PA$  IM;o  IM 腜   IHE1D$    MHH0  H  HMHHP  A  $   tHD$HLL$XMHT$`Ld$ HD\$xHD$0HD$PHD$(:  D\$xAMH$  Ld$`AJA  1tE1ۃAHAD$(    EH5^  HL$P   EHAH+AHH&  HHL$hH   
  E1E1HHHL$XP  HL$hP  HL$pP      HL$PHHh  qHH@ HL   CME-   H|$HH艦  HH  .f     MH$  Ld$`APA   A      ME,   Ld$`t$|H$  E1HIL$`  HL$P/JH   H   H   HL   oHD\$HLD$x   D\$HD$xf.     UWVSH(HL~5BIXItHLCHHl1  uH9u1H([^_]H4 L
	    H P     D  AWAVAUATUWVSH(  HALM~^H54 IHH
  LIH/  HHtHPHHL3    t<   H(  [^_]A\A]A^A_H'4 L
H	    H P     ILH-.  uHGL|$0A      LH@ HD$(ģ  HLy  G(D$$2  AELl$(G(   LeH|$(L$  HDLHY@ IMt$HL2 HL      IV E1MH>I9ItHg-M   LI9uHH|$(L$  1 MIHHHJ_  HtMHHwuLH踣  D$$   E PU    DD$$G(    LHY  ]LD$$HHWAh  D$$?LE-   Lߢ  LHD  H|$(L$  D$$        LE,   D  LHH|$(L$    D$$   MHH   <f.     AWAVAUATUWVSH(  HALM~^H51 IHH
  LIH,  HHtHPHHLA0    t<   H(  [^_]A\A]A^A_Hg1 L
    H P     ILHm+  uHGL|$0A>      LH@ HD$(  HL  G(D$$4  AELl$(G(   LeH|$(L$  HDLH\@ IMt$HL]/ HL      IV MA   H$<I9ItH*M   LI9uHH|$(L$  n MIHHH\  HtMHHuLH  D$$   E PU    DD$$G(    LH$W  ZLD$$HHWAh  D$$<LE-   L  LH肠  H|$(L$  D$$   fD  LE,   D  LHH|$(L$  8  D$$   PHH   ?f.     UWVSHXHLt-H/    L
Q    H P  HX[^_]ÐH-. HL$LH$ IPA   L  HE HL$0HH\$ D$(    	  uXT$Lt   tRHE H
 P 념HE1HHLHL+Id\  E1HHT  S   HX[^_]HHA   HHLHL+I\   HE1HHLHL+I[  A   HHS  @ f.     ATUWVSH HωLL   1L(HA   HHLHL+I[     HUH-u- NHE    HLM E1HA8	     HE1HHLHL+I-[  E1HAHR     EH [^_]A\ÐHUH-,  H, L
a       H P  H [^_]A\fHE HL   H     H
. AHHCff.     AUATUWVSHXHΉLM  L%Y, HL$HH# IPA   L  I$HL$0HD$(    H\$ 	  t   HX[^_]A\A]D  I$HL$LH HWA   D$(    HL$0H\$ L-  H	  u\$L      T$H     o  S  I$H
^ Hh    HHHHU+ L
 H    I   HP  HX[^_]A\A]fD  I$H
" P i  I$L
 I   HP  I$HO 
  LIH%  H\$H   ~\      6  I$L
  6uI   I$L
 I   HP  >fuV  I$L
(  I$H
  P    I$L
  I   HP  I$H
l     P    I$L
  I   HP     I$L
  NLG LHS  oLG LH$S  YLH"T  GLG(HHlR  1LO0LG(HHQ  LO0LG(HHP  HH6Q  D  f.     AWAVAUATUWVSH(  BHALL̓v9H(    L
     H P  H(  [^_]A\A]A^A_fL%( IHI$
  HIH#  HItHPI$HL'    t	   fMHHb"  uIFLl$0A3      LH@ HD$   AtI$IVHh  CLI$HL& L   uMG HD$ IHLD$(}  LD$(   LHK   AF(D$    AF(   AMMLHS  HH   MHL   P   APA   DD$ AF(    HHJN  SI蓊     ILH  LH赗  I$H   P0LG-   L   LH腗  WLG,   D$    TLH]  D$    ,I$L   <I$H   ff.     AWAVAUATUWVSH8  HΉLL~mL-& IHIE 
  HIH*!  HItHPIE HL$    tG   H8  [^_]A\A]A^A_f     H% L
        H P  ID$L|$@A/      LLp \  DGA   HD$<LML|$(LHHD$ $  HH`IT$<LIH
HI/V  M;ftI$E1HIE HHh  HHf.     L9  L	   ̔  LH葕  ff.     H8~'HcIDLL$ II[<  H8fD  Hi$ L
     H P     H8f.     SH@H~u.H'$ L
     HH P     H@[@ H# LL$8LD$0T$,H   T$,LD$0HLL$8IH@[ -  AWAVAUATUWVSH(  HLLt?H# L
-     H P     H(  [^_]A\A]A^A_f.     H-Y# IHHE 
  HIHm  HItHPHE HL!    t	   fMHH2  uIGLfHL! Lh HE L   uID$ Ld$0A&      LHD$(襒  LHZ   Ew(E   HT$(AG(   ML=:H   H茧 MILHHN  HtMHLuLHA   
  P~|AG(    EHHKI  LE D$(HIWAh  D$(fLF-   LC  LH訒  LF,   LHA   臒  HE H   r AWAVAUATUWVSH(  HALM~^H5e! IHH
  LIHz  HHtHPHHL    t<   H(  [^_]A\A]A^A_H! L
(     H P     ILH
  uHGL|$0A      LH@ HD$(褐  HLY  G(D$$2  AELl$(G(   LeH|$(L$  HDLHV@ IMt$HL HL      IV MHJ7I9ItHJM  LI9uHH|$(L$   MIHHH-L  HtMHHZuLH蛐  D$$   E PU    DD$$G(    LHF  `LD$$HHWAh  D$$B@ LE-   L迏  LH$  H|$(L$  D$$        LE,   D  LHH|$(L$  ؏  D$$   JHH   9f.     AWAVAUATUWVSHH  HˉLL~bL% IHI$
  HIH  HItHPI$HL     tF   HH  [^_]A\A]A^A_HD L
e        H P       MHHB  uIGLl$PA      LH@ HD$8ٍ  LH获  Ew(E   HD$LLNDGAG(   LLl$(HD$ H/  HHA   tiT$LILrHHN  HD$8E1H;xtAH   ILHHH  A   P   E1EAAG(    HHUD  I$DHh    HHf.     LF-   L?  LH褍  YLF,   HLA   I$IO`  NI$H   L@ f.     AWAVAUATUWVSHhHMLt4HK    L
     H P  Hh[^_]A\A]A^A_H5 IHH
  HIH,  HHtHPHHL    t	   fIHH  uHEHHLx [  E(D$D  ID$IL$ E(   Mg HD$HM;gHL$0!  HL$\H$   Hl$8HI IAPA   M9g  I$   IHE1D$    MHH0  HtkHMHT$0HP  uQT$\tHT$8MH(F  u5HL$8LD$DIcf.     HL   ZAHl$8H$   PA  A   L|$HHE(    L   L
  E1E1HHAAPA   HL$0P   HEHGA  D$DupHL$DHh    HHID$A,   HD$ A      H  AH$   Hl$8-ID$A-   HD$ HHM`  H   SHHL$H   +HL   E1ff.     AUATUWVSH8  HωLM~   H Hl$@IHH 
  A	      HIΈ  HSIH}  I   Mt#H Hl$0H\$(t$ MLH  H8  [^_]A\A]H L
	     H P     H8  [^_]A\A]@ f.     H(v&H= L
     H P     H(IPMH('  @ f.     AVAUATUWVSHPHALM   L- HL$LH='  IPA   L     IE HL$0HD$(    H|$ 	  uV\$Ld  H
  HcHfA  AHy  tIE HM
  HMH
1  ÉHP[^_]A\A]A^f.     H L
2        H P  A  IE L
y  I   H   P  A   IE L
.  f     A   IE L
  I   HP  JD  AD$H}Hl(IE HHO
  MHH`3     H9uHUMHO;       IE H
  1P f     HUMHq3  f.     HUMH0  f.     IE HM
  LIHd1  IU 1HHh    HH1AIE L
  *D  H(	t'HP L
     H P     H(ÐIPMH(p  UWVSHXAHHDL   H HL$LIQLK  A   H HL$0H
  D$(    HL$ H	     tHX[^_]HcD$LL   IIHHATHX[^_]Hx L
y  I   HH P     HX[^_]ÐUWVSH(H1 HHLLH  HH     Hx  tHPH1AH  HHAH    HAHH  H([^_] AWAVAUATUWVSHXH= HMLL$HLD$@Hx  A   |$@       HD$HE1Ld$DH-  H	  2    HcD$DHD|D$@PHD$HT$@HHD$H~ULA   Ld$0D$(    Hl$ IHHA	  tA   DHX[^_]A\A]A^A_E1f     E>HE1H  HP   AWAVAUATUWVSH8H- HB  AI͉M11HE   HǸ   HCtStHE DHLp    HILAׅtʋB   HU D$,H   D$,I>H8[^_]A\A]A^A_f.     AWAVAUATUWVSHhH L$   IHL$   L$   I$HMLID   AT$,LD$PHI?  HL$PM|$ M  HD$XLt$(ML$   HD$0HD$LHD$8M6M  AFAD$,tH11  HHMFHHh    HLD$PHHp  q  HLD$(HHp  U  MtHMHHp  6  MtHMHHp    HLL$0HLD$8Hx     HA   LD$XT$LH0	     H$    t\HtE PU !  HH@   HHH  HHHL$      uxM   IPHH   M6M@ HL$PLt$(P   APAXHL   Jf.     Lt$(P~pHL$PPH   D  APA~#1HHh[^_]A\A]A^A_Mt6I3HL   ϐHH   HH   MH   5f.     WVSH HHHL  Hv;:tHtIHH:HH [^_À{:uH=b HH  HAHH  HH [^_fD  ATUWVSH  $`  HIL t~Ht$0HINLHH HtKL%
 LHE1HHI$\$ HH  	  I$H  HH  [^_]A\@ 1L%
 D  H
 H
@  1H P fD  AUATUWVSH(  :IHtXHO
 Ht$@HH  HHstHH  H(  [^_]A\A]HHL  HZLj:   D$@    H)  H  HhH9;  H H
  H  HWHƃ HAHA)   HE1H  H   HIUH 	  HE1He  H   HHLA   8	  JB  tr     l$@HLH  @  LD$@HLօuHL  l$@HHwLH   @  HD$     L
  IH  HӋl$@tD  IUH A   HL8	  IMDBD(HD$<   D$<@ HL  HwHLH   @  HD$     L
6  IH{  Hf     HD$<H   D$<    IUH
 E1H>    HL  HHwLH   @  HD$     L
  IH  HӋl$@DD  AWAVAUATUWVSH(IIDL_H=8
 HhH11H  HI$HH@ H9L tVAfD  HH9t?L`  LDHMp  A  HILAׅtƋP~1HH([^_]A\A]A^A_HH1   @ AWAVAUATUWVSHX  $  IHILLRA  Lt$pHLMAIHLHHHD$8  H=	 LH  HD$8H8HHD$P  CIHD@HHHD$XHD$hHD$HHD$dHD$@H$  HLLL$HLD$@Hx    L$dn  |  HEHH
  HD$hHHHL$hHHHHQc  	  HO    D$dPT$dH  HD$PL9Hu!  HH  uH`L9   HHǄ  tH$  H$  
   L Fx  H|  HT$8L葟H$  Lx  A   8f     H$  LQ  )   A   w  H$  LQx  DHX  [^_]A\A]A^A_f.     H$  Iغ   Kw  `fD  H$  H$     L w  8HI6INL I`H
  H$  L;t$XHt$8HE1LLND$    H	  H   H5wHLL|$(A"   E1Ht$ Ht$8H  HHVLh  H$  I"   A   `v  H$  L w  H9 LH   H$  H$     L {v  H$  Lv  HT$8L辝8HT$8LA   覝`AUATUWVSH8Ip1HILÃ>H8[^_]A\A]ÐL% HI$   HC P   I$H5:E1HHLl$(Ht$ A"   1  I$LKE1HHD$    	  HI$H@Ll$(Ht$ A"   E1H  H8[^_]A\A]ÐI$H   i@ f.     ATUWVSH   HLHL
  Hv;:tgHt$@HHITHT$@HA蔜Hu	E   L%o I$HD$8H  HD$8Ht)H   [^_]A\D  {:uHH?HuH\$ A#   E1   HHD$8u  HD$8H   [^_]A\    L% HI$  I$H  I$HAH    I$HHA  HT$@H袛 f.     ATUWVSH0HHHLU0  L%V HI$
  HIHkHH   HPI$HL       HSHBH+HH   H{  t(C,   E1HD$     E1HHC,    I$HH|$(E1HA"   HT$ H  I$E1E1HH   HHJI$H  H0[^_]A\HA'   A      HHD$ /t     H0[^_]A\H  A)   A?      H   HD$ s  H0[^_]A\fD  AVAUATUWVSH0HBH$   IHHMHh IHM i  HHtg H~  PtN,      Ls HM IVzW  uFIغ   Hq  HL;r  P   1HH0[^_]A\A]A^    IVHM a  IIIE LHt     L  HF,   HD$     ILIƋF,    P~8MqL)@ H  HH    M     IH  HH         AWAVAUATUWVSH(L$   AHL$pHT$xDL  HD$xDDH@L` s  II$u0  H   L=   HHH*L&v  LCMHH3  t8HH9tgIH
  I$HHZ  yI   Lo  HL$pLkp  Ls  H10  HH([^_]A\A]A^A_D  HL$xL賩x@H$    t
H$   Lrs  M	   L1o  HL$pLo  L      Lo  ^fD  AWAVAUATUWVSH8H-  HBHHMLL  LLp HE    t   H8[^_]A\A]A^A_D  IM I1c   IăH~  tF,   A$PA$2  P  MtHE L  LH   qIG IHPT  uzH$   M   en  H$   Hn  APA&HE L   Ly  F,   MH\$ HHF,    I*IU HdH$   MILLHD$ tRAPAHE L   HE H   f.     HE L   L$   H$   MHÄMM+FIU E1HI'oH$   M      %m  H$   Hm  @AVAUATUWVSH H=  L$   HIMH$   E1LH8	  t\ttH [^_]A\A]A^HH@  MILHH$   L$   H [^_]A\A]A^@ HHL   H     HAn  HIn  EMHH
  AHH`AWAVAUATUWVSHh  HLHL|uhH=  HMH
  HIHHItHPHHL     tSA   DHh  [^_]A\A]A^A_fD  Hg  A)   A      HA   HD$ l   ID$HULL$pLD$`HLh Hx  uHHL$dL5:  HUA   D$(    HL$0Lt$ Lw  H	  CHHM 
  HIHHIHPHHL     IGHU(LL$xLD$hHH@ HD$PHx  HHL$lHU0A   D$(    Lt$ HL$0LƧ  H	  HcD$lH  HDtE*  D$h9D$`t'HE(A$   HD$ A      Hk  C11
HD$HIE HD$XHD$PH HD$PD$`PT$`   HD$pD       HL$HD;T$`HD$pHD$xJHHD$pL$`   HH
  HL$XHS  HT$xAHH

  HL$PHS  EAyHD$pH A   HD$ HA      j  HL$HJHE0A(   HD$ Ll$H   LL   zLIH謢LD$Pl  |$P [  HL$H   L$   NHHM 
  MHHSHHHHI   HcT$dH
  H@L HHʋJRLxPHcT$lHH
  HHʋJRPT$PHIL$P HT$HLD$HHP(L.y  II$ڕ  LD$Hu=LHtHL  HD$xHHA   IOLx  LD$HA%   A      HHl$ A   h  HL$HHL  HE(A   HD$ fD  AWAVAUATUWVSH  HLAML艏uhH  IL$H
  HIHHD$8HHtHPHHL     tJA   DHĨ  [^_]A\A]A^A_H  A)   A(      HA   HD$ g  HEIT$L$   L$   HLh Hx  Au$   Di  IU Hl$HD|$`HD$@IHTf     HH
  H$   HHH$   HHHI_P    L]n  %  $   P$   Hl$HHT$@D|$`H艟Ad  HT$@I$HBH+HHD$PH
  H$   HIH$   H.HHH$   kHHD$`;  HT$`HMHBH(DhHD$hOv     s  HD$HID$AH  AFH$  D$   HHID(H$   H$   HD$pH$   HD$xHH
  HL$8HH膒  @  HL$HHw     H$  IHHI  HPHHL       IGHWHLL$pLD$xHh Hx  Y  D$P;$   U  H   P(1I1L8ID$IHD$hLHHu  HD$@Lu H(R    HH
  H$   HHH$   HLHIM     HDEL$   P$      LHIOHT$`qt  H;$   yD$   H$  HT$`H&  HL$@f  HL$HJr  HH$      @ Ll$ A   HA(      2d  HL$@xf   Ll$ A    Ld$ A   A(      HH$  c  HL$@A   *f  HL$Hq  H$   HHH$     vID$A   A(      HA   HD$ c  HL$@e  >I$A%   A(      HHD$ c  HL$@e  HH$     H$  9HA$   A(   H@   HH$  HD$ fc  Hl$ A"   A(      HH$  b  HD$8A*   A(      HH$  HD$ b  H$   HA   耏f.     AWAVAUATUWVSH  HLHMLA   L%J  H=  HI$
  
   Hu
HA   I$HK
  LIH4HHtHPI$HL     tPA   DHĸ  [^_]A\A]A^A_ H  A)   A      HA   HD$ a   HGHSL$   L$   HHp I$x  I$HS0L$   L$   Hx  VI$H$   L=  HSD$(    A   HL$0L|$ L  H	  E)  Hc$   H  HHD@EtHu"HCA+   A      HHD$ `  I$H$   HS8D$(    L|$ A   HL$0LO  H	  ~Hc$   H/  HHЋPt@u"HC8A+   A      HHD$ x`  I$HK 
  LIH,HHD$HHPI$HL     HD$HHS(L$   L$   HH@H@ HD$hI$x  I$HK@
  LIHHHD$XHPI$HL     iHD$XHSHL$   L$   HH@H@ H$   I$x  A($   9$     $   9$   tlHCHHD$ A$   A      H"_  I$H$   HS8D$(    L|$ A   HL$0Ln  H	  eI$H
  H$   HIHT$xHLHHD$p耈HHD$`  HD$`HT$HL  L$   H@HPHH8Hc$   HL$PHLP@QA11rHL$PL$   1HA HD$XHA(Hc$   HIA@A0A@A412HL$PHA@HDqHLq H$   HD$hH HD$hxH$   I$H

  H$   HBF  H$   I$H

  HL$hHF  Ao    L!H$   H$   $   P$   o   L	   LHL$HHHHT$PHB^  HT$Pz k  L   HD$PLp@H$   H HD$hxH$   I$H

  H$   H@E  H$   I$H

  HL$hHE  A    LH$   H$   $   P$   o   L   LHL$XHHHT$PHB8]  HT$Pz8 B     LHt$HHT$`HN+k  HD$XHHӇ  tHD$XHT$`HHk  Ht$`HOHj  HHuHL$pLA   I$HL$x  3HC($H$   H A   A      HHD$ ;[  HL$pLA   踇I$HL$x  H$   HD$pHA%   A      HD$ wZ  I$HL$x  H$   jH$   ]HCHHD$ A   A      HZ  WHC(ff.     AWAVAUATUWVSH8  H$  HAMLHD$H%  H=  IH
  HT$PHIHT$8H
HHHD$@zHI.  LhAGD$LI]I^IM g  D$L   AGM|AD  HPHHL     uHIM Lh  INLh  L9   HHHK
  HIH5HIuHL$@HHHL$8  D$H   D$HH8  [^_]A\A]A^A_fH3  A)   A)      HHD$ X  D$H   뷐LHtHHL$8  뗐IA%   A)      HHD$ X  HHL$8  D$H   Z@ AWAVAUATUWVSHXH  LIH11H|$0H  IHHN`H  HthfD     Lx QuLx HHL  t(HLLp    LILAօu3HH  HuHLLh  1HX[^_]A\A]A^A_A$BA$   HD$,L   D$,ĐAVAUATUWVSH H5  LHHH
  HIH+HHtiH11A  HHCL`HI9u[HI9tRHHHJLp  D  HIHAօt͋E PU ~:   H [^_]A\A]A^fD  HHHh  1H [^_]A\A]A^HH   fATUWVSH   HLHLz  Hv;:tgHt$@HHIHL$@HAt~Hu	E   L%  I$HD$8H  HD$8Ht)H   [^_]A\D  {:uHH~HuH\$ A#   E1   HHD$8jU  HD$8H   [^_]A\    L%Y  HI$  I$H  I$HAHK    I$HHA  HL$@H} f.     WVSH0LHIHHHt HHH贁1H0[^_f.     H|$ A&   A      HT     H0[^_D  AWAVAUATUWVSHHHi  LHHHH
  HIHHH  H11  HƃO       ukHH
v  LgLp    HIHAՅg      B   =HD$<H   D$<'f     HH
  P HHHh  1HH[^_]A\A]A^A_ÐHH
  LoLp    HIHAԅlAEHt0HH
  Lp    HIHAԅ4HHOLp    HIHAԅHIM Hp  H	  HIHׅIE ME E1HHP1HIHHHp  IcMIcUH=  HHJLp  H׺  HIHAԅ^HIMLp  H	  HIHAԅ0IEME A   HHPHI
HHHp  IE ME@E1HHPLHIHHHp  IcE0IcU4LHMp  HЃHA  HIHAԅHIM(Hp  H	  HIHׅRIE(ME@A   HHPHI,Hf     HHp  D    LgH
  ID$LhHLp    HIHAօHHOLp    HIHAօID$I$H8HH	Lp  L7  HIHAׅ[I$MFA   HHH8LIHHAp  I9QAHI$MD$E1HHPdHIHHHp  HI9L'HDI$Lp  H	  HIHAׅtfHH
  LoLp    HIHAԅ\HHOLp    HIHAԅ/HIM Hp  H	  HIHׅ IE ME(E1HHPYHLIHHAp  IcEIcUH=  LHMp  HЃHA  HIHAԅHIMLp  H	  HIHAԅVIEME(A   HHPHI0HHHp  IcEIcULHMp  HЃHA  HIHAԅf.     Ht$ A&   A      HM          HHOLp    HIHAՅsI$AH8H;xt;HHDLp  H	  HIHAօ3I$HH9xuHMD$     UWVSH8H5  LHHH
  HIHHnHt-LHHIh  A  HH1H8[^_]H|$ A&   A      HNL     H8[^_]AWAVAUATUWVSH8IH
HHAH+HJH#  H
߹  H Hh    LHӻ   H8[^_]A\A]A^A_    DD$,6[  DD$,IEtBID$HpHH9t     HHtH9u   LsHM [  1X  HID$LxL(M9t9IE H@HpHH9t     HHH!]  H9uIM9uHwHH9u*D  HH9tHLuHX  2HX  ID$HpHH9tfHHH9u1.     WVSH HH
HDZ   u1H [^_ AHHH [^_+f.     VSHHHLLD$<Lt   HH[^    T$<IHىD$,mD$,HH[^fWVSH0HLLD$,LVt   H0[^_ÐT$,IHaH0[^_    AVAUATUWVSH@H-  HH11HE   H[ HH   Ld$(Ll$0DHD$(L   HD$0HCHD$8HE Lp    HIHAօuHHt7SMHtP~:   H@[^_]A\A]A^D  HE HHh  1H@[^_]A\A]A^HE H   ATUWVSH H-  HHLF,@   E1HE 8	  tF,    H [^_]A\    HE HL   H     H
^  AHHD  f.     VSH(HC     HLH P(H   @    HXHH   1H([^f.     AUATUWVSH(H=  H   1ILH
  HHHHtQ{ uHHK
  HHt  uHt>HHL+HKP~2HHLP0Hu1H([^_]A\A]D  L+M$   H        UWVSH8H51  HH11H  H   HHu
MHHtEHLCHHp  tB   .HD$,H   D$,    HHHh  1H8[^_]fD  SH01Hz  Ht5R,u.MC,   E1HD$     HHC,    H0[@ SH0A Hz  HLt0R,u)ML  C,   HD$     H1C,    H0[@ SH0AHz  HLt,R,u%LL$ ML  C,   HC,    H0[     AUATUWVSH8A Hz  HHLLtB,tHH8[^_]A\A]fD  ML  B,   HD$     qHHF,       L%7  Lh HHVL  I$   ukHFIM H@ H)  uIM 6     IHHB  HHC  I$HP0P~b1HH8[^_]A\A]I$H@  HI   B  HHdC  fL  H   |B  HHAC  I$H1   돐f.     SH0Hz  Ht"B,   HD$     E1E16C,    H0[f     AWAVAUATUWVSHHH    HHED  H5  H
  H  EIĺH  H
    IH  A$IHA H@  H   IŃ HD$8HD$ HD$4HHD$(u   D  HH   H11  HHLCHHh  uHMHHp  uHMHHp  uHMHHp  uHLL$ HLD$(Hx     PZHH   If     A$PA$   APA~qAPA~UHLHh  AE PAU 
HL   HH[^_]A\A]A^A_ H
D    IeHL   HL   ~HL   _HA   LD$8T$4H0	  WVSH H  HqH+qHH P(LcH1Hm      H{HCHCHH [^_ff.     VSH(LHHLSHLHI9t'LF     Hc
IHIIII9uH([^@ f.     UWVSH(HHtpHYHIH-  HHqH+qH9u4HH9t'HHtPHE H   H9uHOV$  HE HH@0H([^_]HH([^_]    Ht~Yf     AUATUWVSH(HALjHqLbHhHXH9uED  HH9t7H;LHH(  HHVHMuH([^_]A\A]f   H([^_]A\A]@ f.     VSH(H9HHt0HRHI#  uH([^    HHH([^/   H([^ UWVSH(Hi1HqH]H;]t!@ HHHVHK軷H9]uH([^_]     UWVSH(1HH;ZHHt/    HOHcHHWHIHH`H9]u؉H([^_] f.     ATUWVSH HH;ZHHMI1uOHHH9]tAHOHcLcMD$HWHIHOHyuH [^_]A\f.        H [^_]A\AUATUWVSH89HHMM~H  H
  H P H~HH&  H;GHtgH1MM藳H   H+_HHcHHHVH
HtDEHAD	~@H    H8[^_]A\A]     I   Lp;  1H8[^_]A\A]ÐHI  HD$(H   HHD$(HVM   L;  1f     VSH(HYHH%  H;CtH+CHVHH([^f1ff.     ATUWVSH 1H9HILtGH-  HHKHHNI9t#HHtDA@AHE    H [^_]A\fVSH(HHHNHHFLCHHPH+P`HH([^fD  AVAUATUWVSH HAMaMIHyHLHHhMiMq#  tM)H)H)HH7O,   H [^_]A\A]A^fD  WVSH HYHHKH+KH  HHSLCIHHStHH [^_@ H1v@ VSH(HHIHKHHCLFHHPH+POHH([^D  VSH(LHH\LKHI9t+LVL@     Hc
HIIIHI9uH([^@ f.     AUATUWVSH(BLaHqLHA@@H  H P(Lc1HGHf  I\$I;\$LELOtL   APHHIyIh HDlHV苵I9\$ADIIuɃH([^_]A\A]ø   fWVSH0HHIHLD$(U*  LD$(XHH"H0[^_     AWAVAUATUWVSH8LaLqIpIyLj{H$   I\$I;\$~   L;ILIInHL3HcILLH Lj H|$ HHMHLcMA  I9\$I@IuH+$   A }H8[^_]A\A]A^A_H$   MHfATUWVSH IHZ{HIHLHL)  HcIIHLHZ}B HHHHH)UHݵH [^_]A\UWVSHhH|$@Hl$ HHD$@    HD$     IHHD$H    HD$P    HD$X    D$@   HD$(    HD$0    D$    HH  H P(IIHHHH HHh[^_]Ðf.     UWVSHhHH|$@HIHD$@    HD$     Hl$ HHD$H    HD$P    HD$X    D$@   HD$(    HD$0    D$    '  HY  IHHH HX(HIIHHHkH HHh[^_]ÐAWAVAUATUWVSH  zVUUUIL$p  A)RA)   H$p  Ll$ A      LHX4    H   $@ HHu  H;E   H~dLCHML螪HIuHA     DH  [^_]A\A]A^A_ÐH)  L
     A   H P  HdHuHHhH$p  Lp  HHE IH~dMMLH萫HuHA   `     ML   3  LLe4  L轤"HHHf  HH Hh  (  LH    AUATUWVSH(Ht0H%     L
     H P  H([^_]A\A]H=  IhM`L  HH   t   H([^_]A\A]fHLv  LH   uIT$ HHM L  Hh  1ɅAHHc@ f.     AWAVAUATUWVSH(Ht8HA  A   L
     H P  DH([^_]A\A]A^A_fH5	  IXL  HH   AtA    HC 11ALpH  I^I;^IuvfD  HI;^tfHHDH	  HHLIHp  tˋE PU ~AAE PAU nHL   ]     HLHh  HH   @ f.     UWVSH(Ht0H  L
w        H P  H([^_]@ H-  IxL~  HE H   t   H([^_]HW HE HRHh  HJH+JH  HHՉH([^_] AWAVAUATUWVSH(Ht8HA  A   L
     H P  DH([^_]A\A]A^A_fH-	  IXL  HE H   At
A   fHC 11LpHxHE   I^I9^Iu/   @ HE HLGLHp  u6HI;^tZHE HLp  H	  LIHAׅtA$PA$`HE L   N     HE LHh  f.     AWAVAUATUWVSH8HLt5H     L
r     H P  H8[^_]A\A]A^A_ÐH=  IhLn  HH   t       Lm HHNMe
  HHL  I;D$H  H xt&Ht$ A   A@      H(0  fD  HI+D$IULݼ  HHHL4HL   \M~ IEIOHPH+PHAH+AHHL  H%IT$IILItIGMLHPL@tuGH3  A   A@      Hپ   HD$ V/  L~Ht$ A   MD$HUMLtHLHh  !  HH>fAWAVAUATUWVSHHHAL  H  IhL  HH   A   Le AID$HD$8  H11Ln  HAFLtDHIIM
  HL$8HH]  x9IT$LHHLHAp  u8M9uHHHh  J@ Ht$ A   A      H.  E PU HH        A   DHH[^_]A\A]A^A_fD  Hɻ  L
     A   H P  HHN
  HL$8HH  xIT$HH,6Ht$ A   A      HA   G-  hfAWAVAUATUWVSHX  H$  AL  H-  IXH$  L̹  HE H   AtA   DHX  [^_]A\A]A^A_HGL{ HD$@AFIwÉD$8  HHD$0I   AFH_ Lt()fL@MHL^  HL9   HE H
  HHH  H;FuL`H|$ A   H$  A@      A   ,   fH	  L
ڛ     A   H P  f     L  IWHD$HHBH+BH+D$8H  HHD$P
HD$8HFH;FLptlDd$\MLt$0     IH9~tFI\$LMl$LHN  yLL$8MLL'uHE HH
5  P Dd$\H\$`AD      H)  HE HL$@
  HT$0HHtH|$PHH  H;G   LD$HHL$8IHuNHL$8Ht$HP~oH$  HA   p)  LH|$ A   >HE HL$8Hh    H$  HZI   HL(  vHE H        AWAVAUATUWVSH8  IAM   H5  IXL  HH      EMA   HK Hl$@DL$<bAA      HI'  DL$<E   AI_AIOl/(     HL9   HH
  LCIHL%uHL   (  LI       H8  [^_]A\A]A^A_D  H  L
        H P  H  A	   AA      L   HD$ (  떐HLHh  N  LHvAWAVAUATUWVSHHH$   AMI  H5b  IXH$   L  HH   A   AT$.  HK T$0mT$0HxHD$8HoD  AI]AIKD%(HD$06f     HL՗LIH  H9.  H9\$0   HHH
  IHHK
  LHI5  H9IuH$   Ld$ A   A-      J'  HL$8A   j     A   DHH[^_]A\A]A^A_fD  H  L
     A   H P  H  H$   A	   A-      A   HD$ &  f.     HHL$8Hh  \  H$   H`f.     H$   Ll$ A   A-      ]&       ATUWVSH Ht.HG  L
ŕ        H P  H [^_]A\H=  IhLβ  HH   t   H [^_]A\f.     Le IL$蒭   HHbE1LH贮HHHh  8HH׉H [^_]A\ f.     AWAVAUATUWVSHHIΉM   H5X  IxL
  HH   t   HH[^_]A\A]A^A_ HG SM}щT$4HD$8L`&  T$4Hǅ   CMl      H+  M9   HIIO
  LHH
  yA   A*      LH\$    w$  H&  Df.     Hi  L
"        H P  f.     HL
  HL$8IHHIZ&  HLHh    LHf.     ATUWVSH Ht.Hױ  L
U        H P  H [^_]A\H=  IhL^  HH   t   H [^_]A\f.     HE HH  HHH  IHHP0LLHAh  H [^_]A\fD  AWAVAUATUWVSH(  H͉L~пVUUU)R9tIH  A   L
  Iغ   HH P  DH(  [^_]A\A]A^A_f.     H  M`LN  H L   A     Md$ HIL$)
  LLd$0HILHA      HD$(   HGHHxH+xHyLCHMH=HI   HL  I;Et`LCMLH螗HtdHHGH H  HL$(H Hh  H  HHA   L   L  LHE   HL$(A   uHI  LHH h  tD  AWAVAUATUWVSHHIΉM   H5  IxL  HH   A   ك;  HG M|$HD$8HxD"  HōCMd     H'  M9   HIIO
  HHHn	  yA   A      LH\$ A   F   H"  A   DHH[^_]A\A]A^A_fH)  L
     A   H P  HWH+WHH8'  HH-"  HHb  HL$8IHHH"  HHHh  e  LHb     HHLh  Fff.     ATUWVSH  HLt8Ha     L
O     H P  H  [^_]A\    Hl$ A	         H  HWIH  HtHWLGIHHI  u1Hլ  LH Hh  s  HH׉H  [^_]A\L   H  [^_]A\@ f.     ATUWVSH   HˉL~yH_  IhL  H H   ¸   tH   [^_]A\Le Hl$0A      H  LODFHl$ HL  H   [^_]A\H  L
     H P     H   [^_]A\@ f.     WVSHPAHDL   H  HL$LIQL  A   H HL$0H
  D$(    HL$ H	     tHP[^_ÐHcD$LL
4  IHHATHP[^_    H	  L
E  I   HH P     HP[^_ÐUWVSH(H=  Hc,x   HP(LcH1HI  HCxHK 1HCHCHHCH  HH([^_]VSH(HYH;YHtHH葋H9^uHD  HN Hx  HHH@0H([^H@ Ht~ff.     ATUWVSH HHHIH]HFH+FHH)HHH9t
1H [^_]A\H9Lf w2ED  HcPHFHH9FtHHctHH9]vH;LHV`Huf   H [^_]A\SH HALcJH;YrHC  H
\  H P HH [ f.     AVAUATUWVSH0H9QH=  HHL   Lf LD$,HU LVhIŋD$,toL3LIV`HI   HH+FIT$HH9tHAH
  P LHcЉHLp  HH+IuH0[^_]A\A]A^HLp  HFH0[^_]A\A]A^H+QHH
  HP :fHIH
Í  P Yf.     WVSH0HAHqHHIHHH)H)HHHH9}@HHK LD$,ShT$,tHSHJHKH:HH+SHHPHH0[^_H  H
  H P      AVAUATUWVSH L-  HHJH+
HHLcHHL5D  I9u=HHшHHI9t!HcHGH4H;wHrILP fHH [^_]A\A]A^@ f.     UWVSH(HAHHLLH9   H9   H)HFHHcHH9Fr^tJGHlH9t;HHKHHH`H;FuH袇1H([^_]       H([^_]fH9  H
ʌ  H P f.     H  HH+UH
[  H HP FH  H+QH
:  H HP HE    VSH(HHIH+KHHSLCHIHHH([^fD  VSH(HHIH+KHHSLCHIH~HH([^@ SH HHI S`HtHcPHCHH [f.     HCH [fD  H(HHI P`Ht	H@H(Hf.     WVSH HHIHH+KHHJH+JHHHSLCIHHt/H_H;_u-HH9_t#HHH^H;FuH1HH [^_ AVAUATUWVSH IHIHI+L$HHJH+JHH9MFI\$I;\$ItjLn HI9\$tXH;LHV`HtHcPHFH,H9ntHM H薆tHM YLHI;FuLE1-LH [^_]A\A]A^Ðf.     AWAVAUATUWVSH8I@I+ HHILrH+KILL$   HHA)LH+BH]HSLCHII7MD$M+D$   HIlI\$H$   HD$(H0I9Ht^HHD  HFHHL9t8FHktMIHLHt)D~HAHL9uLH8[^_]A\A]A^A_HCH$      L d  LE1HL$(  H$   H     @ f.     AWAVAUATUWVSH8I@I+ IHJH+JIL$   L$   HHIL$I+L$)HMD$M+D$1H$   HHD$ IMl$I\$HHD$(H$   H2I9H  1L$   HAHLIHFHHI9t<VL{tMILHHI   DvHAHI9uDL$   MFM+F   H$   ISMfI^IH$   Iu I9L(tjLl$(HAHHl$ HFHHI9t<FL{tIILLH   DnHAHI9uHD$ H8[^_]A\A]A^A_@ HCH$      L 7  HL$ HL$(S  H$   H     HD$     f     HCH$      Ll$(L   HL$ XL   H$   H     HD$     I1ff.     AWAVAUATUWVSH(HHIIH+MHt  H]IHEH9}   M} 1E1D  AHLHAU`HItL@1E9DLHA~c  HEH9uHUHcH)HH9tLH([^_]A\A]A^A_H1LE1  D  f.     AWAVAUATUWVSH(HYH;YHHM   Lb E1fL;LIV`HtHHcPHFH<H9~t6HL"t'MtH+~HH+ULIHI;AHH9]uDH([^_]A\A]A^A_E1@ AUATUWVSH(HrHHLLiM` H9t.fD  HcLID HW`HHtHPHH9uH([^_]A\A]Ðf.     ATUWVSH H  HY1H+YHIH HZP(LcHF1H;  I\$I;\$Hvs0H;HHHʈHHl(H I9\$lwЉH [^_]A\D  f.     AVAUATUWVSH HyH;yIIIhHs[L7IHHL膈HILHH Hs HHH I9}H^wHCL)H [^_]A\A]A^Àz tHHWVSH@HR  Ht$ HHD$     HD$0    HD$8    H HHD$(    D$    HX(HIHHH	HH 諊HH@[^_SH HHI OHCH:  CH [f.     SH HI HHC  HI  HCH [    SH HHI HC     HC    H [f     SH HD  HH   HqI  HX HPH@    @    H [ÐUWVSH8H  HHLLL$(LD$$H x     D$$   HƋD$$~ZHD$(/HHH;FtOD$$PHD$(T$$HHD$(~$L@HIH谀HHuH1HH8[^_]fLH   P
  HH  Hm{1HH8[^_]Iغ	   H}
  HH1
  @ f.     AVAUATUWVSH0DHDILN
  H~lFL%  LlfD  H  L9tFI$HHK
  HHHqyA   E11LHt$ U  H1
  HH0[^_]A\A]A^f     VSH8H3  HHLLL$(LD$$HH x  1uLL$(DD$$HHH8[^AVAUATUWVSH@L%ۘ  HLILLD$4LL$8I$HHx     IFHcL$4LhH@L)HHH9   HI^u.   f     IV HHCHBH+BIFH9tuHD$8IIHHPL HT$8IUHtGHHtDBDI$HD$(   HD$(닺   HI[  HH     H@[^_]A\A]A^f.     AWAVAUATUWVSH8H=  HLHMLD$$LL$(HHHx  L  D$$D  HHT$$IHD$(l   fL@IHHN   HFMMHHPLp~HttL+vIUJ2 D$$PHD$(T$$HHD$(~kHH
  IHD$(HLLxhH;CnM   L  LHs  L1H8[^_]A\A]A^A_     HH=D    HxLh AE H@    @    fM   fD  1I	   L  LH  1 f.     AUATUWVSH8H  IMHLLD$$LL$(HLLx     T$$   HD$(%ZfD  D$$PHD$(T$$HHD$(~7HH
  HT$(IHLBH@uHL   ,  H8[^_]A\A]ý   H8[^_]A\A]f     MH	        HL  H8[^_]A\A]ÐAWAVAUATUWVSH8H5ٔ     HMHMLD$$LL$(HLHx  uLD$$M	   u)HGLcHPH+PHI9t.M   L  LH?  H8[^_]A\A]A^A_1U  T$$HHD$(E   HF	     DA   Hm	  D$$PHD$(T$$HHD$(~PHH
  HOHIAyM   L  Hٻ     LH  BH  MMHH!M   L  @ f.     WVSH0HLD$(LLL$pLD$(HHHAu=HVHtHRHt
D$(HҋD$(H=GA  H^ H~H0[^_@ H   H0[^_Ðf.     WVSH0  H  HHLL$8LD$4H x  t   H0  [^_fH|$@E1   Hk  |$4t$HI     HH9      HD$8IHH-HtHT$8IHLBH|$ HH0  [^_ÐAWAVAUATUWVSHXA IHDL     HD$DE1HՑ  L|$HHD$8   fD  D$D0  HT$HHH

  IMHIHcL$DHT$HHT)  IMHE1LHHD$    0  Ht3AHD9tfHMLD$8HU Hx  ffD     HX[^_]A\A]A^A_f.     HALaH  LpHhI9u0@ HHh    HH1fHI9tHU HIML$E1D$    HH	  Huc H$   LE 	      H$   H:  5D  H$   M   K   H$   H  1H  8Ht
DAA    ff.     WVSH HLtAH5ڏ  HYQHH  HAHHH  H [^_H    H [^_     UWVSH(MHˉtOH5w  LH
  HHt4H{HH  HAHHH  H([^_]HH([^_]fD  AUATUWVSHHHHHtHu4H   H=  HHKH  HH[^_]A\A]H    H=Ɏ  HjH  KHH/  L   L,H@  Hxu  IHHD$(    L^u  HT$ LAԋSH  KLЋHن  LHw  HHD$0    Hl$(HHHT$ Hu  0  +HH[^_]A\A]fUWVSH  H=  Ht$0DL$(H͉T$ DD$$H\$ HH  HAH$`  H  HHH  [^_]D  VSH  DD$$H\$ L$P  HΉT$ DHD$(    HH@H  [^D  Hǅ  HfQHV  HÐHH+f     +Ðf.     H8LL$XLL$XLL$((+  H8 ATUWVSH H-ߌ  Lcӹ   HE P(HHE B    P(JHH9HNHNs     HH9wHH [^_]A\    UWVSH(H=q  Hc   HP(HH    P(HHCHHCHH([^_]     UWVSH(H-!  HYH+HϹ   HE P(HHE P(HHLcHHHcHHVHVHh*  HH([^_]ff.     VSH(HHt#H5  H	HP0HHH@0H([^HH([^     WVSH HH	HcHCH)HH9}1HsHU  H)H HHcP8HHHHSHCH [^_    HHIH9s@ HH9r HHIH9s@ HH9rVSH(HcHHHHH;ArH  H
v  H P HH H([^ÐVSH8HcHHHHH;Ar$H}  DD$,H
i  H P HHDD$,D H8[^fD  SH HH;AHt H [    H)  H
R  H P H H [ÐSH HAH9Ht@H [fD  H  H
  H P HC@H [D  f.     VSH(HAHQHH9rH+HTHSHBHC2H([^f.     SH HAH9HtHHCH [ÐHI  H
  H P HCHHCH [f.     AUATUWVSH(ELAHHDI9rHE HHMH)H)HLHH)HHD9   HcHMcIL9r_MELEtAT$HTD  HsH9uHH([^_]A\A]@ Hy  H+H
  H HP LE]J+I)H&  LEHE  HHcHE HLEH8YWVSH LLAHHI9rGL9v"H  HH+H
  H HP LFI)HHZ&  H)H)^HH [^_H  H+H
  H HP LFD  f.     WVSH0HHcHHIHLL9vCIcHH0H9rHX  DH
  H P HH0HDD 
H0[^_H%  DD$,H
  H P HHODD$,D  H(HLAL9rHL9s;u1H(f.        H(ÐUWVSH(HAH+HωщֺH)HHWL
lA   HHH)H$  LLO1M9t3fD  A9}f     HSA9IQM9u9~HS9uHH([^_]D  AUATUWVSH(Hˉ1LkH3II9t^L H-߅  H=~  fD  H   I9t6HcHHII;L$rHE HHP M$I9I   uLH([^_]A\A]Ðf.     HHQL
BA   H)HH#   f.     LHAI9sA;uA9tII9rL     LHII9s!A;t&L	D  9tHH9rL)H1ÐVSH(HHHH9wH;QrH  H
~  H P HH)HHH([^@ f.     ATUWVSH HyHjH˹   HHIH9t9 HHNH9rHH9s;uHH9tLcHH9uLH [^_]A\ff.     UWVSH8HyHjH˹   H`HH9t6     HLFL9rHL9s;uIL9tHH9uH8[^_]HHD$(HD$(D  H(LLA1H
LJM)II)McIM9uIHL!  H(f.     SLYHLBI9tVL
HE1fD  M9ȋvA;t4L;t+HI9w1HAI9uMcI)1IM9[ø   E1@ LRL
LAM9t*H	I9Av;t!H;tHI9wIM9u1    f.     ATUWVSH HAHl$pIHLLH9rH9wYH9osH:  HH+H
|  H HP H)E1HHHALcHHI   H [^_]A\H  HI+$H
|  H HP HƁ  H+H
{  H HP ID$\     AVAUATUWVSH0L5k  L-d  L%1|  IL+	HHLIH;QtS@ LHH)MAyHD$ 1HAHH9uuL9tHH   H0[^_]A\A]A^H       UWVSH(H-  Hcٹ   HE P(HHE     HP(IHFH1H9  HHH^H([^_]fD  UWVSH(H-  HYH+HϹ   HE P(HHE P(HHLcHHHcHHVHVH  HH([^_]ff.     VSH(HHt#H5  H	HP0HHH@0H([^HH([^     WVSH HH	HcHCH)HH9}1HsH  H)H HHcP8HHHHSHCH [^_    HH;As    HHH9AwÐVSH(HcHHHHH;ArHM  H
z  H P HHH H([^VSH8HcHHHHH;Ar$H
  LD$(H
y  H P HHLD$(L H8[^fD  SH HH;AHtH H [fD  H~  H
y  H P HH H [SH HAH9HtH@H [D  Hy~  H
y  H P HCH@H [@ f.     VSH(HAHHQHH9rH+HT.HSHBHCH2H([^     SH HAH9HtHHCH [ÐH}  H
2y  H P HCHHCH [f.     AUATUWVSH(ELAHHLI9|   HE HMLHH)H)H)HHHDH9   HcHMcIL9r\MELEtAD$HTHHsH9uHHE H([^_]A\A]H}  H+H
x  H HP LE`J+I)Hd  LEfD  HcHHHLEH] ^WVSH LLAHHI9rGL9v"H|  HH+H
Nx  H HP LFI)HH  H)H)^HH [^_HI|  H+H
x  H HP LFD  f.     H(HLAL9rHL9sH;u1H(f        H(ÐHHQL
A   H)HH5   f.     LHAI9sI;uI9tII9rL     H(LLA1H
LJM)II)McIM9uIHL  H(f.     SLYHLBI9tXL
HE1fD  M9HvI;t5LH;t+HI9w1HAI9uMcI)1IM9[ø   E1fATUWVSH HAHl$pIHLLH9rH9wYH9osHz  HH+H
v  H HP H)E1HHHALcHHI  H [^_]A\H7z  HI+$H
1v  H HP Hz  H+H
v  H HP ID$\     AVAUATUWVSH0L5  L-  L%Qv  IL+	HHLIH;QtT@ HLHH)MAyHD$ HAHH9uuL9tHHR   H0[^_]A\A]A^H9  ސUWVSHHHYHHDHD$8    H   ;Ϻ   HD$8E1IHu  HD$ SH  ttDE   AHE10H	AuA   D8HHt   D:   HDHEu0	   HT$8Hn  HRHt}H
H
S  HJHRH
<  H-  HH[^_]@ Ht  HF    H1HH[^_] HD$     A   IHt  HSHv1@ H      H      H      HH[^_]fD  HD$     A   IHht  HS1MAUATUWVSHhH=}w  HHT$XHEEHD$X    HE1HT$ IHit  PHHt3HD$XH   PD9u-HD$XPD9   HD$XH  HHh[^_]A\A]L%s  HH  HH
/t  H=,t  H\$0L
s  Ls  Hs  1HL$8HD$H    Ld$@HH|$(Hl$ @  HHh[^_]A\A]f.     L%zs  {@ L%9s  kSH0HH
A    HHD$ tv     H
     H
  HD$   HT$ LD$(HHD$(  HL$ H  HL$(H       H    HH0[Ht  HHHH0[ H(GHH(ÐH(H$  H HtH#  HPH@H#  HuH(Ðf.     VSH(H
#t  Ht9t HH)Ht@ HH9uH
~H([^S 1D@J< Luf.       t@      qATUWVSH@H_#  H2-+  HD$     H9tHHM#  H@[^_]A\fHL$ Q  H|$   A  =  HL$0H  H3|$0DH  H1H1ǉH1H!H9t%HHH"  H"  H@[^_]A\fD  H] fH3-+  f.     UVSHHpHH
    H  E1HUH  HI   HEHD$8    IHU1HD$0HEHD$(H  HD$   H~  1H5  S  	 M     HR  H!  HEH!  HEJ  H
p  ]    	 H9    HEH
  HEH  xH(tt   H(f     {     H(ÐVSH(Hsq  8t    tt@   H([^fH  H5  H9tHHtHH9u   H([^	      1ÐHff.     HÐVSH8HD$XHHT$XLD$`LL$hHD$(  H
ro  LH`A      v  Ht$(  HH`HI  x  D  f.     AUATUWVSHXH΋
O  HLŅ   H>  1HHH9rL@E@LH9   H9uH  HI   HcH[HIL-  IEAE     
  AT$A0   HHT$ IEH  HL:  HttD$DPt,t$H  A@   HT$8HL$ I  t(f  IHHHX[^_]A\A]  18?  H
|n  ]H.  AT$H
*n  LD@H
m  H1UAWAVAUATWVSHhH$   5  tHe[^_A\A]A^A_]     N  HH@H   H
  L%n  Hn  H)      HD$ H  LH)H~H~.DEB  DSE5  DKEu#H     DE  K	  S  HL92L-in  LuI    KSLLL 5      n  @Q  H9LH)ILLEA   JHL9r  1ۅL%  LuL-
  H;5s  HHg  tHHA0   HAH  HA  MHUHMDAD  L99L-pm  H}KA   HHLEL9rCugLII IHH)ILLEA   PLII  fIHH)ILLEA   H
l  HE    LIM	IHH)ILLEA   H6  H
7k  HCLCPGH
k  ;H(=  wc=  s{=  
     =     =     1ҹ   i  H1  H     1H(     =     wC=     =  u1ҹ     HHttHҸ   t   1H(@ =  te=  u@1ҹ   
  H   Hty   1lf     =  t=  t   H(ú      
  	  1H(1ҹ   q
  HHX      U
  1   H(ú      5
  1      
  1     ATUWVSH   Hŋp  u%Ht H
j  V       Ht   H [^_]A\@ HI  HB  0   L%1HH    HI)HH.	DgHPHSPHH)HCSH t2Hu  HuHsD  IH
    W    @ SH HHˋ CCG    =     =     =     w?=     =  u?1ҹ     H4  Ht$   иH [=     =  tuH  HtHH [H@ =     wX=  tF=  u1ҹ   I  H   Ht   иH [     BH [=  t=  n1ҹ     HtJHS   иH [1ҹ     Hw                @       q        X    8UWVSH(H
a  _  H4  Ht3H-  H=e  HׅuHt	HCHH[HuH
  H([^_]H%Z  fD  UWVSH(  1HׅuH([^_]ú        HHt=(HxH
    H  H
  H}  HC  H([^_]þ뚐SH ]  ˅u1H [    H
Y  W  H,  Ht9uK9t)HHHHuH
&  t  1H [     HQHP  H
  J  HPHHƿ  @ SH    r0tu  tG
D       H [Ë     t  uH
`  HtHY  HHuH
c  H8      6      D  fD  "  t        H [ÐH
  _  ېtHcA<H19PE  u	fyff9MZt	1fD  @ f.     HcA<HAHDIt)HLL(D@I9LwHH9rH(L9u1Ðf.     WVSH H  HwiHd  1f:MZtHH [^_H8tHcB<HPH\@t*HH|(A   HH  tH(H9u1HH [^_D  H(L
Ud  1IfA9MZtHH(LtIcA<LL)LPHT@t3HLL(f.     DBL9LrBH9rH(L9u1HH(f     H(Hc  E1f:MZtDH( H(tHcB<DDDH(fH(L
c  1IfA9MZtHH(LtIcA<LPHT@t$HHD(B' t	MtIH(H9u1HH(@ f.     H(Lc  1fA8MZtHH( LhIEHH(f.     H(Lb  1HfA8MZtH( L(tIc@<HL)IAPA@ID t1HLL(     D@L9LrPH9rH(L9u1H(Ë@$H(D  f.     H(L%b  E1AfA;MZtLH(LtIcK<Lً   tQHTItʃHLT(D  DJL9LrJH9rH(L9uE1LH(     Luf     AHHuPtEDHMLH(ÐÐQPH=   HL$rH   H	 H-   H=   wH)H	 XYÐ   Ð   Ð   %  %  %  %  %  %  %  %v  %f  %V  %F  %6  %&  %  %  %  %  %  %  %  %  %  %  %v  %f  %V  %>  %.  %  %  %  %  %  %  %  %  %  %  %~  %n  %^  %N  %>  %.  %  %  %  %  %  %  UH]0m                                                                                                                                                                                                                                                                                                                                                                                                                                                                             m    jm     m    cm     m     m                    Tuple   Relation                      :m    :m    @;m     ;m                                  :m    :m     :m    P:m                                  :m    :m    9m    :m                            Hm    0ɐm     ɐm    Ȑm    Pːm                            hm    om    vm    {m    m    m                    m    m                    m    m    vm    m                                    m    m    m            $m            $m           $m           $m                                          @m    m    m    `m    pm                            @m                                   2-+          ] f                                                                                                                                                                                                                                                        libgcj-16.dll _Jv_RegisterClasses                               8.6 ::ral iso8859-1 relvar      ::ral::relvar   relation        ::ral::relation tuple   ::ral::tuple ::ral pkgname version copyright                            This software is copyrighted 2004 - 2014 by G. Andrew Mangogna. Terms and conditions for use are distributed with the source code. 0.12.2 ral                   booleanHash: cannot convert, "%s" intHash: cannot convert, "%s" longHash: cannot convert, "%s"  doubleHash: cannot convert, "%s"        wideIntHash: cannot convert, "%s" bad boolean value, "%s"       Ral_AttributeDelete: unknown attribute type: %d Ral_AttributeDup: unknown attribute type: %d    Ral_AttributeRename: unknown attribute type: %d Ral_AttributeTypeEqual: unknown attribute type: %d      Ral_AttributeValueEqual: unknown attribute type: %d     Ral_AttributeValueCompare: unknown attribute type: %d   Ral_AttributeValueObj: unknown attribute type: %d       Ral_AttributeConvertValueToType: unknown attribute type: %d     Ral_AttributeHashValue: unknown attribute type: %d      Ral_AttributeScanType: unknown attribute type: %d       Ral_AttributeScanValue: unknown attribute type: %d      Ral_AttributeConvertValue: unknown attribute type: %d   Ral_AttributeTypeScanFlagsFree: unknown flags type: %d  Ral_AttributeValueScanFlagsFree: unknown flags type: %d bignum boolean bytearray dict double int list long string wideInt                       Hm    m    Pm    m     m    Om    P#m    m     m    m    Wm    m    "m    "m    m    am     !m    p"m    "m    !m    fm     m     m    m    m    mm    `m    m    m    @m    qm     m    $m     $m    !m    vm    m    m    0m    m    {m    @!m     "m    !m    P!m    m    m    m    m    m                    4m    4m     !=     Ral_RelationUpdate: attempt to update non-existant tuple while ungrouping relation int while unwrapping tuple while computing quotient  Ral_RelationErase: first iterator out of bounds Ral_RelationErase: last iterator out of bounds  Ral_RelationErase: first iterator greater than last             relation attrName ?-ascending | -descending sort-attr-list? ?-within attr-list -start tag-value? tag option     Ral_TagCmd: unknown option, "%d"        relation1 relation2 ?-using joinAttrs?  relation1 relation2 ?-using joinAttrs relation3 ... ? relationValue     relationValue newAttr ?attr attr2 ...? relation1 relation2 relation1 compareop relation2 compareop      relationValue tupleVarName expr script  
    ("in ::ral::relation update" body line %d) invoked "break" outside of a loop       invoked "continue" outside of a loop unknown return code %d relValue expr relValue tupleVarName expr relationValue attribute    relation1 relation2 ?relation3 ...? relation attribute  relationValue ?name-value-list ...? relation    tupleVarName relationValue ?-ascending | -descending? ?attr-list?script ordering        
    ("::ral::relation foreach" body line %d) heading ?tuple1 tuple2 ...?       heading ?value-list1 value-list2 ...?   relationValue perRelation relationVarName ?attr1 type1 expr1 ... attrN typeN exprN?     attribute / type / expression arguments must be given in triples        relationValue ?oldname newname ... ?    oldname / newname arguments must be given in pairs      relationValue ?-ascending | -descending? rankAttr newAttr int relationValue ?attr ... ? relationValue ?attrName ? ?-ascending | -descending? sortAttrList? ?    relationValue attrName ?attrName2 ...? relation keyAttr valueAttr       relation arrayName keyAttr valueAttr    relation newattribute ?attr1 attr2 ...? attempt to group all attributes in the relation during group operation list attribute type      dictValue keyattr keytype valueattr valuetype dividend divisor mediator relationValue ?attrName | attr-var-list ...? subcommand ?arg? ... subcommand            relationValue tupleVarName ?attr1 type1 expr1 ... attrN typeN exprN? equal == notequal != propersubsetof < propersupersetof > subsetof <= supersetof >=         m    pam    m    pam    m    am    m    am    m    0bm    m    0bm    m    bm    m    bm    m    am    m    am    m    bm    m    bm                    -ascending -descending -within -start           m            m           m           m                           array assign attributes body cardinality compose create degree dict divide dunion eliminate emptyof extend extract foreach fromdict fromlist group heading insert intersect is isempty isnotempty issametype join list minus project rank rename restrict restrictwith semijoin semiminus summarize summarizeby table tag tclose times tuple uinsert ungroup union unwrap update wrap                           Pm     m    Vm    Ɛm    ]m    vm    hm     Ɛm    mm    wm    ym    sm    m    @m    m    xm    m    Pm    m    Đm    m    m    m    @m    m    ~m    m    m    m    @m    m    0m    m    Đm    m     m    m    m    m     m    m    `m    m    m    m    @m    m    0ym    m    ym    m    Pm    m    `qm    "m    pm    'm    `m    -m    @m    5m    m    :m    0m    Am    m    Jm    `m    Wm    nm    `m    @lm    jm    m    tm    Лm    m    Pm    m    @hm    m     m    m    m    m    m    m    @m    m    @m    m    m    m    m    m    m    m    zm                    m            m                                           -using                          (  <== [ ]   (Complete)  [ ] ==>  )     relvarConstraintCleanup: unknown constraint type, %d , in relvar  
 tuple    for association  ] ==> [ for partition   is partitioned [ ])  |  for correlation   Ral_ConstraintDelete: unknown constraint type, %d       is referenced by multiple tuples        is not referenced by any tuple references multiple tuples references no tuple   is referred to by multiple tuples       is not referred to by any tuple correlation      does not form a complete correlation unknown constraint type, %d 1 + ? *       rm    tm    vm    xm    ?pattern?       relvarName ?name-value-list ...? relvarName relvarName relationValue    relvarName tupleVarName idValueList script      relvarName tupleVarName expr script ?relvar1 relvar2 ...?       relvarName ?relationValue ...? transaction option       Unknown transaction option, %d arg ?arg ...?    
    ("in ::ral::relvar eval" body line %d) option type ?arg arg ...? trace option trace type   add variable relvarName ops cmdPrefix   remove variable relvarName ops cmdPrefix info variable relvarName       suspend variable relvarName script Unknown trace option, %d add transaction cmdPrefix remove transaction cmdPrefix info transaction     suspending eval traces not implemented Unknown trace type, %d relvar ?relationValue?    relvarValue attr value ?attr2 value 2 ...?      attribute / value arguments must be given in pairs      name relvarName1 ?relvarName2 ...? script       name super super-attrs sub1 sub1-attrs ?sub2 sub2-attrs sub3 sub3-attrs ...?    relvarName relvarName ?attrName1 value1 attrName2 value2 ...? relvarName tupleVarName expr      relvarName heading id1 ?id2 id3 ...?    ?-complete? name corrRelvar corr-attr-list1 relvar1 attr-list1 spec1 corr-attr-list2 relvar2 attr-list2 spec2 * delete | info | names ?args? | member <relvar> | path <name> constraint subcommand name Unknown association command type, %d    c"name refrngRelvar refrngAttrList refToSpec refToRelvar refToAttrList refrngSpec subcommand ?arg? ... subcommand delete exists info names member path variable transaction add remove suspend begin end rollback association constraint correlation create deleteone dunion eval identifiers insert intersect minus partition procedural restrictone set trace uinsert union unset update updateone updateper                            m    .m    m    @,m    m    +m    m    +m    hm    'm    m    p%m    m    Pm    m    pm    om    m    m    `m    m    m    m    "m    %m     m    {m    pm    +m      m    m     m    5m    m    @m     m    Lm    Pm    Pm    m    m    m    Vm    m    ^m    m    dm     m    jm    
m    qm    pm    {m    Pm                    ::      bad operation list: must be one or more of delete, insert, update, set or unset traceOp relvar may only be modified using "::ral::relvar" command       relvarTraceProc: trace on non-write, flags = %#x
 namespace eval   { } ", failed procedural contraint, " "      returned "continue" from procedural contraint script for constraint, "  returned "break" from procedural contraint script for constraint, "     relvar creation not allowed during a transaction unset  during identifier construction operation        
    ("in ::ral::%s %s" body line %d) association partition correlation -complete procedural    Ral_RelvarObjConstraintInfo: unknown constraint type, %d        end transaction with no beginning       
    ("in ::ral::relvar trace suspend variable" body line %d)  transaction begin end 1 + ? *                    $m    $m    $m    $m    multiplicity specification delete insert set update             $m           $m           $m           1#m           $m                           Ral_TupleUpdate: attempt to update a shared tuple               attr1 type1 value1 ... tuple1 tuple2 tupleValue tupleValue tupleAttribute while unwrapping tuple tupleValue attr ?...?  tupleValue newAttr ?attr attr2 ...?     failed to copy attribute, "%s"  tupleValue ?attr1 value1 attr2 value2?  for attribute name / attribute value arguments  tupleValue ?oldname newname ... ?       for oldname / newname arguments tupleValue ?attr? ...   tupleValue ?name type value ... ? heading name-value-list       tupleValue ?attrName | attr-var-pair ... ? subcommand ?arg? ... subcommand assign attributes create degree eliminate equal extend extract fromlist get heading project relation rename unwrap update wrap                               'm    m    'm    m    'm    m    'm    0m    'm    m    'm     m    'm     m    'm    0m    'm    0m    'm    m    'm    Pm    'm    m    'm    m    (m    m    (m    0m    (m     m    (m    m                    Ral_TupleHeadingFetch: out of bounds access at offset "%d"      Ral_TupleHeadingStore: out of bounds access at offset "%d"      Ral_TupleHeadingStore: cannot find attribute name, "%s", in hash table  Ral_TupleHeadingStore: inconsistent index, expected %u, got %u  Ral_TupleHeadingPushBack: overflow      Ral_TupleHeadingAppend: out of bounds access at source offset "%d"      Ral_TupleHeadingAppend: overflow of destination                         , " " RAL OK UNKNOWN_ATTR DUPLICATE_ATTR HEADING_ERR FORMAT_ERR BAD_VALUE BAD_TYPE BAD_KEYWORD WRONG_NUM_ATTRS BAD_PAIRS_LIST DUPLICATE_OPTION NO_IDENTIFIER IDENTIFIER_FORMAT IDENTIFIER_SUBSET IDENTITY_CONSTRAINT DUP_ATTR_IN_ID DUPLICATE_TUPLE HEADING_NOT_EQUAL DEGREE_ONE DEGREE_TWO CARDINALITY_ONE BAD_TRIPLE_LIST NOT_AN_IDENTIFIER NOT_A_RELATION NOT_A_TUPLE NOT_A_PROJECTION NOT_DISJOINT NOT_UNION TOO_MANY_ATTRS TYPE_MISMATCH SINGLE_IDENTIFIER SINGLE_ATTRIBUTE WITHIN_NOT_SUBSET BAD_RANK_TYPE DUP_NAME UNKNOWN_NAME REFATTR_MISMATCH DUP_CONSTRAINT UNKNOWN_CONSTRAINT CONSTRAINTS_PRESENT BAD_MULT BAD_TRANS_OP SUPER_NAME INCOMPLETE_SPEC ONGOING_CMD ONGOING_MODIFICATION INTERNAL_ERROR                  *+m    -+m    :+m    I+m    U+m    `+m    j+m    s+m    +m    +m    +m    +m    +m    +m    +m    +m    ,m    ,m    &,m    1,m    <,m    L,m    \,m    n,m    },m    ,m    ,m    ,m    ,m    ,m    ,m    ,m    ,m    -m    -m    -m    '-m    8-m    G-m    Z-m    n-m    w-m    -m    -m    -m    -m    -m    no error unknown attribute name duplicate attribute name bad heading format bad value format bad value type for value unknown data type bad type keyword        wrong number of attributes specified bad list of pairs duplicate command option relations of non-zero degree must have at least one identifier  identifiers must have at least one attribute    identifiers must not be subsets of other identifiers    tuple has duplicate values for an identifier    duplicate attribute name in identifier attribute set duplicate tuple headings not equal relation must have degree of one        relation must have degree of two        relation must have cardinality of one bad list of triples       attributes do not constitute an identifier      attribute must be of a Relation type    attribute must be of a Tuple type       relation is not a projection of the summarized relation divisor heading must be disjoint from the dividend heading      mediator heading must be a union of the dividend and divisor headings too many attributes specified     attributes must have the same type      only a single identifier may be specified       identifier must have only a single attribute    "-within" option attributes are not the subset of any identifier        attribute is not a valid type for rank operation duplicate relvar name unknown relvar name      mismatch between referential attributes duplicate constraint name unknown constraint name       relvar has constraints in place referred to identifiers can not have non-singular multiplicities        operation is not allowed during "eval" command  a super set relvar may not be named as one of its own sub sets  correlation spec is not available for a "-complete" correlation recursively invoking a relvar command outside of a transaction  recursive attempt to modify a relvar already being changed serious internal error                               X/m    a/m    x/m    /m    /m    /m    /m    /m    /m    0m    /0m    H0m    0m    0m    0m     1m    U1m    e1m    x1m    1m    1m    1m    2m    82m    `2m    2m    2m     3m    F3m    h3m    3m    3m    3m    84m    i4m    4m    4m    4m    4m    4m    5m    `5m    5m    5m    6m    P6m    6m    NONE array assign association body cardinality compose constraint correlation create degree delete deleteone destroy dict divide dunion eliminate emptyof equal eval extend extract foreach fromdict fromlist get group heading identifiers insert intersect is isempty isnotempty issametype join list minus names partition procedural project rank relation rename restrict restrictone restrictwith semijoin semiminus set summarize summarizeby table tag tclose times trace tuple uinsert ungroup union unset unwrap update updateone updateper wrap              88m    =8m    C8m    J8m    V8m    [8m    g8m    o8m    z8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    8m    9m    
9m    9m    9m    $9m    +9m    59m    89m    @9m    K9m    V9m    [9m    `9m    f9m    l9m    v9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    9m    :m    
:m    :m    :m     :m    &:m    ,:m    3:m    ::m    D:m    N:m    unknown command relvar setfromany updatefromobj         <m    
:m    9m    <m    <m    <m                    Ral_IntVectorFetch: out of bounds access at offset "%d" Ral_IntVectorStore: out of bounds access at offset "%d" Ral_IntVectorFront: access to empty vector      Ral_IntVectorPopBack: access to empty vector    Ral_IntVectorInsert: out of bounds access at offset "%d"        Ral_IntVectorErase: out of bounds access at offset "%d" Ral_IntVectorOffsetOf: out of bounds access     Ral_IntVectorCopy: out of bounds access at source offset "%d"   Ral_IntVectorCopy: out of bounds access at dest offset "%d" %d: %d
     Ral_PtrVectorFetch: out of bounds access at offset "%d" Ral_PtrVectorStore: out of bounds access at offset "%d" Ral_PtrVectorFront: access to empty vector      Ral_PtrVectorPopBack: access to empty vector    Ral_PtrVectorInsert: out of bounds access at offset "%d"        Ral_PtrVectorErase: out of bounds access at offset "%d" Ral_PtrVectorCopy: out of bounds access at source offset "%d"   Ral_PtrVectorCopy: out of bounds access at dest offset "%d" %d: %X
     interpreter uses an incompatible stubs mechanism Tcl            missing stub table pointer epoch number mismatch requires a later revision tcl::tommath  (requested version  Error loading  ):  , actual version                @m    `m    ёm            Mingw-w64 runtime failure:
     Address %p has no image-section   VirtualQuery failed for %d bytes at address %p          VirtualProtect failed with code 0x%x    Unknown pseudo relocation protocol version %d.
         Unknown pseudo relocation bit size %d.
               .pdata           m            m            m             m            @Im            @Im            Am              m            ̲m            m            Hm            @m            0m            8m             m            m            m             m            m            ǐm             /m            8m            @m            m            m            GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119       GCC: (GNU) 6.2.1 20161119                                                                                                                                                                                                          M   p P    p     p     ,p    &  4p 0    <p     Hp     Pp     dp     hp     lp     pp    |  |p     p     p   T  p `  ~  p     p   <  p @    p     p   ,  p 0    p     p      p      q     q     q     (q     0q     <q     Hq   O  Tq P    \q   y   lq       |q    !  q  !  >!  q @!  C!  q P!  !  q !  !  q !  !  q !  !  q !  "  q "  "  q  "  g"  q p"  r"  q "  "  q "  K#  q P#  #  q  $  $  q $  $  q  $  $   r $  %  r  %  x%  r %  %  $r %  @&  0r @&  &  8r &  '  @r '  m'  Hr p'  )  Tr  )  )  dr )  u*  pr *  
,  |r ,  .  r .  /  r /  0  r 0  +0  r 00  0  r 0  1  r 1  A2  r P2  2  r  3  3  r 3  4  r  4  4  r 4  4  s 4  4  s 4  B5  s P5  5   s 5  5  ,s 5  K6  8s P6  6  Ds  7  U7  Ps `7  7  \s 7  8  ls 8  f8  |s p8  8  s 8  8  s  9  E9  s P9  9  s 9  9  s  :  B:  s P:  }:  s :  :  s :  :  s :  :  s :  :  s :  :  s  ;  6;  s @;  ;  s ;  4<  s @<  <  t <  \=  t `=  =  $t  >  n>  8t p>  >  Ht >  ;?  Tt @?  	@  `t @  @  pt @  A  t A  B  t  B  ?C  t @C  C  t C  F  t F  LI  t PI  J  t J  L  u L  !O  ,u 0O  O  Du O  O  Tu O  P  du P  Q  tu Q  S  u S  U  u U  
V  u V  V  u V  Y  u  Z  Z  u Z  _[  v `[  [  v  \  \  (v \  _  <v  _  _  Tv _  `  dv `  ia  xv pa  a  v a  a  v a  #b  v 0b  sb  v b  b  v b  c  v  c  c  v c  d  v d  e  v  e  _f  w `f  f  w f  xg  ,w g  6h  <w @h  =l  Lw @l  n  dw n  Uq  |w `q  s  w s  }v  w v  w  w w  wx  w x  ,y  w 0y  y  w y  z  x z  ~  x ~  G  4x P  <  Dx @    Xx   W  hx `    x     x     x     x   3  x @  6  x @  V  y `     y     8y     Hy    /  `y 0  9  py @  I  y P  ͛  y Л    y     y   -  y 0     z   ;  z @  1  0z @  T  Dz `  b  Xz p  3  lz @  C  z P    z    z  z     z      z     z      {     ${     <{     D{      \{     l{     {     {    +  { 0  W  { `    {   L  { P  J  { P    {     {   |  {     |     $|      0|     D|     P|     `|     l|   I  || P    |    2  | @    |   s  |     |       }   E  } P  t  }   )  $} 0  E  4} P  q  8}     @}     H}     T}     `}     h}   I  p} P    |}     }     }    y  }     }     }   N  } P  /  } 0  y  }     }   (   ~ 0    ~   X  ~ `  7  0~ @    H~   M  T~ P    d~      t~   
  ~   B  ~ P  w  ~     ~   l  ~ p    ~     ~     ~         ]    `  C  , P  h D p 
 \ 
  t      F  P    b  p    A  P          0      8   " @ " f% X p% ' p ' +  + +  + 2,  @, .  .  / Ѐ  / / ؀  0 0  0 1  1 V2  `2 5 ( 5 
6 @ 6 6 L 6 9 ` 9 : t : o>  p> b?  p? @  @ JB ́ PB C ܁ C JE  PE G  G H   H zM 4 M S L S \ d  \ ^ | ^ _  _ _   ` Ca  Pa a Ԃ a Xi  `i i  i xk  k k   k l ,  l il 8 pl `m D `m m X m 6n h @n n t  o o  o o  o <p  @p p  p q   r 7r ă @r t ̃ t t   u Ru  `u u  u v  v v  v v $ v 8w 0 @w w @ w  x P  x 'y ` 0y dy t py y  y z   z z  z z  z +{  0{ { Ȅ { >| Ԅ @| x|  | N}  P} }  } e~  p~ ' , 0  <   ҁ T  " h 0 ݃   %  0 .  0     ؅      ؏       E 0 P  H    X   p        a  p Й  Й  ̆   4 ؆ @  ܆      %  0    R  ` i ( p  8   D  * P 0 U X `  `   l            < ȇ @    A  P    `  `  $   ,   4  / < 0 B D P  T   P h P v t  3  @ O  P Ұ   e ̈ p / ؈ 0 (  0 D   P        .   0  4   D   P    T   X   % \ 0 M ` P  h   x  t          =  @ `  `     ĉ  / Љ 0 q ؉  ƺ  к             ;  @     + ( 0 ž 8 о  L   ( P 0 o T p  X  4 d @  t     |         8  @     ̊  H ܊ P          Z  `       8   @  ,   4   H   _ T `  \   `   d  ~ l  H t P              ċ   ̋  V ԋ `    V  ` [  `               $   0  $    0    D    \   d  b t p  |  _  `                e  p      Č   ̌  B Ԍ P  ܌  !  0       P V  ` f  p u                                                                                                                   20 B0`pP  
 
20`pP   P2P  P b0`pP            b0`   b0`   b0`   B   b   b   b0`   b0`   b0`   b   b0`   b0`   b0`   b   b0`   b0`   R0`p b   b0`   b0`   R0`p 0
 
0`pP
 
0`pP 0`   b   b      b            20`p    20`p    R0`p R0`p
 
R0`pP       0`p 20`p 20`p B0`   B   B   B   B0`   0`pP   0`   20`p
 
r0`pP 0`pP   0`p 20    B   R0`p R0 20`p 20`p 20`p 0`pP        
 
20`pP B0`   20`p 20`p b0`   b0`   B0`pP   B0`pP         B0`pP      B0`pP   20 20 20 20             20 B0`   2
0	`pP
 
20`pP
 
20`pP r
0	`pP B0`pP   B0`   20`p
 
r0`pP B0`pP   20`p R
0	`pP	 b0`
p	P  
 
20`pP
  0`
p	P	 0`
p	P  	 b0`
p	P  
  0`
p	P	 0`
p	P   B0`pP   B0`pP  
 
20`pP
  0`
p	P	 0`
p	P  	 0`
p	P   B0`pP  
 
20`pP
  0`
p	P 0`pP   20`p B0`pP   B0`pP  	 0`
p	P   B0`pP   B0`pP   B0`pP   B0`   B   B0`   B0`   B0`   B0`  
 
20`pP	 B0`
p	P   R0`p	 0`
p	P  
 
20`pP 0`pP   0`pP  
 1 0`
p	P
 ) 0`
p	P
 ) 0`
p	P
 ) 0`
p	P
 ) 0`
p	P	 B0`
p	P   B0`pP   B0`pP   B0`pP   B0`pP  
 - 0`
p	P B0`pP   B0`pP  
 
0`pP	 0`
p	P  	 0`
p	P  	 0`
p	P   # 0`pP
 # 0`
p	P
 # 0`
p	P # 0`pP
 % 0`
p	P
 # 0`
p	P b0`pP  	 b0`
p	P  
 
R0`pP	 0`
p	P  	 " 
0	`pP  
 # 0`
p	P
 5 0`
p	P
 3 0`
p	P
 1 0`
p	P
 # 0`
p	P	 0`
p	P   B0`pP   B0`pP   # 0`pP	 0`
p	P  	 0`
p	P  	 b0`
p	P  	 0`
p	P  
 # 0`
p	P
 
20`pP	 0`
p	P  	 $ 
0	`pP  
 1 0`
p	P B  	 " 
0	`pP  
 
20`pP	 $ 
0	`pP   0`p 20 20 20 20 B0`pP  
 
R0`pP
 
& 0`p   2
0	`pP 2
0	`pP	 0`
p	P   B0`   B0`pP   B0`   B0`pP   20`p 0`pP   2
0	`pP R
0	`pP	 B0`
p	P  	 b0`
p	P  	 b0`
p	P   2
0	`pP b0`pP   20 B  
 
20`pP    20 20 B0`   20`p 20 B   B0p   b0`   B0`   b0`   B0p   b0`   B0`   b0`  
 
20`pP 20	 0`
p	P   B0`pP   R0	 0`
p	P  	 b0`
p	P   20`p
 
20`pP
 
20`pP b0`pP   B0`pP   B0`   20 20`p	 B0`
p	P  	 0`
p	P  
 + 0`
p	P
 + 0`
p	P b0`   b0`  	 0`
p	P  
 5 0`
p	P
 - 0`
p	P
 1 0`
p	P B0`pP  
 % 0`
p	P
 % 0`
p	P 0`pP  
 
20`pP 0`pP  
 % 0`
p	P
 ' 0`
p	P b   r0
 % 0`
p	P
 % 0`
p	P
 ) 0`
p	P	 0`
p	P   ' 0`pP B   
0	`pP B   0`pP   B0`pP  	 0`
p	P  	 b0`
p	P  	 0`
p	P   20`p
 
" 0`pP   % 0`pP	 B0`
p	P  
 + 0`
p	P b0`pP  
 
$ 0`pP  
 
R0`pP R
0	`pP	 B0`
p	P  	 b0`
p	P   2
0	`pP
 - 0`
p	P
 5 0`
p	P
 7 0`
p	P
 ' 0`
p	P	 0`
p	P   2
0	`pP
 
$ 0`pP   R0`p	 0`
p	P   b0`pP  	 b0`
p	P   20`p 0`   R0`p r
0	`pP
 
20`pP B0`   B0`pP   b0`pP   R0 R0 R0 b0`pP   R0	 0`
p	P   20`p B0`   B0`pP      B0`pP   B0`   B0`pP   B0`pP  
 
20`pP b0`pP   B0`  
 
20`pP B0`   2
0	`pP 20`p B0`   B0`   B0`pP   R0`p	 b0`
p	P  
 
20`pP 0`pP   0`pP  
 # 0`
p	P B0`pP  	 B0`
p	P   B0`pP  	 B0`
p	P  	 b0`
p	P  	 0`
p	P  
 + 0`
p	P
 ' 0`
p	P	 0`
p	P  
 
20`pP	 0`
p	P  
 
20`pP
 % 0`
p	P	 0`
p	P  
 
" 0`pP  
 
$ 0`pP   0`p B0`pP   B0`     
 
20`pP 20 R
0	`pP R0`p 2
0	`pP B0`pP   B0`   B0`   20 B   20`p 2
0	`pP	 b0`
p	P  	 b0`
p	P  	 B0`
p	P  	 B0`
p	P   B0`pP  
 
20`pP 2
0	`pP r0`p 20 20 20 20 b0`pP   R
0	`pP b0`   r
0	`pP	 b0`
p	P   b0`pP  	 b0`
p	P   R0`p
 
& 0`p  	 0`
p	P      20`p B0`pP   0`pP   # 0`pP	 	# 0`             b  
 
20`pP B0`pP   B0`pP   B0`   20`p       B0`   b0`   20 20 B0`   20 B0`pP   20`p R0`p B   B0`pP   B0`pP            B0`  
 
20`pP b0`pP   B   0     
 
20`pP R
0	`pP B0`pP   B0`pP   B0`   20`p    B0`   b0`   20 20 B0`   20 B0`pP   20`p B         B   0  
 
20`pP R
0	`pP 0`pP   0`pP   R0 B   B   B0`     
 
r0`pP

0`P   B   B0`            b0`   0`pP  
0`
p	P B  
 
20`pP 20 B0`pP   B0`pP   20 20          20`p B   B   B   B   B   B              P                                                                                                                                                                                                                                                        kπY    F          ( 4 @       R [ j     ral0122.dll Ral_Init Ral_SafeUnload Ral_Unload                                                                                                                                                                                                                                                                                                                                                                                                            <          ܱ          `                      |                    ֳ                    "     <     L     h                    Ĵ     ޴                    ,     :     V     h             x                                   ʵ     ص                                        $     .     8     @     J     T     ^     h     r     |             |                    ֳ                    "     <     L     h                    Ĵ     ޴                    ,     :     V     h             x                                   ʵ     ص                                        $     .     8     @     J     T     ^     h     r     |              DeleteCriticalSection  EnterCriticalSection  GetCurrentProcess GetCurrentProcessId GetCurrentThreadId  GetLastError  &GetModuleHandleA  XGetProcAddress  GetSystemTimeAsFileTime GetTickCount  InitializeCriticalSection MLeaveCriticalSection  QueryPerformanceCounter RtlAddFunctionTable RtlCaptureContext RtlLookupFunctionEntry  RtlVirtualUnwind  SetUnhandledExceptionFilter Sleep TerminateProcess  TlsGetValue UnhandledExceptionFilter  VirtualProtect  VirtualQuery  O __dllonexit T __iob_func  y _amsg_exit  _initterm _lock %_onexit _unlock _vsnprintf  abort bsearch calloc  free  fwrite  malloc  memcmp  memcpy  memmove memset  qsort signal  (strcmp  *strcpy  /strlen  2strncmp 6strrchr Qvfprintf                                                  KERNEL32.dll                              msvcrt.dll                                                                                                                                                                               m                    ёm    `ёm                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     Вm    `Вm    왒m    0m                                                                                                                                                                                                                                                                                                                                                                                                                                                                         (    h     (hpx  (08@`hpxȡСء  0@P   p   ȥХإ  (08@HPX`hpxȦЦئ  (@H  (    (08@HPX`hpx  0  (08@HPX`hpxȣУأ  (08@HPX`hpxȤФؤ  (08@HPX`hpxȥХإ ȯЯد       (08@HPX`hpxȠРؠ  (08@H  0@@HPX`hpxȨШب  (08@H  (08@HPX`hpxȮЮخ  (08@HP 0    ȦЦئ  (08@HPX`hpxȧЧا  (0`hpxȪЪت  (08@HPX`hpxȫЫث  (08@HPX`hpxȬЬج @ @     0@P`pУ  0@P`p     08        (08                                                                                                                                                                                                                                                                                                                                                                                                        ,              m                          ,    d       ͑m                           ,    K       Αm                                                     ,    T      ϑm                          ,    
      `ёm                               ɢ                      ,    g      ґm                           ,          0ґm                          ,    fT       בm    b                      ,    )      pۑm    j                                                    1                      ,           ݑm                          ,    23       m                           ,    3      Pm                           ,    ~}      `m                           ,          pm                                                                                                                                                                                                                                                           d       GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/crtdll.c  m              char size_t #   long long unsigned int long long int intptr_t >   uintptr_t K   wchar_t b   short unsigned int    int long int pthreadlocinfo ?  E  threadlocaleinfostruct `        	lc_codepage   	lc_collate_cp   	lc_handle /  	lc_id `  $	lc_category p  H
lc_clike   
mb_cur_max   
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv   (
ctype1_refcount   0
ctype1 %  8
pctype +  @
pclmap 1  H
pcumap 1  P
lc_time_curr ]  X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct L  	locinfo (   	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID d  
   	locale    	wlocale        	wrefcount            unsigned int   K  ?  ?   sizetype long unsigned int   p  ?       ?   lconv X-  decimal_point .   thousands_sep /  grouping 0  int_curr_symbol 1  currency_symbol 2   mon_decimal_point 3  (mon_thousands_sep 4  0mon_grouping 5  8positive_sign 6  @negative_sign 7  Hint_frac_digits 8   Pfrac_digits 9   Qp_cs_precedes :   Rp_sep_by_space ;   Sn_cs_precedes <   Tn_sep_by_space =   Up_sign_posn >   Vn_sign_posn ?   W        H  unsigned char 7  __lc_time_data M  _PHNDLR ?r  x       _XCPT_ACTION A  XcptNum BK   SigNum C  XcptAction Dc        _XcptActTab G  _XcptActTabCount H  _XcptActTabSize I  _First_FPE_Indx J  _Num_FPE K  E  _EXCEPTION_RECORD 	  	ExceptionCode 	   	ExceptionFlags 	  	ExceptionRecord 	?  	ExceptionAddress 	  	NumberParameters 	  	ExceptionInformation 	,        _CONTEXT 
  	P1Home    	P2Home   	P3Home   	P4Home   	P5Home    	P6Home   (	ContextFlags   0	MxCsr   4	SegCs   8	SegDs   :	SegEs   <	SegFs   >	SegGs   @	SegSs   B	EFlags   D	Dr0   H	Dr1   P	Dr2   X	Dr3   `	Dr6   h	Dr7   p	Rax   x	Rcx   	Rdx   	Rbx   	Rsp   	Rbp   	Rsi   	Rdi   	R8   	R9   	R10   	R11   	R12   	R13   	R14   	R15   	Rip      
VectorRegister    
VectorControl   
DebugControl   
LastBranchToRip   
LastBranchFromRip   
LastExceptionToRip   
LastExceptionFromRip    WINBOOL   BYTE 7  WORD    DWORD K  float LPVOID     __imp__pctype $[  %  __imp__wctype 3[  __imp__pwctype ?[  H       __newclmap H  __newcumap I  __ptlocinfo J(  __ptmbcinfo K  __globallocalestatus L  __locale_changed M  __initiallocinfo NE  __initiallocalestructinfo OL  __imp___mb_cur_max   signed char short int ULONG_PTR 	1   DWORD64 	   PVOID   LONG   HANDLE   LONGLONG    ULONGLONG    _LIST_ENTRY ])
  	Flink ^)
   	Blink _)
     LIST_ENTRY `  _GUID 

  Data1 
K   Data2 
   Data3 
   Data4 

   7  
  ?   GUID 
B
  
  IID 
R
  
  CLSID 
Z
  
  FMTID 
a
  
  EXCEPTION_ROUTINE 
      ?         PEXCEPTION_ROUTINE 1  
  _M128A Gc  	Low H   	High I   M128A J7  c    ?   c    ?       ?  _     double long double       ?    _sys_errlist   _sys_nerr   __imp___argc   __imp___argv )  /    __imp___wargv J  P    __imp__environ )  __imp__wenviron J  __imp__pgmptr /  __imp__wpgmptr P  __imp__fmode   __imp__osplatform  @  __imp__osver 	@  __imp__winver @  __imp__winmajor @  __imp__winminor $@  _amblksiz 5  _XMM_SAVE_AREA32  j  	ControlWord k   	StatusWord l  	TagWord m  	Reserved1 n  	ErrorOpcode o  	ErrorOffset p  	ErrorSelector q  	Reserved2 r  	DataOffset s  	DataSelector t  	Reserved3 u  	MxCsr v  	MxCsr_Mask w  	FloatRegisters xq   	XmmRegisters y  
Reserved4 z   XMM_SAVE_AREA32 {H    	Header    	Legacy q   	Xmm0 c  	Xmm1 c  	Xmm2 c  	Xmm3 c  	Xmm4 c  	Xmm5 c  
Xmm6 c   
Xmm7 c  
Xmm8 c   
Xmm9 c  0
Xmm10 c  @
Xmm11 c  P
Xmm12 c  `
Xmm13 c  p
Xmm14 c  
Xmm15 c   c    ?      FltSave   FloatSave       c  ,  ?     <  ?   !`  Next   prev    _EXCEPTION_REGISTRATION_RECORD   "<   "   `  !  Handler   handler    !  FiberData   Version    _NT_TIB 8  	ExceptionList    	StackBase   	StackLimit   	SubSystemTib   "   	ArbitraryUserPointer   (	Self   0   NT_TIB   PNT_TIB     GUID_MAX_POWER_SAVINGS b
  GUID_MIN_POWER_SAVINGS c
  GUID_TYPICAL_POWER_SAVINGS d
  NO_SUBGROUP_GUID e
  ALL_POWERSCHEMES_GUID f
  GUID_POWERSCHEME_PERSONALITY g
  GUID_ACTIVE_POWERSCHEME h
  GUID_IDLE_RESILIENCY_SUBGROUP i
  GUID_IDLE_RESILIENCY_PERIOD j
  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k
  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l
  GUID_VIDEO_SUBGROUP m
  GUID_VIDEO_POWERDOWN_TIMEOUT n
  GUID_VIDEO_ANNOYANCE_TIMEOUT o
  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p
  GUID_VIDEO_DIM_TIMEOUT q
  GUID_VIDEO_ADAPTIVE_POWERDOWN r
  GUID_MONITOR_POWER_ON s
  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t
  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u
  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v
  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w
  GUID_CONSOLE_DISPLAY_STATE x
  GUID_ALLOW_DISPLAY_REQUIRED y
  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z
  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {
  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |
  GUID_DISK_SUBGROUP }
  GUID_DISK_POWERDOWN_TIMEOUT ~
  GUID_DISK_IDLE_TIMEOUT 
  GUID_DISK_BURST_IGNORE_THRESHOLD 
  GUID_DISK_ADAPTIVE_POWERDOWN 
  GUID_SLEEP_SUBGROUP 
  GUID_SLEEP_IDLE_THRESHOLD 
  GUID_STANDBY_TIMEOUT 
  GUID_UNATTEND_SLEEP_TIMEOUT 
  GUID_HIBERNATE_TIMEOUT 
  GUID_HIBERNATE_FASTS4_POLICY 
  GUID_CRITICAL_POWER_TRANSITION 
  GUID_SYSTEM_AWAYMODE 
  GUID_ALLOW_AWAYMODE 
  GUID_ALLOW_STANDBY_STATES 
  GUID_ALLOW_RTC_WAKE 
  GUID_ALLOW_SYSTEM_REQUIRED 
  GUID_SYSTEM_BUTTON_SUBGROUP 
  GUID_POWERBUTTON_ACTION 
  GUID_SLEEPBUTTON_ACTION 
  GUID_USERINTERFACEBUTTON_ACTION 
  GUID_LIDCLOSE_ACTION 
  GUID_LIDOPEN_POWERSTATE 
  GUID_BATTERY_SUBGROUP 
  GUID_BATTERY_DISCHARGE_ACTION_0 
  GUID_BATTERY_DISCHARGE_LEVEL_0 
  GUID_BATTERY_DISCHARGE_FLAGS_0 
  GUID_BATTERY_DISCHARGE_ACTION_1 
  GUID_BATTERY_DISCHARGE_LEVEL_1 
  GUID_BATTERY_DISCHARGE_FLAGS_1 
  GUID_BATTERY_DISCHARGE_ACTION_2 
  GUID_BATTERY_DISCHARGE_LEVEL_2 
  GUID_BATTERY_DISCHARGE_FLAGS_2 
  GUID_BATTERY_DISCHARGE_ACTION_3 
  GUID_BATTERY_DISCHARGE_LEVEL_3 
  GUID_BATTERY_DISCHARGE_FLAGS_3 
  GUID_PROCESSOR_SETTINGS_SUBGROUP 
  GUID_PROCESSOR_THROTTLE_POLICY 
  GUID_PROCESSOR_THROTTLE_MAXIMUM 
  GUID_PROCESSOR_THROTTLE_MINIMUM 
  GUID_PROCESSOR_ALLOW_THROTTLING 
  GUID_PROCESSOR_IDLESTATE_POLICY 
  GUID_PROCESSOR_PERFSTATE_POLICY 
  GUID_PROCESSOR_PERF_INCREASE_THRESHOLD 
  GUID_PROCESSOR_PERF_DECREASE_THRESHOLD 
  GUID_PROCESSOR_PERF_INCREASE_POLICY 
  GUID_PROCESSOR_PERF_DECREASE_POLICY 
  GUID_PROCESSOR_PERF_INCREASE_TIME 
  GUID_PROCESSOR_PERF_DECREASE_TIME 
  GUID_PROCESSOR_PERF_TIME_CHECK 
  GUID_PROCESSOR_PERF_BOOST_POLICY 
  GUID_PROCESSOR_PERF_BOOST_MODE 
  GUID_PROCESSOR_IDLE_ALLOW_SCALING 
  GUID_PROCESSOR_IDLE_DISABLE 
  GUID_PROCESSOR_IDLE_STATE_MAXIMUM 
  GUID_PROCESSOR_IDLE_TIME_CHECK 
  GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD 
  GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY 
  GUID_PROCESSOR_CORE_PARKING_MAX_CORES 
  GUID_PROCESSOR_CORE_PARKING_MIN_CORES 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD 
  GUID_PROCESSOR_PARKING_CORE_OVERRIDE 
  GUID_PROCESSOR_PARKING_PERF_STATE 
  GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD 
  GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD 
  GUID_PROCESSOR_PERF_HISTORY 
  GUID_PROCESSOR_PERF_LATENCY_HINT 
  GUID_PROCESSOR_DISTRIBUTE_UTILITY 
  GUID_SYSTEM_COOLING_POLICY 
  GUID_LOCK_CONSOLE_ON_WAKE 
  GUID_DEVICE_IDLE_POLICY 
  GUID_ACDC_POWER_SOURCE 
  GUID_LIDSWITCH_STATE_CHANGE 
  GUID_BATTERY_PERCENTAGE_REMAINING 
  GUID_GLOBAL_USER_PRESENCE 
  GUID_SESSION_DISPLAY_STATUS 
  GUID_SESSION_USER_PRESENCE 
  GUID_IDLE_BACKGROUND_TASK 
  GUID_BACKGROUND_TASK_NOTIFICATION 
  GUID_APPLAUNCH_BUTTON 
  GUID_PCIEXPRESS_SETTINGS_SUBGROUP 
  GUID_PCIEXPRESS_ASPM_POLICY 
  GUID_ENABLE_SWITCH_FORCED_SHUTDOWN 
  PPM_PERFSTATE_CHANGE_GUID 
  PPM_PERFSTATE_DOMAIN_CHANGE_GUID 
  PPM_IDLESTATE_CHANGE_GUID 
  PPM_PERFSTATES_DATA_GUID 
  PPM_IDLESTATES_DATA_GUID 
  PPM_IDLE_ACCOUNTING_GUID 
  PPM_IDLE_ACCOUNTING_EX_GUID 
  PPM_THERMALCONSTRAINT_GUID 
  PPM_PERFMON_PERFSTATE_GUID 
  PPM_THERMAL_POLICY_CHANGE_GUID 
  PIMAGE_TLS_CALLBACK t)  S)  z)  )         _RTL_CRITICAL_SECTION_DEBUG 0\*  	Type ]   	CreatorBackTraceIndex ^  	CriticalSection _%+  	ProcessLocksList `/
  	EntryCount a   	ContentionCount b  $	Flags c  (	CreatorBackTraceIndexHigh d  ,	SpareWORD e  . _RTL_CRITICAL_SECTION (w%+  	DebugInfo x++   	LockCount y  	RecursionCount z  	OwningThread {  	LockSemaphore |  	SpinCount }    *  PRTL_CRITICAL_SECTION_DEBUG fO+  )  RTL_CRITICAL_SECTION ~*  CRITICAL_SECTION 
U+  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

  RPC_IF_HANDLE B  #tagCOINITBASE   ,  $COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )+  IWinTypesBase_v0_1_s_ifspec *+  IID_IUnknown W
  IID_AsyncIUnknown 
  IID_IClassFactory m
  IID_IMarshal i
  IID_INoMarshal P
  IID_IAgileObject 
  IID_IMarshal2 
  IID_IMalloc V
  IID_IStdMarshalInfo 

  IID_IExternalConnection p
  IID_IMultiQI 
  IID_AsyncIMultiQI B
  IID_IInternalUnknown 
  IID_IEnumUnknown 
  IID_IEnumString 
  IID_ISequentialStream F
  IID_IStream 
  IID_IRpcChannelBuffer 5	
  IID_IRpcChannelBuffer2 	
  IID_IAsyncRpcChannelBuffer a

  IID_IRpcChannelBuffer3 #
  IID_IRpcSyntaxNegotiate =
  IID_IRpcProxyBuffer 
  IID_IRpcStubBuffer 
  IID_IPSFactoryBuffer 

  IID_IChannelHook C
  IID_IClientSecurity g
  IID_IServerSecurity 
  IID_IRpcOptions 
  IID_IGlobalOptions R
  IID_ISurrogate 
  IID_IGlobalInterfaceTable -
  IID_ISynchronize 
  IID_ISynchronizeHandle 0
  IID_ISynchronizeEvent 
  IID_ISynchronizeContainer 
  IID_ISynchronizeMutex S
  IID_ICancelMethodCalls 
  IID_IAsyncManager .
  IID_ICallFactory 
  IID_IRpcHelper 

  IID_IReleaseMarshalBuffers u
  IID_IWaitMultiple 
  IID_IAddrTrackingControl <
  IID_IAddrExclusionControl 
  IID_IPipeByte 
  IID_IPipeLong }
  IID_IPipeDouble 
  IID_IComThreadingInfo 
  IID_IProcessInitControl V
  IID_IFastRundown 
  IID_IMarshalingStream 
  IID_ICallbackWithNoReentrancyToApplicationSTA 
  GUID_NULL 

  CATID_MARSHALER 
  IID_IRpcChannel 
  IID_IRpcStub 
  IID_IStubManager 
  IID_IRpcProxy 
  IID_IProxyManager 
  IID_IPSFactory 
  IID_IInternalMoniker 
  IID_IDfReserved1 
  IID_IDfReserved2 
  IID_IDfReserved3 
  CLSID_StdMarshal 
  CLSID_AggStdMarshal 
  CLSID_StdAsyncActManager 
  IID_IStub 
  IID_IProxy 
  IID_IEnumGeneric 
  IID_IEnumHolder 
  IID_IEnumCallback  
  IID_IOleManager !
  IID_IOlePresObj "
  IID_IDebug #
  IID_IDebugStream $
  CLSID_PSGenObject %
  CLSID_PSClientSite &
  CLSID_PSClassObject '
  CLSID_PSInPlaceActive (
  CLSID_PSInPlaceFrame )
  CLSID_PSDragDrop *
  CLSID_PSBindCtx +
  CLSID_PSEnumerators ,
  CLSID_StaticMetafile -
  CLSID_StaticDib .
  CID_CDfsVolume /
  CLSID_DCOMAccessControl 0
  CLSID_StdGlobalInterfaceTable 1
  CLSID_ComBinding 2
  CLSID_StdEvent 3
  CLSID_ManualResetEvent 4
  CLSID_SynchronizeContainer 5
  CLSID_AddrControl 6
  CLSID_CCDFormKrnl 7
  CLSID_CCDPropertyPage 8
  CLSID_CCDFormDialog 9
  CLSID_CCDCommandButton :
  CLSID_CCDComboBox ;
  CLSID_CCDTextBox <
  CLSID_CCDCheckBox =
  CLSID_CCDLabel >
  CLSID_CCDOptionButton ?
  CLSID_CCDListBox @
  CLSID_CCDScrollBar A
  CLSID_CCDGroupBox B
  CLSID_CCDGeneralPropertyPage C
  CLSID_CCDGenericPropertyPage D
  CLSID_CCDFontPropertyPage E
  CLSID_CCDColorPropertyPage F
  CLSID_CCDLabelPropertyPage G
  CLSID_CCDCheckBoxPropertyPage H
  CLSID_CCDTextBoxPropertyPage I
  CLSID_CCDOptionButtonPropertyPage J
  CLSID_CCDListBoxPropertyPage K
  CLSID_CCDCommandButtonPropertyPage L
  CLSID_CCDComboBoxPropertyPage M
  CLSID_CCDScrollBarPropertyPage N
  CLSID_CCDGroupBoxPropertyPage O
  CLSID_CCDXObjectPropertyPage P
  CLSID_CStdPropertyFrame Q
  CLSID_CFormPropertyPage R
  CLSID_CGridPropertyPage S
  CLSID_CWSJArticlePage T
  CLSID_CSystemPage U
  CLSID_IdentityUnmarshal V
  CLSID_InProcFreeMarshaler W
  CLSID_Picture_Metafile X
  CLSID_Picture_EnhMetafile Y
  CLSID_Picture_Dib Z
  GUID_TRISTATE [
  IWinTypes_v0_1_c_ifspec (+  IWinTypes_v0_1_s_ifspec )+  %VARENUM    V=  $VT_EMPTY  $VT_NULL $VT_I2 $VT_I4 $VT_R4 $VT_R8 $VT_CY $VT_DATE $VT_BSTR $VT_DISPATCH 	$VT_ERROR 
$VT_BOOL $VT_VARIANT $VT_UNKNOWN 
$VT_DECIMAL $VT_I1 $VT_UI1 $VT_UI2 $VT_UI4 $VT_I8 $VT_UI8 $VT_INT $VT_UINT $VT_VOID $VT_HRESULT $VT_PTR $VT_SAFEARRAY $VT_CARRAY $VT_USERDEFINED $VT_LPSTR $VT_LPWSTR $VT_RECORD $$VT_INT_PTR %$VT_UINT_PTR &$VT_FILETIME @$VT_BLOB A$VT_STREAM B$VT_STORAGE C$VT_STREAMED_OBJECT D$VT_STORED_OBJECT E$VT_BLOB_OBJECT F$VT_CF G$VT_CLSID H$VT_VERSIONED_STREAM I&VT_BSTR_BLOB &VT_VECTOR  &VT_ARRAY   &VT_BYREF  @&VT_RESERVED  &VT_ILLEGAL &VT_ILLEGALMASKED &VT_TYPEMASK  IID_IMallocSpy 
  IID_IBindCtx :
  IID_IEnumMoniker J 
  IID_IRunnableObject  
  IID_IRunningObjectTable !
  IID_IPersist i"
  IID_IPersistStream "
  IID_IMoniker j#
  IID_IROTData X%
  IID_IEnumSTATSTG %
  IID_IStorage X&
  IID_IPersistFile A(
  IID_IPersistStorage (
  IID_ILockBytes )
  IID_IEnumFORMATETC *
  IID_IEnumSTATDATA l+
  IID_IRootStorage ,
  IID_IAdviseSink ,
  IID_AsyncIAdviseSink s-
  IID_IAdviseSink2 .
  IID_AsyncIAdviseSink2 ./
  IID_IDataObject /
  IID_IDataAdviseHolder 1
  IID_IMessageFilter 1
  FMTID_SummaryInformation ]2
  FMTID_DocSummaryInformation _2
  FMTID_UserDefinedProperties a2
  FMTID_DiscardableInformation c2
  FMTID_ImageSummaryInformation e2
  FMTID_AudioSummaryInformation g2
  FMTID_VideoSummaryInformation i2
  FMTID_MediaFileSummaryInformation k2
  IID_IClassActivator s2
  IID_IFillLockBytes 2
  IID_IProgressNotify 3
  IID_ILayoutStorage 3
  IID_IBlockingLock 4
  IID_ITimeAndNoticeControl 4
  IID_IOplockStorage N5
  IID_IDirectWriterLock 5
  IID_IUrlMon M6
  IID_IForegroundTransfer 6
  IID_IThumbnailExtractor 7
  IID_IDummyHICONIncluder 7
  IID_IProcessLock 7
  IID_ISurrogateService H8
  IID_IInitializeSpy 8
  IID_IApartmentShutdown 9
  IID_IOleAdviseHolder 
  IID_IOleCache b
  IID_IOleCache2 )
  IID_IOleCacheControl 
  IID_IParseDisplayName <
  IID_IOleContainer 
  IID_IOleClientSite 
  IID_IOleObject 
  IOLETypes_v0_0_c_ifspec +  IOLETypes_v0_0_s_ifspec +  IID_IOleWindow $
  IID_IOleLink 
  IID_IOleItemContainer 
  IID_IOleInPlaceUIWindow v	
  IID_IOleInPlaceActiveObject 

  IID_IOleInPlaceFrame 

  IID_IOleInPlaceObject 
  IID_IOleInPlaceSite 
  IID_IContinue 

  IID_IViewObject 

  IID_IViewObject2 *
  IID_IDropSource 
  IID_IDropTarget [
  IID_IDropSourceNotify 
  IID_IEnumOLEVERB v
  IID_IServiceProvider M
  IOleAutomationTypes_v1_0_c_ifspec +  IOleAutomationTypes_v1_0_s_ifspec +  IID_ICreateTypeInfo ;
  IID_ICreateTypeInfo2 b
  IID_ICreateTypeLib 
  IID_ICreateTypeLib2 
  IID_IDispatch 	
  IID_IEnumVARIANT 

  IID_ITypeComp 5
  IID_ITypeInfo 
  IID_ITypeInfo2 P
  IID_ITypeLib 
  IID_ITypeLib2 =
  IID_ITypeChangeEvents a
  IID_IErrorInfo 
  IID_ICreateErrorInfo }
  IID_ISupportErrorInfo  
  IID_ITypeFactory u
  IID_ITypeMarshal 
  IID_IRecordInfo 
  IID_IErrorLog  
  IID_IPropertyBag z
  __MIDL_itf_msxml_0000_v0_0_c_ifspec +  __MIDL_itf_msxml_0000_v0_0_s_ifspec +  LIBID_MSXML 
  IID_IXMLDOMImplementation  
  IID_IXMLDOMNode '
  IID_IXMLDOMDocumentFragment 
  IID_IXMLDOMDocument f
  IID_IXMLDOMNodeList u
  IID_IXMLDOMNamedNodeMap 
  IID_IXMLDOMCharacterData 
  IID_IXMLDOMAttribute 
  IID_IXMLDOMElement 
  IID_IXMLDOMText 
  IID_IXMLDOMComment %
  IID_IXMLDOMProcessingInstruction 
  IID_IXMLDOMCDATASection 
  IID_IXMLDOMDocumentType 
  IID_IXMLDOMNotation 
  IID_IXMLDOMEntity 
  IID_IXMLDOMEntityReference 
  IID_IXMLDOMParseError a	
  IID_IXTLRuntime 	
  DIID_XMLDOMDocumentEvents =

  CLSID_DOMDocument \

  CLSID_DOMFreeThreadedDocument `

  IID_IXMLHttpRequest g

  CLSID_XMLHTTPRequest 

  IID_IXMLDSOControl 

  CLSID_XMLDSOControl 

  IID_IXMLElementCollection 
  IID_IXMLDocument J
  IID_IXMLDocument2 
  IID_IXMLElement $
  IID_IXMLElement2 
  IID_IXMLAttribute 
  IID_IXMLError 

  CLSID_XMLDocument .

  CLSID_SBS_StdURLMoniker ~
  CLSID_SBS_HttpProtocol 
  CLSID_SBS_FtpProtocol 
  CLSID_SBS_GopherProtocol 
  CLSID_SBS_HttpSProtocol 
  CLSID_SBS_FileProtocol 
  CLSID_SBS_MkProtocol 
  CLSID_SBS_UrlMkBindCtx 
  CLSID_SBS_SoftDistExt 
  CLSID_SBS_CdlProtocol 
  CLSID_SBS_ClassInstallFilter 
  CLSID_SBS_InternetSecurityManager 
  CLSID_SBS_InternetZoneManager 
  IID_IAsyncMoniker 
  CLSID_StdURLMoniker 
  CLSID_HttpProtocol 
  CLSID_FtpProtocol 
  CLSID_GopherProtocol 
  CLSID_HttpSProtocol 
  CLSID_FileProtocol 
  CLSID_MkProtocol 
  CLSID_StdURLProtocol 
  CLSID_UrlMkBindCtx 
  CLSID_CdlProtocol 
  CLSID_ClassInstallFilter 
  IID_IAsyncBindCtx 
  IID_IPersistMoniker P
  IID_IMonikerProp !
  IID_IBindProtocol 
  IID_IBinding 
  IID_IBindStatusCallback u
  IID_IBindStatusCallbackEx 
  IID_IAuthenticate d
  IID_IAuthenticateEx 
  IID_IHttpNegotiate A
  IID_IHttpNegotiate2 
  IID_IHttpNegotiate3 ;	
  IID_IWinInetFileStream 	
  IID_IWindowForBindingUI 0

  IID_ICodeInstall 

  IID_IUri -
  IID_IUriContainer 

  IID_IUriBuilder 

  IID_IUriBuilderFactory (
  IID_IWinInetInfo 
  IID_IHttpSecurity 
  IID_IWinInetHttpInfo y
  IID_IWinInetHttpTimeouts 
  IID_IWinInetCacheHints Z
  IID_IWinInetCacheHints2 
  SID_BindHost 5
  IID_IBindHost ?
  IID_IInternet M
  IID_IInternetBindInfo 
  IID_IInternetBindInfoEx &
  IID_IInternetProtocolRoot 
  IID_IInternetProtocol 
  IID_IInternetProtocolEx W
  IID_IInternetProtocolSink 
  IID_IInternetProtocolSinkStackable 
  IID_IInternetSession ?
  IID_IInternetThreadSwitch H
  IID_IInternetPriority 
  IID_IInternetProtocolInfo N
  CLSID_InternetSecurityManager 
  CLSID_InternetZoneManager 
  CLSID_PersistentZoneIdentifier 
  IID_IInternetSecurityMgrSite 
  IID_IInternetSecurityManager i
  IID_IInternetSecurityManagerEx 
  IID_IInternetSecurityManagerEx2 V
  IID_IZoneIdentifier  
  IID_IInternetHostSecurityManager !
  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"
  IID_IInternetZoneManager "
  IID_IInternetZoneManagerEx L$
  IID_IInternetZoneManagerEx2 ]%
  CLSID_SoftDistExt &
  IID_ISoftDistExt &
  IID_ICatalogFileInfo x'
  IID_IDataFilter '
  IID_IEncodingFilterFactory (
  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)
  IID_IWrappedProtocol A)
  IID_IGetBindHandle )
  IID_IBindCallbackRedirect 
*
  IID_IPropertyStorage 
  IID_IPropertySetStorage 
  IID_IEnumSTATPROPSTG 
  IID_IEnumSTATPROPSETSTG D
  IID_StdOle 
  GUID_DEVINTERFACE_DISK 
  GUID_DEVINTERFACE_CDROM 

  GUID_DEVINTERFACE_PARTITION 
  GUID_DEVINTERFACE_TAPE 
  GUID_DEVINTERFACE_WRITEONCEDISK 
  GUID_DEVINTERFACE_VOLUME 
  GUID_DEVINTERFACE_MEDIUMCHANGER 
  GUID_DEVINTERFACE_FLOPPY 
  GUID_DEVINTERFACE_CDCHANGER 
  GUID_DEVINTERFACE_STORAGEPORT 
  GUID_DEVINTERFACE_COMPORT 
  GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR 
  _SCARD_IO_REQUEST X  dwProtocol    cbPciLength    SCARD_IO_REQUEST gX  X  g_rgSCardT0Pci  %X  g_rgSCardT1Pci  %X  g_rgSCardRawPci  %X  IID_IPrintDialogCallback !
  IID_IPrintDialogServices !
  _PVFV "  _PIFV "  _commode "  '@",Z  osfhnd "-    osfile ".   pipech "/   	lockinitflag "0  lock "1r+  (textmode "2   8(unicode "3    8pipech2 "4Z  9    Z  ?   ioinfo "5sY  +Z  +Z   1Z  Z  __imp___badioinfo "K Z  __imp___pioinfo "P Z  _dowildcard "m  _newmode "n  __imp___winitenv "qJ  __imp___initenv "v)  __imp__acmdln "{/  __imp__wcmdln "/  )V     "'[  $__uninitialized  $__initializing $__initialized  *V   "Z  '[  __native_startup_state "2[  __native_startup_lock "r[  x[  +__native_dllmain_reason "*  __native_vcclrit_reason "*  __security_cookie #|   
  [      2   _pRawDllMain #\  [  [  VY  \   __xi_a &\  __xi_z '\  IY  :\   __xc_a (/\  __xc_z )/\  __dyn_tls_init_callback ,o)  ,__proc_attached .  	 m    __onexitbegin 0\  IY  __onexitend 1\  mingw_app_type 3  -pcinit ;VY  	m    .__DllMainCRTStartup 
  m    9      
_  /w         /n        /   2  0  0retcode 
    1i__leave m    2m    d  2m    d  3m    )d  ]  4Ru 4Qs 4Xv  3m    _  ]  4Ru 4Qs 4Xv  2m    d  3*m    _  
^  4Ru 4Qs 4Xv  3;m    )d  .^  4Ru 4Qs 4Xv  3Tm    _  Q^  4Ru 4Q04Xv  2jm    4d  3zm    d  ^  4Ru 4Q14Xv  3m    d  ^  4Ru 4Q04Xv  3m    )d  ^  4Ru 4Q04Xv  3m    _  ^  4Ru 4Q04Xv  5m    d  4Ru 4Q24Xv   6DllMainCRTStartup 
  m    O       _  /w       /n     X  /   2    7m    \  _  4RR4QQ4XX 2m    Ed  2m    Pd  8m    \  4RX4Qd4Xh  9_CRT_INIT K
  Pm    ;      b  /w   K    /n   K    /   K2  z  :   a  ;L   V  Q  0fiberid W  u  0nested X    <c  m           W`  =c  m            >c  	  ?m           @c     Alc  m       Z`  >c  .  >c  R  Bc   <c  0m           ua  >Wc  u  BHc   3 m    [d  $a  4R
 Czm    Ca  4R| 4Q24X}  3Ym    jd  [a  DRDQ 3hm    ud  ra  4RO 5m    jd  DRDQ  E    FL     Gm    j       ;b  ;   \    Gm    G       b  0onexitend \    2m    d  5m    d  4Rv   Ac  m       -b  >Wc    BHc   2m    d   Alc  ~m    @   jb  >c  >  >c  b  Bc   3m    [d  b  4R
 5m    ud  4RO   .pre_c_init >   m    M       b  ;   @\    3m    d  b  4R
  5m    d  4Rs   _TEB HNtCurrentTeb  c  b  I_InterlockedExchangePointer _  fc  JTarget _fc  JValue _     I_InterlockedCompareExchangePointer T  c  JDestination Tfc  JExChange T  JComperand T   I__readgsqword    d  JOffset K  Kret     L      %MDllMain DllMain 5L      7M__main __main $L      #tL6   6   MSleep Sleep $L      #L      "L        "Nfree free Nmalloc malloc L      " R     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/atonexit.c ͑m             char long long unsigned int long long int intptr_t >   wchar_t b   short unsigned int    int long int pthreadlocinfo "  (  threadlocaleinfostruct `         	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id C  $	lc_category S  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv j  (
ctype1_refcount   0
ctype1 p  8
pctype v  @
pclmap |  H
pcumap |  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct /  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID G  
   	locale    	wlocale        	wrefcount             unsigned int   .  "  "   sizetype long unsigned int   S  "     c  "   lconv c          unsigned char   __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B.   SigNum C   XcptAction D     !   _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WORD    DWORD .  float   __imp__pctype $  p  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N(  __initiallocalestructinfo O/  __imp___mb_cur_max   signed char short int ULONG_PTR 1   LONG    HANDLE   _LIST_ENTRY ]X  	Flink ^X   	Blink _X   $  LIST_ENTRY `$  _GUID   Data1 .   Data2    Data3    Data4        "   GUID q    IID R    CLSID Z    FMTID a    _onexit_t 	1  $     double long double H    Y  "    _sys_errlist 	I  _sys_nerr 	   __imp___argc 	  __imp___argv 	      __imp___wargv 	      __imp__environ 	Ц  __imp__wenviron 	  __imp__pgmptr 	  __imp__wpgmptr 	  __imp__fmode 	  __imp__osplatform 	   __imp__osver 		  __imp__winver 	  __imp__winmajor 	  __imp__winminor 	$  _amblksiz 
5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   _RTL_CRITICAL_SECTION_DEBUG 0\\  	Type ]   	CreatorBackTraceIndex ^  	CriticalSection _  	ProcessLocksList `^  	EntryCount a   	ContentionCount b  $	Flags c  (	CreatorBackTraceIndexHigh d  ,	SpareWORD e  . _RTL_CRITICAL_SECTION (w  	DebugInfo x     	LockCount y  	RecursionCount z  	OwningThread {  	LockSemaphore |  	SpinCount }    \  PRTL_CRITICAL_SECTION_DEBUG f$   d  RTL_CRITICAL_SECTION ~\  CRITICAL_SECTION *   VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  RPC_IF_HANDLE 
B  tagCOINITBASE      COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )   IWinTypesBase_v0_1_s_ifspec *   IID_IUnknown W  IID_AsyncIUnknown   IID_IClassFactory m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (   IWinTypes_v0_1_s_ifspec )   VARENUM    +2  VT_EMPTY  VT_NULL VT_I2 VT_I4 VT_R4 VT_R8 VT_CY VT_DATE VT_BSTR VT_DISPATCH 	VT_ERROR 
VT_BOOL VT_VARIANT VT_UNKNOWN 
VT_DECIMAL VT_I1 VT_UI1 VT_UI2 VT_UI4 VT_I8 VT_UI8 VT_INT VT_UINT VT_VOID VT_HRESULT VT_PTR VT_SAFEARRAY VT_CARRAY VT_USERDEFINED VT_LPSTR VT_LPWSTR VT_RECORD $VT_INT_PTR %VT_UINT_PTR &VT_FILETIME @VT_BLOB AVT_STREAM BVT_STORAGE CVT_STREAMED_OBJECT DVT_STORED_OBJECT EVT_BLOB_OBJECT FVT_CF GVT_CLSID HVT_VERSIONED_STREAM IVT_BSTR_BLOB VT_VECTOR  VT_ARRAY   VT_BYREF  @VT_RESERVED  VT_ILLEGAL VT_ILLEGALMASKED VT_TYPEMASK  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec    IOLETypes_v0_0_s_ifspec    IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec    IOleAutomationTypes_v1_0_s_ifspec    IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec    __MIDL_itf_msxml_0000_v0_0_s_ifspec    LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST ~M  dwProtocol    cbPciLength    SCARD_IO_REQUEST <M  ~M  g_rgSCardT0Pci %M  g_rgSCardT1Pci %M  g_rgSCardRawPci %M  IID_IPrintDialogCallback   IID_IPrintDialogServices   _PVFV  B  _commode     @ ,N  osfhnd  -    osfile  .   pipech  /   	lockinitflag  0   lock  1G    textmode  2   8 unicode  3    8pipech2  4N  9    N  "   ioinfo  5;N  N  N   N  N  __imp___badioinfo  KN  __imp___pioinfo  PN  _dowildcard  m   _newmode  n   __imp___winitenv  q  __imp___initenv  v  __imp__acmdln  {  __imp__wcmdln    !	     O  __uninitialized  __initializing __initialized  "	   O  O  __native_startup_state  O  __native_startup_lock  :P  @P  #__native_dllmain_reason  
  __native_vcclrit_reason  
  $__onexitbegin P  	0m    N  $__onexitend P  	8m    %
  P  
   __imp__onexit P  P  &atexit 	]   Αm           6Q  'func 8N    (Αm    6Q  )RR  *mingw_onexit !
  ͑m           PR  'func !
  	  +onexitbegin #P  `+onexitend $P  h,retval %
  }	  -͑m    PR  .	Αm    [R  Q  )R8 -Αm    PR  -&Αm    PR  .=Αm    jR  R  )Rs )Q`)Xh -JΑm    uR  -[Αm    uR  .lΑm    R  ?R  )R8 /Αm    )Rs   0       1_lock _lock 0!  !  0       1_unlock _unlock  L   z  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/gccmain.c Αm             char long long unsigned int long long int ptrdiff_t X   wchar_t b   short unsigned int    int long int pthreadlocinfo "  (  threadlocaleinfostruct `  -      	lc_codepage   	lc_collate_cp   	lc_handle 
  	lc_id >  $	lc_category N  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv e  (
ctype1_refcount   0
ctype1 k  8
pctype q  @
pclmap w  H
pcumap w  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct /  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID G  
   	locale    	wlocale   -    	wrefcount             unsigned int )       sizetype long unsigned int   N       ^     lconv ^          unsigned char }  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B)   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   DWORD )  float   __imp__pctype $  k  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N(  __initiallocalestructinfo O/  __imp___mb_cur_max   signed char short int _GUID -  Data1 )   Data2    Data3    Data4 -   }  =     GUID   =  IID R=  N  CLSID Z=  ^  FMTID a=  p  double long double           _sys_errlist   _sys_nerr    __imp___argc   __imp___argv       __imp___wargv    &    __imp__environ   __imp__wenviron    __imp__pgmptr   __imp__wpgmptr &  __imp__fmode   __imp__osplatform    __imp__osver 	  __imp__winver   __imp__winmajor   __imp__winminor $  _amblksiz 5  GUID_MAX_POWER_SAVINGS 	bI  GUID_MIN_POWER_SAVINGS 	cI  GUID_TYPICAL_POWER_SAVINGS 	dI  NO_SUBGROUP_GUID 	eI  ALL_POWERSCHEMES_GUID 	fI  GUID_POWERSCHEME_PERSONALITY 	gI  GUID_ACTIVE_POWERSCHEME 	hI  GUID_IDLE_RESILIENCY_SUBGROUP 	iI  GUID_IDLE_RESILIENCY_PERIOD 	jI  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT 	kI  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT 	lI  GUID_VIDEO_SUBGROUP 	mI  GUID_VIDEO_POWERDOWN_TIMEOUT 	nI  GUID_VIDEO_ANNOYANCE_TIMEOUT 	oI  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE 	pI  GUID_VIDEO_DIM_TIMEOUT 	qI  GUID_VIDEO_ADAPTIVE_POWERDOWN 	rI  GUID_MONITOR_POWER_ON 	sI  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS 	tI  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS 	uI  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS 	vI  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS 	wI  GUID_CONSOLE_DISPLAY_STATE 	xI  GUID_ALLOW_DISPLAY_REQUIRED 	yI  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT 	zI  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP 	{I  GUID_NON_ADAPTIVE_INPUT_TIMEOUT 	|I  GUID_DISK_SUBGROUP 	}I  GUID_DISK_POWERDOWN_TIMEOUT 	~I  GUID_DISK_IDLE_TIMEOUT 	I  GUID_DISK_BURST_IGNORE_THRESHOLD 	I  GUID_DISK_ADAPTIVE_POWERDOWN 	I  GUID_SLEEP_SUBGROUP 	I  GUID_SLEEP_IDLE_THRESHOLD 	I  GUID_STANDBY_TIMEOUT 	I  GUID_UNATTEND_SLEEP_TIMEOUT 	I  GUID_HIBERNATE_TIMEOUT 	I  GUID_HIBERNATE_FASTS4_POLICY 	I  GUID_CRITICAL_POWER_TRANSITION 	I  GUID_SYSTEM_AWAYMODE 	I  GUID_ALLOW_AWAYMODE 	I  GUID_ALLOW_STANDBY_STATES 	I  GUID_ALLOW_RTC_WAKE 	I  GUID_ALLOW_SYSTEM_REQUIRED 	I  GUID_SYSTEM_BUTTON_SUBGROUP 	I  GUID_POWERBUTTON_ACTION 	I  GUID_SLEEPBUTTON_ACTION 	I  GUID_USERINTERFACEBUTTON_ACTION 	I  GUID_LIDCLOSE_ACTION 	I  GUID_LIDOPEN_POWERSTATE 	I  GUID_BATTERY_SUBGROUP 	I  GUID_BATTERY_DISCHARGE_ACTION_0 	I  GUID_BATTERY_DISCHARGE_LEVEL_0 	I  GUID_BATTERY_DISCHARGE_FLAGS_0 	I  GUID_BATTERY_DISCHARGE_ACTION_1 	I  GUID_BATTERY_DISCHARGE_LEVEL_1 	I  GUID_BATTERY_DISCHARGE_FLAGS_1 	I  GUID_BATTERY_DISCHARGE_ACTION_2 	I  GUID_BATTERY_DISCHARGE_LEVEL_2 	I  GUID_BATTERY_DISCHARGE_FLAGS_2 	I  GUID_BATTERY_DISCHARGE_ACTION_3 	I  GUID_BATTERY_DISCHARGE_LEVEL_3 	I  GUID_BATTERY_DISCHARGE_FLAGS_3 	I  GUID_PROCESSOR_SETTINGS_SUBGROUP 	I  GUID_PROCESSOR_THROTTLE_POLICY 	I  GUID_PROCESSOR_THROTTLE_MAXIMUM 	I  GUID_PROCESSOR_THROTTLE_MINIMUM 	I  GUID_PROCESSOR_ALLOW_THROTTLING 	I  GUID_PROCESSOR_IDLESTATE_POLICY 	I  GUID_PROCESSOR_PERFSTATE_POLICY 	I  GUID_PROCESSOR_PERF_INCREASE_THRESHOLD 	I  GUID_PROCESSOR_PERF_DECREASE_THRESHOLD 	I  GUID_PROCESSOR_PERF_INCREASE_POLICY 	I  GUID_PROCESSOR_PERF_DECREASE_POLICY 	I  GUID_PROCESSOR_PERF_INCREASE_TIME 	I  GUID_PROCESSOR_PERF_DECREASE_TIME 	I  GUID_PROCESSOR_PERF_TIME_CHECK 	I  GUID_PROCESSOR_PERF_BOOST_POLICY 	I  GUID_PROCESSOR_PERF_BOOST_MODE 	I  GUID_PROCESSOR_IDLE_ALLOW_SCALING 	I  GUID_PROCESSOR_IDLE_DISABLE 	I  GUID_PROCESSOR_IDLE_STATE_MAXIMUM 	I  GUID_PROCESSOR_IDLE_TIME_CHECK 	I  GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD 	I  GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD 	I  GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD 	I  GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD 	I  GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY 	I  GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY 	I  GUID_PROCESSOR_CORE_PARKING_MAX_CORES 	I  GUID_PROCESSOR_CORE_PARKING_MIN_CORES 	I  GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME 	I  GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME 	I  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR 	I  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD 	I  GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING 	I  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR 	I  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD 	I  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING 	I  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD 	I  GUID_PROCESSOR_PARKING_CORE_OVERRIDE 	I  GUID_PROCESSOR_PARKING_PERF_STATE 	I  GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD 	I  GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD 	I  GUID_PROCESSOR_PERF_HISTORY 	I  GUID_PROCESSOR_PERF_LATENCY_HINT 	I  GUID_PROCESSOR_DISTRIBUTE_UTILITY 	I  GUID_SYSTEM_COOLING_POLICY 	I  GUID_LOCK_CONSOLE_ON_WAKE 	I  GUID_DEVICE_IDLE_POLICY 	I  GUID_ACDC_POWER_SOURCE 	I  GUID_LIDSWITCH_STATE_CHANGE 	I  GUID_BATTERY_PERCENTAGE_REMAINING 	I  GUID_GLOBAL_USER_PRESENCE 	I  GUID_SESSION_DISPLAY_STATUS 	I  GUID_SESSION_USER_PRESENCE 	I  GUID_IDLE_BACKGROUND_TASK 	I  GUID_BACKGROUND_TASK_NOTIFICATION 	I  GUID_APPLAUNCH_BUTTON 	I  GUID_PCIEXPRESS_SETTINGS_SUBGROUP 	I  GUID_PCIEXPRESS_ASPM_POLICY 	I  GUID_ENABLE_SWITCH_FORCED_SHUTDOWN 	I  PPM_PERFSTATE_CHANGE_GUID 	I  PPM_PERFSTATE_DOMAIN_CHANGE_GUID 	I  PPM_IDLESTATE_CHANGE_GUID 	I  PPM_PERFSTATES_DATA_GUID 	I  PPM_IDLESTATES_DATA_GUID 	I  PPM_IDLE_ACCOUNTING_GUID 	I  PPM_IDLE_ACCOUNTING_EX_GUID 	I  PPM_THERMALCONSTRAINT_GUID 	I  PPM_PERFMON_PERFSTATE_GUID 	I  PPM_THERMAL_POLICY_CHANGE_GUID 	I  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

I  RPC_IF_HANDLE B  tagCOINITBASE   6  COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )  IWinTypesBase_v0_1_s_ifspec *  IID_IUnknown 
WI  IID_AsyncIUnknown 
I  IID_IClassFactory 
mI  IID_IMarshal iI  IID_INoMarshal PI  IID_IAgileObject I  IID_IMarshal2 I  IID_IMalloc VI  IID_IStdMarshalInfo 
I  IID_IExternalConnection pI  IID_IMultiQI I  IID_AsyncIMultiQI BI  IID_IInternalUnknown I  IID_IEnumUnknown I  IID_IEnumString I  IID_ISequentialStream FI  IID_IStream I  IID_IRpcChannelBuffer 5	I  IID_IRpcChannelBuffer2 	I  IID_IAsyncRpcChannelBuffer a
I  IID_IRpcChannelBuffer3 #I  IID_IRpcSyntaxNegotiate =I  IID_IRpcProxyBuffer I  IID_IRpcStubBuffer I  IID_IPSFactoryBuffer 
I  IID_IChannelHook CI  IID_IClientSecurity gI  IID_IServerSecurity I  IID_IRpcOptions I  IID_IGlobalOptions RI  IID_ISurrogate I  IID_IGlobalInterfaceTable -I  IID_ISynchronize I  IID_ISynchronizeHandle 0I  IID_ISynchronizeEvent I  IID_ISynchronizeContainer I  IID_ISynchronizeMutex SI  IID_ICancelMethodCalls I  IID_IAsyncManager .I  IID_ICallFactory I  IID_IRpcHelper 
I  IID_IReleaseMarshalBuffers uI  IID_IWaitMultiple I  IID_IAddrTrackingControl <I  IID_IAddrExclusionControl I  IID_IPipeByte I  IID_IPipeLong }I  IID_IPipeDouble I  IID_IComThreadingInfo I  IID_IProcessInitControl VI  IID_IFastRundown I  IID_IMarshalingStream I  IID_ICallbackWithNoReentrancyToApplicationSTA I  GUID_NULL 
Y  CATID_MARSHALER Y  IID_IRpcChannel Y  IID_IRpcStub Y  IID_IStubManager Y  IID_IRpcProxy Y  IID_IProxyManager Y  IID_IPSFactory Y  IID_IInternalMoniker Y  IID_IDfReserved1 Y  IID_IDfReserved2 Y  IID_IDfReserved3 Y  CLSID_StdMarshal k  CLSID_AggStdMarshal k  CLSID_StdAsyncActManager k  IID_IStub Y  IID_IProxy Y  IID_IEnumGeneric Y  IID_IEnumHolder Y  IID_IEnumCallback  Y  IID_IOleManager !Y  IID_IOlePresObj "Y  IID_IDebug #Y  IID_IDebugStream $Y  CLSID_PSGenObject %k  CLSID_PSClientSite &k  CLSID_PSClassObject 'k  CLSID_PSInPlaceActive (k  CLSID_PSInPlaceFrame )k  CLSID_PSDragDrop *k  CLSID_PSBindCtx +k  CLSID_PSEnumerators ,k  CLSID_StaticMetafile -k  CLSID_StaticDib .k  CID_CDfsVolume /k  CLSID_DCOMAccessControl 0k  CLSID_StdGlobalInterfaceTable 1k  CLSID_ComBinding 2k  CLSID_StdEvent 3k  CLSID_ManualResetEvent 4k  CLSID_SynchronizeContainer 5k  CLSID_AddrControl 6k  CLSID_CCDFormKrnl 7k  CLSID_CCDPropertyPage 8k  CLSID_CCDFormDialog 9k  CLSID_CCDCommandButton :k  CLSID_CCDComboBox ;k  CLSID_CCDTextBox <k  CLSID_CCDCheckBox =k  CLSID_CCDLabel >k  CLSID_CCDOptionButton ?k  CLSID_CCDListBox @k  CLSID_CCDScrollBar Ak  CLSID_CCDGroupBox Bk  CLSID_CCDGeneralPropertyPage Ck  CLSID_CCDGenericPropertyPage Dk  CLSID_CCDFontPropertyPage Ek  CLSID_CCDColorPropertyPage Fk  CLSID_CCDLabelPropertyPage Gk  CLSID_CCDCheckBoxPropertyPage Hk  CLSID_CCDTextBoxPropertyPage Ik  CLSID_CCDOptionButtonPropertyPage Jk  CLSID_CCDListBoxPropertyPage Kk  CLSID_CCDCommandButtonPropertyPage Lk  CLSID_CCDComboBoxPropertyPage Mk  CLSID_CCDScrollBarPropertyPage Nk  CLSID_CCDGroupBoxPropertyPage Ok  CLSID_CCDXObjectPropertyPage Pk  CLSID_CStdPropertyFrame Qk  CLSID_CFormPropertyPage Rk  CLSID_CGridPropertyPage Sk  CLSID_CWSJArticlePage Tk  CLSID_CSystemPage Uk  CLSID_IdentityUnmarshal Vk  CLSID_InProcFreeMarshaler Wk  CLSID_Picture_Metafile Xk  CLSID_Picture_EnhMetafile Yk  CLSID_Picture_Dib Zk  GUID_TRISTATE [I  IWinTypes_v0_1_c_ifspec (  IWinTypes_v0_1_s_ifspec )  VARENUM    /  VT_EMPTY  VT_NULL VT_I2 VT_I4 VT_R4 VT_R8 VT_CY VT_DATE VT_BSTR VT_DISPATCH 	VT_ERROR 
VT_BOOL VT_VARIANT VT_UNKNOWN 
VT_DECIMAL VT_I1 VT_UI1 VT_UI2 VT_UI4 VT_I8 VT_UI8 VT_INT VT_UINT VT_VOID VT_HRESULT VT_PTR VT_SAFEARRAY VT_CARRAY VT_USERDEFINED VT_LPSTR VT_LPWSTR VT_RECORD $VT_INT_PTR %VT_UINT_PTR &VT_FILETIME @VT_BLOB AVT_STREAM BVT_STORAGE CVT_STREAMED_OBJECT DVT_STORED_OBJECT EVT_BLOB_OBJECT FVT_CF GVT_CLSID HVT_VERSIONED_STREAM IVT_BSTR_BLOB VT_VECTOR  VT_ARRAY   VT_BYREF  @VT_RESERVED  VT_ILLEGAL VT_ILLEGALMASKED VT_TYPEMASK  IID_IMallocSpy I  IID_IBindCtx :I  IID_IEnumMoniker J I  IID_IRunnableObject  I  IID_IRunningObjectTable !I  IID_IPersist i"I  IID_IPersistStream "I  IID_IMoniker j#I  IID_IROTData X%I  IID_IEnumSTATSTG %I  IID_IStorage X&I  IID_IPersistFile A(I  IID_IPersistStorage (I  IID_ILockBytes )I  IID_IEnumFORMATETC *I  IID_IEnumSTATDATA l+I  IID_IRootStorage ,I  IID_IAdviseSink ,I  IID_AsyncIAdviseSink s-I  IID_IAdviseSink2 .I  IID_AsyncIAdviseSink2 ./I  IID_IDataObject /I  IID_IDataAdviseHolder 1I  IID_IMessageFilter 1I  FMTID_SummaryInformation ]2}  FMTID_DocSummaryInformation _2}  FMTID_UserDefinedProperties a2}  FMTID_DiscardableInformation c2}  FMTID_ImageSummaryInformation e2}  FMTID_AudioSummaryInformation g2}  FMTID_VideoSummaryInformation i2}  FMTID_MediaFileSummaryInformation k2}  IID_IClassActivator s2I  IID_IFillLockBytes 2I  IID_IProgressNotify 3I  IID_ILayoutStorage 3I  IID_IBlockingLock 4I  IID_ITimeAndNoticeControl 4I  IID_IOplockStorage N5I  IID_IDirectWriterLock 5I  IID_IUrlMon M6I  IID_IForegroundTransfer 6I  IID_IThumbnailExtractor 7I  IID_IDummyHICONIncluder 7I  IID_IProcessLock 7I  IID_ISurrogateService H8I  IID_IInitializeSpy 8I  IID_IApartmentShutdown 9I  IID_IOleAdviseHolder I  IID_IOleCache bI  IID_IOleCache2 )I  IID_IOleCacheControl I  IID_IParseDisplayName <I  IID_IOleContainer I  IID_IOleClientSite I  IID_IOleObject I  IOLETypes_v0_0_c_ifspec   IOLETypes_v0_0_s_ifspec   IID_IOleWindow $I  IID_IOleLink I  IID_IOleItemContainer I  IID_IOleInPlaceUIWindow v	I  IID_IOleInPlaceActiveObject 
I  IID_IOleInPlaceFrame 
I  IID_IOleInPlaceObject I  IID_IOleInPlaceSite I  IID_IContinue 
I  IID_IViewObject 
I  IID_IViewObject2 *I  IID_IDropSource I  IID_IDropTarget [I  IID_IDropSourceNotify I  IID_IEnumOLEVERB vI  IID_IServiceProvider MI  IOleAutomationTypes_v1_0_c_ifspec   IOleAutomationTypes_v1_0_s_ifspec   IID_ICreateTypeInfo ;I  IID_ICreateTypeInfo2 bI  IID_ICreateTypeLib I  IID_ICreateTypeLib2 I  IID_IDispatch 	I  IID_IEnumVARIANT 
I  IID_ITypeComp 5I  IID_ITypeInfo I  IID_ITypeInfo2 PI  IID_ITypeLib I  IID_ITypeLib2 =I  IID_ITypeChangeEvents aI  IID_IErrorInfo I  IID_ICreateErrorInfo }I  IID_ISupportErrorInfo  I  IID_ITypeFactory uI  IID_ITypeMarshal I  IID_IRecordInfo I  IID_IErrorLog  I  IID_IPropertyBag zI  __MIDL_itf_msxml_0000_v0_0_c_ifspec   __MIDL_itf_msxml_0000_v0_0_s_ifspec   LIBID_MSXML Y  IID_IXMLDOMImplementation  Y  IID_IXMLDOMNode 'Y  IID_IXMLDOMDocumentFragment Y  IID_IXMLDOMDocument fY  IID_IXMLDOMNodeList uY  IID_IXMLDOMNamedNodeMap Y  IID_IXMLDOMCharacterData Y  IID_IXMLDOMAttribute Y  IID_IXMLDOMElement Y  IID_IXMLDOMText Y  IID_IXMLDOMComment %Y  IID_IXMLDOMProcessingInstruction Y  IID_IXMLDOMCDATASection Y  IID_IXMLDOMDocumentType Y  IID_IXMLDOMNotation Y  IID_IXMLDOMEntity Y  IID_IXMLDOMEntityReference Y  IID_IXMLDOMParseError a	Y  IID_IXTLRuntime 	Y  DIID_XMLDOMDocumentEvents =
Y  CLSID_DOMDocument \
k  CLSID_DOMFreeThreadedDocument `
k  IID_IXMLHttpRequest g
Y  CLSID_XMLHTTPRequest 
k  IID_IXMLDSOControl 
Y  CLSID_XMLDSOControl 
k  IID_IXMLElementCollection Y  IID_IXMLDocument JY  IID_IXMLDocument2 Y  IID_IXMLElement $Y  IID_IXMLElement2 Y  IID_IXMLAttribute Y  IID_IXMLError 
Y  CLSID_XMLDocument .
k  CLSID_SBS_StdURLMoniker ~Y  CLSID_SBS_HttpProtocol Y  CLSID_SBS_FtpProtocol Y  CLSID_SBS_GopherProtocol Y  CLSID_SBS_HttpSProtocol Y  CLSID_SBS_FileProtocol Y  CLSID_SBS_MkProtocol Y  CLSID_SBS_UrlMkBindCtx Y  CLSID_SBS_SoftDistExt Y  CLSID_SBS_CdlProtocol Y  CLSID_SBS_ClassInstallFilter Y  CLSID_SBS_InternetSecurityManager Y  CLSID_SBS_InternetZoneManager Y  IID_IAsyncMoniker Y  CLSID_StdURLMoniker Y  CLSID_HttpProtocol Y  CLSID_FtpProtocol Y  CLSID_GopherProtocol Y  CLSID_HttpSProtocol Y  CLSID_FileProtocol Y  CLSID_MkProtocol Y  CLSID_StdURLProtocol Y  CLSID_UrlMkBindCtx Y  CLSID_CdlProtocol Y  CLSID_ClassInstallFilter Y  IID_IAsyncBindCtx Y  IID_IPersistMoniker PI  IID_IMonikerProp !I  IID_IBindProtocol I  IID_IBinding I  IID_IBindStatusCallback uI  IID_IBindStatusCallbackEx I  IID_IAuthenticate dI  IID_IAuthenticateEx I  IID_IHttpNegotiate AI  IID_IHttpNegotiate2 I  IID_IHttpNegotiate3 ;	I  IID_IWinInetFileStream 	I  IID_IWindowForBindingUI 0
I  IID_ICodeInstall 
I  IID_IUri -I  IID_IUriContainer 
I  IID_IUriBuilder 
I  IID_IUriBuilderFactory (I  IID_IWinInetInfo I  IID_IHttpSecurity I  IID_IWinInetHttpInfo yI  IID_IWinInetHttpTimeouts I  IID_IWinInetCacheHints ZI  IID_IWinInetCacheHints2 I  SID_BindHost 5I  IID_IBindHost ?I  IID_IInternet MI  IID_IInternetBindInfo I  IID_IInternetBindInfoEx &I  IID_IInternetProtocolRoot I  IID_IInternetProtocol I  IID_IInternetProtocolEx WI  IID_IInternetProtocolSink I  IID_IInternetProtocolSinkStackable I  IID_IInternetSession ?I  IID_IInternetThreadSwitch HI  IID_IInternetPriority I  IID_IInternetProtocolInfo NI  CLSID_InternetSecurityManager Y  CLSID_InternetZoneManager Y  CLSID_PersistentZoneIdentifier Y  IID_IInternetSecurityMgrSite I  IID_IInternetSecurityManager iI  IID_IInternetSecurityManagerEx I  IID_IInternetSecurityManagerEx2 VI  IID_IZoneIdentifier  I  IID_IInternetHostSecurityManager !I  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"I  IID_IInternetZoneManager "I  IID_IInternetZoneManagerEx L$I  IID_IInternetZoneManagerEx2 ]%I  CLSID_SoftDistExt &Y  IID_ISoftDistExt &I  IID_ICatalogFileInfo x'I  IID_IDataFilter 'I  IID_IEncodingFilterFactory (I  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)I  IID_IWrappedProtocol A)I  IID_IGetBindHandle )I  IID_IBindCallbackRedirect 
*I  IID_IPropertyStorage I  IID_IPropertySetStorage I  IID_IEnumSTATPROPSTG I  IID_IEnumSTATPROPSETSTG DI  IID_StdOle Y  GUID_DEVINTERFACE_DISK I  GUID_DEVINTERFACE_CDROM 
I  GUID_DEVINTERFACE_PARTITION I  GUID_DEVINTERFACE_TAPE I  GUID_DEVINTERFACE_WRITEONCEDISK I  GUID_DEVINTERFACE_VOLUME I  GUID_DEVINTERFACE_MEDIUMCHANGER I  GUID_DEVINTERFACE_FLOPPY I  GUID_DEVINTERFACE_CDCHANGER I  GUID_DEVINTERFACE_STORAGEPORT I  GUID_DEVINTERFACE_COMPORT I  GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR I  _SCARD_IO_REQUEST J  dwProtocol    cbPciLength    SCARD_IO_REQUEST J  J  g_rgSCardT0Pci %J  g_rgSCardT1Pci %J  g_rgSCardRawPci %J  IID_IPrintDialogCallback I  IID_IPrintDialogServices I  func_ptr   |K  K   __CTOR_LIST__ K  __DTOR_LIST__ 
K  initialized 2   	Pm    __main 5`ϑm           L  ϑm    L    __do_global_ctors  Αm    f       qL  !nptrs ")  	  !i #)  5
  "=ϑm    L  #R	Αm       __do_global_dtors Αm    5       L  p L  	m     |K  $atexit atexit ] P   V  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/natstart.c 
  char long long unsigned int long long int intptr_t >   wchar_t b   short unsigned int    int long int pthreadlocinfo     threadlocaleinfostruct `  N      	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id 3  $	lc_category C  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv Z  (
ctype1_refcount   0
ctype1 `  8
pctype f  @
pclmap l  H
pcumap l  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct   	locinfo     	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID 7  
   	locale    	wlocale   N    	wrefcount    {         unsigned int          sizetype long unsigned int   C       S     lconv S          unsigned char r  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WORD    DWORD   float   __imp__pctype $  `  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J   __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N  __initiallocalestructinfo O  __imp___mb_cur_max   signed char short int ULONG_PTR 1   LONG    HANDLE z  _LIST_ENTRY ]H  	Flink ^H   	Blink _H     LIST_ENTRY `  _GUID   Data1    Data2    Data3    Data4    r       GUID a    IID R    CLSID Z    FMTID a    double long double   &      _sys_errlist   _sys_nerr    __imp___argc   __imp___argv s  y    __imp___wargv Ĕ      __imp__environ s  __imp__wenviron ٔ  __imp__pgmptr y  __imp__wpgmptr   __imp__fmode   __imp__osplatform    __imp__osver 	  __imp__winver   __imp__winmajor   __imp__winminor $  _amblksiz 	5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   _RTL_CRITICAL_SECTION_DEBUG 0\)  	Type ]|   	CreatorBackTraceIndex ^|  	CriticalSection _  	ProcessLocksList `N  	EntryCount a   	ContentionCount b  $	Flags c  (	CreatorBackTraceIndexHigh d|  ,	SpareWORD e|  . _RTL_CRITICAL_SECTION (w  	DebugInfo x   	LockCount y  	RecursionCount z  	OwningThread {  	LockSemaphore |  	SpinCount }    )  PRTL_CRITICAL_SECTION_DEBUG f  1  RTL_CRITICAL_SECTION ~)  CRITICAL_SECTION 
  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  RPC_IF_HANDLE Bz  tagCOINITBASE      COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec 
)Z   IWinTypesBase_v0_1_s_ifspec 
*Z   IID_IUnknown W  IID_AsyncIUnknown   IID_IClassFactory m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (Z   IWinTypes_v0_1_s_ifspec )Z   VARENUM    1  VT_EMPTY  VT_NULL VT_I2 VT_I4 VT_R4 VT_R8 VT_CY VT_DATE VT_BSTR VT_DISPATCH 	VT_ERROR 
VT_BOOL VT_VARIANT VT_UNKNOWN 
VT_DECIMAL VT_I1 VT_UI1 VT_UI2 VT_UI4 VT_I8 VT_UI8 VT_INT VT_UINT VT_VOID VT_HRESULT VT_PTR VT_SAFEARRAY VT_CARRAY VT_USERDEFINED VT_LPSTR VT_LPWSTR VT_RECORD $VT_INT_PTR %VT_UINT_PTR &VT_FILETIME @VT_BLOB AVT_STREAM BVT_STORAGE CVT_STREAMED_OBJECT DVT_STORED_OBJECT EVT_BLOB_OBJECT FVT_CF GVT_CLSID HVT_VERSIONED_STREAM IVT_BSTR_BLOB VT_VECTOR  VT_ARRAY   VT_BYREF  @VT_RESERVED  VT_ILLEGAL VT_ILLEGALMASKED VT_TYPEMASK  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec Z   IOLETypes_v0_0_s_ifspec Z   IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec Z   IOleAutomationTypes_v1_0_s_ifspec Z   IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec Z   __MIDL_itf_msxml_0000_v0_0_s_ifspec Z   LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST KM  dwProtocol    cbPciLength    SCARD_IO_REQUEST 	M  KM  g_rgSCardT0Pci %cM  g_rgSCardT1Pci %cM  g_rgSCardRawPci %cM  IID_IPrintDialogCallback   IID_IPrintDialogServices   _commode    @,N  osfhnd -    osfile .{   pipech /{   	lockinitflag 0   lock 1   textmode 2{   8unicode 3{    8pipech2 4N  9 {   N     ioinfo 5M  N  N   N  N  __imp___badioinfo KN  __imp___pioinfo PN  _dowildcard m   _newmode n   __imp___winitenv q  __imp___initenv vs  __imp__acmdln {y  __imp__wcmdln y  6    O  __uninitialized  __initializing __initialized   6  kO  O  __native_startup_state O  __native_startup_lock O   P  !__native_dllmain_reason   __native_vcclrit_reason   "P   	m    " P   	m    "O   
	@m    "O   	Hm     q+   	  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/gs_support.c ϑm          '
  char long long unsigned int long long int wchar_t b   short unsigned int    int long int pthreadlocinfo     threadlocaleinfostruct `  T      	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id 0  $	lc_category @  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv W  (
ctype1_refcount   0
ctype1 ]  8
pctype c  @
pclmap i  H
pcumap i  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct !  	locinfo     	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID 9  
   	locale    	wlocale   T    	wrefcount             unsigned int        sizetype long unsigned int   @       P     lconv P          unsigned char o  __lc_time_data   double float long double __imp___mb_cur_max r          _sys_errlist   _sys_nerr    __imp___argc   __imp___argv 4  :    __imp___wargv U  [    __imp__environ 4  __imp__wenviron U  __imp__pgmptr :  __imp__wpgmptr [  __imp__fmode   __imp__osplatform      __imp__osver 	  __imp__winver   __imp__winmajor   __imp__winminor $  _amblksiz 5  _PHNDLR ?h  n  y      _XCPT_ACTION A  XcptNum B   SigNum C   XcptAction DY   y     _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   ;  _EXCEPTION_RECORD 	  	ExceptionCode 	
   	ExceptionFlags 	
    	5  	ExceptionAddress 	l  	NumberParameters 	
  	ExceptionInformation 	      _CONTEXT 
  	P1Home ]   	P2Home ]  	P3Home ]  	P4Home ]  	P5Home ]   	P6Home ]  (	ContextFlags 
  0	MxCsr 
  4	SegCs 
  8	SegDs 
  :	SegEs 
  <	SegFs 
  >	SegGs 
  @	SegSs 
  B	EFlags 
  D	Dr0 ]  H	Dr1 ]  P	Dr2 ]  X	Dr3 ]  `	Dr6 ]  h	Dr7 ]  p	Rax ]  x	Rcx ]  	Rdx ]  	Rbx ]  	Rsp ]  	Rbp ]  	Rsi ]  	Rdi ]  	R8 ]  	R9 ]  	R10 ]  	R11 ]  	R12 ]  	R13 ]  	R14 ]  	R15 ]  	Rip ]     
VectorRegister    
VectorControl ]  
DebugControl ]  
LastBranchToRip ]  
LastBranchFromRip ]  
LastExceptionToRip ]  
LastExceptionFromRip ]   BYTE o  WORD    DWORD   __imp__pctype $  ]  __imp__wctype 3  __imp__pwctype ?    P   E  __newclmap HP  __newcumap IP  __ptlocinfo J   __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N  __initiallocalestructinfo O!  signed char short int UINT_PTR 	/   (  ULONG_PTR 	1   ULONG64 	   DWORD64 	   PVOID   LONG    LONGLONG    ULONGLONG    
  	LowPart 
     y   
  	LowPart 
     y   _LARGE_INTEGER 2
    u   QuadPart    LARGE_INTEGER   _GUID 

  Data1 
   Data2 
   Data3 
   Data4 

   o  
     GUID 
H
  
  _M128A G
  	Low H   	High I   M128A J
  
  
     
  
     
      _ _XMM_SAVE_AREA32  jy  	ControlWord k
   	StatusWord l
  	TagWord m
  	Reserved1 n
  	ErrorOpcode o
  	ErrorOffset p
  	ErrorSelector q
  	Reserved2 r
  	DataOffset s
  	DataSelector t
  	Reserved3 u
  	MxCsr v
  	MxCsr_Mask w
  	FloatRegisters x
   	XmmRegisters y
  
Reserved4 z
   XMM_SAVE_AREA32 {    	Header    	Legacy 
   	Xmm0 
  	Xmm1 
  	Xmm2 
  	Xmm3 
  	Xmm4 
  	Xmm5 
  
Xmm6 
   
Xmm7 
  
Xmm8 
   
Xmm9 
  0
Xmm10 
  @
Xmm11 
  P
Xmm12 
  `
Xmm13 
  p
Xmm14 
  
Xmm15 
   
          FltSave y  FloatSave y     
       CONTEXT   PCONTEXT   _RUNTIME_FUNCTION y  	BeginAddress 
   	EndAddress 
  	UnwindData 
   PRUNTIME_FUNCTION     =       EXCEPTION_RECORD 	;  PEXCEPTION_RECORD 	    _EXCEPTION_POINTERS 	$    	   	ContextRecord 	   EXCEPTION_POINTERS 	  $  GUID_MAX_POWER_SAVINGS b
  GUID_MIN_POWER_SAVINGS c
  GUID_TYPICAL_POWER_SAVINGS d
  NO_SUBGROUP_GUID e
  ALL_POWERSCHEMES_GUID f
  GUID_POWERSCHEME_PERSONALITY g
  GUID_ACTIVE_POWERSCHEME h
  GUID_IDLE_RESILIENCY_SUBGROUP i
  GUID_IDLE_RESILIENCY_PERIOD j
  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k
  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l
  GUID_VIDEO_SUBGROUP m
  GUID_VIDEO_POWERDOWN_TIMEOUT n
  GUID_VIDEO_ANNOYANCE_TIMEOUT o
  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p
  GUID_VIDEO_DIM_TIMEOUT q
  GUID_VIDEO_ADAPTIVE_POWERDOWN r
  GUID_MONITOR_POWER_ON s
  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t
  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u
  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v
  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w
  GUID_CONSOLE_DISPLAY_STATE x
  GUID_ALLOW_DISPLAY_REQUIRED y
  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z
  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {
  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |
  GUID_DISK_SUBGROUP }
  GUID_DISK_POWERDOWN_TIMEOUT ~
  GUID_DISK_IDLE_TIMEOUT 
  GUID_DISK_BURST_IGNORE_THRESHOLD 
  GUID_DISK_ADAPTIVE_POWERDOWN 
  GUID_SLEEP_SUBGROUP 
  GUID_SLEEP_IDLE_THRESHOLD 
  GUID_STANDBY_TIMEOUT 
  GUID_UNATTEND_SLEEP_TIMEOUT 
  GUID_HIBERNATE_TIMEOUT 
  GUID_HIBERNATE_FASTS4_POLICY 
  GUID_CRITICAL_POWER_TRANSITION 
  GUID_SYSTEM_AWAYMODE 
  GUID_ALLOW_AWAYMODE 
  GUID_ALLOW_STANDBY_STATES 
  GUID_ALLOW_RTC_WAKE 
  GUID_ALLOW_SYSTEM_REQUIRED 
  GUID_SYSTEM_BUTTON_SUBGROUP 
  GUID_POWERBUTTON_ACTION 
  GUID_SLEEPBUTTON_ACTION 
  GUID_USERINTERFACEBUTTON_ACTION 
  GUID_LIDCLOSE_ACTION 
  GUID_LIDOPEN_POWERSTATE 
  GUID_BATTERY_SUBGROUP 
  GUID_BATTERY_DISCHARGE_ACTION_0 
  GUID_BATTERY_DISCHARGE_LEVEL_0 
  GUID_BATTERY_DISCHARGE_FLAGS_0 
  GUID_BATTERY_DISCHARGE_ACTION_1 
  GUID_BATTERY_DISCHARGE_LEVEL_1 
  GUID_BATTERY_DISCHARGE_FLAGS_1 
  GUID_BATTERY_DISCHARGE_ACTION_2 
  GUID_BATTERY_DISCHARGE_LEVEL_2 
  GUID_BATTERY_DISCHARGE_FLAGS_2 
  GUID_BATTERY_DISCHARGE_ACTION_3 
  GUID_BATTERY_DISCHARGE_LEVEL_3 
  GUID_BATTERY_DISCHARGE_FLAGS_3 
  GUID_PROCESSOR_SETTINGS_SUBGROUP 
  GUID_PROCESSOR_THROTTLE_POLICY 
  GUID_PROCESSOR_THROTTLE_MAXIMUM 
  GUID_PROCESSOR_THROTTLE_MINIMUM 
  GUID_PROCESSOR_ALLOW_THROTTLING 
  GUID_PROCESSOR_IDLESTATE_POLICY 
  GUID_PROCESSOR_PERFSTATE_POLICY 
  GUID_PROCESSOR_PERF_INCREASE_THRESHOLD 
  GUID_PROCESSOR_PERF_DECREASE_THRESHOLD 
  GUID_PROCESSOR_PERF_INCREASE_POLICY 
  GUID_PROCESSOR_PERF_DECREASE_POLICY 
  GUID_PROCESSOR_PERF_INCREASE_TIME 
  GUID_PROCESSOR_PERF_DECREASE_TIME 
  GUID_PROCESSOR_PERF_TIME_CHECK 
  GUID_PROCESSOR_PERF_BOOST_POLICY 
  GUID_PROCESSOR_PERF_BOOST_MODE 
  GUID_PROCESSOR_IDLE_ALLOW_SCALING 
  GUID_PROCESSOR_IDLE_DISABLE 
  GUID_PROCESSOR_IDLE_STATE_MAXIMUM 
  GUID_PROCESSOR_IDLE_TIME_CHECK 
  GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD 
  GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY 
  GUID_PROCESSOR_CORE_PARKING_MAX_CORES 
  GUID_PROCESSOR_CORE_PARKING_MIN_CORES 
  GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME 
  GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING 
  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD 
  GUID_PROCESSOR_PARKING_CORE_OVERRIDE 
  GUID_PROCESSOR_PARKING_PERF_STATE 
  GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD 
  GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD 
  GUID_PROCESSOR_PERF_HISTORY 
  GUID_PROCESSOR_PERF_LATENCY_HINT 
  GUID_PROCESSOR_DISTRIBUTE_UTILITY 
  GUID_SYSTEM_COOLING_POLICY 
  GUID_LOCK_CONSOLE_ON_WAKE 
  GUID_DEVICE_IDLE_POLICY 
  GUID_ACDC_POWER_SOURCE 
  GUID_LIDSWITCH_STATE_CHANGE 
  GUID_BATTERY_PERCENTAGE_REMAINING 
  GUID_GLOBAL_USER_PRESENCE 
  GUID_SESSION_DISPLAY_STATUS 
  GUID_SESSION_USER_PRESENCE 
  GUID_IDLE_BACKGROUND_TASK 
  GUID_BACKGROUND_TASK_NOTIFICATION 
  GUID_APPLAUNCH_BUTTON 
  GUID_PCIEXPRESS_SETTINGS_SUBGROUP 
  GUID_PCIEXPRESS_ASPM_POLICY 
  GUID_ENABLE_SWITCH_FORCED_SHUTDOWN 
  PPM_PERFSTATE_CHANGE_GUID 
  PPM_PERFSTATE_DOMAIN_CHANGE_GUID 
  PPM_IDLESTATE_CHANGE_GUID 
  PPM_PERFSTATES_DATA_GUID 
  PPM_IDLESTATES_DATA_GUID 
  PPM_IDLE_ACCOUNTING_GUID 
  PPM_IDLE_ACCOUNTING_EX_GUID 
  PPM_THERMALCONSTRAINT_GUID 
  PPM_PERFMON_PERFSTATE_GUID 
  PPM_THERMAL_POLICY_CHANGE_GUID 
  _FILETIME #'  dwLowDateTime 
   dwHighDateTime 
   FILETIME &  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

  NTSTATUS y  '   ft_scalar !    ft_struct "#'   FT #q'  !GS_ExceptionRecord %  	@m    !GS_ContextRecord &  	`m    !GS_ExceptionPointers (?  	Am    "__security_cookie ,(  	m    "__security_cookie_complement -(  	 m    #__report_gsfailure h`Бm           *  $StackCookie h=  k
  !cookie j*  P!controlPC lN  S!imgBase lN  !establisherFrame lN  @%fctEntry my  
  !hndData nl  H&zБm    *  =)  'R	`m     &Бm    *  `)  'Rs 'Qvh'X0 &Бm    *  )  'R0'Xs 'w 	`m    'w(vx'w0vp'w80 &ёm    +  )  'R0 &'ёm    +  )  'R	Am     (-ёm    +  &;ёm    "+  )  'Q| (@ёm    -+   8  *     *  )__security_init_cookie 2ϑm           *  %cookie 4(  
  !systime 5'  !perfctr 62
  @&ϑm    =+  *  'R (ϑm    H+  (ϑm    S+  (ϑm    ^+  *ϑm    i+  'R@  +    +    +    ,k  k  ,.  .  ,]  ]  
,    
Q-abort abort I,    &,W  W  
,    
,G  G  L,     "   I  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/tlssup.c `ёm             char long long unsigned int long long int uintptr_t K   wchar_t b   short unsigned int    int long int pthreadlocinfo !  '  threadlocaleinfostruct `  x      	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id =  $	lc_category M  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv d  (
ctype1_refcount   0
ctype1 j  8
pctype p  @
pclmap v  H
pcumap v  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct .  	locinfo 
   	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID F  
   	locale    	wlocale   x    	wrefcount             unsigned int (       sizetype long unsigned int   M       ]     lconv ]          unsigned char |  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B(   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   ULONG (  BOOL    DWORD (  float LPVOID     __imp__pctype $  j  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J
  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N'  __initiallocalestructinfo O.  __imp___mb_cur_max   signed char short int ULONG_PTR 1   PVOID   HANDLE   ULONGLONG    _GUID   Data1 (   Data2    Data3    Data4    |       GUID K    double long double           _sys_errlist 	  _sys_nerr 	   __imp___argc 	  __imp___argv 	0  6    __imp___wargv 	Q  W    __imp__environ 	0  __imp__wenviron 	Q  __imp__pgmptr 	6  __imp__wpgmptr 	W  __imp__fmode 	  __imp__osplatform 	   __imp__osver 		  __imp__winver 	  __imp__winmajor 	  __imp__winminor 	$  _amblksiz 
5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   PIMAGE_TLS_CALLBACK       *         _IMAGE_TLS_DIRECTORY64 (  	StartAddressOfRawData 9   	EndAddressOfRawData 9  	AddressOfIndex 9  	AddressOfCallBacks 9  	SizeOfZeroFill    	Characteristics   $ IMAGE_TLS_DIRECTORY64 *  IMAGE_TLS_DIRECTORY 2  
  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  _PVFV   _tls_index )  	왒m    _tls_start /  	 Вm    _tls_end 0  	`Вm    __xl_a 2  	(m    __xl_z 3  	@m    _tls_used 5&  	 Вm    __xd_a EY  	Hm    __xd_z FY  	Pm    _CRT_MT M   __dyn_tls_init_callback m
  	Am    __xl_c n  	0m    __xl_d   	8m    mingw_initltsdrot_force    	虒m    mingw_initltsdyn_force    	䙒m    mingw_initltssuo_force    	m    __dyn_tls_dtor   `ёm    /       {!    *  D  o    }        ёm    "   __tlregdtor s    ґm           !  func sY  R  __dyn_tls_init R  "  !  R*  !o  R  !  R  "pfunc T"  "ps U    Y  #!  ёm    i       "  $!    $!  z
  $!    %!  %!  &ёm    &       "  $!    $!    $!    'ёm    &       (C"  &  (H"  ^    ёm    "   )        d  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/cinitexe.c   char long long unsigned int long long int short unsigned int int long int unsigned int sizetype long unsigned int unsigned char _PVFV &  ,    =       __xi_a 
-  	m    __xi_z -  	 m    __xc_a -  	 m    __xc_z 
-  	m     7S     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/mingw_helpers.c ґm           
  char long long unsigned int long long int intptr_t >   uintptr_t K   wchar_t b   short unsigned int    int long int pthreadlocinfo 8  >  threadlocaleinfostruct `       	lc_codepage   	lc_collate_cp   	lc_handle (  	lc_id Y  $	lc_category i  H
lc_clike   
mb_cur_max   
lconv_intl_refcount 
  
lconv_num_refcount 
  
lconv_mon_refcount 
   
lconv   (
ctype1_refcount 
  0
ctype1   8
pctype $  @
pclmap *  H
pcumap *  P
lc_time_curr V  X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct E  	locinfo !   	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID ]  
   	locale    	wlocale     
  	wrefcount 
           unsigned int   D  8  8   sizetype long unsigned int   i  8     y  8   lconv X-  decimal_point .   thousands_sep /  grouping 0  int_curr_symbol 1  currency_symbol 2   mon_decimal_point 3  (mon_thousands_sep 4  0mon_grouping 5  8positive_sign 6  @negative_sign 7  Hint_frac_digits 8   Pfrac_digits 9   Qp_cs_precedes :   Rp_sep_by_space ;   Sn_cs_precedes <   Tn_sep_by_space =   Up_sign_posn >   Vn_sign_posn ?   W y     	  A  unsigned char 0  __lc_time_data F  _PHNDLR ?k  q  |     _XCPT_ACTION A  XcptNum BD   SigNum C  XcptAction D\   |     _XcptActTab G  _XcptActTabCount H  _XcptActTabSize I  _First_FPE_Indx J  _Num_FPE K  WORD    DWORD D  float   __imp__pctype $w    __imp__wctype 3w  __imp__pwctype ?w  A       __newclmap H  __newcumap I  __ptlocinfo J!  __ptmbcinfo K  __globallocalestatus L  __locale_changed M  __initiallocinfo N>  __initiallocalestructinfo OE  __imp___mb_cur_max 
  signed char short int ULONG_PTR 1   PVOID 8  LONG   HANDLE 8  _LIST_ENTRY ]	  	Flink ^	   	Blink _	     LIST_ENTRY `  _GUID 	s	  Data1 	D   Data2 	   Data3 	   Data4 	s	   0  	  8   GUID 	,	  	  IID 	R	  	  CLSID 	Z	  	  FMTID 	a	  	  double long double   	  8    _sys_errlist 
	  _sys_nerr 
  __imp___argc 

  __imp___argv 
>
  D
    __imp___wargv 
_
  e
    __imp__environ 
>
  __imp__wenviron 
_
  __imp__pgmptr 
D
  __imp__wpgmptr 
e
  __imp__fmode 

  __imp__osplatform 
 \  __imp__osver 
	\  __imp__winver 
\  __imp__winmajor 
\  __imp__winminor 
$\  _amblksiz 5  GUID_MAX_POWER_SAVINGS b	  GUID_MIN_POWER_SAVINGS c	  GUID_TYPICAL_POWER_SAVINGS d	  NO_SUBGROUP_GUID e	  ALL_POWERSCHEMES_GUID f	  GUID_POWERSCHEME_PERSONALITY g	  GUID_ACTIVE_POWERSCHEME h	  GUID_IDLE_RESILIENCY_SUBGROUP i	  GUID_IDLE_RESILIENCY_PERIOD j	  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k	  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l	  GUID_VIDEO_SUBGROUP m	  GUID_VIDEO_POWERDOWN_TIMEOUT n	  GUID_VIDEO_ANNOYANCE_TIMEOUT o	  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p	  GUID_VIDEO_DIM_TIMEOUT q	  GUID_VIDEO_ADAPTIVE_POWERDOWN r	  GUID_MONITOR_POWER_ON s	  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t	  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u	  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v	  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w	  GUID_CONSOLE_DISPLAY_STATE x	  GUID_ALLOW_DISPLAY_REQUIRED y	  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z	  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {	  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |	  GUID_DISK_SUBGROUP }	  GUID_DISK_POWERDOWN_TIMEOUT ~	  GUID_DISK_IDLE_TIMEOUT 	  GUID_DISK_BURST_IGNORE_THRESHOLD 	  GUID_DISK_ADAPTIVE_POWERDOWN 	  GUID_SLEEP_SUBGROUP 	  GUID_SLEEP_IDLE_THRESHOLD 	  GUID_STANDBY_TIMEOUT 	  GUID_UNATTEND_SLEEP_TIMEOUT 	  GUID_HIBERNATE_TIMEOUT 	  GUID_HIBERNATE_FASTS4_POLICY 	  GUID_CRITICAL_POWER_TRANSITION 	  GUID_SYSTEM_AWAYMODE 	  GUID_ALLOW_AWAYMODE 	  GUID_ALLOW_STANDBY_STATES 	  GUID_ALLOW_RTC_WAKE 	  GUID_ALLOW_SYSTEM_REQUIRED 	  GUID_SYSTEM_BUTTON_SUBGROUP 	  GUID_POWERBUTTON_ACTION 	  GUID_SLEEPBUTTON_ACTION 	  GUID_USERINTERFACEBUTTON_ACTION 	  GUID_LIDCLOSE_ACTION 	  GUID_LIDOPEN_POWERSTATE 	  GUID_BATTERY_SUBGROUP 	  GUID_BATTERY_DISCHARGE_ACTION_0 	  GUID_BATTERY_DISCHARGE_LEVEL_0 	  GUID_BATTERY_DISCHARGE_FLAGS_0 	  GUID_BATTERY_DISCHARGE_ACTION_1 	  GUID_BATTERY_DISCHARGE_LEVEL_1 	  GUID_BATTERY_DISCHARGE_FLAGS_1 	  GUID_BATTERY_DISCHARGE_ACTION_2 	  GUID_BATTERY_DISCHARGE_LEVEL_2 	  GUID_BATTERY_DISCHARGE_FLAGS_2 	  GUID_BATTERY_DISCHARGE_ACTION_3 	  GUID_BATTERY_DISCHARGE_LEVEL_3 	  GUID_BATTERY_DISCHARGE_FLAGS_3 	  GUID_PROCESSOR_SETTINGS_SUBGROUP 	  GUID_PROCESSOR_THROTTLE_POLICY 	  GUID_PROCESSOR_THROTTLE_MAXIMUM 	  GUID_PROCESSOR_THROTTLE_MINIMUM 	  GUID_PROCESSOR_ALLOW_THROTTLING 	  GUID_PROCESSOR_IDLESTATE_POLICY 	  GUID_PROCESSOR_PERFSTATE_POLICY 	  GUID_PROCESSOR_PERF_INCREASE_THRESHOLD 	  GUID_PROCESSOR_PERF_DECREASE_THRESHOLD 	  GUID_PROCESSOR_PERF_INCREASE_POLICY 	  GUID_PROCESSOR_PERF_DECREASE_POLICY 	  GUID_PROCESSOR_PERF_INCREASE_TIME 	  GUID_PROCESSOR_PERF_DECREASE_TIME 	  GUID_PROCESSOR_PERF_TIME_CHECK 	  GUID_PROCESSOR_PERF_BOOST_POLICY 	  GUID_PROCESSOR_PERF_BOOST_MODE 	  GUID_PROCESSOR_IDLE_ALLOW_SCALING 	  GUID_PROCESSOR_IDLE_DISABLE 	  GUID_PROCESSOR_IDLE_STATE_MAXIMUM 	  GUID_PROCESSOR_IDLE_TIME_CHECK 	  GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD 	  GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD 	  GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD 	  GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD 	  GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY 	  GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY 	  GUID_PROCESSOR_CORE_PARKING_MAX_CORES 	  GUID_PROCESSOR_CORE_PARKING_MIN_CORES 	  GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME 	  GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME 	  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR 	  GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD 	  GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING 	  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR 	  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD 	  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING 	  GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD 	  GUID_PROCESSOR_PARKING_CORE_OVERRIDE 	  GUID_PROCESSOR_PARKING_PERF_STATE 	  GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD 	  GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD 	  GUID_PROCESSOR_PERF_HISTORY 	  GUID_PROCESSOR_PERF_LATENCY_HINT 	  GUID_PROCESSOR_DISTRIBUTE_UTILITY 	  GUID_SYSTEM_COOLING_POLICY 	  GUID_LOCK_CONSOLE_ON_WAKE 	  GUID_DEVICE_IDLE_POLICY 	  GUID_ACDC_POWER_SOURCE 	  GUID_LIDSWITCH_STATE_CHANGE 	  GUID_BATTERY_PERCENTAGE_REMAINING 	  GUID_GLOBAL_USER_PRESENCE 	  GUID_SESSION_DISPLAY_STATUS 	  GUID_SESSION_USER_PRESENCE 	  GUID_IDLE_BACKGROUND_TASK 	  GUID_BACKGROUND_TASK_NOTIFICATION 	  GUID_APPLAUNCH_BUTTON 	  GUID_PCIEXPRESS_SETTINGS_SUBGROUP 	  GUID_PCIEXPRESS_ASPM_POLICY 	  GUID_ENABLE_SWITCH_FORCED_SHUTDOWN 	  PPM_PERFSTATE_CHANGE_GUID 	  PPM_PERFSTATE_DOMAIN_CHANGE_GUID 	  PPM_IDLESTATE_CHANGE_GUID 	  PPM_PERFSTATES_DATA_GUID 	  PPM_IDLESTATES_DATA_GUID 	  PPM_IDLE_ACCOUNTING_GUID 	  PPM_IDLE_ACCOUNTING_EX_GUID 	  PPM_THERMALCONSTRAINT_GUID 	  PPM_PERFMON_PERFSTATE_GUID 	  PPM_THERMAL_POLICY_CHANGE_GUID 	  PIMAGE_TLS_CALLBACK      #   8     F     _RTL_CRITICAL_SECTION_DEBUG 0\0!  	Type ]:   	CreatorBackTraceIndex ^:  	CriticalSection _!  	ProcessLocksList `	  	EntryCount aF   	ContentionCount bF  $	Flags cF  (	CreatorBackTraceIndexHigh d:  ,	SpareWORD e:  . _RTL_CRITICAL_SECTION (w!  	DebugInfo x!   	LockCount y  	RecursionCount z  	OwningThread {  	LockSemaphore |  	SpinCount }    0!  PRTL_CRITICAL_SECTION_DEBUG f!  8   RTL_CRITICAL_SECTION ~0!  CRITICAL_SECTION !  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

	  RPC_IF_HANDLE B8  tagCOINITBASE   "  COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )a"  IWinTypesBase_v0_1_s_ifspec *a"  IID_IUnknown W	  IID_AsyncIUnknown 	  IID_IClassFactory m	  IID_IMarshal i	  IID_INoMarshal P	  IID_IAgileObject 	  IID_IMarshal2 	  IID_IMalloc V	  IID_IStdMarshalInfo 
	  IID_IExternalConnection p	  IID_IMultiQI 	  IID_AsyncIMultiQI B	  IID_IInternalUnknown 	  IID_IEnumUnknown 	  IID_IEnumString 	  IID_ISequentialStream F	  IID_IStream 	  IID_IRpcChannelBuffer 5		  IID_IRpcChannelBuffer2 		  IID_IAsyncRpcChannelBuffer a
	  IID_IRpcChannelBuffer3 #	  IID_IRpcSyntaxNegotiate =	  IID_IRpcProxyBuffer 	  IID_IRpcStubBuffer 	  IID_IPSFactoryBuffer 
	  IID_IChannelHook C	  IID_IClientSecurity g	  IID_IServerSecurity 	  IID_IRpcOptions 	  IID_IGlobalOptions R	  IID_ISurrogate 	  IID_IGlobalInterfaceTable -	  IID_ISynchronize 	  IID_ISynchronizeHandle 0	  IID_ISynchronizeEvent 	  IID_ISynchronizeContainer 	  IID_ISynchronizeMutex S	  IID_ICancelMethodCalls 	  IID_IAsyncManager .	  IID_ICallFactory 	  IID_IRpcHelper 
	  IID_IReleaseMarshalBuffers u	  IID_IWaitMultiple 	  IID_IAddrTrackingControl <	  IID_IAddrExclusionControl 	  IID_IPipeByte 	  IID_IPipeLong }	  IID_IPipeDouble 	  IID_IComThreadingInfo 	  IID_IProcessInitControl V	  IID_IFastRundown 	  IID_IMarshalingStream 	  IID_ICallbackWithNoReentrancyToApplicationSTA 	  GUID_NULL 
	  CATID_MARSHALER 	  IID_IRpcChannel 	  IID_IRpcStub 	  IID_IStubManager 	  IID_IRpcProxy 	  IID_IProxyManager 	  IID_IPSFactory 	  IID_IInternalMoniker 	  IID_IDfReserved1 	  IID_IDfReserved2 	  IID_IDfReserved3 	  CLSID_StdMarshal 	  CLSID_AggStdMarshal 	  CLSID_StdAsyncActManager 	  IID_IStub 	  IID_IProxy 	  IID_IEnumGeneric 	  IID_IEnumHolder 	  IID_IEnumCallback  	  IID_IOleManager !	  IID_IOlePresObj "	  IID_IDebug #	  IID_IDebugStream $	  CLSID_PSGenObject %	  CLSID_PSClientSite &	  CLSID_PSClassObject '	  CLSID_PSInPlaceActive (	  CLSID_PSInPlaceFrame )	  CLSID_PSDragDrop *	  CLSID_PSBindCtx +	  CLSID_PSEnumerators ,	  CLSID_StaticMetafile -	  CLSID_StaticDib .	  CID_CDfsVolume /	  CLSID_DCOMAccessControl 0	  CLSID_StdGlobalInterfaceTable 1	  CLSID_ComBinding 2	  CLSID_StdEvent 3	  CLSID_ManualResetEvent 4	  CLSID_SynchronizeContainer 5	  CLSID_AddrControl 6	  CLSID_CCDFormKrnl 7	  CLSID_CCDPropertyPage 8	  CLSID_CCDFormDialog 9	  CLSID_CCDCommandButton :	  CLSID_CCDComboBox ;	  CLSID_CCDTextBox <	  CLSID_CCDCheckBox =	  CLSID_CCDLabel >	  CLSID_CCDOptionButton ?	  CLSID_CCDListBox @	  CLSID_CCDScrollBar A	  CLSID_CCDGroupBox B	  CLSID_CCDGeneralPropertyPage C	  CLSID_CCDGenericPropertyPage D	  CLSID_CCDFontPropertyPage E	  CLSID_CCDColorPropertyPage F	  CLSID_CCDLabelPropertyPage G	  CLSID_CCDCheckBoxPropertyPage H	  CLSID_CCDTextBoxPropertyPage I	  CLSID_CCDOptionButtonPropertyPage J	  CLSID_CCDListBoxPropertyPage K	  CLSID_CCDCommandButtonPropertyPage L	  CLSID_CCDComboBoxPropertyPage M	  CLSID_CCDScrollBarPropertyPage N	  CLSID_CCDGroupBoxPropertyPage O	  CLSID_CCDXObjectPropertyPage P	  CLSID_CStdPropertyFrame Q	  CLSID_CFormPropertyPage R	  CLSID_CGridPropertyPage S	  CLSID_CWSJArticlePage T	  CLSID_CSystemPage U	  CLSID_IdentityUnmarshal V	  CLSID_InProcFreeMarshaler W	  CLSID_Picture_Metafile X	  CLSID_Picture_EnhMetafile Y	  CLSID_Picture_Dib Z	  GUID_TRISTATE [	  IWinTypes_v0_1_c_ifspec (a"  IWinTypes_v0_1_s_ifspec )a"  VARENUM    3  VT_EMPTY  VT_NULL VT_I2 VT_I4 VT_R4 VT_R8 VT_CY VT_DATE VT_BSTR VT_DISPATCH 	VT_ERROR 
VT_BOOL VT_VARIANT VT_UNKNOWN 
VT_DECIMAL VT_I1 VT_UI1 VT_UI2 VT_UI4 VT_I8 VT_UI8 VT_INT VT_UINT VT_VOID VT_HRESULT VT_PTR VT_SAFEARRAY VT_CARRAY VT_USERDEFINED VT_LPSTR VT_LPWSTR VT_RECORD $VT_INT_PTR %VT_UINT_PTR &VT_FILETIME @VT_BLOB AVT_STREAM BVT_STORAGE CVT_STREAMED_OBJECT DVT_STORED_OBJECT EVT_BLOB_OBJECT FVT_CF GVT_CLSID HVT_VERSIONED_STREAM IVT_BSTR_BLOB VT_VECTOR  VT_ARRAY   VT_BYREF  @VT_RESERVED  VT_ILLEGAL VT_ILLEGALMASKED VT_TYPEMASK  IID_IMallocSpy 	  IID_IBindCtx :	  IID_IEnumMoniker J 	  IID_IRunnableObject  	  IID_IRunningObjectTable !	  IID_IPersist i"	  IID_IPersistStream "	  IID_IMoniker j#	  IID_IROTData X%	  IID_IEnumSTATSTG %	  IID_IStorage X&	  IID_IPersistFile A(	  IID_IPersistStorage (	  IID_ILockBytes )	  IID_IEnumFORMATETC *	  IID_IEnumSTATDATA l+	  IID_IRootStorage ,	  IID_IAdviseSink ,	  IID_AsyncIAdviseSink s-	  IID_IAdviseSink2 .	  IID_AsyncIAdviseSink2 ./	  IID_IDataObject /	  IID_IDataAdviseHolder 1	  IID_IMessageFilter 1	  FMTID_SummaryInformation ]2	  FMTID_DocSummaryInformation _2	  FMTID_UserDefinedProperties a2	  FMTID_DiscardableInformation c2	  FMTID_ImageSummaryInformation e2	  FMTID_AudioSummaryInformation g2	  FMTID_VideoSummaryInformation i2	  FMTID_MediaFileSummaryInformation k2	  IID_IClassActivator s2	  IID_IFillLockBytes 2	  IID_IProgressNotify 3	  IID_ILayoutStorage 3	  IID_IBlockingLock 4	  IID_ITimeAndNoticeControl 4	  IID_IOplockStorage N5	  IID_IDirectWriterLock 5	  IID_IUrlMon M6	  IID_IForegroundTransfer 6	  IID_IThumbnailExtractor 7	  IID_IDummyHICONIncluder 7	  IID_IProcessLock 7	  IID_ISurrogateService H8	  IID_IInitializeSpy 8	  IID_IApartmentShutdown 9	  IID_IOleAdviseHolder 	  IID_IOleCache b	  IID_IOleCache2 )	  IID_IOleCacheControl 	  IID_IParseDisplayName <	  IID_IOleContainer 	  IID_IOleClientSite 	  IID_IOleObject 	  IOLETypes_v0_0_c_ifspec a"  IOLETypes_v0_0_s_ifspec a"  IID_IOleWindow $	  IID_IOleLink 	  IID_IOleItemContainer 	  IID_IOleInPlaceUIWindow v		  IID_IOleInPlaceActiveObject 
	  IID_IOleInPlaceFrame 
	  IID_IOleInPlaceObject 	  IID_IOleInPlaceSite 	  IID_IContinue 
	  IID_IViewObject 
	  IID_IViewObject2 *	  IID_IDropSource 	  IID_IDropTarget [	  IID_IDropSourceNotify 	  IID_IEnumOLEVERB v	  IID_IServiceProvider M	  IOleAutomationTypes_v1_0_c_ifspec a"  IOleAutomationTypes_v1_0_s_ifspec a"  IID_ICreateTypeInfo ;	  IID_ICreateTypeInfo2 b	  IID_ICreateTypeLib 	  IID_ICreateTypeLib2 	  IID_IDispatch 		  IID_IEnumVARIANT 
	  IID_ITypeComp 5	  IID_ITypeInfo 	  IID_ITypeInfo2 P	  IID_ITypeLib 	  IID_ITypeLib2 =	  IID_ITypeChangeEvents a	  IID_IErrorInfo 	  IID_ICreateErrorInfo }	  IID_ISupportErrorInfo  	  IID_ITypeFactory u	  IID_ITypeMarshal 	  IID_IRecordInfo 	  IID_IErrorLog  	  IID_IPropertyBag z	  __MIDL_itf_msxml_0000_v0_0_c_ifspec a"  __MIDL_itf_msxml_0000_v0_0_s_ifspec a"  LIBID_MSXML 	  IID_IXMLDOMImplementation  	  IID_IXMLDOMNode '	  IID_IXMLDOMDocumentFragment 	  IID_IXMLDOMDocument f	  IID_IXMLDOMNodeList u	  IID_IXMLDOMNamedNodeMap 	  IID_IXMLDOMCharacterData 	  IID_IXMLDOMAttribute 	  IID_IXMLDOMElement 	  IID_IXMLDOMText 	  IID_IXMLDOMComment %	  IID_IXMLDOMProcessingInstruction 	  IID_IXMLDOMCDATASection 	  IID_IXMLDOMDocumentType 	  IID_IXMLDOMNotation 	  IID_IXMLDOMEntity 	  IID_IXMLDOMEntityReference 	  IID_IXMLDOMParseError a		  IID_IXTLRuntime 		  DIID_XMLDOMDocumentEvents =
	  CLSID_DOMDocument \
	  CLSID_DOMFreeThreadedDocument `
	  IID_IXMLHttpRequest g
	  CLSID_XMLHTTPRequest 
	  IID_IXMLDSOControl 
	  CLSID_XMLDSOControl 
	  IID_IXMLElementCollection 	  IID_IXMLDocument J	  IID_IXMLDocument2 	  IID_IXMLElement $	  IID_IXMLElement2 	  IID_IXMLAttribute 	  IID_IXMLError 
	  CLSID_XMLDocument .
	  CLSID_SBS_StdURLMoniker ~	  CLSID_SBS_HttpProtocol 	  CLSID_SBS_FtpProtocol 	  CLSID_SBS_GopherProtocol 	  CLSID_SBS_HttpSProtocol 	  CLSID_SBS_FileProtocol 	  CLSID_SBS_MkProtocol 	  CLSID_SBS_UrlMkBindCtx 	  CLSID_SBS_SoftDistExt 	  CLSID_SBS_CdlProtocol 	  CLSID_SBS_ClassInstallFilter 	  CLSID_SBS_InternetSecurityManager 	  CLSID_SBS_InternetZoneManager 	  IID_IAsyncMoniker 	  CLSID_StdURLMoniker 	  CLSID_HttpProtocol 	  CLSID_FtpProtocol 	  CLSID_GopherProtocol 	  CLSID_HttpSProtocol 	  CLSID_FileProtocol 	  CLSID_MkProtocol 	  CLSID_StdURLProtocol 	  CLSID_UrlMkBindCtx 	  CLSID_CdlProtocol 	  CLSID_ClassInstallFilter 	  IID_IAsyncBindCtx 	  IID_IPersistMoniker P	  IID_IMonikerProp !	  IID_IBindProtocol 	  IID_IBinding 	  IID_IBindStatusCallback u	  IID_IBindStatusCallbackEx 	  IID_IAuthenticate d	  IID_IAuthenticateEx 	  IID_IHttpNegotiate A	  IID_IHttpNegotiate2 	  IID_IHttpNegotiate3 ;		  IID_IWinInetFileStream 		  IID_IWindowForBindingUI 0
	  IID_ICodeInstall 
	  IID_IUri -	  IID_IUriContainer 
	  IID_IUriBuilder 
	  IID_IUriBuilderFactory (	  IID_IWinInetInfo 	  IID_IHttpSecurity 	  IID_IWinInetHttpInfo y	  IID_IWinInetHttpTimeouts 	  IID_IWinInetCacheHints Z	  IID_IWinInetCacheHints2 	  SID_BindHost 5	  IID_IBindHost ?	  IID_IInternet M	  IID_IInternetBindInfo 	  IID_IInternetBindInfoEx &	  IID_IInternetProtocolRoot 	  IID_IInternetProtocol 	  IID_IInternetProtocolEx W	  IID_IInternetProtocolSink 	  IID_IInternetProtocolSinkStackable 	  IID_IInternetSession ?	  IID_IInternetThreadSwitch H	  IID_IInternetPriority 	  IID_IInternetProtocolInfo N	  CLSID_InternetSecurityManager 	  CLSID_InternetZoneManager 	  CLSID_PersistentZoneIdentifier 	  IID_IInternetSecurityMgrSite 	  IID_IInternetSecurityManager i	  IID_IInternetSecurityManagerEx 	  IID_IInternetSecurityManagerEx2 V	  IID_IZoneIdentifier  	  IID_IInternetHostSecurityManager !	  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"	  IID_IInternetZoneManager "	  IID_IInternetZoneManagerEx L$	  IID_IInternetZoneManagerEx2 ]%	  CLSID_SoftDistExt &	  IID_ISoftDistExt &	  IID_ICatalogFileInfo x'	  IID_IDataFilter '	  IID_IEncodingFilterFactory (	  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)	  IID_IWrappedProtocol A)	  IID_IGetBindHandle )	  IID_IBindCallbackRedirect 
*	  IID_IPropertyStorage 	  IID_IPropertySetStorage 	  IID_IEnumSTATPROPSTG 	  IID_IEnumSTATPROPSETSTG D	  IID_StdOle 	  GUID_DEVINTERFACE_DISK 	  GUID_DEVINTERFACE_CDROM 
	  GUID_DEVINTERFACE_PARTITION 	  GUID_DEVINTERFACE_TAPE 	  GUID_DEVINTERFACE_WRITEONCEDISK 	  GUID_DEVINTERFACE_VOLUME 	  GUID_DEVINTERFACE_MEDIUMCHANGER 	  GUID_DEVINTERFACE_FLOPPY 	  GUID_DEVINTERFACE_CDCHANGER 	  GUID_DEVINTERFACE_STORAGEPORT 	  GUID_DEVINTERFACE_COMPORT 	  GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR 	  _SCARD_IO_REQUEST RO  dwProtocol F   cbPciLength F   SCARD_IO_REQUEST O  RO  g_rgSCardT0Pci %jO  g_rgSCardT1Pci %jO  g_rgSCardRawPci %jO  IID_IPrintDialogCallback  	  IID_IPrintDialogServices  	  _commode !  @!,P  osfhnd !-    osfile !.   pipech !/   	lockinitflag !0  lock !1"  textmode !2   8unicode !3    8pipech2 !4P  9    P  8   ioinfo !5P  P  P   P  P  __imp___badioinfo !KP  __imp___pioinfo !PP  _dowildcard !m  _newmode !n  __imp___winitenv !q_
  __imp___initenv !v>
  __imp__acmdln !{D
  __imp__wcmdln !D
      !Q  __uninitialized  __initializing __initialized     !rQ  Q  __native_startup_state !Q  __native_startup_lock !R  R  !__native_dllmain_reason !#  __native_vcclrit_reason !#  __security_cookie "|   	  __imp__HUGE #_R  signgam #  __dyn_tls_init_callback    "mingw_app_type   	m    #_encode_pointer 8  R  $ptr 8   %_decode_pointer 8  S  $codedptr 8   &R   ґm           'R  R  \     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/pseudo-reloc.c 0ґm          m  __gnuc_va_list    __builtin_va_list    char    va_list    size_t #   long long unsigned int long long int intptr_t >  ptrdiff_t X  wchar_t bB  short unsigned int B  int long int pthreadlocinfo     threadlocaleinfostruct `$  	d  ]   
lc_codepage b  
lc_collate_cp b  
lc_handle w  
lc_id   $
lc_category   Hlc_clike ]  mb_cur_max ]  lconv_intl_refcount \  lconv_num_refcount \  lconv_mon_refcount \   lconv   (ctype1_refcount \  0ctype1   8pctype   @pclmap   Hpcumap   Plc_time_curr 
  X pthreadmbcinfo ;  A  threadmbcinfostruct 
localeinfo_struct   
locinfo p   
mbcinfo $   _locale_tstruct V  
tagLC_ID   
wLanguage B   
wCountry B  
wCodePage B   LC_ID    P  
locale P   
wlocale V  	d  \  
wrefcount \      3  ]  unsigned int b         sizetype long unsigned int               lconv   B  X    unsigned char   __lc_time_data   _PHNDLR ?"  (  3  ]   _XCPT_ACTION A{  XcptNum B   SigNum C]  XcptAction D   3     _XcptActTab G{  _XcptActTabCount H]  _XcptActTabSize I]  _First_FPE_Indx J]  _Num_FPE K]  BYTE   WORD B  DWORD   float PBYTE ,    LPBYTE ,  LPVOID   T  b  __imp__pctype $p    __imp__wctype 3p  __imp__pwctype ?p         __newclmap H  __newcumap I  __ptlocinfo Jp  __ptmbcinfo K$  __globallocalestatus L]  __locale_changed M]  __initiallocinfo N  __initiallocalestructinfo O  __imp___mb_cur_max \  signed char short int ULONG_PTR 1   SIZE_T   PVOID   LONG d  HANDLE   
_LIST_ENTRY ]  
Flink ^   
Blink _     LIST_ENTRY `  _GUID 	z  Data1 	   Data2 	B  Data3 	B  Data4 	z          GUID 	3    IID 	R    CLSID 	Z    FMTID 	a    double long double P        _sys_errlist 
  _sys_nerr 
]  __imp___argc 
\  __imp___argv 
E	  K	  P  __imp___wargv 
f	  l	  V  __imp__environ 
E	  __imp__wenviron 
f	  __imp__pgmptr 
K	  __imp__wpgmptr 
l	  __imp__fmode 
\  __imp__osplatform 
 U  __imp__osver 
	U  __imp__winver 
U  __imp__winmajor 
U  __imp__winminor 
$U  _amblksiz 5b  
_MEMORY_BASIC_INFORMATION 0   
BaseAddress !   
AllocationBase "  
AllocationProtect #	  
RegionSize $  
State %	   
Protect &	  $
Type '	  ( MEMORY_BASIC_INFORMATION (d
  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID          C   PhysicalAddress D	  VirtualSize E	   
_IMAGE_SECTION_HEADER (A,!  
Name B   
Misc F  
VirtualAddress G	  
SizeOfRawData H	  
PointerToRawData I	  
PointerToRelocations J	  
PointerToLinenumbers K	  
NumberOfRelocations L   
NumberOfLinenumbers M  "
Characteristics N	  $ PIMAGE_SECTION_HEADER OJ!     
_RTL_CRITICAL_SECTION_DEBUG 0\H"  
Type ]   
CreatorBackTraceIndex ^  
CriticalSection _"  
ProcessLocksList `   
EntryCount a	   
ContentionCount b	  $
Flags c	  (
CreatorBackTraceIndexHigh d  ,
SpareWORD e  . 
_RTL_CRITICAL_SECTION (w"  
DebugInfo x"   
LockCount y  
RecursionCount z  
OwningThread {  
LockSemaphore |  
SpinCount }    H"  PRTL_CRITICAL_SECTION_DEBUG f#  P!  RTL_CRITICAL_SECTION ~H"  CRITICAL_SECTION #  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

  RPC_IF_HANDLE B  tagCOINITBASE b  #  COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )y#  IWinTypesBase_v0_1_s_ifspec *y#  IID_IUnknown W  IID_AsyncIUnknown   IID_IClassFactory m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (y#  IWinTypes_v0_1_s_ifspec )y#  VARENUM b   5  VT_EMPTY  VT_NULL VT_I2 VT_I4 VT_R4 VT_R8 VT_CY VT_DATE VT_BSTR VT_DISPATCH 	VT_ERROR 
VT_BOOL VT_VARIANT VT_UNKNOWN 
VT_DECIMAL VT_I1 VT_UI1 VT_UI2 VT_UI4 VT_I8 VT_UI8 VT_INT VT_UINT VT_VOID VT_HRESULT VT_PTR VT_SAFEARRAY VT_CARRAY VT_USERDEFINED VT_LPSTR VT_LPWSTR VT_RECORD $VT_INT_PTR %VT_UINT_PTR &VT_FILETIME @VT_BLOB AVT_STREAM BVT_STORAGE CVT_STREAMED_OBJECT DVT_STORED_OBJECT EVT_BLOB_OBJECT FVT_CF GVT_CLSID HVT_VERSIONED_STREAM I VT_BSTR_BLOB  VT_VECTOR   VT_ARRAY    VT_BYREF  @ VT_RESERVED   VT_ILLEGAL  VT_ILLEGALMASKED  VT_TYPEMASK  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec y#  IOLETypes_v0_0_s_ifspec y#  IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec y#  IOleAutomationTypes_v1_0_s_ifspec y#  IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec y#  __MIDL_itf_msxml_0000_v0_0_s_ifspec y#  LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST jP  dwProtocol 	   cbPciLength 	   SCARD_IO_REQUEST (P  jP  g_rgSCardT0Pci %P  g_rgSCardT1Pci %P  g_rgSCardRawPci %P  IID_IPrintDialogCallback    IID_IPrintDialogServices    _commode !]  !@!,Q  osfhnd !-   osfile !.   pipech !/   	lockinitflag !0]  lock !13#  "textmode !2   8"unicode !3    8pipech2 !4Q  9    Q     ioinfo !5Q  Q  Q   Q  Q  __imp___badioinfo !KQ  __imp___pioinfo !PQ  _dowildcard !m]  _newmode !n]  __imp___winitenv !qf	  __imp___initenv !vE	  __imp__acmdln !{K	  __imp__wcmdln !K	  #2  b  !R  __uninitialized  __initializing __initialized  $2  !R  R  __native_startup_state !R  __native_startup_lock !S  S  %__native_dllmain_reason !r  __native_vcclrit_reason !r  __RUNTIME_PSEUDO_RELOC_LIST__ 0   __RUNTIME_PSEUDO_RELOC_LIST_END__ 1   __image_base__ 2   !;S  addend <	   target =	   runtime_pseudo_reloc_item_v1 >S  !F?T  sym G	   target H	  flags I	   runtime_pseudo_reloc_item_v2 J
T  !LT  magic1 M	   magic2 N	  version O	   runtime_pseudo_reloc_v2 PcT  &  T  old_protect 	   sec_start   hash ,!   $  T  'the_secs U  	m    T  'maxSections ]  	m    (_pei386_runtime_relocator  ԑm          {X  )was_init ]  	 m    *mSecs ]    +{X  Yԑm    P  W  ,X  ,X  ,X  -P  .X    /X  .X    .X    . Y    +:Y  ԑm      HV  0kY  .  0_Y  R  0RY  u  1֑m    Z  2Qu 2X2  +:Y  ;Ցm      V  0kY    0_Y    0RY    1VՑm    Z  2Qu 2X8  3Ցm    =       W  .Y    -0  /Y  4:Y  Ցm    `  p0kY  M  0_Y  q  0RY    1֑m    Z  2Qu 2X4    +:Y  %֑m      YW  0kY    0_Y    0RY    1P֑m    Z  2Qu 2X1  +:Y  ֑m      W  0kY  !  0_Y  E  0RY  h  1֑m    Z  2Qu 2X4  5֑m    Z  W  2R	Bm     1֑m    Z  2R	pBm       +xY  fՑm      mX  -  .Y    /Y  /Y  ~5Ցm    )\  8X  2Qu 2X0 5Ցm    4\  PX  2Y~  1֑m    Z  2R	Bm       6Bԑm    ?\   7do_pseudo_reloc 3(Y  8start 3  8end 3  8base 3  9addr_imp 5"  9reldata 5"  9reloc_target 6"  9v2_hdr 7(Y  9r 8.Y  :9o h4Y  :9newval m	     T  ?T  S  7__write_memory xY  8addr   8src N  8len     ;restore_modified_sections Y  <i ]  <b   <oldprot 	   ;mark_section_writable Z  =addr @  <b   <h ,!  <i ]   >__report_error S0ґm    a       Z  ?msg SZ    @'argp    X6Wґm    J\  5rґm    U\  Z  2R	Am    2Q12XK 6|ґm    J\  5ґm    w\  Z  2Qs 2Xt  6ґm    \      A:Y  ґm    `      )\  0RY    0_Y  }  0kY    +Y  ґm      
\  0Y  t  -  /Y  .Y    .Y  ,  5ґm    \  M[  2Rt  61ӑm    \  5Zӑm    )\  y[  2Q2X0 5ӑm    4\  [  2X@2Ys  6ӑm    \  5ӑm    Z  [  2R	HBm     5ӑm    Z  [  2R	Bm     1 ԑm    Z  2R	Am    2Qt    Bӑm    \  2RR2QQ2XX  CW  W  "/C    "AC    Cm  m  #TDfwrite __builtin_fwrite % fwrite E    #Fabort abort 
IC    C    CJ  J  $-Gmemcpy memcpy  ]   H  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/crt_handler.c  בm    b        char size_t #   long long unsigned int long long int wchar_t b   short unsigned int    int long int pthreadlocinfo #  )  threadlocaleinfostruct `        	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id ?  $	lc_category O  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv f  (
ctype1_refcount   0
ctype1 l  8
pctype r  @
pclmap x  H
pcumap x  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct 0  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID H  
   	locale    	wlocale       	wrefcount             unsigned int *       sizetype long unsigned int   O       _     lconv _          unsigned char ~  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B*   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K     _EXCEPTION_RECORD 	6  	ExceptionCode 	P	   	ExceptionFlags 	P	    	  	ExceptionAddress 	
  	NumberParameters 	P	  	ExceptionInformation 	0    >  _CONTEXT +	  	P1Home 
   	P2Home 
  	P3Home 
  	P4Home 
  	P5Home 
   	P6Home 
  (	ContextFlags P	  0	MxCsr P	  4	SegCs D	  8	SegDs D	  :	SegEs D	  <	SegFs D	  >	SegGs D	  @	SegSs D	  B	EFlags P	  D	Dr0 
  H	Dr1 
  P	Dr2 
  X	Dr3 
  `	Dr6 
  h	Dr7 
  p	Rax 
  x	Rcx 
  	Rdx 
  	Rbx 
  	Rsp 
  	Rbp 
  	Rsi 
  	Rdi 
  	R8 
  	R9 
  	R10 
  	R11 
  	R12 
  	R13 
  	R14 
  	R15 
  	Rip 
  j   
VectorRegister    
VectorControl 
  
DebugControl 
  
LastBranchToRip 
  
LastBranchFromRip 
  
LastExceptionToRip 
  
LastExceptionFromRip 
   ULONG *  BYTE ~  WORD    DWORD *  float PBYTE s	  8	  LPBYTE s	    __imp__pctype $	  l  __imp__wctype 3	  __imp__pwctype ?	    	   	  __newclmap H	  __newcumap I	  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N)  __initiallocalestructinfo O0  __imp___mb_cur_max   signed char short int ULONG_PTR 1   DWORD64 ¤   PVOID 6  LONG    LONGLONG    ULONGLONG    _GUID t  Data1 *   Data2    Data3    Data4 t   ~       GUID -    IID R    CLSID Z    FMTID a    _M128A G  	Low H   	High I
   M128A J           #     8	  3    _ double long double   \      _sys_errlist 	L  _sys_nerr 	   __imp___argc 	  __imp___argv 	      __imp___wargv 	      __imp__environ 	Щ  __imp__wenviron 	  __imp__pgmptr 	  __imp__wpgmptr 	  __imp__fmode 	  __imp__osplatform 	 	  __imp__osver 			  __imp__winver 		  __imp__winmajor 		  __imp__winminor 	$	  _amblksiz 
5  _XMM_SAVE_AREA32  j'  	ControlWord kD	   	StatusWord lD	  	TagWord m8	  	Reserved1 n8	  	ErrorOpcode oD	  	ErrorOffset pP	  	ErrorSelector qD	  	Reserved2 rD	  	DataOffset sP	  	DataSelector tD	  	Reserved3 uD	  	MxCsr vP	  	MxCsr_Mask wP	  	FloatRegisters x   	XmmRegisters y  
Reserved4 z#   XMM_SAVE_AREA32 {
  Z  	Header Z   	Legacy    	Xmm0   	Xmm1   	Xmm2   	Xmm3   	Xmm4   	Xmm5   
Xmm6    
Xmm7   
Xmm8    
Xmm9   0
Xmm10   @
Xmm11   P
Xmm12   `
Xmm13   p
Xmm14   
Xmm15      j        FltSave '  FloatSave '  ?          PCONTEXT 8  _RUNTIME_FUNCTION   	BeginAddress P	   	EndAddress P	  	UnwindData P	   RUNTIME_FUNCTION   
  @     EXCEPTION_RECORD 	  PEXCEPTION_RECORD 	s  @  _EXCEPTION_POINTERS 	    	Y     	   EXCEPTION_POINTERS 	y  y  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   8	  &     _IMAGE_DOS_HEADER @'  	e_magic D	   	e_cblp D	  	e_cp D	  	e_crlc D	  	e_cparhdr D	  	e_minalloc D	  
	e_maxalloc D	  	e_ss D	  	e_sp D	  	e_csum D	  	e_ip D	  	e_cs D	  	e_lfarlc D	  	e_ovno D	  	e_res '  	e_oemid D	  $	e_oeminfo D	  &	e_res2 '  (	e_lfanew 
  < D	  '     D	  '    	 IMAGE_DOS_HEADER &  CE(  PhysicalAddress DP	  VirtualSize EP	   _IMAGE_SECTION_HEADER (AY)  	Name Bq&   	Misc F(  	VirtualAddress GP	  	SizeOfRawData HP	  	PointerToRawData IP	  	PointerToRelocations JP	  	PointerToLinenumbers KP	  	NumberOfRelocations LD	   	NumberOfLinenumbers MD	  "	Characteristics NP	  $ PIMAGE_SECTION_HEADER Ow)  E(  )  
  )     PTOP_LEVEL_EXCEPTION_FILTER })  LPTOP_LEVEL_EXCEPTION_FILTER )  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  RPC_IF_HANDLE 
B6  tagCOINITBASE   R*   COINITBASE_MULTITHREADED   IWinTypesBase_v0_1_c_ifspec )*  IWinTypesBase_v0_1_s_ifspec **  IID_IUnknown W  IID_AsyncIUnknown   IID_IClassFactory m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (*  IWinTypes_v0_1_s_ifspec )*  !VARENUM    ;   VT_EMPTY   VT_NULL  VT_I2  VT_I4  VT_R4  VT_R8  VT_CY  VT_DATE  VT_BSTR  VT_DISPATCH 	 VT_ERROR 
 VT_BOOL  VT_VARIANT  VT_UNKNOWN 
 VT_DECIMAL  VT_I1  VT_UI1  VT_UI2  VT_UI4  VT_I8  VT_UI8  VT_INT  VT_UINT  VT_VOID  VT_HRESULT  VT_PTR  VT_SAFEARRAY  VT_CARRAY  VT_USERDEFINED  VT_LPSTR  VT_LPWSTR  VT_RECORD $ VT_INT_PTR % VT_UINT_PTR & VT_FILETIME @ VT_BLOB A VT_STREAM B VT_STORAGE C VT_STREAMED_OBJECT D VT_STORED_OBJECT E VT_BLOB_OBJECT F VT_CF G VT_CLSID H VT_VERSIONED_STREAM I"VT_BSTR_BLOB "VT_VECTOR  "VT_ARRAY   "VT_BYREF  @"VT_RESERVED  "VT_ILLEGAL "VT_ILLEGALMASKED "VT_TYPEMASK  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec *  IOLETypes_v0_0_s_ifspec *  IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec *  IOleAutomationTypes_v1_0_s_ifspec *  IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec *  __MIDL_itf_msxml_0000_v0_0_s_ifspec *  LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST V  dwProtocol P	   cbPciLength P	   SCARD_IO_REQUEST V  V  g_rgSCardT0Pci %W  g_rgSCardT1Pci %W  g_rgSCardRawPci %W  IID_IPrintDialogCallback   IID_IPrintDialogServices   __p_sig_fn_t  0  __image_base__ '  _UNWIND_INFO _X  VersionAndFlags 8	   PrologSize 8	  CountOfUnwindCodes 8	  FrameRegisterAndOffset 8	  AddressOfExceptionHandler  +	   UNWIND_INFO !W    X     #emu_pdata /rX  	@m    _X  X     #emu_xdata 0X  	@m    $__mingw_oldexcpt_handler )  	 m    %_gnu_exception_handler    ّm          Z  &exception_data ¬Z  w  '  Ĺ    (action    "  '       )ّm    y]  Y  *R;*Q0 +ڑm    Y  *R; ,<ڑm    Y  *RR )gڑm    y]  Y  *R8*Q0 +}ڑm    Y  *R8 )ڑm    y]  Y  *R4*Q0 +ڑm    Z  *R4 )ڑm    y]  .Z  *R8*Q0 )	ۑm    y]  JZ  *R8*Q1 )"ۑm    y]  fZ  *R4*Q1 )?ۑm    y]  Z  *R;*Q1 )Xۑm    y]  Z  *R8*Q1 -]ۑm    ]     %__mingw_SEH_error_handler ^    בm          \  .  ^    &EstablisherFrame _6    .  `8  i  &DispatcherContext a6  F  (action c   #  '  d    '  e     )Gבm    y]  [  *R;*Q0 +aבm    [  *R; )בm    y]  [  *R8*Q0 +בm    [  *R8 )בm    y]  [  *R4*Q0 +בm    [  *R4 )'ؑm    y]  \  *R8*Q1 -,ؑm    ]  )?ؑm    y]  B\  *R8*Q0 )[ؑm    y]  ^\  *R8*Q1 ){ؑm    y]  z\  *R4*Q1 /ؑm    y]  *R;*Q1  %__mingw_init_ehandler 3   ؑm           y]  #was_here 5   	(m    (e 6   >   (pSec 7Y)     (_ImageBase 8f	     -ؑm    ]  )ؑm    ]  ?]  *R	Bm     )[ّm    ]  W]  *Rt  /ّm    ]  *R	@m    *Xv   0signal signal  <1    [1x  x  &1    $1    %2     i%     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/tlsthrd.c pۑm    j        char long long unsigned int long long int wchar_t b   short unsigned int    int    long int pthreadlocinfo     threadlocaleinfostruct `  	=      
lc_codepage   
lc_collate_cp   
lc_handle   
lc_id 2  $
lc_category B  Hlc_clike    mb_cur_max    lconv_intl_refcount   lconv_num_refcount   lconv_mon_refcount    lconv Y  (ctype1_refcount   0ctype1 _  8pctype e  @pclmap k  Hpcumap k  Plc_time_curr   X pthreadmbcinfo     threadmbcinfostruct 
localeinfo_struct #  
locinfo     
mbcinfo    _locale_tstruct   
tagLC_ID   
wLanguage     
wCountry    
wCodePage     LC_ID ;     
locale    
wlocale   	=    
wrefcount             unsigned int        sizetype long unsigned int   B       R     lconv R          unsigned char q  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WINBOOL    WORD    DWORD   float LPVOID y    __imp__pctype $  _  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J   __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N  __initiallocalestructinfo O#  __imp___mb_cur_max   signed char short int ULONG_PTR 1   LONG    HANDLE y  
_LIST_ENTRY ]d  
Flink ^d   
Blink _d   0  LIST_ENTRY `0  _GUID   Data1    Data2    Data3    Data4    q       GUID }    double long double         _sys_errlist 	  _sys_nerr 	   __imp___argc 	  __imp___argv 	[  a    __imp___wargv 	|      __imp__environ 	[  __imp__wenviron 	|  __imp__pgmptr 	a  __imp__wpgmptr 	  __imp__fmode 	  __imp__osplatform 	   __imp__osver 		  __imp__winver 	  __imp__winmajor 	  __imp__winminor 	$  _amblksiz 
5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   
_RTL_CRITICAL_SECTION_DEBUG 0\  
Type ]   
CreatorBackTraceIndex ^  
CriticalSection _  
ProcessLocksList `j  
EntryCount a   
ContentionCount b  $
Flags c  (
CreatorBackTraceIndexHigh d  ,
SpareWORD e  . 
_RTL_CRITICAL_SECTION (w  
DebugInfo x   
LockCount y  
RecursionCount z  
OwningThread {!  
LockSemaphore |!  
SpinCount }      PRTL_CRITICAL_SECTION_DEBUG f    RTL_CRITICAL_SECTION ~  CRITICAL_SECTION      %   y   VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  __mingwthr_cs   	m    __mingwthr_cs_init    	Ȝm    __mingwthr_key_t       __mingwthr_key     key !   dtor "   next #       key_dtor_list '   	m    __mingw_TLScallback z{   ݑm           w"  hDllHandle z!  !  reason {  "  reserved |  #  Yݑm    G       !  keyp    $  t    c$  nݑm    $  ݑm    
%   R	m      !w"  ݑm           "  "ݑm           #"  )ݑm    W$    $w"  @ݑm    @  N"  %@  #"  ݑm    W$    5ݑm    %  ݑm    #%   R	m      &__mingwthr_run_key_dtors c"  'keyp e   ('value m    ___w64_mingwthr_remove_key_dtor A   `ܑm           #  key A  $  prev_key C   $  cur_key D   C%  )ܑm    .%  F#   R	m     )ܑm    9%  e#   R	m     ܑm    $  ܑm    9%   R	m      ___w64_mingwthr_add_key_dtor *   ۑm           Q$  key *  %  dtor *   `&  new_key ,Q$  &  )ܑm    D%  $   R1 QH ).ܑm    .%  5$   R	m     Mܑm    9%   R	m         *w"  pۑm    j       $  +"  D'  ۑm           $  +"  g'  ۑm    V%  ۑm    a%  ,ۑm     Rt   )ۑm    .%  $   R	m     -ۑm    9%   R	m      .free free 	/p  p  
./    /
  
  
p/[  [  
+/F  F  
,.calloc calloc 	/1  1  /$  $  -     $  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/tlsmcrt.c   _CRT_MT    	m    int      N  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/pseudo-reloc-list.c   __RUNTIME_PSEUDO_RELOC_LIST_END__    	!m    char __RUNTIME_PSEUDO_RELOC_LIST__    	 m     Z   x  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/pesect.c ݑm            char    size_t #   long long unsigned int long long int wchar_t b   short unsigned int    int long int pthreadlocinfo #  )  threadlocaleinfostruct `        	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id ?  $	lc_category O  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv f  (
ctype1_refcount   0
ctype1 l  8
pctype r  @
pclmap x  H
pcumap x  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct 0  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID H  
   	locale    	wlocale       	wrefcount             unsigned int *       sizetype long unsigned int   O       _     lconv _          unsigned char ~  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B*   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WINBOOL    BYTE ~  WORD    DWORD *  float PBYTE     LPVOID     __imp__pctype $  l  __imp__wctype 3  __imp__pwctype ?    =   2  __newclmap H=  __newcumap I=  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N)  __initiallocalestructinfo O0  __imp___mb_cur_max   signed char short int ULONG_PTR 1   DWORD_PTR /  LONG    ULONGLONG    _GUID   Data1 *   Data2    Data3    Data4    ~       GUID p    IID R    CLSID Z    FMTID a    double long double   5      _sys_errlist 	%  _sys_nerr 	   __imp___argc 	  __imp___argv 	      __imp___wargv 	ģ      __imp__environ 	Ђ  __imp__wenviron 	٣  __imp__pgmptr 	  __imp__wpgmptr 	  __imp__fmode 	  __imp__osplatform 	   __imp__osver 		  __imp__winver 	  __imp__winmajor 	  __imp__winminor 	$  _amblksiz 
5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID     P     _IMAGE_DOS_HEADER @  	e_magic    	e_cblp   	e_cp   	e_crlc   	e_cparhdr   	e_minalloc   
	e_maxalloc   	e_ss   	e_sp   	e_csum   	e_ip   	e_cs   	e_lfarlc   	e_ovno   	e_res   	e_oemid   $	e_oeminfo   &	e_res2   (	e_lfanew Q  <              	 IMAGE_DOS_HEADER P  PIMAGE_DOS_HEADER   P  _IMAGE_FILE_HEADER &   	Machine '   	NumberOfSections (    )  	PointerToSymbolTable *  	NumberOfSymbols +  	SizeOfOptionalHeader ,    -   IMAGE_FILE_HEADER .  _IMAGE_DATA_DIRECTORY b
!    c   	Size d   IMAGE_DATA_DIRECTORY e   
!  7!     _IMAGE_OPTIONAL_HEADER64 e$  	Magic    	MajorLinkerVersion   	MinorLinkerVersion   	SizeOfCode   	SizeOfInitializedData   	SizeOfUninitializedData   	AddressOfEntryPoint   	BaseOfCode   	ImageBase ^  	SectionAlignment    	FileAlignment   $	MajorOperatingSystemVersion   (	MinorOperatingSystemVersion   *	MajorImageVersion   ,	MinorImageVersion   .	MajorSubsystemVersion   0	MinorSubsystemVersion   2	Win32VersionValue   4	SizeOfImage   8	SizeOfHeaders   <	CheckSum   @	Subsystem   D	DllCharacteristics   F	SizeOfStackReserve ^  H	SizeOfStackCommit ^  P	SizeOfHeapReserve ^  X	SizeOfHeapCommit ^  `	LoaderFlags   h	NumberOfRvaAndSizes   l	DataDirectory '!  p IMAGE_OPTIONAL_HEADER64 7!  PIMAGE_OPTIONAL_HEADER64 $  7!  PIMAGE_OPTIONAL_HEADER $  _IMAGE_NT_HEADERS64 )%  	Signature    	FileHeader    	OptionalHeader e$   PIMAGE_NT_HEADERS64 E%  $  PIMAGE_NT_HEADERS )%  C%  PhysicalAddress D  VirtualSize E   _IMAGE_SECTION_HEADER (A&  	Name B@   	Misc Fe%    G  	SizeOfRawData H  	PointerToRawData I  	PointerToRelocations J  	PointerToLinenumbers K  	NumberOfRelocations L   	NumberOfLinenumbers M  "  N  $ PIMAGE_SECTION_HEADER O&  %  ?&    @  OriginalFirstThunk A   _IMAGE_IMPORT_DESCRIPTOR >]'  &     C  	ForwarderChain E  	Name F  	FirstThunk G   IMAGE_IMPORT_DESCRIPTOR H&  PIMAGE_IMPORT_DESCRIPTOR I'  ]'  VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 
  RPC_IF_HANDLE B  IWinTypesBase_v0_1_c_ifspec 
)'  IWinTypesBase_v0_1_s_ifspec 
*'  IID_IUnknown W  IID_AsyncIUnknown   IID_IClassFactory m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec ('  IWinTypes_v0_1_s_ifspec )'  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec '  IOLETypes_v0_0_s_ifspec '  IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec '  IOleAutomationTypes_v1_0_s_ifspec '  IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec '  __MIDL_itf_msxml_0000_v0_0_s_ifspec '  LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST R  dwProtocol    cbPciLength    SCARD_IO_REQUEST Q  R  g_rgSCardT0Pci %R  g_rgSCardT1Pci %R  g_rgSCardRawPci %R  IID_IPrintDialogCallback   IID_IPrintDialogServices   __image_base__   __mingw_enum_import_library_names S  0m           S  i    '        K%  '   importDesc }'  <(    Ř&   importsStartRVA Ư  w(  !X  4m    P  ɡS  "X  #P  $Y  $Y  $Y  %Ym    'Y  &R{    'zX  nm    ?       (X  (  "X  )nm    ?       $Y  $Y  $Y        _IsNonwritableInCurrentImage   m           T  pTarget   )      *rvaTarget @  R  &  !X  m      T  "X  #  $Y  $Y  $Y  %m    'Y  &Rx    +zX  m       (X  `)  "X  #   $Y  $Y  $Y     _GetPEImageBase   Pm    6       ^U      +X  Tm      "X  #  $Y  $Y  $Y  %xm    'Y  &Rx     _FindPESectionExec &  ߑm    r       V  eNo    )        K%  )    &  "*      E*  +X  ߑm    `  "X  #`  $Y  $Y  $Y  %ߑm    'Y  &Ry     __mingw_GetSectionCount p   ߑm    >       V    r    sK%  +X  ߑm       v"X  #   $Y  $Y  $Y  %ߑm    'Y  &Rq     __mingw_GetSectionForAddress b&   ߑm           W  p b  i*    d  *rva e@  R!X  ߑm      hKW  "X  #  $Y  $Y  $Y  %(ߑm    'Y  &Ry    +zX  ,ߑm      l,X  R"X  #  $Y  $Y  $Y     _FindPESectionByName C&  pޑm           zX  pName CS  *    E    FK%  S+    G&  v+    H  +  !X  ޑm    p  OBX  "X  #p  $Y  $Y  $Y  %ޑm    'Y  &Rq    -ޑm    Y  ZX  &Rt  %ޑm    Y  &Rs &Qt &X8  ._FindPESection -&  X  /  -  0rva -@    /K%    0&    1   ._ValidateImageBase   'Y  /    1pDOSHeader     K%  1pOptHeader $   2X  ݑm           aY  (X  +  $X  3	Y  R4Y  +   2X   ޑm           Y  (X  ,  4X  T,  $	Y  $Y  5ޑm    'Y   2zX   ޑm    E       Y  (X  ,  ,X  Q4X  ,  4X  
-  4X  0-   6strlen strlen @6strncmp strncmp V     %  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/CRT_fp10.c  m           8  _fpreset 	 m            I   O  GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/dllentry.c Pm             char long long unsigned int long long int uintptr_t K   wchar_t b   short unsigned int    int long int pthreadlocinfo #  )  threadlocaleinfostruct `        	lc_codepage   	lc_collate_cp   	lc_handle   	lc_id ?  $	lc_category O  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv f  (
ctype1_refcount   0
ctype1 l  8
pctype r  @
pclmap x  H
pcumap x  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct 0  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID H  
   	locale    	wlocale       	wrefcount             unsigned int *       sizetype long unsigned int   O       _     lconv _          unsigned char ~  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B*   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WINBOOL    BOOL    DWORD *  float LPVOID     __imp__pctype $  l  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N)  __initiallocalestructinfo O0  __imp___mb_cur_max   signed char short int HANDLE   _GUID f  Data1 *   Data2    Data3    Data4 f   ~  v     GUID   v  IID Rv    CLSID Zv    FMTID av    double long double         _sys_errlist   _sys_nerr    __imp___argc   __imp___argv 1  7    __imp___wargv R  X    __imp__environ 1  __imp__wenviron R  __imp__pgmptr 7  __imp__wpgmptr X  __imp__fmode   __imp__osplatform    __imp__osver 	  __imp__winver   __imp__winmajor   __imp__winminor $  _amblksiz 	5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

  RPC_IF_HANDLE B  IWinTypesBase_v0_1_c_ifspec )  IWinTypesBase_v0_1_s_ifspec *  IID_IUnknown 
W  IID_AsyncIUnknown 
  IID_IClassFactory 
m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (  IWinTypes_v0_1_s_ifspec )  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec   IOLETypes_v0_0_s_ifspec   IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec   IOleAutomationTypes_v1_0_s_ifspec   IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec   __MIDL_itf_msxml_0000_v0_0_s_ifspec   LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST NH  dwProtocol    cbPciLength    SCARD_IO_REQUEST H  NH  g_rgSCardT0Pci %fH  g_rgSCardT1Pci %fH  g_rgSCardRawPci %fH  IID_IPrintDialogCallback   IID_IPrintDialogServices   __security_cookie |      I         _pRawDllMain :I  I  4I  DllEntryPoint 
  Pm           hDllHandle 
  RdwReason   Qlpreserved   X  I     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/crt/dllmain.c `m           s  char long long unsigned int long long int uintptr_t K   wchar_t b   short unsigned int    int long int pthreadlocinfo "  (  threadlocaleinfostruct `        	lc_codepage   	lc_collate_cp   	lc_handle 
  	lc_id >  $	lc_category N  H
lc_clike    
mb_cur_max    
lconv_intl_refcount   
lconv_num_refcount   
lconv_mon_refcount    
lconv e  (
ctype1_refcount   0
ctype1 k  8
pctype q  @
pclmap w  H
pcumap w  P
lc_time_curr   X pthreadmbcinfo     threadmbcinfostruct localeinfo_struct /  	locinfo    	mbcinfo    _locale_tstruct   tagLC_ID   	wLanguage     	wCountry    	wCodePage     LC_ID G  
   	locale    	wlocale       	wrefcount             unsigned int )       sizetype long unsigned int   N       ^     lconv ^          unsigned char }  __lc_time_data   _PHNDLR ?          _XCPT_ACTION A  XcptNum B)   SigNum C   XcptAction D        _XcptActTab G  _XcptActTabCount H   _XcptActTabSize I   _First_FPE_Indx J   _Num_FPE K   WINBOOL    BOOL    DWORD )  float LPVOID     __imp__pctype $  k  __imp__wctype 3  __imp__pwctype ?         __newclmap H  __newcumap I  __ptlocinfo J  __ptmbcinfo K  __globallocalestatus L   __locale_changed M   __initiallocinfo N(  __initiallocalestructinfo O/  __imp___mb_cur_max   signed char short int HANDLE   _GUID e  Data1 )   Data2    Data3    Data4 e   }  u     GUID   u  IID Ru    CLSID Zu    FMTID au    double long double         _sys_errlist   _sys_nerr    __imp___argc   __imp___argv 0  6    __imp___wargv Q  W    __imp__environ 0  __imp__wenviron Q  __imp__pgmptr 6  __imp__wpgmptr W  __imp__fmode   __imp__osplatform    __imp__osver 	  __imp__winver   __imp__winmajor   __imp__winminor $  _amblksiz 	5  GUID_MAX_POWER_SAVINGS b  GUID_MIN_POWER_SAVINGS c  GUID_TYPICAL_POWER_SAVINGS d  NO_SUBGROUP_GUID e  ALL_POWERSCHEMES_GUID f  GUID_POWERSCHEME_PERSONALITY g  GUID_ACTIVE_POWERSCHEME h  GUID_IDLE_RESILIENCY_SUBGROUP i  GUID_IDLE_RESILIENCY_PERIOD j  GUID_DISK_COALESCING_POWERDOWN_TIMEOUT k  GUID_EXECUTION_REQUIRED_REQUEST_TIMEOUT l  GUID_VIDEO_SUBGROUP m  GUID_VIDEO_POWERDOWN_TIMEOUT n  GUID_VIDEO_ANNOYANCE_TIMEOUT o  GUID_VIDEO_ADAPTIVE_PERCENT_INCREASE p  GUID_VIDEO_DIM_TIMEOUT q  GUID_VIDEO_ADAPTIVE_POWERDOWN r  GUID_MONITOR_POWER_ON s  GUID_DEVICE_POWER_POLICY_VIDEO_BRIGHTNESS t  GUID_DEVICE_POWER_POLICY_VIDEO_DIM_BRIGHTNESS u  GUID_VIDEO_CURRENT_MONITOR_BRIGHTNESS v  GUID_VIDEO_ADAPTIVE_DISPLAY_BRIGHTNESS w  GUID_CONSOLE_DISPLAY_STATE x  GUID_ALLOW_DISPLAY_REQUIRED y  GUID_VIDEO_CONSOLE_LOCK_TIMEOUT z  GUID_ADAPTIVE_POWER_BEHAVIOR_SUBGROUP {  GUID_NON_ADAPTIVE_INPUT_TIMEOUT |  GUID_DISK_SUBGROUP }  GUID_DISK_POWERDOWN_TIMEOUT ~  GUID_DISK_IDLE_TIMEOUT   GUID_DISK_BURST_IGNORE_THRESHOLD   GUID_DISK_ADAPTIVE_POWERDOWN   GUID_SLEEP_SUBGROUP   GUID_SLEEP_IDLE_THRESHOLD   GUID_STANDBY_TIMEOUT   GUID_UNATTEND_SLEEP_TIMEOUT   GUID_HIBERNATE_TIMEOUT   GUID_HIBERNATE_FASTS4_POLICY   GUID_CRITICAL_POWER_TRANSITION   GUID_SYSTEM_AWAYMODE   GUID_ALLOW_AWAYMODE   GUID_ALLOW_STANDBY_STATES   GUID_ALLOW_RTC_WAKE   GUID_ALLOW_SYSTEM_REQUIRED   GUID_SYSTEM_BUTTON_SUBGROUP   GUID_POWERBUTTON_ACTION   GUID_SLEEPBUTTON_ACTION   GUID_USERINTERFACEBUTTON_ACTION   GUID_LIDCLOSE_ACTION   GUID_LIDOPEN_POWERSTATE   GUID_BATTERY_SUBGROUP   GUID_BATTERY_DISCHARGE_ACTION_0   GUID_BATTERY_DISCHARGE_LEVEL_0   GUID_BATTERY_DISCHARGE_FLAGS_0   GUID_BATTERY_DISCHARGE_ACTION_1   GUID_BATTERY_DISCHARGE_LEVEL_1   GUID_BATTERY_DISCHARGE_FLAGS_1   GUID_BATTERY_DISCHARGE_ACTION_2   GUID_BATTERY_DISCHARGE_LEVEL_2   GUID_BATTERY_DISCHARGE_FLAGS_2   GUID_BATTERY_DISCHARGE_ACTION_3   GUID_BATTERY_DISCHARGE_LEVEL_3   GUID_BATTERY_DISCHARGE_FLAGS_3   GUID_PROCESSOR_SETTINGS_SUBGROUP   GUID_PROCESSOR_THROTTLE_POLICY   GUID_PROCESSOR_THROTTLE_MAXIMUM   GUID_PROCESSOR_THROTTLE_MINIMUM   GUID_PROCESSOR_ALLOW_THROTTLING   GUID_PROCESSOR_IDLESTATE_POLICY   GUID_PROCESSOR_PERFSTATE_POLICY   GUID_PROCESSOR_PERF_INCREASE_THRESHOLD   GUID_PROCESSOR_PERF_DECREASE_THRESHOLD   GUID_PROCESSOR_PERF_INCREASE_POLICY   GUID_PROCESSOR_PERF_DECREASE_POLICY   GUID_PROCESSOR_PERF_INCREASE_TIME   GUID_PROCESSOR_PERF_DECREASE_TIME   GUID_PROCESSOR_PERF_TIME_CHECK   GUID_PROCESSOR_PERF_BOOST_POLICY   GUID_PROCESSOR_PERF_BOOST_MODE   GUID_PROCESSOR_IDLE_ALLOW_SCALING   GUID_PROCESSOR_IDLE_DISABLE   GUID_PROCESSOR_IDLE_STATE_MAXIMUM   GUID_PROCESSOR_IDLE_TIME_CHECK   GUID_PROCESSOR_IDLE_DEMOTE_THRESHOLD   GUID_PROCESSOR_IDLE_PROMOTE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_DECREASE_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_INCREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_DECREASE_POLICY   GUID_PROCESSOR_CORE_PARKING_MAX_CORES   GUID_PROCESSOR_CORE_PARKING_MIN_CORES   GUID_PROCESSOR_CORE_PARKING_INCREASE_TIME   GUID_PROCESSOR_CORE_PARKING_DECREASE_TIME   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_AFFINITY_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_AFFINITY_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_DECREASE_FACTOR   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_HISTORY_THRESHOLD   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_WEIGHTING   GUID_PROCESSOR_CORE_PARKING_OVER_UTILIZATION_THRESHOLD   GUID_PROCESSOR_PARKING_CORE_OVERRIDE   GUID_PROCESSOR_PARKING_PERF_STATE   GUID_PROCESSOR_PARKING_CONCURRENCY_THRESHOLD   GUID_PROCESSOR_PARKING_HEADROOM_THRESHOLD   GUID_PROCESSOR_PERF_HISTORY   GUID_PROCESSOR_PERF_LATENCY_HINT   GUID_PROCESSOR_DISTRIBUTE_UTILITY   GUID_SYSTEM_COOLING_POLICY   GUID_LOCK_CONSOLE_ON_WAKE   GUID_DEVICE_IDLE_POLICY   GUID_ACDC_POWER_SOURCE   GUID_LIDSWITCH_STATE_CHANGE   GUID_BATTERY_PERCENTAGE_REMAINING   GUID_GLOBAL_USER_PRESENCE   GUID_SESSION_DISPLAY_STATUS   GUID_SESSION_USER_PRESENCE   GUID_IDLE_BACKGROUND_TASK   GUID_BACKGROUND_TASK_NOTIFICATION   GUID_APPLAUNCH_BUTTON   GUID_PCIEXPRESS_SETTINGS_SUBGROUP   GUID_PCIEXPRESS_ASPM_POLICY   GUID_ENABLE_SWITCH_FORCED_SHUTDOWN   PPM_PERFSTATE_CHANGE_GUID   PPM_PERFSTATE_DOMAIN_CHANGE_GUID   PPM_IDLESTATE_CHANGE_GUID   PPM_PERFSTATES_DATA_GUID   PPM_IDLESTATES_DATA_GUID   PPM_IDLE_ACCOUNTING_GUID   PPM_IDLE_ACCOUNTING_EX_GUID   PPM_THERMALCONSTRAINT_GUID   PPM_PERFMON_PERFSTATE_GUID   PPM_THERMAL_POLICY_CHANGE_GUID   VIRTUAL_STORAGE_TYPE_VENDOR_MICROSOFT 

  RPC_IF_HANDLE B  IWinTypesBase_v0_1_c_ifspec )  IWinTypesBase_v0_1_s_ifspec *  IID_IUnknown 
W  IID_AsyncIUnknown 
  IID_IClassFactory 
m  IID_IMarshal i  IID_INoMarshal P  IID_IAgileObject   IID_IMarshal2   IID_IMalloc V  IID_IStdMarshalInfo 
  IID_IExternalConnection p  IID_IMultiQI   IID_AsyncIMultiQI B  IID_IInternalUnknown   IID_IEnumUnknown   IID_IEnumString   IID_ISequentialStream F  IID_IStream   IID_IRpcChannelBuffer 5	  IID_IRpcChannelBuffer2 	  IID_IAsyncRpcChannelBuffer a
  IID_IRpcChannelBuffer3 #  IID_IRpcSyntaxNegotiate =  IID_IRpcProxyBuffer   IID_IRpcStubBuffer   IID_IPSFactoryBuffer 
  IID_IChannelHook C  IID_IClientSecurity g  IID_IServerSecurity   IID_IRpcOptions   IID_IGlobalOptions R  IID_ISurrogate   IID_IGlobalInterfaceTable -  IID_ISynchronize   IID_ISynchronizeHandle 0  IID_ISynchronizeEvent   IID_ISynchronizeContainer   IID_ISynchronizeMutex S  IID_ICancelMethodCalls   IID_IAsyncManager .  IID_ICallFactory   IID_IRpcHelper 
  IID_IReleaseMarshalBuffers u  IID_IWaitMultiple   IID_IAddrTrackingControl <  IID_IAddrExclusionControl   IID_IPipeByte   IID_IPipeLong }  IID_IPipeDouble   IID_IComThreadingInfo   IID_IProcessInitControl V  IID_IFastRundown   IID_IMarshalingStream   IID_ICallbackWithNoReentrancyToApplicationSTA   GUID_NULL 
  CATID_MARSHALER   IID_IRpcChannel   IID_IRpcStub   IID_IStubManager   IID_IRpcProxy   IID_IProxyManager   IID_IPSFactory   IID_IInternalMoniker   IID_IDfReserved1   IID_IDfReserved2   IID_IDfReserved3   CLSID_StdMarshal   CLSID_AggStdMarshal   CLSID_StdAsyncActManager   IID_IStub   IID_IProxy   IID_IEnumGeneric   IID_IEnumHolder   IID_IEnumCallback    IID_IOleManager !  IID_IOlePresObj "  IID_IDebug #  IID_IDebugStream $  CLSID_PSGenObject %  CLSID_PSClientSite &  CLSID_PSClassObject '  CLSID_PSInPlaceActive (  CLSID_PSInPlaceFrame )  CLSID_PSDragDrop *  CLSID_PSBindCtx +  CLSID_PSEnumerators ,  CLSID_StaticMetafile -  CLSID_StaticDib .  CID_CDfsVolume /  CLSID_DCOMAccessControl 0  CLSID_StdGlobalInterfaceTable 1  CLSID_ComBinding 2  CLSID_StdEvent 3  CLSID_ManualResetEvent 4  CLSID_SynchronizeContainer 5  CLSID_AddrControl 6  CLSID_CCDFormKrnl 7  CLSID_CCDPropertyPage 8  CLSID_CCDFormDialog 9  CLSID_CCDCommandButton :  CLSID_CCDComboBox ;  CLSID_CCDTextBox <  CLSID_CCDCheckBox =  CLSID_CCDLabel >  CLSID_CCDOptionButton ?  CLSID_CCDListBox @  CLSID_CCDScrollBar A  CLSID_CCDGroupBox B  CLSID_CCDGeneralPropertyPage C  CLSID_CCDGenericPropertyPage D  CLSID_CCDFontPropertyPage E  CLSID_CCDColorPropertyPage F  CLSID_CCDLabelPropertyPage G  CLSID_CCDCheckBoxPropertyPage H  CLSID_CCDTextBoxPropertyPage I  CLSID_CCDOptionButtonPropertyPage J  CLSID_CCDListBoxPropertyPage K  CLSID_CCDCommandButtonPropertyPage L  CLSID_CCDComboBoxPropertyPage M  CLSID_CCDScrollBarPropertyPage N  CLSID_CCDGroupBoxPropertyPage O  CLSID_CCDXObjectPropertyPage P  CLSID_CStdPropertyFrame Q  CLSID_CFormPropertyPage R  CLSID_CGridPropertyPage S  CLSID_CWSJArticlePage T  CLSID_CSystemPage U  CLSID_IdentityUnmarshal V  CLSID_InProcFreeMarshaler W  CLSID_Picture_Metafile X  CLSID_Picture_EnhMetafile Y  CLSID_Picture_Dib Z  GUID_TRISTATE [  IWinTypes_v0_1_c_ifspec (  IWinTypes_v0_1_s_ifspec )  IID_IMallocSpy   IID_IBindCtx :  IID_IEnumMoniker J   IID_IRunnableObject    IID_IRunningObjectTable !  IID_IPersist i"  IID_IPersistStream "  IID_IMoniker j#  IID_IROTData X%  IID_IEnumSTATSTG %  IID_IStorage X&  IID_IPersistFile A(  IID_IPersistStorage (  IID_ILockBytes )  IID_IEnumFORMATETC *  IID_IEnumSTATDATA l+  IID_IRootStorage ,  IID_IAdviseSink ,  IID_AsyncIAdviseSink s-  IID_IAdviseSink2 .  IID_AsyncIAdviseSink2 ./  IID_IDataObject /  IID_IDataAdviseHolder 1  IID_IMessageFilter 1  FMTID_SummaryInformation ]2  FMTID_DocSummaryInformation _2  FMTID_UserDefinedProperties a2  FMTID_DiscardableInformation c2  FMTID_ImageSummaryInformation e2  FMTID_AudioSummaryInformation g2  FMTID_VideoSummaryInformation i2  FMTID_MediaFileSummaryInformation k2  IID_IClassActivator s2  IID_IFillLockBytes 2  IID_IProgressNotify 3  IID_ILayoutStorage 3  IID_IBlockingLock 4  IID_ITimeAndNoticeControl 4  IID_IOplockStorage N5  IID_IDirectWriterLock 5  IID_IUrlMon M6  IID_IForegroundTransfer 6  IID_IThumbnailExtractor 7  IID_IDummyHICONIncluder 7  IID_IProcessLock 7  IID_ISurrogateService H8  IID_IInitializeSpy 8  IID_IApartmentShutdown 9  IID_IOleAdviseHolder   IID_IOleCache b  IID_IOleCache2 )  IID_IOleCacheControl   IID_IParseDisplayName <  IID_IOleContainer   IID_IOleClientSite   IID_IOleObject   IOLETypes_v0_0_c_ifspec   IOLETypes_v0_0_s_ifspec   IID_IOleWindow $  IID_IOleLink   IID_IOleItemContainer   IID_IOleInPlaceUIWindow v	  IID_IOleInPlaceActiveObject 
  IID_IOleInPlaceFrame 
  IID_IOleInPlaceObject   IID_IOleInPlaceSite   IID_IContinue 
  IID_IViewObject 
  IID_IViewObject2 *  IID_IDropSource   IID_IDropTarget [  IID_IDropSourceNotify   IID_IEnumOLEVERB v  IID_IServiceProvider M  IOleAutomationTypes_v1_0_c_ifspec   IOleAutomationTypes_v1_0_s_ifspec   IID_ICreateTypeInfo ;  IID_ICreateTypeInfo2 b  IID_ICreateTypeLib   IID_ICreateTypeLib2   IID_IDispatch 	  IID_IEnumVARIANT 
  IID_ITypeComp 5  IID_ITypeInfo   IID_ITypeInfo2 P  IID_ITypeLib   IID_ITypeLib2 =  IID_ITypeChangeEvents a  IID_IErrorInfo   IID_ICreateErrorInfo }  IID_ISupportErrorInfo    IID_ITypeFactory u  IID_ITypeMarshal   IID_IRecordInfo   IID_IErrorLog    IID_IPropertyBag z  __MIDL_itf_msxml_0000_v0_0_c_ifspec   __MIDL_itf_msxml_0000_v0_0_s_ifspec   LIBID_MSXML   IID_IXMLDOMImplementation    IID_IXMLDOMNode '  IID_IXMLDOMDocumentFragment   IID_IXMLDOMDocument f  IID_IXMLDOMNodeList u  IID_IXMLDOMNamedNodeMap   IID_IXMLDOMCharacterData   IID_IXMLDOMAttribute   IID_IXMLDOMElement   IID_IXMLDOMText   IID_IXMLDOMComment %  IID_IXMLDOMProcessingInstruction   IID_IXMLDOMCDATASection   IID_IXMLDOMDocumentType   IID_IXMLDOMNotation   IID_IXMLDOMEntity   IID_IXMLDOMEntityReference   IID_IXMLDOMParseError a	  IID_IXTLRuntime 	  DIID_XMLDOMDocumentEvents =
  CLSID_DOMDocument \
  CLSID_DOMFreeThreadedDocument `
  IID_IXMLHttpRequest g
  CLSID_XMLHTTPRequest 
  IID_IXMLDSOControl 
  CLSID_XMLDSOControl 
  IID_IXMLElementCollection   IID_IXMLDocument J  IID_IXMLDocument2   IID_IXMLElement $  IID_IXMLElement2   IID_IXMLAttribute   IID_IXMLError 
  CLSID_XMLDocument .
  CLSID_SBS_StdURLMoniker ~  CLSID_SBS_HttpProtocol   CLSID_SBS_FtpProtocol   CLSID_SBS_GopherProtocol   CLSID_SBS_HttpSProtocol   CLSID_SBS_FileProtocol   CLSID_SBS_MkProtocol   CLSID_SBS_UrlMkBindCtx   CLSID_SBS_SoftDistExt   CLSID_SBS_CdlProtocol   CLSID_SBS_ClassInstallFilter   CLSID_SBS_InternetSecurityManager   CLSID_SBS_InternetZoneManager   IID_IAsyncMoniker   CLSID_StdURLMoniker   CLSID_HttpProtocol   CLSID_FtpProtocol   CLSID_GopherProtocol   CLSID_HttpSProtocol   CLSID_FileProtocol   CLSID_MkProtocol   CLSID_StdURLProtocol   CLSID_UrlMkBindCtx   CLSID_CdlProtocol   CLSID_ClassInstallFilter   IID_IAsyncBindCtx   IID_IPersistMoniker P  IID_IMonikerProp !  IID_IBindProtocol   IID_IBinding   IID_IBindStatusCallback u  IID_IBindStatusCallbackEx   IID_IAuthenticate d  IID_IAuthenticateEx   IID_IHttpNegotiate A  IID_IHttpNegotiate2   IID_IHttpNegotiate3 ;	  IID_IWinInetFileStream 	  IID_IWindowForBindingUI 0
  IID_ICodeInstall 
  IID_IUri -  IID_IUriContainer 
  IID_IUriBuilder 
  IID_IUriBuilderFactory (  IID_IWinInetInfo   IID_IHttpSecurity   IID_IWinInetHttpInfo y  IID_IWinInetHttpTimeouts   IID_IWinInetCacheHints Z  IID_IWinInetCacheHints2   SID_BindHost 5  IID_IBindHost ?  IID_IInternet M  IID_IInternetBindInfo   IID_IInternetBindInfoEx &  IID_IInternetProtocolRoot   IID_IInternetProtocol   IID_IInternetProtocolEx W  IID_IInternetProtocolSink   IID_IInternetProtocolSinkStackable   IID_IInternetSession ?  IID_IInternetThreadSwitch H  IID_IInternetPriority   IID_IInternetProtocolInfo N  CLSID_InternetSecurityManager   CLSID_InternetZoneManager   CLSID_PersistentZoneIdentifier   IID_IInternetSecurityMgrSite   IID_IInternetSecurityManager i  IID_IInternetSecurityManagerEx   IID_IInternetSecurityManagerEx2 V  IID_IZoneIdentifier    IID_IInternetHostSecurityManager !  GUID_CUSTOM_LOCALMACHINEZONEUNLOCKED t"  IID_IInternetZoneManager "  IID_IInternetZoneManagerEx L$  IID_IInternetZoneManagerEx2 ]%  CLSID_SoftDistExt &  IID_ISoftDistExt &  IID_ICatalogFileInfo x'  IID_IDataFilter '  IID_IEncodingFilterFactory (  GUID_CUSTOM_CONFIRMOBJECTSAFETY 3)  IID_IWrappedProtocol A)  IID_IGetBindHandle )  IID_IBindCallbackRedirect 
*  IID_IPropertyStorage   IID_IPropertySetStorage   IID_IEnumSTATPROPSTG   IID_IEnumSTATPROPSETSTG D  IID_StdOle   GUID_DEVINTERFACE_DISK   GUID_DEVINTERFACE_CDROM 
  GUID_DEVINTERFACE_PARTITION   GUID_DEVINTERFACE_TAPE   GUID_DEVINTERFACE_WRITEONCEDISK   GUID_DEVINTERFACE_VOLUME   GUID_DEVINTERFACE_MEDIUMCHANGER   GUID_DEVINTERFACE_FLOPPY   GUID_DEVINTERFACE_CDCHANGER   GUID_DEVINTERFACE_STORAGEPORT   GUID_DEVINTERFACE_COMPORT   GUID_DEVINTERFACE_SERENUM_BUS_ENUMERATOR   _SCARD_IO_REQUEST MH  dwProtocol    cbPciLength    SCARD_IO_REQUEST H  MH  g_rgSCardT0Pci %eH  g_rgSCardT1Pci %eH  g_rgSCardRawPci %eH  IID_IPrintDialogCallback   IID_IPrintDialogServices   __security_cookie |     I         _pRawDllMain 9I  I  3I  DllMain   `m           hDllHandle   RdwReason   Qlpreserved   X  #     GNU C99 6.2.1 20161119 -m64 -mtune=generic -march=x86-64 -g -O2 -std=gnu99 ./mingw-w64-crt/stdio/vsnprintf.c pm           X  __gnuc_va_list -   __builtin_va_list    char    va_list k   size_t #   long long unsigned int long long int short unsigned int int long int    unsigned int sizetype long unsigned int unsigned char __ms_vsnprintf 
'  pm             s 
:  T-  n 
   -  format 
  -  arg 
   -  	um      
RR
QQ
XX
YY                                                                                                                                                                                                          %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  5 I  I  ! I/  :;  
 :;I8  '   I  !   4 :;I?<     
 I8  'I   'I   '  4 :;I?<  :;  :;  
 :;I   
 I  !:;  "
 I8  #I:;  $(   %I:;  &(   ':;  (
 :;I
8  )I:;  * :;I  +5   ,4 :;I  -4 :;I?  ..:;'I@B  / :;I  04 :;I  1
 :;  2 1  31  4 B  51  6.?:;'I@B  7B1  8B1  9.?:;'I@B  :U  ;4 :;I  <1XY  =1XY  > 1  ?  @4 1  A1RUXY  B 1  C  D   EU  F4 :;I  G  H. ?:;'I   I.?:;'I   J :;I  K4 :;I  L. ?<n:;  M. ?<n:;  N. ?<n:;   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  5 I  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<      'I   '  4 :;I?<  I:;  (   I:;  (   :;   
 :;I
8  !I:;  " :;I  #5   $4 :;I?  %'I  &.?:;'I@B  ' :;I  (1  ) B  *.?:;'I@B  +4 :;I  ,4 :;I  - 1  .1  /  0. ?<n:;  1. ?<n:;   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<      '  4 :;I?<  I:;  (   I:;  (   4 :;I  .?:;'@B   B1   .?:;'@B  !4 :;I  "B1  # B  $. ?<n:;   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  5 I  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     4 :;I?<  I:;  (   I:;  (   :;  
 :;I
8  I:;    :;I  !5   "4 G:;   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  4 :;I?<  4 :;I?<  '   I  :;  
 :;I8  !      
 I8  5 I  :;  
 I  
 :;I  :;  :;  :;   
 :;I  !4 :;I  "4 :;I?  #.?:;'@B  $ :;I  %4 :;I  &1  ' B  ( 1  ).?:;'@B  *1  +. ?<n:;  ,. ?<n:;  -. ?<n:;   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<      '  4 :;I?<  4 :;I?  4 :;I  .:;'I@B   :;I   1  .?:;'I@B   :;I   .?:;'I   ! :;I  "4 :;I  #.1@B  $ 1  %4 1  &  '  (4 1  ). ?<n:;   %  $ >   :;I   I   '  I  ! I/  4 :;I?   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  5 I  I  ! I/  :;  
 :;I8  '   I  !   4 :;I?<     4 :;I?<  I:;  (   I:;  (   :;  
 :;I
8  I:;    :;I  !5   "4 :;I?  #.?:;'I   $ :;I  %.?:;'I  &.1@B  ' 1   %   :;I   I  $ >  & I   :;I   I  :;  	
 :;I8  

 :;I8  
 :;I8   <  
:;  :;  5 I  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     &   4 :;I?<  :;  
 :;I  I:;  (   I:;   (   !:;  "
 :;I
8  #I:;  $ :;I  %5   &:;  '4 :;I  (.?:;'@B  )4 :;I  *4 :;I  +1RUXY  , 1  -U  .4 1  /4 1  0 1  11  2 B  3  41RUXY  51  6 1  7.:;'   8 :;I  94 :;I  :  ;.:;'   <4 :;I  = :;I  >.:;'@B  ? :;I  @   A.1@B  BB1  C. ?<n:;  D. ?<n:;n  E. ?<n:;  F. ?<n:;  G. ?<n   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     
 I8  4 :;I?<  :;  :;  
 :;I  
 I  :;  'I  I:;   (   !I:;  "(   #4 :;I  $4 :;I?  %.?:;'I@B  & :;I  '4 :;I  (4 :;I  )1  * B  +  ,B  - 1  . :;I  /1  0. ?<n:;  1. ?<n:;  2. ?<n:;   %  $ >   :;I  & I  5 I   :;I   I  :;  	
 :;I8  

 :;I8  
 :;I8   <  
:;  :;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     4 :;I?<  4 :;I  .?:;'I@B   :;I    4 :;I   1  1    B  !1XY  "  #4 1  $1RUXY  %U  &.:;'   '4 :;I  (  )1  *.1@B  +4 1  ,  -B1  .. ?<n:;  /. ?<n:;   %  4 :;I?  $ >   %  4 :;I?  $ >   %  $ >  & I   :;I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     4 :;I?<  :;  
 :;I  
 :;I  
 I8  .?:;'I@B   :;I  4 :;I  4 :;I   4 :;I  !1RUXY  " 1  #U  $4 1  %1  & B  '1XY  ( 1  )  *4 :;I  +1RUXY  , 1  -1  ..?:;'I   / :;I  0 :;I  14 :;I  2.1@B  34 1  44 1  5 B1  6. ?<n:;   %  . ?:;'@B   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     4 :;I?<  'I  .?:;'I@B   :;I   %  $ >   :;I  & I   :;I   I  :;  
 :;I8  	
 :;I8  

 :;I8   <  :;  
:;  I  ! I/  '   I  :;  
 :;I8  !   4 :;I?<     4 :;I?<  'I  .?:;'I@B   :;I   %   :;I   I  $ >  & I   I  .?:;'I@B   :;I  	B1  
 B  . ?<n:;                                                                                                                                        {  
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include/psdk_inc ./debian/tmp/usr/x86_64-w64-mingw32/include ./mingw-w64-crt/include  crtdll.c   intrin-impl.h   crtdefs.h   locale.h   excpt.h   winnt.h   minwindef.h   ctype.h   basetsd.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   internal.h   process.h   synchapi.h     	 m    >[;=:1-vc[-g0w.fxXx\"Yf <X.{Jx.xttxV.$tY8;=0/L,[x.fx<YFxhz
/?u*/ .XɁ=A:h`Y>ZYL-=0YYY-=hYKu;g6I7Z2I I     
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include ./mingw-w64-crt/include  atonexit.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   internal.h     	͑m    !XAHZ0sYZZZV>YWuYWuZuK      
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  gccmain.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   guiddef.h   stdlib.h   malloc.h   winnt.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h     	Αm    M/Mq]g91 3  d]uex .      
      ./debian/tmp/usr/x86_64-w64-mingw32/include ./mingw-w64-crt/include ./mingw-w64-crt/crt  crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   internal.h   natstart.c       +  
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  gs_support.c   crtdefs.h   stdlib.h   malloc.h   excpt.h   winnt.h   minwindef.h   ctype.h   basetsd.h   guiddef.h   virtdisk.h   errhandlingapi.h   processthreadsapi.h   sysinfoapi.h   profileapi.h     	ϑm    2Z dȮ^hV0
<tY2C?uuXCu;=h6
tt.|vtg=dX       
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  tlssup.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   virtdisk.h     	`ёm    P,ZkhZXxu*s ;    5   
      ./mingw-w64-crt/crt  cinitexe.c    T   8  
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include ./mingw-w64-crt/include  mingw_helpers.c   crtdefs.h   locale.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   internal.h   process.h   math.h     	 ґm    > d   \  
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include ./mingw-w64-crt/include  pseudo-reloc.c   vadefs.h   crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   internal.h   memoryapi.h   stdio.h   errhandlingapi.h   <built-in>      	0ґm     ?fAX? Z J< ffL9;=kd>VLmuW=Z$ t	wXteuwznY~f=~nfg
fL!<bA	.wJ7<x<D ȑNJ~{u0 A=M=z.[h=gh=yssX[=~ruI
      
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  crt_handler.c   crtdefs.h   excpt.h   winnt.h   minwindef.h   ctype.h   basetsd.h   guiddef.h   stdlib.h   malloc.h   errhandlingapi.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   combaseapi.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   signal.h     	 בm     O7u?.4ֻI=5@< X1u.ֻ[u\ XhXY.eXI=hYt<XNXktC䢄 [sYhus	Xzt+gx<>OF>z< f ^z<B_5[ f  >Vt2[<Xf m0`fgm,
    &  
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  tlsthrd.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   minwinbase.h   virtdisk.h   synchapi.h   processthreadsapi.h   errhandlingapi.h     	pۑm     u0xJXuȅg*[Jr;=1/LwrsMht\b2JivZsJXu_u*<YiJ/X_J ֯KUrofXet :    4   
      ./mingw-w64-crt/crt  tlsmcrt.c    D    >   
      ./mingw-w64-crt/crt  pseudo-reloc-list.c         
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  pesect.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   basetsd.h   winnt.h   guiddef.h   stdlib.h   malloc.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   string.h     	ݑm    	uq<	tvIH`/tCT-tS.=ts.MvIHHxX.J tz.< KJ8JHf>IH4X2J t< tz.MLgJ tx.< oMvIhZzJHT.^~Jt~.-g&~Jty.~<yJJ~f?W"H X4WC~Jtw<~<cMvg~JINHX<q
KzJ twYz O    5   
      ./mingw-w64-crt/crt  CRT_fp10.c     	 m    	      
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  dllentry.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   winnt.h   guiddef.h   stdlib.h   malloc.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   process.h     	Pm          
      ./mingw-w64-crt/crt ./debian/tmp/usr/x86_64-w64-mingw32/include  dllmain.c   crtdefs.h   excpt.h   minwindef.h   ctype.h   winnt.h   guiddef.h   stdlib.h   malloc.h   virtdisk.h   rpcdce.h   wtypesbase.h   unknwnbase.h   objidlbase.h   cguid.h   wtypes.h   objidl.h   oleidl.h   servprov.h   oaidl.h   msxml.h   urlmon.h   propidl.h   oleauto.h   winioctl.h   winsmcrd.h   winscard.h   commdlg.h   process.h     	`m            
      ./mingw-w64-crt/stdio ./debian/tmp/usr/x86_64-w64-mingw32/include  vsnprintf.c   stdarg.h   crtdefs.h   stdio.h     	pm    
      x       4        m    M       AD0{
AAIA              Pm    ;      BBA A(A0A8D`q
8A0A(A ABBK
8A0A(A ABBA
8A0A(A ABBE  L       m    9      BAA A(A0DPn
0A(A AABB $       m    O       DPV
Fj        x       4   `  ͑m           AD@
AAVA         `  Αm           D0T      x            Αm    5       D0p  4     Αm    f       AAD@@
AAH         `ϑm               x       l   X  ϑm           BAA A(A0Dpm
0A(A AABCs
0A(A AABG       ,   X  `Бm           AAA C
          x       $     `ёm    /       D0R
JN    D     ёm    i       AAD@e
AACo
AAA             ґm               x            ґm                 ґm               x       $     0ґm    a       AADP   \     ґm    `      BBA A(A0A8D
8A0A(A ABBE       \      ԑm          ABB B(B0A8A@AH	DH0O
AABBBBAA      x       <      בm          D0c
IK
EW
AZ
Ax
A    L     ؑm           BAA A(A0DP{
0A(A AABE  T     ّm          AD0|
AAe
AGF
AIS
AA}
AA        x       D     pۑm    j       AAA A(DPW(A AAA     \     ۑm           AAA A(DPW
(A AAAAO
(A AAAA    4     `ܑm           AD0R
AHF
AI   4      ݑm           AD0y
AA}
AB       x          (  ݑm              (   ޑm              (   ޑm    E       L   (  pޑm           AAA D@e
 AAAAW AAA    $   (   ߑm           D0[
Af   $   (  ߑm    >       D0X
D]    $   (  ߑm    r       D0[
AQ   $   (  Pm    6       D0X
DU    ,   (  m           D0X
Dc
AL     ,   (  0m           D0\
Af
Iy         x             m               x          	  Pm               x          8	  `m               x          h	  pm                                                                                                                   _decode_pointer onexitbegin _pei386_runtime_relocator __mingw_init_ehandler lock_free __enative_startup_state dwReason hDllHandle __security_init_cookie DllEntryPoint _amsg_exit lpreserved _encode_pointer refcount _initterm _encode_pointer refcount _decode_pointer __enative_startup_state __dllonexit refcount __enative_startup_state refcount GetCurrentProcessId SetUnhandledExceptionFilter HighPart ExceptionRecord RtlCaptureContext RtlVirtualUnwind TerminateProcess RtlLookupFunctionEntry GetCurrentThreadId GetSystemTimeAsFileTime QueryPerformanceCounter UnhandledExceptionFilter GetTickCount refcount GetCurrentProcess dwReason refcount hDllHandle lpreserved __mingw_TLScallback __enative_startup_state refcount _GetPEImageBase VirtualProtect __mingw_GetSectionCount sSecInfo vfprintf __mingw_GetSectionForAddress __enative_startup_state GetLastError VirtualQuery refcount __iob_func _GetPEImageBase old_handler _FindPESectionExec RtlAddFunctionTable refcount ContextRecord reset_fpu _FindPESectionByName ExceptionRecord _fpreset InitializeCriticalSection GetLastError TlsGetValue refcount LeaveCriticalSection EnterCriticalSection DeleteCriticalSection _fpreset pSection TimeDateStamp pNTHeader Characteristics pImageBase VirtualAddress iSection refcount refcount refcount _vsnprintf                                                                                                                                                                                                                                                                     R             U             R             R      %       U%      )       R)             U                             Q      	       S	             Q             Q      %       S%      )       Q)             S                             X             V             X             V                             1             \             P             \             0             \             P      *       1*      :       P@      S       PS      V       \V      i       Pi             \             P             \             P             \             P                             R             R             R             X                             Q             Q             Q             d                             X             X             X             h                P               R               \               R       	       R	             \             R             R             \             R             \                P               Q               Q       	       Q	             Q             Q             Q                P               X               ]               X       	       X	             ]             X             X             ]             X             ]                       	       0                       /       T      %       T                       /       0/             T      *       0*             T                               0                       (       0                       (       T                *      :       0                             P             V                             T             t             T                             0                               0                               1                               P       A        SB       L        S                               R               R                                R       e        Se               R               S               R                e       i        Pi               S               P                U       `        P`       g        Q               P               Q               P                W       `        P`       g        Q                               R              T                      !       P!      R       Y             P             Y                P       Y        UY       ^        p u '^       a        | u 'a       f        | p 'u 'f       n        | v 'u 'n       s        | v 'p 'u 's       y        | v 't 'u '               P               t u '?                        $        R$       /        R                        $        Q$       /        Q                        $        X$       /        X                0       R        RR       ^        R^       s        Rs               R               R               R                0       R        QR       ^        Q^       s        Qs               Q               Q               Q                0       R        XR       ^        X^       s        Xs               X               X               X                ^       s        Rs               R                ^               2                ^       s        Xs               X                s               S               sx                ^       g       
 Hm    g               S                             P                             P             X             p              X             p %      B       XB      I       p Y      l       Xm             X             p                 V             P      %       R             P             s} "             s|} "             R             R%      S       RY      `       R`      l       s} "m             R             P                             S             st             S                      /       S             S                I      Y       2                I      Y       ^                I      S       R                      &       8                      &       ^                      %       R                             S             sx             S                             4                             U                             R                      %       1                      %       ^                             R                             4                             ^                             R                6      T       0T             T             T                        &        R&       a        S                p               R       }       T}             R             R             T                p               Q       ~       U~             Q             Q             Q             U                p               X              V             X             X             V                               R       n       T             T                               P       n       \             \             P             \                               0               S             0                             R             S             R      9       S9      ;       R;      <       R<             S             R             S             R             S             R      b       S                             Pg      |       P             P             P      !       P,      >       PI      W       P                             0      
       	      }       0}             	             0             	      "       0"      ,       	,      ?       0?      I       	I      b       0                      
       0      %       0<      [       0[      q       1             0      I       0I      b       1                        B        RB       h        Rh               R               R               R               R              R      3       R3      :       R:             R                        =        Q=       h        Qh               Q               Q               Q               Q              Q      3       Q3      5       Q5             Q                        F        XF       h        Xh               X               X               X               X              X      3       X3      >       X>             X                        F        YF       h        Yh               Y               Y               Y               Y              Y      3       Y3      >       Y>             Y                       a        1a       c        0h               1               0               1               0       {       1{             0             1             0                G       `        P               P               Q               P      &       PF      Z       Pb      g       Pl      z       P             P                       c        0h               0               1              0      ,       13             0                             0      %       0%             T                %      @       P[      k       P                             P             V             V                             R             R             R             R             R      *       R*      :       R:      O       RO      W       RW      _       R_      d       Rd      j       R                             Q             Q             Q             Q             Q      *       Q*      :       Q:      O       QO      \       Q\      _       Q_      d       Qd      j       Q                             X             X             X             X             X      *       X*      :       X:      O       XO      \       X\      _       X_      d       Xd      j       X                             R             S      
       R                             S                              R      W       SW      X       RX             S                      1       01      C       PX      h       P|             0                $      1       P1      :       R:      >       P>      C       RX      l       R|             P                p               R               V               R               R               V               R               V                p               Q               U               Q               Q               U               Q               U                               P               S               P                       `        S                7       8        P8       T        T                P      i       Ri      q       Xq      t       Rt             X                             R             {< $ &{ "                             { p "             P                             P             {< $ &{ "#             P             {< $ &{ "#                             P                             R             Q      A       R                      4       R                             R             X             R      b       X                #      0       P0      b       y< $ &y "                ,      Z       Q                #      @       0                       8       R8      @       R@      C       RC      v       Xv             R                               R               T               R              T             R                               P                              S                               0                                R               R                               r                        1        R1       2        R                        1        R1       2        R                @       G        RG               R                G       T        RT               R#< $ &R"                P               P                G       d        0                                R               R                                Q               Q                                X               X                                Y               Y                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        ~                                  
                      ~                                                                            
                                                                                     (                                           n                                  )      7      P      6                                                          (      +      B      E      I      Y                                                    &                                                                                                                                    %                      o      r                                              6                                                    `      j                                                                                $      +      0      8      @      H                      L      P      V                                                                                                                                     t      {      }                                                                                                              -                      T      [      a      i      q      y                                                                                                                                      .file   <     gcrtdll.c              j                                u   C                           C                           P                             C                           C                          PC                        9   D                        O  C                        e  C                        {  C                                      C                                      D                    .text            *             .data                            .bss                            .xdata         4                 .pdata         0                         	                             
   d  n                                                                            +         0                    :                             H                            T                              _  D                          j         `               .file   Q     gcygming-crtbegin.c    w              0                                   .text         v                .data                            .bss                            .xdata  4                       .pdata  0      $   	             .rdata         "                     _  D                      .jcr          .file   y     gral.c             Ral_Init                             @D                                                                               p           (  j          4  `           H  T           X              c  c          o            |                                       0D                           D                          pD                    .text                       .data          @                .bss                            .rdata  @      .                .xdata  P                        .pdata  T      0                    _  D                      .file        gral_attribute.c                                                                         %  	      isAnInt `	      intEqual	          3  	      intHash @
      isALong 
          >  
          H  0      longHash          T             ^             j            x  
            
            
                                                PD                          P                              isAList       isADict              @            P      dictHash      listHash                      )            7            C             O  p          Y            j            y  P                                                                                                               @          ,            C            V  p          n  B                           C                                                                                     1            G             `  0           v               !            P"             #            #             $      .text                       .data   @                       .bss                            .xdata  p                      .pdata                        .rdata         2                 _  D                      .file        gral_joinmap.c           $                              $          +  $          :  P%          L  %          c  %          {  P&             '            `'            '            (            p(            @          	  (           	   )          7	  P)      .text   $                    .data   `                        .bss                            .xdata                        .pdata          -             .rdata  @                         _   E                      .file   ?    gral_relation.c        S	  )                           g	   *          	  P*          	  *          	  *          	  *          	  *          	  *          	   +          
  @+          4
  +          E
  @,          R
  ,          c
  `-          
  `           
   .          
             
  p.          
  .          
  @/          
  0            0            1          /   2          A  @3          U  3          g  6          {  P9            :            <            0?            ?            ?            @            A          !  C          5  E          J  F          `  F          s   J            J            `K             L            L             O            O            P          

  pQ          
  Q          1
  Q          F
  0R          a
  R          x
  R          
   S          
  S          
  T          
   U          
  `V            V            W      .text   )     f.               .data   `                      .bss                            .xdata                       .pdata  8                    .rdata  P     $                    _   E                      .file       gral_relationcmd.c     4  @X                           C             P             _  @\          t  ^            `a            c            f            g            h            0i            i            j          #  n          6  Po          L  @p          Z             i  q          {  `u            w            y            z            |            @~            @            `                      %            6             H  0          [             h  @          z  P            Ћ                                    `            0                        @          	  @            `          /  p          ?  @          R  P          b             s                                                                                                               $            0        .text   @X     `              .data                           .bss                           .rdata       
               .xdata  L     D                .pdata       d                   _  @E                      .file       gral_relationobj.c     >  и                           U             l  0            `                        P            P                                          .text   и       2             .data         (                .bss                             .xdata                        .pdata  T	     x                .rdata  @                          _  `E                      .file       gral_relvar.c          1                             D             h            x                                                              P                         @          H            x                                       P                        0            P          !            :            U            l                                      P                                               .            N            j              P            0                                    0                        `          0  `D                        Q  @          a            w  P                                                 P                              .text        %               .data   `                       .bss                             .xdata  $                     .pdata  	     4               .rdata  `     @                   _  E                      .file       gral_relvarcmd.c         p                                        1            A             O            _  `          t  P            p                                                 P                                   p                    #            6            G  P         T            i           }                                    p                                            @           `          #           8            B        .text   p     :  {            .data   `                     .bss                             .rdata         <             .xdata                       .pdata        P  T                 _  E                      .file   W    gral_relvarobj.c       P                              g        opsTable %            !           `"           %           &           &           )           *         $  p.         >  p/         V  0         j  P2           3           P5           7           8           $                         =           C         5   L         S  N         p  O            P           PQ           Q           $            `Y           Y         ,  [         @  [         Y   \         u  p\           `]           ]           @^            _           _           _         :  @`         X  `         s   b           @b     .text         D  4            .data         P                .bss                             .rdata  `!        	             .xdata                       .pdata  P
       {                 _  E                      .file       gral_tuple.c             d                             e           `e           e           f           f           f            @g         2  g         E   h         ^  0i         t  pi           i            j           j           j           0k           k           @l           l            Pm         #   m         5   pn     .text   d    
  !             .data                          .bss                             .xdata       X                .pdata  <       E             .rdata  `%     2                     _  E                      .file       gral_tuplecmd.c        L   0o                          ]    q         k   q         ~   0s            s            0u            0w            x             |            }                                 !  P         !            #!           5!           D!       tupleCmd         S!  @(      .text   0o    1               .data                          .bss                             .rdata  %       "             .xdata  <                     .pdata  P        6                 _   F                      .file       gral_tupleheading.c    a!  p                          u!  Љ         !            !  @         !           !            !  0         "           "  `         1"  p         E"           \"           q"  0         "  `         "           "           "            "           #           ##  @         >#           S#  P         k#       .text   p      2             .data                          .bss                             .xdata       h                .pdata  (       E             .rdata  `)                         _   F                      .file       gral_tupleobj.c        #  `                          #           #           #           #  0         #  P         
$            '$  P         A$           \$  @         s$  P         $           $  p         $  0     .text   `      H             .data        (                .bss                             .xdata  $                      .pdata  <        *                 _  @F                      .file       gral_utils.c           $  0                          $  P         $           %            %  6          %%  `:          0%  <          ;%  -          H%  0         \%           s%           %        .text   0                   .data                          .bss                             .xdata        X                 .pdata       `                .rdata   +                        _  `F                      .file   M    gral_vector.c          %                            %        snprintf0         %  P         %           %           %           &           #&            5&  @         R&  `         e&           x&           &  0         &           &  Ъ         &            &            &           '  @         '           4'  0         L'  Ю         ^'            p'  0         '  p         '           '  @         '           '           '           (  б         (       buf.9816           ,(  @         =(           N(           b(  P         w(           (  д         (           (  `         (           (           (  @         (           )           &)            :)  `         L)           ^)           q)           )           )  P     buf.9939        .text         V             .data                          .bss                            .xdata  X     @                .pdata  D     |               .rdata   =                         _  F                      .file   _    gtclStubLib.c          )                        .text                      .data                          .bss                            .rdata  @     5                 .xdata                        .pdata                           _  F                      .file   q    g    )                )                        .text       ,               .data                          .bss    @                      .rdata  A                      .xdata                        .pdata                           _  F                      .file       gatonexit.c            )                            )  pC                    atexit       .text                       .data                          .bss    P                       .xdata                        .pdata                             d  
   R                                                                           +  0      0                    H       M                   T        M                     _  F                          j  `     p                .file       ggccmain.c             *                        p.92158           $*           6*   C                    __main  `         S*  P      .text                       .data                        .bss    P                      .xdata                        .pdata       $   	                   K  
   L                     z                            	                          +  `      0                    H                          T  -     	                     _   G                          j                       .file       gnatstart.c        .text                         .data                         .bss    `                              
   P  
                   V                         +                             H  
                         T  6     !                     _   G                      .file       ggs_support.c          _*                            v*                           *                            *  `         *  `          *  @	          *  A      .text                      .data                          .bss    `     x                .xdata                         .pdata                       .rdata  A                           T 
   u+  ;                   	     T                       k
                         +        0                    H  '
                        T  W                         _  @G                          j  X                     .file   *    gtlssup.c              *  `                          +           +  C                    __xd_a  H   	    __xd_z  P   	        2+        .text   `                    .data                          .bss    	                      .xdata                        .pdata  ,     $   	             .CRT$XLD8   	                   .CRT$XLC0   	                   .rdata  A                     .CRT$XDZP   	                    .CRT$XDAH   	                    .tls        
   (                .CRT$XLZ@   	                    .CRT$XLA(   	                    .tls$ZZZ`   
                    .tls$AAA    
                          
 
   "  -                   I                            D     Y                   +        0                    H                          T  o     <                     _  `G                          j                       .file   D    gcinitexe.c        .text                         .data                          .bss    	                       .CRT$XCZ   	                    .CRT$XCA    	                    .CRT$XIZ    	                    .CRT$XIA   	                          ɢ 
                        d     Y                     +                            H       ?                     _  G                      .file   a    gmingw_helpers.c       >+                            N+        .text                        .data                          .bss    	                      .xdata                        .pdata  P                           g 
   ;S  	                                            +  0     0                    H  
     X                   T       !                     _  G                          j       H                .file       gpseudo-reloc.c        ^+  0                          m+           +  
      the_secs
          +            +   
          +  0C                        +  @C                        ,  `C                    .text   0      *             .data                          .bss     
                      .rdata  A                     .xdata  $     8                 .pdata  h     $   	                    
   \  q                                                                        +  `     0                    :       0                    H  m     h                   T                            _  G                          j                       .file       gcrt_handler.c         4,                             N,           d,  (
          s,  @          },  @
          ,       .text        b               .data                          .bss     
                     .xdata  \                       .pdata       $   	             .rdata  B                            fT 
   ]  L                   H                            w     	                    +       0                    H                          T  x                          _  G                          j                       .file       gtlsthrd.c             ,  p                          ,            ,            ,           ,            
-  `         *-        .text   p    j  '             .data                          .bss         H                 .xdata  |     0                 .pdata       0                      ) 
   m%  I                        P                       !                         +       0                    :  @     0                     H                          T  
                          _   H                          j       0               .file       gtlsmcrt.c         .text                         .data                         .bss     
                              
                         $     *                     +                            H       >                     _   H                      .file       g    >-            .text                         .data                          .bss     
                            1 
                         N     *                     +                            H       H                     _  @H                      .file   &    gpesect.c              R-                            l-            -            -  p         -            -           -           -  P         -           .  0     .text         	             .data                          .bss    0
                       .xdata       H                 .pdata       x                        
   Z  l                   x                            '                         +  0     0                    :  p                          H                          T       ]                     _  `H                          j  (                    .file   A    gCRT_fp10.c        _fpreset                       fpreset       .text                         .data                          .bss    0
                       .xdata                        .pdata  X                           23 
                         %     *                     +  `     0                    H  8     S                    _  H                          j       0                .file   ]    gdllentry.c            :.  P                      .text   P                     .data                          .bss    0
                       .xdata                        .pdata  d                           3 
   I                     O     J                    +       0                    H                          T       	                     _  H                          j  	     0                .file   y    gdllmain.c         DllMain `                      .text   `                     .data                          .bss    0
                       .xdata                        .pdata  p                           ~} 
   I                          J                    +       0                    H  s                        T       	                     _  H                          j  8	     0                .file   M    gvsnprintf.c           H.  p                      .text   p                    .data                          .bss    0
                       .xdata                         .pdata  |                            
   '                                                  T-                          +       0                    H  X                         T                            _   I                          j  h	     0                .text        .data         .bss    0
      .idata$7\      .idata$5l      .idata$4      .idata$6|      .text        .data         .bss    0
      .idata$7X      .idata$5d      .idata$4      .idata$6r      .text        .data         .bss    0
      .idata$7T      .idata$5\      .idata$4      .idata$6h      .text        .data         .bss    0
      .idata$7P      .idata$5T      .idata$4      .idata$6^      .text        .data         .bss    0
      .idata$7L      .idata$5L      .idata$4      .idata$6T      .text        .data         .bss    0
      .idata$7H      .idata$5D      .idata$4      .idata$6J      .text        .data         .bss    0
      .idata$7D      .idata$5<      .idata$4      .idata$6@      .text        .data         .bss    0
      .idata$7@      .idata$54      .idata$4      .idata$68      .text        .data         .bss    0
      .idata$7<      .idata$5,      .idata$4      .idata$6.      .text        .data         .bss    0
      .idata$78      .idata$5$      .idata$4      .idata$6$      .text        .data         .bss    0
      .idata$74      .idata$5      .idata$4|      .idata$6      .text        .data         .bss    0
      .idata$70      .idata$5      .idata$4t      .idata$6      .text        .data         .bss    0
      .idata$7,      .idata$5      .idata$4l      .idata$6      .text        .data         .bss    0
      .idata$7(      .idata$5      .idata$4d      .idata$6      .text        .data         .bss    0
      .idata$7$      .idata$5      .idata$4\      .idata$6      .text        .data         .bss    0
      .idata$7       .idata$5      .idata$4T      .idata$6      .text         .data         .bss    0
      .idata$7      .idata$5      .idata$4L      .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4D      .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4<      .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$44      .idata$6      .text         .data         .bss    0
      .idata$7      .idata$5      .idata$4,      .idata$6      .text   (     .data         .bss    0
      .idata$7      .idata$5      .idata$4$      .idata$6      .text   0     .data         .bss    0
      .idata$7      .idata$5      .idata$4      .idata$6      .text   8     .data         .bss    0
      .idata$7       .idata$5      .idata$4      .idata$6      .text   @     .data         .bss    0
      .idata$7      .idata$5      .idata$4      .idata$6      .text   H     .data         .bss    0
      .idata$7      .idata$5      .idata$4      .idata$6x      .file   [    gfake              hname         fthunk        .text   P                      .data                          .bss    0
                       .idata$2                      .idata$4      .idata$5      .file       gfake              .text   P                      .data                          .bss    0
                       .idata$4                      .idata$5t                      .idata$7`                      .text   P     .data         .bss    0
      .idata$7      .idata$5      .idata$4       .idata$6h      .text   X     .data         .bss    0
      .idata$7      .idata$5      .idata$4       .idata$6V      .text   `     .data         .bss    0
      .idata$7      .idata$5      .idata$4       .idata$6:      .text   h     .data         .bss    0
      .idata$7      .idata$5|      .idata$4       .idata$6,      .text   p     .data         .bss    0
      .idata$7      .idata$5t      .idata$4       .idata$6      .text   x     .data         .bss    0
      .idata$7      .idata$5l      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5d      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5\      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5T      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5L      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5D      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5<      .idata$4       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$54      .idata$4       .idata$6h      .text        .data         .bss    0
      .idata$7      .idata$5,      .idata$4       .idata$6L      .text        .data         .bss    0
      .idata$7      .idata$5$      .idata$4       .idata$6<      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4|       .idata$6"      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4t       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4l       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4d       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4\       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4T       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4L       .idata$6      .text         .data         .bss    0
      .idata$7      .idata$5      .idata$4D       .idata$6      .text        .data         .bss    0
      .idata$7      .idata$5      .idata$4<       .idata$6|      .file       gfake              hname   <       fthunk        .text                         .data                          .bss    0
                       .idata$2                       .idata$4<       .idata$5      .file   -    gfake              .text                         .data                          .bss    0
                       .idata$4                       .idata$5                      .idata$7     
                 .file   D    gcygming-crtend.c      W.            c.                        .text                         .data                          .bss    0
                       .jcr                              w.      
                    .                            .                           .  (                        _   I                      __xc_z     	        .         strcpy           .  @I          .            .             /            /  0D          ./              =/  8         L/            X/  \          o/           /           /        _lock   (     memmove          /      
        /  C          /    m       /  pC      __xl_a  (   	        /            0           0  	          /0  @I          C0            U0  `         m0  @           0            0  B          0  @         0             0            0  (   	        1        __xl_d  8   	    bsearch       _tls_end`   
        #1   C          91  P         F1     	    qsort            X1            i1  (   	        y1      
    memcpy           1  @C          1   
          1        malloc           1       _CRT_MT           1  h         1  p         
2  D          2              (2  @I          K2           b2            z2  d          2            2            2  C          2  T          2  X         2  	          2  H   	        3  4          "3  $      abort            53  0C          _3  @          r3  H   	    __dll__            3             3             3            3            3  C          3    m       3  C          4           4            34        strrchr      calloc           K4  L          X4      
        b4  ,          o4            }4   D          4  `         4  |          4  (          4  `D          4            4  @I          
5  $          5  D      Sleep   x         25            ?5            L5            Z5              t5           5  P
      __xi_z      	        5       pcinit     	        5             5            5     	        5           5  C          6            +6  	      signal           66  @
          M6      	        _6  0
          m6        strncmp          6  t          6            6            6  PC          6             7  <          7            57             H7  <      memcmp           f7  T          s7            7            7  ,          7           7   C      memset           7            7  l          
8  C           8            .8        __xl_z  @   	    __end__              M8  d          o8  D          8  H           8        strcmp           8           8  8         8           8         __xi_a     	        8  l          8       __xc_a      	        
9            #9  H   	        59            C9  8         N9             k9  |          }9           9  PD          9        __xl_c  0   	        9            9  h   
        9  4          9  8
          9           
:             :            0:  	          G:            X:  A          p:  `C          :  0     fwrite           :  \          :  0          :             :             :           :  8          :  H
          ;  C          ';       _onexit           :;  @I          L;            h;             ;            ;  	          ;  C      strlen           ;   D          ;           ;  L          ;  pD          <  @I          %<          i                    9<  @D          M<  H     _unlock          Y<            q<  C          <  H   	        <            <  `      vfprintf     free             <        <  .debug_aranges .debug_info .debug_abbrev .debug_line .debug_frame .debug_str .debug_loc .debug_ranges pre_c_init .rdata$.refptr.__onexitbegin .rdata$.refptr.__onexitend _CRT_INIT __proc_attached .rdata$.refptr.__native_startup_lock .rdata$.refptr.__native_startup_state .rdata$.refptr.__dyn_tls_init_callback .rdata$.refptr.__xi_z .rdata$.refptr.__xi_a .rdata$.refptr.__xc_z .rdata$.refptr.__xc_a __DllMainCRTStartup .rdata$.refptr.__native_dllmain_reason DllMainCRTStartup .rdata$.refptr.mingw_app_type .CRT$XIAA .debug_info .debug_abbrev .debug_loc .debug_aranges .debug_ranges .debug_line .debug_str .rdata$zzz .debug_frame __JCR_LIST__ __gcc_register_frame __gcc_deregister_frame .rdata$.refptr.tclStubsPtr pkgNamespace.10317 tupleCmdName.10311 tupleStr.10312 relationCmdName.10313 relationStr.10314 ral_pkgname relvarCmdName.10315 relvarStr.10316 ral_config ral_version Ral_SafeInit Ral_Unload Ral_SafeUnload ral_copyright .rdata$.refptr.relvarCmd .rdata$.refptr.relationCmd .rdata$.refptr.tupleCmd booleanEqual booleanCompare booleanHash isAByteArray byteArrayHash intCompare longEqual longCompare isADouble doubleEqual doubleCompare doubleHash isAWideInt wideIntEqual wideIntCompare wideIntHash isABignum .rdata$.refptr.tclTomMathStubsPtr bignumEqual bignumCompare bignumHash isAString stringHash cmpTypeNames stringCompare dictCompare stringEqual dictEqual byteArrayCompare byteArrayEqual isABoolean listCompare listEqual Ral_AttributeNewTclType Ral_Types Ral_AttributeNewTupleType Ral_AttributeNewRelationType Ral_AttributeDelete Ral_AttributeDup Ral_AttributeRename Ral_AttributeTypeEqual Ral_AttributeEqual Ral_AttributeValueEqual .rdata$.refptr.Ral_RelationObjType .rdata$.refptr.Ral_TupleObjType Ral_AttributeValueCompare Ral_AttributeValueObj Ral_AttributeNewFromObjs Ral_AttributeConvertValueToType Ral_AttributeHashValue Ral_AttributeScanName Ral_AttributeConvertName Ral_AttributeScanType Ral_AttributeConvertType Ral_AttributeScanValue Ral_AttributeConvertValue Ral_AttributeTypeScanFlagsFree Ral_AttributeValueScanFlagsFree Ral_AttributeToString attr_0_cmp attr_1_cmp Ral_JoinMapNew Ral_JoinMapDelete Ral_JoinMapAttrReserve Ral_JoinMapTupleReserve Ral_JoinMapAddAttrMapping Ral_JoinMapAddTupleMapping Ral_JoinMapGetAttr Ral_JoinMapAttrMap Ral_JoinMapFindAttr Ral_JoinMapSortAttr cmpFuncs.9202 Ral_JoinMapTupleMap Ral_JoinMapTupleCounts Ral_JoinMapMatchingTupleSet tupleHashEntryAlloc tupleAttrHashEntryAlloc tupleAttrHashEntryFree tupleHashEntryFree tupleAttrHashCompareKeys tupleAttrHashGenKey tupleHashCompareKeys tupleHashGenKey tupleAttrHashMultiEntryFree tupleAttrHashMultiEntryAlloc Ral_TupleCompare Ral_DownHeap Ral_HeadingMatch Ral_RelationIndexByAttrs.isra.1 tupleAttrHashMultiType Ral_RelationNew tupleHashType Ral_RelationDelete Ral_RelationReserve Ral_RelationPushBack Ral_RelationDup Ral_RelationShallowCopy Ral_RelationUpdate Ral_RelationTimes Ral_RelationProject Ral_RelationGroup Ral_RelationUngroup Ral_RelationTag Ral_RelationTagWithin Ral_RelationUnwrap Ral_RelationDisjointCopy Ral_RelationUnionCopy Ral_RelationUnion Ral_RelationFindJoinTuples Ral_RelationJoin Ral_RelationCompose Ral_RelationSemiJoin Ral_RelationSemiMinus Ral_RelationTclose Ral_RelationSort Ral_RelationFind Ral_RelationIntersect Ral_RelationMinus Ral_RelationDivide Ral_RelationExtract Ral_RelationErase Ral_RelationCompare Ral_RelationEqual Ral_RelationNotEqual Ral_RelationSubsetOf Ral_RelationProperSubsetOf Ral_RelationSupersetOf Ral_RelationProperSupersetOf Ral_RelationRenameAttribute Ral_RelationScanValue Ral_RelationScan Ral_RelationConvertValue Ral_RelationConvert Ral_RelationStringOf Ral_RelationValueStringOf RelationTagCmd gotOpt.10416 optTable.10425 RelationSemiminusCmd RelationSemijoinCmd RelationJoinCmd RelationComposeCmd RelationAttributesCmd RelationCardinalityCmd RelationDegreeCmd RelationIsemptyCmd RelationIsnotemptyCmd RelationWrapCmd RelationEmptyofCmd RelationIssametypeCmd RelationIsCmd cmdTable.10063 RelationUpdateCmd RelationRestrictWithCmd RelationRestrictCmd RelationUnwrapCmd RelationUnionCmd RelationDunionCmd RelationUngroupCmd RelationUinsertCmd RelationInsertCmd RelationTupleCmd RelationTimesCmd RelationTcloseCmd RelationForeachCmd orderOptions RelationCreateCmd RelationTableCmd RelationSummarizebyCmd RelationSummarizeCmd RelationExtendCmd usage.9881 RelationRenameCmd RelationRankCmd RelationProjectCmd RelationEliminateCmd RelationMinusCmd RelationListCmd RelationExtractCmd RelationDictCmd RelationArrayCmd RelationIntersectCmd RelationHeadingCmd RelationGroupCmd RelationFromlistCmd RelationFromdictCmd.part.40 RelationFromdictCmd RelationDivideCmd RelationBodyCmd RelationAssignCmd relationCmd cmdTable.9718 UpdateStringOfRelation DupRelationInternalRep FreeRelationInternalRep Ral_RelationObjNew Ral_RelationInsertTupleValue Ral_RelationObjConvert SetRelationFromAny Ral_RelationInsertTupleObj Ral_RelationUpdateTupleObj Ral_RelationObjParseJoinArgs relvarTraceCleanup relvarCorrelationConstraintToString condMultStrings Ral_ConstraintNew relvarCleanup relvarConstraintCleanup Ral_RelvarDeleteInfo relvarEvalAssocTupleCounts relvarFindJoinTuples.isra.1 relvarConstraintErrorMsg.isra.2 relvarAssocConstraintErrorMsg.isra.3.part.4 relvarPartitionConstraintErrorMsg.isra.5.part.6 relvarCorrelationConstraintErrorMsg.isra.7.part.8 Ral_RelvarNew Ral_RelvarDelete Ral_RelvarFind Ral_RelvarNewInfo Ral_RelvarIsTransOnGoing Ral_RelvarTransModifiedRelvar Ral_RelvarNewTransaction Ral_RelvarStartTransaction Ral_RelvarStartCommand Ral_RelvarDeleteTransaction Ral_ConstraintFindByName Ral_ConstraintNewAssociation Ral_ConstraintAssocCreate Ral_ConstraintNewPartition Ral_ConstraintPartitionCreate Ral_ConstraintNewCorrelation Ral_ConstraintCorrelationCreate Ral_ConstraintNewProcedural Ral_ConstraintProceduralCreate Ral_ConstraintDelete Ral_ConstraintDeleteByName Ral_RelvarDeclConstraintEval Ral_RelvarFindIdentifier Ral_RelvarFindById Ral_RelvarIdIndexTuple relvarIndexIds.isra.10 .rdata$.refptr.tupleAttrHashType relvarSetIntRep Ral_RelvarSetRelation Ral_RelvarInsertTuple Ral_RelvarIdUnindexTuple Ral_RelvarDeleteTuple Ral_RelvarRestorePrev Ral_RelvarDiscardPrev Ral_RelvarTraceAdd Ral_RelvarTraceRemove RelvarNamesCmd RelvarUinsertCmd RelvarInsertCmd RelvarPathCmd RelvarExistsCmd RelvarIdentifiersCmd RelvarUpdatePerCmd RelvarUpdateOneCmd RelvarUpdateCmd RelvarUnsetCmd RelvarUnionCmd RelvarDunionCmd RelvarTransactionCmd transactionOptions.10312 RelvarEvalCmd RelvarTraceCmd traceOptions.10281 traceTypes.10285 RelvarSetCmd RelvarRestrictOneCmd RelvarProceduralCmd RelvarPartitionCmd RelvarMinusCmd RelvarIntersectCmd RelvarDeleteOneCmd RelvarDeleteCmd RelvarCreateCmd RelvarCorrelationCmd RelvarConstraintCmd constraintCmds.10019 RelvarAssociationCmd relvarCmd cmdTable.9998 relvarGetNamespaceName Ral_RelvarObjDecodeTraceOps Ral_RelvarObjEncodeTraceFlag Ral_RelvarObjExecTraces relvarResolveName relvarTraceProc relvarObjConstraintEval relvarConstraintAttrNames.isra.3 Ral_RelvarObjNew Ral_RelvarObjCopyOnShared Ral_RelvarObjFindRelvar Ral_RelvarObjDelete Ral_RelvarObjInsertTuple Ral_RelvarObjKeyTuple Ral_RelvarObjTraceUpdate Ral_RelvarObjUpdateTuple Ral_RelvarObjCreateAssoc specErrMsg specTable Ral_RelvarObjCreatePartition Ral_RelvarObjCreateCorrelation Ral_RelvarObjCreateProcedural Ral_RelvarObjConstraintNames Ral_RelvarObjConstraintMember Ral_RelvarObjFindConstraint Ral_RelvarObjConstraintDelete Ral_RelvarObjConstraintInfo condMultStrings.10418 Ral_RelvarObjConstraintPath Ral_RelvarObjEndTrans Ral_RelvarObjEndCmd Ral_RelvarObjTraceVarAdd Ral_RelvarObjTraceVarRemove Ral_RelvarObjTraceVarInfo Ral_RelvarObjTraceVarSuspend Ral_RelvarObjTraceEvalAdd Ral_RelvarObjTraceEvalRemove Ral_RelvarObjTraceEvalInfo Ral_RelvarObjExecDeleteTraces Ral_RelvarObjExecInsertTraces Ral_RelvarObjExecUpdateTraces Ral_RelvarObjExecSetTraces Ral_RelvarObjExecUnsetTraces Ral_RelvarObjExecEvalTraces Ral_TupleNew Ral_TupleSubset Ral_TupleDelete Ral_TupleUnreference Ral_TupleEqualValues Ral_TupleEqual Ral_TupleHash Ral_TupleHashAttr Ral_TupleAttrEqual Ral_TupleUpdateAttrValue Ral_TupleGetAttrValue Ral_TupleCopyValues Ral_TupleExtend Ral_TupleCopy Ral_TupleDup Ral_TupleDupShallow Ral_TupleDupOrdered Ral_TupleScanValue Ral_TupleScan Ral_TupleConvertValue Ral_TupleConvert Ral_TupleStringOf Ral_TupleValueStringOf TupleFromListCmd TupleEqualCmd TupleAttributesCmd TupleDegreeCmd TupleGetCmd TupleUnwrapCmd TupleExtractCmd TupleWrapCmd TupleUpdateCmd TupleRenameCmd TupleRelationCmd TupleProjectCmd TupleHeadingCmd TupleExtendCmd TupleEliminateCmd TupleCreateCmd TupleAssignCmd cmdTable.9031 Ral_TupleHeadingNew Ral_TupleHeadingDelete Ral_TupleHeadingUnreference Ral_TupleHeadingEqual Ral_TupleHeadingFetch Ral_TupleHeadingStore Ral_TupleHeadingPushBack Ral_TupleHeadingSubset Ral_TupleHeadingAppend Ral_TupleHeadingDup Ral_TupleHeadingExtend Ral_TupleHeadingFind Ral_TupleHeadingIndexOf Ral_TupleHeadingUnion Ral_TupleHeadingIntersect Ral_TupleHeadingJoin Ral_TupleHeadingCompose Ral_TupleHeadingNewOrderMap Ral_TupleHeadingCommonAttributes Ral_TupleHeadingMapIndices Ral_TupleHeadingScan Ral_TupleHeadingConvert Ral_TupleHeadingStringOf UpdateStringOfTuple DupTupleInternalRep FreeTupleInternalRep Ral_TupleObjNew Ral_TupleHeadingNewFromObj Ral_TupleHeadingAttrsFromVect Ral_TupleHeadingAttrsFromObj Ral_TupleSetFromValueList Ral_TuplePartialSetFromObj Ral_TupleUpdateFromObj Ral_TupleSetFromObj Ral_TupleObjConvert SetTupleFromAny Ral_TupleAssignToVars Ral_ErrorInfoSetCmd Ral_ErrorInfoSetError Ral_ErrorInfoSetErrorObj Ral_InterpSetError resultStrings optStrings cmdStrings errorStrings Ral_InterpErrorInfo Ral_InterpErrorInfoObj Ral_ErrorInfoGetCommand Ral_ErrorInfoGetOption ptr_ind_compare int_ind_compare Ral_IntVectorNew Ral_IntVectorNewEmpty Ral_IntVectorDup Ral_IntVectorDelete Ral_IntVectorReserve Ral_IntVectorFill Ral_IntVectorFillConsecutive Ral_IntVectorFetch Ral_IntVectorStore Ral_IntVectorFront Ral_IntVectorBack Ral_IntVectorPushBack Ral_IntVectorPopBack Ral_IntVectorInsert Ral_IntVectorErase Ral_IntVectorExchange Ral_IntVectorSetAdd Ral_IntVectorSetComplement Ral_IntVectorBooleanMap Ral_IntVectorSort Ral_IntVectorFind Ral_IntVectorIndexOf Ral_IntVectorOffsetOf Ral_IntVectorIntersect Ral_IntVectorMinus Ral_IntVectorEqual Ral_IntVectorSubsetOf Ral_IntVectorContainsAny Ral_IntVectorCopy Ral_IntVectorPrint Ral_PtrVectorNew Ral_PtrVectorDup Ral_PtrVectorDelete Ral_PtrVectorReserve Ral_PtrVectorFill Ral_PtrVectorFetch Ral_PtrVectorStore Ral_PtrVectorFront Ral_PtrVectorBack Ral_PtrVectorPushBack Ral_PtrVectorPopBack Ral_PtrVectorInsert Ral_PtrVectorErase Ral_PtrVectorSetAdd Ral_PtrVectorSort Ral_PtrVectorFind Ral_PtrVectorEqual Ral_PtrVectorSubsetOf Ral_PtrVectorCopy Ral_PtrVectorPrint Tcl_InitStubs TclTomMathInitializeStubs tclTomMathStubLib.c mingw_onexit .rdata$.refptr.__imp__onexit __do_global_dtors __do_global_ctors .rdata$.refptr.__CTOR_LIST__ initialized __security_init_cookie .data$__security_cookie .data$__security_cookie_complement __report_gsfailure GS_ContextRecord GS_ExceptionRecord GS_ExceptionPointers __dyn_tls_dtor __dyn_tls_init .rdata$.refptr._CRT_MT __tlregdtor _decode_pointer _encode_pointer __report_error __write_memory.part.0 maxSections _pei386_runtime_relocator was_init.93491 .rdata$.refptr.__RUNTIME_PSEUDO_RELOC_LIST_END__ .rdata$.refptr.__RUNTIME_PSEUDO_RELOC_LIST__ .rdata$.refptr.__image_base__ __mingw_SEH_error_handler __mingw_init_ehandler was_here.93323 emu_pdata emu_xdata _gnu_exception_handler __mingwthr_run_key_dtors.part.0 __mingwthr_cs key_dtor_list ___w64_mingwthr_add_key_dtor __mingwthr_cs_init ___w64_mingwthr_remove_key_dtor __mingw_TLScallback pseudo-reloc-list.c _ValidateImageBase.part.0 _ValidateImageBase _FindPESection _FindPESectionByName __mingw_GetSectionForAddress __mingw_GetSectionCount _FindPESectionExec _GetPEImageBase _IsNonwritableInCurrentImage __mingw_enum_import_library_names DllEntryPoint __ms_vsnprintf __JCR_END__ register_frame_ctor .text.startup .xdata.startup .pdata.startup .ctors.65535 Ral_RelationObjType ___RUNTIME_PSEUDO_RELOC_LIST__ __imp__vsnprintf __imp_abort __lib64_libkernel32_a_iname .refptr.relvarCmd __data_start__ ___DTOR_LIST__ __imp__lock __imp_RtlVirtualUnwind SetUnhandledExceptionFilter _vsnprintf __imp_calloc ___tls_start__ .refptr.__native_startup_state __ImageBase .refptr.__imp__onexit GetLastError GetSystemTimeAsFileTime mingw_initltssuo_force __rt_psrelocs_start __imp___dllonexit __dll_characteristics__ ral_tupleTypeName __size_of_stack_commit__ .refptr.Ral_RelationObjType __iob_func __size_of_stack_reserve__ __major_subsystem_version__ ___crt_xl_start__ __imp_DeleteCriticalSection .refptr.__CTOR_LIST__ VirtualQuery ___crt_xi_start__ __imp__amsg_exit ___crt_xi_end__ _tls_start .refptr.__RUNTIME_PSEUDO_RELOC_LIST__ __mingw_oldexcpt_handler __imp_GetCurrentThreadId GetCurrentProcessId TlsGetValue TerminateProcess __imp_strcmp __bss_start__ ___RUNTIME_PSEUDO_RELOC_LIST_END__ RtlLookupFunctionEntry __size_of_heap_commit__ __imp_strrchr __imp_GetLastError __imp_free .refptr.__onexitend __imp_RtlLookupFunctionEntry VirtualProtect mingw_app_type ___crt_xp_start__ __imp_LeaveCriticalSection __imp_GetTickCount .refptr.__RUNTIME_PSEUDO_RELOC_LIST_END__ tclTomMathStubsPtr ___crt_xp_end__ tclIntPlatStubsPtr __minor_os_version__ __imp_GetSystemTimeAsFileTime EnterCriticalSection .refptr.__xi_a __image_base__ .refptr._CRT_MT RtlCaptureContext __section_alignment__ __native_dllmain_reason __imp_strcpy _tls_used __imp_memset __imp_bsearch .refptr.relationCmd UnhandledExceptionFilter __IAT_end__ tclIntStubsPtr .refptr.tupleAttrHashType __imp_memcpy __RUNTIME_PSEUDO_RELOC_LIST__ __imp_memmove __imp_RtlAddFunctionTable __data_end__ __imp_fwrite __CTOR_LIST__ _head_lib64_libkernel32_a GetModuleHandleA __bss_end__ GetTickCount _head_lib64_libmsvcrt_a __native_vcclrit_reason ___crt_xc_end__ RtlAddFunctionTable .refptr.__native_startup_lock __imp_EnterCriticalSection _tls_index __native_startup_state ___crt_xc_start__ __onexitbegin __imp_GetCurrentProcessId __imp_TerminateProcess __imp_GetProcAddress ___CTOR_LIST__ .refptr.__dyn_tls_init_callback .weak._Jv_RegisterClasses.__gcc_register_frame __imp_signal __imp_GetModuleHandleA __rt_psrelocs_size __imp_QueryPerformanceCounter __imp_strlen __imp_malloc __file_alignment__ __imp_InitializeCriticalSection InitializeCriticalSection .refptr.Ral_TupleObjType __major_os_version__ __imp_vfprintf .refptr.__onexitbegin __IAT_start__ __imp_UnhandledExceptionFilter __imp_SetUnhandledExceptionFilter .refptr.mingw_app_type ral_relationTypeName __imp__onexit GetProcAddress __DTOR_LIST__ RtlVirtualUnwind tupleAttrHashType __imp_Sleep LeaveCriticalSection __size_of_heap_reserve__ ___crt_xt_start__ __subsystem__ _amsg_exit __security_cookie_complement __imp_TlsGetValue GetCurrentProcess .refptr.tclTomMathStubsPtr __imp_VirtualProtect Ral_TupleObjType ___tls_end__ __imp_qsort __onexitend QueryPerformanceCounter __imp_VirtualQuery __imp__initterm mingw_initltsdyn_force __imp___iob_func __dyn_tls_init_callback .refptr.__image_base__ _initterm __imp_strncmp tclPlatStubsPtr __major_image_version__ __loader_flags__ ___chkstk_ms tclStubsPtr __native_startup_lock .refptr.__native_dllmain_reason GetCurrentThreadId __rt_psrelocs_end __minor_subsystem_version__ __minor_image_version__ __imp__unlock mingw_initltsdrot_force .refptr.__xc_a .refptr.__xi_z DeleteCriticalSection __imp_RtlCaptureContext .refptr.tupleCmd __RUNTIME_PSEUDO_RELOC_LIST_END__ _Jv_RegisterClasses .refptr.tclStubsPtr __dllonexit __imp_GetCurrentProcess .refptr.__xc_z ___crt_xt_end__ __imp_memcmp __lib64_libmsvcrt_a_iname __security_cookie              Z         p                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              
            __TEXT                                   __text          __TEXT                                      __symbol_stub   __TEXT           H                       __stub_helper   __TEXT                                    __const         __TEXT          | 5  |                         __cstring       __TEXT           *                         __unwind_info   __TEXT           H                                __DATA                                  __nl_symbol_ptr __DATA            ,                          __la_symbol_ptr __DATA          , 0   ,                      __const         __DATA          \ l  \                        __data          __DATA           <                          __bss           __DATA                                          8   __LINKEDIT         `     Z                
   ,             libral0.12.2.dylib  "  0      H   H  8                @             0 )     P                                            00 #                         _w1piv/0!L$       
  
 *                 4              /usr/lib/libSystem.B.dylib  &      L   )                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            USWV,    [ j Wuگ 1A  1VVWuK l  ] } MVVQMQ  E  E N UVVQRu   [ VSuu    u  M؃1OWS   ËWu   PSuuU؃  ]1} a UVVQRMV   q     QWuV  3  u  M؃1KSW   ǋSu   PWuuU؃   } MԋEPQ/  uE z E1S    QRPuE    SWuu  u{u  M1IQW   Ë1IQu   PSuuUu3}j uu܋MV|  uPV  t0UuHQ   1Aȃ,^_[]Ë]}  QRuVMQ   j QVuP1U]U1]U1]USWV    [EEp   UPjj
QRT  1tIM  xQZ P׃ǍGSP6 G1GG^_[]USWV    _]8  p$ $։ƍF\$$ܮ  FF   EF ^_[]USWV    _]  p$虮 $։ƍF\$$z  FF   EF ^_[]UV    ^MAPs
A$u tv 	D$B $Qv  @^]U    YEPtt"u0p0(p0
p0]Ë!   RQP1]UVP    ZEMqtt u-qPqPqP,  ۯ VQP1^]UWVu}9t'67 1uO;Nu^_]   1@^_]UWV    YUEp~srp :urp蜬   ^ VQP1^_]USWV<    _uUE@   &     ]ԋM܉L$$   M؉L$4$   E܋M؉	^  1~   v  E@E;  D$ D$E$D$   D$
   膫   ML$M$P    |$T$$    PP1   |$ML$$    PP   E@MIL$$@@       |$T$$    PP1   |$ML$$    PPurE@MIL$$PZ V 	D$  $Q1:1F5}ԋM$X  ƋM$X  D$4$n <^_[]USWV    X}M $X  Ƌ<$X  D$4$ ^_[]USWV,    [}uMAPrIuqAE  D$} D$E$D$   D$
   蓩 t|$4$PF1 4$X  Ƌ<$X  D$4$w 1 	D$ $Q1,^_[]USWV    X}Mw $X  Ƌ<$X  D$4$ ^_[]USWV    XuMQtBMt>   0 < D$t$$RP1u}F$^ 3l0 8 D$t$$RP1uHF$B  Ƌ4$D$   ǋ4$P0 	T$& $Q1^_[]USWV    [Z uX  ƋMUQRuu   1Ʌ8  MEt;   M1X  ǃWV   WvM1X  ǃ; WP耧 tm2 WPg }t}SjEV VW*QX  PjEV蹏 Vu` 1YEupuE     t9PV(EupW     tPVȃ^_[]USWV,    _MU]uC     j    MQR   } \  CEt   UPjj
QR
    }WuVQ  WjEWێ WV  B;    :    E؋ RP|ǃuWuVs3   uqE؋ W   1     B;    :    E؋ RP|ǃuWuVs胄     U؋H1%  W    	֍ PRQ   E   QRRVsX   1     MV   G0jQRRVsЃ  1     MV   $u_[E؋ W   1tCGp3x u'ujEV VuB 1,^_[]U1]USWV    [}G1Hr|u_ E؋ MQu   } tTGE  O uPjj
QVV  t3uP 	g PRQ^_[]ËE؋ Vu   M1t@IuUVP    X9  MQu   U1t	1AJu^]UVP    XuMQ  VR1p  F^]U    XMU  qu2X  ]UVP    ZMEp1t&t5uE Qpp  FQp1~ 
/Qp~ 
  ' VQP1^]USWV    X]u}Ot-tn     sVwX     N{ PQ聡 fF	 {FSPw~ }TD
T)aN{ PQ; fF {FSPw} }TD
T	)  QPR1^_[]UWV    YuUE}>t0t:uH
 	p  PX  VPVRp7  
VRp%T 
  Ӥ WQP1^_]USWV    Y}UuE][t0t8uDu 	X  PX  wVPWRVp8  WRVpT u  q SQP1^_[]USWV    Z}Hs<w_~UF$N_U  $PG    t 	D$ $Q^_[]USWV    Z}Hs<w_~UF$N_Uc  $PG    tc 	D$̣ $Q^_[]USWV    X}OM MQ7p  ƉuEPWDPQƋuV7X  L ]SQWbSy^_[]UEM01 ]USWV    _.  ]Suu  ƅu>  SP ^_[]USWV,    _ MQuj   1ua]Suj   1u/ً QMQP(MQP   MQP ,^_[]USWV,    _B ]Suj   t1N[uVuj   t1N)R E̋ VSP(ƋE̋ QP R  SP ,^_[]USWV    _  ]Suj   1u)E1t1MAHu  SP ^_[]USWV    ^C ]Suu   tSuu   T]M11B9tBt;	  j u   & PQӃPu  1@^_[]UWV    X MQuj    1u'MQuj    uE;E^_]UWV    X? MQuj    1Nu!MQuj    uu+u^_]USWV    _1u MQuV   t$XuX  ݟ PQӃ1L5Fu^_[]U    Xv  j u   1]USWV    X}G MQu   ƋMQW   M;MuQPV趙 1^_[]USWV    X} MQu   ƋMQW   MU9NуRPVG ^_[]UVP    X  MQu   U1t	1AJu^]U    X>  MQuu  ]USWV    X}M $X  Ƌ<$X  D$4$謘 ^_[]USWV    X}M $X  Ƌ<$X  D$4$] ^_[]UVP    X}  MQu   U1t	1AJu^]U    X<  MQuu   ]UWV    X MQuj    1u-MQuj    uEE f~ƃ^_]UWV    X MQuj    1Nu<MQuj    u!EMf.v1F11Nf.F^_]USWV    _1uu苟/ MQuV   t$XuX  L PQӃ1L5Fu^_[]U    X  MQuu   ]UWV    X MQuj    1u'MQuj    uE;E^_]UWV    X1 MQuj    1Nu!MQuj    uu+u^_]USWV    _1u MQuV   t$XuX   PQӃ1L5Fu^_[]U    Xh  MUQRuu   ]USWV    X}M8 $X  Ƌ<$X  D$4$Ք ^_[]USWV    X}M $X  Ƌ<$X  D$4$膔 ^_[]UVP    X  MQu   U1t	1AJu^]U    Xe  MQuu   ]UWV    X7 MQuj    1u'MQuj    uE;E^_]UWV    Xֿ MQuj    1Nu!MQuj    uu+u^_]USWV    _1uy MQuV   t$XuX  ՘ PQӃ1L5Fu^_[]U    X
  MQuu  ]UWV    X߾ MQuj   1u/MQuj   uEM3M3E	^_]USWV    Xu MQuj   1Nu@MQuj   u%ME}U9ω}1F11N9M^_[]USWV    _1uu苟 MQuV  t$XuX  s PQӃ1L5Fu^_[]USWV    X] $   PƋ    $PF؉F}    $PFFF^_[]UWV    ^}t 	$QGt 	$Q  @^_]USWV    Y}uV)9},^)ËѼ 	    RPQÉ^F^_[]USWV    Y}uFV)9}-^)Ëz 	    RPQFÉ^F^_[]USWVuNF9r"+DPVNU}9t9;t   9Su;QN11@^_[]USWV}]uNF9r#+FDPVNyN^_[]USWV}G+$?{ Ƌ9tM)ǍD$4$[} u^_[]USWV}ju|z Ƌ9t M)Ǎj 3V]| u^_[]UWVEP1H9t+}u9<t	9uFT?)Ћ^_]UVP    XMU2R) jRV$ ^]UME +]UME@+A]USWV}juy ƋG9t M)Ǎj 3Vw{ u^_[]UWVEpH19t#}EA))4>u@^_]USWVu$   y EF~9tAMQTڍT)Ǎ4E9uD$E$} u݋E^_[]UEp0C= ]UEMIq1p0m= ]UVP    Xu  $   PHH@    HNH^]UVP    ^E@ $;   @^]USWV    _ jHP1FFFFFFFF$F F,F(F4F0F<F8FDF@FM RjP  ^_[]USWV}$k^ É]$VEO+OL$$d   wO>$8 ËGOU׋P+PʋCD$T$$9 \$]E$D$    l   ;wuE^_[]USWV    Y}uFV)9}-^)ËY 	    RPQFÉ^F^_[]USWV}]UB9Br$+BDPRn@tW2S= Ƌ@ZBMQVPW@MMt+_XGHO0]Vn9 Sb9 E^_[]USWV}7ƋG+GPV_j 3V;_u^_[]USWV    _u^
$8 ;^u N$  Ft$Q$] @^_[]USWV    [u}9ww  ! $PMPUL$$P<t 	$  MU҉ȋ]tT$D$$< ML$D$}<$S@} t+KH E$7 1@6;Et$K7 MED$L$<$S@} t	+sp1^_[]USWV    XE]}u   1   71PVwwO  37db ǃ} CKtVWVQPr  ;CtQ0}= ÃSjucp E  SPV1WVQP   WYt ^_[]USWVӉ    ^\$<$[    uu<$Ed ǉ}$xd É|$4$D$   o M苹 1JT$U L$4$  1IL$\$4$  M$P$PE^_[]USWVu])Put9t}W3u9u^_[]USWVu])Pu(9t#}W3uit	9uމ^_[]USWVuUE
Z~}$1ɅtpM΋L$$@` ǋ$Ev9t:|$D$E$:   ;EtD$E$D$    9uƉ<$Yr Mȃ^_[]USWVME]t/Q3P8 ǍCWPS<ƃW3 KPQS<ƅtFCC^_[]USWVuUE
Z~}$j1ɅtpM΋L$$*_ ǋ$jEv9t:|$D$E$$;EuD$E$D$    9uƉ<$Cq Mȃ^_[]USWVu}67Z É]1   SEW+WN+NʃQPGW9   NE~9   0ESF1 ËFNP+PʃsRQ2 MAIP+PʋE p+psVuRQ1 j S]uE;~zMQ9YE^_[]USWVuu6T EPǋF+FPW^%uu30 j PW;^u։^_[]USWV       XE}EC+CD$<$s E|$$S EC+CO+@)ȉ$cS ƋE8&CDt$D$T$$(U E;xuՋED$E$wD$4$QT 4$EE$}U  E E ML$<$L  ǅ|n  EtuGG E4$/ pM;Yt5}E}GDt$D$$/ E;Xu}EE$ǉ}<$謅   EpuF+D$<$;N   ME@	EE$n. E};_t0EpE@Dt$D$$/ ;_u֋ED$E$D$    ME;H{ED$E$D$    E M$  ǅ|E$"m E M$  EČ   ^_[]USWV։    XY 	q PjW  ^7EEEEPEPWW0+NQp2n ;^uă^_[]USWV<    _]E0SVT ;F  y  }y}ME̋J+JG+GDPJP S}WuvV"R SvWVR SEppPQ v  SEMqIM9K  ]U싊 M M]uЉ HxA+AEM̋E+HM4E uVj PP  vF+FPuE^FE9ËEM   }}MIMԋ;P+ EpVuun, 4VuuY, OWq+qփPVR:, j uu	E܃9؋EuuЃ;uċ]ESjSju?f 1<^_[]ÃE썀 Pjuf SxQ ΃VjSTf uUSWV,]EM18QUPU܃SPQm  EuPSWVT  1A  PEƉu؋C+CPVK;K   MIU<E܋4uEu* EPGO׋P+PʃWRQ* NQ;Qt>UR<Ѝp;tWVP* <EHPA+A9Ɖu̓j }WuVPEtM;H>uh 9V6uh WZ Pj.ud 1,^_[]USWVl    XEu}]4$D$   ]E$ML4$D$    =E_;_td]EEEED$E$Ut6@E8'+AˋL$D$4$Eً];xuԃ;YuE  M$  E$g E$g l^_[]USWV,uEM9QUԋPUЃVPQEMuPQVSW<S  É]1K  Sǉ}ȋF+FPW N;N   M܋IUԋ4EЋES' HM^K;KuEIE؋NuU2ʍz>t ]SWR)( ]EXHS+S9׉uǋE@H;H]t@MIU2ʍz>tSWR' MAIP+P9׉u̓j u؋}WM܃E;H]u	f ue ,^_[]USWV]uFESVuB6ƋC+CPV{Gj M4V;{u^_[]USWV]}WSu3(ƋC+CPjWeE{C9t*M; tj 7VfE@9uۃud ^_[]USWV,EMK+KMMQ+QUx+xE Ep+p    XEj QSR tE썀} Pj  u؉}܋uuj }WS]R 9]}e  uj QSxR ډ]}99  91  }WE؃VWM ǃW~$ ƉuWS^Q EEH;HE   E܉MU~BH+HBWQP% MY1;YtQE}H@Q+QuRP$ uuW;GƃE;Xu}E9uj uulME;HJV$ ub EE썀} Pju^ 1,^_[]USWV,    ^M]z D$$"ǋ$D$   {I É|$$G ;C   }$69;y   ]] M܉EE@4E$" E@EFNP+Pʋut$T$$t# EE܋ $C   M ED$E$D$    "E;xuE$EML$$D$   k] $H 1,^_[]USWVl    ^]y Puǃj3UH ÃWSF ;C  S]ǉ} E Ψ UQjR  E`MFNP+PʃuVRQV" EE WG   M Exj u}W E;Xt`E@4u! ǋGEuEEEPEPEPU} t}}yF}y;E MQ  !uju[ S;G 1l^_[]USWV,    XE]uE8VWG ;G  y  Y]uEJ+JC+CDPC SuwWE ESwPW|E SEppPiE E  ]SÉ]܋F+FPS V;V  GM)MEP uԋ\ EЉUEX<Eԋ uWj PP   GEuJ ExWSuv  <FH+HNWQS UJRq+qփPVR j u]SUE;PM9VjVjSY 1ۉ؃,^_[]ÃEYy PjuY SE ̃WjuY u2USWV|    XuM~}^] E M$  F+F$g E9   E    ]x7ED$t$E$U} tMHAMt$E$i EwML$t$$U} tMHAMt$E$h u9uEX+]]}E <$PE|$$o U9}tAUE pD$<$UNpL$<$UpUE0E9uE <$  } ~bE    EE}}     t9MEEE1ɋEڊu"7uA@9|M;M|ǋE@E]9|E$UEA+AD$$}    E    }}1? t^E$9 E@EED$]$}f M t$$]gf MA ED$E$D$    FG9|E@E}9|E M$PE$oe E|^_[]USWV,EMq+qux+81ۉ\$4$Y E\$É$[[ ~tEMMMME{+;]؃|#CKE$B   K}]|)}Nt$$D$    ] <$1ɉ   ډ؃,^_[]USWVU΍\69}W}us9}<$  OމދE<$Éى  MyGt$\$$j] ڍ\69|^_[]USWV}E0ƋG+PVE@j 4V;_u߉^_[]USWV    [MuUB9w9rsј  r $PMB9w9Jsј  ΍!s $Pu9vј  ΍Ps $Pu9tHBEј E}D$E$։V<M	$  $ ى9uʋB)ȉD$L$]$։k )ߋB)B9t-r}D$4$S<څt+JH;zuًE^_[]USWV}uFE_76? tT]67E ǋ^1E9t&W3uM;Eƃ9uڃWW 1N^_[]UWVu}VW_x!O+OV+V99 ^_]UuuI]UWVu}VWx!O+OV+V99 ^_]UWVu}VWx!O+OV+V99 ^_]UWVu}VWux!O+OV+V99 ^_]UWVu}VW4x!O+OV+V99 ^_]USWV}EWSF> ;st)}W6軻PVSP> ;Ct
1@WjWjuQ 1^_[]USWV]}S7E ƃuSW   D^_[]USWV    X}U]MK+KJO4I  VPGVPJh CK9˾      E@EEP@1FF9t_Ex]@]UwSW0E28Ut0E;PEuˉ]E@MIM]ǉM9׸    EtM9ˉrEXF11H9D^_[]USWV]M}G{SP1(D tD}LD uSQu-   t0Su)^_[]USWV,MUE2u؋x{ZA;Z   }΋U؉]ԋYMIM {zkEU]uVQP#M܍L ]EPEVuQuu5 u܍L0 T0U؃;zuAN E }@ ]ԃpM;YD}F+u,^_[]USWV<    X}1ɉMW)Eغ   U؉M̉MȉMЉUȋL  @Eă]S7A ƃEPSWDPUăƍEPSVW <^_[]USWV,    ^E 1ɉMW)Eغ   U؉M̉MȉMЉUȃ]SPlA   p}WSul@PփWSVu ,^_[]UE$ ]UEpup ]UVP    Xu  $   Pp@    ^]UVP    ^E@$ ѐ  @^]USWV֋]UxQsQ EVsQ C0@E9Ƹ    t?<ESu
6 MIW44Pu9uvuȉًU19rDȉȃ^_[]UWV    X}  $   PƍFF$   O FF GF^_]UWV    ^}G $ G$%P   @^_]USWV    _]uUj  k QSjR(  *MQj k Qj QsR   t1@ESVuǊ ^_[]USWV    X]U}   s  D$t$$QP1A  FE ECEsX  ƋsX  EV]Sd8 ExzuSN8 Euxqu^;^t{@h   U4U4uu   u 	'k PSjR(  1A=1GVWWju1GuWWjVK  ul  1ɉȃ^_[]USWV       X]}U 	M$ 	i PSjR(  1F   s 	 D$t$$QP1F   @   PPQ)I AI)   9u#QSWu0<G  +jWDI WuI  ;Eu
   ^_[]` USWV    XM]}u3yM U D$|$$QP1At+   M h PQjS(  1@   G1QQ   E{CE9t7j0   ËSuu   u 9}uɋuu  1@HS   UH
   R   ^_[]USWV    XM]}uzq % D$t$$QPu~F$Ƌ   EFE4$^ D$E$UE4$PML$$  1* g D$L$$D$   (  1@^_[]USWV    XMu}uXyV ^ D$|$4$QPuKG  P+PR   PV׃1V f PQjV(  1@^_[]USWV      X}]U͉ M$ 	]f PWjR(  1Ap  w  	 D$t$$QP1A(  Fw  VuPP   F1PPUƃjjPE RVSPQuc   tVR1An  WVSVÃVۋ   UH   H0  QVRPP   ^1PP6VPSSPPWb      VVSxǃSwV5UPRWuMD     S?_  PWփ1ɋ;Eu.ȁ  ^_[]ÃZGoZ USWV       Yu]Ȇ E&  c QVju(  1A   j	jPB Pvua; 1A   PǍCPW|(P6uW_  u?Kދ    W]  Puփ1ɋW1A;Eu
ȁ   ^_[]-Y USWV    XMu}uZyu } D$|$4$QPuMG   P+PR   PV׃1u b PQjV(  1@^_[]USWV    X}U}   _ل  D$\$$QP1A  CE EwX  Ëw}X  ESW`. E   uWF. Å     EExE;xtd@4U4uu  tҋUH
aR   ل 	a PWjR(  5Suu  1ujjjuA  1Aȃ^_[]USWV       Xu]Mw U   vc 	ڋk \$t$$QP1A   FEp SVuPP   FEp SVuPPuyvjjS> SVBt]	  PZ  Puփ1c 	e` PVjS(  1A;Euȁ   ^_[]ÃSu6? U USWV       Xu}U M$ 	_ PVjR(  1A  ^ 	 D$\$$QP1Atq  C ^ SuPP9  C   PjSt= Sj x   M     SQPP   OPjs VÃVۉ؋MuPQ= Puq= =    aX  PWփ1 !1A;Eu
ȁ   ^_[]S USWV    X}]U$ 	$] PWjR(  1F   wu M	 D$t$$QP1F   E@WSu35 ǅtfH+HQW3D ƃW? VSǃV? E   W.W  Puփ1^_[]USWV    XMu}uMy~ ~ D$|$4$QPuQG   $$V  D$4$1*~ p[ D$L$4$D$   (  1@^_[]USWV,      ^]Q~ E}#=~  ~S QSju(  v  S =~ E~ L$T$M$PP1Cb   X AExjjP9 VUUUЍ@9t.[ PjV9 Vuc:   E ЃP $ }|XEHEp S6vuo&  PW" ;G   V	BJ9  P+PVEHS&/ ǃVg  p h   Wj u     CKP+PʃVRQ  }     Q3}W<  Y  P6W虨ǅ  Q1B9	P    8[j ;     VX  1SSP}W H V      R  PWփ   Q	P        SX  1QQPuփHS   .vjSp6 Su6 f! 1C;Eu4؁,  ^_[]ËHI W   /L USWV    Xu}#>y 	V PVju(  ;  VU싘>y Fy D$T$E$QP1A  ]]CHMP)ʹ   9uVE E]P߃6X  ÃSV"    MIM   Sjjju6 {M1QQ   E~q6X  ÃSu!" xfO]UR4uu   tTSjjju5  1Aȃ^_[]ËMURQ  1܃Sjjju5  ]UH
R   ^USWV,    _uEU@r!gw  )U QVjR(  q  FE^gw ow L$\$$PP>  }܉usEt;  F+Fu1VPI6 VV7 ExK}wu6, ǅ     j WVu
ƃW7 ExEUuԉ}؉EM] Rl  1;^M  AEЉ]1落ǃjW覾1VM4WWN  }h   SVuuV     j MQV  ǋHE S   Mu2  U܋  Ћ];^11  uE Mu܍qU z Qj RjV}wuV   uVwVE0@+ t>}ɃQPuEpEԃun5 Eu_1@,^_[]ËH1GuE S   MIE  	  	  MS|	  M܍zU PQ׃PS֋uԋM1G}A;A]t)E    SX  1QQPu։V4 HuE S   H}E V   HE W   E      !       USWV      [Es M  jjV/ VEppu˞  VEppu袞  jƉ ǃPW VW W,ǋss  RRQSMqu   ^      7  ǍS0W     S0W     1PWWQQQ S  19\S    WI  Puփ1   ss  Q Quju(  PVuj. 1@MSuV. W    Q  W1@;Mu  ^_[]D USWV       Yu}Ep Up (  QRvW   1A  jjP, PvvWǅ    j ƃWV+ V萷Ã ~[3:  ƋQ07V     j VSĸH   SG  Puփ1ɋ XO QVjW(  1A;EuWȁ   ^_[]ÃPQd, ԃPuP, V  S' 1AB USWV    [}} n  M QWju(    WUn o L$T$M$PP1A  u]E@E܋ EGEEpVd. É]}|GE 7X  ÃSuH    P]S2 NC+uN+N9}}E uX  ߉ÃSV xPW2 ;G   WSu迾tkM	  PE  Puփ1   E䍀M Pjjjuu+  S\SjjjuZ+  u?E䍀M Pjjju4+  1A'   SPjPu+  W- 1Aȃ^_[]USWV    XM]}ufql l D$t$$QPujF $ Ƌ4$D$   E4$PML$$  1*l I D$L$$D$   (  1@^_[]USWV       X]uJl M&6l 	?K PSju(  1A   S6l >l D$T$E$QP1A   pOǃVW'jjP' ~4NP3uWD  t߃WU1A.   WQC  Puփ1ɋ;Eu
ȁ   ^_[]> USWV       Xu}Uj M$j 	H PVjR(  1A  ^j 	j D$\$$QP1Ato  C ^ SuPP7  CjjSZ& S 蠷   M     SQPP   OPs V&ÃV}ۉ؋MuPQl& PuU& =    EA  PWփ1 1A;Eu
ȁ   ^_[]< USWV    [uU}   ~h E h t$|$$PP1A   uwE MfH k Qj RjW}wu      uw}uVuPPueEvuÐk xX  ɍd$Q   d$d$Pu֍d$1h  HH QVjR(  1Aȃ^_[]ÃVjj ju7&  USWV    XMu}u[yg g D$|$4$QPuNG  P;PP   PV׃1g &D PQjV(  1@^_[]USWV    XMu}u[yf f D$|$4$QPuNG  P;PP   PV׃1f C PQjV(  1@^_[]USWV    XuU}   ~Vf M	^f \$|$$QP1F   GEExE SWuPPufGM	     0E0 P׃Puփ1Vf 	E PVjR(  1F^_[]USWV      X}]Ue M$ne 	*E PWjR(  1A^  wne  	ve D$t$$QP1A  Fw  VuPPutv1PPǃj$jS  SWVVPQuL?   tW1Ao  PWSV蚺ƃW   UH   H0  QVRPP   ^1PPVPSSPPW[>      VVS躹ǃSBV UPR
Pu      W;  PWփ1ɋ;Eu.ȁ  ^_[]Ã讬YF66 USWV    _uEU@r$~b  pB QVjR(  1Az  p~b b L$t$$PP1AH  ]^Eu/H+HE       9tVVjj%jur  )  uMqX  EP3 E   E   C+C]1VP  VV" ]u1QQ   ǉ0E;ptUE@@U4Wu   tˋHW   u! Wu  uk! 1ɉȃ^_[]Ãujj%ju  Xu
E@1EM@ *e Qj RjVuvMQ   }APu3~ ǅVWS]ƃW  USWV       Xu]M` U   ڋ^` ` |$\$$QPM1BuwCEXWSQPP   {j&jS SWtK  Pw7  Puփ1ҋ9` 	? PVjS(  1BSu! 1B;Eu
Ё   ^_[]2 USWV    X}]U$^ 	< PWjR(  1F   wu^ M	^ D$t$$QP1FucE@WSu3 ǅt@WSĮƃW E   VG6  Pu׃1^_[]USWV,    _uEU@r$]  *> QVjR(  1A  v] ] L$t$$PP1Aj  }܋FE0}MtX  ǃWV ;FU   M+FEMtX  M܍6 QPEEjV ƃuV ;F   VuUEE    }   Mu܍; ` Qj RjWMqu   tSu薦Wjj+juv  1AG     WPj+PuV  V |Eƣ` E]ЋEp@9   }q
19tKU؋AM܋M<@WM4u諅}tÃE;puM܉U؃u  ǋwM܋AIP+PʃVRQ  E܋EЋ CS   M܉ j WuSU؃MA9/EЋ   u_3  Puփ1ɉȃ,^_[]USWV       [u}[  E&[  y; QVju(  1A  V[  [ L$T$M$PP1A   @j-jP G   B|YO6X  Ë vX  QPSIt}  1  Puփ1ɋ ;Euiȁ   ^_[]Ã; Pj	V Vu #Pux V1A, USWV    X}U}e  _+Y 3Y D$\$$QP1AT  uu[GE}3>ME؋{CE9tuEP ]h   Pj uV      MQuV(     } tj uu99}u]   M}WX  1QQPVUUH
R   HW     u/  PV׃1+Y 	9 PWjR(  1Aȃ^_[]Ë   uX  1QQPV׃UH
R   UH
R   u-USWV,    XuU}  ~W W D$|$$QP1A  ]HM܋HMvuP"EԋGOM9   E؋ E@X+Xt@Ep}܋M	h   6j 0u      űE MQuu(     } tj uuE؃;E_E܋]9؋}t&ǋM	1VV0R   9uډ׋UH
uR     u-  PWփ1   W 	7 PVjR(  eE܋M9ȋ}ut*ǋ1SS0R   ދM9u֋UH
R   u 1Aȃ,^_[]USWV      Xu]UT M$T 	y1 PVjR(  1AM  ~T  	T D$|$$QP1A  G~  WuPP   G1PPqǃj1jP4 RWSVPQu.   tWm1AJ  WSV'ƃWCH    H0V]SPP   v1PPkPQVVPPS-      WVVÃV訜WfHދ !   S*  Pu׃1ɋ;Eu3ȁ  ^_[]Ã&ݔn% USWV      Xu]UR M$Q 	. PVjR(  1AM  ~Q  	Q D$|$$QP1A  G~  WuPP   G1PP芓ǃj2jPM
 RWSVPQu+   tW膓1AJ  WSV踫ƃW\H    H0V]SPP   v1PP脒PQVVPPS*      WVÃVWHދ !   S'  Pu׃1ɋ;Eu3ȁ  ^_[]Ã?n" USWV,      _uUO E}!O  / QVjR(    vO  O \$t$$PP1At  vEH  SQuPPMb   @j4jPj
 C+CN+NQPjƃVS  K+K9ȋu\EƍNVUUUЍ@9tpa, PjV	 Vu
 / jW
 Wuj
 VhE@ȹVUUUЃP  |XEHEx V7wuwx  PS  ;C   S VWyO;O   P+PWE@11҉RPVj P躲P$  ƃW  x _  h   V1Qu     CKP+PʃWRQ  }   4  Q2}W<  6  P3W6xǅ-  Q1B9	P   > [Gj 苓;H    ?   WX  1QQPuփHW     7"  Pu׃1  `  >Q	P        WX  1QQPuHW   B]wjV Vu1   ȋ1A;EuKȁ,  ^_[]ÃD  H V   ; USWV<      _uUH E}$H  r) QVjR(  1A   vH H L$t$$PP1Af  vQRMqu   E
  j5jS EHVUUUЍ@9t&% PjS Su   E@P  A+AP芉    1 4X  PS  ;{  GPWS^    +{WV^F;|}|RExEpP6vuqO  PSB  ;C?  莍jS藊PVkǉg  SVWO;O   P+PSE@111WP诋ǃWPc  S[  XW  h   W1Qu   L  FNP+PʃSRQ  }    Q3}W<    P6Wzqǅ  Q1B9	P   8[Gj ό;H    A   SX  1QQPu։HS   h  p  Puփ 1  4j4jV  Vu	    蠅 1A  +  >QP        SX  1QQPuփHS   
PPvjV  Vu    肄1A ;EuKȁ<  ^_[]Ã  HW    USWV       Y}]NA E&:A  " QWju(  1@N  j6jP+  Pwu  1@
   QKQP*|S  ƍP7uVJ  u\j V8tOK:A    ?  Puփ1 LVp  $7jV  Vu  ŉ 1@;Mu   ^_[]B USWV      [u}?  Eǅ    Gr ?    QVju(  8V?   ? L$T$M$PPt1@;M    ^_[]Ë@j7jP    vX  1҉H {H H H ǅ      ǅ    ǅ    M  Rj c! RjB R6Q   E  {H  ?  ÷B wC+  1AH {H vu0	    j N  Qn! QP_1A{H {H vu0    jP  rǄ{H    vu0Z  ǅ    e  4Ǆ{H      Qvu   )  Meۺ    u-A+ARP1  ÃVS  t VPWSaVPS  t,	  PC  Puփǅ    Vu  1@P1@6j
Vr  Vu  1@tS  tW  f     D    USWV    XMu}   Y>; F; D$\$4$QP1F   CEAI)   9ugp0at[  u藝P  Puփ1>>; F PQjV(  1FSjSjj8ju   ^_[]USWV    X]}U!H: 	d PSjR(    sH: M	P: ED$t$$QP1A   FEsE uVuPP   vutyMƉuEU   3u RVQPP   OEpuV輈ƃQTMu   uPj9PQ   VPj9Pu   1Aȃ^_[]Ë   u  PWփ1ЃuUSWV    XMu}uqY8 8 D$\$4$QP1Fu{KAI)   9uR  0Ǿ  P@  Puփ168 [ PQjV(  1FSjj;ju   ^_[]USWV       X]}
8 M&7 	 PSju(  1@   S7 7 D$T$E$QPt1@   G pƃ Vj<jPv  |$P3uV3  O⋅   V  Pu׃1;Mu   ^_[]h
 USWV       X}uM6 U   6 	6 D$|$4$QP1F   G MqX  j=jWd  Wt]	  P  Puփ16 	 PWjV(  1F;Eu$   ^_[]ÃPu  /	 USWV       Xu}U5 M$t5 	 PVjR(  1A  ^t5 	|5 D$\$$QP1Atm  C ^ SuPP5  Cj>jS  Sj     M     SQPP   OPj s VtÃV~ۉ؋MuPQ  Pu  =      PWփ1 }1A;Eu
ȁ   ^_[] USWV       X}uMj3 U   V3 	^3 D$|$4$QP1F   G MqX  j@jW
  WPt]	  P
  Puփ1V3 	z PWjV(  1F;Eu$   ^_[]ÃPuS   USWVL    [uU}  ~ 2 (2 L$|$$PP1A  ]̉u}܋EAE quAE $%yEԋO;O    ]  }M$q  E D$M؉L$$D$   D$      G  ML$t$$(  )  }    MЉL$$D$      V  M̋  $މ  ǋ,2 L$|$$PP  _E܋ KL$$  u
   EuԋKL$$  ǉ|$\$4$sy<$  UH
}$   ME;H]u1g   2  \ L$t$$D$   (    u}UH
у  E $     |$]$D$   D$A   D$   P     u  M̍ $D$	  v}u 	  U싸	  $މ|	  D$ $׉D$$Usu   u   $D$   D$$֋}苰 	  	  $|	  D$E̍ $׉D$$1@uEuE    u؉4$X  1ɉL$L$D$$׋UЋH
E $   H}E 4$   HE <$   } tEԉ$Kx1AȃL^_[]ËE   Eԉ$I  D$$1Ӑ!    !      USWV,      XuU- M}/- 	 D$t$$D$   (  1AJ  v- - D$t$$QP1A  F8E@ Ep4$  4$&  }|}1ML$X  D$<$  ;_	  CL$D$\$<$  *  +_\$$  F;|G+G   +MD$K  G+GD$$X  Ë3)GDL$D$T$<$  ;su҉  4$X  L$$Pǉ|$$  ;C  $SsN+NL$$XtN;N	  $Ҭ  x(@D|$D$$x  <;^uӉ$~   ;Xt< p@Dt$D$$  4;Yu͋ $   D$$D$    s;H$4  $&     $  D$E$1ɋ   t$ލ$D$     \$E$I  <$O4$  $  $  $     $  $k  D$E$D$   D$@   D$   A$i  $%  D$   D$D$E$D$@   d  1A;Eu
ȁ,  ^_[]  UVPuF$8s1FF^]UVP    ^E@$qtMAF1 A^]UVPuF$聖F$  F^]USWV       X]( M(  QRSu   1F   j jP  uFQ0EPZ  tJRSquPz    &SjS  Su  ;Eu
   ^_[]  UWV    ^}'     x/ H1ɉHH^_]USWV    _]unƋ'  MUQRuS   1Aux}}uVoEHM~+EHMW0SVM   tكV q1A+}Gt@t	WЃE荀/ Gw1ɉȃ^_[]USWV]}E0  SWuVB  tV  4j Vuo1u!WjS.  Su  1@^_[]USWV    X]UK& 	W& D$\$$QPt1GcE{ OL$$/  Ɖt$|$E$n1u)}\$<$D$     |$E$  1G4$T  ^_[]USWV    X]U% 	% D$\$$QPt1G^E{w0  VWuuo1u$}SjW  WuA  1GV  ^_[]USWV    ^MUE8 t8E$ 3X   QP  UMtu12  1?  E[UƋEMUQRSV   
  ًu E]   QShg} ~~E0X  Pu&  ƋMqX  Pu  xLxRPVSgtLtRMAEEEE E Qj	/E0	EpjEpE0ju V*  Vu~  1@^_[]ÐUSWV    [uEHUT$t$$P41}    F# EkU<UJ $PǋX4$  @$Ӊt$$  E$Ij$G 1@$  G14$  GwwwEGG ML$$+  Ex^_[]UWV    _uEHT$$P0p" 	$  ^_]    USWV    XEFt=Q1M싉," 	$   $C  $kF$  FH;uۉ$  F$  ~ut3E싰," GQ$   <$Pۉu֋EXt1Ex$E싰,"  t<$  G$  <KuދE싰," }$P<$P^_[]UEHuQP01tA]USWV    XE  jxP1~>~~~~~~~$~ ~,~(~4~0~<~8~D~@~L~H~T~P~\~X~d~`~l~h~t~pj  NWQ  N<WQ  M   VQuu  ^_[]USWV    ^E t$    Mq]\$4$L  H$  u4$  Mq<\$4$L  Ha   $  u4$  Eptt*FQ$   4$Pۉu֋M$P^_[]USWV    _w   ^C?1 	D$ $Qr^C$  T^Ct$aC(t;$a1^$  CtQ1 	$   1  $P1  4$P^_[]   5   E   h   USWV    X}]V  $   P$   n  Ft$$  ^_[]UVP    X
  $   P$   (  F^]UE H+]USWV    Y}P;u>  $   P1C$  Ft$$  $  $  18 u@]$  @\$$  1FtC{@$fD$<$  ^_[]UVuE $J  @t$$Y  ^]UVP    ^E@$    @^]UWV}EH<URWQPl1} tW
   F^_]USWV    X}p p<$3  
$։ƍFF|$$  1>$    PFx8xxxxxx^_[]UWV}EH<URWQPl1} tW
   F^_]USWV    X} p<$~  
$։ƍFF|$$\        $PF$     NA^_[]UWV}EH<URWQPl1} tW
   F^_]USWV    X} p<$  
$։ƍFF|$$     $0   PF1ɉHHHHHHHH$H H,H(^_[]UWV}EH<URWQPl1} tW
   F^_]USWV    X}C p<$  
$։ƍFF|$$     $   PF$   .  NF@    ^_[]UWV    _EMQ<D$$Qh1tF$     4$  1@^_]USWV    XMM	w  MAxQ	  y   QPR=  Ep>Pw  O;AtRPQ  Fx9F  u0uv}  N;AtRPQ  9uɋE@8X9   @tPf[9u   MA0XxQv
  N;AtRPQ"  us  K;AtRPQ  uw  O;At[RPQ  FEx;Xt03uvl  N;AtۉRPQ  ƋM^_[][   F        UEH<uQPh1tA]USWV<    XE̋E   S    HMpM܋VU؋A@EBxvvJ  G+G1Q1Pa  ǃWjv\S  É]1P  EVPSvo  É]W$  u:}̍  MUPu]SV    MUPuSV  u  u  MA+A1SP  ǃWSv!\S  É]1P  EVPSv
  ÃWX  u;}̍7  MUPuuV
  R  MUPuuV
  Ã} ǃu  u  FHH   HMXMSU؋A@EBpuȃssX  F+F1Q1Po  ǃWjsZV  Ɖu1P  ESPVs}  Ɖu܃W2  u:}̍  MUPuuVS    MUPuVS  u  u  MA+A1WP  ƃVWs/ZW  ǉ}1P  E1҉PW1P  ǉ}؃Vd  }u9u̍7  MUPWuSK  R  MUPuuS/  W  u
  CHHSUЋBpuăMs(s$
  F+F1Q1P  ǃWjs(5YV  Ɖu1P  ES PVs
  ƉuԃWi  u:}̍  MUPuuVSP    MUPuVS6  u  u  MA+A1VP  ǃWVs(fXV/  ]ÃV  E1҉PSV	  ƉuЃW  uu9}̍7  MUPSuV  R  MUPuuVj  SS  uE  F(HH1@~,    UJ+JuȋV+V}ċw+w9tY}̋U E 1J  RQuV  MVR  E    1JRQV  11Ƀ} t} t} t} t;  HMԋx}MGpu܋A@H+Hj Q-  EG1C9   E̍R  EЉMMxApUWEp  ujWwVN+N9   j.  ËF+FPj W}UƋ>F9t$? tWVx  PS  FՃVx  MUuSuu
  ST  1ۋ}؋GGM;M]1V  EVs  E܋]3;stW>tVS  |PuPup  E    ŰU    QRP1ju̍  }UPuu}W!
    MUPuVuW
  ub  VV  SJ  E<^_[]USWVMEIypL$$裼  Åt0$UǋFN\$|$L$$S[$  
4$WǋE$M   ^_[]USWV׉    [E$  Ft1H$QFxF H 	$   1@P$    R  1^_[]USWV    XE]}GpuSVzVt(FN)QuRtW]   t81@IS輖  ƃVju  E
  VPFPPV6t1^_[]USWV    YEUEz1@tvM ]
MrEPSVR4} tMH8Ou1@=}t4u  ƃVjW  E  VP1^_[]USWV}u6W\   GXFPVSXsE;Ct/u6W,   +Cj P6W ;suԋE^_[]USWV    XuMM~t/  EFML$$V0$  <Ouڃ^_[]USWV]E$  Ɖ4$  {t ]D$4$M  u<Ou1O+]i4$  ^_[]UWVEMQrEkE<T UT$}WRTPt@FF^_]UWVuF$  ǋF$  $    ^_]UVPuF$  $TF$I  ^]USWV    Xu}]q
  $   PXxNF^_[]USWV    XUMZ/
 E $X  E1   M11E9Nu(E ˋN$X  ML$$  مt
6uKtEXFQM	$   E 4$PE@ۋM{E^_[]USWVUM    XYt>My$W	 o	 EW  ujW  <Ku׋u^1@;^tDuj 3}Wt&;^t"+FuP3Wu11@^_[]USWV׋AXj u/MEs;stLkE<L$M|PEEPut+KpQuL;suƋEP  ,^_[]USWV։My1@9tTtFu|Sur  PuMu SuU  Pub  19u^_[]USWV    [t
MA;ux  E@@E U1J  RQV  EM@E1IQuV    1JRQV  1IQM1V    1JRQV  EEHP
 1HP4VE    1JRQV  EUJR
 1HP4VE    1JRQV  1IQM1V    1JRQV  Euuu   ,^_[]USWV,U    _E@E苟 ]1J=  RQV  1IQuV  J  M1JRQV  M	;M   L  E܍S  E؋}1NME@M	4\  EVuS  VuS  uPVuS  VuS  VuS  ME9c,^_[]USWV    _t
MA;uK  E@@E U1JZ  RQV}  ME܋AE@E1IQuV    1JRQV  1IQM1V  i  1OWUQR  M9;}tCE䍀{  E V0u  Vuu  E9uɋuNQV  M䍉  jQV  EuuuQ,^_[]USWV׉    Xt
]K;uTMIIMK 	<  jPV  UV!   EuSu^_[]USWV    [EFEFE 1JRQu    1JRQu  1IQM1MQ    1JRQu  ENV" 1HP4uE    1JRQMQ  1IQM1u  ~, t  jQu    1JRQMQ  NV " 1NV4u    1JRQuV  1IQM1V     1JRQV  ^_[]ÐUSWV    _]uU&  J  QSjR(  *MQj _  Qj QsR   t1@EuSVuǎ ^_[]UVP    ZME}	uuQP$L  "  6g  RQjP(  1@^]USWV    [u}N  E n  QVju(  1F	  M  Qj Rj Qvu   1F  EEE   J  E]1  OM1MX  uPuEf  ƅtn     uEpug  G     uEpuq  %  M	  PRQY}   EpX  PuESf    P   PSփ1   U~VE   QujR7uEpuq  {E   E   Quju(  Tu"  5  Puju(  -E MqX  uPRo  Ɖ^_[]f   )            UVP    ZMEuuuQPY  "  6  RQjP(  1@^]USWV       Y]u  E     QSju(  psX  j	jPm  Psu)  t)QSVPuu8   1@;Mu   ^_[]n  USWV,    [U}}     rX  PuW
?  1A   >EH  T$L$E$WPu]u]SuVuO@  u@CxSVVt6{ ؋MtGqj-,  QRjW(  1Aȃ,^_[]ÃEpj,jju]   @   AE AE }؋w;wE           E    u6d  uh   Wj u]S      MQuS(  Åui} t/WuVuu  ÅuJuVEuHE W   1ۋE;pA*H
H1CE W   E1ɉHE    uVX  1QQPu׃HE V   U܋H
uE R   PV]SRo  !ߋ]ԅۋutMq     S   PWփ1USWV       _uU   E  }  QVjR(  wvX  PuV;  1AtR3H  T$L$}<$VPu# ]SW&=  ut!1A ;E  ȁ   ^_[]ËG@jjP  WSt# tAvj-SE  Svj,V&  Vuz  G   ESPVRWu1?   ǅ   WVÃWz  ǅ    ;X       3  WVu1r  1u+SVG  v   ڃ1@H  W   ǅ    F    Pu]Srl  -       PS׃1  USWV      _MU   E  }΍  QVjR(  1A   qX  PuV9  1A   H  |$L$4$RP    }WVC:     C@j>jV?  SW$t'{ ؋]t^qj-,Epj,Vd  Vu踲  1A;E  ȁ  ^_[]@   uM   8  WQPP   PjwSy@;tS?NuPEV     SSWVOr  tSP1ɅuSu茱  1AH  W   EE1AC    QuPEi  m  sW  1N  USWV    Yu]&     QVju(  1G   M}j W%H+QjWur  uF}}   VSPL  j PuV  Eu9 	  M싘	  V|	    PQӃPVU1G]9EEH+Qj RVq  EPuVyf  D^_[]USWV    ZMu}  u9]qX  PSV4       t j  RQjV(  1@j ЃPV׃1^_[]USWV    ^M}  ?  }qX  PW}WO4  1A,  MǋO  T$L$E$EPP:  G@E1VV   EO    M܉}?}VV   E7;wtH6E0     j0   Puu׃upE;puuuu   uJ}<MI    cuu  `~  RQju(  1FEAUH
R   UH
R   1@Ɖ^_[]USWV      ^U]@   Eǅ    ,  }%  QRjS(  1A   rX  PuSM2  1A   H4  T$L$E$SP    WuVu3  ugjjS藫  WV|t EtVpj-,Epj,Sƫ  Su  1A;E  ȁ  ^_[]A   A@056ǅ    E   ǃj /  SPEpVu4   Å   1sC76  H  S   1~QsO虮  PPEtur3   Åc1F11F]:  @    VuSb  مtv7o˃  tr     OPSփ1SjV让  V]S  1F%z  USWV      _MU   E  }΍_  QVjR(  1A   qX  PuV.  1A   H  |$L$4$RP    }WV0     C@jjV  SWt'{ ؋]t^qj-,Epj,V?  Vu蓨  1A;E  ȁ  ^_[]@   uM   8  WQPP   PwS8;tS4NuPEVΧ     SSWV)h  tSP1ɅuSuf  1AH  W   EE1AC    QuP_  l  sW  1M^  USWV       [U}E  	M     rX  PuW+  1G   H  \$L$E$RP   uu$-     @@EHSSuPPugC j&jS  utNx Mtdqj-@  QRjW(  1G ;E3     ^_[]ÃEpj,S  SuF  @   W 6   PQWSue  tSP!1ɅuSuˤ  1AHW   }EW}W胤  1AEC    QPW\  ǅsQ  1ʺ  USWV    _EMʃ   UZ1҃    u  	pX  1҃Ƌ  }RR   EMQSL  MQ  Å   E@,tu[tVS  t   jS   Puu։}tUH
1FJR   9  	  RPju(  1Fuu  1^_[]USWV    Y}]u   |*ڃt#Vl  uWSV:    QWjV(  1@^_[]USWV    ZMu}C  uJ]qX  PSV'  t:  j0   PV׃1  RQjV(  1@^_[]UVP    ^MUE}$  ;  VQjP(  1@utQRPH  $^]USWV       ^}]Um   EY    QWjR(  iwX  PuW&  1At>	 Ha  T$L$<$ PPt!1A ;E)  ȁ   ^_[]Ë@@ j/jP賟     EQPRSu5*   ǅ   s 3*ƃWǃQfe  ;{tj 7V+   VPu׃1.  Pj	V  Vu蕟  U  USWV       ^UM}s   EA_  rg  QRjW(  lrX  PuW$  1GtIǋOg  t$L$E$RPuSuu%  t!1G ;E+     ^_[]ËC@j3jP蹝  }      EX VSuQPPus60S  t2 Su<te{ ؋M   qj-W6  ÃSjV%  Vu̝   SPEpj,V6  Vu芝  @   SQPu]  ǅtQSWV1؋]uPu2  1FH W   }	1F}]H
 R    @    VSWT  ǅ su  1  USWV    _]`  }  QujS(     MM    Qj RjVދ]qV   1A   E M썗    Qj RjVދ]sV   unMEtp   E sX  PuV!  t3M#    }t  ssPVS  '  1Aȃ^_[]ËEw9  }  sQuW     M	  M	X  PRQ}   sQuV     }   QuW     E     jQ   PuL}   PV)T  D}   sPVU  +E X  QRP}uossPVS  E q  E   E   QSjuE   %E   E 5  E   QSjVuf,  r                USWV    [M}  upU0  Rj Vuj  RqV   uYMEtYt{   j Q辽H+QjVu'[     0  RQju(  1@^_[]ËϋH+Qj WVZ  jWVO  ?ˋH+1QVSEWZ  VSW\O  뢋C  PRQ1USWV       _]uUy   Eǅ    e  ^  QSjR(      sX  PuS  1AtX 	Hm  T$L$$PPu}WS  t!1A ;E  ȁ   ^_[]Ãj<jS誖  W芼t.x ]tDsj-Vݖ  VEpj,SÖ  Su  @   @@ǅ    0C!I  ~nNPP3u   ǅtwW"H  W   w1GPuuV[N  t#l  tq     PWփ1W  USWV      _MU   E  }΍<  QVjR(  1A   qX  PuV  1A   H  |$L$4$RP    }WV     C@   PjV  SWعt'{ ؋]t^qj-,Epj,V  Vul  1A;E  ȁ  ^_[]@   uM   8  WQPP   Pj wS-";tS NuPEV訓     S蝮SWVT  tSP1ɅuSu@  1AH  W   EE1AC    QuPJ  m  sW  1N8  USWV    X}]u!  	\  PWjV(  1@%1~K7uVa  t^_[]USWV      _U]E&  	M     rX  PuSS  1@   J  T$L$$PP    uVS     ߋC@jAjP艐  SVnt:{ Mtsqj-2  QRjS(  1@2Epj,V蓐  VW  1@;MK    ^_[]C   A A A 0؋Xǅ    ;XD  3'   h   Sj W      QW(      ދt1PVSW(      H  V   ;X%   ڋH
1AuJ  R   3H1Au  S   DH  V   1EuB    PVQF  A;At  v        VX  1QQPu׃H
  R   H  V   H
  R    t9 u0   ʨPuփ1@  USWV      [U}Ej   	MV  F  rX  PuW  1A3  >H^  T$L$]$WP   WuS     G@jBjPڋ  QR}qS      QPSW;   tcPSWQ  Su9tJ{ ؋Mt[qj-@l  QRjW(  1A  ;E  ȁ  ^_[]ÃEpj,jBju,   @   31ɋ;S   EX22  h   Wj S]S   tBPWEpSo   u.1@:H1A%W   q      X  1QQPuӃHW   @    1G9PuusB  Dù   Tu7  Puփ}  USWV<      ^UE  M    rX  Puu
  1A  H  t$L$M$RP  uuJ    @@ExVWuPP  G p;ptB>73xt  ;AZ  0W\M   ;puv軕  vY  y uR  y ǅ     H+Q  8;x   1ɉ73p  Ã36t     P6p  PSwtpVی  A  ;xw؋]މxY    )֊   ]ދ   <kA<D 9F;uhVӔ  m  EpjjCjWQ  {  ܴ  QRju(  1A;Ev  ȁ<  ^_[]É H+HQG  ǋF	9t/3;st6W蹍  ;suꋅ ċ H+HQW  ÃW}  jCjW  VEP   ~ u     Epj-S#  SVy  7j7jjCju>   EpjjCjV讆   1@  S臈  Epj,W蓄  Wu  덃0*Q1;Q  F9   0ǅ    1Hǋ k<L$WQTPt9x9tG  9n1N1" )9Ƌm  p8|PM  ;X   AI @0vn  ǍP 47uNF+y8u r1 Y Q   @28x  QPu{   u1;Qo苅  @    Pu}W:  t9u5 󋰴  ŜPW։ރ1@  66܄  ;3u  ×  USWV      _] u   EVtH  Pj)S2  S@  U]P  uuSVDi    Q  E  ]I ǅ    ǅ     QR3uV       k<L  P蹂  ËH~PX  Q1ЃƃVul    PS  u  Sڇ   t47SX  e  S7B  O  <űL$  T$$D$  < eh   sj 3uV      a  uQj"j 3V   sV  1   Sj"E Vx}  Vu~     Q  nVju V=}  .0jS   0j
u V`}  S4  uVu}  1@	;Mu@  ^_[]Ã3jE W}  WVW}  u諝˓  USWV։    [4$  r>:u6F:u-  ]$  t$$D$  ]t$  ^_[]USWV       [U  E u   7  RQP1zUM}P+PW1t+  	upj u        Q  ;Eu
   ^_[]{  USWV    ^]}Sՠt!m  Pj)j?jW|   1G
  þ  E uX  PSW   1G   E K˾  T$L$M$PP   UCH;   { tC   1MWWW8  {E 1URQj"W3MQ   uWW3u   Su.ul  3j'jjR{   ^_[]USWV       X]}z  ES  r':uC:uSu   W  ƃu辛Åusuof  W  W  1Jب  RQW  SuW  uGËf   W  uuj#j ju3z   ;Eu
؁   ^_[]  UWVu~ tMF   1WWW6  ~^_]USWV    YEx1?   ׻  MƋWP|F H1A9W   M1U"   MQRW    Q}7uV   h   w1Q7V   uu"   Q1Q7u   ^_[]USWVU    [  u  W  x tjpu  E    1@E1Og  WQMS  WuS  E^_[]USWV    _Eu@@EuuV0s  1   }SuV   ǋH1A9E싀X   S   u   _s6
b  t.s6g  MuVPSutGWjuVGv  Vuv  HE싀X   W   1^_[]USWV    ZEu ~ t+~ u%MF   1M  WPS3  ~^_[]USWV    Xu}M   EE@X]Px  E39  ƅ~mE  EME 7X  ÃSE0rb  x}Puz  uwSV(=  t|Muu艬x?Mtux  m7j	Vt  Vuu  JSjEf  Pju-t  uut  uAx  V:  1^_[]USWV    ^}   j uu  ug 	  M	  EWv  EWv  ǋMS|	    PWuQUPSU1@Utt)uS  WVPuuS   ,^_[]USWV    XuMUIIM    |$t$$QP  }]EM00i  VSuu  ǋH1A9E S   H]V   M&  QW}WPP  Fp]3]     }7EP謩uVEPWS׏ ߉uYuVuuËE+Gj P1uJu&VjESq  SuFr  1CHE V   DVjESq  Suq  HE V   1@^_[]USWV    ZEu ~ t0~ u*}MF     PWS#/  F    ^_[]USWV      [}e  EW[t&m     Pj)QQur   1Au  MQ  qX  PWuh1A4  ǋOY  t$L$E$RPF  G@QR]su     ׉    Rj P   PWsu     sX  PuumÅ~  suPP_  C@QRwu   ,   Qj jwu      @   t   wj(  ;  1PPoǋ  H   0X  P[  Ë1X  P}[  M  M  PSW)   Qe1FVWVWǃWTƉWtq    j: M1X  MVpuP输ǅ  GʋI  HI  Hp I  H I  HHHWrh|  63覃  tWvF|  M:  1uu˕1G Q        wj$PPum   1A      MqjPPum   {   0jQQum   W됃   j%PPul    Q  1A;Eu
ȁ  ^_[](  USWV   ։    _y   E>
  ǅ    p@j:V  Å4  K9K  e    jQ   I)˃SQWPH1ҋ  RQWPDqW    1RQWPDh   W  ËH1A9W      e  S  SV赕ƅuS  S    F e   	j !F e   	h   P  ËFQ1B9 	P   ؃   9   e     S  QPS׃/  Sl     pDS      j QqRPփ    tx   e   Sl   HDS    ț  j QqRP We  Wl  pDW      j QqRPփ ;Eu
   ^_[]k~  USWV,      _]Ī  ES躌t#  Pj)j(juzh   1@  Mދ  qX  PVuV1@  I  |$L$4$PP  @pQRMqW      i  6ËH~Z  X  Q1ЃǃWVeS  `  PSm  uWjJ  S茝 uM  C+M1X  PuP迌  HYPru  jr  EM  1X  ǃW{    Wzv    WuV  pVQP  C@QQqV     ;o   jPǋ1PPGWps   3PH~D  X  Q1ЃǃWP  y`  jps  @ 1Ʌuuc1Af  p  Q     Wjj(juc   Smf  QEpjj(jVFd   S>f  5E0j%j(jV$d   Sf  Q  1@;M   ,  ^_[]ÃWjj(jV>c    ?j*Wj"j(jVc   pj$j(jVc   }e  o  uQ  Kx  USWV<      ^  Eu蛆t É  Pj)jju[b     }  7X  T  QPw  OEωϋwX  Puur1@X  I  t$L$M$PP  @@QRwu   ދ  QRwu     z     Rj P   PQwu   '   u:@$   t
(   upj+jjua   Qj jqu      u:@$   t
(   upj+jju`   qX  PuEPg0  RpuQP  G@  QRwu      w X  Puuǅ   wuPPq  G@QRq$u   &  ;G  ;E  1X  MW`uP  xG@$  G@(  G1PPhGG@$  G@(  G 1PP'G(G,  wH   0X  PXI  ǋ 1X  P.I      PWV    Qe1GWV;WVKǃWFW$_  ~ c  j@(0H   0X  P H  ǋ1X  PVG      PW   Qe1@VWVWǃWˑF$W]  ~$ V  jvi  06p  tpii  VpNi  MB1uuӂ1FQ        pj$$   q$PjjuZ      j%jju*Z   f$p+ 0jp$jjjuiZ   uQ  1@;Mu<  ^_[]3o  USWV      _]  ES}t&  )   PQQju?Y   1@  Mx   1X  MPSP\M  MuPJFP2e  HM   1@)   1X  PS]Sǅ    wSPPurW0f  wf  @]oMntQ  1,uQ  1@7E0j%j)juX    Q  1@;Mu  ^_[]l  USWV}]uSWV8   tWpA1Sj&jjVW   1@^_[]USWV       X]}  ESl  r':uC:uuS   WpƃuրÅusuo  W  W  1JS  RQW  SuW  u_Ë   W  uuj#j juU   ;Eu
؁   ^_[];k  USWV    XEu]  uX  PSVeÅ  1QQ   Ƌ  M  CE}싸   |  jQ   PVEP׃:  E    js   PVu׉  E    M	j1   PVu׉  EMj p    M	PWu     E    KS]Ǜ  j4   PVu׉G  E    MIj1   PVu׉
  EPMjp2    M	PWu     E    KS]Ǜ  j4   PVu׉  uj&jjVS   1F  }UQ  PRQE Wu  1Z  CE@@E}싸   |  jQ   PVEP׃  E    js   PWuփ  MI1MM	   j0   PWuӃ  EMjEp  _  M	PWu   A  ;uE    Mj1   PWuU   Mj s+    M	PWu     9uucCE}싸   |  jQ   PVEP׃z  Ex, t<E    MG  jQ   PVu׉5  E    js   PVu׉   E    M	j1   PVu׉  EMj p    M	PWu     E    KS]Ǜ  j4   PVu׉B  E    MIj1   PVu׉  EPMjp-    M	PWu     Mj v(    M	PWu     E    KUǛ  S j4   PVu׉E  E    Kj1   PWuփ  SMjs(<     M	PWu      CE}싐   U }  jQ   PVuVU   E    Mjs   PWVUukE 0E ;pt2M	   Uj0   PWSUt!E MqWS   H1FE W   ^_[]ff     &    USWVUM    ^uuË{}䋆T  E 1QQ   9tWEE@@03V4  M	   j0   Puu׃u9]uu'#UH
1E R   ^_[]USWV    X}  1ɉL$$   EO<UT$$L  M$  Åt}E@dtu[ML$$  t   $D$   D$ED$E$ׅtUH
1G$$   ML$M$  1^_[]USWV    Xu  uX  PVu1@   1QQ   EC@E9t6   jp   Puuփu 9]uʋuu  1!UH
R   1@^_[]USWV    X}  uX  ƃVW}Wt'  jp   PWփ1Vj&jjWEJ   1@^_[]USWV    Xu}Q+~]]$TX  ǅt@}   (  	  /x  $D$   D$<$1   E$    U  ËG}@E9t1M@8p9tD$$Y  9uM;MuσE;sE    9t#M9tuE$V  uE 8X} t9t($)9u9t$L9uE$m$W  } ^_[]UVPuE $V  8 t
^]l^]UMUEP'   t1@uuu1]USWV    [G  uVuVRV   1At   Ethu    ~S1EURj kw  Rj  R0u   uBEß  MAEEEE01w  j QV  1@^_[]UMUEPt1@uuu1]USWV,    Xu]P  M؋	M<  E܋ 1ɉL$$   E[txSE$   u?EECEE܋    ML$$      D$ED$4$ׅtuHE܋ 4$   1@ME܋ ML$4$  M1	;Mu,^_[]\  USWV׉M    [F  E 1QQ   E  1@t.E    j3   PuuփuCuċEM1$UH
1GE R   ^_[]USWV    _MEA@     j Pu  ƃu> 	  M	  MQ|	  s  PQӃPuUE@    ^_[]UWV    Xu}   $   P@    xNtFt1^_]USWV    XUMrtȆ  E $X  Etx1 ufE O$X  ML$$EZ  uBۋ7t37EptGQM	$   E <$Pu7u1^_[]USWV    X]  1QQ   ǋ[tۋt7sWu   tH1C#W   Wu  1ۉ؃^_[]UWVu1~ t0~ u*EMF   1WPW   ~^_]USWV<Ӊ]؉    X<  Eԋ j3   SEPt+uԋJ1K  W   15  uЉ}ċuԋ}EEDЋMMȋ@]؋KM܅ɉ  ]܋AC  Ű1QQ   Es]P]S     uuS     uuS     } tuuS   a  tWuS   B  MQMQuS   #  h   uuS    } t~}]̅tHS   }W  Wl  uSWPP}   MɉډDEDȉMŰ]HS   ڃM؋]܋HEȋ }ĉӍHUȉ
R   HW   ؃<^_[]ËUH
R   UȋH
}gR   SUSWV    ^E}  t tÉ؃^_[]ËMG   1ۋN  USPu_t`XF  M	uwuQPteE MS  PjEWa>  WS>     n  PjEV=  Vu>     G@03)  t3K2  ǃWjES=  Su8>  E WPH1ۃE V   USWV    _Ext   ]Ԁ  rf  jQ   E}    tkk  	qk  jQЃǉ}u   ME S  E Ext?   1QQ   EwPS   uʋuuS   uuuS   uuuS   {MQMQuS   uh   uuS  UH
4Q    UH
R   UH
}R   U܋H
R   WS  HW   ^_[]USWV    X}_+_~   $PƉ\$4$2R      ~FF^_[]USWVuE$EV9tEu~p479uE^_[]UWV}uNƋGOP+PʃvRQ	   ^_]USWV    Y}]19tKuG)E싁}  EtQM	$    9uЋE@^_[]USWV    _MtjAX+Xt@Mq}t}  tQ$   uۋE@}$/%  t}   @^_[]^_[]UEtQ]\]UME9t"qp$  1t]   1@]USWVEHyQU1A9tPMQUIMp3ui&  M46S謣1t9}u1@ȃ^_[]USWVEHMY1;Yt$x173M;Yu^_[]USWVE1;Xt.}1GO@44蛫ƋE;Xu׉^_[]USWVM1@;YtTEu8FNM@EEH4E4E4袢1tE;Xu1@^_[]USWV    [}u>|{   i  QP]^WS#  ;st_uu6j ǅtW+sE@0t*ZE{   Q   E@<1@%Wju6  uju
7  1^_[]UWVu~uW"  1;Gt	+GN^_]USWVu}EMQIMH@EBEAEYQVWR   1t'+]MًE))ǋEǃQVW1@^_[]USWV}_C+CP@  PVssWHuV1^_[]UWV}wcƋGOP+PʃvRQ^_]USWVuE$#+Vt19uv4X4;u^_[]USWV]}Sw(  ƃuSW   D^_[]USWV    X]UMy}IMJK4Itx   VPCVPK  H   ;HtECUZ1F}MsPES[71Mt0E;HEuF^_[]USWV]M}G{SPq'(  tD}LD uSQu-   t0S莫u)^_[]USWV,]uMUz}ԋRA~{q;q[   }Eu؋E܋
MUuVSP5ML ]SVQuuVU } L8uu؃]\8M;quM\}C+]؃,^_[]USWV<    XEċ}1EW)E1AM؉ẺEȉEЉMȃ]Sw6&  ƃEPSWDMċv  	PQƍEPSVWC <^_[]USWV,    ^E@1ɉMW)E1BU؉M̉MȉMЉUȃ]SP%  u   p}WSu$@PփWSVu. ,^_[]USWV    _]uUmu   Q  QSjR(  *MQj Q  Qjy  QsR   t1@ESVuy  ^_[]USWV       Y]}t  Et   	c  QSju(  *St  L$T$M$PPt1@H@jjP0  PSWu.   ;Mu   ^_[]G  USWV    XM]}u3yt  t  D$|$$QP1At+   t  ab  PQjS(  1@   GX]1QQ   E{;{t:j0   ËSuu   u#E;xuƋuu  1@HS   UH
   R   ^_[]USWV       _Er  Muzj	jV.  VEpu'  1F   PPEpuQ%  t8Rr   7a  Quju(  1F-r     $  Puփ1;Eu
   ^_[]E  USWV    XMu}u[yq  q  D$|$4$QPuNG  @P+PR   PV׃1q  %`  PQjV(  1@^_[]USWV,    XuU})7q  	_  D$t$$D$   (  #~7q  Cq  D$|$$QPt
1G,^_[]É]M˃   GE@E$0  E|CE $X  ǉ|$E$     D$E$4  K]C+CD$u4$45  ǉ4$0  |$$X  |$D$E$Ɖ<$0  E   4$"  D$E$E |$M$  1|$1G|$|$E$D$   -  E$?0  USWV    YU}o     ruRUo  \$t$M$PP1FusS]SuPPu\     sEp<P׃Puփ1^  QRju(  1F^_[]USWV       Y}n  EVUUUu  Ѝ@)  Epn   n  L$t$$PP1Af  i  qVUUUЃPv   PVjjP+*  u   BX+XZPvvu=Å  S W  ;G   P6Su艚    w      Puփ1ɋ!n   X]  QujV(  1A;Eujȁ   ^_[]Ë Qu  1ЃvjVS)  Vu)  @  USWV    Xu}U$]l  	[  PVjR(  1A(  ^]l  il  D$\$$QP1A   CEPMu81X  ÃSW     MIM   UM1QQ   E~ou6X  ÃSuo  xiOuuUR4uu   tU1FSVjVu")   MURQ  1ɉȃ^_[]Ã1@SPjPu(   uUH
R   USWV       [Mj   EqVUUUЍ@9t"j   zY  Quju(    jjP&  ЃP  M    EXqEQs3P舕:  PW  ;G   EVE  }Qv@]P76S臖     PNj     Q  Puփ1S7jV.%  Vu%  W,k  1@;Mu&   ^_[]Ë f	<  USWV    XM]}u0qQh  ]h  D$t$$QP1At+Qh  V  PQjS(  1@^_[]ËFHM@Eu1QQ   EEHu}ME;HtoM   j0   PuVS׃}uE 7VS   tUHOV   ;uS  1(USWV    XM]}ugqg  g  D$t$$QPukF@$  Ƌ4$D$   E4$PML$$  1*g  `U  D$L$$D$   (  1@^_[]USWV    XuU}[f  	T  PVjR(  )~[f  gf  D$|$$QPt
1G^_[]É]GE@EEÍ{W%  E|BE 6X  ÃSu  xqPu+*  OuVu
  VPuÃV%  E   S/  Puփ1/1GSWj*Wu!#   u%  USWV    XM}}u}qd  e  D$t$<$QPup~w+ƃjVAj WV苭  V<Pu׃1d  MS  PQjW(  1@^_[]USWV    ^}U}$=d   $S  QWjR(  1A@  x=d  Id  L$|$$PP1A  MA   w    PUE؋BEuO}]܉u6X  ǋvX  EW]S  ƋE9   u6膉PVS  9tx]܋EEu{ FS  Pj	j-ju    E؋  u  Puփ1ɉȃ^_[]Ã1@WPj-Pujj-ju    uwUSWV    YuU}i  ^b  b  |$\$$MPP1AU  }CEXMqX  ǃWS
  ك;A#  z1  +QϋMI4uVuEQP   ]܋VUuFH+HBP+PDPX  PSuw}V]   MASppQ?   MSEpQV    E܋   S  Puփ1b   Q  QVjR(  1Aȃ^_[]Ã1FWVj@Vu   ܃Wjj@ju   E<  Pjj@ju   SUSWV       [}u`   E&r`   O  QWju(  1A   Wr`   ~`  L$T$M$PP1A   F   qjAjP  |<N7X  QwP`tfȋ  >  Puփ1ɋ ;EuXȁ   ^_[]ÍO  Pj	jAju   &Pu  n1A2  USWV      YuU^  E}$^   eN  QVjR(  1G9  ~^  ΋^  L$|$$PP1G   @XE@EƍFP  P4|a1  MtX  ƃVS  ;C
  QP  G;|b  @H+H   +EDP  Po؋X;XtnN  ߋ76  _y3SWu  6P;XujDjP   X  ÃS2PWg  ;GtIPWSS"tH  S  Puփ1   SjPg  ShHW   Pu  1GVWj@Wu   )Vjj@jud   1G;Eu
  ^_[].  USWV    XMH   0[  $PƉ\$4$.  FHFFEDHFN$D$      ^_[]USWV    [uF+$vE>FE9t[ً]Z  E荁K  E4    s;srM	D$E$Q$5D$E$T   9}uE^_[]UVP    YEU4    r;rr5Z  J  PQR^]USWV}Gw)W)9    X]|Y  	'K  PQGMQ3PW@} tOQW+OH^_[]UWV}G+GP-VwwW	   ^_]USWV    ^}]UB9s&BY   +JL$I  $PUB9s BY   +JL$I  $P)ω]{;{vUBY   ύaI  $PU1@t<U$s~ǉ|$$;CtUJu1@
<$}1^_[]UWV}G+GEPVwwW^_]USWV    _u^
$};^u$X  4$  @^_[]UEtQ]]USWV}EpX)W+W19uS1@9sLGE3uW<t0@G;Gt"0S~tE;pr1@1^_[]UVPuFuPV<t@FF^]USWV,    _]U9Zw$W   +JL$G  $P}rE ML$D$4$W@}    ME D$4$W<ƅu U싂W   M	L$G  $PFu+w9}tW  	t$D$H  $QE${W   M$  EEpE싀W   $  _؃,^_[]UEHuQP<t@1H]USWV}]C+CO+O$fƋCKt$L$D$$8t(_;_t%$A{D$4$c;Fu݉4$"1^_[]USWVu}G+GN+N9Nȉ$E_;_t~FE;D$E$V<tApEp;pt0|$${t$zD$u4$;FtE;XuuE$fE    E^_[]USWV]U}suO+OM+CȋJ+
)ȃPEPwwWC+CPjuM{9t}0E_> tuSWutMAM1H9]u@0ju   u{u7  E     E    E^_[]USWVuU}G+GJ+JF+)]QEG+GPj VE_GEE    9tFE0E    {> t#uWSu   MAM1H9}uƋ}EƋF+FPju莗^NM9tA0EE{> t PWSu,tYE@EE1I9}uuEp0ju  u u  E30jup  uu  E     1^_[]USWVuF+FjP3  VU1;VtlEE@E111ۋ0uEP<tH9ǃQSCu  UD2M;AuEHE)9uP  1^_[]USWVEXE    ;X   }GEE    36uW<tQxy;yt>7Vrwt,Ut"؋M+AM+yWPRbE}E;XuE^_[]USWVE0@E9t6MEXy0WEP<tH9uuփ^_[]USWV    Xu]K+K<NP   <$PF|$$y#  ؋X1;Xs?v1ɉMEt$$賀ǉt$E$}L8E;XrƉȃ^_[]USWV}EH;Hs`Ep} MEVWP臀ML VQu"MU\|EE;AMrGO E+E^_[]USWV,    X}W)EE    E   N   p]؉\$<$@$։Ɖ\$t$<$ $襂,^_[]UVPuF${1FF^]UWV    ^}E@$G &W  G^_]UVPuF$F$!  F^]USWV       X]N  MM   QRSu   1F   j jP	  uFQ0EP  tJRSquP|    &SjS	  SuQ
  ;Eu
   ^_[]   UWV    ^}M      U  Hx1ɉHH^_]USWV    _]uSuuVJ   tV#1@*]Ct@t	SЃU  Cs1^_[]USWV    X]]L  MUQRuS   1F  EM   UJQ+Q9   }1VPm  9u~oUE2X  ÃSEpƅ   VW     jVW
  MAEUUW	  uuuu  0uj	PujEW  WSD  ^_[]ÃSjSjuVm  W  Vu  1FUSWV    X}]J   MUQRWS   1   MEupωQO} ~sEEuWp0SvÅ   SVZ;FtCMAEEE]u_Wj	Q  WS1  ?;3jWe  Wu  Scou1^_[]UV    XuI   MUQRuV   1uuuVu	   ^]USWV    _]SQ	  E~yuCmI  EE 6X  ǃWu/xPu
  K*1WjPPu   u	  E    E^_[]USWV    X}H  MUQRWu   1Fu~MEuL~D}E0X  MuqPWHt2MAEEE1(Wj	Q  Wuu<  ^_[]USWV    X}H  MUQRWu   1C   uUBp@)9Mup9thZMQUu1PutǅtltQM	P   ;EBH+HJ9u1(WjEP  VuJ  1C؃^_[]USWV    [G  MUQRuu   1k  Mu]܋]E   Q~ƃVEE}    E0X  EMYP}W);G   VQPW   ~uSwu`s   +~MI9 MAEEEM	i   ǋE܍O  GEG 1GG_Qj	S  Su  ?ujujEV  Vu  u1^_[]USWV    XE܋}E   1  G1E܋bE  ]MQMQM4u   O  EH  }M1X  EPߋ]s   MMU?Kh   4j Etu      F}O1@9ǋ]QWAxX19tFAE1E܋bE  EM	h   U4j 0u   tsF9u΋E܋bE     V   Pu׃1@E4j	EV   ujEV;   Vu   1@^_[]UEtMUH@    ]USWV    Xut5}MNC  V  jWV  ^_[]USWV    YEtE}:C  $X  ƅt+EG<$  t$<$D$  ^_[]USWV    _uM      FEB  Ql  HDMu  N1R0  RuE4  RG  PU EV^1PuI  H  H  I4  PuE   tB   V  ^_[]USWV       XMB  UUUA  W  juW  Pu;Eu   ^_[]  USWV       YEoA  UUU1҉UtM[A  $X  ƋE$  t$$D$  UD$$;Eu   ^_[]_  U    XM	}F  ]U    XMI~F  ]USWV    X}x@  $   PƋ    $PNN9sU9r^_[]UMI9sU9r]USWV    X}@  $   PƋ    $PFF^_[]USWV    XMY+?  $   PƋ$PNNM	\$L$$.  ^_[]UVP    YEt^?   $Q@^]^]USWV    Y}uV)9},^)Ë&?  	    RPQÉ^F^_[]UMI9s
UB9r]UWV    Xu};Wr>  	9  VPQ^_]USWV    Xu}];Srt>  	9  WPQ4^_[]UVP    Xu;Nu0>  	9  $Q^]UVP    XuN9u=  	9  $QNA^]UWVu}OG9r"+DPWmOAG1^_]UVP    XuN9u=  	\9  $QNN^]USWV    Y]u}W9s*I=   +M9  RQPOMщU})))9ϋ}U}QW4    7W9s)RVQs  }W}OtMKvu^_[]USWV    [}uUB9s&q<   +
8  QRPUB9s,q<   +
8  ]QuPC)PWV  )){^_[]USWV    ^U]K9ʋ}r ;   7  uQPK9r;   7  WQPM΋^_[]UWVUM1y9s
9u19uRQk1@^_]UE@9sU9t	9r]USWV    XE}uF+)jQ[EN)U썒<  R   RQP  V19u
1@19|F9щu9}
3F9uE^_[]UVP    XMI)      PVQR  ^]USWV    _]sj uE9tG07:  M썏5  M;;HrE WuPE0   9u̓^_[]UME +]UWVEP1H9s}9>t	9r)^_]UWV    Xu}9w9wwy9  	5  $Q)^_]USWV    XuMYNM59  E $   Pǉ}E    $PGGE89t,N9s9u;MtމD$E$͋E^_[]USWV    XuMYNM8  E $   Pǉ}E    $PGGE89t,N9s9u;MuމD$E$N͋E^_[]UVPuMI)v)9uQRP  1^]USWVEM@E19t6EH81ҋ]9ωs9t	9r9;]u׋E+E9^_[]USWVEX19t1Ur:9s9t	9r9u9u11@^_[]USWV    ^]}G;Es$7   M+3  QRPGMU9s'7   +׍3  QRPM9Js%7   +
3  QRPUE)j PQRƃSWV
  ^_[]USWV    Z}E9xt_E+1G?  ^3  uƉu   )7SCujj PQ	   EG?  u;~uÅt	IG?   G?  ^_[]USWV    X}5  $   PƋ<$PF|$$	  >~^_[]USWV    XMY+b5  $   PƋ$PNNM	\$L$$  ^_[]UVP    YEt	5   $Q@^]^]USWV    Y}uV)9},^)Ë4  	    RPQÉ^F^_[]UE;Hs
U;Hr]UWV    Xu};Wrb4  	1  VPQ^_]USWV    Xu}];Sr!4  	|1  WPQ4^_[]UVP    Xu;Nu3  	p1  $Q^]UVP    XuN9u3  	=1  $QNA^]UWVu}OG9r"+DPWoOAG1^_]UVP    XuN9u23  	0  $QNN^]USWV    Y]u}W9s*2   +0  RQPOMщU})))9ϋ}U}QW4    7W9s)RVQ   }W}OtMKvu^_[]USWV    [}uUB9s&2   +
B0  QRPUB9s,2   +
B0  ]QuPC)PWVj  )){^_[]UWVUM1y9s
9u19uRQ1@^_]UE@9sU9t	9r]UVP    XMI)(      PVQR  ^]UME +]UVPuMI)v)9uQRP  1^]USWVEM@E19t6EH81ҋ]9ωs9t	9r9;]u׋E+E9^_[]USWV    ^]}G;Es$D0   M+.  QRPGMU9s'D0   +׍.  QRPM9Js%D0   +
.  QRPUE)j PQRƃSWVL  ^_[]USWV    Z}E9xt_E+1y<  K.  uƉu   )7SCujj PQ   Ey<  u;~uÅt	Iy<   y<  ^_[]Uh]u}    [}E    wt>Ϻt-  G       ED$D$    ED$-  M̉L$<$V
  ME    Eu   A0	EuUԋEEă}tOEEԋMt	M:tM0	wRD$    D$   ED$M̉L$<$V1zD$    D$   ED$M̉L$<$VtPM䉋7  At" 7  A@7  A@7  ǃ7      ǃ7      ǃ7      Ћ]u}UWVS\    [E    Y-  ED$D$    ED$n,  EԉD$E$Rƅ   Et*P{,  ;Eu"EP,  ;EuE䉃I6  v,  Y-  U$l  Y-  D$$    |$ ,  D$t$,  D$ED$,  D$EԉD$,  D$E$   1\[^_%, %0 %4 %8 %< %@ %D %H %L %P %T %X   h %  h    h   h%   h=   hL   hZ   hh   hw   h   h   h   h   |::ral::tuple tuple ::ral::relation relation ::ral::relvar relvar ral ::ral 0.12.2 This software is copyrighted 2004 - 2014 by G. Andrew Mangogna. Terms and conditions for use are distributed with the source code. relationValue tupleVarName ?attr1 type1 expr1 ... attrN typeN exprN? multiplicity specification    8.6 iso8859-1 pkgname version copyright Ral_AttributeDelete: unknown attribute type: %d Ral_AttributeDup: unknown attribute type: %d Ral_AttributeRename: unknown attribute type: %d Ral_AttributeTypeEqual: unknown attribute type: %d Ral_AttributeValueEqual: unknown attribute type: %d Ral_AttributeValueCompare: unknown attribute type: %d Ral_AttributeValueObj: unknown attribute type: %d Ral_AttributeConvertValueToType: unknown attribute type: %d Ral_AttributeHashValue: unknown attribute type: %d Ral_AttributeScanType: unknown attribute type: %d Ral_AttributeScanValue: unknown attribute type: %d Ral_AttributeConvertValue: unknown attribute type: %d Ral_AttributeTypeScanFlagsFree: unknown flags type: %d Ral_AttributeValueScanFlagsFree: unknown flags type: %d bignum boolean bytearray dict double int list long string wideInt bad boolean value, "%s" booleanHash: cannot convert, "%s" doubleHash: cannot convert, "%s" intHash: cannot convert, "%s" longHash: cannot convert, "%s" wideIntHash: cannot convert, "%s" Ral_RelationUpdate: attempt to update non-existant tuple while ungrouping relation while computing quotient while unwrapping tuple Ral_RelationErase: first iterator out of bounds Ral_RelationErase: last iterator out of bounds Ral_RelationErase: first iterator greater than last  !=  array assign attributes body cardinality compose create degree divide dunion eliminate emptyof extend extract foreach fromdict fromlist group heading insert intersect is isempty isnotempty issametype join minus project rank rename restrict restrictwith semijoin semiminus summarize summarizeby table tag tclose times tuple uinsert ungroup union unwrap update wrap subcommand ?arg? ... subcommand relation arrayName keyAttr valueAttr relationValue ?attrName | attr-var-list ...? relationValue relation1 relation2 ?-using joinAttrs? heading ?tuple1 tuple2 ...? relation keyAttr valueAttr dividend divisor mediator relation1 relation2 ?relation3 ...? relationValue ?attr ... ? attribute / type / expression arguments must be given in triples relationValue attrName ?attrName2 ...? tupleVarName relationValue ?-ascending | -descending? ?attr-list?script ordering 
    ("::ral::relation foreach" body line %d) -ascending -descending dictValue keyattr keytype valueattr valuetype list attribute type relation newattribute ?attr1 attr2 ...? attempt to group all attributes in the relation during group operation relationValue ?name-value-list ...? equal == notequal != propersubsetof < propersupersetof > subsetof <= supersetof >= relation1 compareop relation2 compareop relation1 relation2 relation1 relation2 ?-using joinAttrs relation3 ... ? relationValue ?attrName ? ?-ascending | -descending? sortAttrList? ? relationValue ?-ascending | -descending? rankAttr newAttr relationValue ?oldname newname ... ? oldname / newname arguments must be given in pairs relValue tupleVarName expr relValue expr relationValue perRelation relationVarName ?attr1 type1 expr1 ... attrN typeN exprN? heading ?value-list1 value-list2 ...? relation attrName ?-ascending | -descending sort-attr-list? ?-within attr-list -start tag-value? -within -start tag option Ral_TagCmd: unknown option, "%d" relation relation attribute relationValue attribute relationValue tupleVarName expr script 
    ("in ::ral::relation update" body line %d) invoked "break" outside of a loop invoked "continue" outside of a loop unknown return code %d relationValue newAttr ?attr attr2 ...? -using Ral_ConstraintDelete: unknown constraint type, %d unknown constraint type, %d relvarConstraintCleanup: unknown constraint type, %d is referenced by multiple tuples is not referenced by any tuple references multiple tuples references no tuple for association  (  [ ] ==> [ ]  ) 1 + ? * , in relvar  
 tuple    is referred to by multiple tuples is not referred to by any tuple for partition   is partitioned [  |  ]) correlation   does not form a complete correlation for correlation   <== [  (Complete) ] ==>  association constraint correlation delete deleteone eval exists identifiers names partition path procedural restrictone set trace transaction unset updateone updateper name refrngRelvar refrngAttrList refToSpec refToRelvar refToAttrList refrngSpec info member delete | info | names ?args? | member <relvar> | path <name> constraint subcommand name ?pattern? relvarName Unknown association command type, %d ?-complete? name corrRelvar corr-attr-list1 relvar1 attr-list1 spec1 corr-attr-list2 relvar2 attr-list2 spec2 relvarName heading id1 ?id2 id3 ...? relvarName tupleVarName expr relvarName relvarName ?attrName1 value1 attrName2 value2 ...? relvarName ?relationValue ...? arg ?arg ...? 
    ("in ::ral::relvar eval" body line %d) relvarName ?name-value-list ...? relvarName relationValue name super super-attrs sub1 sub1-attrs ?sub2 sub2-attrs sub3 sub3-attrs ...? name relvarName1 ?relvarName2 ...? script relvarValue attr value ?attr2 value 2 ...? attribute / value arguments must be given in pairs relvar ?relationValue? add remove suspend variable option type ?arg arg ...? trace option trace type add variable relvarName ops cmdPrefix remove variable relvarName ops cmdPrefix info variable relvarName suspend variable relvarName script Unknown trace option, %d add transaction cmdPrefix remove transaction cmdPrefix info transaction suspending eval traces not implemented Unknown trace type, %d begin end rollback transaction option Unknown transaction option, %d ?relvar1 relvar2 ...? relvarName tupleVarName expr script relvarName tupleVarName idValueList script relvar creation not allowed during a transaction :: during identifier construction operation 
    ("in ::ral::%s %s" body line %d) -complete Ral_RelvarObjConstraintInfo: unknown constraint type, %d end transaction with no beginning 
    ("in ::ral::relvar trace suspend variable" body line %d)  relvar may only be modified using "::ral::relvar" command relvarTraceProc: trace on non-write, flags = %#x
 bad operation list: must be one or more of delete, insert, update, set or unset traceOp namespace eval   { } procedural contraint, " ", failed returned "continue" from procedural contraint script for constraint, " " returned "break" from procedural contraint script for constraint, " Ral_TupleUpdate: attempt to update a shared tuple get tupleValue ?attrName | attr-var-pair ... ? tupleValue heading name-value-list tupleValue ?attr? ... tuple1 tuple2 tupleValue ?name type value ... ? tupleValue attr ?...? attr1 type1 value1 ... tupleValue ?oldname newname ... ? for oldname / newname arguments tupleValue tupleAttribute tupleValue ?attr1 value1 attr2 value2? for attribute name / attribute value arguments tupleValue newAttr ?attr attr2 ...? failed to copy attribute, "%s" Ral_TupleHeadingAppend: out of bounds access at source offset "%d" Ral_TupleHeadingAppend: overflow of destination Ral_TupleHeadingFetch: out of bounds access at offset "%d" Ral_TupleHeadingStore: out of bounds access at offset "%d" Ral_TupleHeadingStore: cannot find attribute name, "%s", in hash table Ral_TupleHeadingStore: inconsistent index, expected %u, got %u Ral_TupleHeadingPushBack: overflow , " RAL no error unknown attribute name duplicate attribute name bad heading format bad value format bad value type for value unknown data type bad type keyword wrong number of attributes specified bad list of pairs duplicate command option relations of non-zero degree must have at least one identifier identifiers must have at least one attribute identifiers must not be subsets of other identifiers tuple has duplicate values for an identifier duplicate attribute name in identifier attribute set duplicate tuple headings not equal relation must have degree of one relation must have degree of two relation must have cardinality of one bad list of triples attributes do not constitute an identifier attribute must be of a Relation type attribute must be of a Tuple type relation is not a projection of the summarized relation divisor heading must be disjoint from the dividend heading mediator heading must be a union of the dividend and divisor headings too many attributes specified attributes must have the same type only a single identifier may be specified identifier must have only a single attribute "-within" option attributes are not the subset of any identifier attribute is not a valid type for rank operation duplicate relvar name unknown relvar name mismatch between referential attributes duplicate constraint name unknown constraint name relvar has constraints in place referred to identifiers can not have non-singular multiplicities operation is not allowed during "eval" command a super set relvar may not be named as one of its own sub sets correlation spec is not available for a "-complete" correlation recursively invoking a relvar command outside of a transaction recursive attempt to modify a relvar already being changed serious internal error unknown command relvar setfromany updatefromobj NONE destroy OK UNKNOWN_ATTR DUPLICATE_ATTR HEADING_ERR FORMAT_ERR BAD_VALUE BAD_TYPE BAD_KEYWORD WRONG_NUM_ATTRS BAD_PAIRS_LIST DUPLICATE_OPTION NO_IDENTIFIER IDENTIFIER_FORMAT IDENTIFIER_SUBSET IDENTITY_CONSTRAINT DUP_ATTR_IN_ID DUPLICATE_TUPLE HEADING_NOT_EQUAL DEGREE_ONE DEGREE_TWO CARDINALITY_ONE BAD_TRIPLE_LIST NOT_AN_IDENTIFIER NOT_A_RELATION NOT_A_TUPLE NOT_A_PROJECTION NOT_DISJOINT NOT_UNION TOO_MANY_ATTRS TYPE_MISMATCH SINGLE_IDENTIFIER SINGLE_ATTRIBUTE WITHIN_NOT_SUBSET BAD_RANK_TYPE DUP_NAME UNKNOWN_NAME REFATTR_MISMATCH DUP_CONSTRAINT UNKNOWN_CONSTRAINT CONSTRAINTS_PRESENT BAD_MULT BAD_TRANS_OP SUPER_NAME INCOMPLETE_SPEC ONGOING_CMD ONGOING_MODIFICATION INTERNAL_ERROR Ral_IntVectorFetch: out of bounds access at offset "%d" Ral_IntVectorStore: out of bounds access at offset "%d" Ral_IntVectorFront: access to empty vector Ral_IntVectorPopBack: access to empty vector Ral_IntVectorInsert: out of bounds access at offset "%d" Ral_IntVectorErase: out of bounds access at offset "%d" Ral_IntVectorOffsetOf: out of bounds access Ral_IntVectorCopy: out of bounds access at source offset "%d" Ral_IntVectorCopy: out of bounds access at dest offset "%d" %d: %d
 Ral_PtrVectorFetch: out of bounds access at offset "%d" Ral_PtrVectorStore: out of bounds access at offset "%d" Ral_PtrVectorFront: access to empty vector Ral_PtrVectorPopBack: access to empty vector Ral_PtrVectorInsert: out of bounds access at offset "%d" Ral_PtrVectorErase: out of bounds access at offset "%d" Ral_PtrVectorCopy: out of bounds access at source offset "%d" Ral_PtrVectorCopy: out of bounds access at dest offset "%d" %d: %X
   interpreter uses an incompatible stubs mechanism Tcl tcl::tommath epoch number mismatch requires a later revision missing stub table pointer ):  , actual version   (requested version  Error loading                             4   4        4                          `     z @            " , 6 @ J T ^ h r  %  #&  &  Z'   '  c(  (   )   )  )  '*  *   *  *  B+  +   +  +  \,  ,   E-  q-  -  ..   .  .  /  `/   /  /  -0  0             0  $1  1  
2  5  5   a   ~b   c   d   e   Ff   Ki    j   )k   l   m   p   (q  ' q  . v  6 x  > |  G -  P   V   ^ Ƀ  e   o :  r r  z                /         ז      *            Ů   x   Ĵ      =   	   R     !   (   /           Q     \            = X  C X  F ;Y  O ;Y  R Y  a Y  c %Z  t %Z  v bY   bY   Y   Y          Q     \                    D F H J , c  8   C $   z  O L  V K     ` 6  e s  l   ^   e @  \ x  ~ Q
  
  K    
    |    c   (     =#         D F H J O    ^            (             {  {  $}   .~  ~ = n ' ? .  G U 2  V       ŋ !  (  / 3                  D V o    = r       < a    < Z }    F \ p     + Z    S j   z      ,    8 C    O V       = ` ' . 6 > G 2 P V l ^ e o r z     x ~                        ! (   /            , : L ^ r            $ . = K ] n             ( =               Relation Tuple        6  6  7  V7        ^  ^  _  5_        6  6  
`  Z`       "  I  O e $ x )      / 3 $ :     B             D         F        H        J                      ݡ                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  !WA`Bpp`Bp`:BUAp`"B`HTBTB`ATARASApU ppQ  @___stack_chk_guard Qq$@dyld_stub_binder  q,@___bzero  q0@___snprintf_chk  q4@___stack_chk_fail  q8@_bsearch  q<@_memcmp  q@@_memcpy  qD@_memmove  qH@_qsort  qL@_strcmp  qP@_strcpy  qT@_strlen  qX@_strrchr      _  Ral_ r tuple  
In xSafe Unload Attribute JoinMap Rel Constraint Tuple ErrorInfo  PtrVector % it t   "  Init Unload  (  (  (  	New D Rename Equal T Value Convert HashValue Scan  T RelationType FromObjs  clType upleType  (  )  *  elete up  +  +  ,  -  ype oString  Equal ScanFlagsFree  .  Equal Compare Obj ScanFlagsFree  .  3  5  7  Value Name Type  ToType   :  ?  Name Type Value  B  B  B  D  E  G  H  I  J  New Delete A Tuple GetAttr FindAttr SortAttr MatchingTupleSet  d  e  ttr dd  Reserve Map  f  Reserve Map Counts  g  AttrMapping TupleMapping  g  h  i  i  j  k  k  l  l  ation var 
 N D Re P S U In 	Find 	Minus 	T 	Group 	Join 
Co 
E 
ValueStringOf Obj  ew otEqual  o  up elete i  p  serve nameAttribute  q  ushBack ro 	 r  hallowCopy emi 
ort 
u can tringOf  s  t  pdate n  tTupleObj 
 ion group 	wrap 
 vCopy  z  sjointCopy 	vide 
 z  tersect 	sertTuple 
 {  |JoinTuples 
 }  imes 	ag 
close 
 ~  ject 	perSu            mp 
nvert  ose 
are    Join 
Minus 
     Ǘ  Within 
         xtract rase qual  ϭ          bsetOf persetOf    bsetOf persetOf          ׵Value    Value      Ǽ  el al_ ' ationCmd varCmd    New 
Convert 
ParseJoinArgs 
Type '     Value 
Obj 
     ޔ    New 
D Find S I Tra RestorePrev Obj  Info Transaction  e iscardPrev  lete clConstraintEval  Info T  Identifier ById      tart etRelation  Transaction Command      sTransOnGoing nsertTuple d    ɤ  nsModifiedRelvar ce    ransaction uple    AssocCreate New P CorrelationCreate Delete FindByName  Ҧ  Association P Correlation    artitionCreate roceduralCreate    artition rocedural  Ǩ          ByName               IndexTuple UnindexTuple                Add Remove        	New Delete Find E C InsertTuple KeyTuple UpdateTuple Trace      Relvar Constraint    xec nd  U InsertTraces DeleteTraces SetTraces EvalTraces  nsetTraces pdateTraces    o reate  pyOnShared nstraint            Update Var Eval      Assoc P Correlation    artition rocedural  ϊ      Delete Info Names Member Path  ɭ            Trans Cmd      Add Remove Info Suspend          Add Remove Info              New S E Co D U H A GetAttrValue ValueStringOf Obj PartialSetFromObj     ubset can tringOf etFrom    xtend qual    py nvert  Values     elete up    nreference pdate    Values    ash eading  Attr    ttrEqual ssignToVars     AttrValue FromObj        Shallow Ordered      Value    Value        Cmd AttrHashType '   New S F PushBack D A E Un In Join Co MapIndices  ҩOrderMap FromObj  ubset t can    etch ind  ѫ    up elete    ppend ttrsFrom    xtend qual    ޯ  reference ion    ǰ  Ǳ  ore ringOf    dexOf tersect      ȵ    m nvert  pose monAttributes                New Convert Type (     Obj ValueList      Obj Vect              Set  Get ! Cmd  Error     Obj     erp  Vector ! SetError  ErrorInfo     Obj !   Command !Option !     New !F !D "Reserve "S "B #P #In #E #OffsetOf $Minus $Co $ Empty " i "etch "ront # ll "nd $ Consecutive "   up "elete "           tore #et $ort $ubsetOf $     ack #ooleanMap $   ushBack #opBack #rint %     sert #dexOf $tersect $   rase $xchange $qual $     Add $Complement $                       ntainsAny %py %       
New %D %Reserve %F %S &Back &P &Insert 'E 'Copy '   up %elete %       i &etch &ront & ll &nd '     tore &etAdd 'ort 'ubsetOf '       ushBack &opBack &rint '       rase 'qual '                 relationTypeName 'tupleTypeName '             "	bbVpp=mOG@90||Oea\o,l`@,OG@,gwr,a\o-OG@,a\o,i}rhMWYsROQC1R;q#@-YXdLVo|zxubA'AAAAq;s2+P>

	!2'4A)P6,+@u@s@@{O)qvViN:,BvQVn[k	{1PUU2??CDRM<n:yJMr0y^<G<thAy58K0*Vo!4'6yMScZJ]4W=H35C87#21C=lJQ]4W=H35C87#2=l                                 %      %      #&      &  $    Z'  0    '  <    c(  J    (  Z     )  g    )  u    )      '*      *      *      *      B+      +      +      +      \,      ,      E-  
    q-      -  #    ..  ,    .  5    .  @    /  M    `/  W    /  `    /  k    -0  x    0      0      $1      1      
2      5      5      6      6       7      V7  1    \<  C    C  ]    ZV  k    ^  |    ^      _      5_      `_      
`      Z`      a      ~b  -    c  D    d  U    e  m    Ff      Ki      j      )k      l      m      p      (q      q      v  /    x  C    |  X    -  m                Ƀ            :      r              
            /      A    /  U      f      y    ז            *                          Ů       x  "     Ĵ  5       G     =  Y     	  m     R                                                 "  !    I  '!      6!      O!      `!      p!    Y  !      !    X  !      !    <  !      #"    2  H"    c  ^"      s"    $  "    z  "    L  "    K  "      "    6  "    s  "      #      #    @ +#    \ ;#     K#    Q
 _#    
 n#    K #     #    
 #     #    | #     #    c #     #     $     $$    =# 8$    / K$    j0 \$    5 t$    A $    ` $    e $    g $    j $    { %    {  %    $} 0%    .~ @%    ~ S%    n b%    ? r%     %    U %     %     %     %     %    ŋ %     %     &    3 &     *&     ?&    ݡ T&     e&    J v&     &     &     &    | &     &     &     '     #'     7'     D'     Q'     `'    Q y'     '  	  \ '  	  $ '  	  , '  	   '  	   '  	  < (  	  d (  	  t ((  	  T I(  	  d S(  	   f(  	  $ u(  	   (  	   (  	   (  
   (  
   (  
  ( (  
  T (  
  p )  
   *)  
   S)  
   ^)  
   k)  
   |)  
   )  
   )  
    )     )     )          F!        "  6     #  Q     5  r                                        ,       9        1    T  F    
!  ]    v!  t    "      %            #$            n        &    $  G    R  b                :                        G  '      D      c          [     r     W     u     Ȭ            	     ˵ 0     J     ]     r    K               '     Q          3     ݴ     ֱ /    Y E     Z    Ͷ r    e                              >     ܰ      2      N     a     u    :               +     3      T4      4      13  0    2  C    F5  X    4  l    g6      |2      5      ,6      5      3      \           /     A     U     i     }              )          O     Լ      	     	    1 5	    z J	    Ծ ]	     q	    R 	    nX  	    
I  	    >\  	    \  	    :  	    _=  
    K  #
    
8  4
    X  G
    1W  Z
    V  o
    \>  
    H  
    A  
      
    7  
    =      F      >  &    7  7    ;Y  M    o  e    ;  y        
  @     @      Y      %Z      9      fZ  .    8  C    Z  U    [  l    J      2K      9      U      ]      bY      Y  
    M  
    O  0
    R  D
    r?  W
    D  l
    r;  
    =  
    P  
    ~:  
    ^  
    G^  
            %      ;    '  X      o    e                  t                  ]  
    2  '      6      I      c    V     X     tb     a     Uc     +4     ; .    oK N    OE l    T     ?1     Je     c     {j     .o     6 '    m C    3 a    Y;     W     2     5     7     +     h     i 0    :i N    a9 h    |e     f     f     hh     8     +  	    l       I  8      T      h                "      2           ~t     >x     x     u      r 4    r E    qv S    w h    v }    4s     ns     q     u     s     1t          H #     B     d     }         ޗ          G          ѕ 	    ǘ      8    Ț S     i         Ҕ     +     7                     4    : L    A c    )     Eq     ,        
            Rw     w           5    |y H    q Y    s o    t     >     z     +    
     
       `          
       z #        ,        <        N        a        j        r        z                                                                                      @                                                     _Ral_AttributeConvertName _Ral_AttributeConvertType _Ral_AttributeConvertValue _Ral_AttributeConvertValueToType _Ral_AttributeDelete _Ral_AttributeDup _Ral_AttributeEqual _Ral_AttributeHashValue _Ral_AttributeNewFromObjs _Ral_AttributeNewRelationType _Ral_AttributeNewTclType _Ral_AttributeNewTupleType _Ral_AttributeRename _Ral_AttributeScanName _Ral_AttributeScanType _Ral_AttributeScanValue _Ral_AttributeToString _Ral_AttributeTypeEqual _Ral_AttributeTypeScanFlagsFree _Ral_AttributeValueCompare _Ral_AttributeValueEqual _Ral_AttributeValueObj _Ral_AttributeValueScanFlagsFree _Ral_ConstraintAssocCreate _Ral_ConstraintCorrelationCreate _Ral_ConstraintDelete _Ral_ConstraintDeleteByName _Ral_ConstraintFindByName _Ral_ConstraintNewAssociation _Ral_ConstraintNewCorrelation _Ral_ConstraintNewPartition _Ral_ConstraintNewProcedural _Ral_ConstraintPartitionCreate _Ral_ConstraintProceduralCreate _Ral_ErrorInfoGetCommand _Ral_ErrorInfoGetOption _Ral_ErrorInfoSetCmd _Ral_ErrorInfoSetError _Ral_ErrorInfoSetErrorObj _Ral_Init _Ral_IntVectorBack _Ral_IntVectorBooleanMap _Ral_IntVectorContainsAny _Ral_IntVectorCopy _Ral_IntVectorDelete _Ral_IntVectorDup _Ral_IntVectorEqual _Ral_IntVectorErase _Ral_IntVectorExchange _Ral_IntVectorFetch _Ral_IntVectorFill _Ral_IntVectorFillConsecutive _Ral_IntVectorFind _Ral_IntVectorFront _Ral_IntVectorIndexOf _Ral_IntVectorInsert _Ral_IntVectorIntersect _Ral_IntVectorMinus _Ral_IntVectorNew _Ral_IntVectorNewEmpty _Ral_IntVectorOffsetOf _Ral_IntVectorPopBack _Ral_IntVectorPrint _Ral_IntVectorPushBack _Ral_IntVectorReserve _Ral_IntVectorSetAdd _Ral_IntVectorSetComplement _Ral_IntVectorSort _Ral_IntVectorStore _Ral_IntVectorSubsetOf _Ral_InterpErrorInfo _Ral_InterpErrorInfoObj _Ral_InterpSetError _Ral_JoinMapAddAttrMapping _Ral_JoinMapAddTupleMapping _Ral_JoinMapAttrMap _Ral_JoinMapAttrReserve _Ral_JoinMapDelete _Ral_JoinMapFindAttr _Ral_JoinMapGetAttr _Ral_JoinMapMatchingTupleSet _Ral_JoinMapNew _Ral_JoinMapSortAttr _Ral_JoinMapTupleCounts _Ral_JoinMapTupleMap _Ral_JoinMapTupleReserve _Ral_PtrVectorBack _Ral_PtrVectorCopy _Ral_PtrVectorDelete _Ral_PtrVectorDup _Ral_PtrVectorEqual _Ral_PtrVectorErase _Ral_PtrVectorFetch _Ral_PtrVectorFill _Ral_PtrVectorFind _Ral_PtrVectorFront _Ral_PtrVectorInsert _Ral_PtrVectorNew _Ral_PtrVectorPopBack _Ral_PtrVectorPrint _Ral_PtrVectorPushBack _Ral_PtrVectorReserve _Ral_PtrVectorSetAdd _Ral_PtrVectorSort _Ral_PtrVectorStore _Ral_PtrVectorSubsetOf _Ral_RelationCompare _Ral_RelationCompose _Ral_RelationConvert _Ral_RelationConvertValue _Ral_RelationDelete _Ral_RelationDisjointCopy _Ral_RelationDivide _Ral_RelationDup _Ral_RelationEqual _Ral_RelationErase _Ral_RelationExtract _Ral_RelationFind _Ral_RelationFindJoinTuples _Ral_RelationGroup _Ral_RelationInsertTupleObj _Ral_RelationInsertTupleValue _Ral_RelationIntersect _Ral_RelationJoin _Ral_RelationMinus _Ral_RelationNew _Ral_RelationNotEqual _Ral_RelationObjConvert _Ral_RelationObjNew _Ral_RelationObjParseJoinArgs _Ral_RelationObjType _Ral_RelationProject _Ral_RelationProperSubsetOf _Ral_RelationProperSupersetOf _Ral_RelationPushBack _Ral_RelationRenameAttribute _Ral_RelationReserve _Ral_RelationScan _Ral_RelationScanValue _Ral_RelationSemiJoin _Ral_RelationSemiMinus _Ral_RelationShallowCopy _Ral_RelationSort _Ral_RelationStringOf _Ral_RelationSubsetOf _Ral_RelationSupersetOf _Ral_RelationTag _Ral_RelationTagWithin _Ral_RelationTclose _Ral_RelationTimes _Ral_RelationUngroup _Ral_RelationUnion _Ral_RelationUnionCopy _Ral_RelationUnwrap _Ral_RelationUpdate _Ral_RelationUpdateTupleObj _Ral_RelationValueStringOf _Ral_RelvarDeclConstraintEval _Ral_RelvarDelete _Ral_RelvarDeleteInfo _Ral_RelvarDeleteTransaction _Ral_RelvarDeleteTuple _Ral_RelvarDiscardPrev _Ral_RelvarFind _Ral_RelvarFindById _Ral_RelvarFindIdentifier _Ral_RelvarIdIndexTuple _Ral_RelvarIdUnindexTuple _Ral_RelvarInsertTuple _Ral_RelvarIsTransOnGoing _Ral_RelvarNew _Ral_RelvarNewInfo _Ral_RelvarNewTransaction _Ral_RelvarObjConstraintDelete _Ral_RelvarObjConstraintInfo _Ral_RelvarObjConstraintMember _Ral_RelvarObjConstraintNames _Ral_RelvarObjConstraintPath _Ral_RelvarObjCopyOnShared _Ral_RelvarObjCreateAssoc _Ral_RelvarObjCreateCorrelation _Ral_RelvarObjCreatePartition _Ral_RelvarObjCreateProcedural _Ral_RelvarObjDelete _Ral_RelvarObjEndCmd _Ral_RelvarObjEndTrans _Ral_RelvarObjExecDeleteTraces _Ral_RelvarObjExecEvalTraces _Ral_RelvarObjExecInsertTraces _Ral_RelvarObjExecSetTraces _Ral_RelvarObjExecUnsetTraces _Ral_RelvarObjExecUpdateTraces _Ral_RelvarObjFindConstraint _Ral_RelvarObjFindRelvar _Ral_RelvarObjInsertTuple _Ral_RelvarObjKeyTuple _Ral_RelvarObjNew _Ral_RelvarObjTraceEvalAdd _Ral_RelvarObjTraceEvalInfo _Ral_RelvarObjTraceEvalRemove _Ral_RelvarObjTraceUpdate _Ral_RelvarObjTraceVarAdd _Ral_RelvarObjTraceVarInfo _Ral_RelvarObjTraceVarRemove _Ral_RelvarObjTraceVarSuspend _Ral_RelvarObjUpdateTuple _Ral_RelvarRestorePrev _Ral_RelvarSetRelation _Ral_RelvarStartCommand _Ral_RelvarStartTransaction _Ral_RelvarTraceAdd _Ral_RelvarTraceRemove _Ral_RelvarTransModifiedRelvar _Ral_SafeInit _Ral_SafeUnload _Ral_TupleAssignToVars _Ral_TupleAttrEqual _Ral_TupleConvert _Ral_TupleConvertValue _Ral_TupleCopy _Ral_TupleCopyValues _Ral_TupleDelete _Ral_TupleDup _Ral_TupleDupOrdered _Ral_TupleDupShallow _Ral_TupleEqual _Ral_TupleEqualValues _Ral_TupleExtend _Ral_TupleGetAttrValue _Ral_TupleHash _Ral_TupleHashAttr _Ral_TupleHeadingAppend _Ral_TupleHeadingAttrsFromObj _Ral_TupleHeadingAttrsFromVect _Ral_TupleHeadingCommonAttributes _Ral_TupleHeadingCompose _Ral_TupleHeadingConvert _Ral_TupleHeadingDelete _Ral_TupleHeadingDup _Ral_TupleHeadingEqual _Ral_TupleHeadingExtend _Ral_TupleHeadingFetch _Ral_TupleHeadingFind _Ral_TupleHeadingIndexOf _Ral_TupleHeadingIntersect _Ral_TupleHeadingJoin _Ral_TupleHeadingMapIndices _Ral_TupleHeadingNew _Ral_TupleHeadingNewFromObj _Ral_TupleHeadingNewOrderMap _Ral_TupleHeadingPushBack _Ral_TupleHeadingScan _Ral_TupleHeadingStore _Ral_TupleHeadingStringOf _Ral_TupleHeadingSubset _Ral_TupleHeadingUnion _Ral_TupleHeadingUnreference _Ral_TupleNew _Ral_TupleObjConvert _Ral_TupleObjNew _Ral_TupleObjType _Ral_TuplePartialSetFromObj _Ral_TupleScan _Ral_TupleScanValue _Ral_TupleSetFromObj _Ral_TupleSetFromValueList _Ral_TupleStringOf _Ral_TupleSubset _Ral_TupleUnreference _Ral_TupleUpdateAttrValue _Ral_TupleUpdateFromObj _Ral_TupleValueStringOf _Ral_Unload _ral_relationTypeName _ral_tupleTypeName _relationCmd _relvarCmd _tupleAttrHashType _tupleCmd ___bzero ___snprintf_chk ___stack_chk_fail ___stack_chk_guard _bsearch _memcmp _memcpy _memmove _qsort _strcmp _strcpy _strlen _strrchr dyld_stub_binder _stringEqual _stringCompare _isAString _stringHash _cmpTypeNames _isABignum _bignumEqual _bignumCompare _bignumHash _isABoolean _booleanEqual _booleanCompare _booleanHash _isAByteArray _byteArrayEqual _byteArrayCompare _byteArrayHash _isADict _dictEqual _dictCompare _dictHash _isADouble _doubleEqual _doubleCompare _doubleHash _isAnInt _intEqual _intCompare _intHash _isAList _listEqual _listCompare _listHash _isALong _longEqual _longCompare _longHash _isAWideInt _wideIntEqual _wideIntCompare _wideIntHash _attr_0_cmp _attr_1_cmp _tupleAttrHashGenKey _tupleAttrHashCompareKeys _tupleAttrHashEntryAlloc _tupleAttrHashEntryFree _Ral_HeadingMatch _Ral_RelationIndexByAttrs _Ral_DownHeap _tupleHashGenKey _tupleHashCompareKeys _tupleHashEntryAlloc _tupleHashEntryFree _Ral_TupleCompare _tupleAttrHashMultiEntryAlloc _tupleAttrHashMultiEntryFree _RelationArrayCmd _RelationAssignCmd _RelationAttributesCmd _RelationBodyCmd _RelationCardinalityCmd _RelationComposeCmd _RelationCreateCmd _RelationDegreeCmd _RelationDictCmd _RelationDivideCmd _RelationDunionCmd _RelationEliminateCmd _RelationEmptyofCmd _RelationExtendCmd _RelationExtractCmd _RelationForeachCmd _RelationFromdictCmd _RelationFromlistCmd _RelationGroupCmd _RelationHeadingCmd _RelationInsertCmd _RelationIntersectCmd _RelationIsCmd _RelationIsemptyCmd _RelationIsnotemptyCmd _RelationIssametypeCmd _RelationJoinCmd _RelationListCmd _RelationMinusCmd _RelationProjectCmd _RelationRankCmd _RelationRenameCmd _RelationRestrictCmd _RelationRestrictWithCmd _RelationSemijoinCmd _RelationSemiminusCmd _RelationSummarizeCmd _RelationSummarizebyCmd _RelationTableCmd _RelationTagCmd _RelationTcloseCmd _RelationTimesCmd _RelationTupleCmd _RelationUinsertCmd _RelationUngroupCmd _RelationUnionCmd _RelationUnwrapCmd _RelationUpdateCmd _RelationWrapCmd _FreeRelationInternalRep _DupRelationInternalRep _UpdateStringOfRelation _SetRelationFromAny _relvarCleanup _relvarConstraintCleanup _relvarSetIntRep _relvarIndexIds _relvarFindJoinTuples _relvarEvalAssocTupleCounts _relvarAssocConstraintErrorMsg _relvarConstraintErrorMsg _relvarPartitionConstraintErrorMsg _relvarCorrelationConstraintErrorMsg _relvarCorrelationConstraintToString _RelvarAssociationCmd _RelvarConstraintCmd _RelvarCorrelationCmd _RelvarCreateCmd _RelvarDeleteCmd _RelvarDeleteOneCmd _RelvarDunionCmd _RelvarEvalCmd _RelvarExistsCmd _RelvarIdentifiersCmd _RelvarInsertCmd _RelvarIntersectCmd _RelvarMinusCmd _RelvarNamesCmd _RelvarPartitionCmd _RelvarPathCmd _RelvarProceduralCmd _RelvarRestrictOneCmd _RelvarSetCmd _RelvarTraceCmd _RelvarTransactionCmd _RelvarUinsertCmd _RelvarUnionCmd _RelvarUnsetCmd _RelvarUpdateCmd _RelvarUpdateOneCmd _RelvarUpdatePerCmd _relvarResolveName _relvarTraceProc _relvarGetNamespaceName _relvarObjConstraintEval _relvarConstraintAttrNames _Ral_RelvarObjDecodeTraceOps _Ral_RelvarObjEncodeTraceFlag _Ral_RelvarObjExecTraces _TupleAssignCmd _TupleAttributesCmd _TupleCreateCmd _TupleDegreeCmd _TupleEliminateCmd _TupleEqualCmd _TupleExtendCmd _TupleExtractCmd _TupleFromListCmd _TupleGetCmd _TupleHeadingCmd _TupleProjectCmd _TupleRelationCmd _TupleRenameCmd _TupleUnwrapCmd _TupleUpdateCmd _TupleWrapCmd _FreeTupleInternalRep _DupTupleInternalRep _UpdateStringOfTuple _SetTupleFromAny _int_ind_compare _ptr_ind_compare _Tcl_InitStubs _TclTomMathInitializeStubs _Ral_Init.tupleCmdName _Ral_Init.tupleStr _Ral_Init.relationCmdName _Ral_Init.relationStr _Ral_Init.relvarCmdName _Ral_Init.relvarStr _ral_pkgname _ral_version _ral_copyright _RelationExtendCmd.usage _specErrMsg _Ral_Types _Ral_JoinMapSortAttr.cmpFuncs _relationCmd.cmdTable _orderOptions _RelationIsCmd.cmdTable _RelationTagCmd.optTable _condMultStrings _relvarCmd.cmdTable _relvarAssocSpec.condMultStrings _opsTable _tupleCmd.cmdTable _resultStrings _cmdStrings _optStrings _errorStrings _ral_config _tupleHashType _tupleAttrHashMultiType _RelvarConstraintCmd.constraintCmds _RelvarTraceCmd.traceOptions _RelvarTraceCmd.traceTypes _RelvarTransactionCmd.transactionOptions _specTable _tclStubsPtr _tclPlatStubsPtr _tclIntStubsPtr _tclIntPlatStubsPtr _tclTomMathStubsPtr _RelationTagCmd.gotOpt _Ral_IntVectorPrint.buf _Ral_PtrVectorPrint.buf                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  
   @           x  __TEXT                                                     __text          __TEXT                 O                                 __stubs         __TEXT          p     H       p                        __stub_helper   __TEXT                                                __const         __TEXT          @     [      @                            __cstring       __TEXT               *                                 __unwind_info   __TEXT               D                                  __eh_frame      __TEXT                                                    (  __DATA                                                     __got           __DATA                                                    __nl_symbol_ptr __DATA                                                  __la_symbol_ptr __DATA          (      `       (                           __const         __DATA                                                   __data          __DATA               @                                  __bss           __DATA                                                      H   __LINKEDIT              p             b                    
   0             libral0.12.2.dylib      "  0      H   H  0           x     8!         8   X  *     P                                            PX                          h̥3>V=k$       
  
 *                 8              /usr/lib/libSystem.B.dylib      &      P5   )      8                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     UHAWAVAUATSPIH5h 1褲 Hh  H5Q 11L HL  L=- IH5 11L  II  IIH5 H
j 1E1L  IH 1LL    IL  H=l   HH H H=@   LLHHAL=w m  IH5- HO  1E1L  IH 1LL  .  IL  H=   HH
 H H=ȴ   LLHHAL=    H=ƴ LM  MH5 H  E1LHA  IH 1LL     IL  H=e   HHa H H=8   LLHHAL=6 u0IH5" 11LL  ILHL  t0AHAIL      H[A\A]A^A_]IL5 H H
ƴ LL  IH 1LLPUH]UH1]UH1]UHAWAVSH(IHuH5Z L  H}
   (   2 I1MtCH7 H HX(L< x!HH{ L" HIHCC    HC    HH([A^A_]UHAWAVSPIIH H HX(ޱ x!HH{ Lı HH HCC   LsAHH[A^A_]UHAWAVSPIIHu H HX(} x!HH{ Lc HHs HCC   LsAHH[A^A_]UHSPHsFsH{u tH HH=' 1Q H H HH[]`0wtt"u-HHwH?HHwH!HHwHUHH HH= 1Q 1]Hwttu$HwHHwHHwH[UHH= HH= 1Q 1]UHAVSIH   L9t+H;I6 1uKA;NuHL[A^]   [A^]HwNs
HHpL UHSPuHHp蹯 H HH=S 11Q H[]UHAWAVAUATSH8IIHs     M  L-H IE HuLX  IE HuLX  MEA   	'  @E1  @   HCHEH5 L2
  H}
   (   謮 H   LLPA   H HHG E11L      HH' E11L   u|I Iv A  AjHL HH E11L   uGHH E11L   u+I Iv [ AH HH= E11Q DH8[A\A]A^A_]IE L
  HIE L
  HH跭 DUHAWAVSPIL= I
  HIL
  HHu H[A^A_]UHAWAVSH(IHwFs7L=5 IH
  HIL
  HHH([A^A_]
 u;HGHEH5 LH  H}
   (   ¬ Ht'HLPIH HH=ٯ 11Q /L= IH
  HIL
  HH臬 ÉH([A^A_]UHAWAVSPIL=\ I
  HIL
  HHH[A^A_]7 UHAWAVSPI֋vt5t5u~L=
 IH L   1ۅuvI~ k` .LfL= IH L   1ۅuFI~ D  HIH  IIHP0LH HH=ܮ 11Q HH[A^A_]UHAWAVAUATSHIHIL-L IE H
  IIE HUHMLHx  1   Eԃt0urIE HMH9
  HLHoHux   ?IE HMH9
  HH= HŪ tYH= H貪 tp   LHIE H
     LHV LL 1H[A\A]A^A_]HEHpLL臊 H1HtLHKHEHpLL] H1HtLHUHAWAVAUATSH8IIHIs      H ]  H HuLX  } R  HCHEH5 L  H}
   (   r HH  LLQ       ID$H;    A<$   H= H L   IH{LLLMԪ  uwH H L`  1M  H   ID$H;R    A<$   H H L   IH{LLLM҆    H AM AAE 1  HL   1   HH= 1Q L   HDHM   H{LLLM  1   HH H L`  MIE H;H0bH{LLLM 1unHH H L`  MH H L`  1Mt8HIE H;Hpz Lu   LL赏 LL 1H8[A\A]A^A_]UH1]UHAWAVATSH0IIAw1ۍFrruVL%. I$HuLX  } tPIGHEH5  L^  H}
   (   ئ Ht1LP H HH= 11Q H0[A\A^A_]I$HuLX  M1ۅtHuUHHH H HuX  HU1t1HuH]UHSPHGHA H H?Hs  CH[]UHH H H  H?R]UHSPOt*t5H u:H H  HHH[]H{ Ã
#Hk Ã
HH= 11R H[]UHAWAVATSIIIAwt3twHj    H H  IAVL[A\A^A_]I|$A${H5Z ; fAD$	 {It$IL HI\}ALALD)ZI|$A${H5  fAD$ {It$IL6 HI\	}ALAL	D)HH= 11Q [A\A^A_]UHAVSHHw3t6tDuSH
n H	L  H
  HHHL[A^]Hx HH[A^]9  Hx HH[A^]V H HH=> 1Q 1[A^]UHAWAVSPMHHwt<tSukH
 H	L  H
  AVHHLH[A^A_]Hx HHLH[A^A_]:  Hx HHLH[A^A_]V Ho HH=ŧ 1Q 1H[A^A_]UHAWAVSPIA6Fs?A^M~~LM ˃M~H H LP0IF    uH[A^A_]H H HH H=n 1H[A^A_]UHAWAVSPIA6Fs?A^M~~LM˃M~H H LP0IF    uH[A^A_]H] H HH H= 1H[A^A_]UHAWAVAUATSH(IAD$EL- IE I<$Hu  AD}LuLLA|IE P(HIE I<$UH  HHt LL*LbHH([A\A]A^A_]UHH?H6]雡 UHSHH H HU  ÅuH H H}P@H[]UHAWAVSH8IHL=< I1HU1H  uRI1HU1L  u)L=% IH}LuLPPILP@H H H}P@H8[A^A_]UHAWAVSH8IHL= IHU1H  uOIHU1L  u#L= IH}LuLPPILP@Hk H H}P@H8[A^A_]UHSHHH
! H	1HU1H  u,E1ۅt1HMHuH H H}P@H[]UHAWAVATSHHIL= IHU@  tIHULH  QM܃1   9t@Mt6IL   1HX  HH= 1HAILHh     H[A\A^A_]UHAWAVSPIHL= I1HU1H  u$I1HU1L  uE;E؉H[A^A_]UHAWAVSPIHL= IHU1H  uIHU1L  u]+]H[A^A_]UHAWAVSPIE    L=^ I1HU1L      t&ILx L
  HH= 11HA1LHÃuH[A^A_]UHHH
 H	1H  1]UHAWAVSPHL= IHu  IIHuH  HcU;UuLH艝 1H[A^A_]UHAWAVSPHL=n IHu  IIHuH  MU9NHcLH% H[A^A_]UHHH H Hu  HU1t1HuH]UHHH H HU  H]UHAWAVSPIL= I
  HIL
  HH蟜 H[A^A_]UHAWAVSPIL=p I
  HIL
  HHH[A^A_]K UHHH4 H HuX  HU1t1HuH]UHHH H HU(  H]UHAWAVSHIHL= I1HU1H(  u+I1HU1L(  uEE fH~ÃH[A^A_]UHAWAVSHIHL=c IHU1H(  u9IHU1L(  u#EMػ   f.w1f.ȻF؉H[A^A_]UHAWAVSPIHE    L= I1HU1L(      t&ILx L
  HH=0 11HA1LHÃuH[A^A_]UHHHz H HU@  H]UHAWAVSPIHL=P I1HU1H@  u$I1HU1L@  uE;E؉H[A^A_]UHAWAVSPIHL= IHU1H@  uIHU1L@  u]+]H[A^A_]UHAWAVSPIE    L= I1HU1L@      t&ILx L
  HH= 11HA1LHÃuH[A^A_]UHHH% H HUHMx  H]UHAWAVSPIL= I
  HIL
  HHߘ H[A^A_]UHAWAVSPIL= I
  HIL
  HHH[A^A_]鋘 UHHHt H HuX  HU1t1HuH]UHHH: H HUH  H]UHAWAVSHIHL=
 I1HU1HH  u&I1HU1LH  uHEH;E؉H[A^A_]UHAWAVSHIHL= IHU1HH  uIHU1LH  u]+]؉H[A^A_]UHAWAVSPIHE    L=H I1HU1LH      t&ILx L
  HH=ќ 11HA1LHÃuH[A^A_]UHHH H HUH  H]UHAWAVSHIHL= I1HU1HH  u&I1HU1LH  uHEH;E؉H[A^A_]UHAWAVSHIHL=J IHU1HH  u5IHU1LH  uHEHMػ   H9
1H9ȻMډH[A^A_]UHAWAVSPIHE    L= I1HU1LH      t&ILx L
  HH=y 11HA1LHÃuH[A^A_]UHAWAVATSAAL%\ I$0   P(HI$McB<    P(HCHJHCI$McB<    P(HC HCJHC(H[A\A^A_]UHSPHH;Ht
H H P0H{Ht
H H P0H H HH[]`0UHAWAVSPHH;HCH)HLcL9}6L{I)H H B4    P8HIIcHHKJHCH[A^A_]UHAWAVSPHH{HC(H)HLcL9}7L{ I)H H B4    P8HCIIcHHK JHC(H[A^A_]UHAWAVSPA׉IINIFH9r I+HtLINIHH9t   9t   D9zuDyHIN1H[A^A_]UHAWAVSPAAHHK HC(H9r!H+CHtHHK D9DqHHK H[A^A_]UHAWAVATSAIII+?H~ IIML9tI)IcH3LԀ HIuL[A\A^A_]UHAWAVATSAI   } IIML9t I)IcH31L HIuL[A\A^A_]UHHLGL9t)Hc9tHI9uFT)H]UHLHGL)HHcH Hʺ   LH] UH+]UHG+F]UHAWAVATSAI   | IIGM L9t I)IcH31L~ HIuL[A\A^A_]UHLGHO 1I9t*HHAL)HL)HcI4Hc>HHu]UHAWAVAUATSPAAI   | IID$Md$ L9tEAOALALI)IcHHcL<D9+uA7La HIIuLH[A\A]A^A_]UHH>Hv]> UHHN HHwHHIH]? UHSPHHv H 8   P(HH(HH H@    HHH(HKHH0H[]UHSPHHC H8P= H' H HH[]`0UHAWAVSPIL= Ix   P(HHCp    HCh    HC`    HCX    HCP    HCH    HC@    HC8    HC0    HC(    HC     HC    HC    HC    H    H{ L3AIH H
  HH[A^A_]UHAWAVAUATSPIL}I?_ ILIIwI+wHLc   I_EL3L@: MIIFI~HpH+pHIU; 1LLML}   HI;_uLH[A\A]A^A_]UHAWAVSPHH{HCH)HLcL9}7L{I)H~ H B4    P8HCIIcHHKJHCH[A^A_]UHAWAVAUATSPIIIIGI9GrI+GHtL\AAMLtI7LLL? HËMoI HUHAWhDeEtM+oILhIGHHIOHH: L: DH[A\A]A^A_]UHAWAVSPII?]IIwI+wHLI_H31LHI;_uLH[A^A_]UHAWAVSPII_H;:: HI;_uL5 II x  IHtIP0I?J_ ILH[A^A_]`0UHAWAVAUATSHIHUIHL9swH HH= 1Q I6HuLk LS`HtH
y H	Hp  ML}LtH3LL= HI6HULSh} t$LH+KHHHI H}B9    2I>L9t8 HuI6HULSh} tL+sILp1H[A\A]A^A_]UHAWAVAUATSPIAIHH;I7L   E1   H;IH{Hs1L,  H;I7Vd HEIIwLHtCQ  I;Gt=H8>? IǾ   LLs H: H LP0LE1   Hw LH[A\A]A^A_]UHAWAVAUATSPIIHm] AEusHf HLf Iľ   LHr L- IE IH5 L  IE LL  IE HP0IE LP0DH[A\A]A^A_]UHAWAVATSIIIHH)HLL9tH3LLHI9u[A\A^A_]UHAWAVATSIIIHH)HL;L9t H3LLtHI9uIL[A\A^A_]UHAWAVAUATSPIII>MnI7Mg1ɅtYI?I6[b HI>_HEMvM9t*I6LH:   L9tI61H}IM9uHu HMHH[A\A]A^A_]UHAWAVSPHHt(HHH`: IH{ LS`IL5 
H{ S`IMtIcFHHCHCH[A^A_]UHAWAVAUATSPIII>MnI7Mg1ɅtYI?I6]a HI>aHEMvM9t*I6LH<L9uI61H}IM9uHt HMHH[A\A]A^A_]UHAWAVAUATSH(HH]II<$H3x\ IE1M  LILmID$I+D$HHsH+sHL
It$ID$H9   LHUHKHuL{I9   ML&M/H}2 HID$I|$MHpH+pHHSv3 IEI}LmHpH+pHI$HPH+PHSJ3 1LHH]HUHuIL;{wID$LHH9QLH([A\A]A^A_]UHAWAVAUATSPIII<$2V ILIIt$I+t$HLI\$H;LL2 1LH.HI;\$uLH[A\A]A^A_]UHAWAVAUATSH   HH`H@L/IuI+uHHv ILHU HXI}I+}HHCH+IH)T II$IEHcH4HTLLV HI;\$uH`HXLHU LL8sH(LuLH@L  Hg H HhL  HLHML   MtuHH HH IH8H8L9E0 LpI] I;]t/H0IGHcH<HtLMv0 HI;]uH0HXILPL"  I I\$HHHsH+3HLL+L;kH   H@H@IcM L<HX/ H`L3L;st2H`L`IGIcH<HtLMd$70 IL;su1HPH`IHHL;hu1H(H0H H Hh  HL oLp H H H}x  H(H   [A\A]A^A_]UHAWAVAUATSHIHIHP H H H
  LcL;ct<LmI$HELuLHuLAWHHxLH+sHq IL;cuH[A\A]A^A_]UHAWAVAUATSHhIIIIHU ILmL;k  IE x  LxL`IHHH+HHID$I+D$H|Q IH]HsHLLJS LHHSHL4S It$IT$LLS   LHH]MfIFHEI9N  L}I$HHH@LqL+qLIHEH+HHMM|
 H H 1LHl    1  LeM IwI+wHHLMoIOHMI9MHM   HEIHHMHAHEMe H}, HELpH}HuLY- HIH}LHD- HHID$I|$H]HpH+pH - 1HHuHEIL9uLeIL;eLm      LLh 1HHh[A\A]A^A_]HԆ    Hxch LR 1ʾ   HxLh H.1UHAWAVAUATSHHIIL}L'L6HGHEHFHEv  HMLLLMU ILuE1M<  LmLILeIw I+wHL(MoM;o    IcE HML$IcEHML<HEHL* HELpID$I|$HpH+pHL+ IOHQH;QtFIWHM4HLg;tLLp+ HM4IOIWHHAH+AHI9Lu1LeLH]HHEtIL;h Lu1H}Zk -LH}Gk H\ .   H}Hf E1LHH[A\A]A^A_]UHAWAVAUATSHxIIH   LyH`H}LH1LYHhLkIL;ktXIE HpHhHxH}HpUHtLpM&LH+sHA$LIM;fuH H H}x  HhCj H`7j Hx[A\A]A^A_]UHAWAVAUATSHXIILmL'L>HGHEHFHEHMLELLLMoT ILuE1MY  L>IL}Iu I+uHLI]I;]   HcHML,HcCHMHHELH( LpIuHNH;Nu
HEH]QHEH]IMHUL"HLA<$tLL( HM4IuIMIHVH+VHI9LuLmIEHHH;HLetGIMHUHHL;tLL( HM4IEIMHHPH+PHI9Lu1L}LHuH]HI;\$ LuH}vh H}mh LHX[A\A]A^A_]UHAWAVATSIHL{(H;IIv I+vHL
I^HcCI41LXHI;^ uL[A\A^A_]UHAWAVAUATSPIII<$RIIT$I+T$H   LtIMl$ID$I9t(I; tIu 1LID$HII9uLrg LH[A\A]A^A_]UHAWAVAUATSHXHMHUIH}L?IGI+GHEM&MnID$I+D$HEM+nHHEHXH+X1LL8T tH݀      H]LeLmLuLeI1LLuLS D9H]w  HLmI1H}LS AE9Q  9I  LHELHu~N IL6% HH]LLR HEHEHHH;HLeHE   HHEHMH	HML{HAHyHpH+pHL% Mt$M;t$LA    H]LmtXHIHEE1IIHHHxHqH+qHHUk% HHuLLH;CAIL;quH]IHEA9Luu1H}HuHMHI;NM)H}% H}@e HEH    H}` 1HX[A\A]A^A_]UHAWAVAUATSH(LEAIIHH]H5} LIH;   QJ HH]HLH H;C   H,M&M;f   HEHEH@Ic$L,HY# HEHXIEI}HpH+pHH$ MLcH H AOMD  JH]M 1H}HurIM;fEAuHE   H}Lk_ HI 1H([A\A]A^A_]UHAWAVAUATSH   MD`HPIIHHHH5f| LIH;   
I HHLOG H;Cw  HH8IL0H H H H}H
  LH@L L;`  Hc`H(HHH@Ic$L4H8! HL{LpHPHxH}HpHlUȃl tH`L(LhH`LhHXIFI~HpH+pHL5" LcH
 H D  K I H       II HXLh1L0LH`vIH@L;`H H H}x     LL_] HG E1LHĸ   [A\A]A^A_]UHAWAVAUATSH8IHIM<$LRH IM;w  Ix  LmLhLH]HHHH+HHIEI+EH| D IIwLLLE LHIWLLE IuIULLE B  LHEHsH+sHHLkL;k   LeIGI)IMcLuMe I\$N4HT H 1LH       HEL<IF HEH}\ HELpI|$LL'  HIIID$HpH+pIt$LLuH HHHMHAHyHpH+pH 1H}Hu`IHEL;h0HE      LHU[ 1H8[A\A]A^A_]Hy    H}/[ LqE 1о   H}Lm[ H}1UHAWAVAUATSH   HH@HHXLcLsLpH H H}P
  H{H+{Hi H`M9   E1MILpIH}HH|Uȃ| tIcAHHH`H&k I^H}HH|Uȃ| tIcAHHH`Hj LpIM9xLpH`LhL+(ILHEELhH H DP(HHLq M9tBLuI$LxI7LUIwDxELUDxIcHpIL9uH H H}x  E~^McHhAMcE1IHhIA    t)K4DLLފ"HHuMMM9|IME9uHXHPH@HqH+qHHIE   1ǅh    H@HHHpE1HpB<0 tWHX IM|$L`LhQh I LLHD:h IG 1HPLIE9uhhH@DD9\H9 H HP0H`1g HPHĘ   [A\A]A^A_]UHAWAVAUATSH(UIHLkL+kIM|$M+<$I1DZ ILu1L\ E~rH]LeEEM~M+>IA|&DDLe{DLF   ˃A|'L}A1LD__ 1DL   ALH([A\A]A^A_]UHAWAVAUATSPIAAC\$D9}MDkE9}DL,	  AOADDL	  yIDD^ C\-D9E|H[A\A]A^A_]UHAWAVATSIII>IIwI+7HLIIFHcH41L$HI;_uL[A\A^A_]UHAWAVAUATSPIHIIFH9wI9^sH+ HH=t 1Q IFL9wM9~sH HH=t 1Q L9vH HH=t 1Q L9H]t7Mf L-ȱ H]H3LAV`IM Hp  H; HI9uIVL)LeLLm M)IFL)IFL9t/M~ LH3LAV`HtHI+NHHHHI;^uLH[A\A]A^A_]UHAWAVAUATSPIII<$Ml$I6M~? t`L}I>I4$E HMd$1M9Lt6I4$LILMILH;ELMIL9uHY H[A\A]A^A_]UHAVSIHKx'HKH+KHIVI+VH99 [A^]UHI]UHAVSIHx'HKH+KHIVI+VH99 [A^]UHAVSIHx'HKH+KHIVI+VH99 [A^]UHAVSIHox'HKH+KHIVI+VH99 [A^]UHAVSIH-x'HKH+KHIVI+VH99 [A^]UHAWAVAUATSPIIIHH > IL;kt5I} LxHLH+> H   H;Ku   LL   LLR 1H[A\A]A^A_]UHAWAVATSIIHH;GE AHLL   AD[A\A^A_]UHAWAVAUATSH(IHuIL}M/IGHcNI+GHHAFHc؍    <@H% H P(IFH#H4[H Hi IwIGH9A      MfMuIMA   LHUAI9t[HEHXHHuLhI>Iu ID{HH[ LMd$HMF|8IL;quHIHEH@HuIHI9ֹ    EE|HH9LIoHMHqAH1H9DDH([A\A]A^A_]UHAWAVAUATSPHIIIIvA{I<$WD HAD}MlAD LLLIL)   DLeLD)H[A\A]A^A_]UHAWAVAUATSHHHUH}HHUHY{LGHFL;G   IHuLEILaHMLqI {LjhI} H}I$HEILLHIHMIT ALMv IH[H}Hu HUHHuL|0L0IL;juIGAO IE }@ LEILxHHML;A=HuIAGA}HM)HH[A\A]A^A_]UHAWAVAUATSHHIW)E)E   E)EHE    EH6 H HX(I} LuLA ALeLLLA|HLHLLH HHH[A\A]A^A_]UHAWAVATSH@II<$W)E)E   E)EHE    ELuLHA H H HX(L}LLL xHLHLLH HH@[A\A^A_]UHH]/ UHHv ] UHSPHH H (   P(HX H@    H[]UHSPHH{ 	 H H HH[]`0UHAWAVAUATSHHH]AHH{HHULaR HM,H{DR HCL8LpM9    t@HcM$IIc_H}ЉZ5 IMH4IL$HHu	M9MuˉHU19rDȉH[A\A]A^A_]UHAVSIH  H 8   P(HHC(HC    wP HCIHC( IFHC0H[A^]UHSPHHC H8 H{
Q H H HH[]`0UHAWAVAUATSHIAIHk H AH
l       LLP  OIvHMH$L%Ș Ll    E1LL	     uHcMHLDLBT!ÉH[A\A]A^A_]UHAWAVAUATSHHI  L{L- IE H] LL   A      IG HELL(HCHEH{ H
  IHf H H{(
  HLL7    ELH7    L}I_I;_tGLc}LcHH@H
 L	JJA   LHuA0  HtNHHEH;XuHѦ H L  E1(H H H
Wk    LHP  A   DH[A\A]A^A_]A            LML A            LIK UHAWAVAUATSH   IAIH HH]A-H
 H H
j    LLP  A      I_H٥ H H LH   A      HC HH      WI HHAHIH)Hu%H8AILLDLDG A"L   LHuI LLI H HH;]uDH   [A\A]A^A_]` UHAWAVAUATSPI   HZL5 IH LH      HC HI11  HELkL{M9t8IE IH8  HILHuHp  uPIM9uILHuh  1$HP H H
@i    LP     H[A\A]A^A_]ËHIH   H}ЋHI   UHAWAVAUATSPIuzHZL%У I$Hu LH      uqH{ HI$L  L{H_ pLAII$HP0I$LLh  1$HZ H H
Jh    LP     H[A\A]A^A_]UHAWAVSPHuYLrL= IH HL      uQIF ILh  HxH+xH  HHA1$H H H
g    HP     H[A^A_]UHAWAVAUATSH  IAIH HH]DLA,HN H H
Lg    LLP       MgH HHĠ LL        ID$ HMgHH LL     Md$ 11KHH      iE ADI LIHH$HHLHHMMa  t
LR0  LHLLLsIL*M   H   HHHHHH H LHH       L{ 11MHL4$LHHLMIa     LLHLILDHqMMDHLD 7HK H Hh  L]  LH1H HLH H   H;]uH  [A\A]A^A_]H[ UHAWAVAUATSH   I׉IL-: Mm LmЃ,H H H
d    LLP        L   	   L3C IwLL;    H   H3IōsL|.ILILLL]  u;I˃H H Hh  Ln\  LH1L-X Mm LL-C Mm    L;muH   [A\A]A^A_]Z UHAWAVSPHu\LrL= IH, HL      uTIF ILh  H HxH+xH  HHA1$H. H H
c    HP     H[A^A_]UHAWAVAUATSPHI  L{L-ޝ IE H LL   A      IG HEL8IE H{
  IIE H{ 
  HLL-    ALH- AE   IE   HHEHXH;XtEMcMcHH@H
0 LJJLIA     HHEH;XLuH H Lh  E1(H H H
!b    LHP  A   DH[A\A]A^A_]MA            L9B A            LIB LHA   HM H    UHAWAVAUATSH   IIH HH]Ѓ   MeH H H LL   A     Md$ I]Hқ H Hx LH      HC HI] H IIE H@ LH      H[ H      %? LHHHQHtJIM Hh  HX  LHE19H H H
y`    LLP  A   HL? H\ HH;]uDH   [A\A]A^A_]V UHAWAVAUATSH  IAHL- Mm LmA,H H H
_    HLP       MgHX H H HL      Z  ID$ HMgH H HĘ HL         HMd$ H      H=    HLHIM   AI EH    M/H HLH4       AIIu    LHHLHIuHH= DHH=    :H Hh  LV  HH1
L   L-W Mm L;muH  [A\A]A^A_]T UHAWAVAUATSPIՉI-H H H
4^    LLP  A      M}Hi H H LL   A      M II?LL5 HHt]IHpH+pHHD IHh@ LLHLR@ H H Lh  HNU  LHAE1DH[A\A]A^A_]UHAWAVSPHuWLrL= IH< HL      uOIF ILh  H8GHT  HHA1$HC H H
3\    HP     H[A^A_]UHAWAVAUATSHX  IAHL%o M$$LeA'H H H
pT    HLP    MoH H H] HL      }  LHMm IE HIGHMA_H      : HcHiVUUUHH?H R9t/H[ H   H9 HH:   LHH# ILALLL|QHD`I_(HsHLH,Ha  LH! I;F)  AHALMHHHBHHJH9L% M$$  HHPH+PIVHI0LH@HL)L. HL   IMwHԔ H1A   HHHH0  H  LIEI}HpH+pHL  H   HM4DLLHN H I4$HHHx  "  I7HHHHH   HH   9H H    IIIIAAh1HH^HHH;L% M$$L{Hw HHL   LL
  11LLHAAHAHL   HLh  HP  LHA1   H  L%k M$$IH DHHH˒ L	H   LM  L%" M$$LHL   LL
  11LHALH	H   H!6HSH   Ho6 LH6 LK  L% M$$   L;euCHX  [A\A]A^A_]HߋHLH =H   /M UHAWAVAUATSHIԉI,H H H
W    LLP     O  Ml$L5s IH LL         LIE HHHPH)HuPL0HHEЃIIIE RI<$
  HLH^!    HHMHIH4            LM7 @Lu11  HEȅ~pIE I<$
  MIH}L  xpIIHHUHRHLHup  Mt`         LI5 HuIE Lh  1ɉH[A\A]A^A_]þ         LM5 MH}ȋHeIE    VUHAWAVAUATSHHIAHAFr,H H H
U    HLP     y  MgMoH H H' HL      G  H]LeI] AtEA   H{H+{H1L6 MI1L8 IV L5 HEH}   H;IW Hu;,    IHH  1LHMIH07 IV(L5 HEH}L   IH H Iw HMH$HE LT    E1LeL	  I>IW(L+ HH}LH.L}IHL6 HEHP0L5 HELL}HULmHE AE I  M'1M;gC  HCHEH;I趹Hþ   HIc$HMH41H^HK  IAE I1A   H]HHuL0  HtbI1HHu8	  AU JAM IL   ؉LwOH.  HcHL}IM;g>1uAU JAM    IL      LL}FIL   L   IH]H  H=1S 1AHLHAL}   AHCH;CH]t!IL   H
  11H}HAL4 HIH   H}H	I   H}H	I   DHH[A\A]A^A_]f fUHAWAVAUATSH(  IIHGz HH]Ѓ  L      Li/ It$IT$ LL$IM  It$(IT$0HLHHy     ` ILL} LHn LIH# H It$HHLLL  O   M   LI?
  HI4$HHL     HH0HHL     1LH3Hv H HHHH   mH= HHH  HHh  LG  LH1zH H H
P    LLP     [HLy.    EHLc. HR  H H H  Ld   Hx HH;]uH(  [A\A]A^A_]OE UHAWAVAUATSH  HIL-w Mm LmЃH8 H   HsHHLx  A   G  L      L, HsHS LLmIM       HHL H薴IŃ ~\LI}   HI7HHHL     1LHHHH? H Hh  LE  LHE1VH
O    LHP  A   AHL, 0HL, Ht  L蝶A   L-;v Mm L;muDH  [A\A]A^A_]C UHAWAVAUATSHIAHA-Hx H H
lN    HLP  A     MeHG H H HL   A     H]ID$ HEL IEHEEwD. AI|DI H H I} 
  HLH+    ILE3 AEIGI+HIL$I+L$HH9A   }tL- IE H}
  HLH xL3 I;G   H}HLHH]tTIM Lh  HC  HHAE1vL9M          H}MA            /L6M          Hu+ #         H}IX+ L. DH[A\A]A^A_]UHAWAVATSHuoLrL%t I$H HL      ufIF H8S II$L  II$LP0I$HLh  1$H	 H H
I    HP     [A\A^A_]UHAWAVAUATSH   IԉIL-8s Mm LmЃ,H H H
L    LLP        M|$H{ H H! LL         I I農IL英L      L' ~6I$ILLLB  tLǲL-kr Mm    .H H Hh  LLA  LH1L-6r Mm L;muH   [A\A]A^A_]? UHAWAVAUATSH   IAIHq HH]A,Hc H H
H    LLP     9  Mt$H2 HH؁ LL         HI^ Mt$H H LL      LMv L      L& HLLIM   AI E   M4$H H HLH'       AIIv LH蕴HLHIuHH& mHH&    Hp HH;]uVH   [A\A]A^A_]H H Hh  LO?  HH1L臰H+p H   = UHAWAVAUATSHII   I\$L-x IE H LH   A      HC HEIE It$HMH$Hu L.I    E1LH	     Md$ IE H LL   ueIt$ HcMHH}Tx^IE Hh    LHE1(H H H
H    LLP  A   DH[A\A]A^A_]þ          LM& UHAWAVSPHu[LrL=@ IH~ HL      uSIF ILh  HPH;P  HHA1$H H H
D    HP     H[A^A_]UHAWAVSPHu[LrL= IHJ~ HL      uSIF ILh  HPH;P  HHA1$HM H H
=D    HP     H[A^A_]UHAWAVAUATSPHI   LcL-~ IE H} LL   A      Md$ H[IE Ht} LH   ubHC IM Lh  H  I<$H0
 LHAE1(Hs~ H H
uF    LHP  A   DH[A\A]A^A_]UHAWAVAUATSH  IAIHl HH]DLA,H} H H
F    LLP       MgH} HHp| LL        ID$ HMgHH=| LL     I\$ 11IADI LIH   $   H  H$HHLHHMM=  t
L0  LHLLL膸LIڣM   H   HHHHHH| H LHH<{       L{ 11HL4$LHHLMI<     LLHLշILH!MMDHL  7H{ H Hh  Lf9  LH1HPj HL蘪H<j H   H;]uH  [A\A]A^A_]Hy7 UHAWAVAUATSH8IԉICr-Hm{ H H
C    LLP  A   9  M|$H;{ H Hy LL   A   	  IW u2HHHH+H1Ht^   %      LM"!   Hz H I|$IL}
  I?IL
       HULmEHUHzH+zH1! I1Lo# L=kz I11  II] I;]t9LcuHEH@HcHIH@JH}Lp  uaHI;]uIH}Lh  Lh" E1   A      %      LM    Eu9IT$ E1yA$HA$IL   L" A      Hy H It$ HMH$H,m L?    E1L	  uJ}DIT$(LeI<$L| HHt"LmLHDIH{! DH8[A\A]A^A_]UHAWAVAUATSH   IIHSg HH]Ѓ   MfHx HHfw LL   A      Md$ MvHH9w LL      I^ L   &   L LHL莫HtPH
>x H	Hh  H5  LHE19Hx H H
@    LLP  A   HL HWf HH;]uDH   [A\A]A^A_]3 UHAWAVAUATSPIԉI-Hw H H
4=    LLP  A      Ml$Hhw H Hv LL   A   u^Mm II} LL HHt:LH>IHv Hw H Hh  Lr4  LHE1DH[A\A]A^A_]UHAWAVAUATSHXIIFr-Hv H H
C?    LLP  A     HuI\$L-|v IE H!u LH   A     HC HEHIE HMLcK|
  IHL; HH;K   LmHHEH+KHMHv H K|
  H57 HgIA      H HHL H;C   H薡HEE    HE   Hu H It$HMH$L50i L;    E1LL	  taH}A     A      +      LM _     +      LLE H! 7  HcEHBD0EH]L}LeMl$ID$I9   HMHLcLMl$HMH	HMI9Ż    tFHEH@N41IE H@J4H}L}tIM;l$uH}  HELpHMHAHyHpH+pHLJ  LcHt H É  K 1H}Hu賡HMHID$H9.Hs H Hh  H}J1  H}HE1DHX[A\A]A^A_]UHAWAVAUATSH   IAIHb HH]A,Hws H H
B<    LLP     8  Ml$HFs HHq LL         Mm H   -    AF   L裟HHA|PIAI<$
  IHI|$
  HLHH{tbHAIAHh  H/  LH1H` HRH_; H	   H+ LH HL HH` H   H;]uH   [A\A]A^A_]- UHAWAVAUATSH(HIE  LsL-q IE Hup LL      =  Mv LcLeH[ H]I>xA$HEI^MfL9tkL3LZ MM 1A   LHuHA0  H   IE LHuHUP     } t1H}L͞HI9uIE L   H]H
  11LHAH}H
IE    H
IE H   IE Hh  H}.  LH1'Hp H H
9    LHP     H([A\A]A^A_]IE L   H]H
  11LHAH}H
IE    H
IE H   H}ʞUHAWAVAUATSHHIH  M~Ho H Hvn HL        H]M I?LgLoLmI^H]qHEIGIOHMH9   HEH HEH@LpL+pItLHELhIMIM IH4o HH01A   H}	  H   IIIuHn H H}HuHUP     } t1H}Hu肜HEHH;ELmHM9LuL=n t I$IH011L   IM9uH}H	I   IHh  H}+  LH1HGn H H
7    HLP  VLmM9LuL=n t I$IH011L   IM9uH}H	I   H}衜   HH[A\A]A^A_]UHAWAVAUATSH  IAIH\ HH]DLA,Hxm H H
v2    LLP       M|$HGm HHk LL        IG HM|$HHk LL   uM 11yHH   1    ADII LMLL<$HHLHHMM,  tL}HZ H     HLLfILPH   MHHHHHHl H LHHj       L{ 11yHL4$LHHLMI;,     LLH辫ILsH蠒HMPMHxk H Hh  L(  LH1HY HH;]u)H  [A\A]A^A_]LH' UHAWAVAUATSH  IAIHrY HH]DLA,Hj H H
/    LLP       M|$Hj HHMi LL        IG HM|$HHi LL   uM 11ِHH   2   
 ADII LMLL<$HHLHHMM[*  tLݐHLX H     HLL-IL谐H   MHHHHHHxi H LHHh       L{ 11ُHL4$LHHLMI)     LLH腩ILӗH HMPMHh H Hh  LC&  LH1H-W HH;]u)H  [A\A]A^A_]L]Hb$ UHAWAVAUATSHx  HL%V M$$LeЃ$HLh H H
1    HP    HLzIL5h IHf HL      U  M IHMeIHf HL     LHI\$ L3H   4   ^ I~I+~HIwI+wH
ILHL
  LHHOH+OHH9usHHCHcHiVUUUHH?H R9L%U M$$   H, L   L
 HL L9L   LL
 HL_ L贍L%#U M$$   L;e  Hx  [A\A]A^A_]LHLq T  IŃ|UDcHHX0HsHHHvH  LH_  I;E  AHALHALHHMLMLLLLHKH;K  HH HPH+PIT$HxH8H@t1HHML!1LILL蚲H"  HL|  HLhL
 H$e H 1A   HHH0  H  ID$I|$HpH+pHL  H   HM| DtLLxHd H I4$HHHx  <  Iu HHH]vHH@  HH   9H=d H    IIIIAAgH1HH译HHHH;HLLLIL=c IL   HH
  11HHAMHIH   IHh  H   HHL%Q M$$1L=@c H  L%Q M$$IHHL=c HHL%vQ M$$H	I   Hx  LIL   H
  11LHAՋHIH   H^HH"HSH   H HH Lq  HL=:b H  HH߃L%P M$$LH-I    UHAWAVAUATSHx  IL=IP M?L}Ѓ$Ha H H
+    LP  d  HII]L5a IH2` LH      1  H[ LL3H IuHHLx    H   5    H@HcHiVUUUHH?H R9t+H& H   H LHy   LIM LH  IŋHHsH+sHH ~wHE1Hg` H J<
  LHb  IM;f  IT$LLL!    M+fIHDD#ID;|HL|QDxHLp0IvILH\pH6  LH  I;E-  AIALpMLsH   LL_HLLH耔ILxH H LLLM~IMI;M  IE HPH+PIWHhH8H@d1HHL)1H菉ILL
H6  IL  HLxL H^ H 1A   HHHL0  HH  IEI}HpH+pHLl  H   HM4DdLLhH^ H Iu HHHx    I4$HHHoHH  HH   9H] H    IIIIAAgH1HHHHLxI;MLLpOL=8] IL   HH
  11LLHAԋHIH   L跋IHh  HQ  LHL==K M?1  JH   H  HH9 L  H肃   L=J M?4  JH   H  H   H  L=J M?@HHL=J M?H\ H    H  HL%[ I$L   LL
  11HHAAE HAM 
I$L   HxeHYMHLIVH   H  LH   Lw  HIL=I M?   L;}uTHx  [A\A]A^A_]H  AHAL=wI M?HZ H L    UHAWAVAUATSH   IԉIL-.I Mm LmЃ,HZ H H
B$    LLP     "  L   6   L'  It$LL  IƸ   M   L#HsHy|LILS  II$LLH  uT1HL蝇tII˃HY H Hh  H8  LH1L-"H Mm IL8  #I$H   H  LH  H8L-G Mm    L;muH   [A\A]A^A_]" UHAWAVAUATSH(  HAHG H HEǅ    AGr*HX H H
"    HP  A   y  LcIHX H HqW L   A   J  ID$ HH   7   K  HX IIH{
  HW)X 1HA  H AL1H1HIH3L$$   E1LHL LI" 	  HcH
X < 3  HHHL tHwRH
  HHcHM   W H
W LI} HSL  IM  1TI1H=! Q    L   W H
xW LI} HSIz  IM     LLHLs  L5W MLoH
W    HH8HSL  H    H1  H-H
V    IHsLH@     HAAKLMu'HHxH+xH1n  I1LA  DHHt"LHHLH虚#HHHLDI@Ht ILh  HR  LHAE1MHL]  A   6HH
   H  LH3  A   HLMtL  HtH  HC H H;EuDH(  [A\A]A^A_] f|<~UHAWAVAUATSPIuLzL-U IE HS LL         Mg I$HAHIH)Hu]H8Hp`t^IE Hh  LxH  LH1QHT H H
    LP     +   8         8      LM  H[A\A]A^A_]UHAWAVAUATSPIԉI'HT H H
    LLP    Mt$L-S IE HR LL         LMn Mt$H HUR LL      Iv LaIMt]I ~nM4$HgS H LLHR    unIIv LL}ILMML}u   9      LMG  -HR H Hh  Lh  LH1
L褁   H[A\A]A^A_]UHAWAVATSIugLzL%R I$HHQ LL      u|IO HAHIH)HuMI$Hh  H8"  H?  LH1@H@R H H
0    LP        ;      LMH  [A\A^A_]UHAWAVAUATSH   I׉IL-Q@ Mm LmЃ,HQ H H
(    LLP        MgHQ H H;P LL         I|$ DkILD~H   <     |*ILIILLL  ˃HQ H Hh  Ls  LH1L-]? Mm L;muH   [A\A]A^A_] UHAWAVAUATSH   HIL-? Mm LmЃ   LcHP H H-O LL   A      Md$ H{HSP H 
  HH   =     LHH9HtiH
P H	Hh  H~
  LHE1(HO H H
X    LHP  A   L;mu&DH   [A\A]A^A_]HLG   UHAWAVAUATSH   IAIH= HH]A,HcO H H
    LLP     =  Mt$H2O HHM LL        HI^ Mt$H HM LL      LMv L   >   L  1HLL~IM   AI E   M4$HN H HLH%M       AIIv 1LHs~HL
}HIuHH  mHH     H|< HH;]uVH   [A\A]A^A_]HM H Hh  LK  HH1L|H'< H   	 UHAWAVAUATSH   HIL-; Mm LmЃ   LcHcM H H	L LL   A      Md$ H{H/M H 
  HH   @     LHH薓HtiH
L H	Hh  HZ
  LHE1(HL H H
G    LHP  A   L;mu&DH   [A\A]A^A_]HL#  [ UHAWAVAUATSHHHILmȃ  L{L%HL I$HJ LL      4  M L}HCHE HC HE HC(HE I?wHEMw1M;wMk  I>  IA$I1A   LHuL0  H;  ILHuHUP    }    I1LHu8	  H  H  HHcHIL@  LIHLH}LHK      I_ HEH8Hs  u   ILmI} Hs  ILHLxL@  A$HA$L=J IL   I1HEL;pLm$  HJ H H
'    LHP  |  LmA$HA$   HRJ H L         A      H}Mg     IHh  H=N 1   ;ILH]AIHh  H= IHh  H=   LHLH]IL   L   H  H=[ 1AHHAֻ   L=xI LmIL   LeL
  11LHAH}H	I   A$HA$IL   H}H	I   tH}w   HH[A\A]A^A_]IHh  H}I  LH1jjUHAWAVAUATSH8  AHL%7 M$$LeA*HvH H H
    HP  A   <  LrHHH H HHF HL   A     HIF HL0LID$HA_  Hm  HA   HcHE1HG H K| 
  ILL  HI;^  HSLHH`    I+^HHW  IL;|I~I+~H   +4  IIvI+vHH  III;_L%r5 M$$t%IFHcH4HTLL  HI;_uLHF H LL
  HHHPILL  I;E{  LL]rHHHsH+sHHsHKH;KH<H"  I 1HHsHHHH;HL   HIL)H  HLpI$"IEHcH<HtLح  HM4HI;\$uHج  HLL#L;cAHLxIEIc$H<HtL耭  HM<IL;cuH  L  H3E H Hh  H  HHE1   H   L*  HH  LOL  H  H&  H  A      H  H  A      @      HM  L%2 M$$GHv  H     @      HM  L%2 M$$A   L;euDH8  [A\A]A^A_]  UHSPHH{ rHC     HC    H[]UHSPHH jpHtHC H\B HCH[]UHSPHH{ HCH  CH[]UHAWAVAUATSH  IHL-1 Mm LmHIC H HHx  A      H   1  uCHH0HH)  HtGHHQLHHL   A"L   LL  HLH  L;muDH  [A\A]A^A_]g  UHSPHHsB H   HX H
A HHH@    @    H[]UHAWAVAUATSHMIIHnIHB H HUHMHLx  A   uluL=oEԍHMԅ~+HEHHHMHLHLI   tLp)IFHtH@HtLHG@ IFM~ E1DH[A\A]A^A_]UHAWAVAUATSPIIIII} 蕨  HHLLL  t
Hө  4LuE11LHnu"   LLB  H}L  A   DH[A\A]A^A_]UHAWAVAUATSPIHIIH@ H H@ LH   A   uZLeLs I} Ivl  ILLL8nE1u%   LuLH  LL  A   L  DH[A\A]A^A_]UHAWAVAUATSPMHIHuIH@ H H? LH   A   uYLc I>It$  LIHuLLoE1u$   MLH  LLQ  A   L  DH[A\A]A^A_]UHAWAVAUATSH8MLIHuH}> t-L"H[? IH I<$
  H5
 HD  tI?H3L  1_  LmLuMd$M/L3H	? HHHUHML}LLx  H  LLmILeEԨ   H}f} M~uHEIM H8
  H}H  IE HMHy
  LH  x^xcH}މqft`teMԍAEHEHHEȃHE HEH 	   LK  LM:HEHHEHP   HEHPHEH   H]H	  LHn     H8[A\A]A^A_]ÐUHAWAVAUATSHAIIHHxHUPP1ۃ}    L5y= HEIC<8P(HILp(Lg  xAHHLM  LiHHC    S  HC1H  HCHC(    HC     D{0H{8IcHH4@  HEHXHH[A\A]A^A_]UHSPHHHxH3PHH
< H	Hp  HH[]    UHAWAVATSII~Ht4H(Hm< H      H"kI~@  I~HGH;uG  I~>  I^ Ht4L=!< L#H{H	I   IHP0MLuE~0Et1I^@L%; H{ tI$Hx  H{+  H`AuH; HI>P0HL[A\A^A_]`0UHHHxPHH1HtHA]UHAWAVATSIIL%c; I$   P(Hþ   H     q  HI$H{1  I$H{`1  I$H   LLH  H[A\A^A_]UHAWAVATSH II>Ht  L%: I$M~HuL  HtH]HxI$H  HuI$Lx  I$M~`HuL  HtH]Hxq   I$H  HuI$Lx  I   Ht/L;H{H
I$   I$HP0MLuI$LP0H [A\A^A_]UHAVSH3HwH   HcHLsI~(=H9 HH= 1Q rLsI~  SLsI~ Ht`I~@Ht8`1LsI>|  I~HtHHS9 H    HC9 H LP0H39 H H[A^]`0fVxUHAWAVSPAIH8 H    P(Hÿ     HCD;I>HH[A^A_]s  UHSPH8 H    P(Hÿ     HCHH[]UHHHHH+H]UHAWAVATSIII<$HGH;u@HX8 H    P(Hÿ   t  HC   I<$H  I<$  
  E18 u;I<$|  HxL  A   tIFI^Hx XfHH  D[A\A^A_]UHSPHH?-  HxHH[]Q  UHSPHH{  H7 H HH[]`0UHAVSHHHHx`HUH   I1} tH
   IFH[A^]UHAWAVSPIL=*7 IHX(2  xHH{H{L      I0   P(HCH@(    H@     H@    H@    H@    H     HH[A^A_]UHAVSHHHHx`HUH   I1} tH
   IFH[A^]UHAWAVSPIL=^6 IHX(f  xHH{H{LH     I   P(HC   O  HKHAHH[A^A_]UHAVSHHHHx`HUH   I1} tH
   IFH[A^]UHAWAVSPIL=5 IHX(  xHH{H{L     IP   P(HCH@H    H@@    H@8    H@0    H@(    H@     H@    H@    H@    H     HH[A^A_]UHAVSHHHHx`HUH   I1} tH
   IFH[A^]UHAWAVSPIL=4 IHX(  xHH{H{L     I   P(HC     HKHHCH@    HH[A^A_]UHSPHHHx`H   H1Ht!H{   H,4 H Hp     H[]UHAWAVAUATSPIA7Hw0H  HcHIGHLpH{L  H{  H3 HH=<  1Q n  MwIH{Ls  H{H;GtHHH  IFHL`L9.  HL(I}L4  I}H;GtHHHa  HI9uIFHLpL9   HHxHtZHI9u   IGHL`Lp(H{L  H{H;GtHHH  I|$L  I|$H;GtHHH  I~Lx  I~H;GtVHHH  EMwIHHIH;Xt,L#I|$L4  I|$H;GtHHH`  LH[A\A]A^A_]; >y UHHHHx`H   H1HtHA]UHAWAVAUATSHXHu7    h  HGHELoI} H}IuHuHGH@ HEHFHX AU IM(%  H{H+{H1  II}(   L[1  I1  IAUAuLLMw  É]L  u@L
   H]HL}LLHMM  L
  HLLHMM  LA  L9  HEHxH+xH1  HI}(1H*[19  I1/  IAUAuHLM
  AH  Eu>L
  H]HHuLHMM"  L
  HHuLHMM  EAă} L  L{  IE(HHHH D DH  HGHELwI>H}IvHuHGH@ HEHFHX H]AVIN ]  H{H+{H1  II~    LZ1$  I1  IAVAvLLM  É]L  H]uDL
T  LmLH]HLHMMT  L
Q  LHH]LHMM4  H]Lm  Le  H{H+{H1  II~ 1LZY1i  I1_  I11LLM  É]L  u@L
  H]HLmLLHMM  L
  HLLHMM  L  L  IF HHHH Iv(HuHFHX H]AV8IN@H}
  H{H+{H1+  II~@   L{X1  I1  IAV0Av4LLM  É]L+  u@L
  H]HL}LLHMM  L
  HLLHMM  L  L  L}II+H1u  II~@1LW1  HH]1  I11LHMb
  É]Lx  uGL
K  H]HL}LLLeLLE  L
>  HLL}LLM  H}$  L  IF@HHHH    A~H   IOI+OHHuHVH+VHH}HwH+wHHH9]  L%U, I$H5l  L}L  H}LL  I$H5M  L  15  HGHELoLmIM HMIEHXH]HAH@ HxH+xH1  HEIEL A   I9   A   I$H8H}LpHGHX AUHuL%     LHUUHHKH+KHH9        IHSH+SH1L`UHL+HCII9t"A}  tHLc  L  HCHa  H}HuLmLHMML
=    L7  E1IFIF IL;eM1C  HE18  HELeI$HI;\$tND+AtLH  A|H}H}d  E1HQ* HH=  E11Q    L
  H]HLmLLLeLLE
  L
  HLLLH]I
  H}R  HJ  H}A  $]1Ʌt} t} t} tDDHX[A\A]A^A_]UHAWAVAUATSPIIIGLh H^ I} H3N  IMt)I} LUIH{HsLLZL  HWILLLH[A\A]A^A_]    UHAVSIH  H{t6HGPHCLp H{H' HGH( H `  A   Hw E11H  D[A^]UHAWAVATSIHIIGL` LFVtIID$IL$H)HHHcH4LLT      u@IT$HrLu+H%  Hþ   LH  H!( H HP01[A\A^A_]UHAWAVAUATSH(HHuD{0   EtHuHMH8LcLmHHMLsLHuLSP} tL`IXALuҸ   7H]Ht,H}w  Iƾ   HLT  Hs' H LP01H([A\A]A^A_]UHAWAVATSIII7_   IFL` IWLLtIM;|$t2LH3L0   H3HI+T$H1LHI;\$uL[A\A^A_]UHAWAVATSHHHuD{0Et6H8LuL%& HHEH{LSHI$Hp  H`AuH[A\A^A_]UHAWAVAUATSPIHM  IL  En0AEt/I8LH3L  uH`AuL)HDiL}  DH[A\A]A^A_]UHSHHHHHY HUHcHIHHT8HUH|@Hu   HtHc@HHCHCH[]UHAVSHH{	  IH{~  1HL[A^]\UHSPHH{  H,TH{H[]D  UHAWAVSPHAIH6% H    P(DpHXIO HIG H[A^A_]UHAWAVAUATSHAH}H_ L5$ IH
  I1H   E11ED9kuIH{
  HL  t
IHHuWMHt
I$HHEHEHH HMH{H	I   IHP0EH]HyEH[A\A]A^A_]UHAWAVAUATSHHUIH}Dw0Et?HEHX@L=# L%+" IHx  IHLH
  H`AuI]A   I;]tJH31L}LLuLt+HI;]t$H3HI+UHLLnHuE1DH[A\A]A^A_]UHAWAVAUATSHIAIHGLh 1L!LHEI]I;]tRIcH@HMd@M   HHELHuAHtHI+uHPLtKHI;]uHEH2  H[A\A]A^A_]UHAWAVAUATSHLEHMAԉIMu M}   M9tM   At7u|LL  H}D	uLL  H}ȉ  1IM9uH[A\A]A^A_]UHAWAVAUATSH(MIIHHtID$I;$uH([A\A]A^A_]HAH@ HEL=! IH5H  HLMHM  IE HEIEHEIHL  IH5  H  IHMH1H  IH5  H  IIcMIcUHL5C LH4ʺH  IH5  H  IIcMIcUHLH4ʺH  IH5s  H  IHMH1H  IH5F  H  HEH0HHULLEH([A\A]A^A_]    UHAWAVAUATSHLEIHUIHMeL5M  IH5    IHL  IH5  H  M} M9   HEH@IcH<n  IIHH5    IHL  ILP0IHH5W    IHHu  IHH5!    IM9bH[A\A]A^A_]UHAWAVAUATSH(IIHHt	I@I; uH([A\A]A^A_]LELMHAHMH@ HEL= IH5  H  IHEMfMt$IHL  IH5<  H  IHMH1H  IH5  H  M$$M9t?L-  I$H IH0H  IHL  IM9uIsH  IH53  H  HEH0HHUHMLEH([A\A]A^A_]cUHAWAVAUATSHMIIIHHtID$I;$uH[A\A]A^A_]IGH@ HEH H H5  HLM  LLH$   I7HHULLEH[A\A]A^A_]UHAWAVAUATSPHIHM.MfIN(HML= IHH  IH5x  H  II4$H  IH5+  H  IIcNIcVHL% LH4ʺH  IH5  H  IIu H  A~H tIH5  H  IH5  H  IIcN4IcV0HII4̺H  IH5s  H  IHMH1H  IH  H5Y  HH[A\A]A^A_]UHAWAVAUATSHIΉIIH H  H
  A      LLP  WIvHMH$Hr L     E1L	  A   u"LcEILLLH6 AT ADH[A\A]A^A_]Ã	uHHHL  UHH H H
     P     ]UHAWAVAUATSHIIԉIH H #H
     LLP  A     It$HMH$H/ L     E1L	  A     uH   H  HcHIE1  H H I<$I
  LHLe  AEtd     It$LLfg  AC     It$LL7q  A"  H HH=k  E11Q   Hc H   I|$I
  LLHe  I$Hh  H  LHE1   H
 ~XH H
  9It$LLsq  A~H H H
  H H H
     LLP  Gu#H5B  +H
r     LLP  H I|$
  HLLo  ADH[A\A]A^A_] @@XuHHH_Y  UHH H H
&     P     ]UHAWAVAUATSH  IHAIHO H HEH H AH
=     LHP     lH{
  HL   	   L:  HsLL袳     Ht)AH L,$LLHHEI8  H H H;EuH  [A\A]A^A_]  UHAWAVAUATSHHIHIL= I   H{
  LLH?  A   H   IHpHZ LI   ugLLL@  uUIGH@ HELLtLA( tbLC      -   CH
     LHP  A   DHH[A\A]A^A_]LC      ,   LG  AG(   L}HCHE HC HE HMHY1HEH;YA      1HEH]H;蠯  IAH H1A   LHuL0  H   HLHuHUP  AEuw} t5LH]HLu  AEuWHHuHHEHEH]HAHAH H L   E1HEH;X22AHAAHAA   H H L   HE@(    H H L   H]H
  11LHA׋HHu H H   H}HHR H    ELLn  AELuEH tHHMHy`  HHh  D  LHE1UHAWAVAUATSH  IIAIH6 HH]H H A#H
f     LLP  A     I
  LLLH2<  HA   H   HP H HqH LH      LHHz=     LLID$H@ HH      蟶  HHLMt0Lx( LHtmIWH-   Hж  LIWH,   H賶  HH  H  HH;]a  DH  [A\A]A^A_]F(   AILLLDILR?  IM   LLHLz  E1HH;X       H;H贫  IALLL
r  E1u)LHCH H I}`  A   AHAHm H L   E1AE(    LHk  AEH  HH H Lh  D  LHAE1  UHAWAVAUATSH  IIAIHF  HH]H H A#H
     LLP  A     I}
  LLLHB9  HMA   H   H] H HqH LH      LHH:     LLIFH@ HH   >   譳  HLbtA~( tbIUL-   IUL,   L߳  HL@  H  HH;]  DH  [A\A]A^A_]HAF(   AILLE   Mu H= H HLH
         Iv    HH&?IL9tH=AIMLuHLLn  L5:  M6MHL      HIAHLLHL,r  HL5  M6MtHHH1ɅuHL޲     AHAHH H L   L"L5h  M6MLHLAF(    LH[h  LAELH
 H Ivh  E1s  UHAWAVAUATSPII׉I,Hj
 H H
     LLP        1L'I$HHH+H   LLIr  uIwH

 H IL   HH H1L8	  AAu2HL   H   L  H=  1LHAջ   A9AI$HHH+H1LLq  ALL{e  AD߉H[A\A]A^A_]UHAWAVSPHIL=: Iu5Hz
  LHH4  IHh  HH  t#   H
     LP     1LH1H[A^A_]UHAWAVAUATSH(IIH H  Hz
  LLHR4  IM
  HIvH
 L      L}IFH@ HEH11  HEAF0   I8ELuM&H11  IM<$IM;|$t:HEH8A7  HLp  H8  H}LHAօtiHH}HuLp  uQLuI`EpHH}Huh  1H
g     LP     H([A\A]A^A_]AE HAM HL   H}HH   UHAWAVAUATSH  HAIH|  H HEHǅ    L5	 IA H
8     LP  A      HzH
  LHHl2  A   H   IHpH6 LI      LHL3  uvH      
  HHLt A~( tlHHPH-   HHPH,   H*  LH菭  H[  H H;E  DH  [A\A]A^A_]AF(   IFH@ H8L4HE1AA      1  HHPHLLH,4  IM   A1HIIw HH5   AHAH
 H L   1E~MAHS  HJT LHHL3  IIFM   mAA3E1A   (L   LL踫  LL  A   Hί  LAD$(    LHDa  AEtH5ALEH t
HHy`  HHh  HvLHE1  UHAWAVAUATSH  IIAIH5  HH]H H A#H
     LLP  A     I}
  LLLH1/  HMA   H   HL H HqH LH      LHHv0     LLIFH@ HH      蜩  HLQtA~( tbIUL-   IUL,   LΩ  HL/  H  HH;]  DH  [A\A]A^A_]HAF(   AILLE   Mu H, H HLH         Iv HH87IL9tH3AIMLuHLLb  L5.  M6MHL      H
IAHLLHL h  HL5  M6MtHHH1ɅuHLҨ     AHAHH H L   L"L5\  M6MLHLAF(    LHO^  LAEL$H H Ivh  E1g  UHAWAVAUATSH  IHIH  H HEЃL=N I   H{
  LLH+  A   H   IHpH LH      LLHH-     HH@H@ HHsIHj LI   uqIG HH   &   L  LHtfHx( t~HSH-   WH
     LHP  A   H  H H;E1  DH  [A\A]A^A_]HSH,   H-  LH蒦  @(   HHHH5HtyH7IAHLHLUe  HtHHHE1uHL  A   AHA*HW H L   HLߥ  A   C(    LLD[  AEH H HsLh  E1Ӽ  UHAWAVAUATSH(Ỉ   H}M|$1ۃuH  H Hz
  HL5  I11  HEIHuL  
IH}  IM   AD$DtuM I HtILH  tILp  L  H}HuHAՅtH}H   AI   6H  H H
     P     IH}Huh  1ۉH([A\A]A^A_]UHAWAVATSIIԉIH  H |/كt(L  ILLL[A\A^A_]:  H
     LLP     [A\A^A_]UHAWAVSPHIL=
  IuIHz
  LHH'  H   Ht>IHh  H9  LH1H
     LP     H[A^A_]Hȃ}'UHH  H H
a     P     ]Hc΃HLHIH  UHAWAVAUATSH  IIAHH  H HEH  H A#H
     HLP  A   U  I}
  HLH&  IA   M+  H  H It$Hc  H     HID$HX H   /   ?  AA   HILLHLDLz*  HH   LI} )IǋLH
IHe  M;ut
I61L*H  H Hh  LVHHE1*H  H	   H臠  HHC  H  H H;EuDH  [A\A]A^A_]X  UHAWAVAUATSH  HIAIH  H HEAOH;  H r#H
     LLP  A     I|$
  LHH$  HA   Hc  H  H HqHHy  L   6  LHH&    HHH@HX H   3   5  A  IIL$HY  H H  LIHL     HI] I?H3裉  t:HHHtf{( H   IT$H-   WH;  HL   LH虞  HLU  H  H HP0  IT$H,   H跞  HH     F(   AE HLLHL]  IA   MtOHHLE1uHL趞  A   AHAH  H L   AE HAM H  H L   C(    LMHDDT  AEu!H  H HHqLh  E1H  H H;EuDH  [A\A]A^A_]<  UHAWAVAUATSHIIԉIL55  IH
     LLP  1  It$HMH$H  L     E1L	        IIt$HMH$H  L     E1L	     uЃth   II|$ 
  LLH=!  H   uHwtH
  HcHd  IT$(IL$0LHS  VuHw9Hx  HcH   IT$ LLV   IH=  
IH=@  1Q    H[A\A]A^A_]Ã   IT$ LLV  ҃   LLWW  IHh  H=@    LH돃   LHS  y   IT$(LHiU  YuvIT$(IL$0LH4S  8IH
}  IH
  IH
  IH
  IH
  IH
  IH
  Fe*EUHAWAVATSHIIL%i  I$uyHrHMH$H  LW     E1L	     u_u܃tf      11LIHHH+H   LL[  H
     LP     H[A\A^A_]IHHH+H11LLZ     LLN  IHHH+H1LLZ  1LLzN  I$H=|  11Q UHAWAVAUATSH  IIAIH  HH]Hǅ    H  H A#H
_     LLP  A      I|$
  LLLH  LIA   M   H  H IvHL  H   u}HHL  ugH   <   &  HL׻tA~( t`IT$L-   IT$L,   LR  HL跘  H  HH;];  DH  [A\A]A^A_]AF(   AIIFH@ H8Hǅ    H  E~fAI$IHLHLN  IMtIw HH!AHAHG  H L   AF(    1HHM  AEtH! L=  t
II~`  ILh  HQHHAE1袮  UHAWAVAUATSH  IIAIH  HH]H  H A#H
}     LLP  A     I}
  LLLH  HMA   H   H&  H HqH  LH      LHHP     LLIFH@ HH      v  HL+tA~( tbIUL-   IUL,   L訕  HL	  H  HH;]  DH  [A\A]A^A_]HAF(   AILLE   Mu H  H HLH          Iv 1HH IL9tHAIMLuHLL:  L5  M6MHL      HIAHLLHLS  HL5  M6MtHHHy1ɅuHL誔     AHAHH  H L   L"L54  M6MLHLAF(    LH'J  LAEL"Hu  H Ivh  E1?  UHAWAVATSIIԉI)H9  H H
o     LLP     &I1~I$ILL]  t[A\A^A_]UHAWAVAUATSH8  IIIH<  HH]ЃH
  H   HI~
  LLHX  Iĸ   M   HIt$H  L      LLL     ID$HX H   A     LL衵tIA|$(    IVH-   ;H
K     LLP     >HN  H2IVH,   H  LHS  H  H   H;]  H8  [A\A]A^A_]AD$(   IFH IF H IF(H HH;,HLs1L;s5  LI>
  IAH  L1A   LHLA0  H   HLHHP  u{ LLt0HH$LLLHIL  ueHHu  H H   I1HL;p%XAHA   EH6  H L      +HH  H H߉   1ɃEʉAD$(    LLnF  LIGI;GtH  H I|$`  L5  IH   LL
  11LHHH	I   A$HA$H  HIL   HH	I   t/ u&ILh  LArLHADEL   3豦  UHAWAVAUATSH8  HIIH!  H HEЃL5  IP  I}
  LHH@  A   HG  IHpMIH  L   "  LHL    HID$H@ HH   B   Í  H  H Iu HHLx     HLLLL  H   LLHIHLR  HLtsA|$( I   ME   B   -   L肏  H
     LLP  A   H  H H;E   DH8  [A\A]A^A_]ME   B   ,   H  AD$(   LHH;HE1HH;C(  MeH8I~  HH  H 1A   LLH0  HtiIM(HH$LHLIL  Aǅ   AH
$  tfHHHy`  H
  DBH%ǅ   H  IIH   Lǅ   H
  HL   LI
  11LHA׋H
IE H   DH@(       A9LHA  ADEA   uHA   H  H Hh  HzLHE̢  UHAWAVAUATSHh  IIL=?  M?L}ЃL%  I$  HzH
  LLHV  Hø   H  I$HsH  L     LLH  s  LHCH@ HHLpI$H  LL   3  IF HH LxL;xLtDM7I<$I6t  I$H;A<  H0LZ  IHH L;xuLILA~0轗  HA~0  HA~0 )  I^8E1LHHxH+8H݌  HHL0L;p   D1ɉHH8A6q  II<$Iu Yu  AAtoI<$Dp  LHtTHDҎ  IHL;puEDxDHHA  HD菎   H   D5  H`ALAF0H@HID8H9LHHCH;uuH,  H{  HL@   C      H.    H
_     LP     L;}~  Hh  [A\A]A^A_]HHHHH HxH+xH   IL3HCII9t/M&I$I;\$t3L  HI;\$uHH LI$HpH+pHL  IL胋  H   C   Ȇ  HLLrH^  HI   LA~( H  H  HHPL-   Lˆ  H   M   C      L      H  HIM   C      L  H  HI^HL@   C      HL  LMIǅ   Hd  HId  LA  HHPL,   L  HLP  HH8
HpIT$1I;T$Ld  HHI$ID$H9   HxLHL0ǅE1AHHHHIcH@HII|@HA   LHt+DxD9At/DHIHH H9xE1AHH LI$IHxIcH)HH9C  HLpIcM<A?|HxLjM  IHxHHIHL0L,AE IHL;p   HHALaMc.H@IJH0n  HH3KHHIIOIGH+YHHcH<HtQqH  H    IGXLXy  HLLHHLpL  LHxuH1HH;PHm  AF(    LLHw9  t2u.H  H Lh  Hp/LHAHph
ǅ   H  HILHن  M4$I>ˆ  IM;u uL^  x%  UHAWAVAUATSHH  MDIHIIH  H HELjt(H  )   H]H辁  LH~    LHLH  ILLLLIMr  LH  H H  L5  M6  IG8H1HHHE1MMH-  H HH3Lx      LMO<dIKD>8HLh  IL5?  M6H~UH  H H
  HHQHH9HHHl    L  u  LL迉  MHLt0I6L?    I>L,    I`HàuHL(H  H LHI|@H  H
  HIH` L5(  M6LHHXH  H I7IO11A   ML	  HL   HT  IIE I7L  1ҹ"   L  IE IwLh     "   LuLL~  LL  H  H H     L5S  M6      L}LH~  OHH   L}L~  2LLǝRHH
   L}L~  L5  M6HLǂ  HH肝HL~     L;uuGHH  [A\A]A^A_]H   H]HC~  LH~  HL&Ε  UHAWAVSPIHIH  Hr::u2C:u)L=  IL  ILH  MtLHL  IH[A^A_]UHAWAVAUATSH   EIHL-  Mm LmA uH5  HH=1  11DQ gHLHH IHLc1Ht&H
  HHH1LLE	  H  H  H H  L;muHH   [A\A]A^A_]|  UHAWAVAUATSPHIILt)L|     ?   )   L~  A     HE  H H
  LLH   IA   M   H  HHIt$H  L      ID$HHH;u}I|$  t$AD$,   11E1LLZ8  AD$,    HI4$L9E11ҹ"   LM  HI4$11L   LL褚HL  M$      '   L|  DH[A\A]A^A_]UHAWAVAUATSH   IIIH  HH]L  Hr/A$:u%AD$:uLLL>I   HLLL  AHLIEuiMudH  HLL  HL  HH5=  L  HLL  HL蝚IH,  H H  H  HMu   1ҹ#   HM{  H;]uLH   [A\A]A^A_]趑  UHSPHH{  tC,   11E1HH6  C,    H[]UHAWAVAUATSPIIII]E1;   L%p  I$H   IE H   9
I$H   I$Iu H1ҹ"   LIM  I$Iu IM1A   L	  HLDHIu 1ҹ"   LIM  DH[A\A]A^A_]UHAWAVATSHIIL%  I$H  I$L  A   Hx  tI$HpH  E1I$H52  H  I$HL  D[A\A^A_]UHAWAVAUATSHLEHMHIHCLp I>LLt  I1M   AE LHL   IAE HAM    9H  H L   1M   Mo I>Iub  t1I>Iu\h  HMHHLHH]H質tLD   H]HLw  LHw  AAA1H(  H L   1H[A\A]A^A_]UHSPHH H{  t){, u#C,   H  E1HHp3  C,    H[]UHAWAVAUATSH(LIAIA   LEL}HFHuLx DDz  HEI?9  HEE~ZAH[  H I} 
  II?Lb  xyH}Љ8|  IUH}LH.=  txAIAH}Hu耫xEHMHtH}Oz  HE_IU 	   Hv  LHjv  ?   HLH     H|u  H}H;v  H}y  H} :  1H([A\A]A^A_]UHAWAVAUATSHMIIHL}HK  H 1H8	  uhL-1  IE H   HMH   HELx  IL
x  IIE H  H=  1LLUHHUи   "H  LeLmLuЃttH[A\A]A^A_]HH@  HHuHUHLEMH[A\A]A^A_]    UHAWAVAUATSH(LMLEIIHIHCLh H7  H H  L        LmAI?L}i  IHLLHL  IŋH   9H  H H   AHAH  HHL   Hع   LH1  H H[  MLH     HC LuI>Hp]     LeI4$L_LLLHLuMEuXH}LHL&EI4$HEL+`I1LDu"   LHs  LLus  E   HH  H H   MI   IH]HLr  LHL%s  HHm  H H      H([A\A]A^A_]UHSPHHH H{  t&{, u C,   H  HI.  C,    H[]UHAWAVAUATSHH  IIIHZ  HH]L=t)LH        )   L]s  A     H  HI}
  LLHMA   HG  HHpH  LH   %  HCH@ HH=  H IuHHLx    HH	  HIuHH$Hx  Lъ     E1L	    HI} 
  LLH~HH{  H  H HsHD  L   U  HCH@ HHm  H Iu(HHLx    H@  H Iu0HH$H  L     E1L	    HcHH
}  | tME0      (     ;{  H11@HHH HHH HH   HH
f  HHH8
  HH[  AHHH9
  HH~[  E*  +  HDHHHHQcA      HH   HHHHÉHr  r  A      HAHY  H I} 
  HLHPHHL訒IMj  IELL8HcHH  LHHcHLHHHXHcHLHHcHLHH HHH(IL>~  I?H3葅  tH{L#~  LL  E1   HLA      ME(      $   Loo  H  HH;]   DHH  [A\A]A^A_]ME(         (HHL A            Lo  H넾      %   LL@n  HH  H H  >K  UHAWAVAUATSH   IIL-  Mm LmA?  Lǅ    MgMw:   L(  HH$  HL9,  L%  I$H=t    IAE I$Iw)LL   HH5P  11L   HIvL 	  HH5,  11L   H   HL8	  AAE HAM    9HL   L-  Mm    L5*  IHH  LHÅuIHL  IH    IvH  H1IvH  H   H8	  AI~H   9	H   D   9uL5r  IL  HH@  HHHAԅ   IH      IL   H@  HIWH5  H
     AtGAH   L5  IH  IL   H@  HIWH5v  ;L5  IHH  IL   H@  HIWH5  H
y  E11A֋L;muH   [A\A]A^A_]"  UHAWAVAUATSHX  IHAIL-  Mm LmLqt)LΒ     (   )   Lj  A     LL%  I$H{
  HLLLHkA   H  I$HpH4  LH     HH@H@ HI$HsHHLx  p  HLk  HHL H~`H  H H
  HHQHH9ILLT  S  H߉9p  u   (      F  HILAEL-  Mm A  ID$I+$HLL%O  I$HH;
  HLLH8HHHXH  HHHHHDqHzHHw     u  HLAL5    HAHIH;
  IHL~  L?  HLy  L  LHLLIIM  H+  H IuLH       IEL` H  H HsLHHx  \  H;  HH  H    P(IM.L11%IM~HHxL)v  M,$HL A$ILsH~?HC  H H
  HHQHH9ILLoQ  yk     LH{Hu  HHH@L-G  Mm L%  MIHH>E1uHH,A   Hh  Hzs  I   A      (      HMe  Hh  L-  Mm mHL@   (      HGf  Lfh  =L   (   %   L#f  H>h  HϿ  H H  L;m   DHX  [A\A]A^A_]þ   (      HM.e  LHg  Hpr  HLHM  H H  A      (   *   HLd  듾   (   "   HMd  pLC   (   $   Le  Pz  UHAWAVAUATSH  IIIH  HH]Lt#L        )   Ld    He  HI>
  H5z  HRz  LM~MEHI
  LLH   H  HHpH  LI   v  IFL` HIwHHLx  H  LLHIw0HHLx    HIwHH$L%  LG{     E1LL	    LMIDEu6HcHB|  tB|  uMF      +   L9c  HIv8HH$L%g  Lz     E1LL	  Q  EMu6HcHB|  tB|  uMF8      +   Lb  MHI~ 
  LHH%IM  HIt$H  L     ID$LMLh HIv(HHLx    LHI~@
  LHHH]  HHpHj  LI   ;  IELxH@ HpHIvHHHLx    ;&  ;  H?  H I>
  HLH7HhHHH`H  DH`HXHXHHHHCHcHBD8CHcHBD8C11)HC HxHC(HcHBD8C0HcHBD8C411HC@DkHHH HHH HL{ H   HH
  HHH8
  HHIJ  AHHH9
  HH$J  E    LDHHHHQg   LN   LOIHLϒHXCLa  { O     LHXH@@HHpL8H   HH
  HHH8
  HH!I  AHHH9
  LH I  E    HDHHHHQg   HH   HIIHxL蝑HXC8Lr`  {8 P     LHH{H`l  H;HxH0s  tHxHxH`Yl  HHxH`H?l  LH#E1uHhHA   Hc  H H  Hĥ  HD   H  H   MF(MFH      $   LR]           %   LLh\  jHMF(1HHHL          MFH         L\  HhH#H{  H H  Hܤ  H   H;]uHĈ  [A\A]A^A_]!r  UHAWAVAUATSH  MII׉IH  H HELo}t#Lۃ     )   )   L[    HLHL5Ƶ  II?
  HLHHHH}HH$  LHCHLhAE H8LAt$h  D   A   E)ILII?
  LLHHIMtrIIt$LH     uVHH8Li  I|$Hi  LIAuHHOtIH  1(HH,IH     L-  Mm L9M   )   %   LZ  IH     H
  H	H;MuH  [A\A]A^A_]o  UHAWAVSPHIIHL?   HtHxH1      &   LMoY     H[A^A_]UHAWAVAUATSH   IIIH  HH]Lo  Hr/A$:u%AD$:uLLL;I   HLLLpAHLIEuiMudH  HLL  HL  HH5  L  HLL  HL蚀IH  H H  H	  HMu   1ҹ#   HMX  H;]uLH   [A\A]A^A_]2n  UHAWAVAUATSHHIIL--  IE L
  LHH_IM  IE 11  IA4$H  H  HcHMl$H˱  H Hp  H=:    LLHӅ5  H  H Hp  I|$  LLHӅ  Ha  H Hp  IM H9  LLHӅ  Iu IU(1L   H  H
  H	LLHp    H  H Hp  IMHcH HL%  LH<ʾ  LLHӅC  H  H Hp  IMH9  LLHӅ
  IuIU(   L?  H  H
O  H	LLHp    H.  H Hp  IMHcH HII<̾  LLH        &   LMV       IM H=  1Q H  H LLh  1a  Ml$IEH@HEH  H Hp  H=K}    LLHӅ  HT  H Hp  I|$  LLHӅ  IE IML!I$HMH

  H	Hp  H8  LLHӅ{  Iu HEHP   L  HY  H
  H	LLHp  8  L;eI$H  H Lp  HH9  LLHAՅ  H3HS1L*  H  H
:  H	LLHp    IL9euRMl$H  H Hp  H={    LLHӅo  A}H t6Hȭ  H Hp  H=߁    LLHӅ2  H  H Hp  I|$  LLHӅ  H^  H Hp  IM H9  LLHӅ  Iu IU 1L  H  H

  H	LLHp    H  H Hp  IMHcH HL%  LH<ʾ  LLHӅ@  H  H Hp  IMH9  LLHӅ
  IuIU    L<  H  H
L  H	LLHp    Iu IU@1L   H  H
  H	LLHp    H  H Hp  IM0HcH HII<̾  LLHӅJ  H  H Hp  IM(H9  LLHӅ  Iu(IU@   LF  H   H
V  H	LLHp     Ml$H3  H Hp  H=y    LLHӅ   H  H Hp  I|$  LLHӅumIE HHIE H;Xt4HH
  H	Lp  H8  LLHAԅt"H  H IULLp  AHA   HM  H L   H[A\A]A^A_]YUHAWAVAUATSPIH}H׉HLcH  H 11  IL+M9tPIFH@ HAu H5  H
  H	Lp  H8  H}LHAօuIM9uL%AAA1Ho  H L   1H[A\A]A^A_]UHAWAVAUATSH(HIH}L=0  I11  HEIH{`Hu  
IH}  IMtw   tuMv I ILL  tILp  L  H}HuHAԅtH}H   I   IH}Huh  1ۉH([A\A]A^A_]UHAWAVAUATSPIHH]L-=  IE H
  HLHIM   IE 11  IID$HL`L9t7HIM Lp  Hx  H}LHAׅuHI9uIE H}Lh  1 AHA
IE L      H[A\A]A^A_]UHAWAVATSHIL%a  I$H
  ILHLHt&I$Hh  Hx  LH1      &   LML     [A\A^A_]UHAWAVAUATSH(AIHI?HGH+H~H]1[  IEt<H   H  H Lh  H= {    HHA1   HEL}1X  IID$LeL8L`IM9t)IH@HLhL9tH3L\  HI9uHEIM~L9A    t*HLeHsLI9tHuDLX  L}LeHEH HLpEtL9t)H;
HI9uL9tH;"HI9uLmI?vZ  EH([A\A]A^A_]UHAWAVSPAHIH;Y  8 tLHDH[A^A_]?EH[A^A_]UHAVSHIHHEHH$      uuHL1H[A^]UHAWAVAUATSH(IIH  HHUHMx     Eԅ   L}E1A    ~iE1HEL%~  H
o  HH0HEH$   E1LLLy  	  uEHcEHFl MԍAEHEHHEȃHED(HH5@y  1LP  A   DH([A\A]A^A_]UHAVSHIHHEHH   uuHL~1H[A^]UHAWAVAUATSH(IHH  H HEHu  H 11  IMt$ Mt_LeAvHL   ulHEHEIFHEH-  H Lp     Hu  HLHAՅu,M6MuH  H HLh  HW  H 10AHAHâ  H L      H#  H H;EuH([A\A]A^A_]p^  UHAWAVAUATSPHUЉIHn  H 11  IL5_     t/HG  H Lp  I>  LLHAՅuAFIuHEL81&AHA   H  H L   H[A\A]A^A_]UHAWAVAUATSPHIIAF,@   L%  I$1H8	  Ãu4I$L   L   L  H=u  1ALHAAF,    H[A\A]A^A_]UHAVSHIH4  H    P(@    HXI   HI   1[A^]UHAWAVAUATSPHuL   L%  I$H
  IMt}1IL{ ugI$H{
  HL\  uJML3t	Mu L3HEL   H{H
I$   I$HP0MLuL3Mu1H[A\A]A^A_]UHAWAVAUATSPIIL-&  IE 11  II$   HHIM t6HSLLp  tAHA   IE L   LLh  1ۉH[A\A]A^A_]UHSPHH1H{  t*{, u$C,   1E1H   HC,    H[]UHAWAVAUATSHXMIHUIIH<  H I>  HAv,HULt!H5  J1  HR  L}H]L=  MLLeIDLmHMHMMn MLu  AF,AE  II11  IIIUH]HLh    IHLHUp  o  IHLHUp  T  H} tIHLHUp  2  H} tIHLHUp    IHLHUHMx     IuHU   H0	     HEH   LHtH	I   IH]H@  IA$IH  IHLHU   umHUHLHHDHMHHEHMHDHULAHAILH   HLuMm MEIH}H]`AHAIL   H}HH]	I   J1IIH   19IH]H}H	I   HIH   LHX[A\A]A^A_]UHAWAVAUATSPIHHI H{  A  {, 7  C,   H  E1LHHIC,    MtvMn H  H HsH  L   tqH  HHL@     LH?  LL9@  AAA1H   H   H9p     L;?  LL?  pHCH@ I} H0)  tLUI} ;3  Hþ   LH>  LL?  L=	  IHP0AAA1IL   1H[A\A]A^A_]UHAWAVAUATSH(AIHI      ML-  IE H=h    IEIE H  t	H=lm  H=im  ILeIE }  AA$HE IE H@  HE M   M$$M   IE 11  IIE IT$HLh  uIE HLLp  uIE HLHUp  uIE HLHUp  uIE HLHUHMx  uIE uHUȹ   H0	  AHA;IE L   )AHA
IE L   H}H
IE    H}H
IE    IE HH]Hh  H
IE H   H([A\A]A^A_]UHAWAVSPIM~M+~IH  H DP(HIcHcT      LsAHCHCHH[A^A_]UHAWAVSPIIHIIVH9t!HpHc9I_H<H>HHH9uH[A^A_]UHAVSHHDIHCH{HpH+pHIV   L[A^]UHAWAVAUATSPIIHE1L9tJMwI)IL-  I<$HtH
IE    HHI$I I9uADH[A\A]A^A_]UHAWAVATSIMtjI~LL+It;I^IL%0  H;HtH
I$   HIuI~f%  H  H L[A\A^A_]`0[A\A^A_]UHHtH]_]UHAVSIH   L9t$H{Iv!%  1tHL[A^]   [A^]UHAWAVAUATSPHGHXL`L9tFHFHELnLwL;I7H}&  I6HIT L諢1tIHI9uʸ   H[A\A]A^A_]UHAWAVATSLwI^E1I;^t$LgE1H;HI4$IuAI;^uD[A\A^A_]UHAWAVATSIIIE1I;^t,E1IGIOH@HcH<H4$AHI;^uD[A\A^A_]UHAWAVAUATSPIIIII;_tCL)ID$It$HxHcINIcE H<H4H蔡1tHII;_u   H[A\A]A^A_]UHAWAVAUATSHIHUIH;|HƔ  HH=k  1Q L{LL#  IM;wtpH]I611LmLLyLIMt_M+wIIcLuIFH<Ht QHK  H    IFL$A$   !   LL7  1
   LC8  H[A\A]A^A_]UHAVSII^H"  1H;CtH+CINHH[A^]UHAWAVAUATSHIIHGHWHUHPHUHYLyHKHMLkHLHp   1t.L+mMHMI)HEII)ILLLn   H[A\A]A^A_]UHAWAVSPII^H{H+{H%  HMIHsHSLL.uLE1LH[A^A_]UHAVSHH{IHCH{HpH+pHIVL[A^]UHAWAVSPIIHIHI+Vt$1Hc<1I_H<HXH<sHHHuH[A^A_]UHAWAVATSIIHH{(  AHLL   AD[A\A^A_]UHAWAVAUATSPIILgLeLoAGAF@Hޑ  H P(IFHcHM  LL`   L;`tDMvM   I<$Iu IA_LM LMv\HEIL;`uÉH[A\A]A^A_]UHAWAVAUATSPHIIIIvA{I|$(  HAD}MlAD LLLIL)   DL譩L.D)H[A\A]A^A_]UHAWAVAUATSH(HGHEH_LbLq{LhL~L;htHuI} H}HHEHH]LLaHIIT AA H}HuLM\HH]HMHD|I IIL|L;iuL|HuAGA})H([A\A]A^A_]UHAWAVAUATSHHIW)E)E   E)EHE    EI}LuL&  ALeLLLA|H  H P(HLHLL5H HHH[A\A]A^A_]UHAWAVATSH@II|$W)E)E   E)EHE    ELuL%  H1  H HX(L}LLLxHLHLLH HH@[A\A^A_]UHAWAVAUATSHIAIHȎ  H AH
@S        LLP  OIvHMH$L%  L#S     E1LL	     uHcMHLDLBT!ÉH[A\A]A^A_]UHAWAVAUATSH   I׉IL-|  Mm LmH  H H
d     LLP     [MgH  LL      u9Md$ H      m1  ILLLLt/  L;muH   [A\A]A^A_]SI  UHAWAVAUATSPI   HZL5J  IH  LH      HC LhI11  HEM}IM;}   IIH8  HILHuHp  tċHIH   H}ЋH*I   H  H H
Bc     LP     H[A\A]A^A_]ILHuh  1UHAWAVAUATSH   HIL-z  Mm LmЃubLA      	   L/  HsLL5(  Ht}H,IHSHLL&  t4LeNH  H H
vb     LHP  A   $H  H Hh  Lp%  LHE1L;muDH   [A\A]A^A_]=G  UHAWAVSPHu]LrL=<  IH
  HL      uUIF ILh  H@HxH+xH  HHA1$H  H H
a     HP     H[A^A_]UHAWAVAUATSHIAHA-H  H H
ba     HLP  A   N  MnHc  H H1  HL   A     EA   H]IE HELhD1  HEAL%  |>II$I>
  HLHE     IH}Љ^6  AEIuI+uHH]H6  IH2  LL  H}HL&IL1  I$Hh  L\#  H}HH^  H HLh  E1*A            H}I.  H}1  DH[A\A]A^A_]UHAWAVAUATSPIL-  IE uuLrLzH  LL      ukIE H  LL   uPIE Lh  H  I~ Iw LHA1H
__     LP     H[A\A]A^A_]UHAWAVAUATSH  I׉IL-v  Mm LmHcHiVUUUHH?H ȃd  @)W  MgH  H H  LL        ؃E  LMd$ I|$HHiVUUUHH?H w  ILLhHH      &+     HHALhL+hLiI(IwIWHH跗IM   LLF  I;F   IHLHޘH   IE  IIHĆ  H Hh  H   HH1{H  H H
]     LLP     ]Hn  H LLh  1CIWH   Hx*  HH*  H   L-t  Mm L;muH  [A\A]A^A_]A  UHAWAVAUATSHIԉI,Hօ  H H
\     LLP       Ml$L5  IHs  LL         IE HEHHIIM2HI<$
  IHL  xzHHMHIH4   HM11  HEȅ~pIE I<$
  MIH}LW  xpIIHHUHRHLHup  Mt`         LM]*  HuIE Lh  1ɉH[A\A]A^A_]þ         LM*  MH}ȋHIE    UHAWAVAUATSH  IIL=r  M?L}ЍFHHiVUUUII?H B3I9t,H  H H
![     LLP       HH      x'  D  HHÃ   Mt$LDyI6IVLMHIM  HL  H;C   AIAMHsLA   MLI LhHH[AH3ILHH    IE HIIAAL=;q  M?MHHH
  H	Hh  H  LH1BIU L   LI&  LL	'  L1Hx     L=p  M?L;}u/H  [A\A]A^A_]L=p  M?rH=  UHAWAVAUATSHI   LrH  HH  LL      L}IF LpLuL`H11  IMnM;ntdIE HHp  H8  MLeLLHHb  urHILLp  u\IIHEL;hMuHH}Lh  1$H  H H
W     LP     H[A\A]A^A_]AHAHL   UHAWAVATSHupLrL%  I$H  HL      ugIF Hx  II$L  II$LP0I$HLh  1$HF  H H
V     HP     [A\A^A_]UHAWAVAUATSHIAHA-H  H H
V     HLP  A   	  MoH  H H  HL   A      H]IE HELhEfD'  HEAL5y  |9III?
  HLH  xaIH}Љ+  AELH]H
  H}HHIH}'  ILh  L  H}HAE1*A      *      H}I$  H}2'  DH[A\A]A^A_]UHAWAVAUATSPIuuLzL-~  IE Hd~  LL      unM IRIľ   L誫11LLIE Lh  L;LHA$H&~  H H
T     LP     H[A\A]A^A_]UHAWAVAUATSH(IAIA-H}  H H
U     LLP  A   E  I]H}  H Hn}  LH   A     L%u}  AF   L}H{ #LHA   LxHEIGHEIAI} 
  ILeHI}
  HLL  MIHEI9   I?HH]HdLLH  H9   H|  HAIAMy*LT  A      -   	   L4"   HEHh  H}^  H}HE1DH([A\A]A^A_]þ   -         -      H}LE!  H}cUHAWAVAUATSHHI{  LsL-{  IE H{  LL   A     IF HELpIE H{
  HLH
  LIL;n=  IE xF  LH+FHHHMHIHL5c{  LH H.{  ILH   "  L{ H]HCHHH+HHIGHPH+PH|  HDIvLuHLHI"   IGHpHPLL   IHEHPHLL   Hz  H Hh  Lw  LHE1aHwz  H H
Q     LHP  A   7A      @      A      @      LI  DH[A\A]A^A_]Lq<  A      @      L  L{UHAWAVAUATSH   I׉IH2h  H HEЃ-Hy  H H
DQ     LLP  A     MoHvy  H HDy  LL   A      C   I} bIH   A     L-y  IM |9II?
  IWLHH tRIM IHh  L  LHE1CLP  A      A   	   LH  HL0  LA   Hf  H H;EuDH   [A\A]A^A_]74  UHAWAVAUATSH(  IՉHf  H HEЃ*H x  H H
P     LP  A     M}Hw  H Hw  IL   A     LIG HLpIEH{  ILH|nHcH1Hqw  H I| 
  ILLh  I;F  HHHHHc  HH;|H  HHH@HxH+xH   +  HH
HMfHv  HIu 1H=N  Q M;ft9LL+Iu L  LcyHHLHuH   D     L5Lv  IH
  ILLAHHHU  H;CtMHLLLHHtHIHh  L  HHE1   H   L  HLL;HIH   HH  gHA      @      !HA      @      HM  Hc  H H;EuDH(  [A\A]A^A_]0  UHAWAVATSLcD4x   L%t  I$DP(HIcH0  HCxHCHCJDxHCI$H{ 1  H[A\A^A_]UHAWAVAUATSPIIII+?HvIIMoL9tIHc3L<    M|$M;|$rH;t  H1H=L  Q I?HLHR   HI9uLH[A\A]A^A_]UHSPHcHH_H;_rHs  HH=L  1Q HH[]UHAWAVSPIHHCLsLH)HSH)H9|Hs  HH=;M  1Q H{ I7HUSh} tHKHQHSL9LH+KHHHLH[A^A_]UHAVSHH{H+{H:IHsHSHL   L[A^]UHAWAVAUATSPIIIIID$L9s'Hr  HLI+t$HH=
K  1Q ID$L9s"Hr  HLI+t$HH=J  1Q M)MIIIM~M;~vHyr  HH=J  1Q A   Et6AI} w}HLH~I;FtIAuH}E1DH[A\A]A^A_]UHAVSHH{H+{HIHsHSHLL[A^]UHAVSII^H;|HI;^uHq  HI~ x  HL[A^]`0UHHtH]]UHAWAVAUATSPIIMwIwHL)IUI+U1H9uOI9sAMe IH3LAU`Ht2Hc@HIEI;Et H0H|tIM;wrø   1H[A\A]A^A_]UHSPHH{ S`HtHc@HHCHCH[]UHAWAVAUATSHIIHL9cw!Hp  HLH+sHH=qI  1Q Ls I7HULSh}    HEL}M,$Iu LS`IMuH>p  HIu H=YI  1Q AwML+sID9tHp  HH=vI  1DQ LzHo  H Lp  HEI$IcHMHAH
o  H	Hp  LcLH[A\A]A^A_]UHHHx P`HHtA]UHAWAVATSIIIGI+GHI~I+~HcIIwIWLL1t*I^HI;^t%H;GzLHQI;D$uL-E1L[A\A^A_]UHAWAVAUATSHIHH]HCH+CHI}I+}H9NHEL{L;{tsMe IH3LAU`Ht>LcpIMuM;ut,I>HRztI>yH]HHH;CtIHEL;xuH}f1HEHEH[A\A]A^A_]UHAWAVAUATSH(LEHMIHH]ILkMfM+fILH+{HDIGI+H)HEIvIVLHHSH+SH   LHMHL{M9t~HHEMw; t"H}LLHM]tDAL$ẢHM9Mu7IH   H}  H}SH}  HEH     1HEHEH([A\A]A^A_]UHAWAVAUATSHHLMLEILuIL}HuHH]HCH+CHH~H+~HIGI+H)HEHSH+SHE11LЕHEILkH[I9tNHEL8E1MuA? t&H}LLHM9   DAL$AAIL9MuLuIVI+VH   H}PHHEHI^IFHEH9tML1HMLmL{A> t!H}HLLtPDAL$AAIL9}LufLm`IE H   H}^  H}H}  HE)HH   H}2  LtH}  HEH     E1LHH[A\A]A^A_]UHAWAVAUATSHIHH]H{H+{H  LsE1L;steHEIE HE1IH0H}AU`HtP9ADcH}ȉ  IHEL;pDuHEHHMcHELI)IM9u
H  1H[A\A]A^A_]UHAWAVAUATSHHUHIMoE    M;otuHC HEE    Me I4$H}S`HtHLcpILsL;st6I6Lut'H}HtLI+wHL+sIDޑEIM;ouEH[A\A]A^A_]UHAWAVAUATSPIHLfL9t2LoM~ HcID H0LAV`HtHHI9uH[A\A]A^A_]UHAWAVAUATSPIHH]LsL+sLHAD$J<    Hi  H P(ID$I"I HL$  LsE1L;ss>I\$E1M&LHy~ALH~EF|(HEH IL;prDH[A\A]A^A_]UHAWAVAUATSHHuH}LwL;wIsUHZLeM.LLH/~HM<It ALH~HMHNd8 BT8H IL;qrAD$ ID$AEHM)H[A\A]A^A_]UHAWAVSH(IW)E)EE   Hg  H HX(LuLlxHLHLH L@HH([A^A_]UHSPHH{ HC     HC    H[]UHSPHH HC  Hg  HCH[]UHSPHH{ 
HCH(#  CH[]UHAWAVAUATSH  IHL-aU  Mm LmHf  H HHx  A      H   1e
  uCHH0HH  HtGHHQLHHL   A"L   LLt
  HL
  L;muDH  [A\A]A^A_]!  UHSPHHf  H   H
e  HHHX H@    @    H[]UHAWAVAUATSPMIIHILHLLN   tL1   ,IFHtH@HtLHHe  IFMn AE 1H[A\A]A^A_]UHAWAVAUATSH(IIHIH/e  H HUHMHLx  A      EԨ   HcIL$HQH+QHH9   H]L}1  IŃ} ~pHEH
d  H	H8
  HI|$HAE   LD
        LD
  MԍAEHEHHEȃL  LHuLHMk  A"	      LLO  HL  DH([A\A]A^A_]þ      LuLH  LB  H}Ls  A   UHAWAVAUATSHIIIHc  H HUHMx  E1   EԨufIŃ}    HEH0HPLLsHHtpLHwI;Et<MԍAEHEHHEȃN	   LLA  LL  E1.H   L  LL  HmLE1LH[A\A]A^A_]UHAVSHHIHb  H LEHMHHLx  1uUHMLH	   H[A^]UHAWAVAUATSPỈII	  Iǅ~cLuH3b  H I<$
  ILLsxIL  ˃ 11ҹ   H}M  LH
  E1LH[A\A]A^A_]UHAWAVAUATSHIHIIL5a  IHUHMLHx     usEԨuL~DHEIH8
  HMHQLHLft.MԍAEHEHHEȃ1 	   LHY  LL     H[A\A]A^A_]UHAWAVAUATSHHIIIH`  H HUHMLLx  A      L}HcMIFLxH@HL)HH9   L9txMnI7HEHHHMHH}HI]rHHtnI} HtHHE`  H    II] IIFHHH+HINI9LuE1$   HL6  H}H  A   DH[A\A]A^A_]UHAWAVAUATSH8IIIL}IH_  HHUHMLLx  1i  LeEԨ  HEHHEH}    HELeMH8
  IHELpH]HL;H;C   HHHHL   I\$HsH}LMLpH   HI+\$HMHIH MԍAEHEHHEȃH
^  H	P  H
i^  HHHMHH H@    @    K	   LL  LL  ,      LL  H}L  H}1H8[A\A]A^A_]UHAWAVAUATSH8HMAILuIE   E1E,  LEAE1H]  HHMJ4LHUHMx  [  HLEȃ	  D}IIHMH9
  III~L   HMHcUHtMLHINH1A   H}A0  HM   D}AIALuC\IELxHXE1I9tHMmE1KH
\  L	H0KL 1A   LA	  H   HtuIHI9uH\  H Hh  D  LH1CHEJ	   H]H   L   H]HL7   H}H      H8[A\A]A^A_]UHHt7WG    ]UHAWAVSPIHHt;sL=\  IHH  IH  HLH[A^A_]H[A^A_]UHAWAVATSAHHtQL%[  I$H
  IHt5D{I$HH  I$H  HL[A\A^A_][A\A^A_]UHAWAVAUATSPHIM   H   L{L%*[  I$L  I$L   L@  HCHS  H4H4  L0  E11HLAM$HHT  HH H
T  HCH5W  LH$    H5r4  1LMA0  Ht%HZ  H HHH[A\A]A^A_]  H[A\A]A^A_]UHAWAVAUATSH   MIL%H  M$$LeЉHZ  HLL  HLL  HLL;euH   [A\A]A^A_]  UHAWAVAUATSH   IL%H  M$$LeЉǅ    MtFL-uY  IE L
  IǉIE HH  IE HL  HLL;euH   [A\A]A^A_]  UHH
S  H]UHGH
!S  H]UHAWAVATSAL%X  I$   P(II$McB<    P(IJININHH9rL[A\A^A_]UHHHO0HH9r]UHAWAVSPAL=UX  I   P(HIMcB<    P(HHCJHCHH[A^A_]UHAWAVAUATSPIIFI+HHcL<    L-W  IE    P(IIE DP(I$HIL$IL$I6H"H HH  LH[A\A]A^A_]UHAVSHHtL5W  IH;P0IH[A^]`0[A^]UHAWAVSPHH;HCH)HLcL9}6L{I)H6W  H B4    P8HIIcHHKJHCH[A^A_]UHHHO0HH9r]UHAVSHHLcJH;KrHV  HH=\:  1Q HB[A^]UHAWAVSPAHHLcJH;KrHV  HH=R:  1Q HF4H[A^A_]UHSPHHH;CuHJV  HH=P:  1Q H H[]UHSPHHCH9uHV  HH=:  1Q HC@H[]UHAVSAHHKHCH9r H+HtH\HKHAHCD1[A^]UHSPHHCH9uHU  HH=9  1Q HCHHCH[]UHAWAVAUATSPAAHHWH9s*HMU  HHH+7HIH=9  1Q LHWIHHHOH)HH)HI)D9} HHLHHcHHHWH}McH9sJ<H)H  HMHQHMJHAEtAD;H[uIIL)LH[A\A]A^A_]UHAWAVSPHIIIWL9s$HcT  HLI+7HH=8  1Q IWH9s$H:T  HHI+7HH=8  1Q IWH)LH  L)I)_LH[A^A_]UHAWAVATSAII$IL$LcJH9rHS  HH=c7  1Q I$IL$IcHH9rHS  HH=87  1DQ I$BB[A\A^A_]HHWHH9s91u1H9uUHV   ]UHHHGHH9s91uH]UHAWAVATSAIIGI+HD)ǾIII?IwH)HH
        IIW1H9u13HƋ9|HH9щuD9}
HA9uL[A\A^A_]UHHHwH)HH
      H]i  UHAWAVAUATSPHLc1dIHL9t@IL5
6  Lc+JI;OrHR  H1LDQ IB   HI9uLH[A\A]A^A_]UH+]UHHHH9sH92tHH9r	H)H]UHAVSHIIH9wI9^wHQ  HH=[6  1Q IH)H[A^]UHAWAVAUATSPIHLcMoHIQ  H    P(IH4Q  H    P(IIFHIFHHL9t*3IIGHH9s91uHL9tLLH[A\A]A^A_]UHAWAVAUATSPIHLcMoHP  H    P(IHP  H    P(IIFHIFHHL9t*3IIGHH9s91uHL9uLOLH[A\A]A^A_]HHH9HIH)HHcH0H@H)HH9uUHH  ]1UHLLO1M9t@HVLLǉI9Hs7L91tHH9rHH9HL9uHcM)IL9]UHHLFL9t3HwHH9sH9tHH9r
   H9uHL9u1]UHAWAVAUATSPMIHIIIEL9s%HO  HLI+u HH=4  1Q IEH9s!HN  HHI+u HH=3  1Q M9|$s!HN  HLI+4$HH=3  1Q L)HH1LL!IHHLLHr
  LH[A\A]A^A_]UHAWAVAUATSPHII9^tdIM+IL=wN  L1Iž   L)Ea$    H1L3  	  HLHI<I;^EuHtHH=N   HN  H[A\A]A^A_]UHAWAVSPAHM  H   P(IHIcHP(IGIHHa	  II_LH[A^A_]UHAWAVAUATSPIIFI+HHcL<    L-FM  IE    P(IIE DP(I$HIL$IL$I6H#H HH  LH[A\A]A^A_]UHAVSHHtL5L  IH;P0IH[A^]`0[A^]UHAWAVSPHH;HCH)HLcL9}6L{I)HL  H B4    P8HIIcHHKJHCH[A^A_]UHHH0HH;Gr]UHAVSHHLcJH;KrH%L  HH=1  1Q HJ[A^]UHAWAVSPIHHLcJH;KrHK  HH=1  1Q HN4H[A^A_]UHSPHHH;CuHK  HH=1  1Q HH H[]UHSPHHCH9uHtK  HH=a1  1Q HCH@H[]UHAVSIHHKHCH9r H+HtH^HKHAHCL1[A^]UHSPHHCH9uHJ  HH=1  1Q HCHHCH[]UHAWAVAUATSPIAHHWH9s*HJ  HHH+7HIH=0  1Q LHWIHHHOH)HH)HI)D9} HHLHHcHHHWH}McH9sJ<H)H#  HMHQHMJHAEtAL;H[uIIL)LH[A\A]A^A_]UHAWAVSPHIIIWL9s$HI  HLI+7HH=40  1Q IWH9s$HI  HHI+7HH=0  1Q IWH)LHh  L)I)_LH[A^A_]HHWHH9sH91u1H9uUH   ]UHHHGHH9sH91uH]UHHHwH)HH
      H]  UH+]HHH9HIH)HHcH0H@H)HH9uUHH  ]1UHLLO1M9tBHVLLǉI9HsH7LH91tHH9rHH9HL9uHcM)IL9]UHAWAVAUATSPMIHIIIEL9s%HH  HLI+u HH=.  1Q IEH9s!HG  HHI+u HH=.  1Q M9|$s!HG  HLI+4$HH=.  1Q L)HH1LLIHHLLH[  LH[A\A]A^A_]UHAWAVAUATSPHII9^tdIM+IL=`K  L1Iž   L)Ea$    H1L4.    HLHI<I;^EuHtHH=K   HJ  H[A\A]A^A_]UHH]LeLmLuH0HIAHE    LgMt
A<$ϺtH-  HHC       LE1HH5-  AT$HH   L1Eu   H0	uLHt?HHt:t0	wCE1   LH5T-  HAT$1   E1   LH53-  HAT$HHt_HMH
E  HAHt*H HE  HAH@HE  HAH@HE  !HiE      HfE      HcE      HH]LeLmLuUHH]LeLmLuL}HpIHuA׉MHE    L5D  ILE1HH5h,  PHH   HEHt/PL-S,  D9u'HEPL-V,  ;EuHEHD  bL-W,  IL  MHD$    Ll$H,  HD$H$L
>,  LEH
E,  H+  H5L,  L1A@  1HH]LeLmLuL}Ð%2  %2  %2  %2  %2  %2  %2  %2  %2  %2  %2  %2  La2  AS%Q2  h    h   h%   h=   hL   hZ   hh   hw   h   h   h   h   x::ral::tuple tuple              ::ral::relation relation ::ral::relvar relvar ral ::ral 0.12.2  This software is copyrighted 2004 - 2014 by G. Andrew Mangogna. Terms and conditions for use are distributed with the source code.              relationValue tupleVarName ?attr1 type1 expr1 ... attrN typeN exprN?            multiplicity specification      8.6 iso8859-1 pkgname version copyright Ral_AttributeDelete: unknown attribute type: %d Ral_AttributeDup: unknown attribute type: %d Ral_AttributeRename: unknown attribute type: %d Ral_AttributeTypeEqual: unknown attribute type: %d Ral_AttributeValueEqual: unknown attribute type: %d Ral_AttributeValueCompare: unknown attribute type: %d Ral_AttributeValueObj: unknown attribute type: %d Ral_AttributeConvertValueToType: unknown attribute type: %d Ral_AttributeHashValue: unknown attribute type: %d Ral_AttributeScanType: unknown attribute type: %d Ral_AttributeScanValue: unknown attribute type: %d Ral_AttributeConvertValue: unknown attribute type: %d Ral_AttributeTypeScanFlagsFree: unknown flags type: %d Ral_AttributeValueScanFlagsFree: unknown flags type: %d bignum boolean bytearray dict double int list long string wideInt bad boolean value, "%s" booleanHash: cannot convert, "%s" doubleHash: cannot convert, "%s" intHash: cannot convert, "%s" longHash: cannot convert, "%s" wideIntHash: cannot convert, "%s" Ral_RelationUpdate: attempt to update non-existant tuple while ungrouping relation while computing quotient while unwrapping tuple Ral_RelationErase: first iterator out of bounds Ral_RelationErase: last iterator out of bounds Ral_RelationErase: first iterator greater than last  !=  array assign attributes body cardinality compose create degree divide dunion eliminate emptyof extend extract foreach fromdict fromlist group heading insert intersect is isempty isnotempty issametype join minus project rank rename restrict restrictwith semijoin semiminus summarize summarizeby table tag tclose times tuple uinsert ungroup union unwrap update wrap subcommand ?arg? ... subcommand relation arrayName keyAttr valueAttr relationValue ?attrName | attr-var-list ...? relationValue relation1 relation2 ?-using joinAttrs? heading ?tuple1 tuple2 ...? relation keyAttr valueAttr dividend divisor mediator relation1 relation2 ?relation3 ...? relationValue ?attr ... ? attribute / type / expression arguments must be given in triples relationValue attrName ?attrName2 ...? tupleVarName relationValue ?-ascending | -descending? ?attr-list?script ordering 
    ("::ral::relation foreach" body line %d) -ascending -descending dictValue keyattr keytype valueattr valuetype list attribute type relation newattribute ?attr1 attr2 ...? attempt to group all attributes in the relation during group operation relationValue ?name-value-list ...? equal == notequal != propersubsetof < propersupersetof > subsetof <= supersetof >= relation1 compareop relation2 compareop relation1 relation2 relation1 relation2 ?-using joinAttrs relation3 ... ? relationValue ?attrName ? ?-ascending | -descending? sortAttrList? ? relationValue ?-ascending | -descending? rankAttr newAttr relationValue ?oldname newname ... ? oldname / newname arguments must be given in pairs relValue tupleVarName expr relValue expr relationValue perRelation relationVarName ?attr1 type1 expr1 ... attrN typeN exprN? heading ?value-list1 value-list2 ...? relation attrName ?-ascending | -descending sort-attr-list? ?-within attr-list -start tag-value? -within -start tag option Ral_TagCmd: unknown option, "%d" relation relation attribute relationValue attribute relationValue tupleVarName expr script 
    ("in ::ral::relation update" body line %d) invoked "break" outside of a loop invoked "continue" outside of a loop unknown return code %d relationValue newAttr ?attr attr2 ...? -using Ral_ConstraintDelete: unknown constraint type, %d unknown constraint type, %d relvarConstraintCleanup: unknown constraint type, %d is referenced by multiple tuples is not referenced by any tuple references multiple tuples references no tuple for association  (  [ ] ==> [ ]  ) 1 + ? * , in relvar  
 tuple    is referred to by multiple tuples is not referred to by any tuple for partition   is partitioned [  |  ]) correlation   does not form a complete correlation for correlation   <== [  (Complete) ] ==>  association constraint correlation delete deleteone eval exists identifiers names partition path procedural restrictone set trace transaction unset updateone updateper name refrngRelvar refrngAttrList refToSpec refToRelvar refToAttrList refrngSpec info member delete | info | names ?args? | member <relvar> | path <name> constraint subcommand name ?pattern? relvarName Unknown association command type, %d ?-complete? name corrRelvar corr-attr-list1 relvar1 attr-list1 spec1 corr-attr-list2 relvar2 attr-list2 spec2 relvarName heading id1 ?id2 id3 ...? relvarName tupleVarName expr relvarName relvarName ?attrName1 value1 attrName2 value2 ...? relvarName ?relationValue ...? arg ?arg ...? 
    ("in ::ral::relvar eval" body line %d) relvarName ?name-value-list ...? relvarName relationValue name super super-attrs sub1 sub1-attrs ?sub2 sub2-attrs sub3 sub3-attrs ...? name relvarName1 ?relvarName2 ...? script relvarValue attr value ?attr2 value 2 ...? attribute / value arguments must be given in pairs relvar ?relationValue? add remove suspend variable option type ?arg arg ...? trace option trace type add variable relvarName ops cmdPrefix remove variable relvarName ops cmdPrefix info variable relvarName suspend variable relvarName script Unknown trace option, %d add transaction cmdPrefix remove transaction cmdPrefix info transaction suspending eval traces not implemented Unknown trace type, %d begin end rollback transaction option Unknown transaction option, %d ?relvar1 relvar2 ...? relvarName tupleVarName expr script relvarName tupleVarName idValueList script relvar creation not allowed during a transaction :: during identifier construction operation 
    ("in ::ral::%s %s" body line %d) -complete Ral_RelvarObjConstraintInfo: unknown constraint type, %d end transaction with no beginning 
    ("in ::ral::relvar trace suspend variable" body line %d)  relvar may only be modified using "::ral::relvar" command relvarTraceProc: trace on non-write, flags = %#x
 bad operation list: must be one or more of delete, insert, update, set or unset traceOp namespace eval   { } procedural contraint, " ", failed returned "continue" from procedural contraint script for constraint, " " returned "break" from procedural contraint script for constraint, " Ral_TupleUpdate: attempt to update a shared tuple get tupleValue ?attrName | attr-var-pair ... ? tupleValue heading name-value-list tupleValue ?attr? ... tuple1 tuple2 tupleValue ?name type value ... ? tupleValue attr ?...? attr1 type1 value1 ... tupleValue ?oldname newname ... ? for oldname / newname arguments tupleValue tupleAttribute tupleValue ?attr1 value1 attr2 value2? for attribute name / attribute value arguments tupleValue newAttr ?attr attr2 ...? failed to copy attribute, "%s" Ral_TupleHeadingAppend: out of bounds access at source offset "%d" Ral_TupleHeadingAppend: overflow of destination Ral_TupleHeadingFetch: out of bounds access at offset "%d" Ral_TupleHeadingStore: out of bounds access at offset "%d" Ral_TupleHeadingStore: cannot find attribute name, "%s", in hash table Ral_TupleHeadingStore: inconsistent index, expected %u, got %u Ral_TupleHeadingPushBack: overflow , " RAL no error unknown attribute name duplicate attribute name bad heading format bad value format bad value type for value unknown data type bad type keyword wrong number of attributes specified bad list of pairs duplicate command option relations of non-zero degree must have at least one identifier identifiers must have at least one attribute identifiers must not be subsets of other identifiers tuple has duplicate values for an identifier duplicate attribute name in identifier attribute set duplicate tuple headings not equal relation must have degree of one relation must have degree of two relation must have cardinality of one bad list of triples attributes do not constitute an identifier attribute must be of a Relation type attribute must be of a Tuple type relation is not a projection of the summarized relation divisor heading must be disjoint from the dividend heading mediator heading must be a union of the dividend and divisor headings too many attributes specified attributes must have the same type only a single identifier may be specified identifier must have only a single attribute "-within" option attributes are not the subset of any identifier attribute is not a valid type for rank operation duplicate relvar name unknown relvar name mismatch between referential attributes duplicate constraint name unknown constraint name relvar has constraints in place referred to identifiers can not have non-singular multiplicities operation is not allowed during "eval" command a super set relvar may not be named as one of its own sub sets correlation spec is not available for a "-complete" correlation recursively invoking a relvar command outside of a transaction recursive attempt to modify a relvar already being changed serious internal error unknown command relvar setfromany updatefromobj NONE destroy OK UNKNOWN_ATTR DUPLICATE_ATTR HEADING_ERR FORMAT_ERR BAD_VALUE BAD_TYPE BAD_KEYWORD WRONG_NUM_ATTRS BAD_PAIRS_LIST DUPLICATE_OPTION NO_IDENTIFIER IDENTIFIER_FORMAT IDENTIFIER_SUBSET IDENTITY_CONSTRAINT DUP_ATTR_IN_ID DUPLICATE_TUPLE HEADING_NOT_EQUAL DEGREE_ONE DEGREE_TWO CARDINALITY_ONE BAD_TRIPLE_LIST NOT_AN_IDENTIFIER NOT_A_RELATION NOT_A_TUPLE NOT_A_PROJECTION NOT_DISJOINT NOT_UNION TOO_MANY_ATTRS TYPE_MISMATCH SINGLE_IDENTIFIER SINGLE_ATTRIBUTE WITHIN_NOT_SUBSET BAD_RANK_TYPE DUP_NAME UNKNOWN_NAME REFATTR_MISMATCH DUP_CONSTRAINT UNKNOWN_CONSTRAINT CONSTRAINTS_PRESENT BAD_MULT BAD_TRANS_OP SUPER_NAME INCOMPLETE_SPEC ONGOING_CMD ONGOING_MODIFICATION INTERNAL_ERROR Ral_IntVectorFetch: out of bounds access at offset "%d" Ral_IntVectorStore: out of bounds access at offset "%d" Ral_IntVectorFront: access to empty vector Ral_IntVectorPopBack: access to empty vector Ral_IntVectorInsert: out of bounds access at offset "%d" Ral_IntVectorErase: out of bounds access at offset "%d" Ral_IntVectorOffsetOf: out of bounds access Ral_IntVectorCopy: out of bounds access at source offset "%d" Ral_IntVectorCopy: out of bounds access at dest offset "%d" %d: %d
 Ral_PtrVectorFetch: out of bounds access at offset "%d" Ral_PtrVectorStore: out of bounds access at offset "%d" Ral_PtrVectorFront: access to empty vector Ral_PtrVectorPopBack: access to empty vector Ral_PtrVectorInsert: out of bounds access at offset "%d" Ral_PtrVectorErase: out of bounds access at offset "%d" Ral_PtrVectorCopy: out of bounds access at source offset "%d" Ral_PtrVectorCopy: out of bounds access at dest offset "%d" %d: %X
       interpreter uses an incompatible stubs mechanism Tcl tcl::tommath epoch number mismatch requires a later revision missing stub table pointer , actual version   (requested version  Error loading  ):            4       4      Xa    !    L   L   p     L               M  N     
     " Q o   s 7    " D  D w  N  5    D j  Q     z    g" # # # 8$  $ $ `% &&  & .'  ' (  + +  V, ,  9 9  F  G  H +I AI IJ  J K  N 4O OO O  gP P P  JV V  Z r[  a ]b  s t  y Sz   |      3    ߽   d      :   s  _   d  `   [  k     u   ( m     < z  $    Q  $  & 

 
  ## $K% m&& -+r+ hEE QR T]TT UU 0XtX YZ ``aa bbbb pcd {ee ?fff)geg ij nn xXy  z1j a߇  Ӊ~ n d  ;R n_ u͢ɣ ~O ʧ D3  2n"l !J òB            zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x         zR x  4      k             
            zR x  4      
             
           H     p                                                                          "     ,     6                  ,      B,      ,      d-           -      d.      .      /           /      /      0      n0           0      0      1      U1           1      1      2      2           3      13      3      3           d4      4      4      5           O5      q5      5      46           4&            (!      '           6      6      47      7      ;      ;           i           Vk           l           m           jn           o           q           r           s           Vu           v           x           y           }z                 "           *           3     -      <           B           J           Q           [           ^     ;      f     ב      q     s      |     X                                 ܙ                                                       ڣ           z                                            Z           t           k           ׾                                       
                !                                 =             H                            )     	a      /     	a      2     Ka      ;     Ka      >     a      M     a      O     'b      `     'b      b     aa      k     aa      n     a      y     a                      =             H                                                    0     2     4     6                $           /     \                 ;           B                      L     	     Q     D     X          J     p     Q     
               d          j          t     q     y               D          $          @                    F                    ="          "          &          *                     0     2     4     6     ;            J                                                                     X          )          A          ?          ݆     )               K               3     P               B     Ȑ          x                         
                         E                     r     {                                        0     B     [                    )     ^     n                              (     M     o               (     F     i                    2     H     \                              F                    ?             V               f     m     x                                        $     /               ;     B                                   )     L               "     *     3          <     B     X     J     Q     [     ^     f     q     |               d     j     y                                                                                                                   
                                                                                             &     8     J     ^     m     }                                                            )     7     I     Z     l     z                                                            )                                                           Relation Tuple        <      <      =      U=            Tg      ag      og      g            <      <      h      h                                   ;     Q          d          t                                    &                     .                                                 0             2            4            6                                     <     f                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  ! RC\A`Bpp`Bp`:BUAp`"B`/A`KA`/AVETATA`BTBRBSApU      @___stack_chk_guard Qq@dyld_stub_binder     q(@___bzero  q0@___snprintf_chk  q8@___stack_chk_fail  q@@_bsearch  qH@_memcmp  qP@_memcpy  qX@_memmove  q`@_qsort  qh@_strcmp  qp@_strcpy  qx@_strlen  q@_strrchr     _  Ral_ r tuple  
In xSafe Unload Attribute JoinMap Rel Constraint Tuple ErrorInfo  PtrVector % it t   0  Init Unload  6  6  6  	New D Rename Equal T Value Convert HashValue Scan  T RelationType FromObjs  clType upleType  6  7  8  elete up  8  9  :  :  ype oString  Equal ScanFlagsFree  ;  Equal Compare Obj ScanFlagsFree  <  @  B  D  Value Name Type  ToType   G  L  Name Type Value  N  N  O  P  R  S  T  U  V  New Delete A Tuple GetAttr FindAttr SortAttr MatchingTupleSet  p  q  ttr dd  Reserve Map  q  Reserve Map Counts  r  AttrMapping TupleMapping  s  t  u  u  v  v  w  x  x  ation var 
 N D Re P S U In 	Find 	Minus 	T 	Group 
Join 
Co 
E ValueStringOf Obj  ew otEqual  {  up elete i  |  serve nameAttribute  }  ushBack ro 	 ~  hallowCopy emi 
ort u can tringOf      pdate n  ȁTupleObj 
 ion group 
wrap 
 Copy    sjointCopy 	vide 
   tersect 	sertTuple 
 އ  JoinTuples 
 ܉  imes 	ag 
close 
   ject 	perSu            mp 
nvert  ose 
are  ؟  Join 
Minus 
       Within 
         xtract rase qual            bsetOf persetOf    bsetOf persetOf          Value    Value        el al_ ' ationCmd varCmd    New 
Convert 
ParseJoinArgs 
Type (   Ӟ  Value 
Obj 
     ܢ    New 
D Find S I Tra RestorePrev Obj  Info Transaction  e iscardPrev  lete clConstraintEval  ʩInfo T  Identifier ById      tart etRelation  Transaction Command    ϱ  sTransOnGoing nsertTuple d      nsModifiedRelvar ce  ͳ  ransaction uple    AssocCreate New P CorrelationCreate Delete FindByName    Association P Correlation  ڴ  artitionCreate roceduralCreate    artition rocedural      շ      ByName               IndexTuple UnindexTuple                Add Remove        	New Delete Find E C InsertTuple KeyTuple UpdateTuple Trace      Relvar Constraint    xec nd  U InsertTraces DeleteTraces SetTraces EvalTraces  nsetTraces pdateTraces    o reate  pyOnShared nstraint            Update Var Eval    ͆  Assoc P Correlation    artition rocedural  ژ  ҥ  ۶  Delete Info Names Member Path      ʾ        Trans Cmd      Add Remove Info Suspend          Add Remove Info              New S E Co D U H A GetAttrValue ValueStringOf Obj PartialSetFromObj     ubset can tringOf etFrom    xtend qual    py nvert  Values     elete up    nreference pdate    Values    ash eading  Attr    ttrEqual ssignToVars     AttrValue FromObj         Shallow Ordered      Value    Value        Cmd AttrHashType (   New S F PushBack D A E Un In Join Co MapIndices  OrderMap FromObj  ubset t can    etch ind    ѻ  up elete  Ѽ  ppend ttrsFrom    xtend qual      reference ion        ore ringOf    dexOf tersect          m nvert  pose monAttributes                New Convert Type (     Obj ValueList       Obj Vect              Set  Get ! Cmd  Error     Obj     erp  Vector ! SetError  ErrorInfo !   Obj !   Command !Option !     New !F "D "Reserve "S "B #P #In #E #OffsetOf $Minus $Co $ Empty " i "etch "ront # ll "nd $ Consecutive "   up "elete "           tore #et $ort $ubsetOf $     ack #ooleanMap $   ushBack #opBack #rint %     sert #dexOf $tersect $   rase $xchange $qual $     Add $Complement $                 ȁ      ntainsAny %py %   ӄ    
New %D %Reserve %F &S &Back &P &Insert 'E 'Copy ' ̇  up %elete %     ։  i &etch &ront & ll &nd '   Ҋ  tore &etAdd 'ort 'ubsetOf '   ֋    ushBack &opBack 'rint '     Ǎ  rase 'qual '               Ŕ  relationTypeName 'tupleTypeName '   ɟ  П  Ƞ        0
aaO\VG]JA:/v=``[x _X:"JA:"j{y"`[x&JA:"e^y"ewyvJdf~[ZX@1
Y?C+fWcKWfg}_BBBBB;{
0(N7

	)-(>7K3#(BBmBBxK'ixb|Z0)E;>zFs|4EE]LDDDFZP9{@yLUw2]:S<|f79;C. ^n)*(@U[p`P1d<H46J;-'}
2BDdGX1d<H57J;. '
Df                (!          4&          '          +          ,          B,          ,      $    d-      0    -      <    d.      J    .      Z    /      g    /      u    /          0          n0          0          0          1          U1          1          1          2          2          3      
    13          3      #    3      ,    d4      5    4      @    4      M    5      W    O5      `    q5      k    5      x    46          6          6          47          7          ;          ;          <          <           =          U=      1    B      C    'J      ]    d^      k    Tg      |    ag          og          g          g          h          h          i          Vk      -    l      D    m      U    jn      m    o          q          r          s          Vu          v          x          y          }z                /          C          X    -      m                                                      ;          ב          s      
    X                /          A    ܙ      U          f          y                        ڣ          z                                           Z      "     t      5     k      G     ׾      Y           m                                      !                                                  !          '!          6!          O!    {      `!    j      p!    :      !          !    l      !          !    F      !          #"    [      H"          ^"          s"    \      "          "          "          "          "    	     "    D     "         #    p     #    
     +#         ;#         K#         _#    q     n#         #    D     #    $     #    @     #         #    F     #         #    ="     #    "     $    &     $$    *     8$    7     K$    *8     \$    <     t$    1I     $    xg     $    l     $    o     $    4r     $    X     %    )      %    A     0%    ?     @%    ݆     S%         b%    K     r%         %    P     %         %    Ȑ     %    x     %         %         %         %         &    E     &         *&    <     ?&    f     T&         e&         v&         &         &    b     &    @     &    M     &    `     &    p     '    y     #'         7'         D'         Q'         `'    0     y'         '          '          '    0     '    P     '         '    P     (         (         ((         I(         S(     	     f(     
     u(         (         (          (         (         (          (    p     (         )         *)          S)          ^)         k)         |)         )         )         )  
       )  
       )  
            q'           (      6     )      Q     #      r     m                      n           <&           +"                                      1          F    B'      ]    '      t    )          W+                    E*          >                     i!      &    *      G          b                        9                    Z                    &      '          D          c                                [         r         ͵                    i         1     0         J    S     ]          r             d                                                      P         5     /         E    $     Z    ,     r                      /                           .                  1         #     2    o     N    
     a         u             D         ٷ         =         9          ,:          :          8      0    8      C    9;      X    :      l    X<          $8          y;          <          ;          H9                   j              /    $     A         U         i    R     }    :         o                                             E     	    B     	         5	    A     J	         ]	         q	         	    f`      	    O      	    d      	    d      	    e@      	    C      
    R      #
    F>      4
    	a      G
    @_      Z
    ^      o
    vD      
    N      
    =G      
          
          
    C          ;M          D      &    =      7    Ka      M    S      e          y              H         F          a          'b          N?          ib      .    >      C    b      U    %c      l    Q          R          @          ]          %f          aa          a      
    T      
     V      0
    mZ      D
    tE      W
    J      l
    A      
    <C      
    3X      
    @      
    \      
    f      
                    %          ;          X    ?      o                                                                      
          '          6          I          c    ]         J_         <i         Fh         j         ;         C     .    R     N    ZL     l    [[          9         1l         j         q         v         >     '    Nu     C    ;     a    MC         ]         a:         k=         >         X3         Pp         Wq     0    p     N    -A     h    }l         m         m         o         %@               	                     8          T    H      h                                                     1|         8                  }         y     4    4z     E    _~     S    ~     h    ~     }    z         {         y         }         {         {                  ܮ     #    1     B    <     d         }                      Q                                	                  8         S    o     i                      ҭ         q         ѝ         M         ٠              4         L         c             x         ī                  p                  I                  E              5         H    0y     Y    z     o    |         ͯ                                              i                                  #            ,            <            N            a            j            r            z                                                                                                                          @                           _Ral_AttributeConvertName _Ral_AttributeConvertType _Ral_AttributeConvertValue _Ral_AttributeConvertValueToType _Ral_AttributeDelete _Ral_AttributeDup _Ral_AttributeEqual _Ral_AttributeHashValue _Ral_AttributeNewFromObjs _Ral_AttributeNewRelationType _Ral_AttributeNewTclType _Ral_AttributeNewTupleType _Ral_AttributeRename _Ral_AttributeScanName _Ral_AttributeScanType _Ral_AttributeScanValue _Ral_AttributeToString _Ral_AttributeTypeEqual _Ral_AttributeTypeScanFlagsFree _Ral_AttributeValueCompare _Ral_AttributeValueEqual _Ral_AttributeValueObj _Ral_AttributeValueScanFlagsFree _Ral_ConstraintAssocCreate _Ral_ConstraintCorrelationCreate _Ral_ConstraintDelete _Ral_ConstraintDeleteByName _Ral_ConstraintFindByName _Ral_ConstraintNewAssociation _Ral_ConstraintNewCorrelation _Ral_ConstraintNewPartition _Ral_ConstraintNewProcedural _Ral_ConstraintPartitionCreate _Ral_ConstraintProceduralCreate _Ral_ErrorInfoGetCommand _Ral_ErrorInfoGetOption _Ral_ErrorInfoSetCmd _Ral_ErrorInfoSetError _Ral_ErrorInfoSetErrorObj _Ral_Init _Ral_IntVectorBack _Ral_IntVectorBooleanMap _Ral_IntVectorContainsAny _Ral_IntVectorCopy _Ral_IntVectorDelete _Ral_IntVectorDup _Ral_IntVectorEqual _Ral_IntVectorErase _Ral_IntVectorExchange _Ral_IntVectorFetch _Ral_IntVectorFill _Ral_IntVectorFillConsecutive _Ral_IntVectorFind _Ral_IntVectorFront _Ral_IntVectorIndexOf _Ral_IntVectorInsert _Ral_IntVectorIntersect _Ral_IntVectorMinus _Ral_IntVectorNew _Ral_IntVectorNewEmpty _Ral_IntVectorOffsetOf _Ral_IntVectorPopBack _Ral_IntVectorPrint _Ral_IntVectorPushBack _Ral_IntVectorReserve _Ral_IntVectorSetAdd _Ral_IntVectorSetComplement _Ral_IntVectorSort _Ral_IntVectorStore _Ral_IntVectorSubsetOf _Ral_InterpErrorInfo _Ral_InterpErrorInfoObj _Ral_InterpSetError _Ral_JoinMapAddAttrMapping _Ral_JoinMapAddTupleMapping _Ral_JoinMapAttrMap _Ral_JoinMapAttrReserve _Ral_JoinMapDelete _Ral_JoinMapFindAttr _Ral_JoinMapGetAttr _Ral_JoinMapMatchingTupleSet _Ral_JoinMapNew _Ral_JoinMapSortAttr _Ral_JoinMapTupleCounts _Ral_JoinMapTupleMap _Ral_JoinMapTupleReserve _Ral_PtrVectorBack _Ral_PtrVectorCopy _Ral_PtrVectorDelete _Ral_PtrVectorDup _Ral_PtrVectorEqual _Ral_PtrVectorErase _Ral_PtrVectorFetch _Ral_PtrVectorFill _Ral_PtrVectorFind _Ral_PtrVectorFront _Ral_PtrVectorInsert _Ral_PtrVectorNew _Ral_PtrVectorPopBack _Ral_PtrVectorPrint _Ral_PtrVectorPushBack _Ral_PtrVectorReserve _Ral_PtrVectorSetAdd _Ral_PtrVectorSort _Ral_PtrVectorStore _Ral_PtrVectorSubsetOf _Ral_RelationCompare _Ral_RelationCompose _Ral_RelationConvert _Ral_RelationConvertValue _Ral_RelationDelete _Ral_RelationDisjointCopy _Ral_RelationDivide _Ral_RelationDup _Ral_RelationEqual _Ral_RelationErase _Ral_RelationExtract _Ral_RelationFind _Ral_RelationFindJoinTuples _Ral_RelationGroup _Ral_RelationInsertTupleObj _Ral_RelationInsertTupleValue _Ral_RelationIntersect _Ral_RelationJoin _Ral_RelationMinus _Ral_RelationNew _Ral_RelationNotEqual _Ral_RelationObjConvert _Ral_RelationObjNew _Ral_RelationObjParseJoinArgs _Ral_RelationObjType _Ral_RelationProject _Ral_RelationProperSubsetOf _Ral_RelationProperSupersetOf _Ral_RelationPushBack _Ral_RelationRenameAttribute _Ral_RelationReserve _Ral_RelationScan _Ral_RelationScanValue _Ral_RelationSemiJoin _Ral_RelationSemiMinus _Ral_RelationShallowCopy _Ral_RelationSort _Ral_RelationStringOf _Ral_RelationSubsetOf _Ral_RelationSupersetOf _Ral_RelationTag _Ral_RelationTagWithin _Ral_RelationTclose _Ral_RelationTimes _Ral_RelationUngroup _Ral_RelationUnion _Ral_RelationUnionCopy _Ral_RelationUnwrap _Ral_RelationUpdate _Ral_RelationUpdateTupleObj _Ral_RelationValueStringOf _Ral_RelvarDeclConstraintEval _Ral_RelvarDelete _Ral_RelvarDeleteInfo _Ral_RelvarDeleteTransaction _Ral_RelvarDeleteTuple _Ral_RelvarDiscardPrev _Ral_RelvarFind _Ral_RelvarFindById _Ral_RelvarFindIdentifier _Ral_RelvarIdIndexTuple _Ral_RelvarIdUnindexTuple _Ral_RelvarInsertTuple _Ral_RelvarIsTransOnGoing _Ral_RelvarNew _Ral_RelvarNewInfo _Ral_RelvarNewTransaction _Ral_RelvarObjConstraintDelete _Ral_RelvarObjConstraintInfo _Ral_RelvarObjConstraintMember _Ral_RelvarObjConstraintNames _Ral_RelvarObjConstraintPath _Ral_RelvarObjCopyOnShared _Ral_RelvarObjCreateAssoc _Ral_RelvarObjCreateCorrelation _Ral_RelvarObjCreatePartition _Ral_RelvarObjCreateProcedural _Ral_RelvarObjDelete _Ral_RelvarObjEndCmd _Ral_RelvarObjEndTrans _Ral_RelvarObjExecDeleteTraces _Ral_RelvarObjExecEvalTraces _Ral_RelvarObjExecInsertTraces _Ral_RelvarObjExecSetTraces _Ral_RelvarObjExecUnsetTraces _Ral_RelvarObjExecUpdateTraces _Ral_RelvarObjFindConstraint _Ral_RelvarObjFindRelvar _Ral_RelvarObjInsertTuple _Ral_RelvarObjKeyTuple _Ral_RelvarObjNew _Ral_RelvarObjTraceEvalAdd _Ral_RelvarObjTraceEvalInfo _Ral_RelvarObjTraceEvalRemove _Ral_RelvarObjTraceUpdate _Ral_RelvarObjTraceVarAdd _Ral_RelvarObjTraceVarInfo _Ral_RelvarObjTraceVarRemove _Ral_RelvarObjTraceVarSuspend _Ral_RelvarObjUpdateTuple _Ral_RelvarRestorePrev _Ral_RelvarSetRelation _Ral_RelvarStartCommand _Ral_RelvarStartTransaction _Ral_RelvarTraceAdd _Ral_RelvarTraceRemove _Ral_RelvarTransModifiedRelvar _Ral_SafeInit _Ral_SafeUnload _Ral_TupleAssignToVars _Ral_TupleAttrEqual _Ral_TupleConvert _Ral_TupleConvertValue _Ral_TupleCopy _Ral_TupleCopyValues _Ral_TupleDelete _Ral_TupleDup _Ral_TupleDupOrdered _Ral_TupleDupShallow _Ral_TupleEqual _Ral_TupleEqualValues _Ral_TupleExtend _Ral_TupleGetAttrValue _Ral_TupleHash _Ral_TupleHashAttr _Ral_TupleHeadingAppend _Ral_TupleHeadingAttrsFromObj _Ral_TupleHeadingAttrsFromVect _Ral_TupleHeadingCommonAttributes _Ral_TupleHeadingCompose _Ral_TupleHeadingConvert _Ral_TupleHeadingDelete _Ral_TupleHeadingDup _Ral_TupleHeadingEqual _Ral_TupleHeadingExtend _Ral_TupleHeadingFetch _Ral_TupleHeadingFind _Ral_TupleHeadingIndexOf _Ral_TupleHeadingIntersect _Ral_TupleHeadingJoin _Ral_TupleHeadingMapIndices _Ral_TupleHeadingNew _Ral_TupleHeadingNewFromObj _Ral_TupleHeadingNewOrderMap _Ral_TupleHeadingPushBack _Ral_TupleHeadingScan _Ral_TupleHeadingStore _Ral_TupleHeadingStringOf _Ral_TupleHeadingSubset _Ral_TupleHeadingUnion _Ral_TupleHeadingUnreference _Ral_TupleNew _Ral_TupleObjConvert _Ral_TupleObjNew _Ral_TupleObjType _Ral_TuplePartialSetFromObj _Ral_TupleScan _Ral_TupleScanValue _Ral_TupleSetFromObj _Ral_TupleSetFromValueList _Ral_TupleStringOf _Ral_TupleSubset _Ral_TupleUnreference _Ral_TupleUpdateAttrValue _Ral_TupleUpdateFromObj _Ral_TupleValueStringOf _Ral_Unload _ral_relationTypeName _ral_tupleTypeName _relationCmd _relvarCmd _tupleAttrHashType _tupleCmd ___bzero ___snprintf_chk ___stack_chk_fail ___stack_chk_guard _bsearch _memcmp _memcpy _memmove _qsort _strcmp _strcpy _strlen _strrchr dyld_stub_binder _stringEqual _stringCompare _isAString _stringHash _cmpTypeNames _isABignum _bignumEqual _bignumCompare _bignumHash _isABoolean _booleanEqual _booleanCompare _booleanHash _isAByteArray _byteArrayEqual _byteArrayCompare _byteArrayHash _isADict _dictEqual _dictCompare _dictHash _isADouble _doubleEqual _doubleCompare _doubleHash _isAnInt _intEqual _intCompare _intHash _isAList _listEqual _listCompare _listHash _isALong _longEqual _longCompare _longHash _isAWideInt _wideIntEqual _wideIntCompare _wideIntHash _attr_0_cmp _attr_1_cmp _tupleAttrHashGenKey _tupleAttrHashCompareKeys _tupleAttrHashEntryAlloc _tupleAttrHashEntryFree _Ral_HeadingMatch _Ral_RelationIndexByAttrs _Ral_DownHeap _tupleHashGenKey _tupleHashCompareKeys _tupleHashEntryAlloc _tupleHashEntryFree _Ral_TupleCompare _tupleAttrHashMultiEntryAlloc _tupleAttrHashMultiEntryFree _RelationArrayCmd _RelationAssignCmd _RelationAttributesCmd _RelationBodyCmd _RelationCardinalityCmd _RelationComposeCmd _RelationCreateCmd _RelationDegreeCmd _RelationDictCmd _RelationDivideCmd _RelationDunionCmd _RelationEliminateCmd _RelationEmptyofCmd _RelationExtendCmd _RelationExtractCmd _RelationForeachCmd _RelationFromdictCmd _RelationFromlistCmd _RelationGroupCmd _RelationHeadingCmd _RelationInsertCmd _RelationIntersectCmd _RelationIsCmd _RelationIsemptyCmd _RelationIsnotemptyCmd _RelationIssametypeCmd _RelationJoinCmd _RelationListCmd _RelationMinusCmd _RelationProjectCmd _RelationRankCmd _RelationRenameCmd _RelationRestrictCmd _RelationRestrictWithCmd _RelationSemijoinCmd _RelationSemiminusCmd _RelationSummarizeCmd _RelationSummarizebyCmd _RelationTableCmd _RelationTagCmd _RelationTcloseCmd _RelationTimesCmd _RelationTupleCmd _RelationUinsertCmd _RelationUngroupCmd _RelationUnionCmd _RelationUnwrapCmd _RelationUpdateCmd _RelationWrapCmd _FreeRelationInternalRep _DupRelationInternalRep _UpdateStringOfRelation _SetRelationFromAny _relvarCleanup _relvarConstraintCleanup _relvarSetIntRep _relvarIndexIds _relvarFindJoinTuples _relvarEvalAssocTupleCounts _relvarAssocConstraintErrorMsg _relvarConstraintErrorMsg _relvarPartitionConstraintErrorMsg _relvarCorrelationConstraintErrorMsg _relvarCorrelationConstraintToString _RelvarAssociationCmd _RelvarConstraintCmd _RelvarCorrelationCmd _RelvarCreateCmd _RelvarDeleteCmd _RelvarDeleteOneCmd _RelvarDunionCmd _RelvarEvalCmd _RelvarExistsCmd _RelvarIdentifiersCmd _RelvarInsertCmd _RelvarIntersectCmd _RelvarMinusCmd _RelvarNamesCmd _RelvarPartitionCmd _RelvarPathCmd _RelvarProceduralCmd _RelvarRestrictOneCmd _RelvarSetCmd _RelvarTraceCmd _RelvarTransactionCmd _RelvarUinsertCmd _RelvarUnionCmd _RelvarUnsetCmd _RelvarUpdateCmd _RelvarUpdateOneCmd _RelvarUpdatePerCmd _relvarResolveName _relvarTraceProc _relvarGetNamespaceName _relvarObjConstraintEval _relvarConstraintAttrNames _Ral_RelvarObjDecodeTraceOps _Ral_RelvarObjEncodeTraceFlag _Ral_RelvarObjExecTraces _TupleAssignCmd _TupleAttributesCmd _TupleCreateCmd _TupleDegreeCmd _TupleEliminateCmd _TupleEqualCmd _TupleExtendCmd _TupleExtractCmd _TupleFromListCmd _TupleGetCmd _TupleHeadingCmd _TupleProjectCmd _TupleRelationCmd _TupleRenameCmd _TupleUnwrapCmd _TupleUpdateCmd _TupleWrapCmd _FreeTupleInternalRep _DupTupleInternalRep _UpdateStringOfTuple _SetTupleFromAny _int_ind_compare _ptr_ind_compare _Tcl_InitStubs _TclTomMathInitializeStubs _Ral_Init.tupleCmdName _Ral_Init.tupleStr _Ral_Init.relationCmdName _Ral_Init.relationStr _Ral_Init.relvarCmdName _Ral_Init.relvarStr _ral_pkgname _ral_version _ral_copyright _RelationExtendCmd.usage _specErrMsg _Ral_Types _Ral_JoinMapSortAttr.cmpFuncs _relationCmd.cmdTable _orderOptions _RelationIsCmd.cmdTable _RelationTagCmd.optTable _condMultStrings _relvarCmd.cmdTable _relvarAssocSpec.condMultStrings _opsTable _tupleCmd.cmdTable _resultStrings _cmdStrings _optStrings _errorStrings _ral_config _tupleHashType _tupleAttrHashMultiType _RelvarConstraintCmd.constraintCmds _RelvarTraceCmd.traceOptions _RelvarTraceCmd.traceTypes _RelvarTransactionCmd.transactionOptions _specTable _tclStubsPtr _tclPlatStubsPtr _tclIntStubsPtr _tclIntPlatStubsPtr _tclTomMathStubsPtr _RelationTagCmd.gotOpt _Ral_IntVectorPrint.buf _Ral_PtrVectorPrint.buf      