# Repository: https://gitlab.com/quantify-os/quantify
# Licensed according to the LICENSE file on the main branch
"""Common python dataclasses for multiple backends."""

from __future__ import annotations

from dataclasses import dataclass
from typing import (
    TYPE_CHECKING,
    Any,
    Literal,
)

import matplotlib.pyplot as plt
import networkx as nx
from matplotlib.patches import Rectangle
from pydantic import (
    field_validator,
)

from quantify.structure.model import (
    DataStructure,
)
from quantify.structure.types import Graph, NDArray  # noqa: TC001 pydantic needs it

if TYPE_CHECKING:
    from matplotlib.axes import Axes

LatencyCorrection = float
"""
Latency correction in seconds for a port-clock combination.

Positive values delay the operations on the corresponding port-clock combination,
while negative values shift the operation backwards in time with respect to other
operations in the schedule.

.. note::

    If the port-clock combination of a signal is not specified in the corrections,
    it is set to zero in compilation. The minimum correction over all port-clock
    combinations is then subtracted to allow for negative latency corrections and to
    ensure minimal wait time (see
    :meth:`~quantify.backends.corrections.determine_relative_latency_corrections`).

.. admonition:: Example
    :class: dropdown

    Let's say we have specified two latency corrections in the CompilationConfig:

    .. code-block:: python

        compilation_config.hardware_options.latency_corrections = {
            "q0:res-q0.ro": LatencyCorrection(-20e-9),
            "q0:mw-q0.01": LatencyCorrection(120e9),
        }

    In this case, all operations on port ``"q0:mw"`` and clock ``"q0.01"`` will
    be delayed by 140 ns with respect to operations on port ``"q0:res"`` and
    clock ``"q0.ro"``.
"""


class SoftwareDistortionCorrection(DataStructure):
    """
    Software distortion correction information for a port-clock combination.

    .. admonition:: Example
        :class: dropdown

        .. jupyter-execute::

            from quantify.backends.types.common import (
                SoftwareDistortionCorrection
            )

            distortion_corrections = {
                "q0:fl-cl0.baseband": SoftwareDistortionCorrection(
                    filter_func="scipy.signal.lfilter",
                    input_var_name="x",
                    kwargs={
                        "b": [0, 0.25, 0.5],
                        "a": [1]
                    },
                    clipping_values=[-2.5, 2.5]
                )
            }
    """

    filter_func: str
    """The function applied to the waveforms."""
    input_var_name: str
    """The argument to which the waveforms will be passed in the filter_func."""
    kwargs: dict[str, list | NDArray]
    """The keyword arguments that are passed to the filter_func."""
    clipping_values: list | None = None
    """
    The optional boundaries to which the corrected pulses will be clipped,
    upon exceeding."""
    sampling_rate: float = 1e9
    """The sample rate of the corrected pulse, in Hz."""

    @field_validator("clipping_values")
    @classmethod
    def _only_two_clipping_values(
        cls,
        clipping_values,  # noqa: ANN001
    ) -> list | None | ValueError:  # noqa: ANN001
        if clipping_values and len(clipping_values) != 2:
            raise KeyError(
                f"Clipping values should contain only two values, min and max.\n"
                f"clipping_values: {clipping_values}"
            )
        return clipping_values


# See <https://docs.pydantic.dev/2.10/errors/usage_errors/#class-not-fully-defined>.
SoftwareDistortionCorrection.model_rebuild()


class ModulationFrequencies(DataStructure):
    """
    Modulation frequencies for a port-clock combination.

    .. admonition:: Example
        :class: dropdown

        .. jupyter-execute::

            from quantify.backends.types.common import (
                ModulationFrequencies
            )

            modulation_frequencies = {
                "q0:res-q0.ro": ModulationFrequencies(
                    interm_freq=None,
                    lo_freq=6e9,
                )
            }
    """

    interm_freq: float | None = None
    """The intermodulation frequency (IF) used for this port-clock combination."""
    lo_freq: float | None = None
    """The local oscillator frequency (LO) used for this port-clock combination."""


class MixerCorrections(DataStructure):
    """
    Mixer corrections for a port-clock combination.

    .. admonition:: Example
        :class: dropdown

        .. jupyter-execute::

            from quantify.backends.types.common import (
                MixerCorrections
            )

            mixer_corrections = {
                "q0:mw-q0.01": MixerCorrections(
                    dc_offset_i = -0.0542,
                    dc_offset_q = -0.0328,
                    amp_ratio = 0.95,
                    phase_error= 0.07,
                )
            }
    """

    dc_offset_i: float = 0.0
    """The DC offset on the I channel used for this port-clock combination."""
    dc_offset_q: float = 0.0
    """The DC offset on the Q channel used for this port-clock combination."""
    amp_ratio: float = 1.0
    """The mixer gain ratio used for this port-clock combination."""
    phase_error: float = 0.0
    """The mixer phase error used for this port-clock combination."""


class LocalOscillatorDescription(DataStructure):
    """Information needed to specify a Local Oscillator in the :class:`~.CompilationConfig`."""  # noqa: E501

    instrument_type: Literal["LocalOscillator"]
    """The field discriminator for this HardwareDescription datastructure."""
    instrument_name: str | None = None
    """The QCoDeS instrument name corresponding to this Local Oscillator."""
    generic_icc_name: str | None = None
    """The name of the :class:`~.GenericInstrumentCoordinatorComponent`
    corresponding to this Local Oscillator."""
    frequency_param: str = "frequency"
    """The QCoDeS parameter that is used to set the LO frequency."""
    power_param: str = "power"
    """The QCoDeS parameter that is used to set the LO power."""
    power: int | None = None
    """The power setting for this Local Oscillator."""


class Connectivity(DataStructure):
    """
    Connectivity between ports on the quantum device and control hardware
    inputs/outputs.

    The connectivity graph can be parsed from a list of edges, which are given by a
    set of two strings that each correspond to an input/output on an instrument or
    a port on the quantum device.

    .. note::
        To specify connections between more than one pair of ports at once, one can
        also specify a list of ports within the edge input (see example below, and also
        see :ref:`sec-connectivity-examples`).

    The connectivity graph can be drawn using :meth:`~.draw`, which groups the nodes
    according to the instrument name (specified by the string before the first ``"."``
    in the node name; the name is omitted for the quantum device).

    .. admonition:: Example
        :class: dropdown

        .. jupyter-execute::

            from quantify.backends.types.common import (
                Connectivity
            )

            connectivity_dict = {
                "graph": [
                    ("awg0.channel_0", "q0:mw"),
                    ("awg0.channel_1", "q1:mw"),
                    ("rom0.channel_0", ["q0:res", "q1:res"]),
                ]
            }

            connectivity = Connectivity.model_validate(connectivity_dict)
            connectivity.draw()
    """

    graph: Graph
    """
    The connectivity graph consisting of i/o ports (nodes) on the quantum device and on
    the control hardware, and their connections (edges).
    """

    @field_validator("graph", mode="before")
    @classmethod
    def _unroll_lists_of_ports_in_edges_input(cls, graph) -> list[tuple[Any, Any]]:  # type: ignore  # noqa: ANN001
        if isinstance(graph, list):
            list_of_edges = []
            for edge_input in graph:
                ports_0 = edge_input[0]
                ports_1 = edge_input[1]
                if not isinstance(ports_0, list):
                    ports_0 = [ports_0]
                if not isinstance(ports_1, list):
                    ports_1 = [ports_1]
                list_of_edges.extend([(p0, p1) for p0 in ports_0 for p1 in ports_1])
            graph = list_of_edges
        return graph

    def draw(
        self,
        ax: Axes | None = None,
        figsize: tuple[float, float] = (20, 10),
        **options,
    ) -> Axes:
        """
        Draw the connectivity graph using matplotlib.

        The nodes are positioned using a multipartite layout, where the nodes
        are grouped by instrument (identified by the first part of the node name).


        Parameters
        ----------
        ax
            Matplotlib axis to plot the figure on.
        figsize
            Optional figure size, defaults to something slightly larger that fits the
            size of the nodes.
        options
            optional keyword arguments that are passed to
            :code:`networkx.draw_networkx`.

        Returns
        -------
        :
            Matplotlib axis on which the figure is plotted.

        """
        if ax is None:
            _, ax = plt.subplots(figsize=figsize)

        options_dict = {
            "font_size": 10,
            "node_size": 1000,
            "node_color": "white",
            "edgecolors": "C0",
        }
        options_dict.update(options)

        # Group nodes by instrument:
        node_labels = {}
        grouped_nodes = {}
        for node in self.graph:
            if "." in node:
                node_instrument, node_port = node.split(sep=".", maxsplit=1)
                self.graph.nodes[node]["subset"] = node_instrument
            else:
                node_instrument = ""
                node_port = node
                self.graph.nodes[node]["subset"] = "quantum_device"
            if node_instrument not in grouped_nodes:
                grouped_nodes[node_instrument] = []
            grouped_nodes[node_instrument].append(node)
            node_labels[node] = node_port

        pos = nx.drawing.multipartite_layout(self.graph)

        # Draw boxes around instrument ports:
        for instrument, nodes in grouped_nodes.items():
            min_node_pos_x = min(pos[node][0] for node in nodes)
            max_node_pos_x = max(pos[node][0] for node in nodes)
            min_node_pos_y = min(pos[node][1] for node in nodes)
            max_node_pos_y = max(pos[node][1] for node in nodes)

            instrument_anchor = (min_node_pos_x - 0.05, min_node_pos_y - 0.05)  # type: ignore
            instrument_width = max_node_pos_x - min_node_pos_x + 0.1  # type: ignore
            instrument_height = max_node_pos_y - min_node_pos_y + 0.1  # type: ignore
            ax.add_patch(  # type: ignore
                Rectangle(
                    xy=instrument_anchor,
                    width=instrument_width,
                    height=instrument_height,
                    fill=False,
                    color="b",
                )
            )
            ax.text(x=min_node_pos_x, y=max_node_pos_y + 0.1, s=instrument, color="b")  # type: ignore

        nx.draw_networkx(self.graph, pos=pos, ax=ax, labels=node_labels, **options_dict)
        ax.set_axis_off()  # type: ignore

        return ax  # type: ignore


# See <https://docs.pydantic.dev/2.10/errors/usage_errors/#class-not-fully-defined>.
Connectivity.model_rebuild()


@dataclass
class ThresholdedTriggerCountMetadata:
    """Metadata specifically for the ThresholdedTriggerCount acquisition."""

    threshold: int
    """The threshold of the ThresholdedTriggerCount acquisition."""
    # condition: TriggerCondition
    # """
    # The comparison condition (greater-equal, less-than) for the
    # ThresholdedTriggerCount acquisition.
    # """
