""" CsvPaths' intent is to help you manage and automate your use
    of the CsvPath library. it makes it easier to scale your CSV quality control. """

import os
import traceback
from uuid import uuid4, UUID
from abc import ABC, abstractmethod
from typing import List, Any
from datetime import datetime, timezone
from .managers.errors.error import Error
from .managers.errors.error_comms import ErrorCommunications
from .managers.errors.error_manager import ErrorManager
from .managers.errors.error_collector import ErrorCollector
from .util.config import Config
from .util.log_utility import LogUtility
from .util.metadata_parser import MetadataParser
from .util.exceptions import InputException, CsvPathsException
from .util.references.reference_parser import ReferenceParser
from .util.run_home_maker import RunHomeMaker
from .managers.paths.paths_manager import PathsManager
from .managers.files.file_manager import FileManager
from .managers.results.results_manager import ResultsManager
from .managers.results.result import Result
from .util.box import Box
from . import CsvPath


class CsvPathsCoordinator(ABC):
    """
        @private
    This abstract class defines callbacks for CsvPath instances to
    broadcast state to their siblings through CsvPaths. A CsvPath
    instance might stop the entire run, rather than each CsvPath
    instance needing to contain the same logic that stops their
    participation in a run.
    """

    @abstractmethod
    def stop_all(self) -> None:  # pragma: no cover
        """
        @private
        Stops every CsvPath instance in a run"""

    @abstractmethod
    def fail_all(self) -> None:  # pragma: no cover
        """
        @private
        Fails every CsvPath instance in a run"""

    @abstractmethod
    def skip_all(self) -> None:  # pragma: no cover
        """
        @private
        skips the line for every CsvPath instance in a run"""

    @abstractmethod
    def advance_all(self, lines: int) -> None:  # pragma: no cover
        """
        @private
        advances every CsvPath instance in a run"""


class CsvPaths(CsvPathsCoordinator, ErrorCollector):
    """
    a CsvPaths instance manages applying any number of csvpaths
    to any number of files. CsvPaths applies sets of csvpaths
    to a given file, on demand. Think of CsvPaths as a session
    object. It gives you a way to manage files, csvpaths, and
    the results generated by applying paths to files. It is not
    intended for concurrent use. If you need multiple threads,
    create multiple CsvPaths instances.
    """

    # pylint: disable=too-many-instance-attributes
    def __init__(
        self,
        *,
        delimiter=",",
        quotechar='"',
        skip_blank_lines=True,
        print_default=True
        # config: Config = None,
    ):
        #
        # we gain nothing by allowing the config to be passed in. and it adds complexity.
        #
        self._config = Config()  # if config is None else config
        #
        # in a few cases, mainly s3 and sftp connection or config sharing
        # we need to pass some state around. it's ugly, but logically not
        # terrible and better than other options bar major refactoring.
        #
        box = Box()
        box.add(Box.CSVPATHS_CONFIG, self._config)
        #
        # managers centralize activities, offer async potential, and
        # are where integrations hook in. ErrorManager functionality
        # must be available in CsvPath too. The others are CsvPaths
        # only.
        #
        self._paths_manager = None
        self._file_manager = None
        self._results_manager = None
        self._ecoms = None
        self._error_manager = None
        self._set_managers()
        #
        #
        #
        self.print_default = print_default
        """ @private """
        self.delimiter = delimiter
        """ @private """
        self.quotechar = quotechar
        """ @private """
        self.skip_blank_lines = skip_blank_lines
        """ @private """
        self.current_matcher: CsvPath = None
        """ @private """
        self._config._assure_logs_path()
        self.logger = LogUtility.logger(self)
        """ @private """
        self._errors = []
        # coordinator attributes
        self._stop_all = False
        self._fail_all = False
        self._skip_all = False
        self._advance_all = 0
        #
        # TODO: we probably don't need all three of these
        #
        self._current_run_time = None
        self._run_time_str = None
        self._last_run_dir = None
        #
        #
        #
        self.named_paths_name = None
        """ @private """
        self.named_file_name = None
        """ @private """
        #
        # metrics is for OTLP OpenTelemetry. it should only
        # be used by the OTLP listener. it is here because
        # the integration may need a long-lived presence. if
        # needed, the first OTLP listener will set it up
        # before spinning up a thread. any other OTLP
        # listener threads that need to use a long-lived metric
        # will work with this property.
        #
        self.metrics = None
        """ @private """
        #
        # this metadata is generated at the run start. it is
        # the coordinating metadata for the run in the sense
        # that its UUID is the correlation ID all metadata
        # objects from the same run use to understand their
        # relationship.
        #
        self.run_metadata = None
        #
        # for many purposes csvpaths can clean up at the end of
        # runs. in some cases you may want to do it manually
        # later. e.g. if you are doing a run and then want to
        # inspect or use the results in another run you might
        # want to keep the connections live. when ready, call
        # wrap_up(). defaults to True.
        #
        self._wrap_up_automatically = True
        """ @private """
        self.logger.info("initialized CsvPaths")

    def _set_managers(self) -> None:
        self.paths_manager = PathsManager(csvpaths=self)
        self.file_manager = FileManager(csvpaths=self)
        self.results_manager = ResultsManager(csvpaths=self)
        self.ecoms = ErrorCommunications(csvpaths=self)
        self.error_manager = ErrorManager(csvpaths=self)

    @property
    def wrap_up_automatically(self) -> bool:
        return self._wrap_up_automatically

    @wrap_up_automatically.setter
    def wrap_up_automatically(self, auto: bool) -> None:
        self._wrap_up_automatically = auto

    @property
    def ecoms(self) -> ErrorCommunications:
        """@private"""
        return self._ecoms

    @ecoms.setter
    def ecoms(self, ec: ErrorCommunications) -> None:
        """@private"""
        self._ecoms = ec

    @property
    def last_run_dir(self) -> str:
        return self._last_run_dir

    @property
    def file_manager(self) -> FileManager:
        return self._file_manager

    @file_manager.setter
    def file_manager(self, m: FileManager) -> None:
        self._file_manager = m

    @property
    def results_manager(self) -> ResultsManager:
        return self._results_manager

    @results_manager.setter
    def results_manager(self, m: ResultsManager) -> None:
        self._results_manager = m

    @property
    def paths_manager(self) -> PathsManager:
        return self._paths_manager

    @paths_manager.setter
    def paths_manager(self, m: PathsManager) -> None:
        self._paths_manager = m

    @property
    def error_manager(self) -> ErrorManager:
        return self._error_manager

    @error_manager.setter
    def error_manager(self, em: ErrorManager) -> None:
        if em.csvpaths is None:
            raise Exception("CsvPaths cannot be None")
        self._error_manager = em

    @property
    def current_run_time(self) -> datetime:
        maker = RunHomeMaker(self)
        return maker.current_run_time

    def csvpath(self) -> CsvPath:
        """Gets a CsvPath object primed with a reference to this CsvPaths"""
        path = CsvPath(
            csvpaths=self,
            delimiter=self.delimiter,
            quotechar=self.quotechar,
            skip_blank_lines=self.skip_blank_lines,
            #
            # in the usual case we don't want csvpaths and its csvpath children
            # to share the same config. sharing doesn't offer much. the flexibility
            # of having separate configs is valuable.
            #
            # config=None,
            print_default=self.print_default,
            error_manager=self.error_manager,
        )
        return path

    def stop_all(self) -> None:  # pragma: no cover
        self._stop_all = True

    def fail_all(self) -> None:  # pragma: no cover
        self._fail_all = True

    def skip_all(self) -> None:  # pragma: no cover
        self._skip_all = True

    def advance_all(self, lines: int) -> None:  # pragma: no cover
        self._advance_all = lines

    @property
    def errors(self) -> List[Error]:  # pylint: disable=C0116
        """@private
        generally you should be looking at results_manager or error_manager for errors.
        """
        return self._errors

    def collect_error(self, error: Error) -> None:  # pylint: disable=C0116
        """@private"""
        self._errors.append(error)

    def has_errors(self) -> bool:  # pylint: disable=C0116
        """@private
        generally you should be looking at results_manager or error_manager for errors.
        """
        return len(self._errors) > 0

    @property
    def config(self) -> Config:  # pylint: disable=C0116
        """@private"""
        if not self._config:
            self._config = Config()  # pragma: no cover
        return self._config

    #
    # this is the preferred way to update config. it is preferred because
    # csvpath and csvpaths work off the same config file, even though they,
    # in some cases, have separate keys. if you update the config directly
    # before a run starts using the CsvPaths's Config you have to remember
    # to save and reload for it to effect both CsvPaths and CsvPath. this
    # method does the save and reload every time.
    #
    def add_to_config(self, section, key, value) -> None:
        """@private"""
        self.config.add_to_config(section=section, key=key, value=value)
        self.config.save_config()
        self.config.reload()
        self._set_managers()

    # =========================
    # cleanup, wrapup calls
    # =========================

    def clear_run_coordination(self) -> None:
        """@private
        run coordination is the set of signals that csvpaths send to affect
        one another through the CsvPaths instance"""
        self._stop_all = False
        self._fail_all = False
        self._skip_all = False
        self._advance_all = 0
        self._current_run_time = None
        self._run_time_str = None
        self.logger.debug("Cleared run coordination")

    #
    # we do not currently use the paths param. what was the purpose?
    #
    def clean(self, *, paths=None) -> None:
        """@private
        at this time we do not recommend reusing CsvPaths, but it is doable
        you should clean before reuse unless you want to accumulate results."""
        self._set_managers()
        self.clear_run_coordination()
        self._errors = []
        self.named_file_name = None
        self.current_matcher = None
        self.named_paths_name = None
        self.run_metadata = None

    def wrap_up(self) -> None:
        #
        # leave clean() and clear_run_coordination() in case there
        # is some reason we might need to see the residual. those are
        # called at top of run.
        #
        # cleanup shared connections, etc.
        #
        box = Box()
        ds = []
        for k, v in box.get_my_stuff().items():
            self.logger.debug(f"csvpaths.wrapping up: {k}: {v}")
            if hasattr(v, "close"):
                ds.append(k)
        for _ in ds:
            try:
                v = box.get(_)
                v.close()
                box.remove(_)
            except Exception as e:
                msg = f"Error in wrapping up: {e}"
                self.error_manager.handle_error(source=self, msg=msg)
                if self.ecoms.do_i_raise():
                    raise CsvPathsException(msg)

    # =========================
    # prep csvpath children
    # =========================

    def _load_csvpath(
        self,
        *,
        csvpath: CsvPath,
        path: str,
        file: str,
        pathsname: str = None,
        filename,
        by_line: bool = False,
        crt: str,
        index: int = -1,
    ) -> None:
        """@private"""
        # file is the physical file (+/- if preceding mode) filename is the named-file name
        self.logger.debug("Beginning to load csvpath %s with file %s", path, file)
        csvpath.named_paths_name = pathsname
        self.named_paths_name = pathsname
        csvpath.named_file_name = filename
        self.named_file_name = filename
        #
        # by_line==True means we are starting a run that is breadth-first ultimately using
        # next_by_line(). by_line runs cannot be source-mode preceding and have different
        # semantics around csvpaths influencing one another.
        #
        # we strip comments from above the path so we need to extract them first
        #
        path = MetadataParser(self).extract_metadata(instance=csvpath, csvpath=path)
        identity = csvpath.identity
        self.logger.debug("Csvpath %s after metadata extract: %s", identity, path)
        #
        # update the run settings using the metadata fields we just collected
        #
        csvpath.update_settings_from_metadata()
        #
        # historic note: file references are resolved in the filemanager. and we don't care
        # for if we are using source-mode preceding for that.
        #
        if csvpath.data_from_preceding is True and index > 0:
            #
            # if index  == 0 we can't look to the preceding csvpath in the run.
            # if it is -1 we didn't pass in an index -- possibly because by_line.
            #
            if by_line is True:
                raise CsvPathsException(
                    "Breadth-first runs do not support source-mode preceding because each line of data flows through each csvpath in order already"
                )
            #
            # we are in source-mode: preceding that means we ignore the original data file path and
            # instead use the data.csv from the preceding csvpath. that is, assuming there is a
            # preceding csvpath and it created and saved data.
            #
            # find the preceding csvpath in the named-paths
            #
            # we may have a reference like: $sourcemode.csvpaths.source2:from. if so
            # we just need the named-paths name.
            #
            # cannot do this anymore because ref templates.
            # resman should get the whole reference
            #
            if pathsname.startswith("$"):
                ref = ReferenceParser(pathsname, csvpaths=self)
                idname = self.paths_manager._get_from_names(ref.root_major, identity)[0]
                idnames = self.paths_manager.get_identified_path_names_in(pathsname)
                i = idnames.index(idname)
                identity = idnames[i - 1]
                file = f"{crt}{os.sep}{identity}{os.sep}data.csv"
                self.logger.debug("csvpaths.load_csvpath: idnames: %s", idnames)
                self.logger.debug("csvpaths.load_csvpath: idname: %s", idname)
                self.logger.debug("csvpaths.load_csvpath: identity: %s", identity)
                self.logger.debug("csvpaths.load_csvpath: ref: %s", ref)
                self.logger.debug("csvpaths.load_csvpath: pathsname: %s", pathsname)
                self.logger.debug("csvpaths.load_csvpath: file: %s", file)
            else:
                #
                # not a reference
                #
                result = self.results_manager.get_last_named_result(
                    name=pathsname, before=csvpath.identity
                )
                if result is not None:
                    # get its data.csv path for this present run
                    # swap in that path for the regular origin path
                    file = result.data_file_path
                    #
                    # this was added when the mode was created. it duplicates info available in manifests.
                    #
                    csvpath.metadata["source-mode-source"] = file
                    self.logger.info(
                        "Csvpath identified as %s uses last csvpath's data.csv at %s as source",
                        csvpath.identity,
                        file,
                    )
                else:
                    self.logger.warning(
                        "No preceding data file for csvpath %s running in source-mode",
                        csvpath.identity,
                    )

        f = path.find("[")
        self.logger.debug("Csvpath matching part starts at char # %s", f)
        apath = f"${file}{path[f:]}"
        self.logger.info("Parsing csvpath %s", apath)
        csvpath.parse(apath)
        #
        # ready to run. time to register the run. this is separate from
        # the run_register.py (ResultsRegister) event
        #
        self.logger.debug("Done loading csvpath")

    def _get_named_paths(self, pathsname: str) -> list:
        if pathsname is None:
            raise ValueError("Named-paths name cannot be None")
        paths = self.paths_manager.get_named_paths(pathsname)
        if paths is None:
            raise InputException(f"No named-paths found for {pathsname}")
        if not isinstance(paths, list):
            raise InputException(
                f"Named-paths group {pathsname} must be represented as a list[str]"
            )
        if len(paths) == 0:
            raise InputException(f"Named-paths group {pathsname} is empty")
        if "" in paths:
            raise InputException(
                f"Named-paths group {pathsname} has one or more empty csvpaths"
            )
        return paths

    # =========================
    # main functions
    # =========================

    def collect_paths(
        self, *, pathsname: str, filename: str, template: str = None
    ) -> None:
        """
        Sequentially does a CsvPath.collect() on filename for every named-path in the
        specified named-paths group. if a file reference is passed in, we iterate on
        every concrete data file selected by the reference. unlike the named-paths
        iteration, each file we iterate on becomes its own call to collect_paths. lines
        are collected into a results object, not returned.
        """
        files = self.file_manager.get_named_file(filename)
        if files is None:
            raise InputException(f"No named-file found for {filename}")
        if isinstance(files, list):
            for file in files:
                self._collect_paths(
                    pathsname=pathsname, filename=filename, template=template, file=file
                )
        else:
            self._collect_paths(
                pathsname=pathsname, filename=filename, template=template, file=files
            )

    def _collect_paths(
        self, *, pathsname: str, file: str, filename: str, template: str = None
    ) -> None:
        #
        # if template is None we need to go find any template that was given when
        # the named-paths were loaded.
        #
        paths = self._get_named_paths(pathsname)
        if template is None:
            template = self.paths_manager.get_template_for_paths(pathsname)
        self.logger.info(
            "Prepping %s and %s with template %s", filename, pathsname, template
        )
        self.clean(paths=pathsname)
        self.logger.info(
            "Beginning collect_paths %s with %s paths using template %s",
            pathsname,
            len(paths),
            template,
        )
        #
        # run identification and directories created here
        #
        maker = RunHomeMaker(self)
        crt = maker.get_run_dir(
            paths_name=pathsname, file_name=filename, template=template
        )
        #
        # capture the last run dir for the benefit of the caller
        #
        self._last_run_dir = crt

        results = []
        #
        # exp
        # adding uuid for the run as a whole
        run_uuid = uuid4()
        #
        # run starts here
        #
        self.run_metadata = self.results_manager.start_run(
            run_dir=crt,
            pathsname=pathsname,
            filename=filename,
            file=file,
            run_uuid=run_uuid,
        )
        #
        #
        #
        for i, path in enumerate(paths):
            csvpath = self.csvpath()
            if not csvpath.will_run:
                continue
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
                run_uuid=run_uuid,
            )
            # casting a broad net because if "raise" not in the error policy we
            # want to never fail during a run
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                    index=i,
                )
                #
                # if run-mode: no-run we skip ahead without saving results
                #
                if not csvpath.will_run:
                    continue
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                lines = result.lines
                self.logger.debug("Collecting lines using a %s", type(lines))
                csvpath.collect(lines=lines)
                if lines is None:
                    self.logger.error(  # pragma: no cover
                        "Unexpected None for lines after collect_paths: file: %s, match: %s",
                        file,
                        csvpath.match,
                    )
                #
                # TODO: unmatched needs additional support for streaming very large files
                #
                result.unmatched = csvpath.unmatched
            except Exception as ex:  # pylint: disable=W0718
                if self.error_manager.csvpaths is None:
                    raise Exception("ErrorManager's CsvPaths cannot be None")
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        #
        # update/write run manifests here
        #  - validity (are all paths valid)
        #  - paths-completeness (did they all run and complete)
        #  - method (collect, fast_forward, next)
        #  - timestamp
        #
        self.clear_run_coordination()
        self.logger.info(
            "Completed collect_paths %s with %s paths", pathsname, len(paths)
        )
        if self.wrap_up_automatically:
            self.wrap_up()

    def fast_forward_paths(
        self, *, pathsname: str, filename: str, template: str = None
    ) -> None:
        """
        Sequentially does a CsvPath.fast_forward() on filename for every named path. No matches are collected.
        """
        files = self.file_manager.get_named_file(filename)
        if files is None:
            raise InputException(f"No named-file found for {filename}")
        if isinstance(files, list):
            for file in files:
                self._fast_forward_paths(
                    pathsname=pathsname, filename=filename, template=template, file=file
                )
        else:
            self._fast_forward_paths(
                pathsname=pathsname, filename=filename, template=template, file=files
            )

    def _fast_forward_paths(
        self, *, pathsname: str, file: str, filename: str, template: str = None
    ) -> None:
        """Sequentially does a CsvPath.fast_forward() on filename for every named path. No matches are collected."""
        paths = self._get_named_paths(pathsname)
        if template is None:
            template = self.paths_manager.get_template_for_paths(pathsname)
        self.logger.info(
            "Prepping %s and %s with template %s", filename, pathsname, template
        )
        self.clean(paths=pathsname)
        self.logger.info(
            "Beginning FF %s with %s paths against file %s. No match results will be held.",
            pathsname,
            len(paths),
            filename,
        )
        #
        # run identification and directories created here
        #
        maker = RunHomeMaker(self)
        crt = maker.get_run_dir(
            paths_name=pathsname, file_name=filename, template=template
        )
        #
        # capture the last run dir for the benefit of the caller
        #
        self._last_run_dir = crt
        #
        #
        #
        run_uuid = uuid4()
        #
        # run starts here
        #
        self.run_metadata = self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename, run_uuid=run_uuid
        )
        #
        #
        #
        results = []
        for i, path in enumerate(paths):
            csvpath = self.csvpath()
            self.logger.debug("Beginning to FF CsvPath instance: %s", csvpath)
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
                run_uuid=run_uuid,
            )
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                )
                #
                # if run-mode: no-run we skip ahead without saving results
                #
                if not csvpath.will_run:
                    continue
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                self.logger.info(
                    "Parsed csvpath %s pointed at %s and starting to fast-forward",
                    i,
                    file,
                )
                csvpath.fast_forward()
                self.logger.info(
                    "Completed fast forward of csvpath %s against %s", i, file
                )
            except Exception as ex:  # pylint: disable=W0718
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()
        self.logger.info(
            "Completed fast_forward_paths %s with %s paths", pathsname, len(paths)
        )
        if self.wrap_up_automatically:
            self.wrap_up()

    def next_paths(
        self,
        *,
        pathsname: str,
        filename: str,
        collect: bool = False,
        template: str = None,
    ):  # pylint: disable=R0914
        """
        Does a CsvPath.next() on filename for every line against every named path in sequence
        """
        paths = self._get_named_paths(pathsname)
        if template is None:
            template = self.paths_manager.get_template_for_paths(pathsname)

        file = self.file_manager.get_named_file(filename)
        if isinstance(file, list):
            raise ValueError(
                "Multiple file runs are not supported with this iterative next() method"
            )
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        self.logger.info("Beginning next_paths with %s paths", len(paths))

        #
        # run identification and directories created here
        #
        maker = RunHomeMaker(self)
        crt = maker.get_run_dir(
            paths_name=pathsname, file_name=filename, template=template
        )
        #
        # capture the last run dir for the benefit of the caller
        #
        self._last_run_dir = crt
        #
        #
        #
        run_uuid = uuid4()
        #
        # run starts here
        #
        self.run_metadata = self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename, run_uuid=run_uuid
        )
        #
        #
        #
        results = []
        for i, path in enumerate(paths):
            if self._skip_all:
                skip_err = "Found the skip-all signal set. skip_all() is"
                skip_err = f"{skip_err} only for breadth-first runs using the"
                skip_err = f"{skip_err} '_by_line' methods. It has the same"
                skip_err = f"{skip_err} effect as skip() in a"
                skip_err = f"{skip_err} serial run like this one."
                self.logger.error(skip_err)
            if self._stop_all:
                self.logger.warning("Stop-all set. Shutting down run.")
                break
            if self._advance_all > 0:
                advance_err = "Found the advance-all signal set. advance_all() is"
                advance_err = f"{advance_err} only for breadth-first runs using the"
                advance_err = f"{advance_err} '_by_line' methods. It has the same"
                advance_err = f"{advance_err} effect as advance() in a"
                advance_err = f"{advance_err} serial run like this one."
                self.logger.error(advance_err)
            csvpath = self.csvpath()
            result = Result(
                csvpath=csvpath,
                file_name=filename,
                paths_name=pathsname,
                run_index=i,
                run_time=self.current_run_time,
                run_dir=crt,
                run_uuid=run_uuid,
            )
            if self._fail_all:
                self.logger.warning(
                    "Fail-all set. Failing all remaining CsvPath instances in the run."
                )
                csvpath.is_valid = False
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=file,
                    pathsname=pathsname,
                    filename=filename,
                    crt=crt,
                )
                #
                # if run-mode: no-run we skip ahead without saving results
                #
                if not csvpath.will_run:
                    continue
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
                for line in csvpath.next():
                    if collect:
                        result.append(line)
                        result.unmatched = csvpath.unmatched
                    yield line
            except Exception as ex:  # pylint: disable=W0718
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    self.results_manager.save(result)
                    raise
            self.results_manager.save(result)
            results.append(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=crt, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()
        if self.wrap_up_automatically:
            self.wrap_up()

    # =============== breadth first processing ================

    def collect_by_line(
        self,
        *,
        pathsname,
        filename,
        if_all_agree=False,
        collect_when_not_matched=False,
        template: str = None,
    ):
        """
        Does a CsvPath.collect() on filename where each row is considered by
        every named path before the next row starts next_by_line for if_all_agree
        and collect_when_not_matched.
        """
        files = self.file_manager.get_named_file(filename)
        if isinstance(files, str):
            files = [files]
        self.logger.info(
            "Starting collect_by_line for paths: %s and file: %s with template: %s",
            pathsname,
            filename,
            template,
        )
        lines = []
        for file in files:
            for line in self._next_by_line(  # pylint: disable=W0612
                pathsname=pathsname,
                filename=filename,
                collect=True,
                if_all_agree=if_all_agree,
                collect_when_not_matched=collect_when_not_matched,
                file=file,
                template=template,
            ):
                # re: W0612: we need 'line' in order to do the iteration. we have to iterate.
                lines.append(line)
        self.logger.info(
            "Completed collect_by_line for paths: %s and file: %s", pathsname, filename
        )
        #
        # the results have all the lines according to what CsvPath captured them, but
        # since we're doing if_all_agree T/F we should return the union here. for some
        # files this obviously makes the data in memory problem even bigger, but it's
        # operator's responsibility to know if that will be a problem for their use
        # case.
        #
        return lines

    def fast_forward_by_line(
        self,
        *,
        pathsname,
        filename,
        if_all_agree=False,
        collect_when_not_matched=False,
        template: str = None,
    ):
        """
        Does a CsvPath.fast_forward() on filename where each row is considered by
        every named path before the next row starts next_by_line for if_all_agree
        and collect_when_not_matched.
        """
        files = self.file_manager.get_named_file(filename)
        if isinstance(files, str):
            files = [files]
        self.logger.info(
            "Starting fast_forward_by_line for paths: %s and file: %s with template: %s",
            pathsname,
            filename,
            template,
        )
        for file in files:
            for line in self._next_by_line(  # pylint: disable=W0612
                pathsname=pathsname,
                filename=filename,
                collect=False,
                if_all_agree=if_all_agree,
                collect_when_not_matched=collect_when_not_matched,
                file=file,
                template=template,
            ):
                # re: W0612: we need 'line' in order to do the iteration. we have to iterate.
                pass
        self.logger.info(
            "Completed fast_forward_by_line for paths: %s and file: %s with template: %s",
            pathsname,
            filename,
            template,
        )

    def next_by_line(  # pylint: disable=R0912,R0915,R0914
        self,
        *,
        pathsname,
        filename,
        collect: bool = False,
        if_all_agree=False,
        collect_when_not_matched=False,
        template: str = None,
    ) -> List[Any]:
        #
        # we're doing a programmatic use when we use next_by_line() so we don't allow
        # multiple file runs. we check that here; the other _by_line()s skip it.
        #
        file = self.file_manager.get_named_file(filename)
        if isinstance(file, list):
            raise ValueError(
                "Multi-file runs are not supported with iterative next_by_line()"
            )
        return self._next_by_line(
            pathsname=pathsname,
            filename=filename,
            collect=collect,
            if_all_agree=if_all_agree,
            collect_when_not_matched=collect_when_not_matched,
            template=template,
            file=file,
        )

    def _next_by_line(  # pylint: disable=R0912,R0915,R0914
        self,
        *,
        pathsname,
        filename,
        collect: bool = False,
        if_all_agree=False,
        collect_when_not_matched=False,
        template: str = None,
        file: str,
    ) -> List[Any]:
        """Does a CsvPath.next() on filename where each row is considered
        by every named path before the next row starts.

        if_all_agree=True means all the CsvPath instances must match for
        the line to be kept. However, every CsvPath instance will keep its
        own matches in its results regardless of if every line kept was
        returned to the caller by CsvPaths.

        collect_when_not_matched=True inverts the match so that lines
        which did not match are returned, rather than the default behavior.
        """
        # re: R0912 -- absolutely. plan to refactor.
        if isinstance(file, list):
            ValueError(
                "This method only supports iterating through a single file, not a list of files"
            )
        self.logger.info("Prepping %s and %s", filename, pathsname)
        self.clean(paths=pathsname)
        paths = self._get_named_paths(pathsname)
        if template is None:
            template = self.paths_manager.get_template_for_paths(pathsname)
        #
        # create run identity and directories
        #
        maker = RunHomeMaker(self)
        crt = maker.get_run_dir(
            paths_name=pathsname, file_name=filename, template=template
        )
        #
        # capture the last run dir for the benefit of the caller
        #
        self._last_run_dir = crt
        #
        # also use of crt below
        #
        csvpath_objects = self._load_csvpath_objects(
            paths=paths,
            named_file=file,
            collect_when_not_matched=collect_when_not_matched,
            filename=filename,
            pathsname=pathsname,
            crt=crt,
        )
        #
        # prep has to come after _load_csvpath_objects because we need the identity or
        # indexes to be stable and the identity is found in the load, if it exists.
        #
        self._prep_csvpath_results(
            csvpath_objects=csvpath_objects,
            filename=filename,
            pathsname=pathsname,
            crt=crt,
        )
        #
        # setting file into the csvpath is less obviously useful at CsvPaths
        # but we'll do it for consistency.
        #
        self.logger.info("Beginning next_by_line with %s paths", len(csvpath_objects))
        reader = FileManager.get_reader(
            file, delimiter=self.delimiter, quotechar=self.quotechar
        )
        stopped_count: List[int] = []
        for line in reader.next():
            # for line in reader:  # pylint: disable=R1702
            # question to self: should this default be in a central place
            # so that we can switch to OR, in part by changing the default?
            keep = if_all_agree
            self._skip_all = False
            self._advance_all = 0
            try:
                # p is a (CsvPath, List[List[str]]) where the second item is
                # the line-by-line results of the first item's matching
                for p in csvpath_objects:
                    #
                    # if run-mode: no-run we skip ahead without saving results
                    #
                    if not p[0].will_run:
                        continue
                    self.current_matcher = p[0]
                    if self._fail_all:
                        self.logger.warning(
                            "Fail-all set. Setting CsvPath is_valid to False."
                        )
                        self.current_matcher.is_valid = False
                    if self._stop_all:
                        self.logger.warning("Stop-all set. Shutting down run.")
                        self.current_matcher.stopped = True
                        continue
                    if self._skip_all:
                        self.logger.warning("Skip-all set. Continuing to next.")
                        #
                        # all following CsvPaths must have their line_monitors incremented
                        #
                        self.current_matcher.track_line(line)
                        continue
                    if self._advance_all > 0:
                        logtxt = "Advance-all set. Setting advance. "
                        logtxt = f"{logtxt}CsvPath and its Matcher will handle the advancing."
                        self.logger.info(logtxt)
                        #
                        # CsvPath will handle advancing so we don't need to do
                        # anything, including track_line(line). we just need to
                        # see if we're setting advance or increasing it.
                        #
                        a = self.current_matcher.advance_count
                        if self._advance_all > a:
                            self.current_matcher.advance_count = self._advance_all
                        #
                        # all following CsvPaths must have their
                        # advance incremented -- with the advance not being simply
                        # additive, have to be mindful of any existing advance
                        # count!
                        #
                    if self.current_matcher.stopped:  # pylint: disable=R1724
                        continue
                    #
                    # allowing the match to happen regardless of keep
                    # because we may want side-effects or to have different
                    # results in different named-results, as well as the
                    # union
                    #
                    self.logger.debug(
                        "considering line with csvpath identified as: %s",
                        self.current_matcher.identity,
                    )
                    matched = False
                    self.current_matcher.track_line(line)
                    #
                    # re: W0212: treating _consider_line something like package private
                    #
                    matched = (
                        self.current_matcher._consider_line(  # pylint:disable=W0212
                            line
                        )
                    )
                    if self.current_matcher.stopped:
                        stopped_count.append(1)
                    if if_all_agree:
                        keep = keep and matched
                    else:
                        keep = keep or matched
                    #
                    # not doing continue if we have if_all_agree and not keep as we
                    # used to do allows individual results to have lines that in
                    # aggregate we do not keep.
                    #
                    if matched and collect:
                        line = self.current_matcher.limit_collection(line)
                        p[1].append(line)
            except Exception as ex:  # pylint: disable=W0718
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                if self.ecoms.do_i_raise():
                    for r in csvpath_objects:
                        result = r[1]
                        result.unmatched = r[0].unmatched
                        self.results_manager.save(result)
                    raise
            # we yield even if we stopped in this iteration.
            # caller needs to see what we stopped on.
            #
            # ! we only yield if keep is True
            #
            if keep:
                yield line
            if sum(stopped_count) == len(csvpath_objects):
                break
        results = []
        for r in csvpath_objects:
            result = r[1]
            results.append(result)
            result.unmatched = r[0].unmatched
            self.results_manager.save(result)
        #
        # run ends here
        #
        self.results_manager.complete_run(
            run_dir=results[0].run_dir, pathsname=pathsname, results=results
        )
        self.clear_run_coordination()
        if self.wrap_up_automatically:
            self.wrap_up()

    def _load_csvpath_objects(
        self,
        *,
        paths: List[str],
        named_file: str,
        collect_when_not_matched=False,
        filename,
        pathsname,
        crt: str,
    ):
        """@private"""
        csvpath_objects = []
        for path in paths:
            csvpath = self.csvpath()
            csvpath.collect_when_not_matched = collect_when_not_matched
            try:
                self._load_csvpath(
                    csvpath=csvpath,
                    path=path,
                    file=named_file,
                    filename=filename,
                    pathsname=pathsname,
                    by_line=True,
                    crt=crt,
                )
                if csvpath.data_from_preceding is True:
                    # this exception raise may be redundant.
                    raise CsvPathsException(
                        "Csvpath identified as {csvpath.identity} is set to use preceding data, but CsvPaths's by_line methods do not permit that"
                    )
                csvpath_objects.append([csvpath, []])
            except Exception as ex:  # pylint: disable=W0718
                ex.trace = traceback.format_exc()
                ex.source = self
                # the error collector is the Results. it registers itself with
                # the csvpath as the error collector. not as straightforward but
                # effectively same as we do above
                self.error_manager.handle_error(source=self, msg=f"{ex}")
        return csvpath_objects

    def _prep_csvpath_results(self, *, csvpath_objects, filename, pathsname, crt: str):
        """@private"""
        #
        #
        #
        run_uuid = uuid4()
        #
        # run starts here
        #
        self.run_metadata = self.results_manager.start_run(
            run_dir=crt, pathsname=pathsname, filename=filename, run_uuid=run_uuid
        )
        #
        #
        #
        for i, csvpath in enumerate(csvpath_objects):
            try:
                #
                # Result sets itself into its CsvPath as error collector, printer, etc.
                #
                result = Result(
                    csvpath=csvpath[0],
                    file_name=filename,
                    paths_name=pathsname,
                    lines=csvpath[1],
                    run_index=i,
                    run_time=self.current_run_time,
                    run_dir=crt,
                    by_line=True,
                    run_uuid=run_uuid,
                )
                csvpath[1] = result
                #
                # the add has to come after _load_csvpath because we need the identity or index
                # to be stable and the identity is found in load, if it exists.
                #
                self.results_manager.add_named_result(result)
            except Exception as ex:  # pylint: disable=W0718
                self.error_manager.handle_error(source=self, msg=f"{ex}")
                #
                # keep this comment for modelines avoidance
                #
                ...
