# pylint: disable=C0114
from typing import Any
from csvpath.matching.util.expression_utility import ExpressionUtility
from csvpath.matching.productions import Term, Variable, Header
from ..function_focus import ValueProducer
from ..function import Function
from ..args import Args


class Subtotal(ValueProducer):
    """returns the running sum of values aggregated by another value"""

    def check_valid(self) -> None:
        self.description = [
            self._cap_name(),
            self.wrap(
                """\
                    Returns the running sum of values of a type generated by another source.

                    Each value of the type is used as a tracking value that holds the subtotal.
                    This allows you to collect information using a function call like this:

                        subtotal.mycustomers( #company, #days )

                    And access the subtotal like this:

                        @mycustomers.Acme == 365 -> print("Customer for a year")

                    By default subtotal() tracks its value in the variable @subtotal. Add a
                    name qualifier as shown above to use another name.
                """
            ),
        ]
        self.name_qualifier = True
        self.args = Args(matchable=self)
        a = self.args.argset(2)
        a.arg(
            name="type to subtotal",
            types=[Variable, Function, Header],
            actuals=[None, Any],
        )
        a.arg(
            name="value to subtotal",
            types=[Variable, Function, Header],
            actuals=[float, int],
        )
        self.args.validate(self.siblings())
        super().check_valid()

    def _produce_value(self, skip=None) -> None:
        t = self._value_one(skip=skip)
        c = self._value_two(skip=skip)
        varname = self.first_non_term_qualifier(self.name)
        val = self.matcher.get_variable(varname, tracking=t, set_if_none=0)
        # this may blow up if c is not convertable, but that is fine
        val += ExpressionUtility.to_float(c)
        self.matcher.set_variable(varname, tracking=t, value=val)
        self.value = val

    def _decide_match(self, skip=None) -> None:
        self.to_value(skip=skip)
        self.match = self.default_match()
