"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsManagedMicrosoftAdR53 = exports.AwsManagedMicrosoftAd = exports.AwsManagedMicrosoftConfigurationStoreType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const skylight = require("../index");
var AwsManagedMicrosoftConfigurationStoreType;
(function (AwsManagedMicrosoftConfigurationStoreType) {
    AwsManagedMicrosoftConfigurationStoreType["SSM"] = "AWS Systems Manager Parameter Store";
})(AwsManagedMicrosoftConfigurationStoreType || (exports.AwsManagedMicrosoftConfigurationStoreType = AwsManagedMicrosoftConfigurationStoreType = {}));
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
class AwsManagedMicrosoftAd extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.props.domainName = props.domainName ?? 'domain.aws';
        this.props.edition = props.edition ?? 'Standard';
        this.props.secretName = props.secretName ?? `${props.domainName}-secret`;
        this.props.createWorker = props.createWorker ?? true;
        this.adParameters = props.configurationStore ?? {
            configurationStoreType: AwsManagedMicrosoftConfigurationStoreType.SSM,
        };
        this.adParameters.secretPointer =
            this.adParameters.secretPointer ?? 'domain-secret';
        this.adParameters.directoryIDPointer =
            this.adParameters.directoryIDPointer ?? 'directoryID';
        if (this.adParameters.namespace) {
            this.adParameters.namespace = `${this.adParameters.namespace}/authentication/mad`;
        }
        else {
            this.adParameters.namespace = 'cdk-skylight/authentication/mad';
        }
        this.secret =
            this.props.secret ??
                new aws_cdk_lib_1.aws_secretsmanager.Secret(this, 'Secret', {
                    generateSecretString: {
                        secretStringTemplate: JSON.stringify({
                            Domain: props.domainName,
                            UserID: 'Admin',
                        }),
                        generateStringKey: 'Password',
                        excludePunctuation: true,
                    },
                    secretName: props.secretName,
                });
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'mad-secretName-pointer', {
            parameterName: `/${this.adParameters.namespace}/${this.adParameters.secretPointer}`,
            stringValue: this.props.secretName,
        });
        let subnets;
        if (props.vpcSubnets) {
            if (props.vpcSubnets.hasPublic || props.vpcSubnets.subnets.length !== 2) {
                throw new Error('A public subnet or not exactly 2 subnets where passed in, please pass in two private subnets');
            }
            subnets = props.vpcSubnets;
        }
        else {
            subnets =
                props.vpc.selectSubnets({
                    subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
                }) ??
                    props.vpc.selectSubnets({
                        subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED,
                    });
        }
        new aws_cdk_lib_1.CfnOutput(this, 'secret-value-hint', {
            value: `aws secretsmanager get-secret-value --secret-id ${this.secret.secretArn} --query SecretString --output text --region ${aws_cdk_lib_1.Stack.of(scope).region}`,
        });
        this.microsoftAD = new aws_cdk_lib_1.aws_directoryservice.CfnMicrosoftAD(this, 'AWS-Managed-Microsoft-AD', {
            password: this.secret
                .secretValueFromJson('Password')
                .unsafeUnwrap.toString(),
            edition: props.edition,
            name: this.props.domainName,
            vpcSettings: {
                subnetIds: [subnets.subnetIds[0], subnets.subnetIds[1]],
                vpcId: props.vpc.vpcId,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-dns-ips', {
            value: `${aws_cdk_lib_1.Fn.join(',', this.microsoftAD.attrDnsIpAddresses)}`,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-dns-name', {
            value: `${this.props.domainName}`,
        });
        new aws_cdk_lib_1.CfnOutput(this, 'mad-directoyID', {
            value: `${this.microsoftAD.ref}`,
        });
        new aws_cdk_lib_1.aws_ssm.StringParameter(this, 'mad-directoryID-pointer', {
            parameterName: `/${this.adParameters.namespace}/${this.adParameters.directoryIDPointer}`,
            stringValue: this.microsoftAD.ref,
        });
        if (this.props.createWorker) {
            this.domainWindowsNode = this.createWorker(this.props.domainName, this.secret);
            this.domainWindowsNode.runPSwithDomainAdmin([
                'Add-WindowsFeature RSAT-AD-PowerShell',
                'Stop-Computer -ComputerName localhost',
            ], 'ad-powershell');
            this.domainWindowsNode.instance.node.addDependency(this.microsoftAD);
        }
        else {
            this.domainWindowsNode = undefined;
        }
    }
    // Creates DomainWindowsNode that will be used to run admin-tasks to this directory
    createWorker(domainName, domainPassword) {
        return new skylight.compute.DomainWindowsNode(this, 'madWorker', {
            domainName: domainName,
            passwordObject: domainPassword,
            vpc: this.props.vpc,
            instanceType: 't3.small',
            usePrivateSubnet: true,
        });
    }
    // The function creates a Lambda to Start the Windows Worker, then creates SSM Document and Desired state in State Manager to schedule this document on the Worker.
    createADGroup(groupName, groupDescription) {
        if (this.domainWindowsNode) {
            this.domainWindowsNode.startInstance();
            this.domainWindowsNode.runPSwithDomainAdmin([
                `New-ADGroup -Name "${groupDescription}" -SamAccountName "${groupName}" -GroupScope DomainLocal`,
                'Stop-Computer -ComputerName localhost',
            ], 'createAdGroup');
        }
        else {
            console.log("Can't create AD group when no Worker is defined");
        }
    }
    // Experimental
    createServiceAccount(adServiceAccountName, servicePrincipalNames, principalsAllowedToRetrieveManagedPassword) {
        if (this.domainWindowsNode) {
            this.domainWindowsNode.runPSwithDomainAdmin([
                `New-ADServiceAccount -Name "${adServiceAccountName}" -DnsHostName "${adServiceAccountName}.${this.props.domainName}" -ServicePrincipalNames "${servicePrincipalNames}" -PrincipalsAllowedToRetrieveManagedPassword "${principalsAllowedToRetrieveManagedPassword}"`,
            ], 'createServiceAccount');
        }
        else {
            console.log("Can't createServiceAccount when no Worker is defined");
        }
    }
}
exports.AwsManagedMicrosoftAd = AwsManagedMicrosoftAd;
_a = JSII_RTTI_SYMBOL_1;
AwsManagedMicrosoftAd[_a] = { fqn: "cdk-skylight.authentication.AwsManagedMicrosoftAd", version: "1.1.851" };
/**
 * A Ad Authentication represents an integration pattern of Managed AD and Route 53 Resolver in a specific VPC
 *
 * The Construct creates Managed AD with the provided Secret (Secrets Manager) or generates a new Secret.
 * The secret saved to SSM parameter store so others can use it with other Constructs (Such as Windows node or FSx)
 * The provided VPC or the new created VPC will be configured to forward DNS requests to the Managed AD with Route53 Resolvers
 * The construct also creates (optionally) t3.nano machine that is part of the domain that can be used to run admin-tasks (such as createADGroup)
 *
 * The createADGroup() method creates an Active Directory permission group in the domain, using the domain admin user.
 * Please note: When calling createADGroup() API, a Lambda will be created to start the worker machine (Using AWS-SDK),
 * then each command will be scheduled with State Manager, and the instance will be shut down after complete.
 *
 */
class AwsManagedMicrosoftAdR53 extends AwsManagedMicrosoftAd {
    constructor(scope, id, props) {
        super(scope, id, props);
        let subnets;
        if (props.vpcSubnets) {
            if (props.vpcSubnets.hasPublic || props.vpcSubnets.subnets.length !== 2) {
                throw new Error('A public subnet or not exactly 2 subnets where passed in, please pass in two private subnets');
            }
            subnets = props.vpcSubnets;
        }
        else {
            subnets =
                props.vpc.selectSubnets({
                    subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
                }) ??
                    props.vpc.selectSubnets({
                        subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_ISOLATED,
                    });
        }
        const sg = new aws_cdk_lib_1.aws_ec2.SecurityGroup(this, 'r53-outbound-resolver-SG', {
            vpc: props.vpc,
        });
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.udp(53));
        sg.addIngressRule(aws_cdk_lib_1.aws_ec2.Peer.ipv4(props.vpc.vpcCidrBlock), aws_cdk_lib_1.aws_ec2.Port.tcp(53));
        const outBoundResolver = new aws_cdk_lib_1.aws_route53resolver.CfnResolverEndpoint(this, 'R53-Resolver-Endpoint', {
            direction: 'OUTBOUND',
            ipAddresses: subnets.subnetIds.map((s) => {
                return { subnetId: s };
            }),
            securityGroupIds: [sg.securityGroupId],
        });
        const resolverRules = new aws_cdk_lib_1.aws_route53resolver.CfnResolverRule(this, 'R53-Resolve-Rule', {
            domainName: this.props.domainName,
            resolverEndpointId: outBoundResolver.ref,
            ruleType: 'FORWARD',
            targetIps: [
                { ip: aws_cdk_lib_1.Fn.select(0, this.microsoftAD.attrDnsIpAddresses) },
                { ip: aws_cdk_lib_1.Fn.select(1, this.microsoftAD.attrDnsIpAddresses) },
            ],
        });
        new aws_cdk_lib_1.aws_route53resolver.CfnResolverRuleAssociation(this, 'R53-Resolver-Association', {
            resolverRuleId: resolverRules.attrResolverRuleId,
            vpcId: props.vpc.vpcId,
        });
    }
}
exports.AwsManagedMicrosoftAdR53 = AwsManagedMicrosoftAdR53;
_b = JSII_RTTI_SYMBOL_1;
AwsManagedMicrosoftAdR53[_b] = { fqn: "cdk-skylight.authentication.AwsManagedMicrosoftAdR53", version: "1.1.851" };
//# sourceMappingURL=data:application/json;base64,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