# Copyright 2021 CR.Sparse Development Team
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     https://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""
Orthogonal wavelets
"""
import math

from jax import jit
import jax.numpy as jnp

from .dyad import *
from .transform import *


def wavelet_function(qmf, j, k, n):
    """
    Returns the (j,k)-th wavelet function for an orthogonal wavelet

    Inputs: 
      j, k - scale and location indices
      n - signal length (dyadic)
    """
    w = jnp.zeros(n)
    # identify the index of the j-th scale at k-th translation
    index = 2**j + k
    w = w.at[index].set(1)         
    return inverse_periodized_orthogonal(qmf, w, j)

def scaling_function(qmf, j, k, n):
    """
    Returns the (j,k)-th scaling function for an orthogonal wavelet

    Inputs: 
      j, k - scale and location indices
      n - signal length (dyadic)
    """
    w = jnp.zeros(n)
    # k-th translate in the coarsest part of the wavelet coefficients
    w = w.at[k].set(1)
    return inverse_periodized_orthogonal(qmf, w, j)

def haar():
    """
    Returns the quadrature mirror filter for the Haar wavelets
    """
    return jnp.array([1., 1]) / math.sqrt(2)



def db4():
    return jnp.array([.482962913145, .836516303738, .224143868042, -.129409522551, ])

def db6():
    return jnp.array([.332670552950, .806891509311, .459877502118, 
        -.135011020010, -.085441273882,  .035226291882])

def db8():
    return jnp.array([.230377813309, .714846570553, .630880767930, -.027983769417, 
        -.187034811719, .030841381836, .032883011667, -.010597401785])

def db10():
    return jnp.array([.160102397974,   .603829269797,   .724308528438, 
                .138428145901,   -.242294887066,  -.032244869585,
                .077571493840,   -.006241490213,  -.012580751999,
                .003335725285])

def db12():
    return jnp.array([.111540743350,   .494623890398,   .751133908021, 
                .315250351709,   -.226264693965,  -.129766867567,
                .097501605587,   .027522865530,   -.031582039317,
                .000553842201,   .004777257511,   -.001077301085,])

def db14():
    return jnp.array([.077852054085,   .396539319482,   .729132090846, 
                .469782287405,   -.143906003929,  -.224036184994,
                .071309219267,   .080612609151,   -.038029936935,
                -.016574541631,  .012550998556,   .000429577973,
                -.001801640704,  .000353713800, ])

def db16():
    return jnp.array([.054415842243,   .312871590914,   .675630736297, 
                .585354683654,   -.015829105256,  -.284015542962,
                .000472484574,   .128747426620,   -.017369301002,
                -.044088253931,  .013981027917,   .008746094047,
                -.004870352993,  -.000391740373,  .000675449406, 
                -.000117476784, ])

def db18():
    return jnp.array([.038077947364,   .243834674613,   .604823123690, 
                .657288078051,   .133197385825,   -.293273783279,
                -.096840783223,  .148540749338,   .030725681479, 
                -.067632829061,  .000250947115,   .022361662124, 
                -.004723204758,  -.004281503682,  .001847646883, 
                .000230385764,   -.000251963189,  .000039347320,])

def db20():
    return jnp.array([.026670057901,   .188176800078,   .527201188932, 
                .688459039454,   .281172343661,   -.249846424327,
                -.195946274377,  .127369340336,   .093057364604, 
                -.071394147166,  -.029457536822,  .033212674059, 
                .003606553567,   -.010733175483,  .001395351747, 
                .001992405295,   -.000685856695,  -.000116466855,
                .000093588670,   -.000013264203,])


def baylkin():
    return jnp.array([.099305765374,   .424215360813,   .699825214057, 
            .449718251149,   -.110927598348,  -.264497231446,
            .026900308804,   .155538731877,   -.017520746267,
            -.088543630623,  .019679866044,   .042916387274,
            -.017460408696,  -.014365807969,  .010040411845, 
            .001484234782,   -.002736031626,  .000640485329,])


def coif1():
    return jnp.array([.038580777748,  -.126969125396, -.077161555496,
                    .607491641386,  .745687558934,  .226584265197,])

def coif2():
    return jnp.array([.016387336463,  -.041464936782, -.067372554722,
                    .386110066823,  .812723635450,  .417005184424,
                    -.076488599078, -.059434418646, .023680171947,
                    .005611434819,  -.001823208871, -.000720549445,])

def coif3():
    return jnp.array([-.003793512864, .007782596426,  .023452696142,
                    -.065771911281, -.061123390003, .405176902410,
                    .793777222626,  .428483476378,  -.071799821619,
                    -.082301927106, .034555027573,  .015880544864,
                    -.009007976137, -.002574517688, .001117518771,
                    .000466216960,  -.000070983303, -.000034599773,])

def coif4():
    return jnp.array([.000892313668,  -.001629492013, -.007346166328,
                    .016068943964,  .026682300156,  -.081266699680,
                    -.056077313316, .415308407030,  .782238930920,
                    .434386056491,  -.066627474263, -.096220442034,
                    .039334427123,  .025082261845,  -.015211731527,
                    -.005658286686, .003751436157,  .001266561929,
                    -.000589020757, -.000259974552, .000062339034,
                    .000031229876,  -.000003259680, -.000001784985,])

def coif5():
    return jnp.array([-.000212080863, .000358589677,  .002178236305,
                    -.004159358782, -.010131117538, .023408156762,
                    .028168029062,  -.091920010549, -.052043163216,
                    .421566206729,  .774289603740,  .437991626228,
                    -.062035963906, -.105574208706, .041289208741,
                    .032683574283,  -.019761779012, -.009164231153,
                    .006764185419,  .002433373209,  -.001662863769,
                    -.000638131296, .000302259520,  .000140541149,
                    -.000041340484, -.000021315014, .000003734597,
                    .000002063806,  -.000000167408, -.000000095158,])


def symm4():
    return jnp.array([-.107148901418, -.041910965125, .703739068656, 
                1.136658243408, .421234534204,  -.140317624179,
                -.017824701442, .045570345896]) / math.sqrt(2)

def symm5():
    return jnp.array([.038654795955,  .041746864422,  -.055344186117,
                .281990696854,  1.023052966894, .896581648380, 
                .023478923136,  -.247951362613, -.029842499869,
                .027632152958]) / math.sqrt(2)

def symm6():
    return jnp.array([.021784700327,  .004936612372,  -.166863215412,
                -.068323121587, .694457972958,  1.113892783926,
                .477904371333,  -.102724969862, -.029783751299,
                .063250562660,  .002499922093,  -.011031867509]) / math.sqrt(2)

def symm7():
    return jnp.array([.003792658534,  -.001481225915, -.017870431651,
                .043155452582,  .096014767936,  -.070078291222,
                .024665659489,  .758162601964,  1.085782709814,
                .408183939725,  -.198056706807, -.152463871896,
                .005671342686,  .014521394762]) / math.sqrt(2)

def symm8():
    return jnp.array([.002672793393,  -.000428394300, -.021145686528,
                .005386388754,  .069490465911,  -.038493521263,
                -.073462508761, .515398670374,  1.099106630537,
                .680745347190,  -.086653615406, -.202648655286,
                .010758611751,  .044823623042,  -.000766690896, 
                -.004783458512]) / math.sqrt(2)

def symm9():
    return jnp.array([.001512487309,  -.000669141509, -.014515578553,
                .012528896242,  .087791251554,  -.025786445930,
                -.270893783503, .049882830959,  .873048407349, 
                1.015259790832, .337658923602,  -.077172161097,
                .000825140929,  .042744433602,  -.016303351226,
                -.018769396836, .000876502539,  .001981193736]) / math.sqrt(2)

def symm10():
    return jnp.array([.001089170447,  .000135245020,  -.012220642630,
                -.002072363923, .064950924579,  .016418869426, 
                -.225558972234, -.100240215031, .667071338154, 
                1.088251530500, .542813011213,  -.050256540092,
                -.045240772218, .070703567550,  .008152816799, 
                -.028786231926, -.001137535314, .006495728375, 
                .000080661204,  -.000649589896]) / math.sqrt(2)

def vaidyanathan():
    return jnp.array([-.000062906118, .000343631905,  -.000453956620,
            -.000944897136, .002843834547,  .000708137504, 
            -.008839103409, .003153847056,  .019687215010, 
            -.014853448005, -.035470398607, .038742619293, 
            .055892523691,  -.077709750902, -.083928884366,
            .131971661417,  .135084227129,  -.194450471766,
            -.263494802488, .201612161775,  .635601059872, 
            .572797793211,  .250184129505,  .045799334111])