ckan.module("files--resource-select", function ($) {
    return {
        options: {
            successEvent: "files-file-created",
            errorEvent: "files-file-failed",
            select: null,
            initialValue: null,
        },
        initialize() {
            if (!this.options.select) {
                console.error("files--resource-select cannot be initialized without `select` option");
                return;
            }
            $.proxyAll(this, /_on/);
            this.el.on({
                [this.options.successEvent]: this._onSuccess,
                [this.options.errorEvent]: this._onError,
            });
            this.url = document.getElementById("field-resource-url");
            this.select = $(this.options.select);
            if (this.options.initialValue) {
                this.select.select2("data", this.options.initialValue);
            }
        },
        _onSuccess({ detail }) {
            this.select.select2("data", { id: detail.id, text: detail.name });
        },
        _onError({ detail: err, }) {
            if (typeof err === "string") {
                this.reportError("Error", err);
            }
            else {
                for (let [field, problems] of Object.entries(err)) {
                    if (field.startsWith("__"))
                        continue;
                    this.reportError(field, problems.join(","));
                }
            }
        },
        reportError(label, message) {
            const box = this.sandbox.notify.initialize(this.sandbox.notify.create(label, message));
            this.el.closest("label").parent().after(box);
        },
    };
});
/**
 * Upload file and trigger event with file data on specified element.
 *
 */
ckan.module("files--auto-upload", function ($) {
    return {
        options: {
            adapter: "Standard",
            spinner: null,
            action: null,
            successEvent: "files-file-created",
            errorEvent: "files-file-failed",
            eventTarget: null,
            copyIdInto: null,
            requestParams: null,
        },
        queue: null,
        initialize() {
            if (!this.options.action) {
                console.error("files--auto-upload cannot be initialized without `action` option");
                return;
            }
            $.proxyAll(this, /_on/);
            this.queue = new Set();
            this.el.on("change", (event) => this.upload(...event.target.files));
            this.spinner = $(this.options.spinner);
            this.field = $(this.options.field);
            this.submits = this.el
                .closest("form")
                .find("input[type=submit],button[type=submit]");
            this.idTarget = $(this.options.copyIdInto);
            this.uploader = this.sandbox.files.makeUploader(this.options.adapter, {
                uploadAction: this.options.action,
            });
            this.uploader.addEventListener("progress", console.log);
        },
        upload(...files) {
            files.forEach(async (file) => {
                this.queue.add(file);
                this.refreshFormState();
                const options = {
                    uploader: this.uploader,
                    requestParams: { ...this.options.requestParams, multipart: this.uploader instanceof ckan.CKANEXT_FILES.adapters.Multipart },
                };
                this.sandbox.files
                    .upload(file, options)
                    .then((result) => {
                    this.idTarget.val(result.id);
                    this.dispatchResult(this.options.successEvent, result);
                }, (err) => this.dispatchResult(this.options.errorEvent, err))
                    .then(() => {
                    this.queue.delete(file);
                    this.refreshFormState();
                });
            });
        },
        dispatchResult(event, detail) {
            const target = this.options.eventTarget
                ? $(this.options.eventTarget)
                : this.el;
            target[0].dispatchEvent(new CustomEvent(event, { detail }));
        },
        refreshFormState() {
            this.spinner.prop("hidden", !this.queue.size);
            this.submits.prop("disabled", !!this.queue.size);
        },
    };
});
/**
 * Add selected file to upload queue whenever `[data-queue-scheduler]`
 * dispatches `change` event.
 *
 */
ckan.module("files--scheduler", function ($) {
    return {
        options: {
            immediate: false,
        },
        initialize() {
            const scheduler = this.$("[data-queue-scheduler]");
            scheduler.on("drop", (event) => event.preventDefault());
            scheduler.on("change", (event) => this.push(...event.target.files));
        },
        push(...files) {
            files.forEach((file) => this.sandbox.publish(ckan.CKANEXT_FILES.topics.addFileToQueue, file, { immediate: this.options.immediate }));
        },
    };
});
/**
 * Add file/directories to upload queue via drag'n'drop.
 *
 */
ckan.module("files--dropzone", function ($) {
    return {
        options: {
            immediate: false,
        },
        initialize() {
            $.proxyAll(this, /_on/);
            const element = this.el[0];
            element.addEventListener("dragover", this._onDragOver);
            element.addEventListener("dragenter", this._onDragEnter);
            element.addEventListener("dragleave", this._onDragLeave);
            element.addEventListener("drop", this._onDrop);
        },
        _onDragOver(event) {
            event.preventDefault();
        },
        _onDragEnter(event) { },
        _onDragLeave(event) { },
        _onDrop(event) {
            event.preventDefault();
            if (!event.dataTransfer) {
                return;
            }
            for (let entry of event.dataTransfer.items) {
                this.scanEntry(entry.webkitGetAsEntry(), (file) => this.push(file));
            }
        },
        scanEntry(entry, cb) {
            if (entry.isFile) {
                entry.file(cb);
            }
            else {
                entry
                    .createReader()
                    .readEntries((entries) => entries.forEach((e) => this.scanEntry(e, cb)));
            }
        },
        push(file) {
            this.sandbox.publish(ckan.CKANEXT_FILES.topics.addFileToQueue, file, { immediate: this.options.immediate });
        },
    };
});
/**
 * Add to queue a file, that has associated incomplete upload.
 *
 * Supports a number of properties to verify that the new file matches
 * previously uploaded file.
 *
 *
 */
ckan.module("files--restorer", function ($) {
    return {
        options: {
            name: "",
            size: 0,
            uploaded: 0,
            id: "",
            immediate: false,
        },
        initialize() {
            $.proxyAll(this, /_on/);
            this.el.on("change", this._onChange);
        },
        _onChange(event) {
            const file = event.target.files?.[0];
            if (!file) {
                return;
            }
            if (this.options.name && file.name !== this.options.name) {
                this.sandbox.notify("Name mismatch.", `Expected name: ${this.options.name}`);
                this.sandbox.notify.el[0].scrollIntoView();
                return;
            }
            if (this.options.size && file.size !== this.options.size) {
                this.sandbox.notify("Size mismatch.", `Expected size: ${this.options.size.toLocaleString()} bytes`);
                this.sandbox.notify.el[0].scrollIntoView();
                return;
            }
            this.sandbox.publish(ckan.CKANEXT_FILES.topics.restoreFileInQueue, file, {
                id: this.options.id,
                uploaded: this.options.uploaded,
                immediate: this.options.immediate,
            });
        },
    };
});
ckan.module("files--shared-queue", function ($) {
    return {
        initialize() {
            $.proxyAll(this, /_on/);
            this.worker = new SharedWorker(this.sandbox.url("ckanext-files/scripts/files--shared-uploader.js"));
            this.worker.port.onmessage = console.debug;
        },
    };
});
ckan.module("files--queue", function ($) {
    return {
        options: {
            storage: "default",
            uploader: "Standard",
        },
        initialize() {
            $.proxyAll(this, /_on/);
            ckan.pubsub.subscribe(ckan.CKANEXT_FILES.topics.addFileToQueue, this._onFile);
            ckan.pubsub.subscribe(ckan.CKANEXT_FILES.topics.restoreFileInQueue, this._onFile);
            this.tpl = this.$("[data-upload-template]")
                .remove()
                .removeAttr("data-upload-template hidden");
            this.widgets = new WeakMap();
        },
        teardown() {
            ckan.pubsub.unsubscribe(ckan.CKANEXT_FILES.topics.addFileToQueue, this._onFile);
            ckan.pubsub.unsubscribe(ckan.CKANEXT_FILES.topics.restoreFileInQueue, this._onFile);
        },
        _onFile(file, options = {
            immediate: false,
            id: "",
            uploaded: 0,
            uploaderInstance: null,
            uploader: null,
            storage: null,
        }) {
            const widget = this.tpl.clone(true).appendTo(this.el);
            const info = {
                file,
                id: options.id,
                uploaded: options.uploaded || 0,
                uploader: options.uploaderInstance ||
                    this.sandbox.files.makeUploader(options.uploader || this.options.uploader, { storage: options.storage || this.options.storage }),
            };
            this.widgets.set(widget[0], info);
            widget.on("click", "[data-upload-resume]", this._onWidgetResume);
            widget.on("click", "[data-upload-pause]", this._onWidgetPause);
            info.uploader.addEventListener("fail", ({ detail: { reasons, file }, }) => {
                this.sandbox.notify(file.name, Object.entries(reasons)
                    .filter(([k, v]) => k[0] !== "_")
                    .map(([k, v]) => Array.isArray(v) ? v.join("; ") : v)
                    .join("; "));
                this.sandbox.notify.el[0].scrollIntoView();
                this.toggleAnimation(widget, false);
                widget
                    .find("[data-upload-progress]")
                    .removeClass("bg-primary bg-secondary")
                    .addClass("bg-danger progress-bar-danger");
            });
            info.uploader.addEventListener("error", ({ detail: { message, file }, }) => {
                this.sandbox.notify(file.name, message);
                this.sandbox.notify.el[0].scrollIntoView();
                this.toggleAnimation(widget, false);
                widget
                    .find("[data-upload-progress]")
                    .removeClass("bg-primary bg-secondary")
                    .addClass("bg-danger progress-bar-danger");
            });
            info.uploader.addEventListener("progress", ({ detail: { loaded, total } }) => this.setWidgetCompletion(widget, loaded, total));
            info.uploader.addEventListener("finish", ({ detail: { file, result } }) => {
                this.toggleAnimation(widget, false);
                widget
                    .find("[data-upload-progress]")
                    .removeClass("bg-primary bg-secondary")
                    .addClass("bg-success progress-bar-success");
                this.sandbox.publish(ckan.CKANEXT_FILES.topics.queueItemUploaded, file, result);
            });
            this.setWidgetName(widget, info.file.name);
            this.setWidgetCompletion(widget, info.uploaded, info.file.size);
            if (options.immediate) {
                widget.find("[data-upload-resume]").trigger("click");
            }
        },
        setWidgetName(widget, name) {
            widget.find("[data-item-name]").text(name);
        },
        setWidgetCompletion(widget, uploaded, total) {
            const value = (uploaded * 100) / total;
            const info = this.widgets.get(widget[0]);
            info.uploaded = uploaded;
            const completion = value.toFixed(0) + "%";
            widget
                .find("[data-upload-progress]")
                .text(completion)
                .css("width", completion);
        },
        toggleAnimation(widget, state) {
            widget
                .find("[data-upload-progress]")
                .toggleClass("progress-bar-animated active", state);
        },
        _onWidgetResume(event) {
            const info = this.widgets.get(event.delegateTarget);
            if (info.uploaded >= info.total)
                return;
            const widget = $(event.delegateTarget);
            widget
                .find("[data-upload-progress]")
                .removeClass("bg-secondary")
                .addClass("bg-primary");
            if (info.id) {
                info.uploader.resume(info.file, info.id);
            }
            else {
                info.uploader.upload(info.file);
            }
            this.toggleAnimation(widget, true);
        },
        _onWidgetPause(event) {
            const info = this.widgets.get(event.delegateTarget);
            if (info.uploaded >= info.total)
                return;
            const widget = $(event.delegateTarget);
            widget
                .find("[data-upload-progress]")
                .removeClass("bg-primary")
                .addClass("bg-secondary");
            info.uploader.pause(info.file);
            this.toggleAnimation(widget, false);
        },
    };
});
//# sourceMappingURL=data:application/json;base64,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
